"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 * @stability stable
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     *
     * The format is 'resource-server-identifier/scope'.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     * @stability stable
     */
    static custom(name) {
        return new OAuthScope(name);
    }
    /**
     * Adds a custom scope that's tied to a resource server in your stack.
     *
     * @stability stable
     */
    static resourceServer(server, scope) {
        return new OAuthScope(`${server.userPoolResourceServerId}/${scope.scopeName}`);
    }
}
exports.OAuthScope = OAuthScope;
_a = JSII_RTTI_SYMBOL_1;
OAuthScope[_a] = { fqn: "@aws-cdk/aws-cognito.OAuthScope", version: "1.90.1" };
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client.
 *
 * @stability stable
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.
 *
 * @stability stable
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient.
 *
 * @stability stable
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     *
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     * @stability stable
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
_b = JSII_RTTI_SYMBOL_1;
UserPoolClientIdentityProvider[_b] = { fqn: "@aws-cdk/aws-cognito.UserPoolClientIdentityProvider", version: "1.90.1" };
/**
 * Allow users to sign in using 'Facebook Login'.
 *
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 *
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 *
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client.
 *
 * @stability stable
 */
class UserPoolClient extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_e = (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.flows) !== null && _e !== void 0 ? _e : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_f = props.oAuth) === null || _f === void 0 ? void 0 : _f.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_g = props.oAuth) === null || _g === void 0 ? void 0 : _g.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
            readAttributes: (_h = props.readAttributes) === null || _h === void 0 ? void 0 : _h.attributes(),
            writeAttributes: (_j = props.writeAttributes) === null || _j === void 0 ? void 0 : _j.attributes(),
        });
        this.configureTokenValidity(resource, props);
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     *
     * @stability stable
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization, throws an error otherwise.
     *
     * @stability stable
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _d;
        const scopes = (_d = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _d !== void 0 ? _d : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
    configureTokenValidity(resource, props) {
        this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.idTokenValidity);
        this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.accessTokenValidity);
        this.validateDuration('refreshTokenValidity', core_1.Duration.minutes(60), core_1.Duration.days(10 * 365), props.refreshTokenValidity);
        if (props.refreshTokenValidity) {
            this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.idTokenValidity);
            this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.accessTokenValidity);
        }
        if (props.accessTokenValidity || props.idTokenValidity || props.refreshTokenValidity) {
            resource.tokenValidityUnits = {
                idToken: props.idTokenValidity ? 'minutes' : undefined,
                accessToken: props.accessTokenValidity ? 'minutes' : undefined,
                refreshToken: props.refreshTokenValidity ? 'minutes' : undefined,
            };
        }
        ;
        resource.idTokenValidity = props.idTokenValidity ? props.idTokenValidity.toMinutes() : undefined;
        resource.refreshTokenValidity = props.refreshTokenValidity ? props.refreshTokenValidity.toMinutes() : undefined;
        resource.accessTokenValidity = props.accessTokenValidity ? props.accessTokenValidity.toMinutes() : undefined;
    }
    validateDuration(name, min, max, value) {
        if (value === undefined) {
            return;
        }
        if (value.toMilliseconds() < min.toMilliseconds() || value.toMilliseconds() > max.toMilliseconds()) {
            throw new Error(`${name}: Must be a duration between ${min.toHumanString()} and ${max.toHumanString()} (inclusive); received ${value.toHumanString()}.`);
        }
    }
}
exports.UserPoolClient = UserPoolClient;
_c = JSII_RTTI_SYMBOL_1;
UserPoolClient[_c] = { fqn: "@aws-cdk/aws-cognito.UserPoolClient", version: "1.90.1" };
//# sourceMappingURL=data:application/json;base64,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