from nvidia.dali.pipeline import Pipeline
from nvidia.dali.plugin.pytorch import DALIClassificationIterator, LastBatchPolicy
import nvidia.dali.fn as fn
import nvidia.dali.types as types
import nvidia.dali.tfrecord as tfrec
import torch
import os
import glob
import numpy as np
from titans.dataloader.utils import RandAugment
from colossalai.core import global_context as gpc
from colossalai.context import ParallelMode

__all__ = ['DaliDataloaderWithRandAug', 'DaliDataloader', 'build_dali_imagenet']


class DaliDataloaderWithRandAug(DALIClassificationIterator):

    def __init__(self,
                 tfrec_filenames,
                 tfrec_idx_filenames,
                 shard_id=0,
                 num_shards=1,
                 batch_size=128,
                 num_threads=4,
                 resize=256,
                 crop=224,
                 prefetch=2,
                 training=True,
                 gpu_aug=False,
                 cuda=True,
                 mixup_alpha=0.0,
                 randaug_magnitude=10,
                 randaug_num_layers=0):
        self.mixup_alpha = mixup_alpha
        self.training = training
        self.randaug_magnitude = randaug_magnitude
        self.randaug_num_layers = randaug_num_layers
        pipe = Pipeline(batch_size=batch_size,
                        num_threads=num_threads,
                        device_id=torch.cuda.current_device() if cuda else None,
                        seed=42)
        with pipe:
            inputs = fn.readers.tfrecord(path=tfrec_filenames,
                                         index_path=tfrec_idx_filenames,
                                         random_shuffle=training,
                                         shard_id=shard_id,
                                         num_shards=num_shards,
                                         initial_fill=10000,
                                         read_ahead=True,
                                         prefetch_queue_depth=prefetch,
                                         name='Reader',
                                         features={
                                             'image/encoded': tfrec.FixedLenFeature((), tfrec.string, ""),
                                             'image/class/label': tfrec.FixedLenFeature([1], tfrec.int64, -1),
                                         })
            images = inputs["image/encoded"]
            images = fn.decoders.image(images, device='mixed' if gpu_aug else 'cpu', output_type=types.RGB)
            if training:
                images = fn.random_resized_crop(images, size=crop, device='gpu' if gpu_aug else 'cpu')
                if randaug_num_layers == 0:
                    flip_lr = fn.random.coin_flip(probability=0.5)
                    images = fn.flip(images, horizontal=flip_lr)
            else:
                images = fn.resize(images,
                                   device='gpu' if gpu_aug else 'cpu',
                                   resize_x=resize,
                                   resize_y=resize,
                                   dtype=types.FLOAT,
                                   interp_type=types.INTERP_TRIANGULAR)
                images = fn.crop(images, dtype=types.FLOAT, crop=(crop, crop))
            label = inputs["image/class/label"] - 1    # 0-999
            if cuda:    # transfer data to gpu
                pipe.set_outputs(images.gpu(), label.gpu())
            else:
                pipe.set_outputs(images, label)

        pipe.build()
        last_batch_policy = LastBatchPolicy.DROP if training else LastBatchPolicy.PARTIAL
        super().__init__(pipe, reader_name="Reader", auto_reset=True, last_batch_policy=last_batch_policy)

    def __iter__(self):
        # if not reset (after an epoch), reset; if just initialize, ignore
        if self._counter >= self._size or self._size < 0:
            self.reset()
        return self

    def __next__(self):
        data = super().__next__()
        img, label = data[0]['data'], data[0]['label']
        img = img.permute(0, 3, 1, 2)
        if self.randaug_num_layers > 0 and self.training:
            img = RandAugment(img, num_layers=self.randaug_num_layers, magnitude=self.randaug_magnitude)
        img = (img - 127.5) / 127.5
        label = label.squeeze()
        if self.mixup_alpha > 0.0:
            if self.training:
                lam = np.random.beta(self.mixup_alpha, self.mixup_alpha)
                idx = torch.randperm(img.size(0)).to(img.device)
                img = lam * img + (1 - lam) * img[idx, :]
                label_a, label_b = label, label[idx]
                lam = torch.tensor([lam], device=img.device, dtype=img.dtype)
                label = {'targets_a': label_a, 'targets_b': label_b, 'lam': lam}
            else:
                label = {
                    'targets_a': label,
                    'targets_b': label,
                    'lam': torch.ones(1, device=img.device, dtype=img.dtype)
                }
            return img, label
        return img, label


class DaliDataloader(DALIClassificationIterator):

    def __init__(self,
                 tfrec_filenames,
                 tfrec_idx_filenames,
                 shard_id=0,
                 num_shards=1,
                 batch_size=128,
                 num_threads=4,
                 resize=256,
                 crop=224,
                 prefetch=2,
                 training=True,
                 gpu_aug=False,
                 cuda=True):
        pipe = Pipeline(batch_size=batch_size,
                        num_threads=num_threads,
                        device_id=torch.cuda.current_device() if cuda else None,
                        seed=1024)
        with pipe:
            inputs = fn.readers.tfrecord(path=tfrec_filenames,
                                         index_path=tfrec_idx_filenames,
                                         random_shuffle=training,
                                         shard_id=shard_id,
                                         num_shards=num_shards,
                                         initial_fill=10000,
                                         read_ahead=True,
                                         prefetch_queue_depth=prefetch,
                                         name='Reader',
                                         features={
                                             'image/encoded': tfrec.FixedLenFeature((), tfrec.string, ""),
                                             'image/class/label': tfrec.FixedLenFeature([1], tfrec.int64, -1),
                                         })
            images = inputs["image/encoded"]

            if training:
                images = fn.decoders.image(images, device='mixed' if gpu_aug else 'cpu', output_type=types.RGB)
                images = fn.random_resized_crop(images, size=crop, device='gpu' if gpu_aug else 'cpu')
                flip_lr = fn.random.coin_flip(probability=0.5)
            else:
                # decode jpeg and resize
                images = fn.decoders.image(images, device='mixed' if gpu_aug else 'cpu', output_type=types.RGB)
                images = fn.resize(images,
                                   device='gpu' if gpu_aug else 'cpu',
                                   resize_x=resize,
                                   resize_y=resize,
                                   dtype=types.FLOAT,
                                   interp_type=types.INTERP_TRIANGULAR)
                flip_lr = False

            # center crop and normalise
            images = fn.crop_mirror_normalize(images,
                                              dtype=types.FLOAT,
                                              crop=(crop, crop),
                                              mean=[127.5],
                                              std=[127.5],
                                              mirror=flip_lr)
            label = inputs["image/class/label"] - 1    # 0-999
            # LSG: element_extract will raise exception, let's flatten outside
            if cuda:    # transfer data to gpu
                pipe.set_outputs(images.gpu(), label.gpu())
            else:
                pipe.set_outputs(images, label)

        pipe.build()
        last_batch_policy = LastBatchPolicy.FILL
        super().__init__(pipe, reader_name="Reader", auto_reset=True, last_batch_policy=last_batch_policy)

    def __iter__(self):
        # if not reset (after an epoch), reset; if just initialize, ignore
        if self._counter >= self._size or self._size < 0:
            self.reset()
        return self

    def __next__(self):
        try:
            data = super().__next__()
        except StopIteration:
            data = super().__next__()
        img, label = data[0]['data'], data[0]['label']
        label = label.squeeze()

        return img, label


def _build_dali_imagenet_train(root, rand_augment=False):
    train_pat = os.path.join(root, 'train/*')
    train_idx_pat = os.path.join(root, 'idx_files/train/*')
    if rand_augment:
        train_dataloader = DaliDataloaderWithRandAug(sorted(glob.glob(train_pat)),
                                                     sorted(glob.glob(train_idx_pat)),
                                                     batch_size=gpc.config.BATCH_SIZE,
                                                     shard_id=gpc.get_local_rank(ParallelMode.DATA),
                                                     num_shards=gpc.get_world_size(ParallelMode.DATA),
                                                     gpu_aug=gpc.config.dali.gpu_aug,
                                                     cuda=True,
                                                     mixup_alpha=gpc.config.dali.mixup_alpha,
                                                     randaug_num_layers=2)
    else:
        train_dataloader = DaliDataloader(
            sorted(glob.glob(train_pat)),
            sorted(glob.glob(train_idx_pat)),
            batch_size=gpc.config.BATCH_SIZE,
            shard_id=gpc.get_local_rank(ParallelMode.DATA),
            num_shards=gpc.get_world_size(ParallelMode.DATA),
            training=True,
            gpu_aug=False,
            cuda=False,
        )
    return train_dataloader


def _build_dali_imagenet_test(root, rand_augment=False):
    val_pat = os.path.join(root, 'validation/*')
    val_idx_pat = os.path.join(root, 'idx_files/validation/*')
    if rand_augment:
        test_dataloader = DaliDataloaderWithRandAug(sorted(glob.glob(val_pat)),
                                                    sorted(glob.glob(val_idx_pat)),
                                                    batch_size=gpc.config.BATCH_SIZE,
                                                    shard_id=gpc.get_local_rank(ParallelMode.DATA),
                                                    num_shards=gpc.get_world_size(ParallelMode.DATA),
                                                    training=False,
                                                    gpu_aug=False,
                                                    cuda=True,
                                                    mixup_alpha=gpc.config.dali.mixup_alpha)
    else:
        test_dataloader = DaliDataloader(
            sorted(glob.glob(val_pat)),
            sorted(glob.glob(val_idx_pat)),
            batch_size=gpc.config.BATCH_SIZE,
            shard_id=gpc.get_local_rank(ParallelMode.DATA),
            num_shards=gpc.get_world_size(ParallelMode.DATA),
            training=False,
            gpu_aug=False,
            cuda=False,
        )
    return test_dataloader


def build_dali_imagenet(root, rand_augment=False):
    train_dataloader = _build_dali_imagenet_train(root, rand_augment=rand_augment)
    test_dataloader = _build_dali_imagenet_test(root, rand_augment=rand_augment)
    return train_dataloader, test_dataloader
