"""
Paraview Result Processing
==========================

This module provides a class for loading and processing results from Paraview solution files.
It is designed to read VTU files generated by a simulation and organize the data for further
analysis or visualization.

"""

import os
import numpy as np
import pyvista as pv


class ParaviewResult:
    """
    A class to load and process Paraview solution files.

    Parameters
    ----------
    folder_path : str
        Path to the folder containing the Paraview solution files.
    steps : list of int
        List of time steps to load the corresponding solution files.

    Attributes
    ----------
    folder_path : str
        Path to the folder containing the Paraview solution files.
    paraview_folder : str
        Path to the folder where Paraview solution files are stored.
    steps : list of int
        List of time steps to load the corresponding solution files.
    data : numpy.ndarray
        Array of PyVista datasets loaded from the solution files.
    file : numpy.ndarray
        Array of file paths to the loaded solution files.
    """

    def __init__(self, folder_path, steps):
        self.folder_path = folder_path
        self.paraview_folder = os.path.join(
            self.folder_path, 'paraview-solutions_vtu')
        self.steps = steps
        self.data = np.array([])
        self.file = np.array([])

        for k in range(0, len(self.steps)):
            if self.steps[k] < 10:
                aux = '0000'
            elif 10 <= self.steps[k] < 100:
                aux = '000'
            elif 100 <= self.steps[k] < 1000:
                aux = '00'
            tN = os.path.join('fenicsx_p0_', str(
                aux), str(self.steps[k]), '.vtu')
            pN = pv.read(tN)
            # pN = pv.read(os.path.join(self.paraview_folder, tN))

            self.file = np.append(self.file, [tN])
            self.data = np.append(self.data, [pN])
