#!/usr/bin/env python
# coding: utf-8
"""script that makes source data .csv files for figures or tables
that summarize test set error of trained models,
with and without clean-up transformations applied to predictions

Note this script expects directories with the following structure:
results
└── experiment_name  # <-- you want to specify this path as `results_root` argument when calling this script
    ├── bird1  # <-- animal IDs
    │   └──results_210409_092101  # <-- results directories generated by runs of `vak train`
    |
    ├── bird2
    │   └──results_210409_011292
    ├ ...

Each results directory must contain an `eval` config file alone with the `train` config.
"""
from argparse import ArgumentParser
import configparser  # used to load 'min_segment_dur.ini'
from pathlib import Path

import pandas as pd
import pyprojroot

import article


def main(results_root,
         min_segment_dur_ini,
         csv_filename):
    results_root = Path(results_root)
    indiv_roots = sorted(
        [subdir for subdir in results_root.iterdir() if subdir.is_dir()]
    )

    # get minimum segment durations to use for clean up. Fail early if they're not there.
    config = configparser.ConfigParser()
    config.read(Path(min_segment_dur_ini).expanduser().resolve())
    min_segment_durs = {k: float(v) for k, v in config['min_segment_dur'].items()}
    for indiv_root in indiv_roots:
        if indiv_root.name not in min_segment_durs:
            raise ValueError(
                f"could not find a minimum segment duration for animal id: {indiv_root.name}. "
                f"Found the following animal IDs in the min_segment_dur.ini file: {min_segment_durs.keys()}"
            )

    all_eval_dfs = []

    for indiv_root in indiv_roots:
        results_dirs = sorted(indiv_root.glob('results_*'))
        most_recent_results = results_dirs[-1]
        min_segment_dur = min_segment_durs[indiv_root.name]
        eval_dfs = article.eval.train_with_transforms(results_root=most_recent_results,
                                                      min_segment_dur=min_segment_dur)
        eval_dfs['avg_error'] = 1 - eval_dfs['avg_acc']
        eval_dfs['animal_id'] = indiv_root.name
        all_eval_dfs.append(eval_dfs)

    result_eval_df = pd.concat(all_eval_dfs)
    result_eval_df['animal_id'] = result_eval_df['animal_id'].astype('category')
    result_eval_df['avg_error'] = result_eval_df['avg_error'] * 100
    
    csv_path = results_root.joinpath(csv_filename)
    result_eval_df.to_csv(csv_path, index=False)


PROJ_ROOT = pyprojroot.here()
BR_RESULTS_ROOT = PROJ_ROOT / 'results' / 'BirdsongRecognition' / 'default_spect_params'
MIN_SEGMENT_DUR_INI = PROJ_ROOT / 'data' / 'configs' / 'min_segment_dur.ini'


def get_parser():
    parser = ArgumentParser()
    parser.add_argument('--results_root',
                        help=('root directory where subdirectories are individuals from dataset, '
                              'and each subdirectory contains results folders from a run '
                              'of `vak learncurve`'),
                        default=BR_RESULTS_ROOT)
    parser.add_argument('--min_segment_dur_ini',
                        help=('path to .ini file with minimum segment durations '
                              'where option name is '),
                        default=MIN_SEGMENT_DUR_INI)
    parser.add_argument('--csv_filename',
                        help='filename of .csv that will be saved by this script in results_root',
                        default='error_across_birds_with_cleanup.csv')
    return parser


if __name__ == '__main__':
    parser = get_parser()
    args = parser.parse_args()
    main(results_root=args.results_root,
         min_segment_dur_ini=args.min_segment_dur_ini,
         csv_filename=args.csv_filename)
