from pathlib import Path

import joblib
import numpy as np
import pandas as pd

from sklearn.pipeline import make_pipeline
from sklearn.preprocessing import StandardScaler
from sklearn.utils.fixes import loguniform
# need to import this for HalvingRandomSearchCV to work!!!
from sklearn.experimental import enable_halving_search_cv
from sklearn.model_selection import HalvingRandomSearchCV
from sklearn.svm import SVC

from tqdm import tqdm


def fit(extract_csv_path,
        learncurve_csv_path,
        split='train',
        max_fits=4):
    """fit an SVM to features extracted
    by ``article.hvc.extract.extract``.

    Parameters
    ----------
    extract_csv_path : str, pathlib.Path
        path to .csv saved by ``article.hvc.extract``
    learncurve_csv_path : str, pathlib.Path
        path to original .csv generated by ``vak prep``,
         that was used to create the ``extract`` csv
    split : str
        split to use to fit models,
        default is 'train'
    max_fits : int
        maximum number of times to attempt to ``fit`` the data.
        After each ``fit`` this function tests that the number
        of classes the model predicts is equal to the true number
        of classes in the data. If this is True, the model is
        returned. If not, this function attempts to fit again,
        up to ``max_fits`` times. This loop is added to deal with
        "pathological" behavior where random hyperparameter search finds
        "best" values that cause the SVM to predict only
        one class, or less than the n classes. The current ranges used
        mostly mitigate this behavior, but for small duration training sets
        (30, 45s) it can still occur, roughly every ~1/10 replicates.

    Returns
    -------
    clf : sklearn.SVM.SVC
        instance, fit by pipeline

    Notes
    -----
    Performs an inner merge with
    the .csv saved by the ``extract``
    function and the original .csv saved by
    ``vak prep``, so that feature file paths in
    the ``extract`` csv are aligned with
    defined dataset splits in the ``prep`` csv.

    Then trains SVM using the specified split.

    Training is done with a ``sklearn`` pipeline
    that makes use of random search
    for hyperparameters, with halving.
    """
    extract_df = pd.read_csv(extract_csv_path)
    learncurve_df = pd.read_csv(learncurve_csv_path)

    # add 'features_path' from DataFrame saved by extract script
    # to DataFrame generated by `learncurve`
    # by doing an INNER JOIN
    learncurve_df = pd.merge(learncurve_df, extract_df, how='inner')

    # just use the split we want
    learncurve_df = learncurve_df[learncurve_df.split == split]

    ftr_paths = learncurve_df.features_path.values.tolist()
    df = []
    for ftr_path in tqdm(ftr_paths):
        df.append(
            pd.read_csv(ftr_path)
        )
    df = pd.concat(df)
    y = df.labels.values
    x = df[df.columns.drop('labels')].values

    tuned_parameters = {
        'svc__kernel': ['rbf'],
        # in practice found we never want large gamma,
        # and that even gamma < 1 can still cause pathological behavior
        # (model predicts all one class) when 0 < C < ~50.
        'svc__gamma': loguniform(1e-9, 1e-3),
        'svc__C': loguniform(60, 1e10)
    }

    n_classes = np.unique(y).shape[0]

    for fit_num in range(max_fits):
        print(
            f'fit {fit_num} of (maximum) {max_fits}.'
        )
        pipe = make_pipeline(StandardScaler(), SVC())
        # need to dynamically set min_resources so default isn't greater than number of samples in dataset
        DEFAULT_CV = 5
        # in next line, 2 is ``magic_factor`` from ``sklearn.model_selection._search_successive_halving.py``
        min_resources_ = DEFAULT_CV * 2 * n_classes
        if min_resources_ > x.shape[0]:
            for cv in (4, 3,):
                min_resources_ = cv * 2 * n_classes
                if min_resources_ < x.shape[0]:
                    break
            if min_resources_ > x.shape[0]:
                raise ValueError("could not find min_resources_ value smaller than number of samples")
        else:
            cv = DEFAULT_CV
        clf = HalvingRandomSearchCV(
            pipe, tuned_parameters, factor=3, min_resources=min_resources_, cv=cv, n_jobs=-2, verbose=1,
        )
        clf.fit(x, y)
        n_classes_preds = np.unique(clf.predict(x)).shape[0]
        if n_classes_preds == n_classes:
            print(
                f'number of classes predicted by model, {n_classes_preds}, matches '
                f'number of classes in ground truth data. Done fitting.'
            )
            break
        elif n_classes_preds < n_classes:
            if fit_num < max_fits - 1:
                print(
                    f'number of classes predicted by model, {n_classes_preds}, was '
                    f'less than number of classes in ground truth data, {n_classes}. '
                    f'Attempting random hyperparameter search again.'
                )
            else:
                raise ValueError(
                    f'number of classes predicted by model, {n_classes_preds}, was '
                    f'less than number of classes in ground truth data, {n_classes}. '
                    f'Was unable to fit model in less than {max_fits} attempts.'
                )
        elif n_classes_preds > n_classes:
            raise ValueError(
                f'number of classes predicted by model, {n_classes_preds}, was '
                f'more than number of classes in ground truth data, {n_classes}. '
            )

    return clf


def save_clf(clf,
             csv_path,
             clf_dst):
    """save trained scikit-learn
    classifier, using ``joblib``

    Parameters
    ----------
    clf :
    csv_path : str, pathlib.Path
        path to csv created by ``extract``
        with features used to train classifier.
        Associates classifier with training set
        via the filename.
    clf_dst : str, pathlib.Path
        directory where ``joblib`` should save
        binary blob of classifier

    Returns
    -------
    clf_path : pathlib.Path
        path to saved classifier.
        Used when running ``article.hvc.predict.predict``.
    """
    csv_path = Path(csv_path)
    clf_fname = f'{csv_path.name}.svm.joblib'
    clf_path = clf_dst / clf_fname
    joblib.dump(clf, clf_path)
    return clf_path
