"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Map = exports.isPositiveInteger = void 0;
const chain_1 = require("../chain");
const state_graph_1 = require("../state-graph");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Returns true if the value passed is a positive integer
 * @param value the value ti validate
 */
exports.isPositiveInteger = (value) => {
    const isFloat = Math.floor(value) !== value;
    const isNotPositiveInteger = value < 0 || value > Number.MAX_SAFE_INTEGER;
    return !isFloat && !isNotPositiveInteger;
};
/**
 * Define a Map state in the state machine
 *
 * A `Map` state can be used to run a set of steps for each element of an input array.
 * A Map state will execute the same steps for multiple entries of an array in the state input.
 *
 * While the Parallel state executes multiple branches of steps using the same input, a Map state
 * will execute the same steps for multiple entries of an array in the state input.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-map-state.html
 */
class Map extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [this];
        this.maxConcurrency = props.maxConcurrency;
        this.itemsPath = props.itemsPath;
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Define iterator state machine in Map
     */
    iterator(iterator) {
        const name = `Map ${this.stateId} Iterator`;
        super.addIterator(new state_graph_1.StateGraph(iterator.startState, name));
        return this;
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.MAP,
            Comment: this.comment,
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderNextEnd(),
            ...this.renderInputOutput(),
            ...this.renderRetryCatch(),
            ...this.renderIterator(),
            ...this.renderItemsPath(),
            MaxConcurrency: this.maxConcurrency,
        };
    }
    /**
     * Validate this state
     */
    validate() {
        const errors = [];
        if (this.iteration === undefined) {
            errors.push('Map state must have a non-empty iterator');
        }
        if (this.maxConcurrency !== undefined && !exports.isPositiveInteger(this.maxConcurrency)) {
            errors.push('maxConcurrency has to be a positive integer');
        }
        return errors;
    }
    renderItemsPath() {
        return {
            ItemsPath: state_1.renderJsonPath(this.itemsPath),
        };
    }
}
exports.Map = Map;
//# sourceMappingURL=data:application/json;base64,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