from .core.managed_processes import ManagedProcesses
from .core.concurrent_futures_process_pool import ConcurrentFuturesProcessPool

import psutil
from .utils import isnotebook

if isnotebook():
    # https://docs.python.org/3/library/multiprocessing.html#contexts-and-start-methods
    # https://stackoverflow.com/questions/64095876/multiprocessing-fork-vs-spawn
    print('ezFutures used on Jupyter Notebook: defaulting to fork subprocess start method')
    import multiprocessing
    multiprocessing.set_start_method('fork')

class ezFutures():
    
    def __init__(self, n_procs=None, verbose=False, timeout=60*60, n_retries=3, parallelism_mode='managed.processes'):

        if n_procs is None:
            n_procs = psutil.cpu_count(logical=False)
        
        self.parallelism_mode = parallelism_mode

        if self.parallelism_mode=='managed.processes':
            self.core = ManagedProcesses(n_procs=n_procs, verbose=verbose, timeout=timeout, n_retries=n_retries)
        elif self.parallelism_mode=='concurrent.futures.process.pool':
            self.core = ConcurrentFuturesProcessPool(n_procs=n_procs)
    
    def submit(self, func, *args, **kwargs):
        
        self.core.submit(func, *args, **kwargs)

    def globalize(self, local_var, global_name):

        if self.parallelism_mode=='managed.processes':
            return self.core.globalize(local_var, global_name)
        else:    
            raise Exception('[globalize] method only supported for parallelism_mode: managed.processes')
        
    def results(self, show_progress=True):
        
        return self.core.results(show_progress=show_progress)

    def errors(self):

        if self.parallelism_mode=='managed.processes':
            return self.core.shared_error_dict.items()
        else:    
            raise Exception('[errors] method only supported for parallelism_mode: managed.processes')

