"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToMediaStore = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const constructs_1 = require("constructs");
const defaults = require("@aws-solutions-constructs/core");
const aws_cdk_lib_1 = require("aws-cdk-lib");
class CloudFrontToMediaStore extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of CloudFrontToMediaStore class.
     * @param {Construct} scope - represents the scope for all the resources.
     * @param {string} id - this is a scope-unique id.
     * @param {CloudFrontToMediaStoreProps} props - user provided props for the construct
     * @since 1.76.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // All our tests are based upon this behavior being on, so we're setting
        // context here rather than assuming the client will set it
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        defaults.CheckMediaStoreProps(props);
        defaults.CheckCloudFrontProps(props);
        let cloudFrontDistributionProps = props.cloudFrontDistributionProps;
        if (props.existingMediaStoreContainerObj) {
            this.mediaStoreContainer = props.existingMediaStoreContainerObj;
        }
        else {
            let mediaStoreProps;
            if (props.mediaStoreContainerProps) {
                mediaStoreProps = props.mediaStoreContainerProps;
            }
            else {
                this.cloudFrontOriginAccessIdentity = defaults.CloudFrontOriginAccessIdentity(this);
                mediaStoreProps = {
                    containerName: aws_cdk_lib_1.Aws.STACK_NAME,
                    policy: JSON.stringify({
                        Version: '2012-10-17',
                        Statement: [{
                                Sid: 'MediaStoreDefaultPolicy',
                                Effect: 'Allow',
                                Principal: '*',
                                Action: [
                                    'mediastore:GetObject',
                                    'mediastore:DescribeObject'
                                ],
                                Resource: `arn:${aws_cdk_lib_1.Aws.PARTITION}:mediastore:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:container/${aws_cdk_lib_1.Aws.STACK_NAME}/*`,
                                Condition: {
                                    StringEquals: {
                                        'aws:UserAgent': this.cloudFrontOriginAccessIdentity.originAccessIdentityId
                                    },
                                    Bool: {
                                        'aws:SecureTransport': 'true'
                                    }
                                }
                            }]
                    })
                };
                const userAgentHeader = {
                    'User-Agent': this.cloudFrontOriginAccessIdentity.originAccessIdentityId
                };
                if (cloudFrontDistributionProps) {
                    cloudFrontDistributionProps.customHeaders = userAgentHeader;
                }
                else {
                    cloudFrontDistributionProps = {
                        customHeaders: userAgentHeader
                    };
                }
            }
            this.mediaStoreContainer = defaults.MediaStoreContainer(this, mediaStoreProps);
        }
        const DistributionResponse = defaults.CloudFrontDistributionForMediaStore(this, this.mediaStoreContainer, cloudFrontDistributionProps, props.insertHttpSecurityHeaders, props.cloudFrontLoggingBucketProps, props.responseHeadersPolicyProps);
        this.cloudFrontWebDistribution = DistributionResponse.distribution;
        this.cloudFrontLoggingBucket = DistributionResponse.loggingBucket;
        this.cloudFrontOriginRequestPolicy = DistributionResponse.requestPolicy;
        this.cloudFrontFunction = DistributionResponse.cloudfrontFunction;
    }
}
exports.CloudFrontToMediaStore = CloudFrontToMediaStore;
_a = JSII_RTTI_SYMBOL_1;
CloudFrontToMediaStore[_a] = { fqn: "@aws-solutions-constructs/aws-cloudfront-mediastore.CloudFrontToMediaStore", version: "2.48.0" };
//# sourceMappingURL=data:application/json;base64,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