import abc
from typing import List, Sequence, FrozenSet

from sneks.core.cell import Cell
from sneks.core.direction import Direction


class Snek(abc.ABC):
    body: List[Cell] = []
    occupied: FrozenSet[Cell] = []
    food: FrozenSet[Cell] = []

    def get_next_direction(self) -> Direction:
        """
        :return: the next direction for your snake to move
        """
        raise NotImplementedError()

    def get_head(self) -> Cell:
        """
        :return: the cell representing the head of the snake
        """
        return self.body[0]

    def get_body(self) -> List[Cell]:
        """
        :return: the list of cells making up the snake, including the head
        """
        return self.body

    def get_food(self) -> FrozenSet[Cell]:
        return self.food

    def get_closest_food(self) -> Cell:
        """
        :return: the Cell representing the location of the nearest food
        """
        return min(self.food, key=lambda food: food.get_distance(self.get_head()))

    def look(self, direction: Direction) -> int:
        """
        :param direction: the direction to look
        :return: the distance until the closest obstacle in the specified direction
        """

        current = self.get_head().get_neighbor(direction)
        current_distance = 1

        while current not in self.occupied and current.is_valid():
            current = current.get_neighbor(direction)
            current_distance += 1

        return current_distance - 1

    def get_direction_to_destination(
        self,
        destination: Cell,
        directions: Sequence[Direction] = (
            Direction.UP,
            Direction.DOWN,
            Direction.LEFT,
            Direction.RIGHT,
        ),
    ) -> Direction:
        """
        Get the next direction to travel in order to reach the destination.

        :param destination: the Cell to travel towards
        :param directions: the directions to evaluate
        :return: the direction to travel that will close the most distance
        """

        return min(
            directions,
            key=lambda direction: self.get_head()
            .get_neighbor(direction)
            .get_distance(destination),
        )
