"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const aws_credentials_1 = require("./aws-credentials");
const posix_utils_1 = require("./private/posix-utils");
const stage_1 = require("./stage");
const github_action_step_1 = require("./steps/github-action-step");
const wave_1 = require("./wave");
const github = require("./workflows-model");
const yaml_file_1 = require("./yaml-file");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        // in order to keep track of if this pipeline has been built so we can
        // catch later calls to addWave() or addStage()
        this.builtGH = false;
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = props.preSynthed ?? false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = props.preBuildSteps ?? [];
        this.postBuildSteps = props.postBuildSteps ?? [];
        this.jobSettings = props.jobSettings;
        this.dockerAssetJobSettings = props.dockerAssetJobSettings;
        this.awsCredentials = this.getAwsCredentials(props);
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.workflowPath = props.workflowPath ? (0, posix_utils_1.posixPath)(props.workflowPath) : '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowFile = new yaml_file_1.YamlFile(this.workflowPath);
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        if (props.concurrency) {
            this.concurrency = {
                group: props.concurrency.group,
                cancelInProgress: props.concurrency.cancelInProgress ?? false,
            };
        }
        this.runner = props.runner ?? github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = props.publishAssetsAuthRegion ?? 'us-west-2';
    }
    /**
     * Parse AWS credential configuration from deprecated properties For backwards compatibility.
     */
    getAwsCredentials(props) {
        if (props.gitHubActionRoleArn) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove githubActionRoleArn)');
            }
            return aws_credentials_1.AwsCredentials.fromOpenIdConnect({
                gitHubActionRoleArn: props.gitHubActionRoleArn,
            });
        }
        if (props.awsCredentials) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove awsCredentials)');
            }
            return aws_credentials_1.AwsCredentials.fromGitHubSecrets({
                accessKeyId: 'AWS_ACCESS_KEY_ID',
                secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
                ...props.awsCredentials,
            });
        }
        return props.awsCreds ?? aws_credentials_1.AwsCredentials.fromGitHubSecrets();
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', options?.jobSettings);
        return stageDeployment;
    }
    /**
     * Add a Wave to the pipeline, for deploying multiple Stages in parallel
     *
     * Use the return object of this method to deploy multiple stages in parallel.
     *
     * Example:
     *
     * ```ts
     * declare const pipeline: GitHubWorkflow; // assign pipeline a value
     *
     * const wave = pipeline.addWave('MyWave');
     * wave.addStage(new MyStage(this, 'Stage1'));
     * wave.addStage(new MyStage(this, 'Stage2'));
     * ```
     */
    addWave(id, options) {
        return this.addGitHubWave(id, options);
    }
    addGitHubWave(id, options) {
        if (this.builtGH) {
            throw new Error("addWave: can't add Waves anymore after buildPipeline() has been called");
        }
        const wave = new wave_1.GitHubWave(id, this, options);
        this.waves.push(wave);
        return wave;
    }
    /**
     * Support adding stages with GitHub options to waves - should ONLY be called internally.
     *
     * Use `pipeline.addWave()` and it'll call this when `wave.addStage()` is called.
     *
     * `pipeline.addStage()` will also call this, since it calls `pipeline.addWave().addStage()`.
     *
     *  @internal
     */
    _addStageFromWave(stage, stageDeployment, options) {
        if (!(stage instanceof stage_1.GitHubStage) && options === undefined) {
            return;
        }
        const ghStage = stage instanceof stage_1.GitHubStage ? stage : undefined;
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', ghStage?.props?.gitHubEnvironment ?? options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', ghStage?.props?.stackCapabilities ?? options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', ghStage?.props?.jobSettings ?? options?.jobSettings);
    }
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        this.builtGH = true;
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false, // we create and execute the changeset in a single job
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!(0, helpers_internal_1.isGraph)(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            ...(this.concurrency ? {
                concurrency: {
                    'group': this.concurrency.group,
                    'cancel-in-progress': this.concurrency.cancelInProgress,
                },
            } : {}),
            jobs: jobmap,
        };
        // write as a yaml file
        this.workflowFile.update(workflow);
        // create directory if it does not exist
        (0, fs_1.mkdirSync)(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const contextValue = this.node.tryGetContext('cdk-pipelines-github:diffProtection');
        const diffProtection = contextValue === 'false' ? false : contextValue ?? true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!(0, fs_1.existsSync)(this.workflowPath) || this.workflowFile.toYaml() !== (0, fs_1.readFileSync)(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        this.workflowFile.writeFile();
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Workflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else if (node.data.step instanceof github_action_step_1.GitHubActionStep) {
                    return this.jobForGitHubActionStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
            default:
                // The 'as any' is temporary, until the change upstream rolls out
                throw new Error(`GitHubWorfklow does not support graph nodes of type '${node.data?.type}'. You are probably using a feature this CDK Pipelines implementation does not support.`);
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const { assetId, assetManifestPath } = assets[0];
        const preBuildSteps = [];
        let permissions = {
            contents: github.JobPermission.READ,
            idToken: this.awsCredentials.jobPermission(),
        };
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset')) {
            if (this.dockerCredentials.length > 0) {
                for (const creds of this.dockerCredentials) {
                    dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
                }
            }
            if (this.dockerAssetJobSettings?.setupSteps) {
                preBuildSteps.push(...this.dockerAssetJobSettings.setupSteps);
            }
            permissions = {
                ...permissions,
                ...this.dockerAssetJobSettings?.permissions,
            };
        }
        // create one file and make one step
        const relativeToAssembly = (p) => (0, posix_utils_1.posixPath)(path.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p)));
        const fileContents = ['set -ex'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '${ASSET_HASH_NAME}=${assetId}' >> $GITHUB_OUTPUT`);
        const publishStepFile = (0, posix_utils_1.posixPath)(path.join(path.dirname(relativeToAssembly(assetManifestPath)), `publish-${jobId}-step.sh`));
        (0, fs_1.mkdirSync)(path.dirname(publishStepFile), { recursive: true });
        (0, fs_1.writeFileSync)(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${(0, posix_utils_1.posixPath)(path.relative(cdkoutDir, publishStepFile))}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                ...this.renderJobSettingParameters(),
                needs: this.renderDependencies(node),
                permissions,
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.publishAssetsAuthRegion),
                    ...dockerLoginSteps,
                    ...preBuildSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            const partition = process.env.CDK_AWS_PARTITION ?? 'aws';
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: partition,
            });
        };
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split('/').pop() ?? '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            const updated_template = template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
            return process.env.CDK_AWS_PARTITION == 'aws-cn'
                ? updated_template.replace('.amazonaws.com', '.amazonaws.com.cn')
                : updated_template;
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = this.stackProperties[stack.stackArtifactId]?.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                ...this.renderJobSettingParameters(),
                ...this.stackProperties[stack.stackArtifactId]?.settings,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.awsCredentials.jobPermission(),
                },
                ...this.renderGitHubEnvironment(this.stackProperties[stack.stackArtifactId]?.environment),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(region, assumeRoleArn),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.awsCredentials.jobPermission(),
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        for (const dep of node.allDeps) {
            if (dep.data?.type === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v4',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v4',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    jobForGitHubActionStep(node, step) {
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.WRITE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                steps: step.jobSteps,
            },
        };
    }
    stepsToConfigureAws(region, assumeRoleArn) {
        return this.awsCredentials.credentialSteps(region, assumeRoleArn);
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: dockerCredential.username,
                password: dockerCredential.password,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: dockerCredential.username,
                password: dockerCredential.password,
            };
        }
        if (dockerCredential.name === 'ghcr') {
            this.dockerAssetJobSettings = {
                ...this.dockerAssetJobSettings,
                permissions: {
                    ...this.dockerAssetJobSettings?.permissions,
                    packages: github.JobPermission.READ,
                },
            };
        }
        return [
            {
                uses: 'docker/login-action@v2',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v4',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [
            {
                name: 'Checkout',
                uses: 'actions/checkout@v4',
            },
        ];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v4',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
    renderJobSettingParameters() {
        return this.jobSettings;
    }
    renderGitHubEnvironment(environment) {
        if (!environment) {
            return {};
        }
        if (environment.url === undefined) {
            return { environment: environment.name };
        }
        return { environment };
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.4.116" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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