"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubActionRole = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * GitHub OIDC thumbprints updated 2023-07-27
 *
 * https://github.blog/changelog/2023-06-27-github-actions-update-on-oidc-integration-with-aws/
 */
const GITHUB_OIDC_THUMBPRINTS = [
    '6938fd4d98bab03faadb97b34396831e3780aea1',
    '1c58a3a8518e8759bf075b76b750d4f2df264fcd',
];
/**
 * Creates or references a GitHub OIDC provider and accompanying role that trusts the provider.
 * This role can be used to authenticate against AWS instead of using long-lived AWS user credentials
 * stored in GitHub secrets.
 *
 * You can do this manually in the console, or create a separate stack that uses this construct.
 * You must `cdk deploy` once (with your normal AWS credentials) to have this role created for you.
 *
 * You can then make note of the role arn in the stack output and send it into the Github Workflow app via
 * the `gitHubActionRoleArn` property. The role arn will be `arn:<partition>:iam::<accountId>:role/GithubActionRole`.
 *
 * @see https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
 */
class GitHubActionRole extends constructs_1.Construct {
    /**
     * Reference an existing GitHub Actions provider.
     * You do not need to pass in an arn because the arn for such
     * a provider is always the same.
     */
    static existingGitHubActionsProvider(scope) {
        return iam.OpenIdConnectProvider.fromOpenIdConnectProviderArn(scope, 'GitHubActionProvider', `arn:${aws_cdk_lib_1.Aws.PARTITION}:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:oidc-provider/token.actions.githubusercontent.com`);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const rawEndpoint = 'token.actions.githubusercontent.com';
        const providerUrl = `https://${rawEndpoint}`;
        // uses the given provider or creates a new one.
        const provider = props.provider ?? new iam.OpenIdConnectProvider(this, 'github-provider', {
            url: providerUrl,
            clientIds: ['sts.amazonaws.com'],
            thumbprints: props.thumbprints ?? GITHUB_OIDC_THUMBPRINTS,
        });
        // create a role that references the provider as a trusted entity
        const principal = new iam.FederatedPrincipal(provider.openIdConnectProviderArn, {
            StringLike: {
                [`${rawEndpoint}:sub`]: formatRepos(props.repos ?? []).concat(props.subjectClaims ?? []),
            },
        }, 'sts:AssumeRoleWithWebIdentity');
        // permit this role from assuming all of the CDK bootstrap roles
        const oidcPolicyStatement = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: ['*'],
            conditions: {
                'ForAnyValue:StringEquals': {
                    'iam:ResourceTag/aws-cdk:bootstrap-role': [
                        'deploy',
                        'lookup',
                        'file-publishing',
                        'image-publishing',
                    ],
                },
            },
        });
        // permit this role from accessing ecr repositories for docker assets
        const ecrPolicyStatement = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['ecr:GetAuthorizationToken'],
            resources: ['*'],
        });
        this.role = new iam.Role(this, 'github-action-role', {
            roleName: props.roleName ?? 'GitHubActionRole',
            assumedBy: principal,
            inlinePolicies: {
                AssumeBootstrapRoles: new iam.PolicyDocument({
                    statements: [oidcPolicyStatement, ecrPolicyStatement],
                }),
            },
        });
        // show the role arn in the stack output
        new aws_cdk_lib_1.CfnOutput(this, 'roleArn', {
            value: this.role.roleArn,
        });
    }
}
exports.GitHubActionRole = GitHubActionRole;
_a = JSII_RTTI_SYMBOL_1;
GitHubActionRole[_a] = { fqn: "cdk-pipelines-github.GitHubActionRole", version: "0.4.116" };
function formatRepos(repos) {
    const formattedRepos = [];
    for (const repo of repos) {
        formattedRepos.push(`repo:${repo}:*`);
    }
    return formattedRepos;
}
//# sourceMappingURL=data:application/json;base64,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