"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubOIDCStack = exports.TokenActions = exports.ProviderUrl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const utils_1 = require("../../common/utils");
const base_1 = require("../base");
var ProviderUrl;
(function (ProviderUrl) {
    ProviderUrl["GITHUB"] = "https://token.actions.githubusercontent.com";
})(ProviderUrl = exports.ProviderUrl || (exports.ProviderUrl = {}));
var TokenActions;
(function (TokenActions) {
    TokenActions[TokenActions["ALL"] = 0] = "ALL";
    TokenActions[TokenActions["ALL_BRANCH"] = 1] = "ALL_BRANCH";
    TokenActions[TokenActions["ALL_TAGS"] = 2] = "ALL_TAGS";
    TokenActions[TokenActions["CUSTOM"] = 3] = "CUSTOM";
})(TokenActions = exports.TokenActions || (exports.TokenActions = {}));
class GithubOIDCStack extends base_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.githubUser = props.githubUser;
        this.githubRepository = props.githubRepository;
        this.tokenAction = props.tokenAction;
        let token = this.createTokenAction(props.tokenAction, props.githubUser, props.githubRepository, props.tokenActionCustom);
        this.oidcRole = this.createOidcRole(ProviderUrl.GITHUB, token);
        this.cdkBootstrapRole = this.createCdkBootstrapRole();
        this.cdkDeployRoleManagedPolicies = props.cdkDeployRoleManagedPolicies;
        this.cdkDeployRolePolicyStatements = props.cdkDeployRolePolicyStatements;
        this.cdkDeployRole = this.createCdkDeployRole(this.cdkDeployRoleManagedPolicies, this.cdkDeployRolePolicyStatements);
    }
    createTokenAction(tokenAction, githubUser, githubRepository, tokenActionCustom) {
        if (tokenAction === TokenActions.ALL) {
            return `repo:${githubUser}/*`;
        }
        if (tokenAction === TokenActions.ALL_BRANCH) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/heads/*`;
        }
        if (tokenAction === TokenActions.ALL_TAGS) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/tags/*`;
        }
        if (tokenAction === TokenActions.CUSTOM) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/${tokenActionCustom}`;
        }
        throw new Error('tokenAction not found');
    }
    createCdkDeployRole(managed_policies, policy_statements) {
        let basePolicy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'iam:PassRole',
                        'iam:CreateRole',
                        'iam:DeleteRole',
                        'iam:UpdateRole',
                        'iam:DetachRolePolicy',
                        'iam:AttachRolePolicy',
                        'iam:DeleteRolePolicy',
                        'iam:PutRolePolicy',
                        'ssm:GetParameters',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-deploy-role', {
            roleName: 'cdk-oidc-deploy-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKDeployBasePolicy: basePolicy,
            },
        });
        if (policy_statements) {
            for (let index = 0; index < policy_statements.length; index++) {
                role.addToPolicy(policy_statements[index]);
            }
        }
        if (managed_policies) {
            for (let index = 0; index < managed_policies.length; index++) {
                role.addManagedPolicy(managed_policies[index]);
            }
        }
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-deploy-role-output', {
            value: role.roleArn,
            description: 'Role for cdk deploy role arn',
            exportName: 'cdk-oidc-deploy-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createCdkBootstrapRole() {
        let policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                        'iam:*Role*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:cloudformation:${this.region}:${this.account}:stack/CDKToolkit/*`],
                    actions: [
                        'cloudformation:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        's3:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ecr:${this.region}:${this.account}:repository/cdk-*`],
                    actions: [
                        'ecr:SetRepositoryPolicy',
                        'ecr:GetLifecyclePolicy',
                        'ecr:PutImageScanningConfiguration',
                        'ecr:DescribeRepositories',
                        'ecr:CreateRepository',
                        'ecr:DeleteRepository',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ssm:${this.region}:${this.account}:parameter/cdk-bootstrap/*`],
                    actions: [
                        'ssm:GetParameter*',
                        'ssm:PutParameter*',
                        'ssm:DeleteParameter*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-bootstrap-role', {
            roleName: 'cdk-oidc-bootstrap-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKBootstrapPolicy: policy,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-bootstrap-role-output', {
            value: role.roleArn,
            description: 'Role for cdk bootstrap role arn',
            exportName: 'cdk-oidc-bootstrap-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createOidcRole(providerUrl, token) {
        let oidcProvider = new iam.OpenIdConnectProvider(this, 'oidc-provider', {
            url: providerUrl,
            clientIds: ['sts.amazonaws.com'],
        });
        let role = new iam.Role(this, 'oidc-role', {
            roleName: 'github-oidc-workflow-role',
            assumedBy: new iam.WebIdentityPrincipal(oidcProvider.openIdConnectProviderArn, {
                StringLike: {
                    'token.actions.githubusercontent.com:sub': token,
                },
            }),
        });
        role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'iam:PassRole',
                'ssm:GetParameter*',
                'cloudformation:*',
                's3:*',
                'ecr:*',
            ],
            effect: iam.Effect.ALLOW,
        }));
        role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
            actions: ['sts:AssumeRole'],
            effect: iam.Effect.ALLOW,
        }));
        new aws_cdk_lib_1.CfnOutput(this, 'github-oidc-workflow-role-output', {
            value: role.roleArn,
            description: 'Role for OIDC Github Workflow role arn',
            exportName: 'github-oidc-workflow-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
}
exports.GithubOIDCStack = GithubOIDCStack;
_a = JSII_RTTI_SYMBOL_1;
GithubOIDCStack[_a] = { fqn: "neulabs-cdk-constructs.stacks.oidc.github.GithubOIDCStack", version: "0.1.4" };
//# sourceMappingURL=data:application/json;base64,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