import os
from typing import Optional

import typer
from git import Repo
from github import Github

from repo_tools.common import utils
from repo_tools.common.help_texts import HelpText

github_app = typer.Typer()

pr_app = typer.Typer()
github_app.add_typer(pr_app, name="pr")


@pr_app.command(help="Create a PR on github")
def create(
    target_branch: str = typer.Option(...),
    message: str = typer.Option(...),
    reviewers: str = typer.Option(
        ..., help="Comma separated list of github usernames to add for review"
    ),
    glob: Optional[str] = typer.Option(
        "",
        "--glob",
        "-g",
        help=HelpText.GLOB,
    ),
    feature: Optional[str] = typer.Option(
        "",
        "--feature",
        "-f",
        help=HelpText.FEATURE,
    ),
    include: Optional[str] = typer.Option("", "--include", "-i", help=HelpText.INCLUDE),
    exclude: Optional[str] = typer.Option("", "--exclude", "-e", help=HelpText.EXCLUDE),
    dry_run: bool = typer.Option(False, help="execute a dry run, does not create PRs"),
    draft: bool = typer.Option(False, help="Create a PR in draft state"),
):
    github_api_client = Github(utils.get_github_oauth_token())
    github_org = utils.get_github_org()
    projects = utils.get_projects(feature, include, exclude, glob_pattern=glob)
    reviewers = reviewers.split(",")

    for project in projects:
        try:
            repo = Repo(project.abs_path)
            current_branch = str(repo.active_branch)

            if dry_run:
                typer.echo(
                    typer.style(
                        f"DRY RUN --> {project.name}:",
                        fg=typer.colors.YELLOW,
                        bold=True,
                    )
                )
                typer.echo(
                    f"   Running without --dry-run flag will create a PR for {project.name}\n"
                    f"   Merge '{current_branch}' into '{target_branch}'\n"
                    f"   Title: {message}"
                )
                continue

            repository = github_api_client.get_repo(f"{github_org}/{project.name}")
            pull_request = repository.create_pull(
                title=message,
                body=f"{message} \n\n - autogenerated PR",
                head=current_branch,
                base=target_branch,
                draft=draft,
            )
            pull_request.create_review_request(reviewers=reviewers)
            typer.echo(
                typer.style(f"{project.name}:", fg=typer.colors.GREEN, bold=True)
            )
            typer.echo(
                f"   PR created: {pull_request.html_url}, review assigned to {', '.join(reviewers)}"
            )
        except Exception as exc:
            typer.echo(typer.style(f"{project.name}:", fg=typer.colors.RED, bold=True))
            typer.echo(f"   GitHub PR creation failed: {exc}")
            continue


@github_app.command(
    help="Clone all repositories from the configured github organisation"
)
def clone_all(
    path: Optional[str] = typer.Argument(
        ".", help="target path to clone all repositories to."
    )
):
    cwd = os.path.abspath(os.path.join(os.getcwd(), path))
    os.makedirs(cwd, exist_ok=True)
    github_api_client = Github(utils.get_github_oauth_token())
    github_org_name = utils.get_github_org()

    org = github_api_client.get_organization(github_org_name)

    repositories = org.get_repos()
    typer.echo(f"Found {repositories.totalCount} repositories in {github_org_name}.")
    confirm = typer.confirm(
        f"Are you sure you want to clone all repositories to {cwd} ?"
    )

    if not confirm:
        typer.echo("Not continuing...")
        raise typer.Abort()

    for repo in repositories:
        typer.echo(typer.style(f"Cloning {repo.full_name}...", bold=True))
        clone_command = f"git clone {repo.ssh_url}"
        out, status_code = utils.execute_command(clone_command, cwd=cwd)
        utils.display_command_output(repo.full_name, out, status_code)
