# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sqlakeyset', 'sqlakeyset.serial']

package_data = \
{'': ['*']}

install_requires = \
['packaging>=20.0', 'python-dateutil', 'sqlalchemy>=1.3.11']

setup_kwargs = {
    'name': 'sqlakeyset',
    'version': '1.0.1655855963',
    'description': 'offset-free paging for sqlalchemy',
    'long_description': "sqlakeyset: offset-free paging for sqlalchemy\n=============================================\n\n.. image:: https://img.shields.io/circleci/build/gh/djrobstep/sqlakeyset?label=tests\n    :alt: Tests\n    :target: https://circleci.com/gh/djrobstep/sqlakeyset\n    \n.. image:: https://img.shields.io/pypi/v/sqlakeyset\n    :alt: PyPI\n    :target: https://pypi.org/project/sqlakeyset/\n    \n.. image:: https://img.shields.io/conda/vn/conda-forge/sqlakeyset.svg\n    :alt: conda-forge\n    :target: https://anaconda.org/conda-forge/sqlakeyset\n\nsqlakeyset implements keyset-based paging for SQLAlchemy (both ORM and core).\n\nThis library is tested with PostgreSQL, MariaDB/MySQL and SQLite. It should work with other SQLAlchemy-supported databases, too; but you should verify the results are correct.\n\n**Notice:** In accordance with Python 2's end-of-life, we've stopped supporting Python versions earlier than 3.4. If you really need it, the latest version to support Python 2 is 0.1.1559103842, but you'll miss out on all the latest features and bugfixes from the latest version. You should be upgrading anyway!\n\nBackground\n----------\n\nA lot of people use SQL's ``OFFSET`` syntax to implement paging of query results. The trouble with that is, the more pages you get through, the slower your query gets. Also, if the results you're paging through change frequently, it's possible to skip over or repeat results between pages. Keyset paging avoids these problems: Selecting even the millionth page is as fast as selecting the first.\n\n\nGetting Started\n---------------\n\nHere's how it works with a typical ORM query:\n\n.. code-block:: python\n\n    from sqlakeyset import get_page\n    from sqlbag import S\n\n    from models import Book\n\n    with S('postgresql:///books') as s:  # create a session\n        q = s.query(Book).order_by(Book.author, Book.title, Book.id)  #\n\n        # gets the first page\n        page1 = get_page(q, per_page=20)\n\n        # gets the key for the next page\n        next_page = page1.paging.next\n\n        # gets the second page\n        page2 = get_page(q, per_page=20, page=next_page)\n\n        # returning to the first page, getting the key\n        previous_page = page2.paging.previous\n\n        # the first page again, backwards from the previous page\n        page1 = get_page(q, per_page=20, page=previous_page)\n\n        # what if new items were added at the start?\n        if page1.paging.has_previous:\n\n            # go back even further\n            previous_page = page1.paging.previous\n            page1 = get_page(q, per_page=20, page=previous_page)\n\n\nUnder the Hood\n--------------\n\nsqlakeyset does the following to your query in order to get the paged contents:\n\n- adds a where clause, to get only rows after the specified row key.\n- if getting the previous page, reverses the ``order by`` direction in order the get the rows *before* the specified bookmark.\n- adds a limit clause, to fetch only enough items to fill the page, plus one additional (this additional row is used only to test for the existence of further pages after the current one, and is discarded from the results).\n- returns the page contents as an ordinary list that has an attached ``.paging`` attribute with the paging information for this and related pages.\n\n\nPage objects\n------------\n\nPaged items/rows are returned in a ``Page`` object, which is a vanilla python list extended by an attached ``Paging`` object containing paging information.\n\nProperties such as `next` and `previous` return a pair containing the ordering key for the row, and a boolean to specify if the direction is forwards or backwards. We refer to such a pair ``(keyset, backwards)`` as a *marker*.\n\nIn our above example, the marker specifying the second page might look like:\n\n.. code-block:: python\n\n    ('Joseph Heller', 'Catch 22', 123), False\n\nThe `False` means the query will fetch the page *after* the row containing Catch 22. This tuple contains two elements, title and id, to match the order by clause of the query.\n\nThe page before this row would be specified as:\n\n.. code-block:: python\n\n    ('Joseph Heller', 'Catch 22', 123), True\n\nThe first and last pages are fetched with `None` instead of a tuple, so for the first page (this is also the default if the page parameter is not specified):\n\n.. code-block:: python\n\n    None, False\n\nAnd the last page:\n\n.. code-block:: python\n\n    None, True\n\nKeyset Serialization\n--------------------\n\nYou will probably want to turn these markers into strings for passing around. ``sqlakeyset`` includes code to do this, and calls the resulting strings *bookmarks*. To get a serialized bookmark, just add ``bookmark_`` to the name of the property that holds the keyset you want.\n\nMost commonly you'll want ``next`` and ``previous``, so:\n\n.. code-block:: python\n\n    >>> page.paging.bookmark_previous\n    <i:1~i:2015~s:Bad Blood~i:34\n    >>> page.paging.bookmark_next\n    >i:1~i:2014~s:Shake It Off~i:31\n\n``sqlakeyset`` uses the python csv row serializer to serialize the bookmark values (using ``~`` instead of a ``,`` as the separator). Direction is indicated by ``>`` (forwards/next), or ``<`` (backwards/previous) at the start of the string.\n\nLimitations\n-----------\n\n- **Golden Rule:** Always ensure your keysets are unique per row. If you violate this condition you risk skipped rows and other nasty problems. The simplest way to do this is to always include your primary key column(s) at the end of your ordering columns.\n\n- Any rows containing null values in their keysets **will be omitted from the results**, so your ordering columns should be ``NOT NULL``. (This is a consequence of the fact that comparisons against ``NULL`` are always false in SQL.) This may change in the future if we work out an alternative implementation; but for now we recommend using ``coalesce`` as a workaround if you need to sort by nullable columns:\n\n.. code-block:: python\n\n    from sqlakeyset import get_page\n    from sqlalchemy import func\n    from sqlbag import S\n    from models import Book\n    with S('postgresql:///books') as s:\n        # If Book.cost can be NULL:\n        q = s.query(Book).order_by(func.coalesce(Book.cost, 0), Book.id)\n        # Assuming cost is non-negative, page1 will start with books where cost is null:\n        page1 = get_page(q, per_page=20)\n\n- If you're using the in-built keyset serialization, this only handles basic data/column types so far (strings, ints, floats, datetimes, dates, booleans, and a few others). The serialization can be extended to serialize more advanced types as necessary (documentation on this is forthcoming).\n\n\nDocumentation\n-------------\n\nOther than this README, there is some basic sphinx documentation, which you can build yourself with e.g. ``make -C doc html``. Hopefully this will be available more conveniently soon - watch this space.\n\n\nInstallation\n------------\n\nAssuming you have `pip <https://pip.pypa.io>`_ installed, all you need to do is install as follows:\n\n.. code-block:: shell\n\n    $ pip install sqlakeyset\n\nThis will install sqlakeyset and also sqlalchemy if not already installed. Obviously you'll need the necessary database driver for your chosen database to be installed also.\n",
    'author': 'Robert Lechte',
    'author_email': 'robertlechte@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/djrobstep/sqlakeyset',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.4',
}


setup(**setup_kwargs)
