"""Option Dataclasses"""

import re
from dataclasses import dataclass
from typing import Optional, Union


@dataclass
class _DefaultField:
    """Field info in options classes"""

    name: str
    type: type
    default: object


@dataclass
class _Options:
    def __init__(self, **kwargs):
        default = self.defaults()
        for option in kwargs:
            if option in default:
                setattr(self, option, kwargs[option])

    @staticmethod
    def convert_name(value: str) -> str:
        """Add flag marker and replace underscores with dashes in name"""
        return "--" + value.replace("_", "-")

    # pylint: disable=no-member
    @classmethod
    def defaults(cls) -> dict[_DefaultField]:
        """Get list of fields for an Options dataclass"""
        values = {}
        for field in cls.__dataclass_fields__.values():
            values[field.name] = _DefaultField(field.name, field.type, field.default)
        return values

    def parse(self) -> list[Optional[Union[str, int]]]:
        """Turn options into list for argv

        :return: options for the command line
        :rtype: list[Optional[Union[str, int]]]
        """
        args = []
        # pylint: disable=no-member
        for key, value in self.__dataclass_fields__.items():
            attr = getattr(self, key)
            if attr is not None and value.default != attr:
                flag = self.convert_name(key)
                if value.type is bool:
                    if attr is not value.default:
                        args.append(flag)
                elif value.type is list:
                    for val in attr:
                        args.extend([flag, val])
                else:
                    args.extend([flag, attr])
        return args


# pylint: disable=too-many-instance-attributes
@dataclass
class CommonOptions(_Options):
    """Common Options for all Borg commands

    :param critical: work on log level CRITICAL
    :type critical: bool
    :param error: work on log level ERROR
    :type error: bool
    :param warning: work on log level WARNING
    :type warning: bool
    :param info: work on log level INFO
    :type info: bool
    :param verbose: work on log level INFO
    :type verbose: bool
    :param debug: work on log level DEBUG
    :type debug: bool
    :param debug_topic: enable TOPIC debugging (can be specified multiple times). The logger path
        is borg.debug.<TOPIC> if TOPIC is not fully qualified.
    :type debug_topic: list[str]
    :param progress: show progress information
    :type progress: bool
    :param log_json: output one JSON object per log line instead of formatted text.
    :type log_json: bool
    :param lock_wait: wait at most SECONDS for acquiring a repository/cache lock (default: 1).
    :type lock_wait: int
    :param bypass_lock: bypass locking mechanism
    :type bypass_lock: bool
    :param show_version: show/log the borg version
    :type show_version: bool
    :param show_rc: show/log the return code (rc)
    :type show_rc: bool
    :param umask: set umask to M (local and remote, default: 0077)
    :type umask: str
    :param remote_path: use PATH as borg executable on the remote (default: “borg”)
    :type remote_path: str
    :param remote_ratelimit: set remote network upload rate limit in kiByte/s (default: 0=unlimited)
    :type remote_ratelimit: int
    :param consider_part_files: treat part files like normal files (e.g. to list/extract them)
    :type consider_part_files: bool
    :param debug_profile: write execution profile in Borg format into FILE. For local use a
        Python-compatible file can be generated by suffixing FILE with “.pyprof”
    :type debug_profile: str
    :param rsh: Use this command to connect to the ‘borg serve’ process (default: ‘ssh’)
    :type rsh: str
    """

    critical: bool = False
    error: bool = False
    warning: bool = False
    info: bool = False
    verbose: bool = False
    debug: bool = False
    debug_topic: list[str] = None
    progress: bool = False
    log_json: bool = False
    lock_wait: int = None
    bypass_lock: bool = False
    show_version: bool = False
    show_rc: bool = False
    umask: str = None
    remote_path: str = None
    remote_ratelimit: int = None
    consider_part_files: bool = False
    debug_profile: str = None
    rsh: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        if isinstance(self.debug_topic, str):
            self.exclude = [self.exclude]
        if self.umask and not re.match(r"^[0-9]{4}", self.umask):
            raise ValueError("umask must be in format 0000 permission code, eg: 0077")


@dataclass
class ExclusionOptions(_Options):
    """Options for excluding various files from backup

    :param exclude: exclude paths matching PATTERN
    :type exclude: list[str]
    :param exclude_from: read exclude patterns from EXCLUDEFILE, one per line
    :type exclude_from: str
    :param pattern: include/exclude paths matching PATTERN (experimental)
    :type pattern: str
    :param patterns_from: read include/exclude patterns from PATTERNFILE, one per
        line (experimental)
    :type patterns_from: str
    """

    exclude: list[str] = None
    exclude_from: str = None
    pattern: str = None
    patterns_from: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        if isinstance(self.exclude, str):
            self.exclude = [self.exclude]


@dataclass
class ExclusionInput(ExclusionOptions):
    """Exclusion Options when inputing data to the archive

    :param exclude_caches: exclude directories that contain a CACHEDIR.TAG file
        (http://www.bford.info/cachedir/spec.html)
    :type exclude_caches: bool
    :param exclude_if_present: exclude directories that are tagged by containing a filesystem
        object with the given NAME
    :type exclude_if_present: list[str]
    :param keep_exclude_tags: if tag objects are specified with --exclude-if-present, don’t omit
        the tag objects themselves from the backup archive
    :type keep_exclude_tags: bool
    :param keep_tag_files: alternate to keep_exclude_tags
    :type keep_tag_files: bool
    :param exclude_nodump: exclude files flagged NODUMP
    :type exclude_nodump: bool
    """

    exclude_caches: bool = False
    exclude_if_present: list[str] = None
    keep_exclude_tags: bool = False
    keep_tag_files: bool = False
    exclude_nodump: bool = False

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        if isinstance(self.exclude_if_present, str):
            self.exclude_if_present = [self.exclude_if_present]


@dataclass
class ExclusionOutput(ExclusionOptions):
    """Exclusion Options when outputing data in the archive

    :param strip_componts: Remove the specified number of leading path elements. Paths with fewer
        elements will be silently skipped
    :type strip_componts: int
    """

    strip_componts: int = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class FilesystemOptions(_Options):
    """Options for how to handle filesystem attributes

    :param one_file_system: stay in the same file system and do not store mount points of other
        file systems. This might behave different from your expectations, see the docs.
    :type one_file_system: bool
    :param numeric_owner: only store numeric user and group identifiers
    :type numeric_owner: bool
    :param noatime: do not store atime into archive
    :type noatime: bool
    :param noctime: do not store ctime into archive
    :type noctime: bool
    :param nobirthtime: do not store birthtime (creation date) into archive
    :type nobirthtime: bool
    :param nobsdflags: do not read and store bsdflags (e.g. NODUMP, IMMUTABLE) into archive
    :type nobsdflags: bool
    :param noacls: do not read and store ACLs into archive
    :type noacls: bool
    :param noxattrs: do not read and store xattrs into archive
    :type noxattrs: bool
    :param ignore_inode: ignore inode data in the file metadata cache used to detect
        unchanged files.
    :type ignore_inode: bool
    :param files_cache: operate files cache in MODE. default: ctime,size,inode
    :type files_cache: str
    :param read_special: open and read block and char device files as well as FIFOs as if they were
        regular files. Also follows symlinks pointing to these kinds of files.
    :type read_special: bool
    """

    one_file_system: bool = False
    numeric_owner: bool = False
    noatime: bool = False
    noctime: bool = False
    nobirthtime: bool = False
    nobsdflags: bool = False
    noacls: bool = False
    noxattrs: bool = False
    ignore_inode: bool = False
    files_cache: str = None
    read_special: bool = False

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchiveOptions(_Options):
    """Options related to the archive"""

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchiveInput(ArchiveOptions):
    """Archive Options when inputing data to the archive

    :param comment: add a comment text to the archive
    :type comment: str
    :param timestamp: manually specify the archive creation date/time
        (UTC, yyyy-mm-ddThh:mm:ss format). Alternatively, give a reference file/directory.
    :type timestamp: str
    :param checkpoint_interval: write checkpoint every SECONDS seconds (Default: 1800)
    :type checkpoint_interval: int
    :param chunker_params: specify the chunker parameters (CHUNK_MIN_EXP, CHUNK_MAX_EXP,
        HASH_MASK_BITS, HASH_WINDOW_SIZE). default: 19,23,21,4095
    :type chunker_params: str
    :param compression: select compression algorithm, see the output of the “borg help compression”
        command for details.
    :type compression: str
    """

    comment: str = None
    timestamp: str = None
    checkpoint_interval: int = None
    chunker_params: str = None
    compression: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchivePattern(ArchiveOptions):
    """Archive Options when outputing data in the archive

    :param prefix: only consider archive names starting with this prefix.
    :type prefix: str
    :param glob_archives: only consider archive names matching the glob.
        sh: rules apply, see “borg help patterns”. --prefix and --glob-archives
        are mutually exclusive.
    :type glob_archives: str
    """

    prefix: str = None
    glob_archives: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchiveOutput(ArchivePattern):
    """Archive options when filtering output

    :param sort_by: Comma-separated list of sorting keys; valid keys are: timestamp, name, id;
        default is: timestamp
    :type sort_by: str
    :param first: consider first N archives after other filters were applied
    :type first: int
    :param last: consider last N archives after other filters were applied
    :type last: int
    """

    sort_by: str = None
    first: int = None
    last: int = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
