from typing import Union

import pytest
from pytest_lazyfixture import lazy_fixture

from quantus.evaluation import evaluate
from quantus.functions.explanation_func import explain

from quantus.metrics.complexity import Sparseness
from quantus.metrics.robustness import MaxSensitivity
from tests.fixtures import *

@pytest.mark.evaluate_func
@pytest.mark.parametrize(
    "model,data,params,expected",
    [
        (
            lazy_fixture("load_1d_3ch_conv_model"),
            lazy_fixture("almost_uniform_1d_no_abatch"),
            {
                "explain_func_kwargs": {"method": "Gradient",},
                "explain_func": explain,
                "eval_metrics": "{'max-Sensitivity': MaxSensitivity(**{'disable_warnings': True,})}",
                "eval_xai_methods": "{params['explain_func_kwargs']['method']: a_batch}",
                "call_kwargs": {"explain_func": explain,},
            },
            {"min": 0.0, "max": 1.0},
        ),
        (
            lazy_fixture("load_mnist_model"),
            lazy_fixture("load_mnist_images"),
            {
                "explain_func_kwargs": {"method": "Saliency",},
                "explain_func": explain,
                "eval_metrics": "{'max-Sensitivity': MaxSensitivity(**{'disable_warnings': True,})}",
                "eval_xai_methods": "{params['explain_func_kwargs']['method']: a_batch}",
                "call_kwargs": {"explain_func": explain,},
            },
            {"min": 0.0, "max": 1.0},
        ),
        (
            lazy_fixture("load_mnist_model"),
            lazy_fixture("load_mnist_images"),
            {
                "explain_func_kwargs": {"method": "IntegratedGradients",},
                "explain_func": explain,
                "eval_metrics": "{'Sparseness': Sparseness(**{'disable_warnings': True,'normalise': True,})}",
                "eval_xai_methods": "{params['explain_func_kwargs']['method']: a_batch}",
                "call_kwargs": {"explain_func": explain,},
            },
            {"min": 0.0, "max": 1.0},
        ),
        (
            lazy_fixture("load_mnist_model"),
            lazy_fixture("load_mnist_images"),
            {
                "explain_func_kwargs": {"method": "Gradient",},
                "explain_func": explain,
                "eval_metrics": "{'max-Sensitivity': MaxSensitivity(**{'disable_warnings': True,'normalise': True,})}",
                "eval_xai_methods": "{params['explain_func_kwargs']['method']: a_batch}",
                "call_kwargs": {"explain_func": explain,},
            },
            {"min": 0.0, "max": 1.0},
        ),
        (
            lazy_fixture("load_mnist_model"),
            lazy_fixture("load_mnist_images"),
            {
                "explain_func_kwargs": {"method": "IntegratedGradients",},
                "explain_func": explain,
                "eval_metrics": "{'max-Sensitivity': Sparseness(**{'disable_warnings': True,'normalise': False,})}",
                "eval_xai_methods": "[params['explain_func_kwargs']['method']]",
                "call_kwargs": {"explain_func": explain,},
            },
            {"min": 0.0, "max": 1.0},
        ),
        (
            lazy_fixture("load_mnist_model"),
            lazy_fixture("load_mnist_images"),
            {
                "explain_func_kwargs": {"method": "IntegratedGradients",},
                "explain_func": explain,
                "eval_metrics": "{'Sparseness': Sparseness(**{'disable_warnings': True, 'normalise': True,})}",
                "eval_xai_methods": "{params['explain_func_kwargs']['method'] : params['explain_func']}",
                "call_kwargs": {"explain_func": explain,},
            },
            {"min": 0.0, "max": 1.0},
        ),
        (
            lazy_fixture("load_mnist_model"),
            lazy_fixture("load_mnist_images"),
            {
                "explain_func_kwargs": {"method": "IntegratedGradients",},
                "explain_func": explain,
                "eval_metrics": "None",
                "eval_xai_methods": "None",
                "call_kwargs": {"explain_func": explain,},
            },
            {"None": None},
        ),
    ],
)
def test_evaluate_func(
    model, data: np.ndarray, params: dict, expected: Union[float, dict, bool],
):
    x_batch, y_batch = data["x_batch"], data["y_batch"]
    explain = params["explain_func"]
    call_kwargs = params.get("call_kwargs", {})
    a_batch = explain(
        model=model, inputs=x_batch, targets=y_batch, **params["explain_func_kwargs"],
    )

    if "None" in expected:
        results = evaluate(
            metrics=eval(params["eval_metrics"]),
            xai_methods=eval(params["eval_xai_methods"]),
            model=model,
            x_batch=x_batch,
            y_batch=y_batch,
            a_batch=a_batch,
            agg_func=np.mean,
            explain_func_kwargs=params["explain_func_kwargs"],
            **call_kwargs,
        )
        assert results == None, "Test failed."

    results = evaluate(
        metrics=eval(params["eval_metrics"]),
        xai_methods=eval(params["eval_xai_methods"]),
        model=model,
        x_batch=x_batch,
        y_batch=y_batch,
        a_batch=a_batch,
        agg_func=np.mean,
        explain_func_kwargs=params["explain_func_kwargs"],
        **call_kwargs,
    )

    if "min" in expected and "max" in expected:
        assert (
            results[params["explain_func_kwargs"]["method"]][
                list(eval(params["eval_metrics"]).keys())[0]
            ]
            >= expected["min"]
        ), "Test failed."
        assert (
            results[params["explain_func_kwargs"]["method"]][
                list(eval(params["eval_metrics"]).keys())[0]
            ]
            <= expected["max"]
        ), "Test failed."
