# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['turbulette',
 'turbulette.apps',
 'turbulette.apps.auth',
 'turbulette.apps.auth.policy',
 'turbulette.apps.auth.resolvers',
 'turbulette.apps.auth.resolvers.queries',
 'turbulette.apps.base',
 'turbulette.apps.base.resolvers',
 'turbulette.conf',
 'turbulette.db',
 'turbulette.management',
 'turbulette.management.templates.app',
 'turbulette.management.templates.project',
 'turbulette.management.templates.project.alembic',
 'turbulette.middleware',
 'turbulette.test',
 'turbulette.type',
 'turbulette.validation']

package_data = \
{'': ['*'],
 'turbulette.apps.auth': ['graphql/queries/*', 'graphql/types/*'],
 'turbulette.apps.base': ['graphql/*'],
 'turbulette.management.templates.app': ['graphql/*', 'resolvers/*']}

install_requires = \
['alembic>=1.4.2,<2.0.0',
 'ariadne>=0.11,<0.13',
 'async-caches>=0.3.0,<0.4.0',
 'ciso8601>=2.1.3,<3.0.0',
 'click>=7.1.2,<8.0.0',
 'gino[starlette]>=1.0.1,<2.0.0',
 'passlib[bcrypt]>=1.7.2,<2.0.0',
 'psycopg2>=2.8.5,<3.0.0',
 'pydantic[email]>=1.6.1,<2.0.0',
 'python-jwt>=3.2.6,<4.0.0',
 'simple-settings>=0.19.1,<1.1.0']

extras_require = \
{'argon2': ['argon2-cffi>=20.1.0,<21.0.0'],
 'dev_doc': ['mkdocs-material>=6.0.1,<7.0.0',
             'mkdocs-git-revision-date-plugin>=0.3,<0.4',
             'mkdocstrings>=0.14.0,<0.15.0',
             'pygments-graphql-lexer>=0.1.0,<0.2.0',
             'pymdown-extensions>=8.0.1,<9.0.0'],
 'dev_lint': ['black>=20.8b1,<21.0',
              'pylint-pytest>=0.3.0,<0.4.0',
              'prospector[with_bandit,with_mypy]>=1.3.1,<2.0.0'],
 'dev_profiling': ['tuna>=0.5.0,<0.6.0', 'memory_profiler>=0.58.0,<0.59.0'],
 'dev_test': ['pytest>=6.2.1,<7.0.0',
              'pytest-cov>=2.11.1,<3.0.0',
              'pytest-asyncio>=0.14.0,<0.15.0',
              'async-asgi-testclient>=1.4.5,<2.0.0',
              'python-multipart>=0.0.5,<0.0.6',
              'coverage[toml]>=5.3,<6.0']}

entry_points = \
{'console_scripts': ['turb = turbulette.management.cli:cli'],
 'pytest11': ['turbulette = turbulette.test.pytest_plugin']}

setup_kwargs = {
    'name': 'turbulette',
    'version': '0.5.1',
    'description': 'A batteries-included framework to build high performance, async GraphQL APIs',
    'long_description': '# Turbulette\n\n<p align="center">\n<a class="badge" href="https://github.com/turbulette/turbulette/actions?query=workflow%3ATest">\n    <img src="https://github.com/turbulette/turbulette/workflows/Test/badge.svg" alt="test"/>\n</a>\n<a class="badge" href="https://www.codacy.com/gh/turbulette/turbulette/dashboard?utm_source=github.com&utm_medium=referral&utm_content=turbulette/turbulette&utm_campaign=Badge_Coverage">\n    <img src="https://app.codacy.com/project/badge/Coverage/e244bb031e044079af419dabd40bb7fc" alt="codacy-coverage"/>\n</a>\n<a class="badge" href="https://www.codacy.com/gh/turbulette/turbulette/dashboard?utm_source=github.com&amp;utm_medium=referral&amp;utm_content=turbulette/turbulette&amp;utm_campaign=Badge_Grade">\n    <img src="https://app.codacy.com/project/badge/Grade/e244bb031e044079af419dabd40bb7fc" alt="codacy-grade"/>\n</a>\n<a class="badge" href="https://pypi.org/project/turbulette/">\n    <img src="https://img.shields.io/pypi/v/turbulette" alt="pypi"/>\n</a>\n<a class="badge" href="https://img.shields.io/pypi/pyversions/turbulette">\n    <img src="https://img.shields.io/pypi/pyversions/turbulette" alt="py-version"/>\n</a>\n<a class="badge" href="https://github.com/turbulette/turbulette/blob/main/LICENSE">\n    <img src="https://img.shields.io/pypi/l/Turbulette" alt="license"/>\n</a>\n<a class="badge" href="http://mypy-lang.org/">\n    <img src="https://img.shields.io/badge/mypy-checked-blue" alt="mypy"/>\n</a>\n<a class="badge" href="https://github.com/psf/black">\n    <img src="https://img.shields.io/badge/code%20style-black-000000.svg" alt="black"/>\n</a>\n<a class="badge" href="https://github.com/PyCQA/bandit">\n    <img src="https://img.shields.io/badge/security-bandit-yellow.svg" alt="bandit"/>\n</a>\n<a class="badge" href="https://pre-commit.com/">\n    <img src="https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white" alt="pre-commit"/>\n</a>\n<a class="badge" href="https://gitter.im/turbulette/turbulette">\n    <img src="https://badges.gitter.im/turbulette/turbulette.svg" alt="gitter"/>\n</a>\n<a class="badge" href="https://app.netlify.com/sites/turbulette/deploys">\n    <img src="https://api.netlify.com/api/v1/badges/3d71e7d8-f219-41c3-9dce-1dc0c5b92251/deploy-status" alt="netlify"/>\n</a>\n</p>\n\n<p align="center">Turbulette packages all you need to build great GraphQL APIs :</p>\n\n<p align="center"><strong><em>ASGI framework, GraphQL library, ORM and data validation</em></strong></p>\n\n---\n\nDocumentation : https://turbulette.netlify.app\n\n---\n\nFeatures :\n\n- Split your API in small, independent applications\n- Generate Pydantic models from GraphQL types\n- JWT authentication with refresh and fresh tokens\n- Declarative, powerful and extendable policy-based access control (PBAC)\n- Extendable auth user model with role management\n- Async caching (provided by async-caches)\n- Built-in CLI to manage project, apps, and DB migrations\n- Built-in pytest plugin to quickly test your resolvers\n- Settings management at project and app-level (thanks to simple-settings)\n- CSRF middleware\n- 100% test coverage\n- 100% typed, your IDE will thank you ;)\n- Handcrafted with ❤️, from 🇫🇷\n\n## Requirements\n\nPython 3.6+\n\n👍 Turbulette makes use of great tools/frameworks and wouldn\'t exist without them :\n\n- [Ariadne](https://ariadnegraphql.org/) - Schema-first GraphQL library\n- [Starlette](https://www.starlette.io/) - The little ASGI framework that shines\n- [GINO](https://python-gino.org/docs/en/master/index.html) - Lightweight, async ORM\n- [Pydantic](https://pydantic-docs.helpmanual.io/) - Powerful data validation with type annotations\n- [Alembic](https://alembic.sqlalchemy.org/en/latest/index.html) - Lightweight database migration tool\n- [simple-settings](https://github.com/drgarcia1986/simple-settings) - A generic settings system inspired by Django\'s one\n- [async-caches](https://github.com/rafalp/async-caches) - Async caching library\n- [Click](https://palletsprojects.com/p/click/) - A "Command Line Interface Creation Kit"\n\n## Installation\n\n``` bash\npip install turbulette\n```\n\nYou will also need an ASGI server, such as [uvicorn](https://www.uvicorn.org/) :\n\n``` bash\npip install uvicorn\n```\n\n----\n\n## 🚀 Quick Start\n\nHere is a short example that demonstrates a minimal project setup.\n\nWe will see how to scaffold a simple Turbulette project, create a Turbulette application, and write some GraphQL schema/resolver. It\'s advisable to start the project in a virtualenv to isolate your dependencies.\nHere we will be using [poetry](https://python-poetry.org/) :\n\n``` bash\npoetry init\n```\n\nThen, install Turbulette from PyPI :\n\n``` bash\npoetry add turbulette\n```\n\nFor the rest of the tutorial, we will assume that commands will be executed under the virtualenv. To spawn a  shell inside the virtualenv, run :\n\n```bash\npoetry shell\n```\n\n### 1: Create a project\n\nFirst, create a directory that will contain the whole project.\n\nNow, inside this folder, create your Turbulette project using the `turb` CLI :\n\n``` bash\nturb project eshop\n```\n\nYou should get with something like this :\n\n```console\n.\n└── 📁 eshop\n    ├── 📁 alembic\n    │   ├── 📄 env.py\n    │   └── 📄 script.py.mako\n    ├── 📄 .env\n    ├── 📄 alembic.ini\n    ├── 📄 app.py\n    └── 📄 settings.py\n```\n\nLet\'s break down the structure :\n\n- `📁 eshop` : Here is the so-called *Turbulette project* folder, it will contain applications and project-level configuration files\n- `📁 alembic` : Contains the [Alembic](https://alembic.sqlalchemy.org/en/latest/) scripts used when generating/applying DB migrations\n  - `📄 env.py`\n  - `📄 script.py.mako`\n- `📄 .env` : The actual project settings live here\n- `📄 app.py` : Your API entrypoint, it contains the ASGI app\n- `📄 settings.py` : Will load settings from `.env` file\n\n\nWhy have both `.env` and `settings.py`?\n\nYou don\'t *have to*. You can also put all your settings in `settings.py`.\nBut Turbulette encourage you to follow the [twelve-factor methodology](https://12factor.net),\nthat recommend to separate settings from code because config varies substantially across deploys, *code does not*.\nThis way, you can untrack `.env` from version control and only keep tracking `settings.py`, which will load settings\nfrom `.env` using Starlette\'s `Config` object.\n\n### 2: Create the first app\n\nNow it\'s time to create a Turbulette application!\n\nRun this command under the project directory (`eshop`) :\n\n```bash\nturb app --name account\n```\n\nYou need to run `turb app` under the project dir because the CLI needs to access the `almebic.ini` file to create the initial database migration.\n\nYou should see your new app under the project folder :\n\n```console\n.\n└── 📁 eshop\n    ...\n    |\n    └── 📁 account\n        ├── 📁 graphql\n        ├── 📁 migrations\n        │   └── 📄 20200926_1508_auto_ef7704f9741f_initial.py\n        ├── 📁 resolvers\n        └── 📄 models.py\n```\n\nDetails :\n\n- `📁 graphql` : All the GraphQL schema will live here\n- `📁 migrations` : Will contain database migrations generated by Alembic\n- `📁 resolvers` : Python package where you will write resolvers binded to the schema\n- `📄 models.py` : Will hold GINO models for this app\n\nWhat is this "initial" python file under `📁 migrations`?\n\nWe won\'t cover database connection in this quickstart, but note that it\'s the initial database migration\nfor the `account` app that creates its dedicated Alembic branch, needed to generate/apply per-app migrations.\n\nBefore writing some code, the only thing to do is make Turbulette aware of our lovely account app.\n\nTo do this, open `📄 eshop/settings.py` and add `"eshop.account"` to `INSTALLED_APPS`,\nso the application is registered and can be picked up by Turbulette at startup :\n\n``` python\n# List installed Turbulette apps that defines some GraphQL schema\nINSTALLED_APPS = ["eshop.account"]\n```\n\n### 3: GraphQL schema\n\nNow that we have our project scaffold, we can start writing actual schema/code.\n\nCreate a `schema.gql` file in the `📁 graphql` folder and add this base schema :\n\n``` graphql\nextend type Mutation {\n    registerCard(input: CreditCard!): SuccessOut!\n}\n\ninput CreditCard {\n    number: String!\n    expiration: Date!\n    name: String!\n}\n\ntype SuccessOut {\n    success: Boolean\n    errors: [String]\n}\n\n```\n\nNote that we *extend* the type `Mutation` because Turbulette already defines it. The same goes for `Query` type\n\nNotice that with use the `Date` scalar, it\'s one of the custom scalars provided by Turbulette. It parses string in the ISO8601 date format YYY-MM-DD.\n\n### 4: Add pydantic model\n\nWe want to validate our `CreditCard` input to ensure the user has entered a valid card number and date.\nFortunately, Turbulette integrates with [Pydantic](https://pydantic-docs.helpmanual.io/), a data validation library that uses python type annotations,\nand offers a convenient way to generate a Pydantic model from a schema type.\n\nCreate a new `📄 pyd_models.py` under `📁 account` :\n\n```python\nfrom turbulette.validation import GraphQLModel\nfrom pydantic import PaymentCardNumber\n\n\nclass CreditCard(GraphQLModel):\n    class GraphQL:\n        gql_type = "CreditCard"\n        fields = {"number": PaymentCardNumber}\n```\n\nWhat\'s happening here?\n\nThe inherited `GraphQLModel` class is a pydantic model that knows about the GraphQL schema and can produce pydantic fields from a given GraphQL type. We specify the GraphQL type with the `gql_type` attribute; it\'s the only one required.\n\nBut we also add a `fields` attribute to override the type of `number` field because it is string typed in our schema. If we don\'t add this, Turbulette will assume that `number` is a string and will annotate the number field as `str`.\n`fields` is a mapping between GraphQL field names and the type that will override the schema\'s one.\n\nLet\'s add another validation check: the expiration date. We want to ensure the user has entered a valid date (i.e., at least greater than now) :\n\n```python hl_lines="3 11 12 13 14 15"\nfrom datetime import datetime\nfrom pydantic import PaymentCardNumber\nfrom turbulette.validation import GraphQLModel, validator\n\n\nclass CreditCard(GraphQLModel):\n    class GraphQL:\n        gql_type = "CreditCard"\n        fields = {"number": PaymentCardNumber}\n\n    @validator("expiration")\n    def check_expiration_date(cls, value):\n        if value < datetime.now():\n            raise ValueError("Expiration date is invalid")\n        return value\n```\n\nWhy don\'t we use the `@validator` from Pydantic?\n\nFor those who have already used Pydantic, you probably know about the `@validator` decorator used add custom validation rules on fields.\n\nBut here, we use a `@validator` imported from `turbulette.validation`, why?\n\nThey\'re almost identical. Turbulette\'s validator is just a shortcut to the Pydantic one with `check_fields=False` as a default, instead of `True`, because we use an inherited `BaseModel`. The above snippet would correctly work if we used Pydantic\'s validator and explicitly set `@validator("expiration", check_fields=False)`.\n\n### 5: Add a resolver\n\nThe last missing piece is the resolver for our `user` mutation, to make the API returning something when querying for it.\n\nThe GraphQL part is handled by [Ariadne](https://ariadnegraphql.org/), a schema-first GraphQL library that allows binding the logic to the schema with minimal code.\n\nAs you may have guessed, we will create a new Python module in our `📁 resolvers` package.\n\nLet\'s call it `📄 user.py` :\n\n``` python\nfrom turbulette import mutation\nfrom ..pyd_models import CreditCard\n\n@mutation.field("registerCard")\nasync def register(obj, info, **kwargs):\n    return {"success": True}\n```\n\n`mutation` is the base mutation type defined by Turbulette and is used to register all mutation resolvers (hence the use of `extend type Mutation` on the schema).\nFor now, our resolver is very simple and doesn\'t do any data validation on inputs and doesn\'t handle errors.\n\nTurbulette has a `@validate` decorator that can be used to validate resolver input using a pydantic model (like the one defined in [Step 4](#4-add-pydantic-model)).\n\nHere\'s how to use it:\n\n``` python hl_lines="3 6 7"\nfrom turbulette import mutation\nfrom ..pyd_models import CreditCard\nfrom turbulette.validation import validate\n\n@mutation.field("registerCard")\n@validate(CreditCard)\nasync def register(obj, info, **kwargs):\n    return {"success": True}\n```\n\nIf the validation succeeds, you can access the validated input data in `kwargs["_val_data"]`\nBut what happens otherwise? Normally, if the validation fails, pydantic will raise a `ValidationError`,\nbut here the `@validate` decorator handles the exception and will add error messages returned by pydantic into a dedicated error field in the GraphQL response.\n\n### 5: Run it\n\nOur `registerCard` mutation is now binded to the schema, so let\'s test it.\n\nStart the server in the root directory (the one containing `📁 eshop` folder) :\n\n```bash\nuvicorn eshop.app:app --port 8000\n```\n\nNow, go to [http://localhost:8000/graphql](http://localhost:8000/graphql), you will see the [GraphQL Playground](https://github.com/graphql/graphql-playground) IDE.\nFinally, run the `registerCard` mutation, for example :\n\n``` graphql\nmutation card {\n  registerCard(\n    input: {\n      number: "4000000000000002"\n      expiration: "2023-05-12"\n      name: "John Doe"\n    }\n  ) {\n    success\n    errors\n  }\n}\n```\n\nShould give you the following expected result :\n\n``` json\n{\n  "data": {\n    "registerCard": {\n      "success": true,\n      "errors": null\n    }\n  }\n}\n```\n\nNow, try entering a wrong date (before *now*). You should see the validation error as expected:\n\n```json\n{\n  "data": {\n    "registerCard": {\n      "success": null,\n      "errors": [\n        "expiration: Expiration date is invalid"\n      ]\n    }\n  }\n}\n```\n\nHow the error message end in the `errors` key?\n\nIndeed, we didn\'t specify anywhere that validation errors should be passed to the `errors` key in our `SuccessOut` GraphQL type.\nThat is because Turbulette has a setting called `ERROR_FIELD`, which defaults to `"errors"`.\nThis setting indicates the error field on the GraphLQ output type used by Turbulette when collecting query errors.\n\nIt means that if you didn\'t specify `ERROR_FIELD` on the GraphQL type, you would get an exception telling you that the field is missing.\n\nIt\'s the default (and recommended) way of handling errors in Turbulette. Still, as all happens in the `@validate`, you can always remove it and manually instantiate your Pydantic models in resolvers.\n\nGood job! 👏\n\nThat was a straightforward example, showing off a simple Turbulette API set up. To get the most of it, follow the  [User Guide](https://python-turbulette.github.io/turbulette/user-guide/) .\n',
    'author': 'Matthieu MN',
    'author_email': 'matthieu.macnab@pm.me',
    'maintainer': 'Matthieu MN',
    'maintainer_email': 'matthieu.macnab@pm.me',
    'url': 'https://github.com/turbulette/turbulette/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
