"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
describe('UserPoolIdentityProvider', () => {
    describe('facebook', () => {
        test('defaults', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
            });
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                ProviderName: 'Facebook',
                ProviderType: 'Facebook',
                ProviderDetails: {
                    client_id: 'fb-client-id',
                    client_secret: 'fb-client-secret',
                    authorize_scopes: 'public_profile',
                },
            });
        });
        test('scopes & api_version', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
                scopes: ['scope1', 'scope2'],
                apiVersion: 'version1',
            });
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                ProviderName: 'Facebook',
                ProviderType: 'Facebook',
                ProviderDetails: {
                    client_id: 'fb-client-id',
                    client_secret: 'fb-client-secret',
                    authorize_scopes: 'scope1,scope2',
                    api_version: 'version1',
                },
            });
        });
        test('registered with user pool', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            const provider = new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
            });
            // THEN
            expect(pool.identityProviders).toContain(provider);
        });
        test('attribute mapping', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
                attributeMapping: {
                    givenName: lib_1.ProviderAttribute.FACEBOOK_NAME,
                    address: lib_1.ProviderAttribute.other('fb-address'),
                    custom: {
                        customAttr1: lib_1.ProviderAttribute.FACEBOOK_EMAIL,
                        customAttr2: lib_1.ProviderAttribute.other('fb-custom-attr'),
                    },
                },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                AttributeMapping: {
                    given_name: 'name',
                    address: 'fb-address',
                    customAttr1: 'email',
                    customAttr2: 'fb-custom-attr',
                },
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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