"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    /**
     * Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    /**
     * Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    /**
     * Email only
     */
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    /**
     * Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    /**
     * (Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.
     */
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    /**
     * None – users will have to contact an administrator to reset their passwords
     */
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole) !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                    },
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn,
            };
        }
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _a, _b;
                return ({
                    name: attr_names_1.StandardAttributeNames[attrName],
                    mutable: (_a = attr.mutable) !== null && _a !== void 0 ? _a : true,
                    required: (_b = attr.required) !== null && _b !== void 0 ? _b : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        var _a;
        const accountRecovery = (_a = props.accountRecovery) !== null && _a !== void 0 ? _a : AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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