"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolDomain = void 0;
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) &&
            !core_1.Token.isUnresolved((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_c = props.cognitoDomain) === null || _c === void 0 ? void 0 : _c.domainPrefix) || ((_d = props.customDomain) === null || _d === void 0 ? void 0 : _d.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        if (!this.cloudFrontCustomResource) {
            const sdkCall = {
                service: 'CognitoIdentityServiceProvider',
                action: 'describeUserPoolDomain',
                parameters: {
                    Domain: this.domainName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
            };
            this.cloudFrontCustomResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
                resourceType: 'Custom::UserPoolCloudFrontDomainName',
                onCreate: sdkCall,
                onUpdate: sdkCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    // DescribeUserPoolDomain only supports access level '*'
                    // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                    resources: ['*'],
                }),
            });
        }
        return this.cloudFrontCustomResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize the behaviour of this method.
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLWRvbWFpbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVzZXItcG9vbC1kb21haW4udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esd0NBQTZFO0FBQzdFLGdFQUF1SDtBQUN2SCwyREFBd0Q7QUEyRXhEOztHQUVHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsZUFBUTtJQU0xQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCOztRQUNsRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDbEQsTUFBTSxJQUFJLEtBQUssQ0FBQywwRUFBMEUsQ0FBQyxDQUFDO1NBQzdGO1FBRUQsSUFBSSxPQUFBLEtBQUssQ0FBQyxhQUFhLDBDQUFFLFlBQVk7WUFDbkMsQ0FBQyxZQUFLLENBQUMsWUFBWSxPQUFDLEtBQUssQ0FBQyxhQUFhLDBDQUFFLFlBQVksQ0FBQztZQUN0RCxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUV4RCxNQUFNLElBQUksS0FBSyxDQUFDLDBGQUEwRixDQUFDLENBQUM7U0FDN0c7UUFFRCxJQUFJLENBQUMsZUFBZSxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBRTdDLE1BQU0sVUFBVSxHQUFHLE9BQUEsS0FBSyxDQUFDLGFBQWEsMENBQUUsWUFBWSxNQUFJLE1BQUEsS0FBSyxDQUFDLFlBQVksMENBQUUsVUFBVyxDQUFBLENBQUM7UUFDeEYsTUFBTSxRQUFRLEdBQUcsSUFBSSxxQ0FBaUIsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3ZELFVBQVUsRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLFVBQVU7WUFDckMsTUFBTSxFQUFFLFVBQVU7WUFDbEIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDdkgsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFVBQVUsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsb0JBQW9CO1FBQzdCLElBQUksQ0FBQyxJQUFJLENBQUMsd0JBQXdCLEVBQUU7WUFDbEMsTUFBTSxPQUFPLEdBQWU7Z0JBQzFCLE9BQU8sRUFBRSxnQ0FBZ0M7Z0JBQ3pDLE1BQU0sRUFBRSx3QkFBd0I7Z0JBQ2hDLFVBQVUsRUFBRTtvQkFDVixNQUFNLEVBQUUsSUFBSSxDQUFDLFVBQVU7aUJBQ3hCO2dCQUNELGtCQUFrQixFQUFFLHFDQUFrQixDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDO2FBQzNELENBQUM7WUFDRixJQUFJLENBQUMsd0JBQXdCLEdBQUcsSUFBSSxvQ0FBaUIsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7Z0JBQ2xGLFlBQVksRUFBRSxzQ0FBc0M7Z0JBQ3BELFFBQVEsRUFBRSxPQUFPO2dCQUNqQixRQUFRLEVBQUUsT0FBTztnQkFDakIsTUFBTSxFQUFFLDBDQUF1QixDQUFDLFlBQVksQ0FBQztvQkFDM0Msd0RBQXdEO29CQUN4RCxrSUFBa0k7b0JBQ2xJLFNBQVMsRUFBRSxDQUFFLEdBQUcsQ0FBRTtpQkFDbkIsQ0FBQzthQUNILENBQUMsQ0FBQztTQUNKO1FBQ0QsT0FBTyxJQUFJLENBQUMsd0JBQXdCLENBQUMsZ0JBQWdCLENBQUMsMENBQTBDLENBQUMsQ0FBQztJQUNwRyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3hCLE9BQU8sV0FBVyxJQUFJLENBQUMsVUFBVSxTQUFTLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxvQkFBb0IsQ0FBQztTQUNyRjtRQUNELE9BQU8sV0FBVyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxTQUFTLENBQUMsTUFBc0IsRUFBRSxPQUF5Qjs7UUFDaEUsSUFBSSxZQUFvQixDQUFDO1FBQ3pCLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxzQkFBc0IsRUFBRTtZQUM1QyxZQUFZLEdBQUcsTUFBTSxDQUFDO1NBQ3ZCO2FBQU0sSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLGlCQUFpQixFQUFFO1lBQzlDLFlBQVksR0FBRyxPQUFPLENBQUM7U0FDeEI7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMseUdBQXlHLENBQUMsQ0FBQztTQUM1SDtRQUNELE1BQU0sSUFBSSxTQUFHLE9BQU8sQ0FBQyxVQUFVLG1DQUFJLFFBQVEsQ0FBQztRQUM1QyxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sRUFBRSxHQUFHLElBQUksY0FBYyxNQUFNLENBQUMsZ0JBQWdCLGtCQUFrQixZQUFZLGlCQUFpQixPQUFPLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDM0ksQ0FBQztDQUNGO0FBdEZELHdDQXNGQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElDZXJ0aWZpY2F0ZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIFJlc291cmNlLCBTdGFjaywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEF3c0N1c3RvbVJlc291cmNlLCBBd3NDdXN0b21SZXNvdXJjZVBvbGljeSwgQXdzU2RrQ2FsbCwgUGh5c2ljYWxSZXNvdXJjZUlkIH0gZnJvbSAnQGF3cy1jZGsvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDZm5Vc2VyUG9vbERvbWFpbiB9IGZyb20gJy4vY29nbml0by5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVVzZXJQb29sIH0gZnJvbSAnLi91c2VyLXBvb2wnO1xuaW1wb3J0IHsgVXNlclBvb2xDbGllbnQgfSBmcm9tICcuL3VzZXItcG9vbC1jbGllbnQnO1xuXG4vKipcbiAqIFJlcHJlc2VudHMgYSB1c2VyIHBvb2wgZG9tYWluLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElVc2VyUG9vbERvbWFpbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgZG9tYWluIHRoYXQgd2FzIHNwZWNpZmllZCB0byBiZSBjcmVhdGVkLlxuICAgKiBJZiBgY3VzdG9tRG9tYWluYCB3YXMgc2VsZWN0ZWQsIHRoaXMgaG9sZHMgdGhlIGZ1bGwgZG9tYWluIG5hbWUgdGhhdCB3YXMgc3BlY2lmaWVkLlxuICAgKiBJZiB0aGUgYGNvZ25pdG9Eb21haW5gIHdhcyB1c2VkLCBpdCBjb250YWlucyB0aGUgcHJlZml4IHRvIHRoZSBDb2duaXRvIGhvc3RlZCBkb21haW4uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBPcHRpb25zIHdoaWxlIHNwZWNpZnlpbmcgY3VzdG9tIGRvbWFpblxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29nbml0by11c2VyLXBvb2xzLWFkZC1jdXN0b20tZG9tYWluLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDdXN0b21Eb21haW5PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBjdXN0b20gZG9tYWluIG5hbWUgdGhhdCB5b3Ugd291bGQgbGlrZSB0byBhc3NvY2lhdGUgd2l0aCB0aGlzIFVzZXIgUG9vbC5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNlcnRpZmljYXRlIHRvIGFzc29jaWF0ZSB3aXRoIHRoaXMgZG9tYWluLlxuICAgKi9cbiAgcmVhZG9ubHkgY2VydGlmaWNhdGU6IElDZXJ0aWZpY2F0ZTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIHdoaWxlIHNwZWNpZnlpbmcgYSBjb2duaXRvIHByZWZpeCBkb21haW4uXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbHMtYXNzaWduLWRvbWFpbi1wcmVmaXguaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvZ25pdG9Eb21haW5PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwcmVmaXggdG8gdGhlIENvZ25pdG8gaG9zdGVkIGRvbWFpbiBuYW1lIHRoYXQgd2lsbCBiZSBhc3NvY2lhdGVkIHdpdGggdGhlIHVzZXIgcG9vbC5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpblByZWZpeDogc3RyaW5nO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gY3JlYXRlIGEgVXNlclBvb2xEb21haW5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBVc2VyUG9vbERvbWFpbk9wdGlvbnMge1xuICAvKipcbiAgICogQXNzb2NpYXRlIGEgY3VzdG9tIGRvbWFpbiB3aXRoIHlvdXIgdXNlciBwb29sXG4gICAqIEVpdGhlciBgY3VzdG9tRG9tYWluYCBvciBgY29nbml0b0RvbWFpbmAgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hZGQtY3VzdG9tLWRvbWFpbi5odG1sXG4gICAqIEBkZWZhdWx0IC0gbm90IHNldCBpZiBgY29nbml0b0RvbWFpbmAgaXMgc3BlY2lmaWVkLCBvdGhlcndpc2UsIHRocm93cyBhbiBlcnJvci5cbiAgICovXG4gIHJlYWRvbmx5IGN1c3RvbURvbWFpbj86IEN1c3RvbURvbWFpbk9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZSBhIGNvZ25pdG8gcHJlZml4IGRvbWFpbiB3aXRoIHlvdXIgdXNlciBwb29sXG4gICAqIEVpdGhlciBgY3VzdG9tRG9tYWluYCBvciBgY29nbml0b0RvbWFpbmAgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hc3NpZ24tZG9tYWluLXByZWZpeC5odG1sXG4gICAqIEBkZWZhdWx0IC0gbm90IHNldCBpZiBgY3VzdG9tRG9tYWluYCBpcyBzcGVjaWZpZWQsIG90aGVyd2lzZSwgdGhyb3dzIGFuIGVycm9yLlxuICAgKi9cbiAgcmVhZG9ubHkgY29nbml0b0RvbWFpbj86IENvZ25pdG9Eb21haW5PcHRpb25zO1xufVxuXG4vKipcbiAqIFByb3BzIGZvciBVc2VyUG9vbERvbWFpbiBjb25zdHJ1Y3RcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBVc2VyUG9vbERvbWFpblByb3BzIGV4dGVuZHMgVXNlclBvb2xEb21haW5PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSB1c2VyIHBvb2wgdG8gd2hpY2ggdGhpcyBkb21haW4gc2hvdWxkIGJlIGFzc29jaWF0ZWQuXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbDogSVVzZXJQb29sO1xufVxuXG4vKipcbiAqIERlZmluZSBhIHVzZXIgcG9vbCBkb21haW5cbiAqL1xuZXhwb3J0IGNsYXNzIFVzZXJQb29sRG9tYWluIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJVXNlclBvb2xEb21haW4ge1xuICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuICBwcml2YXRlIGlzQ29nbml0b0RvbWFpbjogYm9vbGVhbjtcblxuICBwcml2YXRlIGNsb3VkRnJvbnRDdXN0b21SZXNvdXJjZT86IEF3c0N1c3RvbVJlc291cmNlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBVc2VyUG9vbERvbWFpblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmICghIXByb3BzLmN1c3RvbURvbWFpbiA9PT0gISFwcm9wcy5jb2duaXRvRG9tYWluKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09uZSBvZiwgYW5kIG9ubHkgb25lIG9mLCBjb2duaXRvRG9tYWluIG9yIGN1c3RvbURvbWFpbiBtdXN0IGJlIHNwZWNpZmllZCcpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5jb2duaXRvRG9tYWluPy5kb21haW5QcmVmaXggJiZcbiAgICAgICFUb2tlbi5pc1VucmVzb2x2ZWQocHJvcHMuY29nbml0b0RvbWFpbj8uZG9tYWluUHJlZml4KSAmJlxuICAgICAgIS9eW2EtejAtOS1dKyQvLnRlc3QocHJvcHMuY29nbml0b0RvbWFpbi5kb21haW5QcmVmaXgpKSB7XG5cbiAgICAgIHRocm93IG5ldyBFcnJvcignZG9tYWluUHJlZml4IGZvciBjb2duaXRvRG9tYWluIGNhbiBjb250YWluIG9ubHkgbG93ZXJjYXNlIGFscGhhYmV0cywgbnVtYmVycyBhbmQgaHlwaGVucycpO1xuICAgIH1cblxuICAgIHRoaXMuaXNDb2duaXRvRG9tYWluID0gISFwcm9wcy5jb2duaXRvRG9tYWluO1xuXG4gICAgY29uc3QgZG9tYWluTmFtZSA9IHByb3BzLmNvZ25pdG9Eb21haW4/LmRvbWFpblByZWZpeCB8fCBwcm9wcy5jdXN0b21Eb21haW4/LmRvbWFpbk5hbWUhO1xuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmblVzZXJQb29sRG9tYWluKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHVzZXJQb29sSWQ6IHByb3BzLnVzZXJQb29sLnVzZXJQb29sSWQsXG4gICAgICBkb21haW46IGRvbWFpbk5hbWUsXG4gICAgICBjdXN0b21Eb21haW5Db25maWc6IHByb3BzLmN1c3RvbURvbWFpbiA/IHsgY2VydGlmaWNhdGVBcm46IHByb3BzLmN1c3RvbURvbWFpbi5jZXJ0aWZpY2F0ZS5jZXJ0aWZpY2F0ZUFybiB9IDogdW5kZWZpbmVkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5kb21haW5OYW1lID0gcmVzb3VyY2UucmVmO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBkb21haW4gbmFtZSBvZiB0aGUgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24gYXNzb2NpYXRlZCB3aXRoIHRoZSB1c2VyIHBvb2wgZG9tYWluLlxuICAgKi9cbiAgcHVibGljIGdldCBjbG91ZEZyb250RG9tYWluTmFtZSgpOiBzdHJpbmcge1xuICAgIGlmICghdGhpcy5jbG91ZEZyb250Q3VzdG9tUmVzb3VyY2UpIHtcbiAgICAgIGNvbnN0IHNka0NhbGw6IEF3c1Nka0NhbGwgPSB7XG4gICAgICAgIHNlcnZpY2U6ICdDb2duaXRvSWRlbnRpdHlTZXJ2aWNlUHJvdmlkZXInLFxuICAgICAgICBhY3Rpb246ICdkZXNjcmliZVVzZXJQb29sRG9tYWluJyxcbiAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgIERvbWFpbjogdGhpcy5kb21haW5OYW1lLFxuICAgICAgICB9LFxuICAgICAgICBwaHlzaWNhbFJlc291cmNlSWQ6IFBoeXNpY2FsUmVzb3VyY2VJZC5vZih0aGlzLmRvbWFpbk5hbWUpLFxuICAgICAgfTtcbiAgICAgIHRoaXMuY2xvdWRGcm9udEN1c3RvbVJlc291cmNlID0gbmV3IEF3c0N1c3RvbVJlc291cmNlKHRoaXMsICdDbG91ZEZyb250RG9tYWluTmFtZScsIHtcbiAgICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpVc2VyUG9vbENsb3VkRnJvbnREb21haW5OYW1lJyxcbiAgICAgICAgb25DcmVhdGU6IHNka0NhbGwsXG4gICAgICAgIG9uVXBkYXRlOiBzZGtDYWxsLFxuICAgICAgICBwb2xpY3k6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5LmZyb21TZGtDYWxscyh7XG4gICAgICAgICAgLy8gRGVzY3JpYmVVc2VyUG9vbERvbWFpbiBvbmx5IHN1cHBvcnRzIGFjY2VzcyBsZXZlbCAnKidcbiAgICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jb2duaXRvdXNlcnBvb2xzLmh0bWwjYW1hem9uY29nbml0b3VzZXJwb29scy1hY3Rpb25zLWFzLXBlcm1pc3Npb25zXG4gICAgICAgICAgcmVzb3VyY2VzOiBbICcqJyBdLFxuICAgICAgICB9KSxcbiAgICAgIH0pO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5jbG91ZEZyb250Q3VzdG9tUmVzb3VyY2UuZ2V0UmVzcG9uc2VGaWVsZCgnRG9tYWluRGVzY3JpcHRpb24uQ2xvdWRGcm9udERpc3RyaWJ1dGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBVUkwgdG8gdGhlIGhvc3RlZCBVSSBhc3NvY2lhdGVkIHdpdGggdGhpcyBkb21haW5cbiAgICovXG4gIHB1YmxpYyBiYXNlVXJsKCk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMuaXNDb2duaXRvRG9tYWluKSB7XG4gICAgICByZXR1cm4gYGh0dHBzOi8vJHt0aGlzLmRvbWFpbk5hbWV9LmF1dGguJHtTdGFjay5vZih0aGlzKS5yZWdpb259LmFtYXpvbmNvZ25pdG8uY29tYDtcbiAgICB9XG4gICAgcmV0dXJuIGBodHRwczovLyR7dGhpcy5kb21haW5OYW1lfWA7XG4gIH1cblxuICAvKipcbiAgICogVGhlIFVSTCB0byB0aGUgc2lnbiBpbiBwYWdlIGluIHRoaXMgZG9tYWluIHVzaW5nIGEgc3BlY2lmaWMgVXNlclBvb2xDbGllbnRcbiAgICogQHBhcmFtIGNsaWVudCBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXSB0aGUgdXNlciBwb29sIGNsaWVudCB0aGF0IHRoZSBVSSB3aWxsIHVzZSB0byBpbnRlcmFjdCB3aXRoIHRoZSBVc2VyUG9vbFxuICAgKiBAcGFyYW0gb3B0aW9ucyBvcHRpb25zIHRvIGN1c3RvbWl6ZSB0aGUgYmVoYXZpb3VyIG9mIHRoaXMgbWV0aG9kLlxuICAgKi9cbiAgcHVibGljIHNpZ25JblVybChjbGllbnQ6IFVzZXJQb29sQ2xpZW50LCBvcHRpb25zOiBTaWduSW5VcmxPcHRpb25zKTogc3RyaW5nIHtcbiAgICBsZXQgcmVzcG9uc2VUeXBlOiBzdHJpbmc7XG4gICAgaWYgKGNsaWVudC5vQXV0aEZsb3dzLmF1dGhvcml6YXRpb25Db2RlR3JhbnQpIHtcbiAgICAgIHJlc3BvbnNlVHlwZSA9ICdjb2RlJztcbiAgICB9IGVsc2UgaWYgKGNsaWVudC5vQXV0aEZsb3dzLmltcGxpY2l0Q29kZUdyYW50KSB7XG4gICAgICByZXNwb25zZVR5cGUgPSAndG9rZW4nO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3NpZ25JblVybCBpcyBub3Qgc3VwcG9ydGVkIGZvciBjbGllbnRzIHdpdGhvdXQgYXV0aG9yaXphdGlvbkNvZGVHcmFudCBvciBpbXBsaWNpdENvZGVHcmFudCBmbG93IGVuYWJsZWQnKTtcbiAgICB9XG4gICAgY29uc3QgcGF0aCA9IG9wdGlvbnMuc2lnbkluUGF0aCA/PyAnL2xvZ2luJztcbiAgICByZXR1cm4gYCR7dGhpcy5iYXNlVXJsKCl9JHtwYXRofT9jbGllbnRfaWQ9JHtjbGllbnQudXNlclBvb2xDbGllbnRJZH0mcmVzcG9uc2VfdHlwZT0ke3Jlc3BvbnNlVHlwZX0mcmVkaXJlY3RfdXJpPSR7b3B0aW9ucy5yZWRpcmVjdFVyaX1gO1xuICB9XG59XG5cbi8qKlxuICogT3B0aW9ucyB0byBjdXN0b21pemUgdGhlIGJlaGF2aW91ciBvZiBgc2lnbkluVXJsKClgXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2lnbkluVXJsT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBXaGVyZSB0byByZWRpcmVjdCB0byBhZnRlciBzaWduIGluXG4gICAqL1xuICByZWFkb25seSByZWRpcmVjdFVyaTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGF0aCBpbiB0aGUgVVJJIHdoZXJlIHRoZSBzaWduLWluIHBhZ2UgaXMgbG9jYXRlZFxuICAgKiBAZGVmYXVsdCAnL2xvZ2luJ1xuICAgKi9cbiAgcmVhZG9ubHkgc2lnbkluUGF0aD86IHN0cmluZztcbn1cbiJdfQ==