#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Sources:
KAMAE et al. (2006); CPARAMLIB (Nikalas Karlsson)

@author: Sergey Koldobskiy, Armen Lskavyan
"""
import numpy as np

# GAMMA-RAYS (TAB-3)
##############################################################################
##############################################################################
def gamma_param_nd(Tp):
    y = np.log10(Tp*0.001)
    a = np.zeros(9)
    if ((Tp > 0.488)):
        z = y + 3.3
        a[0] = -0.51187 * z +7.6179 *np.power(z, 2) - 2.1332*np.power(z, 3) + 0.22184*np.power(z, 4)
        a[1] = -(1.2592 * 1e-5) + (1.4439 * 1e-5)*np.exp(-0.29360 * (y + 3.4)) + (5.9363*1e-5)/(y+ 4.1485) + 2.2640 * 1e-6*y - (3.3723*1e-7)*np.power(y,2)
        a[2] = -(174.83) + 152.78 * np.log10(1.5682*( y + 3.4)) -  808.74/( y + 4.6157)
        a[3] = 0.81177+ 0.56385*y+ 0.0040031*np.power(y,2) - 0.0057658*np.power(y,3) + 0.00012057*np.power(y,4)
        z = y + 3.32
        a[4] = 0.68631*(z+0.02) + 10.145*np.power(z+0.02,2) - 4.6176*np.power(z+0.02,3) + 0.86824*np.power(z+0.02,4) - 0.053741*np.power(z+0.02,5)
        a[5] = (9.0466* 1e-7)+(1.4539* 1e-6)* np.log10 (0.015204 *( y + 3.4)) +(1.3253* 1e-4)/np.power(( y + 4.7171),2)-(4.1228*1e-7)*y + (2.2036 * 1e-7)*y*y
        a[6] = -339.45+(618.73)*np.log10 (0.31595 *( y + 3.9)) + 250.20/np.power(( y+ 4.4395),2)
        a[7] = -35.105 + 36.167 *y - 9.3575*np.power(y,2) + 0.33717*np.power(y,3)
        a[8] = 0.17554 + 0.37300 *y - 0.014938*np.power(y,2) + 0.0032314*np.power(y,3) + 0.0025579*np.power(y,4)
    return a
##########################################
def gamma_param_diff(Tp):
    y = np.log10(Tp*0.001)
    b = np.zeros(8)
    if ((Tp > 1.94)):
        if (Tp > 5.52):
            b[0] = 60.142 * np.tanh( - 0.37555 *( y + 2.2)) - 5.9564*np.power(( y + 0.59913),2) + 6.0162*1e-3*np.power((y+ 9.4773),4)
            b[1] = 35.322 + 3.8026*np.tanh(- 2.5979*( y + 1.9)) - 2.1870*1e-4*np.power( y+369.13,2)
            b[2] = -15.732 -0.082064* np.tanh (-1.9621*( y+ 2.1))+ 2.3355 *1e-4*np.power(y + 252.43,2)
            b[3] = -0.086827+ 0.37646 * np.exp( -0.53053*np.power(( y+ 1.0444)/(1.0 + 0.27437*( y+ 1.0444)),2))
        else :
            b[0] = 0.0
            b[1] = 0.0
            b[2] = 0.0
            b[3] = 0.0
        b[4] = 2.5982 +0.39131*np.power(y + 2.95,2)-0.0049693*np.power(y+ 2.95,4)+0.94131*np.exp(-24.347*np.power(y+ 2.45-0.19717*np.power(y+ 2.45,2),2))
        b[5] = 0.11198- 0.64582*y+ 0.16114*np.power(y,2)+ 2.2853*np.exp(-0.0032432*np.power(( y-0.83562)/(1.0+ 0.33933*(y-0.83562)),2))
        b[6] = 1.7843 + 0.91914*y + 0.050118*np.power(y,2) + 0.038096*np.power(y,3)- 0.027334*np.power(y,4) -0.0035556*np.power(y,5) +0.0025742*np.power(y,6)
        b[7] = -0.19870- 0.071003*y+ 0.019328*np.power(y,2)- 0.28321*np.exp(-6.0516*np.power( y+ 1.8441,2))
    return b
##########################################
def gamma_param_delta(Tp):
    y = np.log10(Tp*0.001)
    c = np.zeros(5)
    if ((Tp >= 0.488) and (Tp <= 1.95)):
        z =  y+ 3.1301
        c[0] = 2.4316*np.exp(-69.484*np.power(z/(1.0+ 1.24921*z),2)) - 6.3003 - 9.5349/y+ 0.38121*np.power(y,2)
        c[1] = 56.872+ 40.627*y+ 7.7528*np.power(y,2)
        c[2] = -5.4918- 6.7872*np.tanh(4.7128*( y+ 2.1))+0.68048*y
        c[3] = -0.36414+ 0.039777*y
        c[4] = -0.72807 -0.48828*y- 0.092876*np.power(y,2)
    return c
##########################################
def gamma_param_res(Tp):
    y = np.log10(Tp*0.001)
    d = np.zeros(5)
    if ((Tp >= 0.69) and (Tp <= 2.76)):
        d[0] = 3.2433* np.exp(-57.133*np.power(( y + 2.9507)/(1.0 + 1.2912*( y + 2.9507)),2)) - 1.0640 - 0.43925*y
        d[1] = 16.901+ 5.9539*y- 2.1257*np.power(y,2)- 0.92057*np.power(y,3)
        d[2] = -6.6638- 7.5010*np.tanh (30.322*( y+ 2.1)) + 0.54662*y
        d[3] = -1.50648 -0.87211*y- 0.17097*np.power(y,2)
        d[4] = 0.42795+ 0.55136*y + 0.20707*np.power(y,2)+ 0.027552*np.power(y,3)
    return d


# ELECTRON (TAB-4)
##########################################
##########################################
def elec_param_nd(Tp):
    y = np.log10(Tp*0.001)
    a = np.zeros(9)
    if ((Tp > 0.4)): # and (Tp < 512000.1)):
        a[0] = -0.018639*(y + 3.3) + 2.4315*np.power(y + 3.3, 2) - 0.57719*np.power(y + 3.3, 3) + 0.063435*np.power(y + 3.3, 4);
        a[1] = 7.1827e-6 - 3.5067e-6*y + 1.3264e-6*y*y - 3.3481e-7*y*y*y + 2.3551e-8*y*y*y*y + 3.4297e-8*y*y*y*y*y;
        a[2] = 563.91 - 362.18*np.log10(2.7187*(y + 3.4)) - 2.8924e4/np.power(y + 7.9031, 2);
        a[3] = 0.52684 + 0.57717*y + 0.0045336*y*y - 0.0089066*y*y*y;
        a[4] = 0.36108*(y + 3.32) + 1.6963*np.power(y + 3.32, 2) - 0.074456*np.power(y + 3.32, 3) - 0.071455*np.power(y + 3.32, 4) + 0.010473*np.power(y + 3.32, 5);
        a[5] = 9.7387e-5 + 7.8573e-5*np.log10(0.0036055*(y + 4.3)) + 0.00024660/(y + 4.9390) - 3.8097e-7*y*y;
        a[6] = -273.00 - 106.22*np.log10(0.34100*(y + 3.4)) + 89.037*y - 12.546*y*y;
        a[7] = 432.53 - 883.99*np.log10(0.19737*(y + 3.9)) - 4.1938e4/np.power(y + 8.5518, 2);
        a[8] = -0.12756 + 0.43478*y - 0.0027797*y*y -0.0083074*y*y*y;

    return a
##########################################
def elec_param_diff(Tp):
    y = np.log10(Tp*0.001)
    b = np.zeros(8)
    if ((Tp > 1.38)): # and (Tp < 512000.1)):
        if (Tp > 5.52):
            b[0] = 0.20463*np.tanh(-6.2370*(y + 2.2)) - 0.16362*np.power(y + 1.6878, 2) + 3.5183e-4*np.power(y + 9.6400, 4);
            b[1] = 1.6537 + 3.8530*np.exp(-3.2027*np.power((y + 2.0154)/(1.0 + 0.62779*(y + 2.0154)), 2));
            b[2] = -10.722 - 0.082672*np.tanh(-1.8879*(y + 2.1)) + 0.00014895*np.power(y + 256.63, 2);
            b[3] = -0.023752 - 0.51734*np.exp(-3.3087*np.power((y + 1.9877)/(1.0 + 0.40300*(y + 1.988)), 2));
        else :
            b[0]=0.0
            b[1]=0.0
            b[2]=0.0
            b[3]=0.0

        b[4] = 0.94921 + 0.12280*np.power(y + 2.9, 2) - 7.1585e-4*np.power(y + 2.9, 4) + 0.52130*np.log10(y + 2.9);
        b[5] = -4.2295 - 1.0025*np.tanh(9.0733*(y + 1.9)) - 0.11452*(y - 62.382);
        b[6] = 1.4862 + 0.99544*y - 0.042763*y*y - 0.0040065*y*y*y + 0.0057987*y*y*y*y;
        b[7] = 6.2629 + 6.9517*np.tanh(-0.36480*(y + 2.1)) - 0.26033*np.power(y - 2.8542, 2);
    return b
##########################################
def elec_param_delta(Tp):
    c = np.zeros(5)
    return c
##########################################
def elec_param_res(Tp):
    y = np.log10(Tp*0.001)
    d = np.zeros(5)
    if ((Tp >= 0.6) and (Tp <= 2.9)):
        d[0] = 0.37790*np.exp(-56.826*np.power((y + 2.9537)/(1.0 + 1.5221*(y + 2.9537)), 2)) - 0.059458 + 0.0096583*y*y;
        d[1] = -5.5135 - 3.3988*y;
        d[2] = -7.1209 - 7.1850*np.tanh(30.801*(y + 2.1)) + 0.35108*y;
        d[3] = -6.7841 - 4.8385*y - 0.91523*y*y;
        d[4] = -134.03 - 139.63*y - 48.316*y*y - 5.5526*y*y*y;
    return d



# POSITRON
##########################################
##########################################
def posi_param_nd(Tp):
    y = np.log10(Tp*0.001)
    a = np.zeros(9)
    if ((Tp > 0.4)): # and (Tp < 512000.1)):
        a[0] = -0.79606*(y + 3.3) + 7.7496*np.power(y + 3.3, 2) - 3.9326*np.power(y + 3.3, 3) + 0.80202*np.power(y + 3.3, 4) - 0.054994*np.power(y + 3.3, 5);
        a[1] = 6.7943e-6 - 3.5345e-6*y + 6.0927e-7*y*y + 2.0219e-7*y*y*y + 5.1005e-8*y*y*y*y - 4.2622e-8*y*y*y*y*y;
        a[2] = 44.827 + 81.378*np.log10(0.027733*(y + 3.5)) - 1.3886e4/np.power(y + 8.4417, 2);
        a[3] = 0.52010 + 0.59336*y + 0.012032*y*y - 0.0064242*y*y*y;
        a[4] = 2.1361*(y + 3.32) + 1.8514*np.power(y + 3.32, 2) - 0.47572*np.power(y + 3.32, 3) + 0.0032043*np.power(y + 3.32, 4) + 0.0082955*np.power(y + 3.32, 5);
        a[5] = 1.0845e-6 + 1.4336e-6*np.log10(0.0077255*(y + 4.3)) + 0.00013018/np.power(y + 4.8188, 2) + 9.3601e-8*y;
        a[6] = -267.74 + 14.175*np.log10(0.35391*(y + 3.4)) + 64.669*y - 7.7036*y*y;
        a[7] = 138.26 - 529.84*np.log10(0.12467*(y + 3.9)) - 1.9869e4/np.power(y + 7.6884, 2) + 1.0675*y*y;
        a[8] = -0.14707 + 0.40135*y + 0.0039899*y*y - 0.0016602*y*y*y;
    return a
##########################################
def posi_param_diff(Tp):
    y = np.log10(Tp*0.001)
    b = np.zeros(8)
    if ((Tp > 1.38)): # and (Tp < 512000.1)):
        if (Tp > 11.05):
            b[0] = 29.192*np.tanh(-0.37879*(y + 2.2)) - 3.2196*np.power(y + 0.67500, 2) + 3.6687e-3*np.power(y + 9.0824, 4);
            b[1] = -142.97 + 147.86*np.exp(-0.37194*np.power((y + 1.8781)/(1.0 + 3.8389*(y + 1.8781)), 2));
            b[2] = -14.487 - 4.2223*np.tanh(-13.546*(y + 2.2)) + 0.00016988*np.power(y + 234.65, 2);
            b[3] = -0.0036974 - 0.41976*np.exp(-6.1527*np.power((y + 1.8194)/(1.0 + 0.99946*(y + 1.8194)), 2));
        else :
            b[0]=0.0
            b[1]=0.0
            b[2]=0.0
            b[3]=0.0

        b[4] = 1.8108 + 0.18545*np.power(y + 2.95, 2) - 2.0049e-3*np.power(y + 2.95, 4) + 0.85084*np.exp(-14.987*np.power(y + 2.29 - 0.18967*(y + 2.29), 2));
        b[5] = 2.0404 - 0.51548*np.tanh(2.2758*(y + 1.9)) - 0.035009*(y - 6.6555);
        b[6] = 1.5258 + 1.0132*y - 0.064388*y*y - 0.0040209*y*y*y + 0.0082772*y*y*y*y;
        b[7] = 3.0551 + 3.5240*np.tanh(-0.36739*(y + 2.1)) - 0.13382*np.power(y - 2.7718, 2);

    return b
##########################################
def posi_param_delta(Tp):
    y = np.log10(Tp*0.001)
    c = np.zeros(5)
    if ((Tp >= 0.4) and (Tp <= 2)):
        c[0] = 2.9841*np.exp(-67.857*np.power(((y + 3.1272)/(1.0 + 0.22831*(y + 3.1272))), 2)) - (6.5855 + 9.6984/y - 0.41256*y*y);
        c[1] = 6.8276 + 5.2236*y + 1.4630*y*y;
        c[2] = -6.0291 - 6.4581*np.tanh(5.0830*(y + 2.1)) + 0.46352*y;
        c[3] = 0.59300 + 0.36093*y;
        c[4] = 0.77368 + 0.44776*y + 0.056409*y*y;

    return c
##########################################
def posi_param_res(Tp):
    y = np.log10(Tp*0.001)
    d = np.zeros(5)
    if ((Tp >= 0.6) and (Tp < 2.9)):
        d[0] = 1.9186*np.exp(-56.544*np.power(((y + 2.9485)/(1.0 + 1.2892*(y + 2.9485))), 2)) - (0.23720 - 0.041315*y*y);
        d[1] = -4.9866 - 3.1435*y;
        d[2] = -7.0550 - 7.2165*np.tanh(31.033*(y + 2.1)) + 0.38541*y;
        d[3] = -2.8915 - 2.1495*y - 0.45006*y*y;
        d[4] = -1.2970 - 0.13947*y + 0.41197*y*y + 0.10641*y*y*y;
    return d


#Electron Neutrino (TAB-6)
###############################################################################
###############################################################################
def nue_param_nd(Tp):
    y = np.log10(Tp*0.001)
    a = np.zeros(9)
    if ((Tp > 0.4)): # and (Tp < 512000.1)):
        a[0] = 0.0074087 + 2.9161*(y + 3.31) + 0.99061*pow(y + 3.31, 2) - 0.28694*pow(y + 3.31, 3) + 0.038799*pow(y + 3.31, 4);
        a[1] = -3.2480e-5 + 7.1944e-5*np.exp(-0.21814*(y + 3.4)) + 2.0467e-5/(y + 4.1640) + 5.6954e-6*y - 3.4105e-7*y*y;
        a[2] = -230.50 + 58.802*y - 9.9393*y*y + 1.2473*y*y*y - 0.26322*y*y*y*y;
        a[3] = 0.45064 + 0.56930*y + 0.012428*y*y - 0.0070889*y*y*y;
        a[4] = -0.011883 + 1.7992*(y + 3.32) + 3.5264*pow(y + 3.32, 2) - 1.7478*pow(y + 3.32, 3) + 0.32077*pow(y + 3.32, 4) - 0.017667*pow(y + 3.32, 5);
        a[5] = -1.6238e-7 + 1.8116e-6*np.exp(-0.30111*(y + 3.4)) + 9.6112e-5/pow(y + 4.8229, 2);
        a[6] = -261.30 - 43.351*np.log10(0.35298*(y + 3.4)) + 70.925*y - 8.7147*y*y;
        a[7] = 184.45 - 1473.6/(y + 6.8788) - 4.0536*y*y;
        a[8] = -0.24019 + 0.38504*y + 0.0096869*y*y - 0.0015046*y*y*y;
    return a
##########################################
def nue_param_diff(Tp):
    y = np.log10(Tp*0.001)
    b = np.zeros(8)
    if ((Tp > 1.38)): # and (Tp < 512000.1)):
        if Tp > 11.0 :
            b[0] = 53.809*np.tanh(-0.41421*(y + 2.2)) - 6.7538*pow(y + 0.76010, 2) + 0.0088080*pow(y + 8.5075, 4);
            b[1] = -50.211 + 55.131*np.exp(-1.3651*pow((y + 1.8901)/(1.0 + 4.4440*(y + 1.8901)), 2));
            b[2] = -17.231 + 0.041100*np.tanh(7.9638*(y + 1.9)) - 0.055449*y + 0.00025866*pow(y + 250.68, 2);
            b[3] = 12.335 - 12.893*np.exp(-1.4412*pow((y + 1.8998)/(1.0 + 5.5969*(y + 1.8998)), 2));
        else:
            b[0]=0.0
            b[1]=0.0
            b[2]=0.0
            b[3]=0.0
            
        b[4] = 1.3558 + 0.46601*(y + 2.95) + 0.052978*pow(y + 2.2, 2) + 0.79575*np.exp(-5.4007*pow(y + 2.2 + 4.6121*pow(y + 2.2, 2), 2));
        b[5] = 1.8756 - 0.42169*np.tanh(1.6100*(y + 1.9)) - 0.051026*(y - 3.9573);
        b[6] = 1.5016 + 1.0118*y - 0.072787*y*y - 0.0038858*y*y*y + 0.0093650*y*y*y*y;
        b[7] = 4.9735 + 5.5674*np.tanh(-0.36249*(y + 2.1)) - 0.20660*pow(y - 2.8604, 2);
    return b
##########################################
def nue_param_delta(Tp):
    y = np.log10(Tp*0.001)
    c = np.zeros(5)
    if ((Tp >= 0.4) and (Tp <= 2)):
        c[0] = 2.8290*np.exp(-71.339*np.power(((y + 3.1282)/(1.0 + 0.48420*(y + 3.1282))), 2)) - (9.6339 + 15.733/y - 0.52413*y*y)
        c[1] = -24.571 - 15.831*y - 2.1200*y*y
        c[2] = -5.9593 - 6.4695*np.tanh(4.7225*(y + 2.1)) + 0.50003*y
        c[3] = 0.26022 + 0.24545*y
        c[4] = 0.076498 + 0.061678*y + 0.0040028*y*y
    return c
##########################################
def nue_param_res(Tp):
    y = np.log10(Tp*0.001)
    d = np.zeros(5)
    if ((Tp >= 0.6) and (Tp < 2.9)):
        d[0] = 1.7951*np.exp(-57.260*np.power((y + 2.9509)/(1.0 + 1.4101*(y + 2.9509)), 2)) - (0.58604 + 0.23868*y)
        d[1] = -2.6395 - 1.5105*y + 0.22174*y*y
        d[2] = -7.0512 - 7.1970*np.tanh(31.074*(y + 2.1)) + 0.39007*y
        d[3] = -1.4271 - 1.0399*y - 0.24179*y*y
        d[4] = 0.74875 + 0.63616*y + 0.17396*y*y + 0.017636*y*y*y
    return d


#Electron Antineutrino (TAB-7)
###############################################################################
###############################################################################
def anue_param_nd(Tp):
    y = np.log10(Tp*0.001)
    a = np.zeros(9)
    if ((Tp > 0.4)): # and (Tp < 512000.1)):
        a[0] = 0.0013113 + 0.36538*(y + 3.31) + 1.5178*np.power(y + 3.31, 2) - 0.20668*np.power(y + 3.31, 3) + 0.024255*np.power(y + 3.31, 4);
        a[1] = -4.7833e-6 + 4.5837e-5*np.exp(-0.42980*(y + 3.4)) + 6.1559e-6/(y + 4.1731) + 1.1928e-6*y;
        a[2] = -245.22 + 73.223*y - 19.652*y*y + 0.83138*y*y*y + 0.71564*y*y*y*y;
        a[3] = 0.45232 + 0.52934*y + 0.010078*y*y - 0.0017092*y*y*y;
        a[4] = -0.0025734 + 0.38424*(y + 3.32) + 1.5517*np.power(y + 3.32, 2) + 0.17336*np.power(y + 3.32, 3) - 0.17160*np.power(y + 3.32, 4) + 0.021059*np.power(y + 3.32, 5);
        a[5] = 4.7673e-5 + 5.4936e-5*np.log10(0.0067905*(y + 4.3)) + 0.00020740/(y + 4.9772);
        a[6] = -270.30 - 114.47*np.log10(0.34352*(y + 3.4)) + 80.085*y - 7.9240*y*y;
        a[7] = 3271.9 - 2.9161e5/(y + 87.847) - 6.2330*y*y;
        a[8] = -0.17787 + 0.36771*y - 0.025397*y*y + 0.0019238*y*y*y + 0.0032725*y*y*y*y;

    return a
##########################################
def anue_param_diff(Tp):
    y = np.log10(Tp*0.001)
    b = np.zeros(8)
    if ((Tp > 1.38)): # and (Tp < 512000.1)):
        if Tp > 11.05:
            b[0] = 41.307*np.tanh(-0.37411*(y + 2.2)) - 4.1223*np.power(y + 0.55505, 2) + 0.0042652*np.power(y + 9.2685, 4);
            b[1] = -132.50 + 142.12*np.exp(-8.0289*np.power((y + 1.9196)/(1.0 + 11.530*(y + 1.9196)), 2));
            b[2] = -17.223 + 0.011285*np.tanh(69.746*(y + 1.9)) - 0.048233*y + 0.00025881*np.power(y + 250.77, 2);
            b[3] = 8.1991 - 9.6437*np.exp(-45.261*np.power((y + 1.9292)/(1.0 + 16.682*(y + 1.9292)), 2));
        else:
            b[0]=0.0
            b[1]=0.0
            b[2]=0.0
            b[3]=0.0        
            
        b[4] = 0.55919 + 0.36647*(y + 2.95) + 0.056194*np.power(y + 2.95, 2) + 0.49957*np.exp(-5.5317*np.power(y + 2.2 + 0.43867*np.power(y + 2.2, 2), 2));
        b[5] = 1.2544 - 0.52362*np.tanh(2.7638*(y + 1.9)) - 0.055837*(y - 17.638);
        b[6] = 1.4788 + 1.0278*y - 0.092852*y*y - 0.0062734*y*y*y + 0.011920*y*y*y*y;
        b[7] = 5.1651 + 5.7398*np.tanh(-0.37356*(y + 2.1)) - 0.22234*np.power(y - 2.7889, 2);
    return b
##########################################
def anue_param_delta(Tp):
    c = np.zeros(5)
    return c
##########################################
def  anue_param_res(Tp) :
    y = np.log10(Tp*0.001)
    d = np.zeros(5)
    if ((Tp >= 0.6) and (Tp < 2.9)):
        d[0] = 0.36459*np.exp(-58.210*np.power((y + 2.9537)/(1.0 + 1.4320*(y + 2.9537)), 2)) - (0.11283 + 0.046244*y);
        d[1] = -9.5066 - 5.4655*y - 0.31769*y*y;
        d[2] = -7.1831 - 7.1551*np.tanh(30.354*(y + 2.1)) + 0.33757*y;
        d[3] = 2.7938 + 1.6992*y + 0.20161*y*y;
        d[4] = 0.61878 + 0.62371*y + 0.18913*y*y + 0.019118*y*y*y;
    return d



#Muon Neutrino (TAB-8)
###############################################################################
###############################################################################
def numu_param_nd(Tp):
    y = np.log10(Tp*0.001)
    a = np.zeros(9)
    if ((Tp > 0.4)): # and (Tp < 512000.1)):
        a[0] = -0.63611*(y + 3.3) + 9.9015*np.power(y + 3.3, 2) - 4.5897*np.power(y + 3.3, 3) + 0.91778*np.power(y + 3.3, 4) - 0.060724*np.power(y + 3.3, 5);
        a[1] = 6.8700e-6 - 2.8245e-6*y + 7.6032e-7*y*y - 3.2953e-7*y*y*y + 7.4292e-8*y*y*y*y;
        a[2] = -240.46 + 58.405*y - 9.8556*y*y + 3.1401*y*y*y - 0.88932*y*y*y*y;
        a[3] = 0.49935 + 0.60919*y + 0.0024963*y*y - 0.0099910*y*y*y;
        a[4] = 2.5094*(y + 3.32) + 4.1350*np.power(y + 3.32, 2) - 0.89534*np.power(y + 3.32, 3) - 2.7577e-3*np.power(y + 3.32, 4) + 0.014511*np.power(y + 3.32, 5);
        a[5] = 8.2046e-7 + 1.4085e-6*np.log10(0.016793*(y + 4.3)) + 0.00013340/np.power(y + 4.7136, 2);
        a[6] = -267.55 - 0.21018*np.log10(0.35217*(y + 3.4)) + 69.586*y - 9.9930*y*y;
        a[7] = 2741.8 + 222.01*np.log10(9.7401e-13*(y + 3.9)) - 4772.5/(y + 19.773) - 6.1001*y*y;
        a[8] = -0.11857 + 0.39072*y - 0.037813*y*y + 0.0022265*y*y*y + 0.0046931*y*y*y*y;
    return a
##########################################
def numu_param_diff(Tp):
    y = np.log10(Tp*0.001)
    b = np.zeros(8)
    if ((Tp > 1.38)): # and (Tp < 512000.1)):
        if Tp > 11.05:
            b[0] = 64.682*np.tanh(-0.34313*(y + 2.2)) - 5.5955*np.power(y + 0.44754, 2) + 0.0050117*np.power(y + 9.9165, 4);
            b[1] = -7.6016 + 3042.7*np.exp(-1.0134e4*np.power((y + 2.3066)/(1.0 + 41.612*(y + 2.3066)), 2));
            b[2] = -1.4978 - 0.58163*np.tanh(-0.36488*(y + 1.9)) + 0.031825*(y + 2.8097) + 0.022796*np.power(y - 1.8861, 2);
            b[3] = -0.0061483 - 65.799*np.exp(-4.8239*np.power((y + 3.8835)/(1.0 + 0.53343*(y + 3.8835)), 2));
        else:
            b[0]=0.0
            b[1]=0.0
            b[2]=0.0
            b[3]=0.0  
            
        b[4] = 2.8009 + 0.35351*np.power(y + 2.95, 2) - 0.0039779*np.power(y + 2.95, 4) + 1.3012*np.exp(-10.592*np.power(y + 2.28 - 0.19149*np.power(y + 2.28, 2), 2));
        b[5] = 1.8016 - 0.69847*np.tanh(2.8627*(y + 1.9)) - 0.015722*(y - 45.410);
        b[6] = 1.4617 + 1.0167*y - 0.078617*y*y - 0.0038336*y*y*y + 0.010141*y*y*y*y;
        b[7] = 3.5599 + 4.0041*np.tanh(-0.41889*(y + 2.1)) - 0.18182*np.power(y - 2.4209, 2);
    return b
##########################################
def numu_param_delta(Tp):
    y = np.log10(Tp*0.001)
    c = np.zeros(5)
    if ((Tp >= 0.4) and (Tp <= 2)):
        c[0] = 3.6052*np.exp(-60.914*np.power((y + 3.1278)/(1.0 - 0.19497*(y + 3.1278)), 2)) - (0.92514 - 2.1315/y - 0.23548*y*y);
        c[1] = 95.310 + 70.497*y + 13.636*y*y;
        c[2] = -6.2158 - 6.2939*np.tanh(21.592*(y + 2.1)) + 0.37440*y;
        c[3] = 2.7485 + 1.1692*y;
        c[4] = -2.7568 - 1.8461*y - 0.31376*y*y;
    return c
##########################################
def numu_param_res(Tp):
    y = np.log10(Tp*0.001)
    d = np.zeros(5)
    if ((Tp >= 0.6) and (Tp < 2.9)):   
        d[0] = 2.5489*np.exp(-58.488*np.power((y + 2.9509)/(1.0 + 1.3154*(y + 2.9509)), 2)) - (0.83039 + 0.34412*y);
        d[1] = 88.173 + 65.148*y + 12.585*y*y;
        d[2] = -7.0962 - 7.1690*np.tanh(30.890*(y + 2.1)) + 0.38032*y;
        d[3] = -4.1440 - 3.2717*y - 0.70537*y*y;
        d[4] = 2.2624 + 1.1806*y + 0.0043450*y*y - 0.043020*y*y*y;
    return d



#Muon Antineutrino (TAB-9)
###############################################################################
###############################################################################
def anumu_param_nd(Tp):
    y = np.log10(Tp*0.001)
    a = np.zeros(9)
    if ((Tp > 0.4)): #and (Tp < 512000.1)):
        a[0] = -1.5243*(y + 3.3) + 10.107*np.power(y + 3.3, 2) - 4.3126*np.power(y + 3.3, 3) + 0.80081*np.power(y + 3.3, 4) - 0.048724*np.power(y + 3.3, 5);
        a[1] = -2.6297e-5 + 9.3858e-5*np.exp(-0.32384*(y + 3.4)) + 7.7821e-6/(y + 4.0560) + 7.6149e-6*y - 8.4091e-7*y*y;
        a[2] = -243.62 + 59.374*y - 5.7356*y*y + 1.9815*y*y*y - 1.0478*y*y*y*y;
        a[3] = 0.50807 + 0.60221*y + 0.0034120*y*y - 0.011139*y*y*y;
        a[4] = 2.6483*(y + 3.32) + 4.4585*np.power(y + 3.32, 2) - 1.2744*np.power(y + 3.32, 3) + 0.11659*np.power(y + 3.32, 4) + 0.0030477*np.power(y + 3.32, 5);
        a[5] = 9.1101e-7 + 1.3880e-6*np.log10(0.016998*(y + 4.3)) + 0.00012583/np.power(y + 4.7707, 2);
        a[6] = -272.11 + 53.477*np.log10(0.35531*(y + 3.4)) + 56.041*y - 6.0876*y*y;
        a[7] = 6431.8 + 893.92*np.log10(5.7013e-9*(y + 3.9)) + 2103.6/(y + 5.6740) - 6.1125*y*y;
        a[8] = -0.11120 + 0.38144*y - 0.040128*y*y + 0.0047484*y*y*y + 0.0054707*y*y*y*y;
    return a
##########################################
def anumu_param_diff(Tp):
    y = np.log10(Tp*0.001)
    b = np.zeros(8)
    if ((Tp > 1.38)): #and (Tp < 512000.1)):
        if Tp > 11.05:
            b[0] = 70.430*np.tanh(-0.35816*(y + 2.2)) - 6.6796*np.power(y + 0.52273, 2) + 0.0065659*np.power(y + 9.5266, 4);
            b[1] = -8.1145 + 7686.0*np.exp(-44046*np.power((y + 2.2190)/(1.0 + 81.105*(y + 2.2190)), 2));
            b[2] = -1.3095 + 0.071270*np.tanh(0.0075463*(y + 1.9)) + 0.067759*(y + 5.3433) - 0.0044205*np.power(y - 1.8683, 2);
            b[3] = 0.082149 - 2190.1*np.exp(-533.75*np.power((y + 2.8363)/(1.0 + 7.0976*(y + 2.8363)), 2));
        else:
            b[0]=0.0
            b[1]=0.0
            b[2]=0.0
            b[3]=0.0  
            
        b[4] = 2.7540 + 0.33859*np.power(y + 2.95, 2) - 0.0034274*np.power(y + 2.95, 4) + 1.1679*np.exp(-10.408*np.power(y + 2.28 - 0.18922*np.power(y + 2.2, 2), 2));
        b[5] = 2.1817 - 0.59584*np.tanh(2.7054*(y + 1.9)) - 0.010909*(y - 14.880);
        b[6] = 1.4591 + 1.0275*y - 0.074949*y*y - 0.0060396*y*y*y + 0.0097568*y*y*y*y;
        b[7] = 3.7609 + 4.2843*np.tanh(-0.37148*(y + 2.1)) - 0.16479*np.power(y - 2.7653, 2);    
    return b
##########################################
def anumu_param_delta(Tp):
    y = np.log10(Tp*0.001)
    c = np.zeros(5)
    if ((Tp >= 0.4) and (Tp <= 2)):
        c[0] = 2.8262*np.exp(-62.894*np.power((y + 3.1250)/(1.0 - 0.47567*(y + 3.1250)), 2)) + 5.6845 + 13.409/y - 0.097296*y*y;
        c[1] = 16.721 + 11.750*y + 2.4637*y*y;
        c[2] = -6.0557 - 6.3378*np.tanh(21.984*(y + 2.1)) + 0.43173*y;
        c[3] = 0.37009 + 0.27706*y; 
        c[4] = 0.047507 + 0.061570*y + 0.0070117*y*y;
    return c
##########################################
def anumu_param_res(Tp):
    y = np.log10(Tp*0.001)
    d = np.zeros(5)
    if ((Tp >= 0.6) and (Tp < 2.9)):   
        d[0] = 2.2400*np.exp(-57.159*np.power((y + 2.9492)/(1.0 + 1.2994*(y + 2.9492)), 2)) - (0.66521 + 0.27554*y);
        d[1] = -7.0650 - 4.2773*y - 0.17648*y*y;
        d[2] = -7.0410 - 7.1977*np.tanh(31.095*(y + 2.1)) + 0.40238*y;
        d[3] = -1.2354 - 0.87581*y - 0.20829*y*y;
        d[4] = -0.11395 + 0.34418*y + 0.27103*y*y + 0.050248*y*y*y;
    return d


##########################################
##########################################
# Cross sections
##########################################
##########################################

# Non-diffracrive
##########################################
##########################################
def sigma_incl_nond(E, Tp, particle):

    x = np.log10(E)
    y = np.log10(Tp*0.001)
    Lmin = -2.6
    r_factor = 1.0

    if particle == 'gamma':
        Lmax, Wl, Wh, a = 0.96*(y + 3.0), 15, 44, gamma_param_nd(Tp)
        if (Tp <= 1.95):
            r_factor = 3.05*np.exp(-107.0 * pow((y + 3.25)/(1.0 + 8.08*(y + 3.25)), 2))
        else:
            r_factor = 1.01
    elif particle == 'elec':
        Lmax, Wl, Wh, a = 0.96*(y + 3.0), 20, 45, elec_param_nd(Tp)
        if (Tp <= 15.6):
            r_factor = 3.63*np.exp(-106 * pow((y + 3.26)/(1.0 +9.21*(y + 3.26)), 2)) - 0.182*y - 0.175*y*y
        else:
            r_factor = 1.01
    elif particle == 'posi':
        Lmax, Wl, Wh, a = 0.94*(y + 3.0), 15, 47, posi_param_nd(Tp)
        if (Tp <= 5.52):
            r_factor = 2.22*np.exp(-98.9 * pow((y + 3.25)/(1.0 + 1.04*(y + 3.25)), 2))
    elif particle == 'nue':
        Lmax, Wl, Wh, a = 0.98*(y + 3.0), 15, 42, nue_param_nd(Tp)
        if (Tp <= 7.81):
            r_factor = 0.329*np.exp(-249 * pow((y + 3.26)/(1.0 + 6.56*(y + 3.26)), 2)) - 0.957*y - 0.229*y*y
    elif particle == 'anue':
        Lmax, Wl, Wh, a = 0.98*(y + 3.0), 15, 40, anue_param_nd(Tp)
        if (Tp <= 15.6):
            r_factor = 2.67*np.exp(-45.7 * pow((y + 3.27)/(1.0 + 6.59*(y + 3.27)), 2)) - 0.301*y - 0.208*y*y
    elif particle == 'numu':
        Lmax, Wl, Wh, a = 0.94*(y + 3.0), 20, 45, numu_param_nd(Tp)
        if (Tp <= 15.6):
            r_factor = 2.23*np.exp(-93.4 * pow((y + 3.25)/(1.0 + 8.38*(y + 3.25)), 2)) - 0.376*y - 0.121*y*y
    elif particle == 'anumu':
        Lmax, Wl, Wh, a = 0.98*(y + 3.0), 15, 40, anumu_param_nd(Tp)
        if (Tp <= 15.6):
            r_factor = 2.56*np.exp(-107 * pow((y + 3.25)/(1.0 + 8.34*(y + 3.25)), 2)) - 0.385*y - 0.125*y*y
# ============================
    sigma = a[0]*np.exp(-a[1]*pow(x - a[3] + a[2]*pow(x - a[3], 2), 2)) +\
        a[4]*np.exp(-a[5]*pow(x - a[8] + a[6]*pow(x - a[8], 2) +
                              a[7]*pow(x - a[8], 3), 2))
    factor = (1.0/(1.0 + np.exp(Wl*(Lmin -x)))) * (1.0/(1.0 + np.exp(Wh*(x - Lmax))))

    sigma = sigma * factor * r_factor
# ============================
    if (sigma < 0.0):
        sigma = 0.0

    return sigma

# Diffractive
##########################################
##########################################
def sigma_incl_diff(E, Tp, particle):
    x = np.log10(E)
    Lmax = np.log10(Tp)
    Wdiff = 75.0
    if particle == 'gamma':
        b = gamma_param_diff(Tp)
    elif particle == 'elec':
        b = elec_param_diff(Tp)
    elif particle == 'posi':
        b = posi_param_diff(Tp)
    elif particle == 'nue':
        b = nue_param_diff(Tp)
    elif particle == 'anue':
        b = anue_param_diff(Tp)
    elif particle == 'numu':
        b = numu_param_diff(Tp)
    elif particle == 'anumu':
        b = anumu_param_diff(Tp)
# ============================
    sigma = b[0]*np.exp(-b[1]*pow((x - b[2])/(1.0 + b[3]*(x - b[2])), 2)) +\
        b[4]*np.exp(-b[5]*pow((x - b[6])/(1.0 + b[7]*(x - b[6])), 2))
    cutoff = 1.0/(1.0 + np.exp(Wdiff*(x - Lmax)))
    sigma = sigma*cutoff
# ============================
    if (sigma < 0.0):
        sigma = 0.0

    return sigma


# Delta-resonance
##########################################
##########################################
def sigma_incl_delta(E, Tp, particle):
    x = np.log10(E)
    Lmax = np.log10(Tp)
    Wdiff = 75.0
# ============================
    if particle == 'gamma':
        c = gamma_param_delta(Tp)
    elif particle == 'elec':
        c = elec_param_delta(Tp)
    elif particle == 'posi':
        c = posi_param_delta(Tp)
    elif particle == 'nue':
        c = nue_param_delta(Tp)
    elif particle == 'anue':
        c = anue_param_delta(Tp)
    elif particle == 'numu':
        c = numu_param_delta(Tp)
    elif particle == 'anumu':
        c = anumu_param_delta(Tp)
# ============================
    sigma = c[0]*np.exp(-c[1]*pow((x - c[2])/(1.0 + c[3]*(x - c[2]) +
                                              c[4]*pow(x - c[2], 2)), 2))
    cutoff = 1.0/(1.0 + np.exp(Wdiff*(x - Lmax)))
    sigma = sigma*cutoff
    if (sigma < 0.0):
        sigma = 0.0

    return sigma

# 1600-resonance
##########################################
##########################################
def sigma_incl_res(E, Tp, particle):
    x = np.log10(E)
    Lmax = np.log10(Tp)
    Wdiff = 75.0
# ============================
    if particle == 'gamma':
        d = gamma_param_res(Tp)
    elif particle == 'elec':
        d = elec_param_res(Tp)
    elif particle == 'posi':
        d = posi_param_res(Tp)
    elif particle == 'nue':
        d = nue_param_res(Tp)
    elif particle == 'anue':
        d = anue_param_res(Tp)
    elif particle == 'numu':
        d = numu_param_res(Tp)
    elif particle == 'anumu':
        d = anumu_param_res(Tp)
# ============================
    sigma = d[0]*np.exp(-d[1]*pow((x - d[2])/(1.0 + d[3]*(x - d[2]) +
                                              d[4]*pow(x - d[2], 2)), 2))
    cutoff = 1.0/(1.0 + np.exp(Wdiff*(x - Lmax)))
    sigma = sigma*cutoff
    if (sigma < 0.0):
        sigma = 0.0

    return sigma

##########################################
##########################################
##########################################
def dXSdE_gamma_Kamae2006(Tp, E, diffractive):
    particle = 'gamma'
    xx = sigma_incl_nond(E, Tp, particle) +\
        sigma_incl_delta(E, Tp, particle)+sigma_incl_res(E, Tp, particle)
    if diffractive is True:
        xx += sigma_incl_diff(E, Tp, particle)
    xx = np.where(E >= Tp, 0, xx)
    return xx/E
###########################################
def dXSdE_elec_Kamae2006(Tp, E, diffractive):
    particle = 'elec'
    xx = sigma_incl_nond(E, Tp, particle) +\
        sigma_incl_res(E, Tp, particle)
    if diffractive is True:
        xx += sigma_incl_diff(E, Tp, particle)
    xx = np.where(E >= Tp, 0, xx)
    return xx/E
##########################################
def dXSdE_posi_Kamae2006(Tp, E, diffractive):
    particle = 'posi'
    xx = sigma_incl_nond(E, Tp, particle) +\
        sigma_incl_delta(E, Tp, particle)+sigma_incl_res(E, Tp, particle)
    if diffractive is True:
        xx += sigma_incl_diff(E, Tp, particle)
    xx = np.where(E >= Tp, 0, xx)
    return xx/E
##########################################
def dXSdE_elec_nu_Kamae2006(Tp, E, diffractive):
    particle = 'nue'
    xx = sigma_incl_nond(E, Tp, particle) +\
        sigma_incl_delta(E, Tp, particle)+sigma_incl_res(E, Tp, particle)
    if diffractive is True:
        xx += sigma_incl_diff(E, Tp, particle)
    xx = np.where(E >= Tp, 0, xx)
    return xx/E
##########################################
def dXSdE_elec_anti_nu_Kamae2006(Tp, E, diffractive):
    particle = 'anue'
    xx = sigma_incl_nond(E, Tp, particle) +\
        sigma_incl_res(E, Tp, particle)
    if diffractive is True:
        xx += sigma_incl_diff(E, Tp, particle)
    xx = np.where(E >= Tp, 0, xx)
    return xx/E
##########################################
def dXSdE_mu_nu_Kamae2006(Tp, E, diffractive):
    particle = 'numu'
    xx = sigma_incl_nond(E, Tp, particle) +\
        sigma_incl_delta(E, Tp, particle)+sigma_incl_res(E, Tp, particle)
    if diffractive is True:
        xx += sigma_incl_diff(E, Tp, particle)
    xx = np.where(E >= Tp, 0, xx)
    return xx/E
##########################################
def dXSdE_mu_anti_nu_Kamae2006(Tp, E, diffractive):
    particle = 'anumu'
    xx = sigma_incl_nond(E, Tp, particle) +\
        sigma_incl_delta(E, Tp, particle)+sigma_incl_res(E, Tp, particle)
    if diffractive is True:
        xx += sigma_incl_diff(E, Tp, particle)
    xx = np.where(E >= Tp, 0, xx)
    return xx/E