"""Simple chatterbox."""

from typing import List, Union

import openai

from .display import Markdown
from .messaging import Message, ai, deltas, human


class Murkrow:
    """A simple chat class that uses OpenAI's Chat API.

    Messages stream in as they are generated by the API.

    History is tracked and can be used to continue a conversation.

    Args:
        initial_context (str | Message): The initial context for the conversation.


    >>> from murkrow import Murkrow
    >>> murkrow = Murkrow("Hello!")
    >>> murkrow.chat("How are you?")
    Hello!
    How are you?
    >>> murkrow.chat("I'm fine, thanks.")
    Nice to hear!

    """

    def __init__(self, *initial_context: Union[Message, str], model="gpt-3.5-turbo"):
        """Initialize a `Murkrow` object with an optional initial context of messages.

        >>> from murkrow import Murkrow, narrate
        >>> murkrow = Murkrow(narrate("You are a large bird"))
        >>> murkrow.chat("What are you?")
        I am a large bird.

        """
        if initial_context is None:
            initial_context = []

        self.messages: List[Message] = []

        self.append(*initial_context)
        self.model = model

    def chat(self, *messages: Union[Message, str]):
        """Send messages to the chat model and display the response.

        Args:
            messages (str | Message): One or more messages to send to the chat, can be strings or Message objects.

        """
        self.append(*messages)

        mark = Markdown()
        mark.display()

        resp = openai.ChatCompletion.create(
            model=self.model,
            messages=self.messages,
            stream=True,
        )

        mark.extend(deltas(resp))

        self.messages.append(ai(mark.message))

    def append(self, *messages: Union[Message, str]):
        """Append messages to the conversation history.

        Note: this does not send the messages on until `chat` is called.

        Args:
            messages (str | Message): One or more messages to append to the conversation.

        """
        # Messages are either a dict respecting the {role, content} format or a str that we convert to a human message
        for message in messages:
            if isinstance(message, str):
                self.messages.append(human(message))
            else:
                self.messages.append(message)
