"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
function deploy(stack, props = {}) {
    return new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, 'test-stream-firehose-s3', props);
}
test('snapshot test KinesisStreamsToKinesisFirehoseToS3 default params', () => {
    const stack = new cdk.Stack();
    deploy(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test kinesisFirehose override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
test('test kinesisFirehose.deliveryStreamType override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisFirehoseProps: {
            deliveryStreamType: 'DirectPut'
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        DeliveryStreamType: 'KinesisStreamAsSource'
    });
});
test('test kinesisFirehose.kinesisStreamSourceConfiguration override ', () => {
    const stack = new cdk.Stack();
    const kinesisStream = defaults.buildKinesisStream(stack, {
        existingStreamObj: undefined,
        kinesisStreamProps: undefined
    });
    deploy(stack, {
        kinesisFirehoseProps: {
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: kinesisStream.streamArn,
                roleArn: new aws_iam_1.Role(stack, 'myRole', {
                    assumedBy: new aws_iam_1.ServicePrincipal('firehose.amazonaws.com')
                }).roleArn
            }
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        KinesisStreamSourceConfiguration: {
            KinesisStreamARN: {
                "Fn::GetAtt": [
                    "teststreamfirehoses3KinesisStream3165E68E",
                    "Arn"
                ]
            },
            RoleARN: {
                "Fn::GetAtt": [
                    "KinesisStreamsRole2BFD39A5",
                    "Arn"
                ]
            }
        }
    });
});
test('test kinesisStreamProps override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisStreamProps: {
            shardCount: 3
        }
    });
    expect(stack).toHaveResourceLike("AWS::Kinesis::Stream", {
        ShardCount: 3
    });
});
test('Test All properties', () => {
    const stack = new cdk.Stack();
    const construct = deploy(stack);
    expect(construct.cloudwatchAlarms).not.toEqual(undefined);
    expect(construct.kinesisFirehose).not.toEqual(undefined);
    expect(construct.kinesisFirehoseRole).not.toEqual(undefined);
    expect(construct.kinesisFirehoseLogGroup).not.toEqual(undefined);
    expect(construct.kinesisStream).not.toEqual(undefined);
    expect(construct.kinesisStreamRole).not.toEqual(undefined);
    expect(construct.s3Bucket).not.toEqual(undefined);
    expect(construct.s3LoggingBucket).not.toEqual(undefined);
});
test('Test properties with no CW Alarms', () => {
    const stack = new cdk.Stack();
    const construct = deploy(stack, {
        createCloudWatchAlarms: false
    });
    expect(construct.cloudwatchAlarms).toEqual(undefined);
});
test('Test properties with existing S3 bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = defaults.CreateScrapBucket(stack, {});
    const mybucket = s3.Bucket.fromBucketName(stack, 'mybucket', existingBucket.bucketName);
    const construct = deploy(stack, {
        existingBucketObj: mybucket
    });
    expect(construct.s3Bucket).toEqual(undefined);
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
test('Test properties with existing logging S3 bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = defaults.CreateScrapBucket(stack, {});
    const myLoggingBucket = s3.Bucket.fromBucketName(stack, 'myLoggingBucket', existingBucket.bucketName);
    const construct = deploy(stack, {
        existingLoggingBucketObj: myLoggingBucket
    });
    expect(construct.s3Bucket).not.toEqual(undefined);
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5raW5lc2lzZmlyZWhvc2UtczMudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3Qua2luZXNpc2ZpcmVob3NlLXMzLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILDRDQUE2QztBQUM3QyxnQ0FBdUc7QUFDdkcscUNBQXFDO0FBQ3JDLGdDQUE4QjtBQUM5QiwyREFBMkQ7QUFDM0QsOENBQTBEO0FBQzFELHNDQUFzQztBQUV0QyxTQUFTLE1BQU0sQ0FBQyxLQUFnQixFQUFFLFFBQWtELEVBQUU7SUFDcEYsT0FBTyxJQUFJLHlDQUFtQyxDQUFDLEtBQUssRUFBRSx5QkFBeUIsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUMxRixDQUFDO0FBRUQsSUFBSSxDQUFDLGtFQUFrRSxFQUFFLEdBQUcsRUFBRTtJQUM1RSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDZCxNQUFNLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO0FBQy9ELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLEdBQUcsRUFBRTtJQUMxQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLENBQUMsS0FBSyxFQUFFO1FBQ1osb0JBQW9CLEVBQUU7WUFDcEIsa0NBQWtDLEVBQUU7Z0JBQ2xDLGNBQWMsRUFBRTtvQkFDZCxpQkFBaUIsRUFBRSxHQUFHO29CQUN0QixTQUFTLEVBQUUsRUFBRTtpQkFDZDthQUNGO1NBQ0Y7UUFDRCxXQUFXLEVBQUU7WUFDWCxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPO1NBQ3pDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLHNDQUFzQyxFQUFFO1FBQ3ZFLGtDQUFrQyxFQUFFO1lBQ2xDLGNBQWMsRUFBRTtnQkFDZCxpQkFBaUIsRUFBRSxHQUFHO2dCQUN0QixTQUFTLEVBQUUsRUFBRTthQUNkO1NBQ0Y7S0FBQyxDQUFDLENBQUM7QUFDUixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtREFBbUQsRUFBRSxHQUFHLEVBQUU7SUFDN0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxDQUFDLEtBQUssRUFBRTtRQUNaLG9CQUFvQixFQUFFO1lBQ3BCLGtCQUFrQixFQUFFLFdBQVc7U0FDaEM7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsc0NBQXNDLEVBQUU7UUFDdkUsa0JBQWtCLEVBQUUsdUJBQXVCO0tBQzVDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlFQUFpRSxFQUFFLEdBQUcsRUFBRTtJQUMzRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFO1FBQ3ZELGlCQUFpQixFQUFFLFNBQVM7UUFDNUIsa0JBQWtCLEVBQUUsU0FBUztLQUM5QixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxFQUFFO1FBQ1osb0JBQW9CLEVBQUU7WUFDcEIsZ0NBQWdDLEVBQUU7Z0JBQ2hDLGdCQUFnQixFQUFFLGFBQWEsQ0FBQyxTQUFTO2dCQUN6QyxPQUFPLEVBQUUsSUFBSSxjQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtvQkFDakMsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsd0JBQXdCLENBQUM7aUJBQzFELENBQUMsQ0FBQyxPQUFPO2FBQ1g7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxzQ0FBc0MsRUFBRTtRQUN2RSxnQ0FBZ0MsRUFBRTtZQUNoQyxnQkFBZ0IsRUFBRTtnQkFDaEIsWUFBWSxFQUFFO29CQUNaLDJDQUEyQztvQkFDM0MsS0FBSztpQkFDTjthQUNGO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLFlBQVksRUFBRTtvQkFDWiw0QkFBNEI7b0JBQzVCLEtBQUs7aUJBQ047YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbUNBQW1DLEVBQUUsR0FBRyxFQUFFO0lBQzdDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sQ0FBQyxLQUFLLEVBQUU7UUFDWixrQkFBa0IsRUFBRTtZQUNsQixVQUFVLEVBQUUsQ0FBQztTQUNkO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQixFQUFFO1FBQ3ZELFVBQVUsRUFBRSxDQUFDO0tBQ2QsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO0lBQy9CLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sU0FBUyxHQUF3QyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFckUsTUFBTSxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDMUQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ3pELE1BQU0sQ0FBQyxTQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQzdELE1BQU0sQ0FBQyxTQUFTLENBQUMsdUJBQXVCLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ2pFLE1BQU0sQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUN2RCxNQUFNLENBQUMsU0FBUyxDQUFDLGlCQUFpQixDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMzRCxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbEQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBQzNELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtJQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBeUMsTUFBTSxDQUFDLEtBQUssRUFBRTtRQUNwRSxzQkFBc0IsRUFBRSxLQUFLO0tBQzlCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7QUFDeEQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMseUNBQXlDLEVBQUUsR0FBRyxFQUFFO0lBQ25ELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDN0QsTUFBTSxRQUFRLEdBQWUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxjQUFjLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDcEcsTUFBTSxTQUFTLEdBQXdDLE1BQU0sQ0FBQyxLQUFLLEVBQUU7UUFDbkUsaUJBQWlCLEVBQUUsUUFBUTtLQUM1QixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUM5QyxNQUFNLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztBQUN2RCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxpREFBaUQsRUFBRSxHQUFHLEVBQUU7SUFDM0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxjQUFjLEdBQUcsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUM3RCxNQUFNLGVBQWUsR0FBZSxFQUFFLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsY0FBYyxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ2xILE1BQU0sU0FBUyxHQUF5QyxNQUFNLENBQUMsS0FBSyxFQUFFO1FBQ3BFLHdCQUF3QixFQUFFLGVBQWU7S0FDMUMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ2xELE1BQU0sQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBQ3ZELENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLG9EQUFvRDtBQUNwRCxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLG1EQUFtRCxFQUFFLEdBQUcsRUFBRTtJQUM3RCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxVQUFVLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFM0QsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YscUJBQXFCO1FBQ3JCLElBQUkseUNBQW1DLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtZQUM1RCxpQkFBaUIsRUFBRSxVQUFVO1lBQzdCLFdBQVcsRUFBRTtnQkFDWCxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPO2FBQ3pDO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBQ0YsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIxIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgU3ludGhVdGlscyB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgeyBLaW5lc2lzU3RyZWFtc1RvS2luZXNpc0ZpcmVob3NlVG9TMywgS2luZXNpc1N0cmVhbXNUb0tpbmVzaXNGaXJlaG9zZVRvUzNQcm9wcyB9IGZyb20gJy4uL2xpYic7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJ0Bhd3Mtc29sdXRpb25zLWNvbnN0cnVjdHMvY29yZSc7XG5pbXBvcnQgeyBSb2xlLCBTZXJ2aWNlUHJpbmNpcGFsIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuXG5mdW5jdGlvbiBkZXBsb3koc3RhY2s6IGNkay5TdGFjaywgcHJvcHM6IEtpbmVzaXNTdHJlYW1zVG9LaW5lc2lzRmlyZWhvc2VUb1MzUHJvcHMgPSB7fSkge1xuICByZXR1cm4gbmV3IEtpbmVzaXNTdHJlYW1zVG9LaW5lc2lzRmlyZWhvc2VUb1MzKHN0YWNrLCAndGVzdC1zdHJlYW0tZmlyZWhvc2UtczMnLCBwcm9wcyk7XG59XG5cbnRlc3QoJ3NuYXBzaG90IHRlc3QgS2luZXNpc1N0cmVhbXNUb0tpbmVzaXNGaXJlaG9zZVRvUzMgZGVmYXVsdCBwYXJhbXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBkZXBsb3koc3RhY2spO1xuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxudGVzdCgndGVzdCBraW5lc2lzRmlyZWhvc2Ugb3ZlcnJpZGUgJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgZGVwbG95KHN0YWNrLCB7XG4gICAga2luZXNpc0ZpcmVob3NlUHJvcHM6IHtcbiAgICAgIGV4dGVuZGVkUzNEZXN0aW5hdGlvbkNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgYnVmZmVyaW5nSGludHM6IHtcbiAgICAgICAgICBpbnRlcnZhbEluU2Vjb25kczogNjAwLFxuICAgICAgICAgIHNpemVJbk1CczogNTVcbiAgICAgICAgfSxcbiAgICAgIH1cbiAgICB9LFxuICAgIGJ1Y2tldFByb3BzOiB7XG4gICAgICByZW1vdmFsUG9saWN5OiBjZGsuUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OktpbmVzaXNGaXJlaG9zZTo6RGVsaXZlcnlTdHJlYW1cIiwge1xuICAgIEV4dGVuZGVkUzNEZXN0aW5hdGlvbkNvbmZpZ3VyYXRpb246IHtcbiAgICAgIEJ1ZmZlcmluZ0hpbnRzOiB7XG4gICAgICAgIEludGVydmFsSW5TZWNvbmRzOiA2MDAsXG4gICAgICAgIFNpemVJbk1CczogNTVcbiAgICAgIH1cbiAgICB9fSk7XG59KTtcblxudGVzdCgndGVzdCBraW5lc2lzRmlyZWhvc2UuZGVsaXZlcnlTdHJlYW1UeXBlIG92ZXJyaWRlICcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGRlcGxveShzdGFjaywge1xuICAgIGtpbmVzaXNGaXJlaG9zZVByb3BzOiB7XG4gICAgICBkZWxpdmVyeVN0cmVhbVR5cGU6ICdEaXJlY3RQdXQnXG4gICAgfVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6S2luZXNpc0ZpcmVob3NlOjpEZWxpdmVyeVN0cmVhbVwiLCB7XG4gICAgRGVsaXZlcnlTdHJlYW1UeXBlOiAnS2luZXNpc1N0cmVhbUFzU291cmNlJ1xuICB9KTtcbn0pO1xuXG50ZXN0KCd0ZXN0IGtpbmVzaXNGaXJlaG9zZS5raW5lc2lzU3RyZWFtU291cmNlQ29uZmlndXJhdGlvbiBvdmVycmlkZSAnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IGtpbmVzaXNTdHJlYW0gPSBkZWZhdWx0cy5idWlsZEtpbmVzaXNTdHJlYW0oc3RhY2ssIHtcbiAgICBleGlzdGluZ1N0cmVhbU9iajogdW5kZWZpbmVkLFxuICAgIGtpbmVzaXNTdHJlYW1Qcm9wczogdW5kZWZpbmVkXG4gIH0pO1xuXG4gIGRlcGxveShzdGFjaywge1xuICAgIGtpbmVzaXNGaXJlaG9zZVByb3BzOiB7XG4gICAgICBraW5lc2lzU3RyZWFtU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBraW5lc2lzU3RyZWFtQXJuOiBraW5lc2lzU3RyZWFtLnN0cmVhbUFybixcbiAgICAgICAgcm9sZUFybjogbmV3IFJvbGUoc3RhY2ssICdteVJvbGUnLCB7XG4gICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnZmlyZWhvc2UuYW1hem9uYXdzLmNvbScpXG4gICAgICAgIH0pLnJvbGVBcm5cbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpLaW5lc2lzRmlyZWhvc2U6OkRlbGl2ZXJ5U3RyZWFtXCIsIHtcbiAgICBLaW5lc2lzU3RyZWFtU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgS2luZXNpc1N0cmVhbUFSTjoge1xuICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgIFwidGVzdHN0cmVhbWZpcmVob3NlczNLaW5lc2lzU3RyZWFtMzE2NUU2OEVcIixcbiAgICAgICAgICBcIkFyblwiXG4gICAgICAgIF1cbiAgICAgIH0sXG4gICAgICBSb2xlQVJOOiB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgXCJLaW5lc2lzU3RyZWFtc1JvbGUyQkZEMzlBNVwiLFxuICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgXVxuICAgICAgfVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgndGVzdCBraW5lc2lzU3RyZWFtUHJvcHMgb3ZlcnJpZGUgJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBkZXBsb3koc3RhY2ssIHtcbiAgICBraW5lc2lzU3RyZWFtUHJvcHM6IHtcbiAgICAgIHNoYXJkQ291bnQ6IDNcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpLaW5lc2lzOjpTdHJlYW1cIiwge1xuICAgIFNoYXJkQ291bnQ6IDNcbiAgfSk7XG59KTtcblxudGVzdCgnVGVzdCBBbGwgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IGNvbnN0cnVjdDogS2luZXNpc1N0cmVhbXNUb0tpbmVzaXNGaXJlaG9zZVRvUzMgPSBkZXBsb3koc3RhY2spO1xuXG4gIGV4cGVjdChjb25zdHJ1Y3QuY2xvdWR3YXRjaEFsYXJtcykubm90LnRvRXF1YWwodW5kZWZpbmVkKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5raW5lc2lzRmlyZWhvc2UpLm5vdC50b0VxdWFsKHVuZGVmaW5lZCk7XG4gIGV4cGVjdChjb25zdHJ1Y3Qua2luZXNpc0ZpcmVob3NlUm9sZSkubm90LnRvRXF1YWwodW5kZWZpbmVkKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5raW5lc2lzRmlyZWhvc2VMb2dHcm91cCkubm90LnRvRXF1YWwodW5kZWZpbmVkKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5raW5lc2lzU3RyZWFtKS5ub3QudG9FcXVhbCh1bmRlZmluZWQpO1xuICBleHBlY3QoY29uc3RydWN0LmtpbmVzaXNTdHJlYW1Sb2xlKS5ub3QudG9FcXVhbCh1bmRlZmluZWQpO1xuICBleHBlY3QoY29uc3RydWN0LnMzQnVja2V0KS5ub3QudG9FcXVhbCh1bmRlZmluZWQpO1xuICBleHBlY3QoY29uc3RydWN0LnMzTG9nZ2luZ0J1Y2tldCkubm90LnRvRXF1YWwodW5kZWZpbmVkKTtcbn0pO1xuXG50ZXN0KCdUZXN0IHByb3BlcnRpZXMgd2l0aCBubyBDVyBBbGFybXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBjb25zdHJ1Y3Q6IEtpbmVzaXNTdHJlYW1zVG9LaW5lc2lzRmlyZWhvc2VUb1MzICA9IGRlcGxveShzdGFjaywge1xuICAgIGNyZWF0ZUNsb3VkV2F0Y2hBbGFybXM6IGZhbHNlXG4gIH0pO1xuXG4gIGV4cGVjdChjb25zdHJ1Y3QuY2xvdWR3YXRjaEFsYXJtcykudG9FcXVhbCh1bmRlZmluZWQpO1xufSk7XG5cbnRlc3QoJ1Rlc3QgcHJvcGVydGllcyB3aXRoIGV4aXN0aW5nIFMzIGJ1Y2tldCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IGV4aXN0aW5nQnVja2V0ID0gZGVmYXVsdHMuQ3JlYXRlU2NyYXBCdWNrZXQoc3RhY2ssIHt9KTtcbiAgY29uc3QgbXlidWNrZXQ6IHMzLklCdWNrZXQgPSBzMy5CdWNrZXQuZnJvbUJ1Y2tldE5hbWUoc3RhY2ssICdteWJ1Y2tldCcsIGV4aXN0aW5nQnVja2V0LmJ1Y2tldE5hbWUpO1xuICBjb25zdCBjb25zdHJ1Y3Q6IEtpbmVzaXNTdHJlYW1zVG9LaW5lc2lzRmlyZWhvc2VUb1MzID0gZGVwbG95KHN0YWNrLCB7XG4gICAgZXhpc3RpbmdCdWNrZXRPYmo6IG15YnVja2V0XG4gIH0pO1xuXG4gIGV4cGVjdChjb25zdHJ1Y3QuczNCdWNrZXQpLnRvRXF1YWwodW5kZWZpbmVkKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5zM0xvZ2dpbmdCdWNrZXQpLnRvRXF1YWwodW5kZWZpbmVkKTtcbn0pO1xuXG50ZXN0KCdUZXN0IHByb3BlcnRpZXMgd2l0aCBleGlzdGluZyBsb2dnaW5nIFMzIGJ1Y2tldCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IGV4aXN0aW5nQnVja2V0ID0gZGVmYXVsdHMuQ3JlYXRlU2NyYXBCdWNrZXQoc3RhY2ssIHt9KTtcbiAgY29uc3QgbXlMb2dnaW5nQnVja2V0OiBzMy5JQnVja2V0ID0gczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHN0YWNrLCAnbXlMb2dnaW5nQnVja2V0JywgZXhpc3RpbmdCdWNrZXQuYnVja2V0TmFtZSk7XG4gIGNvbnN0IGNvbnN0cnVjdDogS2luZXNpc1N0cmVhbXNUb0tpbmVzaXNGaXJlaG9zZVRvUzMgID0gZGVwbG95KHN0YWNrLCB7XG4gICAgZXhpc3RpbmdMb2dnaW5nQnVja2V0T2JqOiBteUxvZ2dpbmdCdWNrZXRcbiAgfSk7XG5cbiAgZXhwZWN0KGNvbnN0cnVjdC5zM0J1Y2tldCkubm90LnRvRXF1YWwodW5kZWZpbmVkKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5zM0xvZ2dpbmdCdWNrZXQpLnRvRXF1YWwodW5kZWZpbmVkKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBiYWQgY2FsbCB3aXRoIGV4aXN0aW5nQnVja2V0IGFuZCBidWNrZXRQcm9wc1xuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoXCJUZXN0IGJhZCBjYWxsIHdpdGggZXhpc3RpbmdCdWNrZXQgYW5kIGJ1Y2tldFByb3BzXCIsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgdGVzdEJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICd0ZXN0LWJ1Y2tldCcsIHt9KTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgLy8gSGVscGVyIGRlY2xhcmF0aW9uXG4gICAgbmV3IEtpbmVzaXNTdHJlYW1zVG9LaW5lc2lzRmlyZWhvc2VUb1MzKHN0YWNrLCBcImJhZC1zMy1hcmdzXCIsIHtcbiAgICAgIGV4aXN0aW5nQnVja2V0T2JqOiB0ZXN0QnVja2V0LFxuICAgICAgYnVja2V0UHJvcHM6IHtcbiAgICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWVxuICAgICAgfSxcbiAgICB9KTtcbiAgfTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigpO1xufSk7Il19