"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
const default_1 = require("./default");
/**
 * A namespace for kernel types, interfaces, and type checker functions.
 */
var Kernel;
(function (Kernel) {
    /**
     * Find a kernel by id.
     *
     * @param id - The id of the kernel of interest.
     *
     * @param settings - The optional server settings.
     *
     * @returns A promise that resolves with the model for the kernel.
     *
     * #### Notes
     * If the kernel was already started via `startNewKernel`, we return its
     * `Kernel.IModel`. Otherwise, we attempt to find the existing kernel. The
     * promise is fulfilled when the kernel is found, otherwise the promise is
     * rejected.
     */
    function findById(id, settings) {
        return default_1.DefaultKernel.findById(id, settings);
    }
    Kernel.findById = findById;
    /**
     * Fetch all of the kernel specs.
     *
     * @param settings - The optional server settings.
     *
     * @returns A promise that resolves with the kernel specs.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernelspecs).
     */
    function getSpecs(settings) {
        return default_1.DefaultKernel.getSpecs(settings);
    }
    Kernel.getSpecs = getSpecs;
    /**
     * Fetch the running kernels.
     *
     * @param settings - The optional server settings.
     *
     * @returns A promise that resolves with the list of running kernels.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
     *
     * The promise is fulfilled on a valid response and rejected otherwise.
     */
    function listRunning(settings) {
        return default_1.DefaultKernel.listRunning(settings);
    }
    Kernel.listRunning = listRunning;
    /**
     * Start a new kernel.
     *
     * @param options - The options used to create the kernel.
     *
     * @returns A promise that resolves with a kernel object.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
     *
     * If no options are given or the kernel name is not given, the
     * default kernel will by started by the server.
     *
     * Wraps the result in a Kernel object. The promise is fulfilled
     * when the kernel is started by the server, otherwise the promise is rejected.
     */
    function startNew(options = {}) {
        return default_1.DefaultKernel.startNew(options);
    }
    Kernel.startNew = startNew;
    /**
     * Connect to a running kernel.
     *
     * @param model - The model of the running kernel.
     *
     * @param settings - The server settings for the request.
     *
     * @returns The kernel object.
     *
     * #### Notes
     * If the kernel was already started via `startNewKernel`, the existing
     * Kernel object info is used to create another instance.
     */
    function connectTo(model, settings) {
        return default_1.DefaultKernel.connectTo(model, settings);
    }
    Kernel.connectTo = connectTo;
    /**
     * Shut down a kernel by id.
     *
     * @param id - The id of the running kernel.
     *
     * @param settings - The server settings for the request.
     *
     * @returns A promise that resolves when the kernel is shut down.
     */
    function shutdown(id, settings) {
        return default_1.DefaultKernel.shutdown(id, settings);
    }
    Kernel.shutdown = shutdown;
    /**
     * Shut down all kernels.
     *
     * @returns A promise that resolves when all of the kernels are shut down.
     */
    function shutdownAll(settings) {
        return default_1.DefaultKernel.shutdownAll(settings);
    }
    Kernel.shutdownAll = shutdownAll;
})(Kernel = exports.Kernel || (exports.Kernel = {}));
//# sourceMappingURL=kernel.js.map