"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const iam = require("aws-cdk-lib/aws-iam");
const route53 = require("aws-cdk-lib/aws-route53");
const patterns = require("aws-cdk-lib/aws-route53-patterns");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const origin_request_function_1 = require("./origin-request-function");
/**
 * A CloudFront static website hosted on S3
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName: props.domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(this.bucket),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                        functionVersion: new origin_request_function_1.OriginRequestFunction(this, 'OriginRequest'),
                    },
                ],
                responseHeadersPolicy: props.responseHeadersPolicy ?? new cloudfront.ResponseHeadersPolicy(this, 'ResponseHeadersPolicy', {
                    securityHeadersBehavior: StaticWebsite.defaultSecurityHeadersBehavior,
                }),
            },
            defaultRootObject: 'index.html',
            domainNames: [props.domainName],
            certificate,
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2021,
        });
        new route53.ARecord(this, 'ARecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        if (props.backendConfiguration) {
            // Save backend config to bucket, can be queried by the frontend
            new cr.AwsCustomResource(this, 'PutConfig', {
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['s3:PutObject'],
                        resources: [this.bucket.arnForObjects('config.json')],
                    }),
                ]),
                onUpdate: {
                    service: 'S3',
                    action: 'putObject',
                    parameters: {
                        Bucket: this.bucket.bucketName,
                        Key: 'config.json',
                        Body: aws_cdk_lib_1.Stack.of(this).toJsonString(props.backendConfiguration),
                        ContentType: 'application/json',
                        CacheControl: 'max-age=0, no-cache, no-store, must-revalidate',
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('config'),
                },
            });
        }
        if (shouldAddRedirect(props)) {
            const httpsRedirect = new patterns.HttpsRedirect(this, 'HttpsRedirect', {
                targetDomain: props.domainName,
                zone: props.hostedZone,
                recordNames: props.redirects,
            });
            // Force minimum protocol version
            const redirectDistribution = httpsRedirect.node.tryFindChild('RedirectDistribution');
            const cfnDistribution = redirectDistribution.node.tryFindChild('CFDistribution');
            cfnDistribution.addPropertyOverride('DistributionConfig.ViewerCertificate.MinimumProtocolVersion', 'TLSv1.2_2021');
        }
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "cloudstructs.StaticWebsite", version: "0.6.1" };
/**
* Best practice security headers used as default
*/
StaticWebsite.defaultSecurityHeadersBehavior = {
    contentTypeOptions: {
        override: true,
    },
    frameOptions: {
        frameOption: cloudfront.HeadersFrameOption.DENY,
        override: true,
    },
    referrerPolicy: {
        referrerPolicy: cloudfront.HeadersReferrerPolicy.STRICT_ORIGIN_WHEN_CROSS_ORIGIN,
        override: true,
    },
    strictTransportSecurity: {
        accessControlMaxAge: aws_cdk_lib_1.Duration.seconds(31536000),
        includeSubdomains: true,
        preload: true,
        override: true,
    },
    xssProtection: {
        protection: true,
        modeBlock: true,
        override: true,
    },
};
function shouldAddRedirect(props) {
    if (props.redirects && props.redirects.length === 0) {
        return false;
    }
    if (!props.redirects && !aws_cdk_lib_1.Token.isUnresolved(props.domainName)
        && !aws_cdk_lib_1.Token.isUnresolved(props.hostedZone.zoneName)
        && props.domainName === props.hostedZone.zoneName) {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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