"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
/* eslint-disable no-console */
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line import/no-extraneous-dependencies
const signature_1 = require("./signature");
/**
 * Handle Slack events
 */
async function handler(event) {
    console.log('Event: %j', event);
    // Base API gateway response
    const response = {
        statusCode: 200,
        body: '',
    };
    try {
        if (!process.env.SLACK_SIGNING_SECRET)
            throw new Error('The environment variable SLACK_SIGNING_SECRET is not defined');
        if (!event.body)
            throw new Error('Missing body');
        if (!event.headers['X-Slack-Signature'])
            throw new Error('Missing X-Slack-Signature');
        if (!event.headers['X-Slack-Request-Timestamp'])
            throw new Error('Missing X-Slack-Request-Timestamp');
        if (!signature_1.verifyRequestSignature({
            body: event.body,
            requestSignature: event.headers['X-Slack-Signature'],
            requestTimestamp: parseInt(event.headers['X-Slack-Request-Timestamp'], 10),
            signingSecret: process.env.SLACK_SIGNING_SECRET,
        })) {
            response.statusCode = 403;
            return response;
        }
        const body = JSON.parse(event.body);
        console.log('Body: %j', body);
        if (body.type === 'url_verification') { // Slack URL verification, respond with challenge
            console.log('URL verification');
            response.body = JSON.stringify({ challenge: body.challenge });
            return response;
        }
        const eventBridge = new aws_sdk_1.EventBridge({ apiVersion: '2015-10-07' });
        const putEvents = await eventBridge.putEvents({
            Entries: [
                {
                    Detail: event.body,
                    DetailType: 'Slack Event',
                    Source: 'slack',
                    Resources: [body.api_app_id],
                    EventBusName: process.env.EVENT_BUS_NAME,
                    Time: new Date(body.event_time),
                },
            ],
        }).promise();
        console.log('Put events: %j', putEvents);
        return response;
    }
    catch (err) {
        console.log(err);
        response.statusCode = 500;
        return response;
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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