"""
BSD 3-Clause License

Copyright (c) 2019, Andrew Riha
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""

import os
import tempfile

import numpy as np
import pandas as pd

from snps import SNPs
from snps.resources import Resources, ReferenceSequence
from snps.utils import gzip_file
from tests import BaseSNPsTestCase


class TestWriter(BaseSNPsTestCase):
    def test_save_snps(self):
        with tempfile.TemporaryDirectory() as tmpdir:
            snps = SNPs("tests/input/generic.csv", output_dir=tmpdir)
            dest = os.path.join(tmpdir, "generic_GRCh37.txt")
            self.assertEqual(snps.save(), dest)
            self.run_parsing_tests(dest, "generic")

    def test_save_snps_false_positive_build(self):
        with tempfile.TemporaryDirectory() as tmpdir:
            snps = SNPs("tests/input/generic.csv", output_dir=tmpdir)
            output = os.path.join(tmpdir, "generic_GRCh37.txt")
            self.assertEqual(snps.save(), output)

            s = ""
            with open(output, "r") as f:
                for line in f.readlines():
                    if "snps v" in line:
                        s += "# Generated by snps v1.2.3.post85.dev0+gb386302, https://pypi.org/project/snps/\n"
                    else:
                        s += line

            with open(output, "w") as f:
                f.write(s)

            self.run_parsing_tests(output, "generic")

    def test_save_snps_csv(self):
        with tempfile.TemporaryDirectory() as tmpdir:
            snps = SNPs("tests/input/generic.csv", output_dir=tmpdir)
            dest = os.path.join(tmpdir, "generic_GRCh37.csv")
            self.assertEqual(snps.save(sep=","), dest)
            self.run_parsing_tests(dest, "generic")

    def test_save_snps_csv_filename(self):
        with tempfile.TemporaryDirectory() as tmpdir:
            snps = SNPs("tests/input/generic.csv", output_dir=tmpdir)
            dest = os.path.join(tmpdir, "generic.csv")
            self.assertEqual(
                snps.save("generic.csv", sep=","), dest,
            )
            self.run_parsing_tests(dest, "generic")

    def test_save_snps_tsv_filename(self):
        with tempfile.TemporaryDirectory() as tmpdir:
            snps = SNPs("tests/input/generic.tsv", output_dir=tmpdir)
            dest = os.path.join(tmpdir, "generic.tsv")
            self.assertEqual(
                snps.save("generic.tsv", sep="\t"), dest,
            )
            self.run_parsing_tests(dest, "generic")

    def test_save_snps_vcf(self):
        with tempfile.TemporaryDirectory() as tmpdir1:
            s = SNPs("tests/input/testvcf.vcf", output_dir=tmpdir1)

            r = Resources()
            r._reference_sequences["GRCh37"] = {}

            output = os.path.join(tmpdir1, "vcf_GRCh37.vcf")
            with tempfile.TemporaryDirectory() as tmpdir2:
                dest = os.path.join(tmpdir2, "generic.fa.gz")
                gzip_file("tests/input/generic.fa", dest)

                seq = ReferenceSequence(ID="1", path=dest)

                r._reference_sequences["GRCh37"]["1"] = seq

                self.assertEqual(s.save(vcf=True), output)

            self.run_parsing_tests_vcf(output)

    def test_save_snps_vcf_false_positive_build(self):
        with tempfile.TemporaryDirectory() as tmpdir1:
            snps = SNPs("tests/input/testvcf.vcf", output_dir=tmpdir1)

            r = Resources()
            r._reference_sequences["GRCh37"] = {}

            output = os.path.join(tmpdir1, "vcf_GRCh37.vcf")
            with tempfile.TemporaryDirectory() as tmpdir2:
                dest = os.path.join(tmpdir2, "generic.fa.gz")
                gzip_file("tests/input/generic.fa", dest)

                seq = ReferenceSequence(ID="1", path=dest)

                r._reference_sequences["GRCh37"]["1"] = seq

                self.assertEqual(snps.save(vcf=True), output)

                s = ""
                with open(output, "r") as f:
                    for line in f.readlines():
                        if "snps v" in line:
                            s += '##source="vcf; snps v1.2.3.post85.dev0+gb386302; https://pypi.org/project/snps/"\n'
                        else:
                            s += line

                with open(output, "w") as f:
                    f.write(s)

            self.run_parsing_tests_vcf(output)

    def test_save_snps_vcf_discrepant_pos(self):
        with tempfile.TemporaryDirectory() as tmpdir1:
            s = SNPs("tests/input/testvcf.vcf", output_dir=tmpdir1)

            r = Resources()
            r._reference_sequences["GRCh37"] = {}

            output = os.path.join(tmpdir1, "vcf_GRCh37.vcf")
            with tempfile.TemporaryDirectory() as tmpdir2:
                dest = os.path.join(tmpdir2, "generic.fa.gz")
                gzip_file("tests/input/generic.fa", dest)

                seq = ReferenceSequence(ID="1", path=dest)

                r._reference_sequences["GRCh37"]["1"] = seq

                # create discrepant SNPs by setting positions outside reference sequence
                s._snps.loc["rs1", "pos"] = 0
                s._snps.loc["rs17", "pos"] = 118

                # esnure this is the right type after manual tweaking
                s._snps = s._snps.astype({"pos": np.uint32})

                self.assertEqual(s.save(vcf=True), output)

            pd.testing.assert_frame_equal(
                s.discrepant_vcf_position,
                self.create_snp_df(
                    rsid=["rs1", "rs17"],
                    chrom=["1", "1"],
                    pos=[0, 118],
                    genotype=["AA", np.nan],
                ),
                check_exact=True,
            )

            expected = self.generic_snps_vcf().drop(["rs1", "rs17"])
            self.run_parsing_tests_vcf(output, snps_df=expected)

    def test_save_snps_vcf_phased(self):
        with tempfile.TemporaryDirectory() as tmpdir1:
            # read phased data
            s = SNPs("tests/input/testvcf_phased.vcf", output_dir=tmpdir1)

            # setup resource to use test FASTA reference sequence
            r = Resources()
            r._reference_sequences["GRCh37"] = {}

            output = os.path.join(tmpdir1, "vcf_GRCh37.vcf")
            with tempfile.TemporaryDirectory() as tmpdir2:
                dest = os.path.join(tmpdir2, "generic.fa.gz")
                gzip_file("tests/input/generic.fa", dest)

                seq = ReferenceSequence(ID="1", path=dest)

                r._reference_sequences["GRCh37"]["1"] = seq

                # save phased data to VCF
                self.assertEqual(s.save(vcf=True), output)

            # read saved VCF
            self.run_parsing_tests_vcf(output, phased=True)

    def test_save_snps_phased(self):
        with tempfile.TemporaryDirectory() as tmpdir:
            # read phased data
            s = SNPs("tests/input/testvcf_phased.vcf", output_dir=tmpdir)
            dest = os.path.join(tmpdir, "vcf_GRCh37.txt")
            # save phased data to TSV
            self.assertEqual(s.save(), dest)
            # read saved TSV
            self.run_parsing_tests_vcf(dest, phased=True)

    def test_save_snps_specify_file(self):
        with tempfile.TemporaryDirectory() as tmpdir:
            s = SNPs("tests/input/generic.csv", output_dir=tmpdir)
            dest = os.path.join(tmpdir, "snps.csv")
            self.assertEqual(s.save("snps.csv"), dest)
            self.run_parsing_tests(dest, "generic")
