"""
Converts a cache into a TSV with one column per feature, a column for the
target label, and (optionally) a column for score document.

Usage:
    cache2feature_tsv.py <features> <label> [<model>] [<additional-field>]...
                         [--input=<path>] [--output=<path>]
                         [--verbose]
                         [--debug]


Options:
    -h --help               Prints this documentation
    <features>              Classpath to an list of features to use when
                            constructing the model
    <label>                 The name of the target label field
    <model>                 If set, include a column with score documents
                            generated by the model file.
    <additional-field>      Fields from the observations to include in the output
    --input=<path>          The location of the input cache file to read
                            [default: <stdin>]
    --output=<path>         The location of the output TSV file to write
                            [default: <stdout>]
    --verbose               Prints dots and stuff to indicate progress.
    --debug                 Print debug logging.
"""
import logging
import json
import sys
import traceback

import docopt
import mysqltsv
import yamlconf
from revscoring import Model
from revscoring.dependencies import solve
from revscoring.utilities.util import read_observations


def main(argv=None):
    args = docopt.docopt(__doc__, argv=argv)

    logging.basicConfig(
        level=logging.INFO if not args['--debug'] else logging.DEBUG,
        format='%(asctime)s %(levelname)s:%(name)s -- %(message)s'
    )

    sys.path.insert(0, ".")  # Search local directory first
    features = yamlconf.import_module(args['<features>'])
    label_name = args['<label>']
    if args['<model>'] is not None:
        model = Model.load(open(args['<model>']))
    else:
        model = None

    additional_fields = args['<additional-field>']

    if args['--input'] == "<stdin>":
        observations = read_observations(sys.stdin)
    else:
        observations = read_observations(open(args['--input']))

    if args['--output'] == "<stdout>":
        output = sys.stdout
    else:
        output = open(args['--output'], 'w')

    verbose = args['--verbose']

    run(observations, output, features, label_name, model, additional_fields,
        verbose)


def run(observations, output, features, label_name, model, additional_fields,
        verbose):
    headers = [str(f) for f in features] + [label_name]
    if model is not None:
        headers.append("score_doc")
    headers.extend(additional_fields)
    writer = mysqltsv.Writer(output, headers=headers)

    for ob in observations:
        try:
            feature_values = list(solve(features, cache=ob['cache']))
            row = feature_values + [ob[label_name]]
            if model is not None:
                score_doc = model.score(feature_values)
                row += [json.dumps(score_doc)]
            for field_name in additional_fields:
                row += [ob[field_name]]
            writer.write(row)
            if verbose:
                sys.stderr.write(".")
                sys.stderr.flush()
        except:  # noqa: E722
            # Naughty indiscriminate exception consumption.
            sys.stderr.write(traceback.format_exc())

    if verbose:
        sys.stderr.write("\n")


if __name__ == "__main__":
    main(sys.argv[1:])
