# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['einspect',
 'einspect._compat',
 'einspect.protocols',
 'einspect.structs',
 'einspect.structs.include',
 'einspect.views']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions>=4.4.0,<5.0.0']

setup_kwargs = {
    'name': 'einspect',
    'version': '0.5.9',
    'description': 'Extended Inspect - view and modify memory structs of runtime objects.',
    'long_description': '# einspect\n\n<!-- start badges -->\n\n[![Build](https://github.com/ionite34/einspect/actions/workflows/build.yml/badge.svg)](https://github.com/ionite34/einspect/actions/workflows/build.yml)\n[![codecov](https://codecov.io/gh/ionite34/einspect/branch/main/graph/badge.svg?token=v71SdG5Bo6)](https://codecov.io/gh/ionite34/einspect)\n[![security](https://snyk-widget.herokuapp.com/badge/pip/einspect/badge.svg)](https://security.snyk.io/package/pip/einspect)\n\n[![PyPI](https://img.shields.io/pypi/v/einspect)][pypi]\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/einspect)][pypi]\n\n[pypi]: https://pypi.org/project/einspect/\n\n<!-- end badges -->\n\n> Extended Inspections for CPython\n\n## [Documentation](https://docs.ionite.io/einspect)\n\n- [View and modify memory structures of live objects.](#check-detailed-states-of-built-in-objects)\n- [Able to mutate immutable objects like tuples and ints.](#mutate-tuples-strings-ints-or-other-immutable-types)\n- [Modify slot functions or attributes of built-in types.](#modify-attributes-of-built-in-types-get-original-attributes-with-orig)\n- [Fully typed, extensible framework in pure Python.](#move-objects-in-memory)\n\n<!-- start intro -->\n\n## Check detailed states of built-in objects\n```python\nfrom einspect import view\n\nls = [1, 2, 3]\nv = view(ls)\nprint(v.info())\n```\n```python\nPyListObject(at 0x2833738):\n   ob_refcnt: Py_ssize_t = 5\n   ob_type: *PyTypeObject = &[list]\n   ob_item: **PyObject = &[&[1], &[2], &[3]]\n   allocated: Py_ssize_t = 4\n```\n\n[doc_tuple_view]: https://docs.ionite.io/einspect/api/views/view_tuple.html#einspect.views.view_tuple\n[doc_str_view]: https://docs.ionite.io/einspect/api/views/view_str.html#einspect.views.view_str\n[py_doc_mutable_seq]: https://docs.python.org/3/library/stdtypes.html#mutable-sequence-types\n## Mutate tuples, strings, ints, or other immutable types\n> [TupleView][doc_tuple_view] and [StrView][doc_str_view] supports all [MutableSequence][py_doc_mutable_seq] methods (append, extend, insert, pop, remove, reverse, clear).\n\n> ⚠️ A note on [safety.](#safety)\n```python\nfrom einspect import view\n\ntup = (1, 2)\nv = view(tup)\n\nv[1] = 500\nprint(tup)      # (1, 500)\nv.append(3)\nprint(tup)      # (1, 500, 3)\n\ndel v[:2]\nprint(tup)      # (3,)\nprint(v.pop())  # 3\n\nv.extend([1, 2])\nprint(tup)      # (1, 2)\n\nv.clear()\nprint(tup)      # ()\n```\n```python\nfrom einspect import view\n\ntext = "hello"\n\nv = view(text)\nv[1] = "3"\nv[4:] = "o~"\nv.append("!")\n\nprint(text)  # h3llo~!\nv.reverse()\nprint(text)  # !~oll3h\n```\n```python\nfrom einspect import view\n\nn = 500\nview(n).value = 10\n\nprint(500)        # 10\nprint(500 == 10)  # True\n```\n\n## Modify attributes of built-in types, get original attributes with `orig`\n```python\nfrom einspect import view, orig\n\nv = view(int)\nv["__name__"] = "custom_int"\nv["__iter__"] = lambda s: iter(range(s))\nv["__repr__"] = lambda s: "custom: " + orig(int).__repr__(s)\n\nprint(int)\nfor i in 3:\n    print(i)\n```\n```\n<class \'custom_int\'>\ncustom: 0\ncustom: 1\ncustom: 2\n```\n\n## Implement methods on built-in types\n> See the [Extending Types](https://docs.ionite.io/einspect/extending_types.html) docs page for more information.\n```python\nfrom einspect import impl, orig\n\n@impl(int)\ndef __add__(self, other):\n    other = int(other)\n    return orig(int).__add__(self, other)\n\nprint(50 + "25")  # 75\n```\n\n## Move objects in memory\n```python\nfrom einspect import view\n\ns = "meaning of life"\n\nv = view(s)\nwith v.unsafe():\n    v <<= 42\n\nprint("meaning of life")        # 42\nprint("meaning of life" == 42)  # True\n```\n\n<!-- end intro -->\n\n## Fully typed interface\n<img width="551" alt="image" src="https://user-images.githubusercontent.com/13956642/211129165-38a1c405-9d54-413c-962e-6917f1f3c2a1.png">\n\n## Safety\n\n This project is mainly for learning purposes or inspecting and debugging CPython internals for development and fun. You should not violate language conventions like mutability in production software and libraries.\n\nThe interpreter makes assumptions regarding types that are immutable, and changing them causes all those usages to be affected. While the intent of the project is to make a memory-correct mutation without further side effects, there can be very significant runtime implications of mutating interned strings with lots of shared references, including interpreter crashes.\n\nFor example, some strings like "abc" are interned and used by the interpreter. Changing them changes all usages of them, even attribute calls like `collections.abc`.\n\n> The spirit of safety maintained by einspect is to do with memory layouts, not functional effects.\n\n### For example, appending to tuple views (without an unsafe context) will check that the resize can fit within allocated memory\n```python\nfrom einspect import view\n\ntup = (1, 2)\nv = view(tup)\n\nv.append(3)\nprint(tup)  # (1, 2, 3)\n\nv.append(4)\n# UnsafeError: insert required tuple to be resized beyond current memory allocation. Enter an unsafe context to allow this.\n```\n- Despite this, mutating shared references like empty tuples can cause issues in interpreter shutdown and other runtime operations.\n```python\nfrom einspect import view\n\ntup = ()\nview(tup).append(1)\n```\n```\nException ignored in: <module \'threading\' from \'/lib/python3.11/threading.py\'>\nTraceback (most recent call last):\n  File "/lib/python3.11/threading.py", line 1563, in _shutdown\n    _main_thread._stop()\n  File "/lib/python3.11/threading.py", line 1067, in _stop\n    with _shutdown_locks_lock:\nTypeError: \'str\' object cannot be interpreted as an integer\n```\n\n### Similarly, memory moves are also checked for GC-header compatibility and allocation sizes\n```python\nfrom einspect import view\n\nv = view(101)\nv <<= 2\n\nprint(101)  # 2\n\nv <<= "hello"\n# UnsafeError: memory move of 54 bytes into allocated space of 32 bytes is out of bounds. Enter an unsafe context to allow this.\n```\n\n- However, this will not check the fact that small integers between (-5, 256) are interned and used by the interpreter. Changing them may cause issues in any library or interpreter Python code.\n```python\nfrom einspect import view\n\nview(0) << 100\n\nexit()\n# sys:1: ImportWarning: can\'t resolve package from __spec__ or __package__, falling back on __name__ and __path__\n# IndexError: string index out of range\n```\n\n## Table of Contents\n- [Views](#views)\n  - [Using the `einspect.view` constructor](#using-the-einspectview-constructor)\n  - [Inspecting struct attributes](#inspecting-struct-attributes)\n\n## Views\n\n### Using the `einspect.view` constructor\n\nThis is the recommended and simplest way to create a `View` onto an object. Equivalent to constructing a specific `View` subtype from `einspect.views`, except the choice of subtype is automatic based on object type.\n\n```python\nfrom einspect import view\n\nprint(view(1))\nprint(view("hello"))\nprint(view([1, 2]))\nprint(view((1, 2)))\n```\n> ```\n> IntView(<PyLongObject at 0x102058920>)\n> StrView(<PyUnicodeObject at 0x100f12ab0>)\n> ListView(<PyListObject at 0x10124f800>)\n> TupleView(<PyTupleObject at 0x100f19a00>)\n> ```\n\n### Inspecting struct attributes\n\nAttributes of the underlying C Struct of objects can be accessed through the view\'s properties.\n```python\nfrom einspect import view\n\nls = [1, 2]\nv = view(ls)\n\n# Inherited from PyObject\nprint(v.ref_count)  # ob_refcnt\nprint(v.type)       # ob_type\n# Inherited from PyVarObject\nprint(v.size)       # ob_size\n# From PyListObject\nprint(v.item)       # ob_item\nprint(v.allocated)  # allocated\n```\n> ```\n> 4\n> <class \'tuple\'>\n> 3\n> <einspect.structs.c_long_Array_3 object at 0x105038ed0>\n> ```\n\n## 2. Writing to view attributes\n\nWriting to these attributes will affect the underlying object of the view.\n\nNote that most memory-unsafe attribute modifications require entering an unsafe context manager with `View.unsafe()`\n```python\nwith v.unsafe():\n    v.size -= 1\n\nprint(obj)\n```\n> `(1, 2)`\n\nSince `items` is an array of integer pointers to python objects, they can be replaced by `id()` addresses to modify\nindex items in the tuple.\n```python\nfrom einspect import view\n\ntup = (100, 200)\n\nwith view(tup).unsafe() as v:\n    s = "dog"\n    v.item[0] = id(s)\n\nprint(tup)\n```\n> ```\n> (\'dog\', 200)\n>\n> >> Process finished with exit code 139 (interrupted by signal 11: SIGSEGV)\n> ```\n\nSo here we did set the item at index 0 with our new item, the string `"dog"`, but this also caused a segmentation fault.\nNote that the act of setting an item in containers like tuples and lists "steals" a reference to the object, even\nif we only supplied the address pointer.\n\nTo make this safe, we will have to manually increment a ref-count before the new item is assigned. To do this we can\neither create a `view` of our new item, and increment its `ref_count += 1`, or use the apis from `einspect.api`, which\nare pre-typed implementations of `ctypes.pythonapi` methods.\n```python\nfrom einspect import view\nfrom einspect.api import Py\n\ntup = (100, 200)\n\nwith view(tup).unsafe() as v:\n    a = "bird"\n    Py.IncRef(a)\n    v.item[0] = id(a)\n\n    b = "kitten"\n    Py.IncRef(b)\n    v.item[1] = id(b)\n\nprint(tup)\n```\n> `(\'bird\', \'kitten\')`\n\n🎉 No more seg-faults, and we just successfully set both items in an otherwise immutable tuple.\n\nTo make the above routine easier, you can access an abstraction by simply indexing the view.\n\n```python\nfrom einspect import view\n\ntup = ("a", "b", "c")\n\nv = view(tup)\nv[0] = 123\nv[1] = "hm"\nv[2] = "🤔"\n\nprint(tup)\n```\n> `(123, \'hm\', \'🤔\')`\n',
    'author': 'ionite34',
    'author_email': 'dev@ionite.io',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/ionite34/einspect',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.13',
}


setup(**setup_kwargs)
