"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cdk = require("@aws-cdk/core");
const validation_1 = require("./private/validation");
const servicecatalogappregistry_generated_1 = require("./servicecatalogappregistry.generated");
class ApplicationBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedAttributeGroups = new Set();
        this.associatedResources = new Set();
    }
    /**
     * Associate an attribute group with application
     * If the attribute group is already associated, it will ignore duplicate request.
     */
    associateAttributeGroup(attributeGroup) {
        if (!this.associatedAttributeGroups.has(attributeGroup.node.addr)) {
            const hashId = this.generateUniqueHash(attributeGroup.node.addr);
            new servicecatalogappregistry_generated_1.CfnAttributeGroupAssociation(this, `AttributeGroupAssociation${hashId}`, {
                application: this.applicationId,
                attributeGroup: attributeGroup.attributeGroupId,
            });
            this.associatedAttributeGroups.add(attributeGroup.node.addr);
        }
    }
    /**
     * Associate a stack with the application
     * If the resource is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     */
    associateStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            const hashId = this.generateUniqueHash(stack.node.addr);
            new servicecatalogappregistry_generated_1.CfnResourceAssociation(this, `ResourceAssociation${hashId}`, {
                application: this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
        }
    }
}
/**
 * A Service Catalog AppRegistry Application.
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalogappregistry_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        this.validateApplicationProps(props);
        const application = new servicecatalogappregistry_generated_1.CfnApplication(this, 'Resource', {
            name: props.applicationName,
            description: props.description,
        });
        this.applicationArn = application.attrArn;
        this.applicationId = application.attrId;
        this.nodeAddress = cdk.Names.nodeUniqueId(application.node);
    }
    /**
     * Imports an Application construct that represents an external application.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param applicationArn the Amazon Resource Name of the existing AppRegistry Application
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const arn = cdk.Stack.of(scope).splitArn(applicationArn, cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME);
        const applicationId = arn.resourceName;
        if (!applicationId) {
            throw new Error('Missing required Application ID from Application ARN: ' + applicationArn);
        }
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationArn = applicationArn;
                this.applicationId = applicationId;
            }
            generateUniqueHash(resourceAddress) {
                return hashValues(this.applicationArn, resourceAddress);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: applicationArn,
        });
    }
    generateUniqueHash(resourceAddress) {
        return hashValues(this.nodeAddress, resourceAddress);
    }
    validateApplicationProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'application name', 1, 256, props.applicationName);
        validation_1.InputValidator.validateRegex(this.node.path, 'application name', /^[a-zA-Z0-9-_]+$/, props.applicationName);
        validation_1.InputValidator.validateLength(this.node.path, 'application description', 0, 1024, props.description);
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-servicecatalogappregistry.Application", version: "1.163.1" };
/**
 * Generates a unique hash identfifer using SHA256 encryption algorithm
 */
function hashValues(...values) {
    const sha256 = crypto.createHash('sha256');
    values.forEach(val => sha256.update(val));
    return sha256.digest('hex').slice(0, 12);
}
//# sourceMappingURL=data:application/json;base64,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