"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const absent_1 = require("./private/matchers/absent");
const type_1 = require("./private/type");
/**
 * Partial and special matching during template assertions.
 */
class Match {
    /**
     * Use this matcher in the place of a field's value, if the field must not be present.
     */
    static absent() {
        return new absent_1.AbsentMatch('absent');
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     * The set of elements (or matchers) must be in the same order as would be found.
     * @param pattern the pattern to match
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     * The set of elements (or matchers) must match exactly and in order.
     * @param pattern the pattern to match
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * Deep exact matching of the specified pattern to the target.
     * @param pattern the pattern to match
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     * @param pattern the pattern to match
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     * The keys and their values (or matchers) must match exactly with the target.
     * @param pattern the pattern to match
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * Matches any target which does NOT follow the specified pattern.
     * @param pattern the pattern to NOT match
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
    /**
     * Matches any string-encoded JSON and applies the specified pattern after parsing it.
     * @param pattern the pattern to match after parsing the encoded JSON.
     */
    static serializedJson(pattern) {
        return new SerializedJson('serializedJson', pattern);
    }
    /**
     * Matches any non-null value at the target.
     */
    static anyValue() {
        return new AnyMatch('anyValue');
    }
    /**
     * Matches targets according to a regular expression
     */
    static stringLikeRegexp(pattern) {
        return new StringLikeRegexpMatch('stringLikeRegexp', pattern);
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.147.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false, partialObjects: this.partialObjects }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected type ${typeof this.pattern} but received ${type_1.getType(actual)}`,
            });
            return result;
        }
        if (actual !== this.pattern) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected ${this.pattern} but received ${actual}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b, _c;
        super();
        this.name = name;
        this.pattern = pattern;
        this.subsequence = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
        this.partialObjects = (_c = options.partialObjects) !== null && _c !== void 0 ? _c : false;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type array but received ${type_1.getType(actual)}`,
            });
        }
        if (!this.subsequence && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected array of length ${this.pattern.length} but received ${actual.length}`,
            });
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement)
                ? patternElement
                : new LiteralMatch(this.name, patternElement, { partialObjects: this.partialObjects });
            const matcherName = matcher.name;
            if (this.subsequence && (matcherName == 'absent' || matcherName == 'anyValue')) {
                // array subsequence matcher is not compatible with anyValue() or absent() matcher. They don't make sense to be used together.
                throw new Error(`The Matcher ${matcherName}() cannot be nested within arrayWith()`);
            }
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.subsequence || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Missing element${element}at pattern index ${patternIdx}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type object but received ${type_1.getType(actual)}`,
            });
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.recordFailure({
                        matcher: this,
                        path: [`/${a}`],
                        message: 'Unexpected key',
                    });
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (!(patternKey in actual) && !(patternVal instanceof absent_1.AbsentMatch)) {
                result.recordFailure({
                    matcher: this,
                    path: [`/${patternKey}`],
                    message: 'Missing key',
                });
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class SerializedJson extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    ;
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (type_1.getType(actual) !== 'string') {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected JSON as a string but found ${type_1.getType(actual)}`,
            });
            return result;
        }
        let parsed;
        try {
            parsed = JSON.parse(actual);
        }
        catch (err) {
            if (err instanceof SyntaxError) {
                result.recordFailure({
                    matcher: this,
                    path: [],
                    message: `Invalid JSON string: ${actual}`,
                });
                return result;
            }
            else {
                throw err;
            }
        }
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(parsed);
        result.compose(`(${this.name})`, innerResult);
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`,
            });
        }
        return result;
    }
}
class AnyMatch extends matcher_1.Matcher {
    constructor(name) {
        super();
        this.name = name;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: 'Expected a value but found none',
            });
        }
        return result;
    }
}
class StringLikeRegexpMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        const regex = new RegExp(this.pattern, 'gm');
        if (typeof actual !== 'string') {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected a string, but got '${typeof actual}'`,
            });
        }
        if (!regex.test(actual)) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `String '${actual}' did not match pattern '${this.pattern}'`,
            });
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,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