# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['temporalio',
 'temporalio.api',
 'temporalio.api.batch',
 'temporalio.api.batch.v1',
 'temporalio.api.cluster',
 'temporalio.api.cluster.v1',
 'temporalio.api.command',
 'temporalio.api.command.v1',
 'temporalio.api.common',
 'temporalio.api.common.v1',
 'temporalio.api.dependencies',
 'temporalio.api.dependencies.gogoproto',
 'temporalio.api.enums',
 'temporalio.api.enums.v1',
 'temporalio.api.errordetails',
 'temporalio.api.errordetails.v1',
 'temporalio.api.failure',
 'temporalio.api.failure.v1',
 'temporalio.api.filter',
 'temporalio.api.filter.v1',
 'temporalio.api.history',
 'temporalio.api.history.v1',
 'temporalio.api.namespace',
 'temporalio.api.namespace.v1',
 'temporalio.api.operatorservice',
 'temporalio.api.operatorservice.v1',
 'temporalio.api.query',
 'temporalio.api.query.v1',
 'temporalio.api.replication',
 'temporalio.api.replication.v1',
 'temporalio.api.schedule',
 'temporalio.api.schedule.v1',
 'temporalio.api.taskqueue',
 'temporalio.api.taskqueue.v1',
 'temporalio.api.testservice',
 'temporalio.api.testservice.v1',
 'temporalio.api.update',
 'temporalio.api.update.v1',
 'temporalio.api.version',
 'temporalio.api.version.v1',
 'temporalio.api.workflow',
 'temporalio.api.workflow.v1',
 'temporalio.api.workflowservice',
 'temporalio.api.workflowservice.v1',
 'temporalio.bridge',
 'temporalio.bridge.proto',
 'temporalio.bridge.proto.activity_result',
 'temporalio.bridge.proto.activity_task',
 'temporalio.bridge.proto.bridge',
 'temporalio.bridge.proto.child_workflow',
 'temporalio.bridge.proto.common',
 'temporalio.bridge.proto.external_data',
 'temporalio.bridge.proto.health',
 'temporalio.bridge.proto.health.v1',
 'temporalio.bridge.proto.workflow_activation',
 'temporalio.bridge.proto.workflow_commands',
 'temporalio.bridge.proto.workflow_completion',
 'temporalio.contrib',
 'temporalio.testing',
 'temporalio.worker',
 'temporalio.worker.workflow_sandbox']

package_data = \
{'': ['*'],
 'temporalio.bridge': ['sdk-core/*',
                       'sdk-core/.buildkite/*',
                       'sdk-core/.buildkite/docker/*',
                       'sdk-core/.cargo/*',
                       'sdk-core/arch_docs/*',
                       'sdk-core/arch_docs/diagrams/*',
                       'sdk-core/bridge-ffi/*',
                       'sdk-core/bridge-ffi/include/*',
                       'sdk-core/bridge-ffi/src/*',
                       'sdk-core/client/*',
                       'sdk-core/client/src/*',
                       'sdk-core/client/src/workflow_handle/*',
                       'sdk-core/core-api/*',
                       'sdk-core/core-api/src/*',
                       'sdk-core/core/*',
                       'sdk-core/core/benches/*',
                       'sdk-core/core/src/*',
                       'sdk-core/core/src/core_tests/*',
                       'sdk-core/core/src/ephemeral_server/*',
                       'sdk-core/core/src/pollers/*',
                       'sdk-core/core/src/protosext/*',
                       'sdk-core/core/src/replay/*',
                       'sdk-core/core/src/telemetry/*',
                       'sdk-core/core/src/test_help/*',
                       'sdk-core/core/src/worker/*',
                       'sdk-core/core/src/worker/activities/*',
                       'sdk-core/core/src/worker/client/*',
                       'sdk-core/core/src/worker/workflow/*',
                       'sdk-core/core/src/worker/workflow/machines/*',
                       'sdk-core/core/src/worker/workflow/machines/workflow_machines/*',
                       'sdk-core/core/src/worker/workflow/managed_run/*',
                       'sdk-core/etc/*',
                       'sdk-core/fsm/*',
                       'sdk-core/fsm/rustfsm_procmacro/*',
                       'sdk-core/fsm/rustfsm_procmacro/src/*',
                       'sdk-core/fsm/rustfsm_procmacro/tests/*',
                       'sdk-core/fsm/rustfsm_procmacro/tests/trybuild/*',
                       'sdk-core/fsm/rustfsm_trait/*',
                       'sdk-core/fsm/rustfsm_trait/src/*',
                       'sdk-core/fsm/src/*',
                       'sdk-core/histories/*',
                       'sdk-core/protos/api_upstream/*',
                       'sdk-core/protos/api_upstream/.buildkite/*',
                       'sdk-core/protos/api_upstream/.github/*',
                       'sdk-core/protos/api_upstream/.github/workflows/*',
                       'sdk-core/protos/api_upstream/dependencies/gogoproto/*',
                       'sdk-core/protos/api_upstream/temporal/api/batch/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/command/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/common/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/enums/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/errordetails/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/failure/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/filter/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/history/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/namespace/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/operatorservice/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/query/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/replication/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/schedule/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/taskqueue/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/update/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/version/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/workflow/v1/*',
                       'sdk-core/protos/api_upstream/temporal/api/workflowservice/v1/*',
                       'sdk-core/protos/grpc/health/v1/*',
                       'sdk-core/protos/local/temporal/sdk/core/*',
                       'sdk-core/protos/local/temporal/sdk/core/activity_result/*',
                       'sdk-core/protos/local/temporal/sdk/core/activity_task/*',
                       'sdk-core/protos/local/temporal/sdk/core/bridge/*',
                       'sdk-core/protos/local/temporal/sdk/core/child_workflow/*',
                       'sdk-core/protos/local/temporal/sdk/core/common/*',
                       'sdk-core/protos/local/temporal/sdk/core/external_data/*',
                       'sdk-core/protos/local/temporal/sdk/core/workflow_activation/*',
                       'sdk-core/protos/local/temporal/sdk/core/workflow_commands/*',
                       'sdk-core/protos/local/temporal/sdk/core/workflow_completion/*',
                       'sdk-core/protos/testsrv_upstream/*',
                       'sdk-core/protos/testsrv_upstream/dependencies/gogoproto/*',
                       'sdk-core/protos/testsrv_upstream/temporal/api/testservice/v1/*',
                       'sdk-core/sdk-core-protos/*',
                       'sdk-core/sdk-core-protos/src/*',
                       'sdk-core/sdk/*',
                       'sdk-core/sdk/src/*',
                       'sdk-core/sdk/src/workflow_context/*',
                       'sdk-core/test-utils/*',
                       'sdk-core/test-utils/src/*',
                       'sdk-core/tests/*',
                       'sdk-core/tests/integ_tests/*',
                       'sdk-core/tests/integ_tests/workflow_tests/*',
                       'src/*']}

install_requires = \
['protobuf>=4.21,<4.22',
 'types-protobuf>=3.20,<3.21',
 'typing-extensions>=4.2.0,<5.0.0']

extras_require = \
{':python_version < "3.11"': ['python-dateutil>=2.8.2,<3.0.0'],
 'opentelemetry': ['opentelemetry-api>=1.11.1,<2.0.0',
                   'opentelemetry-sdk>=1.11.1,<2.0.0']}

setup_kwargs = {
    'name': 'temporalio',
    'version': '0.1b3',
    'description': 'Temporal.io Python SDK',
    'long_description': '# Temporal Python SDK\n\n[![Python 3.7+](https://img.shields.io/pypi/pyversions/temporalio.svg?style=for-the-badge)](https://pypi.org/project/temporalio)\n[![PyPI](https://img.shields.io/pypi/v/temporalio.svg?style=for-the-badge)](https://pypi.org/project/temporalio)\n[![MIT](https://img.shields.io/pypi/l/temporalio.svg?style=for-the-badge)](LICENSE)\n\n[Temporal](https://temporal.io/) is a distributed, scalable, durable, and highly available orchestration engine used to\nexecute asynchronous long-running business logic in a scalable and resilient way.\n\n"Temporal Python SDK" is the framework for authoring workflows and activities using the Python programming language.\n\nAlso see:\n\n* [Code Samples](https://github.com/temporalio/samples-python)\n* [API Documentation](https://python.temporal.io)\n* [Application Development Guide](https://docs.temporal.io/application-development?lang=python)\n\nIn addition to features common across all Temporal SDKs, the Python SDK also has the following interesting features:\n\n**Type Safe**\n\nThis library uses the latest typing and MyPy support with generics to ensure all calls can be typed. For example,\nstarting a workflow with an `int` parameter when it accepts a `str` parameter would cause MyPy to fail.\n\n**Different Activity Types**\n\nThe activity worker has been developed to work with `async def`, threaded, and multiprocess activities. While\n`async def` activities are the easiest and recommended, care has been taken to make heartbeating and cancellation also\nwork across threads/processes.\n\n**Custom `asyncio` Event Loop**\n\nThe workflow implementation basically turns `async def` functions into workflows backed by a distributed, fault-tolerant\nevent loop. This means task management, sleep, cancellation, etc have all been developed to seamlessly integrate with\n`asyncio` concepts.\n\n**⚠️ UNDER DEVELOPMENT**\n\nThe Python SDK is under development. There are no compatibility guarantees at this time.\n\n## Quick Start\n\n### Installation\n\nInstall the `temporalio` package from [PyPI](https://pypi.org/project/temporalio).\n\nThese steps can be followed to use with a virtual environment and `pip`:\n\n* [Create a virtual environment](https://packaging.python.org/en/latest/tutorials/installing-packages/#creating-virtual-environments)\n* Update `pip` - `python -m pip install -U pip`\n  * Needed because older versions of `pip` may not pick the right wheel\n* Install Temporal SDK - `python -m pip install temporalio`\n\nThe SDK is now ready for use. To build from source, see "Building" near the end of this documentation.\n\n**NOTE: This README is for the current branch and not necessarily what\'s released on `PyPI`.**\n\n### Implementing a Workflow\n\nCreate the following script at `run_worker.py`:\n\n```python\nimport asyncio\nfrom datetime import datetime, timedelta\nfrom temporalio import workflow, activity\nfrom temporalio.client import Client\nfrom temporalio.worker import Worker\n\n@activity.defn\nasync def say_hello(name: str) -> str:\n    return f"Hello, {name}!"\n\n@workflow.defn\nclass SayHello:\n    @workflow.run\n    async def run(self, name: str) -> str:\n        return await workflow.execute_activity(\n            say_hello, name, schedule_to_close_timeout=timedelta(seconds=5)\n        )\n\nasync def main():\n    # Create client connected to server at the given address\n    client = await Client.connect("localhost:7233")\n\n    # Run the worker\n    worker = Worker(client, task_queue="my-task-queue", workflows=[SayHello], activities=[say_hello])\n    await worker.run()\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\nAssuming you have a [Temporal server running on localhost](https://docs.temporal.io/docs/server/quick-install/), this\nwill run the worker:\n\n    python run_worker.py\n\n### Running a Workflow\n\nCreate the following script at `run_workflow.py`:\n\n```python\nimport asyncio\nfrom temporalio.client import Client\n\n# Import the workflow from the previous code\nfrom run_worker import SayHello\n\nasync def main():\n    # Create client connected to server at the given address\n    client = await Client.connect("localhost:7233")\n\n    # Execute a workflow\n    result = await client.execute_workflow(SayHello.run, "my name", id="my-workflow-id", task_queue="my-task-queue")\n\n    print(f"Result: {result}")\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\nAssuming you have `run_worker.py` running from before, this will run the workflow:\n\n    python run_workflow.py\n\nThe output will be:\n\n    Result: Hello, my-name!\n\n## Usage\n\n### Client\n\nA client can be created and used to start a workflow like so:\n\n```python\nfrom temporalio.client import Client\n\nasync def main():\n    # Create client connected to server at the given address and namespace\n    client = await Client.connect("localhost:7233", namespace="my-namespace")\n\n    # Start a workflow\n    handle = await client.start_workflow(MyWorkflow.run, "some arg", id="my-workflow-id", task_queue="my-task-queue")\n\n    # Wait for result\n    result = await handle.result()\n    print(f"Result: {result}")\n```\n\nSome things to note about the above code:\n\n* A `Client` does not have an explicit "close"\n* To enable TLS, the `tls` argument to `connect` can be set to `True` or a `TLSConfig` object\n* A single positional argument can be passed to `start_workflow`. If there are multiple arguments, only the\n  non-type-safe form of `start_workflow` can be used (i.e. the one accepting a string workflow name) and it must be in\n  the `args` keyword argument.\n* The `handle` represents the workflow that was started and can be used for more than just getting the result\n* Since we are just getting the handle and waiting on the result, we could have called `client.execute_workflow` which\n  does the same thing\n* Clients can have many more options not shown here (e.g. data converters and interceptors)\n* A string can be used instead of the method reference to call a workflow by name (e.g. if defined in another language)\n\nClients also provide a shallow copy of their config for use in making slightly different clients backed by the same\nconnection. For instance, given the `client` above, this is how to have a client in another namespace:\n\n```python\nconfig = client.config()\nconfig["namespace"] = "my-other-namespace"\nother_ns_client = Client(**config)\n```\n\n#### Data Conversion\n\nData converters are used to convert raw Temporal payloads to/from actual Python types. A custom data converter of type\n`temporalio.converter.DataConverter` can be set via the `data_converter` client parameter. Data converters are a\ncombination of payload converters, payload codecs, and failure converters. Payload converters convert Python values\nto/from serialized bytes. Payload codecs convert bytes to bytes (e.g. for compression or encryption). Failure converters\nconvert exceptions to/from serialized failures.\n\nThe default data converter supports converting multiple types including:\n\n* `None`\n* `bytes`\n* `google.protobuf.message.Message` - As JSON when encoding, but has ability to decode binary proto from other languages\n* Anything that can be converted to JSON including:\n  * Anything that [`json.dump`](https://docs.python.org/3/library/json.html#json.dump) supports natively\n  * [dataclasses](https://docs.python.org/3/library/dataclasses.html)\n  * Iterables including ones JSON dump may not support by default, e.g. `set`\n  * Any class with a `dict()` method and a static `parse_obj()` method, e.g.\n    [Pydantic models](https://pydantic-docs.helpmanual.io/usage/models)\n    * Note, this doesn\'t mean every Pydantic field can be converted, only fields which the data converter supports\n  * [IntEnum, StrEnum](https://docs.python.org/3/library/enum.html) based enumerates\n  * [UUID](https://docs.python.org/3/library/uuid.html)\n\nThis notably doesn\'t include any `date`, `time`, or `datetime` objects as they may not work across SDKs.\n\nFor converting from JSON, the workflow/activity type hint is taken into account to convert to the proper type. Care has\nbeen taken to support all common typings including `Optional`, `Union`, all forms of iterables and mappings, `NewType`,\netc in addition to the regular JSON values mentioned before.\n\nUsers are strongly encouraged to use a single `dataclass` for parameter and return types so fields with defaults can be\neasily added without breaking compatibility.\n\n### Workers\n\nWorkers host workflows and/or activities. Here\'s how to run a worker:\n\n```python\nimport asyncio\nimport logging\nfrom temporalio.client import Client\nfrom temporalio.worker import Worker\n# Import your own workflows and activities\nfrom my_workflow_package import MyWorkflow, my_activity\n\nasync def run_worker(stop_event: asyncio.Event):\n    # Create client connected to server at the given address\n    client = await Client.connect("localhost:7233", namespace="my-namespace")\n\n    # Run the worker until the event is set\n    worker = Worker(client, task_queue="my-task-queue", workflows=[MyWorkflow], activities=[my_activity])\n    async with worker:\n        await stop_event.wait()\n```\n\nSome things to note about the above code:\n\n* This creates/uses the same client that is used for starting workflows\n* While this example accepts a stop event and uses `async with`, `run()` and `shutdown()` may be used instead\n* Workers can have many more options not shown here (e.g. data converters and interceptors)\n\n### Workflows\n\n#### Definition\n\nWorkflows are defined as classes decorated with `@workflow.defn`. The method invoked for the workflow is decorated with\n`@workflow.run`. Methods for signals and queries are decorated with `@workflow.signal` and `@workflow.query`\nrespectively. Here\'s an example of a workflow:\n\n```python\nimport asyncio\nfrom dataclasses import dataclass\nfrom datetime import timedelta\nfrom temporalio import activity, workflow\nfrom temporalio.client import Client\nfrom temporalio.worker import Worker\n\n@dataclass\nclass GreetingInfo:\n    salutation: str = "Hello"\n    name: str = "<unknown>"\n\n@workflow.defn\nclass GreetingWorkflow:\n    def __init__() -> None:\n        self._current_greeting = "<unset>"\n        self._greeting_info = GreetingInfo()\n        self._greeting_info_update = asyncio.Event()\n        self._complete = asyncio.Event()\n\n    @workflow.run\n    async def run(self, name: str) -> str:\n        self._greeting_info.name = name\n        while True:\n            # Store greeting\n            self._current_greeting = await workflow.execute_activity(\n                create_greeting_activity,\n                self._greeting_info,\n                start_to_close_timeout=timedelta(seconds=5),\n            )\n            workflow.logger.debug("Greeting set to %s", self._current_greeting)\n            \n            # Wait for salutation update or complete signal (this can be\n            # cancelled)\n            await asyncio.wait(\n                [self._greeting_info_update.wait(), self._complete.wait()],\n                return_when=asyncio.FIRST_COMPLETED,\n            )\n            if self._complete.is_set():\n                return self._current_greeting\n            self._greeting_info_update.clear()\n\n    @workflow.signal\n    async def update_salutation(self, salutation: str) -> None:\n        self._greeting_info.salutation = salutation\n        self._greeting_info_update.set()\n\n    @workflow.signal\n    async def complete_with_greeting(self) -> None:\n        self._complete.set()\n\n    @workflow.query\n    async def current_greeting(self) -> str:\n        return self._current_greeting\n\n@activity.defn\nasync def create_greeting_activity(info: GreetingInfo) -> str:\n    return f"{info.salutation}, {info.name}!"\n```\n\nSome things to note about the above code:\n\n* Workflows run in a sandbox by default. Users are encouraged to define workflows in files with no side effects or other\n  complicated code or unnecessary imports to other third party libraries. See the [Workflow Sandbox](#workflow-sandbox)\n  section for more details.\n* This workflow continually updates the queryable current greeting when signalled and can complete with the greeting on\n  a different signal\n* Workflows are always classes and must have a single `@workflow.run` which is an `async def` function\n* Workflow code must be deterministic. This means no threading, no randomness, no external calls to processes, no\n  network IO, and no global state mutation. All code must run in the implicit `asyncio` event loop and be deterministic.\n* `@activity.defn` is explained in a later section. For normal simple string concatenation, this would just be done in\n  the workflow. The activity is for demonstration purposes only.\n* `workflow.execute_activity(create_greeting_activity, ...` is actually a typed signature, and MyPy will fail if the\n  `self._greeting_info` parameter is not a `GreetingInfo`\n\nHere are the decorators that can be applied:\n\n* `@workflow.defn` - Defines a workflow class\n  * Must be defined on the class given to the worker (ignored if present on a base class)\n  * Can have a `name` param to customize the workflow name, otherwise it defaults to the unqualified class name\n* `@workflow.run` - Defines the primary workflow run method\n  * Must be defined on the same class as `@workflow.defn`, not a base class (but can _also_ be defined on the same\n    method of a base class)\n  * Exactly one method name must have this decorator, no more or less\n  * Must be defined on an `async def` method\n  * The method\'s arguments are the workflow\'s arguments\n  * The first parameter must be `self`, followed by positional arguments. Best practice is to only take a single\n    argument that is an object/dataclass of fields that can be added to as needed.\n* `@workflow.signal` - Defines a method as a signal\n  * Can be defined on an `async` or non-`async` function at any hierarchy depth, but if decorated method is overridden,\n    the override must also be decorated\n  * The method\'s arguments are the signal\'s arguments\n  * Can have a `name` param to customize the signal name, otherwise it defaults to the unqualified method name\n  * Can have `dynamic=True` which means all otherwise unhandled signals fall through to this. If present, cannot have\n    `name` argument, and method parameters must be `self`, a string signal name, and a `*args` varargs param.\n  * Non-dynamic method can only have positional arguments. Best practice is to only take a single argument that is an\n    object/dataclass of fields that can be added to as needed.\n  * Return value is ignored\n* `@workflow.query` - Defines a method as a query\n  * All the same constraints as `@workflow.signal` but should return a value\n  * Temporal queries should never mutate anything in the workflow\n\n#### Running\n\nTo start a locally-defined workflow from a client, you can simply reference its method like so:\n\n```python\nfrom temporalio.client import Client\nfrom my_workflow_package import GreetingWorkflow\n\nasync def create_greeting(client: Client) -> str:\n    # Start the workflow\n    handle = await client.start_workflow(GreetingWorkflow.run, "my name", id="my-workflow-id", task_queue="my-task-queue")\n    # Change the salutation\n    await handle.signal(GreetingWorkflow.update_salutation, "Aloha")\n    # Tell it to complete\n    await handle.signal(GreetingWorkflow.complete_with_greeting)\n    # Wait and return result\n    return await handle.result()\n```\n\nSome things to note about the above code:\n\n* This uses the `GreetingWorkflow` from the previous section\n* The result of calling this function is `"Aloha, my name!"`\n* `id` and `task_queue` are required for running a workflow\n* `client.start_workflow` is typed, so MyPy would fail if `"my name"` were something besides a string\n* `handle.signal` is typed, so MyPy would fail if `"Aloha"` were something besides a string or if we provided a\n  parameter to the parameterless `complete_with_greeting`\n* `handle.result` is typed to the workflow itself, so MyPy would fail if we said this `create_greeting` returned\n  something besides a string\n\n#### Invoking Activities\n\n* Activities are started with non-async `workflow.start_activity()` which accepts either an activity function reference\n  or a string name.\n* A single argument to the activity is positional. Multiple arguments are not supported in the type-safe form of\n  start/execute activity and must be supplied via the `args` keyword argument.\n* Activity options are set as keyword arguments after the activity arguments. At least one of `start_to_close_timeout`\n  or `schedule_to_close_timeout` must be provided.\n* The result is an activity handle which is an `asyncio.Task` and supports basic task features\n* An async `workflow.execute_activity()` helper is provided which takes the same arguments as\n  `workflow.start_activity()` and `await`s on the result. This should be used in most cases unless advanced task\n  capabilities are needed.\n* Local activities work very similarly except the functions are `workflow.start_local_activity()` and\n  `workflow.execute_local_activity()`\n* Activities can be methods of a class. Invokers should use `workflow.start_activity_method()`,\n  `workflow.execute_activity_method()`, `workflow.start_local_activity_method()`, and\n  `workflow.execute_local_activity_method()` instead.\n* Activities can callable classes (i.e. that define `__call__`). Invokers should use `workflow.start_activity_class()`,\n  `workflow.execute_activity_class()`, `workflow.start_local_activity_class()`, and\n  `workflow.execute_local_activity_class()` instead.\n\n#### Invoking Child Workflows\n\n* Child workflows are started with async `workflow.start_child_workflow()` which accepts either a workflow run method\n  reference or a string name. The arguments to the workflow are positional.\n* A single argument to the child workflow is positional. Multiple arguments are not supported in the type-safe form of\n  start/execute child workflow and must be supplied via the `args` keyword argument.\n* Child workflow options are set as keyword arguments after the arguments. At least `id` must be provided.\n* The `await` of the start does not complete until the start has been accepted by the server\n* The result is a child workflow handle which is an `asyncio.Task` and supports basic task features. The handle also has\n  some child info and supports signalling the child workflow\n* An async `workflow.execute_child_workflow()` helper is provided which takes the same arguments as\n  `workflow.start_child_workflow()` and `await`s on the result. This should be used in most cases unless advanced task\n  capabilities are needed.\n\n#### Timers\n\n* A timer is represented by normal `asyncio.sleep()`\n* Timers are also implicitly started on any `asyncio` calls with timeouts (e.g. `asyncio.wait_for`)\n* Timers are Temporal server timers, not local ones, so sub-second resolution rarely has value\n\n#### Conditions\n\n* `workflow.wait_condition` is an async function that doesn\'t return until a provided callback returns true\n* A `timeout` can optionally be provided which will throw a `asyncio.TimeoutError` if reached (internally backed by\n  `asyncio.wait_for` which uses a timer)\n\n#### Asyncio and Cancellation\n\nWorkflows are backed by a custom [asyncio](https://docs.python.org/3/library/asyncio.html) event loop. This means many\nof the common `asyncio` calls work as normal. Some asyncio features are disabled such as:\n\n* Thread related calls such as `to_thread()`, `run_coroutine_threadsafe()`, `loop.run_in_executor()`, etc\n* Calls that alter the event loop such as `loop.close()`, `loop.stop()`, `loop.run_forever()`,\n  `loop.set_task_factory()`, etc\n* Calls that use a specific time such as `loop.call_at()`\n* Calls that use anything external such as networking, subprocesses, disk IO, etc\n\nCancellation is done the same way as `asyncio`. Specifically, a task can be requested to be cancelled but does not\nnecessarily have to respect that cancellation immediately. This also means that `asyncio.shield()` can be used to\nprotect against cancellation. The following tasks, when cancelled, perform a Temporal cancellation:\n\n* Activities - when the task executing an activity is cancelled, a cancellation request is sent to the activity\n* Child workflows - when the task starting or executing a child workflow is cancelled, a cancellation request is sent to\n  cancel the child workflow\n* Timers - when the task executing a timer is cancelled (whether started via sleep or timeout), the timer is cancelled\n\nWhen the workflow itself is requested to cancel, `Task.cancel` is called on the main workflow task. Therefore,\n`asyncio.CancelledError` can be caught in order to handle the cancel gracefully.\n\nWorkflows follow `asyncio` cancellation rules exactly which can cause confusion among Python developers. Cancelling a\ntask doesn\'t always cancel the thing it created. For example, given\n`task = asyncio.create_task(workflow.start_child_workflow(...`, calling `task.cancel` does not cancel the child\nworkflow, it only cancels the starting of it, which has no effect if it has already started. However, cancelling the\nresult of `handle = await workflow.start_child_workflow(...` or\n`task = asyncio.create_task(workflow.execute_child_workflow(...` _does_ cancel the child workflow.\n\nAlso, due to Temporal rules, a cancellation request is a state not an event. Therefore, repeated cancellation requests\nare not delivered, only the first. If the workflow chooses swallow a cancellation, it cannot be requested again.\n\n#### Workflow Utilities\n\nWhile running in a workflow, in addition to features documented elsewhere, the following items are available from the\n`temporalio.workflow` package:\n\n* `continue_as_new()` - Async function to stop the workflow immediately and continue as new\n* `info()` - Returns information about the current workflow\n* `logger` - A logger for use in a workflow (properly skips logging on replay)\n* `now()` - Returns the "current time" from the workflow\'s perspective\n\n#### Exceptions\n\n* Workflows can raise exceptions to fail the workflow or the "workflow task" (i.e. suspend the workflow retrying).\n* Exceptions that are instances of `temporalio.exceptions.FailureError` will fail the workflow with that exception\n  * For failing the workflow explicitly with a user exception, use `temporalio.exceptions.ApplicationError`. This can\n    be marked non-retryable or include details as needed.\n  * Other exceptions that come from activity execution, child execution, cancellation, etc are already instances of\n    `FailureError` and will fail the workflow when uncaught.\n* All other exceptions fail the "workflow task" which means the workflow will continually retry until the workflow is\n  fixed. This is helpful for bad code or other non-predictable exceptions. To actually fail the workflow, use an\n  `ApplicationError` as mentioned above.\n\n#### External Workflows\n\n* `workflow.get_external_workflow_handle()` inside a workflow returns a handle to interact with another workflow\n* `workflow.get_external_workflow_handle_for()` can be used instead for a type safe handle\n* `await handle.signal()` can be called on the handle to signal the external workflow\n* `await handle.cancel()` can be called on the handle to send a cancel to the external workflow\n\n#### Testing\n\nWorkflow testing can be done in an integration-test fashion against a real server, however it is hard to simulate\ntimeouts and other long time-based code. Using the time-skipping workflow test environment can help there.\n\nThe time-skipping `temporalio.testing.WorkflowEnvironment` can be created via the static async `start_time_skipping()`.\nThis internally downloads the Temporal time-skipping test server to a temporary directory if it doesn\'t already exist,\nthen starts the test server which has special APIs for skipping time.\n\n##### Automatic Time Skipping\n\nAnytime a workflow result is waited on, the time-skipping server automatically advances to the next event it can. To\nmanually advance time before waiting on the result of a workflow, the `WorkflowEnvironment.sleep` method can be used.\n\nHere\'s a simple example of a workflow that sleeps for 24 hours:\n\n```python\nimport asyncio\nfrom temporalio import workflow\n\n@workflow.defn\nclass WaitADayWorkflow:\n    @workflow.run\n    async def run(self) -> str:\n        await asyncio.sleep(24 * 60 * 60)\n        return "all done"\n```\n\nAn integration test of this workflow would be way too slow. However the time-skipping server automatically skips to the\nnext event when we wait on the result. Here\'s a test for that workflow:\n\n```python\nfrom temporalio.testing import WorkflowEnvironment\nfrom temporalio.worker import Worker\n\nasync def test_wait_a_day_workflow():\n    async with await WorkflowEnvironment.start_time_skipping() as env:\n        async with Worker(env.client, task_queue="tq1", workflows=[WaitADayWorkflow]):\n            assert "all done" == await env.client.execute_workflow(WaitADayWorkflow.run, id="wf1", task_queue="tq1")\n```\n\nThat test will run almost instantly. This is because by calling `execute_workflow` on our client, we have asked the\nenvironment to automatically skip time as much as it can (basically until the end of the workflow or until an activity\nis run).\n\nTo disable automatic time-skipping while waiting for a workflow result, run code inside a\n`with env.auto_time_skipping_disabled():` block.\n\n##### Manual Time Skipping\n\nUntil a workflow is waited on, all time skipping in the time-skipping environment is done manually via\n`WorkflowEnvironment.sleep`.\n\nHere\'s workflow that waits for a signal or times out:\n\n```python\nimport asyncio\nfrom temporalio import workflow\n\n@workflow.defn\nclass SignalWorkflow:\n    def __init__(self) -> None:\n        self.signal_received = False\n\n    @workflow.run\n    async def run(self) -> str:\n        # Wait for signal or timeout in 45 seconds\n        try:\n            await workflow.wait_condition(lambda: self.signal_received, timeout=45)\n            return "got signal"\n        except asyncio.TimeoutError:\n            return "got timeout"\n\n    @workflow.signal\n    def some_signal(self) -> None:\n        self.signal_received = True\n```\n\nTo test a normal signal, you might:\n\n```python\nfrom temporalio.testing import WorkflowEnvironment\nfrom temporalio.worker import Worker\n\nasync def test_signal_workflow():\n    async with await WorkflowEnvironment.start_time_skipping() as env:\n        async with Worker(env.client, task_queue="tq1", workflows=[SignalWorkflow]):\n            # Start workflow, send signal, check result\n            handle = await env.client.start_workflow(SignalWorkflow.run, id="wf1", task_queue="tq1")\n            await handle.signal(SignalWorkflow.some_signal)\n            assert "got signal" == await handle.result()\n```\n\nBut how would you test the timeout part? Like so:\n\n```python\nfrom temporalio.testing import WorkflowEnvironment\nfrom temporalio.worker import Worker\n\nasync def test_signal_workflow_timeout():\n    async with await WorkflowEnvironment.start_time_skipping() as env:\n        async with Worker(env.client, task_queue="tq1", workflows=[SignalWorkflow]):\n            # Start workflow, advance time past timeout, check result\n            handle = await env.client.start_workflow(SignalWorkflow.run, id="wf1", task_queue="tq1")\n            await env.sleep(50)\n            assert "got timeout" == await handle.result()\n```\n\nAlso, the current time of the workflow environment can be obtained via the async `WorkflowEnvironment.get_current_time`\nmethod.\n\n##### Mocking Activities\n\nActivities are just functions decorated with `@activity.defn`. Simply write different ones and pass those to the worker\nto have different activities called during the test.\n\n#### Workflow Sandbox\n\nBy default workflows are run in a sandbox to help avoid non-deterministic code. If a call that is known to be\nnon-deterministic is performed, an exception will be thrown in the workflow which will "fail the task" which means the\nworkflow will not progress until fixed.\n\nThe sandbox is not foolproof and non-determinism can still occur. It is simply a best-effort way to catch bad code\nearly. Users are encouraged to define their workflows in files with no other side effects.\n\n##### How the Sandbox Works\n\nThe sandbox is made up of two components that work closely together:\n\n* Global state isolation\n* Restrictions preventing known non-deterministic library calls\n\nGlobal state isolation is performed by using `exec`. Upon workflow start, the file that the workflow is defined in is\nimported into a new sandbox created for that workflow run. In order to keep the sandbox performant a known set of\n"passthrough modules" are passed through from outside of the sandbox when they are imported. These are expected to be\nside-effect free on import and have their non-deterministic aspects restricted. By default the entire Python standard\nlibrary, `temporalio`, and a couple of other modules are passed through from outside of the sandbox. To update this\nlist, see "Customizing the Sandbox".\n\nRestrictions preventing known non-deterministic library calls are achieved using proxy objects on modules wrapped around\nthe custom importer set in the sandbox. Many restrictions apply at workflow import time and workflow run time, while\nsome restrictions only apply at workflow run time. A default set of restrictions is included that prevents most\ndangerous standard library calls. However it is known in Python that some otherwise-non-deterministic invocations, like\nreading a file from disk via `open` or using `os.environ`, are done as part of importing modules. To customize what is\nand isn\'t restricted, see "Customizing the Sandbox".\n\n##### Avoiding the Sandbox\n\nThere are three increasingly-scoped ways to avoid the sandbox. Users are discouraged from avoiding the sandbox if\npossible.\n\nTo remove restrictions around a particular block of code, use `with temporalio.workflow.unsafe.sandbox_unrestricted():`.\nThe workflow will still be running in the sandbox, but no restrictions for invalid library calls will be applied.\n\nTo run an entire workflow outside of a sandbox, set `sandboxed=False` on the `@workflow.defn` decorator when defining\nit. This will run the entire workflow outside of the workflow which means it can share global state and other bad\nthings.\n\nTo disable the sandbox entirely for a worker, set the `Worker` init\'s `workflow_runner` keyword argument to \n`temporalio.worker.UnsandboxedWorkflowRunner()`. This value is defaulted to\n`temporalio.worker.workflow_sandbox.SandboxedWorkflowRunner()` so by changing it to the unsandboxed runner, the sandbox\nwill not be used at all.\n\n##### Customizing the Sandbox\n\n⚠️ WARNING: APIs in the `temporalio.worker.workflow_sandbox` module are not yet considered stable and may change in\nfuture releases.\n\nWhen creating the `Worker`, the `workflow_runner` is defaulted to\n`temporalio.worker.workflow_sandbox.SandboxedWorkflowRunner()`. The `SandboxedWorkflowRunner`\'s init accepts a\n`restrictions` keyword argument that is defaulted to `SandboxRestrictions.default`. The `SandboxRestrictions` dataclass\nis immutable and contains three fields that can be customized, but only two have notable value\n\n###### Passthrough Modules\n\nTo make the sandbox quicker and use less memory when importing known third party libraries, they can be added to the\n`SandboxRestrictions.passthrough_modules` set like so:\n\n```python\nmy_restrictions = dataclasses.replace(\n    SandboxRestrictions.default,\n    passthrough_modules=SandboxRestrictions.passthrough_modules_default | SandboxMatcher(access={"pydantic"}),\n)\nmy_worker = Worker(..., runner=SandboxedWorkflowRunner(restrictions=my_restrictions))\n```\n\nIf an "access" match succeeds for an import, it will simply be forwarded from outside of the sandbox. See the API for\nmore details on exact fields and their meaning.\n\n###### Invalid Module Members\n\n`SandboxRestrictions.invalid_module_members` contains a root matcher that applies to all module members. This already\nhas a default set which includes things like `datetime.date.today()` which should never be called from a workflow. To\nremove this restriction:\n\n```python\nmy_restrictions = dataclasses.replace(\n    SandboxRestrictions.default,\n    invalid_module_members=SandboxRestrictions.invalid_module_members_default.with_child_unrestricted(\n      "datetime", "date", "today",\n    ),\n)\nmy_worker = Worker(..., runner=SandboxedWorkflowRunner(restrictions=my_restrictions))\n```\n\nRestrictions can also be added by `|`\'ing together matchers, for example to restrict the `datetime.date` class from\nbeing used altogether:\n\n```python\nmy_restrictions = dataclasses.replace(\n    SandboxRestrictions.default,\n    invalid_module_members=SandboxRestrictions.invalid_module_members_default | SandboxMatcher(\n      children={"datetime": SandboxMatcher(use={"date"})},\n    ),\n)\nmy_worker = Worker(..., runner=SandboxedWorkflowRunner(restrictions=my_restrictions))\n```\n\nSee the API for more details on exact fields and their meaning.\n\n##### Known Sandbox Issues\n\nBelow are known sandbox issues. As the sandbox is developed and matures, some may be resolved.\n\n###### Global Import/Builtins\n\nCurrently the sandbox references/alters the global `sys.modules` and `builtins` fields while running workflow code. In\norder to prevent affecting other sandboxed code, thread locals are leveraged to only intercept these values during the\nworkflow thread running. Therefore, technically if top-level import code starts a thread, it may lose sandbox\nprotection.\n\n###### Sandbox is not Secure\n\nThe sandbox is built to catch many non-deterministic and state sharing issues, but it is not secure. Some known bad\ncalls are intercepted, but for performance reasons, every single attribute get/set cannot be checked. Therefore a simple\ncall like `setattr(temporalio.common, "__my_key", "my value")` will leak across sandbox runs.\n\nThe sandbox is only a helper, it does not provide full protection.\n\n###### Sandbox Performance\n\nThe sandbox does not add significant CPU or memory overhead for workflows that are in files which only import standard\nlibrary modules. This is because they are passed through from outside of the sandbox. However, every\nnon-standard-library import that is performed at the top of the same file the workflow is in will add CPU overhead (the\nmodule is re-imported every workflow run) and memory overhead (each module independently cached as part of the workflow\nrun for isolation reasons). This becomes more apparent for large numbers of workflow runs.\n\nTo mitigate this, users should:\n\n* Define workflows in files that have as few non-standard-library imports as possible\n* Alter the max workflow cache and/or max concurrent workflows settings if memory grows too large\n* Set third-party libraries as passthrough modules if they are known to be side-effect free\n\n###### Extending Restricted Classes\n\nCurrently, extending classes marked as restricted causes an issue with their `__init__` parameters. This does not affect\nmost users, but if there is a dependency that is, say, extending `zipfile.ZipFile` an error may occur and the module\nwill have to be marked as pass through.\n\n###### is_subclass of ABC-based Restricted Classes\n\nDue to [https://bugs.python.org/issue44847](https://bugs.python.org/issue44847), classes that are wrapped and then\nchecked to see if they are subclasses of another via `is_subclass` may fail (see also\n[this wrapt issue](https://github.com/GrahamDumpleton/wrapt/issues/130)).\n\n### Activities\n\n#### Definition\n\nActivities are decorated with `@activity.defn` like so:\n\n```python\nfrom temporalio import activity\n\n@activity.defn\nasync def say_hello_activity(name: str) -> str:\n    return f"Hello, {name}!"\n```\n\nSome things to note about activity definitions:\n\n* The `say_hello_activity` is `async` which is the recommended activity type (see "Types of Activities" below)\n* A custom name for the activity can be set with a decorator argument, e.g. `@activity.defn(name="my activity")`\n* Long running activities should regularly heartbeat and handle cancellation\n* Activities can only have positional arguments. Best practice is to only take a single argument that is an\n  object/dataclass of fields that can be added to as needed.\n* Activities can be defined on methods instead of top-level functions. This allows the instance to carry state that an\n  activity may need (e.g. a DB connection). The instance method should be what is registered with the worker.\n* Activities can also be defined on callable classes (i.e. classes with `__call__`). An instance of the class should be\n  what is registered with the worker.\n\n#### Types of Activities\n\nThere are 3 types of activity callables accepted and described below: asynchronous, synchronous multithreaded, and\nsynchronous multiprocess/other. Only positional parameters are allowed in activity callables.\n\n##### Asynchronous Activities\n\nAsynchronous activities, i.e. functions using `async def`, are the recommended activity type. When using asynchronous\nactivities no special worker parameters are needed.\n\nCancellation for asynchronous activities is done via\n[`asyncio.Task.cancel`](https://docs.python.org/3/library/asyncio-task.html#asyncio.Task.cancel). This means that\n`asyncio.CancelledError` will be raised (and can be caught, but it is not recommended). An activity must heartbeat to\nreceive cancellation and there are other ways to be notified about cancellation (see "Activity Context" and\n"Heartbeating and Cancellation" later).\n\n##### Synchronous Activities\n\nSynchronous activities, i.e. functions that do not have `async def`, can be used with workers, but the\n`activity_executor` worker parameter must be set with a `concurrent.futures.Executor` instance to use for executing the\nactivities.\n\nCancellation for synchronous activities is done in the background and the activity must choose to listen for it and\nreact appropriately. If after cancellation is obtained an unwrapped `temporalio.exceptions.CancelledError` is raised,\nthe activity will be marked cancelled. An activity must heartbeat to receive cancellation and there are other ways to be\nnotified about cancellation (see "Activity Context" and "Heartbeating and Cancellation" later).\n\nNote, all calls from an activity to functions in the `temporalio.activity` package are powered by\n[contextvars](https://docs.python.org/3/library/contextvars.html). Therefore, new threads starting _inside_ of\nactivities must `copy_context()` and then `.run()` manually to ensure `temporalio.activity` calls like `heartbeat` still\nfunction in the new threads.\n\n###### Synchronous Multithreaded Activities\n\nIf `activity_executor` is set to an instance of `concurrent.futures.ThreadPoolExecutor` then the synchronous activities\nare considered multithreaded activities. Besides `activity_executor`, no other worker parameters are required for\nsynchronous multithreaded activities.\n\n###### Synchronous Multiprocess/Other Activities\n\nIf `activity_executor` is set to an instance of `concurrent.futures.Executor` that is _not_\n`concurrent.futures.ThreadPoolExecutor`, then the synchronous activities are considered multiprocess/other activities.\n\nThese require special primitives for heartbeating and cancellation. The `shared_state_manager` worker parameter must be\nset to an instance of `temporalio.worker.SharedStateManager`. The most common implementation can be created by passing a\n`multiprocessing.managers.SyncManager` (i.e. result of `multiprocessing.managers.Manager()`) to\n`temporalio.worker.SharedStateManager.create_from_multiprocessing()`.\n\nAlso, all of these activity functions must be\n["picklable"](https://docs.python.org/3/library/pickle.html#what-can-be-pickled-and-unpickled).\n\n#### Activity Context\n\nDuring activity execution, an implicit activity context is set as a\n[context variable](https://docs.python.org/3/library/contextvars.html). The context variable itself is not visible, but\ncalls in the `temporalio.activity` package make use of it. Specifically:\n\n* `in_activity()` - Whether an activity context is present\n* `info()` - Returns the immutable info of the currently running activity\n* `heartbeat(*details)` - Record a heartbeat\n* `is_cancelled()` - Whether a cancellation has been requested on this activity\n* `wait_for_cancelled()` - `async` call to wait for cancellation request\n* `wait_for_cancelled_sync(timeout)` - Synchronous blocking call to wait for cancellation request\n* `is_worker_shutdown()` - Whether the worker has started graceful shutdown\n* `wait_for_worker_shutdown()` - `async` call to wait for start of graceful worker shutdown\n* `wait_for_worker_shutdown_sync(timeout)` - Synchronous blocking call to wait for start of graceful worker shutdown\n* `raise_complete_async()` - Raise an error that this activity will be completed asynchronously (i.e. after return of\n  the activity function in a separate client call)\n\nWith the exception of `in_activity()`, if any of the functions are called outside of an activity context, an error\noccurs. Synchronous activities cannot call any of the `async` functions.\n\n##### Heartbeating and Cancellation\n\nIn order for an activity to be notified of cancellation requests, they must invoke `temporalio.activity.heartbeat()`.\nIt is strongly recommended that all but the fastest executing activities call this function regularly. "Types of\nActivities" has specifics on cancellation for asynchronous and synchronous activities.\n\nIn addition to obtaining cancellation information, heartbeats also support detail data that is persisted on the server\nfor retrieval during activity retry. If an activity calls `temporalio.activity.heartbeat(123, 456)` and then fails and\nis retried, `temporalio.activity.info().heartbeat_details` will return an iterable containing `123` and `456` on the\nnext run.\n\n##### Worker Shutdown\n\nAn activity can react to a worker shutdown. Using `is_worker_shutdown` or one of the `wait_for_worker_shutdown`\nfunctions an activity can react to a shutdown.\n\nWhen the `graceful_shutdown_timeout` worker parameter is given a `datetime.timedelta`, on shutdown the worker will\nnotify activities of the graceful shutdown. Once that timeout has passed (or if wasn\'t set), the worker will perform\ncancellation of all outstanding activities.\n\nThe `shutdown()` invocation will wait on all activities to complete, so if a long-running activity does not at least\nrespect cancellation, the shutdown may never complete.\n\n#### Testing\n\nUnit testing an activity or any code that could run in an activity is done via the\n`temporalio.testing.ActivityEnvironment` class. Simply instantiate this and any callable + params passed to `run` will\nbe invoked inside the activity context. The following are attributes/methods on the environment that can be used to\naffect calls activity code might make to functions on the `temporalio.activity` package.\n\n* `info` property can be set to customize what is returned from `activity.info()`\n* `on_heartbeat` property can be set to handle `activity.heartbeat()` calls\n* `cancel()` can be invoked to simulate a cancellation of the activity\n* `worker_shutdown()` can be invoked to simulate a worker shutdown during execution of the activity\n\n### Workflow Replay\n\nGiven a workflow\'s history, it can be replayed locally to check for things like non-determinism errors. For example,\nassuming `history_json_str` is populated with a JSON string history either exported from the web UI or from `tctl`, the\nfollowing function will replay it:\n\n```python\nfrom temporalio.worker import Replayer\n\nasync def run_replayer(history_json_str: str):\n  replayer = Replayer(workflows=[SayHello])\n  await replayer.replay_workflow(history_json_str)\n```\n\nThis will throw an error if any non-determinism is detected.\n\n### OpenTelemetry Support\n\nOpenTelemetry support requires the optional `opentelemetry` dependencies which are part of the `opentelemetry` extra.\nWhen using `pip`, running\n\n    pip install temporalio[opentelemetry]\n\nwill install needed dependencies. Then the `temporalio.contrib.opentelemetry.TracingInterceptor` can be created and set\nas an interceptor on the `interceptors` argument of `Client.connect`. When set, spans will be created for all client\ncalls and for all activity and workflow invocations on the worker, spans will be created and properly serialized through\nthe server to give one proper trace for a workflow execution.\n\n## Development\n\nThe Python SDK is built to work with Python 3.7 and newer. It is built using\n[SDK Core](https://github.com/temporalio/sdk-core/) which is written in Rust.\n\n### Building\n\n#### Prepare\n\nTo build the SDK from source for use as a dependency, the following prerequisites are required:\n\n* [Python](https://www.python.org/) >= 3.7\n* [Rust](https://www.rust-lang.org/)\n* [poetry](https://github.com/python-poetry/poetry) (e.g. `python -m pip install poetry`)\n* [poe](https://github.com/nat-n/poethepoet) (e.g. `python -m pip install poethepoet`)\n\nmacOS note: If errors are encountered, it may be better to install Python and Rust as recommended from their websites\ninstead of via `brew`.\n\nWith the prerequisites installed, first clone the SDK repository recursively:\n\n```bash\ngit clone --recursive https://github.com/temporalio/sdk-python.git\ncd sdk-python\n```\n\nUse `poetry` to install the dependencies with `--no-root` to not install this package (because we still need to build\nit):\n\n```bash\npoetry install --no-root\n```\n\n#### Build\n\nNow perform the release build:\n\n> This will take a while because Rust will compile the core project in release mode (see [Local SDK development\nenvironment](#local-sdk-development-environment) for the quicker approach to local development).\n\n```bash\npoetry build\n```\n\nThe compiled wheel doesn\'t have the exact right tags yet for use, so run this script to fix it:\n\n```bash\npoe fix-wheel\n```\n\nThe `whl` wheel file in `dist/` is now ready to use.\n\n#### Use\n\nThe wheel can now be installed into any virtual environment.\n\nFor example,\n[create a virtual environment](https://packaging.python.org/en/latest/tutorials/installing-packages/#creating-virtual-environments)\nsomewhere and then run the following inside the virtual environment:\n\n```bash\npip install wheel\n```\n\n```bash\npip install /path/to/cloned/sdk-python/dist/*.whl\n```\n\nCreate this Python file at `example.py`:\n\n```python\nimport asyncio\nfrom temporalio import workflow, activity\nfrom temporalio.client import Client\nfrom temporalio.worker import Worker\n\n@workflow.defn\nclass SayHello:\n    @workflow.run\n    async def run(self, name: str) -> str:\n        return f"Hello, {name}!"\n\nasync def main():\n    client = await Client.connect("localhost:7233")\n    async with Worker(client, task_queue="my-task-queue", workflows=[SayHello]):\n        result = await client.execute_workflow(SayHello.run, "Temporal",\n            id="my-workflow-id", task_queue="my-task-queue")\n        print(f"Result: {result}")\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\nAssuming there is a [local Temporal server](https://docs.temporal.io/docs/server/quick-install/) running, execute the\nfile with `python` (or `python3` if necessary):\n\n```bash\npython example.py\n```\n\nIt should output:\n\n    Result: Hello, Temporal!\n\n### Local SDK development environment\n\nFor local development, it is often quicker to use debug builds and a local virtual environment.\n\nWhile not required, it often helps IDEs if we put the virtual environment `.venv` directory in the project itself. This\ncan be configured system-wide via:\n\n```bash\npoetry config virtualenvs.in-project true\n```\n\nNow perform the same steps as the "Prepare" section above by installing the prerequisites, cloning the project,\ninstalling dependencies, and generating the protobuf code:\n\n```bash\ngit clone --recursive https://github.com/temporalio/sdk-python.git\ncd sdk-python\npoetry install --no-root\n```\n\nNow compile the Rust extension in develop mode which is quicker than release mode:\n\n```bash\npoe build-develop\n```\n\nThat step can be repeated for any Rust changes made.\n\nThe environment is now ready to develop in.\n\n#### Testing\n\nTo execute tests:\n\n```bash\npoe test\n```\n\nThis runs against [Temporalite](https://github.com/temporalio/temporalite). To run against the time-skipping test\nserver, pass `--workflow-environment time-skipping`. To run against the `default` namespace of an already-running\nserver, pass the `host:port` to `--workflow-environment`.\n\n### Style\n\n* Mostly [Google Style Guide](https://google.github.io/styleguide/pyguide.html). Notable exceptions:\n  * We use [Black](https://github.com/psf/black) for formatting, so that takes precedence\n  * In tests and example code, can import individual classes/functions to make it more readable. Can also do this for\n    rarely in library code for some Python common items (e.g. `dataclass` or `partial`), but not allowed to do this for\n    any `temporalio` packages (except `temporalio.types`) or any classes/functions that aren\'t clear when unqualified.\n  * We allow relative imports for private packages\n  * We allow `@staticmethod`\n',
    'author': 'Temporal Technologies Inc',
    'author_email': 'sdk@temporal.io',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/temporalio/sdk-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
