# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['numbers_parser', 'numbers_parser.generated']

package_data = \
{'': ['*'], 'numbers_parser': ['data/*']}

install_requires = \
['compact-json>=1.1.3,<2.0.0',
 'pendulum>=2.1.2,<3.0.0',
 'protobuf>=4.21.1,<5.0.0',
 'python-magic>=0.4.27,<0.5.0',
 'python-snappy>=0.6.1,<0.7.0',
 'regex>=2022.9.13,<2023.0.0',
 'roman>=3.3,<4.0']

entry_points = \
{'console_scripts': ['cat-numbers = numbers_parser._cat_numbers:main',
                     'unpack-numbers = numbers_parser._unpack_numbers:main']}

setup_kwargs = {
    'name': 'numbers-parser',
    'version': '3.9.3',
    'description': 'Read and write Apple Numbers spreadsheets',
    'long_description': '# numbers-parser\n\n[![build:](https://github.com/masaccio/numbers-parser/actions/workflows/run-all-tests.yml/badge.svg)](https://github.com/masaccio/numbers-parser/actions/workflows/run-all-tests.yml)\n[![build:](https://github.com/masaccio/numbers-parser/actions/workflows/codeql.yml/badge.svg)](https://github.com/masaccio/numbers-parser/actions/workflows/codeql.yml)\n[![codecov](https://codecov.io/gh/masaccio/numbers-parser/branch/main/graph/badge.svg?token=EKIUFGT05E)](https://codecov.io/gh/masaccio/numbers-parser)\n\n`numbers-parser` is a Python module for parsing [Apple Numbers](https://www.apple.com/numbers/)\n`.numbers` files. It supports Numbers files generated by Numbers version 10.3, and up with the latest tested version being 12.1\n(current as of June 2022).\n\nIt supports and is tested against Python versions from 3.8 onwards. It is not compatible with earlier versions of Python.\n\nCurrently supported features of Numbers files are:\n\n* Multiple sheets per document\n* Multiple tables per sheet\n* Text, numeric, date, currency, duration, percentage cell types\n\nFormulas rely on Numbers storing current values which should usually be\nthe case. Formulas themselves rather than the computed values can optionally\nbe extracted. Styles are not supported.\n\nAs of version 3.0, `numbers-parser` has limited support for creating Numbers files.\n\n## Installation\n\n``` bash\npython3 -m pip install numbers-parser\n```\n\nA pre-requisite for this package is [python-snappy](https://pypi.org/project/python-snappy/) which will be installed by Python automatically, but python-snappy also requires that the binary libraries for snappy compression are present. Since version 3.9.0, `numbers-parser` also has a dependency on `libmagic` through [python-magic](https://pypi.org/project/python-magic/).\n\nThe most straightforward way to install the binary dependencies is to use [Homebrew](https://brew.sh) and source Python from Homebrew rather than from macOS as described in the [python-snappy github](https://github.com/andrix/python-snappy):\n\nFor Intel Macs:\n\n``` bash\nbrew install snappy libmagic python3\nCPPFLAGS="-I/usr/local/include -L/usr/local/lib" python3 -m pip install python-snappy\n```\n\nAnd on Apple Silicon:\n\n``` bash\nbrew install snappy libmagic python3\nCPPFLAGS="-I/opt/homebrew/include -L/opt/homebrew/lib" python3 -m pip install python-snappy\n```\n\n## Usage\n\nReading documents:\n\n``` python\nfrom numbers_parser import Document\ndoc = Document("my-spreasdsheet.numbers")\nsheets = doc.sheets\ntables = sheets[0].tables\nrows = tables[0].rows()\n```\n\n### Referring to sheets and tables\n\nBoth sheets and names can be accessed from lists of these objects using an integer index (`list` syntax) and using the name\nof the sheet/table (`dict` syntax):\n\n``` python\n# list access method\nsheet_1 = doc.sheets[0]\nprint("Opened sheet", sheet_1.name)\n\n# dict access method\ntable_1 = sheets["Table 1"]\nprint("Opened table", table_1.name)\n```\n\nAs of version 3.0, the `Document.sheets()` and `Sheet.tables()` methods are deprecated and will issue a `DeprecationWarning` if used. Instead of these functions, you should use the properties as demonstrated above. The legacy methods will be removed in a future version of `numbers-parser`.\n\n### Accessing data\n\n`Table` objects have a `rows` method which contains a nested list with an entry for each row of the table. Each row is\nitself a list of the column values. Empty cells in Numbers are returned as `None` values.\n\n``` python\ndata = sheets["Table 1"].rows()\nprint("Cell A1 contains", data[0][0])\nprint("Cell C2 contains", data[2][1])\n```\n\nCells are objects with a common base class of `Cell`. All cell types have a property `value` which returns the contents of the cell in as a native Python datatype. `DurationCell` object values are `datetime.timedelta` objects which are additionally available as a formatted value matching that stored in the Numbers spreadsheet. The formatted value is returned using the `formatted_value` property.\n\n### Cell references\n\nIn addition to extracting all data at once, individual cells can be referred to as methods\n\n``` python\ndoc = Document("my-spreasdsheet.numbers")\nsheets = doc.sheets\ntables = sheets["Sheet 1"].tables\ntable = tables["Table 1"]\n\n# row, column syntax\nprint("Cell A1 contains", table.cell(0, 0))\n# Excel/Numbers-style cell references\nprint("Cell C2 contains", table.cell("C2"))\n```\n\n### Merged cells\n\nWhen extracting data using ```rows()``` merged cells are ignored since only text values\nare returned. The ```cell()``` method of ```Table``` objects returns a ```Cell``` type\nobject which is typed by the type of cell in the Numbers table. ```MergeCell``` objects\nindicates cells removed in a merge.\n\n``` python\ndoc = Document("my-spreasdsheet.numbers")\nsheets = doc.sheets\ntables = sheets["Sheet 1"].tables\ntable = tables["Table 1"]\n\ncell = table.cell("A1")\nprint(cell.merge_range)\nprint(f"Cell A1 merge size is {cell.size[0]},{cell.size[1]})\n```\n\n### Row and column iterators\n\nTables have iterators for row-wise and column-wise iteration with each iterator\nreturning a list of the cells in that row or column\n\n``` python\nfor row in table.iter_rows(min_row=2, max_row=7, values_only=True):\n    sum += row\nfor col in table.iter_cols(min_row=2, max_row=7):\n    sum += col.value\n```\n\n### Pandas\n\nSince the return value of `data()` is a list of lists, you can pass this directly to pandas. Assuming you have a Numbers table with a single header which contains the names of the pandas series you want to create you can construct a pandas dataframe using:\n\n``` python\nimport pandas as pd\n\ndoc = Document("simple.numbers")\nsheets = doc.sheets\ntables = sheets[0].tables\ndata = tables[0].rows(values_only=True)\ndf = pd.DataFrame(data[1:], columns=data[0])\n```\n\n### Bullets and lists\n\nCells that contain bulleted or numbered lists can be identified by the `is_bulleted` property. Data from such cells is returned using the `value` property as with other cells, but can additionally extracted using the `bullets` property. `bullets` returns a list of the paragraphs in the cell without the bullet or numbering character. Newlines are not included when bullet lists are extracted using `bullets`.\n\n``` python\ndoc = Document("bullets.numbers")\nsheets = doc.sheets\ntables = sheets[0].tables\ntable = tables[0]\nif not table.cell(0, 1).is_bulleted:\n    print(table.cell(0, 1).value)\nelse:\n    bullets = ["* " + s for s in table.cell(0, 1).bullets]\n    print("\\n".join(bullets))\n```\n\nBulleted and numbered data can also be extracted with the bullet or number characters present in the text for each line in the cell in the same way as above but using the `formatted_bullets` property. A single space is inserted between the bullet character and the text string and in the case of bullets, this will be the Unicode character seen in Numbers, for example `"• some text"`.\n\n### \xa0Cell images\n\nQuerying cell formats is currently limited to image backrgounds only. If a cell has no background image, `None` is returned for all calls.\n\n``` python\ncell = table.cell("B1")\nwith open (cell.image_filename, "wb") as f:\n    f.write(cell.image_data)\nprint("Wrote file of type", cell.image_type)\n```\n\n## Writing Numbers files\n\n*This is considered experimental*: you are highly recommened not to overwrite working Numbers files and instead save data to a new file.\n\n### Limitations\n\nSince version 3.4.0, adding tables and sheets is supported. Known limitations to write support are:\n\n* Creating cells of type `BulletedTextCell` is not supported\n* Formats cannot be defined for `DurationCell` or `DateCell`\n* New tables are inserted with a fixed offset below the last table in a worksheet which does not take into account title or caption size\n* New sheets insert tables with formats copied from the first table in the previous sheet rather than default table formats\n\n### Editing cells\n\n`numbers-parser` will automatically empty rows and columns for any cell references that are out of range of the current table. The `write` method accepts the same cell numbering notation as `cell` plus an additional argument representing the new cell value. The type of the new value will be used to determine the cell type.\n\n``` python\ndoc = Document("old-sheet.numbers")\nsheets = doc.sheets\ntables = sheets[0].tables\ntable = tables[0]\ntable.write(1, 1, "This is new text")\ntable.write("B7", datetime(2020, 12, 25))\ndoc.save("new-sheet.numbers")\n```\n\nSheet names and table names can be changed by assigning a new value to the `name` of each:\n\n```python\nsheets[0].name = "My new sheet"\ntables[0].name = "Edited table"\n````\n\n### Adding tables and sheets\n\nAdditional tables and worksheets can be added to a `Document` before saving. If no sheet name or table name is supplied, `numbers-parser` will use `Sheet 1`, `Sheet 2`, etc.\n\n```python\ndoc = Document()\ndoc.add_sheet("New Sheet", "New Table")\nsheet = doc.sheets["New Sheet"]\ntable = sheet.tables["New Table"]\ntable.write(1, 1, 1000)\ntable.write(1, 2, 2000)\ntable.write(1, 3, 3000)\n\ndoc.save("sheet.numbers")\n```\n\n## Table geometries\n\n`numbers-parser` can query and change the position and size of tables. Changes made to a table\'s row height or column width is retained when files are saved.\n\n### \xa0Row and column sizes\n\nRow heights and column widths are queried and set using the `row_height` and `col_width` methods:\n\n```python\ndoc = Document("sheet.numbers")\ntable = doc.sheets[0].tables[0]\nprint(f"Table size is {table.height} x {table.width}")\nprint(f"Table row 1 height is {table.row_height(0)}")\ntable.row_height(0, 40)\nprint(f"Table row 1 height is now {table.row_height(0)}")\nprint(f"Table column A width is {table.col_width(0)}")\ntable.col_width(0, 200)\nprint(f"Table column A width is {table.col_width(0)}")\n```\n\n### \xa0Header row and columns\n\nWhen new tables are created, `numbers-parser` follows the Numbers convention of creating a table with one row header and one column header. You can change the number of headers by modifying the appopriate property:\n\n```python\ndoc = Document("sheet.numbers")\ntable = doc.sheets[0].tables[0]\ntable.num_header_rows = 2\ntable.num_header_cols = 0\ndoc.save("saved.numbers")\n```\n\nA zero header count will remove the headers from the table. Attempting to set a negative number of headers, or using more headers that rows or columns in the table will raise a `ValueError` exception.\n\n### Positioning tables\n\nBy default, new tables are positioned at a fixed offset below the last table vertically in a sheet and on the left side of the sheet. Large table headers and captions may result in new tables overlapping existing ones. The `add_table` method takes optional coordinates for positioning a table. A table\'s height and coordinates can also be queried to help aligning new tables:\n\n```python\n(x, y) = sheet.table[0].coordinates\ny += sheet.table[0].height + 200.0\nnew_table = sheet.add_table("Offset Table", x, y)\n```\n\n## Command-line scripts\n\nWhen installed from [PyPI](https://pypi.org/project/numbers-parser/), a command-like script `cat-numbers` is installed in Python\'s scripts folder. This script dumps Numbers spreadsheets into Excel-compatible CSV format, iterating through all the spreadsheets passed on the command-line.\n\n``` text\nusage: cat-numbers [-h] [-T | -S | -b] [-V] [--debug] [--formulas]\n                   [--formatting] [-s SHEET] [-t TABLE] [document ...]\n\nExport data from Apple Numbers spreadsheet tables\n\npositional arguments:\n  document                 Document(s) to export\n\noptional arguments:\n  -h, --help               show this help message and exit\n  -T, --list-tables        List the names of tables and exit\n  -S, --list-sheets        List the names of sheets and exit\n  -b, --brief              Don\'t prefix data rows with name of sheet/table (default: false)\n  -V, --version\n  --debug                  Enable debug output\n  --formulas               Dump formulas instead of formula results\n  --formatting             Dump formatted cells (durations) as they appear in Numbers\n  -s SHEET, --sheet SHEET  Names of sheet(s) to include in export\n  -t TABLE, --table TABLE  Names of table(s) to include in export\n```\n\nNote: `--formatting` will return different capitalisation for 12-hour times due to differences between Numbers\' representation of these dates and `datetime.strftime`. Numbers in English locales displays 12-hour times with \'am\' and \'pm\', but `datetime.strftime` on macOS at least cannot return lower-case versions of AM/PM.\n\n## Numbers File Formats\n\nNumbers uses a proprietary, compressed binary format to store its tables.\nThis format is comprised of a zip file containing images, as well as\n[Snappy](https://github.com/google/snappy)-compressed\n[Protobuf](https://github.com/protocolbuffers/protobuf) `.iwa` files containing\nmetadata, text, and all other definitions used in the spreadsheet.\n\n### Protobuf updates\n\nAs `numbers-parser` includes private Protobuf definitions extracted from a copy of Numbers, new versions of Numbers will inevitably create `.numbers` files that cannot be read by `numbers-parser`. As new versions of Numbers are released, running `make bootstrap` will perform all the steps necessary to recreate the protobuf files used `numbers-parser` to read Numbers spreadsheets.\n\nOn Apple Silicon Macs, the default protobuf package installation does not include the C++ optimised version which is required by the bootstrapping scripts to extract protobufs. You will receive the following error during build if this is the case:\n\n `This script requires the Protobuf installation to use the C++ implementation. Please reinstall Protobuf with C++ support.`\n\n To include the C++ support, download a released version of Google protobuf [from github](https://github.com/protocolbuffers/protobuf). Build instructions are in the [`src/README.md`](https://github.com/protocolbuffers/protobuf/blob/main/src/README.md) in the distribution but for macOS with [Homebrew](https://brew.sh) the two steps are, firstly to install the native protobuf libraries, which must be on your `LD_LIBRARY_PATH`:\n\n``` bash\nbrew install autoconf automake libtool\n./autogen.sh\n./configure --prefix=/usr/local\nmake check -j`sysctl -n hw.ncpu`\nsudo make install\n```\n\nAnd then to install the Python libraries with C++ support. If you already have protobuf install via Homebrew, you will need to `brew unlink` the installation.\n\n``` bash\ncd python\npython3 setup.py build --cpp_implementation\npython3 setup.py test --cpp_implementation\npython3 setup.py install --cpp_implementation\n```\n  \n  This will install protobuf in a folder above the source installation which can then be used by `make bootstrap` in the `numbers-parser` source tree.\n\n## Credits\n\n`numbers-parser` was built by [Jon Connell](http://github.com/masaccio) but relies heavily on from [prior work](https://github.com/psobot/keynote-parser) by [Peter Sobot](https://petersobot.com) to read the IWA format archives used by Apple\'s iWork family of applications, and to regenerate the mapping files required for Python. Both modules are derived from [previous work](https://github.com/obriensp/iWorkFileFormat/blob/master/Docs/index.md) by [Sean Patrick O\'Brien](http://www.obriensp.com).\n\nDecoding the data structures inside Numbers files was helped greatly by [Stingray-Reader](https://github.com/slott56/Stingray-Reader) by [Steven Lott](https://github.com/slott56).\n\nFormula tests were adapted from JavaScript tests used in\n[fast-formula-parser](https://github.com/LesterLyu/fast-formula-parser).\n\nDecimal128 conversion to and from byte storage was adapted from work done by the\n[SheetsJS project](https://github.com/SheetJS/sheetjs). SheetJS also helped greatly with some of the steps required to successfully save a Numbers spreadsheet.\n\n## License\n\nAll code in this repository is licensed under the [MIT License](https://github.com/masaccio/numbers-parser/blob/master/LICENSE.rst)\n',
    'author': 'Jon Connell',
    'author_email': 'python@figsandfudge.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/masaccio/numbers-parser',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
