#
#  Copyright (C) 2010,2021  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#


import sherpa.astro.ui as ui

pars = {}

pars['acis0i'] = {
    'pow1.gamma': -2.06919,
    'pow1.ref':           4,
    'pow1.ampl': 5.71382e-05,
    'pow2.gamma':    0.596993,
    'pow2.ref':           4,
    'pow2.ampl': 0.000210897,
    'g1.fwhm':    0.235916,
    'g1.pos':    0.520526,
    'g1.ampl': 0.000119669,
    'g2.fwhm':  0.00131921,
    'g2.pos':     1.49967,
    'g2.ampl':  0.00946143,
    'g3.fwhm':    0.220895,
    'g3.pos':     1.80814,
    'g3.ampl': 0.000137385,
    'g4.fwhm':    0.152387,
    'g4.pos':      2.1634,
    'g4.ampl': 0.000536424,
    'g5.fwhm':   0.0519383,
    'g5.pos':     7.49607,
    'g5.ampl':  0.00202631,
    'g6.fwhm':    0.326552,
    'g6.pos':     8.27745,
    'g6.ampl': 0.000116593,
    }

pars['acis1i'] = {
    'pow1.gamma': -2.08968,
    'pow1.ref':             4,
    'pow1.ampl':   5.71616e-05,
    'pow2.gamma':       0.60286,
    'pow2.ref':             4,
    'pow2.ampl':   0.000219046,
    'g1.fwhm':      0.204261,
    'g1.pos':      0.565457,
    'g1.ampl':   9.26722e-05,
    'g2.fwhm':    0.00477969,
    'g2.pos':       1.48143,
    'g2.ampl':    0.00235951,
    'g3.fwhm':      0.234647,
    'g3.pos':       1.81784,
    'g3.ampl':   0.000134789,
    'g4.fwhm':      0.148377,
    'g4.pos':       2.16246,
    'g4.ampl':   0.000535352,
    'g5.fwhm':     0.0735449,
    'g5.pos':        7.4826,
    'g5.ampl':    0.00142288,
    'g6.fwhm':      0.506373,
    'g6.pos':        8.2381,
    'g6.ampl':   9.59387e-05,
    }

pars['acis2i'] = {
    'pow1.gamma': -2.08968,
    'pow1.ref':           4,
    'pow1.ampl': 5.71616e-05,
    'pow2.gamma':     0.60286,
    'pow2.ref':           4,
    'pow2.ampl': 0.000219046,
    'g1.fwhm':    0.358572,
    'g1.pos':    0.576604,
    'g1.ampl': 0.000149777,
    'g2.fwhm':  0.00300716,
    'g2.pos':     1.48873,
    'g2.ampl':  0.00576769,
    'g3.fwhm':    0.245171,
    'g3.pos':     1.80394,
    'g3.ampl': 0.000133464,
    'g4.fwhm':    0.139058,
    'g4.pos':     2.16976,
    'g4.ampl': 0.000581377,
    'g5.fwhm':   0.0656173,
    'g5.pos':     7.50531,
    'g5.ampl':  0.00167657,
    'g6.fwhm':    0.260118,
    'g6.pos':     8.32323,
    'g6.ampl': 0.000121175,
    }

pars['acis2s'] = {
    'pow1.gamma': -2.06335,
    'pow1.ref':           4,
    'pow1.ampl': 5.84052e-05,
    'pow2.gamma':    0.651823,
    'pow2.ref':           4,
    'pow2.ampl': 0.000224642,
    'g1.fwhm':    0.403368,
    'g1.pos':    0.615889,
    'g1.ampl': 5.24234e-05,
    'g2.fwhm':  0.00292955,
    'g2.pos':     1.48135,
    'g2.ampl':  0.00609344,
    'g3.fwhm':     0.15541,
    'g3.pos':      1.7742,
    'g3.ampl': 0.000162127,
    'g4.fwhm':    0.158258,
    'g4.pos':     2.16079,
    'g4.ampl': 0.000525568,
    'g5.fwhm':    0.124188,
    'g5.pos':     7.49188,
    'g5.ampl': 0.000936898,
    'g6.fwhm':    0.319197,
    'g6.pos':     8.30065,
    'g6.ampl': 0.000122796,
    }

pars['acis3i'] = {
    'pow1.gamma': -2.08968,
    'pow1.ref':           4,
    'pow1.ampl': 5.71616e-05,
    'pow2.gamma':     0.60286,
    'pow2.ref':           4,
    'pow2.ampl': 0.000219046,
    'g1.fwhm':     0.44995,
    'g1.pos':    0.525641,
    'g1.ampl': 7.98613e-05,
    'g2.fwhm': 0.000880626,
    'g2.pos':     1.48143,
    'g2.ampl':   0.0190133,
    'g3.fwhm':    0.233277,
    'g3.pos':     1.81784,
    'g3.ampl':  0.00013719,
    'g4.fwhm':    0.145338,
    'g4.pos':     2.16635,
    'g4.ampl': 0.000576123,
    'g5.fwhm':   0.0394237,
    'g5.pos':     7.48964,
    'g5.ampl':  0.00300823,
    'g6.fwhm':    0.436435,
    'g6.pos':     8.31374,
    'g6.ampl': 0.000132079,
    }

pars['acis3s'] = {
    'pow1.gamma': -2.22521,
    'pow1.ref':           4,
    'pow1.ampl': 5.11749e-05,
    'pow2.gamma':    0.574363,
    'pow2.ref':           4,
    'pow2.ampl': 0.000237404,
    'g1.fwhm':    0.494623,
    'g1.pos':    0.647695,
    'g1.ampl': 4.76084e-05,
    'g2.fwhm':  0.00289383,
    'g2.pos':     1.48004,
    'g2.ampl':  0.00562661,
    'g3.fwhm':    0.215497,
    'g3.pos':     1.77779,
    'g3.ampl': 0.000145856,
    'g4.fwhm':    0.141732,
    'g4.pos':     2.16067,
    'g4.ampl': 0.000589601,
    'g5.fwhm':    0.077452,
    'g5.pos':     7.48077,
    'g5.ampl':  0.00153332,
    'g6.fwhm':    0.405486,
    'g6.pos':     8.29021,
    'g6.ampl': 0.000118158,
    }

pars['acis5s'] = {
    'pow1.gamma': -6.12125,
    'pow1.ref':             4,
    'pow1.ampl':   3.31976e-05,
    'pow2.gamma':      0.615669,
    'pow2.ref':             4,
    'pow2.ampl':   0.000483355,
    'g1.fwhm':           0.1,
    'g1.pos':      0.525641,
    'g1.ampl':    0.00147523,
    'g2.fwhm':   0.000880626,
    'g2.pos':       1.48143,
    'g2.ampl':      0.020896,
    'g3.fwhm':      0.186781,
    'g3.pos':       1.78293,
    'g3.ampl':    0.00125368,
    'g4.fwhm':      0.286822,
    'g4.pos':        2.1294,
    'g4.ampl':   0.000704949,
    'g5.fwhm':             2,
    'g5.pos':       7.76993,
    'g5.ampl':    0.00121701,
    'g6.fwhm':      0.436435,
    'g6.pos':       8.31374,
    'g6.ampl':   0.000191996,
    }

pars['acis6i'] = {
    'pow1.gamma': -2.20319,
    'pow1.ref':            4,
    'pow1.ampl':   5.1385e-05,
    'pow2.gamma':     0.615709,
    'pow2.ref':            4,
    'pow2.ampl':  0.000246014,
    'g1.fwhm':        1e-08,
    'g1.pos':     0.738391,
    'g1.ampl':  6.74342e-05,
    'g2.fwhm':  0.000962075,
    'g2.pos':      1.48813,
    'g2.ampl':    0.0129933,
    'g3.fwhm':     0.217325,
    'g3.pos':      1.80989,
    'g3.ampl':  0.000164641,
    'g4.fwhm':     0.148726,
    'g4.pos':      2.17042,
    'g4.ampl':  0.000578537,
    'g5.fwhm':          0.1,
    'g5.pos':      7.49067,
    'g5.ampl':  0.000817549,
    'g6.fwhm':     0.469257,
    'g6.pos':      8.36108,
    'g6.ampl':  7.95877e-05,
    }

pars['acis6s'] = {
    'pow1.gamma': -2.23266,
    'pow1.ref':            4,
    'pow1.ampl':  5.49859e-05,
    'pow2.gamma':     0.622359,
    'pow2.ref':            4,
    'pow2.ampl':  0.000252563,
    'g1.fwhm':  1e-8,
    'g1.pos':     0.738391,
    'g1.ampl':  6.74342e-05,
    'g2.fwhm':  0.000877951,
    'g2.pos':      1.48813,
    'g2.ampl':    0.0104668,
    'g3.fwhm':     0.320222,
    'g3.pos':      1.80989,
    'g3.ampl':  0.000121991,
    'g4.fwhm':     0.139571,
    'g4.pos':      2.17042,
    'g4.ampl':  0.000623203,
    'g5.fwhm':          0.1,
    'g5.pos':      7.51181,
    'g5.ampl':  0.000767453,
    'g6.fwhm':     0.180799,
    'g6.pos':      8.32003,
    'g6.ampl':  0.000147529,
    }


pars['acis7s'] = {
    'pow1.gamma': -7.27677,
    'pow1.ref':           7,
    'pow1.ampl': 0.000350106,
    'pow2.gamma':     0.44274,
    'pow2.ref':           7,
    'pow2.ampl': 0.000367266,
    'g1.fwhm':           1,
    'g1.pos':           0,
    'g1.ampl':   0.0010143,
    'g2.fwhm':    0.112428,
    'g2.pos':         1.5,
    'g2.ampl': 0.000204453,
    'g3.fwhm':  0.00479983,
    'g3.pos':     1.75243,
    'g3.ampl':   0.0300776,
    'g4.fwhm':   0.0864722,
    'g4.pos':     2.12312,
    'g4.ampl':  0.00160632,
    'g5.fwhm':         0.1,
    'g5.pos':     7.45356,
    'g5.ampl': 0.000626361,
    'g6.fwhm':    0.122775,
    'g6.pos':     1.86313,
    'g6.ampl': 0.000902448,
    }

pars['acis8s'] = {
    'pow1.gamma': -1.41246,
    'pow1.ref':             4,
    'pow1.ampl':   0.000123478,
    'pow2.gamma':       1.11102,
    'pow2.ref':             4,
    'pow2.ampl':   0.000165159,
    'g1.fwhm':      0.358572,
    'g1.pos':      0.576604,
    'g1.ampl':   0.000149777,
    'g2.fwhm':    0.00300716,
    'g2.pos':       1.48873,
    'g2.ampl':    0.00169218,
    'g3.fwhm':      0.245171,
    'g3.pos':       1.80394,
    'g3.ampl':   0.000101901,
    'g4.fwhm':      0.139058,
    'g4.pos':       2.16976,
    'g4.ampl':   0.000598291,
    'g5.fwhm':     0.0656173,
    'g5.pos':       7.45382,
    'g5.ampl':    0.00112055,
    'g6.fwhm':      0.260118,
    'g6.pos':       8.32323,
    'g6.ampl':   0.000120482,
    }

pars['acis4s'] = pars['acis6s']
pars['acis9s'] = pars['acis8s']


def acis_bkg_model(detnam, root='bkg_', as_str=False):
    """Empirically derived background model for the ACIS detector, based on
    fitting a broken powerlaw plus 6 gaussians to ACIS background data.  These
    models *require* that the corresponding ARF be set to a constant value of 100
    and that the RMF be the correct RMF for the source and detector.  The model
    is only calibrated between 0.5 and 9 keV.  The following code is an example::

       from acis_bkg_model import acis_bkg_model

       load_pha(1, 'acisf04938_000N002_r0043_pha3.fits')
       arf = get_arf()
       rmf = get_rmf()

       # Load the background ARF/RMF.  This must be done in addition
       # to load_pha, otherwise the source and background arfs are
       # always identical.
       load_bkg_arf(1, arf.name)
       load_bkg_rmf(1, rmf.name)
       bkg_arf = get_bkg_arf(1)
       bkg_rmf = get_bkg_rmf(1)

       # Stub the bkg_arf to be a constant.  This is required for use
       # of the acis_bkg_model models.
       bkg_arf.specresp = bkg_arf.specresp * 0 + 100.

       # Set scaling between background and source apertures
       # Roughly equivalent to
       # bkg_scale = get_exposure() * get_backscal() / (get_exposure(bkg_id=1) * get_backscal(bkg_id=1))
       bkg_scale = get_data(1).sum_background_data(lambda x,y: 1)

       # Set source and background models.  This source is on ACIS-I CCDID = 2 (acis2i).
       bkg_model = const1d.c1 * acis_bkg_model('acis2i')
       set_full_model(rsp(powlaw1d.pow1) + bkg_scale * bkg_rsp(bkg_model))
       set_bkg_full_model(bkg_rmf(bkg(arf(bkg_model))))

       set_full_model(powlaw1d.pow1)
       set_bkg_full_model(bkg_rmf(bkg_arf( const1d.c1 * acis_bkg_model('acis2i'))))

       fit()  # or fit_bkg() to only fit the background

    :param detnam: detector name 'acis<CCD_ID><aimpoint det: i or s>'
    :returns: sherpa model for background
    """
    comps = (('powlaw1d', 'pow1'),
             ('powlaw1d', 'pow2'),
             ('gauss1d', 'g1'),
             ('gauss1d', 'g2'),
             ('gauss1d', 'g3'),
             ('gauss1d', 'g4'),
             ('gauss1d', 'g5'),
             ('gauss1d', 'g6'))

    model_comps = dict()
    for mtype, name in comps:
        ui.create_model_component(mtype, root + name)
        model_comp = model_comps[name] = eval(root + name)
        if mtype == 'gauss1d':
            model_comp.ampl.min = 0.0
        ui.freeze(model_comp)
        model_comp.integrate = True

    if detnam in pars:
        for parname, parval in pars[detnam].items():
            name, attr = parname.split('.')
            setattr(model_comps[name], attr, parval)
    else:
        raise ValueError('No background model available for "{0}".  Must be one of {1}'.format(
            detnam, sorted(pars.keys())))

    if as_str:
        out = ' + '.join([root + name for mtype, name in comps])
    else:
        mc = model_comps
        out = mc['pow1'] + mc['pow2'] + mc['g1'] + mc['g2'] + mc['g3'] + mc['g4'] + mc['g5'] + mc['g6']
    return out
