"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var PackageDetails_1 = require("./PackageDetails");
describe("getPackageDetailsFromPatchFilename", function () {
    it("parses old-style patch filenames", function () {
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("@types/banana:3.4.2-beta.2.patch")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"@types/banana\",\n  \"isDevOnly\": false,\n  \"isNested\": false,\n  \"name\": \"@types/banana\",\n  \"packageNames\": Array [\n    \"@types/banana\",\n  ],\n  \"patchFilename\": \"@types/banana:3.4.2-beta.2.patch\",\n  \"path\": \"node_modules/@types/banana\",\n  \"pathSpecifier\": \"@types/banana\",\n  \"version\": \"3.4.2-beta.2\",\n}\n");
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("banana:0.4.2.patch"))
            .toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"banana\",\n  \"isDevOnly\": false,\n  \"isNested\": false,\n  \"name\": \"banana\",\n  \"packageNames\": Array [\n    \"banana\",\n  ],\n  \"patchFilename\": \"banana:0.4.2.patch\",\n  \"path\": \"node_modules/banana\",\n  \"pathSpecifier\": \"banana\",\n  \"version\": \"0.4.2\",\n}\n");
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("banana+0.4.2.patch"))
            .toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"banana\",\n  \"isDevOnly\": false,\n  \"isNested\": false,\n  \"name\": \"banana\",\n  \"packageNames\": Array [\n    \"banana\",\n  ],\n  \"patchFilename\": \"banana+0.4.2.patch\",\n  \"path\": \"node_modules/banana\",\n  \"pathSpecifier\": \"banana\",\n  \"version\": \"0.4.2\",\n}\n");
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("banana-0.4.2.patch")).toBe(null);
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("@types+banana-0.4.2.patch")).toBe(null);
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("banana+0.4.2.dev.patch"))
            .toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"banana\",\n  \"isDevOnly\": true,\n  \"isNested\": false,\n  \"name\": \"banana\",\n  \"packageNames\": Array [\n    \"banana\",\n  ],\n  \"patchFilename\": \"banana+0.4.2.dev.patch\",\n  \"path\": \"node_modules/banana\",\n  \"pathSpecifier\": \"banana\",\n  \"version\": \"0.4.2.dev\",\n}\n");
    });
    it("parses new-style patch filenames", function () {
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("banana++apple+0.4.2.patch"))
            .toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"banana => apple\",\n  \"isDevOnly\": false,\n  \"isNested\": true,\n  \"name\": \"apple\",\n  \"packageNames\": Array [\n    \"banana\",\n    \"apple\",\n  ],\n  \"patchFilename\": \"banana++apple+0.4.2.patch\",\n  \"path\": \"node_modules/banana/node_modules/apple\",\n  \"pathSpecifier\": \"banana/apple\",\n  \"version\": \"0.4.2\",\n}\n");
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("@types+banana++@types+apple++@mollusc+man+0.4.2-banana-tree.patch")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"@types/banana => @types/apple => @mollusc/man\",\n  \"isDevOnly\": false,\n  \"isNested\": true,\n  \"name\": \"@mollusc/man\",\n  \"packageNames\": Array [\n    \"@types/banana\",\n    \"@types/apple\",\n    \"@mollusc/man\",\n  ],\n  \"patchFilename\": \"@types+banana++@types+apple++@mollusc+man+0.4.2-banana-tree.patch\",\n  \"path\": \"node_modules/@types/banana/node_modules/@types/apple/node_modules/@mollusc/man\",\n  \"pathSpecifier\": \"@types/banana/@types/apple/@mollusc/man\",\n  \"version\": \"0.4.2-banana-tree\",\n}\n");
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("@types+banana.patch++hello+0.4.2-banana-tree.patch")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"@types/banana.patch => hello\",\n  \"isDevOnly\": false,\n  \"isNested\": true,\n  \"name\": \"hello\",\n  \"packageNames\": Array [\n    \"@types/banana.patch\",\n    \"hello\",\n  ],\n  \"patchFilename\": \"@types+banana.patch++hello+0.4.2-banana-tree.patch\",\n  \"path\": \"node_modules/@types/banana.patch/node_modules/hello\",\n  \"pathSpecifier\": \"@types/banana.patch/hello\",\n  \"version\": \"0.4.2-banana-tree\",\n}\n");
        expect(PackageDetails_1.getPackageDetailsFromPatchFilename("@types+banana.patch++hello+0.4.2-banana-tree.dev.patch")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"@types/banana.patch => hello\",\n  \"isDevOnly\": true,\n  \"isNested\": true,\n  \"name\": \"hello\",\n  \"packageNames\": Array [\n    \"@types/banana.patch\",\n    \"hello\",\n  ],\n  \"patchFilename\": \"@types+banana.patch++hello+0.4.2-banana-tree.dev.patch\",\n  \"path\": \"node_modules/@types/banana.patch/node_modules/hello\",\n  \"pathSpecifier\": \"@types/banana.patch/hello\",\n  \"version\": \"0.4.2-banana-tree\",\n}\n");
    });
});
describe("getPatchDetailsFromCliString", function () {
    it("handles a minimal package name", function () {
        expect(PackageDetails_1.getPatchDetailsFromCliString("patch-package")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"patch-package\",\n  \"isNested\": false,\n  \"name\": \"patch-package\",\n  \"packageNames\": Array [\n    \"patch-package\",\n  ],\n  \"path\": \"node_modules/patch-package\",\n  \"pathSpecifier\": \"patch-package\",\n}\n");
    });
    it("handles a scoped package name", function () {
        expect(PackageDetails_1.getPatchDetailsFromCliString("@david/patch-package")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"@david/patch-package\",\n  \"isNested\": false,\n  \"name\": \"@david/patch-package\",\n  \"packageNames\": Array [\n    \"@david/patch-package\",\n  ],\n  \"path\": \"node_modules/@david/patch-package\",\n  \"pathSpecifier\": \"@david/patch-package\",\n}\n");
    });
    it("handles a nested package name", function () {
        expect(PackageDetails_1.getPatchDetailsFromCliString("david/patch-package")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"david => patch-package\",\n  \"isNested\": true,\n  \"name\": \"patch-package\",\n  \"packageNames\": Array [\n    \"david\",\n    \"patch-package\",\n  ],\n  \"path\": \"node_modules/david/node_modules/patch-package\",\n  \"pathSpecifier\": \"david/patch-package\",\n}\n");
    });
    it("handles a nested package name with scopes", function () {
        expect(PackageDetails_1.getPatchDetailsFromCliString("@david/patch-package/banana")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"@david/patch-package => banana\",\n  \"isNested\": true,\n  \"name\": \"banana\",\n  \"packageNames\": Array [\n    \"@david/patch-package\",\n    \"banana\",\n  ],\n  \"path\": \"node_modules/@david/patch-package/node_modules/banana\",\n  \"pathSpecifier\": \"@david/patch-package/banana\",\n}\n");
        expect(PackageDetails_1.getPatchDetailsFromCliString("@david/patch-package/@david/banana")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"@david/patch-package => @david/banana\",\n  \"isNested\": true,\n  \"name\": \"@david/banana\",\n  \"packageNames\": Array [\n    \"@david/patch-package\",\n    \"@david/banana\",\n  ],\n  \"path\": \"node_modules/@david/patch-package/node_modules/@david/banana\",\n  \"pathSpecifier\": \"@david/patch-package/@david/banana\",\n}\n");
        expect(PackageDetails_1.getPatchDetailsFromCliString("david/patch-package/@david/banana")).toMatchInlineSnapshot("\nObject {\n  \"humanReadablePathSpecifier\": \"david => patch-package => @david/banana\",\n  \"isNested\": true,\n  \"name\": \"@david/banana\",\n  \"packageNames\": Array [\n    \"david\",\n    \"patch-package\",\n    \"@david/banana\",\n  ],\n  \"path\": \"node_modules/david/node_modules/patch-package/node_modules/@david/banana\",\n  \"pathSpecifier\": \"david/patch-package/@david/banana\",\n}\n");
    });
});
//# sourceMappingURL=data:application/json;base64,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