# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['wraps']

package_data = \
{'': ['*']}

install_requires = \
['attrs>=22.1.0', 'typing-extensions>=4.3.0']

setup_kwargs = {
    'name': 'wraps',
    'version': '0.2.0',
    'description': 'Meaningful and safe wrapping types.',
    'long_description': '# `wraps`\n\n[![License][License Badge]][License]\n[![Version][Version Badge]][Package]\n[![Downloads][Downloads Badge]][Package]\n[![Discord][Discord Badge]][Discord]\n\n[![Documentation][Documentation Badge]][Documentation]\n[![Check][Check Badge]][Actions]\n[![Test][Test Badge]][Actions]\n[![Coverage][Coverage Badge]][Coverage]\n\n**Meaningful and safe wrapping types.**\n\n## Installing\n\n**Python 3.7 or above is required.**\n\n### pip\n\nInstalling the library with `pip` is quite simple:\n\n```console\n$ pip install wraps\n```\n\nAlternatively, the library can be installed from source:\n\n```console\n$ git clone https://github.com/nekitdev/wraps.git\n$ cd wraps\n$ python -m pip install .\n```\n\n### poetry\n\nYou can add `wraps` as a dependency with the following command:\n\n```console\n$ poetry add wraps\n```\n\nOr by directly specifying it in the configuration like so:\n\n```toml\n[tool.poetry.dependencies]\nwraps = "^0.2.0"\n```\n\nAlternatively, you can add it directly from the source:\n\n```toml\n[tool.poetry.dependencies.wraps]\ngit = "https://github.com/nekitdev/wraps.git"\n```\n\n## Examples\n\n### Option\n\n[`Option[T]`][wraps.option.Option] type represents an optional value: every option is either\n[`Some[T]`][wraps.option.Some] and contains a value, or [`Null`][wraps.option.Null], and does not.\n\nHere is an example of using [`wrap_option`][wraps.option.wrap_option] to catch any errors:\n\n```python\nfrom typing import List, TypeVar\nfrom wraps import wrap_option\n\nT = TypeVar("T", covariant=True)\n\n\nclass SafeList(List[T]):\n    @wrap_option\n    def get(self, index: int) -> T:\n        return self[index]\n\n\narray = SafeList([0, 1, 2, 3])\n\nprint(array.get(0))  # Some(value=0)\nprint(array.get(7))  # Null()\n```\n\n### Result\n\n[`Result[T, E]`][wraps.result.Result] is the type used for returning and propagating errors.\nIt has two variants, [`Ok[T]`][wraps.result.Ok], representing success and containing a value,\nand [`Error[E]`][wraps.result.Error], representing error and containing an error value.\n\n```python\nfrom wraps import Result, wrap_result\n\n\n@wrap_result[ValueError]\ndef parse(string: str) -> int:\n    return int(string)\n\n\ndef multiply(x: str, y: str) -> Result[int, ValueError]:\n    # try to parse two strings and multiply results\n    return parse(x).and_then(lambda m: parse(y).map(lambda n: m * n))\n\n\nprint(multiply("21", "2").unwrap())  # 42\nprint(multiply("!", "42").unwrap_error())  # invalid literal for `int` with base 10: `!`\n```\n\nIn python versions before 3.9 (where grammar restrictions on decorators were relaxed),\none can use [`wrap_result`][wraps.result.wrap_result] without a concrete type:\n\n```python\n@wrap_result\ndef parse(string: str) -> int:\n    return int(string)\n```\n\nHowever this makes the types less specific, since [`Exception`][Exception]\nis caught instead of [`ValueError`][ValueError].\n\nInstead, you can create a new concrete [`WrapResult[E]`][wraps.result.WrapResult] instance:\n\n```python\nfrom wraps import WrapResult\n\nwrap_value_error = WrapResult(ValueError)\n\n@wrap_value_error\ndef parse(string: str) -> int:\n    return int(string)\n```\n\n### Early Return\n\nEarly return functionality (`?` operator in Rust) is implemented via `Q` property\n(for both [`Option`][wraps.option.Option] and [`Result`][wraps.result.Result] types)\ncombined with the [`@option_shortcut`][wraps.shortcuts.option_shortcut] or\n[`@result_shortcut`][wraps.shortcuts.result_shortcut] decorator respectively.\n\n```python\nfrom wraps import Option, Some, option_shortcut, wrap_option\n\n\n@wrap_option\ndef parse(string: str) -> int:\n    return int(string)\n\n\n@option_shortcut\ndef try_add(x: str, y: str) -> Option[int]:\n    m = parse(x).Q\n    n = parse(y).Q\n\n    return Some(m + n)\n```\n\n## Documentation\n\nYou can find the documentation [here][Documentation].\n\n## Support\n\nIf you need support with the library, you can send us an [email][Email]\nor refer to the official [Discord server][Discord].\n\n## Changelog\n\nYou can find the changelog [here][Changelog].\n\n## Security Policy\n\nYou can find the Security Policy of `wraps` [here][Security].\n\n## Contributing\n\nIf you are interested in contributing to `wraps`, make sure to take a look at the\n[Contributing Guide][Contributing Guide], as well as the [Code of Conduct][Code of Conduct].\n\n## License\n\n`wraps` is licensed under the MIT License terms. See [License][License] for details.\n\n[Email]: mailto:support@nekit.dev\n\n[Discord]: https://nekit.dev/discord\n\n[Actions]: https://github.com/nekitdev/wraps/actions\n\n[Changelog]: https://github.com/nekitdev/wraps/blob/main/CHANGELOG.md\n[Code of Conduct]: https://github.com/nekitdev/wraps/blob/main/CODE_OF_CONDUCT.md\n[Contributing Guide]: https://github.com/nekitdev/wraps/blob/main/CONTRIBUTING.md\n[Security]: https://github.com/nekitdev/wraps/blob/main/SECURITY.md\n\n[License]: https://github.com/nekitdev/wraps/blob/main/LICENSE\n\n[Package]: https://pypi.org/project/wraps\n[Coverage]: https://codecov.io/gh/nekitdev/wraps\n[Documentation]: https://nekitdev.github.io/wraps\n\n[Discord Badge]: https://img.shields.io/badge/chat-discord-5865f2\n[License Badge]: https://img.shields.io/pypi/l/wraps\n[Version Badge]: https://img.shields.io/pypi/v/wraps\n[Downloads Badge]: https://img.shields.io/pypi/dm/wraps\n\n[Documentation Badge]: https://github.com/nekitdev/wraps/workflows/docs/badge.svg\n[Check Badge]: https://github.com/nekitdev/wraps/workflows/check/badge.svg\n[Test Badge]: https://github.com/nekitdev/wraps/workflows/test/badge.svg\n[Coverage Badge]: https://codecov.io/gh/nekitdev/wraps/branch/main/graph/badge.svg\n\n[wraps.option.Option]: https://nekitdev.github.io/wraps/reference/option#wraps.option.Option\n[wraps.option.Some]: https://nekitdev.github.io/wraps/reference/option#wraps.option.Some\n[wraps.option.Null]: https://nekitdev.github.io/wraps/reference/option#wraps.option.Null\n[wraps.option.wrap_option]: https://nekitdev.github.io/wraps/reference/option#wraps.option.wrap_option\n\n[wraps.result.Result]: https://nekitdev.github.io/wraps/reference/result#wraps.result.Result\n[wraps.result.Ok]: https://nekitdev.github.io/wraps/reference/result#wraps.result.Ok\n[wraps.result.Error]: https://nekitdev.github.io/wraps/reference/result#wraps.result.Error\n[wraps.result.wrap_result]: https://nekitdev.github.io/wraps/reference/result#wraps.result.wrap_result\n\n[wraps.result.WrapResult]: https://nekitdev.github.io/wraps/reference/result#wraps.result.WrapResult\n\n[wraps.shortcuts.option_shortcut]: https://nekitdev.github.io/wraps/reference/shortcuts#wraps.shortcuts.option_shortcut\n[wraps.shortcuts.result_shortcut]: https://nekitdev.github.io/wraps/reference/shortcuts#wraps.shortcuts.result_shortcut\n\n[Exception]: https://docs.python.org/3/library/exceptions#Exception\n[ValueError]: https://docs.python.org/3/library/exceptions#ValueError\n',
    'author': 'nekitdev',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/nekitdev/wraps',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7',
}


setup(**setup_kwargs)
