"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretRotation = exports.SecretRotationApplication = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const serverless = require("@aws-cdk/aws-sam");
const core_1 = require("@aws-cdk/core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    /**
     *
     */
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme.
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme.
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme.
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme.
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme.
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme.
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme.
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme.
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme.
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme.
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme.
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme.
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme.
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme.
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database.
 */
class SecretRotation extends core_2.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        // Max length of 64 chars, get the last 64 chars
        const uniqueId = core_1.Names.uniqueId(this);
        const rotationFunctionName = uniqueId.substring(Math.max(uniqueId.length - 64, 0), uniqueId.length);
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.excludeCharacters !== undefined) {
            parameters.excludeCharacters = props.excludeCharacters;
        }
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent master secret deletion when rotation is in place
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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