"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'CodeCommit Repositories': {
        'add an SNS trigger to repository'(test) {
            const stack = new core_1.Stack();
            const props = {
                repositoryName: 'MyRepository',
            };
            const snsArn = 'arn:aws:sns:*:123456789012:my_topic';
            new lib_1.Repository(stack, 'MyRepository', props).notify(snsArn);
            assert_1.expect(stack).toMatch({
                Resources: {
                    MyRepository4C4BD5FC: {
                        Type: 'AWS::CodeCommit::Repository',
                        Properties: {
                            RepositoryName: 'MyRepository',
                            Triggers: [
                                {
                                    Events: [
                                        'all'
                                    ],
                                    DestinationArn: 'arn:aws:sns:*:123456789012:my_topic',
                                    Name: 'MyRepository/arn:aws:sns:*:123456789012:my_topic'
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'fails when triggers have duplicate names'(test) {
            const stack = new core_1.Stack();
            const myRepository = new lib_1.Repository(stack, 'MyRepository', {
                repositoryName: 'MyRepository',
            }).notify('myTrigger');
            test.throws(() => myRepository.notify('myTrigger'));
            test.done();
        },
        'can be imported using a Repository ARN'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const repositoryArn = 'arn:aws:codecommit:us-east-1:585695036304:my-repo';
            // WHEN
            const repo = lib_1.Repository.fromRepositoryArn(stack, 'ImportedRepo', repositoryArn);
            // THEN
            test.deepEqual(stack.resolve(repo.repositoryArn), repositoryArn);
            test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
            test.done();
        },
        'can be imported using just a Repository name (the ARN is deduced)'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const repo = lib_1.Repository.fromRepositoryName(stack, 'ImportedRepo', 'my-repo');
            // THEN
            test.deepEqual(stack.resolve(repo.repositoryArn), {
                'Fn::Join': ['', [
                        'arn:',
                        { Ref: 'AWS::Partition' },
                        ':codecommit:',
                        { Ref: 'AWS::Region' },
                        ':',
                        { Ref: 'AWS::AccountId' },
                        ':my-repo'
                    ]],
            });
            test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
            test.done();
        },
        'grant push'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const repository = new lib_1.Repository(stack, 'Repo', {
                repositoryName: 'repo-name'
            });
            const role = new aws_iam_1.Role(stack, 'Role', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com')
            });
            // WHEN
            repository.grantPullPush(role);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'codecommit:GitPull',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'Repo02AC86CF',
                                    'Arn'
                                ]
                            },
                        },
                        {
                            Action: 'codecommit:GitPush',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'Repo02AC86CF',
                                    'Arn'
                                ]
                            },
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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