      module params
      implicit none

      save   

      integer, parameter :: kr=selected_real_kind(10) 

      integer, parameter ::
     &     mxly=65,             ! maximum number of layers
     &     maxmom=299,          ! maximum number of legendre moments
     &     nstrms=40,           ! maximum number of radiation streams
     &     ncldz=5,             ! number of cloud layers
     &     naerz=5,             ! number of stratospheric aerosol layers
     &     mxq=63,              ! number of lowtran species
     &     mxkd=20              ! max size of k-distribution arrays

      real(kr), parameter ::
     &     pzero=1013.25,       ! standard pressure (mb)
     &     tzero=273.15,        ! standard temperature (K)
     &     re=6371.2,           ! radius of earth (km)
     &     pmo=2.6568e-23,      ! oxygen atomic mass (g)
     &     grav=9.80665,        ! gravitational acceleration (m/s/s)
     &     alosch=2.6868e19,    ! loschmidt number molecules/cm3 at stp
     &     zip=-1.,             ! undefined read value
     &     nan=huge(0.),        ! another undefined read value
     &     one=1.,              ! 1 of selected real kind kr
     &     zero=0.,             ! 0 of selected real kind kr
     &     wl55=0.55,           ! standard visible wavelength
     &     pi=3.1415926536_kr   ! circumference/diameter of a circle 
                                ! the _kr ensures precision matches 
                                ! selected real kind kr

      end module params
c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
c RCS version control information:
c $Header: /home/paul/rt/sbrt/RCS/disutil.f,v 1.7 2002/08/14 22:56:35 paul Exp $
c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

c ---------------------------------------------------------------------
c  Fortran-90 versions of machine-constant routines R1MACH, D1MACH, I1MACH
c
c  {R,D,I}1MACH revisited: no more uncommenting DATA statements
c  
c  Presented at the IFIP WG 2.5 International Workshop on 
c  "Current Directions in Numerical Software and High Performance 
c  Computing", 19 - 20 October 1995, Kyoto, Japan. 
c  
c  The widely-used original routines were modified to use Fortran-90 
c  intrinsic functions.  This was not completely possible with I1MACH, 
c  which returns some parameters (logical unit numbers of standard
c  input, standard output, and standard error) that may require
c  user customization. 
c  
c  David Gay (dmg@bell-labs.com)
c  Eric Grosse (ehg@bell-labs.com)
c  Bell Laboratories
c  700 Mountain Avenue
c  Murray Hill, New Jersey 07974-0636
c  USA 
c  
c  References:
c  
c  David Gay and Eric Grosse, Comment on Algorithm 528, Bell Labs, Murray 
c    Hill, NJ. submitted to ACM Transactions on Mathematical Software,
c    August 1996.
c
c http://www.nsc.liu.se/~boein/ifip/kyoto/workshop-info/proceedings/einarsson
c    /d1mach.html  (THIS WEB SITE WORKED AS OF APR 2000)
c -------------------------------------------------------------------------


      FUNCTION R1MACH (I)
c
c   R1MACH can be used to obtain machine-dependent parameters for
c   single precision numbers.  The results for various values of I are:
c
c   R1MACH(1) = B**(EMIN-1), the smallest positive magnitude.
c   R1MACH(2) = B**EMAX*(1 - B**(-T)), the largest magnitude.
c   R1MACH(3) = B**(-T), the smallest relative spacing.
c   R1MACH(4) = B**(1-T), the largest relative spacing.
c   R1MACH(5) = LOG10(B)
c
c   Assume single precision numbers are represented in the T-digit,
c   base-B form
c
c              sign (B**E)*( (X(1)/B) + ... + (X(T)/B**T) )
c
c   where 0 <= X(I) < B for I=1,...,T; 0 < X(1); and EMIN <= E <= EMAX.
c
c   The values of B, T, EMIN and EMAX are provided in I1MACH as follows:
c   I1MACH(10) = B, the base.
c   I1MACH(11) = T, the number of base-B digits.
c   I1MACH(12) = EMIN, the smallest exponent E.
c   I1MACH(13) = EMAX, the largest exponent E.
c
c***REFERENCES  
c
c  P. Fox, A. Hall and N. Schryer, Framework for a portable library,
c     ACM Transactions on Mathematical Software 4, 177-188 (1978).
c
c  David Gay and Eric Grosse, Comment on Algorithm 528, Bell Labs, Murray 
c     Hill, NJ. submitted to ACM Transactions on Mathematical Software,
c     August 1996. 
c
c***REVISION HISTORY  (YYMMDD)
c   790101  DATE WRITTEN
c   960329  Modified for Fortran 90 (BE after suggestions by Eric Grosse)      
c --------------------------------------------------------------------

      use params, only: kr
      IMPLICIT NONE
      INTEGER :: I
      REAL(KR) :: B, X = 1.0, r1mach

      B = RADIX(X)

      SELECT CASE (I)
        CASE (1) 
          R1MACH = TINY(X)            ! smallest positive magnitude.
        CASE (2)
          R1MACH = HUGE(X)            ! largest magnitude.
        CASE (3)
          R1MACH = B**(-DIGITS(X))    ! smallest relative spacing.
        CASE (4)
          R1MACH = B**(1-DIGITS(X))   ! largest relative spacing.
        CASE (5)
          R1MACH = LOG10(B)
        CASE DEFAULT
          STOP 'R1MACH -- input argument out of bounds'
      END SELECT

      RETURN
      END FUNCTION R1MACH


      DOUBLE PRECISION FUNCTION D1MACH (I)
c
c   D1MACH can be used to obtain machine-dependent parameters for
c   double precision numbers.  The results for various values of I are:
c
c   D1MACH(1) = B**(EMIN-1), the smallest positive magnitude.
c   D1MACH(2) = B**EMAX*(1 - B**(-T)), the largest magnitude.
c   D1MACH(3) = B**(-T), the smallest relative spacing.
c   D1MACH(4) = B**(1-T), the largest relative spacing.
c   D1MACH(5) = LOG10(B)
c
c   Assume double precision numbers are represented in the T-digit,
c   base-B form
c
c        sign (B**E)*( (X(1)/B) + ... + (X(T)/B**T) )
c
c   where 0 <= X(I) < B for I=1,...,T; 0 < X(1); and EMIN <= E <= EMAX.
c
c   The values of B, T, EMIN and EMAX are provided in I1MACH as follows:
c   I1MACH(10) = B, the base.
c   I1MACH(11) = T, the number of base-B digits.
c   I1MACH(12) = EMIN, the smallest exponent E.
c   I1MACH(13) = EMAX, the largest exponent E.
c
c***REFERENCES  
c
c  P. Fox, A. Hall and N. Schryer, Framework for a portable library,
c     ACM Transactions on Mathematical Software 4, 177-188 (1978).
c
c  David Gay and Eric Grosse, Comment on Algorithm 528, Bell Labs, Murray 
c    Hill, NJ. submitted to ACM Transactions on Mathematical Software,
c    August 1996. 
c
c***REVISION HISTORY  (YYMMDD)
c   790101  DATE WRITTEN
c   960329  Modified for Fortran 90 (BE after suggestions by Eric Grosse)      
c --------------------------------------------------------------------

      IMPLICIT NONE
      INTEGER :: I
      DOUBLE PRECISION :: B, X = 1.D0

      B = RADIX(X)

      SELECT CASE (I)
        CASE (1)
          D1MACH = TINY(X)            ! smallest positive magnitude.
        CASE (2)
          D1MACH = HUGE(X)            ! largest magnitude.
        CASE (3)
          D1MACH = B**(-DIGITS(X))    ! smallest relative spacing.
        CASE (4)
          D1MACH = B**(1-DIGITS(X))   ! largest relative spacing.
        CASE (5)
          D1MACH = LOG10(B)
        CASE DEFAULT
          STOP 'D1MACH -- input arg out of bounds'
      END SELECT

      RETURN
      END FUNCTION D1MACH


      INTEGER FUNCTION I1MACH (I)
c
c   I1MACH can be used to obtain machine-dependent parameters for the
c   local machine environment.  The results for various values of I are:
c
c   I/O unit numbers (**MAY REQUIRE USER CUSTOMIZATION**):
c     I1MACH( 1) = the standard input unit.
c     I1MACH( 2) = the standard output unit.
c     I1MACH( 3) = the standard punch unit (obsolete, will cause error)
c     I1MACH( 4) = the standard error message unit.
c                  (the error message unit is usually 0 in UNIX systems)
c
c   Words:
c     I1MACH( 5) = the number of bits per integer storage unit.
c     I1MACH( 6) = the number of characters per integer storage unit.
c                  (obsolete, will cause an error)
c
c   Integers:
c     assume integers are represented in the S-digit, base-A form
c
c          sign ( X(S-1)*A**(S-1) + ... + X(1)*A + X(0) )
c
c     where 0 <= X(I) < A for I=0,...,S-1.
c
c     I1MACH( 7) = A, the base.
c     I1MACH( 8) = S, the number of base-A digits.
c     I1MACH( 9) = A**S - 1, the largest magnitude.
c
c   Floating-Point Numbers:
c     Assume floating-point numbers are represented in the T-digit,
c     base-B form
c                sign (B**E)*( (X(1)/B) + ... + (X(T)/B**T) )
c
c     where 0 <= X(I) .LT. B for I=1,...,T; 0 < X(1); and EMIN <= E <= EMAX.
c
c     I1MACH(10) = B, the base.
c
c   Single-Precision:
c     I1MACH(11) = T, the number of base-B digits.
c     I1MACH(12) = EMIN, the smallest exponent E.
c     I1MACH(13) = EMAX, the largest exponent E.
c
c   Double-Precision:
c     I1MACH(14) = T, the number of base-B digits.
c     I1MACH(15) = EMIN, the smallest exponent E.
c     I1MACH(16) = EMAX, the largest exponent E.
c
c***REFERENCES  
c
c  P. Fox, A. Hall and N. Schryer, Framework for a portable library,
c     ACM Transactions on Mathematical Software 4, 177-188 (1978).
c
c  David Gay and Eric Grosse, Comment on Algorithm 528, Bell Labs, Murray 
c    Hill, NJ. submitted to ACM Transactions on Mathematical Software,
c    August 1996. 
c
c***REVISION HISTORY  (YYMMDD)
c   750101  DATE WRITTEN
c   960411  Modified for Fortran 90 (BE after suggestions by Eric Grosse)    
c --------------------------------------------------------------------

      use params, only: kr
      IMPLICIT NONE
      INTEGER :: I
      REAL(KR) :: X_single  = 1.0
      DOUBLE PRECISION :: X_double = 1.D0

      SELECT CASE (I)
        CASE (1)
          I1MACH = 5 ! Input unit
        CASE (2)
          I1MACH = 6 ! Output unit
        CASE (3)
          STOP 'I1MACH: input arg = 3 is obsolete'
        CASE (4)
          I1MACH = 0 ! Error message unit
        CASE (5)
          I1MACH = BIT_SIZE(I)
        CASE (6)
          STOP 'I1MACH: input arg = 6 is obsolete'
        CASE (7)
          I1MACH = RADIX(1)
        CASE (8)
          I1MACH = BIT_SIZE(I) - 1
        CASE (9)
          I1MACH = HUGE(1)
        CASE (10)
          I1MACH = RADIX(X_single)
        CASE (11)
          I1MACH = DIGITS(X_single)
        CASE (12)
          I1MACH = MINEXPONENT(X_single)
        CASE (13)
          I1MACH = MAXEXPONENT(X_single)
        CASE (14)
          I1MACH = DIGITS(X_double)
        CASE (15)
          I1MACH = MINEXPONENT(X_double)
        CASE (16)
          I1MACH = MAXEXPONENT(X_double) 
        CASE DEFAULT
          STOP 'I1MACH: input argument out of bounds'
      END SELECT

      RETURN
      END FUNCTION I1MACH

c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
c RCS version control information:
c $Header: /home/paul/rt/sbrt/RCS/disutil.f,v 1.7 2002/08/14 22:56:35 paul Exp $
c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      subroutine  errmsg(msgnum,messag)

c        print out a warning or error messages to file disort_warning.??
c
c input:
c  msgnum    message number, used to identify whether a given message
c            has been issued more than once (0<msgnum<=nmsg)
c            or to idicate a fatal error (msgnum=0)
c
c  messag    a warning or fatal error message (character string)

      implicit none
      integer, parameter :: nmsg=20
      character*(*) messag
      character (len=2)  :: num
      character (len=132) :: line
      integer ::    msgnum
      integer,save :: msgset(nmsg)=0
      logical,save :: first=.true.

      if(msgnum.eq.0) then
        line='ERROR  >>>>>>'
      else
        if(msgset(msgnum).eq.1) return
        line='WARNING >>>>>'
      endif

      write(num,'(i2.2)') msgnum

      open(16,file='SBDART_WARNING.'//num,status='unknown',
     &     form='formatted')    
  
      write(16,'(a,1x,a)') trim(line),messag
      write(16,'(/70("#")/)')

      do 
        read(11,'(a)',end=100) line
        write(16,'(a)') trim(line)
      enddo
 100  continue

      rewind 11
      close(16)

      if(msgnum.eq.0) stop
      msgset(msgnum)=1
      return
      end

      LOGICAL FUNCTION  WrtBad ( VarNam )

c          Write names of erroneous variables and return 'TRUE'

c      INPUT :   VarNam = Name of erroneous variable to be written
c                         ( CHARACTER, any length )

      implicit none
      CHARACTER*(*)  VarNam
      INTEGER        MaxMsg, NumMsg
      SAVE  NumMsg, MaxMsg
      DATA  NumMsg / 0 /,  MaxMsg / 50 /


      WrtBad = .TRUE.
      NumMsg = NumMsg + 1
      WRITE ( *, '(3A)' )  ' ****  Input variable  ', VarNam,
     &                     '  in error  ****'
      IF ( NumMsg.EQ.MaxMsg )
     &   call  errmsg (12,'Too many input errors.  Aborting...')

      RETURN
      END

      LOGICAL FUNCTION  WrtDim ( DimNam, MinVal )

c          Write name of too-small symbolic dimension and
c          the value it should be increased to;  return 'TRUE'

c      INPUT :  DimNam = Name of symbolic dimension which is too small
c                        ( CHARACTER, any length )
c               Minval = Value to which that dimension should be
c                        increased (at least)

      implicit none
      CHARACTER*(*)  DimNam
      INTEGER        MinVal


      WRITE ( *, '(/,3A,I7)' )  ' ****  Symbolic dimension  ', DimNam,
     &                     '  should be increased to at least ', MinVal
      WrtDim = .TRUE.

      RETURN
      END

      LOGICAL FUNCTION  TstBad( VarNam, RelErr )

c       Write name (VarNam) of variable failing self-test and its
c       percent error from the correct value;  return  'FALSE'.

      use params, only: kr
      CHARACTER*(*)  VarNam
      REAL(KR)           RelErr


      TstBad = .FALSE.
      WRITE( *, '(/,3A,1P,E11.2,A)' )
     &       ' Output variable ', VarNam,' differed by ', 100.*RelErr,
     &       ' per cent from correct value.  Self-test failed.'

      RETURN
      END

c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
c RCS version control information:
c $Header: /home/paul/rt/sbrt/RCS/disutil.f,v 1.7 2002/08/14 22:56:35 paul Exp $
c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

c Call tree:
c
c    SGBCO
c       SASUM
c       SDOT
c       SAXPY
c       SGBFA
c           ISAMAX
c           SAXPY
c           SSCAL
c       SSCAL
c   SGBSL
c       SDOT
c       SAXPY
c   SGECO
c       SASUM
c       SDOT
c       SAXPY
c       SGEFA
c           ISAMAX
c           SAXPY
c           SSCAL
c       SSCAL
c   SGESL
c       SDOT
c       SAXPY
c   SSWAP
c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


      SUBROUTINE SGBCO( ABD, LDA, N, ML, MU, IPVT, RCOND, Z )

c         Factors a real band matrix by Gaussian elimination
c         and estimates the condition of the matrix.

c         Revision date:  8/1/82
c         Author:  Moler, C. B. (U. of New Mexico)

c     If  RCOND  is not needed, SGBFA is slightly faster.
c     To solve  A*X = B , follow SBGCO by SGBSL.

c     input:

C        ABD     REAL(LDA, N)
c                contains the matrix in band storage.  The columns
c                of the matrix are stored in the columns of  ABD  and
c                the diagonals of the matrix are stored in rows
c                ML+1 through 2*ML+MU+1 of  ABD .
c                See the comments below for details.

C        LDA     INTEGER
c                the leading dimension of the array  ABD .
c                LDA must be .GE. 2*ML + MU + 1 .

C        N       INTEGER
c                the order of the original matrix.

C        ML      INTEGER
c                number of diagonals below the main diagonal.
c                0 .LE. ML .LT. N .

C        MU      INTEGER
c                number of diagonals above the main diagonal.
c                0 .LE. MU .LT. N .
c                more efficient if  ML .LE. MU .

c     on return

c        ABD     an upper triangular matrix in band storage and
c                the multipliers which were used to obtain it.
c                The factorization can be written  A = L*U  where
c                L  is a product of permutation and unit lower
c                triangular matrices and  U  is upper triangular.

C        IPVT    INTEGER(N)
c                an integer vector of pivot indices.

C        RCOND   REAL
c                an estimate of the reciprocal condition of  A .
c                For the system  A*X = B , relative perturbations
c                in  A  and  B  of size  epsilon  may cause
c                relative perturbations in  X  of size  epsilon/RCOND .
c                If  RCOND  is so small that the logical expression
c                           1.0 + RCOND .EQ. 1.0
c                is true, then  A  may be singular to working
c                precision.  In particular,  RCOND  is zero  if
c                exact singularity is detected or the estimate
c                underflows.

C        Z       REAL(N)
c                a work vector whose contents are usually unimportant.
c                If  A  is close to a singular matrix, then  Z  is
c                an approximate null vector in the sense that
c                norm(a*z) = rcond*norm(a)*norm(z) .

c     Band storage

c           If  A  is a band matrix, the following program segment
c           will set up the input.

c                   ML = (band width below the diagonal)
c                   MU = (band width above the diagonal)
c                   M = ML + MU + 1
c                   DO 20 J = 1, N
c                      I1 = MAX(1, J-MU)
c                      I2 = MIN(N, J+ML)
c                      DO 10 I = I1, I2
c                         K = I - J + M
c                         ABD(K,J) = A(I,J)
c                10    CONTINUE
c                20 CONTINUE

c           This uses rows  ML+1  through  2*ML+MU+1  of  ABD .
c           In addition, the first  ML  rows in  ABD  are used for
c           elements generated during the triangularization.
c           The total number of rows needed in  ABD  is  2*ML+MU+1 .
c           The  ML+MU by ML+MU  upper left triangle and the
c           ML by ML  lower right triangle are not referenced.

c     Example:  if the original matrix is

c           11 12 13  0  0  0
c           21 22 23 24  0  0
c            0 32 33 34 35  0
c            0  0 43 44 45 46
c            0  0  0 54 55 56
c            0  0  0  0 65 66

c      then  N = 6, ML = 1, MU = 2, LDA .GE. 5  and ABD should contain

c            *  *  *  +  +  +  , * = not used
c            *  * 13 24 35 46  , + = used for pivoting
c            * 12 23 34 45 56
c           11 22 33 44 55 66
c           21 32 43 54 65  *

c --------------------------------------------------------------------


c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   LDA, ML, MU, N
      REAL(KR)      RCOND
c     ..
c     .. Array Arguments ..

      INTEGER   IPVT( * )
      REAL(KR)      ABD( LDA, * ), Z( * )
c     ..
c     .. Local Scalars ..

      INTEGER   INFO, IS, J, JU, K, KB, KP1, L, LA, LM, LZ, M, MM
      REAL(KR)      ANORM, EK, S, SM, T, WK, WKM, YNORM
c     ..
c     .. External Functions ..

      REAL(KR)      SASUM, SDOT
      EXTERNAL  SASUM, SDOT
c     ..
c     .. External Subroutines ..

      EXTERNAL  SAXPY, SGBFA, SSCAL
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC ABS, MAX, MIN, SIGN
c     ..


c                       ** compute 1-norm of A
      ANORM  = 0.0E0
      L  = ML + 1
      IS = L + MU

      DO 10 J = 1, N

         ANORM  = MAX( ANORM, SASUM( L,ABD( IS,J ),1 ) )

         IF( IS.GT.ML + 1 ) IS = IS - 1

         IF( J.LE.MU ) L  = L + 1

         IF( J.GE.N - ML ) L  = L - 1

   10 CONTINUE
c                                               ** factor

      CALL SGBFA( ABD, LDA, N, ML, MU, IPVT, INFO )

c     RCOND = 1/(norm(A)*(estimate of norm(inverse(A)))) .
c     estimate = norm(Z)/norm(Y) where  A*Z = Y  and  trans(A)*Y = E.
c     trans(A) is the transpose of A.  The components of E  are
c     chosen to cause maximum local growth in the elements of W  where
c     trans(U)*W = E.  The vectors are frequently rescaled to avoid
c     overflow.

c                     ** solve trans(U)*W = E
      EK = 1.0E0

      DO 20 J = 1, N
         Z( J ) = 0.0E0
   20 CONTINUE


      M  = ML + MU + 1
      JU = 0

      DO 50 K = 1, N

         IF( Z( K ).NE.0.0E0 ) EK = SIGN( EK, -Z( K ) )

         IF( ABS( EK - Z( K ) ).GT.ABS( ABD( M,K ) ) ) THEN

            S  = ABS( ABD( M,K ) ) / ABS( EK - Z( K ) )

            CALL SSCAL( N, S, Z, 1 )

            EK = S*EK

         END IF

         WK   = EK - Z( K )
         WKM  = -EK - Z( K )
         S    = ABS( WK )
         SM   = ABS( WKM )

         IF( ABD( M,K ).NE.0.0E0 ) THEN

            WK   = WK / ABD( M, K )
            WKM  = WKM / ABD( M, K )

         ELSE

            WK   = 1.0E0
            WKM  = 1.0E0

         END IF

         KP1  = K + 1
         JU   = MIN( MAX( JU,MU + IPVT( K ) ), N )
         MM   = M

         IF( KP1.LE.JU ) THEN

            DO 30 J = KP1, JU
               MM     = MM - 1
               SM     = SM + ABS( Z( J ) + WKM*ABD( MM,J ) )
               Z( J ) = Z( J ) + WK*ABD( MM, J )
               S      = S + ABS( Z( J ) )
   30       CONTINUE

            IF( S.LT.SM ) THEN

               T  = WKM - WK
               WK = WKM
               MM = M

               DO 40 J = KP1, JU
                  MM = MM - 1
                  Z( J ) = Z( J ) + T*ABD( MM, J )
   40          CONTINUE

            END IF

         END IF

         Z( K ) = WK

   50 CONTINUE


      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )

c                         ** solve trans(L)*Y = W
      DO 60 KB = 1, N
         K  = N + 1 - KB
         LM = MIN( ML, N - K )

         IF( K.LT.N )
     &       Z( K ) = Z( K ) + SDOT( LM, ABD( M+1, K ), 1, Z( K+1 ), 1 )

         IF( ABS( Z( K ) ).GT.1.0E0 ) THEN

            S  = 1.0E0 / ABS( Z( K ) )

            CALL SSCAL( N, S, Z, 1 )

         END IF

         L      = IPVT( K )
         T      = Z( L )
         Z( L ) = Z( K )
         Z( K ) = T

   60 CONTINUE


      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )

      YNORM  = 1.0E0
c                         ** solve L*V = Y
      DO 70 K = 1, N

         L      = IPVT( K )
         T      = Z( L )
         Z( L ) = Z( K )
         Z( K ) = T
         LM     = MIN( ML, N - K )

         IF( K.LT.N )
     &       CALL SAXPY( LM, T, ABD( M+1, K ), 1, Z( K+1 ), 1 )

         IF( ABS( Z(K) ).GT.1.0E0 ) THEN

            S  = 1.0E0 / ABS( Z(K) )

            CALL SSCAL( N, S, Z, 1 )

            YNORM  = S*YNORM

         END IF

   70 CONTINUE


      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )

      YNORM  = S*YNORM

c                           ** solve  U*Z = W
      DO 80 KB = 1, N

         K  = N + 1 - KB

         IF( ABS( Z( K ) ).GT.ABS( ABD( M,K ) ) ) THEN

            S  = ABS( ABD( M,K ) ) / ABS( Z( K ) )

            CALL SSCAL( N, S, Z, 1 )

            YNORM  = S*YNORM

         END IF

         IF( ABD( M,K ).NE.0.0E0 ) Z( K ) = Z( K ) / ABD( M, K )
         IF( ABD( M,K ).EQ.0.0E0 ) Z( K ) = 1.0E0

         LM = MIN( K, M ) - 1
         LA = M - LM
         LZ = K - LM
         T  = -Z( K )

         CALL SAXPY( LM, T, ABD( LA,K ), 1, Z( LZ ), 1 )

   80 CONTINUE
c                              ** make znorm = 1.0

      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )

      YNORM  = S*YNORM
      IF( ANORM.NE.0.0E0 ) RCOND  = YNORM / ANORM
      IF( ANORM.EQ.0.0E0 ) RCOND  = 0.0E0

      END

      SUBROUTINE SGBFA( ABD, LDA, N, ML, MU, IPVT, INFO )

c         Factors a real band matrix by elimination.

c         Revision date:  8/1/82
c         Author:  Moler, C. B. (U. of New Mexico)

c     SGBFA is usually called by SBGCO, but it can be called
c     directly with a saving in time if  RCOND  is not needed.

c     Input:  same as SGBCO

c     On return:

c        ABD,IPVT    same as SGBCO

c        INFO    INTEGER
c                = 0  normal value.
c                = k  if  u(k,k) .eq. 0.0 .  This is not an error
c                     condition for this subroutine, but it does
c                     indicate that SGBSL will divide by zero if
c                     called.  Use  RCOND  in SBGCO for a reliable
c                     indication of singularity.

c     (see SGBCO for description of band storage mode)

c ----------------------------------------------------------------


c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INFO, LDA, ML, MU, N
c     ..
c     .. Array Arguments ..

      INTEGER   IPVT( * )
      REAL(KR)      ABD( LDA, * )
c     ..
c     .. Local Scalars ..

      INTEGER   I, I0, J, J0, J1, JU, JZ, K, KP1, L, LM, M, MM, NM1
      REAL(KR)      T
c     ..
c     .. External Functions ..

      INTEGER   ISAMAX
      EXTERNAL  ISAMAX
c     ..
c     .. External Subroutines ..

      EXTERNAL  SAXPY, SSCAL
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC MAX, MIN
c     ..


      M    = ML + MU + 1
      INFO = 0
c                        ** zero initial fill-in columns
      J0 = MU + 2
      J1 = MIN( N, M ) - 1

      DO 20 JZ = J0, J1

         I0 = M + 1 - JZ

         DO 10 I = I0, ML
            ABD( I, JZ ) = 0.0E0
   10    CONTINUE

   20 CONTINUE

      JZ = J1
      JU = 0
c                       ** Gaussian elimination with partial pivoting
      NM1  = N - 1

      DO 50 K = 1, NM1

         KP1 = K + 1
c                                  ** zero next fill-in column
         JZ = JZ + 1

         IF( JZ.LE.N ) THEN

            DO 30 I = 1, ML
               ABD( I, JZ ) = 0.0E0
   30       CONTINUE

         END IF
c                                  ** find L = pivot index
         LM  = MIN( ML, N - K )
         L   = ISAMAX( LM + 1, ABD( M, K ), 1 ) + M - 1
         IPVT( K ) = L + K - M

         IF( ABD( L,K ).EQ.0.0E0 ) THEN
c                                      ** zero pivot implies this column
c                                      ** already triangularized
            INFO = K

         ELSE
c                                ** interchange if necessary
            IF( L.NE.M ) THEN

               T           = ABD( L, K )
               ABD( L, K ) = ABD( M, K )
               ABD( M, K ) = T
            END IF
c                                      ** compute multipliers
            T  = - 1.0E0 / ABD( M, K )

            CALL SSCAL( LM, T, ABD( M + 1,K ), 1 )

c                               ** row elimination with column indexing

            JU = MIN( MAX( JU,MU + IPVT( K ) ), N )
            MM = M

            DO 40 J = KP1, JU

               L  = L - 1
               MM = MM - 1
               T  = ABD( L, J )

               IF( L.NE.MM ) THEN

                  ABD( L, J ) = ABD( MM, J )
                  ABD( MM, J ) = T

               END IF

               CALL SAXPY( LM, T, ABD( M+1, K ), 1, ABD( MM+1, J ), 1)

   40       CONTINUE

         END IF

   50 CONTINUE


      IPVT( N ) = N
      IF( ABD( M,N ).EQ.0.0E0 ) INFO = N

      END

      SUBROUTINE SGBSL( ABD, LDA, N, ML, MU, IPVT, B, JOB )

c         Solves the real band system
c            A * X = B  or  transpose(A) * X = B
c         using the factors computed by SBGCO or SGBFA.

c         Revision date:  8/1/82
c         Author:  Moler, C. B. (U. of New Mexico)

c     Input:

C        ABD     REAL(LDA, N)
c                the output from SBGCO or SGBFA.

C        LDA     INTEGER
c                the leading dimension of the array  ABD .

C        N       INTEGER
c                the order of the original matrix.

C        ML      INTEGER
c                number of diagonals below the main diagonal.

C        MU      INTEGER
c                number of diagonals above the main diagonal.

C        IPVT    INTEGER(N)
c                the pivot vector from SBGCO or SGBFA.

C        B       REAL(N)
c                the right hand side vector.

C        JOB     INTEGER
c                = 0         to solve  A*X = B ,
c                = nonzero   to solve  transpose(A)*X = B

c     On return

c        B       the solution vector  X

c     Error condition

c        A division by zero will occur if the input factor contains a
c        zero on the diagonal.  Technically, this indicates singularity,
c        but it is often caused by improper arguments or improper
c        setting of LDA .  It will not occur if the subroutines are
c        called correctly and if SBGCO has set RCOND .GT. 0.0
c        or SGBFA has set INFO .EQ. 0 .

c     To compute  inverse(a) * c  where  c  is a matrix
c     with  p  columns
c           call sgbco(abd,lda,n,ml,mu,ipvt,rcond,z)
c           if (rcond is too small) go to ...
c           do 10 j = 1, p
c              call sgbsl(abd,lda,n,ml,mu,ipvt,c(1,j),0)
c        10 continue

c --------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   JOB, LDA, ML, MU, N
c     ..
c     .. Array Arguments ..

      INTEGER   IPVT( * )
      REAL(KR)      ABD( LDA, * ), B( * )
c     ..
c     .. Local Scalars ..

      INTEGER   K, KB, L, LA, LB, LM, M, NM1
      REAL(KR)      T
c     ..
c     .. External Functions ..

      REAL(KR)      SDOT
      EXTERNAL  SDOT
c     ..
c     .. External Subroutines ..

      EXTERNAL  SAXPY
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC MIN
c     ..


      M   = MU + ML + 1
      NM1 = N - 1

      IF( JOB.EQ.0 ) THEN
c                           ** solve  A * X = B

c                               ** first solve L*Y = B
         IF( ML.NE.0 ) THEN

            DO 10 K = 1, NM1

               LM = MIN( ML, N - K )
               L  = IPVT( K )
               T  = B( L )

               IF( L.NE.K ) THEN

                  B( L ) = B( K )
                  B( K ) = T

               END IF

               CALL SAXPY( LM, T, ABD( M + 1,K ), 1, B( K + 1 ), 1 )

   10       CONTINUE

         END IF

c                           ** now solve  U*X = Y
         DO 20 KB = 1, N

            K      = N + 1 - KB
            B( K ) = B( K ) / ABD( M, K )
            LM     = MIN( K, M ) - 1
            LA     = M - LM
            LB     = K - LM
            T      = -B( K )

            CALL SAXPY( LM, T, ABD( LA,K ), 1, B( LB ), 1 )

   20    CONTINUE


      ELSE
c                          ** solve  trans(A) * X = B

c                                  ** first solve  trans(U)*Y = B
         DO 30 K = 1, N

            LM     = MIN( K, M ) - 1
            LA     = M - LM
            LB     = K - LM
            T      = SDOT( LM, ABD( LA,K ), 1, B( LB ), 1 )
            B( K ) = ( B( K ) - T ) / ABD( M, K )

   30    CONTINUE

c                                  ** now solve trans(L)*X = Y
         IF( ML.NE.0 ) THEN

            DO 40 KB = 1, NM1

               K      = N - KB
               LM     = MIN( ML, N - K )
               B( K ) = B( K ) + SDOT( LM, ABD( M+1, K ), 1,
     &                                 B( K+1 ), 1 )
               L      = IPVT( K )

               IF( L.NE.K ) THEN

                  T    = B( L )
                  B( L ) = B( K )
                  B( K ) = T

               END IF

   40       CONTINUE

         END IF

      END IF

      END

      SUBROUTINE SGECO( A, LDA, N, IPVT, RCOND, Z )

c         Factors a real matrix by Gaussian elimination
c         and estimates the condition of the matrix.

c         Revision date:  8/1/82
c         Author:  Moler, C. B. (U. of New Mexico)

c         If  RCOND  is not needed, SGEFA is slightly faster.
c         To solve  A*X = B , follow SGECO by SGESL.

c     On entry

c        A       REAL(LDA, N)
c                the matrix to be factored.

c        LDA     INTEGER
c                the leading dimension of the array  A .

c        N       INTEGER
c                the order of the matrix  A .

c     On return

c        A       an upper triangular matrix and the multipliers
c                which were used to obtain it.
c                The factorization can be written  A = L*U , where
c                L  is a product of permutation and unit lower
c                triangular matrices and  U  is upper triangular.

c        IPVT    INTEGER(N)
c                an integer vector of pivot indices.

c        RCOND   REAL
c                an estimate of the reciprocal condition of  A .
c                For the system  A*X = B , relative perturbations
c                in  A  and  B  of size  epsilon  may cause
c                relative perturbations in  X  of size  epsilon/RCOND .
c                If  RCOND  is so small that the logical expression
c                           1.0 + RCOND .EQ. 1.0
c                is true, then  A  may be singular to working
c                precision.  In particular,  RCOND  is zero  if
c                exact singularity is detected or the estimate
c                underflows.

C        Z       REAL(N)
c                a work vector whose contents are usually unimportant.
c                If  A  is close to a singular matrix, then  Z  is
c                an approximate null vector in the sense that
c                norm(A*Z) = RCOND*norm(A)*norm(Z) .

c ------------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   LDA, N
      REAL(KR)      RCOND
c     ..
c     .. Array Arguments ..

      INTEGER   IPVT( * )
      REAL(KR)      A( LDA, * ), Z( * )
c     ..
c     .. Local Scalars ..

      INTEGER   INFO, J, K, KB, KP1, L
      REAL(KR)      ANORM, EK, S, SM, T, WK, WKM, YNORM
c     ..
c     .. External Functions ..

      REAL(KR)      SASUM, SDOT
      EXTERNAL  SASUM, SDOT
c     ..
c     .. External Subroutines ..

      EXTERNAL  SAXPY, SGEFA, SSCAL
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC ABS, MAX, SIGN
c     ..


c                        ** compute 1-norm of A
      ANORM  = 0.0E0
      DO 10 J = 1, N
         ANORM  = MAX( ANORM, SASUM( N,A( 1,J ),1 ) )
   10 CONTINUE
c                                      ** factor

      CALL SGEFA( A, LDA, N, IPVT, INFO )

c     RCOND = 1/(norm(A)*(estimate of norm(inverse(A)))) .
c     estimate = norm(Z)/norm(Y) where  A*Z = Y  and  trans(A)*Y = E .
c     trans(A) is the transpose of A.  The components of E  are
c     chosen to cause maximum local growth in the elements of W  where
c     trans(U)*W = E.  The vectors are frequently rescaled to avoid
c     overflow.

c                        ** solve trans(U)*W = E
      EK = 1.0E0

      DO 20 J = 1, N
         Z( J ) = 0.0E0
   20 CONTINUE


      DO 50 K = 1, N

         IF( Z( K ).NE.0.0E0 ) EK = SIGN( EK, -Z( K ) )

         IF( ABS( EK - Z( K ) ).GT.ABS( A( K,K ) ) ) THEN

            S  = ABS( A( K,K ) ) / ABS( EK - Z( K ) )

            CALL SSCAL( N, S, Z, 1 )

            EK = S*EK

         END IF

         WK   = EK - Z( K )
         WKM  = -EK - Z( K )
         S    = ABS( WK )
         SM   = ABS( WKM )

         IF( A( K,K ).NE.0.0E0 ) THEN

            WK   = WK / A( K, K )
            WKM  = WKM / A( K, K )

         ELSE

            WK   = 1.0E0
            WKM  = 1.0E0

         END IF

         KP1  = K + 1

         IF( KP1.LE.N ) THEN

            DO 30 J = KP1, N
               SM     = SM + ABS( Z( J ) + WKM*A( K,J ) )
               Z( J ) = Z( J ) + WK*A( K, J )
               S      = S + ABS( Z( J ) )
   30       CONTINUE

            IF( S.LT.SM ) THEN

               T  = WKM - WK
               WK = WKM

               DO 40 J = KP1, N
                  Z( J ) = Z( J ) + T*A( K, J )
   40          CONTINUE

            END IF

         END IF

         Z( K ) = WK

   50 CONTINUE


      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )
c                                ** solve trans(L)*Y = W
      DO 60 KB = 1, N
         K  = N + 1 - KB

         IF( K.LT.N )
     &       Z( K ) = Z( K ) + SDOT( N - K, A( K+1, K ), 1, Z( K+1 ), 1)

         IF( ABS( Z( K ) ).GT.1.0E0 ) THEN

            S  = 1.0E0 / ABS( Z( K ) )

            CALL SSCAL( N, S, Z, 1 )

         END IF

         L      = IPVT( K )
         T      = Z( L )
         Z( L ) = Z( K )
         Z( K ) = T
   60 CONTINUE


      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )
c                                 ** solve L*V = Y
      YNORM  = 1.0E0

      DO 70 K = 1, N
         L      = IPVT( K )
         T      = Z( L )
         Z( L ) = Z( K )
         Z( K ) = T

         IF( K.LT.N ) CALL SAXPY( N - K, T, A( K + 1,K ), 1, Z( K + 1 ),
     &                            1 )

         IF( ABS( Z( K ) ).GT.1.0E0 ) THEN

            S  = 1.0E0 / ABS( Z( K ) )

            CALL SSCAL( N, S, Z, 1 )

            YNORM  = S*YNORM
         END IF

   70 CONTINUE


      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )
c                                  ** solve  U*Z = V
      YNORM  = S*YNORM

      DO 80 KB = 1, N

         K  = N + 1 - KB

         IF( ABS( Z( K ) ).GT.ABS( A( K,K ) ) ) THEN

            S  = ABS( A( K,K ) ) / ABS( Z( K ) )

            CALL SSCAL( N, S, Z, 1 )

            YNORM  = S*YNORM

         END IF

         IF( A( K,K ).NE.0.0E0 ) Z( K ) = Z( K ) / A( K, K )

         IF( A( K,K ).EQ.0.0E0 ) Z( K ) = 1.0E0

         T  = -Z( K )

         CALL SAXPY( K - 1, T, A( 1,K ), 1, Z( 1 ), 1 )

   80 CONTINUE
c                                   ** make znorm = 1.0
      S  = 1.0E0 / SASUM( N, Z, 1 )

      CALL SSCAL( N, S, Z, 1 )

      YNORM  = S*YNORM

      IF( ANORM.NE.0.0E0 ) RCOND = YNORM / ANORM
      IF( ANORM.EQ.0.0E0 ) RCOND = 0.0E0

      END

      SUBROUTINE SGEFA( A, LDA, N, IPVT, INFO )

c         Factors a real matrix by Gaussian elimination.

c         Revision date:  8/1/82
c         Author:  Moler, C. B. (U. of New Mexico)

c     SGEFA is usually called by SGECO, but it can be called
c     directly with a saving in time if  RCOND  is not needed.
c     (time for SGECO) = (1 + 9/N) * (time for SGEFA) .

c     Input:  same as SGECO

c     On return:

c        A,IPVT  same as SGECO

c        INFO    INTEGER
c                = 0  normal value.
c                = k  if  u(k,k) .eq. 0.0 .  This is not an error
c                     condition for this subroutine, but it does
c                     indicate that SGESL or SGEDI will divide by zero
c                     if called.  Use  RCOND  in SGECO for a reliable
c                     indication of singularity.

c ---------------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INFO, LDA, N
c     ..
c     .. Array Arguments ..

      INTEGER   IPVT( * )
      REAL(KR)      A( LDA, * )
c     ..
c     .. Local Scalars ..

      INTEGER   J, K, KP1, L, NM1
      REAL(KR)      T
c     ..
c     .. External Functions ..

      INTEGER   ISAMAX
      EXTERNAL  ISAMAX
c     ..
c     .. External Subroutines ..

      EXTERNAL  SAXPY, SSCAL
c     ..


c                      ** Gaussian elimination with partial pivoting
      INFO = 0
      NM1  = N - 1

      DO 20 K = 1, NM1

         KP1  = K + 1
c                                            ** find L = pivot index

         L  = ISAMAX( N - K + 1, A( K,K ), 1 ) + K - 1
         IPVT( K ) = L

         IF( A( L,K ).EQ.0.0E0 ) THEN
c                                     ** zero pivot implies this column
c                                     ** already triangularized
            INFO = K

         ELSE
c                                     ** interchange if necessary
            IF( L.NE.K ) THEN

               T         = A( L, K )
               A( L, K ) = A( K, K )
               A( K, K ) = T

            END IF
c                                     ** compute multipliers
            T  = -1.0E0 / A( K, K )

            CALL SSCAL( N - K, T, A( K + 1,K ), 1 )

c                              ** row elimination with column indexing
            DO 10 J = KP1, N

               T  = A( L, J )

               IF( L.NE.K ) THEN

                  A( L, J ) = A( K, J )
                  A( K, J ) = T

               END IF

               CALL SAXPY( N-K, T, A( K+1, K ), 1, A( K+1, J ), 1 )

   10       CONTINUE

         END IF

   20 CONTINUE


      IPVT( N ) = N
      IF( A( N,N ) .EQ. 0.0E0 ) INFO = N

      END

      SUBROUTINE SGESL( A, LDA, N, IPVT, B, JOB )

c         Solves the real system
c            A * X = B  or  transpose(A) * X = B
c         using the factors computed by SGECO or SGEFA.

c         Revision date:  8/1/82
c         Author:  Moler, C. B. (U. of New Mexico)

c     On entry

c        A       REAL(LDA, N)
c                the output from SGECO or SGEFA.

c        LDA     INTEGER
c                the leading dimension of the array  A

c        N       INTEGER
c                the order of the matrix  A

c        IPVT    INTEGER(N)
c                the pivot vector from SGECO or SGEFA.

c        B       REAL(N)
c                the right hand side vector.

c        JOB     INTEGER
c                = 0         to solve  A*X = B ,
c                = nonzero   to solve  transpose(A)*X = B

c     On return

c        B       the solution vector  X

c     Error condition

c        A division by zero will occur if the input factor contains a
c        zero on the diagonal.  Technically, this indicates singularity,
c        but it is often caused by improper arguments or improper
c        setting of LDA.  It will not occur if the subroutines are
c        called correctly and if SGECO has set RCOND .GT. 0.0
c        or SGEFA has set INFO .EQ. 0 .

c     To compute  inverse(a) * c  where  c  is a matrix
c     with  p  columns
c           call sgeco(a,lda,n,ipvt,rcond,z)
c           if (rcond is too small) go to ...
c           do 10 j = 1, p
c              call sgesl(a,lda,n,ipvt,c(1,j),0)
c        10 continue

c ---------------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   JOB, LDA, N
c     ..
c     .. Array Arguments ..

      INTEGER   IPVT( * )
      REAL(KR)      A( LDA, * ), B( * )
c     ..
c     .. Local Scalars ..

      INTEGER   K, KB, L, NM1
      REAL(KR)      T
c     ..
c     .. External Functions ..

      REAL(KR)      SDOT
      EXTERNAL  SDOT
c     ..
c     .. External Subroutines ..

      EXTERNAL  SAXPY
c     ..


      NM1  = N - 1

      IF( JOB.EQ.0 ) THEN
c                                 ** solve  A * X = B

c                                     ** first solve  L*Y = B
         DO 10 K = 1, NM1

            L  = IPVT( K )
            T  = B( L )

            IF( L.NE.K ) THEN

               B( L ) = B( K )
               B( K ) = T

            END IF

            CALL SAXPY( N - K, T, A( K+1, K ), 1, B( K+1 ), 1 )

   10    CONTINUE
c                                    ** now solve  U*X = Y
         DO 20 KB = 1, N

            K      = N + 1 - KB
            B( K ) = B( K ) / A( K, K )
            T      = - B( K )

            CALL SAXPY( K-1, T, A( 1, K ), 1, B(1), 1 )

   20    CONTINUE


      ELSE
c                         ** solve  trans(A) * X = B

c                                    ** first solve  trans(U)*Y = B
         DO 30 K = 1, N

            T      = SDOT( K - 1, A( 1,K ), 1, B( 1 ), 1 )
            B( K ) = ( B( K ) - T ) / A( K, K )

   30    CONTINUE

c                                    ** now solve  trans(l)*x = y
         DO 40 KB = 1, NM1

            K      = N - KB
            B( K ) = B( K ) + SDOT( N - K, A( K+1, K ), 1, B( K+1 ), 1)
            L      = IPVT( K )

            IF( L.NE.K ) THEN

               T      = B( L )
               B( L ) = B( K )
               B( K ) = T

            END IF

   40    CONTINUE

      END IF

      END

      FUNCTION SASUM( N, SX, INCX )

c  INPUT--    N  Number of elements in vector to be summed
c            SX  Sing-prec array, length 1+(N-1)*INCX, containing vector
c          INCX  Spacing of vector elements in SX

c  OUTPUT-- SASUM   Sum from 0 to N-1 of  ABS(SX(1+I*INCX))
c ----------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INCX, N
c     ..
c     .. Array Arguments ..

      REAL(KR)      SX( * ), sasum
c     ..
c     .. Local Scalars ..

      INTEGER   I, M
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC ABS, MOD
c     ..

      SASUM  = 0.0

      IF( N.LE.0 ) RETURN

      IF( INCX.NE.1 ) THEN
c                                          ** non-unit increments
         DO 10 I = 1, 1 + ( N - 1 )*INCX, INCX
            SASUM  = SASUM + ABS( SX( I ) )
   10    CONTINUE

      ELSE
c                                          ** unit increments
         M  = MOD( N, 6 )

         IF( M.NE.0 ) THEN
c                             ** clean-up loop so remaining vector
c                             ** length is a multiple of 6.
            DO 20 I = 1, M
               SASUM  = SASUM + ABS( SX( I ) )
   20       CONTINUE

         END IF
c                              ** unroll loop for speed
         DO 30 I = M + 1, N, 6
            SASUM  = SASUM + ABS( SX( I ) ) + ABS( SX( I + 1 ) ) +
     &               ABS( SX( I + 2 ) ) + ABS( SX( I + 3 ) ) +
     &               ABS( SX( I + 4 ) ) + ABS( SX( I + 5 ) )
   30    CONTINUE

      END IF

      END

      SUBROUTINE SAXPY( N, SA, SX, INCX, SY, INCY )

c          Y = A*X + Y  (X, Y = vectors, A = scalar)

c  INPUT--
c        N  Number of elements in input vectors X and Y
c       SA  Single precision scalar multiplier A
c       SX  Sing-prec array containing vector X
c     INCX  Spacing of elements of vector X in SX
c       SY  Sing-prec array containing vector Y
c     INCY  Spacing of elements of vector Y in SY

c OUTPUT--
c       SY   For I = 0 to N-1, overwrite  SY(LY+I*INCY) with
c                 SA*SX(LX+I*INCX) + SY(LY+I*INCY),
c            where LX = 1          if INCX .GE. 0,
c                     = (-INCX)*N  if INCX .LT. 0
c            and LY is defined analogously using INCY.
c ------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INCX, INCY, N
      REAL(KR)      SA
c     ..
c     .. Array Arguments ..

      REAL(KR)      SX( * ), SY( * )
c     ..
c     .. Local Scalars ..

      INTEGER   I, IX, IY, M
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC MOD
c     ..


      IF( N.LE.0 .OR. SA.EQ.0.0 ) RETURN

      IF( INCX.EQ.INCY .AND. INCX.GT.1 ) THEN

         DO 10 I = 1, 1 + ( N - 1 )*INCX, INCX
            SY( I ) = SY( I ) + SA*SX( I )
   10    CONTINUE

      ELSE IF( INCX.EQ.INCY .AND. INCX.EQ.1 ) THEN

c                                        ** equal, unit increments
         M  = MOD( N, 4 )

         IF( M.NE.0 ) THEN
c                            ** clean-up loop so remaining vector length
c                            ** is a multiple of 4.
            DO 20 I = 1, M
               SY( I ) = SY( I ) + SA*SX( I )
   20       CONTINUE

         END IF
c                              ** unroll loop for speed
         DO 30 I = M + 1, N, 4
            SY( I ) = SY( I ) + SA*SX( I )
            SY( I + 1 ) = SY( I + 1 ) + SA*SX( I + 1 )
            SY( I + 2 ) = SY( I + 2 ) + SA*SX( I + 2 )
            SY( I + 3 ) = SY( I + 3 ) + SA*SX( I + 3 )
   30    CONTINUE


      ELSE
c               ** nonequal or nonpositive increments.
         IX = 1
         IY = 1
         IF( INCX.LT.0 ) IX = 1 + ( N - 1 )*( -INCX )
         IF( INCY.LT.0 ) IY = 1 + ( N - 1 )*( -INCY )

         DO 40 I = 1, N
            SY( IY ) = SY( IY ) + SA*SX( IX )
            IX = IX + INCX
            IY = IY + INCY
   40    CONTINUE

      END IF

      END

      FUNCTION SDOT( N, SX, INCX, SY, INCY )

c        Single-prec dot product of vectors  X  and  Y

c  INPUT--
c        N  Number of elements in input vectors X and Y
c       SX  Sing-prec array containing vector X
c     INCX  Spacing of elements of vector X in SX
c       SY  Sing-prec array containing vector Y
c     INCY  Spacing of elements of vector Y in SY

c OUTPUT--
c     SDOT   Sum for I = 0 to N-1 of  SX(LX+I*INCX) * SY(LY+I*INCY),
c            where  LX = 1          if INCX .GE. 0,
c                      = (-INCX)*N  if INCX .LT. 0,
c            and LY is defined analogously using INCY.
c ------------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INCX, INCY, N
c     ..
c     .. Array Arguments ..

      REAL(KR)      SX( * ), SY( * ), sdot
c     ..
c     .. Local Scalars ..

      INTEGER   I, IX, IY, M
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC MOD
c     ..


      SDOT = 0.0

      IF( N.LE.0 ) RETURN

      IF( INCX.EQ.INCY .AND. INCX.GT.1 ) THEN

         DO 10 I = 1, 1 + ( N - 1 )*INCX, INCX
            SDOT = SDOT + SX( I )*SY( I )
   10    CONTINUE


      ELSE IF( INCX.EQ.INCY .AND. INCX.EQ.1 ) THEN

c                                        ** equal, unit increments
         M  = MOD( N, 5 )

         IF( M.NE.0 ) THEN
c                            ** clean-up loop so remaining vector length
c                            ** is a multiple of 4.
            DO 20 I = 1, M
               SDOT = SDOT + SX( I )*SY( I )
   20       CONTINUE

         END IF
c                              ** unroll loop for speed
         DO 30 I = M + 1, N, 5
            SDOT = SDOT + SX( I )*SY( I ) + SX( I + 1 )*SY( I + 1 ) +
     &               SX( I + 2 )*SY( I + 2 ) + SX( I + 3 )*SY( I + 3 ) +
     &               SX( I + 4 )*SY( I + 4 )
   30    CONTINUE

      ELSE
c               ** nonequal or nonpositive increments.
         IX = 1
         IY = 1

         IF( INCX.LT.0 ) IX = 1 + ( N - 1 )*( -INCX )
         IF( INCY.LT.0 ) IY = 1 + ( N - 1 )*( -INCY )

         DO 40 I = 1, N
            SDOT = SDOT + SX( IX )*SY( IY )
            IX   = IX + INCX
            IY   = IY + INCY
   40    CONTINUE

      END IF

      END

      SUBROUTINE SSCAL( N, SA, SX, INCX )

c         Multiply vector SX by scalar SA

c  INPUT--  N  Number of elements in vector
c          SA  Single precision scale factor
c          SX  Sing-prec array, length 1+(N-1)*INCX, containing vector
c        INCX  Spacing of vector elements in SX

c OUTPUT-- SX  Replace  SX(1+I*INCX)  with  SA * SX(1+I*INCX)
c                for I = 0 to N-1
c ---------------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INCX, N
      REAL(KR)      SA
c     ..
c     .. Array Arguments ..

      REAL(KR)      SX( * )
c     ..
c     .. Local Scalars ..

      INTEGER   I, M
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC MOD
c     ..


      IF( N.LE.0 ) RETURN

      IF( INCX.NE.1 ) THEN

         DO 10 I = 1, 1 + ( N - 1 )*INCX, INCX
            SX( I ) = SA*SX( I )
   10    CONTINUE


      ELSE

         M  = MOD( N, 5 )

         IF( M.NE.0 ) THEN
c                           ** clean-up loop so remaining vector length
c                           ** is a multiple of 5.
            DO 20 I = 1, M
               SX( I ) = SA*SX( I )
   20       CONTINUE

         END IF
c                             ** unroll loop for speed
         DO 30 I = M + 1, N, 5
            SX( I ) = SA*SX( I )
            SX( I + 1 ) = SA*SX( I + 1 )
            SX( I + 2 ) = SA*SX( I + 2 )
            SX( I + 3 ) = SA*SX( I + 3 )
            SX( I + 4 ) = SA*SX( I + 4 )
   30    CONTINUE

      END IF

      END

      SUBROUTINE SSWAP( N, SX, INCX, SY, INCY )

c          Interchange s.p vectors  X  and  Y, as follows:

c     For I = 0 to N-1, interchange  SX(LX+I*INCX) and SY(LY+I*INCY),
c     where LX = 1          if INCX .GE. 0,
c              = (-INCX)*N  if INCX .LT. 0
c     and LY is defined analogously using INCY.


c  INPUT--
c        N  Number of elements in input vectors X and Y
c       SX  Sing-prec array containing vector X
c     INCX  Spacing of elements of vector X in SX
c       SY  Sing-prec array containing vector Y
c     INCY  Spacing of elements of vector Y in SY

c OUTPUT--
c       SX  Input vector SY (unchanged if N .LE. 0)
c       SY  Input vector SX (unchanged IF N .LE. 0)
c --------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INCX, INCY, N
c     ..
c     .. Array Arguments ..

      REAL(KR)      SX( * ), SY( * )
c     ..
c     .. Local Scalars ..

      INTEGER   I, IX, IY, M
      REAL(KR)      STEMP1, STEMP2, STEMP3
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC MOD
c     ..


      IF( N.LE.0 ) RETURN

      IF( INCX.EQ.INCY .AND. INCX.GT.1 ) THEN

         DO 10 I = 1, 1 + ( N-1 )*INCX, INCX
            STEMP1 = SX( I )
            SX( I ) = SY( I )
            SY( I ) = STEMP1
   10    CONTINUE


      ELSE IF( INCX.EQ.INCY .AND. INCX.EQ.1 ) THEN

c                                        ** equal, unit increments
         M  = MOD( N, 3 )

         IF( M.NE.0 ) THEN
c                            ** clean-up loop so remaining vector length
c                            ** is a multiple of 3.
            DO 20 I = 1, M
               STEMP1 = SX( I )
               SX( I ) = SY( I )
               SY( I ) = STEMP1
   20       CONTINUE

         END IF
c                              ** unroll loop for speed
         DO 30 I = M + 1, N, 3
            STEMP1 = SX( I )
            STEMP2 = SX( I + 1 )
            STEMP3 = SX( I + 2 )
            SX( I ) = SY( I )
            SX( I + 1 ) = SY( I + 1 )
            SX( I + 2 ) = SY( I + 2 )
            SY( I ) = STEMP1
            SY( I + 1 ) = STEMP2
            SY( I + 2 ) = STEMP3
   30    CONTINUE


      ELSE
c               ** nonequal or nonpositive increments.
         IX = 1
         IY = 1

         IF( INCX.LT.0 ) IX = 1 + ( N - 1 )*( -INCX )
         IF( INCY.LT.0 ) IY = 1 + ( N - 1 )*( -INCY )

         DO 40 I = 1, N
            STEMP1 = SX( IX )
            SX( IX ) = SY( IY )
            SY( IY ) = STEMP1
            IX   = IX + INCX
            IY   = IY + INCY
   40    CONTINUE

      END IF

      END

      INTEGER FUNCTION ISAMAX( N, SX, INCX )

c INPUT--  N     Number of elements in vector of interest
c          SX    Sing-prec array, length 1+(N-1)*INCX, containing vector
c          INCX  Spacing of vector elements in SX

c OUTPUT-- ISAMAX   First I, I = 1 to N, to maximize
c                         ABS(SX(1+(I-1)*INCX))
c ---------------------------------------------------------------------

c     .. Scalar Arguments ..

      use params, only: kr
      implicit none
      INTEGER   INCX, N
c     ..
c     .. Array Arguments ..

      REAL(KR)      SX( * )
c     ..
c     .. Local Scalars ..

      INTEGER   I, II
      REAL(KR)      SMAX, XMAG
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC ABS
c     ..
c     2017 new initialization
      ISAMAX = 0

      IF( N.LE.0 ) THEN

         ISAMAX = 0

      ELSE IF( N.EQ.1 ) THEN

         ISAMAX = 1

      ELSE

         SMAX = 0.0
         II   = 1

         DO 10 I = 1, 1 + ( N-1 )*INCX, INCX

            XMAG = ABS( SX( I ) )

            IF( SMAX.LT.XMAG ) THEN

               SMAX   = XMAG
               ISAMAX = II

            END IF

            II = II + 1

   10    CONTINUE

      END IF

      END

C================================================================================       
C        CALCULATE PHASE FUNCTION LEGENDRE EXPANSION COEFFICIENTS
C        IN VARIOUS SPECIAL CASES
C--------------------------------------------------------------------------------
C       INPUT: IPHAS   PHASE FUNCTION OPTIONS
C                      1 : ISOTROPIC
C                      2 : RAYLEIGH
C                      3 : HENYEY-GREENSTEIN WITH ASYMMETRY FACTOR -GG-
C                      4 : HAZE L AS SPECIFIED BY GARCIA/SIEWERT
C                      5 : CLOUD C.1 AS SPECIFIED BY GARCIA/SIEWERT
C              GG      ASYMMETRY FACTOR FOR HENYEY-GREENSTEIN CASE
C              NMOM    INDEX OF HIGHEST LEGENDRE COEFFICIENT NEEDED
C                        ( = NUMBER OF STREAMS 'NSTR'  CHOSEN
C                         FOR THE DISCRETE ORDINATE METHOD)
C--------------------------------------------------------------------------------
C      OUTPUT: PMOM(K)  LEGENDRE EXPANSION COEFFICIENTS (K=0 TO NMOM)
C                         (BE SURE TO DIMENSION '0:maxval' IN CALLING PROGRAM)
C--------------------------------------------------------------------------------
C      REFERENCE:  GARCIA, R. AND C. SIEWERT, 1985: BENCHMARK RESULTS
C                     IN RADIATIVE TRANSFER, TRANSP. THEORY AND STAT.
C                     PHYSICS 14, 437-484, TABLES 10 AND 17
C--------------------------------------------------------------------------------
      SUBROUTINE  GETMOM( IPHAS, GG, NMOM, PMOM )
      
      use params, only: kr
      implicit none

      INTEGER  IPHAS, NMOM, k

      REAL(KR)  GG, PMOM(0:*), HAZELM(82), CLDMOM(299)
      
      DATA HAZELM /  2.41260, 3.23047, 3.37296, 3.23150, 2.89350,
     A               2.49594, 2.11361, 1.74812, 1.44692, 1.17714,
     B               0.96643, 0.78237, 0.64114, 0.51966, 0.42563,
     C               0.34688, 0.28351, 0.23317, 0.18963, 0.15788,
     D               0.12739, 0.10762, 0.08597, 0.07381, 0.05828,
     E               0.05089, 0.03971, 0.03524, 0.02720, 0.02451,
     F               0.01874, 0.01711, 0.01298, 0.01198, 0.00904,
     G               0.00841, 0.00634, 0.00592, 0.00446, 0.00418,
     H               0.00316, 0.00296, 0.00225, 0.00210, 0.00160,
     I               0.00150, 0.00115, 0.00107, 0.00082, 0.00077,
     J               0.00059, 0.00055, 0.00043, 0.00040, 0.00031,
     K               0.00029, 0.00023, 0.00021, 0.00017, 0.00015,
     L               0.00012, 0.00011, 0.00009, 0.00008, 0.00006,
     M               0.00006, 0.00005, 0.00004, 0.00004, 0.00003,
     N               0.00003, 3*0.00002, 8*0.00001 /

      DATA  ( CLDMOM(K), K = 1, 159 ) /
     A  2.544,  3.883,  4.568,  5.235,  5.887,  6.457,  7.177,  7.859,
     B  8.494,  9.286,  9.856, 10.615, 11.229, 11.851, 12.503, 13.058,
     C 13.626, 14.209, 14.660, 15.231, 15.641, 16.126, 16.539, 16.934,
     D 17.325, 17.673, 17.999, 18.329, 18.588, 18.885, 19.103, 19.345,
     E 19.537, 19.721, 19.884, 20.024, 20.145, 20.251, 20.330, 20.401,
     F 20.444, 20.477, 20.489, 20.483, 20.467, 20.427, 20.382, 20.310,
     G 20.236, 20.136, 20.036, 19.909, 19.785, 19.632, 19.486, 19.311,
     H 19.145, 18.949, 18.764, 18.551, 18.348, 18.119, 17.901, 17.659,
     I 17.428, 17.174, 16.931, 16.668, 16.415, 16.144, 15.883, 15.606,
     J 15.338, 15.058, 14.784, 14.501, 14.225, 13.941, 13.662, 13.378,
     K 13.098, 12.816, 12.536, 12.257, 11.978, 11.703, 11.427, 11.156,
     L 10.884, 10.618, 10.350, 10.090,  9.827,  9.574,  9.318,  9.072,
     M  8.822, 8.584, 8.340, 8.110, 7.874, 7.652, 7.424, 7.211, 6.990,
     N  6.785, 6.573, 6.377, 6.173, 5.986, 5.790, 5.612, 5.424, 5.255,
     O  5.075, 4.915, 4.744, 4.592, 4.429, 4.285, 4.130, 3.994, 3.847,
     P  3.719, 3.580, 3.459, 3.327, 3.214, 3.090, 2.983, 2.866, 2.766,
     Q  2.656, 2.562, 2.459, 2.372, 2.274, 2.193, 2.102, 2.025, 1.940,
     R  1.869, 1.790, 1.723, 1.649, 1.588, 1.518, 1.461, 1.397, 1.344,
     S  1.284, 1.235, 1.179, 1.134, 1.082, 1.040, 0.992, 0.954, 0.909 /
      DATA  ( CLDMOM(K), K = 160, 299 ) /
     T  0.873, 0.832, 0.799, 0.762, 0.731, 0.696, 0.668, 0.636, 0.610,
     U  0.581, 0.557, 0.530, 0.508, 0.483, 0.463, 0.440, 0.422, 0.401,
     V  0.384, 0.364, 0.349, 0.331, 0.317, 0.301, 0.288, 0.273, 0.262,
     W  0.248, 0.238, 0.225, 0.215, 0.204, 0.195, 0.185, 0.177, 0.167,
     X  0.160, 0.151, 0.145, 0.137, 0.131, 0.124, 0.118, 0.112, 0.107,
     Y  0.101, 0.097, 0.091, 0.087, 0.082, 0.079, 0.074, 0.071, 0.067,
     Z  0.064, 0.060, 0.057, 0.054, 0.052, 0.049, 0.047, 0.044, 0.042,
     A  0.039, 0.038, 0.035, 0.034, 0.032, 0.030, 0.029, 0.027, 0.026,
     B  0.024, 0.023, 0.022, 0.021, 0.020, 0.018, 0.018, 0.017, 0.016,
     C  0.015, 0.014, 0.013, 0.013, 0.012, 0.011, 0.011, 0.010, 0.009,
     D  0.009, 3*0.008, 2*0.007, 3*0.006, 4*0.005, 4*0.004, 6*0.003,
     E  9*0.002, 18*0.001 /

      IF ( IPHAS.LT.1 .OR. IPHAS.GT.5 )
     $     call errmsg(13,'GETMOM--BAD INPUT VARIABLE IPHAS')
      IF ( IPHAS.EQ.3 .AND. (GG.LE.-1.0 .OR. GG.GE.1.0) )
     $     call errmsg(14,'GETMOM--BAD INPUT VARIABLE GG')
      IF ( NMOM.LT.2 )
     $     call errmsg(15,'GETMOM--BAD INPUT VARIABLE NMOM')
      
      PMOM(0) = 1.0
      DO  10  K = 1, NMOM
        PMOM(K) = 0.0
 10   CONTINUE
      
      IF ( IPHAS.EQ.2 )  THEN
C---------------------------
C    RAYLEIGH PHASE FUNCTION
C---------------------------

        PMOM(2) = 0.1
        
      ELSE IF ( IPHAS.EQ.3 ) THEN
C------------------------------
C   HENYEY-GREENSTEIN PHASE FCN
C------------------------------
        DO  20  K = 1, NMOM
          PMOM(K) = GG**K
 20     CONTINUE
        
      ELSE IF ( IPHAS.EQ.4 ) THEN
C---------------------------
C    HAZE-L PHASE FUNCTION
C---------------------------
        DO  30  K = 1, MIN0(82,NMOM)
          PMOM(K) = HAZELM(K) / ( 2*K+1 )
 30     CONTINUE
        
      ELSE IF ( IPHAS.EQ.5 ) THEN
C---------------------------
C   CLOUD C.1 PHASE FUNCTION
C---------------------------
        DO  40  K = 1, MIN0(298,NMOM)
          PMOM(K) = CLDMOM(K) / ( 2*K+1 )
 40     CONTINUE
        
      END IF
      
      RETURN
      END
c file:                  spectra.f
c
c external routines:     salbedo,suralb,raysig,rayleigh,zensun,solirr
c                        setfilt,filter
c
c internal routines:     rdspec,snow,clearw,lakew,seaw,sand,vegeta
c                        meteo, goese,goesw,avhr81,avhr82,avhr91,avhr92,
c                        avhr101,avhr102,avhr111,avhr112,gtr1,gtr2,nm410
c                        nm936,usersat,mfrsr1,mfrsr2,mfrsr3,mfrsr4,mfrsr5,
c                        mfrsr6,avhr83,avhr84,avhr85,setlow,sun1s,sunlow,
c                        sunmod
c
c internal modules:      albblk,  sunblk, fltblk
c=======================================================================
      module albblk             ! albedo data base
      use params, only: kr
      implicit none
      save
      integer, parameter :: mxwl=5000, mxbc=101
      real(kr), dimension(mxwl) :: wlalb,alb
      real(kr) :: wndspd,wndang,chlor,salin
      real(kr) :: hssa,hasym,hotspt,hotwdth
      real(kr) :: rliso,rlvol,rlgeo,rlhot,rlwdth

      integer :: nna=0,ibdrf
      end module albblk
c-----------------------------------------------------------------------
      function salbedo(wl)
c 
c input:   wl
c output:  surface albedo
c
      use params, only : one, zero, kr
      use albblk, only : nna, alb, wlalb
      implicit none
      integer :: j
      real(kr) :: wl, wt, salbedo
c mfl debugging
      character (len=9)  :: num1
      character (len=9)  :: num2

      if(nna.eq.0) return ! such as for isalb eq 7

      write(num1, '(f9.3)') wlalb(1)
      write(num2, '(f9.3)') wlalb(nna)

      if(wl.lt.wlalb(1))
     & call errmsg(18,'SALBEDO--spectral range error, wlinf lt '//num1)

      if(wl.gt.wlalb(nna))
     & call errmsg(18,'SALBEDO--spectral range error, wlsup gt '//num2)
      
      call locate(wlalb,nna,wl,j)
      wt=(wl-wlalb(j))/(wlalb(j+1)-wlalb(j))
      wt=max(zero,min(one,wt))
      salbedo=alb(j)*(1.-wt)+alb(j+1)*wt

      return
      end
c-----------------------------------------------------------------------
      subroutine suralb(isalb,albcon,sc)
c
c purpose: specify surface composition.  Must be called before SALBEDO
c
c input:   isalb    surface type
c                    -1 -spectral surface albedo read from "albedo.dat"
c                     0 -user specified, spectrally uniform, surface albedo
c                     1 -snow
c                     2 -clear water
c                     3 -lake water
c                     4 -sea  water
c                     5 -sand
c                     6 -vegetation
c                     7 -ocean bdrf model (from 6s)
c                     8 -hapke bdrf model
c                    10 -combination of snow, seawater, sand and vegetation
c
c         isalb
c           0    albcon --  constant surface albedo
c
c           7    sc     --  ocean reflection parameters 
c                           sc(1)=Oceanic pigment concentration in mg/m3
c                           sc(2)=wind speed in meters/second
c                           sc(3)=wind direction azimuth angle (degrees)
c                           sc(4)=oceanic salinity in parts per thousand
c
c           8    sc     --  hapke bdrf model parameters 
c                           sc(1)=surface particle single scattering albedo
c                           sc(2)=surface particle asymmetry factor
c                           sc(3)=hot spot magnitude
c                           sc(4)=width of hotspot
c
c           9    sc     --  Ross-thick Li-sparse bdrf model parameters 
c                           sc(1)=isotropic coefficient
c                           sc(2)=volumetric coefficient
c                           sc(3)=geometric shadowing coefficient
c                           sc(4)=hot spot magnitude
c                           sc(5)=hot spot width
c
c          10   sc      --  surface component fractions
c                           sc(1)=fraction of snow
c                           sc(2)=fraction of ocean
c                           sc(3)=fraction of sand
c                           sc(4)=fraction of vegetation
c

      use params, only: kr
      use albblk, only: mxwl, nna, alb, wlalb,ibdrf,
     &     wndspd, wndang, salin, chlor, 
     &     hssa, hasym, hotspt, hotwdth,
     &     rliso,rlvol,rlgeo,rlhot,rlwdth
      implicit none
      integer :: isalb
      real(kr) :: albcon, sc(5), albx(mxwl), mui

      nna=mxwl

c  mfl     if(isalb.eq.-1) then
c        return
c      endif

c mfl
c      print *, 'isalb = ', isalb

      select case (isalb)
      case (-1) ; call rdspec('albedo.dat',wlalb,alb,nna)

      case (0)  
        alb(1:2)=albcon
        wlalb(1:2)=(/0._kr,huge(0._kr)/)
        nna=2

      case (1)  ; call snow(wlalb,alb,nna)
      case (2)  ; call clearw(wlalb,alb,nna)
      case (3)  ; call lakew(wlalb,alb,nna)
      case (4)  ; call seaw(wlalb,alb,nna)
      case (5)  ; call sand(wlalb,alb,nna)
      case (6)  ; call vegeta(wlalb,alb,nna)

      case (-7,7)                ! ocean reflectance model
        ibdrf=1
        nna=0
        chlor=sc(1)
        wndspd=sc(2)
        salin=sc(3)

      case (-8,8)                ! hapke bdrf model
        ibdrf=2
        nna=0
        hssa=sc(1)
        hasym=sc(2)
        hotspt=sc(3)
        hotwdth=sc(4)

      case (-9,9)                ! ross-thick, li-sparse bdrf model
        ibdrf=3
        nna=0
        rliso=sc(1)
        rlvol=sc(2)
        rlgeo=sc(3)
        rlhot=sc(4)
        rlwdth=sc(5)

      case (10)               ! spectral mixture
        call snow(wlalb,albx,nna)   ; alb=albx*sc(1)
        call seaw(wlalb,albx,nna)   ; alb=albx*sc(2)+alb
        call sand(wlalb,albx,nna)   ; alb=albx*sc(3)+alb
        call vegeta(wlalb,albx,nna) ; alb=albx*sc(4)+alb

      case default 
        print *,'ERROR in suralb --- illegal value of isalb',isalb
        stop
      end select

      return
      end

c=======================================================================
      function raysig(v)

c purpose:
c    calculate molecular rayleigh scattering coefficient
c    using approximation of shettle et al., 1980 (appl. opt., 2873-4)
c    with the depolarization = 0.0279 instead of 0.035
c    for temperature = 273 k & pressure = 1 atm.
c
c input:
c  v         wavenumber cm-1
c
c output:
c  raysig    scattering coefficient (km-1) 
c            optical depth = raysig * (p/pzero)*(tzero/t)*dz

      use params, only: kr
      implicit none
      real(kr), parameter :: fit1=9.38076e+18,
     &                       fit2=-1.08426e+09
      real(kr) :: v, raysig


      raysig = v**4/(fit1+fit2*v**2)
      return
      end
c=======================================================================

      subroutine rayleigh(wl,z,p,t,nz,dtaur)

c  purpose:
c
c  input:  
c    wl        wavelength in microns
c    z         altitude array, z(1)=0 (km)
c    p         pressure array, p(1) at surface (millibars)
c    t         temperature array, t(1) at surface (kelvin)
c    nz        number of atmospheric layers
c
c  output: 
c    dtaur    increments of rayleigh scattering optical depth
c             dtaur(nz) represents the optical depth of the bottom layer
c
      use params, only: mxly, one, zero, pzero, tzero, kr
      implicit none
      integer :: i, im, nz 
      real(kr) :: z(*),p(*),t(*),dtaur(*),raysig,sig, dz, rhom, rhop, wl
c
      dtaur(1:nz)=0.

c rayleigh scattering coefficient (1/km) 

      sig=raysig(10000./wl)

      dtaur(1)=sig*(p(nz)/pzero)/(t(nz)/tzero)*5. ! assume 5km scale ht.

      do i=2,nz
        im=nz-i+1
        rhom=(p(im)/pzero)/(t(im)/tzero)
        rhop=(p(im+1)/pzero)/(t(im+1)/tzero)
        dz=z(im+1)-z(im)
        if(rhom.eq.rhop) then
          dtaur(i)=.5*sig*dz*(rhom+rhop)
        else
          dtaur(i)=sig*dz*(rhop-rhom)/log(rhop/rhom)
        endif
      enddo

      return
      end
c=======================================================================
       function bdref(wvnmlo, wvnmhi, mur, mui, phir )

c      Supplies surface bi-directional reflectivity.
c
c      NOTE 1: Bidirectional reflectivity in DISORT is defined
c              by Eq. 39 in STWL.
c      NOTE 2: Both MUR and MUI (cosines of reflection and incidence
c              angles) are positive.
c
c  INPUT:
c
c    WVNMLO : Lower wavenumber (inv cm) of spectral interval
c
c    WVNMHI : Upper wavenumber (inv cm) of spectral interval
c
c    MUR    : Cosine of angle of reflection (positive)
c
c    MUI    : Cosine of angle of incidence (positive)
c
c    PHIR   : Difference of azimuth angles of incidence and reflection
c                (radians)
c
c
c   Called by- DREF, SURFAC, BDREFCHK

c +-------------------------------------------------------------------+
c
c     .. Scalar Arguments ..

      use albblk, only: ibdrf
      use params, only: kr

      implicit none
      real(kr) :: phid, phir, mur, mui, wvnmhi, wvnmlo, wl, bdref, blim

      wl=20000./(wvnmhi+wvnmlo)
      select case (ibdrf)
      case (1); call seabdrf(wl,mui,mur,phir,bdref)
      case (2); call hapkbdrf(mui,mur,phir,bdref)
      case (3); call rtlsbdrf(mui,mur,phir,bdref)

      case default
        print *, 'Error in BDREF --- illegal value of ibdrf',ibdrf
        stop
      end select

      return
      end
c=======================================================================
      subroutine hapkbdrf(ui,ur,phir,bdrf)

c returns hapke bdrf factor 
c (http://stratus.ssec.wisc.edu/streamer_web/userman/surfalb.html) 
c
c   ui    cosine of incident angle (positive)
c   ur    cosine of reflection angle (positive)
c   phir  difference of solar and viewing azimuth (radians)
c
c output:
c   bdref surface reflectance
 
      use params, only: kr, pi
      use albblk, only:
     &     hssa,      ! ground particle single scattering albedo (0,1)
     &     hasym,     ! ground particle asymmetry factor         (-1,1)
     &     hotspt,    ! hot spot amplitude factor 
     &     hotwdth    ! hot spot width factor

      implicit none
      
      real(kr) ::
     &     bdrf,                ! returned bdrf value
     &     s,                   ! angle between incomming and reflected ray
     &     coss,                ! cosine of s
     &     ur,                  ! cosine of reflected zenith angle
     &     ui,                  ! cosine of incident zenith angle
     &     phir,                ! azimuth between incident and reflected rays
     &     pfun,                ! henyey greenstein scattering function
     &     pfun0,               ! h.g. function at coss=1
     &     b0,                  ! hot spot factor
     &     bfun,                ! hot spot function
     &     hfunr,               ! multiple scattering function (reflection)
     &     hfuni                ! multiple scattering function (incidence)

      coss=ui * ur +sqrt(1.-ur**2)*sqrt(1.-ui**2)*cos(pi-phir)

      s=acos(coss)

      pfun=(1.-hasym**2)/(1+hasym**2+2*hasym*coss)**1.5
      pfun0=(1.-hasym**2)/(1+hasym)**3
      b0=hotspt/(hssa*pfun0)
      bfun=b0/(1.+tan(s/2)/hotwdth)
      hfunr=(1.+2*ur)/(1.+2.*ur*sqrt(1.-hssa))
      hfuni=(1.+2*ui)/(1.+2.*ui*sqrt(1.-hssa))

      bdrf=(1.+bfun)*pfun + hfunr*hfuni - 1.
      bdrf=.25*hssa*bdrf/(ur+ui)

      return
      end
c=======================================================================
      subroutine rtlsbdrf(mui,mur,phir,bdrf)

c returns ross-thick, li-sparse  bdrf factor 
c (Ross, 1981; Li and Strahler, 1992).
c (http://stratus.ssec.wisc.edu/streamer_web/userman/surfalb.html) 
c
c   mui    cosine of incident angle (positive)
c   mur    cosine of reflection angle (positive)
c   phir  difference of solar and viewing azimuth (radians)
c
c output:
c   bdref surface reflectance
 
      use params, only: kr, pi
      use albblk, only:
     &     rliso,               ! isotropic coefficient
     &     rlvol,               ! volumetric coefficient
     &     rlgeo,               ! geometric coefficient
     &     rlhot,               ! hot spot magnitude
     &     rlwdth               ! hot spot width

      implicit none
      
      real(kr) :: sp, vza, sza, tanvzap, tanszap, vzap, szap, cossp,
     &     dd, secsum, cost, t, coss, sins, s, f1, f2, mui, mur,
     &     ui, ur, phir, bdrf, cosra

      ui=max(mui,.01_kr)
      ur=max(mur,.01_kr)
      cosra=cos(pi-phir)
      coss=ui * ur +sqrt(1.-ur**2)*sqrt(1.-ui**2)*cosra
      coss=max(-1._kr,min(coss,1._kr))
      s=acos(coss)
      sins=sin(s)

                                ! thick canopy reflectance kernel
      f1=(pi/2-s)*coss+sins     
      f1=f1/(ui+ur)-pi/4.

      vza=acos(ur)
      sza=acos(ui)
      tanvzap=rlwdth*tan(vza)
      tanszap=rlwdth*tan(sza)
      if(rlwdth.eq.1._kr) then
        vzap=vza
        szap=sza
      else
        vzap=atan(tanvzap)
        szap=atan(tanszap)
      endif
      
      cossp=cos(szap)*cos(vzap)+sin(szap)*sin(vzap)*cosra
      cossp=max(-1._kr,min(cossp,1._kr))
      dd=tanszap**2+tanvzap**2-2*tanszap*tanvzap*cosra

      secsum=1./cos(szap)+1./cos(vzap)
      cost=rlhot*sqrt(dd+(tanszap*tanvzap*sin(pi-phir))**2)
      cost=cost/secsum
      cost=max(-1._kr,min(cost,1._kr))

      t=acos(cost)
                                ! sparse object reflectance kernel
      
      f2=(t-sin(t)*cost)*secsum/pi ! sparse object kernel
      f2=f2-1./cos(vzap)+.5*(1.+cossp)/(cos(szap)*cos(vzap))

      bdrf=rliso+rlvol*f1+rlgeo*f2

      return
      end
c=======================================================================
      subroutine seabdrf(wl,mus,muv,phir,bdref)

c returns reflectance of sea surface due to surface reflection,
c foam and subsurface particulates and rayleigh scattering
c
c input:
c   wl    wavelength (um)
c   mus   cosine of solar zenith angle (0-1)
c   muv   cosine of viewing zenith angle (0-1)
c   phir  difference of solar and viewing azimuth (radians)
c
c output:
c   bdref surface reflectance
 
      use params, only: kr, pi
      use albblk, only:
     &     wndspd,    ! wind speed in m/s
     &     wndang,    ! solar_azimuth - wind_azimuth (degrees)
     &     chlor,     ! chlorphyll conentration (mg/m3) 
     &     salin      ! salinity (ppt) sea water 32-37  fresh water < .5
      implicit none
      
      real(kr) :: wl,ts,tv,sunaz,rgl,bdref,mus,muv,phir
      real(kr), save :: wllast=0., wndwt=-1., nr, ni, rsw, rfoam

      ! foam parameters from koepke, 1984

      real(kr), parameter ::
     &     wndc1=2.951e-6, ! coefficient 1 of foam area formula
     &     wndc2=3.52,     ! coefficient 2 of foam area formula
     &     rfco=0.22       ! average albedo of foam +- 0.11 (koepke)

      if(wllast.ne.wl) then 
        call indwat(wl,chlor,nr,ni)   ! index of refraction
        call morcasiwat(wl,chlor,rsw) ! sub-surface contribution (lambertian)
        if(chlor .eq. 0.) rsw=0.
        wndwt=wndc1*wndspd**wndc2     ! area covered by foam
        rfoam=wndwt*rfco              ! foam reflectance (lambertian)
        wllast=wl
      endif
      call sunglint(wndspd,nr,ni,mus,muv,phir,rgl)
      bdref=rfoam+(1.-wndwt)*rgl+(1.-rfoam)*rsw
      return
      end
c=======================================================================
            
      subroutine morcasiwat(wl,c,rsw)
c 
c spectral diffuse attenuation coefficient of case i waters as predicted 
c by morel within the spectral range 400-700nm (1988, journal of geophysical 
c research, vol.93, no c9, pp 10749-10768)
c
c this subroutine copied whole from tanre's 6s code.
c
c input parameters:     wl wavelength (in micrometers)
c                       c  pigment concentration 
c                          (chlorophyll + pheophytin mg/m3) 
c output parameter:     rsw  reflectance of water
c
c according morel,1988, we use:
c
c kd    spectral value of the attenuation coefficient for 
c        downwelling irradiance
c        with: kd=kw+xc*c**e
c kw    spectral value of the diffuse attenuation coefficient 
c        for pure oceanic water
c xc, e spectral coefficients to compute the diffuse attenuation 
c        coefficient for pigment
c bb    total backscattering coefficient
c        with: bb=0.5*bw+bbt*b
c bw    spectral value of the molecular scattering coefficient of water
c bbt,b parameters to compute the scattering coefficients of pigments
c
c rsw   subsurface contribution to ocean reflectivity (transfered to above
c       the surface by averaging fresnel transmission through surface)
c
c        with: rsw=(0.33/u)*(bb/kd)     where u is depends on rsw
c
      use params, only: kr
      implicit none
      real(kr) :: kw,kd,tkw(61),txc(61),te(61),tbw(61),
     &     wl,c,rsw,xc,e,bw,bb,b,bbt,u1,r1,u2,err
      integer :: iwl

      data tkw/
     & 0.0209,0.0200,0.0196,0.0189,0.0183,
     & 0.0182,0.0171,0.0170,0.0168,0.0166,
     & 0.0168,0.0170,0.0173,0.0174,0.0175,
     & 0.0184,0.0194,0.0203,0.0217,0.0240,
     & 0.0271,0.0320,0.0384,0.0445,0.0490,
     & 0.0505,0.0518,0.0543,0.0568,0.0615,
     & 0.0640,0.0640,0.0717,0.0762,0.0807,
     & 0.0940,0.1070,0.1280,0.1570,0.2000,
     & 0.2530,0.2790,0.2960,0.3030,0.3100,
     & 0.3150,0.3200,0.3250,0.3300,0.3400,
     & 0.3500,0.3700,0.4050,0.4180,0.4300,
     & 0.4400,0.4500,0.4700,0.5000,0.5500,
     & 0.6500/
      data txc/
     & 0.1100,0.1110,0.1125,0.1135,0.1126,
     & 0.1104,0.1078,0.1065,0.1041,0.0996,
     & 0.0971,0.0939,0.0896,0.0859,0.0823,
     & 0.0788,0.0746,0.0726,0.0690,0.0660,
     & 0.0636,0.0600,0.0578,0.0540,0.0498,
     & 0.0475,0.0467,0.0450,0.0440,0.0426,
     & 0.0410,0.0400,0.0390,0.0375,0.0360,
     & 0.0340,0.0330,0.0328,0.0325,0.0330,
     & 0.0340,0.0350,0.0360,0.0375,0.0385,
     & 0.0400,0.0420,0.0430,0.0440,0.0445,
     & 0.0450,0.0460,0.0475,0.0490,0.0515,
     & 0.0520,0.0505,0.0440,0.0390,0.0340,
     & 0.0300/
      data te/
     & 0.668,0.672,0.680,0.687,0.693,
     & 0.701,0.707,0.708,0.707,0.704,
     & 0.701,0.699,0.700,0.703,0.703,
     & 0.703,0.703,0.704,0.702,0.700,
     & 0.700,0.695,0.690,0.685,0.680,
     & 0.675,0.670,0.665,0.660,0.655,
     & 0.650,0.645,0.640,0.630,0.623,
     & 0.615,0.610,0.614,0.618,0.622,
     & 0.626,0.630,0.634,0.638,0.642,
     & 0.647,0.653,0.658,0.663,0.667,
     & 0.672,0.677,0.682,0.687,0.695,
     & 0.697,0.693,0.665,0.640,0.620,
     & 0.600/
      data tbw/
     & 0.0076,0.0072,0.0068,0.0064,0.0061,
     & 0.0058,0.0055,0.0052,0.0049,0.0047,
     & 0.0045,0.0043,0.0041,0.0039,0.0037,
     & 0.0036,0.0034,0.0033,0.0031,0.0030,
     & 0.0029,0.0027,0.0026,0.0025,0.0024,
     & 0.0023,0.0022,0.0022,0.0021,0.0020,
     & 0.0019,0.0018,0.0018,0.0017,0.0017,
     & 0.0016,0.0016,0.0015,0.0015,0.0014,
     & 0.0014,0.0013,0.0013,0.0012,0.0012,
     & 0.0011,0.0011,0.0010,0.0010,0.0010,
     & 0.0010,0.0009,0.0008,0.0008,0.0008,
     & 0.0007,0.0007,0.0007,0.0007,0.0007,
     & 0.0007/
      if (wl.lt.0.400.or.wl.gt.0.700)then
        rsw=0.000
        goto 60
      endif

      iwl=1+nint((wl-0.400)/0.005)
      kw=tkw(iwl)
      xc=txc(iwl)
      e=te(iwl)
      bw=tbw(iwl)
c
      if (abs(c).lt.0.0001)then
        bb=0.5*bw
        kd=kw
      else
        b=0.30*c**0.62
        bbt=0.002+0.02*(0.5-0.25*log10(c))*0.550/wl
        bb=0.5*bw+bbt*b
        kd=kw+xc*c**e
      endif

      u1=0.75
      r1=0.33*bb/u1/kd

 50   u2=0.90*(1.-r1)/(1.+2.25*r1)
      rsw=0.33*bb/u2/kd
      err=abs((rsw-r1)/rsw)
      if (err.lt.0.0001)goto 60
      r1=rsw
      goto 50
 60   return
      end
c=======================================================================
       subroutine indwat(wl,xsal,nr,ni)
c
c input parameters:  wl=wavelength (in micrometers)
c                    xsal=salinity (in ppt), if xsal<0 then 34.3ppt by default
c output parameters: nr=index of refraction of sea water
c                    ni=extinction coefficient of sea water
c
c temperature dependence of ni ignored
c 
c
       use params, only: kr
       implicit none

       integer, parameter :: ntab=860
       integer :: i
       real(kr), parameter :: nrc=0.006, nic=0.000
       real(kr) ::  wltab(ntab),mrtab(ntab),mitab(ntab),nr,ni,wl,xsal,wt

c      (1) for 0.01 - 1.e7 microns :  Segelstein, D., 1981:
c             "The Complex Refractive Index of Water", M.S. Thesis,
c             University of Missouri--Kansas City

c      (2) for 10. - 1.e7 microns:  Ray, P., 1972:  Broadband Complex
c             Refractive Indices of Ice and Water, Appl. Opt. 11,
c             1836-1844

c      (There is a new reference, Wieliczka, D. et al., Appl. Opt.
c       28, 1714-1719, 1989, with some updated data for the IR)

      data wltab(1:50)/
     &  1.000e-02, 1.099e-02, 1.200e-02, 1.300e-02, 1.400e-02,
     &  1.600e-02, 1.799e-02, 2.000e-02, 2.198e-02, 2.399e-02,
     &  2.600e-02, 2.799e-02, 2.999e-02, 3.199e-02, 3.396e-02,
     &  3.598e-02, 3.802e-02, 3.999e-02, 4.198e-02, 4.395e-02,
     &  4.603e-02, 4.797e-02, 5.000e-02, 5.200e-02, 5.395e-02,
     &  5.598e-02, 5.794e-02, 5.998e-02, 6.194e-02, 6.397e-02,
     &  6.607e-02, 6.808e-02, 6.998e-02, 7.194e-02, 7.396e-02,
     &  7.603e-02, 7.798e-02, 7.998e-02, 8.203e-02, 8.395e-02,
     &  8.590e-02, 8.790e-02, 8.995e-02, 9.205e-02, 9.397e-02,
     &  9.594e-02, 9.795e-02, 1.000e-01, 1.021e-01, 1.040e-01/

      data wltab(51:100)/
     &  1.059e-01, 1.079e-01, 1.099e-01, 1.119e-01, 1.140e-01,
     &  1.159e-01, 1.180e-01, 1.200e-01, 1.219e-01, 1.239e-01,
     &  1.259e-01, 1.279e-01, 1.300e-01, 1.321e-01, 1.340e-01,
     &  1.361e-01, 1.380e-01, 1.400e-01, 1.419e-01, 1.439e-01,
     &  1.459e-01, 1.479e-01, 1.500e-01, 1.520e-01, 1.542e-01,
     &  1.560e-01, 1.581e-01, 1.600e-01, 1.622e-01, 1.641e-01,
     &  1.660e-01, 1.679e-01, 1.698e-01, 1.722e-01, 1.742e-01,
     &  1.750e-01, 1.799e-01, 1.849e-01, 1.901e-01, 1.950e-01,
     &  2.000e-01, 2.051e-01, 2.099e-01, 2.148e-01, 2.198e-01,
     &  2.249e-01, 2.301e-01, 2.350e-01, 2.399e-01, 2.449e-01/
      data wltab(101:150)/
     &  2.500e-01, 2.553e-01, 2.600e-01, 2.648e-01, 2.698e-01,
     &  2.748e-01, 2.799e-01, 2.851e-01, 2.897e-01, 2.951e-01,
     &  2.999e-01, 3.048e-01, 3.097e-01, 3.148e-01, 3.199e-01,
     &  3.251e-01, 3.304e-01, 3.350e-01, 3.396e-01, 3.451e-01,
     &  3.499e-01, 3.548e-01, 3.598e-01, 3.648e-01, 3.698e-01,
     &  3.750e-01, 3.802e-01, 3.846e-01, 3.899e-01, 3.954e-01,
     &  3.999e-01, 4.046e-01, 4.102e-01, 4.149e-01, 4.198e-01,
     &  4.246e-01, 4.295e-01, 4.345e-01, 4.395e-01, 4.446e-01,
     &  4.498e-01, 4.550e-01, 4.603e-01, 4.645e-01, 4.699e-01,
     &  4.753e-01, 4.797e-01, 4.853e-01, 4.898e-01, 4.955e-01/
      data wltab(151:200)/
     &  5.000e-01, 5.047e-01, 5.105e-01, 5.152e-01, 5.200e-01,
     &  5.248e-01, 5.297e-01, 5.346e-01, 5.395e-01, 5.445e-01,
     &  5.495e-01, 5.546e-01, 5.598e-01, 5.649e-01, 5.702e-01,
     &  5.754e-01, 5.794e-01, 5.848e-01, 5.902e-01, 5.957e-01,
     &  5.998e-01, 6.053e-01, 6.095e-01, 6.152e-01, 6.194e-01,
     &  6.252e-01, 6.295e-01, 6.353e-01, 6.397e-01, 6.456e-01,
     &  6.501e-01, 6.546e-01, 6.607e-01, 6.653e-01, 6.699e-01,
     &  6.745e-01, 6.808e-01, 6.855e-01, 6.902e-01, 6.950e-01,
     &  6.998e-01, 7.047e-01, 7.096e-01, 7.145e-01, 7.195e-01,
     &  7.244e-01, 7.295e-01, 7.345e-01, 7.396e-01, 7.447e-01/
      data wltab(201:250)/
     &  7.499e-01, 7.551e-01, 7.603e-01, 7.656e-01, 7.691e-01,
     &  7.745e-01, 7.798e-01, 7.852e-01, 7.907e-01, 7.943e-01,
     &  7.998e-01, 8.054e-01, 8.091e-01, 8.147e-01, 8.204e-01,
     &  8.241e-01, 8.298e-01, 8.356e-01, 8.395e-01, 8.453e-01,
     &  8.492e-01, 8.551e-01, 8.590e-01, 8.650e-01, 8.710e-01,
     &  8.750e-01, 8.790e-01, 8.851e-01, 8.892e-01, 8.954e-01,
     &  8.995e-01, 9.057e-01, 9.099e-01, 9.141e-01, 9.204e-01,
     &  9.247e-01, 9.290e-01, 9.354e-01, 9.397e-01, 9.441e-01,
     &  9.506e-01, 9.550e-01, 9.594e-01, 9.660e-01, 9.705e-01,
     &  9.750e-01, 9.795e-01, 9.840e-01, 9.908e-01, 9.954e-01/
      data wltab(251:300)/
     &  1.000e+00, 1.009e+00, 1.021e+00, 1.030e+00, 1.040e+00,
     &  1.050e+00, 1.059e+00, 1.069e+00, 1.079e+00, 1.089e+00,
     &  1.099e+00, 1.109e+00, 1.119e+00, 1.130e+00, 1.140e+00,
     &  1.151e+00, 1.159e+00, 1.169e+00, 1.180e+00, 1.191e+00,
     &  1.200e+00, 1.211e+00, 1.219e+00, 1.230e+00, 1.239e+00,
     &  1.250e+00, 1.259e+00, 1.271e+00, 1.279e+00, 1.291e+00,
     &  1.300e+00, 1.309e+00, 1.321e+00, 1.331e+00, 1.340e+00,
     &  1.349e+00, 1.361e+00, 1.371e+00, 1.380e+00, 1.390e+00,
     &  1.400e+00, 1.409e+00, 1.419e+00, 1.429e+00, 1.439e+00,
     &  1.449e+00, 1.459e+00, 1.469e+00, 1.479e+00, 1.489e+00/
      data wltab(301:350)/
     &  1.500e+00, 1.510e+00, 1.520e+00, 1.531e+00, 1.542e+00,
     &  1.549e+00, 1.560e+00, 1.570e+00, 1.581e+00, 1.589e+00,
     &  1.600e+00, 1.611e+00, 1.622e+00, 1.629e+00, 1.641e+00,
     &  1.648e+00, 1.660e+00, 1.671e+00, 1.679e+00, 1.690e+00,
     &  1.698e+00, 1.710e+00, 1.722e+00, 1.730e+00, 1.742e+00,
     &  1.750e+00, 1.762e+00, 1.770e+00, 1.778e+00, 1.791e+00,
     &  1.799e+00, 1.811e+00, 1.820e+00, 1.828e+00, 1.841e+00,
     &  1.849e+00, 1.862e+00, 1.871e+00, 1.879e+00, 1.888e+00,
     &  1.901e+00, 1.910e+00, 1.919e+00, 1.932e+00, 1.941e+00,
     &  1.950e+00, 1.959e+00, 1.968e+00, 1.982e+00, 1.991e+00/
      data wltab(351:400)/
     &  2.000e+00, 2.009e+00, 2.018e+00, 2.028e+00, 2.042e+00,
     &  2.051e+00, 2.061e+00, 2.070e+00, 2.080e+00, 2.089e+00,
     &  2.099e+00, 2.109e+00, 2.118e+00, 2.128e+00, 2.138e+00,
     &  2.148e+00, 2.158e+00, 2.168e+00, 2.178e+00, 2.188e+00,
     &  2.198e+00, 2.208e+00, 2.218e+00, 2.228e+00, 2.239e+00,
     &  2.249e+00, 2.259e+00, 2.270e+00, 2.280e+00, 2.291e+00,
     &  2.301e+00, 2.312e+00, 2.317e+00, 2.328e+00, 2.339e+00,
     &  2.350e+00, 2.361e+00, 2.371e+00, 2.382e+00, 2.388e+00,
     &  2.399e+00, 2.410e+00, 2.421e+00, 2.432e+00, 2.438e+00,
     &  2.449e+00, 2.460e+00, 2.472e+00, 2.477e+00, 2.489e+00/
      data wltab(401:450)/
     &  2.500e+00, 2.512e+00, 2.518e+00, 2.529e+00, 2.541e+00,
     &  2.553e+00, 2.564e+00, 2.570e+00, 2.576e+00, 2.582e+00,
     &  2.588e+00, 2.594e+00, 2.606e+00, 2.612e+00, 2.618e+00,
     &  2.624e+00, 2.630e+00, 2.636e+00, 2.648e+00, 2.655e+00,
     &  2.661e+00, 2.667e+00, 2.673e+00, 2.679e+00, 2.685e+00,
     &  2.698e+00, 2.704e+00, 2.710e+00, 2.716e+00, 2.723e+00,
     &  2.729e+00, 2.742e+00, 2.748e+00, 2.754e+00, 2.761e+00,
     &  2.767e+00, 2.780e+00, 2.786e+00, 2.792e+00, 2.799e+00,
     &  2.812e+00, 2.818e+00, 2.825e+00, 2.831e+00, 2.838e+00,
     &  2.851e+00, 2.858e+00, 2.864e+00, 2.871e+00, 2.884e+00/
      data wltab(451:500)/
     &  2.891e+00, 2.897e+00, 2.904e+00, 2.917e+00, 2.924e+00,
     &  2.931e+00, 2.938e+00, 2.951e+00, 2.958e+00, 2.965e+00,
     &  2.978e+00, 2.985e+00, 2.999e+00, 3.048e+00, 3.097e+00,
     &  3.148e+00, 3.199e+00, 3.251e+00, 3.304e+00, 3.350e+00,
     &  3.396e+00, 3.451e+00, 3.499e+00, 3.548e+00, 3.598e+00,
     &  3.647e+00, 3.698e+00, 3.750e+00, 3.802e+00, 3.846e+00,
     &  3.899e+00, 3.954e+00, 3.999e+00, 4.046e+00, 4.102e+00,
     &  4.149e+00, 4.198e+00, 4.246e+00, 4.295e+00, 4.345e+00,
     &  4.395e+00, 4.446e+00, 4.498e+00, 4.550e+00, 4.603e+00,
     &  4.645e+00, 4.699e+00, 4.753e+00, 4.797e+00, 4.853e+00/
      data wltab(501:550)/
     &  4.898e+00, 4.955e+00, 5.000e+00, 5.047e+00, 5.105e+00,
     &  5.152e+00, 5.200e+00, 5.248e+00, 5.297e+00, 5.346e+00,
     &  5.395e+00, 5.445e+00, 5.495e+00, 5.546e+00, 5.598e+00,
     &  5.649e+00, 5.702e+00, 5.754e+00, 5.794e+00, 5.848e+00,
     &  5.902e+00, 5.957e+00, 5.998e+00, 6.053e+00, 6.095e+00,
     &  6.152e+00, 6.194e+00, 6.252e+00, 6.295e+00, 6.353e+00,
     &  6.397e+00, 6.457e+00, 6.501e+00, 6.546e+00, 6.607e+00,
     &  6.653e+00, 6.699e+00, 6.745e+00, 6.808e+00, 6.855e+00,
     &  6.902e+00, 6.950e+00, 6.998e+00, 7.047e+00, 7.096e+00,
     &  7.145e+00, 7.194e+00, 7.244e+00, 7.295e+00, 7.345e+00/
      data wltab(551:600)/
     &  7.396e+00, 7.447e+00, 7.499e+00, 7.551e+00, 7.603e+00,
     &  7.656e+00, 7.691e+00, 7.745e+00, 7.798e+00, 7.852e+00,
     &  7.907e+00, 7.943e+00, 7.998e+00, 8.054e+00, 8.091e+00,
     &  8.147e+00, 8.204e+00, 8.241e+00, 8.299e+00, 8.356e+00,
     &  8.395e+00, 8.453e+00, 8.492e+00, 8.551e+00, 8.590e+00,
     &  8.650e+00, 8.710e+00, 8.750e+00, 8.790e+00, 8.851e+00,
     &  8.892e+00, 8.954e+00, 8.995e+00, 9.057e+00, 9.099e+00,
     &  9.141e+00, 9.204e+00, 9.247e+00, 9.290e+00, 9.354e+00,
     &  9.397e+00, 9.441e+00, 9.506e+00, 9.550e+00, 9.594e+00,
     &  9.661e+00, 9.705e+00, 9.750e+00, 9.795e+00, 9.840e+00/
      data wltab(601:650)/
     &  9.908e+00, 9.954e+00, 1.000e+01, 1.005e+01, 1.009e+01,
     &  1.014e+01, 1.021e+01, 1.026e+01, 1.030e+01, 1.035e+01,
     &  1.040e+01, 1.045e+01, 1.049e+01, 1.054e+01, 1.059e+01,
     &  1.064e+01, 1.069e+01, 1.074e+01, 1.079e+01, 1.084e+01,
     &  1.089e+01, 1.094e+01, 1.099e+01, 1.104e+01, 1.109e+01,
     &  1.114e+01, 1.119e+01, 1.125e+01, 1.130e+01, 1.135e+01,
     &  1.140e+01, 1.146e+01, 1.151e+01, 1.156e+01, 1.159e+01,
     &  1.164e+01, 1.170e+01, 1.175e+01, 1.180e+01, 1.186e+01,
     &  1.191e+01, 1.194e+01, 1.199e+01, 1.205e+01, 1.211e+01,
     &  1.216e+01, 1.219e+01, 1.225e+01, 1.230e+01, 1.236e+01/
      data wltab(651:700)/
     &  1.239e+01, 1.245e+01, 1.250e+01, 1.256e+01, 1.259e+01,
     &  1.265e+01, 1.271e+01, 1.276e+01, 1.279e+01, 1.285e+01,
     &  1.291e+01, 1.294e+01, 1.300e+01, 1.306e+01, 1.309e+01,
     &  1.315e+01, 1.321e+01, 1.324e+01, 1.330e+01, 1.334e+01,
     &  1.340e+01, 1.346e+01, 1.349e+01, 1.355e+01, 1.361e+01,
     &  1.365e+01, 1.371e+01, 1.374e+01, 1.380e+01, 1.384e+01,
     &  1.390e+01, 1.396e+01, 1.400e+01, 1.406e+01, 1.409e+01,
     &  1.416e+01, 1.419e+01, 1.426e+01, 1.429e+01, 1.436e+01,
     &  1.439e+01, 1.445e+01, 1.449e+01, 1.455e+01, 1.459e+01,
     &  1.465e+01, 1.469e+01, 1.476e+01, 1.479e+01, 1.486e+01/
      data wltab(701:750)/
     &  1.489e+01, 1.496e+01, 1.500e+01, 1.507e+01, 1.510e+01,
     &  1.514e+01, 1.521e+01, 1.524e+01, 1.531e+01, 1.535e+01,
     &  1.542e+01, 1.545e+01, 1.549e+01, 1.556e+01, 1.560e+01,
     &  1.567e+01, 1.570e+01, 1.574e+01, 1.581e+01, 1.585e+01,
     &  1.588e+01, 1.596e+01, 1.600e+01, 1.603e+01, 1.611e+01,
     &  1.614e+01, 1.622e+01, 1.625e+01, 1.629e+01, 1.637e+01,
     &  1.641e+01, 1.644e+01, 1.648e+01, 1.656e+01, 1.660e+01,
     &  1.663e+01, 1.671e+01, 1.675e+01, 1.679e+01, 1.687e+01,
     &  1.690e+01, 1.694e+01, 1.698e+01, 1.706e+01, 1.710e+01,
     &  1.714e+01, 1.722e+01, 1.726e+01, 1.730e+01, 1.734e+01/
      data wltab(751:800)/
     &  1.742e+01, 1.746e+01, 1.750e+01, 1.754e+01, 1.762e+01,
     &  1.766e+01, 1.770e+01, 1.774e+01, 1.778e+01, 1.786e+01,
     &  1.791e+01, 1.795e+01, 1.799e+01, 1.803e+01, 1.811e+01,
     &  1.816e+01, 1.820e+01, 1.824e+01, 1.828e+01, 1.836e+01,
     &  1.841e+01, 1.845e+01, 1.849e+01, 1.854e+01, 1.862e+01,
     &  1.866e+01, 1.871e+01, 1.875e+01, 1.879e+01, 1.884e+01,
     &  1.888e+01, 1.897e+01, 1.901e+01, 1.905e+01, 1.910e+01,
     &  1.914e+01, 1.919e+01, 1.923e+01, 1.932e+01, 1.936e+01,
     &  1.941e+01, 1.945e+01, 1.950e+01, 1.954e+01, 1.959e+01,
     &  1.963e+01, 1.968e+01, 1.977e+01, 1.982e+01, 1.986e+01/
      data wltab(801:860)/
     &  1.991e+01, 1.995e+01, 2.000e+01, 2.099e+01, 2.198e+01,
     &  2.301e+01, 2.399e+01, 2.500e+01, 2.600e+01, 2.698e+01,
     &  2.799e+01, 2.897e+01, 2.999e+01, 3.097e+01, 3.199e+01,
     &  3.304e+01, 3.396e+01, 3.499e+01, 3.598e+01, 3.698e+01,
     &  3.802e+01, 3.899e+01, 3.999e+01, 4.102e+01, 4.198e+01,
     &  4.295e+01, 4.395e+01, 4.498e+01, 4.603e+01, 4.699e+01,
     &  4.797e+01, 4.898e+01, 5.000e+01, 5.200e+01, 5.395e+01,
     &  5.598e+01, 5.794e+01, 5.998e+01, 6.194e+01, 6.397e+01,
     &  6.607e+01, 6.808e+01, 6.998e+01, 7.194e+01, 7.396e+01,
     &  7.603e+01, 7.798e+01, 7.998e+01, 8.203e+01, 8.395e+01,
     &  8.590e+01, 8.790e+01, 8.995e+01, 9.205e+01, 9.397e+01,
     &  9.594e+01, 9.795e+01, 1.000e+02, 1.099e+02, 1.200e+02/

      data mrtab(1:50)/
     &  9.684e-01, 9.648e-01, 9.610e-01, 9.570e-01, 9.528e-01,
     &  9.441e-01, 9.347e-01, 9.246e-01, 9.140e-01, 9.027e-01,
     &  8.908e-01, 8.788e-01, 8.665e-01, 8.541e-01, 8.422e-01,
     &  8.307e-01, 8.198e-01, 8.100e-01, 8.023e-01, 7.977e-01,
     &  7.970e-01, 8.056e-01, 8.207e-01, 8.310e-01, 8.352e-01,
     &  8.353e-01, 8.316e-01, 8.309e-01, 8.406e-01, 8.670e-01,
     &  9.035e-01, 9.418e-01, 9.817e-01, 1.021e+00, 1.050e+00,
     &  1.069e+00, 1.088e+00, 1.112e+00, 1.141e+00, 1.173e+00,
     &  1.215e+00, 1.259e+00, 1.303e+00, 1.347e+00, 1.388e+00,
     &  1.425e+00, 1.456e+00, 1.477e+00, 1.493e+00, 1.507e+00/
      data mrtab(51:100)/
     &  1.516e+00, 1.524e+00, 1.529e+00, 1.535e+00, 1.543e+00,
     &  1.548e+00, 1.553e+00, 1.561e+00, 1.570e+00, 1.585e+00,
     &  1.606e+00, 1.627e+00, 1.634e+00, 1.619e+00, 1.586e+00,
     &  1.536e+00, 1.496e+00, 1.471e+00, 1.461e+00, 1.461e+00,
     &  1.469e+00, 1.490e+00, 1.521e+00, 1.560e+00, 1.597e+00,
     &  1.620e+00, 1.641e+00, 1.650e+00, 1.653e+00, 1.653e+00,
     &  1.647e+00, 1.635e+00, 1.606e+00, 1.568e+00, 1.549e+00,
     &  1.543e+00, 1.513e+00, 1.492e+00, 1.475e+00, 1.463e+00,
     &  1.452e+00, 1.442e+00, 1.435e+00, 1.428e+00, 1.422e+00,
     &  1.416e+00, 1.411e+00, 1.406e+00, 1.402e+00, 1.399e+00/
      data mrtab(101:150)/
     &  1.395e+00, 1.392e+00, 1.389e+00, 1.386e+00, 1.384e+00,
     &  1.381e+00, 1.379e+00, 1.377e+00, 1.375e+00, 1.373e+00,
     &  1.371e+00, 1.370e+00, 1.368e+00, 1.367e+00, 1.365e+00,
     &  1.364e+00, 1.363e+00, 1.362e+00, 1.360e+00, 1.359e+00,
     &  1.358e+00, 1.357e+00, 1.356e+00, 1.355e+00, 1.354e+00,
     &  1.354e+00, 1.353e+00, 1.352e+00, 1.351e+00, 1.350e+00,
     &  1.350e+00, 1.349e+00, 1.348e+00, 1.348e+00, 1.347e+00,
     &  1.347e+00, 1.346e+00, 1.346e+00, 1.345e+00, 1.344e+00,
     &  1.344e+00, 1.343e+00, 1.343e+00, 1.342e+00, 1.342e+00,
     &  1.341e+00, 1.341e+00, 1.341e+00, 1.340e+00, 1.340e+00/
      data mrtab(151:200)/
     &  1.339e+00, 1.339e+00, 1.339e+00, 1.338e+00, 1.338e+00,
     &  1.338e+00, 1.337e+00, 1.337e+00, 1.337e+00, 1.336e+00,
     &  1.336e+00, 1.336e+00, 1.335e+00, 1.335e+00, 1.335e+00,
     &  1.334e+00, 1.334e+00, 1.334e+00, 1.334e+00, 1.333e+00,
     &  1.333e+00, 1.333e+00, 1.333e+00, 1.332e+00, 1.332e+00,
     &  1.332e+00, 1.332e+00, 1.331e+00, 1.331e+00, 1.331e+00,
     &  1.331e+00, 1.330e+00, 1.330e+00, 1.330e+00, 1.330e+00,
     &  1.330e+00, 1.329e+00, 1.329e+00, 1.329e+00, 1.329e+00,
     &  1.329e+00, 1.329e+00, 1.328e+00, 1.328e+00, 1.328e+00,
     &  1.328e+00, 1.328e+00, 1.328e+00, 1.328e+00, 1.327e+00/
      data mrtab(201:250)/
     &  1.327e+00, 1.327e+00, 1.327e+00, 1.327e+00, 1.327e+00,
     &  1.327e+00, 1.326e+00, 1.326e+00, 1.326e+00, 1.326e+00,
     &  1.326e+00, 1.326e+00, 1.326e+00, 1.326e+00, 1.325e+00,
     &  1.325e+00, 1.325e+00, 1.325e+00, 1.325e+00, 1.325e+00,
     &  1.325e+00, 1.325e+00, 1.325e+00, 1.324e+00, 1.324e+00,
     &  1.324e+00, 1.324e+00, 1.324e+00, 1.324e+00, 1.324e+00,
     &  1.324e+00, 1.324e+00, 1.323e+00, 1.323e+00, 1.323e+00,
     &  1.323e+00, 1.323e+00, 1.323e+00, 1.323e+00, 1.323e+00,
     &  1.323e+00, 1.323e+00, 1.322e+00, 1.322e+00, 1.322e+00,
     &  1.322e+00, 1.322e+00, 1.322e+00, 1.322e+00, 1.322e+00/
      data mrtab(251:300)/
     &  1.322e+00, 1.322e+00, 1.321e+00, 1.321e+00, 1.321e+00,
     &  1.321e+00, 1.321e+00, 1.320e+00, 1.320e+00, 1.320e+00,
     &  1.320e+00, 1.320e+00, 1.319e+00, 1.319e+00, 1.319e+00,
     &  1.319e+00, 1.319e+00, 1.319e+00, 1.318e+00, 1.318e+00,
     &  1.318e+00, 1.318e+00, 1.318e+00, 1.317e+00, 1.317e+00,
     &  1.317e+00, 1.317e+00, 1.317e+00, 1.316e+00, 1.316e+00,
     &  1.316e+00, 1.316e+00, 1.316e+00, 1.315e+00, 1.315e+00,
     &  1.315e+00, 1.315e+00, 1.315e+00, 1.314e+00, 1.314e+00,
     &  1.314e+00, 1.314e+00, 1.314e+00, 1.313e+00, 1.313e+00,
     &  1.313e+00, 1.313e+00, 1.313e+00, 1.313e+00, 1.312e+00/
      data mrtab(301:350)/
     &  1.312e+00, 1.312e+00, 1.312e+00, 1.311e+00, 1.311e+00,
     &  1.311e+00, 1.311e+00, 1.310e+00, 1.310e+00, 1.310e+00,
     &  1.310e+00, 1.309e+00, 1.309e+00, 1.309e+00, 1.309e+00,
     &  1.308e+00, 1.308e+00, 1.308e+00, 1.307e+00, 1.307e+00,
     &  1.307e+00, 1.306e+00, 1.306e+00, 1.306e+00, 1.305e+00,
     &  1.305e+00, 1.305e+00, 1.304e+00, 1.304e+00, 1.304e+00,
     &  1.303e+00, 1.303e+00, 1.303e+00, 1.302e+00, 1.302e+00,
     &  1.301e+00, 1.301e+00, 1.300e+00, 1.300e+00, 1.300e+00,
     &  1.299e+00, 1.299e+00, 1.299e+00, 1.299e+00, 1.299e+00,
     &  1.298e+00, 1.298e+00, 1.298e+00, 1.298e+00, 1.297e+00/
      data mrtab(351:400)/
     &  1.297e+00, 1.296e+00, 1.296e+00, 1.296e+00, 1.295e+00,
     &  1.294e+00, 1.294e+00, 1.293e+00, 1.293e+00, 1.292e+00,
     &  1.292e+00, 1.291e+00, 1.291e+00, 1.290e+00, 1.290e+00,
     &  1.289e+00, 1.288e+00, 1.288e+00, 1.287e+00, 1.286e+00,
     &  1.286e+00, 1.285e+00, 1.284e+00, 1.284e+00, 1.283e+00,
     &  1.282e+00, 1.281e+00, 1.280e+00, 1.280e+00, 1.279e+00,
     &  1.278e+00, 1.277e+00, 1.276e+00, 1.275e+00, 1.274e+00,
     &  1.273e+00, 1.272e+00, 1.271e+00, 1.270e+00, 1.269e+00,
     &  1.268e+00, 1.266e+00, 1.265e+00, 1.264e+00, 1.263e+00,
     &  1.261e+00, 1.260e+00, 1.258e+00, 1.257e+00, 1.255e+00/
      data mrtab(401:450)/
     &  1.253e+00, 1.251e+00, 1.250e+00, 1.248e+00, 1.246e+00,
     &  1.243e+00, 1.240e+00, 1.239e+00, 1.237e+00, 1.235e+00,
     &  1.234e+00, 1.232e+00, 1.228e+00, 1.225e+00, 1.223e+00,
     &  1.221e+00, 1.218e+00, 1.216e+00, 1.210e+00, 1.207e+00,
     &  1.203e+00, 1.199e+00, 1.195e+00, 1.191e+00, 1.188e+00,
     &  1.180e+00, 1.175e+00, 1.169e+00, 1.161e+00, 1.153e+00,
     &  1.150e+00, 1.142e+00, 1.136e+00, 1.133e+00, 1.132e+00,
     &  1.133e+00, 1.131e+00, 1.128e+00, 1.128e+00, 1.129e+00,
     &  1.128e+00, 1.126e+00, 1.125e+00, 1.128e+00, 1.133e+00,
     &  1.142e+00, 1.146e+00, 1.152e+00, 1.162e+00, 1.178e+00/
      data mrtab(451:500)/
     &  1.185e+00, 1.196e+00, 1.208e+00, 1.230e+00, 1.240e+00,
     &  1.252e+00, 1.264e+00, 1.286e+00, 1.298e+00, 1.308e+00,
     &  1.326e+00, 1.335e+00, 1.353e+00, 1.412e+00, 1.452e+00,
     &  1.467e+00, 1.462e+00, 1.449e+00, 1.433e+00, 1.417e+00,
     &  1.405e+00, 1.393e+00, 1.384e+00, 1.376e+00, 1.369e+00,
     &  1.363e+00, 1.357e+00, 1.352e+00, 1.347e+00, 1.344e+00,
     &  1.340e+00, 1.337e+00, 1.334e+00, 1.331e+00, 1.329e+00,
     &  1.326e+00, 1.324e+00, 1.322e+00, 1.320e+00, 1.318e+00,
     &  1.316e+00, 1.315e+00, 1.314e+00, 1.312e+00, 1.312e+00,
     &  1.312e+00, 1.311e+00, 1.311e+00, 1.311e+00, 1.310e+00/
      data mrtab(501:550)/
     &  1.309e+00, 1.307e+00, 1.306e+00, 1.304e+00, 1.302e+00,
     &  1.300e+00, 1.298e+00, 1.295e+00, 1.292e+00, 1.289e+00,
     &  1.286e+00, 1.282e+00, 1.278e+00, 1.274e+00, 1.269e+00,
     &  1.263e+00, 1.257e+00, 1.248e+00, 1.242e+00, 1.235e+00,
     &  1.229e+00, 1.232e+00, 1.243e+00, 1.268e+00, 1.295e+00,
     &  1.330e+00, 1.342e+00, 1.340e+00, 1.336e+00, 1.329e+00,
     &  1.325e+00, 1.320e+00, 1.318e+00, 1.315e+00, 1.311e+00,
     &  1.309e+00, 1.307e+00, 1.305e+00, 1.302e+00, 1.300e+00,
     &  1.298e+00, 1.297e+00, 1.295e+00, 1.294e+00, 1.292e+00,
     &  1.291e+00, 1.289e+00, 1.288e+00, 1.287e+00, 1.285e+00/
      data mrtab(551:600)/
     &  1.284e+00, 1.283e+00, 1.281e+00, 1.280e+00, 1.279e+00,
     &  1.277e+00, 1.276e+00, 1.275e+00, 1.273e+00, 1.272e+00,
     &  1.271e+00, 1.270e+00, 1.268e+00, 1.267e+00, 1.266e+00,
     &  1.264e+00, 1.263e+00, 1.261e+00, 1.260e+00, 1.258e+00,
     &  1.257e+00, 1.255e+00, 1.254e+00, 1.252e+00, 1.251e+00,
     &  1.249e+00, 1.247e+00, 1.246e+00, 1.245e+00, 1.243e+00,
     &  1.241e+00, 1.239e+00, 1.238e+00, 1.236e+00, 1.234e+00,
     &  1.233e+00, 1.230e+00, 1.229e+00, 1.227e+00, 1.224e+00,
     &  1.223e+00, 1.221e+00, 1.218e+00, 1.216e+00, 1.214e+00,
     &  1.211e+00, 1.209e+00, 1.207e+00, 1.204e+00, 1.202e+00/
      data mrtab(601:650)/
     &  1.199e+00, 1.196e+00, 1.193e+00, 1.190e+00, 1.187e+00,
     &  1.184e+00, 1.181e+00, 1.178e+00, 1.174e+00, 1.171e+00,
     &  1.167e+00, 1.164e+00, 1.161e+00, 1.157e+00, 1.154e+00,
     &  1.150e+00, 1.147e+00, 1.144e+00, 1.140e+00, 1.137e+00,
     &  1.134e+00, 1.131e+00, 1.129e+00, 1.125e+00, 1.122e+00,
     &  1.119e+00, 1.116e+00, 1.113e+00, 1.110e+00, 1.108e+00,
     &  1.105e+00, 1.103e+00, 1.101e+00, 1.098e+00, 1.097e+00,
     &  1.096e+00, 1.094e+00, 1.092e+00, 1.091e+00, 1.089e+00,
     &  1.086e+00, 1.086e+00, 1.087e+00, 1.088e+00, 1.088e+00,
     &  1.087e+00, 1.087e+00, 1.090e+00, 1.091e+00, 1.091e+00/
      data mrtab(651:700)/
     &  1.092e+00, 1.096e+00, 1.098e+00, 1.100e+00, 1.101e+00,
     &  1.105e+00, 1.107e+00, 1.109e+00, 1.110e+00, 1.114e+00,
     &  1.117e+00, 1.118e+00, 1.122e+00, 1.125e+00, 1.126e+00,
     &  1.131e+00, 1.134e+00, 1.136e+00, 1.140e+00, 1.141e+00,
     &  1.146e+00, 1.150e+00, 1.152e+00, 1.156e+00, 1.160e+00,
     &  1.162e+00, 1.166e+00, 1.168e+00, 1.172e+00, 1.174e+00,
     &  1.179e+00, 1.182e+00, 1.185e+00, 1.189e+00, 1.191e+00,
     &  1.196e+00, 1.198e+00, 1.202e+00, 1.204e+00, 1.207e+00,
     &  1.209e+00, 1.214e+00, 1.215e+00, 1.219e+00, 1.221e+00,
     &  1.226e+00, 1.228e+00, 1.232e+00, 1.234e+00, 1.238e+00/
      data mrtab(701:750)/
     &  1.239e+00, 1.243e+00, 1.245e+00, 1.249e+00, 1.252e+00,
     &  1.254e+00, 1.257e+00, 1.259e+00, 1.263e+00, 1.265e+00,
     &  1.268e+00, 1.270e+00, 1.273e+00, 1.276e+00, 1.278e+00,
     &  1.283e+00, 1.285e+00, 1.287e+00, 1.291e+00, 1.293e+00,
     &  1.295e+00, 1.299e+00, 1.301e+00, 1.303e+00, 1.307e+00,
     &  1.309e+00, 1.313e+00, 1.315e+00, 1.317e+00, 1.321e+00,
     &  1.323e+00, 1.325e+00, 1.327e+00, 1.331e+00, 1.333e+00,
     &  1.335e+00, 1.339e+00, 1.341e+00, 1.343e+00, 1.347e+00,
     &  1.350e+00, 1.351e+00, 1.353e+00, 1.357e+00, 1.359e+00,
     &  1.361e+00, 1.365e+00, 1.367e+00, 1.369e+00, 1.371e+00/
      data mrtab(751:800)/
     &  1.375e+00, 1.377e+00, 1.379e+00, 1.380e+00, 1.384e+00,
     &  1.386e+00, 1.388e+00, 1.389e+00, 1.391e+00, 1.394e+00,
     &  1.396e+00, 1.398e+00, 1.400e+00, 1.401e+00, 1.405e+00,
     &  1.407e+00, 1.409e+00, 1.410e+00, 1.412e+00, 1.415e+00,
     &  1.418e+00, 1.420e+00, 1.422e+00, 1.423e+00, 1.426e+00,
     &  1.428e+00, 1.430e+00, 1.431e+00, 1.433e+00, 1.435e+00,
     &  1.436e+00, 1.440e+00, 1.442e+00, 1.443e+00, 1.444e+00,
     &  1.445e+00, 1.447e+00, 1.448e+00, 1.450e+00, 1.452e+00,
     &  1.454e+00, 1.456e+00, 1.457e+00, 1.458e+00, 1.459e+00,
     &  1.460e+00, 1.460e+00, 1.463e+00, 1.465e+00, 1.467e+00/
      data mrtab(801:860)/
     &  1.467e+00, 1.467e+00, 1.468e+00, 1.484e+00, 1.499e+00,
     &  1.516e+00, 1.529e+00, 1.538e+00, 1.544e+00, 1.547e+00,
     &  1.546e+00, 1.543e+00, 1.536e+00, 1.527e+00, 1.519e+00,
     &  1.512e+00, 1.506e+00, 1.499e+00, 1.493e+00, 1.487e+00,
     &  1.481e+00, 1.478e+00, 1.477e+00, 1.476e+00, 1.477e+00,
     &  1.481e+00, 1.485e+00, 1.492e+00, 1.499e+00, 1.509e+00,
     &  1.520e+00, 1.531e+00, 1.542e+00, 1.567e+00, 1.594e+00,
     &  1.619e+00, 1.644e+00, 1.669e+00, 1.690e+00, 1.710e+00,
     &  1.729e+00, 1.747e+00, 1.763e+00, 1.777e+00, 1.791e+00,
     &  1.806e+00, 1.819e+00, 1.831e+00, 1.842e+00, 1.852e+00,
     &  1.860e+00, 1.867e+00, 1.874e+00, 1.881e+00, 1.886e+00,
     &  1.891e+00, 1.895e+00, 1.899e+00, 1.908e+00, 1.912e+00/

      data mitab(1:50)/
     &  1.745e-03, 2.370e-03, 3.146e-03, 4.072e-03, 5.174e-03,
     &  7.958e-03, 1.164e-02, 1.636e-02, 2.227e-02, 2.950e-02,
     &  3.818e-02, 4.850e-02, 6.064e-02, 7.461e-02, 9.074e-02,
     &  1.093e-01, 1.303e-01, 1.534e-01, 1.798e-01, 2.088e-01,
     &  2.414e-01, 2.766e-01, 2.998e-01, 3.154e-01, 3.310e-01,
     &  3.498e-01, 3.739e-01, 4.119e-01, 4.558e-01, 5.033e-01,
     &  5.355e-01, 5.634e-01, 5.792e-01, 5.859e-01, 5.805e-01,
     &  5.859e-01, 5.981e-01, 6.135e-01, 6.292e-01, 6.453e-01,
     &  6.573e-01, 6.573e-01, 6.528e-01, 6.439e-01, 6.292e-01,
     &  6.050e-01, 5.752e-01, 5.430e-01, 5.185e-01, 4.929e-01/
      data mitab(51:100)/
     &  4.708e-01, 4.485e-01, 4.303e-01, 4.148e-01, 3.988e-01,
     &  3.826e-01, 3.705e-01, 3.596e-01, 3.490e-01, 3.387e-01,
     &  3.220e-01, 2.876e-01, 2.392e-01, 1.870e-01, 1.489e-01,
     &  1.333e-01, 1.422e-01, 1.678e-01, 1.927e-01, 2.167e-01,
     &  2.409e-01, 2.641e-01, 2.772e-01, 2.772e-01, 2.581e-01,
     &  2.338e-01, 2.022e-01, 1.670e-01, 1.351e-01, 1.039e-01,
     &  7.241e-02, 3.997e-02, 3.998e-03, 2.004e-03, 1.182e-03,
     &  8.391e-04, 5.995e-05, 1.250e-06, 3.622e-07, 1.850e-07,
     &  1.101e-07, 6.711e-08, 3.844e-08, 1.999e-08, 1.270e-08,
     &  1.158e-08, 1.101e-08, 1.071e-08, 1.049e-08, 9.903e-09/
      data mitab(101:150)/
     &  9.307e-09, 8.606e-09, 7.994e-09, 7.444e-09, 6.852e-09,
     &  6.292e-09, 5.792e-09, 5.405e-09, 4.795e-09, 4.403e-09,
     &  4.148e-09, 3.826e-09, 3.546e-09, 3.325e-09, 3.190e-09,
     &  3.082e-09, 2.984e-09, 2.883e-09, 2.766e-09, 2.653e-09,
     &  2.528e-09, 2.420e-09, 2.316e-09, 2.217e-09, 2.117e-09,
     &  2.031e-09, 1.940e-09, 1.840e-09, 1.761e-09, 1.663e-09,
     &  1.581e-09, 1.489e-09, 1.422e-09, 1.339e-09, 1.258e-09,
     &  1.169e-09, 1.088e-09, 1.018e-09, 9.393e-10, 8.685e-10,
     &  8.087e-10, 7.795e-10, 7.600e-10, 7.495e-10, 7.291e-10,
     &  7.011e-10, 7.092e-10, 7.158e-10, 7.342e-10, 7.849e-10/
      data mitab(151:200)/
     &  9.242e-10, 1.078e-09, 1.267e-09, 1.461e-09, 1.570e-09,
     &  1.640e-09, 1.757e-09, 1.887e-09, 2.098e-09, 2.269e-09,
     &  2.442e-09, 2.659e-09, 2.869e-09, 3.132e-09, 3.434e-09,
     &  3.844e-09, 4.434e-09, 5.221e-09, 6.365e-09, 7.723e-09,
     &  9.634e-09, 1.132e-08, 1.238e-08, 1.330e-08, 1.399e-08,
     &  1.472e-08, 1.502e-08, 1.552e-08, 1.570e-08, 1.606e-08,
     &  1.674e-08, 1.777e-08, 1.940e-08, 2.031e-08, 2.098e-08,
     &  2.177e-08, 2.300e-08, 2.470e-08, 2.653e-08, 2.963e-08,
     &  3.348e-08, 4.100e-08, 4.998e-08, 5.995e-08, 7.291e-08,
     &  9.137e-08, 1.150e-07, 1.348e-07, 1.458e-07, 1.530e-07/
      data mitab(201:250)/
     &  1.559e-07, 1.580e-07, 1.580e-07, 1.570e-07, 1.527e-07,
     &  1.478e-07, 1.409e-07, 1.339e-07, 1.282e-07, 1.258e-07,
     &  1.250e-07, 1.270e-07, 1.330e-07, 1.448e-07, 1.621e-07,
     &  1.819e-07, 2.041e-07, 2.243e-07, 2.459e-07, 2.690e-07,
     &  2.929e-07, 3.153e-07, 3.348e-07, 3.546e-07, 3.748e-07,
     &  3.907e-07, 4.053e-07, 4.234e-07, 4.403e-07, 4.622e-07,
     &  4.862e-07, 5.150e-07, 5.699e-07, 6.696e-07, 8.304e-07,
     &  1.060e-06, 1.368e-06, 1.771e-06, 2.169e-06, 2.557e-06,
     &  2.932e-06, 3.190e-06, 3.358e-06, 3.464e-06, 3.502e-06,
     &  3.480e-06, 3.418e-06, 3.337e-06, 3.253e-06, 3.131e-06/
      data mitab(251:300)/
     &  3.000e-06, 2.688e-06, 2.352e-06, 2.001e-06, 1.690e-06,
     &  1.419e-06, 1.299e-06, 1.259e-06, 1.329e-06, 1.499e-06,
     &  1.708e-06, 2.038e-06, 2.628e-06, 3.869e-06, 5.951e-06,
     &  9.306e-06, 1.069e-05, 1.120e-05, 1.160e-05, 1.181e-05,
     &  1.199e-05, 1.191e-05, 1.179e-05, 1.160e-05, 1.139e-05,
     &  1.100e-05, 1.079e-05, 1.090e-05, 1.139e-05, 1.221e-05,
     &  1.400e-05, 1.639e-05, 1.912e-05, 2.251e-05, 2.849e-05,
     &  4.047e-05, 4.505e-05, 5.804e-05, 7.802e-05, 1.060e-04,
     &  1.530e-04, 2.540e-04, 3.197e-04, 3.538e-04, 3.629e-04,
     &  3.637e-04, 3.604e-04, 3.387e-04, 3.018e-04, 2.659e-04/
      data mitab(301:350)/
     &  2.248e-04, 1.958e-04, 1.741e-04, 1.602e-04, 1.441e-04,
     &  1.348e-04, 1.240e-04, 1.140e-04, 1.071e-04, 9.940e-05,
     &  9.347e-05, 8.804e-05, 8.310e-05, 8.096e-05, 7.903e-05,
     &  7.591e-05, 7.398e-05, 7.404e-05, 7.495e-05, 7.601e-05,
     &  7.743e-05, 8.050e-05, 8.410e-05, 8.900e-05, 9.510e-05,
     &  1.000e-04, 1.051e-04, 1.120e-04, 1.219e-04, 1.330e-04,
     &  1.359e-04, 1.371e-04, 1.380e-04, 1.418e-04, 1.552e-04,
     &  1.861e-04, 3.205e-04, 5.209e-04, 7.224e-04, 9.221e-04,
     &  1.161e-03, 1.678e-03, 1.827e-03, 1.922e-03, 1.909e-03,
     &  1.848e-03, 1.717e-03, 1.548e-03, 1.402e-03, 1.250e-03/
      data mitab(351:400)/
     &  1.101e-03, 9.904e-04, 8.888e-04, 8.050e-04, 7.392e-04,
     &  6.742e-04, 6.206e-04, 5.725e-04, 5.294e-04, 4.884e-04,
     &  4.643e-04, 4.403e-04, 4.176e-04, 3.970e-04, 3.826e-04,
     &  3.705e-04, 3.588e-04, 3.506e-04, 3.434e-04, 3.395e-04,
     &  3.379e-04, 3.387e-04, 3.410e-04, 3.458e-04, 3.571e-04,
     &  3.739e-04, 3.898e-04, 4.081e-04, 4.293e-04, 4.506e-04,
     &  4.686e-04, 4.918e-04, 5.114e-04, 5.430e-04, 5.995e-04,
     &  6.365e-04, 6.852e-04, 7.427e-04, 7.921e-04, 8.488e-04,
     &  9.095e-04, 9.904e-04, 1.071e-03, 1.150e-03, 1.250e-03,
     &  1.348e-03, 1.472e-03, 1.581e-03, 1.709e-03, 1.811e-03/
      data mitab(401:450)/
     &  1.900e-03, 1.953e-03, 1.990e-03, 2.017e-03, 2.069e-03,
     &  2.142e-03, 2.269e-03, 2.311e-03, 2.338e-03, 2.387e-03,
     &  2.425e-03, 2.476e-03, 2.575e-03, 2.703e-03, 2.977e-03,
     &  3.302e-03, 4.016e-03, 4.363e-03, 4.828e-03, 5.368e-03,
     &  6.278e-03, 7.325e-03, 8.547e-03, 1.049e-02, 1.270e-02,
     &  1.451e-02, 1.640e-02, 1.861e-02, 2.050e-02, 2.817e-02,
     &  3.800e-02, 4.622e-02, 5.480e-02, 6.483e-02, 7.444e-02,
     &  8.352e-02, 9.285e-02, 1.020e-01, 1.119e-01, 1.210e-01,
     &  1.312e-01, 1.422e-01, 1.541e-01, 1.670e-01, 1.798e-01,
     &  1.940e-01, 2.060e-01, 2.182e-01, 2.290e-01, 2.392e-01/
      data mitab(451:500)/
     &  2.493e-01, 2.581e-01, 2.647e-01, 2.715e-01, 2.759e-01,
     &  2.798e-01, 2.804e-01, 2.823e-01, 2.817e-01, 2.785e-01,
     &  2.759e-01, 2.721e-01, 2.721e-01, 2.398e-01, 1.918e-01,
     &  1.348e-01, 9.242e-02, 6.107e-02, 3.688e-02, 2.611e-02,
     &  1.949e-02, 1.321e-02, 9.393e-03, 6.789e-03, 5.150e-03,
     &  4.234e-03, 3.596e-03, 3.402e-03, 3.402e-03, 3.530e-03,
     &  3.800e-03, 4.157e-03, 4.600e-03, 5.067e-03, 5.621e-03,
     &  6.220e-03, 6.883e-03, 7.600e-03, 8.449e-03, 9.307e-03,
     &  1.030e-02, 1.140e-02, 1.238e-02, 1.361e-02, 1.472e-02,
     &  1.548e-02, 1.570e-02, 1.552e-02, 1.499e-02, 1.441e-02/
      data mitab(501:550)/
     &  1.370e-02, 1.312e-02, 1.241e-02, 1.180e-02, 1.111e-02,
     &  1.061e-02, 1.011e-02, 9.904e-03, 9.790e-03, 9.881e-03,
     &  1.030e-02, 1.078e-02, 1.158e-02, 1.258e-02, 1.418e-02,
     &  1.659e-02, 2.031e-02, 2.482e-02, 3.295e-02, 4.323e-02,
     &  6.220e-02, 8.646e-02, 1.069e-01, 1.250e-01, 1.308e-01,
     &  1.172e-01, 8.786e-02, 6.947e-02, 5.699e-02, 4.952e-02,
     &  4.485e-02, 4.176e-02, 3.925e-02, 3.731e-02, 3.563e-02,
     &  3.450e-02, 3.371e-02, 3.310e-02, 3.272e-02, 3.242e-02,
     &  3.220e-02, 3.212e-02, 3.197e-02, 3.190e-02, 3.197e-02,
     &  3.205e-02, 3.205e-02, 3.220e-02, 3.220e-02, 3.227e-02/
      data mitab(551:600)/
     &  3.242e-02, 3.249e-02, 3.257e-02, 3.272e-02, 3.279e-02,
     &  3.302e-02, 3.310e-02, 3.325e-02, 3.348e-02, 3.371e-02,
     &  3.395e-02, 3.410e-02, 3.426e-02, 3.450e-02, 3.466e-02,
     &  3.490e-02, 3.514e-02, 3.530e-02, 3.563e-02, 3.579e-02,
     &  3.604e-02, 3.637e-02, 3.654e-02, 3.688e-02, 3.714e-02,
     &  3.748e-02, 3.783e-02, 3.809e-02, 3.844e-02, 3.880e-02,
     &  3.916e-02, 3.952e-02, 3.988e-02, 4.034e-02, 4.072e-02,
     &  4.109e-02, 4.147e-02, 4.196e-02, 4.234e-02, 4.293e-02,
     &  4.333e-02, 4.373e-02, 4.434e-02, 4.475e-02, 4.537e-02,
     &  4.600e-02, 4.664e-02, 4.718e-02, 4.784e-02, 4.850e-02/
      data mitab(601:650)/
     &  4.929e-02, 4.998e-02, 5.079e-02, 5.174e-02, 5.270e-02,
     &  5.380e-02, 5.805e-02, 5.634e-02, 5.845e-02, 5.995e-02,
     &  6.191e-02, 6.394e-02, 6.619e-02, 6.852e-02, 7.092e-02,
     &  7.358e-02, 7.652e-02, 7.958e-02, 8.295e-02, 8.646e-02,
     &  8.970e-02, 9.328e-02, 9.678e-02, 9.995e-02, 1.039e-01,
     &  1.083e-01, 1.129e-01, 1.172e-01, 1.218e-01, 1.270e-01,
     &  1.321e-01, 1.370e-01, 1.422e-01, 1.472e-01, 1.520e-01,
     &  1.570e-01, 1.621e-01, 1.678e-01, 1.741e-01, 1.802e-01,
     &  1.865e-01, 1.927e-01, 1.990e-01, 2.055e-01, 2.112e-01,
     &  2.177e-01, 2.238e-01, 2.295e-01, 2.359e-01, 2.420e-01/
      data mitab(651:700)/
     &  2.476e-01, 2.528e-01, 2.593e-01, 2.641e-01, 2.690e-01,
     &  2.740e-01, 2.791e-01, 2.837e-01, 2.883e-01, 2.929e-01,
     &  2.977e-01, 3.012e-01, 3.060e-01, 3.103e-01, 3.139e-01,
     &  3.183e-01, 3.227e-01, 3.257e-01, 3.295e-01, 3.325e-01,
     &  3.363e-01, 3.402e-01, 3.426e-01, 3.466e-01, 3.490e-01,
     &  3.514e-01, 3.546e-01, 3.571e-01, 3.596e-01, 3.621e-01,
     &  3.646e-01, 3.679e-01, 3.696e-01, 3.722e-01, 3.739e-01,
     &  3.757e-01, 3.774e-01, 3.791e-01, 3.809e-01, 3.826e-01,
     &  3.844e-01, 3.862e-01, 3.871e-01, 3.897e-01, 3.916e-01,
     &  3.934e-01, 3.943e-01, 3.961e-01, 3.970e-01, 3.988e-01/
      data mitab(701:750)/
     &  3.997e-01, 4.016e-01, 4.025e-01, 4.044e-01, 4.053e-01,
     &  4.053e-01, 4.072e-01, 4.081e-01, 4.100e-01, 4.100e-01,
     &  4.119e-01, 4.128e-01, 4.138e-01, 4.148e-01, 4.157e-01,
     &  4.176e-01, 4.176e-01, 4.186e-01, 4.196e-01, 4.205e-01,
     &  4.205e-01, 4.225e-01, 4.225e-01, 4.225e-01, 4.234e-01,
     &  4.244e-01, 4.254e-01, 4.254e-01, 4.254e-01, 4.264e-01,
     &  4.264e-01, 4.274e-01, 4.274e-01, 4.283e-01, 4.283e-01,
     &  4.283e-01, 4.293e-01, 4.293e-01, 4.293e-01, 4.303e-01,
     &  4.293e-01, 4.293e-01, 4.293e-01, 4.303e-01, 4.303e-01,
     &  4.293e-01, 4.303e-01, 4.303e-01, 4.293e-01, 4.293e-01/
      data mitab(751:800)/
     &  4.293e-01, 4.293e-01, 4.283e-01, 4.283e-01, 4.283e-01,
     &  4.283e-01, 4.274e-01, 4.274e-01, 4.264e-01, 4.274e-01,
     &  4.264e-01, 4.264e-01, 4.254e-01, 4.254e-01, 4.254e-01,
     &  4.254e-01, 4.244e-01, 4.244e-01, 4.234e-01, 4.234e-01,
     &  4.234e-01, 4.225e-01, 4.215e-01, 4.205e-01, 4.205e-01,
     &  4.196e-01, 4.186e-01, 4.176e-01, 4.176e-01, 4.167e-01,
     &  4.157e-01, 4.157e-01, 4.138e-01, 4.128e-01, 4.119e-01,
     &  4.109e-01, 4.100e-01, 4.091e-01, 4.091e-01, 4.081e-01,
     &  4.072e-01, 4.062e-01, 4.044e-01, 4.034e-01, 4.025e-01,
     &  4.016e-01, 4.007e-01, 4.007e-01, 3.988e-01, 3.970e-01/
      data mitab(801:860)/
     &  3.952e-01, 3.943e-01, 3.934e-01, 3.818e-01, 3.722e-01,
     &  3.629e-01, 3.482e-01, 3.356e-01, 3.227e-01, 3.103e-01,
     &  2.991e-01, 2.889e-01, 2.817e-01, 2.791e-01, 2.798e-01,
     &  2.830e-01, 2.863e-01, 2.916e-01, 2.991e-01, 3.068e-01,
     &  3.190e-01, 3.317e-01, 3.442e-01, 3.588e-01, 3.739e-01,
     &  3.880e-01, 4.025e-01, 4.176e-01, 4.323e-01, 4.465e-01,
     &  4.579e-01, 4.675e-01, 4.773e-01, 4.975e-01, 5.079e-01,
     &  5.162e-01, 5.233e-01, 5.258e-01, 5.245e-01, 5.245e-01,
     &  5.233e-01, 5.209e-01, 5.174e-01, 5.138e-01, 5.103e-01,
     &  5.079e-01, 5.021e-01, 4.964e-01, 4.907e-01, 4.839e-01,
     &  4.773e-01, 4.718e-01, 4.654e-01, 4.600e-01, 4.548e-01,
     &  4.485e-01, 4.434e-01, 4.383e-01, 4.176e-01, 4.167e-01/

      call locate(wltab,ntab,wl,i)
      wt=(wl-wltab(i))/(wltab(i+1)-wltab(i))
      wt=max(0.0_kr,min(1.0_kr,wt))     ! no extrapolation
      nr=mrtab(i)*(mrtab(i+1)/mrtab(i))**wt
      ni=mitab(i)*(mitab(i+1)/mitab(i))**wt

c 
c correction to be applied to the index of refraction and to the extinction 
c coefficients of the pure water to obtain the ocean water one (see for 
c example friedman). by default, a typical sea water is assumed 
c (salinity=34.3ppt, chlorinity=19ppt) as reported by sverdrup. 
c in that case there is no correction for the extinction coefficient between 
c 0.25 and 4 microns. for the index of refraction, a correction of +0.006 
c has to be applied (mclellan). for a chlorinity of 19.0ppt the correction 
c is a linear function of the salt concentration. 

c references:
c friedman d., applied optics, 1969, vol.8, no.10, pp.2073-2078.
c mclellan h.j., elements of physical oceanography, pergamon press, inc.,
c        new-york, 1965, p 129.
c sverdrup h.v. et al., the oceans (prentice-hall, inc., englewood cliffs,
c        n.j., 1942, p 173.

        nr=nr+nrc*(xsal/34.3)
        ni=ni+nic*(xsal/34.3)

        return
        end
c
c=======================================================================
      subroutine sunglint(wndspd,nr,ni,csin,cvin,phi,rog)

c input parameters:   wndspd=speed of the wind (in m/s)
c                     nr=index of refraction of the sea water
c                     ni=extinction coefficient of the sea water
c                     cs=cosine of solar zenith angle
c                     cv=cosine of view zenith angle
c                     phi=relative azimuth (sun-satellite) (in radians)
c
c output parameters:  rog=reflectance of the sun glint
c
      use params, only: kr, pi, zip
      implicit none

      real(kr), intent(in) :: wndspd,nr,ni,csin,cvin,phi
      real(kr), intent(out):: rog

      real(kr) ::  d2r,phw,cs,cv,ss,sv,zx,zy,tantilt,tilt,proba,
     &     xe,xn,xe2,xn2,coef,cos2chi,coschi,sinchi,r1,sigmac,
     &     sigmau,c21,c03,c40,c04,c22,axe2,axn2,axe4,axn4,axe2xn2

      d2r=pi/180.

c  no wave shodowing, so set a floor on incident and viewing angle 

      cs=max(csin,0.05_kr) 
      cv=max(cvin,0.05_kr) 
      ss=sqrt(1.-cs**2)
      sv=sqrt(1.-cv**2)

      zx=-sv*sin(pi-phi)/(cs+cv)
      zy=(ss+sv*cos(pi-phi))/(cs+cv)
      tantilt=sqrt(zx*zx+zy*zy)
      tilt=atan(tantilt)

c  anisotropic gaussian distribution

      sigmac=0.003+0.00192*wndspd
      sigmau=0.00316*wndspd
      c21=0.01-0.0086*wndspd
      c03=0.04-0.033*wndspd
      c40=0.40
      c22=0.12
      c04=0.23

                                ! brdf dependence on wndang, 
                                ! original code from 6s:
                                
c     phw=wndang*d2r            ! phw=phi_sun-phi_wind
c     xe=(cos(phw)*zx+sin(phw)*zy)/sqrt(sigmac)
c     xn=(-sin(phw)*zx+cos(phw)*zy)/sqrt(sigmau)
c     xe2=xe*xe
c     xn2=xn*xn
c     coef=1-c21/2.*(xe2-1)*xn-c03/6.*(xn2-3)*xn
c     coef=coef+c40/24.*(xe2*xe2-6*xe2+3)
c     coef=coef+c04/24.*(xn2*xn2-6*xn2+3)
c     coef=coef+c22/4.*(xe2-1)*(xn2-1)
c     coef=coef/(2.*pi*sqrt(sigmau)*sqrt(sigmac))
c     proba=coef*exp(-(xe2+xn2)/2.)

                                ! angle average over wind directions
                                ! axe2=<xe**2> averaged over all angles phw
                                ! <xe**3>=<xn**3>=0
                                ! <xe>=<xn>=<xn2*xe>=<xe2*xn>=0
                                ! <sin2>=1/2 <sin4>=3/8 <sin2 cos2>=1/8

      axe2=.5*(zx**2+zy**2)/sigmac
      axn2=.5*(zx**2+zy**2)/sigmau
      axe4=(3*zx**4+6*zx**2*zy**2+3*zy**4)/(8*sigmac**2)
      axn4=(3*zx**4+6*zx**2*zy**2+3*zy**4)/(8*sigmau**2)
      axe2xn2=(zx**4+10*zx**2*zy**2+zy**4)/(8*sigmau*sigmac)        
      coef=1.
      coef=coef+c40/24.*(axe4-6*axe2+3)
      coef=coef+c04/24.*(axn4-6*axn2+3)
      coef=coef+c22/4.*(axe2xn2-axn2-axe2+1)
      coef=coef/(2.*pi*sqrt(sigmau)*sqrt(sigmac))
      proba=coef*exp(-(axe2+axn2)/2.)

c compute fresnel's coefficient r1

      cos2chi=cv*cs+sv*ss*cos(pi-phi)
      if (cos2chi.gt.1.0)cos2chi=0.99999999999
      if (cos2chi.lt.-1.0)cos2chi=-0.99999999999
      coschi=sqrt(0.5*(1+cos2chi))
      sinchi=sqrt(0.5*(1-cos2chi))
      call fresnel(nr,ni,coschi,sinchi,r1)

c compute reflectance of the sun glint

      rog=pi*r1*proba/(4.*cs*cv*cos(tilt)**4)

      return
      end
c
c
c=======================================================================
      subroutine fresnel(nr,ni,coschi,sinchi,r1)
c
c to compute the fresnel's coefficient of reflection (see for
c example m. born and e. wolf, principles of optics, pergamon press, fifth
c edition, 1975, pp 628
c input parameters: nr=index of refraction of the sea water
c                   ni=extinction coefficient of the sea water
c                   coschi & sinchi=cosine and sine of the incident radiation 
c                                   with respect of the wave facet normal.
c output parameter: r1=fresnel's coefficient for reflection (intensity)
c
c rr2=intensity reflection coefficient for perpendicular component
c rl2=intensity reflection coefficient for parallel component
c r1=total unpolarized reflection coefficient

      use params, only: kr
      implicit none
      real(kr) ::  nr,ni,a1,a2,u,v,rr2,rl2,b1,b2,r1,coschi,sinchi

c absolute value for a1 to get v=0 when ni=0

      a1=abs(nr*nr-ni*ni-sinchi*sinchi)
      a2=sqrt((nr*nr-ni*ni-sinchi*sinchi)**2.+4*nr*nr*ni*ni)
      u=sqrt(0.5*(a1+a2))
      v=sqrt(0.5*(-a1+a2))
      rr2=((coschi-u)**2+v*v)/((coschi+u)**2+v*v)
      b1=(nr*nr-ni*ni)*coschi
      b2=2*nr*ni*coschi
      rl2=((b1-u)**2+(b2+v)**2)/((b1+u)**2+(b2-v)**2)

      r1=(rr2+rl2)/2.

      return
      end
c
c
c=======================================================================

      module sunblk             ! solar data base
      use params, only: kr
      implicit none
      save
      integer, parameter :: mxwl=5000
      real(kr), dimension(mxwl) :: wlsun,sun
      integer :: nns=0
      end module sunblk
c=======================================================================
      function solirr(wl,nf)
c
c input:
c     wl         wavelength in microns
c
c     nf         solar spectrum database switch
c               -1 = read solar.dat, replace data in wlsun sun
c                0 = spectrally uniform
c                1 = 5s 
c                2 = lowtran7 
c                3 = modtran3 (degraded to 20 cm-1 resolution)
c
c    note: the value of nf is not used after the first call to solirr
c    note: the 5s model uses a 1/w**4 power law for w > 4 um
c
c output:
c                extra-terestrial solar flux in w/m2/micron
c
      use params, only: zero, one, kr
      use sunblk, only: mxwl, nns, wlsun, sun
      implicit none
      integer :: j, nf
      real(kr) :: wt, wl, solirr

      if(nf.eq.0) then
        solirr=1.
        return
      endif

      if(nns.eq.0) then
        nns=mxwl
        select case (nf)
        case (-1) ; call rdspec('solar.dat',wlsun,sun,nns)
        case (1)  ; call sun1s(wlsun,sun,nns)
        case (2)  ; call sunlow(wlsun,sun,nns)
        case (3)  ; call sunmod(wlsun,sun,nns)
        case default 
           print *,'ERROR in solirr --- illegal value of nf',nf
           stop
        end select
      endif

      call locate(wlsun,nns,wl,j)
      wt=(wl-wlsun(j))/(wlsun(j+1)-wlsun(j))
      wt=max(zero,min(one,wt))
      solirr=sun(j)*(1.-wt)+sun(j+1)*wt

      return
      end      
c=======================================================================
      subroutine sun1s(wlsun,sun,nns)
      use params, only: kr
      implicit none
      integer :: i, nns
      real(kr) :: wlsun(*),sun(*),sun1(751)
c
c block    source           range           resolution      n_elements
c -----    ------           -----           ----------      ----------
c sun1     5s           0.25-4.00 um        .005 um            751
c
c units are w/m2/micron
c

      data sun1(1:60)/
     &  6.930e+01,8.600e+01,1.141e+02,1.600e+02,1.790e+02,1.770e+02,
     &  2.104e+02,2.730e+02,3.638e+02,5.050e+02,5.580e+02,5.370e+02,
     &  5.810e+02,6.580e+02,7.308e+02,8.180e+02,9.260e+02,8.981e+02,
     &  9.433e+02,9.087e+02,9.635e+02,1.026e+03,9.351e+02,1.137e+03,
     &  1.190e+03,1.029e+03,1.149e+03,9.181e+02,1.146e+03,9.245e+02,
     &  1.614e+03,1.648e+03,1.706e+03,1.783e+03,1.717e+03,1.693e+03,
     &  1.502e+03,1.760e+03,1.790e+03,1.929e+03,2.057e+03,2.021e+03,
     &  2.040e+03,2.013e+03,1.986e+03,2.019e+03,2.057e+03,1.879e+03,
     &  1.933e+03,1.974e+03,1.897e+03,1.936e+03,1.938e+03,1.825e+03,
     &  1.803e+03,1.861e+03,1.911e+03,1.899e+03,1.853e+03,1.878e+03/
      data sun1(61:120)/
     &  1.871e+03,1.874e+03,1.827e+03,1.847e+03,1.835e+03,1.864e+03,
     &  1.838e+03,1.843e+03,1.763e+03,1.795e+03,1.760e+03,1.769e+03,
     &  1.737e+03,1.689e+03,1.717e+03,1.675e+03,1.669e+03,1.658e+03,
     &  1.637e+03,1.623e+03,1.589e+03,1.515e+03,1.543e+03,1.555e+03,
     &  1.534e+03,1.517e+03,1.499e+03,1.469e+03,1.459e+03,1.438e+03,
     &  1.408e+03,1.407e+03,1.389e+03,1.368e+03,1.336e+03,1.352e+03,
     &  1.328e+03,1.314e+03,1.276e+03,1.286e+03,1.270e+03,1.257e+03,
     &  1.242e+03,1.220e+03,1.204e+03,1.199e+03,1.190e+03,1.181e+03,
     &  1.168e+03,1.144e+03,1.143e+03,1.123e+03,1.110e+03,1.112e+03,
     &  1.078e+03,1.077e+03,1.070e+03,1.047e+03,1.044e+03,1.030e+03/
      data sun1(121:180)/
     &  9.909e+02,9.429e+02,1.003e+03,9.550e+02,9.669e+02,9.997e+02,
     &  9.841e+02,9.740e+02,9.636e+02,9.529e+02,9.430e+02,9.330e+02,
     &  9.230e+02,9.134e+02,9.041e+02,8.940e+02,8.844e+02,8.751e+02,
     &  8.654e+02,8.565e+02,8.475e+02,8.385e+02,8.295e+02,8.205e+02,
     &  8.115e+02,8.025e+02,7.939e+02,7.857e+02,7.769e+02,7.690e+02,
     &  7.606e+02,7.515e+02,7.425e+02,7.339e+02,7.256e+02,7.169e+02,
     &  7.090e+02,7.006e+02,6.919e+02,6.840e+02,6.760e+02,6.676e+02,
     &  6.589e+02,6.518e+02,6.464e+02,6.411e+02,6.350e+02,6.294e+02,
     &  6.241e+02,6.180e+02,6.124e+02,6.075e+02,6.021e+02,5.964e+02,
     &  5.915e+02,5.861e+02,5.804e+02,5.755e+02,5.705e+02,5.655e+02/
      data sun1(181:240)/
     &  5.605e+02,5.505e+02,5.505e+02,5.455e+02,5.405e+02,5.355e+02,
     &  5.309e+02,5.270e+02,5.226e+02,5.179e+02,5.130e+02,5.062e+02,
     &  4.996e+02,4.931e+02,4.865e+02,4.800e+02,4.735e+02,4.669e+02,
     &  4.604e+02,4.538e+02,4.473e+02,4.408e+02,4.342e+02,4.277e+02,
     &  4.211e+02,4.146e+02,4.080e+02,4.015e+02,3.950e+02,3.884e+02,
     &  3.967e+02,4.071e+02,4.035e+02,3.998e+02,3.962e+02,3.925e+02,
     &  3.889e+02,3.852e+02,3.816e+02,3.780e+02,3.743e+02,3.707e+02,
     &  3.670e+02,3.634e+02,3.597e+02,3.560e+02,3.524e+02,3.488e+02,
     &  3.451e+02,3.415e+02,3.422e+02,3.437e+02,3.409e+02,3.381e+02,
     &  3.353e+02,3.325e+02,3.297e+02,3.270e+02,3.242e+02,3.214e+02/
      data sun1(241:300)/
     &  3.186e+02,3.158e+02,3.130e+02,3.102e+02,3.075e+02,3.047e+02,
     &  3.019e+02,2.991e+02,2.963e+02,2.935e+02,2.933e+02,2.934e+02,
     &  2.911e+02,2.888e+02,2.865e+02,2.843e+02,2.820e+02,2.797e+02,
     &  2.774e+02,2.751e+02,2.728e+02,2.705e+02,2.682e+02,2.659e+02,
     &  2.636e+02,2.613e+02,2.591e+02,2.568e+02,2.545e+02,2.522e+02,
     &  2.501e+02,2.481e+02,2.459e+02,2.436e+02,2.414e+02,2.391e+02,
     &  2.369e+02,2.346e+02,2.324e+02,2.301e+02,2.279e+02,2.256e+02,
     &  2.234e+02,2.212e+02,2.189e+02,2.167e+02,2.144e+02,2.122e+02,
     &  2.099e+02,2.077e+02,2.063e+02,2.050e+02,2.029e+02,2.009e+02,
     &  1.988e+02,1.967e+02,1.946e+02,1.925e+02,1.905e+02,1.884e+02/
      data sun1(301:360)/
     &  1.863e+02,1.842e+02,1.821e+02,1.801e+02,1.780e+02,1.759e+02,
     &  1.738e+02,1.717e+02,1.697e+02,1.676e+02,1.676e+02,1.679e+02,
     &  1.663e+02,1.646e+02,1.629e+02,1.613e+02,1.596e+02,1.579e+02,
     &  1.562e+02,1.546e+02,1.529e+02,1.512e+02,1.496e+02,1.479e+02,
     &  1.462e+02,1.446e+02,1.429e+02,1.412e+02,1.395e+02,1.379e+02,
     &  1.381e+02,1.386e+02,1.373e+02,1.360e+02,1.347e+02,1.334e+02,
     &  1.321e+02,1.308e+02,1.295e+02,1.282e+02,1.269e+02,1.256e+02,
     &  1.243e+02,1.230e+02,1.217e+02,1.204e+02,1.191e+02,1.178e+02,
     &  1.165e+02,1.152e+02,1.153e+02,1.156e+02,1.146e+02,1.136e+02,
     &  1.125e+02,1.115e+02,1.105e+02,1.095e+02,1.084e+02,1.074e+02/
      data sun1(361:420)/
     &  1.064e+02,1.054e+02,1.043e+02,1.033e+02,1.023e+02,1.013e+02,
     &  1.003e+02,9.922e+01,9.820e+01,9.718e+01,9.707e+01,9.711e+01,
     &  9.626e+01,9.542e+01,9.456e+01,9.372e+01,9.289e+01,9.203e+01,
     &  9.119e+01,9.034e+01,8.949e+01,8.866e+01,8.781e+01,8.697e+01,
     &  8.612e+01,8.526e+01,8.443e+01,8.359e+01,8.275e+01,8.189e+01,
     &  8.189e+01,8.202e+01,8.135e+01,8.065e+01,7.999e+01,7.930e+01,
     &  7.861e+01,7.794e+01,7.726e+01,7.657e+01,7.590e+01,7.522e+01,
     &  7.454e+01,7.386e+01,7.318e+01,7.250e+01,7.182e+01,7.114e+01,
     &  7.047e+01,6.979e+01,6.973e+01,6.980e+01,6.924e+01,6.868e+01,
     &  6.814e+01,6.757e+01,6.702e+01,6.648e+01,6.590e+01,6.536e+01/
      data sun1(421:480)/
     &  6.480e+01,6.425e+01,6.369e+01,6.314e+01,6.257e+01,6.203e+01,
     &  6.147e+01,6.092e+01,6.036e+01,5.981e+01,5.980e+01,5.985e+01,
     &  5.940e+01,5.895e+01,5.850e+01,5.806e+01,5.760e+01,5.715e+01,
     &  5.670e+01,5.626e+01,5.579e+01,5.536e+01,5.490e+01,5.443e+01,
     &  5.400e+01,5.355e+01,5.309e+01,5.265e+01,5.220e+01,5.175e+01,
     &  5.168e+01,5.167e+01,5.130e+01,5.092e+01,5.055e+01,5.018e+01,
     &  4.979e+01,4.943e+01,4.904e+01,4.867e+01,4.830e+01,4.793e+01,
     &  4.754e+01,4.718e+01,4.679e+01,4.642e+01,4.606e+01,4.567e+01,
     &  4.530e+01,4.493e+01,4.482e+01,4.479e+01,4.445e+01,4.414e+01,
     &  4.383e+01,4.349e+01,4.318e+01,4.286e+01,4.255e+01,4.221e+01/
      data sun1(481:540)/
     &  4.190e+01,4.158e+01,4.126e+01,4.094e+01,4.062e+01,4.031e+01,
     &  3.997e+01,3.966e+01,3.934e+01,3.903e+01,3.896e+01,3.892e+01,
     &  3.866e+01,3.838e+01,3.812e+01,3.786e+01,3.758e+01,3.730e+01,
     &  3.705e+01,3.677e+01,3.650e+01,3.623e+01,3.596e+01,3.569e+01,
     &  3.542e+01,3.515e+01,3.488e+01,3.461e+01,3.434e+01,3.407e+01,
     &  3.403e+01,3.403e+01,3.379e+01,3.357e+01,3.335e+01,3.312e+01,
     &  3.290e+01,3.267e+01,3.244e+01,3.223e+01,3.200e+01,3.177e+01,
     &  3.155e+01,3.131e+01,3.110e+01,3.087e+01,3.066e+01,3.042e+01,
     &  3.020e+01,2.997e+01,2.990e+01,2.987e+01,2.966e+01,2.946e+01,
     &  2.927e+01,2.908e+01,2.888e+01,2.867e+01,2.849e+01,2.830e+01/
      data sun1(541:600)/
     &  2.810e+01,2.791e+01,2.771e+01,2.752e+01,2.731e+01,2.712e+01,
     &  2.693e+01,2.674e+01,2.654e+01,2.635e+01,2.631e+01,2.628e+01,
     &  2.612e+01,2.595e+01,2.579e+01,2.564e+01,2.545e+01,2.530e+01,
     &  2.512e+01,2.498e+01,2.480e+01,2.463e+01,2.447e+01,2.431e+01,
     &  2.414e+01,2.397e+01,2.381e+01,2.366e+01,2.347e+01,2.331e+01,
     &  2.326e+01,2.321e+01,2.306e+01,2.292e+01,2.276e+01,2.263e+01,
     &  2.247e+01,2.235e+01,2.219e+01,2.204e+01,2.190e+01,2.176e+01,
     &  2.160e+01,2.147e+01,2.131e+01,2.118e+01,2.103e+01,2.089e+01,
     &  2.073e+01,2.060e+01,2.055e+01,2.053e+01,2.040e+01,2.028e+01,
     &  2.014e+01,2.003e+01,1.990e+01,1.977e+01,1.965e+01,1.953e+01/
      data sun1(601:660)/
     &  1.939e+01,1.927e+01,1.915e+01,1.902e+01,1.890e+01,1.878e+01,
     &  1.864e+01,1.853e+01,1.840e+01,1.827e+01,1.825e+01,1.824e+01,
     &  1.814e+01,1.803e+01,1.793e+01,1.783e+01,1.771e+01,1.762e+01,
     &  1.750e+01,1.741e+01,1.729e+01,1.721e+01,1.708e+01,1.700e+01,
     &  1.687e+01,1.679e+01,1.666e+01,1.657e+01,1.646e+01,1.636e+01,
     &  1.633e+01,1.630e+01,1.622e+01,1.613e+01,1.604e+01,1.596e+01,
     &  1.584e+01,1.578e+01,1.568e+01,1.559e+01,1.550e+01,1.540e+01,
     &  1.532e+01,1.524e+01,1.513e+01,1.505e+01,1.496e+01,1.487e+01,
     &  1.478e+01,1.469e+01,1.464e+01,1.463e+01,1.455e+01,1.445e+01,
     &  1.438e+01,1.432e+01,1.421e+01,1.415e+01,1.405e+01,1.398e+01/
      data sun1(661:720)/
     &  1.391e+01,1.382e+01,1.374e+01,1.367e+01,1.358e+01,1.350e+01,
     &  1.343e+01,1.335e+01,1.325e+01,1.318e+01,1.316e+01,1.312e+01,
     &  1.306e+01,1.300e+01,1.292e+01,1.285e+01,1.278e+01,1.270e+01,
     &  1.265e+01,1.257e+01,1.250e+01,1.243e+01,1.236e+01,1.228e+01,
     &  1.222e+01,1.215e+01,1.207e+01,1.201e+01,1.194e+01,1.186e+01,
     &  1.185e+01,1.184e+01,1.178e+01,1.172e+01,1.166e+01,1.160e+01,
     &  1.155e+01,1.147e+01,1.142e+01,1.136e+01,1.130e+01,1.124e+01,
     &  1.118e+01,1.112e+01,1.106e+01,1.100e+01,1.094e+01,1.089e+01,
     &  1.082e+01,1.075e+01,1.072e+01,1.070e+01,1.064e+01,1.059e+01,
     &  1.052e+01,1.047e+01,1.042e+01,1.037e+01,1.031e+01,1.025e+01/
      data sun1(721:751)/
     &  1.020e+01,1.015e+01,1.010e+01,1.003e+01,9.980e+00,9.920e+00,
     &  9.860e+00,9.820e+00,9.750e+00,9.710e+00,9.700e+00,9.700e+00,
     &  9.630e+00,9.590e+00,9.560e+00,9.500e+00,9.450e+00,9.410e+00,
     &  9.360e+00,9.320e+00,9.270e+00,9.220e+00,9.180e+00,9.130e+00,
     &  9.080e+00,9.050e+00,8.990e+00,8.940e+00,8.900e+00,8.850e+00,
     &  8.810e+00/

      nns=751
      sun(1:nns)=sun1(1:nns)
      do i=1,nns 
        wlsun(i)=.25+(4.0-.25)*real(i-1)/(nns-1)
      enddo
      return
      end
c=======================================================================
      subroutine sunlow(wlsun,sun,nns)
      use params, only: one,zero,kr
      implicit none
      integer :: i, nn1, nn2, nns
      real(kr) :: wn, wnmax, wnmin,
     &     wlsun(*), sun(*), sun2a(1440), sun2b(2910)

c block    source           range           resolution      n_elements
c -----    ------           -----           ----------      ----------
c sun2a   lowtran7        0-28780 cm-1      20 cm-1           1440
c sun2b   lowtran7    28400-57490 cm-1      10 cm-1           2910

c
      data sun2a(1:60)/
     &  0.000e+00,4.576e-08,7.010e-07,3.458e-06,1.073e-05,2.570e-05,
     &  5.250e-05,9.600e-05,1.619e-04,2.577e-04,3.910e-04,5.692e-04,
     &  8.020e-04,1.101e-03,1.477e-03,1.946e-03,2.521e-03,3.215e-03,
     &  4.044e-03,5.023e-03,6.170e-03,7.515e-03,9.068e-03,1.085e-02,
     &  1.289e-02,1.521e-02,1.776e-02,2.064e-02,2.389e-02,2.752e-02,
     &  3.154e-02,3.596e-02,4.085e-02,4.624e-02,5.213e-02,5.854e-02,
     &  6.549e-02,7.302e-02,8.117e-02,9.000e-02,9.954e-02,1.098e-01,
     &  1.208e-01,1.326e-01,1.452e-01,1.586e-01,1.731e-01,1.885e-01,
     &  2.049e-01,2.224e-01,2.411e-01,2.609e-01,2.820e-01,3.043e-01,
     &  3.279e-01,3.527e-01,3.789e-01,4.065e-01,4.355e-01,4.660e-01/
      data sun2a(61:120)/
     &  4.980e-01,5.316e-01,5.669e-01,6.039e-01,6.426e-01,6.832e-01,
     &  7.256e-01,7.699e-01,8.162e-01,8.644e-01,9.147e-01,9.671e-01,
     &  1.022e+00,1.078e+00,1.137e+00,1.199e+00,1.263e+00,1.329e+00,
     &  1.399e+00,1.471e+00,1.546e+00,1.625e+00,1.706e+00,1.791e+00,
     &  1.880e+00,1.971e+00,2.067e+00,2.166e+00,2.268e+00,2.374e+00,
     &  2.484e+00,2.597e+00,2.714e+00,2.835e+00,2.960e+00,3.089e+00,
     &  3.221e+00,3.357e+00,3.498e+00,3.642e+00,3.790e+00,3.944e+00,
     &  4.104e+00,4.273e+00,4.445e+00,4.615e+00,4.791e+00,4.983e+00,
     &  5.195e+00,5.421e+00,5.656e+00,5.893e+00,6.127e+00,6.356e+00,
     &  6.582e+00,6.808e+00,7.036e+00,7.270e+00,7.517e+00,7.789e+00/
      data sun2a(121:180)/
     &  8.091e+00,8.407e+00,8.712e+00,8.990e+00,9.249e+00,9.500e+00,
     &  9.755e+00,1.001e+01,1.025e+01,1.048e+01,1.070e+01,1.095e+01,
     &  1.123e+01,1.155e+01,1.190e+01,1.225e+01,1.260e+01,1.293e+01,
     &  1.325e+01,1.353e+01,1.378e+01,1.404e+01,1.432e+01,1.466e+01,
     &  1.507e+01,1.553e+01,1.601e+01,1.643e+01,1.677e+01,1.708e+01,
     &  1.747e+01,1.796e+01,1.843e+01,1.873e+01,1.891e+01,1.914e+01,
     &  1.949e+01,1.984e+01,2.016e+01,2.051e+01,2.102e+01,2.177e+01,
     &  2.257e+01,2.319e+01,2.358e+01,2.390e+01,2.433e+01,2.483e+01,
     &  2.524e+01,2.565e+01,2.631e+01,2.721e+01,2.798e+01,2.842e+01,
     &  2.882e+01,2.957e+01,3.053e+01,3.125e+01,3.167e+01,3.222e+01/
      data sun2a(181:240)/
     &  3.309e+01,3.397e+01,3.460e+01,3.500e+01,3.540e+01,3.603e+01,
     &  3.699e+01,3.789e+01,3.840e+01,3.889e+01,3.986e+01,4.093e+01,
     &  4.157e+01,4.213e+01,4.308e+01,4.435e+01,4.552e+01,4.598e+01,
     &  4.628e+01,4.833e+01,5.199e+01,5.437e+01,5.408e+01,5.217e+01,
     &  5.071e+01,5.215e+01,5.571e+01,5.655e+01,5.441e+01,5.327e+01,
     &  5.608e+01,6.197e+01,6.441e+01,6.065e+01,5.515e+01,5.307e+01,
     &  5.748e+01,6.464e+01,6.835e+01,6.906e+01,6.987e+01,7.094e+01,
     &  7.166e+01,7.277e+01,7.433e+01,7.526e+01,7.488e+01,7.361e+01,
     &  7.321e+01,7.489e+01,7.804e+01,8.020e+01,8.088e+01,8.267e+01,
     &  8.498e+01,8.624e+01,8.836e+01,9.200e+01,9.538e+01,9.812e+01/
      data sun2a(241:300)/
     &  1.003e+02,1.006e+02,1.000e+02,1.018e+02,1.051e+02,1.075e+02,
     &  1.100e+02,1.124e+02,1.139e+02,1.138e+02,1.192e+02,1.220e+02,
     &  1.246e+02,1.271e+02,1.252e+02,1.244e+02,1.250e+02,1.279e+02,
     &  1.307e+02,1.320e+02,1.337e+02,1.367e+02,1.362e+02,1.350e+02,
     &  1.374e+02,1.384e+02,1.373e+02,1.364e+02,1.426e+02,1.445e+02,
     &  1.484e+02,1.519e+02,1.516e+02,1.554e+02,1.576e+02,1.597e+02,
     &  1.623e+02,1.684e+02,1.714e+02,1.698e+02,1.703e+02,1.723e+02,
     &  1.767e+02,1.819e+02,1.861e+02,1.879e+02,1.860e+02,1.898e+02,
     &  1.894e+02,1.929e+02,2.020e+02,2.096e+02,2.058e+02,2.129e+02,
     &  2.156e+02,2.165e+02,2.192e+02,2.203e+02,2.211e+02,2.271e+02/
      data sun2a(301:360)/
     &  2.300e+02,2.332e+02,2.339e+02,2.345e+02,2.344e+02,2.358e+02,
     &  2.398e+02,2.431e+02,2.412e+02,2.423e+02,2.437e+02,2.428e+02,
     &  2.462e+02,2.461e+02,2.468e+02,2.518e+02,2.554e+02,2.587e+02,
     &  2.603e+02,2.634e+02,2.687e+02,2.718e+02,2.730e+02,2.739e+02,
     &  2.747e+02,2.744e+02,2.797e+02,2.878e+02,2.877e+02,2.880e+02,
     &  2.900e+02,2.919e+02,2.953e+02,2.968e+02,3.005e+02,3.022e+02,
     &  2.991e+02,3.014e+02,3.057e+02,3.093e+02,3.106e+02,3.132e+02,
     &  3.146e+02,3.096e+02,3.188e+02,3.205e+02,3.216e+02,3.286e+02,
     &  3.317e+02,3.372e+02,3.456e+02,3.455e+02,3.430e+02,3.444e+02,
     &  3.462e+02,3.492e+02,3.518e+02,3.547e+02,3.570e+02,3.583e+02/
      data sun2a(361:420)/
     &  3.623e+02,3.641e+02,3.650e+02,3.678e+02,3.690e+02,3.691e+02,
     &  3.722e+02,3.778e+02,3.813e+02,3.842e+02,3.887e+02,3.936e+02,
     &  3.970e+02,3.987e+02,4.006e+02,4.041e+02,4.082e+02,4.125e+02,
     &  4.156e+02,4.162e+02,4.165e+02,4.195e+02,4.259e+02,4.333e+02,
     &  4.377e+02,4.381e+02,4.398e+02,4.415e+02,4.387e+02,4.343e+02,
     &  4.375e+02,4.490e+02,4.489e+02,4.395e+02,4.371e+02,4.393e+02,
     &  4.443e+02,4.550e+02,4.670e+02,4.730e+02,4.696e+02,4.675e+02,
     &  4.738e+02,4.775e+02,4.775e+02,4.810e+02,4.839e+02,4.822e+02,
     &  4.791e+02,4.821e+02,4.934e+02,4.984e+02,4.920e+02,4.895e+02,
     &  4.933e+02,4.955e+02,4.965e+02,4.996e+02,5.046e+02,5.097e+02/
      data sun2a(421:480)/
     &  5.120e+02,5.120e+02,5.123e+02,5.150e+02,5.207e+02,5.273e+02,
     &  5.319e+02,5.322e+02,5.305e+02,5.323e+02,5.393e+02,5.486e+02,
     &  5.530e+02,5.490e+02,5.460e+02,5.510e+02,5.564e+02,5.572e+02,
     &  5.578e+02,5.610e+02,5.640e+02,5.656e+02,5.664e+02,5.679e+02,
     &  5.715e+02,5.767e+02,5.815e+02,5.865e+02,5.936e+02,6.007e+02,
     &  6.028e+02,6.014e+02,6.030e+02,6.069e+02,6.060e+02,6.010e+02,
     &  6.008e+02,6.072e+02,6.129e+02,6.141e+02,6.144e+02,6.166e+02,
     &  6.205e+02,6.252e+02,6.298e+02,6.338e+02,6.373e+02,6.405e+02,
     &  6.425e+02,6.426e+02,6.419e+02,6.431e+02,6.467e+02,6.506e+02,
     &  6.543e+02,6.610e+02,6.721e+02,6.823e+02,6.849e+02,6.822e+02/
      data sun2a(481:540)/
     &  6.825e+02,6.878e+02,6.914e+02,6.896e+02,6.881e+02,6.937e+02,
     &  7.033e+02,7.081e+02,7.062e+02,7.046e+02,7.090e+02,7.173e+02,
     &  7.254e+02,7.311e+02,7.342e+02,7.354e+02,7.366e+02,7.393e+02,
     &  7.429e+02,7.450e+02,7.443e+02,7.424e+02,7.495e+02,7.557e+02,
     &  7.588e+02,7.663e+02,7.615e+02,7.621e+02,7.697e+02,7.642e+02,
     &  7.638e+02,7.689e+02,7.627e+02,7.539e+02,7.624e+02,7.658e+02,
     &  7.722e+02,7.607e+02,7.621e+02,7.668e+02,7.670e+02,7.693e+02,
     &  7.735e+02,7.668e+02,7.636e+02,7.738e+02,7.772e+02,7.796e+02,
     &  7.925e+02,7.975e+02,7.878e+02,7.938e+02,8.060e+02,8.048e+02,
     &  8.066e+02,8.217e+02,8.303e+02,8.275e+02,8.311e+02,8.302e+02/
      data sun2a(541:600)/
     &  8.262e+02,8.233e+02,8.222e+02,8.339e+02,8.546e+02,8.598e+02,
     &  8.626e+02,8.712e+02,8.752e+02,8.677e+02,8.639e+02,8.833e+02,
     &  8.934e+02,8.977e+02,9.052e+02,9.054e+02,9.111e+02,9.302e+02,
     &  9.392e+02,9.347e+02,9.352e+02,9.424e+02,9.481e+02,9.470e+02,
     &  9.519e+02,9.601e+02,9.519e+02,9.542e+02,9.591e+02,9.634e+02,
     &  9.802e+02,9.837e+02,9.788e+02,9.794e+02,9.852e+02,9.771e+02,
     &  9.199e+02,8.997e+02,9.629e+02,9.972e+02,9.999e+02,9.957e+02,
     &  9.999e+02,1.015e+03,9.516e+02,8.935e+02,9.551e+02,1.003e+03,
     &  9.901e+02,9.788e+02,1.011e+03,1.035e+03,1.032e+03,1.030e+03,
     &  1.040e+03,1.046e+03,1.044e+03,1.050e+03,1.056e+03,1.050e+03/
      data sun2a(601:660)/
     &  1.038e+03,1.052e+03,1.072e+03,1.076e+03,1.077e+03,1.079e+03,
     &  1.078e+03,1.076e+03,1.080e+03,1.081e+03,1.070e+03,1.078e+03,
     &  1.104e+03,1.111e+03,1.112e+03,1.118e+03,1.120e+03,1.109e+03,
     &  1.101e+03,1.113e+03,1.123e+03,1.120e+03,1.124e+03,1.136e+03,
     &  1.144e+03,1.141e+03,1.141e+03,1.152e+03,1.149e+03,1.138e+03,
     &  1.141e+03,1.151e+03,1.160e+03,1.171e+03,1.178e+03,1.180e+03,
     &  1.182e+03,1.182e+03,1.180e+03,1.182e+03,1.188e+03,1.190e+03,
     &  1.191e+03,1.197e+03,1.196e+03,1.192e+03,1.201e+03,1.210e+03,
     &  1.209e+03,1.208e+03,1.205e+03,1.193e+03,1.193e+03,1.220e+03,
     &  1.243e+03,1.245e+03,1.242e+03,1.240e+03,1.241e+03,1.244e+03/
      data sun2a(661:720)/
     &  1.249e+03,1.253e+03,1.257e+03,1.260e+03,1.262e+03,1.264e+03,
     &  1.266e+03,1.270e+03,1.277e+03,1.284e+03,1.284e+03,1.283e+03,
     &  1.287e+03,1.287e+03,1.272e+03,1.262e+03,1.271e+03,1.289e+03,
     &  1.305e+03,1.312e+03,1.312e+03,1.314e+03,1.320e+03,1.326e+03,
     &  1.328e+03,1.325e+03,1.323e+03,1.325e+03,1.335e+03,1.346e+03,
     &  1.354e+03,1.354e+03,1.347e+03,1.338e+03,1.331e+03,1.330e+03,
     &  1.338e+03,1.352e+03,1.363e+03,1.369e+03,1.372e+03,1.376e+03,
     &  1.382e+03,1.388e+03,1.389e+03,1.388e+03,1.392e+03,1.402e+03,
     &  1.413e+03,1.418e+03,1.411e+03,1.396e+03,1.386e+03,1.388e+03,
     &  1.405e+03,1.424e+03,1.428e+03,1.422e+03,1.424e+03,1.435e+03/
      data sun2a(721:780)/
     &  1.445e+03,1.451e+03,1.452e+03,1.452e+03,1.454e+03,1.460e+03,
     &  1.467e+03,1.471e+03,1.469e+03,1.463e+03,1.460e+03,1.469e+03,
     &  1.482e+03,1.491e+03,1.495e+03,1.498e+03,1.501e+03,1.505e+03,
     &  1.510e+03,1.512e+03,1.513e+03,1.516e+03,1.522e+03,1.524e+03,
     &  1.521e+03,1.520e+03,1.532e+03,1.546e+03,1.548e+03,1.542e+03,
     &  1.542e+03,1.554e+03,1.564e+03,1.564e+03,1.560e+03,1.561e+03,
     &  1.570e+03,1.582e+03,1.578e+03,1.530e+03,1.447e+03,1.397e+03,
     &  1.429e+03,1.506e+03,1.567e+03,1.594e+03,1.606e+03,1.614e+03,
     &  1.609e+03,1.589e+03,1.568e+03,1.567e+03,1.587e+03,1.610e+03,
     &  1.624e+03,1.630e+03,1.631e+03,1.628e+03,1.622e+03,1.617e+03/
      data sun2a(781:840)/
     &  1.619e+03,1.632e+03,1.648e+03,1.658e+03,1.660e+03,1.658e+03,
     &  1.658e+03,1.659e+03,1.660e+03,1.659e+03,1.654e+03,1.645e+03,
     &  1.642e+03,1.653e+03,1.674e+03,1.694e+03,1.701e+03,1.703e+03,
     &  1.698e+03,1.655e+03,1.644e+03,1.662e+03,1.676e+03,1.708e+03,
     &  1.703e+03,1.711e+03,1.732e+03,1.717e+03,1.720e+03,1.730e+03,
     &  1.683e+03,1.629e+03,1.684e+03,1.727e+03,1.708e+03,1.689e+03,
     &  1.698e+03,1.733e+03,1.738e+03,1.714e+03,1.735e+03,1.750e+03,
     &  1.750e+03,1.760e+03,1.764e+03,1.765e+03,1.769e+03,1.780e+03,
     &  1.793e+03,1.765e+03,1.729e+03,1.746e+03,1.753e+03,1.758e+03,
     &  1.775e+03,1.768e+03,1.768e+03,1.790e+03,1.807e+03,1.799e+03/
      data sun2a(841:900)/
     &  1.783e+03,1.779e+03,1.792e+03,1.810e+03,1.808e+03,1.794e+03,
     &  1.819e+03,1.774e+03,1.649e+03,1.674e+03,1.789e+03,1.847e+03,
     &  1.848e+03,1.813e+03,1.796e+03,1.840e+03,1.868e+03,1.865e+03,
     &  1.873e+03,1.872e+03,1.856e+03,1.845e+03,1.842e+03,1.824e+03,
     &  1.795e+03,1.820e+03,1.862e+03,1.858e+03,1.839e+03,1.841e+03,
     &  1.864e+03,1.877e+03,1.884e+03,1.895e+03,1.875e+03,1.821e+03,
     &  1.779e+03,1.810e+03,1.855e+03,1.832e+03,1.837e+03,1.882e+03,
     &  1.866e+03,1.820e+03,1.805e+03,1.831e+03,1.862e+03,1.867e+03,
     &  1.863e+03,1.852e+03,1.835e+03,1.835e+03,1.845e+03,1.832e+03,
     &  1.804e+03,1.793e+03,1.822e+03,1.846e+03,1.832e+03,1.848e+03/
      data sun2a(901:960)/
     &  1.894e+03,1.909e+03,1.901e+03,1.891e+03,1.870e+03,1.854e+03,
     &  1.866e+03,1.874e+03,1.869e+03,1.882e+03,1.897e+03,1.884e+03,
     &  1.856e+03,1.841e+03,1.855e+03,1.885e+03,1.904e+03,1.900e+03,
     &  1.887e+03,1.888e+03,1.879e+03,1.845e+03,1.844e+03,1.877e+03,
     &  1.847e+03,1.785e+03,1.793e+03,1.849e+03,1.894e+03,1.909e+03,
     &  1.893e+03,1.867e+03,1.886e+03,1.960e+03,1.972e+03,1.896e+03,
     &  1.884e+03,1.918e+03,1.854e+03,1.793e+03,1.876e+03,1.974e+03,
     &  1.976e+03,1.944e+03,1.926e+03,1.914e+03,1.903e+03,1.883e+03,
     &  1.813e+03,1.711e+03,1.718e+03,1.860e+03,1.965e+03,1.970e+03,
     &  1.941e+03,1.903e+03,1.852e+03,1.836e+03,1.879e+03,1.902e+03/
      data sun2a(961:1020)/
     &  1.863e+03,1.839e+03,1.841e+03,1.780e+03,1.685e+03,1.677e+03,
     &  1.719e+03,1.697e+03,1.684e+03,1.785e+03,1.898e+03,1.910e+03,
     &  1.877e+03,1.867e+03,1.863e+03,1.860e+03,1.900e+03,1.971e+03,
     &  2.000e+03,1.971e+03,1.937e+03,1.923e+03,1.923e+03,1.924e+03,
     &  1.917e+03,1.912e+03,1.926e+03,1.960e+03,1.995e+03,1.996e+03,
     &  1.939e+03,1.884e+03,1.895e+03,1.933e+03,1.935e+03,1.899e+03,
     &  1.853e+03,1.820e+03,1.822e+03,1.865e+03,1.936e+03,1.966e+03,
     &  1.920e+03,1.881e+03,1.932e+03,2.016e+03,2.050e+03,2.021e+03,
     &  1.961e+03,1.938e+03,1.997e+03,2.051e+03,2.003e+03,1.912e+03,
     &  1.880e+03,1.895e+03,1.898e+03,1.899e+03,1.938e+03,1.994e+03/
      data sun2a(1021:1080)/
     &  2.010e+03,1.982e+03,1.949e+03,1.927e+03,1.912e+03,1.878e+03,
     &  1.792e+03,1.680e+03,1.645e+03,1.727e+03,1.845e+03,1.926e+03,
     &  1.973e+03,2.005e+03,2.022e+03,2.022e+03,2.026e+03,2.054e+03,
     &  2.087e+03,2.083e+03,2.053e+03,2.047e+03,2.070e+03,2.072e+03,
     &  2.038e+03,2.020e+03,2.050e+03,2.074e+03,2.038e+03,1.979e+03,
     &  1.964e+03,1.997e+03,2.038e+03,2.058e+03,2.048e+03,2.018e+03,
     &  1.999e+03,2.011e+03,2.040e+03,2.056e+03,2.040e+03,1.982e+03,
     &  1.911e+03,1.892e+03,1.938e+03,1.992e+03,2.006e+03,2.001e+03,
     &  2.011e+03,2.023e+03,1.998e+03,1.948e+03,1.936e+03,1.987e+03,
     &  2.038e+03,2.033e+03,1.996e+03,1.984e+03,2.012e+03,2.056e+03/
      data sun2a(1081:1140)/
     &  2.092e+03,2.107e+03,2.095e+03,2.070e+03,2.053e+03,2.047e+03,
     &  2.044e+03,2.036e+03,2.017e+03,1.988e+03,1.973e+03,1.999e+03,
     &  2.057e+03,2.104e+03,2.109e+03,2.089e+03,2.069e+03,2.052e+03,
     &  2.031e+03,2.006e+03,1.987e+03,1.982e+03,1.979e+03,1.964e+03,
     &  1.944e+03,1.952e+03,2.007e+03,2.083e+03,2.139e+03,2.158e+03,
     &  2.143e+03,2.103e+03,2.051e+03,2.002e+03,1.975e+03,1.988e+03,
     &  2.038e+03,2.075e+03,2.051e+03,1.972e+03,1.885e+03,1.829e+03,
     &  1.821e+03,1.866e+03,1.935e+03,1.974e+03,1.959e+03,1.925e+03,
     &  1.920e+03,1.950e+03,1.985e+03,1.996e+03,1.966e+03,1.885e+03,
     &  1.782e+03,1.727e+03,1.759e+03,1.817e+03,1.800e+03,1.693e+03/
      data sun2a(1141:1200)/
     &  1.593e+03,1.599e+03,1.700e+03,1.824e+03,1.910e+03,1.938e+03,
     &  1.903e+03,1.822e+03,1.738e+03,1.683e+03,1.667e+03,1.683e+03,
     &  1.715e+03,1.734e+03,1.712e+03,1.668e+03,1.655e+03,1.698e+03,
     &  1.727e+03,1.637e+03,1.416e+03,1.204e+03,1.156e+03,1.278e+03,
     &  1.450e+03,1.561e+03,1.595e+03,1.588e+03,1.571e+03,1.566e+03,
     &  1.590e+03,1.641e+03,1.688e+03,1.708e+03,1.704e+03,1.701e+03,
     &  1.719e+03,1.749e+03,1.772e+03,1.773e+03,1.745e+03,1.690e+03,
     &  1.625e+03,1.589e+03,1.619e+03,1.701e+03,1.783e+03,1.816e+03,
     &  1.801e+03,1.765e+03,1.734e+03,1.715e+03,1.705e+03,1.702e+03,
     &  1.697e+03,1.682e+03,1.661e+03,1.657e+03,1.693e+03,1.763e+03/
      data sun2a(1201:1260)/
     &  1.827e+03,1.842e+03,1.806e+03,1.756e+03,1.726e+03,1.724e+03,
     &  1.737e+03,1.749e+03,1.756e+03,1.760e+03,1.762e+03,1.770e+03,
     &  1.792e+03,1.827e+03,1.849e+03,1.820e+03,1.721e+03,1.596e+03,
     &  1.514e+03,1.523e+03,1.602e+03,1.706e+03,1.793e+03,1.838e+03,
     &  1.820e+03,1.738e+03,1.631e+03,1.553e+03,1.539e+03,1.574e+03,
     &  1.624e+03,1.661e+03,1.677e+03,1.673e+03,1.653e+03,1.626e+03,
     &  1.607e+03,1.604e+03,1.621e+03,1.655e+03,1.701e+03,1.752e+03,
     &  1.796e+03,1.823e+03,1.827e+03,1.809e+03,1.767e+03,1.714e+03,
     &  1.667e+03,1.644e+03,1.644e+03,1.653e+03,1.655e+03,1.639e+03,
     &  1.592e+03,1.506e+03,1.377e+03,1.210e+03,1.011e+03,8.076e+02/
      data sun2a(1261:1320)/
     &  6.668e+02,6.645e+02,8.352e+02,1.100e+03,1.331e+03,1.423e+03,
     &  1.364e+03,1.194e+03,9.618e+02,7.250e+02,5.513e+02,5.040e+02,
     &  5.963e+02,7.752e+02,9.756e+02,1.150e+03,1.287e+03,1.386e+03,
     &  1.448e+03,1.474e+03,1.469e+03,1.435e+03,1.377e+03,1.296e+03,
     &  1.196e+03,1.085e+03,9.854e+02,9.173e+02,8.946e+02,9.109e+02,
     &  9.515e+02,1.002e+03,1.046e+03,1.071e+03,1.061e+03,1.021e+03,
     &  9.772e+02,9.592e+02,9.821e+02,1.021e+03,1.033e+03,9.834e+02,
     &  8.798e+02,7.627e+02,6.753e+02,6.433e+02,6.627e+02,7.215e+02,
     &  8.083e+02,9.132e+02,1.027e+03,1.140e+03,1.236e+03,1.293e+03,
     &  1.287e+03,1.210e+03,1.102e+03,1.022e+03,1.023e+03,1.109e+03/
      data sun2a(1321:1380)/
     &  1.233e+03,1.337e+03,1.383e+03,1.373e+03,1.325e+03,1.258e+03,
     &  1.189e+03,1.134e+03,1.107e+03,1.114e+03,1.137e+03,1.148e+03,
     &  1.121e+03,1.054e+03,9.681e+02,8.892e+02,8.379e+02,8.176e+02,
     &  8.237e+02,8.510e+02,8.965e+02,9.598e+02,1.041e+03,1.138e+03,
     &  1.231e+03,1.294e+03,1.300e+03,1.241e+03,1.155e+03,1.092e+03,
     &  1.097e+03,1.170e+03,1.264e+03,1.322e+03,1.307e+03,1.234e+03,
     &  1.146e+03,1.091e+03,1.093e+03,1.135e+03,1.189e+03,1.229e+03,
     &  1.246e+03,1.249e+03,1.250e+03,1.261e+03,1.275e+03,1.280e+03,
     &  1.262e+03,1.214e+03,1.145e+03,1.070e+03,1.001e+03,9.525e+02,
     &  9.305e+02,9.417e+02,9.903e+02,1.064e+03,1.135e+03,1.172e+03/
      data sun2a(1381:1440)/
     &  1.149e+03,1.076e+03,9.843e+02,9.063e+02,8.682e+02,8.738e+02,
     &  9.153e+02,9.844e+02,1.067e+03,1.137e+03,1.163e+03,1.116e+03,
     &  9.905e+02,8.309e+02,6.923e+02,6.274e+02,6.541e+02,7.392e+02,
     &  8.389e+02,9.117e+02,9.419e+02,9.444e+02,9.396e+02,9.461e+02,
     &  9.702e+02,1.005e+03,1.042e+03,1.074e+03,1.097e+03,1.114e+03,
     &  1.129e+03,1.143e+03,1.153e+03,1.152e+03,1.132e+03,1.084e+03,
     &  1.017e+03,9.460e+02,8.904e+02,8.662e+02,8.765e+02,9.131e+02,
     &  9.661e+02,1.025e+03,1.080e+03,1.119e+03,1.103e+03,1.244e+03,
     &  1.210e+03,1.079e+03,8.522e+02,9.568e+02,8.423e+02,8.974e+02,
     &  1.082e+03,9.142e+02,9.931e+02,1.050e+03,8.450e+02,8.392e+02/

      data (sun2b(i),i=1,60)/
     &  8.765e+02,8.922e+02,9.131e+02,9.382e+02,9.661e+02,9.956e+02,
     &  1.025e+03,1.054e+03,1.080e+03,1.102e+03,1.119e+03,1.132e+03,
     &  1.103e+03,1.159e+03,1.244e+03,1.238e+03,1.210e+03,1.196e+03,
     &  1.079e+03,8.956e+02,8.522e+02,9.356e+02,9.568e+02,8.971e+02,
     &  8.423e+02,8.212e+02,8.974e+02,1.043e+03,1.082e+03,9.888e+02,
     &  9.142e+02,9.294e+02,9.931e+02,1.042e+03,1.050e+03,9.843e+02,
     &  8.450e+02,7.708e+02,8.392e+02,9.397e+02,1.026e+03,1.121e+03,
     &  1.163e+03,1.143e+03,1.078e+03,1.027e+03,1.078e+03,1.094e+03,
     &  9.698e+02,8.537e+02,8.499e+02,9.091e+02,9.957e+02,1.095e+03,
     &  1.147e+03,1.086e+03,1.010e+03,1.065e+03,1.129e+03,1.081e+03/
      data (sun2b(i),i=61,120)/
     &  9.879e+02,8.982e+02,8.352e+02,7.716e+02,6.871e+02,6.145e+02,
     &  6.061e+02,7.371e+02,9.081e+02,9.976e+02,1.081e+03,1.126e+03,
     &  1.057e+03,1.028e+03,1.142e+03,1.253e+03,1.225e+03,1.103e+03,
     &  1.039e+03,1.043e+03,1.003e+03,9.655e+02,1.035e+03,1.151e+03,
     &  1.201e+03,1.152e+03,1.069e+03,9.958e+02,8.895e+02,8.185e+02,
     &  9.070e+02,1.042e+03,1.056e+03,1.001e+03,9.720e+02,9.857e+02,
     &  1.027e+03,1.055e+03,1.078e+03,1.127e+03,1.205e+03,1.246e+03,
     &  1.201e+03,1.145e+03,1.098e+03,1.030e+03,9.268e+02,8.367e+02,
     &  8.641e+02,9.935e+02,1.075e+03,1.033e+03,1.009e+03,1.066e+03,
     &  1.067e+03,1.005e+03,9.715e+02,9.232e+02,8.157e+02,7.997e+02/
      data (sun2b(i),i=121,180)/
     &  9.462e+02,1.100e+03,1.126e+03,1.032e+03,8.951e+02,7.843e+02,
     &  7.348e+02,7.265e+02,7.269e+02,7.655e+02,8.639e+02,9.922e+02,
     &  1.071e+03,1.028e+03,8.588e+02,6.472e+02,5.632e+02,6.800e+02,
     &  9.064e+02,1.094e+03,1.155e+03,1.124e+03,1.098e+03,1.110e+03,
     &  1.076e+03,9.442e+02,8.492e+02,9.285e+02,1.062e+03,1.119e+03,
     &  1.119e+03,1.075e+03,1.006e+03,9.800e+02,9.991e+02,1.002e+03,
     &  9.398e+02,8.381e+02,8.161e+02,9.087e+02,1.015e+03,1.058e+03,
     &  1.044e+03,9.875e+02,9.463e+02,9.814e+02,1.056e+03,1.094e+03,
     &  1.028e+03,9.164e+02,9.090e+02,9.918e+02,1.050e+03,1.076e+03,
     &  1.094e+03,1.076e+03,1.015e+03,9.496e+02,9.473e+02,1.001e+03/
      data (sun2b(i),i=181,240)/
     &  1.052e+03,1.073e+03,1.068e+03,1.013e+03,9.078e+02,8.663e+02,
     &  9.509e+02,1.038e+03,1.080e+03,1.184e+03,1.291e+03,1.269e+03,
     &  1.199e+03,1.189e+03,1.188e+03,1.187e+03,1.198e+03,1.171e+03,
     &  1.133e+03,1.132e+03,1.096e+03,9.711e+02,8.471e+02,8.366e+02,
     &  9.228e+02,9.910e+02,9.875e+02,9.692e+02,9.815e+02,9.814e+02,
     &  9.720e+02,9.853e+02,1.003e+03,1.037e+03,1.071e+03,1.066e+03,
     &  1.027e+03,9.848e+02,1.003e+03,1.070e+03,1.118e+03,1.116e+03,
     &  1.049e+03,9.653e+02,9.723e+02,1.046e+03,1.097e+03,1.128e+03,
     &  1.134e+03,1.100e+03,1.079e+03,1.083e+03,1.027e+03,9.275e+02,
     &  8.791e+02,8.588e+02,8.310e+02,8.078e+02,7.896e+02,8.138e+02/
      data (sun2b(i),i=241,300)/
     &  8.935e+02,9.376e+02,9.016e+02,8.645e+02,8.733e+02,8.910e+02,
     &  8.625e+02,8.103e+02,7.874e+02,7.529e+02,7.153e+02,7.081e+02,
     &  7.289e+02,7.868e+02,8.077e+02,7.363e+02,6.451e+02,6.169e+02,
     &  6.492e+02,6.918e+02,7.492e+02,8.202e+02,8.207e+02,7.913e+02,
     &  8.543e+02,9.406e+02,9.564e+02,9.094e+02,8.242e+02,7.672e+02,
     &  7.221e+02,6.534e+02,6.247e+02,6.337e+02,6.551e+02,7.079e+02,
     &  7.849e+02,8.808e+02,9.612e+02,9.856e+02,9.862e+02,9.665e+02,
     &  9.215e+02,8.889e+02,8.558e+02,8.517e+02,8.868e+02,8.510e+02,
     &  7.670e+02,7.390e+02,7.245e+02,6.576e+02,5.878e+02,6.169e+02,
     &  7.606e+02,9.032e+02,9.173e+02,8.385e+02,7.848e+02,7.594e+02/
      data (sun2b(i),i=301,360)/
     &  7.196e+02,6.715e+02,6.246e+02,5.886e+02,5.747e+02,5.967e+02,
     &  6.980e+02,8.664e+02,9.748e+02,9.604e+02,9.301e+02,9.627e+02,
     &  1.007e+03,1.002e+03,9.263e+02,8.166e+02,7.633e+02,7.729e+02,
     &  7.627e+02,7.294e+02,7.250e+02,7.272e+02,6.727e+02,5.814e+02,
     &  5.210e+02,4.888e+02,4.786e+02,5.421e+02,6.637e+02,7.495e+02,
     &  7.859e+02,8.110e+02,8.182e+02,8.138e+02,8.245e+02,8.366e+02,
     &  7.997e+02,7.280e+02,6.604e+02,5.593e+02,4.733e+02,5.502e+02,
     &  7.520e+02,8.858e+02,9.068e+02,9.122e+02,9.293e+02,8.997e+02,
     &  8.302e+02,7.746e+02,7.364e+02,7.241e+02,7.401e+02,7.541e+02,
     &  7.650e+02,7.808e+02,7.889e+02,7.849e+02,7.588e+02,7.259e+02/
      data (sun2b(i),i=361,420)/
     &  7.518e+02,8.042e+02,7.777e+02,7.034e+02,6.653e+02,6.640e+02,
     &  6.794e+02,7.061e+02,7.576e+02,8.361e+02,8.800e+02,8.812e+02,
     &  9.079e+02,9.293e+02,8.943e+02,8.740e+02,9.186e+02,9.535e+02,
     &  9.223e+02,8.666e+02,8.365e+02,8.253e+02,7.525e+02,5.860e+02,
     &  4.275e+02,3.740e+02,4.372e+02,5.343e+02,5.567e+02,5.631e+02,
     &  6.293e+02,6.313e+02,5.188e+02,4.383e+02,4.603e+02,5.305e+02,
     &  6.085e+02,6.580e+02,6.621e+02,6.862e+02,7.752e+02,8.431e+02,
     &  7.975e+02,6.853e+02,6.113e+02,6.287e+02,7.114e+02,7.549e+02,
     &  7.288e+02,7.228e+02,7.264e+02,6.797e+02,6.658e+02,7.105e+02,
     &  7.231e+02,7.241e+02,7.602e+02,7.840e+02,7.428e+02,6.343e+02/
      data (sun2b(i),i=421,480)/
     &  5.465e+02,5.635e+02,6.110e+02,6.232e+02,6.654e+02,7.435e+02,
     &  7.645e+02,6.711e+02,5.132e+02,4.019e+02,4.058e+02,5.157e+02,
     &  6.399e+02,6.778e+02,6.795e+02,7.593e+02,8.481e+02,8.199e+02,
     &  7.518e+02,7.105e+02,6.153e+02,5.251e+02,5.833e+02,7.152e+02,
     &  7.675e+02,7.391e+02,6.640e+02,5.806e+02,5.728e+02,6.341e+02,
     &  6.488e+02,5.613e+02,4.977e+02,5.917e+02,7.378e+02,7.942e+02,
     &  8.025e+02,7.993e+02,7.358e+02,6.584e+02,6.595e+02,7.182e+02,
     &  7.617e+02,6.972e+02,5.451e+02,4.745e+02,5.270e+02,5.977e+02,
     &  5.847e+02,4.473e+02,2.914e+02,2.613e+02,3.303e+02,4.020e+02,
     &  4.663e+02,5.313e+02,5.723e+02,5.849e+02,5.852e+02,5.695e+02/
      data (sun2b(i),i=481,540)/
     &  5.583e+02,5.594e+02,5.120e+02,4.264e+02,3.781e+02,3.983e+02,
     &  4.735e+02,5.422e+02,5.318e+02,4.375e+02,3.419e+02,3.058e+02,
     &  2.999e+02,3.281e+02,4.400e+02,5.865e+02,6.603e+02,6.252e+02,
     &  5.103e+02,4.189e+02,4.474e+02,5.349e+02,6.059e+02,6.671e+02,
     &  6.873e+02,6.368e+02,5.496e+02,4.729e+02,4.195e+02,3.701e+02,
     &  3.280e+02,3.205e+02,3.540e+02,3.992e+02,4.510e+02,5.283e+02,
     &  6.083e+02,6.961e+02,7.743e+02,7.608e+02,6.906e+02,6.482e+02,
     &  5.806e+02,4.780e+02,4.539e+02,4.887e+02,4.640e+02,4.216e+02,
     &  4.443e+02,4.466e+02,3.760e+02,3.421e+02,3.975e+02,5.110e+02,
     &  6.464e+02,7.251e+02,7.031e+02,6.391e+02,6.191e+02,6.547e+02/
      data (sun2b(i),i=541,600)/
     &  6.660e+02,6.114e+02,5.802e+02,6.073e+02,5.910e+02,5.423e+02,
     &  5.838e+02,6.730e+02,6.732e+02,5.824e+02,4.657e+02,3.773e+02,
     &  3.770e+02,4.873e+02,6.079e+02,6.175e+02,5.835e+02,6.017e+02,
     &  6.159e+02,5.755e+02,5.416e+02,5.421e+02,5.223e+02,4.725e+02,
     &  4.233e+02,4.381e+02,5.567e+02,6.643e+02,6.699e+02,6.575e+02,
     &  6.847e+02,7.057e+02,6.831e+02,6.008e+02,5.099e+02,4.976e+02,
     &  5.111e+02,4.961e+02,5.003e+02,5.187e+02,5.299e+02,5.630e+02,
     &  6.092e+02,6.265e+02,6.221e+02,6.157e+02,6.004e+02,5.913e+02,
     &  5.981e+02,5.931e+02,5.909e+02,6.316e+02,6.965e+02,7.185e+02,
     &  6.761e+02,6.316e+02,6.196e+02,6.205e+02,6.241e+02,6.366e+02/
      data (sun2b(i),i=601,660)/
     &  6.580e+02,6.888e+02,7.248e+02,7.426e+02,7.223e+02,6.759e+02,
     &  6.660e+02,7.047e+02,7.037e+02,6.450e+02,5.983e+02,5.878e+02,
     &  5.909e+02,5.759e+02,5.280e+02,4.779e+02,4.575e+02,4.568e+02,
     &  4.549e+02,4.486e+02,4.455e+02,4.454e+02,4.444e+02,4.460e+02,
     &  4.559e+02,4.680e+02,4.543e+02,3.933e+02,3.012e+02,2.114e+02,
     &  1.671e+02,1.940e+02,2.540e+02,3.054e+02,3.530e+02,3.851e+02,
     &  3.870e+02,3.916e+02,4.062e+02,4.153e+02,4.353e+02,4.698e+02,
     &  4.921e+02,4.727e+02,4.099e+02,3.533e+02,3.407e+02,3.553e+02,
     &  3.798e+02,4.018e+02,4.097e+02,4.069e+02,3.932e+02,3.789e+02,
     &  3.752e+02,3.735e+02,3.602e+02,3.228e+02,2.735e+02,2.378e+02/
      data (sun2b(i),i=661,720)/
     &  2.123e+02,1.848e+02,1.562e+02,1.278e+02,9.627e+01,6.881e+01,
     &  6.205e+01,7.714e+01,1.005e+02,1.276e+02,1.599e+02,1.941e+02,
     &  2.252e+02,2.546e+02,2.858e+02,3.001e+02,2.944e+02,3.089e+02,
     &  3.408e+02,3.463e+02,3.363e+02,3.475e+02,3.738e+02,3.888e+02,
     &  3.727e+02,3.253e+02,2.944e+02,3.176e+02,3.603e+02,3.781e+02,
     &  3.742e+02,3.740e+02,3.833e+02,3.879e+02,3.775e+02,3.570e+02,
     &  3.407e+02,3.287e+02,3.140e+02,3.169e+02,3.445e+02,3.555e+02,
     &  3.357e+02,3.187e+02,3.186e+02,3.224e+02,3.186e+02,3.049e+02,
     &  2.848e+02,2.681e+02,2.658e+02,2.735e+02,2.742e+02,2.524e+02,
     &  2.150e+02,1.886e+02,1.813e+02,1.813e+02,1.808e+02,1.752e+02/
      data (sun2b(i),i=721,780)/
     &  1.621e+02,1.451e+02,1.288e+02,1.138e+02,9.808e+01,8.307e+01,
     &  7.622e+01,7.836e+01,7.843e+01,7.424e+01,7.584e+01,8.032e+01,
     &  7.786e+01,7.030e+01,6.465e+01,6.705e+01,7.781e+01,8.317e+01,
     &  7.529e+01,7.120e+01,8.055e+01,9.201e+01,1.002e+02,1.086e+02,
     &  1.194e+02,1.308e+02,1.423e+02,1.589e+02,1.771e+02,1.864e+02,
     &  1.866e+02,1.815e+02,1.753e+02,1.755e+02,1.790e+02,1.770e+02,
     &  1.726e+02,1.727e+02,1.790e+02,1.938e+02,2.151e+02,2.336e+02,
     &  2.521e+02,2.777e+02,2.989e+02,2.984e+02,2.808e+02,2.742e+02,
     &  2.865e+02,2.855e+02,2.597e+02,2.414e+02,2.470e+02,2.599e+02,
     &  2.743e+02,2.985e+02,3.169e+02,3.032e+02,2.637e+02,2.293e+02/
      data (sun2b(i),i=781,840)/
     &  2.279e+02,2.561e+02,2.816e+02,3.002e+02,3.106e+02,2.795e+02,
     &  2.119e+02,1.522e+02,1.299e+02,1.475e+02,1.816e+02,2.154e+02,
     &  2.395e+02,2.331e+02,1.916e+02,1.394e+02,1.105e+02,1.189e+02,
     &  1.348e+02,1.291e+02,1.244e+02,1.435e+02,1.583e+02,1.418e+02,
     &  1.163e+02,1.116e+02,1.289e+02,1.492e+02,1.534e+02,1.456e+02,
     &  1.485e+02,1.593e+02,1.558e+02,1.542e+02,1.773e+02,2.034e+02,
     &  2.074e+02,2.053e+02,2.229e+02,2.532e+02,2.713e+02,2.793e+02,
     &  3.022e+02,3.215e+02,2.888e+02,2.301e+02,2.064e+02,2.132e+02,
     &  2.165e+02,2.075e+02,1.962e+02,1.952e+02,2.020e+02,1.943e+02,
     &  1.649e+02,1.366e+02,1.239e+02,1.281e+02,1.619e+02,2.170e+02/
      data (sun2b(i),i=841,900)/
     &  2.537e+02,2.493e+02,2.229e+02,2.131e+02,2.436e+02,2.931e+02,
     &  3.094e+02,2.864e+02,2.696e+02,2.722e+02,2.717e+02,2.658e+02,
     &  2.656e+02,2.648e+02,2.660e+02,2.895e+02,3.257e+02,3.373e+02,
     &  3.212e+02,3.003e+02,2.826e+02,2.871e+02,3.221e+02,3.358e+02,
     &  2.972e+02,2.541e+02,2.435e+02,2.395e+02,2.193e+02,2.119e+02,
     &  2.393e+02,2.714e+02,2.794e+02,2.723e+02,2.648e+02,2.505e+02,
     &  2.299e+02,2.221e+02,2.353e+02,2.568e+02,2.753e+02,2.869e+02,
     &  2.849e+02,2.695e+02,2.551e+02,2.535e+02,2.632e+02,2.748e+02,
     &  2.792e+02,2.702e+02,2.494e+02,2.290e+02,2.216e+02,2.314e+02,
     &  2.527e+02,2.806e+02,3.101e+02,3.283e+02,3.250e+02,2.903e+02/
      data (sun2b(i),i=901,960)/
     &  2.390e+02,2.234e+02,2.572e+02,2.826e+02,2.643e+02,2.433e+02,
     &  2.532e+02,2.729e+02,2.713e+02,2.561e+02,2.602e+02,2.714e+02,
     &  2.571e+02,2.366e+02,2.387e+02,2.489e+02,2.559e+02,2.720e+02,
     &  2.918e+02,2.974e+02,2.881e+02,2.833e+02,2.929e+02,3.017e+02,
     &  3.091e+02,3.220e+02,3.204e+02,2.954e+02,2.696e+02,2.544e+02,
     &  2.409e+02,2.282e+02,2.212e+02,2.137e+02,2.012e+02,1.972e+02,
     &  2.123e+02,2.334e+02,2.476e+02,2.617e+02,2.862e+02,3.225e+02,
     &  3.495e+02,3.383e+02,2.971e+02,2.615e+02,2.523e+02,2.646e+02,
     &  2.869e+02,2.989e+02,2.805e+02,2.444e+02,2.135e+02,1.930e+02,
     &  1.821e+02,1.685e+02,1.431e+02,1.141e+02,8.961e+01,7.359e+01/
      data (sun2b(i),i=961,1020)/
     &  7.399e+01,8.791e+01,9.626e+01,9.481e+01,9.660e+01,1.023e+02,
     &  1.022e+02,1.031e+02,1.178e+02,1.374e+02,1.461e+02,1.443e+02,
     &  1.379e+02,1.281e+02,1.228e+02,1.282e+02,1.307e+02,1.173e+02,
     &  9.891e+01,9.340e+01,1.056e+02,1.227e+02,1.264e+02,1.131e+02,
     &  9.232e+01,7.634e+01,6.903e+01,6.632e+01,7.128e+01,8.743e+01,
     &  1.059e+02,1.140e+02,1.079e+02,9.187e+01,7.521e+01,6.912e+01,
     &  7.593e+01,9.093e+01,1.097e+02,1.257e+02,1.358e+02,1.411e+02,
     &  1.381e+02,1.213e+02,9.181e+01,6.350e+01,5.211e+01,5.956e+01,
     &  8.101e+01,1.067e+02,1.190e+02,1.164e+02,1.108e+02,1.009e+02,
     &  8.906e+01,9.043e+01,1.044e+02,1.149e+02,1.248e+02,1.489e+02/
      data (sun2b(i),i=1021,1080)/
     &  1.717e+02,1.672e+02,1.423e+02,1.184e+02,9.865e+01,7.891e+01,
     &  6.813e+01,7.729e+01,1.009e+02,1.201e+02,1.255e+02,1.318e+02,
     &  1.557e+02,1.808e+02,1.818e+02,1.668e+02,1.501e+02,1.332e+02,
     &  1.161e+02,9.773e+01,8.163e+01,7.669e+01,8.761e+01,1.102e+02,
     &  1.349e+02,1.491e+02,1.476e+02,1.399e+02,1.352e+02,1.351e+02,
     &  1.380e+02,1.367e+02,1.288e+02,1.222e+02,1.205e+02,1.220e+02,
     &  1.231e+02,1.163e+02,1.014e+02,8.630e+01,7.472e+01,6.880e+01,
     &  7.133e+01,8.063e+01,9.049e+01,9.674e+01,1.007e+02,1.008e+02,
     &  9.368e+01,8.474e+01,8.153e+01,8.289e+01,8.456e+01,8.758e+01,
     &  9.178e+01,9.127e+01,8.701e+01,8.739e+01,9.015e+01,8.492e+01/
      data (sun2b(i),i=1081,1140)/
     &  7.127e+01,5.787e+01,5.186e+01,5.388e+01,5.791e+01,5.851e+01,
     &  5.702e+01,5.743e+01,6.067e+01,6.467e+01,6.736e+01,6.751e+01,
     &  6.423e+01,5.903e+01,5.570e+01,5.664e+01,5.940e+01,5.907e+01,
     &  5.652e+01,5.583e+01,5.586e+01,5.404e+01,5.198e+01,5.234e+01,
     &  5.467e+01,5.645e+01,5.675e+01,5.677e+01,5.800e+01,6.003e+01,
     &  5.960e+01,5.313e+01,4.293e+01,3.559e+01,3.345e+01,3.517e+01,
     &  3.938e+01,4.437e+01,4.774e+01,4.693e+01,4.244e+01,3.788e+01,
     &  3.560e+01,3.646e+01,4.105e+01,4.730e+01,5.110e+01,5.002e+01,
     &  4.533e+01,4.128e+01,4.008e+01,4.000e+01,3.910e+01,3.733e+01,
     &  3.663e+01,3.779e+01,3.919e+01,4.106e+01,4.521e+01,5.074e+01/
      data (sun2b(i),i=1141,1200)/
     &  5.428e+01,5.501e+01,5.614e+01,6.093e+01,6.738e+01,6.953e+01,
     &  6.516e+01,5.637e+01,4.733e+01,4.432e+01,4.994e+01,5.970e+01,
     &  6.793e+01,7.133e+01,6.990e+01,6.562e+01,5.930e+01,5.402e+01,
     &  5.588e+01,6.515e+01,7.407e+01,7.622e+01,7.351e+01,7.141e+01,
     &  7.085e+01,6.975e+01,6.927e+01,7.138e+01,7.272e+01,6.893e+01,
     &  6.167e+01,5.490e+01,4.742e+01,3.833e+01,3.222e+01,3.124e+01,
     &  3.331e+01,3.536e+01,3.562e+01,3.684e+01,4.155e+01,4.750e+01,
     &  5.118e+01,5.034e+01,4.536e+01,3.834e+01,3.313e+01,3.380e+01,
     &  4.014e+01,4.912e+01,5.538e+01,5.517e+01,5.045e+01,4.651e+01,
     &  4.749e+01,5.188e+01,5.635e+01,5.960e+01,6.158e+01,6.322e+01/
      data (sun2b(i),i=1201,1260)/
     &  6.460e+01,6.410e+01,5.903e+01,5.096e+01,4.763e+01,5.254e+01,
     &  5.888e+01,5.983e+01,5.762e+01,5.673e+01,5.737e+01,5.790e+01,
     &  5.718e+01,5.513e+01,5.295e+01,5.202e+01,5.219e+01,5.204e+01,
     &  5.027e+01,4.659e+01,4.251e+01,4.076e+01,4.189e+01,4.412e+01,
     &  4.654e+01,4.886e+01,5.049e+01,5.192e+01,5.408e+01,5.471e+01,
     &  5.193e+01,4.945e+01,4.987e+01,5.093e+01,5.050e+01,4.862e+01,
     &  4.672e+01,4.607e+01,4.626e+01,4.673e+01,4.801e+01,5.019e+01,
     &  5.242e+01,5.354e+01,5.251e+01,5.138e+01,5.321e+01,5.699e+01,
     &  6.061e+01,6.314e+01,6.400e+01,6.387e+01,6.510e+01,6.939e+01,
     &  7.474e+01,7.818e+01,7.810e+01,7.411e+01,6.737e+01,6.085e+01/
      data (sun2b(i),i=1261,1320)/
     &  5.892e+01,6.268e+01,6.803e+01,6.912e+01,6.460e+01,5.911e+01,
     &  5.600e+01,5.684e+01,6.178e+01,6.587e+01,6.508e+01,6.304e+01,
     &  6.481e+01,6.991e+01,7.484e+01,7.644e+01,7.359e+01,6.885e+01,
     &  6.750e+01,7.268e+01,8.060e+01,8.342e+01,7.896e+01,7.223e+01,
     &  6.674e+01,6.284e+01,6.153e+01,6.357e+01,6.925e+01,7.658e+01,
     &  7.992e+01,7.775e+01,7.394e+01,7.052e+01,6.800e+01,6.634e+01,
     &  6.398e+01,6.110e+01,5.942e+01,5.810e+01,5.574e+01,5.255e+01,
     &  4.808e+01,4.258e+01,3.837e+01,3.730e+01,3.746e+01,3.486e+01,
     &  3.048e+01,2.963e+01,3.473e+01,4.246e+01,4.707e+01,4.585e+01,
     &  4.016e+01,3.429e+01,3.158e+01,3.065e+01,2.905e+01,2.779e+01/
      data (sun2b(i),i=1321,1380)/
     &  3.043e+01,3.757e+01,4.420e+01,4.688e+01,4.785e+01,4.917e+01,
     &  4.918e+01,4.500e+01,3.813e+01,3.506e+01,3.810e+01,4.175e+01,
     &  4.090e+01,3.572e+01,2.888e+01,2.483e+01,2.867e+01,3.965e+01,
     &  5.031e+01,5.572e+01,5.740e+01,5.811e+01,5.941e+01,5.936e+01,
     &  5.342e+01,4.300e+01,3.479e+01,3.370e+01,3.968e+01,4.755e+01,
     &  5.260e+01,5.363e+01,5.100e+01,4.527e+01,3.784e+01,3.103e+01,
     &  2.594e+01,2.280e+01,2.188e+01,2.348e+01,2.786e+01,3.345e+01,
     &  3.732e+01,3.919e+01,4.283e+01,5.040e+01,5.875e+01,6.330e+01,
     &  6.109e+01,5.353e+01,4.605e+01,4.112e+01,3.765e+01,3.630e+01,
     &  4.043e+01,5.089e+01,6.155e+01,6.539e+01,6.268e+01,5.809e+01/
      data (sun2b(i),i=1381,1440)/
     &  5.462e+01,5.133e+01,4.687e+01,4.287e+01,4.055e+01,3.976e+01,
     &  4.022e+01,4.036e+01,3.956e+01,4.067e+01,4.626e+01,5.341e+01,
     &  5.604e+01,5.257e+01,4.667e+01,4.107e+01,3.551e+01,3.123e+01,
     &  3.108e+01,3.596e+01,4.520e+01,5.546e+01,6.180e+01,6.351e+01,
     &  6.185e+01,5.641e+01,4.939e+01,4.637e+01,5.006e+01,5.669e+01,
     &  6.088e+01,6.103e+01,5.811e+01,5.430e+01,5.194e+01,5.051e+01,
     &  4.675e+01,3.915e+01,3.153e+01,2.896e+01,3.097e+01,3.267e+01,
     &  3.157e+01,2.934e+01,2.727e+01,2.518e+01,2.426e+01,2.707e+01,
     &  3.430e+01,4.247e+01,4.723e+01,4.742e+01,4.444e+01,4.054e+01,
     &  3.687e+01,3.302e+01,2.941e+01,2.875e+01,3.408e+01,4.425e+01/
      data (sun2b(i),i=1441,1500)/
     &  5.378e+01,5.797e+01,5.638e+01,5.120e+01,4.531e+01,4.027e+01,
     &  3.590e+01,3.334e+01,3.401e+01,3.686e+01,4.128e+01,4.737e+01,
     &  5.309e+01,5.620e+01,5.563e+01,5.084e+01,4.400e+01,3.877e+01,
     &  3.625e+01,3.638e+01,4.076e+01,5.070e+01,6.337e+01,7.343e+01,
     &  7.642e+01,7.037e+01,5.874e+01,4.703e+01,3.860e+01,3.466e+01,
     &  3.579e+01,4.208e+01,4.997e+01,5.434e+01,5.396e+01,5.229e+01,
     &  5.278e+01,5.557e+01,5.903e+01,6.027e+01,5.625e+01,4.736e+01,
     &  3.806e+01,3.289e+01,3.174e+01,3.173e+01,3.248e+01,3.506e+01,
     &  3.909e+01,4.340e+01,4.813e+01,5.357e+01,5.875e+01,6.360e+01,
     &  6.897e+01,7.342e+01,7.386e+01,6.900e+01,6.056e+01,5.187e+01/
      data (sun2b(i),i=1501,1560)/
     &  4.488e+01,4.206e+01,4.480e+01,4.795e+01,4.688e+01,4.297e+01,
     &  3.929e+01,3.771e+01,3.714e+01,3.522e+01,3.224e+01,3.049e+01,
     &  3.260e+01,4.043e+01,5.110e+01,5.771e+01,5.715e+01,5.299e+01,
     &  5.028e+01,4.999e+01,4.978e+01,4.837e+01,4.642e+01,4.460e+01,
     &  4.273e+01,4.124e+01,4.156e+01,4.381e+01,4.701e+01,4.899e+01,
     &  4.643e+01,4.060e+01,3.784e+01,4.235e+01,5.225e+01,6.053e+01,
     &  6.157e+01,5.680e+01,5.204e+01,5.226e+01,5.708e+01,6.102e+01,
     &  6.071e+01,5.705e+01,5.148e+01,4.635e+01,4.437e+01,4.495e+01,
     &  4.548e+01,4.494e+01,4.383e+01,4.210e+01,3.947e+01,3.683e+01,
     &  3.591e+01,3.636e+01,3.566e+01,3.395e+01,3.369e+01,3.443e+01/
      data (sun2b(i),i=1561,1620)/
     &  3.400e+01,3.265e+01,3.141e+01,3.028e+01,2.941e+01,2.913e+01,
     &  2.933e+01,2.987e+01,3.060e+01,3.131e+01,3.210e+01,3.278e+01,
     &  3.276e+01,3.210e+01,3.198e+01,3.348e+01,3.605e+01,3.917e+01,
     &  4.337e+01,4.724e+01,4.821e+01,4.579e+01,4.159e+01,3.878e+01,
     &  4.075e+01,4.675e+01,5.168e+01,5.260e+01,5.145e+01,5.068e+01,
     &  4.945e+01,4.675e+01,4.537e+01,4.769e+01,5.024e+01,4.896e+01,
     &  4.669e+01,4.860e+01,5.369e+01,5.647e+01,5.434e+01,5.072e+01,
     &  4.988e+01,5.125e+01,5.209e+01,5.276e+01,5.625e+01,6.333e+01,
     &  6.974e+01,7.107e+01,6.835e+01,6.512e+01,6.255e+01,5.919e+01,
     &  5.371e+01,4.816e+01,4.624e+01,4.771e+01,4.966e+01,5.080e+01/
      data (sun2b(i),i=1621,1680)/
     &  5.184e+01,5.454e+01,5.965e+01,6.471e+01,6.584e+01,6.163e+01,
     &  5.557e+01,5.408e+01,5.878e+01,6.489e+01,6.978e+01,7.401e+01,
     &  7.649e+01,7.623e+01,7.475e+01,7.494e+01,7.780e+01,7.962e+01,
     &  7.619e+01,6.719e+01,5.523e+01,4.581e+01,4.314e+01,4.565e+01,
     &  4.947e+01,5.223e+01,5.222e+01,4.889e+01,4.472e+01,4.261e+01,
     &  4.338e+01,4.597e+01,4.812e+01,4.900e+01,4.988e+01,5.071e+01,
     &  4.989e+01,4.832e+01,4.820e+01,5.028e+01,5.383e+01,5.591e+01,
     &  5.482e+01,5.294e+01,5.194e+01,4.944e+01,4.296e+01,3.461e+01,
     &  2.810e+01,2.450e+01,2.420e+01,2.784e+01,3.460e+01,4.162e+01,
     &  4.532e+01,4.544e+01,4.553e+01,4.718e+01,4.576e+01,3.686e+01/
      data (sun2b(i),i=1681,1740)/
     &  2.604e+01,2.057e+01,2.033e+01,2.426e+01,3.086e+01,3.594e+01,
     &  3.671e+01,3.569e+01,3.726e+01,4.086e+01,4.442e+01,4.880e+01,
     &  5.418e+01,5.765e+01,5.843e+01,5.997e+01,6.394e+01,6.682e+01,
     &  6.546e+01,5.948e+01,4.940e+01,3.942e+01,3.418e+01,3.539e+01,
     &  4.288e+01,5.203e+01,5.760e+01,5.909e+01,5.727e+01,5.217e+01,
     &  4.549e+01,3.942e+01,3.558e+01,3.590e+01,4.035e+01,4.673e+01,
     &  5.331e+01,5.878e+01,6.178e+01,5.926e+01,5.003e+01,4.157e+01,
     &  4.052e+01,4.358e+01,4.488e+01,4.275e+01,4.008e+01,3.994e+01,
     &  4.098e+01,3.957e+01,3.496e+01,3.042e+01,3.104e+01,3.869e+01,
     &  4.748e+01,4.983e+01,4.679e+01,4.483e+01,4.655e+01,5.042e+01/
      data (sun2b(i),i=1741,1800)/
     &  5.460e+01,5.766e+01,5.846e+01,5.728e+01,5.562e+01,5.451e+01,
     &  5.334e+01,5.003e+01,4.282e+01,3.364e+01,2.713e+01,2.552e+01,
     &  2.790e+01,3.139e+01,3.213e+01,2.946e+01,2.658e+01,2.596e+01,
     &  2.774e+01,3.117e+01,3.496e+01,3.767e+01,3.864e+01,3.796e+01,
     &  3.673e+01,3.568e+01,3.388e+01,3.085e+01,2.806e+01,2.761e+01,
     &  2.932e+01,2.938e+01,2.539e+01,2.066e+01,1.948e+01,2.230e+01,
     &  2.728e+01,3.247e+01,3.591e+01,3.714e+01,3.790e+01,3.913e+01,
     &  3.978e+01,3.987e+01,4.078e+01,4.232e+01,4.293e+01,4.043e+01,
     &  3.423e+01,2.770e+01,2.388e+01,2.217e+01,2.164e+01,2.259e+01,
     &  2.518e+01,2.902e+01,3.298e+01,3.611e+01,3.858e+01,4.124e+01/
      data (sun2b(i),i=1801,1860)/
     &  4.443e+01,4.694e+01,4.701e+01,4.417e+01,3.966e+01,3.556e+01,
     &  3.284e+01,3.155e+01,3.268e+01,3.696e+01,4.233e+01,4.493e+01,
     &  4.370e+01,4.094e+01,3.797e+01,3.520e+01,3.357e+01,3.334e+01,
     &  3.419e+01,3.635e+01,3.996e+01,4.396e+01,4.716e+01,4.899e+01,
     &  4.898e+01,4.795e+01,4.800e+01,4.989e+01,5.106e+01,4.783e+01,
     &  4.049e+01,3.267e+01,2.680e+01,2.446e+01,2.666e+01,3.193e+01,
     &  3.763e+01,4.135e+01,4.096e+01,3.683e+01,3.211e+01,2.861e+01,
     &  2.648e+01,2.660e+01,2.883e+01,3.088e+01,3.098e+01,3.006e+01,
     &  2.989e+01,3.031e+01,2.997e+01,2.826e+01,2.652e+01,2.707e+01,
     &  3.040e+01,3.454e+01,3.710e+01,3.760e+01,3.725e+01,3.706e+01/
      data (sun2b(i),i=1861,1920)/
     &  3.650e+01,3.417e+01,2.981e+01,2.419e+01,1.851e+01,1.509e+01,
     &  1.504e+01,1.716e+01,2.081e+01,2.568e+01,3.035e+01,3.420e+01,
     &  3.790e+01,4.253e+01,4.783e+01,5.051e+01,4.802e+01,4.262e+01,
     &  3.832e+01,3.737e+01,4.017e+01,4.440e+01,4.613e+01,4.391e+01,
     &  3.840e+01,3.138e+01,2.627e+01,2.508e+01,2.665e+01,2.896e+01,
     &  3.117e+01,3.417e+01,3.805e+01,4.023e+01,3.835e+01,3.274e+01,
     &  2.620e+01,2.186e+01,2.025e+01,2.018e+01,2.173e+01,2.556e+01,
     &  3.032e+01,3.343e+01,3.445e+01,3.496e+01,3.637e+01,3.787e+01,
     &  3.665e+01,3.197e+01,2.592e+01,2.126e+01,2.066e+01,2.466e+01,
     &  3.026e+01,3.402e+01,3.434e+01,3.136e+01,2.693e+01,2.311e+01/
      data (sun2b(i),i=1921,1980)/
     &  2.087e+01,2.068e+01,2.242e+01,2.488e+01,2.678e+01,2.733e+01,
     &  2.654e+01,2.521e+01,2.401e+01,2.294e+01,2.180e+01,2.045e+01,
     &  1.929e+01,1.953e+01,2.174e+01,2.413e+01,2.399e+01,2.156e+01,
     &  1.956e+01,1.892e+01,1.834e+01,1.733e+01,1.655e+01,1.648e+01,
     &  1.721e+01,1.844e+01,1.929e+01,1.898e+01,1.791e+01,1.716e+01,
     &  1.710e+01,1.726e+01,1.692e+01,1.584e+01,1.448e+01,1.368e+01,
     &  1.365e+01,1.391e+01,1.401e+01,1.377e+01,1.346e+01,1.340e+01,
     &  1.355e+01,1.376e+01,1.406e+01,1.443e+01,1.464e+01,1.444e+01,
     &  1.399e+01,1.375e+01,1.393e+01,1.439e+01,1.476e+01,1.482e+01,
     &  1.468e+01,1.445e+01,1.419e+01,1.394e+01,1.375e+01,1.366e+01/
      data (sun2b(i),i=1981,2040)/
     &  1.354e+01,1.331e+01,1.305e+01,1.284e+01,1.270e+01,1.274e+01,
     &  1.281e+01,1.266e+01,1.235e+01,1.210e+01,1.200e+01,1.201e+01,
     &  1.207e+01,1.222e+01,1.244e+01,1.247e+01,1.216e+01,1.173e+01,
     &  1.152e+01,1.162e+01,1.187e+01,1.203e+01,1.195e+01,1.172e+01,
     &  1.140e+01,1.093e+01,1.047e+01,1.020e+01,1.022e+01,1.069e+01,
     &  1.147e+01,1.201e+01,1.208e+01,1.190e+01,1.150e+01,1.089e+01,
     &  1.057e+01,1.085e+01,1.135e+01,1.161e+01,1.141e+01,1.088e+01,
     &  1.038e+01,1.031e+01,1.069e+01,1.124e+01,1.164e+01,1.183e+01,
     &  1.192e+01,1.186e+01,1.167e+01,1.151e+01,1.141e+01,1.130e+01,
     &  1.122e+01,1.114e+01,1.104e+01,1.098e+01,1.100e+01,1.090e+01/
      data (sun2b(i),i=2041,2100)/
     &  1.065e+01,1.056e+01,1.078e+01,1.119e+01,1.160e+01,1.181e+01,
     &  1.158e+01,1.106e+01,1.057e+01,1.034e+01,1.041e+01,1.073e+01,
     &  1.116e+01,1.154e+01,1.165e+01,1.137e+01,1.093e+01,1.052e+01,
     &  9.997e+00,9.378e+00,8.988e+00,9.016e+00,9.413e+00,9.918e+00,
     &  1.028e+01,1.047e+01,1.055e+01,1.057e+01,1.052e+01,1.022e+01,
     &  9.682e+00,9.150e+00,8.706e+00,8.389e+00,8.344e+00,8.624e+00,
     &  9.137e+00,9.718e+00,1.019e+01,1.044e+01,1.046e+01,1.029e+01,
     &  9.977e+00,9.583e+00,9.310e+00,9.319e+00,9.469e+00,9.518e+00,
     &  9.433e+00,9.248e+00,8.820e+00,7.981e+00,7.000e+00,6.486e+00,
     &  6.746e+00,7.541e+00,8.281e+00,8.726e+00,9.068e+00,9.167e+00/
      data (sun2b(i),i=2101,2160)/
     &  8.864e+00,8.464e+00,8.239e+00,8.166e+00,8.188e+00,8.358e+00,
     &  8.649e+00,8.898e+00,9.012e+00,9.066e+00,9.196e+00,9.421e+00,
     &  9.553e+00,9.424e+00,9.129e+00,8.844e+00,8.614e+00,8.424e+00,
     &  8.298e+00,8.260e+00,8.286e+00,8.347e+00,8.453e+00,8.628e+00,
     &  8.831e+00,8.887e+00,8.675e+00,8.331e+00,8.009e+00,7.730e+00,
     &  7.624e+00,7.869e+00,8.273e+00,8.409e+00,8.252e+00,8.091e+00,
     &  8.000e+00,7.937e+00,7.927e+00,7.958e+00,8.049e+00,8.238e+00,
     &  8.426e+00,8.481e+00,8.331e+00,8.026e+00,7.763e+00,7.699e+00,
     &  7.812e+00,7.939e+00,8.018e+00,8.082e+00,8.043e+00,7.892e+00,
     &  7.696e+00,7.497e+00,7.401e+00,7.429e+00,7.449e+00,7.406e+00/
      data (sun2b(i),i=2161,2220)/
     &  7.420e+00,7.526e+00,7.525e+00,7.324e+00,7.126e+00,7.142e+00,
     &  7.334e+00,7.505e+00,7.548e+00,7.532e+00,7.516e+00,7.499e+00,
     &  7.573e+00,7.810e+00,8.059e+00,8.095e+00,7.914e+00,7.698e+00,
     &  7.512e+00,7.214e+00,6.806e+00,6.543e+00,6.523e+00,6.587e+00,
     &  6.561e+00,6.398e+00,6.189e+00,6.059e+00,6.068e+00,6.199e+00,
     &  6.314e+00,6.253e+00,6.093e+00,6.028e+00,6.094e+00,6.303e+00,
     &  6.659e+00,6.940e+00,6.957e+00,6.831e+00,6.737e+00,6.681e+00,
     &  6.656e+00,6.834e+00,7.202e+00,7.401e+00,7.295e+00,7.049e+00,
     &  6.797e+00,6.629e+00,6.587e+00,6.598e+00,6.601e+00,6.650e+00,
     &  6.763e+00,6.785e+00,6.632e+00,6.486e+00,6.520e+00,6.649e+00/
      data (sun2b(i),i=2221,2280)/
     &  6.727e+00,6.723e+00,6.670e+00,6.619e+00,6.598e+00,6.619e+00,
     &  6.711e+00,6.834e+00,6.875e+00,6.825e+00,6.788e+00,6.827e+00,
     &  6.856e+00,6.807e+00,6.838e+00,7.047e+00,7.278e+00,7.414e+00,
     &  7.471e+00,7.462e+00,7.407e+00,7.359e+00,7.337e+00,7.322e+00,
     &  7.294e+00,7.253e+00,7.205e+00,7.134e+00,7.030e+00,6.853e+00,
     &  6.554e+00,6.223e+00,6.014e+00,5.938e+00,5.904e+00,5.857e+00,
     &  5.791e+00,5.733e+00,5.774e+00,5.961e+00,6.187e+00,6.368e+00,
     &  6.500e+00,6.544e+00,6.464e+00,6.271e+00,6.045e+00,5.956e+00,
     &  6.085e+00,6.254e+00,6.245e+00,6.080e+00,5.910e+00,5.845e+00,
     &  5.922e+00,6.106e+00,6.290e+00,6.327e+00,6.110e+00,5.742e+00/
      data (sun2b(i),i=2281,2340)/
     &  5.445e+00,5.298e+00,5.326e+00,5.494e+00,5.682e+00,5.825e+00,
     &  5.893e+00,5.963e+00,6.170e+00,6.452e+00,6.633e+00,6.696e+00,
     &  6.719e+00,6.624e+00,6.311e+00,5.924e+00,5.699e+00,5.665e+00,
     &  5.743e+00,5.879e+00,5.971e+00,5.962e+00,5.967e+00,6.075e+00,
     &  6.254e+00,6.430e+00,6.497e+00,6.456e+00,6.408e+00,6.302e+00,
     &  6.014e+00,5.643e+00,5.396e+00,5.299e+00,5.264e+00,5.223e+00,
     &  5.128e+00,4.931e+00,4.635e+00,4.317e+00,4.015e+00,3.662e+00,
     &  3.291e+00,3.103e+00,3.135e+00,3.199e+00,3.260e+00,3.414e+00,
     &  3.595e+00,3.653e+00,3.630e+00,3.628e+00,3.588e+00,3.429e+00,
     &  3.266e+00,3.228e+00,3.258e+00,3.300e+00,3.454e+00,3.737e+00/
      data (sun2b(i),i=2341,2400)/
     &  4.057e+00,4.356e+00,4.600e+00,4.778e+00,4.885e+00,4.900e+00,
     &  4.839e+00,4.762e+00,4.706e+00,4.698e+00,4.767e+00,4.845e+00,
     &  4.824e+00,4.729e+00,4.686e+00,4.713e+00,4.773e+00,4.871e+00,
     &  4.960e+00,4.953e+00,4.896e+00,4.925e+00,5.074e+00,5.223e+00,
     &  5.251e+00,5.154e+00,5.016e+00,4.888e+00,4.769e+00,4.655e+00,
     &  4.553e+00,4.483e+00,4.466e+00,4.504e+00,4.590e+00,4.703e+00,
     &  4.786e+00,4.833e+00,4.928e+00,5.038e+00,5.006e+00,4.847e+00,
     &  4.683e+00,4.559e+00,4.481e+00,4.431e+00,4.390e+00,4.383e+00,
     &  4.407e+00,4.390e+00,4.297e+00,4.198e+00,4.146e+00,4.108e+00,
     &  4.149e+00,4.390e+00,4.686e+00,4.821e+00,4.794e+00,4.678e+00/
      data (sun2b(i),i=2401,2460)/
     &  4.522e+00,4.396e+00,4.336e+00,4.295e+00,4.226e+00,4.145e+00,
     &  4.106e+00,4.146e+00,4.215e+00,4.255e+00,4.306e+00,4.374e+00,
     &  4.374e+00,4.272e+00,4.139e+00,4.041e+00,3.946e+00,3.813e+00,
     &  3.710e+00,3.734e+00,3.859e+00,3.960e+00,3.953e+00,3.838e+00,
     &  3.671e+00,3.536e+00,3.493e+00,3.537e+00,3.638e+00,3.789e+00,
     &  3.967e+00,4.100e+00,4.142e+00,4.130e+00,4.123e+00,4.162e+00,
     &  4.251e+00,4.295e+00,4.223e+00,4.099e+00,3.968e+00,3.947e+00,
     &  4.196e+00,4.513e+00,4.610e+00,4.513e+00,4.347e+00,4.175e+00,
     &  4.047e+00,3.996e+00,4.008e+00,4.037e+00,4.025e+00,3.938e+00,
     &  3.819e+00,3.753e+00,3.767e+00,3.826e+00,3.865e+00,3.852e+00/
      data (sun2b(i),i=2461,2520)/
     &  3.815e+00,3.803e+00,3.810e+00,3.793e+00,3.766e+00,3.794e+00,
     &  3.869e+00,3.898e+00,3.886e+00,3.892e+00,3.857e+00,3.694e+00,
     &  3.469e+00,3.322e+00,3.282e+00,3.289e+00,3.304e+00,3.322e+00,
     &  3.331e+00,3.333e+00,3.348e+00,3.381e+00,3.419e+00,3.460e+00,
     &  3.497e+00,3.515e+00,3.516e+00,3.516e+00,3.526e+00,3.541e+00,
     &  3.548e+00,3.534e+00,3.534e+00,3.582e+00,3.640e+00,3.645e+00,
     &  3.587e+00,3.511e+00,3.480e+00,3.526e+00,3.624e+00,3.729e+00,
     &  3.802e+00,3.802e+00,3.727e+00,3.658e+00,3.644e+00,3.642e+00,
     &  3.637e+00,3.640e+00,3.641e+00,3.649e+00,3.661e+00,3.625e+00,
     &  3.521e+00,3.402e+00,3.284e+00,3.123e+00,2.948e+00,2.852e+00/
      data (sun2b(i),i=2521,2580)/
     &  2.843e+00,2.864e+00,2.897e+00,2.951e+00,3.003e+00,3.055e+00,
     &  3.111e+00,3.118e+00,3.060e+00,2.985e+00,2.932e+00,2.890e+00,
     &  2.859e+00,2.850e+00,2.845e+00,2.812e+00,2.763e+00,2.742e+00,
     &  2.767e+00,2.802e+00,2.816e+00,2.811e+00,2.813e+00,2.857e+00,
     &  2.966e+00,3.106e+00,3.199e+00,3.213e+00,3.209e+00,3.239e+00,
     &  3.266e+00,3.236e+00,3.117e+00,2.909e+00,2.695e+00,2.532e+00,
     &  2.396e+00,2.295e+00,2.251e+00,2.224e+00,2.181e+00,2.130e+00,
     &  2.148e+00,2.326e+00,2.586e+00,2.723e+00,2.649e+00,2.451e+00,
     &  2.244e+00,2.085e+00,1.989e+00,1.984e+00,2.082e+00,2.223e+00,
     &  2.325e+00,2.355e+00,2.348e+00,2.361e+00,2.430e+00,2.536e+00/
      data (sun2b(i),i=2581,2640)/
     &  2.613e+00,2.622e+00,2.607e+00,2.620e+00,2.646e+00,2.643e+00,
     &  2.582e+00,2.468e+00,2.332e+00,2.240e+00,2.264e+00,2.397e+00,
     &  2.552e+00,2.689e+00,2.851e+00,3.010e+00,3.068e+00,3.006e+00,
     &  2.911e+00,2.861e+00,2.852e+00,2.837e+00,2.789e+00,2.729e+00,
     &  2.714e+00,2.773e+00,2.871e+00,2.954e+00,2.995e+00,2.991e+00,
     &  2.940e+00,2.855e+00,2.773e+00,2.730e+00,2.737e+00,2.765e+00,
     &  2.770e+00,2.737e+00,2.683e+00,2.622e+00,2.566e+00,2.534e+00,
     &  2.535e+00,2.568e+00,2.612e+00,2.631e+00,2.602e+00,2.543e+00,
     &  2.484e+00,2.455e+00,2.468e+00,2.510e+00,2.545e+00,2.553e+00,
     &  2.556e+00,2.589e+00,2.662e+00,2.755e+00,2.847e+00,2.929e+00/
      data (sun2b(i),i=2641,2700)/
     &  2.996e+00,3.017e+00,2.977e+00,2.899e+00,2.800e+00,2.693e+00,
     &  2.617e+00,2.593e+00,2.581e+00,2.528e+00,2.428e+00,2.336e+00,
     &  2.316e+00,2.385e+00,2.487e+00,2.546e+00,2.577e+00,2.681e+00,
     &  2.972e+00,3.576e+00,4.401e+00,5.004e+00,5.030e+00,4.514e+00,
     &  3.707e+00,2.906e+00,2.360e+00,2.142e+00,2.112e+00,2.087e+00,
     &  2.030e+00,2.004e+00,2.036e+00,2.096e+00,2.152e+00,2.188e+00,
     &  2.195e+00,2.186e+00,2.190e+00,2.217e+00,2.257e+00,2.289e+00,
     &  2.278e+00,2.215e+00,2.164e+00,2.234e+00,2.473e+00,2.812e+00,
     &  3.129e+00,3.298e+00,3.221e+00,2.886e+00,2.447e+00,2.144e+00,
     &  2.073e+00,2.139e+00,2.217e+00,2.258e+00,2.265e+00,2.248e+00/
      data (sun2b(i),i=2701,2760)/
     &  2.210e+00,2.166e+00,2.136e+00,2.132e+00,2.144e+00,2.146e+00,
     &  2.140e+00,2.146e+00,2.159e+00,2.158e+00,2.148e+00,2.134e+00,
     &  2.110e+00,2.075e+00,2.040e+00,2.012e+00,1.993e+00,1.980e+00,
     &  1.970e+00,1.961e+00,1.954e+00,1.945e+00,1.931e+00,1.906e+00,
     &  1.873e+00,1.847e+00,1.847e+00,1.869e+00,1.884e+00,1.863e+00,
     &  1.814e+00,1.762e+00,1.719e+00,1.685e+00,1.666e+00,1.671e+00,
     &  1.704e+00,1.752e+00,1.812e+00,1.901e+00,2.012e+00,2.098e+00,
     &  2.138e+00,2.148e+00,2.135e+00,2.109e+00,2.095e+00,2.106e+00,
     &  2.109e+00,2.069e+00,2.000e+00,1.946e+00,1.919e+00,1.909e+00,
     &  1.902e+00,1.898e+00,1.905e+00,1.930e+00,1.959e+00,1.964e+00/
      data (sun2b(i),i=2761,2820)/
     &  1.936e+00,1.902e+00,1.889e+00,1.894e+00,1.902e+00,1.904e+00,
     &  1.898e+00,1.875e+00,1.829e+00,1.772e+00,1.730e+00,1.733e+00,
     &  1.790e+00,1.878e+00,1.955e+00,1.991e+00,1.981e+00,1.943e+00,
     &  1.917e+00,1.922e+00,1.920e+00,1.872e+00,1.783e+00,1.696e+00,
     &  1.639e+00,1.608e+00,1.592e+00,1.582e+00,1.558e+00,1.514e+00,
     &  1.462e+00,1.424e+00,1.425e+00,1.483e+00,1.597e+00,1.741e+00,
     &  1.877e+00,1.978e+00,2.045e+00,2.087e+00,2.091e+00,2.038e+00,
     &  1.957e+00,1.900e+00,1.882e+00,1.866e+00,1.819e+00,1.754e+00,
     &  1.687e+00,1.622e+00,1.573e+00,1.545e+00,1.529e+00,1.531e+00,
     &  1.570e+00,1.641e+00,1.719e+00,1.768e+00,1.755e+00,1.662e+00/
      data (sun2b(i),i=2821,2880)/
     &  1.512e+00,1.359e+00,1.260e+00,1.235e+00,1.261e+00,1.309e+00,
     &  1.359e+00,1.388e+00,1.380e+00,1.348e+00,1.322e+00,1.316e+00,
     &  1.344e+00,1.414e+00,1.495e+00,1.544e+00,1.552e+00,1.528e+00,
     &  1.490e+00,1.461e+00,1.446e+00,1.440e+00,1.440e+00,1.454e+00,
     &  1.476e+00,1.478e+00,1.451e+00,1.423e+00,1.418e+00,1.422e+00,
     &  1.412e+00,1.391e+00,1.378e+00,1.381e+00,1.393e+00,1.409e+00,
     &  1.427e+00,1.442e+00,1.441e+00,1.419e+00,1.386e+00,1.355e+00,
     &  1.326e+00,1.298e+00,1.277e+00,1.273e+00,1.286e+00,1.300e+00,
     &  1.306e+00,1.299e+00,1.282e+00,1.259e+00,1.229e+00,1.187e+00,
     &  1.143e+00,1.118e+00,1.114e+00,1.112e+00,1.101e+00,1.080e+00/
      data (sun2b(i),i=2881,2910)/
     &  1.052e+00,1.028e+00,1.025e+00,1.058e+00,1.120e+00,1.179e+00,
     &  1.206e+00,1.201e+00,1.194e+00,1.200e+00,1.204e+00,1.182e+00,
     &  1.149e+00,1.140e+00,1.154e+00,1.163e+00,1.162e+00,1.159e+00,
     &  1.156e+00,1.157e+00,1.156e+00,1.140e+00,1.104e+00,1.063e+00,
     &  1.034e+00,1.022e+00,1.020e+00,1.019e+00,1.017e+00,1.016e+00/

c sun2a   lowtran7        0-28780 cm-1      20 cm-1           1440
c sun2b   lowtran7    28400-57490 cm-1      10 cm-1           2910

      wnmin=28400.
      wnmax=57490.
      nn1=2910
      do i=1,nn1
        sun(i)=sun2b(2910-i+1)
        wn=wnmin+(wnmax-wnmin)*real(nn1-i)/(nn1-1)
        wlsun(i)=10000./wn
      enddo
      wnmin=0.
      wnmax=28780.
      nn2=1440
      do i=1,nn2
        sun(i+nn1)=sun2a(nn2-i+1)
        wn=wnmin+(wnmax-wnmin)*real(nn2-i)/(nn2-1)
        wlsun(i+nn1)=10000./max(wn,one)
      enddo
      nns=nn1+nn2
      return
      end
c=======================================================================
      subroutine sunmod(wlsun,sun,nns)
      use params, only: kr
      implicit none
      integer :: i, nns
      real(kr) :: wn, wnmax, wnmin, wlsun(*), sun(*), sun3(2494)

c block    source           range           resolution      n_elements
c -----    ------           -----           ----------      ----------
c sun3    modtran3      100-49960 cm-1      20 cm-1           2494

      data (sun3(i),i=1,60)/
     &  2.616e-05,5.360e-05,9.862e-05,1.675e-04,2.676e-04,4.072e-04,
     &  5.959e-04,8.435e-04,1.161e-03,1.561e-03,2.056e-03,2.660e-03,
     &  3.387e-03,4.254e-03,5.277e-03,6.473e-03,7.861e-03,9.461e-03,
     &  1.129e-02,1.338e-02,1.574e-02,1.840e-02,2.138e-02,2.471e-02,
     &  2.840e-02,3.250e-02,3.702e-02,4.200e-02,4.744e-02,5.341e-02,
     &  5.993e-02,6.699e-02,7.469e-02,8.299e-02,9.200e-02,1.017e-01,
     &  1.121e-01,1.234e-01,1.354e-01,1.482e-01,1.620e-01,1.767e-01,
     &  1.924e-01,2.090e-01,2.268e-01,2.455e-01,2.653e-01,2.864e-01,
     &  3.087e-01,3.323e-01,3.571e-01,3.832e-01,4.111e-01,4.403e-01,
     &  4.709e-01,5.029e-01,5.366e-01,5.721e-01,6.095e-01,6.483e-01/
      data (sun3(i),i=61,120)/
     &  6.888e-01,7.313e-01,7.738e-01,8.219e-01,8.701e-01,9.209e-01,
     &  9.728e-01,1.028e+00,1.085e+00,1.143e+00,1.205e+00,1.269e+00,
     &  1.334e+00,1.403e+00,1.473e+00,1.545e+00,1.623e+00,1.697e+00,
     &  1.780e+00,1.861e+00,1.942e+00,2.036e+00,2.125e+00,2.218e+00,
     &  2.311e+00,2.410e+00,2.511e+00,2.613e+00,2.724e+00,2.833e+00,
     &  2.950e+00,3.064e+00,3.188e+00,3.305e+00,3.446e+00,3.570e+00,
     &  3.708e+00,3.854e+00,3.995e+00,4.136e+00,4.285e+00,4.433e+00,
     &  4.562e+00,4.727e+00,4.871e+00,5.046e+00,5.262e+00,5.429e+00,
     &  5.615e+00,5.912e+00,6.135e+00,6.392e+00,6.773e+00,6.996e+00,
     &  7.194e+00,7.452e+00,7.702e+00,7.942e+00,8.132e+00,8.433e+00/
      data (sun3(i),i=121,180)/
     &  8.682e+00,8.981e+00,9.277e+00,9.561e+00,9.832e+00,1.015e+01,
     &  1.044e+01,1.071e+01,1.099e+01,1.126e+01,1.160e+01,1.202e+01,
     &  1.238e+01,1.279e+01,1.313e+01,1.347e+01,1.385e+01,1.425e+01,
     &  1.464e+01,1.502e+01,1.543e+01,1.579e+01,1.616e+01,1.665e+01,
     &  1.709e+01,1.754e+01,1.788e+01,1.826e+01,1.892e+01,1.939e+01,
     &  1.989e+01,2.037e+01,2.090e+01,2.138e+01,2.187e+01,2.239e+01,
     &  2.291e+01,2.350e+01,2.405e+01,2.451e+01,2.505e+01,2.579e+01,
     &  2.634e+01,2.700e+01,2.760e+01,2.817e+01,2.878e+01,2.947e+01,
     &  3.010e+01,3.050e+01,3.142e+01,3.210e+01,3.282e+01,3.350e+01,
     &  3.423e+01,3.481e+01,3.531e+01,3.637e+01,3.704e+01,3.783e+01/
      data (sun3(i),i=181,240)/
     &  3.860e+01,3.937e+01,3.996e+01,4.046e+01,4.133e+01,4.156e+01,
     &  4.284e+01,4.417e+01,4.461e+01,4.582e+01,4.649e+01,4.738e+01,
     &  4.823e+01,4.949e+01,4.999e+01,5.126e+01,5.166e+01,5.270e+01,
     &  5.403e+01,5.429e+01,5.603e+01,5.703e+01,5.750e+01,5.929e+01,
     &  6.004e+01,6.080e+01,6.217e+01,6.299e+01,6.477e+01,6.565e+01,
     &  6.636e+01,6.828e+01,6.955e+01,7.028e+01,7.174e+01,7.342e+01,
     &  7.435e+01,7.547e+01,7.690e+01,7.851e+01,7.960e+01,8.073e+01,
     &  8.188e+01,8.337e+01,8.455e+01,8.548e+01,8.388e+01,8.903e+01,
     &  9.049e+01,9.183e+01,9.305e+01,9.434e+01,9.624e+01,9.754e+01,
     &  9.832e+01,1.002e+02,1.022e+02,1.031e+02,1.050e+02,1.068e+02/
      data (sun3(i),i=241,300)/
     &  1.079e+02,1.091e+02,1.110e+02,1.137e+02,1.151e+02,1.161e+02,
     &  1.179e+02,1.188e+02,1.198e+02,1.230e+02,1.240e+02,1.251e+02,
     &  1.209e+02,1.291e+02,1.299e+02,1.335e+02,1.357e+02,1.372e+02,
     &  1.386e+02,1.373e+02,1.416e+02,1.409e+02,1.381e+02,1.469e+02,
     &  1.496e+02,1.534e+02,1.525e+02,1.566e+02,1.586e+02,1.597e+02,
     &  1.551e+02,1.651e+02,1.670e+02,1.699e+02,1.722e+02,1.742e+02,
     &  1.752e+02,1.789e+02,1.810e+02,1.821e+02,1.851e+02,1.847e+02,
     &  1.874e+02,1.813e+02,1.922e+02,1.946e+02,1.992e+02,2.003e+02,
     &  2.034e+02,2.056e+02,2.103e+02,2.102e+02,2.065e+02,2.106e+02,
     &  2.174e+02,2.200e+02,2.246e+02,2.266e+02,2.268e+02,2.266e+02/
      data (sun3(i),i=301,360)/
     &  2.236e+02,2.355e+02,2.401e+02,2.393e+02,2.363e+02,2.408e+02,
     &  2.434e+02,2.464e+02,2.490e+02,2.479e+02,2.454e+02,2.538e+02,
     &  2.494e+02,2.602e+02,2.603e+02,2.664e+02,2.656e+02,2.683e+02,
     &  2.712e+02,2.753e+02,2.748e+02,2.787e+02,2.791e+02,2.815e+02,
     &  2.844e+02,2.852e+02,2.877e+02,2.798e+02,2.883e+02,2.928e+02,
     &  2.983e+02,2.898e+02,2.998e+02,3.005e+02,3.003e+02,3.058e+02,
     &  3.080e+02,3.095e+02,3.128e+02,3.122e+02,3.177e+02,3.178e+02,
     &  3.161e+02,3.260e+02,3.263e+02,3.245e+02,3.261e+02,3.317e+02,
     &  3.368e+02,3.350e+02,3.419e+02,3.424e+02,3.439e+02,3.496e+02,
     &  3.515e+02,3.534e+02,3.553e+02,3.570e+02,3.580e+02,3.626e+02/
      data (sun3(i),i=361,420)/
     &  3.608e+02,3.661e+02,3.684e+02,3.691e+02,3.716e+02,3.755e+02,
     &  3.792e+02,3.820e+02,3.808e+02,3.857e+02,3.868e+02,3.864e+02,
     &  3.926e+02,3.953e+02,3.973e+02,3.951e+02,3.974e+02,4.050e+02,
     &  4.067e+02,4.071e+02,4.118e+02,4.145e+02,4.156e+02,4.172e+02,
     &  4.172e+02,3.925e+02,4.252e+02,4.295e+02,4.324e+02,4.321e+02,
     &  4.345e+02,4.379e+02,4.412e+02,4.429e+02,4.450e+02,4.501e+02,
     &  4.513e+02,4.513e+02,4.551e+02,4.610e+02,4.613e+02,4.633e+02,
     &  4.643e+02,4.702e+02,4.716e+02,4.748e+02,4.787e+02,4.798e+02,
     &  4.759e+02,4.728e+02,4.782e+02,4.883e+02,4.758e+02,4.890e+02,
     &  4.985e+02,4.974e+02,4.925e+02,5.032e+02,4.967e+02,5.067e+02/
      data (sun3(i),i=421,480)/
     &  5.043e+02,5.155e+02,5.195e+02,5.145e+02,5.168e+02,5.220e+02,
     &  5.154e+02,5.328e+02,5.352e+02,5.348e+02,5.380e+02,5.442e+02,
     &  5.393e+02,5.367e+02,5.405e+02,5.500e+02,5.499e+02,5.546e+02,
     &  5.535e+02,5.538e+02,5.655e+02,5.632e+02,5.667e+02,5.714e+02,
     &  5.733e+02,5.755e+02,5.800e+02,5.817e+02,5.851e+02,5.802e+02,
     &  5.812e+02,5.811e+02,5.676e+02,6.020e+02,6.002e+02,5.986e+02,
     &  6.136e+02,5.933e+02,6.165e+02,6.227e+02,6.190e+02,6.219e+02,
     &  6.277e+02,6.196e+02,6.351e+02,6.417e+02,6.398e+02,6.411e+02,
     &  6.543e+02,6.575e+02,6.563e+02,6.618e+02,6.636e+02,6.595e+02,
     &  6.708e+02,6.734e+02,6.752e+02,6.738e+02,6.779e+02,6.821e+02/
      data (sun3(i),i=481,540)/
     &  6.895e+02,6.921e+02,6.972e+02,6.996e+02,6.967e+02,6.977e+02,
     &  7.032e+02,7.055e+02,7.114e+02,7.164e+02,7.207e+02,7.219e+02,
     &  6.953e+02,7.163e+02,7.332e+02,7.349e+02,7.385e+02,7.406e+02,
     &  7.446e+02,7.453e+02,7.400e+02,7.501e+02,7.525e+02,7.556e+02,
     &  7.620e+02,7.627e+02,7.606e+02,7.633e+02,7.630e+02,7.749e+02,
     &  7.766e+02,7.803e+02,7.783e+02,7.758e+02,7.792e+02,7.886e+02,
     &  7.940e+02,7.932e+02,7.923e+02,7.738e+02,8.072e+02,8.093e+02,
     &  8.181e+02,8.155e+02,8.121e+02,8.103e+02,7.890e+02,8.237e+02,
     &  8.339e+02,8.325e+02,8.386e+02,8.418e+02,8.482e+02,8.440e+02,
     &  8.558e+02,8.305e+02,8.363e+02,8.231e+02,8.442e+02,8.718e+02/
      data (sun3(i),i=541,600)/
     &  8.636e+02,8.772e+02,8.843e+02,8.800e+02,8.759e+02,8.658e+02,
     &  8.900e+02,8.899e+02,9.093e+02,8.799e+02,8.652e+02,9.115e+02,
     &  9.090e+02,9.277e+02,9.107e+02,9.087e+02,9.335e+02,9.345e+02,
     &  9.326e+02,8.978e+02,9.425e+02,9.270e+02,9.424e+02,9.147e+02,
     &  9.525e+02,9.536e+02,9.335e+02,9.519e+02,9.660e+02,9.639e+02,
     &  9.387e+02,9.488e+02,8.045e+02,9.736e+02,9.908e+02,9.743e+02,
     &  9.846e+02,9.786e+02,1.002e+03,9.952e+02,7.876e+02,9.779e+02,
     &  9.915e+02,9.268e+02,1.020e+03,1.008e+03,1.027e+03,1.017e+03,
     &  1.015e+03,1.034e+03,1.039e+03,1.019e+03,1.049e+03,1.047e+03,
     &  1.036e+03,1.025e+03,1.055e+03,1.048e+03,1.065e+03,1.075e+03/
      data (sun3(i),i=601,660)/
     &  1.076e+03,1.072e+03,1.060e+03,1.069e+03,1.062e+03,1.060e+03,
     &  1.072e+03,1.097e+03,1.100e+03,1.109e+03,1.112e+03,1.111e+03,
     &  1.103e+03,1.099e+03,1.114e+03,1.134e+03,1.105e+03,1.137e+03,
     &  1.126e+03,1.145e+03,1.128e+03,1.151e+03,1.153e+03,1.148e+03,
     &  1.125e+03,1.125e+03,1.153e+03,1.159e+03,1.173e+03,1.176e+03,
     &  1.181e+03,1.173e+03,1.187e+03,1.173e+03,1.198e+03,1.192e+03,
     &  1.196e+03,1.208e+03,1.199e+03,1.219e+03,1.179e+03,1.217e+03,
     &  1.200e+03,1.203e+03,1.220e+03,1.226e+03,1.235e+03,1.192e+03,
     &  1.207e+03,1.256e+03,1.257e+03,1.220e+03,1.250e+03,1.256e+03,
     &  1.243e+03,1.253e+03,1.261e+03,1.265e+03,1.284e+03,1.256e+03/
      data (sun3(i),i=661,720)/
     &  1.274e+03,1.273e+03,1.257e+03,1.287e+03,1.289e+03,1.291e+03,
     &  1.273e+03,1.294e+03,1.279e+03,1.260e+03,1.270e+03,1.293e+03,
     &  1.283e+03,1.326e+03,1.312e+03,1.315e+03,1.340e+03,1.308e+03,
     &  1.335e+03,1.312e+03,1.330e+03,1.287e+03,1.347e+03,1.343e+03,
     &  1.353e+03,1.356e+03,1.369e+03,1.332e+03,1.359e+03,1.344e+03,
     &  1.315e+03,1.341e+03,1.382e+03,1.353e+03,1.343e+03,1.393e+03,
     &  1.374e+03,1.384e+03,1.387e+03,1.416e+03,1.396e+03,1.413e+03,
     &  1.393e+03,1.416e+03,1.432e+03,1.422e+03,1.421e+03,1.385e+03,
     &  1.418e+03,1.419e+03,1.454e+03,1.427e+03,1.452e+03,1.450e+03,
     &  1.453e+03,1.453e+03,1.462e+03,1.459e+03,1.455e+03,1.480e+03/
      data (sun3(i),i=721,780)/
     &  1.488e+03,1.469e+03,1.492e+03,1.480e+03,1.449e+03,1.489e+03,
     &  1.479e+03,1.500e+03,1.500e+03,1.490e+03,1.504e+03,1.506e+03,
     &  1.508e+03,1.518e+03,1.513e+03,1.521e+03,1.519e+03,1.530e+03,
     &  1.530e+03,1.519e+03,1.526e+03,1.533e+03,1.554e+03,1.554e+03,
     &  1.552e+03,1.523e+03,1.563e+03,1.547e+03,1.550e+03,1.553e+03,
     &  1.583e+03,1.565e+03,1.586e+03,1.526e+03,1.579e+03,1.543e+03,
     &  1.435e+03,1.230e+03,1.547e+03,1.564e+03,1.595e+03,1.605e+03,
     &  1.590e+03,1.618e+03,1.545e+03,1.536e+03,1.614e+03,1.605e+03,
     &  1.573e+03,1.598e+03,1.616e+03,1.598e+03,1.611e+03,1.634e+03,
     &  1.609e+03,1.605e+03,1.614e+03,1.630e+03,1.644e+03,1.675e+03/
      data (sun3(i),i=781,840)/
     &  1.638e+03,1.606e+03,1.647e+03,1.644e+03,1.618e+03,1.687e+03,
     &  1.584e+03,1.655e+03,1.645e+03,1.683e+03,1.703e+03,1.684e+03,
     &  1.690e+03,1.672e+03,1.587e+03,1.655e+03,1.623e+03,1.648e+03,
     &  1.712e+03,1.658e+03,1.734e+03,1.715e+03,1.684e+03,1.729e+03,
     &  1.712e+03,1.645e+03,1.588e+03,1.678e+03,1.695e+03,1.643e+03,
     &  1.676e+03,1.668e+03,1.735e+03,1.739e+03,1.702e+03,1.756e+03,
     &  1.746e+03,1.738e+03,1.802e+03,1.756e+03,1.783e+03,1.802e+03,
     &  1.806e+03,1.809e+03,1.762e+03,1.743e+03,1.758e+03,1.758e+03,
     &  1.782e+03,1.796e+03,1.786e+03,1.799e+03,1.816e+03,1.827e+03,
     &  1.817e+03,1.782e+03,1.828e+03,1.826e+03,1.777e+03,1.840e+03/
      data (sun3(i),i=841,900)/
     &  1.788e+03,1.820e+03,1.834e+03,1.642e+03,1.648e+03,1.816e+03,
     &  1.864e+03,1.845e+03,1.760e+03,1.838e+03,1.855e+03,1.874e+03,
     &  1.862e+03,1.886e+03,1.874e+03,1.862e+03,1.834e+03,1.878e+03,
     &  1.851e+03,1.802e+03,1.848e+03,1.859e+03,1.875e+03,1.891e+03,
     &  1.803e+03,1.876e+03,1.891e+03,1.911e+03,1.933e+03,1.895e+03,
     &  1.808e+03,1.811e+03,1.842e+03,1.877e+03,1.835e+03,1.861e+03,
     &  1.913e+03,1.844e+03,1.695e+03,1.853e+03,1.898e+03,1.868e+03,
     &  1.869e+03,1.936e+03,1.832e+03,1.772e+03,1.966e+03,1.856e+03,
     &  1.834e+03,1.789e+03,1.792e+03,1.881e+03,1.826e+03,1.832e+03,
     &  1.882e+03,1.873e+03,1.928e+03,1.919e+03,1.884e+03,1.836e+03/
      data (sun3(i),i=901,960)/
     &  1.905e+03,1.944e+03,1.852e+03,1.869e+03,1.874e+03,1.870e+03,
     &  1.886e+03,1.883e+03,1.784e+03,1.945e+03,1.906e+03,1.866e+03,
     &  2.008e+03,1.862e+03,1.984e+03,1.871e+03,1.841e+03,1.908e+03,
     &  1.879e+03,1.839e+03,1.742e+03,1.822e+03,1.897e+03,1.957e+03,
     &  1.913e+03,2.002e+03,1.799e+03,1.910e+03,2.024e+03,1.954e+03,
     &  1.870e+03,1.780e+03,1.853e+03,1.694e+03,1.869e+03,1.932e+03,
     &  2.035e+03,2.026e+03,1.871e+03,1.908e+03,2.022e+03,1.882e+03,
     &  1.964e+03,1.777e+03,1.548e+03,1.731e+03,1.895e+03,1.950e+03,
     &  1.917e+03,2.002e+03,1.849e+03,1.749e+03,1.967e+03,1.864e+03,
     &  1.984e+03,1.736e+03,1.954e+03,1.783e+03,1.773e+03,1.450e+03/
      data (sun3(i),i=961,1020)/
     &  1.883e+03,1.551e+03,1.453e+03,1.813e+03,1.994e+03,1.858e+03,
     &  1.854e+03,1.853e+03,1.855e+03,1.940e+03,1.821e+03,1.956e+03,
     &  2.079e+03,1.982e+03,1.940e+03,1.864e+03,2.056e+03,2.043e+03,
     &  1.902e+03,1.846e+03,1.950e+03,1.976e+03,2.007e+03,2.095e+03,
     &  2.067e+03,1.899e+03,1.970e+03,1.782e+03,1.980e+03,1.921e+03,
     &  1.883e+03,1.896e+03,1.826e+03,1.933e+03,1.817e+03,1.904e+03,
     &  1.971e+03,2.043e+03,1.701e+03,1.918e+03,2.093e+03,1.947e+03,
     &  2.036e+03,1.940e+03,2.122e+03,2.134e+03,2.066e+03,1.939e+03,
     &  1.960e+03,2.112e+03,1.978e+03,1.692e+03,2.074e+03,1.826e+03,
     &  2.060e+03,2.111e+03,2.142e+03,1.782e+03,1.950e+03,2.053e+03/
      data (sun3(i),i=1021,1080)/
     &  1.944e+03,1.821e+03,1.932e+03,1.422e+03,1.792e+03,1.975e+03,
     &  2.007e+03,2.026e+03,1.999e+03,2.095e+03,2.090e+03,2.096e+03,
     &  2.156e+03,2.103e+03,2.064e+03,2.069e+03,2.068e+03,2.139e+03,
     &  2.077e+03,1.972e+03,2.155e+03,2.180e+03,2.155e+03,2.122e+03,
     &  2.006e+03,2.114e+03,2.065e+03,2.182e+03,2.096e+03,2.127e+03,
     &  2.013e+03,2.049e+03,1.991e+03,2.172e+03,2.127e+03,2.039e+03,
     &  2.109e+03,1.982e+03,1.994e+03,2.058e+03,2.160e+03,2.077e+03,
     &  2.084e+03,2.027e+03,2.220e+03,2.086e+03,1.889e+03,2.053e+03,
     &  2.162e+03,2.042e+03,1.953e+03,2.039e+03,2.161e+03,1.941e+03,
     &  2.095e+03,1.999e+03,2.096e+03,1.983e+03,2.085e+03,2.082e+03/
      data (sun3(i),i=1081,1140)/
     &  2.112e+03,2.002e+03,2.077e+03,1.993e+03,2.120e+03,2.017e+03,
     &  1.869e+03,2.139e+03,2.088e+03,2.201e+03,2.035e+03,2.193e+03,
     &  1.990e+03,2.160e+03,1.951e+03,2.020e+03,2.062e+03,1.927e+03,
     &  1.922e+03,1.784e+03,2.011e+03,2.039e+03,2.137e+03,2.166e+03,
     &  2.242e+03,2.191e+03,2.135e+03,1.894e+03,1.964e+03,2.069e+03,
     &  1.913e+03,2.091e+03,2.059e+03,1.914e+03,2.001e+03,1.814e+03,
     &  1.783e+03,1.703e+03,1.915e+03,2.018e+03,1.790e+03,2.047e+03,
     &  1.767e+03,1.838e+03,1.930e+03,1.858e+03,2.124e+03,1.644e+03,
     &  2.019e+03,1.787e+03,1.647e+03,1.700e+03,1.783e+03,1.752e+03,
     &  1.634e+03,1.187e+03,1.864e+03,1.751e+03,1.724e+03,1.737e+03/
      data (sun3(i),i=1141,1200)/
     &  2.002e+03,1.917e+03,1.827e+03,1.669e+03,1.568e+03,1.834e+03,
     &  1.518e+03,1.219e+03,1.926e+03,1.877e+03,1.690e+03,1.182e+03,
     &  1.956e+03,1.933e+03,1.284e+03,8.052e+02,1.086e+03,9.452e+02,
     &  1.145e+03,1.526e+03,1.192e+03,1.498e+03,1.667e+03,1.576e+03,
     &  1.515e+03,1.254e+03,1.602e+03,1.841e+03,1.509e+03,1.805e+03,
     &  1.577e+03,1.566e+03,1.627e+03,1.814e+03,1.623e+03,1.554e+03,
     &  1.743e+03,1.616e+03,1.227e+03,1.923e+03,1.812e+03,1.572e+03,
     &  1.848e+03,1.978e+03,1.688e+03,1.446e+03,1.410e+03,1.780e+03,
     &  1.822e+03,1.558e+03,1.759e+03,1.856e+03,1.663e+03,1.530e+03,
     &  1.777e+03,1.944e+03,1.931e+03,1.905e+03,1.656e+03,1.954e+03/
      data (sun3(i),i=1201,1260)/
     &  1.836e+03,1.583e+03,1.752e+03,1.960e+03,1.672e+03,1.668e+03,
     &  1.729e+03,1.854e+03,1.915e+03,1.920e+03,1.893e+03,1.877e+03,
     &  1.851e+03,1.583e+03,1.310e+03,1.619e+03,1.943e+03,1.964e+03,
     &  1.933e+03,1.559e+03,1.932e+03,1.421e+03,1.840e+03,1.441e+03,
     &  1.633e+03,1.266e+03,1.996e+03,1.645e+03,1.640e+03,1.914e+03,
     &  1.521e+03,1.420e+03,1.697e+03,1.998e+03,1.573e+03,1.522e+03,
     &  1.889e+03,1.730e+03,1.811e+03,1.671e+03,1.828e+03,1.749e+03,
     &  1.840e+03,1.497e+03,1.957e+03,1.658e+03,1.594e+03,1.896e+03,
     &  1.756e+03,1.631e+03,1.444e+03,1.691e+03,1.382e+03,8.553e+02,
     &  4.010e+02,3.110e+02,8.529e+02,1.101e+03,1.349e+03,1.541e+03/
      data (sun3(i),i=1261,1320)/
     &  1.358e+03,1.272e+03,1.161e+03,1.074e+03,8.786e+02,3.822e+02,
     &  2.173e+02,5.268e+02,9.261e+02,1.253e+03,1.066e+03,1.399e+03,
     &  1.506e+03,1.544e+03,1.582e+03,1.102e+03,1.306e+03,1.203e+03,
     &  1.403e+03,1.399e+03,9.520e+02,7.840e+02,1.046e+03,9.084e+02,
     &  7.732e+02,1.194e+03,6.025e+02,9.981e+02,1.144e+03,1.157e+03,
     &  4.782e+02,8.259e+02,1.030e+03,9.441e+02,1.151e+03,1.107e+03,
     &  8.535e+02,5.215e+02,7.978e+02,4.610e+02,8.221e+02,6.668e+02,
     &  7.040e+02,7.287e+02,1.121e+03,8.432e+02,1.049e+03,1.427e+03,
     &  1.115e+03,1.509e+03,1.312e+03,6.955e+02,1.074e+03,1.133e+03,
     &  1.021e+03,1.020e+03,1.725e+03,1.730e+03,1.467e+03,1.406e+03/
      data (sun3(i),i=1321,1380)/
     &  1.436e+03,1.132e+03,9.268e+02,1.102e+03,1.193e+03,7.077e+02,
     &  1.646e+03,1.241e+03,5.566e+02,6.371e+02,7.420e+02,1.420e+03,
     &  9.540e+02,4.338e+02,8.207e+02,1.289e+03,1.063e+03,1.279e+03,
     &  7.421e+02,7.780e+02,1.525e+03,1.708e+03,1.203e+03,8.153e+02,
     &  9.097e+02,1.317e+03,1.352e+03,1.334e+03,1.235e+03,1.418e+03,
     &  1.152e+03,9.400e+02,1.236e+03,1.122e+03,1.159e+03,1.363e+03,
     &  1.387e+03,1.103e+03,1.227e+03,1.348e+03,1.342e+03,1.454e+03,
     &  1.448e+03,1.483e+03,1.161e+03,7.835e+02,9.843e+02,9.535e+02,
     &  1.307e+03,1.182e+03,9.650e+02,8.898e+02,8.146e+02,1.513e+03,
     &  1.224e+03,1.119e+03,5.361e+02,1.182e+03,1.276e+03,1.034e+03/
      data (sun3(i),i=1381,1440)/
     &  6.753e+02,1.069e+03,9.200e+02,1.243e+03,1.355e+03,1.119e+03,
     &  1.181e+03,1.125e+03,7.264e+02,3.851e+02,7.448e+02,5.304e+02,
     &  8.958e+02,9.643e+02,8.306e+02,5.733e+02,9.510e+02,5.896e+02,
     &  1.246e+03,1.098e+03,1.125e+03,1.083e+03,1.063e+03,1.326e+03,
     &  1.075e+03,1.334e+03,1.190e+03,1.331e+03,9.672e+02,1.218e+03,
     &  1.075e+03,1.229e+03,9.998e+02,7.448e+02,9.717e+02,1.197e+03,
     &  1.305e+03,9.595e+02,8.832e+02,1.129e+03,1.162e+03,1.184e+03,
     &  1.342e+03,1.263e+03,9.291e+02,9.784e+02,9.976e+02,8.360e+02,
     &  1.326e+03,1.304e+03,1.144e+03,1.307e+03,1.108e+03,5.392e+02,
     &  1.027e+03,1.248e+03,1.085e+03,8.488e+02,1.214e+03,9.699e+02/
      data (sun3(i),i=1441,1500)/
     &  9.096e+02,1.369e+03,1.118e+03,9.123e+02,1.176e+03,1.046e+03,
     &  8.571e+02,6.842e+02,5.575e+02,9.405e+02,1.046e+03,8.434e+02,
     &  9.864e+02,9.891e+02,7.194e+02,6.664e+02,9.103e+02,1.127e+03,
     &  7.969e+02,6.468e+02,8.385e+02,8.230e+02,8.359e+02,9.173e+02,
     &  8.132e+02,1.177e+03,1.016e+03,9.794e+02,5.923e+02,8.344e+02,
     &  9.039e+02,9.497e+02,8.562e+02,8.964e+02,6.116e+02,1.092e+03,
     &  9.351e+02,6.175e+02,5.049e+02,6.799e+02,8.828e+02,9.557e+02,
     &  4.216e+02,3.581e+02,1.047e+03,1.029e+03,1.087e+03,8.082e+02,
     &  9.585e+02,1.027e+03,1.031e+03,9.652e+02,9.996e+02,7.608e+02,
     &  1.049e+03,1.053e+03,9.285e+02,1.007e+03,1.213e+03,8.617e+02/
      data (sun3(i),i=1501,1560)/
     &  9.966e+02,1.115e+03,1.092e+03,7.323e+02,9.399e+02,9.989e+02,
     &  9.204e+02,7.284e+02,1.037e+03,1.177e+03,1.178e+03,9.469e+02,
     &  1.137e+03,1.201e+03,1.149e+03,9.677e+02,8.670e+02,9.166e+02,
     &  9.096e+02,9.328e+02,9.598e+02,1.088e+03,1.204e+03,9.403e+02,
     &  1.045e+03,1.137e+03,1.043e+03,1.217e+03,9.163e+02,1.141e+03,
     &  1.146e+03,8.977e+02,8.804e+02,8.169e+02,6.701e+02,1.193e+03,
     &  7.830e+02,9.112e+02,5.866e+02,7.196e+02,5.980e+02,7.290e+02,
     &  7.352e+02,5.010e+02,4.252e+02,7.595e+02,6.654e+02,7.978e+02,
     &  8.962e+02,7.806e+02,5.683e+02,6.081e+02,5.947e+02,9.489e+02,
     &  9.340e+02,1.035e+03,7.730e+02,6.878e+02,1.069e+03,6.922e+02/
      data (sun3(i),i=1561,1620)/
     &  6.132e+02,4.768e+02,1.128e+03,8.246e+02,6.237e+02,5.390e+02,
     &  5.126e+02,5.142e+02,1.013e+03,1.022e+03,9.397e+02,1.115e+03,
     &  7.794e+02,6.980e+02,5.552e+02,8.115e+02,4.396e+02,4.824e+02,
     &  4.740e+02,8.546e+02,9.784e+02,8.321e+02,9.878e+02,6.204e+02,
     &  6.790e+02,3.038e+02,9.230e+02,8.131e+02,1.036e+03,8.095e+02,
     &  6.136e+02,7.761e+02,7.880e+02,8.014e+02,7.458e+02,8.269e+02,
     &  6.337e+02,6.414e+02,6.504e+02,8.785e+02,8.504e+02,1.136e+03,
     &  9.529e+02,1.086e+03,9.323e+02,7.545e+02,7.780e+02,2.312e+02,
     &  5.092e+02,5.440e+02,7.925e+02,3.707e+02,5.603e+02,7.485e+02,
     &  5.859e+02,9.702e+02,7.035e+02,5.693e+02,9.062e+02,6.349e+02/
      data (sun3(i),i=1621,1680)/
     &  6.152e+02,8.680e+02,6.633e+02,8.073e+02,6.675e+02,4.866e+02,
     &  5.959e+02,7.526e+02,7.618e+02,3.963e+02,4.099e+02,7.378e+02,
     &  7.068e+02,8.668e+02,7.861e+02,4.051e+02,8.343e+02,7.763e+02,
     &  4.941e+02,6.495e+02,5.904e+02,5.761e+02,7.947e+02,8.966e+02,
     &  6.009e+02,6.533e+02,8.680e+02,3.771e+02,6.289e+02,4.458e+02,
     &  2.255e+02,3.380e+02,5.338e+02,6.043e+02,5.928e+02,5.851e+02,
     &  4.306e+02,3.351e+02,5.880e+02,4.450e+02,3.235e+02,2.487e+02,
     &  5.846e+02,6.711e+02,3.432e+02,5.508e+02,6.835e+02,6.591e+02,
     &  4.658e+02,4.033e+02,2.514e+02,4.175e+02,5.239e+02,6.458e+02,
     &  8.497e+02,6.651e+02,3.887e+02,5.942e+02,3.061e+02,5.717e+02/
      data (sun3(i),i=1681,1740)/
     &  2.725e+02,4.818e+02,7.439e+02,6.226e+02,6.662e+02,5.715e+02,
     &  6.673e+02,3.917e+02,7.972e+02,6.082e+02,3.719e+02,4.311e+02,
     &  7.118e+02,5.424e+02,5.332e+02,6.237e+02,4.479e+02,3.117e+02,
     &  7.760e+02,5.962e+02,7.501e+02,6.620e+02,4.200e+02,4.756e+02,
     &  5.516e+02,5.493e+02,6.381e+02,6.239e+02,5.589e+02,6.256e+02,
     &  6.069e+02,7.299e+02,6.370e+02,6.039e+02,6.705e+02,6.279e+02,
     &  7.775e+02,6.866e+02,7.114e+02,6.659e+02,5.818e+02,6.354e+02,
     &  4.635e+02,4.454e+02,4.498e+02,4.285e+02,4.388e+02,4.641e+02,
     &  4.066e+02,2.264e+02,1.395e+02,2.680e+02,4.347e+02,3.362e+02,
     &  4.051e+02,4.642e+02,5.097e+02,3.298e+02,3.507e+02,4.057e+02/
      data (sun3(i),i=1741,1800)/
     &  4.052e+02,3.658e+02,3.675e+02,3.581e+02,2.417e+02,1.870e+02,
     &  1.356e+02,6.816e+01,5.988e+01,1.213e+02,1.771e+02,2.441e+02,
     &  3.429e+02,2.237e+02,3.777e+02,3.196e+02,3.687e+02,3.892e+02,
     &  2.577e+02,4.301e+02,3.660e+02,3.880e+02,3.655e+02,3.284e+02,
     &  2.890e+02,3.682e+02,3.186e+02,3.097e+02,3.144e+02,2.747e+02,
     &  2.556e+02,2.859e+02,1.762e+02,1.761e+02,1.817e+02,1.524e+02,
     &  1.149e+02,8.834e+01,7.683e+01,7.632e+01,7.916e+01,7.886e+01,
     &  5.946e+01,1.126e+02,5.685e+01,8.710e+01,1.066e+02,1.352e+02,
     &  1.435e+02,1.906e+02,1.908e+02,1.672e+02,1.839e+02,1.708e+02,
     &  1.688e+02,2.351e+02,2.706e+02,3.222e+02,2.620e+02,3.370e+02/
      data (sun3(i),i=1801,1860)/
     &  2.246e+02,2.711e+02,2.740e+02,3.223e+02,2.464e+02,2.250e+02,
     &  2.712e+02,3.674e+02,1.455e+02,1.025e+02,1.822e+02,2.591e+02,
     &  1.733e+02,7.562e+01,1.675e+02,8.840e+01,1.955e+02,9.706e+01,
     &  1.313e+02,1.521e+02,1.425e+02,1.314e+02,2.077e+02,2.148e+02,
     &  2.221e+02,2.609e+02,3.694e+02,2.561e+02,1.989e+02,2.048e+02,
     &  1.751e+02,2.295e+02,1.492e+02,1.115e+02,1.910e+02,2.912e+02,
     &  2.025e+02,2.581e+02,3.009e+02,2.744e+02,2.616e+02,3.026e+02,
     &  2.379e+02,3.989e+02,2.989e+02,2.512e+02,3.752e+02,2.299e+02,
     &  2.515e+02,2.060e+02,2.521e+02,2.668e+02,2.880e+02,2.163e+02,
     &  2.388e+02,2.681e+02,3.094e+02,2.222e+02,2.844e+02,2.888e+02/
      data (sun3(i),i=1861,1920)/
     &  2.518e+02,1.974e+02,2.464e+02,3.416e+02,3.662e+02,1.890e+02,
     &  3.015e+02,2.317e+02,2.568e+02,2.766e+02,2.468e+02,2.684e+02,
     &  2.399e+02,2.336e+02,3.133e+02,2.578e+02,3.153e+02,2.888e+02,
     &  3.633e+02,2.550e+02,2.250e+02,2.327e+02,1.786e+02,2.246e+02,
     &  2.416e+02,2.701e+02,3.772e+02,2.879e+02,2.425e+02,2.990e+02,
     &  2.995e+02,2.105e+02,1.778e+02,1.457e+02,7.720e+01,6.322e+01,
     &  1.064e+02,7.888e+01,1.098e+02,9.072e+01,1.461e+02,1.476e+02,
     &  1.062e+02,1.460e+02,8.945e+01,9.599e+01,1.301e+02,7.194e+01,
     &  7.451e+01,5.351e+01,1.079e+02,1.027e+02,6.458e+01,6.250e+01,
     &  1.209e+02,1.227e+02,1.337e+02,8.561e+01,3.841e+01,5.427e+01/
      data (sun3(i),i=1921,1980)/
     &  1.080e+02,1.280e+02,7.147e+01,1.374e+02,9.922e+01,1.996e+02,
     &  1.352e+02,1.245e+02,5.456e+01,1.015e+02,1.376e+02,1.946e+02,
     &  1.687e+02,1.519e+02,1.291e+02,7.997e+01,8.464e+01,1.539e+02,
     &  1.388e+02,1.282e+02,1.479e+02,1.222e+02,1.136e+02,1.363e+02,
     &  9.164e+01,6.305e+01,6.575e+01,9.692e+01,1.085e+02,9.025e+01,
     &  7.116e+01,7.687e+01,1.037e+02,6.924e+01,9.677e+01,6.201e+01,
     &  4.701e+01,6.277e+01,4.705e+01,6.230e+01,6.935e+01,5.770e+01,
     &  5.294e+01,6.882e+01,5.187e+01,5.422e+01,4.341e+01,5.599e+01,
     &  5.054e+01,6.250e+01,6.156e+01,3.641e+01,3.311e+01,4.122e+01,
     &  4.713e+01,3.622e+01,3.476e+01,4.600e+01,5.285e+01,3.377e+01/
      data (sun3(i),i=1981,2040)/
     &  4.219e+01,3.859e+01,4.106e+01,3.686e+01,5.796e+01,5.901e+01,
     &  5.489e+01,7.432e+01,6.193e+01,3.488e+01,5.671e+01,8.196e+01,
     &  6.374e+01,5.236e+01,5.935e+01,7.155e+01,7.274e+01,6.400e+01,
     &  7.918e+01,6.098e+01,5.309e+01,3.760e+01,2.594e+01,3.493e+01,
     &  3.272e+01,5.058e+01,5.836e+01,3.784e+01,2.624e+01,5.231e+01,
     &  6.284e+01,4.004e+01,5.164e+01,5.762e+01,7.071e+01,6.756e+01,
     &  4.732e+01,5.337e+01,5.979e+01,5.802e+01,5.745e+01,5.218e+01,
     &  5.144e+01,4.906e+01,4.635e+01,3.534e+01,4.767e+01,5.670e+01,
     &  5.252e+01,5.597e+01,4.499e+01,5.471e+01,4.781e+01,4.579e+01,
     &  4.603e+01,5.262e+01,5.848e+01,4.638e+01,6.333e+01,6.969e+01/
      data (sun3(i),i=2041,2100)/
     &  6.198e+01,6.684e+01,8.475e+01,7.516e+01,5.147e+01,6.639e+01,
     &  7.242e+01,5.904e+01,4.354e+01,7.080e+01,5.808e+01,7.827e+01,
     &  7.732e+01,6.127e+01,7.943e+01,8.996e+01,6.309e+01,5.738e+01,
     &  6.335e+01,8.789e+01,7.135e+01,6.003e+01,6.743e+01,6.353e+01,
     &  5.726e+01,5.364e+01,3.072e+01,4.746e+01,2.748e+01,3.586e+01,
     &  4.882e+01,4.172e+01,2.564e+01,2.906e+01,2.866e+01,4.954e+01,
     &  4.829e+01,5.449e+01,3.629e+01,4.069e+01,4.651e+01,2.370e+01,
     &  2.414e+01,4.737e+01,5.656e+01,4.891e+01,6.514e+01,3.573e+01,
     &  3.176e+01,5.154e+01,5.273e+01,4.244e+01,2.847e+01,2.383e+01,
     &  2.527e+01,4.103e+01,3.772e+01,5.802e+01,6.462e+01,4.367e+01/
      data (sun3(i),i=2101,2160)/
     &  4.600e+01,2.743e+01,6.466e+01,6.665e+01,5.024e+01,4.406e+01,
     &  4.300e+01,3.468e+01,4.870e+01,3.738e+01,5.671e+01,4.959e+01,
     &  4.490e+01,2.841e+01,3.720e+01,6.337e+01,5.811e+01,5.665e+01,
     &  4.692e+01,5.812e+01,6.171e+01,5.185e+01,5.224e+01,3.369e+01,
     &  2.443e+01,3.862e+01,2.890e+01,2.843e+01,2.034e+01,4.568e+01,
     &  4.942e+01,4.139e+01,3.106e+01,2.490e+01,4.091e+01,6.734e+01,
     &  4.449e+01,4.418e+01,3.171e+01,3.274e+01,4.878e+01,6.330e+01,
     &  5.296e+01,3.797e+01,2.738e+01,5.176e+01,8.211e+01,7.092e+01,
     &  3.145e+01,3.651e+01,3.577e+01,5.826e+01,4.686e+01,5.272e+01,
     &  6.938e+01,4.323e+01,3.563e+01,3.304e+01,3.057e+01,4.194e+01/
      data (sun3(i),i=2161,2220)/
     &  5.544e+01,5.358e+01,7.675e+01,7.967e+01,5.509e+01,3.162e+01,
     &  5.621e+01,4.262e+01,3.411e+01,3.832e+01,2.636e+01,2.769e+01,
     &  6.848e+01,4.867e+01,4.911e+01,4.705e+01,4.805e+01,4.111e+01,
     &  3.913e+01,5.750e+01,3.711e+01,3.647e+01,7.327e+01,5.108e+01,
     &  4.743e+01,6.879e+01,5.846e+01,4.600e+01,4.756e+01,4.743e+01,
     &  4.324e+01,3.099e+01,4.147e+01,3.100e+01,3.869e+01,3.353e+01,
     &  3.266e+01,3.007e+01,2.612e+01,3.186e+01,3.489e+01,3.122e+01,
     &  2.619e+01,3.952e+01,4.272e+01,5.374e+01,3.969e+01,3.272e+01,
     &  6.732e+01,4.805e+01,5.042e+01,3.893e+01,6.223e+01,3.210e+01,
     &  6.969e+01,4.771e+01,4.838e+01,5.313e+01,5.035e+01,8.284e+01/
      data (sun3(i),i=2221,2280)/
     &  5.962e+01,7.108e+01,4.481e+01,4.740e+01,5.227e+01,4.850e+01,
     &  5.727e+01,7.848e+01,4.580e+01,6.122e+01,6.765e+01,8.359e+01,
     &  6.884e+01,8.798e+01,7.355e+01,4.449e+01,4.375e+01,5.330e+01,
     &  5.738e+01,3.991e+01,3.993e+01,5.581e+01,4.410e+01,5.360e+01,
     &  4.338e+01,5.720e+01,5.419e+01,5.738e+01,3.930e+01,2.626e+01,
     &  1.729e+01,3.990e+01,4.838e+01,4.419e+01,5.060e+01,2.202e+01,
     &  1.686e+01,3.075e+01,4.232e+01,3.349e+01,4.126e+01,5.598e+01,
     &  5.419e+01,6.771e+01,6.643e+01,5.781e+01,2.666e+01,3.649e+01,
     &  5.739e+01,6.306e+01,5.101e+01,3.100e+01,3.587e+01,6.027e+01,
     &  6.500e+01,4.575e+01,3.671e+01,4.481e+01,3.958e+01,4.664e+01/
      data (sun3(i),i=2281,2340)/
     &  3.204e+01,2.037e+01,6.077e+01,4.514e+01,3.080e+01,6.122e+01,
     &  5.890e+01,5.913e+01,5.740e+01,4.575e+01,2.203e+01,2.816e+01,
     &  3.919e+01,2.405e+01,2.529e+01,3.516e+01,4.197e+01,3.179e+01,
     &  4.408e+01,2.210e+01,3.246e+01,2.951e+01,1.504e+01,1.979e+01,
     &  3.906e+01,3.498e+01,4.719e+01,3.388e+01,4.383e+01,4.237e+01,
     &  2.460e+01,2.003e+01,2.041e+01,3.133e+01,3.428e+01,3.913e+01,
     &  5.984e+01,3.579e+01,3.764e+01,1.886e+01,5.297e+01,4.162e+01,
     &  3.732e+01,3.322e+01,3.610e+01,3.482e+01,4.446e+01,5.240e+01,
     &  4.404e+01,4.902e+01,5.670e+01,2.677e+01,1.921e+01,3.275e+01,
     &  4.919e+01,3.033e+01,2.480e+01,2.635e+01,3.751e+01,2.508e+01/
      data (sun3(i),i=2341,2400)/
     &  3.481e+01,2.401e+01,2.568e+01,3.745e+01,3.819e+01,3.457e+01,
     &  3.925e+01,2.334e+01,1.154e+01,1.498e+01,2.897e+01,2.974e+01,
     &  4.410e+01,5.718e+01,3.462e+01,3.537e+01,4.820e+01,4.615e+01,
     &  2.739e+01,2.411e+01,3.083e+01,3.577e+01,4.155e+01,2.980e+01,
     &  1.823e+01,1.893e+01,2.293e+01,3.506e+01,2.732e+01,4.131e+01,
     &  3.547e+01,1.718e+01,1.806e+01,3.905e+01,2.913e+01,1.955e+01,
     &  2.049e+01,2.506e+01,2.768e+01,1.744e+01,2.600e+01,1.984e+01,
     &  1.572e+01,2.821e+01,2.124e+01,1.929e+01,1.685e+01,1.679e+01,
     &  1.421e+01,2.265e+01,1.402e+01,1.903e+01,1.463e+01,1.278e+01,
     &  1.428e+01,1.396e+01,1.367e+01,1.307e+01,1.378e+01,1.407e+01/
      data (sun3(i),i=2401,2460)/
     &  1.296e+01,1.356e+01,1.480e+01,1.382e+01,1.323e+01,1.381e+01,
     &  1.313e+01,1.231e+01,1.190e+01,1.277e+01,1.166e+01,1.112e+01,
     &  1.217e+01,1.218e+01,1.171e+01,1.057e+01,1.253e+01,1.129e+01,
     &  1.238e+01,8.687e+00,1.062e+01,1.212e+01,1.270e+01,1.028e+01,
     &  1.004e+01,1.203e+01,1.108e+01,9.293e+00,1.157e+01,1.150e+01,
     &  1.198e+01,1.150e+01,1.105e+01,1.174e+01,1.049e+01,1.140e+01,
     &  9.622e+00,1.142e+01,1.230e+01,1.116e+01,1.043e+01,1.001e+01,
     &  1.170e+01,1.139e+01,1.078e+01,9.622e+00,8.734e+00,9.449e+00,
     &  1.113e+01,9.840e+00,1.103e+01,9.832e+00,8.365e+00,7.702e+00,
     &  9.123e+00,1.067e+01,1.097e+01,9.437e+00,9.154e+00,8.843e+00/
      data (sun3(i),i=2461,2494)/
     &  1.021e+01,8.275e+00,6.018e+00,7.045e+00,8.009e+00,1.007e+01,
     &  8.980e+00,7.486e+00,7.783e+00,9.095e+00,9.285e+00,8.224e+00,
     &  1.030e+01,8.923e+00,8.154e+00,8.251e+00,8.283e+00,8.517e+00,
     &  9.428e+00,8.940e+00,7.486e+00,7.695e+00,9.173e+00,8.089e+00,
     &  8.193e+00,7.251e+00,8.160e+00,8.710e+00,8.561e+00,7.154e+00,
     &  8.829e+00,8.004e+00,8.322e+00,8.312e+00/

c sun3    modtran3      100-49960 cm-1      20 cm-1           2494

      nns=2494
      wnmin=100.
      wnmax=49960.
      do i=1,nns
        sun(i)=sun3(nns-i+1)
        wn=wnmin+(wnmax-wnmin)*real(nns-i)/(nns-1)
        wlsun(i)=10000./wn
      enddo
      return
      end


c============================================================

      subroutine snow(wlalb,alb,nna)
      use params, only: kr
      implicit none
      integer, parameter :: mxwl=751
      integer :: i, nna
      real(kr) :: albx(mxwl), wlalb(*), alb(*), wlmax, wlmin

c  
c  Wiscombe, W.J. and and S.G. Warren, 1980: "A model for the spectral
c  albedo of snow. I: pure snow."  J. Atmospheric Sciences, 37, 2712-2733.
c  (based on there Mie/delta-eddington model with Reff=125um )

      data (albx(i),i=1,110) /
     & .970,.970,.970,.970,.970,.970,.970,.970,.970,.970,.970,
     & .971,.972,.973,.974,.975,.975,.975,.975,.976,.977,.978,
     & .979,.980,.980,.980,.980,.980,.980,.980,.980,.981,.982,
     & .983,.984,.985,.985,.985,.985,.986,.987,.988,.989,.990,
     & .990,.990,.990,.990,.990,.990,.990,.990,.990,.990,.990,
     & .988,.987,.986,.985,.983,.982,.981,.980,.978,.977,.976,
     & .975,.973,.972,.971,.970,.968,.967,.966,.965,.965,.965,
     & .965,.965,.963,.962,.961,.960,.953,.952,.951,.950,.950,
     & .950,.950,.950,.948,.945,.943,.940,.938,.935,.933,.930,
     & .928,.925,.923,.920,.918,.915,.913,.910,.908,.907,.906/
      data (albx(i),i=111,220) /
     & .905,.903,.902,.901,.900,.898,.895,.893,.890,.885,.880,
     & .875,.870,.865,.860,.855,.850,.843,.842,.841,.840,.835,
     & .830,.825,.820,.810,.800,.790,.780,.771,.762,.753,.745,
     & .739,.733,.727,.720,.718,.715,.713,.710,.715,.720,.725,
     & .730,.733,.735,.738,.740,.742,.743,.744,.745,.745,.745,
     & .745,.745,.743,.742,.741,.740,.732,.724,.716,.710,.692,
     & .674,.656,.640,.630,.620,.610,.600,.584,.568,.552,.535,
     & .524,.513,.502,.490,.490,.490,.490,.490,.490,.490,.490,
     & .490,.492,.493,.494,.495,.495,.495,.495,.495,.495,.495,
     & .495,.495,.493,.492,.491,.490,.488,.485,.483,.480,.470/
      data (albx(i),i=221,330) /
     & .460,.450,.440,.424,.408,.392,.375,.320,.264,.198,.150,
     & .134,.118,.101,.085,.074,.063,.051,.040,.036,.032,.028,
     & .025,.025,.025,.025,.025,.028,.030,.033,.035,.039,.043,
     & .047,.050,.053,.055,.058,.060,.063,.065,.068,.070,.073,
     & .075,.078,.080,.083,.085,.088,.090,.093,.095,.098,.100,
     & .104,.108,.112,.115,.119,.123,.126,.130,.135,.140,.145,
     & .150,.153,.155,.158,.160,.163,.165,.168,.170,.173,.175,
     & .178,.180,.184,.188,.192,.195,.204,.213,.222,.230,.238,
     & .246,.254,.260,.254,.248,.242,.235,.206,.177,.148,.120,
     & .104,.088,.072,.055,.049,.043,.037,.030,.038,.035,.033/
      data (albx(i),i=331,mxwl) /
     & .030,.028,.025,.023,.020,.023,.022,.021,.020,.018,.017,
     & .016,.015,.013,.012,.011,.010,.010,.010,.010,.010,.010,
     & .010,.010,.010,.012,.013,.014,.015,.017,.018,.019,.020,
     & .022,.023,.024,.025,.025,.025,.025,.025,.027,.030,.032,
     & .035,.040,.045,.050,.055,.059,.063,.067,.070,.074,.078,
     & .082,.085,.089,.093,.097,.100,.102,.103,.104,.105,.109,
     & .113,.117,.120,.124,.128,.132,.135,.130,.125,.120,.115,
     & .108,.100,.092,.085,.081,.077,.073,.070,.068,.065,.063,
     & .062,.058,.055,.053,.050,.048,.047,.046,.045,.042,.040,
     & .038,.035,.033,.032,.031,.030,.030,.030,.030,.030,.028,
     & .027,.026,.025,.025,.025,.025,.025,.025,.025,.025,.025,
     & .025,.025,.025,.025,.025,.025,.025,.025,.025,.025,.025,
     & .025,.025,.025,.025,.025,.023,.022,.021,.020,.018,.017,
     & .016,.015,.013,.012,.011,.010,.008,.007,.006,.005,.003,
     & .002,.001,.000,264*0.0/

      nna=mxwl
      alb(1:nna)=albx(1:nna)
      wlmin=.25
      wlmax=4.0
      do i=1,nna
        wlalb(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nna-1)      
      enddo
      return
      end

c=============================================================================

      subroutine clearw(wlalb,alb,nna)
      use params, only: kr
      implicit none
      integer, parameter :: mxwl=751
      integer :: i, nna
      real(kr) :: albx(mxwl), wlalb(*), alb(*), wlmax, wlmin

c     clear water reflectance
c     warning : values of dry sand ground reflectance are given
c     between 0.5 and 1.0 microns. outside this interval the
c     values are set to 0.
c
c     Viollier, M. "Teledetection des concentrations de seston et
c     pigments chlorophylliens contenus dans l'Ocean", These de Doctorat
c     d'Etat, no 503, 1980

      data (albx(i),i=1,mxwl) / 30*0.,
     & .041,.041,.041,.041,.041,.041,.041,.041,.041,.041,.041,
     & .041,.041,.041,.041,.041,.041,.041,.041,.041,.041,.042,
     & .043,.044,.044,.046,.047,.049,.050,.052,.054,.055,.056,
     & .059,.060,.061,.061,.059,.057,.054,.053,.051,.050,.049,
     & .047,.046,.046,.045,.044,.043,.043,.041,.040,.038,.037,
     & .037,.036,.033,.032,.031,.029,.027,.024,.023,.021,.018,
     & .015,.012,.009,.008,.006,.004,.002,.001,.000,.000,.000,
     & 644*0./
      nna=mxwl
      alb(1:nna)=albx(1:nna)
      wlmin=.25
      wlmax=4.0
      do i=1,nna
        wlalb(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nna-1)      
      enddo
      return
      end

c=============================================================================

      subroutine lakew(wlalb,alb,nna)
      use params, only: kr
      implicit none
      integer, parameter :: mxwl=751
      integer :: i, nna
      real(kr) :: albx(mxwl), wlalb(*), alb(*), wlmax, wlmin

c     lake water reflectance
c     warning : values of lake water reflectance are given
c     between 0.35 and 1.0 microns. outside this interval the
c     values are set to 0.
c
c     Kondratyev, K. Y., 1969: "Radiation in the atmosphere", Academic
c     Press, N.Y. 10003 USA


      data (albx(i),i=1,mxwl) / 21*0.,
     & .045,.047,.048,.050,.051,.051,.053,.053,.055,.057,.058,
     & .059,.060,.060,.062,.064,.065,.067,.068,.070,.070,.069,
     & .071,.071,.072,.074,.074,.075,.076,.076,.077,.077,.077,
     & .077,.079,.080,.080,.081,.081,.082,.082,.082,.083,.082,
     & .083,.082,.082,.082,.082,.082,.080,.079,.079,.077,.075,
     & .074,.072,.071,.070,.068,.066,.065,.064,.063,.061,.060,
     & .059,.057,.056,.054,.053,.051,.050,.048,.047,.046,.045,
     & .044,.042,.041,.040,.039,.037,.036,.035,.034,.032,.032,
     & .031,.030,.030,.028,.027,.027,.025,.024,.024,.023,.022,
     & .022,.020,.020,.020,.019,.019,.019,.019,.019,.019,.018,
     & .016,.016,.015,.014,.012,.012,.012,.011,.011,.009,.008,
     & .008,.006,.005,.005,.003,.002,.000,602*0./
      nna=mxwl
      alb(1:nna)=albx(1:nna)
      wlmin=.25
      wlmax=4.0
      do i=1,nna
        wlalb(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nna-1)      
      enddo
      return
      end

c=============================================================================

      subroutine seaw(wlalb,alb,nna)
      use params, only: kr
      implicit none
      integer, parameter :: mxwl=751
      integer :: i, nna
      real(kr) :: albx(mxwl), wlalb(*), alb(*), wlmax, wlmin
c
c  sea water (actually just clear water)
c
c     Viollier, M. "Teledetection des concentrations de seston et
c     pigments chlorophylliens contenus dans l'Ocean", These de Doctorat
c     d'Etat, no 503, 1980


      data (albx(i),i=1,mxwl) / 41*0.041,
     & .041,.041,.041,.041,.041,.041,.041,.041,.041,.041,.042,
     & .043,.044,.044,.046,.047,.049,.050,.052,.054,.055,.056,
     & .059,.060,.061,.061,.059,.057,.054,.053,.051,.050,.049,
     & .047,.046,.046,.045,.044,.043,.043,.041,.040,.038,.037,
     & .037,.036,.033,.032,.031,.029,.027,.024,.023,.021,.018,
     & .015,.012,.009,.008,.006,.004,.002,.001,.000,.000,.000,
     & 644*0./
      nna=mxwl
      alb(1:nna)=albx(1:nna)
      wlmin=.25
      wlmax=4.0
      do i=1,nna
        wlalb(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nna-1)      
      enddo
      return
      end

c=============================================================================

      subroutine sand(wlalb,alb,nna)
      use params, only: kr
      implicit none
      integer, parameter :: mxwl=751
      integer :: i, nna
      real(kr) :: albx(mxwl), wlalb(*), alb(*), wlmax, wlmin

c     sand average reflectance
c     warning : values of dry sand ground reflectance are given
c     between 0.4 and 2.2 microns. outside this interval the
c     values are set to 0.
c
c     Staetter, R., and M. Schroeder, 1978: "Spectral characteristics of
c     natural surfaces", Proceeding of the tenth Int. Conf. on Earth
c     Obs. from Space, 6-11 March 1978, (ESA-SP, 134)


      data (albx(i),i=1,140) / 30*0.,
     & .091,.091,.091,.091,.091,.091,.095,.095,.095,.095,.095,
     & .095,.095,.097,.097,.100,.103,.103,.107,.107,.107,.107,
     & .110,.114,.117,.121,.121,.125,.125,.125,.128,.126,.131,
     & .134,.134,.134,.134,.134,.134,.141,.145,.149,.149,.154,
     & .157,.160,.163,.163,.166,.169,.173,.177,.181,.185,.189,
     & .195,.199,.204,.204,.208,.214,.220,.224,.228,.233,.235,
     & .239,.242,.246,.246,.248,.251,.254,.256,.260,.263,.263,
     & .263,.266,.270,.273,.276,.279,.279,.282,.284,.284,.284,
     & .286,.286,.286,.290,.292,.292,.292,.292,.292,.292,.292,
     & .295,.295,.298,.298,.301,.301,.306,.306,.310,.310,.314/
      data (albx(i),i=141,240) /
     & .314,.314,.316,.316,.316,.316,.316,.319,.321,.321,
     & .321,.321,.324,.324,.324,.324,.322,.322,.322,.320,
     & .320,.317,.317,.317,.320,.320,.320,.323,.323,.327,
     & .327,.327,.329,.329,.332,.332,.332,.332,.333,.335,
     & .338,.341,.345,.348,.353,.353,.353,.353,.355,.355,
     & .355,.355,.353,.353,.353,.353,.353,.353,.356,.356,
     & .356,.356,.360,.360,.360,.363,.363,.363,.367,.367,
     & .367,.370,.370,.370,.370,.370,.370,.367,.367,.364,
     & .362,.362,.362,.359,.359,.359,.359,.359,.363,.363,
     & .363,.363,.363,.365,.369,.369,.369,.369,.369,.372/
      data (albx(i),i=241,340) /
     & .372,.372,.372,.375,.375,.375,.375,.375,.375,.375,
     & .375,.375,.375,.375,.375,.375,.375,.375,.377,.377,
     & .377,.377,.377,.377,.377,.377,.377,.377,.377,.377,
     & .379,.379,.379,.379,.382,.382,.382,.382,.385,.385,
     & .385,.385,.385,.385,.388,.388,.388,.388,.388,.388,
     & .388,.391,.391,.391,.391,.391,.391,.391,.391,.391,
     & .391,.391,.394,.394,.394,.394,.394,.394,.394,.396,
     & .396,.396,.396,.396,.396,.393,.393,.393,.393,.389,
     & .386,.386,.382,.382,.378,.378,.374,.369,.369,.369,
     & .371,.371,.371,.371,.371,.371,.371,.371,.371,.371/
      data (albx(i),i=341,mxwl) /
     & .374,.374,.377,.377,.377,.377,.379,.379,.379,.379,
     & .383,.385,.388,.388,.388,.392,.392,.392,.395,.395,
     & .395,.395,.395,.393,.393,.393,.388,.388,.388,.388,
     & .388,.385,.385,.385,.381,.381,.381,.381,.381,.381,
     & .381,.381,.374,.374,.374,.374,.374,.374,.374,.374,
     & .372,.369,.369,.369,.369,.369,.369,.369,.369,.369,
     & .369,.369,.369,.369,.369,.369,.369,.369,.369,.369,
     & .369,340*0./
      nna=mxwl
      alb(1:nna)=albx(1:nna)
      wlmin=.25
      wlmax=4.0
      do i=1,nna
        wlalb(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nna-1)      
      enddo
      return
      end

c=============================================================================

      subroutine vegeta(wlalb,alb,nna)
      use params, only: kr
      implicit none
      integer, parameter :: mxwl=751
      integer :: i, nna
      real(kr) :: albx(mxwl), wlalb(*), alb(*), wlmax, wlmin

c     vegetation average reflectance
c     warning : values of dry sand ground reflectance are given
c     between 0.4 and 2.2 microns. outside this interval the
c     values are set to 0.
c
c     Manual of Remote Sensing American Society of Photogrammetry,
c     R.G. Reeves, A. Anson, D. Landen, eds.  1st ed. Falls Church, Va.,
c     1975.

      data (albx(i),i=1,140) / 30*0.,
     & .060,.060,.062,.064,.066,.069,.070,.072,.074,.078,.080,
     & .083,.084,.089,.093,.098,.102,.104,.106,.110,.115,.119,
     & .121,.125,.127,.130,.133,.133,.134,.133,.131,.127,.121,
     & .115,.110,.105,.101,.098,.094,.090,.087,.083,.081,.080,
     & .078,.076,.075,.074,.073,.073,.073,.074,.079,.100,.138,
     & .169,.199,.228,.259,.290,.316,.350,.378,.403,.436,.462,
     & .487,.509,.511,.514,.519,.520,.520,.522,.522,.522,.523,
     & .524,.524,.524,.524,.526,.526,.526,.527,.527,.527,.528,
     & .528,.528,.529,.529,.529,.529,.531,.531,.531,.531,.531,
     & .532,.532,.532,.532,.532,.533,.533,.533,.534,.534,.534/
      data (albx(i),i=141,240) /
     & .534,.535,.535,.536,.536,.537,.537,.536,.536,.535,
     & .535,.534,.532,.531,.530,.528,.528,.527,.527,.526,
     & .525,.524,.522,.521,.519,.518,.515,.513,.512,.510,
     & .508,.507,.506,.505,.502,.500,.498,.496,.495,.493,
     & .492,.492,.492,.492,.492,.493,.495,.495,.496,.496,
     & .496,.497,.497,.497,.498,.498,.497,.497,.497,.495,
     & .493,.492,.491,.488,.486,.482,.478,.476,.472,.467,
     & .462,.451,.441,.429,.421,.408,.399,.385,.371,.365,
     & .349,.339,.330,.321,.309,.298,.289,.279,.272,.267,
     & .259,.251,.243,.233,.229,.224,.218,.215,.215,.215/
      data (albx(i),i=241,340) /
     & .215,.219,.223,.229,.234,.240,.249,.256,.260,.267,
     & .273,.279,.286,.293,.300,.306,.312,.319,.325,.331,
     & .337,.341,.345,.351,.355,.360,.362,.367,.369,.372,
     & .376,.378,.379,.381,.382,.384,.386,.387,.389,.388,
     & .388,.388,.388,.388,.388,.384,.383,.381,.380,.378,
     & .376,.374,.373,.371,.370,.368,.367,.366,.365,.365,
     & .363,.362,.361,.359,.358,.357,.355,.353,.350,.347,
     & .346,.345,.343,.340,.337,.335,.331,.330,.321,.312,
     & .296,.273,.221,.186,.158,.138,.129,.121,.110,.102,
     & .095,.091,.089,.086,.086,.084,.084,.084,.086,.087/
      data (albx(i),i=341,mxwl) /
     & .093,.098,.105,.114,.116,.124,.133,.134,.141,.147,
     & .151,.156,.162,.166,.170,.174,.175,.178,.181,.185,
     & .187,.188,.192,.196,.199,.201,.205,.208,.212,.213,
     & .214,.217,.219,.220,.221,.224,.227,.229,.231,.233,
     & .237,.238,.239,.241,.242,.243,.245,.245,.246,.248,
     & .248,.250,.246,.242,.238,.234,.230,.226,.222,.218,
     & .214,.210,.206,.202,.198,.194,.190,.186,.182,.178,
     & .174,.170,.166,.162,.158,.154,.150,.146,.142,.138,
     & .134,.130,.126,.122,.118,.114,.110,.106,.102,.098,
     & .094,.090,.086,.082,.078,.074,.070,.066,.062,.058,
     & .054,.050,.046,.042,.038,.034,.030,.026,.022,.018,
     & .014,.010,.010,.010,.010,.010,.010,.010,.010,.010,
     & .010,.010,.010,.010,.010,.010,.010,.010,.010,.010,
     & .010,280*0./
      nna=mxwl
      alb(1:nna)=albx(1:nna)
      wlmin=.25
      wlmax=4.0
      do i=1,nna
        wlalb(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nna-1)      
      enddo
      return
      end

c=======================================================================
c=======================================================================
      module fltblk             ! filter data base
      use params, only: kr
      save
      integer, parameter :: mxwl=1000
      real(kr), dimension(mxwl) :: wlfilt,filt
      integer :: nnf=0
      end module fltblk
c=======================================================================
      subroutine setfilt(isat,wlinf,wlsup,wlinc,wlmin,wlmax,nwl)
c
c input   isat      filter selection flag
c 
c         wlinf     lower wavelength limit
c                      or central wavelength for isat=-2,-3
c         wlsup     upper wavelength limit
c                      or equivalent width for isat=-2,-3
c         wlinc     wavelength increment, used to determine number
c                      of wavelength sample points in wavelength iteration
c output
c         wlmin     lower wavelength limit
c         wlmax     upper wavelength limit
c         nwl       number of sample points
c
c         wlinc     is set to 0.001 if single wavelength sample point 
c                     is specified, i.e., either if
c
c                     isat.eq.-2 .and. wlsup.eq.0   
c                          -or-
c                     isal.eq.0 .and. wlinf.eq.wlsup
c
c                   otherwise it is not changed
c
c NOTE: actual filter values are returned by the function "filter"
c
      use params, only: kr, pi
      use fltblk, only: nnf, mxwl, filt, wlfilt
      implicit none
      integer :: i, isat, nwl
      real(kr) :: sqpi, wlinc, wlinf, wlmax, wlmin, wlsup, xc, xlim, xx 

      if(wlsup.eq.0.and.isat.lt.-2) then
        write(*,*) 'Error -- WLSUP must be non-zero when ISAT=',isat
        write(*,*) '         WLSUP =',wlsup
        stop
      endif

      select case (isat)

      case (-4)                 ! center, equivalent width, gaussian
        sqpi=sqrt(pi)
        xc=2
        xlim=xc*sqpi
        nnf=1000
        wlmin=wlinf-xc*wlsup
        wlmax=wlinf+xc*wlsup
        do i=1,nnf
          xx=-xlim+(i-1)*(2*xlim)/(nnf-1)
          filt(i)=exp(-xx**2)
          wlfilt(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nnf-1)
        enddo

      case (-3)               ! center, equivalent width, triangular
        nnf=3
        wlmin=wlinf-wlsup
        wlmax=wlinf+wlsup
        filt(1:3)=(/0.,1.,0./)
        wlfilt(1:3)=(/wlmin,wlinf,wlmax/)

      case(-2)               ! center, width, flat top
        nnf=0
        wlmin=wlinf-.5*wlsup
        wlmax=wlinf+.5*wlsup
        if(wlsup.eq.0.) then
          nwl=1
          wlinc=.001
          return
        endif

      case (-1)              ! read from filter.dat
        nnf=mxwl
        call rdspec('filter.dat',wlfilt,filt,nnf)
        wlmin=wlfilt(1)
        wlmax=wlfilt(nnf)

      case (0)               ! wl_min, wl_max, flat  top
        nnf=0
        wlmin=wlinf
        wlmax=wlsup
        if(wlinf.eq.wlsup) then
          nwl=1
          wlinc=.001
          return
        endif

      case (1)  ;  call meteo(filt,wlmin,wlmax,nnf)
      case (2)  ;  call goese(filt,wlmin,wlmax,nnf)
      case (3)  ;  call goesw(filt,wlmin,wlmax,nnf)
      case (4)  ;  call avhr81(filt,wlmin,wlmax,nnf)
      case (5)  ;  call avhr82(filt,wlmin,wlmax,nnf)
      case (6)  ;  call avhr91(filt,wlmin,wlmax,nnf)
      case (7)  ;  call avhr92(filt,wlmin,wlmax,nnf)
      case (8)  ;  call avhr101(filt,wlmin,wlmax,nnf)
      case (9)  ;  call avhr102(filt,wlmin,wlmax,nnf)
      case (10) ;  call avhr111(filt,wlmin,wlmax,nnf)
      case (11) ;  call avhr112(filt,wlmin,wlmax,nnf)
      case (12) ;  call gtr1(filt,wlmin,wlmax,nnf)
      case (13) ;  call gtr2(filt,wlmin,wlmax,nnf)
      case (14) ;  call nm410(filt,wlmin,wlmax,nnf)
      case (15) ;  call nm936(filt,wlmin,wlmax,nnf)
      case (16) ;  call mfrsr1(filt,wlmin,wlmax,nnf)
      case (17) ;  call mfrsr2(filt,wlmin,wlmax,nnf)
      case (18) ;  call mfrsr3(filt,wlmin,wlmax,nnf)
      case (19) ;  call mfrsr4(filt,wlmin,wlmax,nnf)
      case (20) ;  call mfrsr5(filt,wlmin,wlmax,nnf)
      case (21) ;  call mfrsr6(filt,wlmin,wlmax,nnf)
      case (22) ;  call avhr83(filt,wlmin,wlmax,nnf)
      case (23) ;  call avhr84(filt,wlmin,wlmax,nnf)
      case (24) ;  call avhr85(filt,wlmin,wlmax,nnf)
      case (25) ;  call setlow(filt,wlmin,wlmax,nnf)
      case (26) ;  call airs1(filt,wlmin,wlmax,nnf)
      case (27) ;  call airs2(filt,wlmin,wlmax,nnf)
      case (28) ;  call airs3(filt,wlmin,wlmax,nnf)
      case (29) ;  call airs4(filt,wlmin,wlmax,nnf)

      end select

      if(wlmin.lt.0.199) then
        write(*,*) 'Error in SETFILT -- illegal wavelength limits '
        write(*,*) wlmin,wlmax
        stop
      endif

      if(isat.gt.0) then
        do i=1,nnf
          wlfilt(i)=wlmin+(wlmax-wlmin)*real(i-1)/(nnf-1)
        enddo
      endif
      
      if(wlinc.gt.1.) then
c       equal spacing in wavenumber
        nwl=((10000./wlmin) - (10000./wlmax))/wlinc + 1.
      elseif(wlinc.lt.0.) then
c       equal spacing in ln(wavelength) and ln(wavenumber)
        nwl=1+log(wlmax/wlmin)/abs(wlinc)
      else
c       use default wavelength increment
        if(wlinc.eq.0)
     &       wlinc=(wlmax-wlmin)/max(10,1+int((wlmax-wlmin)/0.005))
c       equal spacing in wavelength
        nwl=nint((wlmax-wlmin)/wlinc)+1
      endif

      if(wlmin.ne.wlmax.and.nwl.eq.1) nwl=2

      return
      end     
      
c=======================================================================
      function filter(w)
c
c input:  w       wavelength in microns
c output:         filter functions
c 
c NOTE: "setfilt" must be called before "filter"
c
      use params, only: one, zero, kr
      use fltblk, only: nnf, filt, wlfilt
      implicit none
      integer :: i
      real(kr) :: w, wt, filter

      if(nnf.eq.0) then 
        filter=1.
      else 
        call locate(wlfilt,nnf,w,i)
        wt=(w-wlfilt(i))/(wlfilt(i+1)-wlfilt(i))
        wt=max(zero,min(one,wt))
        filter=filt(i)*(1.-wt)+filt(i+1)*wt
      endif
      return
      end
c=======================================================================
      subroutine meteo(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.355, wmx=1.11
      integer, parameter :: mwv=152
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
c
c    spectral response of meteosat
c
      data sr/ 
     & 0.0050,0.0094,0.0154,0.0191,0.0208,0.0263,0.0357,0.0404,0.0513,
     & 0.0570,0.0689,0.0746,0.0820,0.0900,0.1001,0.1071,0.1152,0.1222,
     & 0.1358,0.1482,0.1561,0.1683,0.1789,0.1908,0.2047,0.2146,0.2280,
     & 0.2426,0.2582,0.2758,0.2897,0.3029,0.3209,0.3388,0.3514,0.3705,
     & 0.3886,0.4035,0.4240,0.4387,0.4644,0.4885,0.5095,0.5348,0.5611,
     & 0.5841,0.6109,0.6332,0.6478,0.6617,0.6736,0.6857,0.7006,0.7120,
     & 0.7254,0.7430,0.7698,0.7834,0.8002,0.8206,0.8434,0.8563,0.8753,
     & 0.8949,0.9214,0.9410,0.9571,0.9722,0.9844,0.9893,0.9948,0.9968,
     & 0.9960,1.0000,0.9960,0.9953,0.9921,0.9874,0.9824,0.9770,0.9735,
     & 0.9710,0.9660,0.9660,0.9623,0.9623,0.9628,0.9599,0.9569,0.9509,
     & 0.9405,0.9271,0.9150,0.9029,0.8865,0.8753,0.8644,0.8548,0.8426,
     & 0.8283,0.8151,0.8037,0.7901,0.7804,0.7660,0.7591,0.7489,0.7393,
     & 0.7326,0.7192,0.6984,0.6664,0.6434,0.6178,0.6005,0.5824,0.5603,
     & 0.5385,0.5232,0.5038,0.4862,0.4706,0.4471,0.4211,0.4007,0.3809,
     & 0.3618,0.3388,0.3147,0.2924,0.2758,0.2548,0.2332,0.2099,0.1908,
     & 0.1678,0.1509,0.1348,0.1138,0.0994,0.0838,0.0751,0.0642,0.0538,
     & 0.0456,0.0372,0.0263,0.0178,0.0097,0.0084,0.0020,0.0000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
      return
      end
c=======================================================================
      subroutine goese(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.500, wmx=.895
      integer, parameter :: mwv=80
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
c
c    spectral response of goes east
c
      data sr/
     & 0.013,0.046,0.133,0.302,0.455,0.584,0.680,0.727,0.772,0.807,
     & 0.838,0.859,0.879,0.899,0.919,0.934,0.948,0.965,0.983,0.991,
     & 0.983,0.974,0.961,0.949,0.937,0.925,0.916,0.912,0.907,0.911,
     & 0.914,0.907,0.892,0.877,0.843,0.808,0.772,0.734,0.695,0.678,
     & 0.660,0.639,0.613,0.588,0.556,0.521,0.486,0.457,0.427,0.397,
     & 0.363,0.330,0.301,0.279,0.257,0.234,0.210,0.186,0.165,0.146,
     & 0.127,0.110,0.096,0.082,0.069,0.058,0.047,0.038,0.031,0.024,
     & 0.018,0.014,0.010,0.006,0.004,0.003,0.002,0.001,0.001,0.000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
      return
      end
c=======================================================================
      subroutine goesw(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.500, wmx=.905
      integer, parameter :: mwv=82
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
c
c    spectral response of goes west
c
      data sr/
     & 0.002,0.020,0.065,0.183,0.409,0.611,0.775,0.889,0.925,0.957,
     & 0.973,0.987,0.993,0.999,0.998,0.997,0.990,0.983,0.974,0.964,
     & 0.956,0.951,0.944,0.928,0.912,0.893,0.874,0.853,0.832,0.811,
     & 0.795,0.780,0.760,0.738,0.716,0.686,0.656,0.625,0.592,0.560,
     & 0.536,0.513,0.489,0.464,0.440,0.413,0.385,0.358,0.331,0.305,
     & 0.279,0.256,0.233,0.211,0.192,0.174,0.157,0.143,0.129,0.115,
     & 0.101,0.087,0.077,0.070,0.062,0.054,0.046,0.038,0.032,0.027,
     & 0.022,0.018,0.015,0.011,0.008,0.006,0.004,0.003,0.002,0.001,
     & 0.001,0.000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
      return
      end
c=======================================================================
      subroutine avhr81(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.505, wmx=0.745
      integer, parameter :: mwv=49
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
c
c    1st spectral response of avhrr  (noaa 8)
c
      data sr/
     &  .004,.008,.019,.028,.035,.048, .060, .070, .078, .087,
     &  .095,.099,.367,.527,.665,.788, .815, .811, .807, .828,
     &  .870,.903,.931,.956,.978,.993,1.000,1.000,1.000,1.000,
     & 1.000,.976,.900,.817,.743,.622, .438, .306, .217, .140,
     &  .088,.067,.047,.032,.024,.015, .003, .001, .000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine avhr82(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.70, wmx=1.085
      integer, parameter :: mwv=78
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
c
c    2nd spectral response of avhr  (noaa 8)
c
      data sr/
     & .035,.168,.350,.510,.622,.756,.837,.896,.929,.949,
     & .952,.946,.945,.986,.998,.998,.998,.988,.939,.910,
     & .889,.869,.853,.840,.827,.817,.809,.802,.797,.793,
     & .789,.785,.781,.775,.768,.759,.753,.749,.745,.741,
     & .738,.719,.695,.678,.662,.648,.636,.627,.618,.612,
     & .603,.600,.597,.593,.587,.582,.562,.523,.485,.433,
     & .376,.288,.230,.189,.145,.107,.075,.051,.030,.020,
     & .014,.008,.006,.003,.001,.001,.001,.000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine avhr91(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.54, wmx=.80
      integer, parameter :: mwv=53
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
c
c     revised  jan94
c
c    1st spectral response of avhr (noaa 9)
c
      data sr/
     &    0.0010, 0.0040, 0.0400, 0.1398, 0.2700, 0.3929, 0.5000,
     &    0.5898, 0.6700, 0.7448, 0.8000, 0.8203, 0.8150, 0.7973,
     &    0.7850, 0.7933, 0.8200, 0.8587, 0.8950, 0.9134, 0.9050,
     &    0.8663, 0.8250, 0.8087, 0.8150, 0.8397, 0.8900, 0.9645,
     &    1.0000, 0.9363, 0.8100, 0.6633, 0.5000, 0.3232, 0.1800,
     &    0.1138, 0.0900, 0.0697, 0.0500, 0.0351, 0.0250, 0.0188,
     &    0.0150, 0.0124, 0.0100, 0.0073, 0.0050, 0.0038, 0.0030,
     &    0.0019, 0.0010, 0.0009, 0.0010/

c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine avhr92(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.645, wmx=1.190
      integer, parameter :: mwv=110
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
c
c    2nd spectral response of avhr  (noaa 9)
c
      data sr/
     & .0036,.0071,.0071,.0071,.0075,.0079,.0079,.0079,.0100,.0120,
     & .0864,.1607,.3715,.4222,.5381,.6540,.7888,.9235,.9481,.9727,
     & .9845,.9963,.9982,1.000,.9945,.9891,.9891,.9891,.9593,.9294,
     & .9366,.9438,.9276,.9114,.8919,.8724,.8724,.8724,.8724,.8724,
     & .8724,.8724,.8724,.8724,.8724,.8724,.8710,.8695,.8647,.8600,
     & .8533,.8465,.8418,.8372,.8284,.8197,.8041,.7886,.7845,.7805,
     & .7762,.7719,.7656,.7594,.7374,.7154,.6489,.5824,.4963,.4103,
     & .3259,.2415,.1862,.1309,.1009,.0709,.0537,.0364,.0299,.0234,
     & .0188,.0142,.0106,.0070,.0070,.0070,.0061,.0051,.0038,.0025,
     & .0025,.0025,.0025,.0025,.0025,.0025,.0025,.0025,.0025,.0025,
     & .0025,.0025,.0025,.0025,.0025,.0025,.0025,.0025,.0025,.0000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine avhr101(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.555, wmx=.750
      integer, parameter :: mwv=40
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/ 
     $    0.039998,    0.079998,    0.274987,    0.469986,    0.604990,
     $    0.739990,    0.779997,    0.819997,    0.815000,    0.810000,
     $    0.829999,    0.849999,    0.894997,    0.939997,    0.964998,
     $    0.989998,    0.975001,    0.960001,    0.955000,    0.950000,
     $    0.974998,    0.999998,    0.995000,    0.990000,    0.810015,
     $    0.630015,    0.455016,    0.280017,    0.192508,    0.105008,
     $    0.077503,    0.050003,    0.040001,    0.030001,    0.025000,
     $    0.020000,    0.015000,    0.010000,    0.008000,    0.000000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
       return
       end
c======================================================================       
      subroutine avhr102(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.685, wmx=1.180
      integer, parameter :: mwv=100
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr(1:50)/
     $    0.001000,    0.002000,    0.002000,    0.002000,    0.040996,
     $    0.079996,    0.239985,    0.399985,    0.564984,    0.729984,
     $    0.804993,    0.879993,    0.909997,    0.939997,    0.959998,
     $    0.979998,    0.989999,    0.999999,    0.985002,    0.970002,
     $    0.955002,    0.940002,    0.920002,    0.900002,    0.894501,
     $    0.889001,    0.883501,    0.878001,    0.872501,    0.867001,
     $    0.861501,    0.856001,    0.850501,    0.845001,    0.838001,
     $    0.831001,    0.824001,    0.817001,    0.810001,    0.803001,
     $    0.796001,    0.789001,    0.782001,    0.775001,    0.763501,
     $    0.752001,    0.740501,    0.729001,    0.717502,    0.706002/
      data sr(51:100)/
     $    0.694502,    0.683002,    0.671502,    0.660002,    0.617006,
     $    0.574006,    0.531006,    0.488006,    0.445006,    0.402006,
     $    0.359006,    0.316007,    0.273006,    0.230006,    0.211503,
     $    0.193003,    0.174503,    0.156003,    0.137503,    0.119003,
     $    0.100003,    0.081003,    0.062503,    0.044003,    0.025503,
     $    0.007003,    0.006500,    0.006000,    0.005500,    0.005000,
     $    0.004000,    0.003000,    0.002500,    0.002000,    0.002000,
     $    0.002000,    0.002000,    0.002000,    0.001500,    0.001000,
     $    0.001000,    0.001000,    0.001000,    0.001000,    0.001000,
     $    0.001000,    0.001000,    0.001000,    0.000500,    0.000000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c======================================================================       
      subroutine avhr111(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.545, wmx=.750
      integer, parameter :: mwv=42
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     $    0.000500,    0.001000,    0.047997,    0.094997,    0.297487,
     $    0.499986,    0.569995,    0.639995,    0.699996,    0.759996,
     $    0.779999,    0.799999,    0.785001,    0.770001,    0.779999,
     $    0.789999,    0.839996,    0.889996,    0.885000,    0.880000,
     $    0.850002,    0.820002,    0.802501,    0.785002,    0.799999,
     $    0.814999,    0.907492,    0.999992,    0.900009,    0.800010,
     $    0.605019,    0.410019,    0.290011,    0.170011,    0.122505,
     $    0.075005,    0.057502,    0.040002,    0.030001,    0.020001,
     $    0.017000,    0.000000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
       return
       end
c======================================================================       
      subroutine avhr112(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.685, wmx=1.100
      integer, parameter :: mwv=84
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     $    0.004000,    0.008000,    0.018999,    0.029999,    0.189985,
     $    0.349985,    0.494986,    0.639986,    0.749990,    0.859990,
     $    0.899996,    0.939996,    0.964997,    0.989997,    0.994000,
     $    0.998000,    0.983002,    0.968002,    0.946502,    0.925002,
     $    0.892503,    0.860004,    0.855001,    0.850001,    0.847500,
     $    0.845000,    0.842500,    0.840000,    0.837500,    0.835000,
     $    0.832500,    0.830000,    0.827500,    0.825000,    0.824500,
     $    0.824000,    0.823500,    0.823000,    0.822500,    0.822000,
     $    0.821500,    0.821000,    0.820500,    0.820000,    0.801003,
     $    0.782003,    0.763003,    0.744003,    0.725003,    0.706003,
     $    0.687003,    0.668003,    0.649003,    0.630003,    0.585006,
     $    0.540006,    0.495006,    0.450006,    0.405006,    0.360006,
     $    0.315007,    0.270007,    0.225007,    0.180006,    0.165502,
     $    0.151002,    0.136002,    0.121003,    0.106502,    0.092002,
     $    0.077502,    0.063002,    0.048003,    0.033003,    0.018502,
     $    0.004002,    0.003500,    0.003000,    0.002500,    0.002000,
     $    0.001500,    0.001000,    0.000500,    0.000000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c======================================================================       
      subroutine gtr1(srr,wmin,wmax,nnf)
c      gtr-100 ch 1 (bsi's predeployment calibration measurement 
c      linearly interpolated to 5nm intervals)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.555, wmx=.805
      integer, parameter :: mwv=51
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &    0.0048,    0.0501,    0.2077,    0.4466,    0.6530,
     &    0.7900,    0.8780,    0.9232,    0.9468,    0.9712,
     &    0.9877,    0.9930,    0.9944,    1.0000,    0.9940,
     &    0.9785,    0.9700,    0.9538,    0.9241,    0.9531,
     &    0.9024,    0.8780,    0.8550,    0.8311,    0.8164,
     &    0.7743,    0.7516,    0.7424,    0.6844,    0.6400,
     &    0.5675,    0.5034,    0.4515,    0.3408,    0.2231,
     &    0.1521,    0.1057,    0.0716,    0.0511,    0.0414,
     &    0.0332,    0.0247,    0.0320,    0.0226,    0.0195,
     &    0.0167,    0.0153,    0.0139,    0.0113,    0.0117,
     &    0.0000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c======================================================================       
      subroutine gtr2(srr,wmin,wmax,nnf)
c      gtr-100 ch 2 (sbrc's postdeployment calibration measurements
c      spline fit to 5nm intervals)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.705, wmx=1.130
      integer, parameter :: mwv=86
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr(1:50)/
     &   0.0713, 0.1737, 0.2703, 0.3611, 0.4461,
     &   0.5248, 0.5972, 0.6629, 0.7213, 0.7720,
     &   0.8146, 0.8500, 0.8790, 0.9026, 0.9217,
     &   0.9368, 0.9490, 0.9586, 0.9666, 0.9734,
     &   0.9797, 0.9854, 0.9904, 0.9946, 0.9977,
     &   0.9996, 1.0000, 0.9988, 0.9956, 0.9903,
     &   0.9828, 0.9735, 0.9628, 0.9511, 0.9387,
     &   0.9261, 0.9134, 0.9012, 0.8897, 0.8793,
     &   0.8704, 0.8626, 0.8555, 0.8489, 0.8424,
     &   0.8356, 0.8281, 0.8196, 0.8096, 0.7978/
      data sr(51:86)/
     &   0.7839, 0.7679, 0.7499, 0.7300, 0.7084,
     &   0.6851, 0.6603, 0.6338, 0.6057, 0.5760,
     &   0.5447, 0.5122, 0.4787, 0.4447, 0.4106,
     &   0.3766, 0.3432, 0.3106, 0.2792, 0.2493,
     &   0.2212, 0.1950, 0.1704, 0.1476, 0.1263,
     &   0.1067, 0.0886, 0.0721, 0.0572, 0.0440,
     &   0.0325, 0.0227, 0.0144, 0.0077, 0.0025,
     &   0.0000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c======================================================================       
      subroutine nm410(srr,wmin,wmax,nnf)
c      gtr-100 410nm channel
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.389, wmx=.423   
      integer, parameter :: mwv=35
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &   0.0000,    0.0124,    0.0116,    0.0157,    0.0101,    0.0293,
     &   0.0280,    0.0510,    0.0549,    0.0566,    0.0797,    0.1107,
     &   0.1277,    0.1723,    0.2512,    0.3532,    0.5287,    0.7189,
     &   0.8874,    0.9670,    1.0000,    0.9932,    0.9853,    0.9357,
     &   0.8522,    0.6905,    0.4765,    0.2993,    0.1549,    0.0775,
     &   0.0400,    0.0263,    0.0131,    0.0035,    0.0000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine nm936(srr,wmin,wmax,nnf)
c      gtr-100 410nm channel
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.878, wmx=.955
      integer, parameter :: mwv=78
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &  0.0004,    0.0018,    0.0038,    0.0059,    0.0071,    0.0070,
     &  0.0070,    0.0080,    0.0095,    0.0117,    0.0134,    0.0149,
     &  0.0152,    0.0160,    0.0187,    0.0246,    0.0324,    0.0389,
     &  0.0383,    0.0349,    0.0383,    0.0416,    0.0383,    0.0373,
     &  0.0436,    0.0621,    0.0888,    0.1028,    0.0833,    0.0734,
     &  0.0849,    0.0926,    0.0928,    0.0824,    0.0775,    0.0888,
     &  0.1136,    0.1431,    0.1580,    0.1482,    0.1292,    0.1255,
     &  0.1252,    0.1264,    0.1263,    0.1236,    0.1229,    0.1270,
     &  0.1392,    0.1690,    0.2288,    0.3384,    0.4835,    0.6716,
     &  0.8228,    0.9022,    0.9113,    0.9200,    0.9597,    1.0000,
     &  0.9954,    0.9095,    0.7721,    0.5937,    0.4003,    0.2264,
     &  0.1207,    0.0613,    0.0323,    0.0172,    0.0100,    0.0062,
     &  0.0042,    0.0030,    0.0022,    0.0014,    0.0010,    0.0000/
c
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine mfrsr1(srr,wmin,wmax,nnf)
c      MFRSR channel 1
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.397, wmx=0.433
      integer, parameter :: mwv=37
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &    0.0001,    0.0003,    0.0008,    0.0020,    0.0044,
     &    0.0092,    0.0185,    0.0349,    0.0625,    0.1058,
     &    0.1696,    0.2570,    0.3686,    0.5000,    0.6417,
     &    0.7792,    0.8950,    0.9727,    1.0000,    0.9727,
     &    0.8950,    0.7792,    0.6417,    0.5000,    0.3686,
     &    0.2570,    0.1696,    0.1058,    0.0625,    0.0349,
     &    0.0185,    0.0092,    0.0044,    0.0020,    0.0008,
     &    0.0003,    0.0001/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine mfrsr2(srr,wmin,wmax,nnf)
c      MFRSR channel 2
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.482, wmx=0.518
      integer, parameter :: mwv=37
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &    0.0001,    0.0003,    0.0008,    0.0020,    0.0044,
     &    0.0092,    0.0185,    0.0349,    0.0625,    0.1058,
     &    0.1696,    0.2570,    0.3686,    0.5000,    0.6417,
     &    0.7792,    0.8950,    0.9727,    1.0000,    0.9727,
     &    0.8950,    0.7792,    0.6417,    0.5000,    0.3686,
     &    0.2570,    0.1696,    0.1058,    0.0625,    0.0349,
     &    0.0185,    0.0092,    0.0044,    0.0020,    0.0008,
     &    0.0003,    0.0001/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine mfrsr3(srr,wmin,wmax,nnf)
c      MFRSR channel 3
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.592, wmx=0.628
      integer, parameter :: mwv=37
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &    0.0001,    0.0003,    0.0008,    0.0020,    0.0044,
     &    0.0092,    0.0185,    0.0349,    0.0625,    0.1058,
     &    0.1696,    0.2570,    0.3686,    0.5000,    0.6417,
     &    0.7792,    0.8950,    0.9727,    1.0000,    0.9727,
     &    0.8950,    0.7792,    0.6417,    0.5000,    0.3686,
     &    0.2570,    0.1696,    0.1058,    0.0625,    0.0349,
     &    0.0185,    0.0092,    0.0044,    0.0020,    0.0008,
     &    0.0003,    0.0001/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine mfrsr4(srr,wmin,wmax,nnf)
c      MFRSR channel 4
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.647, wmx=0.683
      integer, parameter :: mwv=37
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &    0.0001,    0.0003,    0.0008,    0.0020,    0.0044,
     &    0.0092,    0.0185,    0.0349,    0.0625,    0.1058,
     &    0.1696,    0.2570,    0.3686,    0.5000,    0.6417,
     &    0.7792,    0.8950,    0.9727,    1.0000,    0.9727,
     &    0.8950,    0.7792,    0.6417,    0.5000,    0.3686,
     &    0.2570,    0.1696,    0.1058,    0.0625,    0.0349,
     &    0.0185,    0.0092,    0.0044,    0.0020,    0.0008,
     &    0.0003,    0.0001/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine mfrsr5(srr,wmin,wmax,nnf)
c      MFRSR channel 5
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.844, wmx=0.880
      integer, parameter :: mwv=37
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &    0.0001,    0.0003,    0.0008,    0.0020,    0.0044,
     &    0.0092,    0.0185,    0.0349,    0.0625,    0.1058,
     &    0.1696,    0.2570,    0.3686,    0.5000,    0.6417,
     &    0.7792,    0.8950,    0.9727,    1.0000,    0.9727,
     &    0.8950,    0.7792,    0.6417,    0.5000,    0.3686,
     &    0.2570,    0.1696,    0.1058,    0.0625,    0.0349,
     &    0.0185,    0.0092,    0.0044,    0.0020,    0.0008,
     &    0.0003,    0.0001/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine mfrsr6(srr,wmin,wmax,nnf)
c      MFRSR channel 6
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.922, wmx=0.958
      integer, parameter :: mwv=37
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &    0.0001,    0.0003,    0.0008,    0.0020,    0.0044,
     &    0.0092,    0.0185,    0.0349,    0.0625,    0.1058,
     &    0.1696,    0.2570,    0.3686,    0.5000,    0.6417,
     &    0.7792,    0.8950,    0.9727,    1.0000,    0.9727,
     &    0.8950,    0.7792,    0.6417,    0.5000,    0.3686,
     &    0.2570,    0.1696,    0.1058,    0.0625,    0.0349,
     &    0.0185,    0.0092,    0.0044,    0.0020,    0.0008,
     &    0.0003,    0.0001/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine avhr83(srr,wmin,wmax,nnf)
c     noaa-8    ahvrr channel 3
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=3.40, wmx=4.05
      integer, parameter :: mwv=27
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &   0.0000, 0.0084, 0.0953, 0.2715, 0.4908, 0.7088, 0.8807, 0.9713,
     &   1.0000, 0.9932, 0.9719, 0.9551, 0.9506, 0.9427, 0.9295, 0.9197,
     &   0.9128, 0.9067, 0.8991, 0.8869, 0.8325, 0.6741, 0.3732, 0.1003,
     &   0.0306, 0.0056, 0.0000/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine avhr84(srr,wmin,wmax,nnf)
c     noaa-8    ahvrr channel 4
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=10.225, wmx=11.675
      integer, parameter :: mwv=30
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &   0.0000, 0.1228, 0.3305, 0.5884, 0.8287, 0.9459, 0.9889, 1.0000,
     &   0.9901, 0.9670, 0.9371, 0.9061, 0.8765, 0.8479, 0.8193, 0.7899,
     &   0.7590, 0.7255, 0.6881, 0.6450, 0.5895, 0.5109, 0.3960, 0.2395,
     &   0.1127, 0.0509, 0.0238, 0.0101, 0.0013, 0.0000/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================
      subroutine avhr85(srr,wmin,wmax,nnf)
c     noaa-8    ahvrr channel 5
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=11.300, wmx=12.600
      integer, parameter :: mwv=27
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data sr/
     &   0.0000, 0.0613, 0.2334, 0.5287, 0.8555, 0.9537, 0.9249, 0.9351,
     &   0.9946, 1.0000, 0.9491, 0.8988, 0.8862, 0.8969, 0.9098, 0.9071,
     &   0.8867, 0.8515, 0.8029, 0.7409, 0.6642, 0.5720, 0.4648, 0.3412,
     &   0.2026, 0.0694, 0.0000/
      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
c
      return
      end
c=======================================================================

      subroutine setlow(srr,wmin,wmax,nnf)
c     action spectra for DNA damage by UVB radiation (Setlow, R.B, 1974)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=.280, wmx=0.341
      integer, parameter :: mwv=61

      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data  sr /
     $  7.6459E-01, 6.8615E-01, 6.1575E-01, 5.5258E-01, 4.9589E-01,
     $  4.4501E-01, 3.9935E-01, 3.5838E-01, 3.2161E-01, 2.8862E-01,
     $  2.5901E-01, 2.1868E-01, 1.8459E-01, 1.5582E-01, 1.3153E-01,
     $  1.1103E-01, 8.7078E-02, 6.8320E-02, 5.3602E-02, 4.2055E-02,
     $  3.2996E-02, 2.3459E-02, 1.6681E-02, 1.1862E-02, 8.4351E-03,
     $  5.9981E-03, 4.1937E-03, 2.9307E-03, 2.0480E-03, 1.4312E-03,
     $  1.0002E-03, 6.9896E-04, 4.8845E-04, 3.4134E-04, 2.3854E-04,
     $  1.6670E-04, 1.1649E-04, 8.1409E-05, 5.6891E-05, 3.9757E-05,
     $  2.7783E-05, 1.9416E-05, 1.3568E-05, 9.4818E-06, 6.6261E-06,
     $  4.6305E-06, 3.2359E-06, 2.2614E-06, 1.5803E-06, 1.1044E-06,
     $  7.7176E-07, 5.3932E-07, 3.7689E-07, 2.6338E-07, 1.8406E-07,
     $  1.2863E-07, 8.9888E-08, 6.2816E-08, 4.3898E-08, 3.0677E-08,
     $  2.1438E-08 /

      wmin=wmn
      wmax=wmx
      nnf=mwv
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
      return
      end
c=======================================================================
C spectral response of AIRS channel 1
C=======================================================================
      subroutine airs1(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.380, wmx=0.460
      integer,  parameter :: mwv= 35
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data (sr  (i),i=  1, 35) /
     $  0.005128, 0.002886, 0.001374, 0.003698, 0.006864,
     $  0.013090, 0.033231, 0.076092, 0.161704, 0.245375,
     $  0.259199, 0.267823, 0.274717, 0.286149, 0.298826,
     $  0.309475, 0.317824, 0.317501, 0.310042, 0.307051,
     $  0.292431, 0.294837, 0.338277, 0.363464, 0.327798,
     $  0.239252, 0.149011, 0.085511, 0.046337, 0.024886,
     $  0.014469, 0.009347, 0.007380, 0.005831, 0.004215 /

       wmin=wmn
       wmax=wmx
       nnf=mwv

       do 10 i=1,nnf
         srr(i)=sr(i)
 10   continue
       return
       end

C=======================================================================
C spectral response of AIRS channel 2
C=======================================================================
      subroutine airs2(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.520, wmx=0.700
      integer,  parameter :: mwv= 75
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax

      data (sr  (i),i=    1,   50) / 
     $  0.000000, 0.000533, 0.000945, 0.001361, 0.001783,
     $  0.002211, 0.002644, 0.003083, 0.003527, 0.003976,
     $  0.004142, 0.004254, 0.004366, 0.004958, 0.007938,
     $  0.010861, 0.018275, 0.027527, 0.047194, 0.072460,
     $  0.112121, 0.173171, 0.264447, 0.374261, 0.510745,
     $  0.652248, 0.738061, 0.773845, 0.777518, 0.778838,
     $  0.784899, 0.792303, 0.796726, 0.793400, 0.789225,
     $  0.789243, 0.791918, 0.794664, 0.793183, 0.782235,
     $  0.774498, 0.763391, 0.764142, 0.767907, 0.782163,
     $  0.794344, 0.804426, 0.810698, 0.811115, 0.809902 /

      data (sr  (i),i= 51, 75) / 
     $  0.815325, 0.826189, 0.836451, 0.852932, 0.852806,
     $  0.844644, 0.833519, 0.828699, 0.841141, 0.851687,
     $  0.851055, 0.846426, 0.825965, 0.713892, 0.432026,
     $  0.216361, 0.114826, 0.063494, 0.038689, 0.021043,
     $  0.011658, 0.006644, 0.004261, 0.001918, 0.000000 /
       wmin=wmn
       wmax=wmx
       nnf=mwv

       do 10 i=1,nnf
         srr(i)=sr(i)
 10   continue
       return
       end

C=======================================================================
C spectral response of AIRS channel 3
C=======================================================================
      subroutine airs3(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.670, wmx=0.975
      integer,  parameter :: mwv=125
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
      
      data (sr  (i),i=    1,   50) / 
     $     0.000000, 0.003379, 0.003411, 0.003444, 0.003477,
     $     0.004078, 0.006951, 0.011480, 0.016599, 0.023960,
     $     0.036043, 0.057332, 0.088698, 0.148032, 0.220794,
     $     0.325841, 0.444009, 0.547685, 0.660726, 0.775603,
     $     0.853849, 0.906058, 0.918434, 0.916672, 0.907973,
     $     0.901806, 0.908660, 0.918596, 0.928232, 0.935273,
     $     0.937799, 0.940109, 0.942381, 0.944376, 0.947208,
     $     0.956354, 0.963515, 0.969355, 0.973644, 0.977545,
     $     0.974620, 0.968519, 0.960392, 0.952298, 0.945408,
     $     0.938639, 0.932385, 0.930490, 0.928068, 0.923987 /
      
      data (sr  (i),i=   51,  100) / 
     $     0.917406, 0.908386, 0.898606, 0.890075, 0.880939,
     $     0.875101, 0.873721, 0.875880, 0.879767, 0.885273,
     $     0.890469, 0.894408, 0.887393, 0.879201, 0.866884,
     $     0.850730, 0.835656, 0.824285, 0.815378, 0.807416,
     $     0.804449, 0.803807, 0.804861, 0.805892, 0.807177,
     $     0.807003, 0.804723, 0.795822, 0.784649, 0.776168,
     $     0.765594, 0.752811, 0.744292, 0.737139, 0.735197,
     $     0.731744, 0.726148, 0.715571, 0.707805, 0.702938,
     $     0.697320, 0.691714, 0.684362, 0.676938, 0.667892,
     $     0.656787, 0.648466, 0.641009, 0.634255, 0.626658 /
      
      data (sr  (i),i=101,125) / 
     $     0.620257, 0.611595, 0.600326, 0.591389, 0.580594,
     $     0.565131, 0.546977, 0.530590, 0.514660, 0.504639,
     $     0.502949, 0.509015, 0.509009, 0.469097, 0.392224,
     $     0.264232, 0.148339, 0.070440, 0.035891, 0.016855,
     $     0.009114, 0.005118, 0.003445, 0.002607, 0.000000 /
      wmin=wmn
      wmax=wmx
      nnf=mwv
      
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
      return
      end
      
C=======================================================================
C spectral response of AIRS channel 4
C=======================================================================
      subroutine airs4(srr,wmin,wmax,nnf)
      use params, only: kr
      implicit none
      real(kr), parameter :: wmn=0.415, wmx=1.110
      integer,  parameter :: mwv=259
      integer :: i,nnf
      real(kr) :: srr(*),sr(mwv),wmin,wmax
      
      data (sr  (i),i=    1,   50) / 
     $     0.000000, 0.000835, 0.001808, 0.002801, 0.003818,
     $     0.004859, 0.008237, 0.014668, 0.028408, 0.053174,
     $     0.226002, 0.461725, 0.491061, 0.458212, 0.422702,
     $     0.417594, 0.427578, 0.449909, 0.483016, 0.512304,
     $     0.533860, 0.546248, 0.556111, 0.554623, 0.553001,
     $     0.554188, 0.555123, 0.562048, 0.570187, 0.581164,
     $     0.593501, 0.603637, 0.611861, 0.620140, 0.628474,
     $     0.636258, 0.639973, 0.643660, 0.647318, 0.650875,
     $     0.657180, 0.664420, 0.671698, 0.679299, 0.688091,
     $     0.696974, 0.705916, 0.713593, 0.720423, 0.727282 /
      
      data (sr  (i),i=   51,  100) / 
     $     0.734167, 0.736732, 0.739088, 0.741418, 0.743194,
     $     0.743984, 0.744725, 0.745639, 0.746794, 0.747931,
     $     0.749048, 0.750125, 0.753838, 0.762325, 0.770839,
     $     0.778745, 0.785975, 0.792448, 0.798838, 0.805255,
     $     0.811696, 0.818163, 0.819922, 0.821466, 0.822827,
     $     0.821900, 0.820401, 0.818876, 0.816191, 0.812171,
     $     0.808835, 0.805440, 0.805978, 0.807495, 0.808469,
     $     0.809383, 0.810284, 0.811171, 0.813025, 0.817150,
     $     0.821234, 0.825328, 0.829432, 0.837509, 0.845904,
     $     0.854338, 0.862465, 0.870015, 0.876968, 0.883365 /
      
      data (sr  (i),i=  101,  150) / 
     $     0.889779, 0.896213, 0.901125, 0.904839, 0.908558,
     $     0.912286, 0.913276, 0.913860, 0.914441, 0.914464,
     $     0.913303, 0.912134, 0.911242, 0.910351, 0.909405,
     $     0.908314, 0.906574, 0.904577, 0.903073, 0.901843,
     $     0.900614, 0.899385, 0.898151, 0.896913, 0.895868,
     $     0.897468, 0.899067, 0.900667, 0.902265, 0.904212,
     $     0.906268, 0.908322, 0.910378, 0.912433, 0.915588,
     $     0.919173, 0.922757, 0.926341, 0.929925, 0.933505,
     $     0.934687, 0.935659, 0.936632, 0.936368, 0.935805,
     $     0.935232, 0.934783, 0.934669, 0.934546, 0.933097 /
      
      data (sr  (i),i=  151,  200) / 
     $     0.930430, 0.927784, 0.925141, 0.922344, 0.918725,
     $     0.913528, 0.908044, 0.902574, 0.897117, 0.892657,
     $     0.888827, 0.885007, 0.880852, 0.875619, 0.870432,
     $     0.865259, 0.860098, 0.854950, 0.849800, 0.844656,
     $     0.839525, 0.832753, 0.826716, 0.822859, 0.818949,
     $     0.814976, 0.809735, 0.804009, 0.798282, 0.792555,
     $     0.787259, 0.782719, 0.778161, 0.773586, 0.768983,
     $     0.764229, 0.759209, 0.754179, 0.743430, 0.734616,
     $     0.729730, 0.724829, 0.719912, 0.713143, 0.706277,
     $     0.698287, 0.688177, 0.680948, 0.674610, 0.667855 /
      
      data (sr  (i),i=  201,  250) / 
     $     0.659749, 0.651586, 0.641477, 0.630669, 0.622755,
     $     0.614839, 0.604736, 0.592914, 0.583120, 0.573417,
     $     0.561795, 0.547182, 0.534179, 0.521942, 0.495328,
     $     0.483003, 0.470702, 0.458425, 0.446218, 0.433655,
     $     0.419071, 0.405505, 0.392962, 0.370931, 0.358746,
     $     0.346591, 0.334465, 0.314971, 0.303614, 0.295342,
     $     0.285874, 0.273863, 0.254465, 0.243492, 0.235679,
     $     0.205100, 0.183823, 0.166805, 0.147939, 0.131689,
     $     0.119560, 0.108231, 0.097669, 0.087686, 0.078087,
     $     0.071138, 0.064213, 0.058244, 0.052492, 0.046142 /
      
      data (sr  (i),i=251,259) / 
     $     0.039644, 0.035735, 0.031886, 0.027426, 0.025916,
     $     0.022519, 0.018541, 0.006526, 0.000000 /
      wmin=wmn
      wmax=wmx
      nnf=mwv
      
      do 10 i=1,nnf
        srr(i)=sr(i)
 10   continue
      return
      end
c=======================================================================
      subroutine rdspec(file,wl,r,nn)
c
c purpose:     read user specified spectral input from files
c              filter.dat, albedo.dat and solar.dat
c input:
c  file        input file name
c  nn          maximum number of input values 
c              (an error is triggered if the number of 
c              values read from file exceeds nn)
c
c output:
c  wl          wavelength array
c  r           spectral array
c  nn          number of values read from file
c
      use params, only: kr
      implicit none
      character*(*) file
      character*(80) line
      integer :: nn, nnsv, i,j
      real(kr) :: wl(*),r(*)
c mfl
c     print *, 'in redspec()'     
      wl(1:nn)=0.
      nnsv=nn
      open(unit=13,file=file,status='old',form='formatted')

      do i=1, nn
         read(13,*,end=10) wl(i),r(i)
      end do

 10   continue

      close(unit=13)

      nn=count(wl(1:nn).ne.0.)
      
c mfl 
c     do i=1,nn
c       print *, file, wl(i),r(i)
c     enddo

      if(nn.gt.nnsv) then
c        print *,'ERROR in rdspec --- too many values read from ',file
        print *,'file should not specify more than ',nnsv,' values'
        stop
      endif

      if(wl(1).gt.wl(nn)) then  ! reverse if necessary
c        print *, 'Reversing wavelength array!!!!!!'
        wl(nn:1:-1)=wl(1:nn)
        r(nn:1:-1)=r(1:nn)
      endif

      return
      end
      
c============================================================
      subroutine zensun(iday,time,alat,alon,zenith,azimuth,solfac)
c
c   routine:      zensun
c  
c   purpose:  compute the solar zenith and azimuth angles and solar flux
c             multiplier for a given location, time and day.
c
c   input:
c     iday    day of year (used as a fraction of 365.24 days year)
c  
c     time    universal time in decimal hours
c  
c     alat    geographic latitude of point on earth's surface (degrees)
c  
c     alon    geographic longitude of point on earth's surface (degrees)
c  
c   output:
c  
c     zenith  solar zenith angle (degrees)
c  
c     azimuth solar azimuth measured clockwise from due north (degrees)
c
c     solfac  solar flux multiplier. solfac=1./rsun**2 
c             where rsun is the current earth-sun distance in 
c             astronomical units.  
c  
      use params, only: pi, kr
      implicit none
      
      integer :: nday(74), iday, i

      real(kr), parameter ::
     &     degpday=360./365.242, ! degree per day
     &     eccen=0.01671,        ! eccentricity of earth orbit
     &     dayph=2.              ! day of perihelion


      real(kr) :: eqt(74), dec(74), dtor, frac, eqtime, decang,
     &     sunlat, sunlon, t0, t1, p0, alon, alat, p1, zz, xx, yy,
     &     azimuth, zenith, rsun, solfac, dd, time

      data nday/
     &     1,   6,  11,  16,  21,  26,  31,  36,  41,
     &     46,  51,  56,  61,  66,  71,  76,  81,  86,
     &     91,  96, 101, 106, 111, 116, 121, 126, 131,
     &     136, 141, 146, 151, 156, 161, 166, 171, 176,
     &     181, 186, 191, 196, 201, 206, 211, 216, 221,
     &     226, 231, 236, 241, 246, 251, 256, 261, 266,
     &     271, 276, 281, 286, 291, 296, 301, 306, 311,
     &     316, 321, 326, 331, 336, 341, 346, 351, 356,
     &     361, 366/
      
      data eqt/
     &     -3.23, -5.49, -7.60, -9.48,-11.09,-12.39,-13.34,-13.95,
     &     -14.23,-14.19,-13.85,-13.22,-12.35,-11.26,-10.01, -8.64,
     &     -7.18, -5.67, -4.16, -2.69, -1.29, -0.02,  1.10,  2.05,
     &     2.80,  3.33,  3.63,  3.68,  3.49,  3.09,  2.48,  1.71,
     &     0.79, -0.24, -1.33, -2.41, -3.45, -4.39, -5.20, -5.84,
     &     -6.28, -6.49, -6.44, -6.15, -5.60, -4.82, -3.81, -2.60,
     &     -1.19,  0.36,  2.03,  3.76,  5.54,  7.31,  9.04, 10.69,
     &     12.20, 13.53, 14.65, 15.52, 16.12, 16.41, 16.36, 15.95,
     &     15.19, 14.09, 12.67, 10.93,  8.93,  6.70,  4.32,  1.86,
     &     -0.62, -3.23/
      
      data dec/
     &     -23.06,-22.57,-21.91,-21.06,-20.05,-18.88,-17.57,-16.13,
     &     -14.57,-12.91,-11.16, -9.34, -7.46, -5.54, -3.59, -1.62,
     &     0.36,  2.33,  4.28,  6.19,  8.06,  9.88, 11.62, 13.29,
     &     14.87, 16.34, 17.70, 18.94, 20.04, 21.00, 21.81, 22.47,
     &     22.95, 23.28, 23.43, 23.40, 23.21, 22.85, 22.32, 21.63,
     &     20.79, 19.80, 18.67, 17.42, 16.05, 14.57, 13.00, 11.33,
     &     9.60,  7.80,  5.95,  4.06,  2.13,  0.19, -1.75, -3.69,
     &     -5.62, -7.51, -9.36,-11.16,-12.88,-14.53,-16.07,-17.50,
     &     -18.81,-19.98,-20.99,-21.85,-22.52,-23.02,-23.33,-23.44,
     &     -23.35,-23.06/
      
cccc
c
c compute the subsolar coordinates
c
      dtor=pi/180.
      
      dd=mod(iday-1, 365)+1
      do 10 i=1,74
        if(nday(i) .gt. dd ) goto 20
 10   continue
 20   continue
      frac=(dd-nday(i-1))/(nday(i)-nday(i-1))
      eqtime=eqt(i-1)*(1.-frac)+frac*eqt(i)
      decang=dec(i-1)*(1.-frac)+frac*dec(i)
      sunlat=decang
      sunlon=-15.*(time-12.+eqtime/60.)
      
c      write(12,'(2i4,9es11.3)') i,nday(i),frac,eqtime,sunlat,sunlon
c      write(12,'(9es11.3)') eqt(i-1),eqt(i),eqtime
c      write(12,'(9es11.3)') dec(i-1),dec(i),decang
c
c compute the solar zenith, azimuth and flux multiplier
c
      t0=(90.-alat)*dtor                            
      t1=(90.-sunlat)*dtor                         
      p0=alon*dtor                                  
      p1=sunlon*dtor                               
      zz=cos(t0)*cos(t1)+sin(t0)*sin(t1)*cos(p1-p0) 
c      write(12,'(9es11.3)') t0,t1,p0,p1,zz
c      write(12,'(9es11.3)') cos(t0),cos(t1),sin(t0),sin(t1),cos(p1-p0)
      xx=sin(t1)*sin(p1-p0)
      yy=sin(t0)*cos(t1)-cos(t0)*sin(t1)*cos(p1-p0)
      azimuth=atan2(xx,yy)/dtor
      zenith=acos(zz)/dtor                         
c
      rsun=1.-eccen*cos(degpday*(dd-dayph)*dtor)      
      solfac=1./rsun**2                              
      return
      end
      
      
c=======================================================================

c file:                  tauaero.f
c
c
c external routines:     tauaero,relhum
c
c required routines:     locate, numset
c
c internal routines:     aeroden,aerzstd,phaerw,aerbwi,aestrat,
c                        usraer,stdaer, aervint,aeread

c
c=======================================================================
      module aeroblk

      use params, only: mxly, maxmom, zip, kr
      implicit none

      integer, parameter ::
     &     naerz=5,             ! number of stratospheric aerosol layers
     &     naerb=150,           ! max number of BLA wavelengths
     &     naerw=47,            ! number of wavelengths in standard models
     &     naerwm=naerb*maxmom  ! max pmaer buffer size


      ! quantities set in main program:

      real(kr) ::
     &     zaer(naerz)=0. ,     ! stratospheric aerosol layer altitudes
     &     taerst(naerz)=0.,    ! stratospheric aerosol layer optical depth
     &     zbaer(mxly)=zip,     ! user definded layer altitudes (BLA only)
     &     dbaer(mxly)=zip,     ! user defined density profile (BLA only)
     &     vis=zip,             ! visibility at 0.55um (BLA only)
     &     tbaer=zip,           ! BLA optical depth at 0.55um
     &     abaer=0.,            ! BLA lambda exponent 
     &     wlbaer(naerb)=zip,   ! BLA wavelengths 
     &     qbaer(naerb)=zip,    ! BLA extinction efficiency
     &     wbaer(naerb)=zip,    ! BLA single scattering albedo
     &     gbaer(naerb)=zip,    ! BLA asymmetry factor
     &     pmaer(naerwm)=zip,   ! BLA phase function moments
     &     rhaer=zip            ! relative humidity (BLA only)

      integer ::
     &     iaer=0,              ! BLA aerosol type 
     &     imoma=3,             ! flag to control use of pmaer (BLA only)
     &     jaer(naerz)=0        ! stratospheric aerosol types

      ! the following quantities not accessed by main program

      integer ::
     &     nzbaer,              ! number of user specified BLA levels
     &     ndbaer,              ! number of user specified BLA densities
     &     nwlbaer=naerw,       ! number of user specified BLA wavelengths
     &     npmaer               ! number of user specified BLA pmaer moments

      real(kr), dimension(naerb) ::
     &     aerext(naerb),       ! BLA extinction
     &     aerabs(naerb),       ! BLA absorption
     &     aerasm(naerb)        ! BLA asymmetry

      real(kr), dimension(naerw) :: ! wavelengths for standard BLA and strat models
     &     awl=(/
     &   .2000,   .3000,   .3371,   .5500,   .6943,  1.0600,  1.5360,
     &  2.0000,  2.2500,  2.5000,  2.7000,  3.0000,  3.3923,  3.7500,
     &  4.5000,  5.0000,  5.5000,  6.0000,  6.2000,  6.5000,  7.2000,
     &  7.9000,  8.2000,  8.7000,  9.0000,  9.2000, 10.0000, 10.5910,
     & 11.0000, 11.5000, 12.5000, 14.8000, 15.0000, 16.4000, 17.2000,
     & 18.5000, 21.3000, 25.0000, 30.0000, 40.0000, 50.0000, 60.0000,
     & 80.0000, 100.000, 150.000, 200.000, 300.000/)
                  
      contains
c-----------------------------------------------------------------------

      subroutine aerzstd
c
c purpose 
c       copy standard vertical profile into zbaer and dbaer and change
c       nzbaer to number of elements in standard profile
c
c the standard vertical distributions of the boundary layer aerosol 
c density is based on the 5s vertical profile models for 5 and 23 km 
c visibility.
c
c above 5 km, the aden05 and aden23 models are the same 
c below 5 km, the models differ as follows;
c aden05     0.99 km scale height (94% of extinction occurs below 5 km)
c aden23     1.45 km scale height (80% of extinction occurs below 5 km)
c    

      use params, only: one, zero

      implicit none

      integer, parameter ::  mz=33 ! number of standard BLA layer altitudes
      real(kr), dimension(mz) :: alt=(/
     &        0.0,        1.0,        2.0,        3.0,        4.0,  
     &        5.0,        6.0,        7.0,        8.0,        9.0,
     &       10.0,       11.0,       12.0,       13.0,       14.0, 
     &       15.0,       16.0,       17.0,       18.0,       19.0,
     &       20.0,       21.0,       22.0,       23.0,       24.0, 
     &       25.0,       30.0,       35.0,       40.0,       45.0,
     &       50.0,       70.0,      100.0/)
        real(kr), dimension(mz) :: aden05=(/
     &  1.378E+04,  5.030E+03,  1.844E+03,  6.731E+02,  2.453E+02,      
     &  8.987E+01,  6.337E+01,  5.890E+01,  6.069E+01,  5.818E+01,
     &  5.675E+01,  5.317E+01,  5.585E+01,  5.156E+01,  5.048E+01,
     &  4.744E+01,  4.511E+01,  4.458E+01,  4.314E+01,  3.634E+01,
     &  2.667E+01,  1.933E+01,  1.455E+01,  1.113E+01,  8.826E+00,
     &  7.429E+00,  2.238E+00,  5.890E-01,  1.550E-01,  4.082E-02,
     &  1.078E-02,  5.550E-05,  1.969E-08/)
        real(kr), dimension(mz) :: aden23=(/
     &  2.828E+03,  1.244E+03,  5.371E+02,  2.256E+02,  1.192E+02,
     &  8.987E+01,  6.337E+01,  5.890E+01,  6.069E+01,  5.818E+01,
     &  5.675E+01,  5.317E+01,  5.585E+01,  5.156E+01,  5.048E+01,
     &  4.744E+01,  4.511E+01,  4.458E+01,  4.314E+01,  3.634E+01,
     &  2.667E+01,  1.933E+01,  1.455E+01,  1.113E+01,  8.826E+00,
     &  7.429E+00,  2.238E+00,  5.890E-01,  1.550E-01,  4.082E-02,
     &  1.078E-02,  5.550E-05,  1.969E-08/)

      integer :: i
      real(kr) :: wtv

      if(mz.gt.mxly) then
        print *,'mz gt mxly in aerzstd'
        stop
      endif

      nzbaer=mz

      if(vis.le.0.) then
        wtv=(1./vis-1/5.)/(1./23.-1./5.)
      else
        wtv=1.                  ! default to 23 km vertical profile
      endif
      wtv=max(zero,min(one,wtv))

      do i=1,mz
        zbaer(i)=alt(i)
        dbaer(i)=aden05(i)*(1.-wtv)+aden23(i)*wtv
      enddo
      return
      end subroutine aerzstd
      
c=======================================================================
      subroutine phaerw(w,p)
c
c purpose:
c   Interpolate phase function moments to wavelength w.
c   Must use linear interpolation because moments can be pos or neg.
c   if w>wmax then p(w)=ptable(wmax)
c   if w<wmin then p(w)=ptable(wmin)
c
c input
c   w      wavelength in microns
c
c output
c   p      legendre moments of scattering phase function (0:nstrms)
c          due to boundary layer aerosol
c
      use params, only: one, zero,kr
      implicit none
      integer :: l,j,i
      real(kr) :: wt,w,p(*)

      call locate(wlbaer,nwlbaer,w,l)
      wt=(w-wlbaer(l))/(wlbaer(l+1)-wlbaer(l))
      wt=max(zero,min(wt,one))     ! no extrapolation

      do i=1,npmaer
        j=l+(i-1)*nwlbaer
        p(i)=pmaer(j)*(1.-wt)+pmaer(j+1)*wt
      enddo

      return
      end subroutine phaerw

c=======================================================================
      subroutine aerbwi(wl,extinc,wa,ga)
c 
c purpose
c     At a given wavelength find the extinction efficiency,
c     single scattering albedo and asymmetry factor of boundary
c     layer aerosols by interpolation
c
c input
c  wl         wavelength
c 
c output
c  extinc     extinction efficiency
c  wa         single scattering albedo
c  ga         asymmetry factor

      use params, only: one, zero, kr
      implicit none

      real(kr),intent(in) :: wl
      real(kr),intent(out) :: extinc,wa,ga
      real(kr) :: wt,absorp

      integer :: l
      character*(80) errmes(10)
c
      LOGICAL, parameter :: debug=.false.


c . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

      IF (debug) write(*,*) 'db_aerbwi.1',wl

      if(iaer.eq.0) then   ! no boundary layer aerosol
        extinc=0.
        wa=0.
        ga=0.
        return
      endif        

c     print '(a/(10es11.3))','wlbaer',wlbaer(1:nwlbaer)
c     print '(a/(10es11.3))','aerext',aerext(1:nwlbaer)
c     stop !!!!!!!!!!!!!!!!!!!!

      call locate(wlbaer,nwlbaer,wl,l)

      if(wl.le.wlbaer(1)) then 
        extinc=aerext(1)*(wlbaer(1)/wl)**abaer
        wa=1.-(aerabs(1)/aerext(1))
        ga=aerasm(1)
      elseif(wl.ge.wlbaer(nwlbaer)) then 
        extinc=aerext(nwlbaer)*(wlbaer(nwlbaer)/wl)**abaer
        wa=1.-(aerabs(nwlbaer)/aerext(nwlbaer))
        ga=aerasm(nwlbaer)
      else
        wt=log(wl/wlbaer(l))/log(wlbaer(l+1)/wlbaer(l))

        extinc=aerext(l)*(aerext(l+1)/aerext(l))**wt
        if(aerabs(l).gt.0. .and. aerabs(l+1).gt.0. ) then 
          absorp=aerabs(l)*(aerabs(l+1)/aerabs(l))**wt
        else
          absorp=aerabs(l)*(1.-wt)+aerabs(l+1)*wt
        endif
        if(extinc.gt.0.) wa=max(zero,min(one-absorp/extinc,one))
        ga=(1.-wt)*aerasm(l)+wt*aerasm(l+1)  
        ! linear interpolation because aerasm can be negative
      endif
      !write(*,*) 'wlbaer: ',wlbaer(1),wlbaer(2)
      !write(*,*) 'extinc: ',extinc
      !write(*,*) 'aerext: ',aerext(1),aerext(2)
      !write(*,*) 'wl,l ',wl,l

      return
      end subroutine aerbwi

c=======================================================================

      subroutine aestrat(ja,wl,qa,wa,ga)

c 
c purpose
c     compute extinction efficiency, single scattering albedo and
c     asymmetry factor of STRATOSPHERIC aerosols by interpolation
c
c input
c  ja         stratospheric aerosol type
c  wl         wavelength
c 
c output
c  qa         extinction efficiency of stratospheric aerosol of type ja
c  wa         single scattering albedo
c  ga         asymmetry factor
c

      use params, only: zero, one, kr
      implicit none
      real(kr) :: wl,wt,qa,wa,ga,absorp
      integer :: l,ja
      real(kr), dimension(naerw,3,4) :: aerstr

                                ! background extinction
      data aerstr(1:naerw,1,1) /       
     & 2.0752e0,  1.8656e0,  1.7246e0,  1.0000e0,  7.0156e-1, 3.0170e-1,
     & 1.1440e-1, 5.1225e-2, 3.4285e-2, 2.3475e-2, 1.6878e-2, 6.6506e-2,
     & 1.0943e-1, 8.9653e-2, 6.7609e-2, 5.2855e-2, 6.7496e-2, 5.7975e-2,
     & 4.2471e-2, 2.4176e-2, 4.6102e-2, 1.2339e-1, 1.7699e-1, 1.2389e-1,
     & 9.0220e-2, 8.5793e-2, 3.2838e-2, 2.6528e-2, 5.0703e-2, 1.9471e-2,
     & 1.1710e-2, 1.6106e-2, 1.7716e-2, 3.9533e-2, 3.7954e-2, 5.4871e-3,
     & 8.8409e-3, 1.2289e-3, 1.0647e-3, 3.3151e-3, 4.5164e-3, 4.1496e-3,
     & 3.2801e-3, 2.4481e-3, 1.0714e-3, 5.0381e-4, 1.2101e-4/
                                ! background absorption
      data aerstr(1:naerw,2,1) /
     & 2.4347e-7, 1.4949e-7, 1.3020e-7, 6.8716e-8, 1.0053e-7, 4.2384e-6,
     & 2.3075e-4, 1.4889e-3, 1.8377e-3, 3.3645e-3, 4.6858e-3, 5.9424e-2,
     & 1.0129e-1, 8.2396e-2, 6.3621e-2, 5.0327e-2, 6.5798e-2, 5.6149e-2,
     & 4.0939e-2, 2.3226e-2, 4.5800e-2, 1.2276e-1, 1.7568e-1, 1.2216e-1,
     & 8.9058e-2, 8.4861e-2, 3.1979e-2, 2.6097e-2, 5.0244e-2, 1.9042e-2,
     & 1.1497e-2, 1.6024e-2, 1.7639e-2, 3.9452e-2, 3.7847e-2, 5.4251e-3,
     & 8.8160e-3, 1.2118e-3, 1.0579e-3, 3.3122e-3, 4.5150e-3, 4.1489e-3,
     & 3.2799e-3, 2.4481e-3, 1.0713e-3, 5.0381e-4, 1.2101e-4/
                                ! background asymmetry
      data aerstr(1:naerw,3,1) /
     & 0.6749, 0.6943, 0.6991, 0.6846, 0.6572, 0.5861, 0.4965, 0.4164,
     & 0.3772, 0.3385, 0.3069, 0.2599, 0.2234, 0.2028, 0.1554, 0.1291,
     & 0.1057, 0.0962, 0.0909, 0.0802, 0.0595, 0.0458, 0.0413, 0.0479,
     & 0.0483, 0.0451, 0.0504, 0.0379, 0.0346, 0.0365, 0.0273, 0.0180,
     & 0.0174, 0.0144, 0.0151, 0.0141, 0.0094, 0.0075, 0.0049, 0.0031,
     & 0.0020, 0.0014, 0.0008, 0.0005, 0.0002, 0.0001, 0.0001/
                                ! aged volcanic extinction
      data aerstr(1:naerw,1,2) /
     & 1.14880, 1.19171, 1.18013, 1.00, .84873, .53019, .27968, .14551,
     &  .11070, .08633, .07184, .06076, .04506, .03399, .02095, .01538,
     &  .01266, .01019, .00994, .01044, .01361, .01791, .02278, .02918,
     &  .03108, .03234, .03456, .03184, .02772, .02475, .01715, .01563,
     &  .01665, .01646, .01734, .01772, .01076, .01051, .01133, .01329,
     &  .01492, .01277, .00766, .00562, .00318, .00231, 0.0/
                                ! aged volcanic absorption
      data aerstr(1:naerw,2,2) /
     &  .44816, .11259, .08500, .05272, .04082, .02449, .01487, .01019,
     &  .00867, .00842, .00842, .00949, .00741, .00487, .00316, .00335,
     &  .00399, .00449, .00525, .00665, .01114, .01652, .02177, .02437,
     &  .02506, .02658, .03006, .02861, .02513, .02285, .01620, .01532,
     &  .01633, .01620, .01709, .01741, .01057, .01038, .01127, .01329,
     &  .01492, .01277, .00766, .00562, .00318, .00231, 0.0/
                                ! aged volcanic asymmetry
      data aerstr(1:naerw,3,2) /
     &  .8272, .7148, .7076, .6978, .6886, .6559, .6062, .5561, .5255,
     &  .4958, .4729, .4401, .4015, .3699, .3125, .2773, .2472, .2173,
     &  .2054, .1908, .1623, .1348, .1233, .1615, .1757, .1712, .1521,
     &  .1326, .1230, .1081, .0801, .0528, .0514, .0461, .0446, .0449,
     &  .0415, .0330, .0198, .0097, .0044, .0032, .0020, .0013, .0006,
     &  .0004, .0000/
                                ! fresh volcanic extinction
      data aerstr(1:naerw,1,3) /
     &  .88715, .92532, .94013, 1.0000, 1.03013, 1.05975, 1.01171,
     &  .88677, .82538, .76361, .71563, .67424, .60589, .55057, .45222,
     &  .37646, .32316, .25519, .22728, .20525, .17810, .14481, .14152,
     &  .37639, .44551, .44405, .42222, .36462, .32551, .27519, .16728,
     &  .10627, .10861, .10886, .11665, .13127, .10108, .08557, .06411,
     &  .05741, .05531, .04707, .02792, .02028, .01136, .00820, 0.0/
                                ! fresh volcanic absorption
      data aerstr(1:naerw,2,3) /
     &  .41582, .22892, .19108, .14468, .12475, .09158, .06601, .04943,
     &  .04367, .04342, .04399, .05076, .04133, .02829, .01924, .01981,
     &  .02297, .02475, .02778, .03411, .05335, .07133, .08816, .15342,
     &  .18506, .19354, .20791, .18449, .16101, .13759, .08456, .06886,
     &  .07278, .07367, .07956, .08785, .06032, .05747, .05133, .05323,
     &  .05453, .04657, .02773, .02020, .01135, .00820, 0.0/
                                ! fresh volcanic asymmetry
      data aerstr(1:naerw,3,3) /
     &  .9295, .8115, .7897, .7473, .7314, .7132, .7113, .7238, .7199,
     &  .7165, .7134, .6989, .6840, .6687, .6409, .6325, .6199, .6148,
     &  .6142, .6072, .5853, .5632, .5486, .4753, .4398, .4329, .4091,
     &  .4105, .4120, .4136, .4140, .3637, .3577, .3344, .3220, .3052,
     &  .2957, .2564, .2055, .1229, .0632, .0483, .0321, .0216, .0103,
     &  .0059, .0000/
                                ! meteoric extinction
      data aerstr(1:naerw,1,4) /

     & 1.05019, 1.05880, 1.05259, 1.00, .94949, .81456, .66051, .54380,
     &  .49133, .44677, .41671, .38063, .34778, .32804, .29722, .27506,
     &  .25082, .22620, .21652, .20253, .17266, .14905, .14234, .14082,
     &  .15057, .16399, .23608, .24481, .27791, .25076, .15272, .09601,
     &  .09456, .14576, .12373, .18348, .12190, .12924, .08538, .04108,
     &  .04714, .04069, .02480, .01789, .00980, .00693, 0.0/
                                ! meteoric absorption
      data aerstr(1:naerw,2,4) /
     &  .00063, .00152, .00184, .00506, .00791, .01829, .03728, .06158,
     &  .07538, .08943, .10051, .11614, .13310, .14348, .14633, .13728,
     &  .12462, .11184, .10709, .10076, .09006, .08734, .09000, .10304,
     &  .11905, .13437, .19551, .20095, .22494, .18418, .09285, .06665,
     &  .06823, .12329, .10551, .16184, .09835, .10582, .06759, .03247,
     &  .04405, .03816, .02327, .01696, .00946, .00677, 0.0/
                                ! meteoric asymmetry
      data aerstr(1:naerw,3,4) /
     &  .7173, .7039, .7020, .6908, .6872, .6848, .6891, .6989, .7046,
     &  .7099, .7133, .7159, .7134, .7058, .6827, .6687, .6583, .6513,
     &  .6494, .6475, .6467, .6496, .6506, .6461, .6334, .6177, .5327,
     &  .5065, .4632, .4518, .5121, .5450, .5467, .4712, .4853, .3984,
     &  .4070, .3319, .3427, .3766, .3288, .2969, .2808, .2661, .2409,
     &  .2098, .0000/

      call locate(awl,naerw,wl,l)
      if(wl.le.awl(1)) then 
        qa=aerstr(1,1,ja)*(awl(1)/wl)**abaer
        wa=1.-(aerstr(1,2,ja)/aerstr(1,1,ja))
        ga=aerstr(l,3,ja)
      elseif(wl.ge.awl(naerw)) then 
        qa=aerstr(naerw,1,ja)*(awl(1)/wl)**abaer
        wa=1.-(aerstr(naerw,2,ja)/aerstr(naerw,1,ja))
        ga=aerstr(naerw,3,ja)
      else
        wt=log(wl/awl(l))/log(awl(l+1)/awl(l))
        qa=aerstr(l,1,ja)*(aerstr(l+1,1,ja)/aerstr(l,1,ja))**wt
        absorp=aerstr(l,2,ja)*(aerstr(l+1,2,ja)/aerstr(l,2,ja))**wt
        if(qa.gt.0.) wa=max(zero,min(one-absorp/qa,one))
        ga=(1.-wt)*aerstr(l,3,ja)+wt*aerstr(l+1,3,ja)  
        ! linear interpolation because aerasm can be negative
      endif
      !write(*,*) 'awl: ',awl(1),awl(2)
      !write(*,*) 'qa: ',qa
      !write(*,*) 'aerext: ',aerext(1),aerext(2)
      !write(*,*) 'wl,l ',wl,l

      return
      end subroutine aestrat

c=======================================================================

      subroutine usraer(q55)
c
c purpose 
c       Replaces data in wlbaer,aerext,aerabs,aerasm with user specified
c       aerosol spectral data.  
c
c       A similar function is performed by subroutine STDAER, which
c       fills the same variables with spectral properties of standard
c       aerosol models
c
c input:
c
c  wlbaer    wavelengths at which aerosol model is specified
c  qbaer    aerosol extinction efficiency
c  wbaer    aerosol single scattering albedo
c  gbaer    aerosol asymmetry factor
c  abaer    power law slope used to extrapolate optical depth
c           outside of wlbaer range  tauaer=(wlbaer(j)/wl)**abaer
c           where j is 1 or n (n = total number of assigned values
c           of wlbaer). 
c  pmaer    user defined aerosol phase function (may be undefined)
c
c input/output:
c
c  q55      extinction efficiency at 0.55 um
c
c output:
c
c  imoma    >0 => aerosol phase function from subroutine getmom
c            0 => user defined aerosol phase function
c

c check input parameters

      use params, only: wl55, zero, kr

      implicit none
      integer, external :: numset
      integer :: nqbaer, nwbaer, ngbaer, ne, i, j
      real(kr) :: f,q55
      character (len=80) :: errmes(10)

      nwlbaer=numset(zip,wlbaer,naerb)
      nqbaer=numset(zip,qbaer,naerb)
      nwbaer=numset(zip,wbaer,naerb)
      ngbaer=numset(zip,gbaer,naerb)
      npmaer=numset(zip,pmaer,naerwm)

      ne=0

      if(nwlbaer.eq.0) then         ! wavelength points not set

        qbaer(1)=1.
        nqbaer=1
        if(nwbaer.ne.1) then
          ne=ne+1
          errmes(ne)='specify one value of wbaer when wlbaer not set'
        endif

      elseif(nwlbaer.eq.1) then     ! only one wavelength point

        if(nqbaer.gt.1) then 
          ne=ne+1
          errmes(ne)='number of elements must match: wlbaer, qbaer'
        elseif(nqbaer.eq.0) then
          qbaer(1)=1.
        endif          
        if(nwbaer.ne.1) then
          ne=ne+1
          errmes(ne)='number of elements must match: wlbaer, wbaer'
        endif

      else                          ! many wavelength points

        if(nwlbaer.ne.nqbaer) then
          ne=ne+1
          errmes(ne)='number of elements must match: wlbaer, qbaer'
        endif
        if(nwlbaer.ne.nwbaer) then
          ne=ne+1
          errmes(ne)='number of elements must match: wlbaer, wbaer'
        endif
        if(ngbaer.eq.0) then
          if(npmaer.ge.1) then 
            if(mod(npmaer,nwlbaer).ne.0) then
              ne=ne+1
              errmes(ne)='incorrect number of phase function moments'
            else
              npmaer=npmaer/nwlbaer
            endif
          else
            ne=ne+1
            errmes(ne)='must specify either gbaer or pmaer'
          endif
        else
          if(ngbaer.ne.nwbaer) then
            ne=ne+1
            errmes(ne)='number of elements must match: wlbaer, gbaer'
          endif
        endif
      
      endif

      if((ngbaer.eq.0).eqv.(npmaer.eq.0).and.imoma.eq.3) then 
        ne=ne+1
        errmes(ne)='must specify either gbaer or pmaer, not both'
      endif

      if(ne.gt.0) then 
        write(*,*) 'Error in user specified aerosols (iaer=5)'
        write(*,'(/,1x,5a8)')
     &          'nwlbaer','nqbaer','nwbaer','ngbaer','npmaer'
        write(*,'(5i8,/)')  nwlbaer,nqbaer,nwbaer,ngbaer,npmaer
        do i=1,ne
          write(*,'(2a)') 'Error in USRAER -- ',errmes(i)
        enddo
        stop
      endif
      
c duplicate input to two wavelength points if a 
c single wavelength specified in the input file

      !write(*,*) 'db usraer 1'
      if(nwbaer.eq.1) then  
        if(nwlbaer.eq.0) wlbaer(1)=wl55
        wlbaer(2)=2*wlbaer(1)
        nwlbaer=2
        qbaer(2)=qbaer(1)*(wlbaer(1)/wlbaer(2))**abaer
        wbaer(2)=wbaer(1)
        gbaer(2)=gbaer(1)

c duplicate phase function entries

        !write(*,*) 'db usraer 2'
        if(npmaer.gt.0) then
          pmaer(2:2*npmaer:2)=pmaer(1:npmaer)
          pmaer(1:2*npmaer-1:2)=pmaer(2:2*npmaer:2)
        endif
      endif

      !write(*,'(a10,/,10es11.3)')  'wlbaer',(wlbaer(i),i=1,2)
      !write(*,'(a10,/,10es11.3)')  'qbaer',(qbaer(i),i=1,2)
        
      !write(*,*) 'db usraer 3'
      if(nwlbaer.gt.naerb) then
        write(*,*) 'Too many spectral points in usraer'
        stop
      endif
      do i=1,nwlbaer 
        wlbaer(i)=wlbaer(i)
        aerext(i)=qbaer(i)
        aerabs(i)=(1.-wbaer(i))*aerext(i)
        aerasm(i)=gbaer(i)
      enddo

      !write(*,*) 'db usraer 4'
      !write(*,'(a10/10es11.3)')  'wlbaer',(wlbaer(i),i=1,nwlbaer)
      !write(*,'(a10/10es11.3)')  'aerext',(aerext(i),i=1,nwlbaer)
      !write(*,'(a10/10es11.3)')  'aerabs',(aerabs(i),i=1,nwlbaer)
      !write(*,'(a10/10es11.3)')  'aerasm',(aerasm(i),i=1,nwlbaer)
      !write(*,*) 'pmaer'
      !do j=1,npmaer
      !  write(*,'(10es11.3)') (pmaer((j-1)*nwlbaer+i),i=1,nwlbaer)
      !enddo
        
      call locate(wlbaer,nwlbaer,wl55,j)

      !write(*,*) 'db usraer 5'
      !write(*,*) 'wlbaer ',j,wlbaer(j),wlbaer(j+1)
      f=log(wl55/wlbaer(j))/log(wlbaer(j+1)/wlbaer(j))
      !write(*,*) 'f ',f
      if(wl55.lt.wlbaer(1)) then
        !write(*,*) 'db usraer 5.1',abaer
        q55=qbaer(1)*(wlbaer(1)/wl55)**abaer
      elseif(wl55.gt.wlbaer(nwlbaer)) then
        !write(*,*) 'db usraer 5.2',abaer
        q55=qbaer(nwlbaer)*(wlbaer(nwlbaer)/wl55)**abaer
      else
        !write(*,*) 'db usraer 5.3',qbaer(j),f
        q55=qbaer(j)*(qbaer(j+1)/qbaer(j))**f
      endif

c if defined, save aerosol phase function for use in phaerw

      if(npmaer.gt.0) imoma=0

      return

      end subroutine usraer

c===========================================================================
      subroutine stdaer(humid)
c
c purpose 
c   Replaces data in wlbaer,aerext,aerabs,aerasm with standard
c   extinction, absorption, and asymmetry parameters for boundary
c   layer aerosols.  The rural, urban, oceanic or tropospheric
c   aerosols models are interpolated on relative humidity.  
c
c   A similar function is performed by entry USRAER, which fills the
c   same variables with spectral properties of user defined aerosol model

c
c input:
c
c   humid    relative humidity at the surface (0.0 - 1.0)
c
      
      use params, only: wl55, zero, one, kr
      implicit none

      integer :: j,i

      real(kr) :: rhzone(4)=(/0.,.7,.8,.99/),
     &        tiny=.00000001,
     &        rhum,humid,wt,ex1=0.0_kr,ex2=0.0_kr,
     &        ab1=0.0_kr,ab2=0.0_kr,g1=0.0_kr,g2=0.0_kr

      real(kr), dimension(naerw,4) ::
     &     rure,rura,rurg,      ! rural (four humidity values)
     &     urbe,urba,urbg,      ! urban
     &     ocne,ocna,ocng,      ! oceanic
     &     troe,troa,trog       ! tropospheric

c
c
c   altitude regions for aerosol extinction coefficients
c
c         aerosol extinction normalized to one at 0.55 microns
c
c         0-2km
c           rure,rura,rurg=rural extinction, absorption, asymmetry
c           urbe,urba,urbg=urban extinction, absorption, asymmetry
c           ocne,ocna,ocng=maritime extinction, absorption, asymmetry
c           troe,troa,trog=tropspheric extinction,absorption, asymmetry
c           fg1ext=fog1 .2km vis extinction  fg1abs=fog1 absorption
c           fg1sym=fog1 asymmetry factors
c           fg2ext=fog2 .5km vis extinction  fg2abs=fog2 absorption
c           fg2sym=fog2 asymmetry factors
c         >2-10km
c           troext=tropospher extinction  troabs=tropospher absorption
c           trosym=tropospheric asymmetry factors
c         >10-30km
c           bstext=background stratospheric extinction
c           bstabs=background stratospheric absorption
c           bstsym=background stratospheric asymmetry factors
c           avoext=aged volcanic extinction
c           avoabs=aged volcanic absorption
c           avosym=aged volcanic asymmetry factors
c           fvoext=fresh volcanic extinction
c           fvoabs=fresh volcanic absorption
c           fvosym=fresh volcanic asymmetry factors
c         >30-100km
c           dmeext=meteoric dust extinction
c           dmeabs=meteoric dust absorption
c           dmesym=meteoric dust asymmetry factors
c
c     aerosol extinction and absorption data
c
c     modified to include asymmetry data - jan 1986 (a.e.r. inc.)
c
      data rure /
     & 2.09291, 1.74582, 1.60500, 1.00000,  .75203,  .41943,  .24070,
     &  .14709,  .13304,  .12234,  .13247,  .11196,  .10437,  .09956,
     &  .09190,  .08449,  .07861,  .07025,  .07089,  .07196,  .07791,
     &  .04481,  .04399,  .12184,  .12658,  .12829,  .09152,  .08076,
     &  .07456,  .06880,  .06032,  .04949,  .05854,  .06000,  .06962,
     &  .05722,  .06051,  .05177,  .04589,  .04304,
     &  .03582,  .03155,  .02018,  .01469,  .00798,  .00551, 0.,

     & 2.09544, 1.74165, 1.59981, 1.00000,  .75316,  .42171,  .24323,
     &  .15108,  .13608,  .12430,  .13222,  .13823,  .11076,  .10323,
     &  .09475,  .08728,  .08076,  .07639,  .07797,  .07576,  .07943,
     &  .04899,  .04525,  .12165,  .12741,  .12778,  .09032,  .07962,
     &  .07380,  .06880,  .06329,  .05791,  .06646,  .06639,  .07443,
     &  .06304,  .06443,  .05538,  .04867,  .04519,
     &  .03821,  .03374,  .02173,  .01587,  .00862,  .00594, 0.,

     & 2.07082, 1.71456, 1.57962, 1.00000,  .76095,  .43228,  .25348,
     &  .16456,  .14677,  .13234,  .13405,  .20316,  .12873,  .11506,
     &  .10481,  .09709,  .08918,  .09380,  .09709,  .08791,  .08601,
     &  .06247,  .05601,  .11905,  .12595,  .12348,  .08741,  .07703,
     &  .07266,  .07044,  .07443,  .08146,  .08810,  .08563,  .08962,
     &  .08051,  .07677,  .06658,  .05747,  .05184,
     &  .04572,  .04074,  .02689,  .01981,  .01084,  .00714, 0.,

     & 1.66076, 1.47886, 1.40139, 1.00000,  .80652,  .50595,  .32259,
     &  .23468,  .20772,  .18532,  .17348,  .35114,  .20006,  .17386,
     &  .16139,  .15424,  .14557,  .16215,  .16766,  .14994,  .14032,
     &  .12968,  .12601,  .13551,  .13582,  .13228,  .11070,  .09994,
     &  .09873,  .10418,  .13241,  .15924,  .16139,  .15949,  .15778,
     &  .15184,  .13848,  .12563,  .11076,  .09601,
     &  .09312,  .08720,  .06644,  .05264,  .03181,  .02196, 0.0/

      data rura /
     &  .67196,  .11937,  .08506,  .05930,  .05152,  .05816,  .05006,
     &  .01968,  .02070,  .02101,  .05652,  .02785,  .01316,  .00867,
     &  .01462,  .01310,  .01627,  .02013,  .02165,  .02367,  .03538,
     &  .02823,  .03962,  .06778,  .07285,  .08120,  .04032,  .03177,
     &  .02557,  .02342,  .02177,  .02627,  .03943,  .03114,  .03696,
     &  .02956,  .03500,  .03241,  .03297,  .03380,
     &  .03170,  .02794,  .01769,  .01305,  .00730,  .00518, 0.0,

     &  .62968,  .10816,  .07671,  .05380,  .04684,  .05335,  .04614,
     &  .01829,  .01899,  .01962,  .05525,  .06816,  .01652,  .00867,
     &  .01544,  .01373,  .01627,  .02892,  .02829,  .02532,  .03487,
     &  .02835,  .03854,  .06684,  .07272,  .08038,  .03987,  .03247,
     &  .02816,  .02816,  .03101,  .03741,  .04829,  .04032,  .04399,
     &  .03734,  .03956,  .03601,  .03525,  .03563,
     & .03357,  .02965,  .01887,  .01395,  .00782,  .00555, 0.0,

     &  .51899,  .08278,  .05816,  .04082,  .03570,  .04158,  .03620,
     &  .01513,  .01481,  .01633,  .05278,  .13690,  .02494,  .00886,
     &  .01804,  .01582,  .01677,  .04816,  .04367,  .03013,  .03443,
     &  .02930,  .03677,  .06209,  .06911,  .07475,  .03892,  .03494,
     &  .03513,  .03968,  .05152,  .06241,  .06937,  .06203,  .06215,
     &  .05614,  .05209,  .04608,  .04196,  .04095,
     &  .03916,  .03486,  .02262,  .01686,  .00951,  .00674, 0.0,

     &  .21943,  .02848,  .01943,  .01342,  .01171,  .01437,  .01323,
     &  .01152,  .00696,  .01329,  .06108,  .24690,  .05323,  .01430,
     &  .03361,  .02949,  .02652,  .09437,  .08506,  .05348,  .04627,
     &  .04380,  .04557,  .05380,  .05715,  .05899,  .04861,  .05253,
     &  .06171,  .07437,  .10152,  .12019,  .12190,  .11734,  .11411,
     &  .10766,  .09487,  .08430,  .07348,  .06861,
     &  .06936,  .06458,  .04735,  .03761,  .02313,  .01668, 0.0/

      data rurg /
     &  .7581,   .6785,   .6712,   .6479,   .6342,   .6176,   .6334,
     &  .7063,   .7271,   .7463,   .7788,   .7707,   .7424,   .7312,
     &  .7442,   .7516,   .7662,   .7940,   .7886,   .7797,   .7664,
     &  .8525,   .8700,   .5846,   .5570,   .5992,   .6159,   .6271,
     &  .6257,   .6374,   .6546,   .6861,   .6859,   .6120,   .5570,
     &  .5813,   .5341,   .5284,   .5137,   .4348,   .4223,   .3775,
     &  .3435,   .3182,   .2791,   .2494,   .0000,

     &  .7632,   .6928,   .6865,   .6638,   .6498,   .6314,   .6440,
     &  .7098,   .7303,   .7522,   .7903,   .7804,   .7380,   .7319,
     &  .7508,   .7584,   .7738,   .8071,   .7929,   .7843,   .7747,
     &  .8507,   .8750,   .6112,   .5851,   .6272,   .6466,   .6616,
     &  .6653,   .6798,   .6965,   .7026,   .6960,   .6360,   .5848,
     &  .6033,   .5547,   .5445,   .5274,   .4518,   .4318,   .3863,
     &  .3516,   .3257,   .2853,   .2548,   .0000,

     &  .7725,   .7240,   .7197,   .6997,   .6858,   .6650,   .6702,
     &  .7181,   .7378,   .7653,   .8168,   .7661,   .7286,   .7336,
     &  .7654,   .7735,   .7910,   .8303,   .8025,   .7957,   .7946,
     &  .8468,   .8734,   .6831,   .6619,   .6994,   .7250,   .7449,
     &  .7547,   .7665,   .7644,   .7265,   .7170,   .6769,   .6409,
     &  .6442,   .6031,   .5854,   .5646,   .4977,   .4602,   .4127,
     &  .3751,   .3476,   .3048,   .2721,   .0000,

     &  .7778,   .7793,   .7786,   .7717,   .7628,   .7444,   .7365,
     &  .7491,   .7609,   .7921,   .8688,   .7537,   .7294,   .7413,
     &  .7928,   .8016,   .8225,   .8761,   .8359,   .8285,   .8385,
     &  .8559,   .8654,   .8414,   .8415,   .8527,   .8740,   .8903,
     &  .8952,   .8923,   .8611,   .8033,   .7989,   .7758,   .7632,
     &  .7508,   .7314,   .7091,   .6867,   .6419,   .5790,   .5259,
     &  .4749,   .4415,   .3886,   .3489,   .0000/

      data urbe /
     & 1.88816, 1.63316, 1.51867, 1.00000,  .77785,  .47095,  .30006,
     &  .21392,  .19405,  .17886,  .18127,  .16133,  .14785,  .14000,
     &  .12715,  .11880,  .11234,  .10601,  .10500,  .10361,  .10342,
     &  .08766,  .08652,  .11937,  .12139,  .12297,  .09797,  .09057,
     &  .08595,  .08196,  .07563,  .06696,  .07209,  .06842,  .07177,
     &  .06354,  .06177,  .05373,  .04728,  .04051,
     &  .03154,  .02771,  .01759,  .01278,  .00693,  .00480, 0.0,

     & 1.95582, 1.64994, 1.53070, 1.00000,  .77614,  .46639,  .29487,
     &  .21051,  .18943,  .17285,  .17209,  .21418,  .15354,  .14051,
     &  .12728,  .11861,  .11089,  .11329,  .11323,  .10563,  .10247,
     &  .08696,  .08361,  .12013,  .12418,  .12304,  .09614,  .08842,
     &  .08487,  .08285,  .08361,  .08430,  .08880,  .08449,  .08601,
     &  .07835,  .07323,  .06367,  .05500,  .04747,
     &  .03901,  .03454,  .02240,  .01638,  .00891,  .00612, 0.0,

     & 1.96430, 1.64032, 1.52392, 1.00000,  .77709,  .46253,  .28690,
     &  .20310,  .17981,  .16101,  .15614,  .26475,  .15456,  .13563,
     &  .12215,  .11361,  .10500,  .11715,  .11753,  .10392,  .09766,
     &  .08443,  .08057,  .10943,  .11342,  .11063,  .08703,  .08025,
     &  .07886,  .08032,  .09101,  .10070,  .10386,  .09943,  .09886,
     &  .09152,  .08247,  .07152,  .06089,  .05253,
     &  .04582,  .04091,  .02717,  .02008,  .01103,  .00754, 0.0,

     & 1.41266, 1.33816, 1.29114, 1.00000,  .83646,  .55025,  .35342,
     &  .25285,  .21576,  .18310,  .16215,  .37854,  .20494,  .16665,
     &  .14778,  .13892,  .12943,  .15525,  .15709,  .13513,  .12481,
     &  .11759,  .11494,  .11487,  .11329,  .11108,  .09911,  .09209,
     &  .09342,  .10120,  .13177,  .15696,  .15766,  .15513,  .15203,
     &  .14532,  .13038,  .11785,  .10411,  .09101,
     &  .08907,  .08399,  .06579,  .05337,  .03372,  .02379, 0.0/

      data urba /
     &  .78437,  .58975,  .54285,  .36184,  .29222,  .20886,  .15658,
     &  .12329,  .11462,  .10747,  .11797,  .10025,  .08759,  .08184,
     &  .07506,  .07006,  .06741,  .06601,  .06544,  .06449,  .06665,
     &  .06278,  .06949,  .07316,  .07462,  .08101,  .05753,  .05272,
     &  .04899,  .04734,  .04494,  .04443,  .05133,  .04348,  .04443,
     &  .03994,  .03981,  .03633,  .03468,  .03146,
     &  .02809,  .02471,  .01556,  .01145,  .00639,  .00454, 0.0,

     &  .69032,  .49367,  .45165,  .29741,  .24070,  .17399,  .13146,
     &  .10354,  .09589,  .09025,  .10411,  .15101,  .07880,  .06949,
     &  .06570,  .06095,  .05829,  .07171,  .06797,  .05975,  .06013,
     &  .05589,  .06051,  .07139,  .07494,  .07956,  .05525,  .05184,
     &  .05089,  .05291,  .05886,  .06380,  .06880,  .06127,  .06019,
     &  .05525,  .05070,  .04500,  .04076,  .03741,
     &  .03400,  .03010,  .01926,  .01427,  .00800,  .00567, 0.0,

     &  .54848,  .37101,  .33734,  .21949,  .17785,  .12968,  .09854,
     &  .07804,  .07165,  .06791,  .08563,  .19639,  .06722,  .05316,
     &  .05316,  .04886,  .04620,  .07570,  .06899,  .05291,  .05101,
     &  .04734,  .05025,  .06171,  .06570,  .06854,  .04892,  .04797,
     &  .05057,  .05665,  .07127,  .08095,  .08411,  .07728,  .07475,
     &  .06886,  .06019,  .05222,  .04538,  .04171,
     &  .03911,  .03486,  .02271,  .01697,  .00961,  .00681, 0.0,

     &  .15975,  .10000,  .09013,  .05785,  .04671,  .03424,  .02633,
     &  .02525,  .01975,  .02354,  .06241,  .26690,  .05810,  .02285,
     &  .03810,  .03386,  .03044,  .09627,  .08557,  .05405,  .04576,
     &  .04392,  .04424,  .04671,  .04791,  .04861,  .04684,  .05177,
     &  .06158,  .07475,  .10342,  .12146,  .12177,  .11734,  .11335,
     &  .10608,  .09171,  .08063,  .06968,  .06475,
     &  .06559,  .06131,  .04591,  .03714,  .02365,  .01734, 0.0/

      data urbg /
     &  .7785,   .7182,   .7067,   .6617,   .6413,   .6166,   .6287,
     &  .6883,   .7070,   .7243,   .7370,   .7446,   .7391,   .7371,
     &  .7414,   .7435,   .7466,   .7543,   .7498,   .7424,   .7270,
     &  .7674,   .7850,   .5880,   .5616,   .5901,   .6159,   .6238,
     &  .6240,   .6281,   .6306,   .6298,   .6252,   .5785,   .5378,
     &  .5512,   .5072,   .4930,   .4709,   .4009,   .4110,   .3672,
     &  .3344,   .3093,   .2717,   .2426,   .0000,

     &  .7906,   .7476,   .7385,   .6998,   .6803,   .6536,   .6590,
     &  .7066,   .7258,   .7484,   .7769,   .7405,   .7351,   .7459,
     &  .7625,   .7673,   .7759,   .7910,   .7732,   .7703,   .7644,
     &  .7966,   .8142,   .6635,   .6428,   .6700,   .6935,   .7050,
     &  .7092,   .7145,   .7094,   .6762,   .6684,   .6316,   .5997,
     &  .6013,   .5625,   .5433,   .5198,   .4552,   .4387,   .3928,
     &  .3575,   .3310,   .2899,   .2588,   .0000,

     &  .7949,   .7713,   .7650,   .7342,   .7162,   .6873,   .6820,
     &  .7131,   .7312,   .7583,   .8030,   .7171,   .7185,   .7400,
     &  .7698,   .7778,   .7923,   .8142,   .7864,   .7867,   .7891,
     &  .8147,   .8298,   .7276,   .7136,   .7361,   .7590,   .7729,
     &  .7783,   .7808,   .7624,   .7094,   .7022,   .6714,   .6480,
     &  .6417,   .6104,   .5887,   .5651,   .5058,   .4692,   .4212,
     &  .3825,   .3549,   .3112,   .2778,   .0000,

     &  .7814,   .7993,   .7995,   .7948,   .7870,   .7682,   .7751,
     &  .7501,   .7565,   .7809,   .8516,   .7137,   .7039,   .7241,
     &  .7728,   .7846,   .8093,   .8576,   .8125,   .8140,   .8304,
     &  .8472,   .8549,   .8525,   .8569,   .8640,   .8853,   .9017,
     &  .9061,   .9021,   .8685,   .8126,   .8091,   .7897,   .7802,
     &  .7691,   .7550,   .7353,   .7146,   .6754,   .6134,   .5601,
     &  .5056,   .4701,   .4134,   .3714,   .0000/

      data ocne /
     & 1.47576, 1.32614, 1.26171, 1.00000,  .88133,  .70297,  .56487,
     &  .46006,  .42044,  .38310,  .35076,  .42266,  .32278,  .28810,
     &  .24905,  .21184,  .16734,  .14791,  .21532,  .15076,  .12057,
     &  .10038,  .10703,  .15070,  .15665,  .14639,  .10228,  .08367,
     &  .07373,  .06829,  .05044,  .04373,  .04962,  .06158,  .07703,
     &  .07234,  .06297,  .05481,  .05329,  .08741,
     &  .04608,  .03959,  .02382,  .01712,  .00936,  .00665, 0.0,

     & 1.36924, 1.25443, 1.20835, 1.00000,  .91367,  .77089,  .64987,
     &  .54886,  .50247,  .45038,  .38209,  .50589,  .43766,  .38076,
     &  .31658,  .27475,  .22215,  .21019,  .27570,  .21057,  .16949,
     &  .14209,  .14215,  .16956,  .17082,  .16025,  .11665,  .09759,
     &  .09215,  .09373,  .10532,  .12570,  .13000,  .13633,  .14291,
     &  .13506,  .11475,  .09658,  .08291,  .10348,
     &  .06693,  .05786,  .03522,  .02519,  .01358,  .00954, 0.0,

     & 1.22259, 1.14627, 1.11842, 1.00000,  .94766,  .87538,  .80418,
     &  .72930,  .68582,  .62165,  .49962,  .67949,  .66468,  .59253,
     &  .49551,  .44671,  .37886,  .35924,  .43367,  .37019,  .30842,
     &  .26437,  .25228,  .24905,  .23975,  .22766,  .17804,  .15316,
     &  .15373,  .16791,  .22361,  .28348,  .28677,  .29082,  .29038,
     &  .27810,  .23867,  .20209,  .16430,  .14943,
     &  .12693,  .11177,  .07095,  .05084,  .02690,  .01838, 0.0,

     & 1.09133, 1.06601, 1.05620, 1.00000,  .97506,  .94791,  .94203,
     &  .93671,  .92867,  .90411,  .80253,  .89222,  .94462,  .92146,
     &  .85797,  .82595,  .76747,  .68646,  .78209,  .75266,  .68658,
     &  .62722,  .60228,  .56335,  .53728,  .51861,  .43449,  .37196,
     &  .35899,  .37316,  .46854,  .58234,  .58690,  .60348,  .60563,
     &  .60000,  .55392,  .50367,  .43576,  .35949,
     &  .34729,  .32254,  .23600,  .17953,  .10071,  .06714, 0.0/

      data ocna /
     &  .30987,  .04354,  .02880,  .01797,  .01468,  .01766,  .01582,
     &  .00816,  .01146,  .01677,  .03310,  .03380,  .00715,  .00443,
     &  .00500,  .00601,  .00753,  .01595,  .02943,  .00994,  .01367,
     &  .01671,  .02538,  .03481,  .03405,  .03601,  .01608,  .01310,
     &  .01152,  .01082,  .01070,  .01563,  .02063,  .03171,  .03810,
     &  .03741,  .03804,  .03759,  .04209,  .07892,
     &  .04347,  .03754,  .02269,  .01649,  .00917,  .00657, 0.0,

     &  .23367,  .03127,  .02070,  .01297,  .01063,  .01285,  .01190,
     &  .00937,  .00911,  .01576,  .05576,  .23487,  .03949,  .00905,
     &  .02057,  .01816,  .01665,  .08025,  .08044,  .03677,  .03139,
     &  .03190,  .03766,  .04532,  .04544,  .04715,  .03405,  .03614,
     &  .04329,  .05424,  .07823,  .09728,  .10057,  .10247,  .10222,
     &  .09551,  .08241,  .07158,  .06506,  .09203,
     &  .06133,  .05332,  .03258,  .02366,  .01308,  .00932, 0.0,

     &  .13025,  .01557,  .01013,  .00646,  .00532,  .00665,  .00722,
     &  .01335,  .00728,  .01810,  .09835,  .37329,  .09703,  .01968,
     &  .05114,  .04342,  .03709,  .17456,  .16468,  .08785,  .06880,
     &  .06589,  .06791,  .07247,  .07329,  .07449,  .07025,  .07962,
     &  .09899,  .12481,  .17867,  .22019,  .22228,  .22051,  .21595,
     &  .20335,  .17278,  .14677,  .12171,  .12430,
     &  .10890,  .09644,  .06106,  .04465,  .02457,  .01732, 0.0,

     &  .03506,  .00323,  .00215,  .00139,  .00114,  .00171,  .00532,
     &  .03082,  .01101,  .03741,  .20101,  .47608,  .21165,  .05234,
     &  .12886,  .11215,  .09684,  .32810,  .31778,  .20513,  .16658,
     &  .15956,  .15842,  .15905,  .15968,  .16051,  .16506,  .18323,
     &  .21709,  .25652,  .33222,  .39639,  .39854,  .40297,  .40025,
     &  .39025,  .35468,  .32006,  .27715,  .25348,
     &  .25632,  .23876,  .17092,  .13198,  .07692,  .05407, 0.0/

      data ocng /
     &  .7516,   .6960,   .6920,   .6756,   .6767,   .6844,   .6936,
     &  .7055,   .7110,   .7177,   .7367,   .6287,   .6779,   .6784,
     &  .6599,   .6659,   .6859,   .6887,   .6095,   .6558,   .6665,
     &  .6697,   .6594,   .5851,   .5644,   .5760,   .5903,   .5991,
     &  .6024,   .5979,   .6087,   .5837,   .5763,   .5348,   .4955,
     &  .4821,   .4635,   .4373,   .3944,   .2344,   .2754,   .2447,
     &  .2266,   .2088,   .1766,   .1481,   .0000,

     &  .7708,   .7288,   .7243,   .7214,   .7211,   .7330,   .7445,
     &  .7579,   .7649,   .7790,   .8182,   .7673,   .7171,   .7205,
     &  .7235,   .7251,   .7397,   .7537,   .6934,   .7137,   .7193,
     &  .7206,   .7151,   .6732,   .6620,   .6696,   .6821,   .6895,
     &  .6898,   .6819,   .6556,   .5925,   .5869,   .5511,   .5284,
     &  .5124,   .4912,   .4646,   .4302,   .3124,   .3101,   .2752,
     &  .2529,   .2335,   .2021,   .1738,   .0000,

     &  .7954,   .7782,   .7752,   .7717,   .7721,   .7777,   .7872,
     &  .8013,   .8089,   .8301,   .8844,   .8332,   .7557,   .7597,
     &  .7823,   .7822,   .7944,   .8157,   .7712,   .7738,   .7784,
     &  .7807,   .7800,   .7682,   .7659,   .7692,   .7780,   .7828,
     &  .7776,   .7621,   .7115,   .6342,   .6294,   .5999,   .5854,
     &  .5700,   .5512,   .5265,   .4996,   .4236,   .3765,   .3357,
     &  .3066,   .2830,   .2466,   .2184,   .0000,

     &  .8208,   .8270,   .8260,   .8196,   .8176,   .8096,   .8096,
     &  .8202,   .8255,   .8520,   .9228,   .8950,   .7965,   .7847,
     &  .8242,   .8244,   .8376,   .8857,   .8463,   .8332,   .8379,
     &  .8441,   .8467,   .8502,   .8534,   .8562,   .8688,   .8789,
     &  .8785,   .8683,   .8252,   .7562,   .7519,   .7261,   .7141,
     &  .6980,   .6789,   .6540,   .6294,   .5783,   .5100,   .4595,
     &  .4164,   .3868,   .3404,   .3042,   .0000/

      data troe /
     & 2.21222, 1.82753, 1.67032, 1.00000,  .72424,  .35272,  .15234,
     &  .05165,  .03861,  .02994,  .04671,  .02462,  .01538,  .01146,
     &  .01032,  .00816,  .00861,  .00994,  .01057,  .01139,  .01747,
     &  .01494,  .02418,  .03165,  .03386,  .04247,  .01601,  .01215,
     &  .00937,  .00861,  .00823,  .01139,  .01924,  .01234,  .01348,
     &  .01114,  .01297,  .01266,  .01418,  .01487,
     &  .01543,  .01321,  .00793,  .00582,  .00330,  .00239, 0.0,

     & 2.21519, 1.82266, 1.66557, 1.00000,  .72525,  .35481,  .15449,
     &  .05475,  .04044,  .03082,  .04620,  .05272,  .01867,  .01266,
     &  .01127,  .00886,  .00886,  .01449,  .01399,  .01228,  .01728,
     &  .01475,  .02285,  .03215,  .03494,  .04285,  .01652,  .01304,
     &  .01101,  .01120,  .01297,  .01753,  .02468,  .01741,  .01766,
     &  .01513,  .01557,  .01456,  .01532,  .01582,
     &  .01619,  .01386,  .00832,  .00610,  .00346,  .00251, 0.0,

     & 2.19082, 1.79462, 1.64456, 1.00000,  .73297,  .36443,  .16278,
     &  .06468,  .04658,  .03399,  .04538,  .11892,  .02835,  .01646,
     &  .01386,  .01076,  .00968,  .02551,  .02222,  .01468,  .01690,
     &  .01437,  .01994,  .03127,  .03513,  .04076,  .01722,  .01513,
     &  .01519,  .01791,  .02538,  .03272,  .03816,  .03038,  .02886,
     &  .02551,  .02228,  .01937,  .01804,  .01791,
     &  .01798,  .01539,  .00924,  .00678,  .00384,  .00278, 0.0,

     & 1.75696, 1.54829, 1.45962, 1.00000,  .77816,  .43139,  .21778,
     &  .11329,  .08101,  .05506,  .04943,  .25291,  .06816,  .03703,
     &  .02601,  .01968,  .01468,  .04962,  .04247,  .02234,  .01797,
     &  .01532,  .01633,  .02259,  .02487,  .02595,  .01728,  .01892,
     &  .02399,  .03247,  .05285,  .06462,  .06608,  .05930,  .05525,
     &  .04861,  .03753,  .02968,  .02348,  .02165,
     &  .02152,  .01841,  .01104,  .00809,  .00458,  .00332, 0.0/

      data troa /
     &  .69671,  .09905,  .06563,  .04101,  .03354,  .03627,  .02810,
     &  .00873,  .00918,  .00930,  .03215,  .01285,  .00513,  .00316,
     &  .00557,  .00494,  .00646,  .00867,  .00937,  .01025,  .01646,
     &  .01481,  .02418,  .02886,  .03070,  .04032,  .01494,  .01139,
     &  .00873,  .00816,  .00797,  .01133,  .01911,  .01215,  .01329,
     &  .01101,  .01291,  .01266,  .01418,  .01487,
     &  .01543,  .01321,  .00793,  .00582,  .00330,  .00239, 0.0,

     &  .65000,  .08791,  .05816,  .03652,  .02994,  .03278,  .02557,
     &  .00810,  .00842,  .00867,  .03139,  .03949,  .00646,  .00316,
     &  .00595,  .00519,  .00646,  .01304,  .01247,  .01095,  .01620,
     &  .01449,  .02278,  .02930,  .03184,  .04063,  .01544,  .01234,
     &  .01044,  .01076,  .01272,  .01741,  .02462,  .01722,  .01747,
     &  .01506,  .01551,  .01456,  .01532,  .01582,
     &  .01619,  .01386,  .00832,  .00610,  .00346,  .00251, 0.0,

     &  .52804,  .06367,  .04158,  .02633,  .02184,  .02443,  .01937,
     &  .00658,  .00646,  .00709,  .02949,  .10013,  .00968,  .00310,
     &  .00677,  .00582,  .00646,  .02361,  .01994,  .01266,  .01544,
     &  .01386,  .01968,  .02848,  .03203,  .03854,  .01620,  .01449,
     &  .01462,  .01747,  .02513,  .03253,  .03797,  .03019,  .02861,
     &  .02538,  .02215,  .01930,  .01797,  .01791,
     &  .01797,  .01539,  .00924,  .00677,  .00384,  .00278, 0.0,

     &  .19829,  .01842,  .01215,  .00791,  .00665,  .00778,  .00652,
     &  .00361,  .00253,  .00399,  .02570,  .20690,  .01715,  .00316,
     &  .00873,  .00728,  .00658,  .04481,  .03525,  .01646,  .01405,
     &  .01310,  .01468,  .01956,  .02184,  .02367,  .01608,  .01816,
     &  .02342,  .03203,  .05234,  .06399,  .06538,  .05867,  .05456,
     &  .04810,  .03715,  .02949,  .02335,  .02158,
     &  .02149,  .01840,  .01104,  .00809,  .00458,  .00332, 0.0/

      data trog /
     &  .7518,   .6710,   .6638,   .6345,   .6152,   .5736,   .5280,
     &  .4949,   .4700,   .4467,   .4204,   .4028,   .3777,   .3563,
     &  .3150,   .2919,   .2695,   .2465,   .2402,   .2313,   .2101,
     &  .1760,   .1532,   .2091,   .2079,   .1843,   .1811,   .1687,
     &  .1626,   .1526,   .1356,   .1030,   .0962,   .1024,   .1086,
     &  .0928,   .0836,   .0643,   .0451,   .0290,   .0156,   .0118,
     &  .0076,   .0050,   .0024,   .0015,   .0000,

     &  .7571,   .6858,   .6790,   .6510,   .6315,   .5887,   .5418,
     &  .5075,   .4829,   .4598,   .4338,   .4043,   .3890,   .3680,
     &  .3259,   .3026,   .2800,   .2541,   .2494,   .2414,   .2196,
     &  .1873,   .1657,   .2123,   .2110,   .1890,   .1836,   .1709,
     &  .1640,   .1534,   .1354,   .1044,   .0984,   .1026,   .1073,
     &  .0935,   .0842,   .0661,   .0477,   .0309,   .0171,   .0129,
     &  .0084,   .0056,   .0027,   .0017,   .0000,

     &  .7667,   .7176,   .7128,   .6879,   .6690,   .6255,   .5769,
     &  .5403,   .5167,   .4947,   .4703,   .4143,   .4190,   .3993,
     &  .3563,   .3325,   .3095,   .2767,   .2751,   .2693,   .2464,
     &  .2175,   .1992,   .2247,   .2215,   .2042,   .1952,   .1814,
     &  .1726,   .1604,   .1398,   .1111,   .1065,   .1068,   .1086,
     &  .0984,   .0888,   .0724,   .0549,   .0358,   .0216,   .0166,
     &  .0109,   .0073,   .0036,   .0023,   .0000,

     &  .7696,   .7719,   .7710,   .7606,   .7478,   .7142,   .6727,
     &  .6381,   .6201,   .6050,   .5912,   .4849,   .5137,   .5019,
     &  .4625,   .4389,   .4169,   .3696,   .3707,   .3708,   .3473,
     &  .3232,   .3112,   .3022,   .2938,   .2850,   .2675,   .2494,
     &  .2347,   .2165,   .1857,   .1536,   .1509,   .1441,   .1416,
     &  .1354,   .1245,   .1088,   .0905,   .0614,   .0440,   .0354,
     &  .0257,   .0179,   .0089,   .0059,   .0000/

c
      if(iaer.le.0.) then
        wlbaer(1)=0
        return
      endif
    
      abaer=0.                       ! angstrom exponent      

      rhum=max(zero,min(one,humid))
      if(rhum.lt.rhzone(2)) then
        j=1
      elseif(rhum.lt.rhzone(3)) then
        j=2
      else
        j=3
      endif
      
      wt=(rhum-rhzone(j))/(rhzone(j+1)-rhzone(j))      
      
      nwlbaer=naerw

      do 20 i=1,naerw
        wlbaer(i)=awl(i)
        if(abs(iaer).eq.1) then
          ex1=max(rure(i,j),tiny)
          ex2=max(rure(i,j+1),tiny)
          ab1=max(rura(i,j),tiny)
          ab2=max(rura(i,j+1),tiny)
          g1=max(rurg(i,j),tiny)
          g2=max(rurg(i,j+1),tiny)
        elseif(abs(iaer).eq.2) then
          ex1=max(urbe(i,j),tiny)
          ex2=max(urbe(i,j+1),tiny)
          ab1=max(urba(i,j),tiny)
          ab2=max(urba(i,j+1),tiny)
          g1=max(urbg(i,j),tiny)
          g2=max(urbg(i,j+1),tiny)
        elseif(abs(iaer).eq.3) then
          ex1=max(ocne(i,j),tiny)
          ex2=max(ocne(i,j+1),tiny)
          ab1=max(ocna(i,j),tiny)
          ab2=max(ocna(i,j+1),tiny)
          g1=max(ocng(i,j),tiny)
          g2=max(ocng(i,j+1),tiny)
        elseif(abs(iaer).eq.4) then
          ex1=max(troe(i,j),tiny)
          ex2=max(troe(i,j+1),tiny)
          ab1=max(troa(i,j),tiny)
          ab2=max(troa(i,j+1),tiny)
          g1=max(trog(i,j),tiny)
          g2=max(trog(i,j+1),tiny)
        endif
        if(abs(iaer).ge.1.and.abs(iaer).le.4) then
          aerext(i)=ex1*(ex2/ex1)**wt
          aerabs(i)=ab1*(ab2/ab1)**wt
          aerasm(i)=g1*(g2/g1)**wt
        endif
 20   continue

c        write(*,1000) 'wlbaer:',wlbaer
c        write(*,1000) 'aerext:',aerext
c        write(*,1000) 'aerabs:',aerabs
c        write(*,1000) 'aerasm:',aerasm
 1000 format(a/5(10es11.3,/))
      return
      end subroutine stdaer
c
c-----------------------------------------------------------------------

      end module aeroblk

c=======================================================================
      function aeroden(zz)

c purpose:   find number density of boundary layer aerosols, aeroden,
c            at a given altitude, zz, and for a specified visibility
c input:
c   zz       altitude   (km)
c
c output:
c            aerosol density at altitude zz
c

      use params, only: zero,one,kr
      use aeroblk, only: dbaer, zbaer, nzbaer
      implicit none

      integer :: k,kp

      real(kr) :: zz,z,f,aeroden
      real(kr), parameter :: z1=0., z2=100.

      z=max(z1,min(z2,zz))
      aeroden=0.
c
      if(z.gt.zbaer(nzbaer)) return
      call locate(zbaer,nzbaer,z,k)
      kp=k+1
      f=(z-zbaer(k))/(zbaer(kp)-zbaer(k))
      
      if(min(dbaer(k),dbaer(kp)).le.zero) then
        aeroden=max(dbaer(k)*(1.-f)+dbaer(kp)*f,0.0_kr)
      else
        aeroden=dbaer(k)*(dbaer(kp)/dbaer(k))**f
      endif
      
        !write(*,*) 'aeroden k,kp,z,dbaer(k),dbaer(kp),f,aeroden'
        !write(*,'(2i5,5es11.3)') k,kp,z,dbaer(k),dbaer(kp),f,aeroden
        
      RETURN
      end function aeroden

c=======================================================================
      subroutine tauaero(wl,nz,z,nosct,dtaua,waer,nmom,pmom,idb)
c
c purpose:  
c     compute the optical depth and scattering parameters at all
c     atmospheric levels of both boundary layer and stratospheric
c     aerosols
c
c input:
c   wl      wavelength
c   nz      number of atmospheric levels
c   z       altitude array (km)
c   idb     if 1 then print diagnostic
c   nosct   0 => normal scattering; 1 => no scattering assumption
C
c output:
c   dtaua   aerosol optical depth increment at each atmospheric layer
c   waer    single scatter albedo due to aerosols at each atmospheric layer
c
c input/output:
c   nmom   number of phase function moments
c   pmom   input:  (moments of phase function of cloud)*taucld*wcld
c          output: (moments of phase function clouds)*taucld*wcld +
c                  (moments of phase function aerosol)*tauaer*waer
c
c NOTE:     visibility is defined as 3.912/extinction_coefficient(km-1)
c           for a horizontal path 
c           (exp(-3.912) = 0.02 = visible contrast threshold)
c
c revisions:
c
c  971028: corrected expression for net asymmetry factor including both 
c          stratospheric and boundary layer aerosols (suggested by Xiang Li)
c
c  040428: corrected bug that prevented computation of stratospheric aerosols
c          when iaer=0
c
      use params, only: mxly, maxmom, zero, kr, zip
      use aeroblk, only: naerz, iaer, imoma, tbaer, jaer, zaer,
     &     taerst, npmaer, aestrat, phaerw, aerbwi

      implicit none

      real(kr), intent(in)    :: wl,z(*)
      real(kr), intent(out)   :: dtaua(*),waer(*)
      real(kr), intent(inout) :: pmom(0:maxmom,*)
      real(kr), save          :: dtsv(mxly)

      real(kr) :: dtauab(mxly),gaer(mxly),pm(0:maxmom),extinc,wa,ga,dt

      integer, intent(in) :: nz,nosct,nmom,idb
      integer             :: i,j,nl,ja,namom 
      integer, external   :: numset
      integer, save       :: init=1, laer(naerz)

      logical, parameter  :: debug=.false.

c - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

                                ! find stratospheric aerosol layers

      if(any(jaer(1:naerz).ne.0).and.init.eq.1) then
        call zlayer(nz,z,numset(zero,taerst,naerz),zaer,laer)
        if(debug) then
          write(*,'(a,5f10.4)') '      zaer:' ,zaer
          write(*,'(a,5i10)')   '      laer:' ,laer
        endif
      endif

      if(iaer.eq.0) then        ! no BL aerosols

        dtaua(1:nz)=0.
        dtauab(1:nz)=0.
        waer(1:nz)=0.
        gaer(1:nz)=0.

      elseif(iaer.eq.-1) then   ! read in BLA model from aerosol.dat

        gaer(1:nz)=pmom(1,1:nz)            ! save cloud effect
        call aeread(imoma,nz,wl,dtauab,waer,pmom)
        gaer(1:nz)=pmom(1,1:nz)-gaer(1:nz) ! subtract cloud effect 

      else                      ! use one of the standard BLA models

c integrate vertical profile of boundary layer aerosol density
c dtsv = sigma*n*dz, where sigma is geometric cross-section
c dtsv is independent of wavelenth and is is saved for later calls to tauaero
      
        if(init.eq.1) call denprfl(nz,z,dtsv)
        
c compute scattering parameters (extinction efficiency , single 
c scattering albedo and asymmetry factor) for boundary layer 
c aerosols at a given wavelength. Note that extinction efficiency
c is normalized to 1 at 550nm.
        
        if (debug) write(*,*) ' call aerbwi'
        call aerbwi(wl,extinc,wa,ga)
        
c nosct options:
c  1.  tauaer is absorption optical depth
c  2.  tauaer is extinction optical depth, ssa = 0
c  3.  tauaer is absorption + backscatter optical depth, ssa = 0

        if(nosct.eq.1) extinc = extinc*(1.-wa)
        if(nosct.eq.3) extinc = extinc*(1.-wa*ga)
        if (nosct.ne.0) then
          wa=0.
          ga=0.
        endif 
        
c use standard phasefunction if imoma is set

        if(imoma.gt.0) then
          namom=nmom
          call getmom(imoma,ga,nmom,pm(0))
        else
          namom=npmaer
          call phaerw(wl,pm(1))
        endif
c        print *,"z,dtauab,extinc,dtsv,wa,pm"
        do i=1,nz
          dtauab(i)=extinc*dtsv(i)
c          print '(6es11.3)', z(nz-i+1),dtauab(i),extinc,dtsv(i),wa,pm(1)
          waer(i)=wa
          gaer(i)=pm(1)*waer(i)*dtauab(i)
          do j=1,namom
            pmom(j,i)=pmom(j,i)+pm(j)*dtauab(i)*waer(i)
          enddo
        enddo
        
      endif                     ! end use one of the standard models

c      print *,'total optical depth ',sum(dtauab(1:nz))

c add in stratospheric aerosols at specified scattering layers

c      print *, "boundary layer"
      do i=1,nz
        dtaua(i)=dtauab(i)
      enddo

      do i=1,naerz

        if(jaer(i).ne.0 .and. taerst(i).gt.0.) then
          nl=laer(i) 
          ja=jaer(i)

c find scattering parameters for stratospheric
c aerosols at a given wavelength

          call aestrat(ja,wl,extinc,wa,ga)
          dt=taerst(i)*extinc
          
          call getmom(3,ga,nmom,pm(0))

          do j=1,nmom
            pmom(j,nl)=pmom(j,nl)+pm(j)*dt*wa
          enddo

          waer(nl)=(waer(nl)*dtaua(nl)+wa*dt)/(dtaua(nl)+dt)
          dtaua(nl)=dtaua(nl)+dt
          gaer(nl)=gaer(nl)+pm(1)*dt*waer(nl)
          
        endif
      enddo

      if (idb.gt.0) then
        where(dtaua(1:nz)*waer(1:nz).ne.0)
     &       gaer(1:nz)=gaer(1:nz)/(dtaua(1:nz)*waer(1:nz))
 
        print '(a3,es11.3,a8,i3,a8,5i4)', 
     &       'wl=', wl,'   iaer=',iaer,'   jaer=',jaer 
        print '(a3,5a11)',
     &       'i','z','dtaua','dtauab','waer','gaer'
        print '(i3,5es11.3)',(i,z(nz-i+1),dtaua(i),dtauab(i),waer(i),
     &       gaer(i),i=1,nz)
        print '(3x,11x,2a11)','---------','----------'
        print '(3x,11x,2es11.3)',sum(dtaua(1:nz)),sum(dtauab(1:nz))
        print *
      end if

      init=0                    ! first call is done

      return

      end
c========================================================================
      subroutine denprfl(nz,z,dtsv)

c  input:
c    nz     number of altitude layers
c    z      altitude array (km)
c
c  output:
c    dtsv  the array of values sigma * n * dz where sigma is the geometric
c          cross-section, n is the aerosol number density.  The optical
c          depth increments of boundary layer aerosols is ext*dtsv where
c          ext is the wavelength dependent extinction efficiency.

      use params, only: mxly, zero, wl55, kr, zip
      use aeroblk, only: iaer, nzbaer, zbaer, ndbaer,
     &     dbaer, vis, tbaer, rhaer, stdaer, usraer, aerzstd, aerbwi

      implicit none

      real(kr), intent(in)    :: z(*)
      real(kr), intent(out)   :: dtsv(*)
      real(kr), parameter     :: visfac=3.912
      real(kr), external      :: aeroden
      real(kr) :: sigma

      real(kr) :: ext55,w55,g55,vint,aden0

      integer, intent(in) :: nz
      integer             :: i
      integer, external   :: numset

      logical, parameter  :: debug=.false.
      
      sigma=0.
      
      nzbaer=numset(zip,zbaer,mxly)
      ndbaer=numset(zip,dbaer,mxly)
      
      if(ndbaer.gt.0) then
        if(nzbaer.eq.1) then
          write(*,*) 'Error -- only one value of zbaer set'
          stop
        endif
        if(nzbaer.ne.ndbaer.and.nzbaer.gt.1) then
          write(*,*) 'Error -- number of elements must match:'
          write(*,'(a,/,(10es11.3))') 'zbaer',(zbaer(i),i=1,nzbaer)
          write(*,'(a,/,(10es11.3))') 'dbaer',(dbaer(i),i=1,ndbaer)
          stop
        endif
        if(nzbaer.eq.0) then
          nzbaer=ndbaer
          zbaer=z(1:nzbaer)
        endif
      else
        call aerzstd            ! copy standard profile to zbaer,dbaer
      endif
      
      if(iaer.eq.5) then
        call usraer(ext55)      ! user defined BLA spectral profile
        if(vis.eq.zip.and.tbaer.eq.zip) tbaer=ext55
      else
        call stdaer(rhaer)      ! standard BLA spectral profile
        call aerbwi(wl55,ext55,w55,g55)
        if(vis.eq.zip.and.tbaer.eq.zip) then
          print *,'must specify either tbaer or vis'
          stop
        endif
      endif
      
      if (debug) write(*,*) ' call aervint',vis,nz
      call aervint(nz,z,dtsv)  ! at this point dtsv = n(i) dz 
      if (debug) write(*,*) ' after call aervint'
      
      if (debug) write(*,*) ' call aerbwi'
      if(ext55.gt.0.) then 
        if(tbaer.ge.0) then
          vint=sum(dtsv(1:nz))
          if(vint.ne.0) sigma=tbaer/(ext55*vint)
        else
          if (debug) write(*,*) ' call aeroden'
          aden0=aeroden(zero)
          sigma=visfac/(ext55*vis*aden0) 
        endif
      endif
      dtsv(1:nz)=sigma*dtsv(1:nz)
      
      end subroutine denprfl
c=======================================================================
 
      subroutine aervint(nz,z,vint)
c
c purpose  
c   compute aerosol density vertical integral (actually the 
c   integral increment at each level, vint=n(i)*dz  (km-1)
c   (the aerosol density can be considered dimensionless since
c   any constant factors are normalized away in tauaero)
c
c input
c   z        monitonically increasing or decreasing array of altitudes (km)
c   nz       number of altitudes
c
c output
c   vint   = n(i) dz, where n(i) is the aerosol density at level i
c            averaged over the interval z(i) to z(i+1)
c            i.e., the column depth increment of aerosol density


      use params, only: kr
      implicit none
      real(kr), parameter :: ztop=100.
      real(kr) :: z(*),vint(*),zu,zd,aden2,aden,dz
      real(kr), external :: aeroden
      integer :: ii,nz,i
      logical :: bup

      bup=z(1).lt.z(nz) ! true if z(1) is at surface
      !print *,'z1,znz,bup ', z1,z(nz),bup

      aden2=aeroden(ztop)
      vint(1)=aden2*5.  ! (aerosol density at nz) * (5 km scale height)

      !print '(6a12)','z(i)','dz','aden','zu','zd','vint(i)'
      !print '(6f12.4)',z(nz),0,aden2,0.,0.,vint(1)

      zu=max(z(1),z(nz))
      do 30 i=2,nz
        ii=i
        if(bup) ii=nz-i+1
        zd=z(ii)
        aden=aeroden(zd)
        dz=zu-zd
        vint(i)=dz*aden
        !print '(6f12.4)',z(ii),dz,aden,zu,zd,vint(i)
        zu=zd
 30   CONTINUE

      return
      end
c===========================================================================
      function relhum(t,h2o)
c
c purpose:
c     return relative humidity for a give temperature and water vapor 
c     density
c
c input:
c   t          temperature (kelvin)
c   h2o        water vapor density (g/m3)
c output:
c   relhum     relative humidity 
c
      use params, only: kr, tzero
      implicit none
      real(kr) :: h2osat, a, t, h2o, relhum

      a=tzero/t
c
c h2osat is the mass density (g/m3) of water vapor at 100% saturation 
c (source: handbook of chemistry and physics, h2o vapor pressure table d-112,
c assuming density related to pressure by ideal gas law)
c
      h2osat=a*exp(18.916758_kr-a*(14.845878_kr+a*2.4918766_kr))
      relhum=h2o/h2osat
      return
      end
c======================================================================
      subroutine aeread(imoma,nz,wl,dtau,wbaer,pmom)
c
c  purpose:  open and read aerosol.dat, perform wavelength interpolation
c
c  input:    
c    imoma   phase function selector
c    nz      number of atmospheric levels 
c    wl      wavelength at which aerosol information is required
c    pmom    (legendre moments, cloud)*tcld*wcld
c
c  output:
c    dtau    array of  aerosol optical depth increments
c    wbaer   array of aerosol single scattering albedo
c    pmom    (legendre moments, cloud)*tcld*wcld +
c            (legendre moments, aerosols)*tauaer*waer
c            
c  method:
c    input file aerosol.dat is opened and read each time wl extends
c    beyond the range spanned by wlbaer(ind) - wlbaer(3-ind).  ind
c    toggles between values 1 and 2.  If aerosol.dat contains
c    information for only one wavelength then that info is assumed
c    spectrally uniform and will be used for all wavelengths.  If
c    aerosol.dat contains info for more than one wavelength,
c    interpolations are done if wl is within the wavelength
c    range. Otherwise if wl is outside of the range, values at the range
c    limits are used for extrapolation. It is best to call aeread with 
c    wavelength arguments that increase with each new call of aeread.
c    However, if aeread is called with a wavelength less than the
c    wavelength range established by previous calls it will rewind
c    aerosol.dat and re-establish the correct wavelengths in
c    wlbaer(ind) and wlbaer(3-ind).
c 
c  file format:
c
c    nn  nmom
c    wl_1
c    (taer_1(i),waer_1(i),(pm_1(j,i),j=1,nmom),i=ns,nz)          
c    wl_2
c    (taer_2(i),waer_2(i),(pm_2(j,i),j=1,nmom),i=ns,nz)
c    .
c    .
c    wl_n
c    (taer_n(i),waer_n(i),(pm_n(j,i),j=1,nmom),i=ns,nz)
c    ...
c    etc.....
c
c    where, wl is the wavelength, with wl_1 < wl_2 < wl_n
c           taer_w (i) is the total optical depth at wavelength w. level i
c           waer_w is the single scattering albedo
c           pm_w(j,i) is the jth moment of the phase function
c           ns=nz-nn+1, 
c           nz is the total number of computational layers
c           nn is the number of aerosol layers read from aerosol.dat
c           
c           taer(nz) is aerosol optical depth at surface
c    

      use params, only: mxly, zero, one, kr, maxmom

      implicit none
      
      real(kr), parameter :: taumin=1.e-30

      real(kr) ::
     &     taer(mxly,2)=0.,
     &     waer(mxly,2)=0.,
     &     gaer(maxmom,mxly,2)=0.,
     &     wlbaer(2)=0.,
     &     wl0=0.

      real(kr) ::
     &     dtau(*), wbaer(*), pmom(0:maxmom,*),
     &     pm(0:maxmom), wl, wt, wlmx, wlinp, gg

      integer :: ind=1,more=1

      integer :: i,nz,nn,ns,nmom,nmomg,imoma

      logical, parameter :: debug=.false.


c mfl commented out
c     data wlbaer/0.,0./
      save taer,waer,gaer,ind,more,wlbaer,ns,nmom,wl0

      if(wlbaer(1).eq.0.) then 
        open(unit=15,file='aerosol.dat',form='formatted',status='old')

        read(15,*) nn, nmom
        ns=nz-nn+1
        if(ns.le.0) then
          print *,'nz  nn ',nz,nn
          print *,'too many layers specified in aerosol.dat'
          stop
        endif

        read(15,*,end=30) wlbaer(1)
        wl0=wlbaer(1)
        do i=ns,nz
          read(15,*,end=20) taer(i,1),waer(i,1),gaer(1:nmom,i,1)
        enddo
        ind=1
      elseif(wl.lt.minval(wlbaer).and.wl.gt.wl0) then
        ! reposition to earlier record if wl < current wavelength interval
        rewind 15
        read(15,*)
        read(15,*) wlbaer(1)
        do i=ns,nz
          taer(i,1)=0.
          waer(i,1)=0.
          gaer(1:nmom,i,1)=0.
          read(15,*,end=20) taer(i,1),waer(i,1),gaer(1:nmom,i,1)
        enddo
        wlbaer(2)=0.
        ind=1
        more=1
      endif
  
      if(more.eq.1) then
        
        do while (wl.gt.maxval(wlbaer))
          more=0
          read(15,*,end=10) wlinp
          ind=3-ind
          do i=ns,nz
            taer(i,ind)=0.
            waer(i,ind)=0.
            gaer(1:nmom,i,ind)=0.
            read(15,*,end=20) taer(i,ind),waer(i,ind),gaer(1:nmom,i,ind)
          enddo
          wlbaer(ind)=wlinp
          if(debug) print '(a,i5,4f12.5)','ind,wl,wbaer ',
     &         ind,wl,wlbaer(1:2)
        enddo
        more=1                  ! didnt hit eof, so there must be more
 10     continue
      endif

      if(wlbaer(2).eq.0) then  
        ! only one wavelength in file, assume spectrally uniform
        wlbaer(1)=.5*wl
        wlbaer(2)=2*wl
        taer(ns:nz,2)=taer(ns:nz,1)
        waer(ns:nz,2)=waer(ns:nz,1)
        gaer(1:nmom,ns:nz,2)=gaer(1:nmom,ns:nz,1)
        wl0=0.                  ! turn off later rewind operations
      endif

      wt=log(wl/wlbaer(ind))/log(wlbaer(3-ind)/wlbaer(ind))
      wt=max(zero,min(wt,one))
      if(debug) then
        print *,'ns,nmom,imoma ',ns,nmom,imoma
        print '(a,f8.3,a,2f8.3)','wl=',wl,'    wl1-wl2:',wlbaer(1:2)
        print '(5x,3a11)','tau','wbaer','pmom'
      endif
      do i=ns,nz
        if(min(taer(i,ind),taer(i,3-ind)).gt.0.) then
          dtau(i)=taer(i,ind)*(taer(i,3-ind)/taer(i,ind))**wt
        else
          dtau(i)=taer(i,ind)*(1.-wt)+taer(i,3-ind)*wt
        endif
        wbaer(i)=waer(i,ind)*(1.-wt)+waer(i,3-ind)*wt

        if(nmom.eq.1) then
          nmomg=maxmom
          gg=gaer(1,i,ind)*(1.-wt)+gaer(1,i,3-ind)*wt
          call getmom(imoma,gg,nmomg,pm(0))
        else
          nmomg=nmom
          pm(1:nmom)=gaer(1:nmom,i,ind)*(1.-wt)+gaer(1:nmom,i,3-ind)*wt
        endif
        pmom(1:nmomg,i)=pmom(1:nmomg,i)+pm(1:nmomg)*dtau(i)*wbaer(i)
        if(debug) print '(i5,14es11.3)',
     &       i,dtau(i),wbaer(i),pm(1:min(nmomg,12))
      enddo

      return
 20   continue
      print *,'not enough aerosol records'
      print *,'need ',nz-ns+1,' records'
      print '(a/(10es11.3))','taer',taer(ns:nz,ind)
      stop
 30   continue
      print *,'no data found in aerosol.dat'
      stop
      end
c=======================================================================
 
c file:                  taucloud.f
c
c external routines:     taucloud
c
c internal routines:     cloudpar, icepar, usrcloud 
c
c internal data block:   cloudblk
c
c=======================================================================
      subroutine taucloud(nz,ncldz,wl,lcld,lwp,tcloud,nre,taucld,wcld,
     &                    imomc,nmom,pmom)
c
c purpose:  compute cloud radiometric properties
c
c input: 
c
c  nz      number of atmospheric layers
c  ncldz   maximum number of cloud layers
c  wl      wavelength
c  lcld    cloud layer indecies
c  lwp     liquid water path of each cloud layer
c  tcloud  optical depth of each cloud layer
c  nre     effective radius within each cloud layer
c
c  imomc   selects a cloud phase function
c          1   isotropic scattering
c          2   rayleigh scattering
c          3   henyey_greenstein(g(re))
c          4   haze L as specified by garcia/siewert
C          5   cloud c.1 as specified by garcia/siewert
c 
c output:
c
c  taucld  increment of cloud optical depth at each vertical grid point
c  wcld    cloud single scatter albedo at each vertical grid point
c  pmom    phase function moments multiplied by taucld*wcld
c
      use params, only: mxly,nstrms,maxmom,wl55,kr

      implicit none
c                            
      real(kr), parameter :: rhoice=.917   ! bulk density of ice (g/cm3):

      real(kr) :: pmcld(0:maxmom), pmom(0:maxmom,*),
     &     lwp(*), tcloud(*), nre(*), lwpth, wcld(*), taucld(*),
     &     qcld(mxly), q550(mxly), reff, tcld, wt, wl, qc, wc, gc,
     &     qc550, wc550, gc550

      integer :: i, j, k, nz, lcld(*), icnt(mxly), ncmom=0, 
     &     ncldz, nmom, nmomc, imomc, lbot, ltop, ifirst=1

      save q550
     
      do j=1,nz
        qcld(j)=0.
        wcld(j)=0.
        taucld(j)=0.
        icnt(j)=0
      enddo

      do 20 i=1,ncldz
        call levrng(ncldz,lcld,i,lbot,ltop)
        if(lbot.eq.0) goto 20
        if(tcloud(i).eq.0..and.lwp(i).eq.0.) goto 20
        do 10 j=ltop,lbot
          if(ltop.eq.lbot) then
            reff=nre(i)
            tcld=tcloud(i)
            lwpth=lwp(i)
          else
            wt=float(j-ltop)/(lbot-ltop)
            reff=nre(i+1)*(nre(i)/nre(i+1))**wt

c           if tcloud(i+1) is zero, spread the opacity uniformly over
c           the affected layers.  if tcloud(i+1) is nonzero then the
c           opacity is distributed such that it linearly increases
c           (or decreases) from the bottom to the top layer.  The rate
c           of increase is controled by tcloud(i+1) which sets the ratio
c           of the the opacity in the top layer to that of the bottom
c           layer.  The total opacity over all layers adds up to
c           tcloud(i)

            if(tcloud(i+1).eq.0.) then
              tcld=tcloud(i)/(lbot-ltop+1)
            else
              tcld=2*tcloud(i)/((lbot-ltop+1)*(1.+tcloud(i+1)))
              tcld=tcld+(lbot-j)*tcld*(tcloud(i+1)-1.)/(lbot-ltop)
            endif
            
            if(lwp(i+1).eq.0.) then
              lwpth=lwp(i)/(lbot-ltop+1)
            else
              lwpth=2*lwp(i)/((lbot-ltop+1)*(1.+lwp(i+1)))
              lwpth=lwpth+(lbot-j)*lwpth*(lwp(i+1)-1.)/(lbot-ltop)
            endif
          endif

          ncmom=nmom
          call cloudpar(wl,reff,qc,wc,gc)
          call getmom(imomc,gc,nmom,pmcld)

          do k=1,min(nmom,ncmom)
            pmom(k,j)=pmcld(k)+pmom(k,j)
          enddo
          qcld(j)=qc+qcld(j)
          wcld(j)=wc+wcld(j)
          icnt(j)=1+icnt(j)
          if(tcloud(i).ne.0.) then
            if(ifirst.eq.1) then
              nmomc=nmom
              call cloudpar(wl55,reff,qc550,wc550,gc550)
              q550(j)=qc550
            endif
            taucld(j)=tcld*qc/q550(j)+taucld(j)
          else
            if(lwpth.ne.0.) then
              if(reff.lt.0.) then
                taucld(j)=-.75*qc*lwpth/reff/rhoice+taucld(j)
              else
                taucld(j)=.75*qc*lwpth/reff+taucld(j)
              endif
            endif
          endif
 10     continue
 20   continue

      do 30 j=1,nz
        if(icnt(j).ne.0) then
          qcld(j)=qcld(j)/icnt(j)
          wcld(j)=wcld(j)/icnt(j)
          do i=1,min(nmom,ncmom)
            pmom(i,j)=taucld(j)*wcld(j)*pmom(i,j)/icnt(j)
          enddo
        endif
        
 30   continue

      ifirst=0
      return
      end
c=======================================================================
      subroutine usrcloud(nz,wl,p,taucld,wcld,imomc,nmom,pmom,idb)
c
c purpose:  read user specified cloud properties
c           this routine is called when tcloud(1) < 0
c
c input:   
c   wl      wavelength
c   p       pressure array (mb), pressure at surface=p(1)
c   imomc   phase function type passed to getmom
c   nmom    number of phase function moments required
c   idb     if 1 then print diagnostics
c
c output:
c
c  taucld   increment of cloud optical depth at each vertical grid point
c  wcld     cloud single scatter albedo at each vertical grid point
c  pmom     phase function moments multiplied by taucld*wcld
c
c usrcloud format (read on first call only):
c
c read(13,*) (lwp(i),reff(i),fwp(i),reice(i),cldfrac(i)),i=1,nzz)
cc
c   lwp     liquid water path within layer (g/m2)          (default=0)
c   reff    effective radius of water droplets (microns).  (default=8.)
c   fwp     frozen water path within layer (g/m2)          (default=0)
c           negative fwp => ccm3 model parameters
c   reice   effective radius of ice particles (microns)    (default=-1)
c           reice.le.0 .and. fwp.gt.0 => ccm3 effective 
c           radius is used (see icepar)
c   cldfrac fraction of cloud within layer.  to simulate   (default=1)
c           the effect of partial cloud cover, optical depth is
c           reduced by cldfrac**1.5, an empirical result from
c           the NCAR Community Climate Model (CCM)
c 
c   DISCUSSION:

c           Cloud records are read from file usrcld.dat starting with
c           the lowest layer in the atmosphere. The input file need not
c           specify values above the altitude where clouds are present.
c           In addition the input record may be terminated before
c           reading all the input variables by inserting a forward slash
c           in place of the first variable to be ignord.  If left
c           unspecified, input quantities retain their default of
c           lwp=0. reff=10, fwp=0, reice=-1, and cldfrac=1.  The radiative
c           properties of ice are computed from a CCM3 model
c           parameterization contained in subroutine icepar

c
      use params, only: maxmom,mxly,kr
      implicit none

      integer, parameter :: rhoice=.917 ! the bulk density of ice (g/cm3)

      real(kr) :: p(*), pmom(0:maxmom,*), lwp(mxly), reff(mxly),
     &     fwp(mxly), reice(mxly), cldfrac(mxly), wcld(*),taucld(*),
     &     ww, gw, qw, wi, gi, qi, qcld(mxly), wl, tauw, prat, ri,
     &     taui, gcld

      integer :: i, j, nz, imomc, idb, nmom, ifirst=1
     
      data lwp/mxly*0./
      data reff/mxly*8./
      data fwp/mxly*0./
      data reice/mxly*-1./
      data cldfrac/mxly*1./

      save lwp,reff,fwp,cldfrac

      if(ifirst.eq.1) then 
        open(unit=13,file='usrcld.dat',form='formatted',status='old')
        do i=nz,1,-1
          read(13,*,end=100) lwp(i),reff(i),fwp(i),reice(i),cldfrac(i)
        enddo
        close(unit=13)
 100    continue
        ifirst=0
      endif

      if(imomc.lt.0) then
        print *,'imomc < 0 not allowed with usrcld.dat option'
        stop
      endif

      if(idb.gt.0) print '(a5,10a11)',
     &   'i','p','taucld','tauw','taui','qw','qi','ww','wi','gw','gi'

      do i=1,nz

        if(lwp(i).gt.0.) then 
          call cloudpar(wl,reff(i),qw,ww,gw)
          tauw=.75*qw*lwp(i)/reff(i)
        else
          qw=0.
          ww=0.
          gw=0.
          tauw=0.
        endif

        if(fwp(i).ne.0.) then
          prat=p(nz-i+1)/p(1)
          ri=reice(i)
          if(fwp(i).lt.0) then 
            call icepar(wl,prat,qi,wi,gi,ri)
          else 
            if(ri.le.0) call icepar(wl,prat,qi,wi,gi,ri)
            ri=-abs(ri)
            call cloudpar(wl,ri,qi,wi,gi)
          endif
          taui=.75*qi*abs(fwp(i))/(rhoice*abs(ri))
        else
          qi=0.
          wi=0.
          gi=0.
          taui=0.
        endif

        taucld(i)=tauw+taui
        if(taucld(i).ne.0.) then
          wcld(i)=(tauw*ww+taui*wi)/taucld(i)
          gcld=(tauw*gw+taui*gi)/taucld(i)
          call getmom(imomc,gcld,nmom,pmom(0,i))
        endif
        taucld(i)=taucld(i)*cldfrac(i)**1.5
        do j=1,nmom
          pmom(j,i)=taucld(i)*wcld(i)*pmom(j,i)
        enddo
        if(idb.gt.0) print '(i5,10es11.3)',
     &       i,p(nz-i+1),taucld(i),tauw,taui,qw,qi,ww,wi,gw,gi
      enddo

      return
      end
c_end
c=======================================================================
      subroutine icepar(wl,prat,qice,wice,gice,rei)
c
c purpose: compute ice radiative parameters using CCM3 ice coefficients.
c
c input:
c   wl        wavelength (um)
c   prat      pressure ratio p(z)/p_surface
c
c input/output:
c   rei       effective radius.  if rei.le.0 then replace with
c             effective radius from ccm3 ice model 
c             
c
c output:
c   qice      extinction coefficient
c   wice      single scattering albedo of ice
c   gice      asymmetry factor of ice 
c   rei       effective radius of ice (um)
c
      use params, only: kr

      implicit none

      real(kr) :: abari(4), bbari(4), cbari(4), dbari(4), ebari(4),
     &     fbari(4), pllow, plhgh, reimax, reimin, wl, rei, prat,
     &     qice, wice, gice

      integer :: iceindx
 
      data abari /3.448e-03,3.448e-03,3.448e-03,3.448e-03/
      data bbari /2.431    ,2.431    ,2.431    ,2.431    /
      data cbari /1.00e-05 ,1.10e-04 ,1.861e-02,.46658   /
      data dbari /0.0      ,1.405e-05,8.328e-04,2.05e-05 /
      data ebari /0.7661   ,0.7730   ,0.794    ,0.9595   /
      data fbari /5.851e-04,5.665e-04,7.267e-04,1.076e-04/
      data pllow  /.0/
      data plhgh  /.4/
      data reimax /30./
      data reimin /10./
 
c     CCM3 code - partitions solar spectrum to 4 bands.
 
      if(wl.lt.0.69) then
        iceindx=1
      else if(wl.lt.1.19) then
        iceindx=2
      else if(wl.lt.2.38) then
        iceindx=3
      else
        iceindx=4
      endif
 
      if(rei.le.0.) then 
        if(prat.gt.plhgh) then
          rei=reimin
        else
          rei=reimax-((reimax-reimin)*((prat-plhgh)/(plhgh-pllow)))
        endif
      endif

      gice=ebari(iceindx)+fbari(iceindx)*rei
      wice=1.-cbari(iceindx)-dbari(iceindx)*rei
      qice=(abari(iceindx)+(bbari(iceindx)/rei))

      return
      end
c=======================================================================

      SUBROUTINE cloudpar(wl,re,qc,wc,gc)
c
c input:
c   wl      wavelength in microns
c   re      effective cloud drop radius (microns)
c output:
c   qc      mean extinction efficiency (unitless)
c   wc      single scatter albedo      (unitless)
c   gc      asymmetry factor           (unitless)
c     

c the parameters contained in this subroutine were computed using 
c Wiscombe Mie scattering code, MIEV0.  The radius integrations
c were performed over a modified gamma distribution
c f(x)=(x/xm)^(p-1)*exp(-x/xm) where p=7.

c NOTE:

c re < 0 selects scattering parameters for ice spheres
c Mie scattering parameters for water and ice clouds
c
c Wavelength range is  0.29 to  333.33 microns 
c
c Wavelengths steps are in equal increments of log(wavelength)
c Hence, the i'th table entry is at wavelenth
c
c          wl = 0.29 * (333.33/0.29)**((i-1)/(mxwv-1))
c
c Included in the table are entries for effective radii of
c
c          re = 2**((j+1)/2) for j=1,13
c
c This Mie database was created by 
c    /home/paul/mie/wiscombe/{water,ice}/pmom.cmd 
c and formated with
c    /home/paul/mie/wiscombe/{water,ice}/mkpmom.pro
 
      use params, only: one, kr
      implicit none

      real(kr), parameter :: wlmin=0.29, wlmax=333.33, eps=.000001

      integer, parameter :: mxwv=400, mre=13
      integer ::  icall, i, iw, ir

      real(kr) :: qq(mxwv,mre),ww(mxwv,mre),gg(mxwv,mre),
     &        qqi(mxwv,mre),wwi(mxwv,mre),ggi(mxwv,mre),
     &        fr, fw, gc, qc, re, wc, wl, wmax, wmin, wstep

      data icall,wmin,wmax,wstep/0,0.,0.,0./

c 
c        ---------------- extinction efficiency ----------------

      data (qq(i,1),i=1,mxwv)/
     &   2.17502666, 2.17705488, 2.17945981, 2.18155527, 2.18386197,
     &   2.18641090, 2.18857670, 2.19129729, 2.19366312, 2.19596457,
     &   2.19853497, 2.20104814, 2.20371795, 2.20627928, 2.20893717,
     &   2.21180654, 2.21439219, 2.21722698, 2.22005177, 2.22292018,
     &   2.22576642, 2.22874284, 2.23176241, 2.23473716, 2.23774362,
     &   2.24082780, 2.24402070, 2.24720716, 2.25038552, 2.25361443,
     &   2.25692534, 2.26020098, 2.26354837, 2.26690340, 2.27032566,
     &   2.27374268, 2.27719760, 2.28062248, 2.28411269, 2.28760719,
     &   2.29113030, 2.29466820, 2.29824281, 2.30179715, 2.30538130,
     &   2.30899572, 2.31262636, 2.31630254, 2.32001519, 2.32378149,
     &   2.32762146, 2.33154297, 2.33557439, 2.33973503, 2.34405398,
     &   2.34856272, 2.35329628, 2.35829353, 2.36359835, 2.36925769,
     &   2.37532187, 2.38184500, 2.38888407, 2.39649868, 2.40475059,
     &   2.41370153, 2.42341542, 2.43395329, 2.44537401, 2.45773554,
     &   2.47109199, 2.48549771, 2.50098848, 2.51760030, 2.53535986,
     &   2.55428433, 2.57437634, 2.59563327, 2.61801910, 2.64149570,
     &   2.66603732, 2.69156599, 2.71801019, 2.74528360, 2.77326369,
     &   2.80183864, 2.83086061, 2.86014557, 2.88957477, 2.91879225,
     &   2.94668722, 2.97444415, 3.00203633, 3.02939057, 3.05576015,
     &   3.08073235, 3.10413814, 3.12579226, 3.14543080, 3.16288662,
     &   3.17790985, 3.19019771, 3.19952631, 3.20562434, 3.20852327,
     &   3.20691943, 3.19620252, 3.18033361, 3.17178464, 3.16243815,
     &   3.14695644, 3.12438679, 3.09612036, 3.06194472, 3.02178931,
     &   2.97530246, 2.92187381, 2.86102247, 2.79152226, 2.71235824,
     &   2.62156391, 2.51724195, 2.39780188, 2.25397897, 2.06965280,
     &   1.81314278, 1.47268379, 1.15169585, 1.28466785, 1.53216481,
     &   1.85487890, 2.12174034, 2.31310582, 2.47859597, 2.64122748,
     &   2.79979491, 2.90341711, 2.95882654, 2.94658995, 2.86584187,
     &   2.77218103, 2.66592813, 2.55328059, 2.43995881, 2.33058119,
     &   2.22600079, 2.12766719, 2.03446722, 1.94598961, 1.86261523,
     &   1.78330398, 1.70802879, 1.63683224, 1.57049978, 1.50879097,
     &   1.45302904, 1.40313184, 1.36117184, 1.31957674, 1.27231741,
     &   1.21919715, 1.16273332, 1.10242403, 1.03997958, 0.97503036,
     &   0.91024303, 0.84782869, 0.78598338, 0.72648561, 0.67700797,
     &   0.68004543, 0.80958587, 1.06148815, 1.21916318, 1.06732845,
     &   0.92990327, 0.84266120, 0.77429891, 0.71917790, 0.67489249,
     &   0.63671166, 0.60254693, 0.57224268, 0.54376388, 0.51730049,
     &   0.49202278, 0.46848267, 0.44672537, 0.42581266, 0.40601236,
     &   0.38732231, 0.36951771, 0.35282731, 0.33719108, 0.32282090,
     &   0.30936635, 0.29687786, 0.28551263, 0.27485919, 0.26570359,
     &   0.25775653, 0.25239041, 0.25175655, 0.25901312, 0.27342898,
     &   0.29718980, 0.32283768, 0.35670570, 0.39677656, 0.44318610,
     &   0.48895615, 0.54749864, 0.60162365, 0.65336853, 0.69634545,
     &   0.73471516, 0.76891905, 0.79815042, 0.81967932, 0.83824831,
     &   0.85156107, 0.85920340, 0.86628318, 0.86854428, 0.86856580,
     &   0.86789280, 0.86537731, 0.86100680, 0.85260862, 0.84360075,
     &   0.83239943, 0.81652278, 0.80135447, 0.78186607, 0.76419902,
     &   0.74566597, 0.72170585, 0.69796336, 0.67450345, 0.64958209,
     &   0.62229031, 0.60202730, 0.58223885, 0.56328470, 0.54505736,
     &   0.52727163, 0.50990611, 0.49297634, 0.47528338, 0.45593211,
     &   0.43725908, 0.42013198, 0.40361109, 0.38725349, 0.37129331,
     &   0.35583365, 0.34084553, 0.32662845, 0.31328207, 0.30051482,
     &   0.28836867, 0.27716801, 0.26738128, 0.25871325, 0.25169498,
     &   0.24573471, 0.24069007, 0.23662263, 0.23302822, 0.22975561,
     &   0.22687808, 0.22466733, 0.22326955, 0.22239502, 0.22159633,
     &   0.22182521, 0.22313239, 0.22480163, 0.22632575, 0.22770067,
     &   0.22977827, 0.23218539, 0.23447281, 0.23606516, 0.23758155,
     &   0.23897120, 0.24005422, 0.24075750, 0.24140766, 0.24041410,
     &   0.23857823, 0.23654707, 0.23449720, 0.23238145, 0.22981887,
     &   0.22545232, 0.22110908, 0.21650995, 0.21198845, 0.20732638,
     &   0.20268758, 0.19721176, 0.19181943, 0.18613218, 0.18070966,
     &   0.17568652, 0.17069620, 0.16572662, 0.16076547, 0.15587592,
     &   0.15096237, 0.14622231, 0.14167967, 0.13731468, 0.13312168,
     &   0.12914601, 0.12484229, 0.12050779, 0.11640624, 0.11245794,
     &   0.10855427, 0.10464919, 0.10097606, 0.09758507, 0.09428326,
     &   0.09104992, 0.08808544, 0.08513276, 0.08217223, 0.07938257,
     &   0.07679242, 0.07432113, 0.07205009, 0.06984399, 0.06770088,
     &   0.06561888, 0.06359617, 0.06214036, 0.06082397, 0.05953438,
     &   0.05827096, 0.05703313, 0.05586062, 0.05471718, 0.05359612,
     &   0.05249694, 0.05142131, 0.05037135, 0.04934165, 0.04833176,
     &   0.04735898, 0.04651768, 0.04569063, 0.04487758, 0.04408547,
     &   0.04333149, 0.04258981, 0.04186019, 0.04115284, 0.04046454,
     &   0.03978714, 0.03913825, 0.03857718, 0.03802390, 0.03747830,
     &   0.03693016, 0.03638894, 0.03585538, 0.03529256, 0.03472875,
     &   0.03417354, 0.03363883, 0.03311307, 0.03259677, 0.03209499,
     &   0.03160060, 0.03111210, 0.03062983, 0.03015471, 0.02970541,
     &   0.02926306, 0.02880034, 0.02832900, 0.02786013, 0.02738631,
     &   0.02692121, 0.02647947, 0.02604464, 0.02561771, 0.02519749,
     &   0.02479647, 0.02440309, 0.02401588, 0.02363460, 0.02327066,
     &   0.02291242, 0.02255957, 0.02221145, 0.02186741, 0.02152859/
      data (qq(i,2),i=1,mxwv)/
     &   2.13798332, 2.13925219, 2.14121342, 2.14302087, 2.14454389,
     &   2.14633369, 2.14811230, 2.15019870, 2.15216684, 2.15378094,
     &   2.15584373, 2.15777731, 2.15973592, 2.16166997, 2.16378689,
     &   2.16593575, 2.16784549, 2.17012668, 2.17208719, 2.17420554,
     &   2.17655826, 2.17878199, 2.18132281, 2.18330407, 2.18574071,
     &   2.18803787, 2.19039845, 2.19302201, 2.19534039, 2.19793582,
     &   2.20031524, 2.20289874, 2.20567107, 2.20836806, 2.21095133,
     &   2.21373105, 2.21643114, 2.21926165, 2.22209811, 2.22489476,
     &   2.22785592, 2.23078966, 2.23379755, 2.23679447, 2.23991466,
     &   2.24296117, 2.24611163, 2.24927950, 2.25246811, 2.25567436,
     &   2.25902772, 2.26228857, 2.26560974, 2.26899385, 2.27235270,
     &   2.27572775, 2.27916002, 2.28259277, 2.28602648, 2.28948903,
     &   2.29299402, 2.29648638, 2.30001974, 2.30355215, 2.30712700,
     &   2.31072402, 2.31436896, 2.31805515, 2.32180691, 2.32563615,
     &   2.32955813, 2.33359814, 2.33777022, 2.34213328, 2.34669137,
     &   2.35149336, 2.35658002, 2.36199999, 2.36780286, 2.37404275,
     &   2.38078094, 2.38808084, 2.39600778, 2.40463352, 2.41402841,
     &   2.42426777, 2.43542647, 2.44758058, 2.46080685, 2.47520447,
     &   2.49055290, 2.50686765, 2.52442575, 2.54340053, 2.56385159,
     &   2.58563137, 2.60874343, 2.63316607, 2.65884924, 2.68580270,
     &   2.71406054, 2.74336243, 2.77359009, 2.80458641, 2.83639789,
     &   2.86914539, 2.89929056, 2.92447400, 2.95171785, 2.98308206,
     &   3.01519752, 3.04522467, 3.07343721, 3.09903026, 3.12162685,
     &   3.14059973, 3.15504026, 3.16401362, 3.16631794, 3.15989852,
     &   3.14246082, 3.11055136, 3.06122589, 2.98802567, 2.86877966,
     &   2.64914417, 2.24823785, 1.68687320, 1.65082586, 1.80276322,
     &   2.05663753, 2.25913787, 2.40496802, 2.52363515, 2.64168501,
     &   2.77308202, 2.89902425, 3.02475595, 3.13580918, 3.19440866,
     &   3.22508168, 3.23115849, 3.21329999, 3.17614865, 3.12688971,
     &   3.06819391, 3.00411606, 2.93599343, 2.86500192, 2.79205775,
     &   2.71830487, 2.64325356, 2.56808257, 2.49391270, 2.42130661,
     &   2.35278726, 2.28868771, 2.23401618, 2.18567824, 2.13337469,
     &   2.07345867, 2.00666738, 1.93272746, 1.85209572, 1.76304054,
     &   1.66803360, 1.56938481, 1.46480370, 1.35238814, 1.23800766,
     &   1.17062557, 1.25814593, 1.53864408, 1.80666018, 1.74136353,
     &   1.60048425, 1.49206913, 1.39938819, 1.31875503, 1.24944663,
     &   1.18757772, 1.13048494, 1.07846689, 1.02890050, 0.98189175,
     &   0.93626744, 0.89275277, 0.85150170, 0.81159115, 0.77302444,
     &   0.73593479, 0.69997483, 0.66549397, 0.63244063, 0.60123473,
     &   0.57131159, 0.54277855, 0.51594448, 0.48996422, 0.46606418,
     &   0.44387582, 0.42454937, 0.41302297, 0.41110086, 0.42056698,
     &   0.44397256, 0.47131950, 0.50981212, 0.55733794, 0.61353487,
     &   0.66908497, 0.74138194, 0.80845982, 0.87330765, 0.92801374,
     &   0.97697091, 1.02180791, 1.06119645, 1.09199321, 1.11951602,
     &   1.14143550, 1.15647972, 1.17098117, 1.18009853, 1.18668222,
     &   1.19206452, 1.19590342, 1.19766498, 1.19458222, 1.19039881,
     &   1.18394518, 1.17135620, 1.15939510, 1.14124632, 1.12467051,
     &   1.10712790, 1.08210421, 1.05639410, 1.02959049, 0.99969471,
     &   0.96512556, 0.93898529, 0.91308397, 0.88795686, 0.86347985,
     &   0.83926386, 0.81547731, 0.79197890, 0.76698780, 0.73913848,
     &   0.71192718, 0.68603122, 0.66079444, 0.63542688, 0.61038649,
     &   0.58576566, 0.56151837, 0.53822154, 0.51596385, 0.49447951,
     &   0.47369748, 0.45428184, 0.43682915, 0.42111406, 0.40789378,
     &   0.39639977, 0.38641477, 0.37806916, 0.37057266, 0.36368626,
     &   0.35748842, 0.35236308, 0.34856409, 0.34564441, 0.34291050,
     &   0.34179229, 0.34235492, 0.34365222, 0.34482867, 0.34586433,
     &   0.34801191, 0.35076204, 0.35344568, 0.35519850, 0.35690305,
     &   0.35849330, 0.35969409, 0.36039451, 0.36111709, 0.35949779,
     &   0.35661972, 0.35342786, 0.35023332, 0.34698740, 0.34307635,
     &   0.33652633, 0.32998708, 0.32302523, 0.31616625, 0.30910769,
     &   0.30207407, 0.29380319, 0.28564483, 0.27701414, 0.26877081,
     &   0.26110730, 0.25349978, 0.24593145, 0.23838133, 0.23094511,
     &   0.22347386, 0.21626568, 0.20935442, 0.20271324, 0.19634008,
     &   0.19030578, 0.18380356, 0.17726648, 0.17107494, 0.16512010,
     &   0.15924217, 0.15337250, 0.14784652, 0.14274411, 0.13778329,
     &   0.13293341, 0.12848461, 0.12406534, 0.11964417, 0.11547906,
     &   0.11161148, 0.10792426, 0.10453162, 0.10124227, 0.09805273,
     &   0.09495969, 0.09195992, 0.08978231, 0.08781121, 0.08588430,
     &   0.08400035, 0.08215816, 0.08041602, 0.07872003, 0.07706007,
     &   0.07543521, 0.07384758, 0.07230001, 0.07078455, 0.06930044,
     &   0.06787216, 0.06663447, 0.06541952, 0.06422679, 0.06306614,
     &   0.06196215, 0.06087751, 0.05981183, 0.05877952, 0.05777590,
     &   0.05678929, 0.05584471, 0.05502670, 0.05422094, 0.05342721,
     &   0.05263123, 0.05184607, 0.05107273, 0.05025868, 0.04944408,
     &   0.04864248, 0.04787079, 0.04711253, 0.04636840, 0.04564557,
     &   0.04493385, 0.04423108, 0.04353771, 0.04285501, 0.04220954,
     &   0.04157440, 0.04091078, 0.04023536, 0.03956383, 0.03888561,
     &   0.03822013, 0.03758813, 0.03696627, 0.03635591, 0.03575535,
     &   0.03518232, 0.03462039, 0.03406746, 0.03352317, 0.03300372,
     &   0.03249256, 0.03198923, 0.03149279, 0.03100232, 0.03051943/
      data (qq(i,3),i=1,mxwv)/
     &   2.10869312, 2.10994959, 2.11140633, 2.11277103, 2.11422896,
     &   2.11536121, 2.11695004, 2.11851144, 2.11983109, 2.12128448,
     &   2.12290406, 2.12427378, 2.12593651, 2.12752199, 2.12906098,
     &   2.13054109, 2.13223219, 2.13380599, 2.13549471, 2.13716388,
     &   2.13883781, 2.14053965, 2.14241076, 2.14389253, 2.14597774,
     &   2.14763069, 2.14959931, 2.15132356, 2.15322566, 2.15502787,
     &   2.15699601, 2.15899014, 2.16102433, 2.16296744, 2.16500926,
     &   2.16714239, 2.16919565, 2.17152119, 2.17347479, 2.17582440,
     &   2.17812586, 2.18023086, 2.18252206, 2.18482947, 2.18741202,
     &   2.18976593, 2.19206786, 2.19451046, 2.19712734, 2.19927979,
     &   2.20195913, 2.20460868, 2.20721841, 2.21003222, 2.21257138,
     &   2.21535015, 2.21808338, 2.22079682, 2.22379136, 2.22658467,
     &   2.22949386, 2.23260593, 2.23552299, 2.23868084, 2.24167609,
     &   2.24473333, 2.24787998, 2.25107288, 2.25427008, 2.25750327,
     &   2.26077747, 2.26406288, 2.26738620, 2.27073312, 2.27406740,
     &   2.27748823, 2.28088045, 2.28429341, 2.28773665, 2.29119539,
     &   2.29467297, 2.29816723, 2.30168939, 2.30523252, 2.30881763,
     &   2.31244469, 2.31613111, 2.31988573, 2.32373452, 2.32769465,
     &   2.33177805, 2.33604360, 2.34052014, 2.34524989, 2.35028577,
     &   2.35567641, 2.36148620, 2.36778593, 2.37465405, 2.38218617,
     &   2.39050651, 2.39969325, 2.40986061, 2.42113304, 2.43366981,
     &   2.44789767, 2.46370196, 2.47966862, 2.49573636, 2.51447392,
     &   2.53611541, 2.56007242, 2.58641195, 2.61510921, 2.64629149,
     &   2.68005490, 2.71643901, 2.75540113, 2.79707813, 2.84088016,
     &   2.88646746, 2.93235302, 2.97695661, 3.02118087, 3.05438161,
     &   3.03401637, 2.82029462, 2.20736504, 1.96685779, 1.99823010,
     &   2.16106415, 2.28940988, 2.37913322, 2.44473457, 2.50376511,
     &   2.56359792, 2.62238908, 2.68873930, 2.76887298, 2.84917426,
     &   2.92530346, 2.99790573, 3.06162143, 3.11368728, 3.15467215,
     &   3.18417263, 3.20343637, 3.21363235, 3.21554637, 3.20906758,
     &   3.19627380, 3.17594481, 3.14931393, 3.11678267, 3.07932758,
     &   3.03928494, 2.99739718, 2.95979095, 2.93239808, 2.90570259,
     &   2.87324166, 2.83226228, 2.78262997, 2.72195435, 2.64654684,
     &   2.55602670, 2.45047402, 2.32708144, 2.17662430, 1.99636972,
     &   1.82395816, 1.79886127, 2.02898788, 2.35301685, 2.44095850,
     &   2.36579680, 2.27746248, 2.18968177, 2.10384727, 2.02282238,
     &   1.94670093, 1.87331951, 1.80390608, 1.73616374, 1.67005050,
     &   1.60433435, 1.53977203, 1.47672570, 1.41483545, 1.35354042,
     &   1.29323709, 1.23349452, 1.17478526, 1.11712825, 1.06127715,
     &   1.00641990, 0.95282000, 0.90102839, 0.84956110, 0.80019069,
     &   0.75258124, 0.70687234, 0.67304814, 0.64910203, 0.64281952,
     &   0.65700501, 0.67927259, 0.71648842, 0.76698202, 0.82928628,
     &   0.89143950, 0.97439528, 1.05156362, 1.12681627, 1.19099045,
     &   1.24820960, 1.30166233, 1.34947407, 1.38847625, 1.42405236,
     &   1.45427072, 1.47696316, 1.49915755, 1.51610875, 1.53059542,
     &   1.54341340, 1.55549169, 1.56569850, 1.57107580, 1.57504904,
     &   1.57740808, 1.57322407, 1.56969309, 1.55904830, 1.54945028,
     &   1.53938389, 1.52113044, 1.50125325, 1.47851932, 1.45108402,
     &   1.41654611, 1.38996530, 1.36316288, 1.33678174, 1.31068814,
     &   1.28441143, 1.25846064, 1.23237038, 1.20382631, 1.17095113,
     &   1.13821161, 1.10528207, 1.07261980, 1.03890014, 1.00488675,
     &   0.97053176, 0.93571740, 0.90145522, 0.86771047, 0.83448863,
     &   0.80137235, 0.76968843, 0.74003637, 0.71266198, 0.68858790,
     &   0.66707826, 0.64786100, 0.63126296, 0.61609554, 0.60200340,
     &   0.58903724, 0.57771480, 0.56852263, 0.56079382, 0.55345875,
     &   0.54865283, 0.54646742, 0.54585260, 0.54528731, 0.54470569,
     &   0.54592073, 0.54831171, 0.55089056, 0.55235320, 0.55390370,
     &   0.55547148, 0.55662739, 0.55720031, 0.55807185, 0.55571741,
     &   0.55144519, 0.54661435, 0.54182684, 0.53707272, 0.53129399,
     &   0.52163976, 0.51189679, 0.50135422, 0.49089104, 0.48011509,
     &   0.46930960, 0.45660460, 0.44399253, 0.43051162, 0.41755992,
     &   0.40541157, 0.39333448, 0.38130838, 0.36929464, 0.35744789,
     &   0.34551895, 0.33398399, 0.32289293, 0.31221557, 0.30197191,
     &   0.29228595, 0.28192002, 0.27151796, 0.26163468, 0.25213447,
     &   0.24277574, 0.23345183, 0.22464956, 0.21651274, 0.20861855,
     &   0.20092036, 0.19384757, 0.18685448, 0.17988528, 0.17331909,
     &   0.16721793, 0.16140804, 0.15604682, 0.15086702, 0.14586158,
     &   0.14102377, 0.13634714, 0.13289264, 0.12975894, 0.12670794,
     &   0.12373675, 0.12084254, 0.11811424, 0.11546731, 0.11288538,
     &   0.11036633, 0.10791252, 0.10552726, 0.10319855, 0.10092463,
     &   0.09874038, 0.09683983, 0.09497969, 0.09315877, 0.09139106,
     &   0.08971205, 0.08806676, 0.08645427, 0.08489495, 0.08338176,
     &   0.08189763, 0.08047837, 0.07924533, 0.07803354, 0.07684249,
     &   0.07565257, 0.07448127, 0.07332975, 0.07212290, 0.07091793,
     &   0.06973404, 0.06859525, 0.06747789, 0.06638284, 0.06532034,
     &   0.06427555, 0.06324537, 0.06223033, 0.06123209, 0.06028872,
     &   0.05936145, 0.05839503, 0.05741312, 0.05643792, 0.05545427,
     &   0.05448988, 0.05357417, 0.05267388, 0.05179091, 0.05092278,
     &   0.05009467, 0.04928317, 0.04848522, 0.04770028, 0.04695141,
     &   0.04621495, 0.04549025, 0.04477591, 0.04407059, 0.04337656/
      data (qq(i,4),i=1,mxwv)/
     &   2.08608413, 2.08718085, 2.08819842, 2.08928776, 2.09026051,
     &   2.09133458, 2.09247351, 2.09354448, 2.09475374, 2.09592819,
     &   2.09698820, 2.09825253, 2.09944415, 2.10058975, 2.10188723,
     &   2.10301042, 2.10440993, 2.10564876, 2.10713172, 2.10819149,
     &   2.10947013, 2.11083865, 2.11228514, 2.11354899, 2.11504149,
     &   2.11647153, 2.11772227, 2.11944032, 2.12073398, 2.12217474,
     &   2.12372160, 2.12518692, 2.12670422, 2.12827706, 2.12999678,
     &   2.13142085, 2.13316607, 2.13475418, 2.13644004, 2.13801360,
     &   2.13973999, 2.14158034, 2.14336085, 2.14500785, 2.14754248,
     &   2.14867210, 2.15053105, 2.15235233, 2.15427756, 2.15617537,
     &   2.15817475, 2.16008782, 2.16209793, 2.16414022, 2.16627002,
     &   2.16842127, 2.17039990, 2.17264581, 2.17476940, 2.17722082,
     &   2.17922306, 2.18139601, 2.18377066, 2.18595934, 2.18852735,
     &   2.19084144, 2.19325399, 2.19597149, 2.19838595, 2.20085573,
     &   2.20338416, 2.20608950, 2.20859051, 2.21130514, 2.21415663,
     &   2.21691465, 2.21960235, 2.22249460, 2.22532177, 2.22823811,
     &   2.23119760, 2.23420048, 2.23719168, 2.24030447, 2.24336004,
     &   2.24646950, 2.24964976, 2.25286150, 2.25608301, 2.25932121,
     &   2.26253939, 2.26582241, 2.26915288, 2.27252769, 2.27591467,
     &   2.27931309, 2.28272390, 2.28614759, 2.28958941, 2.29304862,
     &   2.29653072, 2.30004168, 2.30359244, 2.30719543, 2.31088018,
     &   2.31466842, 2.31852102, 2.32249141, 2.32671952, 2.33127499,
     &   2.33619928, 2.34156895, 2.34752536, 2.35422683, 2.36188507,
     &   2.37076735, 2.38122392, 2.39368606, 2.40880442, 2.42733002,
     &   2.45043492, 2.47953343, 2.51644492, 2.56633377, 2.63695979,
     &   2.73646903, 2.80438280, 2.48966694, 2.15578151, 2.10177422,
     &   2.18972707, 2.26456690, 2.31655908, 2.35262156, 2.38277960,
     &   2.40946722, 2.43095112, 2.45043802, 2.47089243, 2.49408054,
     &   2.52004957, 2.55064130, 2.58556533, 2.62389326, 2.66443777,
     &   2.70609426, 2.74769092, 2.78883410, 2.82893705, 2.86680984,
     &   2.90274501, 2.93540955, 2.96442008, 2.98866057, 3.00792336,
     &   3.02198839, 3.03077078, 3.03648925, 3.04872918, 3.06593990,
     &   3.08406568, 3.09908509, 3.11101890, 3.11588383, 3.10966635,
     &   3.08762121, 3.04506540, 2.97790289, 2.86905861, 2.69852328,
     &   2.45939159, 2.28735828, 2.37881136, 2.63521552, 2.83085465,
     &   2.87047887, 2.85724258, 2.82630253, 2.78239822, 2.73039556,
     &   2.67559981, 2.61768126, 2.55856657, 2.49776578, 2.43501234,
     &   2.36961031, 2.30192137, 2.23245740, 2.16206431, 2.08944917,
     &   2.01527452, 1.93915737, 1.86155772, 1.78262067, 1.70341992,
     &   1.62305439, 1.54199219, 1.46106601, 1.37812603, 1.29532707,
     &   1.21259439, 1.12785673, 1.05840755, 0.99532914, 0.95742053,
     &   0.94795388, 0.95371586, 0.97912616, 1.02392077, 1.08473706,
     &   1.14701998, 1.23365045, 1.31460798, 1.39419651, 1.46257877,
     &   1.52303052, 1.58031464, 1.63210475, 1.67565691, 1.71578753,
     &   1.75132167, 1.77945089, 1.80704463, 1.83019590, 1.85123301,
     &   1.87025559, 1.88947523, 1.90730131, 1.92113745, 1.93348849,
     &   1.94530141, 1.95149374, 1.95836437, 1.95874786, 1.95952487,
     &   1.96076453, 1.95526111, 1.94776297, 1.93628299, 1.91968000,
     &   1.89515257, 1.87585616, 1.85601318, 1.83620429, 1.81630564,
     &   1.79586506, 1.77574813, 1.75512981, 1.73167431, 1.70336866,
     &   1.67456388, 1.64312804, 1.61127734, 1.57719040, 1.54180443,
     &   1.50471973, 1.46561122, 1.42580879, 1.38490987, 1.34344804,
     &   1.30030930, 1.25759029, 1.21547580, 1.17526007, 1.13806653,
     &   1.10376072, 1.07215190, 1.04398108, 1.01774323, 0.99301237,
     &   0.96975309, 0.94857353, 0.93038440, 0.91433358, 0.89892870,
     &   0.88709855, 0.87900382, 0.87400436, 0.86951578, 0.86537468,
     &   0.86397922, 0.86478364, 0.86644882, 0.86713207, 0.86827159,
     &   0.86983430, 0.87118530, 0.87207621, 0.87397659, 0.87195694,
     &   0.86715698, 0.86135793, 0.85571909, 0.85037875, 0.84350461,
     &   0.83126980, 0.81861663, 0.80435240, 0.78994828, 0.77498752,
     &   0.75973231, 0.74151307, 0.72311902, 0.70289445, 0.68316793,
     &   0.66430342, 0.64537662, 0.62637138, 0.60720849, 0.58815211,
     &   0.56875134, 0.54981065, 0.53141022, 0.51354426, 0.49632597,
     &   0.48001266, 0.46263373, 0.44515380, 0.42837855, 0.41219971,
     &   0.39624578, 0.38034523, 0.36520466, 0.35113859, 0.33749765,
     &   0.32421276, 0.31194299, 0.29987898, 0.28790689, 0.27660036,
     &   0.26606229, 0.25602967, 0.24670805, 0.23774523, 0.22912604,
     &   0.22083592, 0.21286091, 0.20678002, 0.20123938, 0.19587976,
     &   0.19069350, 0.18567327, 0.18096623, 0.17642576, 0.17202219,
     &   0.16775009, 0.16361059, 0.15960635, 0.15571795, 0.15194090,
     &   0.14832486, 0.14515376, 0.14206679, 0.13906078, 0.13615564,
     &   0.13340358, 0.13071999, 0.12810239, 0.12557934, 0.12313955,
     &   0.12075731, 0.11848422, 0.11649707, 0.11455292, 0.11265031,
     &   0.11076353, 0.10891392, 0.10710233, 0.10522015, 0.10334931,
     &   0.10151698, 0.09975738, 0.09803587, 0.09635346, 0.09472474,
     &   0.09312759, 0.09155734, 0.09001437, 0.08850069, 0.08707152,
     &   0.08566996, 0.08421671, 0.08274542, 0.08128752, 0.07982088,
     &   0.07838541, 0.07702288, 0.07568563, 0.07437608, 0.07309065,
     &   0.07186520, 0.07066607, 0.06948867, 0.06833217, 0.06722954,
     &   0.06614668, 0.06508251, 0.06403495, 0.06300194, 0.06198668/
      data (qq(i,5),i=1,mxwv)/
     &   2.06820655, 2.06907678, 2.06985450, 2.07070923, 2.07153821,
     &   2.07240009, 2.07327151, 2.07405877, 2.07502890, 2.07594180,
     &   2.07686520, 2.07781076, 2.07874894, 2.07965040, 2.08064580,
     &   2.08158016, 2.08250928, 2.08357215, 2.08455801, 2.08565044,
     &   2.08637929, 2.08769488, 2.08875489, 2.08978438, 2.09119201,
     &   2.09195542, 2.09307051, 2.09417486, 2.09566021, 2.09656596,
     &   2.09772825, 2.09896684, 2.10003686, 2.10122228, 2.10266829,
     &   2.10368466, 2.10505676, 2.10627294, 2.10756540, 2.10885644,
     &   2.10994577, 2.11153722, 2.11289072, 2.11438799, 2.11563444,
     &   2.11702752, 2.11848927, 2.12003207, 2.12152100, 2.12287998,
     &   2.12438536, 2.12596536, 2.12761021, 2.12916422, 2.13065577,
     &   2.13222671, 2.13384008, 2.13563156, 2.13722587, 2.13888812,
     &   2.14061761, 2.14231634, 2.14399195, 2.14580345, 2.14766812,
     &   2.14946795, 2.15147948, 2.15331149, 2.15531325, 2.15730000,
     &   2.15920115, 2.16103840, 2.16323447, 2.16517234, 2.16729069,
     &   2.16925478, 2.17180204, 2.17365289, 2.17592597, 2.17805815,
     &   2.18034935, 2.18262458, 2.18497086, 2.18732190, 2.18993735,
     &   2.19229698, 2.19462013, 2.19715142, 2.19970536, 2.20224977,
     &   2.20480633, 2.20744205, 2.21013236, 2.21287704, 2.21566105,
     &   2.21848178, 2.22133803, 2.22423530, 2.22717381, 2.23014522,
     &   2.23316383, 2.23621774, 2.23930860, 2.24243641, 2.24560738,
     &   2.24880743, 2.25185871, 2.25477719, 2.25798988, 2.26141644,
     &   2.26488400, 2.26830220, 2.27171969, 2.27512717, 2.27853608,
     &   2.28195477, 2.28539276, 2.28887916, 2.29246712, 2.29624224,
     &   2.30038857, 2.30524015, 2.31143141, 2.32040977, 2.33603311,
     &   2.37137532, 2.45426416, 2.44946384, 2.20604682, 2.13419557,
     &   2.18014622, 2.22571373, 2.25818729, 2.28112984, 2.30040073,
     &   2.31750512, 2.33114266, 2.34305668, 2.35338426, 2.36136699,
     &   2.36845207, 2.37524247, 2.38214278, 2.38966513, 2.39820385,
     &   2.40807438, 2.41940808, 2.43239141, 2.44712353, 2.46356678,
     &   2.48175073, 2.50162148, 2.52295828, 2.54527211, 2.56819105,
     &   2.59079742, 2.61251926, 2.63217568, 2.65303588, 2.67775917,
     &   2.70673823, 2.73902822, 2.77534389, 2.81459999, 2.85631204,
     &   2.89715695, 2.93249226, 2.95910835, 2.96320677, 2.91557240,
     &   2.75280142, 2.52676201, 2.48015690, 2.59384298, 2.74544239,
     &   2.83394647, 2.88375354, 2.91826963, 2.93902302, 2.94725871,
     &   2.94816136, 2.94250584, 2.93109941, 2.91520643, 2.89412355,
     &   2.86783385, 2.83567119, 2.79767823, 2.75550127, 2.70736241,
     &   2.65363932, 2.59395409, 2.52823567, 2.45648146, 2.37949419,
     &   2.29647303, 2.20777822, 2.11415458, 2.01308823, 1.90620482,
     &   1.79376423, 1.67024875, 1.55999899, 1.44535625, 1.36056030,
     &   1.31077492, 1.28606808, 1.28603566, 1.31325352, 1.36183822,
     &   1.41526771, 1.49541938, 1.57121015, 1.64651060, 1.71159971,
     &   1.76844800, 1.82287967, 1.87235153, 1.91487622, 1.95417786,
     &   1.98994458, 2.01923084, 2.04786587, 2.07324982, 2.09701180,
     &   2.11866832, 2.14117742, 2.16281676, 2.18176293, 2.19935513,
     &   2.21735668, 2.23155284, 2.24632907, 2.25650620, 2.26651025,
     &   2.27776742, 2.28519630, 2.29097319, 2.29291487, 2.29087257,
     &   2.28247237, 2.27554417, 2.26810193, 2.26050091, 2.25264978,
     &   2.24424005, 2.23628497, 2.22780561, 2.21717906, 2.20294452,
     &   2.18797851, 2.16855264, 2.14828658, 2.12532520, 2.10047889,
     &   2.07296491, 2.04221296, 2.00941515, 1.97365880, 1.93596601,
     &   1.89444149, 1.85140610, 1.80609035, 1.76090717, 1.71656728,
     &   1.67406821, 1.63346839, 1.59599853, 1.56027949, 1.52597260,
     &   1.49290502, 1.46160078, 1.43346810, 1.40771902, 1.38269222,
     &   1.36168027, 1.34498692, 1.33325493, 1.32279706, 1.31332016,
     &   1.30742574, 1.30502141, 1.30461967, 1.30402958, 1.30455196,
     &   1.30626345, 1.30836785, 1.31056821, 1.31500280, 1.31589973,
     &   1.31346834, 1.30959499, 1.30618668, 1.30368543, 1.29933238,
     &   1.28896904, 1.27771175, 1.26376152, 1.24925303, 1.23394084,
     &   1.21783960, 1.19770801, 1.17674363, 1.15228570, 1.12778544,
     &   1.10357714, 1.07878339, 1.05338395, 1.02719164, 1.00060153,
     &   0.97278589, 0.94501305, 0.91738498, 0.88998175, 0.86317515,
     &   0.83752573, 0.80999488, 0.78189069, 0.75425148, 0.72720861,
     &   0.70021945, 0.67300582, 0.64650851, 0.62151313, 0.59706151,
     &   0.57307357, 0.55058616, 0.52846700, 0.50646114, 0.48545322,
     &   0.46565524, 0.44668886, 0.42877817, 0.41157955, 0.39506987,
     &   0.37922588, 0.36402419, 0.35186824, 0.34069008, 0.32993704,
     &   0.31959191, 0.30963805, 0.30035883, 0.29146108, 0.28288439,
     &   0.27461582, 0.26665196, 0.25899169, 0.25160208, 0.24447189,
     &   0.23767234, 0.23163033, 0.22579171, 0.22014777, 0.21472745,
     &   0.20961125, 0.20465828, 0.19986169, 0.19526058, 0.19083521,
     &   0.18654466, 0.18246461, 0.17885970, 0.17535831, 0.17195611,
     &   0.16862442, 0.16538107, 0.16222458, 0.15899503, 0.15581037,
     &   0.15270871, 0.14973888, 0.14684851, 0.14403814, 0.14132886,
     &   0.13868555, 0.13610087, 0.13357413, 0.13110690, 0.12878151,
     &   0.12651105, 0.12418008, 0.12183660, 0.11952472, 0.11721121,
     &   0.11495440, 0.11281383, 0.11072018, 0.10867617, 0.10667633,
     &   0.10477211, 0.10291418, 0.10109536, 0.09931409, 0.09761816,
     &   0.09595724, 0.09432944, 0.09273133, 0.09115964, 0.08961877/
      data (qq(i,6),i=1,mxwv)/
     &   2.05411696, 2.05476093, 2.05542445, 2.05602145, 2.05673957,
     &   2.05736589, 2.05815721, 2.05881834, 2.05952716, 2.06018996,
     &   2.06094384, 2.06168818, 2.06234145, 2.06310344, 2.06386614,
     &   2.06469393, 2.06542754, 2.06618309, 2.06695819, 2.06776166,
     &   2.06858897, 2.06941533, 2.07024527, 2.07111263, 2.07197547,
     &   2.07282686, 2.07369161, 2.07456946, 2.07545376, 2.07639432,
     &   2.07728839, 2.07823086, 2.07916021, 2.08000159, 2.08109283,
     &   2.08203435, 2.08302379, 2.08403444, 2.08505797, 2.08608317,
     &   2.08717680, 2.08816314, 2.08928466, 2.09025407, 2.09145880,
     &   2.09244537, 2.09361911, 2.09477377, 2.09595418, 2.09715438,
     &   2.09819961, 2.09934044, 2.10062861, 2.10186815, 2.10311317,
     &   2.10433388, 2.10559273, 2.10696149, 2.10810876, 2.10942388,
     &   2.11090374, 2.11212969, 2.11371994, 2.11499071, 2.11631060,
     &   2.11772799, 2.11924076, 2.12064791, 2.12207150, 2.12361336,
     &   2.12504840, 2.12671852, 2.12837768, 2.12970209, 2.13151193,
     &   2.13294792, 2.13467789, 2.13643956, 2.13797808, 2.13971138,
     &   2.14145851, 2.14318466, 2.14502048, 2.14688516, 2.14867878,
     &   2.15072083, 2.15239573, 2.15428305, 2.15624714, 2.15819168,
     &   2.16015983, 2.16214848, 2.16418982, 2.16628361, 2.16840315,
     &   2.17051125, 2.17272449, 2.17495179, 2.17723060, 2.17938137,
     &   2.18178248, 2.18411541, 2.18653321, 2.18895006, 2.19142032,
     &   2.19391632, 2.19634652, 2.19869804, 2.20127082, 2.20402932,
     &   2.20685768, 2.20969129, 2.21256971, 2.21548724, 2.21845365,
     &   2.22147202, 2.22454119, 2.22766376, 2.23084426, 2.23406315,
     &   2.23731828, 2.24058032, 2.24386501, 2.24731398, 2.25083137,
     &   2.25469613, 2.26597571, 2.29258537, 2.18143511, 2.13118839,
     &   2.15849757, 2.18855834, 2.21036124, 2.22608805, 2.23931837,
     &   2.25110126, 2.26072168, 2.26951075, 2.27760100, 2.28415132,
     &   2.29011202, 2.29568028, 2.30075884, 2.30541325, 2.30978560,
     &   2.31392241, 2.31792593, 2.32188511, 2.32587838, 2.32995987,
     &   2.33425617, 2.33882332, 2.34377050, 2.34916782, 2.35509562,
     &   2.36152601, 2.36842346, 2.37546253, 2.38278770, 2.39101505,
     &   2.40057182, 2.41173005, 2.42501092, 2.44101119, 2.46091676,
     &   2.48543286, 2.51500559, 2.55087304, 2.59300518, 2.63166356,
     &   2.61139894, 2.48055840, 2.40311646, 2.43157744, 2.48926401,
     &   2.53388190, 2.57037735, 2.60534668, 2.63847399, 2.66853070,
     &   2.69659019, 2.72295141, 2.74690151, 2.76941872, 2.78971767,
     &   2.80791521, 2.82277799, 2.83356071, 2.84106755, 2.84384060,
     &   2.84131551, 2.83282924, 2.81728816, 2.79389930, 2.76205349,
     &   2.72076297, 2.66931129, 2.60717225, 2.53173470, 2.44213629,
     &   2.33791590, 2.20967865, 2.08132863, 1.92886686, 1.79913509,
     &   1.70239902, 1.63908076, 1.60349882, 1.60286760, 1.62921667,
     &   1.66550636, 1.72914433, 1.79113674, 1.85388613, 1.90855753,
     &   1.95569682, 2.00122190, 2.04268622, 2.07890534, 2.11233115,
     &   2.14328790, 2.16921377, 2.19439626, 2.21748805, 2.23942614,
     &   2.25948095, 2.28053308, 2.30108595, 2.32012796, 2.33808374,
     &   2.35684037, 2.37357402, 2.39066625, 2.40525365, 2.41938996,
     &   2.43496060, 2.44939351, 2.46288323, 2.47371864, 2.48249650,
     &   2.48787975, 2.49208951, 2.49613523, 2.50011563, 2.50398088,
     &   2.50761199, 2.51184559, 2.51587033, 2.51912761, 2.52113247,
     &   2.52276111, 2.52008677, 2.51676702, 2.51152277, 2.50480032,
     &   2.49571466, 2.48363519, 2.46916986, 2.45112824, 2.43061233,
     &   2.40546346, 2.37728953, 2.34430909, 2.30919838, 2.27173185,
     &   2.23384666, 2.19584727, 2.15913296, 2.12301302, 2.08740306,
     &   2.05200076, 2.01700735, 1.98404872, 1.95275223, 1.92183030,
     &   1.89396334, 1.86962521, 1.85129035, 1.83491778, 1.82013488,
     &   1.80899131, 1.80216825, 1.79846919, 1.79589880, 1.79511058,
     &   1.79632819, 1.79877126, 1.80216968, 1.80898035, 1.81408882,
     &   1.81623924, 1.81686163, 1.81838071, 1.82157373, 1.82323492,
     &   1.82032061, 1.81634116, 1.80921245, 1.80132675, 1.79284811,
     &   1.78335631, 1.77006781, 1.75547171, 1.73624921, 1.71625805,
     &   1.69552636, 1.67367744, 1.65065360, 1.62609124, 1.60038745,
     &   1.57227528, 1.54323494, 1.51326966, 1.48253644, 1.45175755,
     &   1.42187381, 1.38910294, 1.35464907, 1.31931233, 1.28376579,
     &   1.24732709, 1.20954096, 1.17119706, 1.13393533, 1.09659779,
     &   1.05911875, 1.02291322, 0.98677623, 0.95014876, 0.91422421,
     &   0.87946033, 0.84543943, 0.81223166, 0.77996767, 0.74865896,
     &   0.71831310, 0.68893379, 0.66402185, 0.64068013, 0.61813128,
     &   0.59636211, 0.57535821, 0.55577022, 0.53695840, 0.51880342,
     &   0.50129026, 0.48441535, 0.46817762, 0.45253184, 0.43746081,
     &   0.42307484, 0.41003764, 0.39748421, 0.38539773, 0.37382850,
     &   0.36291718, 0.35240680, 0.34228238, 0.33259776, 0.32331842,
     &   0.31437802, 0.30589294, 0.29827940, 0.29093742, 0.28385562,
     &   0.27702653, 0.27043021, 0.26405671, 0.25766641, 0.25142890,
     &   0.24539651, 0.23963928, 0.23407461, 0.22870089, 0.22354980,
     &   0.21856016, 0.21371973, 0.20902373, 0.20447034, 0.20018937,
     &   0.19603764, 0.19184370, 0.18767549, 0.18359342, 0.17954448,
     &   0.17561692, 0.17189540, 0.16827704, 0.16476318, 0.16134495,
     &   0.15809684, 0.15494408, 0.15187432, 0.14888415, 0.14604434,
     &   0.14327748, 0.14057951, 0.13794412, 0.13536531, 0.13284895/
      data (qq(i,7),i=1,mxwv)/
     &   2.04292679, 2.04345584, 2.04394841, 2.04449677, 2.04498100,
     &   2.04555130, 2.04607344, 2.04664254, 2.04707193, 2.04773879,
     &   2.04831982, 2.04891729, 2.04947448, 2.05006957, 2.05066252,
     &   2.05136728, 2.05188251, 2.05246615, 2.05314517, 2.05378556,
     &   2.05458999, 2.05503869, 2.05568361, 2.05637288, 2.05705523,
     &   2.05771661, 2.05844760, 2.05907869, 2.05981398, 2.06053400,
     &   2.06122065, 2.06193972, 2.06268358, 2.06346512, 2.06412601,
     &   2.06501889, 2.06571960, 2.06660628, 2.06734228, 2.06816840,
     &   2.06910706, 2.06983709, 2.07069325, 2.07150078, 2.07231116,
     &   2.07326126, 2.07406402, 2.07494712, 2.07588601, 2.07678294,
     &   2.07762551, 2.07860970, 2.07953358, 2.08056903, 2.08146000,
     &   2.08259320, 2.08351636, 2.08446288, 2.08550620, 2.08639598,
     &   2.08754086, 2.08863521, 2.08964777, 2.09077358, 2.09197688,
     &   2.09304690, 2.09418678, 2.09521651, 2.09642196, 2.09757328,
     &   2.09883595, 2.09991646, 2.10112095, 2.10241914, 2.10365915,
     &   2.10490036, 2.10601854, 2.10737395, 2.10869360, 2.11008978,
     &   2.11148000, 2.11276555, 2.11419702, 2.11561060, 2.11698246,
     &   2.11843348, 2.11982489, 2.12152028, 2.12280083, 2.12435079,
     &   2.12588573, 2.12740803, 2.12898946, 2.13058972, 2.13220334,
     &   2.13388658, 2.13556790, 2.13719106, 2.13896751, 2.14072323,
     &   2.14240813, 2.14421701, 2.14604497, 2.14788318, 2.14982080,
     &   2.15166569, 2.15351629, 2.15531826, 2.15728188, 2.15936255,
     &   2.16149378, 2.16363215, 2.16580367, 2.16800785, 2.17025161,
     &   2.17254281, 2.17487764, 2.17727113, 2.17972875, 2.18224859,
     &   2.18484569, 2.18752074, 2.19031525, 2.19340110, 2.19677806,
     &   2.20011997, 2.20142627, 2.19591856, 2.14499521, 2.11690831,
     &   2.13536215, 2.15609050, 2.17118526, 2.18218684, 2.19137931,
     &   2.19948387, 2.20612431, 2.21224546, 2.21794057, 2.22267699,
     &   2.22710562, 2.23136377, 2.23539996, 2.23923302, 2.24294806,
     &   2.24654555, 2.25006080, 2.25352168, 2.25693989, 2.26028681,
     &   2.26362133, 2.26688528, 2.27009749, 2.27323484, 2.27631211,
     &   2.27934670, 2.28233504, 2.28537798, 2.28885293, 2.29267859,
     &   2.29670906, 2.30083036, 2.30511904, 2.30955291, 2.31425476,
     &   2.31943583, 2.32553458, 2.33349633, 2.34518576, 2.36308503,
     &   2.37687349, 2.33985448, 2.30043578, 2.31498837, 2.33906937,
     &   2.35306096, 2.36411214, 2.37522674, 2.38690662, 2.39918685,
     &   2.41229367, 2.42645383, 2.44143510, 2.45760489, 2.47487473,
     &   2.49348688, 2.51311493, 2.53345180, 2.55460024, 2.57626629,
     &   2.59797573, 2.61938453, 2.63957763, 2.65768433, 2.67237854,
     &   2.68265247, 2.68690038, 2.68316579, 2.66907382, 2.64075994,
     &   2.59540057, 2.52163148, 2.42910123, 2.29360533, 2.15646219,
     &   2.03412366, 1.94308162, 1.87650323, 1.84765303, 1.84900343,
     &   1.86534834, 1.90745282, 1.95132077, 1.99729753, 2.03795505,
     &   2.07271862, 2.10661554, 2.13756537, 2.16495347, 2.19017744,
     &   2.21379566, 2.23390079, 2.25330520, 2.27145505, 2.28881383,
     &   2.30472827, 2.32141757, 2.33779430, 2.35337687, 2.36817288,
     &   2.38364816, 2.39816523, 2.41281939, 2.42630529, 2.43934369,
     &   2.45345116, 2.46764350, 2.48141193, 2.49375653, 2.50543261,
     &   2.51599813, 2.52488136, 2.53391647, 2.54307127, 2.55232930,
     &   2.56169391, 2.57165980, 2.58175254, 2.59211874, 2.60297918,
     &   2.61400032, 2.62250805, 2.63096619, 2.63888264, 2.64639163,
     &   2.65289927, 2.65800500, 2.66176081, 2.66321921, 2.66313028,
     &   2.65991879, 2.65403247, 2.64343381, 2.62985611, 2.61218119,
     &   2.59224629, 2.57032895, 2.54737806, 2.52352238, 2.49891210,
     &   2.47318935, 2.44611764, 2.41896105, 2.39189434, 2.36444306,
     &   2.33778596, 2.31246471, 2.29227257, 2.27408814, 2.25759649,
     &   2.24368310, 2.23374009, 2.22726917, 2.22305179, 2.22085643,
     &   2.22100616, 2.22294450, 2.22649193, 2.23376918, 2.24171162,
     &   2.24785662, 2.25278163, 2.25879288, 2.26679277, 2.27401400,
     &   2.27952790, 2.28417325, 2.28622842, 2.28766608, 2.28906918,
     &   2.28970218, 2.28828931, 2.28575087, 2.27923036, 2.27171707,
     &   2.26284385, 2.25299072, 2.24209833, 2.22974467, 2.21617842,
     &   2.20008945, 2.18258739, 2.16349220, 2.14295197, 2.12182307,
     &   2.10113215, 2.07785749, 2.05238986, 2.02459455, 1.99560940,
     &   1.96481466, 1.93160546, 1.89582396, 1.85961556, 1.82205987,
     &   1.78305328, 1.74372792, 1.70350206, 1.66140103, 1.61832249,
     &   1.57487011, 1.53076375, 1.48544359, 1.44022238, 1.39517975,
     &   1.35039401, 1.30594182, 1.26559842, 1.22653472, 1.18808663,
     &   1.15026963, 1.11312151, 1.07805526, 1.04384744, 1.01031101,
     &   0.97747111, 0.94536084, 0.91401523, 0.88342518, 0.85360384,
     &   0.82474309, 0.79783303, 0.77168643, 0.74630278, 0.72180635,
     &   0.69849086, 0.67589438, 0.65400976, 0.63292384, 0.61260086,
     &   0.59295690, 0.57419252, 0.55700046, 0.54040283, 0.52438498,
     &   0.50907892, 0.49430847, 0.48004684, 0.46595258, 0.45226145,
     &   0.43904594, 0.42641178, 0.41423175, 0.40250522, 0.39128989,
     &   0.38047075, 0.37003249, 0.35996124, 0.35024542, 0.34111440,
     &   0.33230707, 0.32356760, 0.31498984, 0.30665052, 0.29845878,
     &   0.29055834, 0.28306866, 0.27583531, 0.26885310, 0.26210791,
     &   0.25570986, 0.24954014, 0.24357499, 0.23780654, 0.23234518,
     &   0.22706228, 0.22194828, 0.21698979, 0.21217412, 0.20750852/
      data (qq(i,8),i=1,mxwv)/
     &   2.03409195, 2.03448105, 2.03489804, 2.03530455, 2.03574800,
     &   2.03614974, 2.03657389, 2.03700686, 2.03747296, 2.03791094,
     &   2.03835917, 2.03877401, 2.03929281, 2.03973341, 2.04024744,
     &   2.04067302, 2.04115009, 2.04165006, 2.04214668, 2.04264116,
     &   2.04315805, 2.04366279, 2.04417443, 2.04472423, 2.04525638,
     &   2.04574156, 2.04634523, 2.04685712, 2.04750466, 2.04800057,
     &   2.04852676, 2.04916525, 2.04978752, 2.05033755, 2.05093026,
     &   2.05153990, 2.05212212, 2.05279970, 2.05340719, 2.05407929,
     &   2.05466604, 2.05531549, 2.05597448, 2.05659604, 2.05730271,
     &   2.05805349, 2.05871820, 2.05937719, 2.06012940, 2.06085539,
     &   2.06144619, 2.06232929, 2.06298208, 2.06386471, 2.06454563,
     &   2.06533241, 2.06608367, 2.06694388, 2.06778264, 2.06849313,
     &   2.06926894, 2.07017803, 2.07093024, 2.07186890, 2.07273865,
     &   2.07356000, 2.07443881, 2.07538271, 2.07627702, 2.07712650,
     &   2.07808161, 2.07902193, 2.07999992, 2.08093429, 2.08202982,
     &   2.08285427, 2.08392286, 2.08491611, 2.08603597, 2.08694625,
     &   2.08802629, 2.08906388, 2.09014988, 2.09125614, 2.09235644,
     &   2.09346247, 2.09453702, 2.09575915, 2.09693122, 2.09810066,
     &   2.09927440, 2.10039735, 2.10170078, 2.10293865, 2.10419941,
     &   2.10545444, 2.10681272, 2.10808325, 2.10936117, 2.11069441,
     &   2.11210871, 2.11342502, 2.11493516, 2.11630774, 2.11775494,
     &   2.11921978, 2.12064910, 2.12204456, 2.12355256, 2.12514472,
     &   2.12676716, 2.12839437, 2.13004255, 2.13171005, 2.13341022,
     &   2.13513088, 2.13689613, 2.13868451, 2.14052343, 2.14239740,
     &   2.14431787, 2.14628696, 2.14833307, 2.15057421, 2.15302491,
     &   2.15553546, 2.15670848, 2.14728212, 2.11633348, 2.10072613,
     &   2.11380601, 2.12835598, 2.13896060, 2.14674926, 2.15321541,
     &   2.15885830, 2.16349316, 2.16778779, 2.17179608, 2.17516732,
     &   2.17834997, 2.18142986, 2.18436670, 2.18718195, 2.18993378,
     &   2.19262481, 2.19528127, 2.19792366, 2.20056152, 2.20317674,
     &   2.20581079, 2.20842195, 2.21102190, 2.21358585, 2.21611333,
     &   2.21860814, 2.22104883, 2.22349954, 2.22630215, 2.22941303,
     &   2.23271680, 2.23610020, 2.23961782, 2.24318075, 2.24674726,
     &   2.25020742, 2.25344920, 2.25650620, 2.25912738, 2.26108980,
     &   2.26009393, 2.24443913, 2.23007250, 2.24403191, 2.26100516,
     &   2.26971340, 2.27555680, 2.28067517, 2.28533864, 2.28971052,
     &   2.29410148, 2.29862785, 2.30333567, 2.30838656, 2.31382656,
     &   2.31980586, 2.32638693, 2.33366442, 2.34181142, 2.35098839,
     &   2.36131430, 2.37298870, 2.38607669, 2.40064406, 2.41649032,
     &   2.43361378, 2.45158386, 2.46953058, 2.48674846, 2.50050831,
     &   2.50798917, 2.50147963, 2.47489643, 2.40910244, 2.31824183,
     &   2.21594810, 2.12855935, 2.05404663, 2.01064205, 1.99500227,
     &   1.99631953, 2.01945996, 2.04691410, 2.07735825, 2.10497737,
     &   2.12864375, 2.15200806, 2.17347598, 2.19270921, 2.21045041,
     &   2.22718883, 2.24163556, 2.25555110, 2.26870418, 2.28133035,
     &   2.29297829, 2.30512023, 2.31704760, 2.32850504, 2.33943748,
     &   2.35078597, 2.36159492, 2.37243485, 2.38264132, 2.39253688,
     &   2.40305591, 2.41374850, 2.42421460, 2.43393707, 2.44344401,
     &   2.45263267, 2.46076131, 2.46909285, 2.47758865, 2.48624396,
     &   2.49508691, 2.50435567, 2.51382422, 2.52374125, 2.53442597,
     &   2.54542685, 2.55545688, 2.56575108, 2.57624912, 2.58698964,
     &   2.59774518, 2.60840964, 2.61889648, 2.62877846, 2.63843656,
     &   2.64721990, 2.65498948, 2.66042519, 2.66411018, 2.66484261,
     &   2.66354728, 2.66024709, 2.65514851, 2.64868021, 2.64100027,
     &   2.63177776, 2.62049747, 2.60764456, 2.59362006, 2.57862043,
     &   2.56234121, 2.54514503, 2.53061891, 2.51734805, 2.50519609,
     &   2.49395919, 2.48521304, 2.47918320, 2.47560477, 2.47366953,
     &   2.47370481, 2.47547722, 2.47892213, 2.48535609, 2.49397779,
     &   2.50208426, 2.50949192, 2.51780367, 2.52775455, 2.53762627,
     &   2.54822516, 2.55829859, 2.56688476, 2.57511592, 2.58370566,
     &   2.59191751, 2.59997296, 2.60741901, 2.61268330, 2.61721063,
     &   2.62046194, 2.62328291, 2.62565899, 2.62730145, 2.62828660,
     &   2.62778473, 2.62622952, 2.62337136, 2.61930823, 2.61469460,
     &   2.61022210, 2.60486174, 2.59815574, 2.58913279, 2.57890749,
     &   2.56721759, 2.55360961, 2.53693962, 2.51882911, 2.49904037,
     &   2.47744846, 2.45416451, 2.42966914, 2.40293121, 2.37384462,
     &   2.34273624, 2.30956244, 2.27293658, 2.23511052, 2.19612813,
     &   2.15603805, 2.11489344, 2.07412601, 2.03304458, 1.99144244,
     &   1.94937754, 1.90690947, 1.86604476, 1.82517612, 1.78406262,
     &   1.74275875, 1.70131993, 1.65980470, 1.61827314, 1.57678246,
     &   1.53555870, 1.49562418, 1.45597863, 1.41658139, 1.37776375,
     &   1.34002185, 1.30269921, 1.26583481, 1.22955716, 1.19388533,
     &   1.15878451, 1.12458396, 1.09232998, 1.06071031, 1.02974343,
     &   1.00002718, 0.97098076, 0.94256669, 0.91443670, 0.88685554,
     &   0.85992628, 0.83381891, 0.80839092, 0.78367817, 0.75982207,
     &   0.73662692, 0.71411031, 0.69225967, 0.67105651, 0.65096444,
     &   0.63149410, 0.61231315, 0.59354216, 0.57526028, 0.55731559,
     &   0.53997177, 0.52339602, 0.50738311, 0.49191684, 0.47698897,
     &   0.46277267, 0.44908273, 0.43587732, 0.42314628, 0.41108274,
     &   0.39945793, 0.38825315, 0.37744167, 0.36699799, 0.35693419/
      data (qq(i,9),i=1,mxwv)/
     &   2.02706456, 2.02738762, 2.02769160, 2.02802467, 2.02832866,
     &   2.02870750, 2.02903938, 2.02938008, 2.02970338, 2.03006792,
     &   2.03045630, 2.03080034, 2.03115582, 2.03152800, 2.03191280,
     &   2.03229547, 2.03267145, 2.03305078, 2.03344083, 2.03384042,
     &   2.03423452, 2.03465509, 2.03505230, 2.03550577, 2.03590846,
     &   2.03633690, 2.03676200, 2.03722191, 2.03764534, 2.03809428,
     &   2.03854060, 2.03899193, 2.03942204, 2.03994226, 2.04045773,
     &   2.04086900, 2.04133153, 2.04185772, 2.04237986, 2.04288411,
     &   2.04338384, 2.04391384, 2.04443431, 2.04493284, 2.04544902,
     &   2.04600072, 2.04654241, 2.04708195, 2.04768729, 2.04823685,
     &   2.04877973, 2.04937148, 2.04998255, 2.05057120, 2.05120730,
     &   2.05176497, 2.05239773, 2.05307794, 2.05365062, 2.05433059,
     &   2.05495453, 2.05556202, 2.05627084, 2.05688620, 2.05757189,
     &   2.05828929, 2.05907989, 2.05966377, 2.06039596, 2.06110954,
     &   2.06183386, 2.06257701, 2.06345081, 2.06412053, 2.06484747,
     &   2.06566453, 2.06648183, 2.06724524, 2.06810451, 2.06882524,
     &   2.06966782, 2.07048106, 2.07134295, 2.07217717, 2.07309365,
     &   2.07394576, 2.07484651, 2.07575560, 2.07664967, 2.07758784,
     &   2.07848716, 2.07943296, 2.08039021, 2.08139348, 2.08239269,
     &   2.08334589, 2.08437324, 2.08535028, 2.08639693, 2.08745694,
     &   2.08852720, 2.08957958, 2.09065318, 2.09178782, 2.09291649,
     &   2.09405613, 2.09515667, 2.09624934, 2.09742165, 2.09865117,
     &   2.09990144, 2.10115409, 2.10242605, 2.10370731, 2.10500646,
     &   2.10634518, 2.10767531, 2.10903764, 2.11040854, 2.11184978,
     &   2.11329412, 2.11476588, 2.11627960, 2.11791515, 2.11966133,
     &   2.12150049, 2.12205315, 2.11473298, 2.09441423, 2.08528495,
     &   2.09466076, 2.10497022, 2.11249757, 2.11806941, 2.12267661,
     &   2.12666559, 2.12995481, 2.13301921, 2.13589001, 2.13833547,
     &   2.14066386, 2.14292574, 2.14509630, 2.14718580, 2.14923573,
     &   2.15124726, 2.15323853, 2.15522313, 2.15720749, 2.15918207,
     &   2.16116810, 2.16314435, 2.16511703, 2.16707134, 2.16900754,
     &   2.17092729, 2.17282081, 2.17473054, 2.17688966, 2.17926335,
     &   2.18177390, 2.18434930, 2.18703222, 2.18976951, 2.19254255,
     &   2.19528675, 2.19792080, 2.20045137, 2.20256662, 2.20351243,
     &   2.20016313, 2.18825698, 2.18170929, 2.19289041, 2.20491266,
     &   2.21122646, 2.21558881, 2.21945190, 2.22294664, 2.22613430,
     &   2.22922921, 2.23227262, 2.23527002, 2.23830414, 2.24134707,
     &   2.24443650, 2.24754119, 2.25066876, 2.25391650, 2.25727272,
     &   2.26078510, 2.26452971, 2.26857209, 2.27302933, 2.27800965,
     &   2.28373218, 2.29038572, 2.29813337, 2.30733371, 2.31782746,
     &   2.32934260, 2.34023809, 2.34478474, 2.33277202, 2.29834437,
     &   2.24271178, 2.18508863, 2.12755895, 2.08808708, 2.06858230,
     &   2.06360292, 2.07496762, 2.09127879, 2.11062574, 2.12875581,
     &   2.14450455, 2.16026139, 2.17488813, 2.18815136, 2.20046353,
     &   2.21215296, 2.22237468, 2.23225284, 2.24164319, 2.25069094,
     &   2.25911760, 2.26784921, 2.27643704, 2.28470492, 2.29263973,
     &   2.30081081, 2.30859947, 2.31640410, 2.32376552, 2.33095145,
     &   2.33849239, 2.34604430, 2.35344172, 2.36038613, 2.36716580,
     &   2.37371111, 2.37976909, 2.38596058, 2.39226937, 2.39869213,
     &   2.40524054, 2.41204119, 2.41897011, 2.42614841, 2.43373775,
     &   2.44152617, 2.44890332, 2.45648527, 2.46426678, 2.47228074,
     &   2.48045182, 2.48877716, 2.49723387, 2.50568938, 2.51434588,
     &   2.52303314, 2.53166080, 2.53969908, 2.54731417, 2.55385303,
     &   2.55962181, 2.56454444, 2.56850100, 2.57175994, 2.57437587,
     &   2.57612371, 2.57655454, 2.57562017, 2.57356119, 2.57069206,
     &   2.56618571, 2.56010008, 2.55461764, 2.54953837, 2.54486489,
     &   2.53999114, 2.53607583, 2.53363347, 2.53294373, 2.53321624,
     &   2.53469968, 2.53739071, 2.54130602, 2.54703045, 2.55498147,
     &   2.56313658, 2.57103634, 2.57950401, 2.58897758, 2.59869242,
     &   2.60991764, 2.62091136, 2.63133740, 2.64163065, 2.65232992,
     &   2.66293454, 2.67423081, 2.68534708, 2.69581771, 2.70595789,
     &   2.71534753, 2.72475457, 2.73420143, 2.74357867, 2.75287127,
     &   2.76181054, 2.77036881, 2.77837110, 2.78587103, 2.79317403,
     &   2.80051494, 2.80838895, 2.81610394, 2.82264066, 2.82867074,
     &   2.83428907, 2.83938265, 2.84270954, 2.84484768, 2.84613419,
     &   2.84651518, 2.84530306, 2.84355998, 2.84073520, 2.83598948,
     &   2.82923722, 2.82062960, 2.80869079, 2.79536510, 2.78061485,
     &   2.76440644, 2.74670935, 2.72556996, 2.70271802, 2.67859459,
     &   2.65320754, 2.62656832, 2.60023999, 2.57302022, 2.54469728,
     &   2.51528883, 2.48479724, 2.45322466, 2.42064023, 2.38707280,
     &   2.35259223, 2.31751513, 2.28167486, 2.24510956, 2.20804667,
     &   2.17101455, 2.13340902, 2.09527230, 2.05666113, 2.01763201,
     &   1.97822452, 1.93874037, 1.90014684, 1.86139333, 1.82252336,
     &   1.78476572, 1.74702537, 1.70924222, 1.67133427, 1.63338828,
     &   1.59547508, 1.55777061, 1.52020788, 1.48288810, 1.44607472,
     &   1.40951252, 1.37331140, 1.33748305, 1.30201316, 1.26772225,
     &   1.23385918, 1.20018911, 1.16681325, 1.13374329, 1.10077119,
     &   1.06835234, 1.03668070, 1.00561583, 0.97514355, 0.94532347,
     &   0.91643113, 0.88825256, 0.86075085, 0.83395207, 0.80823338,
     &   0.78322130, 0.75891024, 0.73527455, 0.71228892, 0.69000363/
      data (qq(i,10),i=1,mxwv)/
     &   2.02147675, 2.02172947, 2.02200747, 2.02225709, 2.02272439,
     &   2.02278566, 2.02306294, 2.02334714, 2.02374697, 2.02388453,
     &   2.02417088, 2.02445960, 2.02470636, 2.02502751, 2.02532864,
     &   2.02562976, 2.02592611, 2.02623630, 2.02637529, 2.02685618,
     &   2.02720785, 2.02751517, 2.02783895, 2.02816486, 2.02848506,
     &   2.02883649, 2.02917671, 2.02951646, 2.02990079, 2.03023982,
     &   2.03061032, 2.03095460, 2.03132558, 2.03167629, 2.03206921,
     &   2.03244925, 2.03284502, 2.03321195, 2.03361368, 2.03397870,
     &   2.03441906, 2.03482938, 2.03522515, 2.03564739, 2.03609538,
     &   2.03650141, 2.03692579, 2.03738785, 2.03781891, 2.03828096,
     &   2.03873539, 2.03917623, 2.03963327, 2.04010248, 2.04059672,
     &   2.04103661, 2.04158521, 2.04216194, 2.04254341, 2.04300737,
     &   2.04357433, 2.04411316, 2.04460979, 2.04514980, 2.04569697,
     &   2.04623652, 2.04676580, 2.04735374, 2.04789972, 2.04850340,
     &   2.04903936, 2.04963541, 2.05018258, 2.05078959, 2.05140066,
     &   2.05199862, 2.05261564, 2.05328894, 2.05387568, 2.05455470,
     &   2.05519652, 2.05587029, 2.05656505, 2.05728483, 2.05786777,
     &   2.05857587, 2.05926442, 2.05999112, 2.06070018, 2.06142879,
     &   2.06215525, 2.06289697, 2.06365967, 2.06441450, 2.06519032,
     &   2.06597161, 2.06676960, 2.06757021, 2.06805110, 2.06920528,
     &   2.07003784, 2.07087803, 2.07172990, 2.07255316, 2.07346630,
     &   2.07436180, 2.07523084, 2.07613683, 2.07700896, 2.07796621,
     &   2.07893777, 2.07991219, 2.08089685, 2.08189154, 2.08290386,
     &   2.08392835, 2.08496237, 2.08601975, 2.08709145, 2.08817816,
     &   2.08928275, 2.09040236, 2.09154844, 2.09276652, 2.09404469,
     &   2.09526467, 2.09566092, 2.09030700, 2.07687855, 2.07135296,
     &   2.07810140, 2.08545446, 2.09084129, 2.09486294, 2.09818316,
     &   2.10104322, 2.10341430, 2.10563540, 2.10772562, 2.10953021,
     &   2.11126280, 2.11295271, 2.11458302, 2.11616135, 2.11771369,
     &   2.11924291, 2.12076020, 2.12227511, 2.12375855, 2.12530088,
     &   2.12682295, 2.12833786, 2.12985158, 2.13135386, 2.13284683,
     &   2.13433123, 2.13579583, 2.13728881, 2.13895106, 2.14075899,
     &   2.14265704, 2.14459348, 2.14660358, 2.14864445, 2.15070367,
     &   2.15273690, 2.15469122, 2.15656996, 2.15815496, 2.15891409,
     &   2.15654969, 2.14837670, 2.14497161, 2.15331197, 2.16165018,
     &   2.16607857, 2.16922259, 2.17204762, 2.17463875, 2.17703485,
     &   2.17937994, 2.18169975, 2.18399501, 2.18632078, 2.18865323,
     &   2.19101191, 2.19336748, 2.19570541, 2.19808841, 2.20047307,
     &   2.20286083, 2.20525622, 2.20764041, 2.21001387, 2.21236968,
     &   2.21473122, 2.21712136, 2.21956944, 2.22216797, 2.22494698,
     &   2.22808456, 2.23142910, 2.23398280, 2.23227978, 2.22178268,
     &   2.19836807, 2.16905904, 2.13491702, 2.10834002, 2.09336996,
     &   2.08833480, 2.09429550, 2.10436440, 2.11691260, 2.12900496,
     &   2.13970351, 2.15051818, 2.16066551, 2.16996241, 2.17866540,
     &   2.18697286, 2.19432569, 2.20147085, 2.20828986, 2.21488857,
     &   2.22109628, 2.22750258, 2.23381543, 2.23990011, 2.24577570,
     &   2.25179362, 2.25752401, 2.26327920, 2.26870084, 2.27403426,
     &   2.27958965, 2.28507376, 2.29045272, 2.29553080, 2.30046177,
     &   2.30516911, 2.30967855, 2.31427073, 2.31894326, 2.32369304,
     &   2.32851934, 2.33350444, 2.33857179, 2.34374261, 2.34908938,
     &   2.35454559, 2.35974050, 2.36504745, 2.37043333, 2.37592864,
     &   2.38148284, 2.38708806, 2.39275408, 2.39840865, 2.40417171,
     &   2.40995002, 2.41574812, 2.42133713, 2.42684722, 2.43201113,
     &   2.43699145, 2.44175911, 2.44625258, 2.45058703, 2.45478177,
     &   2.45872808, 2.46219563, 2.46508646, 2.46748281, 2.46954989,
     &   2.47072363, 2.47093153, 2.47116423, 2.47148681, 2.47192168,
     &   2.47201514, 2.47238588, 2.47341490, 2.47540450, 2.47787404,
     &   2.48097515, 2.48476171, 2.48926878, 2.49473619, 2.50166845,
     &   2.50889707, 2.51607442, 2.52357793, 2.53162980, 2.53991556,
     &   2.54929423, 2.55859137, 2.56771016, 2.57680202, 2.58616376,
     &   2.59553075, 2.60547280, 2.61539578, 2.62523365, 2.63498187,
     &   2.64442420, 2.65400290, 2.66373968, 2.67362142, 2.68362069,
     &   2.69372225, 2.70382214, 2.71385217, 2.72384548, 2.73389602,
     &   2.74401331, 2.75489259, 2.76610613, 2.77710152, 2.78814507,
     &   2.79942012, 2.81100869, 2.82224822, 2.83312082, 2.84403372,
     &   2.85501623, 2.86541033, 2.87603235, 2.88683295, 2.89709496,
     &   2.90660906, 2.91556454, 2.92318439, 2.93033242, 2.93695211,
     &   2.94298434, 2.94836855, 2.95001125, 2.95014143, 2.94935536,
     &   2.94760919, 2.94485998, 2.94154096, 2.93731356, 2.93207979,
     &   2.92580795, 2.91844463, 2.90993023, 2.90029001, 2.88949943,
     &   2.87743449, 2.86355138, 2.84852409, 2.83234382, 2.81507993,
     &   2.79694867, 2.77770758, 2.75735736, 2.73581886, 2.71312761,
     &   2.68935728, 2.66456842, 2.63901377, 2.61253667, 2.58515596,
     &   2.55817580, 2.53045917, 2.50191736, 2.47287703, 2.44308758,
     &   2.41248536, 2.38108945, 2.34894276, 2.31613445, 2.28290224,
     &   2.24901462, 2.21461558, 2.17970276, 2.14422512, 2.10903907,
     &   2.07339764, 2.03740549, 2.00100780, 1.96401107, 1.92618895,
     &   1.88799989, 1.84947896, 1.81071222, 1.77165866, 1.73245895,
     &   1.69337583, 1.65428913, 1.61518610, 1.57615292, 1.53774071,
     &   1.49950778, 1.46149075, 1.42369270, 1.38611686, 1.34889627/
      data (qq(i,11),i=1,mxwv)/
     &   2.01705885, 2.01725364, 2.01745462, 2.01768255, 2.01787853,
     &   2.01808667, 2.01830935, 2.01851845, 2.01874614, 2.01896572,
     &   2.01919580, 2.01940870, 2.01964378, 2.01986694, 2.02000594,
     &   2.02033734, 2.02058840, 2.02082372, 2.02108407, 2.02132821,
     &   2.02157927, 2.02183557, 2.02211046, 2.02240729, 2.02262783,
     &   2.02290177, 2.02315354, 2.02345300, 2.02372336, 2.02400041,
     &   2.02428150, 2.02456665, 2.02484894, 2.02514815, 2.02541971,
     &   2.02577281, 2.02604866, 2.02636933, 2.02667022, 2.02699304,
     &   2.02731180, 2.02764153, 2.02795672, 2.02831459, 2.02862978,
     &   2.02898359, 2.02930617, 2.02966785, 2.03001499, 2.03036737,
     &   2.03072119, 2.03110361, 2.03145957, 2.03184032, 2.03218198,
     &   2.03258729, 2.03298640, 2.03337836, 2.03378534, 2.03416300,
     &   2.03457975, 2.03505421, 2.03540301, 2.03583026, 2.03625107,
     &   2.03667498, 2.03708100, 2.03755593, 2.03800225, 2.03844643,
     &   2.03888988, 2.03937101, 2.03984165, 2.04030776, 2.04077005,
     &   2.04127884, 2.04175138, 2.04224873, 2.04326868, 2.04326773,
     &   2.04378319, 2.04430485, 2.04483294, 2.04535961, 2.04589963,
     &   2.04642892, 2.04700041, 2.04755235, 2.04815197, 2.04868436,
     &   2.04926920, 2.04985523, 2.05044985, 2.05104470, 2.05166435,
     &   2.05228090, 2.05290651, 2.05305171, 2.05418038, 2.05482459,
     &   2.05548382, 2.05614471, 2.05681682, 2.05749893, 2.05818629,
     &   2.05889821, 2.05957127, 2.06025076, 2.06097293, 2.06172323,
     &   2.06248307, 2.06325054, 2.06401348, 2.06479549, 2.06558466,
     &   2.06638288, 2.06718707, 2.06800556, 2.06884456, 2.06968784,
     &   2.07054210, 2.07140565, 2.07228446, 2.07321072, 2.07416940,
     &   2.07505846, 2.07533360, 2.07151031, 2.06256914, 2.05916095,
     &   2.06403255, 2.06930351, 2.07318449, 2.07610846, 2.07852530,
     &   2.08060145, 2.08233595, 2.08396721, 2.08551097, 2.08686185,
     &   2.08816886, 2.08944821, 2.09068918, 2.09189606, 2.09308791,
     &   2.09426498, 2.09543610, 2.09660673, 2.09778023, 2.09895015,
     &   2.10013032, 2.10130668, 2.10248327, 2.10365391, 2.10481906,
     &   2.10598135, 2.10713482, 2.10830593, 2.10960007, 2.11099339,
     &   2.11244678, 2.11392617, 2.11545205, 2.11699677, 2.11854935,
     &   2.12008023, 2.12154722, 2.12295270, 2.12413621, 2.12472034,
     &   2.12314391, 2.11776686, 2.11604309, 2.12218595, 2.12801600,
     &   2.13115859, 2.13344598, 2.13552523, 2.13744807, 2.13924336,
     &   2.14100862, 2.14275813, 2.14449525, 2.14625454, 2.14802027,
     &   2.14980531, 2.15158749, 2.15336061, 2.15516567, 2.15697217,
     &   2.15878057, 2.16059065, 2.16238642, 2.16415977, 2.16589522,
     &   2.16758633, 2.16921687, 2.17075753, 2.17218137, 2.17338824,
     &   2.17434716, 2.17466402, 2.17420268, 2.17114353, 2.16479421,
     &   2.15297270, 2.13799548, 2.11934972, 2.10381079, 2.09455538,
     &   2.09134841, 2.09515166, 2.10185766, 2.11036015, 2.11869526,
     &   2.12618995, 2.13381004, 2.14102221, 2.14768505, 2.15397120,
     &   2.15999866, 2.16539121, 2.17065954, 2.17570567, 2.18060827,
     &   2.18526149, 2.19004965, 2.19477868, 2.19934440, 2.20377803,
     &   2.20830345, 2.21261501, 2.21695566, 2.22104788, 2.22509933,
     &   2.22929955, 2.23341012, 2.23745131, 2.24128985, 2.24500942,
     &   2.24854350, 2.25201011, 2.25553417, 2.25911832, 2.26275969,
     &   2.26645422, 2.27025962, 2.27411914, 2.27803493, 2.28202653,
     &   2.28609037, 2.28997755, 2.29393578, 2.29792714, 2.30197620,
     &   2.30604458, 2.31012011, 2.31421614, 2.31827736, 2.32238030,
     &   2.32644296, 2.33075094, 2.33436775, 2.33818626, 2.34180117,
     &   2.34533000, 2.34876776, 2.35211039, 2.35541320, 2.35869050,
     &   2.36188960, 2.36490631, 2.36770964, 2.37034869, 2.37288904,
     &   2.37506819, 2.37684941, 2.37873340, 2.38070869, 2.38277984,
     &   2.38472009, 2.38684535, 2.38934875, 2.39238524, 2.39571071,
     &   2.39941144, 2.40351701, 2.40804791, 2.41315484, 2.41907406,
     &   2.42516589, 2.43125391, 2.43755817, 2.44420242, 2.45115590,
     &   2.45832825, 2.46563673, 2.47284532, 2.48005819, 2.48742580,
     &   2.49481058, 2.50246072, 2.51010942, 2.51769543, 2.52526116,
     &   2.53270960, 2.54024839, 2.54788971, 2.55562377, 2.56344628,
     &   2.57135105, 2.57930446, 2.58728361, 2.59531140, 2.60342836,
     &   2.61162901, 2.62023425, 2.62907195, 2.63793468, 2.64692616,
     &   2.65614700, 2.66566586, 2.67525935, 2.68487096, 2.69471288,
     &   2.70482683, 2.71492600, 2.72536993, 2.73623919, 2.74724770,
     &   2.75830412, 2.76953673, 2.78077245, 2.79221869, 2.80386710,
     &   2.81570601, 2.82772279, 2.83812761, 2.84811020, 2.85800552,
     &   2.86778307, 2.87740993, 2.88667798, 2.89571810, 2.90452456,
     &   2.91306305, 2.92128515, 2.92913294, 2.93659782, 2.94363856,
     &   2.95009565, 2.95525002, 2.95982337, 2.96377683, 2.96704388,
     &   2.96953249, 2.97130919, 2.97233939, 2.97248626, 2.97173834,
     &   2.97013807, 2.96753669, 2.96352601, 2.95861840, 2.95279360,
     &   2.94672322, 2.93982029, 2.93201256, 2.92378259, 2.91474605,
     &   2.90475702, 2.89367962, 2.88161755, 2.86859179, 2.85467625,
     &   2.83977580, 2.82397890, 2.80725908, 2.78954577, 2.77122831,
     &   2.75196815, 2.73217726, 2.71165991, 2.69008732, 2.66732550,
     &   2.64353466, 2.61848617, 2.59245682, 2.56535125, 2.53728437,
     &   2.50827956, 2.47837162, 2.44753695, 2.41583848, 2.38366485,
     &   2.35071254, 2.31700206, 2.28252101, 2.24725580, 2.21132469/
      data (qq(i,12),i=1,mxwv)/
     &   2.01354194, 2.01370192, 2.01387286, 2.01403880, 2.01418281,
     &   2.01436043, 2.01453876, 2.01471233, 2.01484990, 2.01505995,
     &   2.01522923, 2.01541471, 2.01556444, 2.01577878, 2.01596642,
     &   2.01615214, 2.01634836, 2.01653957, 2.01673484, 2.01694298,
     &   2.01714182, 2.01734066, 2.01754427, 2.01774359, 2.01795840,
     &   2.01817441, 2.01839352, 2.01859856, 2.01882958, 2.01904893,
     &   2.01927423, 2.01949430, 2.01974797, 2.01996326, 2.02019763,
     &   2.02045727, 2.02066445, 2.02092195, 2.02117181, 2.02143431,
     &   2.02168846, 2.02193904, 2.02217889, 2.02245903, 2.02273178,
     &   2.02296925, 2.02329850, 2.02360320, 2.02382398, 2.02410507,
     &   2.02439094, 2.02467680, 2.02496719, 2.02526641, 2.02556205,
     &   2.02587008, 2.02618313, 2.02647424, 2.02680731, 2.02711105,
     &   2.02743530, 2.02776122, 2.02809978, 2.02843332, 2.02876091,
     &   2.02911019, 2.02945375, 2.02979088, 2.03015113, 2.03053951,
     &   2.03086782, 2.03122282, 2.03163075, 2.03199458, 2.03236508,
     &   2.03275037, 2.03313732, 2.03351712, 2.03392220, 2.03432822,
     &   2.03473234, 2.03514481, 2.03556418, 2.03601098, 2.03573537,
     &   2.03665733, 2.03727484, 2.03772402, 2.03816867, 2.03862405,
     &   2.03908014, 2.03954363, 2.04001403, 2.04049110, 2.04097748,
     &   2.04146075, 2.04195523, 2.04245496, 2.04293537, 2.04347205,
     &   2.04398918, 2.04451394, 2.04504442, 2.04558969, 2.04612589,
     &   2.04667521, 2.04721856, 2.04775882, 2.04832792, 2.04891753,
     &   2.04930472, 2.05011559, 2.05071902, 2.05133128, 2.05195117,
     &   2.05257821, 2.05322456, 2.05385518, 2.05450654, 2.05516481,
     &   2.05583143, 2.05650377, 2.05718660, 2.05790234, 2.05862808,
     &   2.05930161, 2.05951357, 2.05682778, 2.05083275, 2.04870749,
     &   2.05223417, 2.05602813, 2.05883932, 2.06097865, 2.06275249,
     &   2.06427646, 2.06555915, 2.06677341, 2.06792712, 2.06895065,
     &   2.06994724, 2.07092595, 2.07188034, 2.07281256, 2.07373571,
     &   2.07465053, 2.07556295, 2.07647538, 2.07738900, 2.07830715,
     &   2.07922912, 2.08015084, 2.08107376, 2.08199382, 2.08291149,
     &   2.08382845, 2.08474112, 2.08566976, 2.08668423, 2.08776784,
     &   2.08888865, 2.09002566, 2.09120727, 2.09239244, 2.09358048,
     &   2.09475064, 2.09587455, 2.09694982, 2.09786034, 2.09833956,
     &   2.09732151, 2.09380412, 2.09299684, 2.09751034, 2.10163355,
     &   2.10390115, 2.10559559, 2.10715199, 2.10860491, 2.10996699,
     &   2.11132312, 2.11266422, 2.11399794, 2.11534739, 2.11670160,
     &   2.11806822, 2.11943245, 2.12078977, 2.12216830, 2.12354541,
     &   2.12492156, 2.12629676, 2.12765837, 2.12900043, 2.13031244,
     &   2.13158584, 2.13281083, 2.13396049, 2.13500476, 2.13586116,
     &   2.13647676, 2.13649225, 2.13581681, 2.13306594, 2.12832260,
     &   2.12079525, 2.11196542, 2.10119700, 2.09220481, 2.08685875,
     &   2.08516884, 2.08795071, 2.09267259, 2.09863901, 2.10454035,
     &   2.10991311, 2.11539102, 2.12061048, 2.12546515, 2.13007569,
     &   2.13451409, 2.13852310, 2.14245749, 2.14623904, 2.14992714,
     &   2.15345311, 2.15707421, 2.16065788, 2.16412592, 2.16750979,
     &   2.17095590, 2.17424536, 2.17756319, 2.18069887, 2.18381834,
     &   2.18704057, 2.19017982, 2.19327331, 2.19622922, 2.19909477,
     &   2.20181513, 2.20452785, 2.20728302, 2.21008420, 2.21292877,
     &   2.21581316, 2.21877766, 2.22178245, 2.22481823, 2.22788811,
     &   2.23101044, 2.23401546, 2.23707080, 2.24014401, 2.24325418,
     &   2.24637413, 2.24949312, 2.25262475, 2.25572896, 2.25885701,
     &   2.26194549, 2.26501775, 2.26797199, 2.27088380, 2.27366400,
     &   2.27639127, 2.27906370, 2.28168869, 2.28429365, 2.28688741,
     &   2.28943634, 2.29188061, 2.29422760, 2.29649925, 2.29872537,
     &   2.30079174, 2.30268431, 2.30472851, 2.30687165, 2.30911231,
     &   2.31134677, 2.31376576, 2.31646466, 2.31950784, 2.32276392,
     &   2.32628655, 2.33008361, 2.33416390, 2.33864856, 2.34359741,
     &   2.34862804, 2.35366130, 2.35885739, 2.36429834, 2.36982322,
     &   2.37560272, 2.38137245, 2.38705826, 2.39275789, 2.39855576,
     &   2.40436959, 2.41028070, 2.41619015, 2.42200470, 2.42781734,
     &   2.43357825, 2.43938923, 2.44525504, 2.45116496, 2.45711613,
     &   2.46308756, 2.46908975, 2.47510314, 2.48114920, 2.48725939,
     &   2.49344921, 2.49979877, 2.50624394, 2.51269889, 2.51922631,
     &   2.52585864, 2.53261685, 2.53941369, 2.54625821, 2.55322480,
     &   2.56033158, 2.56748462, 2.57481694, 2.58235693, 2.59001565,
     &   2.59777403, 2.60568953, 2.61373115, 2.62197328, 2.63042545,
     &   2.63909674, 2.64799666, 2.65652609, 2.66509795, 2.67383909,
     &   2.68275094, 2.69183421, 2.70092940, 2.71015334, 2.71952677,
     &   2.72904682, 2.73870564, 2.74849200, 2.75840831, 2.76844668,
     &   2.77854156, 2.78834128, 2.79818606, 2.80806017, 2.81790709,
     &   2.82760501, 2.83726692, 2.84687138, 2.85634303, 2.86566401,
     &   2.87484646, 2.88375044, 2.89194131, 2.89987636, 2.90752697,
     &   2.91495919, 2.92209101, 2.92888808, 2.93564558, 2.94212103,
     &   2.94820404, 2.95375085, 2.95883107, 2.96341419, 2.96744323,
     &   2.97093081, 2.97388268, 2.97626638, 2.97802806, 2.97907424,
     &   2.97946358, 2.97957826, 2.97924423, 2.97822189, 2.97651219,
     &   2.97399735, 2.97038054, 2.96591520, 2.96051526, 2.95421124,
     &   2.94682193, 2.93847179, 2.92913508, 2.91881537, 2.90753913,
     &   2.89527369, 2.88200498, 2.86770368, 2.85233641, 2.83592820/
      data (qq(i,13),i=1,mxwv)/
     &   2.01075816, 2.01087856, 2.01101637, 2.01113558, 2.01126814,
     &   2.01140094, 2.01153755, 2.01167440, 2.01181149, 2.01195431,
     &   2.01210046, 2.01223683, 2.01238489, 2.01252913, 2.01267481,
     &   2.01283646, 2.01297617, 2.01313329, 2.01329041, 2.01343536,
     &   2.01360726, 2.01375556, 2.01392603, 2.01410127, 2.01426005,
     &   2.01442409, 2.01459432, 2.01479340, 2.01494479, 2.01512313,
     &   2.01530075, 2.01548648, 2.01566434, 2.01585054, 2.01605654,
     &   2.01622248, 2.01641464, 2.01661777, 2.01682472, 2.01701951,
     &   2.01725245, 2.01740909, 2.01762247, 2.01782322, 2.01803946,
     &   2.01825190, 2.01847672, 2.01869965, 2.01890802, 2.01913238,
     &   2.01936293, 2.01958942, 2.01982522, 2.02006221, 2.02029228,
     &   2.02053905, 2.02077842, 2.02102613, 2.02127361, 2.02154088,
     &   2.02178240, 2.02203512, 2.02231574, 2.02254009, 2.02283263,
     &   2.02309966, 2.02337050, 2.02365112, 2.02392817, 2.02422071,
     &   2.02450013, 2.02479649, 2.02508092, 2.02537847, 2.02568221,
     &   2.02598739, 2.02629662, 2.02660894, 2.02685738, 2.02724266,
     &   2.02756715, 2.02788925, 2.02822113, 2.02855563, 2.02889562,
     &   2.02923989, 2.02958441, 2.02993560, 2.03008676, 2.03064799,
     &   2.03100967, 2.03140903, 2.03172946, 2.03212833, 2.03251147,
     &   2.03289580, 2.03328657, 2.03368449, 2.03408241, 2.03448749,
     &   2.03489637, 2.03531194, 2.03573155, 2.03615618, 2.03657889,
     &   2.03704977, 2.03745079, 2.03788042, 2.03833055, 2.03879523,
     &   2.03926849, 2.03973937, 2.04021645, 2.04069924, 2.04118729,
     &   2.04168105, 2.04218030, 2.04268622, 2.04319835, 2.04371572,
     &   2.04423857, 2.04476595, 2.04530001, 2.04585171, 2.04641509,
     &   2.04693317, 2.04711390, 2.04524899, 2.04120779, 2.03987670,
     &   2.04243565, 2.04517651, 2.04722238, 2.04879594, 2.05010700,
     &   2.05123615, 2.05219507, 2.05310750, 2.05397844, 2.05476141,
     &   2.05552769, 2.05628252, 2.05702209, 2.05774736, 2.05847025,
     &   2.05917621, 2.05989480, 2.06061459, 2.06133437, 2.06205368,
     &   2.06277990, 2.06350613, 2.06423378, 2.06496143, 2.06568789,
     &   2.06641531, 2.06714153, 2.06787992, 2.06868005, 2.06952834,
     &   2.07040501, 2.07129335, 2.07220292, 2.07312107, 2.07404017,
     &   2.07494545, 2.07581830, 2.07665396, 2.07736993, 2.07777882,
     &   2.07714748, 2.07485414, 2.07453990, 2.07785678, 2.08080339,
     &   2.08246446, 2.08373833, 2.08492041, 2.08603382, 2.08709073,
     &   2.08813787, 2.08917785, 2.09021831, 2.09126806, 2.09231973,
     &   2.09338188, 2.09444237, 2.09549809, 2.09656811, 2.09763646,
     &   2.09870362, 2.09976792, 2.10082102, 2.10185814, 2.10287237,
     &   2.10385561, 2.10479951, 2.10568666, 2.10648942, 2.10715151,
     &   2.10763121, 2.10766459, 2.10720873, 2.10527134, 2.10201001,
     &   2.09705687, 2.09148693, 2.08488345, 2.07950282, 2.07641959,
     &   2.07563186, 2.07776237, 2.08118582, 2.08546495, 2.08971786,
     &   2.09362769, 2.09761906, 2.10144258, 2.10501957, 2.10843611,
     &   2.11173701, 2.11474466, 2.11770749, 2.12056518, 2.12336159,
     &   2.12605214, 2.12881041, 2.13154626, 2.13420081, 2.13680124,
     &   2.13944578, 2.14197707, 2.14453387, 2.14695859, 2.14937949,
     &   2.15187311, 2.15429735, 2.15669227, 2.15899301, 2.16122675,
     &   2.16335034, 2.16549182, 2.16766453, 2.16987348, 2.17211652,
     &   2.17438984, 2.17672133, 2.17908382, 2.18146420, 2.18385887,
     &   2.18629193, 2.18864775, 2.19104075, 2.19344449, 2.19587374,
     &   2.19830990, 2.20074391, 2.20318723, 2.20561171, 2.20805478,
     &   2.21046710, 2.21286941, 2.21519947, 2.21750164, 2.21972036,
     &   2.22191501, 2.22408152, 2.22622609, 2.22836637, 2.23050308,
     &   2.23261380, 2.23466420, 2.23666525, 2.23862982, 2.24056983,
     &   2.24243021, 2.24420714, 2.24612355, 2.24812412, 2.25020623,
     &   2.25232267, 2.25459886, 2.25708771, 2.25981355, 2.26270223,
     &   2.26578879, 2.26907277, 2.27255750, 2.27633905, 2.28040624,
     &   2.28451324, 2.28862596, 2.29286504, 2.29728794, 2.30176497,
     &   2.30636215, 2.31095672, 2.31548572, 2.32003284, 2.32464814,
     &   2.32927990, 2.33393955, 2.33860064, 2.34316754, 2.34774184,
     &   2.35229516, 2.35688114, 2.36150122, 2.36614299, 2.37081361,
     &   2.37547946, 2.38016772, 2.38486385, 2.38958287, 2.39435244,
     &   2.39919376, 2.40409994, 2.40904832, 2.41399813, 2.41899395,
     &   2.42404270, 2.42914653, 2.43426108, 2.43941689, 2.44463658,
     &   2.44992590, 2.45525169, 2.46066976, 2.46618342, 2.47174382,
     &   2.47736454, 2.48304725, 2.48877788, 2.49460673, 2.50053763,
     &   2.50657392, 2.51271939, 2.51875329, 2.52483582, 2.53101707,
     &   2.53730130, 2.54369259, 2.55015254, 2.55671263, 2.56338263,
     &   2.57016659, 2.57706714, 2.58408689, 2.59123278, 2.59850955,
     &   2.60590315, 2.61330748, 2.62084007, 2.62850451, 2.63628745,
     &   2.64414763, 2.65213752, 2.66025829, 2.66849399, 2.67684627,
     &   2.68532658, 2.69388723, 2.70235443, 2.71091461, 2.71956277,
     &   2.72824407, 2.73699689, 2.74581790, 2.75480890, 2.76389456,
     &   2.77304029, 2.78218746, 2.79136539, 2.80055547, 2.80970764,
     &   2.81885386, 2.82798076, 2.83707476, 2.84611869, 2.85493803,
     &   2.86366272, 2.87246227, 2.88126445, 2.88998961, 2.89868832,
     &   2.90722013, 2.91542506, 2.92341113, 2.93114090, 2.93859768,
     &   2.94561195, 2.95227122, 2.95855927, 2.96444917, 2.96978474,
     &   2.97466135, 2.97905111, 2.98292804, 2.98626351, 2.98901725/

c        ---------------- single scatter albedo ---------------

      data (ww(i,1),i=1,mxwv)/
     &   0.99999952, 0.99999958, 0.99999964, 0.99999964, 0.99999970,
     &   0.99999970, 0.99999976, 0.99999976, 0.99999976, 0.99999976,
     &   0.99999982, 0.99999982, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999994, 0.99999994, 0.99999994, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999994, 0.99999994, 0.99999994,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999988, 0.99999982,
     &   0.99999976, 0.99999964, 0.99999946, 0.99999940, 0.99999940,
     &   0.99999940, 0.99999934, 0.99999923, 0.99999917, 0.99999905,
     &   0.99999875, 0.99999797, 0.99999660, 0.99999517, 0.99999487,
     &   0.99999499, 0.99999559, 0.99999613, 0.99999607, 0.99999499,
     &   0.99999338, 0.99999148, 0.99999005, 0.99998879, 0.99998742,
     &   0.99998277, 0.99996358, 0.99993312, 0.99991757, 0.99992037,
     &   0.99993020, 0.99994451, 0.99995929, 0.99997067, 0.99997354,
     &   0.99996907, 0.99995905, 0.99992585, 0.99983281, 0.99979675,
     &   0.99979216, 0.99979687, 0.99980885, 0.99982476, 0.99982220,
     &   0.99978274, 0.99970102, 0.99943924, 0.99916118, 0.99816114,
     &   0.99572027, 0.99538642, 0.99598861, 0.99725771, 0.99804133,
     &   0.99851036, 0.99880695, 0.99903840, 0.99916226, 0.99924058,
     &   0.99922901, 0.99915224, 0.99901688, 0.99877852, 0.99873787,
     &   0.99808365, 0.99248528, 0.98448712, 0.98503619, 0.98936582,
     &   0.99313819, 0.99495143, 0.99624306, 0.99692607, 0.99727446,
     &   0.99735862, 0.99709493, 0.99653637, 0.99570632, 0.99422938,
     &   0.99204558, 0.98870075, 0.98481202, 0.98276567, 0.97890717,
     &   0.95922488, 0.87456477, 0.60092169, 0.39636961, 0.35611215,
     &   0.38785917, 0.42182928, 0.45341501, 0.48493326, 0.52896410,
     &   0.59844899, 0.67909014, 0.76488978, 0.84285098, 0.88823092,
     &   0.92191178, 0.94717973, 0.96350604, 0.97250253, 0.97750479,
     &   0.97891426, 0.97817403, 0.97593778, 0.97256535, 0.96757478,
     &   0.96229076, 0.95535046, 0.94727427, 0.93761307, 0.92654824,
     &   0.91459662, 0.90150052, 0.88984787, 0.88724649, 0.89048690,
     &   0.89601666, 0.90116936, 0.90720439, 0.91192007, 0.91372395,
     &   0.91012162, 0.89826524, 0.87755698, 0.83687615, 0.75934839,
     &   0.60984266, 0.44399109, 0.40469515, 0.49654633, 0.62870169,
     &   0.68542868, 0.70718223, 0.71863103, 0.72157675, 0.71745110,
     &   0.71096838, 0.70258021, 0.69270551, 0.68257952, 0.67147034,
     &   0.65969020, 0.64646697, 0.63167769, 0.61690199, 0.60082132,
     &   0.58359200, 0.56517518, 0.54522538, 0.52371144, 0.50063592,
     &   0.47589877, 0.44957092, 0.42163384, 0.39161080, 0.35907152,
     &   0.32457307, 0.28465000, 0.24550866, 0.19957438, 0.16065010,
     &   0.12966332, 0.10869911, 0.09359177, 0.08540367, 0.08223289,
     &   0.08193011, 0.08527038, 0.08943519, 0.09422144, 0.09848119,
     &   0.10209934, 0.10566373, 0.10885277, 0.11145543, 0.11376107,
     &   0.11578444, 0.11712978, 0.11838754, 0.11935266, 0.12015285,
     &   0.12065148, 0.12131826, 0.12191113, 0.12227121, 0.12243631,
     &   0.12274905, 0.12273871, 0.12276960, 0.12238671, 0.12188148,
     &   0.12162974, 0.12124781, 0.12069120, 0.11966123, 0.11832535,
     &   0.11652559, 0.11441365, 0.11232819, 0.11027414, 0.10824618,
     &   0.10623882, 0.10437321, 0.10253117, 0.10071009, 0.09893207,
     &   0.09719741, 0.09498110, 0.09280986, 0.09058782, 0.08836666,
     &   0.08606426, 0.08364838, 0.08116627, 0.07851537, 0.07584449,
     &   0.07299417, 0.07006160, 0.06685416, 0.06360495, 0.06020356,
     &   0.05685322, 0.05357836, 0.05042693, 0.04742674, 0.04458369,
     &   0.04186176, 0.03922241, 0.03670470, 0.03432534, 0.03208752,
     &   0.02995682, 0.02793958, 0.02618165, 0.02459365, 0.02315000,
     &   0.02181572, 0.02063180, 0.01959152, 0.01867770, 0.01784808,
     &   0.01710322, 0.01643152, 0.01582579, 0.01530599, 0.01486420,
     &   0.01443098, 0.01399682, 0.01359221, 0.01322973, 0.01287905,
     &   0.01256918, 0.01225852, 0.01193481, 0.01161513, 0.01131371,
     &   0.01101675, 0.01073934, 0.01046383, 0.01017510, 0.00988740,
     &   0.00959218, 0.00930665, 0.00903108, 0.00876284, 0.00850220,
     &   0.00824333, 0.00798780, 0.00773332, 0.00748180, 0.00723900,
     &   0.00700847, 0.00679591, 0.00659144, 0.00638216, 0.00617820,
     &   0.00598140, 0.00579133, 0.00559511, 0.00539979, 0.00521059,
     &   0.00502781, 0.00484428, 0.00467142, 0.00450606, 0.00434081,
     &   0.00417539, 0.00401277, 0.00384502, 0.00368480, 0.00353176,
     &   0.00338558, 0.00324592, 0.00309221, 0.00294171, 0.00279859,
     &   0.00266246, 0.00253301, 0.00241182, 0.00229673, 0.00218714,
     &   0.00208281, 0.00198339, 0.00188853, 0.00179823, 0.00171227,
     &   0.00162999, 0.00154891, 0.00147186, 0.00139864, 0.00132906,
     &   0.00126291, 0.00120005, 0.00114032, 0.00108331, 0.00102897,
     &   0.00097735, 0.00092809, 0.00088032, 0.00083500, 0.00079201,
     &   0.00075214, 0.00071434, 0.00067843, 0.00064498, 0.00061332,
     &   0.00058321, 0.00055441, 0.00052701, 0.00050096, 0.00047619,
     &   0.00045264, 0.00043030, 0.00040908, 0.00038891, 0.00036973,
     &   0.00035149, 0.00033445, 0.00031841, 0.00030314, 0.00028863,
     &   0.00027480, 0.00026146, 0.00024876, 0.00023665, 0.00022511,
     &   0.00021406, 0.00020353, 0.00019351, 0.00018398, 0.00017489,
     &   0.00016624, 0.00015802, 0.00015020, 0.00014275, 0.00013568/
      data (ww(i,2),i=1,mxwv)/
     &   0.99999940, 0.99999946, 0.99999952, 0.99999952, 0.99999958,
     &   0.99999964, 0.99999964, 0.99999964, 0.99999970, 0.99999970,
     &   0.99999970, 0.99999976, 0.99999970, 0.99999976, 0.99999982,
     &   0.99999976, 0.99999982, 0.99999982, 0.99999982, 0.99999988,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999994, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999982, 0.99999982, 0.99999976, 0.99999976,
     &   0.99999964, 0.99999946, 0.99999928, 0.99999917, 0.99999911,
     &   0.99999911, 0.99999899, 0.99999887, 0.99999881, 0.99999863,
     &   0.99999815, 0.99999702, 0.99999499, 0.99999291, 0.99999249,
     &   0.99999267, 0.99999350, 0.99999428, 0.99999422, 0.99999255,
     &   0.99999017, 0.99998742, 0.99998528, 0.99998343, 0.99998128,
     &   0.99997431, 0.99994552, 0.99989980, 0.99987602, 0.99987978,
     &   0.99989426, 0.99991554, 0.99993777, 0.99995500, 0.99995917,
     &   0.99995196, 0.99993604, 0.99988347, 0.99973547, 0.99967641,
     &   0.99966687, 0.99967229, 0.99968952, 0.99971348, 0.99970734,
     &   0.99964023, 0.99950194, 0.99906087, 0.99858838, 0.99689466,
     &   0.99276316, 0.99217606, 0.99317580, 0.99532056, 0.99665207,
     &   0.99745274, 0.99796146, 0.99835998, 0.99857551, 0.99871409,
     &   0.99870151, 0.99858171, 0.99836808, 0.99799073, 0.99794477,
     &   0.99691707, 0.98812354, 0.97594953, 0.97705644, 0.98385495,
     &   0.98971093, 0.99254555, 0.99454623, 0.99561977, 0.99619257,
     &   0.99638742, 0.99611533, 0.99547791, 0.99453545, 0.99285358,
     &   0.99043518, 0.98684013, 0.98290801, 0.98133349, 0.97818160,
     &   0.96029204, 0.88629788, 0.65051126, 0.45335221, 0.40789646,
     &   0.42965531, 0.45185778, 0.47039634, 0.48706186, 0.51111752,
     &   0.55724645, 0.62581104, 0.71359581, 0.80470860, 0.86261636,
     &   0.90569735, 0.93765694, 0.95804328, 0.96924108, 0.97549391,
     &   0.97760212, 0.97735327, 0.97557360, 0.97271365, 0.96835762,
     &   0.96384770, 0.95789891, 0.95104575, 0.94289029, 0.93362486,
     &   0.92370141, 0.91287541, 0.90336311, 0.90186918, 0.90555465,
     &   0.91124725, 0.91657686, 0.92260861, 0.92745936, 0.92986423,
     &   0.92785108, 0.91924220, 0.90373415, 0.87244433, 0.81095517,
     &   0.68393046, 0.52605706, 0.47914237, 0.56073743, 0.68641496,
     &   0.74120396, 0.76305342, 0.77524769, 0.77978271, 0.77807325,
     &   0.77423626, 0.76871663, 0.76185495, 0.75474048, 0.74674398,
     &   0.73814428, 0.72825056, 0.71692353, 0.70551914, 0.69287282,
     &   0.67907649, 0.66405845, 0.64744067, 0.62911725, 0.60897517,
     &   0.58685577, 0.56266069, 0.53622192, 0.50689054, 0.47391659,
     &   0.43757614, 0.39347333, 0.34805253, 0.29157364, 0.24104093,
     &   0.19887660, 0.16943091, 0.14767697, 0.13574652, 0.13116366,
     &   0.13088927, 0.13592756, 0.14218697, 0.14930777, 0.15567274,
     &   0.16113099, 0.16650350, 0.17136121, 0.17544205, 0.17913465,
     &   0.18247540, 0.18494239, 0.18730941, 0.18932550, 0.19115071,
     &   0.19258994, 0.19428684, 0.19591540, 0.19727910, 0.19840063,
     &   0.19976383, 0.20075156, 0.20181023, 0.20236614, 0.20274742,
     &   0.20351337, 0.20419501, 0.20465152, 0.20444603, 0.20383513,
     &   0.20259498, 0.20075272, 0.19891614, 0.19708408, 0.19524902,
     &   0.19340676, 0.19174626, 0.19008461, 0.18844388, 0.18688156,
     &   0.18534511, 0.18294172, 0.18054605, 0.17800565, 0.17539743,
     &   0.17257227, 0.16946495, 0.16614150, 0.16240478, 0.15852132,
     &   0.15418683, 0.14956179, 0.14425512, 0.13870710, 0.13268358,
     &   0.12659618, 0.12050370, 0.11451343, 0.10870444, 0.10310488,
     &   0.09764888, 0.09225909, 0.08702935, 0.08201165, 0.07722890,
     &   0.07260516, 0.06816494, 0.06426598, 0.06071645, 0.05746546,
     &   0.05443297, 0.05172713, 0.04934153, 0.04724295, 0.04533013,
     &   0.04360836, 0.04205271, 0.04064818, 0.03944657, 0.03843470,
     &   0.03743645, 0.03642629, 0.03548203, 0.03463707, 0.03381536,
     &   0.03309447, 0.03236434, 0.03159376, 0.03082663, 0.03010079,
     &   0.02938045, 0.02870655, 0.02803204, 0.02731759, 0.02660035,
     &   0.02585787, 0.02513627, 0.02443675, 0.02375264, 0.02308489,
     &   0.02241828, 0.02175712, 0.02109545, 0.02043863, 0.01980242,
     &   0.01919672, 0.01863739, 0.01809792, 0.01754330, 0.01700119,
     &   0.01647685, 0.01596922, 0.01544324, 0.01491820, 0.01440852,
     &   0.01391513, 0.01341846, 0.01295000, 0.01250115, 0.01205164,
     &   0.01160076, 0.01115674, 0.01069771, 0.01025871, 0.00983886,
     &   0.00943731, 0.00905325, 0.00862958, 0.00821424, 0.00781882,
     &   0.00744237, 0.00708401, 0.00674826, 0.00642911, 0.00612499,
     &   0.00583521, 0.00555884, 0.00529499, 0.00504363, 0.00480419,
     &   0.00457485, 0.00434868, 0.00413362, 0.00392915, 0.00373473,
     &   0.00354981, 0.00337400, 0.00320684, 0.00304724, 0.00289504,
     &   0.00275041, 0.00261232, 0.00247835, 0.00235122, 0.00223057,
     &   0.00211866, 0.00201252, 0.00191166, 0.00181767, 0.00172872,
     &   0.00164408, 0.00156309, 0.00148603, 0.00141273, 0.00134305,
     &   0.00127677, 0.00121388, 0.00115416, 0.00109735, 0.00104333,
     &   0.00099195, 0.00094395, 0.00089874, 0.00085571, 0.00081482,
     &   0.00077582, 0.00073821, 0.00070241, 0.00066825, 0.00063573,
     &   0.00060454, 0.00057485, 0.00054658, 0.00051968, 0.00049403,
     &   0.00046963, 0.00044642, 0.00042434, 0.00040333, 0.00038334/
      data (ww(i,3),i=1,mxwv)/
     &   0.99999917, 0.99999928, 0.99999934, 0.99999928, 0.99999940,
     &   0.99999946, 0.99999952, 0.99999952, 0.99999958, 0.99999958,
     &   0.99999964, 0.99999958, 0.99999964, 0.99999964, 0.99999970,
     &   0.99999976, 0.99999976, 0.99999976, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999982, 0.99999988, 0.99999988, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999988, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999976, 0.99999976, 0.99999970, 0.99999958,
     &   0.99999946, 0.99999923, 0.99999899, 0.99999881, 0.99999875,
     &   0.99999827, 0.99999857, 0.99999839, 0.99999827, 0.99999803,
     &   0.99999732, 0.99999565, 0.99999279, 0.99998915, 0.99998927,
     &   0.99998927, 0.99999070, 0.99999183, 0.99999154, 0.99998909,
     &   0.99998540, 0.99998146, 0.99997848, 0.99997544, 0.99997264,
     &   0.99996269, 0.99992073, 0.99985343, 0.99981958, 0.99982470,
     &   0.99984527, 0.99987638, 0.99990886, 0.99993390, 0.99994010,
     &   0.99992919, 0.99990588, 0.99982846, 0.99961066, 0.99952310,
     &   0.99950802, 0.99951512, 0.99953949, 0.99957418, 0.99956387,
     &   0.99946231, 0.99925345, 0.99858880, 0.99787223, 0.99531174,
     &   0.98908353, 0.98814297, 0.98958594, 0.99279594, 0.99480408,
     &   0.99601656, 0.99678808, 0.99739659, 0.99772197, 0.99792838,
     &   0.99789298, 0.99768221, 0.99731505, 0.99667311, 0.99657536,
     &   0.99483740, 0.98018080, 0.96016127, 0.96183759, 0.97290260,
     &   0.98260385, 0.98735601, 0.99074084, 0.99257505, 0.99357086,
     &   0.99393755, 0.99353850, 0.99256557, 0.99114746, 0.98863602,
     &   0.98513579, 0.98011905, 0.97502923, 0.97378892, 0.97095239,
     &   0.95134300, 0.87732273, 0.67051172, 0.49031708, 0.44467977,
     &   0.45737207, 0.47020876, 0.47862408, 0.48441643, 0.49194479,
     &   0.51126546, 0.55337983, 0.62692142, 0.72493762, 0.79927546,
     &   0.85966301, 0.90686882, 0.93791169, 0.95533168, 0.96522480,
     &   0.96902776, 0.96950114, 0.96796286, 0.96513134, 0.96058547,
     &   0.95605785, 0.95004308, 0.94324231, 0.93524051, 0.92630577,
     &   0.91691488, 0.90679765, 0.89813751, 0.89775699, 0.90271699,
     &   0.90972978, 0.91633248, 0.92356056, 0.92953640, 0.93311667,
     &   0.93253571, 0.92599750, 0.91360217, 0.88790166, 0.83683336,
     &   0.72771877, 0.58183151, 0.52751464, 0.59064770, 0.70826775,
     &   0.76383793, 0.78741699, 0.80138403, 0.80796903, 0.80866116,
     &   0.80728340, 0.80433774, 0.80012804, 0.79567492, 0.79043794,
     &   0.78469211, 0.77782631, 0.76971328, 0.76152706, 0.75226837,
     &   0.74199599, 0.73064053, 0.71783692, 0.70345432, 0.68732345,
     &   0.66925496, 0.64906478, 0.62647063, 0.60075158, 0.57091689,
     &   0.53691775, 0.49382934, 0.44729593, 0.38603896, 0.32800245,
     &   0.27696961, 0.23993467, 0.21168204, 0.19576751, 0.18940198,
     &   0.18878667, 0.19497463, 0.20273750, 0.21151233, 0.21931687,
     &   0.22597772, 0.23249297, 0.23837221, 0.24334975, 0.24787061,
     &   0.25200266, 0.25517175, 0.25823256, 0.26093772, 0.26345515,
     &   0.26555657, 0.26797572, 0.27034950, 0.27248591, 0.27437881,
     &   0.27658796, 0.27846465, 0.28044030, 0.28195259, 0.28327638,
     &   0.28507885, 0.28692988, 0.28857699, 0.28952676, 0.29010555,
     &   0.29008725, 0.28924155, 0.28841889, 0.28760645, 0.28679666,
     &   0.28599212, 0.28541559, 0.28485230, 0.28437382, 0.28409219,
     &   0.28385824, 0.28253910, 0.28122342, 0.27973992, 0.27816752,
     &   0.27630550, 0.27405846, 0.27147841, 0.26828146, 0.26481298,
     &   0.26064011, 0.25592932, 0.25007156, 0.24365802, 0.23628671,
     &   0.22856160, 0.22056787, 0.21246286, 0.20440146, 0.19644561,
     &   0.18849455, 0.18041290, 0.17235830, 0.16444524, 0.15674609,
     &   0.14910845, 0.14159213, 0.13490824, 0.12874855, 0.12303955,
     &   0.11762637, 0.11274956, 0.10842826, 0.10462550, 0.10113855,
     &   0.09799003, 0.09514111, 0.09256967, 0.09038886, 0.08860181,
     &   0.08682457, 0.08499306, 0.08327505, 0.08174816, 0.08025210,
     &   0.07897643, 0.07765616, 0.07622296, 0.07477175, 0.07339215,
     &   0.07200155, 0.07070382, 0.06938177, 0.06794605, 0.06647844,
     &   0.06492463, 0.06339803, 0.06190287, 0.06042439, 0.05896550,
     &   0.05749057, 0.05600950, 0.05450822, 0.05300087, 0.05152777,
     &   0.05011496, 0.04880668, 0.04753597, 0.04621365, 0.04491090,
     &   0.04364245, 0.04240669, 0.04111309, 0.03981156, 0.03854068,
     &   0.03730373, 0.03604970, 0.03486226, 0.03371971, 0.03256883,
     &   0.03140800, 0.03025947, 0.02906530, 0.02791914, 0.02681920,
     &   0.02576379, 0.02475123, 0.02362765, 0.02252247, 0.02146737,
     &   0.02046029, 0.01949918, 0.01859670, 0.01773696, 0.01691595,
     &   0.01613203, 0.01538294, 0.01466641, 0.01398261, 0.01333013,
     &   0.01270413, 0.01208560, 0.01149662, 0.01093583, 0.01040188,
     &   0.00989335, 0.00940927, 0.00894850, 0.00850803, 0.00808756,
     &   0.00768757, 0.00730528, 0.00693402, 0.00658139, 0.00624646,
     &   0.00593555, 0.00564046, 0.00535982, 0.00509814, 0.00485033,
     &   0.00461439, 0.00438849, 0.00417339, 0.00396871, 0.00377401,
     &   0.00358872, 0.00341282, 0.00324571, 0.00308667, 0.00293539,
     &   0.00279142, 0.00265689, 0.00253014, 0.00240945, 0.00229471,
     &   0.00218526, 0.00207967, 0.00197912, 0.00188315, 0.00179176,
     &   0.00170412, 0.00162064, 0.00154113, 0.00146547, 0.00139330,
     &   0.00132463, 0.00125930, 0.00119713, 0.00113798, 0.00108170/
      data (ww(i,4),i=1,mxwv)/
     &   0.99999887, 0.99999899, 0.99999905, 0.99999911, 0.99999923,
     &   0.99999911, 0.99999928, 0.99999934, 0.99999940, 0.99999946,
     &   0.99999946, 0.99999952, 0.99999952, 0.99999958, 0.99999958,
     &   0.99999964, 0.99999958, 0.99999970, 0.99999970, 0.99999976,
     &   0.99999976, 0.99999976, 0.99999976, 0.99999982, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999982, 0.99999976, 0.99999976,
     &   0.99999976, 0.99999970, 0.99999964, 0.99999958, 0.99999946,
     &   0.99999928, 0.99999893, 0.99999857, 0.99999833, 0.99999821,
     &   0.99999815, 0.99999803, 0.99999756, 0.99999762, 0.99999678,
     &   0.99999624, 0.99999398, 0.99999011, 0.99998540, 0.99998403,
     &   0.99998415, 0.99998647, 0.99998820, 0.99998796, 0.99998474,
     &   0.99997979, 0.99997467, 0.99996984, 0.99996549, 0.99996096,
     &   0.99994606, 0.99988651, 0.99979222, 0.99973822, 0.99974656,
     &   0.99977833, 0.99982131, 0.99986899, 0.99990559, 0.99991339,
     &   0.99989599, 0.99986440, 0.99975067, 0.99943900, 0.99931085,
     &   0.99929076, 0.99929857, 0.99933290, 0.99938214, 0.99936688,
     &   0.99922097, 0.99891806, 0.99795890, 0.99692863, 0.99327600,
     &   0.98449999, 0.98315758, 0.98514807, 0.98965138, 0.99249220,
     &   0.99421740, 0.99531949, 0.99619287, 0.99665838, 0.99695224,
     &   0.99689221, 0.99657279, 0.99601942, 0.99505502, 0.99488884,
     &   0.99227744, 0.97066647, 0.94169754, 0.94376308, 0.95944023,
     &   0.97354060, 0.98052859, 0.98557317, 0.98830819, 0.98977381,
     &   0.99026293, 0.98952681, 0.98784614, 0.98541331, 0.98115838,
     &   0.97525418, 0.96688467, 0.95853454, 0.95671052, 0.95287877,
     &   0.92544538, 0.83741432, 0.65511763, 0.50525880, 0.46748200,
     &   0.47553518, 0.48387331, 0.48764753, 0.48938096, 0.49024406,
     &   0.49463868, 0.51274043, 0.55958045, 0.64279693, 0.71924150,
     &   0.79127181, 0.85442847, 0.89963853, 0.92649579, 0.94236356,
     &   0.94876999, 0.94995117, 0.94805747, 0.94431269, 0.93817466,
     &   0.93237096, 0.92472178, 0.91635615, 0.90675557, 0.89637601,
     &   0.88583744, 0.87477809, 0.86573219, 0.86687821, 0.87461221,
     &   0.88493383, 0.89477652, 0.90533727, 0.91432559, 0.92047620,
     &   0.92187285, 0.91684639, 0.90615088, 0.88299054, 0.83725315,
     &   0.73965657, 0.60575593, 0.54493225, 0.58336729, 0.69001663,
     &   0.75009263, 0.77825075, 0.79611355, 0.80611801, 0.80990529,
     &   0.81139052, 0.81122148, 0.80970329, 0.80790627, 0.80534387,
     &   0.80232823, 0.79829264, 0.79313236, 0.78792620, 0.78180289,
     &   0.77480936, 0.76690537, 0.75775534, 0.74724162, 0.73517931,
     &   0.72140920, 0.70572525, 0.68780404, 0.66697538, 0.64214760,
     &   0.61307013, 0.57478946, 0.53163868, 0.47177818, 0.41189998,
     &   0.35638931, 0.31445366, 0.28130773, 0.26201409, 0.25386393,
     &   0.25262198, 0.25915343, 0.26748633, 0.27683929, 0.28506193,
     &   0.29198697, 0.29868269, 0.30466282, 0.30970261, 0.31425050,
     &   0.31840214, 0.32161641, 0.32471052, 0.32748109, 0.33008018,
     &   0.33227944, 0.33479723, 0.33728719, 0.33959681, 0.34168798,
     &   0.34411466, 0.34630778, 0.34860191, 0.35055777, 0.35233122,
     &   0.35459948, 0.35707170, 0.35939974, 0.36112005, 0.36259350,
     &   0.36365095, 0.36374980, 0.36389840, 0.36407065, 0.36426190,
     &   0.36448589, 0.36495835, 0.36547357, 0.36616361, 0.36719865,
     &   0.36831990, 0.36836678, 0.36844605, 0.36841890, 0.36834788,
     &   0.36802483, 0.36735117, 0.36634630, 0.36470050, 0.36277446,
     &   0.36010337, 0.35680372, 0.35215285, 0.34676129, 0.34010065,
     &   0.33284795, 0.32508805, 0.31697530, 0.30872646, 0.30042335,
     &   0.29193345, 0.28305614, 0.27396300, 0.26482034, 0.25575772,
     &   0.24650718, 0.23714961, 0.22872876, 0.22088939, 0.21355106,
     &   0.20646422, 0.20002249, 0.19430555, 0.18930928, 0.18471621,
     &   0.18057714, 0.17685220, 0.17352052, 0.17076226, 0.16865648,
     &   0.16655932, 0.16434015, 0.16226850, 0.16048065, 0.15872598,
     &   0.15737888, 0.15592508, 0.15425314, 0.15251103, 0.15086417,
     &   0.14916088, 0.14762111, 0.14599922, 0.14414302, 0.14217509,
     &   0.13998438, 0.13779536, 0.13561723, 0.13342257, 0.13121460,
     &   0.12892659, 0.12656879, 0.12411072, 0.12158048, 0.11906129,
     &   0.11660824, 0.11433789, 0.11210244, 0.10970278, 0.10729378,
     &   0.10491285, 0.10256008, 0.10002839, 0.09742652, 0.09484830,
     &   0.09230429, 0.08967441, 0.08715999, 0.08471477, 0.08221217,
     &   0.07964855, 0.07707956, 0.07436566, 0.07173564, 0.06918841,
     &   0.06672274, 0.06433730, 0.06164570, 0.05897395, 0.05640470,
     &   0.05393530, 0.05156305, 0.04932224, 0.04717493, 0.04511272,
     &   0.04313302, 0.04123148, 0.03940355, 0.03765094, 0.03597111,
     &   0.03435239, 0.03274507, 0.03120859, 0.02974024, 0.02833720,
     &   0.02699645, 0.02571607, 0.02449361, 0.02332157, 0.02219959,
     &   0.02112947, 0.02010402, 0.01910548, 0.01815485, 0.01724998,
     &   0.01640845, 0.01560817, 0.01484566, 0.01413354, 0.01345804,
     &   0.01281384, 0.01219609, 0.01160701, 0.01104566, 0.01051094,
     &   0.01000144, 0.00951716, 0.00905655, 0.00861770, 0.00819981,
     &   0.00780170, 0.00742938, 0.00707828, 0.00674368, 0.00642529,
     &   0.00612135, 0.00582790, 0.00554824, 0.00528112, 0.00502660,
     &   0.00478233, 0.00454951, 0.00432765, 0.00411640, 0.00391478,
     &   0.00372285, 0.00354015, 0.00336623, 0.00320065, 0.00304307/
      data (ww(i,5),i=1,mxwv)/
     &   0.99999827, 0.99999851, 0.99999869, 0.99999881, 0.99999881,
     &   0.99999905, 0.99999899, 0.99999911, 0.99999917, 0.99999923,
     &   0.99999893, 0.99999923, 0.99999934, 0.99999940, 0.99999946,
     &   0.99999952, 0.99999946, 0.99999958, 0.99999958, 0.99999964,
     &   0.99999970, 0.99999970, 0.99999970, 0.99999976, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999976, 0.99999970, 0.99999970,
     &   0.99999964, 0.99999958, 0.99999952, 0.99999940, 0.99999928,
     &   0.99999905, 0.99999845, 0.99999791, 0.99999768, 0.99999768,
     &   0.99999762, 0.99999738, 0.99999690, 0.99999642, 0.99999624,
     &   0.99999493, 0.99999201, 0.99998295, 0.99997985, 0.99997914,
     &   0.99997908, 0.99998230, 0.99998391, 0.99998373, 0.99997807,
     &   0.99997193, 0.99996245, 0.99995834, 0.99995255, 0.99994677,
     &   0.99992687, 0.99984056, 0.99970990, 0.99963808, 0.99965358,
     &   0.99968612, 0.99975443, 0.99981660, 0.99986738, 0.99987733,
     &   0.99985737, 0.99979556, 0.99965656, 0.99920702, 0.99904925,
     &   0.99900645, 0.99902117, 0.99906808, 0.99913728, 0.99910432,
     &   0.99889934, 0.99847943, 0.99714142, 0.99570125, 0.99066454,
     &   0.97868037, 0.97683543, 0.97951096, 0.98563069, 0.98951364,
     &   0.99188161, 0.99339950, 0.99460733, 0.99525052, 0.99565774,
     &   0.99556446, 0.99510586, 0.99431616, 0.99294412, 0.99269658,
     &   0.98900241, 0.95917624, 0.92044318, 0.92303604, 0.94375324,
     &   0.96278375, 0.97235131, 0.97934133, 0.98315072, 0.98518610,
     &   0.98583293, 0.98470944, 0.98219794, 0.97855645, 0.97220153,
     &   0.96333444, 0.95069396, 0.93778253, 0.93398577, 0.92673880,
     &   0.88355070, 0.76343107, 0.60543478, 0.50374311, 0.48122457,
     &   0.48914397, 0.49653402, 0.49924690, 0.50052184, 0.50034410,
     &   0.50020581, 0.50602216, 0.53118163, 0.59188259, 0.65918767,
     &   0.73197234, 0.80295813, 0.85798895, 0.89238095, 0.91332805,
     &   0.92149520, 0.92226446, 0.91853690, 0.91206974, 0.90206772,
     &   0.89267725, 0.88071656, 0.86798245, 0.85384184, 0.83908880,
     &   0.82468057, 0.81015295, 0.79879010, 0.80100536, 0.81207871,
     &   0.82696491, 0.84160978, 0.85761040, 0.87177008, 0.88230020,
     &   0.88655806, 0.88261306, 0.87232518, 0.84883171, 0.80405754,
     &   0.71336055, 0.59438878, 0.53352088, 0.54631108, 0.62969011,
     &   0.69243264, 0.72668505, 0.75047392, 0.76572019, 0.77394283,
     &   0.77938986, 0.78290725, 0.78480917, 0.78634250, 0.78703117,
     &   0.78727025, 0.78648353, 0.78458613, 0.78265750, 0.77990109,
     &   0.77636588, 0.77205396, 0.76665354, 0.76007897, 0.75214350,
     &   0.74276060, 0.73174030, 0.71876371, 0.70330232, 0.68425786,
     &   0.66133195, 0.62993312, 0.59300065, 0.53919178, 0.48265457,
     &   0.42755735, 0.38433865, 0.34895119, 0.32762694, 0.31809533,
     &   0.31612349, 0.32215178, 0.33005863, 0.33886451, 0.34648415,
     &   0.35280025, 0.35880163, 0.36407465, 0.36845443, 0.37235758,
     &   0.37587947, 0.37859002, 0.38117957, 0.38349196, 0.38565677,
     &   0.38748786, 0.38957593, 0.39164153, 0.39358085, 0.39534932,
     &   0.39740473, 0.39932731, 0.40133637, 0.40315419, 0.40482590,
     &   0.40692589, 0.40932977, 0.41167146, 0.41358945, 0.41542530,
     &   0.41710067, 0.41784167, 0.41865551, 0.41950476, 0.42038724,
     &   0.42132595, 0.42249408, 0.42372724, 0.42520896, 0.42715588,
     &   0.42921808, 0.43038887, 0.43161848, 0.43283346, 0.43406591,
     &   0.43513489, 0.43595809, 0.43651140, 0.43651205, 0.43628186,
     &   0.43541083, 0.43393400, 0.43113503, 0.42754358, 0.42258534,
     &   0.41690972, 0.41059387, 0.40375650, 0.39665961, 0.38939673,
     &   0.38182691, 0.37371078, 0.36518854, 0.35645732, 0.34769824,
     &   0.33852866, 0.32902873, 0.32039779, 0.31233096, 0.30475488,
     &   0.29733425, 0.29055685, 0.28456804, 0.27942029, 0.27471524,
     &   0.27052361, 0.26681754, 0.26358268, 0.26102126, 0.25932878,
     &   0.25768870, 0.25590375, 0.25428963, 0.25301823, 0.25180683,
     &   0.25120455, 0.25046924, 0.24947682, 0.24838723, 0.24742316,
     &   0.24638113, 0.24560864, 0.24472474, 0.24355973, 0.24222068,
     &   0.24054006, 0.23884106, 0.23713596, 0.23538755, 0.23359150,
     &   0.23166233, 0.22958685, 0.22731146, 0.22487009, 0.22237784,
     &   0.21990776, 0.21770556, 0.21551722, 0.21299908, 0.21039037,
     &   0.20776133, 0.20511732, 0.20208521, 0.19882190, 0.19550437,
     &   0.19215217, 0.18852834, 0.18501490, 0.18153778, 0.17784585,
     &   0.17392287, 0.16987768, 0.16543354, 0.16103637, 0.15669055,
     &   0.15240027, 0.14816944, 0.14316705, 0.13808939, 0.13312280,
     &   0.12827019, 0.12353409, 0.11899475, 0.11458165, 0.11028384,
     &   0.10610224, 0.10203318, 0.09807210, 0.09422879, 0.09050281,
     &   0.08687121, 0.08321727, 0.07968922, 0.07628519, 0.07300249,
     &   0.06983741, 0.06678957, 0.06385642, 0.06102223, 0.05828896,
     &   0.05566394, 0.05313128, 0.05064725, 0.04826819, 0.04599070,
     &   0.04386223, 0.04182776, 0.03987980, 0.03805298, 0.03631263,
     &   0.03464588, 0.03304086, 0.03150441, 0.03003492, 0.02863020,
     &   0.02728725, 0.02600678, 0.02478521, 0.02361796, 0.02250339,
     &   0.02143877, 0.02044085, 0.01949774, 0.01859690, 0.01773790,
     &   0.01691617, 0.01612116, 0.01536204, 0.01463568, 0.01394237,
     &   0.01327586, 0.01263957, 0.01203232, 0.01145329, 0.01089989,
     &   0.01037237, 0.00986963, 0.00939046, 0.00893374, 0.00849863/
      data (ww(i,6),i=1,mxwv)/
     &   0.99999779, 0.99999803, 0.99999815, 0.99999833, 0.99999851,
     &   0.99999863, 0.99999869, 0.99999881, 0.99999887, 0.99999887,
     &   0.99999899, 0.99999905, 0.99999911, 0.99999917, 0.99999928,
     &   0.99999928, 0.99999934, 0.99999940, 0.99999940, 0.99999952,
     &   0.99999958, 0.99999958, 0.99999964, 0.99999970, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999964, 0.99999958, 0.99999958,
     &   0.99999946, 0.99999940, 0.99999928, 0.99999917, 0.99999899,
     &   0.99999851, 0.99999797, 0.99999714, 0.99999696, 0.99999672,
     &   0.99999666, 0.99999636, 0.99999577, 0.99999517, 0.99999475,
     &   0.99999285, 0.99998879, 0.99998069, 0.99997270, 0.99997175,
     &   0.99997175, 0.99997509, 0.99997723, 0.99997783, 0.99997085,
     &   0.99996144, 0.99995142, 0.99994242, 0.99993420, 0.99992651,
     &   0.99989372, 0.99978262, 0.99960804, 0.99951243, 0.99952787,
     &   0.99957955, 0.99966121, 0.99974710, 0.99981964, 0.99982995,
     &   0.99980646, 0.99973977, 0.99951392, 0.99893206, 0.99868143,
     &   0.99864382, 0.99865776, 0.99871540, 0.99881184, 0.99877453,
     &   0.99848175, 0.99791205, 0.99608070, 0.99410695, 0.98723787,
     &   0.97100145, 0.96852177, 0.97212708, 0.98039186, 0.98565477,
     &   0.98888034, 0.99093372, 0.99257398, 0.99343705, 0.99400586,
     &   0.99386013, 0.99322599, 0.99212170, 0.99023223, 0.98987991,
     &   0.98480844, 0.94469929, 0.89431983, 0.89759159, 0.92429322,
     &   0.94930178, 0.96203774, 0.97142589, 0.97656375, 0.97930455,
     &   0.98014748, 0.97854900, 0.97503030, 0.96994895, 0.96115196,
     &   0.94895494, 0.93172091, 0.91418493, 0.90859026, 0.89802313,
     &   0.83923179, 0.68789148, 0.54769576, 0.49790719, 0.49115402,
     &   0.50062710, 0.50812578, 0.51078123, 0.51220894, 0.51208192,
     &   0.51082146, 0.51121312, 0.52162242, 0.55934936, 0.61264759,
     &   0.68033111, 0.75452387, 0.81690586, 0.85782444, 0.88340223,
     &   0.89320207, 0.89352912, 0.88795704, 0.87872696, 0.86482048,
     &   0.85161948, 0.83504152, 0.81745207, 0.79813778, 0.77819657,
     &   0.75894421, 0.73990202, 0.72509640, 0.72633702, 0.73800731,
     &   0.75453442, 0.77127439, 0.79036361, 0.80777729, 0.82094085,
     &   0.82601845, 0.82028323, 0.80680519, 0.77821529, 0.72972155,
     &   0.64642614, 0.55490118, 0.50947350, 0.50876105, 0.55671793,
     &   0.60750794, 0.64055783, 0.66612941, 0.68434411, 0.69578612,
     &   0.70453066, 0.71143627, 0.71671116, 0.72173482, 0.72598380,
     &   0.72992301, 0.73290980, 0.73484367, 0.73683637, 0.73812288,
     &   0.73873734, 0.73872375, 0.73776776, 0.73581129, 0.73265004,
     &   0.72829282, 0.72257310, 0.71518564, 0.70580566, 0.69335550,
     &   0.67759717, 0.65455604, 0.62577599, 0.58137441, 0.53237879,
     &   0.48222941, 0.44152692, 0.40703008, 0.38548797, 0.37531108,
     &   0.37271580, 0.37758833, 0.38429201, 0.39171141, 0.39801484,
     &   0.40317157, 0.40795425, 0.41206545, 0.41538501, 0.41829151,
     &   0.42083937, 0.42274860, 0.42455667, 0.42612034, 0.42755932,
     &   0.42876539, 0.43012545, 0.43145454, 0.43267551, 0.43378356,
     &   0.43507403, 0.43627325, 0.43754238, 0.43870670, 0.43979588,
     &   0.44118357, 0.44283551, 0.44449821, 0.44594786, 0.44744259,
     &   0.44898903, 0.44985032, 0.45080546, 0.45181695, 0.45288339,
     &   0.45402959, 0.45536569, 0.45678768, 0.45849144, 0.46070597,
     &   0.46306923, 0.46486565, 0.46676037, 0.46875286, 0.47084397,
     &   0.47290668, 0.47488853, 0.47672343, 0.47819361, 0.47953582,
     &   0.48045415, 0.48088700, 0.48021448, 0.47880489, 0.47610408,
     &   0.47264701, 0.46849397, 0.46370420, 0.45856440, 0.45316905,
     &   0.44739127, 0.44098380, 0.43402719, 0.42673072, 0.41931728,
     &   0.41131791, 0.40280104, 0.39493766, 0.38754988, 0.38058457,
     &   0.37365144, 0.36725846, 0.36160558, 0.35680848, 0.35244104,
     &   0.34858087, 0.34522140, 0.34235722, 0.34015980, 0.33893844,
     &   0.33784586, 0.33664334, 0.33562192, 0.33495161, 0.33438855,
     &   0.33457017, 0.33464342, 0.33451194, 0.33430308, 0.33425161,
     &   0.33414680, 0.33441204, 0.33459413, 0.33457085, 0.33438531,
     &   0.33385202, 0.33332759, 0.33282652, 0.33231512, 0.33178043,
     &   0.33115318, 0.33038655, 0.32941738, 0.32827669, 0.32707641,
     &   0.32587686, 0.32505047, 0.32428128, 0.32314694, 0.32190636,
     &   0.32065806, 0.31942075, 0.31771848, 0.31568480, 0.31356093,
     &   0.31136954, 0.30876270, 0.30625364, 0.30376503, 0.30092660,
     &   0.29768655, 0.29418859, 0.29003516, 0.28581765, 0.28153861,
     &   0.27720094, 0.27280796, 0.26699835, 0.26086479, 0.25470918,
     &   0.24853899, 0.24236177, 0.23629387, 0.23024882, 0.22421750,
     &   0.21820731, 0.21221797, 0.20624751, 0.20032099, 0.19444548,
     &   0.18858343, 0.18250582, 0.17651449, 0.17061552, 0.16481219,
     &   0.15910415, 0.15350497, 0.14801894, 0.14262158, 0.13732573,
     &   0.13215627, 0.12708600, 0.12202205, 0.11710132, 0.11232451,
     &   0.10780685, 0.10343372, 0.09919482, 0.09517878, 0.09131093,
     &   0.08756629, 0.08392116, 0.08039680, 0.07699367, 0.07371057,
     &   0.07054438, 0.06750056, 0.06457378, 0.06175584, 0.05904507,
     &   0.05643783, 0.05397955, 0.05164262, 0.04939717, 0.04724408,
     &   0.04517306, 0.04315819, 0.04122469, 0.03936569, 0.03758327,
     &   0.03586195, 0.03421184, 0.03263084, 0.03111764, 0.02966609,
     &   0.02827772, 0.02695023, 0.02568108, 0.02446782, 0.02330868/
      data (ww(i,7),i=1,mxwv)/
     &   0.99999696, 0.99999726, 0.99999750, 0.99999762, 0.99999797,
     &   0.99999809, 0.99999815, 0.99999827, 0.99999839, 0.99999851,
     &   0.99999857, 0.99999869, 0.99999851, 0.99999887, 0.99999893,
     &   0.99999905, 0.99999911, 0.99999917, 0.99999923, 0.99999934,
     &   0.99999940, 0.99999946, 0.99999952, 0.99999958, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999946, 0.99999946, 0.99999940,
     &   0.99999928, 0.99999917, 0.99999905, 0.99999887, 0.99999857,
     &   0.99999809, 0.99999708, 0.99999601, 0.99999577, 0.99999547,
     &   0.99999529, 0.99999505, 0.99999428, 0.99999380, 0.99999285,
     &   0.99999022, 0.99998415, 0.99997383, 0.99996281, 0.99996006,
     &   0.99995869, 0.99996525, 0.99996990, 0.99996901, 0.99996042,
     &   0.99994701, 0.99993157, 0.99992120, 0.99991137, 0.99989760,
     &   0.99985856, 0.99970168, 0.99946046, 0.99932796, 0.99934280,
     &   0.99941963, 0.99953932, 0.99965447, 0.99974871, 0.99977320,
     &   0.99972832, 0.99964190, 0.99935168, 0.99853688, 0.99819899,
     &   0.99813479, 0.99815875, 0.99823213, 0.99836111, 0.99831593,
     &   0.99793899, 0.99714297, 0.99460208, 0.99195045, 0.98254484,
     &   0.96055329, 0.95722944, 0.96207827, 0.97324985, 0.98040140,
     &   0.98476905, 0.98758531, 0.98984998, 0.99102640, 0.99176180,
     &   0.99160558, 0.99072623, 0.98922503, 0.98664337, 0.98614794,
     &   0.97925824, 0.92574805, 0.86137813, 0.86548412, 0.89930022,
     &   0.93179733, 0.94863033, 0.96115303, 0.96803856, 0.97171205,
     &   0.97282249, 0.97062778, 0.96583647, 0.95894688, 0.94711155,
     &   0.93085378, 0.90819287, 0.88545144, 0.87800407, 0.86414713,
     &   0.79163742, 0.62828273, 0.51430357, 0.49666989, 0.49972558,
     &   0.51026201, 0.51780307, 0.52047658, 0.52198386, 0.52193785,
     &   0.52046502, 0.51892745, 0.52125508, 0.53972298, 0.57556754,
     &   0.63175446, 0.70367491, 0.77106827, 0.81826645, 0.84883434,
     &   0.86063880, 0.86081958, 0.85372406, 0.84218729, 0.82511300,
     &   0.80907834, 0.78929752, 0.76866978, 0.74647868, 0.72404939,
     &   0.70285225, 0.68236494, 0.66668802, 0.66723871, 0.67824376,
     &   0.69439167, 0.71106184, 0.73064148, 0.74874330, 0.76213342,
     &   0.76590765, 0.75642806, 0.73726624, 0.70047885, 0.64488518,
     &   0.57066619, 0.51562226, 0.49563545, 0.49591392, 0.51720703,
     &   0.54778099, 0.57052344, 0.58946574, 0.60341918, 0.61214048,
     &   0.61896259, 0.62454045, 0.62902325, 0.63367558, 0.63800019,
     &   0.64245516, 0.64642900, 0.64982152, 0.65369338, 0.65734094,
     &   0.66078472, 0.66408938, 0.66694117, 0.66928709, 0.67088288,
     &   0.67179817, 0.67186391, 0.67076021, 0.66836745, 0.66362113,
     &   0.65635419, 0.64339846, 0.62475526, 0.59286040, 0.55517751,
     &   0.51424795, 0.47986630, 0.44967666, 0.43016621, 0.42056099,
     &   0.41787335, 0.42152852, 0.42683262, 0.43264690, 0.43748897,
     &   0.44144249, 0.44499472, 0.44797298, 0.45026886, 0.45224181,
     &   0.45387471, 0.45502672, 0.45611566, 0.45696723, 0.45771068,
     &   0.45832485, 0.45898971, 0.45960811, 0.46009460, 0.46051937,
     &   0.46100765, 0.46136740, 0.46178961, 0.46209058, 0.46239656,
     &   0.46285695, 0.46339101, 0.46396005, 0.46445397, 0.46500123,
     &   0.46563107, 0.46599215, 0.46644634, 0.46697384, 0.46756977,
     &   0.46825263, 0.46907449, 0.46998698, 0.47113091, 0.47268438,
     &   0.47440302, 0.47591543, 0.47756547, 0.47939780, 0.48140597,
     &   0.48352951, 0.48575589, 0.48800933, 0.49016392, 0.49235114,
     &   0.49442068, 0.49624112, 0.49736533, 0.49795496, 0.49754000,
     &   0.49646416, 0.49475563, 0.49240166, 0.48966914, 0.48664290,
     &   0.48322368, 0.47918665, 0.47453797, 0.46947220, 0.46422803,
     &   0.45830297, 0.45174348, 0.44552276, 0.43962339, 0.43401754,
     &   0.42830691, 0.42294353, 0.41815439, 0.41410413, 0.41039804,
     &   0.40710753, 0.40424913, 0.40182683, 0.39994857, 0.39899695,
     &   0.39822730, 0.39739525, 0.39671689, 0.39633107, 0.39607856,
     &   0.39658380, 0.39701837, 0.39733472, 0.39760372, 0.39803261,
     &   0.39844099, 0.39926174, 0.40004364, 0.40074477, 0.40132892,
     &   0.40162846, 0.40197185, 0.40237418, 0.40281451, 0.40327391,
     &   0.40372035, 0.40407923, 0.40429342, 0.40438780, 0.40444684,
     &   0.40450126, 0.40499541, 0.40561378, 0.40594786, 0.40621892,
     &   0.40654138, 0.40695241, 0.40698671, 0.40671328, 0.40639758,
     &   0.40606648, 0.40533641, 0.40474132, 0.40423581, 0.40341619,
     &   0.40220651, 0.40075895, 0.39867204, 0.39651304, 0.39428037,
     &   0.39197248, 0.38958779, 0.38550645, 0.38096195, 0.37631887,
     &   0.37157178, 0.36672136, 0.36184514, 0.35688305, 0.35182512,
     &   0.34667295, 0.34141862, 0.33605176, 0.33059773, 0.32505965,
     &   0.31937593, 0.31319681, 0.30695316, 0.30065057, 0.29428700,
     &   0.28785008, 0.28137523, 0.27486938, 0.26829374, 0.26166910,
     &   0.25503841, 0.24835582, 0.24145187, 0.23457880, 0.22774509,
     &   0.22115138, 0.21462186, 0.20814767, 0.20190477, 0.19576646,
     &   0.18969476, 0.18365113, 0.17768589, 0.17180875, 0.16602533,
     &   0.16034096, 0.15477599, 0.14932926, 0.14399321, 0.13876948,
     &   0.13366316, 0.12878339, 0.12407988, 0.11949507, 0.11503880,
     &   0.11069335, 0.10640509, 0.10223781, 0.09818173, 0.09424719,
     &   0.09040257, 0.08667712, 0.08307081, 0.07958516, 0.07620896,
     &   0.07295066, 0.06980845, 0.06677964, 0.06386153, 0.06105271/
      data (ww(i,8),i=1,mxwv)/
     &   0.99999571, 0.99999619, 0.99999648, 0.99999678, 0.99999714,
     &   0.99999738, 0.99999750, 0.99999762, 0.99999774, 0.99999791,
     &   0.99999803, 0.99999821, 0.99999833, 0.99999839, 0.99999827,
     &   0.99999863, 0.99999881, 0.99999887, 0.99999899, 0.99999905,
     &   0.99999911, 0.99999923, 0.99999934, 0.99999940, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999928, 0.99999923, 0.99999911,
     &   0.99999899, 0.99999887, 0.99999869, 0.99999845, 0.99999815,
     &   0.99999738, 0.99999613, 0.99999458, 0.99999380, 0.99999374,
     &   0.99999344, 0.99999273, 0.99999201, 0.99999106, 0.99998963,
     &   0.99998653, 0.99997860, 0.99996340, 0.99994862, 0.99994433,
     &   0.99994594, 0.99995279, 0.99995714, 0.99995506, 0.99994540,
     &   0.99992621, 0.99990547, 0.99988961, 0.99987245, 0.99985975,
     &   0.99980903, 0.99959201, 0.99924839, 0.99907559, 0.99910349,
     &   0.99920541, 0.99935955, 0.99952328, 0.99965537, 0.99968237,
     &   0.99963135, 0.99950504, 0.99909872, 0.99796885, 0.99752194,
     &   0.99742806, 0.99746478, 0.99758458, 0.99774879, 0.99769086,
     &   0.99715376, 0.99606317, 0.99260443, 0.98891127, 0.97606587,
     &   0.94637978, 0.94197839, 0.94842881, 0.96346414, 0.97316384,
     &   0.97913885, 0.98297501, 0.98605442, 0.98769617, 0.98873019,
     &   0.98846889, 0.98728544, 0.98521662, 0.98170143, 0.98104608,
     &   0.97165918, 0.90081954, 0.82066298, 0.82560819, 0.86722666,
     &   0.90869105, 0.93071711, 0.94733310, 0.95655078, 0.96148878,
     &   0.96298546, 0.96002597, 0.95359266, 0.94438517, 0.92870623,
     &   0.90745121, 0.87838811, 0.84988409, 0.84060633, 0.82353926,
     &   0.73942572, 0.58155888, 0.50410366, 0.50006592, 0.50712168,
     &   0.51790988, 0.52533835, 0.52800852, 0.52954912, 0.52956444,
     &   0.52812338, 0.52614981, 0.52527016, 0.53153592, 0.55081689,
     &   0.59057903, 0.65284228, 0.72026330, 0.77196527, 0.80724251,
     &   0.82120597, 0.82138956, 0.81291533, 0.79930365, 0.77951872,
     &   0.76131135, 0.73937720, 0.71712875, 0.69393373, 0.67128944,
     &   0.65065366, 0.63144523, 0.61724895, 0.61762714, 0.62734431,
     &   0.64203137, 0.65763968, 0.67656338, 0.69452238, 0.70791548,
     &   0.71118766, 0.70025867, 0.67912430, 0.64063907, 0.58693731,
     &   0.52763253, 0.49975356, 0.49687436, 0.50038028, 0.50803709,
     &   0.52313036, 0.53619015, 0.54788685, 0.55658621, 0.56159216,
     &   0.56522149, 0.56787872, 0.56969255, 0.57163125, 0.57330906,
     &   0.57509106, 0.57652462, 0.57757354, 0.57916224, 0.58074737,
     &   0.58242452, 0.58431560, 0.58625752, 0.58829898, 0.59032673,
     &   0.59249544, 0.59475237, 0.59689248, 0.59898192, 0.60017991,
     &   0.60037875, 0.59744680, 0.58990473, 0.57293612, 0.54981077,
     &   0.52197582, 0.49737710, 0.47477964, 0.45970696, 0.45224896,
     &   0.45038220, 0.45341295, 0.45773000, 0.46232629, 0.46607372,
     &   0.46918657, 0.47188887, 0.47411263, 0.47573653, 0.47712332,
     &   0.47818312, 0.47886923, 0.47953627, 0.47995767, 0.48028380,
     &   0.48055974, 0.48081815, 0.48102009, 0.48105374, 0.48106185,
     &   0.48104978, 0.48084226, 0.48070434, 0.48037475, 0.48011309,
     &   0.47990721, 0.47955656, 0.47922388, 0.47885057, 0.47844091,
     &   0.47797751, 0.47761700, 0.47731656, 0.47707853, 0.47689968,
     &   0.47677901, 0.47674122, 0.47676498, 0.47688511, 0.47717208,
     &   0.47758165, 0.47798744, 0.47851592, 0.47919846, 0.48004359,
     &   0.48104617, 0.48222280, 0.48354158, 0.48496580, 0.48655117,
     &   0.48827028, 0.49001774, 0.49156687, 0.49292228, 0.49377176,
     &   0.49424085, 0.49431369, 0.49391729, 0.49323365, 0.49231362,
     &   0.49109134, 0.48939493, 0.48716658, 0.48454672, 0.48173919,
     &   0.47830203, 0.47425243, 0.47023952, 0.46637306, 0.46264949,
     &   0.45873535, 0.45496231, 0.45153886, 0.44863561, 0.44595274,
     &   0.44354466, 0.44144091, 0.43965071, 0.43822026, 0.43750161,
     &   0.43695912, 0.43639296, 0.43592978, 0.43566197, 0.43551177,
     &   0.43596619, 0.43638298, 0.43675491, 0.43710467, 0.43758050,
     &   0.43805769, 0.43889073, 0.43972120, 0.44057271, 0.44136351,
     &   0.44196999, 0.44263929, 0.44338575, 0.44420338, 0.44507301,
     &   0.44600129, 0.44690421, 0.44774032, 0.44852549, 0.44930539,
     &   0.45007688, 0.45128375, 0.45266217, 0.45388472, 0.45509908,
     &   0.45641956, 0.45789796, 0.45915303, 0.46017474, 0.46121436,
     &   0.46229985, 0.46306923, 0.46399617, 0.46507964, 0.46594489,
     &   0.46650687, 0.46690693, 0.46681890, 0.46668798, 0.46651277,
     &   0.46629190, 0.46602389, 0.46411601, 0.46177402, 0.45933539,
     &   0.45679948, 0.45416573, 0.45144036, 0.44862196, 0.44570914,
     &   0.44270119, 0.43958747, 0.43635434, 0.43302307, 0.42959279,
     &   0.42599064, 0.42181838, 0.41755345, 0.41316116, 0.40866005,
     &   0.40400085, 0.39923838, 0.39437270, 0.38935030, 0.38418511,
     &   0.37891829, 0.37347856, 0.36762190, 0.36167359, 0.35563606,
     &   0.34972176, 0.34374201, 0.33768234, 0.33177167, 0.32584321,
     &   0.31984174, 0.31370595, 0.30750200, 0.30123872, 0.29491660,
     &   0.28854883, 0.28216323, 0.27575970, 0.26932919, 0.26286006,
     &   0.25637922, 0.25007147, 0.24386452, 0.23767109, 0.23151650,
     &   0.22536871, 0.21913663, 0.21293789, 0.20676257, 0.20063575,
     &   0.19450325, 0.18842950, 0.18242370, 0.17649744, 0.17063391,
     &   0.16486311, 0.15919085, 0.15362130, 0.14815830, 0.14280744/
      data (ww(i,9),i=1,mxwv)/
     &   0.99999404, 0.99999464, 0.99999475, 0.99999559, 0.99999607,
     &   0.99999624, 0.99999654, 0.99999666, 0.99999684, 0.99999708,
     &   0.99999726, 0.99999744, 0.99999768, 0.99999785, 0.99999774,
     &   0.99999809, 0.99999827, 0.99999845, 0.99999857, 0.99999869,
     &   0.99999881, 0.99999893, 0.99999905, 0.99999917, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999911, 0.99999887, 0.99999875,
     &   0.99999863, 0.99999845, 0.99999815, 0.99999785, 0.99999744,
     &   0.99999636, 0.99999446, 0.99999261, 0.99999166, 0.99999130,
     &   0.99999106, 0.99999034, 0.99998897, 0.99998784, 0.99998546,
     &   0.99998188, 0.99997008, 0.99994898, 0.99992692, 0.99992388,
     &   0.99992508, 0.99993426, 0.99994141, 0.99993998, 0.99992335,
     &   0.99989915, 0.99987030, 0.99984717, 0.99982709, 0.99980617,
     &   0.99973416, 0.99942654, 0.99896801, 0.99871832, 0.99875355,
     &   0.99889970, 0.99911547, 0.99934524, 0.99951988, 0.99956429,
     &   0.99948490, 0.99931473, 0.99875510, 0.99719375, 0.99656361,
     &   0.99644774, 0.99648577, 0.99665350, 0.99689418, 0.99681121,
     &   0.99606395, 0.99454731, 0.98977774, 0.98470414, 0.96713376,
     &   0.92733806, 0.92149788, 0.93006569, 0.95011801, 0.96320558,
     &   0.97134346, 0.97659338, 0.98080754, 0.98305422, 0.98446804,
     &   0.98412359, 0.98248798, 0.97970134, 0.97487390, 0.97397560,
     &   0.96123880, 0.86875015, 0.77265191, 0.77831656, 0.82735258,
     &   0.87873805, 0.90701669, 0.92878002, 0.94100970, 0.94760847,
     &   0.94960958, 0.94565701, 0.93708622, 0.92492121, 0.90442699,
     &   0.87717646, 0.84091264, 0.80649567, 0.79549813, 0.77560693,
     &   0.68449700, 0.54871678, 0.50434560, 0.50493133, 0.51308179,
     &   0.52368975, 0.53095746, 0.53362697, 0.53519613, 0.53526932,
     &   0.53390634, 0.53193331, 0.53018475, 0.53075910, 0.53834158,
     &   0.56128734, 0.60773057, 0.66828561, 0.72060710, 0.75891036,
     &   0.77460575, 0.77478981, 0.76517802, 0.74997932, 0.72840691,
     &   0.70910048, 0.68656749, 0.66454720, 0.64252847, 0.62200046,
     &   0.60417295, 0.58837819, 0.57722980, 0.57747865, 0.58499628,
     &   0.59679538, 0.60985690, 0.62638408, 0.64270645, 0.65521634,
     &   0.65817130, 0.64749140, 0.62755817, 0.59349418, 0.55094743,
     &   0.51251608, 0.50073999, 0.50425029, 0.50891298, 0.51045990,
     &   0.51587820, 0.52185225, 0.52787751, 0.53257489, 0.53516585,
     &   0.53693587, 0.53807092, 0.53863579, 0.53923851, 0.53960818,
     &   0.53996527, 0.53999466, 0.53966260, 0.53960264, 0.53941596,
     &   0.53917396, 0.53895575, 0.53866750, 0.53837222, 0.53804654,
     &   0.53789169, 0.53794116, 0.53822345, 0.53894281, 0.53982806,
     &   0.54108602, 0.54187447, 0.54088992, 0.53577495, 0.52640712,
     &   0.51261115, 0.49915206, 0.48585257, 0.47674894, 0.47262692,
     &   0.47235355, 0.47552928, 0.47947150, 0.48344275, 0.48661661,
     &   0.48932725, 0.49162400, 0.49350846, 0.49484307, 0.49600008,
     &   0.49684086, 0.49736652, 0.49790561, 0.49819306, 0.49839520,
     &   0.49859226, 0.49874216, 0.49883372, 0.49873531, 0.49863249,
     &   0.49847269, 0.49807492, 0.49775848, 0.49720061, 0.49674633,
     &   0.49630898, 0.49560115, 0.49489620, 0.49414292, 0.49327517,
     &   0.49222109, 0.49145856, 0.49072209, 0.49002972, 0.48937556,
     &   0.48874521, 0.48816031, 0.48760155, 0.48701799, 0.48638541,
     &   0.48580870, 0.48523387, 0.48471850, 0.48424196, 0.48383030,
     &   0.48348591, 0.48322001, 0.48305288, 0.48298237, 0.48303759,
     &   0.48322356, 0.48353454, 0.48391443, 0.48435351, 0.48473415,
     &   0.48506218, 0.48530582, 0.48540068, 0.48540348, 0.48532408,
     &   0.48512626, 0.48471555, 0.48402339, 0.48310295, 0.48206130,
     &   0.48063084, 0.47879905, 0.47686487, 0.47494754, 0.47305602,
     &   0.47099811, 0.46895677, 0.46706593, 0.46544430, 0.46393085,
     &   0.46256009, 0.46135595, 0.46032673, 0.45949793, 0.45906067,
     &   0.45872283, 0.45837650, 0.45809096, 0.45791757, 0.45780969,
     &   0.45801291, 0.45819938, 0.45836237, 0.45851943, 0.45874882,
     &   0.45898536, 0.45941672, 0.45985907, 0.46033314, 0.46079540,
     &   0.46117935, 0.46161947, 0.46212626, 0.46270213, 0.46333757,
     &   0.46405247, 0.46479201, 0.46553540, 0.46629158, 0.46707642,
     &   0.46787009, 0.46899712, 0.47029966, 0.47158030, 0.47291282,
     &   0.47438663, 0.47605723, 0.47768718, 0.47921264, 0.48082471,
     &   0.48254940, 0.48410562, 0.48583555, 0.48777831, 0.48964730,
     &   0.49135771, 0.49301913, 0.49442053, 0.49582791, 0.49723774,
     &   0.49864665, 0.50005138, 0.50007892, 0.49974161, 0.49932608,
     &   0.49883074, 0.49825415, 0.49752435, 0.49670503, 0.49580607,
     &   0.49482647, 0.49375695, 0.49258560, 0.49133003, 0.48998946,
     &   0.48849440, 0.48646173, 0.48433390, 0.48211044, 0.47976923,
     &   0.47725648, 0.47464770, 0.47194314, 0.46909025, 0.46610066,
     &   0.46301225, 0.45974633, 0.45603374, 0.45222092, 0.44830763,
     &   0.44445652, 0.44052249, 0.43649137, 0.43259045, 0.42865169,
     &   0.42461532, 0.42040977, 0.41609955, 0.41168573, 0.40715492,
     &   0.40252465, 0.39781508, 0.39302084, 0.38812730, 0.38308606,
     &   0.37794676, 0.37291804, 0.36791551, 0.36283702, 0.35771403,
     &   0.35248733, 0.34703451, 0.34149152, 0.33584082, 0.33010495,
     &   0.32419968, 0.31820750, 0.31213710, 0.30599910, 0.29975379,
     &   0.29345250, 0.28710207, 0.28070864, 0.27427855, 0.26781899/
      data (ww(i,10),i=1,mxwv)/
     &   0.99999154, 0.99999249, 0.99999315, 0.99999374, 0.99999410,
     &   0.99999481, 0.99999487, 0.99999535, 0.99999547, 0.99999595,
     &   0.99999595, 0.99999642, 0.99999660, 0.99999696, 0.99999720,
     &   0.99999738, 0.99999756, 0.99999774, 0.99999803, 0.99999815,
     &   0.99999833, 0.99999851, 0.99999869, 0.99999887, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999875, 0.99999851, 0.99999833,
     &   0.99999809, 0.99999779, 0.99999714, 0.99999690, 0.99999636,
     &   0.99999493, 0.99999249, 0.99998987, 0.99998844, 0.99998772,
     &   0.99998736, 0.99998623, 0.99998432, 0.99998337, 0.99998015,
     &   0.99997413, 0.99995822, 0.99993020, 0.99989992, 0.99989367,
     &   0.99989611, 0.99990797, 0.99991739, 0.99991751, 0.99989474,
     &   0.99985975, 0.99981785, 0.99978757, 0.99975973, 0.99973071,
     &   0.99962676, 0.99921668, 0.99855906, 0.99821609, 0.99826503,
     &   0.99846816, 0.99876815, 0.99908876, 0.99933749, 0.99939471,
     &   0.99929035, 0.99905360, 0.99827147, 0.99610591, 0.99522638,
     &   0.99506927, 0.99512196, 0.99534708, 0.99567366, 0.99557096,
     &   0.99454027, 0.99244213, 0.98585135, 0.97886097, 0.95490962,
     &   0.90218180, 0.89463973, 0.90574646, 0.93213814, 0.94965965,
     &   0.96064270, 0.96779346, 0.97354442, 0.97671461, 0.97856516,
     &   0.97809803, 0.97585803, 0.97203457, 0.96548039, 0.96426469,
     &   0.94706756, 0.82881528, 0.71952885, 0.72557294, 0.77997142,
     &   0.84104294, 0.87632841, 0.90425920, 0.92024285, 0.92895001,
     &   0.93161029, 0.92637753, 0.91510540, 0.89926368, 0.87306631,
     &   0.83912361, 0.79562330, 0.75616127, 0.74388063, 0.72217172,
     &   0.63179404, 0.52959031, 0.50758821, 0.50938952, 0.51760924,
     &   0.52790254, 0.53501958, 0.53769839, 0.53929943, 0.53942776,
     &   0.53814071, 0.53624463, 0.53438050, 0.53303868, 0.53451931,
     &   0.54466611, 0.57336801, 0.62064832, 0.66825229, 0.70647293,
     &   0.72286558, 0.72301728, 0.71283954, 0.69709510, 0.67542171,
     &   0.65673369, 0.63581055, 0.61632711, 0.59786159, 0.58162230,
     &   0.56834280, 0.55725783, 0.54988527, 0.54997671, 0.55480009,
     &   0.56275725, 0.57203341, 0.58446050, 0.59738922, 0.60768461,
     &   0.61006331, 0.60089445, 0.58450639, 0.55864626, 0.53052944,
     &   0.51033640, 0.50665969, 0.51177692, 0.51659578, 0.51598895,
     &   0.51661372, 0.51827407, 0.52041686, 0.52222663, 0.52314138,
     &   0.52370220, 0.52395427, 0.52392316, 0.52390730, 0.52375799,
     &   0.52357739, 0.52319080, 0.52257162, 0.52206910, 0.52144396,
     &   0.52073109, 0.51995981, 0.51906615, 0.51807833, 0.51698244,
     &   0.51584721, 0.51471019, 0.51359105, 0.51258242, 0.51161897,
     &   0.51092780, 0.51029193, 0.50959748, 0.50818676, 0.50570929,
     &   0.50118840, 0.49608642, 0.49053922, 0.48687029, 0.48602432,
     &   0.48741135, 0.49109644, 0.49503064, 0.49877843, 0.50173390,
     &   0.50431311, 0.50648063, 0.50827461, 0.50955003, 0.51068044,
     &   0.51150548, 0.51204520, 0.51261473, 0.51293415, 0.51317507,
     &   0.51343042, 0.51362926, 0.51377177, 0.51371872, 0.51367104,
     &   0.51355565, 0.51318949, 0.51291168, 0.51237661, 0.51195949,
     &   0.51155007, 0.51083016, 0.51011050, 0.50933880, 0.50842458,
     &   0.50727171, 0.50646877, 0.50567830, 0.50492328, 0.50419760,
     &   0.50348127, 0.50279832, 0.50212294, 0.50138062, 0.50049812,
     &   0.49963945, 0.49874625, 0.49787250, 0.49696398, 0.49605134,
     &   0.49512431, 0.49417606, 0.49324512, 0.49231893, 0.49141866,
     &   0.49051768, 0.48966935, 0.48886353, 0.48813403, 0.48747391,
     &   0.48688352, 0.48634976, 0.48586416, 0.48540962, 0.48497814,
     &   0.48455444, 0.48412001, 0.48365369, 0.48315015, 0.48261616,
     &   0.48199400, 0.48126784, 0.48052007, 0.47976378, 0.47900230,
     &   0.47819948, 0.47742113, 0.47670799, 0.47608343, 0.47551212,
     &   0.47501308, 0.47458851, 0.47424036, 0.47401908, 0.47389984,
     &   0.47377008, 0.47362068, 0.47351608, 0.47348842, 0.47346413,
     &   0.47345230, 0.47342825, 0.47334927, 0.47326836, 0.47321311,
     &   0.47315633, 0.47310504, 0.47305265, 0.47294831, 0.47285077,
     &   0.47273803, 0.47264880, 0.47258931, 0.47255245, 0.47254822,
     &   0.47256935, 0.47261894, 0.47269148, 0.47279370, 0.47293845,
     &   0.47312281, 0.47345018, 0.47387952, 0.47434682, 0.47488499,
     &   0.47553566, 0.47633493, 0.47719988, 0.47809085, 0.47909930,
     &   0.48024616, 0.48139265, 0.48270515, 0.48423266, 0.48583215,
     &   0.48744982, 0.48914790, 0.49084195, 0.49262810, 0.49450317,
     &   0.49646378, 0.49850613, 0.49969435, 0.50067568, 0.50163859,
     &   0.50257909, 0.50349319, 0.50425279, 0.50495934, 0.50562865,
     &   0.50626075, 0.50683790, 0.50735843, 0.50782931, 0.50824797,
     &   0.50855482, 0.50842553, 0.50822258, 0.50794423, 0.50756317,
     &   0.50701421, 0.50638485, 0.50567383, 0.50483686, 0.50388187,
     &   0.50283968, 0.50163400, 0.50000316, 0.49827811, 0.49645847,
     &   0.49463409, 0.49272847, 0.49073365, 0.48884690, 0.48692361,
     &   0.48491400, 0.48275119, 0.48049387, 0.47814044, 0.47567013,
     &   0.47310922, 0.47047171, 0.46775335, 0.46494353, 0.46196574,
     &   0.45889407, 0.45592174, 0.45297226, 0.44995612, 0.44691068,
     &   0.44376299, 0.44039291, 0.43692589, 0.43334460, 0.42966378,
     &   0.42578912, 0.42180189, 0.41770780, 0.41351065, 0.40914690,
     &   0.40467870, 0.40010694, 0.39543277, 0.39065722, 0.38577753/
      data (ww(i,11),i=1,mxwv)/
     &   0.99998826, 0.99998951, 0.99999034, 0.99999130, 0.99999213,
     &   0.99999267, 0.99999315, 0.99999350, 0.99999368, 0.99999428,
     &   0.99999464, 0.99999493, 0.99999541, 0.99999571, 0.99999601,
     &   0.99999630, 0.99999660, 0.99999690, 0.99999714, 0.99999744,
     &   0.99999756, 0.99999791, 0.99999809, 0.99999833, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999827, 0.99999791, 0.99999768,
     &   0.99999726, 0.99999696, 0.99999648, 0.99999589, 0.99999493,
     &   0.99999285, 0.99998951, 0.99998581, 0.99998379, 0.99998283,
     &   0.99998248, 0.99998111, 0.99997783, 0.99997628, 0.99997324,
     &   0.99996394, 0.99994171, 0.99990135, 0.99986047, 0.99985313,
     &   0.99985576, 0.99987149, 0.99988687, 0.99988478, 0.99985260,
     &   0.99980354, 0.99974012, 0.99970430, 0.99966669, 0.99962330,
     &   0.99948180, 0.99891329, 0.99799120, 0.99751437, 0.99758202,
     &   0.99786550, 0.99828392, 0.99872899, 0.99907577, 0.99915826,
     &   0.99900669, 0.99867791, 0.99759334, 0.99457920, 0.99335223,
     &   0.99313205, 0.99321020, 0.99353194, 0.99399441, 0.99383706,
     &   0.99240619, 0.98948658, 0.98038191, 0.97075796, 0.93834168,
     &   0.86975056, 0.86027735, 0.87428302, 0.90826225, 0.93136489,
     &   0.94606501, 0.95572597, 0.96363080, 0.96774822, 0.97040725,
     &   0.96976870, 0.96671301, 0.96150815, 0.95262158, 0.95098943,
     &   0.92799157, 0.78130013, 0.66559464, 0.67143440, 0.72704762,
     &   0.79553103, 0.83779490, 0.87257665, 0.89298767, 0.90426397,
     &   0.90773314, 0.90092915, 0.88640666, 0.86628938, 0.83381474,
     &   0.79324943, 0.74386543, 0.70172065, 0.68907005, 0.66737163,
     &   0.58727348, 0.52125454, 0.51085520, 0.51290441, 0.52090764,
     &   0.53088248, 0.53787529, 0.54057050, 0.54220438, 0.54238307,
     &   0.54116094, 0.53933865, 0.53751868, 0.53574479, 0.53491622,
     &   0.53783709, 0.55182058, 0.58272618, 0.62041819, 0.65449154,
     &   0.66999578, 0.67007560, 0.66025966, 0.64548475, 0.62599725,
     &   0.60996044, 0.59293437, 0.57801348, 0.56478149, 0.55394202,
     &   0.54569107, 0.53928715, 0.53527886, 0.53524321, 0.53767914,
     &   0.54197127, 0.54731357, 0.55499411, 0.56352991, 0.57063991,
     &   0.57217115, 0.56534159, 0.55380243, 0.53726029, 0.52184290,
     &   0.51299405, 0.51249599, 0.51784253, 0.52258897, 0.52146447,
     &   0.52037179, 0.52008021, 0.52019638, 0.52035844, 0.52028197,
     &   0.52012175, 0.51986271, 0.51951057, 0.51917118, 0.51877660,
     &   0.51836056, 0.51784754, 0.51722413, 0.51664025, 0.51598096,
     &   0.51526064, 0.51448327, 0.51361877, 0.51266956, 0.51163274,
     &   0.51051986, 0.50934023, 0.50809717, 0.50679797, 0.50541514,
     &   0.50404149, 0.50255233, 0.50123072, 0.49987581, 0.49881411,
     &   0.49759549, 0.49626884, 0.49493143, 0.49461335, 0.49593323,
     &   0.49843776, 0.50252366, 0.50650394, 0.51016790, 0.51304650,
     &   0.51559108, 0.51773584, 0.51953286, 0.52083665, 0.52201247,
     &   0.52289498, 0.52350938, 0.52415973, 0.52456617, 0.52489865,
     &   0.52525240, 0.52554828, 0.52578992, 0.52583832, 0.52589554,
     &   0.52588326, 0.52562135, 0.52544898, 0.52502024, 0.52471268,
     &   0.52441055, 0.52379322, 0.52317727, 0.52251107, 0.52169913,
     &   0.52064121, 0.51994342, 0.51925528, 0.51860154, 0.51797473,
     &   0.51735491, 0.51676661, 0.51618659, 0.51552397, 0.51470715,
     &   0.51390791, 0.51306248, 0.51222670, 0.51133692, 0.51042408,
     &   0.50947207, 0.50846666, 0.50744760, 0.50639313, 0.50532103,
     &   0.50418299, 0.50301391, 0.50188947, 0.50076807, 0.49969733,
     &   0.49868780, 0.49773532, 0.49686292, 0.49603325, 0.49523497,
     &   0.49446154, 0.49372375, 0.49304709, 0.49241284, 0.49177986,
     &   0.49121657, 0.49072316, 0.49033463, 0.48996609, 0.48961225,
     &   0.48933128, 0.48914230, 0.48902327, 0.48892611, 0.48887861,
     &   0.48889205, 0.48894811, 0.48904189, 0.48925638, 0.48941678,
     &   0.48948872, 0.48952365, 0.48960450, 0.48976126, 0.48986524,
     &   0.48984149, 0.48978671, 0.48963889, 0.48948768, 0.48933986,
     &   0.48918116, 0.48891234, 0.48862675, 0.48820338, 0.48778403,
     &   0.48736411, 0.48693687, 0.48650199, 0.48604777, 0.48558843,
     &   0.48508239, 0.48457673, 0.48406851, 0.48356435, 0.48308903,
     &   0.48267010, 0.48222694, 0.48178303, 0.48133978, 0.48092932,
     &   0.48054633, 0.48019201, 0.47986147, 0.47958946, 0.47937834,
     &   0.47923756, 0.47915673, 0.47916785, 0.47928381, 0.47948849,
     &   0.47977671, 0.48016873, 0.48065081, 0.48125663, 0.48199022,
     &   0.48285523, 0.48385507, 0.48462045, 0.48537570, 0.48619249,
     &   0.48706883, 0.48800245, 0.48889479, 0.48981449, 0.49077284,
     &   0.49176708, 0.49279159, 0.49383947, 0.49491304, 0.49600902,
     &   0.49708924, 0.49794662, 0.49879551, 0.49963215, 0.50043100,
     &   0.50113046, 0.50180322, 0.50244606, 0.50302738, 0.50354880,
     &   0.50402778, 0.50440162, 0.50445557, 0.50444722, 0.50437433,
     &   0.50424838, 0.50406104, 0.50380951, 0.50363374, 0.50343323,
     &   0.50317001, 0.50279057, 0.50233907, 0.50181055, 0.50117886,
     &   0.50047457, 0.49970368, 0.49886355, 0.49794662, 0.49686217,
     &   0.49569660, 0.49460500, 0.49352908, 0.49240464, 0.49127194,
     &   0.49005523, 0.48865545, 0.48717344, 0.48559779, 0.48393694,
     &   0.48210391, 0.48017195, 0.47814772, 0.47603261, 0.47375360,
     &   0.47138065, 0.46891433, 0.46635649, 0.46370876, 0.46096510/
      data (ww(i,12),i=1,mxwv)/
     &   0.99998367, 0.99998516, 0.99998641, 0.99998671, 0.99998891,
     &   0.99998975, 0.99999005, 0.99999088, 0.99999136, 0.99999195,
     &   0.99999237, 0.99999303, 0.99999356, 0.99999398, 0.99999440,
     &   0.99999475, 0.99999523, 0.99999565, 0.99999595, 0.99999642,
     &   0.99999654, 0.99999708, 0.99999738, 0.99999774, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999750, 0.99999702, 0.99999672,
     &   0.99999613, 0.99999571, 0.99999505, 0.99999434, 0.99999261,
     &   0.99999011, 0.99998516, 0.99998015, 0.99997747, 0.99997598,
     &   0.99997556, 0.99997342, 0.99996805, 0.99996746, 0.99996263,
     &   0.99994940, 0.99991870, 0.99986297, 0.99980479, 0.99979115,
     &   0.99979824, 0.99982071, 0.99984294, 0.99983919, 0.99979413,
     &   0.99972707, 0.99964803, 0.99958742, 0.99953276, 0.99947369,
     &   0.99927741, 0.99847138, 0.99719435, 0.99652433, 0.99661607,
     &   0.99701500, 0.99760425, 0.99822330, 0.99870783, 0.99882370,
     &   0.99861240, 0.99815071, 0.99664146, 0.99242145, 0.99072492,
     &   0.99041867, 0.99053085, 0.99097902, 0.99161476, 0.99137789,
     &   0.98942226, 0.98537213, 0.97278976, 0.95964092, 0.91620308,
     &   0.82934898, 0.81791693, 0.83488947, 0.87728554, 0.90709072,
     &   0.92645705, 0.93934751, 0.94987309, 0.95558423, 0.95918828,
     &   0.95832604, 0.95418096, 0.94714957, 0.93522829, 0.93305647,
     &   0.90277272, 0.72818893, 0.61663979, 0.62161571, 0.67265534,
     &   0.74365085, 0.79146600, 0.83295405, 0.85816348, 0.87235892,
     &   0.87676525, 0.86814731, 0.84997398, 0.82531673, 0.78680372,
     &   0.74102783, 0.68900931, 0.64806253, 0.63633054, 0.61697817,
     &   0.55529356, 0.51913083, 0.51342607, 0.51549911, 0.52323282,
     &   0.53292847, 0.53982377, 0.54253823, 0.54420334, 0.54442590,
     &   0.54325706, 0.54149371, 0.53973138, 0.53792590, 0.53648901,
     &   0.53642637, 0.54132771, 0.55749613, 0.58249271, 0.60871667,
     &   0.62156630, 0.62153476, 0.61314666, 0.60097593, 0.58575076,
     &   0.57393336, 0.56218159, 0.55260664, 0.54475439, 0.53882056,
     &   0.53464520, 0.53164095, 0.52988291, 0.52978468, 0.53067207,
     &   0.53239948, 0.53473121, 0.53842795, 0.54288268, 0.54677999,
     &   0.54744238, 0.54310179, 0.53626949, 0.52745527, 0.52028865,
     &   0.51664722, 0.51711321, 0.52239591, 0.52704054, 0.52587491,
     &   0.52438992, 0.52349859, 0.52287918, 0.52237952, 0.52189505,
     &   0.52143091, 0.52096063, 0.52048671, 0.52002162, 0.51954073,
     &   0.51904315, 0.51850718, 0.51792824, 0.51735377, 0.51673776,
     &   0.51608467, 0.51538956, 0.51464206, 0.51383883, 0.51298213,
     &   0.51206148, 0.51107579, 0.51002443, 0.50887591, 0.50762594,
     &   0.50628752, 0.50476021, 0.50336117, 0.50190330, 0.50089693,
     &   0.50034648, 0.50011188, 0.50035065, 0.50150532, 0.50384003,
     &   0.50685602, 0.51108670, 0.51504248, 0.51863509, 0.52146900,
     &   0.52399635, 0.52614397, 0.52796543, 0.52931750, 0.53055203,
     &   0.53150576, 0.53220356, 0.53293908, 0.53343862, 0.53386754,
     &   0.53431964, 0.53471518, 0.53505844, 0.53521305, 0.53537756,
     &   0.53547335, 0.53532451, 0.53526402, 0.53495216, 0.53476071,
     &   0.53457385, 0.53407598, 0.53358060, 0.53303754, 0.53235155,
     &   0.53142375, 0.53085399, 0.53029364, 0.52976733, 0.52926815,
     &   0.52877611, 0.52831501, 0.52786243, 0.52732784, 0.52663970,
     &   0.52596855, 0.52525204, 0.52454430, 0.52378142, 0.52299386,
     &   0.52216494, 0.52127844, 0.52037382, 0.51942778, 0.51845640,
     &   0.51740730, 0.51634568, 0.51525152, 0.51417619, 0.51313752,
     &   0.51214772, 0.51120228, 0.51033401, 0.50949842, 0.50868434,
     &   0.50788593, 0.50711751, 0.50642282, 0.50577909, 0.50513494,
     &   0.50459474, 0.50417155, 0.50390625, 0.50367826, 0.50347930,
     &   0.50340003, 0.50345218, 0.50359178, 0.50374365, 0.50395268,
     &   0.50422800, 0.50454181, 0.50488526, 0.50535673, 0.50572324,
     &   0.50597090, 0.50617272, 0.50642306, 0.50675279, 0.50703073,
     &   0.50709087, 0.50713426, 0.50706822, 0.50699782, 0.50692517,
     &   0.50683814, 0.50660378, 0.50634700, 0.50591826, 0.50549072,
     &   0.50506252, 0.50461549, 0.50414753, 0.50364155, 0.50311673,
     &   0.50251251, 0.50188839, 0.50125057, 0.50059307, 0.49995330,
     &   0.49937397, 0.49870476, 0.49798447, 0.49722430, 0.49646413,
     &   0.49567708, 0.49484241, 0.49396318, 0.49311718, 0.49226695,
     &   0.49141058, 0.49058509, 0.48977375, 0.48894894, 0.48813671,
     &   0.48735335, 0.48659441, 0.48583680, 0.48513409, 0.48449144,
     &   0.48391429, 0.48340845, 0.48296252, 0.48256728, 0.48222676,
     &   0.48194385, 0.48172176, 0.48156258, 0.48145896, 0.48141161,
     &   0.48142233, 0.48149225, 0.48162180, 0.48181364, 0.48206899,
     &   0.48237854, 0.48267868, 0.48302883, 0.48342839, 0.48386738,
     &   0.48431823, 0.48480886, 0.48533773, 0.48589200, 0.48647061,
     &   0.48707908, 0.48768571, 0.48817801, 0.48867664, 0.48917863,
     &   0.48965901, 0.49013594, 0.49060857, 0.49114311, 0.49169242,
     &   0.49223676, 0.49274421, 0.49323672, 0.49370763, 0.49413335,
     &   0.49453697, 0.49491659, 0.49526981, 0.49559346, 0.49580130,
     &   0.49597004, 0.49619228, 0.49643481, 0.49666432, 0.49691519,
     &   0.49712375, 0.49722409, 0.49727795, 0.49727952, 0.49722940,
     &   0.49705952, 0.49682295, 0.49652508, 0.49616399, 0.49566761,
     &   0.49510157, 0.49446541, 0.49376118, 0.49299094, 0.49214512/
      data (ww(i,13),i=1,mxwv)/
     &   0.99997699, 0.99997914, 0.99998087, 0.99998271, 0.99998432,
     &   0.99998552, 0.99998641, 0.99998713, 0.99998784, 0.99998844,
     &   0.99998939, 0.99999022, 0.99999082, 0.99999154, 0.99999213,
     &   0.99999267, 0.99999332, 0.99999380, 0.99999422, 0.99999493,
     &   0.99999529, 0.99999589, 0.99999619, 0.99999678, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999654, 0.99999577, 0.99999541,
     &   0.99999475, 0.99999398, 0.99999309, 0.99999183, 0.99998987,
     &   0.99998599, 0.99997926, 0.99997205, 0.99996847, 0.99996632,
     &   0.99996573, 0.99996263, 0.99995756, 0.99995434, 0.99994737,
     &   0.99992925, 0.99988627, 0.99980742, 0.99972659, 0.99971008,
     &   0.99971664, 0.99974847, 0.99977845, 0.99977440, 0.99971014,
     &   0.99961638, 0.99950755, 0.99942231, 0.99934632, 0.99926257,
     &   0.99898750, 0.99785775, 0.99606615, 0.99513042, 0.99526417,
     &   0.99581885, 0.99664354, 0.99751544, 0.99819231, 0.99835342,
     &   0.99805808, 0.99741244, 0.99529612, 0.98941237, 0.98704678,
     &   0.98662060, 0.98677868, 0.98740399, 0.98830020, 0.98799449,
     &   0.98522592, 0.97961640, 0.96232492, 0.94450611, 0.88726640,
     &   0.78134179, 0.76828617, 0.78776211, 0.83839124, 0.87564391,
     &   0.90055680, 0.91744858, 0.93141639, 0.93904877, 0.94391155,
     &   0.94275421, 0.93719256, 0.92781103, 0.91206563, 0.90922564,
     &   0.87025243, 0.67358041, 0.57815468, 0.58176100, 0.62253773,
     &   0.68885893, 0.73896134, 0.78560424, 0.81530929, 0.83247226,
     &   0.83786315, 0.82734984, 0.80553913, 0.77677506, 0.73380107,
     &   0.68600887, 0.63639218, 0.60120153, 0.59166449, 0.57667440,
     &   0.53647351, 0.51946235, 0.51529539, 0.51734155, 0.52482080,
     &   0.53428578, 0.54110628, 0.54383987, 0.54553270, 0.54579216,
     &   0.54466587, 0.54294807, 0.54123217, 0.53946346, 0.53793836,
     &   0.53699470, 0.53776848, 0.54409206, 0.55735433, 0.57408190,
     &   0.58304834, 0.58288193, 0.57670957, 0.56817120, 0.55816764,
     &   0.55091363, 0.54423040, 0.53920668, 0.53541619, 0.53276902,
     &   0.53103119, 0.52985555, 0.52920467, 0.52909732, 0.52926201,
     &   0.52968055, 0.53033358, 0.53156704, 0.53323275, 0.53474528,
     &   0.53474915, 0.53235066, 0.52888763, 0.52482665, 0.52165365,
     &   0.51982540, 0.52052975, 0.52569902, 0.53024846, 0.52913237,
     &   0.52763742, 0.52665609, 0.52588081, 0.52522361, 0.52464688,
     &   0.52412492, 0.52362269, 0.52314711, 0.52267784, 0.52220684,
     &   0.52172065, 0.52121770, 0.52069759, 0.52016979, 0.51961285,
     &   0.51902980, 0.51841223, 0.51775807, 0.51706308, 0.51633191,
     &   0.51554906, 0.51471114, 0.51381814, 0.51283193, 0.51175648,
     &   0.51057911, 0.50920898, 0.50793040, 0.50652736, 0.50550497,
     &   0.50504297, 0.50505602, 0.50576413, 0.50742036, 0.51009744,
     &   0.51325291, 0.51745033, 0.52132201, 0.52483356, 0.52762604,
     &   0.53013498, 0.53228742, 0.53413320, 0.53553152, 0.53681993,
     &   0.53783953, 0.53861254, 0.53942364, 0.54000670, 0.54052180,
     &   0.54106021, 0.54154438, 0.54197824, 0.54222816, 0.54248822,
     &   0.54268128, 0.54263502, 0.54267538, 0.54246980, 0.54238254,
     &   0.54229957, 0.54191178, 0.54152656, 0.54109567, 0.54052532,
     &   0.53971875, 0.53926480, 0.53882104, 0.53841120, 0.53802848,
     &   0.53765333, 0.53730905, 0.53697371, 0.53655839, 0.53599262,
     &   0.53544432, 0.53485334, 0.53427172, 0.53363669, 0.53297794,
     &   0.53227925, 0.53152478, 0.53075349, 0.52994215, 0.52910578,
     &   0.52819300, 0.52726769, 0.52630955, 0.52536845, 0.52446365,
     &   0.52360415, 0.52278620, 0.52204347, 0.52132982, 0.52063495,
     &   0.51995260, 0.51929742, 0.51871455, 0.51818156, 0.51764596,
     &   0.51721650, 0.51690984, 0.51677066, 0.51667279, 0.51660782,
     &   0.51667255, 0.51687872, 0.51717865, 0.51749021, 0.51786232,
     &   0.51830304, 0.51878226, 0.51928973, 0.51992661, 0.52044719,
     &   0.52084255, 0.52119052, 0.52158695, 0.52206236, 0.52248263,
     &   0.52267432, 0.52284706, 0.52290916, 0.52296698, 0.52302158,
     &   0.52306092, 0.52294701, 0.52281052, 0.52249783, 0.52218646,
     &   0.52187479, 0.52154326, 0.52118880, 0.52079451, 0.52037787,
     &   0.51987779, 0.51935685, 0.51881534, 0.51825202, 0.51770395,
     &   0.51721668, 0.51662940, 0.51598221, 0.51528519, 0.51458043,
     &   0.51383698, 0.51302892, 0.51215541, 0.51130283, 0.51042604,
     &   0.50951910, 0.50862426, 0.50771803, 0.50675750, 0.50577450,
     &   0.50477928, 0.50377035, 0.50269592, 0.50162750, 0.50056690,
     &   0.49951559, 0.49847540, 0.49752092, 0.49658957, 0.49566665,
     &   0.49475387, 0.49385306, 0.49302822, 0.49222529, 0.49143639,
     &   0.49066344, 0.48990822, 0.48917246, 0.48845887, 0.48776966,
     &   0.48710936, 0.48648968, 0.48589623, 0.48533344, 0.48480418,
     &   0.48431745, 0.48386234, 0.48344052, 0.48305261, 0.48269993,
     &   0.48238465, 0.48210701, 0.48185655, 0.48163879, 0.48145369,
     &   0.48130804, 0.48119280, 0.48110735, 0.48106733, 0.48106393,
     &   0.48109344, 0.48114845, 0.48123422, 0.48134902, 0.48148555,
     &   0.48164955, 0.48183998, 0.48205581, 0.48229593, 0.48252445,
     &   0.48276937, 0.48306236, 0.48339355, 0.48375392, 0.48416081,
     &   0.48458549, 0.48500076, 0.48542881, 0.48586783, 0.48631513,
     &   0.48673546, 0.48715225, 0.48756766, 0.48797828, 0.48833558,
     &   0.48867881, 0.48900640, 0.48931867, 0.48961639, 0.48989040/

c         ---------------- asymmetry factor ---------------

      data (gg(i,1),i=1,mxwv)/
     &   0.82582545, 0.82592523, 0.82585710, 0.82588977, 0.82582527,
     &   0.82564992, 0.82560641, 0.82534742, 0.82520568, 0.82506657,
     &   0.82481307, 0.82456845, 0.82425445, 0.82396966, 0.82363415,
     &   0.82320780, 0.82287830, 0.82244104, 0.82199508, 0.82151937,
     &   0.82103729, 0.82049131, 0.81991357, 0.81933504, 0.81872535,
     &   0.81806701, 0.81734812, 0.81660944, 0.81584901, 0.81504452,
     &   0.81418365, 0.81330740, 0.81237507, 0.81141019, 0.81038910,
     &   0.80933768, 0.80824000, 0.80711931, 0.80594260, 0.80473220,
     &   0.80347955, 0.80219209, 0.80086428, 0.79951847, 0.79814166,
     &   0.79673654, 0.79531223, 0.79386520, 0.79240590, 0.79093921,
     &   0.78946817, 0.78800279, 0.78654683, 0.78510952, 0.78369778,
     &   0.78231937, 0.78098428, 0.77969915, 0.77847517, 0.77732205,
     &   0.77624935, 0.77526689, 0.77438474, 0.77361232, 0.77296138,
     &   0.77244097, 0.77206379, 0.77183712, 0.77176172, 0.77184343,
     &   0.77209342, 0.77251720, 0.77311635, 0.77389461, 0.77485329,
     &   0.77599204, 0.77730536, 0.77879637, 0.78048605, 0.78226358,
     &   0.78420770, 0.78628397, 0.78848392, 0.79079866, 0.79321790,
     &   0.79573429, 0.79833770, 0.80104482, 0.80381238, 0.80679411,
     &   0.80994910, 0.81254768, 0.81496698, 0.81730813, 0.81980264,
     &   0.82232887, 0.82485163, 0.82733917, 0.82979083, 0.83220702,
     &   0.83462101, 0.83697563, 0.83926922, 0.84150988, 0.84367168,
     &   0.84602505, 0.84901446, 0.85165107, 0.85245973, 0.85309666,
     &   0.85411465, 0.85543901, 0.85682338, 0.85824859, 0.85968673,
     &   0.86113864, 0.86262619, 0.86414945, 0.86575198, 0.86746395,
     &   0.86934125, 0.87142861, 0.87368357, 0.87624252, 0.87963700,
     &   0.88497370, 0.89308161, 0.90238464, 0.90146482, 0.89375055,
     &   0.88199556, 0.87077630, 0.86228466, 0.85282397, 0.84088981,
     &   0.82518446, 0.81116998, 0.79965270, 0.79298598, 0.79247075,
     &   0.79226148, 0.79255801, 0.79326576, 0.79408979, 0.79461628,
     &   0.79487628, 0.79475111, 0.79429221, 0.79352623, 0.79250586,
     &   0.79118663, 0.78969026, 0.78798294, 0.78607655, 0.78397077,
     &   0.78159428, 0.77897614, 0.77590501, 0.77216172, 0.76814109,
     &   0.76407272, 0.76004201, 0.75599086, 0.75198478, 0.74809271,
     &   0.74431711, 0.74065375, 0.73707110, 0.73376703, 0.73080313,
     &   0.72807783, 0.72367293, 0.71595693, 0.70734733, 0.69888288,
     &   0.69204396, 0.68571329, 0.67933685, 0.67286748, 0.66625392,
     &   0.65941244, 0.65234286, 0.64503986, 0.63748878, 0.62969744,
     &   0.62165493, 0.61336565, 0.60482770, 0.59603113, 0.58697444,
     &   0.57765901, 0.56807870, 0.55823809, 0.54813796, 0.53778493,
     &   0.52719527, 0.51635122, 0.50527877, 0.49393955, 0.48236141,
     &   0.47054112, 0.45834744, 0.44618133, 0.43352759, 0.42102569,
     &   0.40871832, 0.39671010, 0.38475269, 0.37320104, 0.36203402,
     &   0.35123599, 0.34106970, 0.33149248, 0.32256624, 0.31413832,
     &   0.30581799, 0.29793465, 0.29031041, 0.28297898, 0.27577883,
     &   0.26885036, 0.26200894, 0.25526813, 0.24875644, 0.24239631,
     &   0.23606311, 0.23000668, 0.22412345, 0.21843770, 0.21282479,
     &   0.20746984, 0.20228088, 0.19719486, 0.19221528, 0.18725319,
     &   0.18251030, 0.17796002, 0.17343809, 0.16885823, 0.16430296,
     &   0.15972769, 0.15507665, 0.15053827, 0.14611103, 0.14179477,
     &   0.13758878, 0.13352397, 0.12956649, 0.12571009, 0.12194680,
     &   0.11827938, 0.11458509, 0.11099261, 0.10747156, 0.10403536,
     &   0.10066457, 0.09735073, 0.09411050, 0.09092317, 0.08781720,
     &   0.08475711, 0.08177216, 0.07883084, 0.07597638, 0.07319383,
     &   0.07050925, 0.06791995, 0.06543289, 0.06303990, 0.06073668,
     &   0.05851362, 0.05636254, 0.05429049, 0.05229382, 0.05036614,
     &   0.04850581, 0.04670938, 0.04500944, 0.04338047, 0.04181745,
     &   0.04031225, 0.03887641, 0.03750962, 0.03621034, 0.03496517,
     &   0.03377532, 0.03263836, 0.03155303, 0.03052563, 0.02956484,
     &   0.02863942, 0.02774037, 0.02687770, 0.02605632, 0.02526627,
     &   0.02452936, 0.02381342, 0.02311428, 0.02243586, 0.02178509,
     &   0.02115416, 0.02055400, 0.01997020, 0.01939676, 0.01883660,
     &   0.01828493, 0.01775069, 0.01723347, 0.01673143, 0.01624404,
     &   0.01576786, 0.01530308, 0.01484825, 0.01440405, 0.01397328,
     &   0.01355793, 0.01315873, 0.01277072, 0.01238773, 0.01201483,
     &   0.01165209, 0.01129850, 0.01094837, 0.01060642, 0.01027373,
     &   0.00995009, 0.00963359, 0.00932789, 0.00903054, 0.00873936,
     &   0.00845465, 0.00817702, 0.00790264, 0.00763738, 0.00738094,
     &   0.00713302, 0.00689335, 0.00665735, 0.00642852, 0.00620756,
     &   0.00599419, 0.00578815, 0.00559134, 0.00540154, 0.00521817,
     &   0.00504102, 0.00486984, 0.00470440, 0.00454457, 0.00439016,
     &   0.00424092, 0.00409623, 0.00395646, 0.00382146, 0.00369117,
     &   0.00356567, 0.00344443, 0.00332730, 0.00321406, 0.00310461,
     &   0.00299887, 0.00289673, 0.00279810, 0.00270280, 0.00261074,
     &   0.00252270, 0.00243769, 0.00235553, 0.00227646, 0.00220013,
     &   0.00212634, 0.00205495, 0.00198594, 0.00191926, 0.00185490,
     &   0.00179268, 0.00173259, 0.00167456, 0.00161845, 0.00156442,
     &   0.00151218, 0.00146191, 0.00141343, 0.00136652, 0.00132110,
     &   0.00127717, 0.00123457, 0.00119338, 0.00115352, 0.00111497,
     &   0.00107767, 0.00104160, 0.00100672, 0.00097300, 0.00094044,
     &   0.00090897, 0.00087854, 0.00084912, 0.00082066, 0.00079314/
      data (gg(i,2),i=1,mxwv)/
     &   0.83627009, 0.83659846, 0.83663005, 0.83670270, 0.83686882,
     &   0.83690906, 0.83694035, 0.83684474, 0.83677542, 0.83682454,
     &   0.83668655, 0.83658659, 0.83646691, 0.83634633, 0.83614570,
     &   0.83592373, 0.83578694, 0.83549982, 0.83532858, 0.83509028,
     &   0.83475798, 0.83446884, 0.83405447, 0.83384895, 0.83346409,
     &   0.83313030, 0.83276808, 0.83230066, 0.83194649, 0.83148319,
     &   0.83109748, 0.83063084, 0.83008826, 0.82956862, 0.82908487,
     &   0.82852161, 0.82798141, 0.82738441, 0.82677656, 0.82617378,
     &   0.82549828, 0.82482111, 0.82410330, 0.82337433, 0.82258445,
     &   0.82180250, 0.82095975, 0.82008660, 0.81918013, 0.81824225,
     &   0.81722403, 0.81621188, 0.81514990, 0.81403714, 0.81290424,
     &   0.81173629, 0.81051993, 0.80927426, 0.80800170, 0.80669385,
     &   0.80534750, 0.80398440, 0.80258876, 0.80117899, 0.79974592,
     &   0.79830289, 0.79685289, 0.79540467, 0.79394943, 0.79249275,
     &   0.79104722, 0.78961992, 0.78822207, 0.78685427, 0.78553617,
     &   0.78427351, 0.78307498, 0.78195798, 0.78094471, 0.78000134,
     &   0.77915174, 0.77841294, 0.77779859, 0.77732247, 0.77700287,
     &   0.77685970, 0.77690613, 0.77720606, 0.77771062, 0.77871221,
     &   0.78036481, 0.78132290, 0.78213137, 0.78293908, 0.78420019,
     &   0.78578764, 0.78764629, 0.78972620, 0.79203528, 0.79455960,
     &   0.79734278, 0.80031121, 0.80345273, 0.80677307, 0.81018698,
     &   0.81414741, 0.81973636, 0.82522154, 0.82739776, 0.82911015,
     &   0.83158195, 0.83482122, 0.83832872, 0.84206462, 0.84595090,
     &   0.84996974, 0.85414451, 0.85845757, 0.86296248, 0.86770225,
     &   0.87274581, 0.87814724, 0.88379961, 0.88986969, 0.89718425,
     &   0.90732378, 0.92101502, 0.93518555, 0.93590158, 0.92911202,
     &   0.91754192, 0.90637332, 0.89815760, 0.88919264, 0.87704295,
     &   0.85732710, 0.83534956, 0.81497556, 0.80224621, 0.80126292,
     &   0.80209434, 0.80450195, 0.80812585, 0.81229156, 0.81617731,
     &   0.81976593, 0.82282186, 0.82540864, 0.82756245, 0.82936561,
     &   0.83074903, 0.83190638, 0.83278763, 0.83341277, 0.83378547,
     &   0.83379912, 0.83350146, 0.83257240, 0.83071029, 0.82852954,
     &   0.82638258, 0.82439715, 0.82249790, 0.82078964, 0.81939876,
     &   0.81833935, 0.81762302, 0.81725740, 0.81763804, 0.81899768,
     &   0.82127768, 0.82124031, 0.81415951, 0.80162561, 0.79155678,
     &   0.78684068, 0.78330159, 0.77986991, 0.77648473, 0.77305311,
     &   0.76942849, 0.76564348, 0.76167184, 0.75751472, 0.75319499,
     &   0.74871570, 0.74408275, 0.73928791, 0.73429054, 0.72911859,
     &   0.72375411, 0.71818864, 0.71241111, 0.70640588, 0.70015448,
     &   0.69364077, 0.68684214, 0.67973685, 0.67229271, 0.66447908,
     &   0.65624845, 0.64749771, 0.63831258, 0.62834412, 0.61784518,
     &   0.60682100, 0.59547061, 0.58352464, 0.57138306, 0.55910558,
     &   0.54688400, 0.53486782, 0.52341312, 0.51261789, 0.50243384,
     &   0.49230000, 0.48262390, 0.47319272, 0.46409330, 0.45504105,
     &   0.44625315, 0.43751350, 0.42875323, 0.42021552, 0.41176525,
     &   0.40322748, 0.39493841, 0.38678047, 0.37883997, 0.37090263,
     &   0.36324027, 0.35580987, 0.34842101, 0.34120595, 0.33391252,
     &   0.32686862, 0.32016787, 0.31346345, 0.30663735, 0.29984090,
     &   0.29303080, 0.28588077, 0.27882427, 0.27185482, 0.26497954,
     &   0.25820878, 0.25159669, 0.24509515, 0.23872399, 0.23249003,
     &   0.22634941, 0.22007392, 0.21390516, 0.20780051, 0.20178203,
     &   0.19581491, 0.18988562, 0.18402441, 0.17819431, 0.17245887,
     &   0.16675198, 0.16113392, 0.15554406, 0.15007971, 0.14471497,
     &   0.13951389, 0.13447754, 0.12962505, 0.12494500, 0.12043165,
     &   0.11606720, 0.11183643, 0.10775577, 0.10381959, 0.10001610,
     &   0.09634242, 0.09279249, 0.08943407, 0.08621509, 0.08312551,
     &   0.08014889, 0.07730906, 0.07460557, 0.07203546, 0.06957155,
     &   0.06721646, 0.06496548, 0.06281620, 0.06078145, 0.05887942,
     &   0.05704686, 0.05526551, 0.05355568, 0.05192748, 0.05036100,
     &   0.04890113, 0.04748214, 0.04609576, 0.04474987, 0.04345868,
     &   0.04220640, 0.04101567, 0.03985688, 0.03871810, 0.03760508,
     &   0.03650808, 0.03544545, 0.03441646, 0.03341743, 0.03244727,
     &   0.03149910, 0.03057328, 0.02966688, 0.02878136, 0.02792246,
     &   0.02709422, 0.02629833, 0.02552464, 0.02476056, 0.02401645,
     &   0.02329252, 0.02258671, 0.02188747, 0.02120440, 0.02053976,
     &   0.01989313, 0.01926063, 0.01864974, 0.01805548, 0.01747345,
     &   0.01690425, 0.01634916, 0.01580043, 0.01526996, 0.01475714,
     &   0.01426139, 0.01378215, 0.01331013, 0.01285247, 0.01241057,
     &   0.01198387, 0.01157186, 0.01117836, 0.01079888, 0.01043228,
     &   0.01007812, 0.00973592, 0.00940517, 0.00908566, 0.00877699,
     &   0.00847863, 0.00818938, 0.00790999, 0.00764011, 0.00737965,
     &   0.00712877, 0.00688641, 0.00665228, 0.00642591, 0.00620711,
     &   0.00599573, 0.00579156, 0.00559438, 0.00540389, 0.00521985,
     &   0.00504386, 0.00487392, 0.00470968, 0.00455163, 0.00439903,
     &   0.00425152, 0.00410881, 0.00397086, 0.00383756, 0.00370889,
     &   0.00358451, 0.00346439, 0.00334836, 0.00323620, 0.00312818,
     &   0.00302375, 0.00292324, 0.00282631, 0.00273253, 0.00264173,
     &   0.00255391, 0.00246873, 0.00238637, 0.00230667, 0.00222961,
     &   0.00215503, 0.00208291, 0.00201317, 0.00194574, 0.00188065,
     &   0.00181772, 0.00175687, 0.00169804, 0.00164112, 0.00158610/
      data (gg(i,3),i=1,mxwv)/
     &   0.84455985, 0.84485096, 0.84503460, 0.84523511, 0.84538072,
     &   0.84563208, 0.84568828, 0.84574807, 0.84588081, 0.84594083,
     &   0.84592026, 0.84598678, 0.84592396, 0.84588182, 0.84584671,
     &   0.84582376, 0.84571087, 0.84563249, 0.84550291, 0.84537148,
     &   0.84523302, 0.84507620, 0.84484732, 0.84476697, 0.84444332,
     &   0.84428525, 0.84400076, 0.84380674, 0.84354097, 0.84331173,
     &   0.84301376, 0.84270489, 0.84237897, 0.84208763, 0.84175521,
     &   0.84138793, 0.84104955, 0.84060657, 0.84030765, 0.83985507,
     &   0.83942181, 0.83906525, 0.83863693, 0.83820271, 0.83766359,
     &   0.83721220, 0.83677375, 0.83627933, 0.83571291, 0.83531868,
     &   0.83471936, 0.83412755, 0.83354622, 0.83288312, 0.83231533,
     &   0.83165139, 0.83099717, 0.83034146, 0.82957327, 0.82887071,
     &   0.82811481, 0.82727188, 0.82648861, 0.82560199, 0.82476044,
     &   0.82388192, 0.82296246, 0.82201469, 0.82103533, 0.82001179,
     &   0.81894821, 0.81785631, 0.81672752, 0.81556922, 0.81439739,
     &   0.81317675, 0.81194711, 0.81070381, 0.80946487, 0.80816841,
     &   0.80683345, 0.80547619, 0.80409986, 0.80271494, 0.80133480,
     &   0.79997998, 0.79865843, 0.79746318, 0.79630262, 0.79561704,
     &   0.79577571, 0.79469544, 0.79312056, 0.79122537, 0.78971910,
     &   0.78849256, 0.78747916, 0.78662771, 0.78598046, 0.78552574,
     &   0.78532881, 0.78536803, 0.78565985, 0.78625280, 0.78700769,
     &   0.78856128, 0.79322964, 0.79878032, 0.79930472, 0.79850447,
     &   0.79875422, 0.80056041, 0.80311215, 0.80641943, 0.81035179,
     &   0.81488729, 0.82007921, 0.82590097, 0.83241725, 0.83973670,
     &   0.84794080, 0.85712224, 0.86701638, 0.87769902, 0.89064717,
     &   0.90846258, 0.93150747, 0.95311749, 0.95590609, 0.94998789,
     &   0.93872488, 0.92796981, 0.92080271, 0.91387290, 0.90524715,
     &   0.88846231, 0.86165524, 0.82762671, 0.79805332, 0.78733510,
     &   0.78259206, 0.78254759, 0.78648686, 0.79288423, 0.79984164,
     &   0.80701917, 0.81376028, 0.82000190, 0.82570952, 0.83098125,
     &   0.83563316, 0.83999014, 0.84394395, 0.84752542, 0.85073042,
     &   0.85338926, 0.85558379, 0.85681653, 0.85660261, 0.85592538,
     &   0.85535437, 0.85510170, 0.85505164, 0.85536933, 0.85624504,
     &   0.85768920, 0.85970110, 0.86227852, 0.86597824, 0.87107486,
     &   0.87732667, 0.88030589, 0.87369573, 0.85839933, 0.84612209,
     &   0.84258795, 0.84101349, 0.83970332, 0.83854777, 0.83739966,
     &   0.83604926, 0.83455569, 0.83287942, 0.83103675, 0.82906634,
     &   0.82698691, 0.82481235, 0.82253838, 0.82011151, 0.81759226,
     &   0.81496322, 0.81222945, 0.80938870, 0.80643511, 0.80335027,
     &   0.80014056, 0.79679084, 0.79328346, 0.78963310, 0.78581131,
     &   0.78178263, 0.77758008, 0.77298433, 0.76805568, 0.76247734,
     &   0.75616741, 0.74919164, 0.74137366, 0.73283529, 0.72361571,
     &   0.71401465, 0.70386934, 0.69391835, 0.68428326, 0.67516202,
     &   0.66605043, 0.65725845, 0.64864951, 0.64036721, 0.63205945,
     &   0.62395519, 0.61590636, 0.60770988, 0.59967655, 0.59162897,
     &   0.58339453, 0.57524943, 0.56711388, 0.55913305, 0.55104345,
     &   0.54309249, 0.53536582, 0.52752781, 0.51989329, 0.51204062,
     &   0.50430495, 0.49698111, 0.48959526, 0.48207411, 0.47460863,
     &   0.46722409, 0.45919818, 0.45116219, 0.44309568, 0.43501252,
     &   0.42693928, 0.41892570, 0.41093770, 0.40306294, 0.39536369,
     &   0.38767549, 0.37972692, 0.37179247, 0.36385259, 0.35591438,
     &   0.34793031, 0.33987877, 0.33177257, 0.32355452, 0.31531486,
     &   0.30695871, 0.29854858, 0.28997570, 0.28140029, 0.27277303,
     &   0.26423383, 0.25580850, 0.24754657, 0.23946342, 0.23156868,
     &   0.22384234, 0.21626292, 0.20887040, 0.20167324, 0.19466785,
     &   0.18784128, 0.18119201, 0.17486876, 0.16878493, 0.16292517,
     &   0.15725191, 0.15181945, 0.14663446, 0.14169870, 0.13695525,
     &   0.13241124, 0.12806056, 0.12390025, 0.11995343, 0.11626942,
     &   0.11272001, 0.10926517, 0.10594362, 0.10277622, 0.09972786,
     &   0.09689724, 0.09414237, 0.09144932, 0.08883125, 0.08631821,
     &   0.08387809, 0.08156190, 0.07930480, 0.07708517, 0.07491116,
     &   0.07276264, 0.07067894, 0.06865892, 0.06669539, 0.06478614,
     &   0.06291765, 0.06109011, 0.05929774, 0.05754391, 0.05584076,
     &   0.05419688, 0.05261735, 0.05108058, 0.04956020, 0.04807804,
     &   0.04663493, 0.04522685, 0.04382974, 0.04246368, 0.04113378,
     &   0.03983932, 0.03857229, 0.03734849, 0.03615772, 0.03499085,
     &   0.03384928, 0.03273576, 0.03163438, 0.03056980, 0.02954081,
     &   0.02854623, 0.02758493, 0.02663783, 0.02571967, 0.02483334,
     &   0.02397770, 0.02315169, 0.02236311, 0.02160278, 0.02086838,
     &   0.02015901, 0.01947367, 0.01881138, 0.01817165, 0.01755372,
     &   0.01695649, 0.01637751, 0.01581832, 0.01527823, 0.01475703,
     &   0.01425508, 0.01377020, 0.01330180, 0.01284898, 0.01241130,
     &   0.01198851, 0.01158013, 0.01118577, 0.01080479, 0.01043675,
     &   0.01008483, 0.00974502, 0.00941661, 0.00910059, 0.00879549,
     &   0.00850058, 0.00821526, 0.00793944, 0.00767294, 0.00741568,
     &   0.00716701, 0.00692687, 0.00669490, 0.00647065, 0.00625469,
     &   0.00604592, 0.00584498, 0.00565121, 0.00546373, 0.00528220,
     &   0.00510663, 0.00493634, 0.00477169, 0.00461234, 0.00445828,
     &   0.00430916, 0.00416498, 0.00402554, 0.00389072, 0.00376058,
     &   0.00363476, 0.00351311, 0.00339547, 0.00328168, 0.00317167/
      data (gg(i,4),i=1,mxwv)/
     &   0.85081232, 0.85113579, 0.85146528, 0.85174489, 0.85205144,
     &   0.85229868, 0.85249537, 0.85271430, 0.85285896, 0.85299277,
     &   0.85316002, 0.85322952, 0.85331571, 0.85341138, 0.85343051,
     &   0.85350990, 0.85346949, 0.85348326, 0.85338873, 0.85345846,
     &   0.85343117, 0.85336089, 0.85325092, 0.85320866, 0.85306942,
     &   0.85294843, 0.85289627, 0.85264999, 0.85257018, 0.85242772,
     &   0.85223860, 0.85207891, 0.85189581, 0.85168809, 0.85141820,
     &   0.85126549, 0.85098267, 0.85076016, 0.85049802, 0.85028118,
     &   0.85000056, 0.84967661, 0.84937501, 0.84912705, 0.84937263,
     &   0.84848493, 0.84814936, 0.84782654, 0.84745914, 0.84710097,
     &   0.84670240, 0.84633636, 0.84593064, 0.84551305, 0.84506303,
     &   0.84460324, 0.84421033, 0.84371644, 0.84327072, 0.84270066,
     &   0.84230822, 0.84185076, 0.84131831, 0.84085870, 0.84025544,
     &   0.83975512, 0.83923119, 0.83860266, 0.83808148, 0.83751965,
     &   0.83692896, 0.83627075, 0.83568621, 0.83502293, 0.83431572,
     &   0.83365059, 0.83300757, 0.83231050, 0.83168280, 0.83096534,
     &   0.83019263, 0.82938969, 0.82857323, 0.82769787, 0.82684958,
     &   0.82600260, 0.82515913, 0.82445931, 0.82375586, 0.82368267,
     &   0.82475203, 0.82395566, 0.82234728, 0.82012844, 0.81829959,
     &   0.81671995, 0.81527698, 0.81387836, 0.81257057, 0.81130725,
     &   0.81014717, 0.80905986, 0.80804741, 0.80716157, 0.80614150,
     &   0.80580586, 0.80975509, 0.81504214, 0.81339782, 0.80906433,
     &   0.80539101, 0.80350810, 0.80230856, 0.80189788, 0.80210501,
     &   0.80293375, 0.80451918, 0.80689418, 0.81018490, 0.81473875,
     &   0.82077187, 0.82867533, 0.83823228, 0.84930396, 0.86486763,
     &   0.89046401, 0.92773694, 0.96182382, 0.96755475, 0.96242714,
     &   0.95152372, 0.94128400, 0.93526137, 0.93043298, 0.92634338,
     &   0.91900939, 0.90185332, 0.86948657, 0.82818484, 0.80124068,
     &   0.78145677, 0.76838702, 0.76273990, 0.76313430, 0.76671398,
     &   0.77272439, 0.77976054, 0.78729051, 0.79496920, 0.80280739,
     &   0.81020558, 0.81765705, 0.82484084, 0.83175385, 0.83829325,
     &   0.84412849, 0.84935659, 0.85311532, 0.85436392, 0.85473776,
     &   0.85526478, 0.85638726, 0.85790098, 0.86012465, 0.86336493,
     &   0.86764228, 0.87291807, 0.87909293, 0.88695550, 0.89673805,
     &   0.90785444, 0.91431409, 0.90862209, 0.89094377, 0.87365401,
     &   0.86945015, 0.86895913, 0.86916959, 0.86979979, 0.87056822,
     &   0.87112111, 0.87152964, 0.87174284, 0.87177110, 0.87167621,
     &   0.87148762, 0.87122667, 0.87088853, 0.87040097, 0.86985612,
     &   0.86922836, 0.86853516, 0.86777627, 0.86695004, 0.86603528,
     &   0.86505330, 0.86399275, 0.86283880, 0.86163843, 0.86036617,
     &   0.85900027, 0.85766286, 0.85596895, 0.85424614, 0.85188890,
     &   0.84881270, 0.84503669, 0.84049070, 0.83512670, 0.82896310,
     &   0.82231402, 0.81478590, 0.80723333, 0.79974216, 0.79265410,
     &   0.78559810, 0.77874410, 0.77204543, 0.76567292, 0.75928175,
     &   0.75306910, 0.74698240, 0.74073350, 0.73463285, 0.72849774,
     &   0.72219795, 0.71588129, 0.70952016, 0.70327723, 0.69690311,
     &   0.69054306, 0.68438345, 0.67802691, 0.67187929, 0.66547132,
     &   0.65901881, 0.65291739, 0.64672405, 0.64045709, 0.63427418,
     &   0.62827706, 0.62162584, 0.61488318, 0.60801888, 0.60104525,
     &   0.59399271, 0.58686709, 0.57967335, 0.57254034, 0.56557262,
     &   0.55853474, 0.55130911, 0.54400909, 0.53667504, 0.52928299,
     &   0.52180803, 0.51423794, 0.50653684, 0.49866205, 0.49065956,
     &   0.48246896, 0.47406906, 0.46533754, 0.45636645, 0.44706321,
     &   0.43756029, 0.42788863, 0.41808772, 0.40822211, 0.39832425,
     &   0.38838124, 0.37836081, 0.36830625, 0.35826531, 0.34828216,
     &   0.33829290, 0.32831287, 0.31860197, 0.30910546, 0.29982051,
     &   0.29066294, 0.28174645, 0.27312136, 0.26483384, 0.25677997,
     &   0.24898212, 0.24144872, 0.23418625, 0.22721982, 0.22070839,
     &   0.21442641, 0.20828547, 0.20233916, 0.19662584, 0.19111435,
     &   0.18602991, 0.18106507, 0.17621298, 0.17147864, 0.16692272,
     &   0.16248700, 0.15829873, 0.15420739, 0.15019397, 0.14624439,
     &   0.14231730, 0.13849746, 0.13478395, 0.13116483, 0.12763371,
     &   0.12416931, 0.12076423, 0.11740698, 0.11410501, 0.11088280,
     &   0.10775743, 0.10475539, 0.10182589, 0.09890985, 0.09605380,
     &   0.09326269, 0.09052978, 0.08780178, 0.08512080, 0.08250186,
     &   0.07994477, 0.07743157, 0.07499962, 0.07262825, 0.07029751,
     &   0.06801128, 0.06577688, 0.06356100, 0.06141751, 0.05934443,
     &   0.05733975, 0.05540150, 0.05348876, 0.05163427, 0.04984452,
     &   0.04811725, 0.04645030, 0.04486012, 0.04332745, 0.04184749,
     &   0.04041842, 0.03903818, 0.03770475, 0.03641716, 0.03517382,
     &   0.03397249, 0.03280810, 0.03168384, 0.03059833, 0.02955112,
     &   0.02854295, 0.02756932, 0.02662903, 0.02572020, 0.02484196,
     &   0.02399380, 0.02317473, 0.02238391, 0.02162009, 0.02088232,
     &   0.02017713, 0.01949631, 0.01883842, 0.01820550, 0.01759452,
     &   0.01700399, 0.01643272, 0.01588055, 0.01534705, 0.01483212,
     &   0.01433441, 0.01385382, 0.01338962, 0.01294090, 0.01250880,
     &   0.01209109, 0.01168911, 0.01130149, 0.01092647, 0.01056337,
     &   0.01021221, 0.00987158, 0.00954226, 0.00922354, 0.00891541,
     &   0.00861717, 0.00832881, 0.00804993, 0.00778031, 0.00752004,
     &   0.00726841, 0.00702514, 0.00678989, 0.00656233, 0.00634234/
      data (gg(i,5),i=1,mxwv)/
     &   0.85563320, 0.85602611, 0.85642934, 0.85678023, 0.85712159,
     &   0.85742575, 0.85771060, 0.85802025, 0.85823357, 0.85844839,
     &   0.85864413, 0.85881537, 0.85897744, 0.85914212, 0.85925424,
     &   0.85938138, 0.85950154, 0.85955465, 0.85963023, 0.85965389,
     &   0.85981476, 0.85973406, 0.85974711, 0.85976636, 0.85962546,
     &   0.85973829, 0.85970360, 0.85966837, 0.85947204, 0.85950708,
     &   0.85943341, 0.85932559, 0.85928148, 0.85918808, 0.85898405,
     &   0.85895330, 0.85877466, 0.85865641, 0.85850585, 0.85835487,
     &   0.85828495, 0.85800630, 0.85782504, 0.85758477, 0.85744548,
     &   0.85724580, 0.85701180, 0.85674244, 0.85649019, 0.85628843,
     &   0.85602474, 0.85572892, 0.85540909, 0.85511899, 0.85485429,
     &   0.85455692, 0.85424262, 0.85385680, 0.85355061, 0.85322076,
     &   0.85286623, 0.85252750, 0.85219628, 0.85181367, 0.85141426,
     &   0.85104787, 0.85062158, 0.85028034, 0.84986031, 0.84942102,
     &   0.84902501, 0.84864128, 0.84812886, 0.84772360, 0.84726638,
     &   0.84688348, 0.84629887, 0.84599727, 0.84563702, 0.84523875,
     &   0.84477299, 0.84429729, 0.84379554, 0.84329247, 0.84273010,
     &   0.84231198, 0.84197360, 0.84180075, 0.84165174, 0.84237945,
     &   0.84462035, 0.84444076, 0.84322512, 0.84121060, 0.83970523,
     &   0.83853668, 0.83755708, 0.83663619, 0.83583832, 0.83509105,
     &   0.83446813, 0.83392000, 0.83343679, 0.83307809, 0.83244199,
     &   0.83266592, 0.83866966, 0.84606677, 0.84443051, 0.83904845,
     &   0.83410203, 0.83122104, 0.82893401, 0.82738811, 0.82629806,
     &   0.82560354, 0.82541573, 0.82568347, 0.82642299, 0.82802439,
     &   0.83057570, 0.83449143, 0.83935499, 0.84425050, 0.85305637,
     &   0.87465060, 0.91838354, 0.96582550, 0.97472960, 0.97002351,
     &   0.95938611, 0.94948053, 0.94408786, 0.94023293, 0.93829083,
     &   0.93697971, 0.93150645, 0.91420490, 0.88200998, 0.85356599,
     &   0.82767940, 0.80577749, 0.79053813, 0.78191030, 0.77732217,
     &   0.77637595, 0.77755451, 0.78014785, 0.78372443, 0.78844458,
     &   0.79331106, 0.79912007, 0.80537939, 0.81213993, 0.81915784,
     &   0.82593966, 0.83253032, 0.83757389, 0.83869624, 0.83809745,
     &   0.83748662, 0.83783835, 0.83885479, 0.84125024, 0.84561515,
     &   0.85218364, 0.86096013, 0.87158108, 0.88527256, 0.90213203,
     &   0.92095721, 0.93291497, 0.92945647, 0.91155660, 0.88634503,
     &   0.87739742, 0.87581998, 0.87618655, 0.87779272, 0.88003355,
     &   0.88217384, 0.88424939, 0.88613790, 0.88782692, 0.88940191,
     &   0.89089185, 0.89233148, 0.89370960, 0.89490968, 0.89606810,
     &   0.89715308, 0.89818305, 0.89916134, 0.90008616, 0.90092939,
     &   0.90172273, 0.90245342, 0.90310466, 0.90374744, 0.90435123,
     &   0.90489548, 0.90556073, 0.90581805, 0.90616655, 0.90579468,
     &   0.90462792, 0.90270078, 0.90000910, 0.89644581, 0.89204508,
     &   0.88715261, 0.88126206, 0.87529397, 0.86930245, 0.86368835,
     &   0.85814977, 0.85277587, 0.84756732, 0.84270507, 0.83785760,
     &   0.83319962, 0.82873565, 0.82413954, 0.81971198, 0.81527448,
     &   0.81073093, 0.80614471, 0.80152237, 0.79702759, 0.79243410,
     &   0.78781486, 0.78340095, 0.77878529, 0.77439260, 0.76977295,
     &   0.76503599, 0.76058877, 0.75605810, 0.75152385, 0.74709457,
     &   0.74290597, 0.73820293, 0.73339558, 0.72845364, 0.72338611,
     &   0.71821892, 0.71291840, 0.70751894, 0.70213932, 0.69687551,
     &   0.69150925, 0.68611801, 0.68062937, 0.67512399, 0.66956145,
     &   0.66395724, 0.65832376, 0.65260470, 0.64681178, 0.64092660,
     &   0.63498062, 0.62888819, 0.62261558, 0.61611891, 0.60932821,
     &   0.60226041, 0.59491754, 0.58727568, 0.57939565, 0.57129735,
     &   0.56296992, 0.55436850, 0.54547197, 0.53632367, 0.52699274,
     &   0.51733893, 0.50735623, 0.49724823, 0.48707953, 0.47686851,
     &   0.46647856, 0.45603278, 0.44563818, 0.43541983, 0.42525369,
     &   0.41517654, 0.40523326, 0.39545381, 0.38582683, 0.37670127,
     &   0.36782694, 0.35906076, 0.35042527, 0.34196472, 0.33372378,
     &   0.32610542, 0.31860468, 0.31126782, 0.30404913, 0.29704034,
     &   0.29017192, 0.28371057, 0.27737492, 0.27121717, 0.26512331,
     &   0.25902608, 0.25307116, 0.24726143, 0.24158759, 0.23603161,
     &   0.23058526, 0.22520241, 0.21986203, 0.21457312, 0.20936629,
     &   0.20425813, 0.19937396, 0.19459642, 0.18979658, 0.18505114,
     &   0.18038221, 0.17578372, 0.17113484, 0.16649671, 0.16191971,
     &   0.15740684, 0.15290625, 0.14851534, 0.14419943, 0.13990602,
     &   0.13564324, 0.13143528, 0.12721278, 0.12309697, 0.11908830,
     &   0.11518696, 0.11139281, 0.10760438, 0.10391353, 0.10034099,
     &   0.09688427, 0.09354073, 0.09034693, 0.08726393, 0.08428287,
     &   0.08140097, 0.07861479, 0.07592086, 0.07331795, 0.07080332,
     &   0.06837264, 0.06601496, 0.06373856, 0.06154075, 0.05942081,
     &   0.05738054, 0.05541059, 0.05350857, 0.05167064, 0.04989509,
     &   0.04818093, 0.04652613, 0.04492895, 0.04338687, 0.04189798,
     &   0.04047595, 0.03910358, 0.03777786, 0.03650309, 0.03527293,
     &   0.03408432, 0.03293475, 0.03182394, 0.03075101, 0.02971576,
     &   0.02871538, 0.02774968, 0.02681715, 0.02591596, 0.02504839,
     &   0.02420989, 0.02340322, 0.02262557, 0.02187332, 0.02114508,
     &   0.02044090, 0.01975791, 0.01909768, 0.01845881, 0.01784124,
     &   0.01724357, 0.01666576, 0.01610701, 0.01556690, 0.01504556,
     &   0.01454159, 0.01405440, 0.01358331, 0.01312769, 0.01268724/
      data (gg(i,6),i=1,mxwv)/
     &   0.85929167, 0.85976028, 0.86019212, 0.86063051, 0.86099911,
     &   0.86138177, 0.86167926, 0.86202049, 0.86232346, 0.86261982,
     &   0.86286354, 0.86309612, 0.86335456, 0.86355489, 0.86373919,
     &   0.86388850, 0.86406577, 0.86422247, 0.86436141, 0.86447895,
     &   0.86457914, 0.86467016, 0.86475188, 0.86481208, 0.86486536,
     &   0.86491764, 0.86495835, 0.86498827, 0.86501014, 0.86500251,
     &   0.86500931, 0.86499393, 0.86497670, 0.86499381, 0.86490285,
     &   0.86487079, 0.86481661, 0.86474985, 0.86467570, 0.86459881,
     &   0.86449087, 0.86442584, 0.86430335, 0.86424226, 0.86408174,
     &   0.86400896, 0.86385453, 0.86370313, 0.86353713, 0.86335838,
     &   0.86324632, 0.86308134, 0.86286002, 0.86265850, 0.86244744,
     &   0.86224681, 0.86202961, 0.86176515, 0.86159194, 0.86135387,
     &   0.86104822, 0.86084682, 0.86049867, 0.86028200, 0.86004639,
     &   0.85978341, 0.85949945, 0.85927629, 0.85903096, 0.85871029,
     &   0.85843241, 0.85805285, 0.85768116, 0.85744888, 0.85704774,
     &   0.85680598, 0.85646445, 0.85616624, 0.85603315, 0.85575974,
     &   0.85542423, 0.85509664, 0.85472423, 0.85433608, 0.85402268,
     &   0.85368657, 0.85356581, 0.85368198, 0.85381329, 0.85506266,
     &   0.85823816, 0.85838354, 0.85723448, 0.85508394, 0.85358322,
     &   0.85253185, 0.85172462, 0.85101396, 0.85048389, 0.85007709,
     &   0.84980023, 0.84967667, 0.84966195, 0.84984535, 0.84969229,
     &   0.85069638, 0.85910267, 0.86902767, 0.86779785, 0.86194444,
     &   0.85653585, 0.85367090, 0.85151732, 0.85028118, 0.84958971,
     &   0.84934813, 0.84967870, 0.85046208, 0.85163581, 0.85362011,
     &   0.85635072, 0.86013454, 0.86411071, 0.86640042, 0.87073863,
     &   0.88604736, 0.92385626, 0.97045618, 0.97967595, 0.97477221,
     &   0.96424443, 0.95446974, 0.94927704, 0.94566309, 0.94422799,
     &   0.94475108, 0.94465792, 0.93854845, 0.91979569, 0.89799803,
     &   0.87444109, 0.85206801, 0.83507347, 0.82450604, 0.81793797,
     &   0.81521076, 0.81464916, 0.81540346, 0.81696570, 0.81961769,
     &   0.82212168, 0.82553607, 0.82933891, 0.83374876, 0.83854997,
     &   0.84336340, 0.84831989, 0.85206133, 0.85110927, 0.84744126,
     &   0.84298253, 0.83916247, 0.83556783, 0.83341736, 0.83366227,
     &   0.83725107, 0.84480339, 0.85592026, 0.87289488, 0.89640242,
     &   0.92505765, 0.94439995, 0.94432628, 0.93090129, 0.90374762,
     &   0.88687646, 0.88011390, 0.87701094, 0.87668604, 0.87825525,
     &   0.88032520, 0.88276213, 0.88532305, 0.88783354, 0.89040345,
     &   0.89301199, 0.89571756, 0.89848292, 0.90108061, 0.90371174,
     &   0.90631616, 0.90890384, 0.91147548, 0.91401726, 0.91648364,
     &   0.91890728, 0.92126507, 0.92353141, 0.92579752, 0.92802411,
     &   0.93017966, 0.93250895, 0.93431711, 0.93627900, 0.93735570,
     &   0.93750960, 0.93679178, 0.93526882, 0.93280530, 0.92945629,
     &   0.92559671, 0.92066520, 0.91563153, 0.91053802, 0.90582031,
     &   0.90120852, 0.89675355, 0.89248037, 0.88857770, 0.88471848,
     &   0.88106930, 0.87766242, 0.87415129, 0.87083548, 0.86753780,
     &   0.86417490, 0.86077863, 0.85736954, 0.85411203, 0.85078931,
     &   0.84744477, 0.84432554, 0.84102315, 0.83796579, 0.83470976,
     &   0.83132714, 0.82821244, 0.82502866, 0.82187921, 0.81884527,
     &   0.81606352, 0.81287253, 0.80958760, 0.80618149, 0.80266201,
     &   0.79905111, 0.79529691, 0.79144692, 0.78759903, 0.78382635,
     &   0.77994519, 0.77615237, 0.77226698, 0.76838386, 0.76445860,
     &   0.76053238, 0.75663388, 0.75270784, 0.74880815, 0.74487662,
     &   0.74101228, 0.73711175, 0.73323888, 0.72926736, 0.72519070,
     &   0.72093689, 0.71649629, 0.71181434, 0.70692742, 0.70184612,
     &   0.69657362, 0.69108450, 0.68531299, 0.67927724, 0.67303401,
     &   0.66643900, 0.65945828, 0.65209872, 0.64450473, 0.63669407,
     &   0.62852752, 0.62004244, 0.61133218, 0.60253489, 0.59355718,
     &   0.58441556, 0.57516497, 0.56583744, 0.55635160, 0.54712659,
     &   0.53803444, 0.52893031, 0.51975524, 0.51051927, 0.50136918,
     &   0.49274170, 0.48412910, 0.47564912, 0.46718740, 0.45882064,
     &   0.45051351, 0.44260776, 0.43478283, 0.42721927, 0.41967836,
     &   0.41210115, 0.40463924, 0.39730519, 0.39011225, 0.38303441,
     &   0.37612051, 0.36927024, 0.36246416, 0.35570660, 0.34900266,
     &   0.34233505, 0.33600390, 0.32983753, 0.32365528, 0.31751642,
     &   0.31147602, 0.30554634, 0.29953808, 0.29345411, 0.28740442,
     &   0.28139395, 0.27529019, 0.26927748, 0.26332793, 0.25731283,
     &   0.25122163, 0.24510422, 0.23884074, 0.23262228, 0.22645408,
     &   0.22034155, 0.21429011, 0.20801441, 0.20177998, 0.19565679,
     &   0.18964890, 0.18375994, 0.17805597, 0.17248595, 0.16704223,
     &   0.16172631, 0.15653774, 0.15147556, 0.14654395, 0.14174297,
     &   0.13706727, 0.13249052, 0.12804706, 0.12373540, 0.11955727,
     &   0.11551898, 0.11160593, 0.10781575, 0.10414216, 0.10058396,
     &   0.09714140, 0.09381126, 0.09058962, 0.08747548, 0.08446581,
     &   0.08159129, 0.07881538, 0.07613228, 0.07355253, 0.07106236,
     &   0.06865555, 0.06632702, 0.06407673, 0.06190313, 0.05980603,
     &   0.05777977, 0.05582414, 0.05393611, 0.05211189, 0.05035639,
     &   0.04866016, 0.04702923, 0.04545765, 0.04393779, 0.04246686,
     &   0.04104489, 0.03966596, 0.03833342, 0.03704438, 0.03579873,
     &   0.03459356, 0.03342883, 0.03230290, 0.03121489, 0.03016505,
     &   0.02915052, 0.02817006, 0.02722230, 0.02630590, 0.02542028/
      data (gg(i,7),i=1,mxwv)/
     &   0.86207455, 0.86257666, 0.86306560, 0.86350971, 0.86396039,
     &   0.86435080, 0.86474526, 0.86510903, 0.86551118, 0.86579049,
     &   0.86608958, 0.86636627, 0.86664778, 0.86689991, 0.86713850,
     &   0.86731941, 0.86756980, 0.86777949, 0.86793941, 0.86810613,
     &   0.86819422, 0.86842436, 0.86856353, 0.86867672, 0.86878502,
     &   0.86889547, 0.86897022, 0.86908108, 0.86914277, 0.86920488,
     &   0.86927605, 0.86932909, 0.86936700, 0.86938471, 0.86944938,
     &   0.86941278, 0.86945337, 0.86941302, 0.86943287, 0.86941338,
     &   0.86933976, 0.86935657, 0.86931652, 0.86929446, 0.86927021,
     &   0.86918890, 0.86915433, 0.86908621, 0.86898899, 0.86890602,
     &   0.86884302, 0.86871678, 0.86861378, 0.86846226, 0.86836773,
     &   0.86817008, 0.86805546, 0.86793029, 0.86776626, 0.86766773,
     &   0.86746418, 0.86728209, 0.86713338, 0.86693603, 0.86671311,
     &   0.86655134, 0.86638767, 0.86629379, 0.86610472, 0.86590576,
     &   0.86564541, 0.86545140, 0.86521626, 0.86494672, 0.86472255,
     &   0.86452222, 0.86438060, 0.86420000, 0.86415708, 0.86398172,
     &   0.86374831, 0.86354035, 0.86327761, 0.86300689, 0.86280411,
     &   0.86264759, 0.86262721, 0.86290652, 0.86333585, 0.86512309,
     &   0.86935747, 0.86975902, 0.86852658, 0.86602479, 0.86433774,
     &   0.86320662, 0.86239874, 0.86172986, 0.86127347, 0.86094379,
     &   0.86084938, 0.86091030, 0.86112398, 0.86158848, 0.86161458,
     &   0.86311716, 0.87385577, 0.88621402, 0.88506049, 0.87830758,
     &   0.87197226, 0.86874187, 0.86639869, 0.86522096, 0.86475271,
     &   0.86488169, 0.86577117, 0.86727154, 0.86930096, 0.87238264,
     &   0.87640309, 0.88168210, 0.88704604, 0.89003420, 0.89466888,
     &   0.90973759, 0.94324309, 0.97778761, 0.98319238, 0.97774953,
     &   0.96725190, 0.95752180, 0.95237738, 0.94881213, 0.94746643,
     &   0.94839060, 0.95002520, 0.94938946, 0.94093490, 0.92652410,
     &   0.90686762, 0.88531911, 0.86768013, 0.85658240, 0.84993112,
     &   0.84778047, 0.84818071, 0.85008180, 0.85282248, 0.85667843,
     &   0.86014611, 0.86441463, 0.86884248, 0.87365776, 0.87860757,
     &   0.88333714, 0.88802540, 0.89144772, 0.89013404, 0.88581043,
     &   0.88018429, 0.87464923, 0.86862922, 0.86337954, 0.85981280,
     &   0.85916781, 0.86242914, 0.86913294, 0.88225913, 0.90376258,
     &   0.93442363, 0.95580459, 0.95670068, 0.94861346, 0.93219966,
     &   0.91596824, 0.90607238, 0.89903319, 0.89481127, 0.89303768,
     &   0.89222622, 0.89216334, 0.89266610, 0.89337748, 0.89446342,
     &   0.89584047, 0.89765006, 0.89986634, 0.90212381, 0.90468973,
     &   0.90748316, 0.91048831, 0.91370845, 0.91710025, 0.92058939,
     &   0.92417336, 0.92779744, 0.93140864, 0.93508708, 0.93878102,
     &   0.94240910, 0.94629371, 0.94951808, 0.95296246, 0.95527506,
     &   0.95647794, 0.95663744, 0.95592147, 0.95418257, 0.95150518,
     &   0.94828665, 0.94394696, 0.93948275, 0.93493557, 0.93076235,
     &   0.92671180, 0.92281651, 0.91911542, 0.91580313, 0.91255444,
     &   0.90953428, 0.90678710, 0.90395832, 0.90134603, 0.89877629,
     &   0.89616847, 0.89354527, 0.89093292, 0.88849837, 0.88602483,
     &   0.88355142, 0.88133001, 0.87894893, 0.87683713, 0.87454921,
     &   0.87215495, 0.87004328, 0.86787903, 0.86576110, 0.86376363,
     &   0.86201274, 0.85988253, 0.85766900, 0.85534459, 0.85292184,
     &   0.85041523, 0.84777260, 0.84504122, 0.84230274, 0.83961022,
     &   0.83680511, 0.83411443, 0.83133417, 0.82855523, 0.82573348,
     &   0.82292247, 0.82015675, 0.81739283, 0.81470668, 0.81202734,
     &   0.80948502, 0.80698735, 0.80465901, 0.80234158, 0.80007792,
     &   0.79774535, 0.79532844, 0.79276079, 0.79006183, 0.78723145,
     &   0.78428817, 0.78123093, 0.77798194, 0.77454501, 0.77096391,
     &   0.76713169, 0.76301253, 0.75847739, 0.75370234, 0.74870586,
     &   0.74338508, 0.73770672, 0.73172987, 0.72556484, 0.71916372,
     &   0.71251845, 0.70567226, 0.69864374, 0.69130498, 0.68400335,
     &   0.67674839, 0.66943181, 0.66193014, 0.65420926, 0.64645523,
     &   0.63897115, 0.63141298, 0.62392348, 0.61635256, 0.60872155,
     &   0.60103858, 0.59356993, 0.58608156, 0.57881510, 0.57149494,
     &   0.56410319, 0.55671930, 0.54935801, 0.54205120, 0.53477561,
     &   0.52762944, 0.52050149, 0.51339209, 0.50630122, 0.49919593,
     &   0.49201944, 0.48512930, 0.47839844, 0.47171226, 0.46506217,
     &   0.45852342, 0.45214090, 0.44578296, 0.43934736, 0.43296462,
     &   0.42664650, 0.42022955, 0.41389546, 0.40767094, 0.40139902,
     &   0.39503446, 0.38862991, 0.38208720, 0.37552065, 0.36892974,
     &   0.36231411, 0.35567373, 0.34850025, 0.34121543, 0.33392268,
     &   0.32663003, 0.31934234, 0.31208676, 0.30485946, 0.29766282,
     &   0.29050201, 0.28337955, 0.27629736, 0.26926753, 0.26229560,
     &   0.25537178, 0.24842130, 0.24155413, 0.23477581, 0.22809340,
     &   0.22151759, 0.21504791, 0.20868857, 0.20243350, 0.19628976,
     &   0.19026835, 0.18436581, 0.17856580, 0.17290026, 0.16737048,
     &   0.16204309, 0.15685491, 0.15180029, 0.14691252, 0.14216325,
     &   0.13754274, 0.13304298, 0.12866995, 0.12442411, 0.12030791,
     &   0.11631385, 0.11244454, 0.10869617, 0.10506305, 0.10155616,
     &   0.09815933, 0.09488908, 0.09173303, 0.08867542, 0.08571182,
     &   0.08284263, 0.08005548, 0.07735942, 0.07474899, 0.07222472,
     &   0.06978065, 0.06741765, 0.06513274, 0.06292444, 0.06079339,
     &   0.05873407, 0.05674412, 0.05482080, 0.05296148, 0.05116508/
      data (gg(i,8),i=1,mxwv)/
     &   0.86415082, 0.86470020, 0.86520845, 0.86570001, 0.86615551,
     &   0.86660403, 0.86702698, 0.86743528, 0.86780620, 0.86817056,
     &   0.86851180, 0.86885196, 0.86913478, 0.86943799, 0.86969519,
     &   0.86997896, 0.87023062, 0.87046015, 0.87068135, 0.87089401,
     &   0.87108797, 0.87127811, 0.87145698, 0.87161267, 0.87176722,
     &   0.87193519, 0.87204623, 0.87219024, 0.87225848, 0.87240922,
     &   0.87252927, 0.87259823, 0.87266862, 0.87276524, 0.87283915,
     &   0.87290239, 0.87297350, 0.87300140, 0.87305409, 0.87307775,
     &   0.87313449, 0.87316227, 0.87318403, 0.87321877, 0.87321573,
     &   0.87319076, 0.87319630, 0.87319952, 0.87315768, 0.87312305,
     &   0.87314165, 0.87303305, 0.87302113, 0.87290806, 0.87287748,
     &   0.87279671, 0.87272930, 0.87261528, 0.87251043, 0.87245822,
     &   0.87238336, 0.87225145, 0.87218541, 0.87204075, 0.87192309,
     &   0.87183547, 0.87176627, 0.87169826, 0.87160873, 0.87149715,
     &   0.87133116, 0.87116259, 0.87097770, 0.87081748, 0.87061965,
     &   0.87055415, 0.87041134, 0.87036735, 0.87042910, 0.87042522,
     &   0.87029284, 0.87014300, 0.86996472, 0.86978400, 0.86965275,
     &   0.86963290, 0.86974692, 0.87031490, 0.87094545, 0.87338996,
     &   0.87894279, 0.87968528, 0.87821656, 0.87520415, 0.87320584,
     &   0.87192732, 0.87104112, 0.87033731, 0.86993456, 0.86965406,
     &   0.86965293, 0.86988896, 0.87027019, 0.87099916, 0.87116754,
     &   0.87315321, 0.88664716, 0.90159076, 0.90042353, 0.89252526,
     &   0.88491464, 0.88100398, 0.87815440, 0.87674141, 0.87620276,
     &   0.87640339, 0.87753987, 0.87942922, 0.88195461, 0.88576460,
     &   0.89069593, 0.89711684, 0.90359700, 0.90726560, 0.91293132,
     &   0.93018579, 0.96187919, 0.98405242, 0.98544466, 0.97960538,
     &   0.96913284, 0.95942426, 0.95429432, 0.95074743, 0.94943005,
     &   0.95044482, 0.95253944, 0.95404065, 0.95199859, 0.94462174,
     &   0.93044513, 0.91118264, 0.89316422, 0.88099957, 0.87352097,
     &   0.87124997, 0.87206668, 0.87473559, 0.87844884, 0.88347143,
     &   0.88804162, 0.89346004, 0.89895397, 0.90471143, 0.91041011,
     &   0.91565698, 0.92062378, 0.92419791, 0.92344326, 0.91996497,
     &   0.91519511, 0.91035396, 0.90483624, 0.89981353, 0.89617991,
     &   0.89517766, 0.89772904, 0.90304500, 0.91356725, 0.93036991,
     &   0.95338988, 0.96704364, 0.96518725, 0.95902288, 0.95322114,
     &   0.94489378, 0.93828285, 0.93264616, 0.92850924, 0.92599154,
     &   0.92407984, 0.92263377, 0.92159110, 0.92061025, 0.91986811,
     &   0.91927475, 0.91902769, 0.91916168, 0.91933924, 0.91984361,
     &   0.92065424, 0.92178738, 0.92331278, 0.92523396, 0.92754012,
     &   0.93022317, 0.93325114, 0.93658423, 0.94027334, 0.94429559,
     &   0.94850278, 0.95326775, 0.95749336, 0.96211427, 0.96542346,
     &   0.96744597, 0.96822953, 0.96805865, 0.96679002, 0.96453911,
     &   0.96171957, 0.95775354, 0.95364600, 0.94944060, 0.94560605,
     &   0.94190180, 0.93835312, 0.93500626, 0.93205947, 0.92918909,
     &   0.92656058, 0.92422372, 0.92181820, 0.91964871, 0.91753769,
     &   0.91540474, 0.91327417, 0.91117370, 0.90927404, 0.90735394,
     &   0.90545923, 0.90384471, 0.90209353, 0.90064007, 0.89902645,
     &   0.89733958, 0.89598036, 0.89459163, 0.89326131, 0.89207298,
     &   0.89114714, 0.88981378, 0.88841277, 0.88691640, 0.88533175,
     &   0.88367867, 0.88191020, 0.88006604, 0.87822866, 0.87644827,
     &   0.87455797, 0.87274027, 0.87083000, 0.86890316, 0.86692011,
     &   0.86492109, 0.86293536, 0.86092854, 0.85897261, 0.85700941,
     &   0.85516244, 0.85336882, 0.85177433, 0.85024595, 0.84885985,
     &   0.84749174, 0.84612721, 0.84470499, 0.84322476, 0.84167957,
     &   0.84009296, 0.83847815, 0.83675838, 0.83492464, 0.83300519,
     &   0.83092761, 0.82865560, 0.82599431, 0.82312179, 0.82005829,
     &   0.81673300, 0.81307983, 0.80913806, 0.80499786, 0.80064309,
     &   0.79605663, 0.79127485, 0.78631002, 0.78102297, 0.77569026,
     &   0.77039492, 0.76506007, 0.75953996, 0.75377899, 0.74796700,
     &   0.74228400, 0.73652250, 0.73081684, 0.72501558, 0.71909958,
     &   0.71310055, 0.70718139, 0.70119941, 0.69538325, 0.68948323,
     &   0.68351406, 0.67748243, 0.67139417, 0.66528070, 0.65912098,
     &   0.65301645, 0.64687574, 0.64071226, 0.63452029, 0.62824333,
     &   0.62180895, 0.61547744, 0.60920388, 0.60297668, 0.59673238,
     &   0.59052736, 0.58441383, 0.57838833, 0.57230306, 0.56624472,
     &   0.56022912, 0.55416441, 0.54812407, 0.54218316, 0.53626698,
     &   0.53033215, 0.52441192, 0.51852584, 0.51261747, 0.50668722,
     &   0.50073504, 0.49476111, 0.48838347, 0.48187745, 0.47532281,
     &   0.46872017, 0.46207014, 0.45529485, 0.44847125, 0.44161275,
     &   0.43472052, 0.42779359, 0.42083013, 0.41383624, 0.40681353,
     &   0.39974120, 0.39249662, 0.38520989, 0.37793794, 0.37062272,
     &   0.36326882, 0.35590875, 0.34854671, 0.34117097, 0.33379012,
     &   0.32642135, 0.31904757, 0.31160146, 0.30418992, 0.29681903,
     &   0.28956163, 0.28236568, 0.27523109, 0.26822975, 0.26131544,
     &   0.25447536, 0.24769610, 0.24099973, 0.23439258, 0.22788079,
     &   0.22146688, 0.21516205, 0.20896754, 0.20288108, 0.19691488,
     &   0.19106369, 0.18537574, 0.17983045, 0.17440125, 0.16908896,
     &   0.16389233, 0.15878820, 0.15380514, 0.14893770, 0.14419208,
     &   0.13955739, 0.13504395, 0.13065058, 0.12637830, 0.12222937,
     &   0.11819944, 0.11428691, 0.11048917, 0.10680356, 0.10322997/
      data (gg(i,9),i=1,mxwv)/
     &   0.86570579, 0.86627316, 0.86681956, 0.86733145, 0.86779755,
     &   0.86828339, 0.86873442, 0.86917073, 0.86959422, 0.86997551,
     &   0.87033057, 0.87068945, 0.87102950, 0.87134981, 0.87164986,
     &   0.87193936, 0.87222141, 0.87248987, 0.87274337, 0.87298322,
     &   0.87321627, 0.87342876, 0.87364298, 0.87382525, 0.87402141,
     &   0.87419945, 0.87437236, 0.87452382, 0.87468487, 0.87482899,
     &   0.87496865, 0.87510151, 0.87523818, 0.87533218, 0.87542295,
     &   0.87555480, 0.87566102, 0.87573594, 0.87580919, 0.87588662,
     &   0.87596458, 0.87602592, 0.87608939, 0.87615889, 0.87621909,
     &   0.87626052, 0.87630016, 0.87633586, 0.87633747, 0.87636369,
     &   0.87635893, 0.87637556, 0.87636262, 0.87635684, 0.87632316,
     &   0.87631983, 0.87628138, 0.87622112, 0.87620938, 0.87614995,
     &   0.87611711, 0.87609148, 0.87602305, 0.87599266, 0.87593222,
     &   0.87587076, 0.87583029, 0.87590492, 0.87587899, 0.87580025,
     &   0.87569672, 0.87557459, 0.87539381, 0.87531763, 0.87524295,
     &   0.87516278, 0.87510377, 0.87515438, 0.87537348, 0.87545365,
     &   0.87538868, 0.87531090, 0.87518978, 0.87507123, 0.87499201,
     &   0.87508547, 0.87531817, 0.87619644, 0.87712485, 0.88037568,
     &   0.88756454, 0.88851970, 0.88688254, 0.88316619, 0.88072366,
     &   0.87918729, 0.87817532, 0.87737513, 0.87693816, 0.87667322,
     &   0.87677860, 0.87715858, 0.87776077, 0.87876087, 0.87905347,
     &   0.88162667, 0.89844346, 0.91603237, 0.91482747, 0.90577930,
     &   0.89671189, 0.89194459, 0.88841474, 0.88663977, 0.88593906,
     &   0.88613766, 0.88749206, 0.88974845, 0.89274335, 0.89725423,
     &   0.90302020, 0.91041005, 0.91766083, 0.92147869, 0.92741328,
     &   0.94565761, 0.97404289, 0.98733282, 0.98675025, 0.98076940,
     &   0.97032666, 0.96062624, 0.95549929, 0.95196092, 0.95065993,
     &   0.95170647, 0.95391148, 0.95605910, 0.95706600, 0.95487237,
     &   0.94684064, 0.93207133, 0.91532111, 0.90266180, 0.89436305,
     &   0.89169329, 0.89253563, 0.89550281, 0.89964533, 0.90519583,
     &   0.91019273, 0.91599697, 0.92174923, 0.92760092, 0.93319523,
     &   0.93815994, 0.94265550, 0.94579136, 0.94539768, 0.94282341,
     &   0.93912029, 0.93525386, 0.93067020, 0.92643458, 0.92348337,
     &   0.92311096, 0.92619044, 0.93172222, 0.94134641, 0.95450735,
     &   0.96872693, 0.97408485, 0.96965992, 0.96367317, 0.96263880,
     &   0.96023405, 0.95755613, 0.95488662, 0.95277768, 0.95147562,
     &   0.95044219, 0.94961959, 0.94897753, 0.94829738, 0.94769430,
     &   0.94709843, 0.94664198, 0.94635075, 0.94598436, 0.94573158,
     &   0.94558370, 0.94554502, 0.94568688, 0.94602907, 0.94659823,
     &   0.94744015, 0.94854909, 0.94997758, 0.95179760, 0.95411015,
     &   0.95686358, 0.96048772, 0.96415257, 0.96859968, 0.97204489,
     &   0.97432017, 0.97534108, 0.97540921, 0.97437757, 0.97236228,
     &   0.96977526, 0.96604151, 0.96215922, 0.95817345, 0.95455498,
     &   0.95106894, 0.94773823, 0.94461238, 0.94189209, 0.93925470,
     &   0.93686682, 0.93478042, 0.93263322, 0.93073279, 0.92890084,
     &   0.92705578, 0.92522466, 0.92343563, 0.92186278, 0.92028093,
     &   0.91874194, 0.91750497, 0.91614407, 0.91510576, 0.91391778,
     &   0.91268128, 0.91181505, 0.91094255, 0.91014570, 0.90952241,
     &   0.90920025, 0.90843731, 0.90762824, 0.90674227, 0.90578622,
     &   0.90478575, 0.90369719, 0.90255398, 0.90146142, 0.90048724,
     &   0.89943242, 0.89842397, 0.89734226, 0.89626122, 0.89513749,
     &   0.89399552, 0.89285290, 0.89166564, 0.89048034, 0.88925511,
     &   0.88807750, 0.88688982, 0.88579792, 0.88471991, 0.88372427,
     &   0.88274103, 0.88177294, 0.88078195, 0.87977344, 0.87874413,
     &   0.87771624, 0.87670648, 0.87565374, 0.87454766, 0.87340820,
     &   0.87218004, 0.87082809, 0.86915582, 0.86731839, 0.86533248,
     &   0.86313832, 0.86066127, 0.85793573, 0.85503560, 0.85195470,
     &   0.84867424, 0.84522712, 0.84162420, 0.83772933, 0.83379042,
     &   0.82990581, 0.82601041, 0.82195979, 0.81769669, 0.81340295,
     &   0.80921412, 0.80497015, 0.80079460, 0.79654425, 0.79218918,
     &   0.78776568, 0.78338808, 0.77895194, 0.77465773, 0.77028584,
     &   0.76586223, 0.76136482, 0.75679362, 0.75217336, 0.74748415,
     &   0.74281126, 0.73808306, 0.73331618, 0.72850215, 0.72358161,
     &   0.71848762, 0.71337032, 0.70822829, 0.70311564, 0.69794226,
     &   0.69273740, 0.68753779, 0.68242276, 0.67725766, 0.67207170,
     &   0.66687483, 0.66165996, 0.65639079, 0.65114713, 0.64594370,
     &   0.64076197, 0.63560587, 0.63059086, 0.62553954, 0.62045336,
     &   0.61533380, 0.61018234, 0.60496104, 0.59967589, 0.59433323,
     &   0.58893341, 0.58347678, 0.57783246, 0.57211447, 0.56634384,
     &   0.56052184, 0.55464941, 0.54872745, 0.54275668, 0.53673822,
     &   0.53065979, 0.52444249, 0.51817000, 0.51184350, 0.50545025,
     &   0.49895629, 0.49241287, 0.48582074, 0.47917122, 0.47246668,
     &   0.46571642, 0.45889390, 0.45190099, 0.44486344, 0.43778354,
     &   0.43065661, 0.42350060, 0.41631943, 0.40917939, 0.40204087,
     &   0.39489061, 0.38770762, 0.38051581, 0.37331700, 0.36610496,
     &   0.35889679, 0.35170040, 0.34451881, 0.33735266, 0.33017612,
     &   0.32302499, 0.31597349, 0.30899784, 0.30206871, 0.29520088,
     &   0.28837287, 0.28154647, 0.27477068, 0.26804280, 0.26137316,
     &   0.25473723, 0.24816571, 0.24166402, 0.23523824, 0.22888158,
     &   0.22261043, 0.21642868, 0.21033935, 0.20434508, 0.19845000/
      data (gg(i,10),i=1,mxwv)/
     &   0.86686021, 0.86744881, 0.86799252, 0.86853611, 0.86895543,
     &   0.86953557, 0.87000090, 0.87045169, 0.87083346, 0.87130159,
     &   0.87169057, 0.87206268, 0.87244010, 0.87277144, 0.87309670,
     &   0.87341088, 0.87371850, 0.87400341, 0.87433618, 0.87454772,
     &   0.87478763, 0.87503749, 0.87527174, 0.87549752, 0.87571692,
     &   0.87591606, 0.87611336, 0.87630445, 0.87646973, 0.87664890,
     &   0.87680864, 0.87697446, 0.87712395, 0.87727755, 0.87740767,
     &   0.87753981, 0.87766117, 0.87779158, 0.87790227, 0.87802625,
     &   0.87811464, 0.87821335, 0.87831587, 0.87840378, 0.87847877,
     &   0.87856746, 0.87864316, 0.87869686, 0.87875730, 0.87880206,
     &   0.87884343, 0.87888986, 0.87892693, 0.87895787, 0.87896967,
     &   0.87899965, 0.87898052, 0.87894756, 0.87899673, 0.87901521,
     &   0.87899166, 0.87898177, 0.87898558, 0.87897056, 0.87895143,
     &   0.87895536, 0.87902099, 0.87911314, 0.87916017, 0.87910670,
     &   0.87905478, 0.87896180, 0.87888515, 0.87879819, 0.87875277,
     &   0.87874502, 0.87876129, 0.87886900, 0.87927407, 0.87939793,
     &   0.87940049, 0.87935853, 0.87928092, 0.87917894, 0.87921232,
     &   0.87938583, 0.87977445, 0.88099146, 0.88227248, 0.88656443,
     &   0.89577186, 0.89704549, 0.89507163, 0.89045733, 0.88740164,
     &   0.88548803, 0.88425052, 0.88326609, 0.88282037, 0.88249201,
     &   0.88268197, 0.88320935, 0.88403571, 0.88537598, 0.88576812,
     &   0.88906181, 0.90978378, 0.92973238, 0.92852092, 0.91851294,
     &   0.90793192, 0.90216047, 0.89778507, 0.89552546, 0.89457971,
     &   0.89474243, 0.89632291, 0.89899188, 0.90251720, 0.90781522,
     &   0.91447639, 0.92282015, 0.93070608, 0.93453318, 0.94044930,
     &   0.95837849, 0.98088449, 0.98866546, 0.98750383, 0.98151189,
     &   0.97109073, 0.96138501, 0.95625317, 0.95271593, 0.95142490,
     &   0.95249093, 0.95473087, 0.95703691, 0.95912594, 0.95972371,
     &   0.95663917, 0.94748896, 0.93396294, 0.92200840, 0.91344202,
     &   0.91050118, 0.91131347, 0.91441357, 0.91874492, 0.92439854,
     &   0.92937386, 0.93496805, 0.94029838, 0.94547522, 0.95017070,
     &   0.95411140, 0.95747131, 0.95968848, 0.95947570, 0.95779967,
     &   0.95526469, 0.95248771, 0.94899625, 0.94561923, 0.94323462,
     &   0.94311613, 0.94609022, 0.95112145, 0.95909560, 0.96844149,
     &   0.97612560, 0.97728866, 0.97189605, 0.96588910, 0.96636462,
     &   0.96668011, 0.96625668, 0.96554589, 0.96492714, 0.96460730,
     &   0.96437061, 0.96422642, 0.96413827, 0.96402669, 0.96393830,
     &   0.96385384, 0.96383405, 0.96388811, 0.96388161, 0.96391714,
     &   0.96398133, 0.96407139, 0.96422249, 0.96442050, 0.96470135,
     &   0.96505171, 0.96548605, 0.96602982, 0.96672130, 0.96765304,
     &   0.96882635, 0.97057033, 0.97255701, 0.97531992, 0.97769487,
     &   0.97937530, 0.98013508, 0.98010373, 0.97909510, 0.97716653,
     &   0.97469872, 0.97110492, 0.96736628, 0.96352291, 0.96004313,
     &   0.95669538, 0.95350122, 0.95051217, 0.94792974, 0.94543296,
     &   0.94318873, 0.94125086, 0.93925613, 0.93751335, 0.93584430,
     &   0.93416697, 0.93250906, 0.93089908, 0.92951322, 0.92812437,
     &   0.92678648, 0.92576188, 0.92462009, 0.92381424, 0.92286450,
     &   0.92187721, 0.92128247, 0.92069417, 0.92019409, 0.91988826,
     &   0.91991156, 0.91948193, 0.91902107, 0.91849655, 0.91791737,
     &   0.91730845, 0.91663092, 0.91592097, 0.91529202, 0.91483450,
     &   0.91432649, 0.91387600, 0.91338164, 0.91292650, 0.91246617,
     &   0.91202241, 0.91161251, 0.91118222, 0.91076481, 0.91032660,
     &   0.90993643, 0.90951794, 0.90912014, 0.90867430, 0.90820146,
     &   0.90767443, 0.90709656, 0.90644801, 0.90576094, 0.90504396,
     &   0.90431547, 0.90357941, 0.90278995, 0.90195560, 0.90111178,
     &   0.90018398, 0.89914638, 0.89785725, 0.89645648, 0.89495516,
     &   0.89327663, 0.89136374, 0.88924867, 0.88701135, 0.88462818,
     &   0.88208151, 0.87940365, 0.87660491, 0.87355155, 0.87050384,
     &   0.86753935, 0.86458558, 0.86150503, 0.85824525, 0.85498452,
     &   0.85187787, 0.84874016, 0.84569210, 0.84259349, 0.83942139,
     &   0.83620602, 0.83307183, 0.82990098, 0.82688332, 0.82380325,
     &   0.82068020, 0.81750131, 0.81426382, 0.81099129, 0.80765885,
     &   0.80434060, 0.80096751, 0.79755175, 0.79408419, 0.79051226,
     &   0.78677911, 0.78299886, 0.77916563, 0.77532864, 0.77140749,
     &   0.76741654, 0.76337540, 0.75936723, 0.75529140, 0.75115168,
     &   0.74695116, 0.74272257, 0.73838937, 0.73400855, 0.72963804,
     &   0.72528166, 0.72092873, 0.71672416, 0.71247023, 0.70816988,
     &   0.70382553, 0.69943988, 0.69525683, 0.69108391, 0.68686813,
     &   0.68260872, 0.67830497, 0.67383534, 0.66929805, 0.66471082,
     &   0.66006964, 0.65538663, 0.65065324, 0.64586812, 0.64103103,
     &   0.63614351, 0.63120854, 0.62621135, 0.62115180, 0.61601764,
     &   0.61077696, 0.60547191, 0.60010302, 0.59467161, 0.58917648,
     &   0.58361703, 0.57797801, 0.57219887, 0.56634778, 0.56042576,
     &   0.55435091, 0.54820609, 0.54200029, 0.53574818, 0.52945143,
     &   0.52310598, 0.51670229, 0.51025212, 0.50375324, 0.49719042,
     &   0.49058831, 0.48394558, 0.47726658, 0.47055566, 0.46375072,
     &   0.45691743, 0.45010355, 0.44330046, 0.43650109, 0.42973113,
     &   0.42295146, 0.41613054, 0.40930209, 0.40246725, 0.39562780,
     &   0.38874793, 0.38186124, 0.37497419, 0.36808857, 0.36116758,
     &   0.35425279, 0.34734738, 0.34045577, 0.33358216, 0.32672521/
      data (gg(i,11),i=1,mxwv)/
     &   0.86769879, 0.86830479, 0.86887866, 0.86941910, 0.86995268,
     &   0.87045628, 0.87093663, 0.87141192, 0.87185776, 0.87228376,
     &   0.87268829, 0.87308472, 0.87345773, 0.87382299, 0.87425536,
     &   0.87450135, 0.87481707, 0.87512726, 0.87541640, 0.87570256,
     &   0.87597644, 0.87623835, 0.87648392, 0.87671274, 0.87696451,
     &   0.87718719, 0.87741238, 0.87760931, 0.87781465, 0.87800986,
     &   0.87819755, 0.87837833, 0.87855524, 0.87872010, 0.87889177,
     &   0.87902391, 0.87918544, 0.87932426, 0.87946743, 0.87959832,
     &   0.87972724, 0.87984902, 0.87997448, 0.88007826, 0.88019824,
     &   0.88029712, 0.88040406, 0.88048887, 0.88057297, 0.88065052,
     &   0.88072395, 0.88078338, 0.88085365, 0.88091004, 0.88097358,
     &   0.88100523, 0.88103610, 0.88106620, 0.88109136, 0.88113493,
     &   0.88116443, 0.88117802, 0.88122392, 0.88124335, 0.88126785,
     &   0.88131064, 0.88143629, 0.88160843, 0.88170332, 0.88170034,
     &   0.88165879, 0.88158000, 0.88150090, 0.88144118, 0.88143736,
     &   0.88146198, 0.88153738, 0.88174969, 0.88216257, 0.88254052,
     &   0.88259017, 0.88258922, 0.88254511, 0.88248152, 0.88253087,
     &   0.88282365, 0.88337606, 0.88502938, 0.88673556, 0.89237070,
     &   0.90404981, 0.90566897, 0.90328348, 0.89754307, 0.89368612,
     &   0.89125937, 0.88969344, 0.88857144, 0.88782907, 0.88749391,
     &   0.88775873, 0.88845557, 0.88953441, 0.89125228, 0.89176673,
     &   0.89594769, 0.92105860, 0.94258136, 0.94139826, 0.93092847,
     &   0.91901135, 0.91216093, 0.90679568, 0.90392965, 0.90265220,
     &   0.90273219, 0.90455574, 0.90768677, 0.91179812, 0.91794336,
     &   0.92549652, 0.93465507, 0.94290626, 0.94661087, 0.95227814,
     &   0.96858722, 0.98432010, 0.98923337, 0.98796481, 0.98199207,
     &   0.97157800, 0.96185619, 0.95671403, 0.95317286, 0.95188731,
     &   0.95296842, 0.95522892, 0.95757639, 0.95996046, 0.96166778,
     &   0.96148670, 0.95733601, 0.94833308, 0.93849123, 0.93051738,
     &   0.92757487, 0.92833453, 0.93136650, 0.93552524, 0.94083095,
     &   0.94531018, 0.95011455, 0.95444673, 0.95839572, 0.96173847,
     &   0.96434933, 0.96641308, 0.96766859, 0.96757686, 0.96670902,
     &   0.96532935, 0.96372485, 0.96155018, 0.95930046, 0.95767146,
     &   0.95772099, 0.96012956, 0.96397567, 0.96953607, 0.97522020,
     &   0.97902775, 0.97873878, 0.97315747, 0.96715951, 0.96802419,
     &   0.96931630, 0.96988648, 0.97014832, 0.97031194, 0.97050172,
     &   0.97067177, 0.97085249, 0.97103441, 0.97119737, 0.97136986,
     &   0.97155011, 0.97176301, 0.97201234, 0.97224039, 0.97249955,
     &   0.97278112, 0.97308803, 0.97342920, 0.97380525, 0.97421300,
     &   0.97465765, 0.97513765, 0.97565758, 0.97623014, 0.97689450,
     &   0.97763813, 0.97862279, 0.97965258, 0.98107171, 0.98230588,
     &   0.98315412, 0.98346281, 0.98317885, 0.98208773, 0.98016989,
     &   0.97776127, 0.97425348, 0.97060800, 0.96685737, 0.96346617,
     &   0.96020532, 0.95709550, 0.95418948, 0.95168948, 0.94927609,
     &   0.94711614, 0.94526505, 0.94335914, 0.94170755, 0.94013232,
     &   0.93855172, 0.93699270, 0.93548453, 0.93420422, 0.93292421,
     &   0.93169850, 0.93079114, 0.92976981, 0.92909050, 0.92827064,
     &   0.92741728, 0.92696416, 0.92652309, 0.92617661, 0.92603266,
     &   0.92622972, 0.92597336, 0.92569226, 0.92535400, 0.92496747,
     &   0.92455876, 0.92408890, 0.92359430, 0.92319554, 0.92298812,
     &   0.92274362, 0.92256969, 0.92236710, 0.92222512, 0.92209977,
     &   0.92201525, 0.92199337, 0.92197663, 0.92199928, 0.92203152,
     &   0.92214644, 0.92228144, 0.92238599, 0.92247021, 0.92249405,
     &   0.92243588, 0.92230439, 0.92206103, 0.92175967, 0.92141414,
     &   0.92103702, 0.92061985, 0.92010897, 0.91953057, 0.91894144,
     &   0.91822892, 0.91737491, 0.91628599, 0.91511405, 0.91386843,
     &   0.91244704, 0.91081208, 0.90900815, 0.90712595, 0.90512133,
     &   0.90297800, 0.90073067, 0.89838773, 0.89581561, 0.89330280,
     &   0.89090312, 0.88853091, 0.88604951, 0.88340968, 0.88084328,
     &   0.87839675, 0.87598664, 0.87369108, 0.87136465, 0.86899370,
     &   0.86660266, 0.86434829, 0.86208344, 0.86000967, 0.85789454,
     &   0.85574782, 0.85357600, 0.85137928, 0.84917855, 0.84695011,
     &   0.84477043, 0.84255904, 0.84032351, 0.83805454, 0.83570200,
     &   0.83320785, 0.83073115, 0.82823753, 0.82573926, 0.82316840,
     &   0.82054639, 0.81789088, 0.81523770, 0.81248903, 0.80966538,
     &   0.80676520, 0.80378997, 0.80069959, 0.79753119, 0.79430979,
     &   0.79103601, 0.78770423, 0.78441918, 0.78103977, 0.77756798,
     &   0.77400637, 0.77035755, 0.76690704, 0.76346880, 0.75998688,
     &   0.75646311, 0.75289893, 0.74922132, 0.74549109, 0.74172008,
     &   0.73790908, 0.73406154, 0.73018110, 0.72626269, 0.72230655,
     &   0.71832496, 0.71438324, 0.71040094, 0.70637739, 0.70230538,
     &   0.69816750, 0.69398296, 0.68975061, 0.68547952, 0.68116546,
     &   0.67679948, 0.67238116, 0.66790652, 0.66336769, 0.65876353,
     &   0.65398246, 0.64912528, 0.64420074, 0.63917285, 0.63407224,
     &   0.62890851, 0.62368530, 0.61839801, 0.61304212, 0.60760248,
     &   0.60209876, 0.59652245, 0.59087747, 0.58517408, 0.57934386,
     &   0.57345390, 0.56749666, 0.56148392, 0.55543697, 0.54937977,
     &   0.54328197, 0.53714406, 0.53096622, 0.52475727, 0.51851255,
     &   0.51221269, 0.50587606, 0.49950999, 0.49311405, 0.48664403,
     &   0.48014596, 0.47362301, 0.46708125, 0.46052697, 0.45395344/
      data (gg(i,12),i=1,mxwv)/
     &   0.86831170, 0.86892611, 0.86950588, 0.87006736, 0.87061650,
     &   0.87112576, 0.87161839, 0.87210166, 0.87257916, 0.87300217,
     &   0.87342572, 0.87382627, 0.87422889, 0.87459022, 0.87494785,
     &   0.87529498, 0.87562650, 0.87594783, 0.87625712, 0.87655067,
     &   0.87683904, 0.87711751, 0.87738538, 0.87764758, 0.87789375,
     &   0.87813270, 0.87836349, 0.87859350, 0.87880582, 0.87901694,
     &   0.87921977, 0.87941945, 0.87959969, 0.87979174, 0.87996984,
     &   0.88013351, 0.88031197, 0.88047129, 0.88062680, 0.88077426,
     &   0.88092113, 0.88106740, 0.88121545, 0.88134259, 0.88147074,
     &   0.88160163, 0.88170147, 0.88179803, 0.88192976, 0.88202780,
     &   0.88212055, 0.88221151, 0.88230157, 0.88238519, 0.88245958,
     &   0.88251948, 0.88257474, 0.88263494, 0.88268197, 0.88274676,
     &   0.88280630, 0.88286680, 0.88291723, 0.88296872, 0.88302237,
     &   0.88309300, 0.88327658, 0.88354516, 0.88369679, 0.88370341,
     &   0.88367778, 0.88361090, 0.88352108, 0.88346839, 0.88348472,
     &   0.88355798, 0.88367873, 0.88399142, 0.88477212, 0.88510680,
     &   0.88520294, 0.88522756, 0.88519591, 0.88513058, 0.88582873,
     &   0.88569438, 0.88639212, 0.88861156, 0.89090234, 0.89821237,
     &   0.91279000, 0.91476500, 0.91191447, 0.90486270, 0.90000987,
     &   0.89689684, 0.89489949, 0.89328557, 0.89248276, 0.89204133,
     &   0.89237589, 0.89326817, 0.89465004, 0.89682239, 0.89747024,
     &   0.90274251, 0.93240190, 0.95398837, 0.95294517, 0.94287133,
     &   0.93035108, 0.92231518, 0.91589338, 0.91232193, 0.91063148,
     &   0.91058540, 0.91265982, 0.91629642, 0.92102641, 0.92801011,
     &   0.93632239, 0.94594473, 0.95408165, 0.95748490, 0.96259135,
     &   0.97595477, 0.98579675, 0.98953760, 0.98826092, 0.98230350,
     &   0.97188365, 0.96213871, 0.95698321, 0.95343494, 0.95215195,
     &   0.95324516, 0.95552206, 0.95788860, 0.96034259, 0.96238190,
     &   0.96345633, 0.96254766, 0.95800221, 0.95132494, 0.94493496,
     &   0.94237447, 0.94307411, 0.94578785, 0.94940567, 0.95381033,
     &   0.95734525, 0.96090704, 0.96390760, 0.96644437, 0.96843421,
     &   0.96987444, 0.97092569, 0.97149986, 0.97147447, 0.97115546,
     &   0.97062767, 0.96997440, 0.96898466, 0.96787870, 0.96708417,
     &   0.96733034, 0.96907258, 0.97164625, 0.97502488, 0.97819084,
     &   0.98021048, 0.97953385, 0.97394341, 0.96795470, 0.96890521,
     &   0.97047478, 0.97139144, 0.97203684, 0.97253901, 0.97295654,
     &   0.97330928, 0.97363591, 0.97393155, 0.97421092, 0.97448736,
     &   0.97477245, 0.97507119, 0.97538418, 0.97569376, 0.97602737,
     &   0.97637981, 0.97675818, 0.97716480, 0.97760242, 0.97806376,
     &   0.97856259, 0.97909743, 0.97966540, 0.98029608, 0.98098314,
     &   0.98172832, 0.98261595, 0.98342681, 0.98439896, 0.98514861,
     &   0.98555487, 0.98559064, 0.98513323, 0.98397642, 0.98206002,
     &   0.97968984, 0.97624022, 0.97265702, 0.96896666, 0.96563119,
     &   0.96242404, 0.95936501, 0.95650822, 0.95405662, 0.95169181,
     &   0.94958073, 0.94778019, 0.94592583, 0.94432724, 0.94280648,
     &   0.94128221, 0.93978047, 0.93833113, 0.93711191, 0.93589497,
     &   0.93473387, 0.93389410, 0.93294197, 0.93233526, 0.93158978,
     &   0.93081254, 0.93043905, 0.93008006, 0.92981893, 0.92976379,
     &   0.93005407, 0.92989242, 0.92970878, 0.92947048, 0.92918658,
     &   0.92888361, 0.92852235, 0.92813957, 0.92785692, 0.92777091,
     &   0.92765224, 0.92760998, 0.92754418, 0.92754591, 0.92757118,
     &   0.92764586, 0.92779255, 0.92795444, 0.92816734, 0.92840248,
     &   0.92873639, 0.92907625, 0.92945611, 0.92979825, 0.93008447,
     &   0.93029219, 0.93042833, 0.93044806, 0.93041074, 0.93033087,
     &   0.93021911, 0.93006051, 0.92979568, 0.92945373, 0.92910099,
     &   0.92860514, 0.92794603, 0.92704523, 0.92606854, 0.92502654,
     &   0.92380172, 0.92236531, 0.92077053, 0.91911727, 0.91735113,
     &   0.91545671, 0.91347104, 0.91140532, 0.90911734, 0.90691692,
     &   0.90484828, 0.90281838, 0.90068787, 0.89840829, 0.89616877,
     &   0.89418095, 0.89219296, 0.89033717, 0.88846296, 0.88656044,
     &   0.88465244, 0.88291430, 0.88118339, 0.87967390, 0.87813967,
     &   0.87658715, 0.87503219, 0.87347674, 0.87194699, 0.87041092,
     &   0.86896044, 0.86750668, 0.86605018, 0.86458904, 0.86306953,
     &   0.86142761, 0.85986912, 0.85834521, 0.85684979, 0.85531831,
     &   0.85378611, 0.85228217, 0.85081601, 0.84927106, 0.84769505,
     &   0.84609044, 0.84441513, 0.84267652, 0.84092027, 0.83912432,
     &   0.83726996, 0.83536339, 0.83346951, 0.83148152, 0.82939583,
     &   0.82720900, 0.82491755, 0.82259738, 0.82022142, 0.81777608,
     &   0.81526119, 0.81267691, 0.80996311, 0.80717444, 0.80432075,
     &   0.80140257, 0.79842228, 0.79538316, 0.79228258, 0.78912175,
     &   0.78591174, 0.78271145, 0.77946240, 0.77616566, 0.77282053,
     &   0.76942307, 0.76598138, 0.76249629, 0.75897950, 0.75542951,
     &   0.75183886, 0.74821746, 0.74459738, 0.74093604, 0.73723286,
     &   0.73339868, 0.72951096, 0.72557640, 0.72153991, 0.71744168,
     &   0.71329522, 0.70911330, 0.70488280, 0.70059961, 0.69625455,
     &   0.69185656, 0.68739510, 0.68287206, 0.67829376, 0.67362326,
     &   0.66889244, 0.66405725, 0.65913820, 0.65416813, 0.64915788,
     &   0.64409477, 0.63900304, 0.63385499, 0.62866044, 0.62341017,
     &   0.61810613, 0.61274499, 0.60733068, 0.60186023, 0.59630424,
     &   0.59069055, 0.58502078, 0.57929963, 0.57353246, 0.56771332/
      data (gg(i,13),i=1,mxwv)/
     &   0.86874950, 0.86937523, 0.86996394, 0.87053841, 0.87108672,
     &   0.87160975, 0.87211424, 0.87260741, 0.87307900, 0.87352484,
     &   0.87395179, 0.87436712, 0.87476403, 0.87514901, 0.87551814,
     &   0.87583959, 0.87621832, 0.87654752, 0.87686646, 0.87718022,
     &   0.87747312, 0.87776649, 0.87804168, 0.87830687, 0.87856972,
     &   0.87882382, 0.87906879, 0.87929326, 0.87953258, 0.87975377,
     &   0.87996948, 0.88017637, 0.88038152, 0.88057828, 0.88076061,
     &   0.88095659, 0.88113701, 0.88130826, 0.88147444, 0.88164258,
     &   0.88178986, 0.88196921, 0.88212043, 0.88227338, 0.88241756,
     &   0.88255817, 0.88268816, 0.88281339, 0.88293791, 0.88305181,
     &   0.88315976, 0.88326931, 0.88337559, 0.88348174, 0.88357466,
     &   0.88365263, 0.88372821, 0.88379669, 0.88387007, 0.88394415,
     &   0.88403374, 0.88411933, 0.88418859, 0.88428080, 0.88434279,
     &   0.88445109, 0.88470691, 0.88507169, 0.88528919, 0.88531703,
     &   0.88527870, 0.88518900, 0.88509685, 0.88503420, 0.88506156,
     &   0.88517106, 0.88534236, 0.88576943, 0.88687420, 0.88729548,
     &   0.88743240, 0.88747567, 0.88744146, 0.88736647, 0.88749766,
     &   0.88804781, 0.88907462, 0.89202386, 0.89513886, 0.90447342,
     &   0.92223132, 0.92451614, 0.92123073, 0.91278315, 0.90676606,
     &   0.90284044, 0.90024561, 0.89813787, 0.89707673, 0.89647794,
     &   0.89688146, 0.89800590, 0.89975536, 0.90253252, 0.90327531,
     &   0.90980607, 0.94359696, 0.96316302, 0.96236646, 0.95378017,
     &   0.94128442, 0.93276876, 0.92537850, 0.92106515, 0.91890836,
     &   0.91870040, 0.92103314, 0.92517585, 0.93048990, 0.93817258,
     &   0.94690663, 0.95637083, 0.96372402, 0.96663129, 0.97086459,
     &   0.98050284, 0.98635459, 0.98972684, 0.98845464, 0.98250371,
     &   0.97206926, 0.96229768, 0.95712745, 0.95357060, 0.95228815,
     &   0.95339119, 0.95568192, 0.95806253, 0.96053821, 0.96265823,
     &   0.96413678, 0.96477824, 0.96341497, 0.96000099, 0.95585865,
     &   0.95408416, 0.95473015, 0.95690602, 0.95966935, 0.96282035,
     &   0.96521020, 0.96744937, 0.96921140, 0.97059846, 0.97162348,
     &   0.97232646, 0.97280991, 0.97304076, 0.97304714, 0.97300011,
     &   0.97293079, 0.97284108, 0.97263026, 0.97235894, 0.97224039,
     &   0.97266018, 0.97385889, 0.97547901, 0.97746164, 0.97937697,
     &   0.98078090, 0.98002952, 0.97444737, 0.96845937, 0.96943128,
     &   0.97106755, 0.97208017, 0.97284484, 0.97346163, 0.97396207,
     &   0.97438413, 0.97476941, 0.97511011, 0.97543699, 0.97575569,
     &   0.97608328, 0.97641665, 0.97675514, 0.97709650, 0.97745770,
     &   0.97783530, 0.97823769, 0.97866422, 0.97911817, 0.97959244,
     &   0.98010278, 0.98064935, 0.98122925, 0.98187536, 0.98257619,
     &   0.98334038, 0.98423564, 0.98503250, 0.98593891, 0.98659509,
     &   0.98688847, 0.98684496, 0.98633164, 0.98515832, 0.98325598,
     &   0.98091769, 0.97750992, 0.97396809, 0.97031564, 0.96701366,
     &   0.96383786, 0.96080774, 0.95797849, 0.95555395, 0.95321625,
     &   0.95113254, 0.94936079, 0.94753569, 0.94596756, 0.94447821,
     &   0.94298655, 0.94151789, 0.94010264, 0.93891925, 0.93773943,
     &   0.93661636, 0.93581706, 0.93490613, 0.93434328, 0.93364257,
     &   0.93291098, 0.93258572, 0.93227649, 0.93206710, 0.93206584,
     &   0.93241268, 0.93230790, 0.93218237, 0.93200350, 0.93178052,
     &   0.93153977, 0.93124205, 0.93092436, 0.93070871, 0.93069190,
     &   0.93064398, 0.93067515, 0.93068463, 0.93076396, 0.93086922,
     &   0.93102646, 0.93125862, 0.93150884, 0.93181348, 0.93214464,
     &   0.93257880, 0.93302339, 0.93351418, 0.93397123, 0.93437696,
     &   0.93470854, 0.93497258, 0.93512315, 0.93522054, 0.93527919,
     &   0.93530971, 0.93529636, 0.93517762, 0.93498313, 0.93478125,
     &   0.93443388, 0.93392032, 0.93316239, 0.93233097, 0.93143696,
     &   0.93035889, 0.92906940, 0.92762434, 0.92612636, 0.92451924,
     &   0.92278761, 0.92096949, 0.91907638, 0.91696388, 0.91494834,
     &   0.91307294, 0.91124249, 0.90931600, 0.90724456, 0.90522027,
     &   0.90346152, 0.90170640, 0.90009391, 0.89846987, 0.89682484,
     &   0.89518207, 0.89372188, 0.89227760, 0.89106870, 0.88984412,
     &   0.88861006, 0.88738424, 0.88616949, 0.88499063, 0.88382292,
     &   0.88275450, 0.88169515, 0.88065046, 0.87961489, 0.87853497,
     &   0.87734401, 0.87626350, 0.87524211, 0.87427205, 0.87328798,
     &   0.87233150, 0.87143826, 0.87061542, 0.86973715, 0.86886311,
     &   0.86800015, 0.86709261, 0.86616564, 0.86528027, 0.86439383,
     &   0.86349124, 0.86257839, 0.86172563, 0.86082596, 0.85987681,
     &   0.85887539, 0.85781848, 0.85663241, 0.85538512, 0.85409141,
     &   0.85274929, 0.85135686, 0.84982216, 0.84822315, 0.84657174,
     &   0.84486634, 0.84310573, 0.84128928, 0.83941376, 0.83747768,
     &   0.83547878, 0.83341616, 0.83129555, 0.82911557, 0.82687014,
     &   0.82454526, 0.82216167, 0.81971902, 0.81722116, 0.81466764,
     &   0.81205553, 0.80938566, 0.80666482, 0.80389136, 0.80106574,
     &   0.79812801, 0.79513252, 0.79208440, 0.78894991, 0.78575200,
     &   0.78249961, 0.77920234, 0.77585262, 0.77244878, 0.76898670,
     &   0.76547277, 0.76190025, 0.75827098, 0.75459015, 0.75084156,
     &   0.74704069, 0.74314433, 0.73916888, 0.73514116, 0.73106354,
     &   0.72693551, 0.72278655, 0.71858615, 0.71434247, 0.71004838,
     &   0.70571768, 0.70133740, 0.69690901, 0.69242942, 0.68788868,
     &   0.68329453, 0.67864645, 0.67394644, 0.66919690, 0.66439301/

c     ice data

      data (qqi(i,1),i=1,mxwv)/
     &   2.17622876, 2.17840743, 2.18082023, 2.18302011, 2.18548632,
     &   2.18786287, 2.19010639, 2.19279814, 2.19516540, 2.19764400,
     &   2.20041800, 2.20275927, 2.20552278, 2.20830655, 2.21089935,
     &   2.21359396, 2.21642351, 2.21922898, 2.22207069, 2.22500086,
     &   2.22788191, 2.23084331, 2.23384261, 2.23687792, 2.23992443,
     &   2.24301553, 2.24619794, 2.24927902, 2.25247335, 2.25576949,
     &   2.25899458, 2.26225591, 2.26553059, 2.26884747, 2.27218223,
     &   2.27553463, 2.27889204, 2.28227782, 2.28567743, 2.28909802,
     &   2.29251885, 2.29597735, 2.29945612, 2.30296636, 2.30651546,
     &   2.31011558, 2.31378007, 2.31752086, 2.32136011, 2.32531667,
     &   2.32941508, 2.33368397, 2.33815837, 2.34287238, 2.34786654,
     &   2.35318375, 2.35891867, 2.36498785, 2.37157607, 2.37870026,
     &   2.38642001, 2.39478970, 2.40387583, 2.41373992, 2.42443871,
     &   2.43602967, 2.44856882, 2.46211982, 2.47670054, 2.49238348,
     &   2.50916147, 2.52708030, 2.54614997, 2.56637406, 2.58772302,
     &   2.61017585, 2.63372922, 2.65826130, 2.68377376, 2.71013165,
     &   2.73726797, 2.76503778, 2.79333735, 2.82198977, 2.85090923,
     &   2.87991333, 2.90880156, 2.93736267, 2.96550608, 2.99295211,
     &   3.01942420, 3.04412246, 3.06594467, 3.08665228, 3.10705876,
     &   3.12583137, 3.14284706, 3.15725636, 3.16872931, 3.17751122,
     &   3.18325520, 3.18570590, 3.18458247, 3.17972922, 3.17122531,
     &   3.15818191, 3.13956642, 3.11378121, 3.08274722, 3.05257297,
     &   3.02228522, 2.98808670, 2.94926620, 2.90207314, 2.84588504,
     &   2.78239703, 2.71123385, 2.63061213, 2.54344487, 2.44561505,
     &   2.33848596, 2.21706152, 2.07900643, 1.91886818, 1.72695100,
     &   1.48603010, 1.21580958, 0.91254288, 0.65315205, 0.60195059,
     &   1.00365186, 1.58387733, 1.88239682, 2.20290422, 2.48098278,
     &   2.62484097, 2.80629730, 2.98096180, 3.11367393, 3.18251824,
     &   3.15866184, 3.06346154, 2.93697429, 2.79611111, 2.66531825,
     &   2.53107524, 2.40736461, 2.29285240, 2.18345046, 2.08598804,
     &   1.99160314, 1.90376842, 1.81958508, 1.74686456, 1.69509971,
     &   1.66831744, 1.65986133, 1.61267698, 1.54809415, 1.46908534,
     &   1.39147377, 1.31403470, 1.24162161, 1.17426014, 1.10881054,
     &   1.04882538, 0.99551463, 0.95016962, 0.91752762, 0.89655596,
     &   0.91106629, 0.95512557, 0.98618406, 1.00193572, 0.99133372,
     &   0.96269393, 0.91042566, 0.87280458, 0.85396403, 0.83193302,
     &   0.80359888, 0.77466649, 0.74798989, 0.71396124, 0.68591940,
     &   0.66213685, 0.63786584, 0.60576016, 0.56587964, 0.51968181,
     &   0.47602180, 0.45039597, 0.42647868, 0.40444461, 0.38474351,
     &   0.36861840, 0.35014328, 0.32921728, 0.30804271, 0.28535312,
     &   0.26718712, 0.26637501, 0.28706035, 0.33275387, 0.43039846,
     &   0.57374614, 0.72907251, 0.85454166, 0.95909679, 1.03006446,
     &   1.08678842, 1.12298036, 1.14409387, 1.15106499, 1.13177979,
     &   1.10395014, 1.06566918, 1.00922060, 0.95248640, 0.89178675,
     &   0.82101321, 0.74446529, 0.66645521, 0.59811872, 0.53704667,
     &   0.48804408, 0.44788849, 0.41426891, 0.38678095, 0.36748898,
     &   0.34543756, 0.31804702, 0.28799936, 0.25950566, 0.23379852,
     &   0.21510667, 0.19971426, 0.19236670, 0.18734805, 0.17555737,
     &   0.15685396, 0.13550365, 0.11661259, 0.10335232, 0.09162591,
     &   0.08123308, 0.07280353, 0.06721842, 0.06320556, 0.06017902,
     &   0.05795182, 0.05598997, 0.05456356, 0.05396695, 0.05378709,
     &   0.05430345, 0.05572163, 0.05745927, 0.05951934, 0.06190668,
     &   0.06458828, 0.06724051, 0.07020132, 0.07347767, 0.07707818,
     &   0.08124653, 0.08608044, 0.09136911, 0.09713465, 0.10508864,
     &   0.11696254, 0.13124792, 0.14863731, 0.16540682, 0.18176426,
     &   0.19979778, 0.21965837, 0.23817964, 0.25584555, 0.27318257,
     &   0.29061607, 0.30880630, 0.32765067, 0.35039961, 0.37857530,
     &   0.39115745, 0.40933737, 0.42001513, 0.38951984, 0.33109453,
     &   0.28626463, 0.26747465, 0.25387740, 0.24094407, 0.22599810,
     &   0.21020570, 0.19229940, 0.16412415, 0.14013043, 0.11989243,
     &   0.12261753, 0.13981597, 0.16021460, 0.17959534, 0.17506924,
     &   0.17012131, 0.16349205, 0.15453941, 0.14214496, 0.12838894,
     &   0.11683186, 0.10673799, 0.09783781, 0.08968746, 0.08241411,
     &   0.07573187, 0.06907845, 0.06251855, 0.05657453, 0.05141868,
     &   0.04678006, 0.04380222, 0.04141493, 0.03917862, 0.03728823,
     &   0.03549027, 0.03383607, 0.03229109, 0.03081769, 0.02958884,
     &   0.02841510, 0.02732820, 0.02647163, 0.02564286, 0.02484093,
     &   0.02406491, 0.02331390, 0.02258705, 0.02188353, 0.02120255,
     &   0.02054335, 0.01990519, 0.01921475, 0.01853583, 0.01788126,
     &   0.01725013, 0.01664159, 0.01605481, 0.01548898, 0.01494335,
     &   0.01441717, 0.01390973, 0.01342036, 0.01294840, 0.01247364,
     &   0.01194608, 0.01144096, 0.01095730, 0.01049418, 0.01005072,
     &   0.00959127, 0.00893763, 0.00832855, 0.00776099, 0.00721005,
     &   0.00668365, 0.00619571, 0.00572624, 0.00525644, 0.00482520,
     &   0.00443282, 0.00411935, 0.00382806, 0.00355738, 0.00330431,
     &   0.00306892, 0.00285031, 0.00264949, 0.00246466, 0.00229273,
     &   0.00213532, 0.00199453, 0.00186304, 0.00174044, 0.00163781,
     &   0.00154123, 0.00145036, 0.00137313, 0.00130258, 0.00123566,
     &   0.00118145, 0.00113826, 0.00109665, 0.00105834, 0.00102656,
     &   0.00099573, 0.00096584, 0.00093824, 0.00091145, 0.00088544/
      data (qqi(i,2),i=1,mxwv)/
     &   2.13844347, 2.14020967, 2.14191651, 2.14379859, 2.14552712,
     &   2.14745545, 2.14914727, 2.15127063, 2.15299201, 2.15473437,
     &   2.15690827, 2.15877390, 2.16086197, 2.16280413, 2.16478133,
     &   2.16730762, 2.16917205, 2.17120576, 2.17324471, 2.17546558,
     &   2.17773080, 2.18014693, 2.18241286, 2.18474793, 2.18714905,
     &   2.18934178, 2.19198346, 2.19438457, 2.19686413, 2.19929576,
     &   2.20192432, 2.20472288, 2.20711827, 2.20986152, 2.21259117,
     &   2.21536827, 2.21810532, 2.22090411, 2.22387075, 2.22673106,
     &   2.22968006, 2.23255682, 2.23556828, 2.23860598, 2.24170113,
     &   2.24479318, 2.24790931, 2.25106692, 2.25423717, 2.25744915,
     &   2.26070762, 2.26395273, 2.26723504, 2.27052283, 2.27385092,
     &   2.27716708, 2.28051353, 2.28388762, 2.28726268, 2.29066586,
     &   2.29408693, 2.29754019, 2.30102706, 2.30455637, 2.30814004,
     &   2.31178784, 2.31551933, 2.31935453, 2.32331324, 2.32742500,
     &   2.33171844, 2.33622861, 2.34099531, 2.34606194, 2.35147524,
     &   2.35728669, 2.36355591, 2.37033868, 2.37770367, 2.38571095,
     &   2.39443612, 2.40393996, 2.41430068, 2.42557383, 2.43785000,
     &   2.45119357, 2.46566033, 2.48129749, 2.49821568, 2.51643324,
     &   2.53610277, 2.55700517, 2.57854009, 2.60104275, 2.62498665,
     &   2.65020180, 2.67701030, 2.70498109, 2.73373914, 2.76343942,
     &   2.79398084, 2.82515192, 2.85669684, 2.88839674, 2.91969538,
     &   2.95110106, 2.98144031, 3.00848579, 3.03000975, 3.05139709,
     &   3.07364488, 3.09542751, 3.11624742, 3.13357115, 3.14477921,
     &   3.14942765, 3.14683795, 3.13508582, 3.11346745, 3.08003068,
     &   3.03372025, 2.96922565, 2.88140988, 2.76193190, 2.59412169,
     &   2.34119678, 2.00220299, 1.54631793, 1.05867374, 0.82390863,
     &   1.22847044, 1.77170849, 2.01606393, 2.25810289, 2.45560288,
     &   2.54650140, 2.64603257, 2.75744605, 2.89142537, 3.03634715,
     &   3.16649795, 3.25008917, 3.29144359, 3.29079938, 3.26440048,
     &   3.22071433, 3.16458178, 3.09872317, 3.02458405, 2.95260167,
     &   2.87637329, 2.79716229, 2.71426439, 2.63025618, 2.55903172,
     &   2.51861954, 2.50597167, 2.49389577, 2.45457292, 2.37898636,
     &   2.29970217, 2.21191835, 2.12133861, 2.03167748, 1.94079149,
     &   1.84890532, 1.76070380, 1.67803979, 1.60632670, 1.54543018,
     &   1.52200961, 1.55972993, 1.59004354, 1.61774421, 1.62015855,
     &   1.59965158, 1.54210770, 1.48926651, 1.45810449, 1.42946732,
     &   1.39335346, 1.35423195, 1.31725109, 1.27142775, 1.22802567,
     &   1.19125080, 1.15622306, 1.11251223, 1.05680764, 0.98802310,
     &   0.91524905, 0.86232048, 0.81475025, 0.76977462, 0.72773045,
     &   0.69077808, 0.65147018, 0.60738575, 0.56211621, 0.51264364,
     &   0.46627894, 0.44185257, 0.44811985, 0.48908699, 0.60091811,
     &   0.77474719, 0.96311653, 1.11815608, 1.24877810, 1.34379911,
     &   1.42493224, 1.48635495, 1.53401840, 1.56883049, 1.57682109,
     &   1.57609081, 1.56286883, 1.52647507, 1.48473012, 1.43583250,
     &   1.37027097, 1.28872204, 1.19588041, 1.10519755, 1.01775694,
     &   0.94071412, 0.87438101, 0.81669694, 0.76722527, 0.73137200,
     &   0.69463980, 0.65020460, 0.59880012, 0.54828155, 0.50159025,
     &   0.46389210, 0.43090943, 0.41124824, 0.39788011, 0.37629962,
     &   0.34386116, 0.30570680, 0.26976365, 0.24215673, 0.21733588,
     &   0.19500025, 0.17597774, 0.16157694, 0.15015827, 0.14074734,
     &   0.13299713, 0.12598202, 0.12005414, 0.11564625, 0.11213247,
     &   0.10991973, 0.10929409, 0.10934937, 0.11007236, 0.11145350,
     &   0.11341944, 0.11541645, 0.11801048, 0.12119722, 0.12497555,
     &   0.12957683, 0.13514075, 0.14147094, 0.14858384, 0.15887335,
     &   0.17477009, 0.19426578, 0.21832667, 0.24159069, 0.26432636,
     &   0.28951985, 0.31739646, 0.34351948, 0.36854199, 0.39320028,
     &   0.41807386, 0.44410795, 0.47118396, 0.50413126, 0.54586518,
     &   0.56629974, 0.59570009, 0.61682498, 0.58111680, 0.49998391,
     &   0.43374577, 0.40518713, 0.38458696, 0.36533761, 0.34326765,
     &   0.31987113, 0.29343894, 0.25123379, 0.21432634, 0.18262807,
     &   0.18443300, 0.20829123, 0.23743929, 0.26605448, 0.26007557,
     &   0.25316370, 0.24369745, 0.23076627, 0.21266459, 0.19230510,
     &   0.17505014, 0.15990594, 0.14649780, 0.13422830, 0.12325247,
     &   0.11318128, 0.10316494, 0.09330244, 0.08437853, 0.07660814,
     &   0.06962012, 0.06507754, 0.06142005, 0.05800861, 0.05512039,
     &   0.05238067, 0.04986439, 0.04751896, 0.04528759, 0.04342024,
     &   0.04164085, 0.03999520, 0.03869200, 0.03743440, 0.03622060,
     &   0.03504890, 0.03391764, 0.03282528, 0.03177032, 0.03075133,
     &   0.02976696, 0.02881590, 0.02779409, 0.02679164, 0.02582652,
     &   0.02489725, 0.02400242, 0.02314070, 0.02231080, 0.02151148,
     &   0.02074157, 0.01999992, 0.01928545, 0.01859713, 0.01790612,
     &   0.01714122, 0.01640937, 0.01570912, 0.01503908, 0.01439792,
     &   0.01373478, 0.01279589, 0.01192130, 0.01110659, 0.01031613,
     &   0.00956121, 0.00886162, 0.00818886, 0.00751612, 0.00689872,
     &   0.00633703, 0.00588786, 0.00547059, 0.00508293, 0.00472060,
     &   0.00438367, 0.00407082, 0.00378348, 0.00351903, 0.00327310,
     &   0.00304795, 0.00284657, 0.00265851, 0.00248320, 0.00233635,
     &   0.00219820, 0.00206824, 0.00195772, 0.00185676, 0.00176102,
     &   0.00168337, 0.00162140, 0.00156174, 0.00150681, 0.00146117,
     &   0.00141694, 0.00137406, 0.00133448, 0.00129609, 0.00125882/
      data (qqi(i,3),i=1,mxwv)/
     &   2.10912704, 2.11047482, 2.11185670, 2.11333418, 2.11451697,
     &   2.11596370, 2.11755061, 2.11892009, 2.12046695, 2.12174582,
     &   2.12333250, 2.12470937, 2.12653637, 2.12807012, 2.12947583,
     &   2.13122869, 2.13274050, 2.13453484, 2.13634610, 2.13780355,
     &   2.13976932, 2.14120197, 2.14308929, 2.14486170, 2.14653039,
     &   2.14841819, 2.15024185, 2.15211272, 2.15407586, 2.15600681,
     &   2.15803742, 2.15996170, 2.16192961, 2.16433668, 2.16607332,
     &   2.16826630, 2.17059374, 2.17236471, 2.17468643, 2.17677665,
     &   2.17910814, 2.18134856, 2.18357873, 2.18591547, 2.18841672,
     &   2.19085431, 2.19314718, 2.19590282, 2.19823122, 2.20077944,
     &   2.20335174, 2.20599461, 2.20862412, 2.21133494, 2.21392989,
     &   2.21685743, 2.21964192, 2.22243190, 2.22529125, 2.22820115,
     &   2.23110461, 2.23414183, 2.23713183, 2.24016404, 2.24328041,
     &   2.24637675, 2.24951816, 2.25266671, 2.25586319, 2.25907397,
     &   2.26230240, 2.26554990, 2.26884675, 2.27213287, 2.27544141,
     &   2.27876210, 2.28209996, 2.28545332, 2.28883600, 2.29224110,
     &   2.29567170, 2.29914474, 2.30266285, 2.30624413, 2.30990386,
     &   2.31365919, 2.31753778, 2.32156610, 2.32578325, 2.33022785,
     &   2.33495879, 2.34002662, 2.34543991, 2.35125256, 2.35753846,
     &   2.36438370, 2.37193894, 2.38027859, 2.38945651, 2.39960217,
     &   2.41085887, 2.42334628, 2.43719149, 2.45252323, 2.46930289,
     &   2.48811483, 2.50900865, 2.53171468, 2.55479455, 2.57894468,
     &   2.60502243, 2.63386321, 2.66580629, 2.70135212, 2.73960590,
     &   2.77976322, 2.82174850, 2.86532068, 2.90805674, 2.95089626,
     &   2.99170876, 3.02866173, 3.05728602, 3.07104445, 3.05439281,
     &   2.96271729, 2.74242926, 2.29272008, 1.60469317, 1.09595001,
     &   1.45002508, 1.91166735, 2.09488654, 2.26609492, 2.39857769,
     &   2.45013642, 2.49635768, 2.53859949, 2.58617806, 2.64452219,
     &   2.72089672, 2.80699158, 2.89493060, 2.97395849, 3.03641391,
     &   3.09296346, 3.13498378, 3.16245985, 3.17771864, 3.18501472,
     &   3.18402195, 3.17172670, 3.14919043, 3.11012578, 3.06345654,
     &   3.03310537, 3.02376533, 3.06278372, 3.08195829, 3.06136203,
     &   3.03242683, 2.98750710, 2.92848277, 2.86144638, 2.78682327,
     &   2.69871545, 2.60421920, 2.50503850, 2.40495729, 2.30628633,
     &   2.22859573, 2.23733568, 2.25205088, 2.28566027, 2.30720615,
     &   2.30918622, 2.27060366, 2.21733618, 2.17954206, 2.15280604,
     &   2.12011790, 2.08144569, 2.04344058, 1.99806333, 1.94681633,
     &   1.90303886, 1.86425710, 1.81940436, 1.76005924, 1.67999816,
     &   1.58302963, 1.49788773, 1.42233217, 1.34870613, 1.27698743,
     &   1.21047115, 1.14201140, 1.06471801, 0.98346233, 0.89214396,
     &   0.79739177, 0.72496331, 0.69346803, 0.70915234, 0.82013714,
     &   1.01435554, 1.22508085, 1.40000474, 1.54653978, 1.65771794,
     &   1.75563025, 1.83641243, 1.90621173, 1.96637034, 2.00497699,
     &   2.03762293, 2.06005812, 2.06227136, 2.05674267, 2.04502392,
     &   2.01686335, 1.96677935, 1.89656687, 1.81441295, 1.72551215,
     &   1.63601422, 1.55328846, 1.47726953, 1.40784883, 1.35561085,
     &   1.30780327, 1.25126922, 1.18015587, 1.10612345, 1.03474343,
     &   0.96968007, 0.90869385, 0.86627674, 0.83786100, 0.80334443,
     &   0.75340998, 0.69105023, 0.62666255, 0.57183754, 0.52115196,
     &   0.47438848, 0.43252265, 0.39751506, 0.36805269, 0.34262568,
     &   0.32063994, 0.30045673, 0.28247720, 0.26732805, 0.25411877,
     &   0.24343042, 0.23564772, 0.22937135, 0.22454426, 0.22111660,
     &   0.21892329, 0.21695390, 0.21622698, 0.21670479, 0.21835579,
     &   0.22116962, 0.22535719, 0.23096238, 0.23797186, 0.24959111,
     &   0.26914024, 0.29423958, 0.32616884, 0.35726491, 0.38780433,
     &   0.42196566, 0.46006322, 0.49605936, 0.53077066, 0.56518644,
     &   0.60004812, 0.63667256, 0.67494959, 0.72198164, 0.78339040,
     &   0.81756455, 0.86632842, 0.90959901, 0.88131291, 0.77864099,
     &   0.68342340, 0.64017057, 0.60912377, 0.58099228, 0.54904801,
     &   0.51478410, 0.47616655, 0.41193762, 0.35218906, 0.29862487,
     &   0.29401717, 0.32526189, 0.36646384, 0.40989909, 0.40327445,
     &   0.39427873, 0.38118854, 0.36272904, 0.33611083, 0.30517074,
     &   0.27838334, 0.25456762, 0.23324765, 0.21372707, 0.19613935,
     &   0.18001536, 0.16398746, 0.14822228, 0.13398001, 0.12145641,
     &   0.11018909, 0.10266139, 0.09654187, 0.09088080, 0.08606996,
     &   0.08152871, 0.07737080, 0.07350947, 0.06985270, 0.06676956,
     &   0.06384503, 0.06114662, 0.05898772, 0.05691514, 0.05492483,
     &   0.05301298, 0.05117594, 0.04941029, 0.04771279, 0.04608036,
     &   0.04451010, 0.04299925, 0.04140033, 0.03983944, 0.03834114,
     &   0.03690268, 0.03552146, 0.03419498, 0.03292089, 0.03169692,
     &   0.03052095, 0.02939092, 0.02830490, 0.02726102, 0.02621774,
     &   0.02507306, 0.02397957, 0.02293489, 0.02193677, 0.02098306,
     &   0.02000059, 0.01862537, 0.01734515, 0.01615330, 0.01499815,
     &   0.01389590, 0.01287497, 0.01189426, 0.01091518, 0.01001695,
     &   0.00919992, 0.00854486, 0.00793663, 0.00737185, 0.00684429,
     &   0.00635394, 0.00589884, 0.00548092, 0.00509638, 0.00473890,
     &   0.00441166, 0.00411884, 0.00384552, 0.00359084, 0.00337710,
     &   0.00317614, 0.00298719, 0.00282622, 0.00267916, 0.00253981,
     &   0.00242643, 0.00233556, 0.00224819, 0.00216773, 0.00210067,
     &   0.00203577, 0.00197295, 0.00191496, 0.00185878, 0.00180432/
      data (qqi(i,4),i=1,mxwv)/
     &   2.08637619, 2.08733582, 2.08842611, 2.08950615, 2.09065223,
     &   2.09175014, 2.09284091, 2.09383965, 2.09489131, 2.09612989,
     &   2.09734106, 2.09851289, 2.09978771, 2.10105252, 2.10210276,
     &   2.10354662, 2.10474396, 2.10591626, 2.10727596, 2.10860682,
     &   2.10981297, 2.11134982, 2.11269855, 2.11378336, 2.11535501,
     &   2.11676455, 2.11821985, 2.11963844, 2.12145305, 2.12273550,
     &   2.12408733, 2.12577462, 2.12733221, 2.12870502, 2.13030434,
     &   2.13195348, 2.13385749, 2.13525105, 2.13705873, 2.13860583,
     &   2.14035511, 2.14207268, 2.14374876, 2.14577222, 2.14760113,
     &   2.14936996, 2.15115857, 2.15301728, 2.15507650, 2.15708542,
     &   2.15883398, 2.16083527, 2.16298389, 2.16497898, 2.16705346,
     &   2.16919971, 2.17127991, 2.17358375, 2.17574239, 2.17792177,
     &   2.18029904, 2.18236995, 2.18482637, 2.18719554, 2.18978119,
     &   2.19213462, 2.19447112, 2.19699121, 2.19958258, 2.20226550,
     &   2.20458674, 2.20733809, 2.21005058, 2.21274400, 2.21545863,
     &   2.21828365, 2.22108722, 2.22395611, 2.22686195, 2.22977543,
     &   2.23276234, 2.23575830, 2.23881364, 2.24188256, 2.24497175,
     &   2.24810505, 2.25126338, 2.25438356, 2.25763535, 2.26086402,
     &   2.26410127, 2.26732469, 2.27051520, 2.27375221, 2.27706599,
     &   2.28040600, 2.28377676, 2.28716207, 2.29056692, 2.29401731,
     &   2.29752469, 2.30110812, 2.30479288, 2.30861521, 2.31261134,
     &   2.31685972, 2.32143807, 2.32644844, 2.33190513, 2.33783674,
     &   2.34434175, 2.35168314, 2.36008286, 2.37006068, 2.38202214,
     &   2.39624858, 2.41332245, 2.43416071, 2.45882320, 2.48909879,
     &   2.52564621, 2.57095504, 2.62703514, 2.69707489, 2.78407216,
     &   2.88072062, 2.93037391, 2.78529954, 2.17223597, 1.39844012,
     &   1.64626133, 2.00189137, 2.13090229, 2.24829459, 2.33679891,
     &   2.36693525, 2.39179063, 2.41252327, 2.43121004, 2.44788837,
     &   2.46446753, 2.48372364, 2.50807405, 2.53809047, 2.57077360,
     &   2.60942554, 2.64961314, 2.68995166, 2.73056865, 2.76914001,
     &   2.80732846, 2.84171224, 2.87246895, 2.89160395, 2.89631581,
     &   2.89615154, 2.89789367, 2.94628763, 2.99557209, 3.03389764,
     &   3.06906438, 3.09596634, 3.11041546, 3.11476994, 3.11056185,
     &   3.08813906, 3.05047083, 2.99647570, 2.92428279, 2.83698368,
     &   2.73502922, 2.70526671, 2.69538260, 2.71974421, 2.75239015,
     &   2.77927518, 2.78344941, 2.75785112, 2.73153639, 2.72107124,
     &   2.70971274, 2.69159055, 2.67173743, 2.65033150, 2.61419344,
     &   2.58244848, 2.55766749, 2.53388023, 2.49945474, 2.44297576,
     &   2.35605192, 2.25865841, 2.17129350, 2.08190227, 1.98980606,
     &   1.89886510, 1.80608106, 1.69882047, 1.58160162, 1.44403017,
     &   1.28715348, 1.14065719, 1.04025495, 0.99976498, 1.08353901,
     &   1.27777624, 1.49107862, 1.66840136, 1.81396818, 1.92674875,
     &   2.02665734, 2.11248851, 2.18986750, 2.26054764, 2.31750584,
     &   2.37166333, 2.42002869, 2.45725393, 2.48834777, 2.51810169,
     &   2.54033756, 2.54654336, 2.53508186, 2.50283146, 2.45590711,
     &   2.39375639, 2.32884336, 2.26361489, 2.19812560, 2.14589882,
     &   2.10522819, 2.05952501, 1.99371696, 1.91901660, 1.84232521,
     &   1.76079226, 1.67773986, 1.61141121, 1.56728971, 1.52656972,
     &   1.47021914, 1.39292753, 1.30238318, 1.21530604, 1.13114715,
     &   1.05025196, 0.97335202, 0.90302396, 0.84061813, 0.78453475,
     &   0.73412764, 0.68700016, 0.64344925, 0.60443068, 0.56903261,
     &   0.53809428, 0.51216537, 0.48933408, 0.46949074, 0.45252860,
     &   0.43812108, 0.42442867, 0.41331890, 0.40468410, 0.39842474,
     &   0.39371136, 0.39088637, 0.39081490, 0.39339948, 0.40252429,
     &   0.42246720, 0.45115024, 0.49007583, 0.52863681, 0.56691670,
     &   0.61042285, 0.65950638, 0.70641452, 0.75201118, 0.79750943,
     &   0.84374189, 0.89240605, 0.94342530, 1.00657511, 1.09151220,
     &   1.14550054, 1.22135746, 1.30054808, 1.30675840, 1.20567024,
     &   1.08690560, 1.02847099, 0.98698264, 0.95140296, 0.91148263,
     &   0.86745435, 0.81782097, 0.72648573, 0.62974322, 0.53463721,
     &   0.50587660, 0.53943092, 0.59424764, 0.66091073, 0.65857595,
     &   0.65013647, 0.63497168, 0.61141551, 0.57453686, 0.52807832,
     &   0.48581052, 0.44698703, 0.41121691, 0.37820187, 0.34784165,
     &   0.31988499, 0.29192510, 0.26429752, 0.23929028, 0.21676758,
     &   0.19640298, 0.18207903, 0.17021935, 0.15937994, 0.15009585,
     &   0.14138769, 0.13344489, 0.12610441, 0.11919758, 0.11329256,
     &   0.10772855, 0.10261066, 0.09844240, 0.09447247, 0.09069010,
     &   0.08708513, 0.08364797, 0.08036962, 0.07724157, 0.07425582,
     &   0.07140484, 0.06868153, 0.06588019, 0.06317053, 0.06058364,
     &   0.05811334, 0.05575379, 0.05349945, 0.05134509, 0.04928578,
     &   0.04731685, 0.04543388, 0.04363267, 0.04190929, 0.04020274,
     &   0.03836540, 0.03661575, 0.03494937, 0.03336208, 0.03184990,
     &   0.03030562, 0.02819982, 0.02624166, 0.02442067, 0.02265960,
     &   0.02098207, 0.01942971, 0.01794195, 0.01646203, 0.01510505,
     &   0.01387083, 0.01287470, 0.01195065, 0.01109340, 0.01029366,
     &   0.00955100, 0.00886227, 0.00822992, 0.00764824, 0.00710791,
     &   0.00661329, 0.00617015, 0.00575689, 0.00537212, 0.00504764,
     &   0.00474294, 0.00445680, 0.00421196, 0.00398815, 0.00377641,
     &   0.00360276, 0.00346217, 0.00332737, 0.00320320, 0.00309886,
     &   0.00299824, 0.00290118, 0.00281158, 0.00272507, 0.00264147/
      data (qqi(i,5),i=1,mxwv)/
     &   2.06840134, 2.06917286, 2.07003403, 2.07085156, 2.07173562,
     &   2.07264829, 2.07337642, 2.07427597, 2.07516885, 2.07611537,
     &   2.07706308, 2.07797146, 2.07889605, 2.07990432, 2.08089304,
     &   2.08172369, 2.08285546, 2.08371210, 2.08479357, 2.08581638,
     &   2.08680725, 2.08784652, 2.08899927, 2.09001684, 2.09109592,
     &   2.09214067, 2.09338617, 2.09438825, 2.09560418, 2.09680820,
     &   2.09795880, 2.09917188, 2.10032535, 2.10159039, 2.10278964,
     &   2.10406780, 2.10529828, 2.10652757, 2.10790539, 2.10918736,
     &   2.11052871, 2.11198092, 2.11340809, 2.11461234, 2.11601210,
     &   2.11746883, 2.11886144, 2.12043524, 2.12188363, 2.12328315,
     &   2.12492537, 2.12634993, 2.12803745, 2.12961531, 2.13112164,
     &   2.13270140, 2.13440347, 2.13597798, 2.13784099, 2.13953805,
     &   2.14124727, 2.14293361, 2.14478254, 2.14634585, 2.14842224,
     &   2.15030313, 2.15213847, 2.15405345, 2.15583014, 2.15791059,
     &   2.15971184, 2.16179180, 2.16406846, 2.16599011, 2.16802454,
     &   2.17008257, 2.17249346, 2.17464900, 2.17666841, 2.17906451,
     &   2.18136048, 2.18370390, 2.18589664, 2.18853164, 2.19086266,
     &   2.19336724, 2.19577360, 2.19855404, 2.20098042, 2.20357537,
     &   2.20622873, 2.20890141, 2.21156335, 2.21428347, 2.21709037,
     &   2.21994400, 2.22285032, 2.22578907, 2.22875261, 2.23175669,
     &   2.23480058, 2.23787618, 2.24097943, 2.24411416, 2.24726963,
     &   2.25045729, 2.25363922, 2.25673628, 2.25971365, 2.26282024,
     &   2.26607919, 2.26941776, 2.27283669, 2.27627301, 2.27970600,
     &   2.28319144, 2.28681946, 2.29069901, 2.29503441, 2.30015445,
     &   2.30653524, 2.31516051, 2.32771373, 2.34753132, 2.38195634,
     &   2.44882083, 2.56232905, 2.69884896, 2.50515223, 1.68738198,
     &   1.79994905, 2.05135584, 2.13943720, 2.21960449, 2.27977991,
     &   2.29837823, 2.31312561, 2.32606387, 2.33858490, 2.35022449,
     &   2.36025834, 2.36832190, 2.37528086, 2.38158941, 2.38789487,
     &   2.39516330, 2.40344524, 2.41303754, 2.42436981, 2.43697691,
     &   2.45152235, 2.46766996, 2.48564649, 2.50365376, 2.51835155,
     &   2.52800345, 2.53410935, 2.55215406, 2.57512665, 2.60421014,
     &   2.63620186, 2.67144632, 2.70738435, 2.74305677, 2.77916050,
     &   2.81103683, 2.83606434, 2.85055041, 2.84783530, 2.82645798,
     &   2.77004099, 2.73213482, 2.71314311, 2.72039723, 2.74413133,
     &   2.77386403, 2.80627298, 2.81688905, 2.81575346, 2.82521415,
     &   2.83858252, 2.84891987, 2.85699677, 2.87000346, 2.86927629,
     &   2.86782122, 2.87155628, 2.88328433, 2.89546585, 2.89833713,
     &   2.87387133, 2.81677794, 2.76260519, 2.70036721, 2.62831831,
     &   2.54857731, 2.46552420, 2.36362791, 2.24333549, 2.09014130,
     &   1.89154458, 1.66858363, 1.47730279, 1.34942794, 1.37293005,
     &   1.53940380, 1.73051584, 1.88946915, 2.01595616, 2.11444187,
     &   2.20059848, 2.27541757, 2.34328485, 2.40592360, 2.46095872,
     &   2.51436043, 2.56490779, 2.61224627, 2.65633011, 2.70277405,
     &   2.75118423, 2.79617882, 2.83636880, 2.86208224, 2.87693715,
     &   2.87312150, 2.86004806, 2.84029889, 2.81313801, 2.78764653,
     &   2.77672601, 2.76910233, 2.74638605, 2.71261549, 2.67201996,
     &   2.61234307, 2.54235220, 2.47472620, 2.42910361, 2.40133452,
     &   2.36786723, 2.31201291, 2.23024774, 2.13636899, 2.03938270,
     &   1.94033885, 1.83861887, 1.73605573, 1.63942754, 1.54840922,
     &   1.46301568, 1.38083661, 1.30187643, 1.22742963, 1.15740180,
     &   1.09296513, 1.03500283, 0.98174757, 0.93319917, 0.88933307,
     &   0.84973013, 0.81145489, 0.77772868, 0.74843395, 0.72344291,
     &   0.70012480, 0.67909384, 0.66318756, 0.65219313, 0.64996612,
     &   0.66155028, 0.68689269, 0.72738498, 0.76924253, 0.81185335,
     &   0.86172092, 0.91903877, 0.97464699, 1.02914095, 1.08377242,
     &   1.13927460, 1.19756460, 1.25858164, 1.33423209, 1.43811381,
     &   1.51145744, 1.61186886, 1.72698736, 1.78812373, 1.72982061,
     &   1.62023771, 1.56037664, 1.51929426, 1.48819876, 1.45474899,
     &   1.41615498, 1.37358952, 1.27565444, 1.14465642, 0.99289840,
     &   0.90358597, 0.91856778, 0.97861540, 1.07337689, 1.08848250,
     &   1.09051549, 1.08290029, 1.06410956, 1.02602053, 0.96799725,
     &   0.90967745, 0.85233516, 0.79610342, 0.74278831, 0.69131768,
     &   0.64289355, 0.59306967, 0.54254669, 0.49600682, 0.45179304,
     &   0.41097355, 0.37994561, 0.35348606, 0.32952106, 0.30869791,
     &   0.28919494, 0.27140298, 0.25497919, 0.23957260, 0.22612610,
     &   0.21350606, 0.20190273, 0.19223826, 0.18308853, 0.17442583,
     &   0.16622368, 0.15845691, 0.15110153, 0.14413479, 0.13753508,
     &   0.13128191, 0.12535587, 0.11949492, 0.11389305, 0.10857955,
     &   0.10353884, 0.09875617, 0.09421755, 0.08990970, 0.08582009,
     &   0.08193684, 0.07824872, 0.07474511, 0.07141599, 0.06816937,
     &   0.06478877, 0.06158564, 0.05855018, 0.05567316, 0.05294586,
     &   0.05020552, 0.04666094, 0.04337005, 0.04031442, 0.03737089,
     &   0.03457541, 0.03199183, 0.02952653, 0.02709229, 0.02486145,
     &   0.02283146, 0.02116776, 0.01962683, 0.01819946, 0.01687092,
     &   0.01563914, 0.01449831, 0.01345086, 0.01248751, 0.01159388,
     &   0.01077550, 0.01004003, 0.00935532, 0.00871874, 0.00817597,
     &   0.00766754, 0.00719123, 0.00677955, 0.00640272, 0.00604742,
     &   0.00575093, 0.00550560, 0.00527177, 0.00505622, 0.00487205,
     &   0.00469570, 0.00452676, 0.00437085, 0.00422133, 0.00407780/
      data (qqi(i,6),i=1,mxwv)/
     &   2.05414009, 2.05484343, 2.05547309, 2.05614829, 2.05680466,
     &   2.05752754, 2.05816746, 2.05879092, 2.05958652, 2.06034422,
     &   2.06102395, 2.06176543, 2.06248307, 2.06322861, 2.06400514,
     &   2.06483269, 2.06559658, 2.06636500, 2.06709790, 2.06792831,
     &   2.06867027, 2.06954741, 2.07042003, 2.07122993, 2.07217002,
     &   2.07289910, 2.07382560, 2.07470727, 2.07562089, 2.07652378,
     &   2.07741165, 2.07842040, 2.07929254, 2.08026028, 2.08134055,
     &   2.08218384, 2.08319497, 2.08423281, 2.08527708, 2.08613849,
     &   2.08737206, 2.08835506, 2.08945274, 2.09051514, 2.09164906,
     &   2.09271026, 2.09364557, 2.09501958, 2.09612608, 2.09721446,
     &   2.09839177, 2.09967399, 2.10096455, 2.10209394, 2.10340047,
     &   2.10464168, 2.10616946, 2.10711622, 2.10842109, 2.10982442,
     &   2.11104131, 2.11248112, 2.11385560, 2.11526966, 2.11661434,
     &   2.11809349, 2.11970735, 2.12088847, 2.12259912, 2.12393022,
     &   2.12580299, 2.12709117, 2.12871528, 2.13031507, 2.13178086,
     &   2.13353825, 2.13521600, 2.13675404, 2.13873053, 2.14031839,
     &   2.14192390, 2.14389205, 2.14562011, 2.14739490, 2.14918590,
     &   2.15118051, 2.15298128, 2.15498114, 2.15746307, 2.15893912,
     &   2.16097784, 2.16296124, 2.16503263, 2.16711068, 2.16925669,
     &   2.17143273, 2.17365670, 2.17590451, 2.17818952, 2.18049002,
     &   2.18286753, 2.18524480, 2.18771982, 2.19018960, 2.19270897,
     &   2.19526958, 2.19788504, 2.20045519, 2.20295644, 2.20558023,
     &   2.20834827, 2.21123075, 2.21423030, 2.21729875, 2.22038460,
     &   2.22349477, 2.22664237, 2.22981882, 2.23298454, 2.23618817,
     &   2.23941660, 2.24265838, 2.24591470, 2.24931288, 2.25335217,
     &   2.26087332, 2.28386521, 2.36612654, 2.46265244, 1.90700400,
     &   1.90563703, 2.07321739, 2.13306093, 2.18826938, 2.22995281,
     &   2.24190640, 2.25087118, 2.25901461, 2.26738071, 2.27561998,
     &   2.28325796, 2.28993726, 2.29602838, 2.30138588, 2.30612373,
     &   2.31078744, 2.31511474, 2.31916142, 2.32307148, 2.32703543,
     &   2.33110428, 2.33522725, 2.33957815, 2.34396338, 2.34821892,
     &   2.35224175, 2.35603690, 2.36227036, 2.36877966, 2.37612462,
     &   2.38424897, 2.39373732, 2.40482211, 2.41764379, 2.43274093,
     &   2.45044065, 2.47002125, 2.49042296, 2.50862169, 2.52191496,
     &   2.51750088, 2.50290298, 2.49504781, 2.49596810, 2.50562406,
     &   2.52026725, 2.54245090, 2.55951381, 2.56948066, 2.58215451,
     &   2.59831476, 2.61516571, 2.63154483, 2.65310597, 2.67076063,
     &   2.68663597, 2.70434499, 2.72943068, 2.76185799, 2.80038166,
     &   2.83418846, 2.84326959, 2.84923148, 2.84785700, 2.83653808,
     &   2.81335235, 2.78631282, 2.74491382, 2.68342066, 2.58691883,
     &   2.42543864, 2.18780613, 1.93148971, 1.71310532, 1.65343451,
     &   1.76727152, 1.91590166, 2.04095793, 2.13720536, 2.21190524,
     &   2.27585602, 2.33101392, 2.38033199, 2.42514348, 2.46530604,
     &   2.50349736, 2.53948307, 2.57471919, 2.60794210, 2.64299011,
     &   2.68164372, 2.72267532, 2.76681066, 2.80798674, 2.84814239,
     &   2.88049245, 2.90746284, 2.92966247, 2.94547606, 2.95486498,
     &   2.97390175, 3.00209141, 3.02980423, 3.05343676, 3.07220387,
     &   3.07102966, 3.05742431, 3.02874255, 3.00806546, 3.01162267,
     &   3.02593946, 3.03191638, 3.01379442, 2.97040629, 2.91680050,
     &   2.85357428, 2.77745295, 2.68662834, 2.59199452, 2.49568963,
     &   2.39904571, 2.30133367, 2.20203638, 2.10233641, 2.00400066,
     &   1.90835845, 1.81671584, 1.72897267, 1.64561510, 1.56706178,
     &   1.49316049, 1.42029095, 1.35311937, 1.29180408, 1.23644173,
     &   1.18170178, 1.12861598, 1.08345068, 1.04606843, 1.01887786,
     &   1.00716889, 1.01605296, 1.04642928, 1.08225620, 1.12135482,
     &   1.17016292, 1.22840774, 1.28635800, 1.34380162, 1.40166783,
     &   1.46031797, 1.52158463, 1.58537996, 1.66426528, 1.77390254,
     &   1.85718465, 1.96708477, 2.09822130, 2.20214462, 2.20512557,
     &   2.14100218, 2.10021138, 2.07498336, 2.06263328, 2.05302954,
     &   2.04062366, 2.03098059, 1.97579741, 1.85931444, 1.68616581,
     &   1.51365185, 1.47587705, 1.51545286, 1.62261951, 1.66748977,
     &   1.69199848, 1.70672929, 1.71199167, 1.69870281, 1.65721583,
     &   1.60610628, 1.54940557, 1.48744845, 1.42591751, 1.36117327,
     &   1.29756963, 1.22793591, 1.15287805, 1.08051765, 1.00458109,
     &   0.93060005, 0.86807078, 0.81206536, 0.76087564, 0.71510607,
     &   0.67153752, 0.63120979, 0.59350115, 0.55772626, 0.52559066,
     &   0.49517971, 0.46695781, 0.44292811, 0.42007884, 0.39836875,
     &   0.37775555, 0.35819653, 0.33964851, 0.32206863, 0.30541417,
     &   0.28964308, 0.27471411, 0.26044431, 0.24689436, 0.23405482,
     &   0.22189255, 0.21037532, 0.19947186, 0.18915188, 0.17938614,
     &   0.17014648, 0.16140586, 0.15313829, 0.14531891, 0.13780901,
     &   0.13029560, 0.12320205, 0.11650559, 0.11018451, 0.10421816,
     &   0.09834570, 0.09133003, 0.08481809, 0.07877392, 0.07297626,
     &   0.06748662, 0.06241503, 0.05760361, 0.05290620, 0.04859865,
     &   0.04466986, 0.04135692, 0.03829319, 0.03545967, 0.03283034,
     &   0.03039692, 0.02814641, 0.02607840, 0.02417555, 0.02241348,
     &   0.02079758, 0.01933659, 0.01797976, 0.01672091, 0.01562639,
     &   0.01460505, 0.01365187, 0.01281357, 0.01204427, 0.01132288,
     &   0.01070326, 0.01017276, 0.00967169, 0.00920929, 0.00880417,
     &   0.00842033, 0.00805647, 0.00772082, 0.00740228, 0.00709971/
      data (qqi(i,7),i=1,mxwv)/
     &   2.04299641, 2.04347539, 2.04397154, 2.04451704, 2.04506326,
     &   2.04560447, 2.04616117, 2.04668808, 2.04722118, 2.04781461,
     &   2.04820299, 2.04895973, 2.04951787, 2.05010843, 2.05071950,
     &   2.05137110, 2.05185175, 2.05260062, 2.05318785, 2.05395913,
     &   2.05446959, 2.05514812, 2.05582356, 2.05649710, 2.05714536,
     &   2.05781865, 2.05847478, 2.05918527, 2.05989599, 2.06057692,
     &   2.06134200, 2.06205177, 2.06284833, 2.06352210, 2.06432819,
     &   2.06509495, 2.06589985, 2.06660342, 2.06717896, 2.06852555,
     &   2.06905770, 2.06990790, 2.07080817, 2.07158422, 2.07226682,
     &   2.07328510, 2.07423162, 2.07507205, 2.07600760, 2.07685685,
     &   2.07781148, 2.07885551, 2.07969737, 2.08070087, 2.08161998,
     &   2.08258843, 2.08365560, 2.08473372, 2.08580041, 2.08676910,
     &   2.08782864, 2.08877134, 2.08989692, 2.09095526, 2.09213519,
     &   2.09319067, 2.09428477, 2.09547615, 2.09658313, 2.09778738,
     &   2.09897661, 2.10011005, 2.10149837, 2.10259986, 2.10396862,
     &   2.10507703, 2.10646129, 2.10773826, 2.10896945, 2.11041689,
     &   2.11168885, 2.11318231, 2.11448097, 2.11591959, 2.11749816,
     &   2.11876893, 2.12024951, 2.12175417, 2.12322569, 2.12485027,
     &   2.12632990, 2.12788272, 2.12945342, 2.13104248, 2.13268542,
     &   2.13434601, 2.13610554, 2.13775468, 2.13945580, 2.14126468,
     &   2.14306617, 2.14487219, 2.14671803, 2.14857292, 2.15050936,
     &   2.15239835, 2.15441132, 2.15635920, 2.15826321, 2.16025925,
     &   2.16236067, 2.16449237, 2.16681528, 2.16914415, 2.17150044,
     &   2.17390323, 2.17636061, 2.17887998, 2.18144965, 2.18412423,
     &   2.18691468, 2.18985009, 2.19295144, 2.19628358, 2.19986343,
     &   2.20338345, 2.20646787, 2.21510601, 2.27427673, 2.02623439,
     &   1.96984124, 2.07889199, 2.11985946, 2.15826654, 2.18755126,
     &   2.19550347, 2.20106816, 2.20624042, 2.21179318, 2.21740007,
     &   2.22268200, 2.22741389, 2.23185277, 2.23590779, 2.23963594,
     &   2.24341941, 2.24704337, 2.25049043, 2.25383401, 2.25716782,
     &   2.26049304, 2.26365471, 2.26675820, 2.26923466, 2.27128077,
     &   2.27347589, 2.27607322, 2.28145528, 2.28660083, 2.29098439,
     &   2.29533172, 2.29947734, 2.30337548, 2.30723596, 2.31125140,
     &   2.31531882, 2.31971240, 2.32457376, 2.32973146, 2.33490658,
     &   2.33673358, 2.33503556, 2.33488560, 2.33742166, 2.34267974,
     &   2.34927034, 2.35776162, 2.36512947, 2.37085199, 2.37724781,
     &   2.38478327, 2.39288282, 2.40114164, 2.41123986, 2.42139482,
     &   2.43124700, 2.44155860, 2.45442700, 2.47106051, 2.49316716,
     &   2.52038717, 2.54436231, 2.56804776, 2.59184241, 2.61446857,
     &   2.63315845, 2.65226960, 2.67049909, 2.68216395, 2.67964125,
     &   2.63170600, 2.48914385, 2.26327038, 2.01131678, 1.88095009,
     &   1.93488169, 2.03488255, 2.12247181, 2.18848968, 2.23972082,
     &   2.28287029, 2.31974196, 2.35217810, 2.38108444, 2.40664816,
     &   2.43023944, 2.45186663, 2.47254658, 2.49160314, 2.51095176,
     &   2.53155923, 2.55320191, 2.57663488, 2.60009956, 2.62493539,
     &   2.64925528, 2.67325044, 2.69691443, 2.71946049, 2.73778772,
     &   2.75904560, 2.78669357, 2.82083154, 2.85803795, 2.89640260,
     &   2.92941165, 2.95919967, 2.97491741, 2.98542285, 3.00959873,
     &   3.04990935, 3.09956717, 3.14617872, 3.17746663, 3.20218372,
     &   3.21921182, 3.22368574, 3.20923162, 3.18187642, 3.14400673,
     &   3.09685349, 3.04179811, 2.97742224, 2.90339017, 2.82267404,
     &   2.73580098, 2.64382505, 2.54945207, 2.45386791, 2.35823965,
     &   2.26322579, 2.16609740, 2.07185364, 1.98143923, 1.89568675,
     &   1.80696118, 1.71660841, 1.63489473, 1.56239152, 1.49843514,
     &   1.44777226, 1.42441881, 1.42897141, 1.44595683, 1.47069740,
     &   1.50788200, 1.55651689, 1.60757554, 1.65943933, 1.71228468,
     &   1.76586890, 1.82161891, 1.87941158, 1.95068491, 2.05061698,
     &   2.13081217, 2.23207784, 2.35396695, 2.46795940, 2.51293445,
     &   2.50024652, 2.48674631, 2.48250747, 2.49033809, 2.50520372,
     &   2.52120900, 2.54655075, 2.55799341, 2.51145482, 2.39639163,
     &   2.18772554, 2.08666420, 2.07993460, 2.16377711, 2.22968030,
     &   2.27263379, 2.30913854, 2.34236836, 2.36736059, 2.36949205,
     &   2.35712862, 2.33537364, 2.30394864, 2.27060223, 2.22909832,
     &   2.18577743, 2.13318396, 2.07055616, 2.00583410, 1.92614543,
     &   1.84140837, 1.75880599, 1.67922258, 1.60417891, 1.53377450,
     &   1.46408594, 1.39725757, 1.33260572, 1.26917601, 1.20958769,
     &   1.15151608, 1.09609103, 1.04751754, 1.00027108, 0.95439738,
     &   0.90993512, 0.86691445, 0.82535809, 0.78528094, 0.74669093,
     &   0.70958906, 0.67396981, 0.64045894, 0.60838681, 0.57761472,
     &   0.54812777, 0.51990694, 0.49292925, 0.46716866, 0.44259605,
     &   0.41917989, 0.39688650, 0.37568066, 0.35552576, 0.33625799,
     &   0.31750858, 0.29972365, 0.28286576, 0.26689741, 0.25178108,
     &   0.23708622, 0.22078836, 0.20556864, 0.19136283, 0.17770717,
     &   0.16474128, 0.15270717, 0.14130796, 0.13026813, 0.12009300,
     &   0.11074781, 0.10256451, 0.09498315, 0.08796031, 0.08144604,
     &   0.07541162, 0.06982425, 0.06467105, 0.05991494, 0.05550934,
     &   0.05145399, 0.04775423, 0.04432165, 0.04113924, 0.03830560,
     &   0.03566916, 0.03321621, 0.03101414, 0.02898577, 0.02709341,
     &   0.02541591, 0.02392892, 0.02253618, 0.02124907, 0.02009377,
     &   0.01900979, 0.01799243, 0.01705430, 0.01617302, 0.01534467/
      data (qqi(i,8),i=1,mxwv)/
     &   2.03409576, 2.03452325, 2.03492522, 2.03533363, 2.03576612,
     &   2.03618002, 2.03602099, 2.03706980, 2.03749776, 2.03793144,
     &   2.03836179, 2.03883648, 2.03929710, 2.03974628, 2.04023528,
     &   2.04072165, 2.04119349, 2.04169106, 2.04217863, 2.04268503,
     &   2.04319239, 2.04372787, 2.04419446, 2.04474831, 2.04529643,
     &   2.04582500, 2.04635286, 2.04694247, 2.04749298, 2.04805136,
     &   2.04865432, 2.04920316, 2.04977274, 2.05036426, 2.05098200,
     &   2.05157590, 2.05220795, 2.05279803, 2.05344844, 2.05409122,
     &   2.05475664, 2.05538273, 2.05606985, 2.05670309, 2.05746841,
     &   2.05806589, 2.05879378, 2.05949116, 2.06019926, 2.06088829,
     &   2.06164265, 2.06236458, 2.06320357, 2.06388879, 2.06466007,
     &   2.06542182, 2.06621408, 2.06701851, 2.06777787, 2.06852531,
     &   2.06941676, 2.07022500, 2.07107663, 2.07194424, 2.07270718,
     &   2.07371283, 2.07452774, 2.07549644, 2.07638311, 2.07728982,
     &   2.07824063, 2.07917428, 2.08017802, 2.08108354, 2.08204079,
     &   2.08302355, 2.08406758, 2.08509111, 2.08628845, 2.08717632,
     &   2.08822608, 2.08934236, 2.09046292, 2.09148121, 2.09254622,
     &   2.09352589, 2.09484124, 2.09601593, 2.09714150, 2.09842062,
     &   2.09953213, 2.10074544, 2.10195732, 2.10319924, 2.10446596,
     &   2.10575366, 2.10706306, 2.10838246, 2.10970902, 2.11108470,
     &   2.11246920, 2.11391878, 2.11527610, 2.11675024, 2.11815429,
     &   2.11966944, 2.12115359, 2.12264967, 2.12411046, 2.12563515,
     &   2.12723160, 2.12888455, 2.13059640, 2.13234568, 2.13411236,
     &   2.13590407, 2.13773060, 2.13960052, 2.14149857, 2.14347100,
     &   2.14552164, 2.14767504, 2.14995766, 2.15244198, 2.15521789,
     &   2.15828133, 2.16139102, 2.16376305, 2.16923070, 2.06401730,
     &   2.00478601, 2.07603669, 2.10436010, 2.13129568, 2.15207386,
     &   2.15753531, 2.16108918, 2.16444802, 2.16820621, 2.17208457,
     &   2.17577624, 2.17912459, 2.18228912, 2.18521261, 2.18793845,
     &   2.19070864, 2.19339180, 2.19597411, 2.19850659, 2.20105529,
     &   2.20361853, 2.20609331, 2.20850515, 2.21057034, 2.21227384,
     &   2.21407175, 2.21616364, 2.22031283, 2.22433543, 2.22785568,
     &   2.23140621, 2.23484540, 2.23807716, 2.24120665, 2.24432945,
     &   2.24712133, 2.24963903, 2.25178242, 2.25328422, 2.25410509,
     &   2.25293183, 2.25247288, 2.25327873, 2.25620484, 2.26057124,
     &   2.26540160, 2.27046394, 2.27413583, 2.27705479, 2.28072739,
     &   2.28483629, 2.28894925, 2.29304957, 2.29777384, 2.30204391,
     &   2.30632806, 2.31095695, 2.31641555, 2.32269001, 2.32992125,
     &   2.33808160, 2.34620571, 2.35527515, 2.36569357, 2.37761307,
     &   2.39057827, 2.40594482, 2.42553210, 2.44893670, 2.47750568,
     &   2.50145054, 2.47889209, 2.36759710, 2.17385006, 2.02502894,
     &   2.03491163, 2.09516048, 2.15226531, 2.19547153, 2.22935343,
     &   2.25786400, 2.28220797, 2.30352354, 2.32238960, 2.33879614,
     &   2.35374308, 2.36724424, 2.37974477, 2.39114523, 2.40245819,
     &   2.41398025, 2.42545176, 2.43713188, 2.44826555, 2.45950651,
     &   2.47034001, 2.48114467, 2.49207139, 2.50303459, 2.51292706,
     &   2.52351856, 2.53604889, 2.55134916, 2.56850100, 2.58701062,
     &   2.60664177, 2.62798095, 2.64624667, 2.66058612, 2.67719364,
     &   2.70059943, 2.73225760, 2.77090812, 2.81122851, 2.85437131,
     &   2.89985585, 2.94605279, 2.98891139, 3.02643609, 3.05822873,
     &   3.08337402, 3.10278249, 3.11435699, 3.11547399, 3.10687995,
     &   3.08677387, 3.05397797, 3.01141524, 2.95964193, 2.89947486,
     &   2.83180189, 2.75620198, 2.67520881, 2.59036708, 2.50327682,
     &   2.40665174, 2.30126381, 2.19892120, 2.10176229, 2.00557756,
     &   1.91366434, 1.85052168, 1.81812787, 1.80636275, 1.80797637,
     &   1.82439697, 1.85397887, 1.88970113, 1.92824256, 1.96874487,
     &   2.01029825, 2.05371809, 2.09886003, 2.15468812, 2.23385000,
     &   2.30045843, 2.38088107, 2.47699070, 2.57277966, 2.62602282,
     &   2.64224529, 2.64797521, 2.65752268, 2.67524099, 2.70015621,
     &   2.72803283, 2.76564240, 2.81457543, 2.83656812, 2.81958008,
     &   2.66851163, 2.54450011, 2.49552941, 2.53217506, 2.59421158,
     &   2.63722587, 2.67753696, 2.71947098, 2.76339388, 2.79834199,
     &   2.82267714, 2.84056687, 2.85162520, 2.86158109, 2.86505151,
     &   2.86674786, 2.86349106, 2.85380769, 2.84016728, 2.80927992,
     &   2.76905870, 2.71633220, 2.65887356, 2.60186243, 2.54422474,
     &   2.48373604, 2.42256975, 2.36033630, 2.29617548, 2.23205948,
     &   2.16675973, 2.10174918, 2.04252291, 1.98285389, 1.92288041,
     &   1.86273944, 1.80256641, 1.74249399, 1.68265235, 1.62316740,
     &   1.56416142, 1.50575149, 1.45071518, 1.39675212, 1.34345138,
     &   1.29090428, 1.23919773, 1.18841279, 1.13862455, 1.08990276,
     &   1.04231012, 0.99590307, 0.95073152, 0.90683818, 0.86427677,
     &   0.82307559, 0.78316641, 0.74457407, 0.70731634, 0.67140520,
     &   0.63625270, 0.59878933, 0.56307817, 0.52909553, 0.49580854,
     &   0.46364224, 0.43329951, 0.40422350, 0.37587196, 0.34936279,
     &   0.32464477, 0.30214044, 0.28110039, 0.26144493, 0.24309598,
     &   0.22597744, 0.21001559, 0.19516663, 0.18135357, 0.16848685,
     &   0.15654950, 0.14553407, 0.13527206, 0.12571844, 0.11703059,
     &   0.10892849, 0.10137452, 0.09446930, 0.08806790, 0.08209395,
     &   0.07666761, 0.07173526, 0.06712465, 0.06284733, 0.05894117,
     &   0.05528787, 0.05187134, 0.04871543, 0.04576318, 0.04300106/
      data (qqi(i,9),i=1,mxwv)/
     &   2.02707410, 2.02737999, 2.02772069, 2.02804041, 2.02843022,
     &   2.02871370, 2.02905655, 2.02940607, 2.02972746, 2.03009272,
     &   2.03036857, 2.03082561, 2.03122354, 2.03155708, 2.03193307,
     &   2.03231239, 2.03269315, 2.03306532, 2.03346181, 2.03389406,
     &   2.03423953, 2.03466845, 2.03510165, 2.03550243, 2.03591371,
     &   2.03638196, 2.03678393, 2.03723025, 2.03766203, 2.03811598,
     &   2.03856874, 2.03904772, 2.03949666, 2.03997064, 2.04042530,
     &   2.04090977, 2.04142857, 2.04186583, 2.04240727, 2.04294920,
     &   2.04340386, 2.04393697, 2.04445362, 2.04495430, 2.04566073,
     &   2.04605174, 2.04662204, 2.04719663, 2.04769778, 2.04837918,
     &   2.04886174, 2.04944634, 2.05003834, 2.05064225, 2.05123043,
     &   2.05183506, 2.05244279, 2.05306768, 2.05374503, 2.05437350,
     &   2.05498981, 2.05563784, 2.05635905, 2.05698562, 2.05764580,
     &   2.05836177, 2.05905294, 2.05976009, 2.06048846, 2.06114149,
     &   2.06195307, 2.06267357, 2.06338334, 2.06417823, 2.06490850,
     &   2.06574678, 2.06653714, 2.06731153, 2.06816816, 2.06896615,
     &   2.06978512, 2.07060146, 2.07146978, 2.07229972, 2.07317662,
     &   2.07409191, 2.07501602, 2.07588553, 2.07690024, 2.07770896,
     &   2.07865286, 2.07959580, 2.08053756, 2.08150101, 2.08249116,
     &   2.08349586, 2.08451486, 2.08554268, 2.08659983, 2.08764648,
     &   2.08871651, 2.08974123, 2.09090543, 2.09201431, 2.09313393,
     &   2.09429336, 2.09545326, 2.09660363, 2.09773827, 2.09891534,
     &   2.10014796, 2.10141635, 2.10272527, 2.10405803, 2.10540271,
     &   2.10676026, 2.10814953, 2.10955572, 2.11098123, 2.11245155,
     &   2.11397099, 2.11555099, 2.11720824, 2.11898279, 2.12141538,
     &   2.12301135, 2.12512732, 2.12685990, 2.12458944, 2.06434035,
     &   2.02183080, 2.06915641, 2.08890295, 2.10790396, 2.12274575,
     &   2.12660432, 2.12895322, 2.13119483, 2.13379788, 2.13653541,
     &   2.13917017, 2.14158797, 2.14389396, 2.14604306, 2.14806914,
     &   2.15012455, 2.15212679, 2.15406632, 2.15597320, 2.15789723,
     &   2.15983295, 2.16171527, 2.16354895, 2.16515851, 2.16653872,
     &   2.16800833, 2.16970205, 2.17277288, 2.17574453, 2.17836118,
     &   2.18100595, 2.18358254, 2.18603110, 2.18842626, 2.19083548,
     &   2.19303036, 2.19504786, 2.19678664, 2.19800496, 2.19861317,
     &   2.19764185, 2.19751525, 2.19838023, 2.20083022, 2.20424962,
     &   2.20794821, 2.21171999, 2.21442485, 2.21659684, 2.21937227,
     &   2.22244692, 2.22548008, 2.22848582, 2.23191833, 2.23485970,
     &   2.23780751, 2.24105501, 2.24491572, 2.24922371, 2.25380516,
     &   2.25805140, 2.26111364, 2.26447964, 2.26797557, 2.27161717,
     &   2.27539468, 2.28003144, 2.28595066, 2.29364634, 2.30492163,
     &   2.32102036, 2.33209300, 2.30511475, 2.20496607, 2.08951259,
     &   2.08093476, 2.11615133, 2.15291715, 2.18140626, 2.20413160,
     &   2.22344923, 2.24004221, 2.25462985, 2.26757193, 2.27872109,
     &   2.28888392, 2.29805374, 2.30640316, 2.31407428, 2.32168245,
     &   2.32932878, 2.33683252, 2.34434652, 2.35139036, 2.35831976,
     &   2.36470461, 2.37086940, 2.37690592, 2.38274074, 2.38814330,
     &   2.39416528, 2.40099859, 2.40842676, 2.41609740, 2.42387128,
     &   2.43142557, 2.43927169, 2.44649172, 2.45309758, 2.46036863,
     &   2.46908236, 2.47940421, 2.49138832, 2.50475240, 2.51991153,
     &   2.53728533, 2.55755806, 2.58116102, 2.60712361, 2.63523483,
     &   2.66503620, 2.69683242, 2.73007107, 2.76327252, 2.79550529,
     &   2.82466936, 2.84837675, 2.86654949, 2.87791848, 2.88138700,
     &   2.87626743, 2.86300635, 2.83978772, 2.80674314, 2.76442099,
     &   2.70858979, 2.63778138, 2.55912018, 2.47545862, 2.38050604,
     &   2.27472758, 2.18771744, 2.12555480, 2.08813000, 2.06770420,
     &   2.06258059, 2.07109737, 2.08897114, 2.11164546, 2.13736582,
     &   2.16475296, 2.19402218, 2.22501159, 2.26384425, 2.31985593,
     &   2.36908722, 2.42648315, 2.49430251, 2.56200433, 2.60211396,
     &   2.62142015, 2.63237882, 2.64467692, 2.66121602, 2.68223071,
     &   2.70534325, 2.73397994, 2.77637577, 2.81576753, 2.85181427,
     &   2.80064750, 2.71374965, 2.65823150, 2.65854597, 2.69962692,
     &   2.72929931, 2.75829387, 2.78962350, 2.82545853, 2.86123538,
     &   2.89270043, 2.92237520, 2.95053148, 2.97930956, 3.00681567,
     &   3.03456473, 3.06404281, 3.09491920, 3.12496114, 3.14906502,
     &   3.16901398, 3.17440009, 3.17271638, 3.16850615, 3.15983963,
     &   3.14739943, 3.13164306, 3.11253595, 3.08973193, 3.06270981,
     &   3.03226185, 2.99904156, 2.96599960, 2.93051815, 2.89265752,
     &   2.85248685, 2.81008601, 2.76554394, 2.71895862, 2.67043543,
     &   2.62008691, 2.56803274, 2.51873684, 2.46876335, 2.41742134,
     &   2.36481929, 2.31106973, 2.25628805, 2.20059180, 2.14410138,
     &   2.08693767, 2.02922320, 1.97108030, 1.91263187, 1.85443163,
     &   1.79767799, 1.74080372, 1.68392742, 1.62716603, 1.57063401,
     &   1.51392055, 1.45420742, 1.39505649, 1.33661664, 1.27697194,
     &   1.21700883, 1.15836775, 1.10024226, 1.04171085, 0.98506343,
     &   0.93040204, 0.87804401, 0.82772702, 0.77947956, 0.73337197,
     &   0.68935537, 0.64740503, 0.60751837, 0.56965786, 0.53376418,
     &   0.49986601, 0.46800008, 0.43790659, 0.40953076, 0.38319454,
     &   0.35838455, 0.33503577, 0.31333360, 0.29300734, 0.27390742,
     &   0.25624394, 0.23990451, 0.22456138, 0.21021703, 0.19695532,
     &   0.18450384, 0.17281733, 0.16196734, 0.15178496, 0.14223060/
      data (qqi(i,10),i=1,mxwv)/
     &   2.02149177, 2.02173114, 2.02199864, 2.02226591, 2.02252579,
     &   2.02279162, 2.02305508, 2.02328157, 2.02361155, 2.02388763,
     &   2.02418017, 2.02446055, 2.02475286, 2.02503181, 2.02533603,
     &   2.02563858, 2.02595234, 2.02624726, 2.02657533, 2.02688265,
     &   2.02719307, 2.02751732, 2.02784109, 2.02818274, 2.02852440,
     &   2.02884173, 2.02918386, 2.02953410, 2.02988100, 2.03024626,
     &   2.03060198, 2.03096318, 2.03133535, 2.03171754, 2.03208518,
     &   2.03246689, 2.03283381, 2.03323269, 2.03362060, 2.03397799,
     &   2.03443241, 2.03487420, 2.03526545, 2.03568625, 2.03609300,
     &   2.03655076, 2.03696489, 2.03741097, 2.03783250, 2.03827882,
     &   2.03873897, 2.03917813, 2.03939867, 2.04015231, 2.04063606,
     &   2.04108191, 2.04159760, 2.04208136, 2.04262400, 2.04309273,
     &   2.04361653, 2.04412079, 2.04467916, 2.04520702, 2.04573941,
     &   2.04627728, 2.04683971, 2.04735541, 2.04792976, 2.04846501,
     &   2.04906106, 2.04966784, 2.05022240, 2.05085897, 2.05147338,
     &   2.05206203, 2.05269003, 2.05326152, 2.05397034, 2.05464482,
     &   2.05526328, 2.05593777, 2.05658579, 2.05726290, 2.05794668,
     &   2.05866694, 2.05934763, 2.06007910, 2.06075501, 2.06148744,
     &   2.06224084, 2.06298232, 2.06373477, 2.06449151, 2.06526804,
     &   2.06605482, 2.06684494, 2.06764984, 2.06847501, 2.06930542,
     &   2.07013822, 2.07099342, 2.07184458, 2.07273173, 2.07360506,
     &   2.07450414, 2.07540178, 2.07630467, 2.07718992, 2.07810974,
     &   2.07906675, 2.08005047, 2.08106184, 2.08208990, 2.08312440,
     &   2.08416986, 2.08526492, 2.08630919, 2.08739972, 2.08851695,
     &   2.08966470, 2.09085083, 2.09208107, 2.09338093, 2.09477472,
     &   2.09622240, 2.09761548, 2.09856796, 2.09601760, 2.05609179,
     &   2.02878428, 2.06075239, 2.07460999, 2.08807158, 2.09872007,
     &   2.10151005, 2.10312176, 2.10466361, 2.10650706, 2.10847783,
     &   2.11039352, 2.11217332, 2.11388397, 2.11549377, 2.11702681,
     &   2.11858010, 2.12010145, 2.12158084, 2.12304354, 2.12452006,
     &   2.12600589, 2.12745237, 2.12887239, 2.13014007, 2.13126087,
     &   2.13246059, 2.13382936, 2.13612485, 2.13834405, 2.14030385,
     &   2.14228487, 2.14421582, 2.14605761, 2.14786339, 2.14967632,
     &   2.15134215, 2.15288401, 2.15423775, 2.15524006, 2.15582585,
     &   2.15535331, 2.15555525, 2.15644407, 2.15842628, 2.16104054,
     &   2.16381955, 2.16660547, 2.16866302, 2.17038178, 2.17252636,
     &   2.17487049, 2.17718410, 2.17948508, 2.18207622, 2.18432379,
     &   2.18658805, 2.18907237, 2.19199085, 2.19522309, 2.19865322,
     &   2.20183134, 2.20407772, 2.20649409, 2.20888257, 2.21116972,
     &   2.21326542, 2.21565223, 2.21821022, 2.22089100, 2.22389030,
     &   2.22694993, 2.22825456, 2.21912813, 2.17481732, 2.10457802,
     &   2.09460211, 2.11623287, 2.14050460, 2.15983486, 2.17549872,
     &   2.18897438, 2.20063162, 2.21094060, 2.22012544, 2.22800112,
     &   2.23519993, 2.24169946, 2.24756718, 2.25299788, 2.25838375,
     &   2.26375484, 2.26900411, 2.27425432, 2.27922177, 2.28413653,
     &   2.28870106, 2.29313684, 2.29749346, 2.30169845, 2.30564785,
     &   2.31010985, 2.31515694, 2.32052612, 2.32600689, 2.33151746,
     &   2.33655524, 2.34145832, 2.34562922, 2.34964800, 2.35461140,
     &   2.36058903, 2.36714554, 2.37369561, 2.37977552, 2.38589215,
     &   2.39209008, 2.39837909, 2.40483999, 2.41172242, 2.41927958,
     &   2.42772007, 2.43738675, 2.44858122, 2.46151996, 2.47635221,
     &   2.49299550, 2.51108861, 2.53040504, 2.55034041, 2.57007122,
     &   2.58872366, 2.60642076, 2.62063622, 2.63003540, 2.63339853,
     &   2.62913108, 2.61413860, 2.58785367, 2.55071831, 2.49621272,
     &   2.42047501, 2.34565783, 2.28129816, 2.23676753, 2.20739198,
     &   2.19062948, 2.18570995, 2.19090056, 2.20165825, 2.21594429,
     &   2.23231244, 2.25060654, 2.27065301, 2.29629397, 2.33393788,
     &   2.36822820, 2.40759182, 2.45376110, 2.49839211, 2.52312922,
     &   2.53548241, 2.54416776, 2.55371857, 2.56540227, 2.57918763,
     &   2.59379292, 2.61055279, 2.63357854, 2.65762472, 2.68698835,
     &   2.68745780, 2.65238166, 2.61937046, 2.61196280, 2.63462090,
     &   2.65195966, 2.66909528, 2.68730140, 2.70776439, 2.72908425,
     &   2.74897718, 2.76866698, 2.78864861, 2.80924058, 2.83054137,
     &   2.85254645, 2.87729216, 2.90529013, 2.93445706, 2.96618819,
     &   2.99974370, 3.02883339, 3.05599213, 3.08234763, 3.10656118,
     &   3.13027906, 3.15261483, 3.17363286, 3.19345808, 3.21009398,
     &   3.22500515, 3.23765993, 3.24670959, 3.25395632, 3.25927043,
     &   3.26252866, 3.26361561, 3.26242471, 3.25885820, 3.25283003,
     &   3.24426484, 3.23309946, 3.22235560, 3.20980120, 3.19486237,
     &   3.17751098, 3.15772891, 3.13550901, 3.11085391, 3.08377624,
     &   3.05429912, 3.02245498, 2.98828578, 2.95184326, 2.91393995,
     &   2.87668443, 2.83730769, 2.79587889, 2.75247526, 2.70718002,
     &   2.66020584, 2.61194873, 2.56145382, 2.50888181, 2.45207357,
     &   2.39173770, 2.32971191, 2.26522756, 2.19714355, 2.12793708,
     &   2.05781102, 1.98613548, 1.91425502, 1.84240031, 1.77096450,
     &   1.70001626, 1.62971270, 1.56020129, 1.49166703, 1.42430556,
     &   1.35835803, 1.29409003, 1.23137569, 1.17032743, 1.11165118,
     &   1.05475295, 0.99969161, 0.94692320, 0.89617562, 0.84733474,
     &   0.80088335, 0.75676143, 0.71449327, 0.67422539, 0.63622546,
     &   0.59996223, 0.56540412, 0.53288305, 0.50195295, 0.47256675/
      data (qqi(i,11),i=1,mxwv)/
     &   2.01706696, 2.01726270, 2.01746988, 2.01766729, 2.01788950,
     &   2.01809287, 2.01830769, 2.01852012, 2.01873446, 2.01896501,
     &   2.01918650, 2.01929808, 2.01970196, 2.01987743, 2.02011609,
     &   2.02034593, 2.02059007, 2.02083564, 2.02108550, 2.02133036,
     &   2.02158332, 2.02183604, 2.02210855, 2.02235866, 2.02263665,
     &   2.02289891, 2.02317238, 2.02343822, 2.02373123, 2.02400422,
     &   2.02429342, 2.02458191, 2.02486324, 2.02515817, 2.02545094,
     &   2.02576733, 2.02607250, 2.02638268, 2.02669740, 2.02699971,
     &   2.02732825, 2.02769685, 2.02796841, 2.02831316, 2.02854180,
     &   2.02898431, 2.02932954, 2.02967262, 2.03001928, 2.03039908,
     &   2.03074312, 2.03109789, 2.03148437, 2.03184772, 2.03221822,
     &   2.03257179, 2.03300762, 2.03338075, 2.03379369, 2.03418589,
     &   2.03455877, 2.03499556, 2.03542495, 2.03585482, 2.03626299,
     &   2.03668714, 2.03713036, 2.03755832, 2.03801107, 2.03847528,
     &   2.03892899, 2.03939056, 2.04008389, 2.04039931, 2.04079795,
     &   2.04128981, 2.04178619, 2.04228425, 2.04274917, 2.04328823,
     &   2.04381895, 2.04433417, 2.04486060, 2.04539251, 2.04593229,
     &   2.04646802, 2.04702759, 2.04760027, 2.04824996, 2.04873991,
     &   2.04932070, 2.04990363, 2.05049539, 2.05109072, 2.05170655,
     &   2.05232453, 2.05295467, 2.05359197, 2.05421853, 2.05488300,
     &   2.05554366, 2.05620694, 2.05688167, 2.05756927, 2.05825472,
     &   2.05896354, 2.05967188, 2.06037164, 2.06107688, 2.06179929,
     &   2.06254935, 2.06331706, 2.06410456, 2.06490350, 2.06570768,
     &   2.06651974, 2.06732750, 2.06817722, 2.06901860, 2.06987834,
     &   2.07079554, 2.07166362, 2.07259440, 2.07356715, 2.07459378,
     &   2.07563424, 2.07659340, 2.07713652, 2.07490039, 2.04777932,
     &   2.03028774, 2.05216765, 2.06194377, 2.07151222, 2.07917452,
     &   2.08122802, 2.08237505, 2.08346868, 2.08480287, 2.08624649,
     &   2.08766246, 2.08899307, 2.09028172, 2.09150457, 2.09268069,
     &   2.09387088, 2.09504199, 2.09618855, 2.09732270, 2.09846997,
     &   2.09962273, 2.10075283, 2.10186410, 2.10287356, 2.10378861,
     &   2.10476828, 2.10587382, 2.10760736, 2.10928273, 2.11077476,
     &   2.11191344, 2.11375046, 2.11515880, 2.11654282, 2.11792994,
     &   2.11921382, 2.12041187, 2.12147903, 2.12230706, 2.12285161,
     &   2.12269449, 2.12306380, 2.12390518, 2.12549710, 2.12750244,
     &   2.12960458, 2.13168263, 2.13326502, 2.13463283, 2.13630509,
     &   2.13809872, 2.13987494, 2.14164448, 2.14360666, 2.14533281,
     &   2.14707875, 2.14898157, 2.15117836, 2.15357757, 2.15609312,
     &   2.15842032, 2.16011548, 2.16193652, 2.16373634, 2.16545987,
     &   2.16703939, 2.16879988, 2.17061520, 2.17239690, 2.17407537,
     &   2.17481470, 2.17244172, 2.16384459, 2.13931155, 2.09924865,
     &   2.09275317, 2.10697818, 2.12348819, 2.13692570, 2.14794898,
     &   2.15753174, 2.16587710, 2.17330003, 2.17994428, 2.18563294,
     &   2.19084764, 2.19556403, 2.19979525, 2.20373535, 2.20763731,
     &   2.21149659, 2.21525407, 2.21899986, 2.22258162, 2.22612619,
     &   2.22946548, 2.23274231, 2.23598504, 2.23914266, 2.24216509,
     &   2.24555087, 2.24932742, 2.25330639, 2.25737214, 2.26148176,
     &   2.26528025, 2.26899385, 2.27220583, 2.27533627, 2.27918315,
     &   2.28378844, 2.28885293, 2.29396176, 2.29873085, 2.30354619,
     &   2.30840564, 2.31319904, 2.31774235, 2.32213354, 2.32643795,
     &   2.33067703, 2.33496332, 2.33927846, 2.34361148, 2.34811020,
     &   2.35281634, 2.35780859, 2.36332965, 2.36949348, 2.37637305,
     &   2.38400459, 2.39268255, 2.40190506, 2.41126108, 2.42014599,
     &   2.42827296, 2.43412519, 2.43543696, 2.43073678, 2.41506433,
     &   2.38240242, 2.34099483, 2.29805398, 2.26492643, 2.24099469,
     &   2.22497177, 2.21720910, 2.21760798, 2.22277308, 2.23103666,
     &   2.24124408, 2.25315166, 2.26660824, 2.28402424, 2.30971360,
     &   2.33358598, 2.36103511, 2.39311838, 2.42293668, 2.43774271,
     &   2.44440603, 2.45025349, 2.45697641, 2.46502900, 2.47410274,
     &   2.48344016, 2.49369812, 2.50578952, 2.51748133, 2.53102827,
     &   2.53520417, 2.52462983, 2.51201940, 2.51070261, 2.52476096,
     &   2.53615713, 2.54729772, 2.55865455, 2.57050633, 2.58217382,
     &   2.59306359, 2.60376120, 2.61452055, 2.62542009, 2.63658714,
     &   2.64794612, 2.66025233, 2.67372489, 2.68761754, 2.70315957,
     &   2.72002101, 2.73692775, 2.75437331, 2.77231193, 2.79047251,
     &   2.80969834, 2.82952118, 2.85007071, 2.87161684, 2.89321160,
     &   2.91559625, 2.93829322, 2.95931840, 2.98073411, 3.00246549,
     &   3.02443242, 3.04653788, 3.06867981, 3.09074306, 3.11261010,
     &   3.13414693, 3.15521646, 3.17581224, 3.19588876, 3.21531510,
     &   3.23396492, 3.25170875, 3.26841497, 3.28395224, 3.29818821,
     &   3.31099296, 3.32223821, 3.33179927, 3.33955550, 3.34588337,
     &   3.35213113, 3.35642481, 3.35866499, 3.35875940, 3.35662317,
     &   3.35294557, 3.35161543, 3.34729028, 3.33989477, 3.32864690,
     &   3.31333447, 3.29435301, 3.27149296, 3.24415851, 3.21272373,
     &   3.17713547, 3.13551426, 3.09031391, 3.04169297, 2.99004555,
     &   2.93540049, 2.87790966, 2.81766725, 2.75492120, 2.69000816,
     &   2.62319493, 2.55479741, 2.48497844, 2.41397309, 2.34249783,
     &   2.27032781, 2.19766784, 2.12507129, 2.05252457, 1.98009789,
     &   1.90844297, 1.83773613, 1.76766837, 1.69863617, 1.63127947,
     &   1.56492341, 1.49967611, 1.43647873, 1.37454689, 1.31395245/
      data (qqi(i,12),i=1,mxwv)/
     &   2.01354265, 2.01370215, 2.01386571, 2.01402664, 2.01419997,
     &   2.01436305, 2.01452351, 2.01466107, 2.01488471, 2.01505852,
     &   2.01523352, 2.01541877, 2.01559353, 2.01578236, 2.01598287,
     &   2.01616073, 2.01635575, 2.01654506, 2.01673841, 2.01693749,
     &   2.01715159, 2.01733899, 2.01754665, 2.01777101, 2.01796818,
     &   2.01817060, 2.01838708, 2.01860404, 2.01883149, 2.01905394,
     &   2.01927590, 2.01951218, 2.01974273, 2.01996303, 2.02021337,
     &   2.02044463, 2.02068686, 2.02093172, 2.02121234, 2.02143908,
     &   2.02169085, 2.02194428, 2.02220511, 2.02253580, 2.02273083,
     &   2.02300692, 2.02327681, 2.02355218, 2.02382827, 2.02411842,
     &   2.02439046, 2.02471852, 2.02497554, 2.02527165, 2.02557445,
     &   2.02589250, 2.02618480, 2.02649236, 2.02680540, 2.02713037,
     &   2.02745199, 2.02777958, 2.02812147, 2.02845168, 2.02877641,
     &   2.02911448, 2.02946544, 2.02981448, 2.03016543, 2.03051519,
     &   2.03090143, 2.03125286, 2.03161335, 2.03200078, 2.03237844,
     &   2.03275514, 2.03314090, 2.03351521, 2.03394175, 2.03434253,
     &   2.03475380, 2.03514838, 2.03556466, 2.03600407, 2.03643084,
     &   2.03685760, 2.03730631, 2.03774691, 2.03819656, 2.03864932,
     &   2.03911567, 2.03957152, 2.04003882, 2.04051280, 2.04099607,
     &   2.04148507, 2.04198170, 2.04248333, 2.04299092, 2.04350376,
     &   2.04402423, 2.04454994, 2.04507971, 2.04562211, 2.04616642,
     &   2.04672050, 2.04727840, 2.04784274, 2.04838610, 2.04895782,
     &   2.04954672, 2.05014896, 2.05076623, 2.05139160, 2.05204558,
     &   2.05265498, 2.05329585, 2.05394435, 2.05460453, 2.05527282,
     &   2.05595541, 2.05665278, 2.05736804, 2.05810332, 2.05887914,
     &   2.05964565, 2.06033635, 2.06067848, 2.05891538, 2.04039288,
     &   2.02896667, 2.04408884, 2.05101514, 2.05783510, 2.06335974,
     &   2.06489062, 2.06573367, 2.06653237, 2.06751704, 2.06859088,
     &   2.06965160, 2.07065964, 2.07164192, 2.07258201, 2.07349348,
     &   2.07441378, 2.07532477, 2.07622004, 2.07710767, 2.07800651,
     &   2.07891107, 2.07979393, 2.08067751, 2.08148646, 2.08223438,
     &   2.08303332, 2.08392501, 2.08524537, 2.08652329, 2.08767080,
     &   2.08882737, 2.08996916, 2.09103656, 2.09220099, 2.09319997,
     &   2.09420681, 2.09515166, 2.09600639, 2.09669805, 2.09719586,
     &   2.09722519, 2.09765339, 2.09840751, 2.09967947, 2.10122538,
     &   2.10282874, 2.10439992, 2.10563207, 2.10672879, 2.10803413,
     &   2.10942864, 2.11080623, 2.11218190, 2.11368656, 2.11502886,
     &   2.11638999, 2.11786318, 2.11953425, 2.12133765, 2.12320828,
     &   2.12493610, 2.12623167, 2.12761807, 2.12898660, 2.13029790,
     &   2.13150835, 2.13283324, 2.13417649, 2.13546944, 2.13663483,
     &   2.13702440, 2.13491654, 2.12815237, 2.11205435, 2.08809233,
     &   2.08451200, 2.09429336, 2.10576701, 2.11525750, 2.12312102,
     &   2.13001871, 2.13606477, 2.14147496, 2.14634204, 2.15051436,
     &   2.15435266, 2.15783310, 2.16094613, 2.16386151, 2.16674519,
     &   2.16957903, 2.17232561, 2.17505646, 2.17768145, 2.18029857,
     &   2.18278551, 2.18524504, 2.18769336, 2.19009519, 2.19243073,
     &   2.19501996, 2.19786334, 2.20082474, 2.20384264, 2.20689487,
     &   2.20974374, 2.21253586, 2.21501613, 2.21747112, 2.22041631,
     &   2.22385979, 2.22759628, 2.23136020, 2.23491144, 2.23851514,
     &   2.24217534, 2.24582434, 2.24933958, 2.25278544, 2.25619888,
     &   2.25958014, 2.26299572, 2.26639795, 2.26971555, 2.27299047,
     &   2.27614784, 2.27911639, 2.28200150, 2.28480220, 2.28752518,
     &   2.29019737, 2.29295230, 2.29571652, 2.29850030, 2.30127692,
     &   2.30397940, 2.30637717, 2.30798435, 2.30812168, 2.30435944,
     &   2.29281068, 2.27486944, 2.25311351, 2.23471665, 2.22058487,
     &   2.21055555, 2.20542932, 2.20568085, 2.20924306, 2.21503401,
     &   2.22228980, 2.23083329, 2.24056101, 2.25307488, 2.27131081,
     &   2.28834534, 2.30807185, 2.33106565, 2.35176992, 2.36107898,
     &   2.36476207, 2.36884785, 2.37376523, 2.37963653, 2.38608289,
     &   2.39261842, 2.39965606, 2.40701270, 2.41364074, 2.42056966,
     &   2.42156529, 2.41712046, 2.41311407, 2.41624999, 2.42666292,
     &   2.43530631, 2.44362879, 2.45185971, 2.45997286, 2.46751642,
     &   2.47456789, 2.48147655, 2.48835754, 2.49532533, 2.50235558,
     &   2.50947165, 2.51689649, 2.52468300, 2.53256750, 2.54081035,
     &   2.54932785, 2.55766344, 2.56609058, 2.57465672, 2.58330703,
     &   2.59229994, 2.60154343, 2.61110830, 2.62111163, 2.63138723,
     &   2.64216828, 2.65337467, 2.66442966, 2.67598772, 2.68808246,
     &   2.70074749, 2.71401477, 2.72791982, 2.74247742, 2.75772667,
     &   2.77368450, 2.79036760, 2.80715132, 2.82449460, 2.84251380,
     &   2.86120868, 2.88057351, 2.90059495, 2.92125297, 2.94252014,
     &   2.96436048, 2.98672938, 3.00957441, 3.03283310, 3.05644703,
     &   3.08042097, 3.10466290, 3.12908936, 3.15360975, 3.17812467,
     &   3.20308948, 3.23159504, 3.25987005, 3.28775239, 3.31603265,
     &   3.34414434, 3.37115145, 3.39739990, 3.42324567, 3.44702911,
     &   3.46837997, 3.48565984, 3.50008798, 3.51144004, 3.51959229,
     &   3.52431679, 3.52543759, 3.52271032, 3.51604009, 3.50544357,
     &   3.49079013, 3.47198200, 3.44923449, 3.42256856, 3.39187837,
     &   3.35751891, 3.31957793, 3.27808738, 3.23331404, 3.18543100,
     &   3.13461208, 3.08113933, 3.02518940, 2.96711850, 2.90753794,
     &   2.84607840, 2.78291082, 2.71919131, 2.65415311, 2.58794975/
      data (qqi(i,13),i=1,mxwv)/
     &   2.01075125, 2.01087999, 2.01100898, 2.01114488, 2.01127291,
     &   2.01139784, 2.01153874, 2.01167774, 2.01183462, 2.01194906,
     &   2.01209188, 2.01223850, 2.01239705, 2.01253176, 2.01268744,
     &   2.01282692, 2.01298237, 2.01314306, 2.01328754, 2.01344466,
     &   2.01360583, 2.01375937, 2.01392555, 2.01410031, 2.01426530,
     &   2.01442957, 2.01460767, 2.01477909, 2.01494956, 2.01512027,
     &   2.01530647, 2.01549172, 2.01566911, 2.01585436, 2.01604438,
     &   2.01623201, 2.01642251, 2.01661921, 2.01681757, 2.01701117,
     &   2.01721787, 2.01741147, 2.01763415, 2.01783013, 2.01788425,
     &   2.01826239, 2.01847291, 2.01869464, 2.01891685, 2.01912975,
     &   2.01937270, 2.01959133, 2.01983070, 2.02006292, 2.02029133,
     &   2.02056670, 2.02079177, 2.02102566, 2.02127767, 2.02152681,
     &   2.02179527, 2.02204680, 2.02230954, 2.02256536, 2.02283812,
     &   2.02312875, 2.02338505, 2.02366638, 2.02394271, 2.02422166,
     &   2.02449536, 2.02480316, 2.02508783, 2.02543569, 2.02570152,
     &   2.02601409, 2.02630568, 2.02661729, 2.02693033, 2.02725077,
     &   2.02758288, 2.02790356, 2.02823424, 2.02856827, 2.02891183,
     &   2.02943587, 2.02959561, 2.02994728, 2.03034019, 2.03066397,
     &   2.03102708, 2.03135395, 2.03176355, 2.03213906, 2.03252125,
     &   2.03290844, 2.03330135, 2.03369784, 2.03409934, 2.03450727,
     &   2.03491592, 2.03533173, 2.03575134, 2.03616214, 2.03654456,
     &   2.03704667, 2.03748655, 2.03792739, 2.03836536, 2.03881645,
     &   2.03928113, 2.03975582, 2.04024100, 2.04073191, 2.04122615,
     &   2.04172468, 2.04222369, 2.04273748, 2.04325390, 2.04377675,
     &   2.04430985, 2.04485178, 2.04540610, 2.04597664, 2.04658818,
     &   2.04714394, 2.04765749, 2.04789758, 2.04658651, 2.03388643,
     &   2.02630115, 2.03682971, 2.04175520, 2.04662776, 2.05061722,
     &   2.05176854, 2.05240488, 2.05300307, 2.05374169, 2.05455065,
     &   2.05535650, 2.05612636, 2.05688238, 2.05761170, 2.05832148,
     &   2.05904245, 2.05974722, 2.06045699, 2.06115627, 2.06186461,
     &   2.06257749, 2.06328154, 2.06397796, 2.06462812, 2.06523895,
     &   2.06588912, 2.06660724, 2.06761980, 2.06860161, 2.06949186,
     &   2.07038951, 2.07126999, 2.07212210, 2.07296348, 2.07380652,
     &   2.07459950, 2.07535195, 2.07604241, 2.07662129, 2.07706785,
     &   2.07719898, 2.07762671, 2.07828069, 2.07929349, 2.08049011,
     &   2.08172178, 2.08292270, 2.08389068, 2.08477235, 2.08580089,
     &   2.08688998, 2.08796787, 2.08904576, 2.09021139, 2.09126425,
     &   2.09233475, 2.09348512, 2.09477043, 2.09614301, 2.09755564,
     &   2.09886265, 2.09987307, 2.10094881, 2.10201073, 2.10303092,
     &   2.10398030, 2.10500455, 2.10602593, 2.10699725, 2.10784912,
     &   2.10811186, 2.10658646, 2.10182309, 2.09103370, 2.07597995,
     &   2.07410622, 2.08099151, 2.08907819, 2.09585190, 2.10151386,
     &   2.10652089, 2.11093855, 2.11491585, 2.11851358, 2.12160921,
     &   2.12446904, 2.12707210, 2.12939978, 2.13159156, 2.13375759,
     &   2.13587570, 2.13792372, 2.13995504, 2.14192057, 2.14388537,
     &   2.14577127, 2.14764810, 2.14952493, 2.15137720, 2.15320110,
     &   2.15520287, 2.15737128, 2.15960836, 2.16188169, 2.16418195,
     &   2.16634607, 2.16847444, 2.17040563, 2.17234063, 2.17461205,
     &   2.17720938, 2.17998958, 2.18277717, 2.18542194, 2.18810296,
     &   2.19082403, 2.19353771, 2.19616151, 2.19874477, 2.20131350,
     &   2.20386934, 2.20645356, 2.20903921, 2.21157908, 2.21410036,
     &   2.21655393, 2.21887827, 2.22113442, 2.22330236, 2.22535944,
     &   2.22728848, 2.22913408, 2.23080277, 2.23226547, 2.23349214,
     &   2.23432374, 2.23461366, 2.23437333, 2.23345065, 2.23069978,
     &   2.22437525, 2.21531892, 2.20428061, 2.19477391, 2.18739986,
     &   2.18224406, 2.17995644, 2.18097234, 2.18409872, 2.18869400,
     &   2.19428968, 2.20078874, 2.20813084, 2.21743202, 2.23072553,
     &   2.24313188, 2.25758004, 2.27437901, 2.28919911, 2.29540229,
     &   2.29759645, 2.30062127, 2.30436325, 2.30880737, 2.31359601,
     &   2.31840014, 2.32348728, 2.32831478, 2.33251214, 2.33674049,
     &   2.33708692, 2.33489728, 2.33399820, 2.33840609, 2.34647369,
     &   2.35324144, 2.35969234, 2.36595130, 2.37189198, 2.37722802,
     &   2.38225365, 2.38720584, 2.39215255, 2.39719582, 2.40229273,
     &   2.40748334, 2.41285396, 2.41844583, 2.42414856, 2.43003750,
     &   2.43608999, 2.44197631, 2.44785523, 2.45378089, 2.45969343,
     &   2.46570778, 2.47177482, 2.47791243, 2.48415375, 2.49040031,
     &   2.49675417, 2.50318122, 2.50950909, 2.51595163, 2.52251744,
     &   2.52921581, 2.53605890, 2.54305911, 2.55023146, 2.55759335,
     &   2.56516123, 2.57295871, 2.58084679, 2.58893991, 2.59728456,
     &   2.60590410, 2.61482334, 2.62406874, 2.63366842, 2.64365244,
     &   2.65405202, 2.66489935, 2.67622805, 2.68807220, 2.70040751,
     &   2.71308088, 2.72632527, 2.74017310, 2.75465560, 2.76980257,
     &   2.78573871, 2.80308914, 2.82129693, 2.84039593, 2.86108255,
     &   2.88330436, 2.90663099, 2.93144345, 2.95831037, 2.98638630,
     &   3.01561761, 3.04593062, 3.07719517, 3.10928273, 3.14198756,
     &   3.17517710, 3.20867181, 3.24224591, 3.27566600, 3.30869842,
     &   3.34099936, 3.37217402, 3.40217161, 3.43071437, 3.45698953,
     &   3.48132038, 3.50346351, 3.52277875, 3.53934979, 3.55310965,
     &   3.56341362, 3.57017779, 3.57373309, 3.57390571, 3.57055640,
     &   3.56382751, 3.55366683, 3.54032469, 3.52361321, 3.50353217/
      data (wwi(i,1),i=1,mxwv)/
     &   0.99999893, 0.99999899, 0.99999905, 0.99999917, 0.99999917,
     &   0.99999928, 0.99999934, 0.99999934, 0.99999940, 0.99999946,
     &   0.99999952, 0.99999958, 0.99999958, 0.99999964, 0.99999970,
     &   0.99999976, 0.99999976, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999994, 0.99999994, 0.99999994, 0.99999994, 0.99999994,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999976, 0.99999970, 0.99999964, 0.99999958,
     &   0.99999952, 0.99999946, 0.99999934, 0.99999928, 0.99999917,
     &   0.99999899, 0.99999875, 0.99999863, 0.99999845, 0.99999803,
     &   0.99999756, 0.99999690, 0.99999636, 0.99999601, 0.99999595,
     &   0.99999589, 0.99999487, 0.99999338, 0.99999100, 0.99998951,
     &   0.99998868, 0.99998760, 0.99998617, 0.99998116, 0.99997437,
     &   0.99996555, 0.99995404, 0.99995154, 0.99995726, 0.99996454,
     &   0.99996686, 0.99996716, 0.99996305, 0.99995965, 0.99993354,
     &   0.99990219, 0.99985403, 0.99981683, 0.99980050, 0.99980265,
     &   0.99981040, 0.99981564, 0.99981272, 0.99980217, 0.99975735,
     &   0.99927634, 0.99793428, 0.99523455, 0.99389344, 0.99461949,
     &   0.99556917, 0.99665117, 0.99732834, 0.99760228, 0.99796760,
     &   0.99828738, 0.99854553, 0.99868679, 0.99877495, 0.99887246,
     &   0.99888206, 0.99804980, 0.99478960, 0.98900956, 0.98652303,
     &   0.98732007, 0.98947984, 0.99293846, 0.99592048, 0.99739212,
     &   0.99805778, 0.99829984, 0.99786675, 0.99673408, 0.99558574,
     &   0.99478257, 0.99389750, 0.99280512, 0.99200344, 0.99055749,
     &   0.97978437, 0.94881326, 0.86067367, 0.60967690, 0.22692783,
     &   0.24402110, 0.34908041, 0.38435230, 0.42141807, 0.44705504,
     &   0.46337482, 0.50317681, 0.58499479, 0.69056219, 0.79392713,
     &   0.87180734, 0.91696227, 0.94493204, 0.95685589, 0.96005332,
     &   0.96240419, 0.96105951, 0.95565194, 0.94754022, 0.94055450,
     &   0.93296361, 0.92204070, 0.90845293, 0.88575310, 0.85709125,
     &   0.83630675, 0.82634318, 0.86181229, 0.89117855, 0.90125400,
     &   0.91037828, 0.91374570, 0.90981656, 0.90216404, 0.89276022,
     &   0.87364262, 0.84773886, 0.81365931, 0.76795143, 0.71263850,
     &   0.63880861, 0.60081506, 0.58054703, 0.58459353, 0.60079491,
     &   0.61934543, 0.63395274, 0.62827170, 0.61567205, 0.61401254,
     &   0.61566389, 0.61506987, 0.61347836, 0.61699241, 0.61041307,
     &   0.60451680, 0.60382456, 0.61171627, 0.62421691, 0.63684320,
     &   0.63682020, 0.61171287, 0.59207040, 0.57065815, 0.54597348,
     &   0.51754898, 0.49330077, 0.46709991, 0.43758005, 0.40136182,
     &   0.34683588, 0.26755103, 0.18720371, 0.12375581, 0.09360991,
     &   0.09752577, 0.11323323, 0.12963845, 0.14419124, 0.15664892,
     &   0.16863048, 0.18015088, 0.19176209, 0.20369312, 0.21541660,
     &   0.22789824, 0.24071628, 0.25377411, 0.26638499, 0.28071111,
     &   0.29709741, 0.31369233, 0.33045653, 0.34305000, 0.35353214,
     &   0.35761806, 0.35893121, 0.35832030, 0.35484597, 0.34928825,
     &   0.35131142, 0.36001039, 0.36978176, 0.37955168, 0.38935012,
     &   0.38836625, 0.38247770, 0.36575145, 0.35199243, 0.35570320,
     &   0.37472662, 0.40229750, 0.42721802, 0.43801203, 0.44849926,
     &   0.45877716, 0.46303931, 0.45217001, 0.43415219, 0.41211826,
     &   0.38715214, 0.36229724, 0.33568290, 0.30611241, 0.27683201,
     &   0.24706928, 0.21694480, 0.18947335, 0.16467519, 0.14249958,
     &   0.12286275, 0.10575227, 0.09075926, 0.07770611, 0.06640869,
     &   0.05607647, 0.04675244, 0.03896822, 0.03250998, 0.02668021,
     &   0.02141886, 0.01760549, 0.01495353, 0.01326762, 0.01214987,
     &   0.01142772, 0.01102241, 0.01084100, 0.01077711, 0.01078768,
     &   0.01084391, 0.01094757, 0.01109459, 0.01140854, 0.01210978,
     &   0.01268792, 0.01358829, 0.01501046, 0.01692639, 0.01800643,
     &   0.01782241, 0.01725059, 0.01681728, 0.01664759, 0.01671414,
     &   0.01688782, 0.01740030, 0.01839100, 0.01860896, 0.01805196,
     &   0.01428891, 0.01127047, 0.00966097, 0.00928079, 0.00968938,
     &   0.00989777, 0.01011711, 0.01042244, 0.01086094, 0.01125539,
     &   0.01149649, 0.01165929, 0.01174688, 0.01185908, 0.01191653,
     &   0.01199670, 0.01212029, 0.01229469, 0.01249737, 0.01256203,
     &   0.01260173, 0.01230579, 0.01192239, 0.01157933, 0.01120660,
     &   0.01084624, 0.01049369, 0.01015059, 0.00981894, 0.00945324,
     &   0.00909977, 0.00875362, 0.00839303, 0.00804719, 0.00771550,
     &   0.00739740, 0.00709233, 0.00679977, 0.00651921, 0.00625017,
     &   0.00599218, 0.00574478, 0.00554037, 0.00534911, 0.00516451,
     &   0.00498634, 0.00481438, 0.00464839, 0.00448818, 0.00433353,
     &   0.00418425, 0.00404015, 0.00390106, 0.00376678, 0.00364388,
     &   0.00354948, 0.00345762, 0.00336821, 0.00328119, 0.00319648,
     &   0.00312487, 0.00312572, 0.00312688, 0.00312833, 0.00313576,
     &   0.00314757, 0.00315967, 0.00317993, 0.00321868, 0.00325812,
     &   0.00329554, 0.00329372, 0.00329197, 0.00329030, 0.00329054,
     &   0.00329126, 0.00329203, 0.00328997, 0.00328539, 0.00328084,
     &   0.00327255, 0.00325503, 0.00323760, 0.00321988, 0.00317981,
     &   0.00314022, 0.00310113, 0.00304452, 0.00298321, 0.00292311,
     &   0.00284231, 0.00274328, 0.00264768, 0.00255143, 0.00244715,
     &   0.00234711, 0.00225115, 0.00215691, 0.00206657, 0.00198000/
      data (wwi(i,2),i=1,mxwv)/
     &   0.99999857, 0.99999857, 0.99999875, 0.99999881, 0.99999887,
     &   0.99999899, 0.99999911, 0.99999911, 0.99999917, 0.99999928,
     &   0.99999934, 0.99999940, 0.99999946, 0.99999946, 0.99999958,
     &   0.99999952, 0.99999970, 0.99999970, 0.99999976, 0.99999976,
     &   0.99999976, 0.99999982, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999988, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999982, 0.99999976, 0.99999976, 0.99999976,
     &   0.99999970, 0.99999964, 0.99999958, 0.99999952, 0.99999940,
     &   0.99999928, 0.99999917, 0.99999905, 0.99999893, 0.99999881,
     &   0.99999851, 0.99999821, 0.99999797, 0.99999768, 0.99999714,
     &   0.99999636, 0.99999547, 0.99999458, 0.99999404, 0.99999398,
     &   0.99999386, 0.99999243, 0.99999017, 0.99998653, 0.99998426,
     &   0.99998301, 0.99998128, 0.99997902, 0.99997139, 0.99996084,
     &   0.99994707, 0.99992895, 0.99992466, 0.99993318, 0.99994421,
     &   0.99994755, 0.99994767, 0.99994075, 0.99993479, 0.99989200,
     &   0.99983996, 0.99975973, 0.99969667, 0.99966770, 0.99966949,
     &   0.99968088, 0.99968839, 0.99968231, 0.99966341, 0.99958628,
     &   0.99876487, 0.99647784, 0.99190092, 0.98964989, 0.99089575,
     &   0.99252105, 0.99436688, 0.99552828, 0.99601245, 0.99664521,
     &   0.99719727, 0.99764299, 0.99789530, 0.99806041, 0.99823797,
     &   0.99827814, 0.99704534, 0.99225670, 0.98400348, 0.98073733,
     &   0.98216856, 0.98544854, 0.99039799, 0.99455929, 0.99659723,
     &   0.99752426, 0.99788558, 0.99741632, 0.99615049, 0.99494487,
     &   0.99420083, 0.99343139, 0.99251938, 0.99199682, 0.99095076,
     &   0.98160130, 0.95582223, 0.88522732, 0.67619228, 0.29152361,
     &   0.30551463, 0.40425956, 0.43248120, 0.46102765, 0.47438729,
     &   0.47632262, 0.48597237, 0.52856725, 0.61600667, 0.72851557,
     &   0.82838976, 0.89086848, 0.92989016, 0.94715494, 0.95280212,
     &   0.95711452, 0.95696872, 0.95239574, 0.94519669, 0.93932480,
     &   0.93305498, 0.92369884, 0.91207933, 0.89209098, 0.86665779,
     &   0.84825599, 0.83940762, 0.87309009, 0.90115023, 0.91163224,
     &   0.92098618, 0.92509627, 0.92278033, 0.91729969, 0.91045165,
     &   0.89556497, 0.87514788, 0.84795827, 0.81076449, 0.76478988,
     &   0.70064265, 0.66558307, 0.64644486, 0.64964908, 0.66472000,
     &   0.68246102, 0.69751531, 0.69420624, 0.68410349, 0.68372840,
     &   0.68652105, 0.68738174, 0.68729419, 0.69198036, 0.68763900,
     &   0.68373132, 0.68441045, 0.69291615, 0.70569664, 0.71876508,
     &   0.72094256, 0.70096648, 0.68542290, 0.66807580, 0.64745861,
     &   0.62288517, 0.60180908, 0.57857347, 0.55155116, 0.51703018,
     &   0.46143696, 0.37303039, 0.27424991, 0.18889360, 0.14545546,
     &   0.15027069, 0.17136745, 0.19276491, 0.21116352, 0.22661430,
     &   0.24120043, 0.25507790, 0.26893476, 0.28307718, 0.29716191,
     &   0.31212389, 0.32757983, 0.34371549, 0.35946730, 0.37727159,
     &   0.39775208, 0.41893044, 0.44068408, 0.45802507, 0.47314292,
     &   0.48132104, 0.48632586, 0.48909095, 0.48860648, 0.48541075,
     &   0.49035192, 0.50275111, 0.51659834, 0.53041017, 0.54406279,
     &   0.54632390, 0.54339027, 0.52848268, 0.51607549, 0.52268535,
     &   0.54595524, 0.57763642, 0.60558456, 0.61895287, 0.63170552,
     &   0.64394981, 0.65039802, 0.64278865, 0.62817365, 0.60910833,
     &   0.58625871, 0.56245852, 0.53553045, 0.50365341, 0.47014183,
     &   0.43380564, 0.39443886, 0.35618335, 0.31949779, 0.28477150,
     &   0.25235251, 0.22272122, 0.19557977, 0.17097858, 0.14889865,
     &   0.12798254, 0.10847671, 0.09172940, 0.07749474, 0.06433093,
     &   0.05217541, 0.04322435, 0.03692795, 0.03291101, 0.03024849,
     &   0.02853328, 0.02758407, 0.02718678, 0.02708058, 0.02715923,
     &   0.02735126, 0.02766037, 0.02807737, 0.02890508, 0.03069258,
     &   0.03221051, 0.03452437, 0.03815779, 0.04311199, 0.04606583,
     &   0.04584938, 0.04457941, 0.04362706, 0.04332679, 0.04362273,
     &   0.04418838, 0.04561748, 0.04829028, 0.04899029, 0.04769443,
     &   0.03803721, 0.03016800, 0.02594159, 0.02495557, 0.02608044,
     &   0.02667335, 0.02729511, 0.02814514, 0.02934947, 0.03043726,
     &   0.03111591, 0.03158483, 0.03185226, 0.03218313, 0.03236684,
     &   0.03260928, 0.03296650, 0.03345834, 0.03402412, 0.03422131,
     &   0.03435063, 0.03358199, 0.03257618, 0.03167496, 0.03069100,
     &   0.02973729, 0.02880177, 0.02788912, 0.02700499, 0.02602631,
     &   0.02507834, 0.02414792, 0.02317596, 0.02224196, 0.02134454,
     &   0.02048236, 0.01965409, 0.01885850, 0.01809435, 0.01736046,
     &   0.01665569, 0.01597895, 0.01541959, 0.01489571, 0.01438951,
     &   0.01390040, 0.01342782, 0.01297121, 0.01253005, 0.01210381,
     &   0.01169199, 0.01129412, 0.01090973, 0.01053837, 0.01019826,
     &   0.00993718, 0.00968292, 0.00943530, 0.00919412, 0.00895921,
     &   0.00876067, 0.00876403, 0.00876823, 0.00877319, 0.00879479,
     &   0.00882859, 0.00886317, 0.00892047, 0.00902933, 0.00914010,
     &   0.00924520, 0.00924082, 0.00923664, 0.00923263, 0.00923394,
     &   0.00923659, 0.00923934, 0.00923414, 0.00922192, 0.00920974,
     &   0.00918713, 0.00913872, 0.00909055, 0.00904153, 0.00893008,
     &   0.00881996, 0.00871115, 0.00855340, 0.00838242, 0.00821479,
     &   0.00798918, 0.00771251, 0.00744530, 0.00717614, 0.00688438,
     &   0.00660437, 0.00633565, 0.00607163, 0.00581846, 0.00557577/
      data (wwi(i,3),i=1,mxwv)/
     &   0.99999797, 0.99999809, 0.99999821, 0.99999803, 0.99999851,
     &   0.99999863, 0.99999875, 0.99999887, 0.99999887, 0.99999899,
     &   0.99999911, 0.99999911, 0.99999923, 0.99999928, 0.99999946,
     &   0.99999952, 0.99999958, 0.99999958, 0.99999964, 0.99999964,
     &   0.99999970, 0.99999976, 0.99999976, 0.99999976, 0.99999982,
     &   0.99999976, 0.99999982, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999982, 0.99999976, 0.99999976, 0.99999976,
     &   0.99999970, 0.99999964, 0.99999970, 0.99999970, 0.99999964,
     &   0.99999958, 0.99999946, 0.99999946, 0.99999928, 0.99999917,
     &   0.99999899, 0.99999875, 0.99999857, 0.99999845, 0.99999827,
     &   0.99999785, 0.99999738, 0.99999702, 0.99999648, 0.99999589,
     &   0.99999470, 0.99999338, 0.99999213, 0.99999136, 0.99999124,
     &   0.99999112, 0.99998885, 0.99998564, 0.99998027, 0.99997681,
     &   0.99997509, 0.99997258, 0.99996936, 0.99995804, 0.99994266,
     &   0.99992245, 0.99989593, 0.99988919, 0.99990171, 0.99991786,
     &   0.99992269, 0.99992281, 0.99991250, 0.99990356, 0.99983984,
     &   0.99976224, 0.99964201, 0.99954683, 0.99950188, 0.99950266,
     &   0.99951786, 0.99952716, 0.99951559, 0.99948412, 0.99936241,
     &   0.99808753, 0.99452692, 0.98739529, 0.98381990, 0.98565078,
     &   0.98810995, 0.99096310, 0.99276608, 0.99350077, 0.99449104,
     &   0.99536479, 0.99607617, 0.99647552, 0.99673522, 0.99702120,
     &   0.99707931, 0.99498278, 0.98689884, 0.97315419, 0.96781981,
     &   0.97025156, 0.97575086, 0.98401922, 0.99098343, 0.99440980,
     &   0.99598193, 0.99662071, 0.99594933, 0.99409890, 0.99245036,
     &   0.99158871, 0.99079233, 0.98992407, 0.98971587, 0.98902500,
     &   0.97933984, 0.95467716, 0.89368397, 0.71991342, 0.35597938,
     &   0.36294839, 0.44760695, 0.46896651, 0.49101466, 0.49698979,
     &   0.49195448, 0.48743597, 0.49699655, 0.54460973, 0.63827211,
     &   0.74900037, 0.83253992, 0.89060301, 0.91839516, 0.92872238,
     &   0.93694675, 0.93857855, 0.93416202, 0.92662853, 0.92106938,
     &   0.91534007, 0.90619451, 0.89491481, 0.87470204, 0.84903949,
     &   0.83077210, 0.82196420, 0.85876620, 0.89044076, 0.90366453,
     &   0.91534114, 0.92132443, 0.92058009, 0.91668010, 0.91159970,
     &   0.89901149, 0.88158673, 0.85832006, 0.82625198, 0.78655219,
     &   0.72958285, 0.69650561, 0.67804277, 0.67992616, 0.69364184,
     &   0.71069682, 0.72671425, 0.72591513, 0.71818978, 0.71909010,
     &   0.72309375, 0.72546870, 0.72690678, 0.73287809, 0.73078024,
     &   0.72883052, 0.73084658, 0.73999798, 0.75320184, 0.76695657,
     &   0.77141821, 0.75637853, 0.74496543, 0.73193198, 0.71595711,
     &   0.69626629, 0.67957819, 0.66102076, 0.63887697, 0.60961592,
     &   0.55929643, 0.47209683, 0.36490318, 0.26295909, 0.20659688,
     &   0.21099554, 0.23535700, 0.25928351, 0.27904636, 0.29518691,
     &   0.31003377, 0.32392046, 0.33758909, 0.35139573, 0.36531287,
     &   0.38007131, 0.39543122, 0.41194376, 0.42826164, 0.44675690,
     &   0.46836591, 0.49131596, 0.51538950, 0.53545219, 0.55346936,
     &   0.56479460, 0.57283747, 0.57852614, 0.58094192, 0.58024454,
     &   0.58708692, 0.60119194, 0.61704695, 0.63285345, 0.64830089,
     &   0.65328091, 0.65354687, 0.64248669, 0.63312858, 0.64125800,
     &   0.66474921, 0.69542569, 0.72215962, 0.73601115, 0.74902600,
     &   0.76131314, 0.76877147, 0.76512146, 0.75587291, 0.74292994,
     &   0.72666281, 0.70924449, 0.68871361, 0.66313326, 0.63502032,
     &   0.60288113, 0.56592357, 0.52793407, 0.48935369, 0.45066297,
     &   0.41241905, 0.37551603, 0.33983079, 0.30575699, 0.27362531,
     &   0.24161421, 0.21024558, 0.18209258, 0.15719207, 0.13318883,
     &   0.11010216, 0.09258837, 0.07999343, 0.07189626, 0.06652693,
     &   0.06308193, 0.06121970, 0.06053429, 0.06047411, 0.06081149,
     &   0.06139238, 0.06222116, 0.06327982, 0.06519528, 0.06912104,
     &   0.07256513, 0.07764237, 0.08548961, 0.09631079, 0.10336585,
     &   0.10392884, 0.10195810, 0.10053621, 0.10045928, 0.10167964,
     &   0.10350128, 0.10723408, 0.11390983, 0.11633753, 0.11433665,
     &   0.09292936, 0.07471169, 0.06471749, 0.06239028, 0.06528799,
     &   0.06691725, 0.06862637, 0.07089777, 0.07404611, 0.07693660,
     &   0.07883425, 0.08021669, 0.08110239, 0.08212784, 0.08278591,
     &   0.08357335, 0.08463445, 0.08601744, 0.08756842, 0.08821471,
     &   0.08868483, 0.08694308, 0.08459746, 0.08248746, 0.08015265,
     &   0.07787441, 0.07562384, 0.07341421, 0.07126124, 0.06885435,
     &   0.06651006, 0.06419604, 0.06176138, 0.05941038, 0.05714085,
     &   0.05495062, 0.05283751, 0.05079934, 0.04883397, 0.04693926,
     &   0.04511309, 0.04335339, 0.04189736, 0.04053020, 0.03920541,
     &   0.03792183, 0.03667833, 0.03547380, 0.03430715, 0.03317729,
     &   0.03208319, 0.03102382, 0.02999816, 0.02900523, 0.02809460,
     &   0.02739607, 0.02671467, 0.02604999, 0.02540161, 0.02476914,
     &   0.02423464, 0.02424949, 0.02426641, 0.02428515, 0.02434907,
     &   0.02444605, 0.02454499, 0.02470569, 0.02500676, 0.02531286,
     &   0.02560330, 0.02559537, 0.02558784, 0.02558063, 0.02558776,
     &   0.02559840, 0.02560920, 0.02559821, 0.02556794, 0.02553769,
     &   0.02547883, 0.02534947, 0.02522062, 0.02508932, 0.02478738,
     &   0.02448880, 0.02419355, 0.02376417, 0.02329820, 0.02284097,
     &   0.02222438, 0.02146698, 0.02073468, 0.01999615, 0.01919451,
     &   0.01842428, 0.01768429, 0.01695644, 0.01625776, 0.01558734/
      data (wwi(i,4),i=1,mxwv)/
     &   0.99999690, 0.99999720, 0.99999750, 0.99999779, 0.99999803,
     &   0.99999803, 0.99999827, 0.99999839, 0.99999851, 0.99999863,
     &   0.99999875, 0.99999845, 0.99999899, 0.99999911, 0.99999923,
     &   0.99999934, 0.99999940, 0.99999946, 0.99999952, 0.99999952,
     &   0.99999958, 0.99999958, 0.99999964, 0.99999970, 0.99999970,
     &   0.99999976, 0.99999976, 0.99999970, 0.99999976, 0.99999976,
     &   0.99999970, 0.99999970, 0.99999970, 0.99999970, 0.99999940,
     &   0.99999958, 0.99999940, 0.99999958, 0.99999958, 0.99999940,
     &   0.99999940, 0.99999934, 0.99999923, 0.99999905, 0.99999881,
     &   0.99999863, 0.99999839, 0.99999809, 0.99999785, 0.99999696,
     &   0.99999678, 0.99999642, 0.99999577, 0.99999535, 0.99999410,
     &   0.99999255, 0.99998999, 0.99998891, 0.99998713, 0.99998754,
     &   0.99998719, 0.99998409, 0.99997950, 0.99997163, 0.99996680,
     &   0.99996382, 0.99995965, 0.99995548, 0.99993813, 0.99991578,
     &   0.99988824, 0.99985093, 0.99984014, 0.99985707, 0.99988145,
     &   0.99988663, 0.99988759, 0.99987274, 0.99985945, 0.99976832,
     &   0.99965429, 0.99948084, 0.99934053, 0.99927515, 0.99927640,
     &   0.99929756, 0.99931014, 0.99929345, 0.99924713, 0.99906850,
     &   0.99721551, 0.99208367, 0.98192412, 0.97684270, 0.97937059,
     &   0.98281169, 0.98685193, 0.98941767, 0.99045193, 0.99186695,
     &   0.99312198, 0.99414700, 0.99471527, 0.99507779, 0.99548221,
     &   0.99554276, 0.99230957, 0.97992903, 0.95904315, 0.95073581,
     &   0.95400792, 0.96202672, 0.97457939, 0.98543733, 0.99086004,
     &   0.99336326, 0.99436998, 0.99320835, 0.99007154, 0.98728120,
     &   0.98584831, 0.98460054, 0.98336112, 0.98338550, 0.98292482,
     &   0.97006178, 0.94111669, 0.88206238, 0.73682249, 0.41387951,
     &   0.41163549, 0.47926876, 0.49554500, 0.51364166, 0.51610518,
     &   0.50881821, 0.50125271, 0.49812537, 0.51700819, 0.57900298,
     &   0.67619991, 0.76468641, 0.83548468, 0.87218833, 0.88618559,
     &   0.89802551, 0.90050828, 0.89422029, 0.88380665, 0.87681055,
     &   0.87008601, 0.85910672, 0.84601504, 0.82221764, 0.79289371,
     &   0.77284414, 0.76324528, 0.80773807, 0.84883595, 0.86819023,
     &   0.88546151, 0.89544320, 0.89689618, 0.89454031, 0.89102691,
     &   0.87925756, 0.86280787, 0.84099406, 0.81106561, 0.77481395,
     &   0.72218657, 0.68974519, 0.67137045, 0.67147094, 0.68371469,
     &   0.70044667, 0.71844357, 0.72065336, 0.71530974, 0.71771610,
     &   0.72334802, 0.72764039, 0.73094314, 0.73880970, 0.73922181,
     &   0.73941463, 0.74306631, 0.75353950, 0.76819772, 0.78389245,
     &   0.79140508, 0.78081620, 0.77328527, 0.76433641, 0.75281018,
     &   0.73791057, 0.72579348, 0.71238667, 0.69596606, 0.67353213,
     &   0.63192511, 0.55345327, 0.44839990, 0.33894250, 0.27287275,
     &   0.27556446, 0.30030939, 0.32371524, 0.34214598, 0.35663515,
     &   0.36951265, 0.38123545, 0.39251095, 0.40369707, 0.41503990,
     &   0.42704034, 0.43965736, 0.45376852, 0.46803796, 0.48449513,
     &   0.50442481, 0.52665740, 0.55099648, 0.57250202, 0.59256524,
     &   0.60670698, 0.61759663, 0.62606800, 0.63132489, 0.63305616,
     &   0.64126319, 0.65623486, 0.67331511, 0.69038224, 0.70690686,
     &   0.71409214, 0.71709162, 0.70965403, 0.70306194, 0.71170419,
     &   0.73397452, 0.76244360, 0.78726590, 0.80096114, 0.81366360,
     &   0.82548577, 0.83327550, 0.83231491, 0.82695627, 0.81870323,
     &   0.80781567, 0.79599196, 0.78162980, 0.76304656, 0.74208182,
     &   0.71727872, 0.68759894, 0.65599281, 0.62264925, 0.58782929,
     &   0.55193180, 0.51584721, 0.47936654, 0.44293183, 0.40699750,
     &   0.36943814, 0.33074740, 0.29433805, 0.26066911, 0.22658072,
     &   0.19208945, 0.16487387, 0.14470139, 0.13158947, 0.12289254,
     &   0.11734834, 0.11445559, 0.11360842, 0.11385804, 0.11480498,
     &   0.11617858, 0.11797658, 0.12016739, 0.12377104, 0.13066253,
     &   0.13682570, 0.14553519, 0.15855931, 0.17631584, 0.18879677,
     &   0.19171400, 0.19014901, 0.18923990, 0.19041193, 0.19376323,
     &   0.19819941, 0.20589019, 0.21927404, 0.22626847, 0.22620827,
     &   0.19080292, 0.15767540, 0.13861781, 0.13399532, 0.14017294,
     &   0.14396426, 0.14797883, 0.15316658, 0.16021024, 0.16695239,
     &   0.17180833, 0.17567647, 0.17857942, 0.18172130, 0.18412407,
     &   0.18673681, 0.18989514, 0.19368844, 0.19775851, 0.20000853,
     &   0.20185278, 0.19916102, 0.19513771, 0.19148555, 0.18726942,
     &   0.18308161, 0.17886376, 0.17465101, 0.17048416, 0.16569348,
     &   0.16096187, 0.15622249, 0.15113990, 0.14617294, 0.14132275,
     &   0.13659006, 0.13197535, 0.12747873, 0.12310009, 0.11883899,
     &   0.11469480, 0.11066666, 0.10732517, 0.10416777, 0.10108624,
     &   0.09807993, 0.09514812, 0.09229001, 0.08950470, 0.08679126,
     &   0.08414867, 0.08157586, 0.07907175, 0.07663515, 0.07439242,
     &   0.07267463, 0.07099206, 0.06934420, 0.06773057, 0.06615065,
     &   0.06481542, 0.06488576, 0.06495991, 0.06503740, 0.06522968,
     &   0.06550568, 0.06578565, 0.06622311, 0.06701973, 0.06782790,
     &   0.06859459, 0.06859595, 0.06859744, 0.06859890, 0.06863630,
     &   0.06868194, 0.06872728, 0.06871604, 0.06865463, 0.06859259,
     &   0.06845646, 0.06813845, 0.06782097, 0.06749639, 0.06673151,
     &   0.06597377, 0.06522313, 0.06412415, 0.06292796, 0.06175176,
     &   0.06015848, 0.05819366, 0.05628844, 0.05436117, 0.05226212,
     &   0.05023926, 0.04829020, 0.04636759, 0.04451698, 0.04273654/
      data (wwi(i,5),i=1,mxwv)/
     &   0.99999571, 0.99999642, 0.99999648, 0.99999684, 0.99999720,
     &   0.99999738, 0.99999762, 0.99999779, 0.99999779, 0.99999803,
     &   0.99999821, 0.99999839, 0.99999857, 0.99999881, 0.99999887,
     &   0.99999911, 0.99999917, 0.99999928, 0.99999928, 0.99999934,
     &   0.99999946, 0.99999952, 0.99999952, 0.99999952, 0.99999964,
     &   0.99999970, 0.99999970, 0.99999970, 0.99999970, 0.99999964,
     &   0.99999964, 0.99999964, 0.99999958, 0.99999958, 0.99999952,
     &   0.99999946, 0.99999928, 0.99999940, 0.99999940, 0.99999934,
     &   0.99999923, 0.99999911, 0.99999881, 0.99999857, 0.99999839,
     &   0.99999809, 0.99999774, 0.99999738, 0.99999696, 0.99999666,
     &   0.99999583, 0.99999493, 0.99999434, 0.99999350, 0.99999171,
     &   0.99998987, 0.99998730, 0.99998468, 0.99998164, 0.99998093,
     &   0.99998200, 0.99997848, 0.99996895, 0.99996084, 0.99995434,
     &   0.99994832, 0.99994546, 0.99993604, 0.99991548, 0.99988425,
     &   0.99984282, 0.99978954, 0.99977040, 0.99980152, 0.99983352,
     &   0.99984425, 0.99983877, 0.99981791, 0.99980444, 0.99966788,
     &   0.99951112, 0.99926293, 0.99907678, 0.99897170, 0.99897444,
     &   0.99899608, 0.99902570, 0.99898279, 0.99892312, 0.99866891,
     &   0.99605852, 0.98890817, 0.97494942, 0.96801221, 0.97140336,
     &   0.97605962, 0.98157442, 0.98509455, 0.98650831, 0.98846227,
     &   0.99020374, 0.99163312, 0.99242485, 0.99292731, 0.99349242,
     &   0.99356771, 0.98894376, 0.97150588, 0.94273639, 0.93136054,
     &   0.93558359, 0.94632095, 0.96356243, 0.97882515, 0.98657000,
     &   0.99016881, 0.99159712, 0.98979926, 0.98499626, 0.98061687,
     &   0.97818506, 0.97593158, 0.97357845, 0.97312975, 0.97190785,
     &   0.95086074, 0.90789688, 0.83981323, 0.72004807, 0.45874456,
     &   0.44906506, 0.50140125, 0.51464319, 0.53070879, 0.53179508,
     &   0.52394485, 0.51614994, 0.50962806, 0.51396328, 0.54971373,
     &   0.62666029, 0.71087909, 0.78647399, 0.82764715, 0.84271610,
     &   0.85554725, 0.85658431, 0.84601110, 0.82989573, 0.81857991,
     &   0.80777073, 0.79167807, 0.77336162, 0.74265110, 0.70754492,
     &   0.68505913, 0.67468548, 0.72447717, 0.77499813, 0.80053198,
     &   0.82440317, 0.83908021, 0.84234589, 0.84070975, 0.83797604,
     &   0.82504177, 0.80733538, 0.78470600, 0.75474375, 0.72060037,
     &   0.67216825, 0.64120466, 0.62383342, 0.62203586, 0.63189358,
     &   0.64734155, 0.66685683, 0.67229587, 0.66956961, 0.67358196,
     &   0.68103147, 0.68755323, 0.69306958, 0.70341456, 0.70685804,
     &   0.70963466, 0.71543026, 0.72813290, 0.74570233, 0.76533687,
     &   0.77763504, 0.77170604, 0.76840264, 0.76379889, 0.75686824,
     &   0.74678254, 0.73950177, 0.73167652, 0.72153169, 0.70678067,
     &   0.67549229, 0.60986775, 0.51498902, 0.40781400, 0.33724284,
     &   0.33723295, 0.35952792, 0.37977231, 0.39490563, 0.40618727,
     &   0.41576651, 0.42408228, 0.43174544, 0.43905476, 0.44628635,
     &   0.45382908, 0.46176884, 0.47102141, 0.48070803, 0.49227959,
     &   0.50720692, 0.52532303, 0.54683006, 0.56768090, 0.58847594,
     &   0.60496897, 0.61862314, 0.63001370, 0.63836402, 0.64272326,
     &   0.65233964, 0.66831923, 0.68709874, 0.70611846, 0.72453362,
     &   0.73447275, 0.74055976, 0.73656994, 0.73248005, 0.74178505,
     &   0.76364505, 0.79143792, 0.81604803, 0.83054161, 0.84383142,
     &   0.85603762, 0.86458528, 0.86570859, 0.86310881, 0.85811287,
     &   0.85097188, 0.84311551, 0.83324993, 0.81998068, 0.80474705,
     &   0.78626466, 0.76351523, 0.73879498, 0.71211076, 0.68352199,
     &   0.65322232, 0.62195462, 0.58931011, 0.55558532, 0.52113807,
     &   0.48374462, 0.44360363, 0.40421996, 0.36629352, 0.32608670,
     &   0.28334799, 0.24818780, 0.22122918, 0.20343685, 0.19158533,
     &   0.18402475, 0.18014783, 0.17921408, 0.17987368, 0.18154389,
     &   0.18383326, 0.18671885, 0.19014981, 0.19542804, 0.20489724,
     &   0.21337272, 0.22487064, 0.24133372, 0.26297423, 0.27875537,
     &   0.28428361, 0.28419828, 0.28468490, 0.28755471, 0.29312277,
     &   0.30013442, 0.31116003, 0.33031642, 0.34328014, 0.34907261,
     &   0.30852255, 0.26479188, 0.23780739, 0.23057866, 0.24008158,
     &   0.24634682, 0.25304750, 0.26153952, 0.27294552, 0.28445965,
     &   0.29357123, 0.30147812, 0.30820724, 0.31530622, 0.32148260,
     &   0.32795537, 0.33537555, 0.34385279, 0.35266131, 0.35899734,
     &   0.36469665, 0.36362508, 0.36036927, 0.35740325, 0.35335019,
     &   0.34914801, 0.34468633, 0.34003326, 0.33526844, 0.32929918,
     &   0.32322305, 0.31692597, 0.30981618, 0.30269742, 0.29557979,
     &   0.28847313, 0.28138712, 0.27433103, 0.26731384, 0.26034418,
     &   0.25343025, 0.24657983, 0.24089096, 0.23544154, 0.23003358,
     &   0.22467196, 0.21936122, 0.21410558, 0.20890896, 0.20377502,
     &   0.19870704, 0.19370808, 0.18878087, 0.18392788, 0.17942446,
     &   0.17599539, 0.17260179, 0.16924505, 0.16592643, 0.16264707,
     &   0.15987764, 0.16021007, 0.16054167, 0.16087195, 0.16144976,
     &   0.16220631, 0.16296443, 0.16406555, 0.16595612, 0.16786337,
     &   0.16967043, 0.16978228, 0.16988923, 0.16999124, 0.17016809,
     &   0.17035893, 0.17054509, 0.17060238, 0.17054501, 0.17048278,
     &   0.17025302, 0.16961657, 0.16897723, 0.16831829, 0.16667424,
     &   0.16503865, 0.16341168, 0.16099398, 0.15834458, 0.15572689,
     &   0.15214501, 0.14768814, 0.14333655, 0.13890286, 0.13403505,
     &   0.12930973, 0.12472465, 0.12016995, 0.11575629, 0.11148240/
      data (wwi(i,6),i=1,mxwv)/
     &   0.99999470, 0.99999493, 0.99999529, 0.99999577, 0.99999601,
     &   0.99999636, 0.99999666, 0.99999690, 0.99999708, 0.99999720,
     &   0.99999726, 0.99999785, 0.99999809, 0.99999833, 0.99999857,
     &   0.99999869, 0.99999881, 0.99999893, 0.99999905, 0.99999899,
     &   0.99999923, 0.99999928, 0.99999934, 0.99999934, 0.99999946,
     &   0.99999952, 0.99999958, 0.99999952, 0.99999952, 0.99999952,
     &   0.99999952, 0.99999946, 0.99999940, 0.99999940, 0.99999934,
     &   0.99999928, 0.99999899, 0.99999923, 0.99999917, 0.99999911,
     &   0.99999881, 0.99999863, 0.99999845, 0.99999815, 0.99999750,
     &   0.99999738, 0.99999696, 0.99999636, 0.99999589, 0.99999535,
     &   0.99999434, 0.99999315, 0.99999124, 0.99999118, 0.99998873,
     &   0.99998593, 0.99997866, 0.99997890, 0.99997705, 0.99997544,
     &   0.99997610, 0.99996948, 0.99996066, 0.99994707, 0.99993730,
     &   0.99993157, 0.99991971, 0.99991482, 0.99988270, 0.99983984,
     &   0.99977517, 0.99971038, 0.99968499, 0.99972385, 0.99976933,
     &   0.99977845, 0.99977899, 0.99975592, 0.99972248, 0.99954194,
     &   0.99933481, 0.99897343, 0.99871051, 0.99859446, 0.99858415,
     &   0.99861479, 0.99864805, 0.99860424, 0.99852669, 0.99816507,
     &   0.99455482, 0.98479646, 0.96588796, 0.95657581, 0.96110421,
     &   0.96734828, 0.97477561, 0.97953320, 0.98143697, 0.98409063,
     &   0.98644549, 0.98839158, 0.98945570, 0.99013281, 0.99089706,
     &   0.99098766, 0.98457909, 0.96081322, 0.92256647, 0.90768814,
     &   0.91307044, 0.92700070, 0.94980031, 0.97040492, 0.98103279,
     &   0.98601687, 0.98799670, 0.98542607, 0.97862685, 0.97241724,
     &   0.96890450, 0.96558833, 0.96204346, 0.96104026, 0.95868099,
     &   0.92679322, 0.86155975, 0.76680946, 0.66494864, 0.48639616,
     &   0.47543660, 0.51664793, 0.52835000, 0.54348129, 0.54419661,
     &   0.53627580, 0.52858984, 0.52130926, 0.51886147, 0.53458047,
     &   0.58759916, 0.66168153, 0.73852605, 0.78354228, 0.80012691,
     &   0.81426316, 0.81454360, 0.80098510, 0.78090227, 0.76648808,
     &   0.75255865, 0.73265558, 0.71040064, 0.67549634, 0.63823324,
     &   0.61578643, 0.60597807, 0.65280223, 0.70466685, 0.73121166,
     &   0.75716645, 0.77285588, 0.77473134, 0.77062821, 0.76531208,
     &   0.74761462, 0.72512829, 0.69861865, 0.66676211, 0.63446152,
     &   0.59389639, 0.56960905, 0.55673379, 0.55453837, 0.56084377,
     &   0.57206208, 0.58816493, 0.59403747, 0.59299392, 0.59712553,
     &   0.60441583, 0.61135203, 0.61758089, 0.62852359, 0.63390082,
     &   0.63847178, 0.64560992, 0.65956461, 0.67935747, 0.70303953,
     &   0.72100908, 0.72032279, 0.72196585, 0.72247726, 0.72091335,
     &   0.71633983, 0.71476835, 0.71360534, 0.71092457, 0.70527965,
     &   0.68601847, 0.63612288, 0.55740422, 0.46161699, 0.39287212,
     &   0.39002362, 0.40812832, 0.42393029, 0.43520954, 0.44303995,
     &   0.44933280, 0.45436689, 0.45865315, 0.46240443, 0.46564230,
     &   0.46882913, 0.47201312, 0.47570330, 0.47973055, 0.48482683,
     &   0.49207604, 0.50211489, 0.51577210, 0.53105390, 0.54814214,
     &   0.56376076, 0.57796586, 0.59077215, 0.60134566, 0.60793483,
     &   0.61839724, 0.63483304, 0.65524292, 0.67672747, 0.69802028,
     &   0.71176857, 0.72195435, 0.72177136, 0.72028512, 0.73094249,
     &   0.75412762, 0.78407747, 0.81161261, 0.82913148, 0.84511274,
     &   0.85966516, 0.87041372, 0.87397343, 0.87389642, 0.87155199,
     &   0.86725479, 0.86238152, 0.85582477, 0.84639537, 0.83537102,
     &   0.82162863, 0.80426514, 0.78522706, 0.76444346, 0.74186999,
     &   0.71757591, 0.69220263, 0.66514659, 0.63653392, 0.60655469,
     &   0.57309783, 0.53601402, 0.49832359, 0.46069992, 0.41906238,
     &   0.37263796, 0.33271679, 0.30088645, 0.27937797, 0.26482558,
     &   0.25533783, 0.25028342, 0.24890468, 0.24953105, 0.25142017,
     &   0.25408113, 0.25745609, 0.26147342, 0.26750445, 0.27794579,
     &   0.28719562, 0.29935321, 0.31606746, 0.33703572, 0.35251474,
     &   0.35933667, 0.36060739, 0.36223540, 0.36603218, 0.37248543,
     &   0.38044843, 0.39232376, 0.41345140, 0.43048289, 0.44250435,
     &   0.40698865, 0.36182776, 0.33176336, 0.32241929, 0.33299115,
     &   0.34023628, 0.34804645, 0.35785949, 0.37104481, 0.38497403,
     &   0.39669532, 0.40740710, 0.41716969, 0.42739931, 0.43691245,
     &   0.44680214, 0.45804214, 0.47075862, 0.48385271, 0.49482587,
     &   0.50527024, 0.50847465, 0.50923514, 0.51010859, 0.50962573,
     &   0.50891387, 0.50777513, 0.50629973, 0.50459951, 0.50136667,
     &   0.49787045, 0.49393702, 0.48874405, 0.48336250, 0.47779551,
     &   0.47204712, 0.46612185, 0.46002507, 0.45376274, 0.44734144,
     &   0.44076836, 0.43405131, 0.42863321, 0.42334911, 0.41794226,
     &   0.41241917, 0.40678668, 0.40105185, 0.39522201, 0.38930461,
     &   0.38330725, 0.37723765, 0.37110358, 0.36491290, 0.35910130,
     &   0.35483652, 0.35051289, 0.34613588, 0.34171087, 0.33724323,
     &   0.33352023, 0.33489147, 0.33621588, 0.33749497, 0.33914840,
     &   0.34106094, 0.34293723, 0.34534496, 0.34901786, 0.35267323,
     &   0.35612217, 0.35677406, 0.35739139, 0.35797581, 0.35865891,
     &   0.35934314, 0.35999951, 0.36042687, 0.36064959, 0.36084765,
     &   0.36075661, 0.35998532, 0.35919341, 0.35835427, 0.35587651,
     &   0.35338816, 0.35089040, 0.34704405, 0.34276804, 0.33850265,
     &   0.33253762, 0.32497725, 0.31749469, 0.30976051, 0.30112940,
     &   0.29262686, 0.28425729, 0.27582169, 0.26753217, 0.25939491/
      data (wwi(i,7),i=1,mxwv)/
     &   0.99999255, 0.99999309, 0.99999279, 0.99999404, 0.99999464,
     &   0.99999493, 0.99999529, 0.99999571, 0.99999601, 0.99999624,
     &   0.99999660, 0.99999696, 0.99999732, 0.99999768, 0.99999803,
     &   0.99999815, 0.99999851, 0.99999857, 0.99999863, 0.99999881,
     &   0.99999893, 0.99999881, 0.99999911, 0.99999911, 0.99999923,
     &   0.99999934, 0.99999940, 0.99999940, 0.99999934, 0.99999934,
     &   0.99999934, 0.99999928, 0.99999917, 0.99999917, 0.99999881,
     &   0.99999899, 0.99999893, 0.99999893, 0.99999893, 0.99999809,
     &   0.99999851, 0.99999815, 0.99999791, 0.99999726, 0.99999696,
     &   0.99999636, 0.99999559, 0.99999505, 0.99999440, 0.99999344,
     &   0.99999213, 0.99998975, 0.99998909, 0.99998736, 0.99998432,
     &   0.99997973, 0.99997497, 0.99996978, 0.99996734, 0.99996513,
     &   0.99996489, 0.99995852, 0.99994528, 0.99992365, 0.99990976,
     &   0.99990642, 0.99989688, 0.99988323, 0.99983799, 0.99977738,
     &   0.99970424, 0.99960148, 0.99957687, 0.99962360, 0.99968183,
     &   0.99970078, 0.99969620, 0.99965364, 0.99961960, 0.99936873,
     &   0.99907732, 0.99860591, 0.99823815, 0.99805254, 0.99803793,
     &   0.99810493, 0.99812859, 0.99808371, 0.99796081, 0.99746168,
     &   0.99253482, 0.97918892, 0.95365870, 0.94124472, 0.94728035,
     &   0.95563769, 0.96564090, 0.97206491, 0.97465980, 0.97823447,
     &   0.98144251, 0.98408943, 0.98554575, 0.98646468, 0.98749626,
     &   0.98761821, 0.97887510, 0.94689888, 0.89689791, 0.87795103,
     &   0.88475353, 0.90256238, 0.93221515, 0.95957702, 0.97388685,
     &   0.98063713, 0.98331034, 0.97973824, 0.97038430, 0.96185732,
     &   0.95698440, 0.95235258, 0.94737762, 0.94578707, 0.94228941,
     &   0.89902043, 0.81365240, 0.69449663, 0.59222209, 0.49761581,
     &   0.49292842, 0.52717292, 0.53816378, 0.55289865, 0.55370462,
     &   0.54589027, 0.53834963, 0.53107214, 0.52626503, 0.52990526,
     &   0.55941999, 0.61633766, 0.68808460, 0.73487920, 0.75284362,
     &   0.76842284, 0.76854503, 0.75321376, 0.73104674, 0.71541166,
     &   0.70052803, 0.67986441, 0.65743524, 0.62418312, 0.59117740,
     &   0.57266456, 0.56503677, 0.60312712, 0.65007526, 0.67530072,
     &   0.70099223, 0.71648216, 0.71713477, 0.71113735, 0.70357615,
     &   0.68258715, 0.65694791, 0.62817121, 0.59623986, 0.56691623,
     &   0.53631592, 0.52160978, 0.51470423, 0.51382250, 0.51731914,
     &   0.52359128, 0.53273183, 0.53557217, 0.53428328, 0.53651512,
     &   0.54083371, 0.54500997, 0.54886252, 0.55618984, 0.55972993,
     &   0.56288838, 0.56812418, 0.57889402, 0.59516090, 0.61620593,
     &   0.63387561, 0.63516796, 0.63910967, 0.64265555, 0.64497447,
     &   0.64507926, 0.64855272, 0.65373802, 0.65876448, 0.66316700,
     &   0.65792406, 0.62723756, 0.57043421, 0.49463302, 0.43491712,
     &   0.43069193, 0.44469216, 0.45644018, 0.46465677, 0.46992609,
     &   0.47397962, 0.47688034, 0.47908401, 0.48074150, 0.48155433,
     &   0.48217729, 0.48255283, 0.48265144, 0.48285612, 0.48334455,
     &   0.48442167, 0.48663741, 0.49080411, 0.49677753, 0.50486189,
     &   0.51362807, 0.52269119, 0.53178519, 0.54018182, 0.54603112,
     &   0.55436724, 0.56756514, 0.58544016, 0.60578239, 0.62730610,
     &   0.64346081, 0.65706599, 0.66060275, 0.66171908, 0.67364097,
     &   0.69859481, 0.73251933, 0.76603180, 0.78947514, 0.81123781,
     &   0.83126754, 0.84689659, 0.85443175, 0.85773653, 0.85846585,
     &   0.85707623, 0.85513622, 0.85158849, 0.84532654, 0.83768040,
     &   0.82760674, 0.81431746, 0.79970288, 0.78367668, 0.76616323,
     &   0.74719232, 0.72741908, 0.70605963, 0.68313330, 0.65870547,
     &   0.63097107, 0.59955651, 0.56675798, 0.53306597, 0.49432448,
     &   0.44919133, 0.40868551, 0.37505969, 0.35174152, 0.33564109,
     &   0.32480785, 0.31867877, 0.31660613, 0.31675884, 0.31829870,
     &   0.32068288, 0.32381678, 0.32761428, 0.33332115, 0.34304449,
     &   0.35147858, 0.36233926, 0.37671486, 0.39353365, 0.40546077,
     &   0.41149923, 0.41327819, 0.41523635, 0.41882813, 0.42459813,
     &   0.43170053, 0.44200581, 0.46139804, 0.47989991, 0.49689630,
     &   0.47187611, 0.43189496, 0.40290672, 0.39200971, 0.40110400,
     &   0.40743762, 0.41437688, 0.42317769, 0.43525502, 0.44871286,
     &   0.46062449, 0.47195303, 0.48277017, 0.49408692, 0.50503623,
     &   0.51637489, 0.52929807, 0.54393607, 0.55893040, 0.57252097,
     &   0.58578253, 0.59237021, 0.59665626, 0.60090667, 0.60380071,
     &   0.60653293, 0.60883886, 0.61083460, 0.61266881, 0.61305010,
     &   0.61322320, 0.61297584, 0.61134511, 0.60955924, 0.60761452,
     &   0.60550702, 0.60323250, 0.60078704, 0.59816658, 0.59536707,
     &   0.59238464, 0.58921552, 0.58717108, 0.58519995, 0.58306015,
     &   0.58074868, 0.57826298, 0.57560080, 0.57276028, 0.56973982,
     &   0.56653851, 0.56315565, 0.55959117, 0.55584544, 0.55237466,
     &   0.55043787, 0.54833579, 0.54606974, 0.54364175, 0.54105383,
     &   0.53919441, 0.54299808, 0.54665148, 0.55015731, 0.55400401,
     &   0.55804962, 0.56195122, 0.56633908, 0.57200646, 0.57752925,
     &   0.58271080, 0.58482081, 0.58681488, 0.58869851, 0.59061348,
     &   0.59246039, 0.59421247, 0.59566271, 0.59684104, 0.59793931,
     &   0.59868443, 0.59867156, 0.59859180, 0.59841979, 0.59648716,
     &   0.59449530, 0.59244704, 0.58891404, 0.58485645, 0.58074236,
     &   0.57469159, 0.56674981, 0.55873573, 0.55026937, 0.54057080,
     &   0.53080964, 0.52099466, 0.51088047, 0.50072724, 0.49054807/
      data (wwi(i,8),i=1,mxwv)/
     &   0.99998969, 0.99999011, 0.99999064, 0.99999171, 0.99999255,
     &   0.99999321, 0.99999326, 0.99999398, 0.99999416, 0.99999475,
     &   0.99999523, 0.99999559, 0.99999630, 0.99999678, 0.99999726,
     &   0.99999756, 0.99999785, 0.99999797, 0.99999815, 0.99999833,
     &   0.99999851, 0.99999863, 0.99999863, 0.99999875, 0.99999899,
     &   0.99999911, 0.99999911, 0.99999911, 0.99999911, 0.99999911,
     &   0.99999899, 0.99999899, 0.99999893, 0.99999887, 0.99999875,
     &   0.99999863, 0.99999851, 0.99999851, 0.99999839, 0.99999809,
     &   0.99999791, 0.99999762, 0.99999702, 0.99999642, 0.99999547,
     &   0.99999434, 0.99999386, 0.99999291, 0.99999046, 0.99999112,
     &   0.99998903, 0.99998671, 0.99998438, 0.99998271, 0.99997705,
     &   0.99997044, 0.99996608, 0.99995780, 0.99995399, 0.99995506,
     &   0.99995351, 0.99994296, 0.99992490, 0.99989623, 0.99988037,
     &   0.99986863, 0.99985689, 0.99983799, 0.99977654, 0.99969500,
     &   0.99959052, 0.99944580, 0.99941140, 0.99947602, 0.99955958,
     &   0.99958712, 0.99958628, 0.99952239, 0.99947113, 0.99913001,
     &   0.99871629, 0.99807000, 0.99755913, 0.99731642, 0.99732220,
     &   0.99740362, 0.99743330, 0.99735701, 0.99718463, 0.99649984,
     &   0.98973489, 0.97147441, 0.93715960, 0.92077148, 0.92872870,
     &   0.93981159, 0.95318222, 0.96185178, 0.96535343, 0.97021794,
     &   0.97458816, 0.97817659, 0.98017842, 0.98142260, 0.98285961,
     &   0.98300296, 0.97111094, 0.92832935, 0.86407232, 0.84062225,
     &   0.84899884, 0.87119746, 0.90914834, 0.94515944, 0.96435314,
     &   0.97348422, 0.97710621, 0.97223204, 0.95956433, 0.94808847,
     &   0.94153589, 0.93530297, 0.92860019, 0.92631423, 0.92141187,
     &   0.86506021, 0.76146209, 0.63426930, 0.54093045, 0.49880630,
     &   0.50427961, 0.53445405, 0.54512352, 0.55971110, 0.56081378,
     &   0.55318195, 0.54579127, 0.53867877, 0.53338677, 0.53170085,
     &   0.54376727, 0.58018988, 0.63926029, 0.68360615, 0.70164955,
     &   0.71776003, 0.71779943, 0.70174044, 0.67921114, 0.66379941,
     &   0.64950556, 0.63034439, 0.61045742, 0.58285069, 0.55793810,
     &   0.54520673, 0.54032588, 0.56664348, 0.60397708, 0.62584710,
     &   0.64937234, 0.66401261, 0.66435808, 0.65818608, 0.65048957,
     &   0.63027310, 0.60662174, 0.58148426, 0.55559272, 0.53381413,
     &   0.51432484, 0.50710350, 0.50429356, 0.50439656, 0.50619602,
     &   0.50914401, 0.51322502, 0.51393235, 0.51264983, 0.51343077,
     &   0.51527661, 0.51698190, 0.51853931, 0.52208418, 0.52324367,
     &   0.52431196, 0.52681571, 0.53290975, 0.54281521, 0.55638176,
     &   0.56757390, 0.56618935, 0.56737971, 0.56857443, 0.56924862,
     &   0.56883585, 0.57166517, 0.57680058, 0.58326954, 0.59168983,
     &   0.59642875, 0.58529270, 0.55470306, 0.50597143, 0.46227494,
     &   0.45945811, 0.47085798, 0.47995150, 0.48642102, 0.49033925,
     &   0.49333930, 0.49531069, 0.49668834, 0.49759364, 0.49753919,
     &   0.49733704, 0.49686509, 0.49581558, 0.49480954, 0.49381322,
     &   0.49267083, 0.49157387, 0.49093556, 0.49108028, 0.49226460,
     &   0.49412802, 0.49659708, 0.49952355, 0.50251901, 0.50472307,
     &   0.50845939, 0.51507962, 0.52498704, 0.53747195, 0.55205041,
     &   0.56384617, 0.57457417, 0.57787728, 0.57927573, 0.58936965,
     &   0.61141610, 0.64414960, 0.67987311, 0.70743668, 0.73464185,
     &   0.76105893, 0.78313774, 0.79589939, 0.80359745, 0.80823928,
     &   0.81041080, 0.81208432, 0.81208408, 0.80922586, 0.80508190,
     &   0.79856831, 0.78897113, 0.77838892, 0.76673365, 0.75392240,
     &   0.73998976, 0.72571653, 0.71008360, 0.69305760, 0.67463416,
     &   0.65347159, 0.62910873, 0.60306448, 0.57563728, 0.54289728,
     &   0.50314498, 0.46602255, 0.43400630, 0.41130754, 0.39536557,
     &   0.38433525, 0.37778610, 0.37524134, 0.37491146, 0.37595445,
     &   0.37784126, 0.38045576, 0.38370645, 0.38860881, 0.39679497,
     &   0.40358913, 0.41232198, 0.42339751, 0.43466878, 0.44101697,
     &   0.44396722, 0.44505313, 0.44638655, 0.44876432, 0.45254219,
     &   0.45723107, 0.46396324, 0.47768304, 0.49364686, 0.51244569,
     &   0.50072682, 0.47095481, 0.44691303, 0.43603307, 0.44207382,
     &   0.44628677, 0.45101666, 0.45714170, 0.46586752, 0.47630486,
     &   0.48614299, 0.49601543, 0.50601554, 0.51664257, 0.52745193,
     &   0.53877378, 0.55195504, 0.56717187, 0.58288854, 0.59819233,
     &   0.61348552, 0.62289786, 0.63015574, 0.63715178, 0.64274883,
     &   0.64818233, 0.65312725, 0.65772676, 0.66216666, 0.66521841,
     &   0.66807687, 0.67050827, 0.67146361, 0.67229444, 0.67300117,
     &   0.67358410, 0.67404342, 0.67437911, 0.67459106, 0.67467892,
     &   0.67464215, 0.67447990, 0.67521620, 0.67603421, 0.67674661,
     &   0.67735207, 0.67784888, 0.67823535, 0.67850894, 0.67866784,
     &   0.67870933, 0.67863071, 0.67842895, 0.67810118, 0.67802787,
     &   0.67926645, 0.68037993, 0.68136537, 0.68221986, 0.68294042,
     &   0.68430334, 0.69060922, 0.69671112, 0.70260721, 0.70875770,
     &   0.71500599, 0.72101843, 0.72732460, 0.73456037, 0.74151003,
     &   0.74802428, 0.75202245, 0.75582021, 0.75942451, 0.76294303,
     &   0.76630437, 0.76949227, 0.77235961, 0.77493554, 0.77736491,
     &   0.77945501, 0.78092360, 0.78227395, 0.78349084, 0.78339756,
     &   0.78320640, 0.78292096, 0.78153336, 0.77972066, 0.77781349,
     &   0.77446717, 0.76968664, 0.76476043, 0.75940341, 0.75301617,
     &   0.74644816, 0.73970020, 0.73257220, 0.72525775, 0.71776128/
      data (wwi(i,9),i=1,mxwv)/
     &   0.99998528, 0.99998623, 0.99998713, 0.99998850, 0.99998945,
     &   0.99999034, 0.99999106, 0.99999154, 0.99999225, 0.99999273,
     &   0.99999332, 0.99999392, 0.99999475, 0.99999541, 0.99999607,
     &   0.99999666, 0.99999690, 0.99999720, 0.99999744, 0.99999768,
     &   0.99999791, 0.99999809, 0.99999821, 0.99999839, 0.99999857,
     &   0.99999881, 0.99999881, 0.99999881, 0.99999875, 0.99999875,
     &   0.99999869, 0.99999845, 0.99999845, 0.99999833, 0.99999827,
     &   0.99999803, 0.99999791, 0.99999785, 0.99999779, 0.99999756,
     &   0.99999708, 0.99999666, 0.99999601, 0.99999505, 0.99999404,
     &   0.99999285, 0.99999166, 0.99999022, 0.99998897, 0.99998772,
     &   0.99998343, 0.99998134, 0.99997872, 0.99997550, 0.99996990,
     &   0.99996233, 0.99995208, 0.99994361, 0.99993694, 0.99993598,
     &   0.99993479, 0.99991852, 0.99989659, 0.99985713, 0.99983293,
     &   0.99981987, 0.99980068, 0.99977589, 0.99969280, 0.99958175,
     &   0.99943036, 0.99923736, 0.99919081, 0.99927890, 0.99939525,
     &   0.99942529, 0.99942416, 0.99934632, 0.99927682, 0.99880105,
     &   0.99822730, 0.99734342, 0.99663258, 0.99629396, 0.99629128,
     &   0.99638420, 0.99644303, 0.99634665, 0.99609536, 0.99518204,
     &   0.98582214, 0.96089077, 0.91517615, 0.89391404, 0.90420872,
     &   0.91868699, 0.93636811, 0.94795632, 0.95265758, 0.95923680,
     &   0.96516001, 0.97007811, 0.97277099, 0.97448659, 0.97642940,
     &   0.97664094, 0.96047336, 0.90380538, 0.82338244, 0.79552865,
     &   0.80539334, 0.83203304, 0.87922555, 0.92584985, 0.95139802,
     &   0.96371382, 0.96862328, 0.96201414, 0.94497252, 0.92970431,
     &   0.92105329, 0.91286671, 0.90410668, 0.90108699, 0.89449662,
     &   0.82395452, 0.70604104, 0.58524776, 0.51620728, 0.49781516,
     &   0.51171857, 0.53945750, 0.54997706, 0.56453151, 0.56600684,
     &   0.55858439, 0.55133486, 0.54438412, 0.53912741, 0.53577876,
     &   0.53816414, 0.55648386, 0.59779572, 0.63486183, 0.65109849,
     &   0.66616404, 0.66606802, 0.65069473, 0.62999707, 0.61637908,
     &   0.60416484, 0.58852053, 0.57317257, 0.55359715, 0.53788561,
     &   0.53073823, 0.52827525, 0.54312158, 0.56818068, 0.58456367,
     &   0.60352045, 0.61590177, 0.61608911, 0.61062950, 0.60393089,
     &   0.58710158, 0.56853384, 0.55021173, 0.53306353, 0.52014452,
     &   0.51031178, 0.50773400, 0.50710100, 0.50758231, 0.50841516,
     &   0.50949550, 0.51075327, 0.51060838, 0.50977939, 0.50991970,
     &   0.51047319, 0.51093161, 0.51134914, 0.51267028, 0.51278007,
     &   0.51291645, 0.51385188, 0.51669586, 0.52173269, 0.52911967,
     &   0.53508842, 0.53271896, 0.53206283, 0.53125286, 0.52993733,
     &   0.52786297, 0.52779627, 0.52894735, 0.53097761, 0.53458345,
     &   0.53754485, 0.53501779, 0.52483916, 0.50270194, 0.47823194,
     &   0.47916222, 0.48956871, 0.49734437, 0.50305361, 0.50644571,
     &   0.50909686, 0.51080126, 0.51198131, 0.51274389, 0.51253891,
     &   0.51223952, 0.51170409, 0.51054972, 0.50945705, 0.50834548,
     &   0.50693423, 0.50525379, 0.50350702, 0.50198615, 0.50082761,
     &   0.49997735, 0.49951708, 0.49937809, 0.49937129, 0.49935013,
     &   0.50024652, 0.50251305, 0.50627947, 0.51167297, 0.51875657,
     &   0.52428406, 0.52920151, 0.52952576, 0.52924681, 0.53527355,
     &   0.55012923, 0.57428902, 0.60286838, 0.62596142, 0.65041780,
     &   0.67585903, 0.69823104, 0.71125799, 0.71932447, 0.72454697,
     &   0.72752237, 0.73055530, 0.73227745, 0.73134267, 0.72960651,
     &   0.72586036, 0.71940672, 0.71261394, 0.70534724, 0.69747263,
     &   0.68899733, 0.68083209, 0.67165434, 0.66136706, 0.64989781,
     &   0.63649172, 0.62066591, 0.60308564, 0.58388418, 0.55969369,
     &   0.52876365, 0.49857253, 0.47149432, 0.45195988, 0.43809170,
     &   0.42833143, 0.42242455, 0.42004350, 0.41964051, 0.42047086,
     &   0.42208445, 0.42437285, 0.42724746, 0.43150702, 0.43831259,
     &   0.44353333, 0.45043743, 0.45878077, 0.46534488, 0.46640950,
     &   0.46530178, 0.46483320, 0.46491799, 0.46567488, 0.46705875,
     &   0.46881548, 0.47142354, 0.47703764, 0.48545533, 0.49914953,
     &   0.49848372, 0.48186517, 0.46627069, 0.45836964, 0.46141663,
     &   0.46350458, 0.46585277, 0.46887192, 0.47319749, 0.47866747,
     &   0.48422068, 0.49024054, 0.49688739, 0.50429165, 0.51243365,
     &   0.52132571, 0.53225315, 0.54556453, 0.55988950, 0.57524037,
     &   0.59137559, 0.60309309, 0.61308551, 0.62271935, 0.63101763,
     &   0.63922161, 0.64691126, 0.65423989, 0.66142297, 0.66724592,
     &   0.67286640, 0.67800689, 0.68147016, 0.68479311, 0.68797535,
     &   0.69101685, 0.69391805, 0.69667941, 0.69930184, 0.70178622,
     &   0.70413381, 0.70634574, 0.70918399, 0.71204793, 0.71480072,
     &   0.71744329, 0.71997666, 0.72240198, 0.72472036, 0.72693300,
     &   0.72904098, 0.73104548, 0.73294759, 0.73474830, 0.73676270,
     &   0.73985249, 0.74284184, 0.74573141, 0.74852180, 0.75121373,
     &   0.75447100, 0.76191956, 0.76914936, 0.77615988, 0.78339410,
     &   0.79069138, 0.79772902, 0.80494958, 0.81286085, 0.82043546,
     &   0.82756376, 0.83272600, 0.83766627, 0.84238970, 0.84697002,
     &   0.85135752, 0.85554171, 0.85942346, 0.86302757, 0.86645490,
     &   0.86957848, 0.87221968, 0.87471884, 0.87706769, 0.87852120,
     &   0.87986785, 0.88111061, 0.88163042, 0.88185757, 0.88199747,
     &   0.88124061, 0.87960213, 0.87786275, 0.87584335, 0.87318820,
     &   0.87040406, 0.86748797, 0.86431175, 0.86099195, 0.85752696/
      data (wwi(i,10),i=1,mxwv)/
     &   0.99997979, 0.99998081, 0.99998200, 0.99998385, 0.99998540,
     &   0.99998665, 0.99998754, 0.99998826, 0.99998897, 0.99998981,
     &   0.99999064, 0.99999148, 0.99999267, 0.99999356, 0.99999458,
     &   0.99999535, 0.99999577, 0.99999613, 0.99999642, 0.99999672,
     &   0.99999696, 0.99999732, 0.99999744, 0.99999774, 0.99999797,
     &   0.99999833, 0.99999833, 0.99999833, 0.99999833, 0.99999821,
     &   0.99999815, 0.99999809, 0.99999785, 0.99999762, 0.99999762,
     &   0.99999732, 0.99999720, 0.99999708, 0.99999696, 0.99999666,
     &   0.99999601, 0.99999535, 0.99999440, 0.99999285, 0.99999166,
     &   0.99998999, 0.99998838, 0.99998665, 0.99998426, 0.99998248,
     &   0.99997866, 0.99997449, 0.99997175, 0.99996591, 0.99995840,
     &   0.99994767, 0.99993312, 0.99992096, 0.99991316, 0.99991220,
     &   0.99991053, 0.99988854, 0.99985403, 0.99980116, 0.99976766,
     &   0.99974769, 0.99972063, 0.99968898, 0.99957484, 0.99941993,
     &   0.99921381, 0.99894023, 0.99887574, 0.99899697, 0.99915886,
     &   0.99920529, 0.99920511, 0.99909341, 0.99899673, 0.99833751,
     &   0.99754190, 0.99630767, 0.99532640, 0.99485826, 0.99485046,
     &   0.99498266, 0.99506736, 0.99493062, 0.99459398, 0.99332082,
     &   0.98039597, 0.94648480, 0.88647074, 0.85956895, 0.87252933,
     &   0.89100826, 0.91397405, 0.92925662, 0.93551785, 0.94431257,
     &   0.95228577, 0.95891243, 0.96259665, 0.96492767, 0.96757382,
     &   0.96786588, 0.94601011, 0.87214041, 0.77523756, 0.74400938,
     &   0.75493312, 0.78520906, 0.84153700, 0.90039146, 0.93392253,
     &   0.95039684, 0.95700729, 0.94811273, 0.92540389, 0.90538114,
     &   0.89416790, 0.88364208, 0.87247258, 0.86861551, 0.86045718,
     &   0.77533758, 0.65086740, 0.55011916, 0.50729215, 0.49844530,
     &   0.51668364, 0.54283750, 0.55328339, 0.56785572, 0.56971020,
     &   0.56249642, 0.55537301, 0.54855192, 0.54340982, 0.53974509,
     &   0.53817362, 0.54451299, 0.56792945, 0.59421641, 0.60683686,
     &   0.61915570, 0.61886877, 0.60583919, 0.58917457, 0.57871747,
     &   0.56970781, 0.55881631, 0.54881632, 0.53731048, 0.52926302,
     &   0.52609700, 0.52522051, 0.53179425, 0.54541320, 0.55550212,
     &   0.56831735, 0.57719713, 0.57716703, 0.57293272, 0.56785601,
     &   0.55586672, 0.54355514, 0.53244436, 0.52314657, 0.51693672,
     &   0.51302701, 0.51269066, 0.51298720, 0.51360828, 0.51405078,
     &   0.51433623, 0.51432621, 0.51388806, 0.51341653, 0.51332086,
     &   0.51329970, 0.51321030, 0.51312876, 0.51332110, 0.51303160,
     &   0.51281762, 0.51296747, 0.51388633, 0.51577073, 0.51885265,
     &   0.52132320, 0.51941282, 0.51843601, 0.51730597, 0.51581818,
     &   0.51386160, 0.51283371, 0.51213920, 0.51160109, 0.51136726,
     &   0.51026142, 0.50726056, 0.50397617, 0.49726787, 0.48822245,
     &   0.49311444, 0.50330585, 0.51049882, 0.51590151, 0.51912713,
     &   0.52169764, 0.52336931, 0.52454996, 0.52533811, 0.52517587,
     &   0.52494043, 0.52448767, 0.52343047, 0.52244800, 0.52145058,
     &   0.52013749, 0.51850438, 0.51667726, 0.51490253, 0.51324290,
     &   0.51173204, 0.51048130, 0.50945008, 0.50854951, 0.50789654,
     &   0.50773656, 0.50807649, 0.50889862, 0.51054233, 0.51325184,
     &   0.51527297, 0.51699036, 0.51625848, 0.51549429, 0.51856667,
     &   0.52715546, 0.54253596, 0.56236959, 0.57911640, 0.59788871,
     &   0.61844605, 0.63688904, 0.64656454, 0.65143216, 0.65346485,
     &   0.65326172, 0.65315753, 0.65182024, 0.64801031, 0.64380878,
     &   0.63818669, 0.63069510, 0.62383783, 0.61756438, 0.61178017,
     &   0.60647941, 0.60248095, 0.59844089, 0.59414518, 0.58938283,
     &   0.58388346, 0.57717842, 0.56905729, 0.55940944, 0.54575658,
     &   0.52653712, 0.50638998, 0.48735294, 0.47339058, 0.46345055,
     &   0.45649853, 0.45247993, 0.45113575, 0.45132291, 0.45248431,
     &   0.45429754, 0.45669547, 0.45959944, 0.46368015, 0.46975887,
     &   0.47399038, 0.47986123, 0.48664033, 0.49041250, 0.48825750,
     &   0.48424318, 0.48243633, 0.48148799, 0.48112652, 0.48096168,
     &   0.48081622, 0.48088735, 0.48048279, 0.48063210, 0.48387235,
     &   0.48523891, 0.47947499, 0.47311258, 0.47063690, 0.47232452,
     &   0.47345170, 0.47453797, 0.47566760, 0.47683054, 0.47790617,
     &   0.47903854, 0.48048192, 0.48239219, 0.48487449, 0.48803639,
     &   0.49190289, 0.49723473, 0.50451297, 0.51315510, 0.52370137,
     &   0.53592092, 0.54622197, 0.55591154, 0.56566566, 0.57468730,
     &   0.58399343, 0.59310633, 0.60213166, 0.61126912, 0.61934441,
     &   0.62738383, 0.63504791, 0.64094871, 0.64676934, 0.65249854,
     &   0.65812606, 0.66364259, 0.66903991, 0.67431092, 0.67944926,
     &   0.68444973, 0.68930781, 0.69451922, 0.69969803, 0.70475137,
     &   0.70967674, 0.71447194, 0.71913552, 0.72366643, 0.72806406,
     &   0.73232841, 0.73645955, 0.74045801, 0.74432492, 0.74832726,
     &   0.75319374, 0.75792587, 0.76252466, 0.76699108, 0.77132654,
     &   0.77613777, 0.78471017, 0.79300189, 0.80101246, 0.80920565,
     &   0.81740695, 0.82527906, 0.83323425, 0.84173143, 0.84981781,
     &   0.85740888, 0.86327004, 0.86886144, 0.87419283, 0.87932557,
     &   0.88422656, 0.88889331, 0.89325452, 0.89733678, 0.90122277,
     &   0.90481985, 0.90800107, 0.91103196, 0.91390908, 0.91609412,
     &   0.91817641, 0.92015964, 0.92161363, 0.92285246, 0.92402178,
     &   0.92457885, 0.92455393, 0.92447621, 0.92423165, 0.92358857,
     &   0.92288566, 0.92212081, 0.92120576, 0.92022258, 0.91916978/
      data (wwi(i,11),i=1,mxwv)/
     &   0.99997157, 0.99997312, 0.99997473, 0.99997711, 0.99997920,
     &   0.99998111, 0.99998242, 0.99998355, 0.99998462, 0.99998569,
     &   0.99998695, 0.99998838, 0.99998957, 0.99999124, 0.99999243,
     &   0.99999338, 0.99999410, 0.99999452, 0.99999493, 0.99999547,
     &   0.99999589, 0.99999624, 0.99999654, 0.99999678, 0.99999714,
     &   0.99999750, 0.99999768, 0.99999768, 0.99999762, 0.99999756,
     &   0.99999744, 0.99999726, 0.99999708, 0.99999696, 0.99999666,
     &   0.99999630, 0.99999607, 0.99999583, 0.99999571, 0.99999523,
     &   0.99999440, 0.99999338, 0.99999225, 0.99999034, 0.99998814,
     &   0.99998623, 0.99998361, 0.99998111, 0.99997878, 0.99997592,
     &   0.99997032, 0.99996424, 0.99995893, 0.99995357, 0.99994200,
     &   0.99992681, 0.99990833, 0.99989063, 0.99987948, 0.99987596,
     &   0.99987954, 0.99984401, 0.99979836, 0.99972236, 0.99967605,
     &   0.99964833, 0.99961233, 0.99956578, 0.99940604, 0.99918777,
     &   0.99890107, 0.99852163, 0.99840873, 0.99859887, 0.99882597,
     &   0.99889129, 0.99888849, 0.99873692, 0.99860823, 0.99769086,
     &   0.99657488, 0.99485594, 0.99349159, 0.99284047, 0.99283320,
     &   0.99302882, 0.99314034, 0.99295044, 0.99246335, 0.99070680,
     &   0.97287041, 0.92710042, 0.85004085, 0.81722605, 0.83291352,
     &   0.85571396, 0.88474482, 0.90449214, 0.91268474, 0.92427528,
     &   0.93487251, 0.94375205, 0.94871175, 0.95186335, 0.95544517,
     &   0.95584655, 0.92656493, 0.83254957, 0.72177881, 0.68941015,
     &   0.70053363, 0.73247302, 0.79597992, 0.86761600, 0.91069239,
     &   0.93243086, 0.94127816, 0.92939585, 0.89959717, 0.87391114,
     &   0.85967451, 0.84664196, 0.83288068, 0.82813537, 0.81819493,
     &   0.72114515, 0.60157156, 0.52897120, 0.50617892, 0.50053662,
     &   0.52002406, 0.54505867, 0.55546647, 0.57007706, 0.57228178,
     &   0.56526077, 0.55824584, 0.55152339, 0.54648167, 0.54283589,
     &   0.54011321, 0.54057783, 0.55053782, 0.56556612, 0.57361239,
     &   0.58201760, 0.58155215, 0.57202649, 0.56062031, 0.55383313,
     &   0.54823780, 0.54193026, 0.53654355, 0.53103775, 0.52772653,
     &   0.52666354, 0.52656579, 0.52875811, 0.53447056, 0.53929436,
     &   0.54656243, 0.55135339, 0.55110633, 0.54826182, 0.54495907,
     &   0.53780890, 0.53103322, 0.52546042, 0.52128196, 0.51873124,
     &   0.51745594, 0.51794130, 0.51855987, 0.51925069, 0.51961184,
     &   0.51968455, 0.51928270, 0.51879501, 0.51851422, 0.51837891,
     &   0.51818991, 0.51793712, 0.51770318, 0.51746428, 0.51706398,
     &   0.51675540, 0.51659769, 0.51663071, 0.51690781, 0.51755542,
     &   0.51793975, 0.51657826, 0.51565576, 0.51464546, 0.51344901,
     &   0.51204389, 0.51095271, 0.50981796, 0.50856924, 0.50710398,
     &   0.50477469, 0.50145370, 0.49900305, 0.49702924, 0.49609333,
     &   0.50349849, 0.51354831, 0.52042884, 0.52567488, 0.52885407,
     &   0.53142518, 0.53313208, 0.53436613, 0.53521943, 0.53514135,
     &   0.53499466, 0.53463840, 0.53369242, 0.53282601, 0.53194726,
     &   0.53075820, 0.52924830, 0.52752441, 0.52581406, 0.52414441,
     &   0.52257067, 0.52120024, 0.51999944, 0.51892066, 0.51816899,
     &   0.51769751, 0.51730710, 0.51687032, 0.51671720, 0.51708072,
     &   0.51723403, 0.51730090, 0.51644379, 0.51579666, 0.51699334,
     &   0.52091825, 0.52891290, 0.54060149, 0.55139095, 0.56448668,
     &   0.57985961, 0.59433824, 0.60181195, 0.60516316, 0.60592896,
     &   0.60456902, 0.60306656, 0.60021472, 0.59493142, 0.58906841,
     &   0.58182269, 0.57295883, 0.56470776, 0.55714357, 0.55032670,
     &   0.54437524, 0.53989637, 0.53613216, 0.53300661, 0.53039527,
     &   0.52821720, 0.52627802, 0.52414280, 0.52145845, 0.51674145,
     &   0.50878149, 0.49926108, 0.48947683, 0.48213962, 0.47699189,
     &   0.47366509, 0.47227123, 0.47253346, 0.47376996, 0.47565252,
     &   0.47800115, 0.48080930, 0.48401690, 0.48825070, 0.49413782,
     &   0.49794725, 0.50340873, 0.50949800, 0.51200503, 0.50854445,
     &   0.50337601, 0.50103176, 0.49969807, 0.49898252, 0.49835375,
     &   0.49765018, 0.49705356, 0.49464273, 0.49099636, 0.48727807,
     &   0.48436791, 0.48265463, 0.48202023, 0.48423749, 0.48611850,
     &   0.48738393, 0.48841423, 0.48921287, 0.48945656, 0.48880860,
     &   0.48795328, 0.48708713, 0.48626670, 0.48571694, 0.48536608,
     &   0.48538685, 0.48588148, 0.48715982, 0.48935270, 0.49259436,
     &   0.49713695, 0.50138187, 0.50583017, 0.51080859, 0.51582873,
     &   0.52144140, 0.52736473, 0.53365147, 0.54044372, 0.54689884,
     &   0.55370945, 0.56059039, 0.56630397, 0.57220143, 0.57826692,
     &   0.58447951, 0.59082329, 0.59727764, 0.60382336, 0.61043817,
     &   0.61710382, 0.62379944, 0.63074780, 0.63775414, 0.64475906,
     &   0.65174603, 0.65869927, 0.66560394, 0.67244607, 0.67921263,
     &   0.68589181, 0.69247276, 0.69894564, 0.70530176, 0.71176392,
     &   0.71896023, 0.72602147, 0.73294014, 0.73970979, 0.74632525,
     &   0.75338960, 0.76420063, 0.77469277, 0.78485197, 0.79522532,
     &   0.80559063, 0.81553149, 0.82548410, 0.83592981, 0.84583443,
     &   0.85511583, 0.86260581, 0.86970729, 0.87643373, 0.88284355,
     &   0.88891393, 0.89464915, 0.89999115, 0.90497255, 0.90967542,
     &   0.91402203, 0.91790754, 0.92158151, 0.92504662, 0.92783815,
     &   0.93048704, 0.93300092, 0.93502152, 0.93682683, 0.93854415,
     &   0.93973470, 0.94044054, 0.94109994, 0.94162363, 0.94183302,
     &   0.94200522, 0.94214064, 0.94216591, 0.94215584, 0.94211054/
      data (wwi(i,12),i=1,mxwv)/
     &   0.99996018, 0.99996233, 0.99996448, 0.99996817, 0.99997133,
     &   0.99997342, 0.99997538, 0.99997675, 0.99997807, 0.99997973,
     &   0.99998176, 0.99998361, 0.99998558, 0.99998766, 0.99998951,
     &   0.99999064, 0.99999160, 0.99999231, 0.99999297, 0.99999362,
     &   0.99999416, 0.99999470, 0.99999511, 0.99999553, 0.99999607,
     &   0.99999660, 0.99999672, 0.99999672, 0.99999672, 0.99999654,
     &   0.99999636, 0.99999619, 0.99999583, 0.99999577, 0.99999535,
     &   0.99999475, 0.99999452, 0.99999428, 0.99999392, 0.99999332,
     &   0.99999213, 0.99999058, 0.99998903, 0.99998671, 0.99998349,
     &   0.99998057, 0.99997711, 0.99997354, 0.99997044, 0.99996555,
     &   0.99995840, 0.99994969, 0.99994284, 0.99993461, 0.99991959,
     &   0.99989706, 0.99987197, 0.99984741, 0.99983126, 0.99982804,
     &   0.99982464, 0.99978292, 0.99971628, 0.99961179, 0.99954647,
     &   0.99950945, 0.99945909, 0.99939084, 0.99916971, 0.99886501,
     &   0.99846148, 0.99793321, 0.99780399, 0.99804276, 0.99835807,
     &   0.99845052, 0.99844873, 0.99823725, 0.99805319, 0.99677461,
     &   0.99521673, 0.99281961, 0.99092633, 0.99001288, 0.99000132,
     &   0.99027729, 0.99042988, 0.99017227, 0.98951429, 0.98705631,
     &   0.96249783, 0.90147668, 0.80562341, 0.76759136, 0.78556985,
     &   0.81239784, 0.84772259, 0.87248057, 0.88292021, 0.89786154,
     &   0.91168851, 0.92339516, 0.92998582, 0.93419278, 0.93898928,
     &   0.93953145, 0.90087593, 0.78521699, 0.66726077, 0.63703525,
     &   0.64716989, 0.67767406, 0.74395090, 0.82682663, 0.88046443,
     &   0.90856194, 0.92018789, 0.90460217, 0.86638349, 0.83446580,
     &   0.81729609, 0.80162638, 0.78546542, 0.77992159, 0.76844603,
     &   0.66560441, 0.56355190, 0.51906425, 0.50767183, 0.50301433,
     &   0.52225459, 0.54645956, 0.55684704, 0.57150155, 0.57401156,
     &   0.56716001, 0.56023592, 0.55358654, 0.54862088, 0.54503930,
     &   0.54211062, 0.54048026, 0.54292977, 0.54926586, 0.55311370,
     &   0.55755568, 0.55692917, 0.55105895, 0.54456019, 0.54086870,
     &   0.53793663, 0.53485292, 0.53243893, 0.53022784, 0.52907652,
     &   0.52883989, 0.52903610, 0.52954900, 0.53125972, 0.53283924,
     &   0.53561121, 0.53779018, 0.53739613, 0.53571427, 0.53382987,
     &   0.53025913, 0.52711678, 0.52469963, 0.52299541, 0.52192694,
     &   0.52153081, 0.52225387, 0.52297437, 0.52370793, 0.52408701,
     &   0.52415287, 0.52370542, 0.52325958, 0.52308035, 0.52298439,
     &   0.52279472, 0.52254450, 0.52231479, 0.52199244, 0.52160782,
     &   0.52131760, 0.52110368, 0.52086896, 0.52057189, 0.52022207,
     &   0.51967138, 0.51863223, 0.51779628, 0.51691508, 0.51594788,
     &   0.51491398, 0.51392853, 0.51276636, 0.51143175, 0.50975293,
     &   0.50740790, 0.50447679, 0.50211191, 0.50089693, 0.50294864,
     &   0.51135242, 0.52113497, 0.52778858, 0.53292787, 0.53610033,
     &   0.53869790, 0.54046082, 0.54176360, 0.54269302, 0.54270738,
     &   0.54265195, 0.54238957, 0.54154837, 0.54078674, 0.54001302,
     &   0.53893507, 0.53754491, 0.53594339, 0.53435063, 0.53278720,
     &   0.53130585, 0.53000885, 0.52886289, 0.52783120, 0.52714020,
     &   0.52665639, 0.52610922, 0.52530855, 0.52452654, 0.52393568,
     &   0.52326858, 0.52257103, 0.52173513, 0.52122599, 0.52145249,
     &   0.52263433, 0.52559382, 0.53078043, 0.53618699, 0.54357463,
     &   0.55316329, 0.56289589, 0.56802881, 0.57024318, 0.57054341,
     &   0.56923133, 0.56777442, 0.56525278, 0.56087166, 0.55602914,
     &   0.55015647, 0.54315501, 0.53660017, 0.53051263, 0.52491403,
     &   0.51986563, 0.51565224, 0.51191837, 0.50868291, 0.50594914,
     &   0.50363564, 0.50180095, 0.50044602, 0.49939367, 0.49806443,
     &   0.49592346, 0.49312311, 0.49005634, 0.48781326, 0.48648864,
     &   0.48618436, 0.48705888, 0.48876095, 0.49096569, 0.49353313,
     &   0.49639556, 0.49960124, 0.50311077, 0.50754327, 0.51343137,
     &   0.51712030, 0.52244914, 0.52825660, 0.53025872, 0.52642530,
     &   0.52106160, 0.51869333, 0.51737064, 0.51667970, 0.51606399,
     &   0.51537216, 0.51480281, 0.51222259, 0.50795257, 0.50246644,
     &   0.49699733, 0.49548760, 0.49667239, 0.50100672, 0.50334775,
     &   0.50493896, 0.50621611, 0.50719780, 0.50750536, 0.50672090,
     &   0.50563037, 0.50441390, 0.50307375, 0.50188237, 0.50065070,
     &   0.49960119, 0.49854830, 0.49762186, 0.49714014, 0.49682167,
     &   0.49706176, 0.49725640, 0.49762997, 0.49841940, 0.49937105,
     &   0.50066727, 0.50224394, 0.50412381, 0.50637466, 0.50858408,
     &   0.51111382, 0.51385075, 0.51616699, 0.51871538, 0.52150369,
     &   0.52453887, 0.52782685, 0.53137934, 0.53517884, 0.53924811,
     &   0.54358041, 0.54817438, 0.55321187, 0.55853271, 0.56409591,
     &   0.56989306, 0.57591414, 0.58214766, 0.58858061, 0.59519851,
     &   0.60198587, 0.60892588, 0.61600077, 0.62319225, 0.63067621,
     &   0.63898182, 0.64736527, 0.65580344, 0.66427356, 0.67275327,
     &   0.68184304, 0.69498175, 0.70803058, 0.72093362, 0.73434746,
     &   0.74798304, 0.76127589, 0.77471340, 0.78886199, 0.80243373,
     &   0.81529701, 0.82612032, 0.83643210, 0.84623104, 0.85556477,
     &   0.86440313, 0.87274617, 0.88052720, 0.88778245, 0.89460337,
     &   0.90090507, 0.90657741, 0.91190583, 0.91689801, 0.92105675,
     &   0.92497307, 0.92866015, 0.93174976, 0.93454665, 0.93718916,
     &   0.93923986, 0.94076139, 0.94220144, 0.94347417, 0.94441032,
     &   0.94529015, 0.94611591, 0.94681209, 0.94746071, 0.94806463/
      data (wwi(i,13),i=1,mxwv)/
     &   0.99994397, 0.99994683, 0.99994993, 0.99995512, 0.99995947,
     &   0.99996275, 0.99996543, 0.99996746, 0.99996954, 0.99997163,
     &   0.99997431, 0.99997693, 0.99997962, 0.99998254, 0.99998498,
     &   0.99998695, 0.99998820, 0.99998915, 0.99999011, 0.99999100,
     &   0.99999183, 0.99999225, 0.99999315, 0.99999368, 0.99999446,
     &   0.99999529, 0.99999535, 0.99999547, 0.99999535, 0.99999517,
     &   0.99999487, 0.99999464, 0.99999428, 0.99999398, 0.99999338,
     &   0.99999267, 0.99999225, 0.99999195, 0.99999160, 0.99999064,
     &   0.99998891, 0.99998719, 0.99998462, 0.99998093, 0.99997705,
     &   0.99997270, 0.99996823, 0.99996316, 0.99995857, 0.99995267,
     &   0.99994135, 0.99992931, 0.99991935, 0.99990851, 0.99988705,
     &   0.99985594, 0.99981946, 0.99978524, 0.99976349, 0.99975967,
     &   0.99975348, 0.99969482, 0.99960482, 0.99945796, 0.99936438,
     &   0.99931228, 0.99924177, 0.99914789, 0.99883658, 0.99840736,
     &   0.99784654, 0.99710482, 0.99692428, 0.99726140, 0.99769688,
     &   0.99782658, 0.99782604, 0.99753135, 0.99727440, 0.99548447,
     &   0.99330676, 0.98996419, 0.98731917, 0.98605686, 0.98604143,
     &   0.98640394, 0.98664272, 0.98628390, 0.98536098, 0.98196095,
     &   0.94833440, 0.86839569, 0.75435889, 0.71325016, 0.73237950,
     &   0.76195765, 0.80271542, 0.83246469, 0.84529561, 0.86394382,
     &   0.88149941, 0.89658171, 0.90529615, 0.91067415, 0.91701174,
     &   0.91770720, 0.86778182, 0.73201197, 0.61752981, 0.59286350,
     &   0.60085982, 0.62645626, 0.68891060, 0.77840281, 0.84233302,
     &   0.87757158, 0.89249009, 0.87253189, 0.82509375, 0.78717029,
     &   0.76741540, 0.74978489, 0.73200721, 0.72595251, 0.71362531,
     &   0.61457199, 0.53908926, 0.51592219, 0.50949192, 0.50523412,
     &   0.52371132, 0.54728872, 0.55766535, 0.57236242, 0.57512861,
     &   0.56842053, 0.56157064, 0.55497426, 0.55006373, 0.54653651,
     &   0.54361850, 0.54146630, 0.54084122, 0.54224122, 0.54325140,
     &   0.54474169, 0.54400682, 0.54091889, 0.53778684, 0.53601962,
     &   0.53461945, 0.53327858, 0.53220618, 0.53132915, 0.53091210,
     &   0.53092873, 0.53122127, 0.53128290, 0.53152823, 0.53166479,
     &   0.53228545, 0.53276998, 0.53229702, 0.53134251, 0.53030682,
     &   0.52869260, 0.52731389, 0.52623224, 0.52542645, 0.52482742,
     &   0.52467722, 0.52546579, 0.52622551, 0.52698714, 0.52739853,
     &   0.52750003, 0.52709281, 0.52670491, 0.52659017, 0.52654386,
     &   0.52639896, 0.52619535, 0.52601236, 0.52571869, 0.52538681,
     &   0.52514803, 0.52496588, 0.52470845, 0.52429920, 0.52369070,
     &   0.52287382, 0.52200103, 0.52124661, 0.52046305, 0.51963097,
     &   0.51878256, 0.51789367, 0.51679677, 0.51552081, 0.51389050,
     &   0.51173609, 0.50919211, 0.50694948, 0.50577539, 0.50866687,
     &   0.51720780, 0.52664953, 0.53311723, 0.53817564, 0.54135650,
     &   0.54398936, 0.54581290, 0.54718673, 0.54819262, 0.54829651,
     &   0.54832739, 0.54815155, 0.54740387, 0.54673368, 0.54605061,
     &   0.54506814, 0.54378021, 0.54228652, 0.54080451, 0.53935182,
     &   0.53798103, 0.53678972, 0.53574413, 0.53481030, 0.53421420,
     &   0.53380454, 0.53329796, 0.53248525, 0.53160661, 0.53079224,
     &   0.52991778, 0.52901185, 0.52824795, 0.52785152, 0.52775455,
     &   0.52780771, 0.52828956, 0.52964276, 0.53136855, 0.53433359,
     &   0.53885669, 0.54394060, 0.54659522, 0.54757720, 0.54740548,
     &   0.54624057, 0.54498661, 0.54306680, 0.54001260, 0.53671253,
     &   0.53285903, 0.52844328, 0.52437431, 0.52063370, 0.51720387,
     &   0.51408297, 0.51133376, 0.50880611, 0.50650239, 0.50442791,
     &   0.50245237, 0.50062311, 0.49910966, 0.49789754, 0.49686700,
     &   0.49600849, 0.49543124, 0.49521044, 0.49540424, 0.49605757,
     &   0.49741662, 0.49958539, 0.50213015, 0.50490737, 0.50788093,
     &   0.51104528, 0.51448137, 0.51816458, 0.52270836, 0.52860308,
     &   0.53228211, 0.53757149, 0.54326755, 0.54509109, 0.54121184,
     &   0.53595024, 0.53371310, 0.53251505, 0.53193700, 0.53142744,
     &   0.53084552, 0.53038800, 0.52794439, 0.52381068, 0.51830536,
     &   0.51250786, 0.51093882, 0.51250571, 0.51741683, 0.51996905,
     &   0.52172792, 0.52315366, 0.52427340, 0.52471685, 0.52406836,
     &   0.52311176, 0.52201569, 0.52076840, 0.51963669, 0.51840764,
     &   0.51729429, 0.51603967, 0.51469135, 0.51355845, 0.51226097,
     &   0.51115566, 0.51010150, 0.50917864, 0.50852597, 0.50802070,
     &   0.50766909, 0.50748533, 0.50746679, 0.50762594, 0.50778931,
     &   0.50810194, 0.50853235, 0.50885302, 0.50927687, 0.50980818,
     &   0.51045394, 0.51122034, 0.51211447, 0.51314360, 0.51431477,
     &   0.51563585, 0.51711428, 0.51901877, 0.52115303, 0.52347654,
     &   0.52599531, 0.52871525, 0.53164202, 0.53478098, 0.53813708,
     &   0.54171467, 0.54551756, 0.54954910, 0.55381155, 0.55847943,
     &   0.56404990, 0.56988710, 0.57598573, 0.58233917, 0.58893931,
     &   0.59633428, 0.60773814, 0.61954051, 0.63168603, 0.64481789,
     &   0.65870613, 0.67278284, 0.68755698, 0.70369303, 0.71975100,
     &   0.73551220, 0.74931049, 0.76285470, 0.77608883, 0.78901452,
     &   0.80154508, 0.81363171, 0.82514286, 0.83608055, 0.84652323,
     &   0.85632426, 0.86530668, 0.87382573, 0.88187271, 0.88876885,
     &   0.89529455, 0.90145862, 0.90677464, 0.91164428, 0.91624749,
     &   0.92002666, 0.92307717, 0.92597508, 0.92861235, 0.93078244,
     &   0.93284434, 0.93480223, 0.93655455, 0.93821645, 0.93979287/
      data (ggi(i,1),i=1,mxwv)/
     &   0.83874619, 0.83875793, 0.83866620, 0.83844739, 0.83812904,
     &   0.83784223, 0.83761531, 0.83722752, 0.83697462, 0.83668435,
     &   0.83627301, 0.83598143, 0.83544493, 0.83492452, 0.83447802,
     &   0.83397424, 0.83342069, 0.83286768, 0.83227932, 0.83163297,
     &   0.83097708, 0.83025908, 0.82951319, 0.82873517, 0.82793087,
     &   0.82708627, 0.82618356, 0.82529402, 0.82433593, 0.82331395,
     &   0.82229072, 0.82122678, 0.82013166, 0.81899190, 0.81781930,
     &   0.81661677, 0.81538779, 0.81412470, 0.81283450, 0.81151956,
     &   0.81019109, 0.80883986, 0.80747682, 0.80610102, 0.80472231,
     &   0.80334157, 0.80196357, 0.80059606, 0.79924405, 0.79791546,
     &   0.79661429, 0.79534733, 0.79412585, 0.79295444, 0.79184258,
     &   0.79079598, 0.78981405, 0.78894639, 0.78815562, 0.78746992,
     &   0.78689855, 0.78644216, 0.78612000, 0.78593296, 0.78588909,
     &   0.78599322, 0.78625208, 0.78667647, 0.78725672, 0.78800917,
     &   0.78892136, 0.78999937, 0.79123855, 0.79263479, 0.79418236,
     &   0.79587364, 0.79770547, 0.79966879, 0.80175310, 0.80394322,
     &   0.80623436, 0.80860651, 0.81104773, 0.81353170, 0.81606686,
     &   0.81863874, 0.82123142, 0.82382888, 0.82645243, 0.82908064,
     &   0.83182973, 0.83467823, 0.83752465, 0.83997345, 0.84202760,
     &   0.84400803, 0.84602392, 0.84807688, 0.85009962, 0.85205173,
     &   0.85396874, 0.85584259, 0.85767365, 0.85944986, 0.86109680,
     &   0.86280072, 0.86460447, 0.86661416, 0.86852723, 0.86977708,
     &   0.87054926, 0.87123984, 0.87187123, 0.87273359, 0.87383097,
     &   0.87498325, 0.87617630, 0.87748677, 0.87874663, 0.88009584,
     &   0.88142580, 0.88291037, 0.88457304, 0.88648546, 0.88884646,
     &   0.89218503, 0.89597559, 0.90047467, 0.90469503, 0.90548348,
     &   0.89393139, 0.86898398, 0.85222101, 0.82893193, 0.81734544,
     &   0.81707454, 0.80532569, 0.78413570, 0.76797599, 0.75873268,
     &   0.75862682, 0.76342672, 0.76863849, 0.77396429, 0.77784407,
     &   0.78097796, 0.78311825, 0.78463280, 0.78572464, 0.78583872,
     &   0.78560168, 0.78513879, 0.78451830, 0.78396630, 0.78293478,
     &   0.78038275, 0.77640998, 0.77023315, 0.76498204, 0.76154208,
     &   0.75804943, 0.75485033, 0.75184029, 0.74874973, 0.74553114,
     &   0.74247533, 0.73932809, 0.73607588, 0.73272145, 0.72916502,
     &   0.72537106, 0.71990061, 0.71392894, 0.70733136, 0.70054513,
     &   0.69375968, 0.68718827, 0.68088764, 0.67444164, 0.66762286,
     &   0.66062671, 0.65351892, 0.64626241, 0.63877457, 0.63120866,
     &   0.62345958, 0.61551034, 0.60730892, 0.59882468, 0.58999389,
     &   0.58081663, 0.57148474, 0.56191128, 0.55204350, 0.54187435,
     &   0.53145057, 0.52073193, 0.50952369, 0.49787042, 0.48552960,
     &   0.47227919, 0.45816705, 0.44353256, 0.42811969, 0.41264573,
     &   0.39927053, 0.38807437, 0.37964272, 0.37176767, 0.36505762,
     &   0.35880020, 0.35334224, 0.34852433, 0.34433725, 0.34104297,
     &   0.33801129, 0.33507434, 0.33216941, 0.32855204, 0.32477379,
     &   0.32061231, 0.31533492, 0.30884263, 0.30096099, 0.29233268,
     &   0.28303388, 0.27376184, 0.26465940, 0.25572476, 0.24739619,
     &   0.23976064, 0.23226026, 0.22436742, 0.21644016, 0.20869276,
     &   0.20095909, 0.19339371, 0.18635289, 0.18005219, 0.17420787,
     &   0.16834669, 0.16226423, 0.15605254, 0.15000923, 0.14420974,
     &   0.13864677, 0.13329220, 0.12814108, 0.12324487, 0.11857629,
     &   0.11411741, 0.10983279, 0.10570837, 0.10174163, 0.09792793,
     &   0.09426469, 0.09074659, 0.08736110, 0.08410282, 0.08096659,
     &   0.07794465, 0.07501174, 0.07218920, 0.06947257, 0.06685756,
     &   0.06429966, 0.06179715, 0.05939137, 0.05707805, 0.05482369,
     &   0.05260702, 0.05050712, 0.04851654, 0.04663692, 0.04484820,
     &   0.04313698, 0.04149871, 0.03995646, 0.03848788, 0.03707949,
     &   0.03571666, 0.03439042, 0.03309963, 0.03184167, 0.03068038,
     &   0.02977655, 0.02888698, 0.02831453, 0.02878861, 0.02957804,
     &   0.02947856, 0.02874881, 0.02801049, 0.02737986, 0.02686523,
     &   0.02639063, 0.02603038, 0.02569732, 0.02488645, 0.02372128,
     &   0.02203722, 0.02069433, 0.01966441, 0.01900977, 0.01876168,
     &   0.01844225, 0.01814760, 0.01790375, 0.01770471, 0.01743343,
     &   0.01706745, 0.01665195, 0.01619747, 0.01574667, 0.01527589,
     &   0.01481480, 0.01434643, 0.01387076, 0.01340701, 0.01291592,
     &   0.01243109, 0.01195248, 0.01149363, 0.01106209, 0.01065247,
     &   0.01025789, 0.00988114, 0.00952000, 0.00917197, 0.00883853,
     &   0.00851722, 0.00820918, 0.00791995, 0.00764089, 0.00737166,
     &   0.00711190, 0.00686129, 0.00661949, 0.00638621, 0.00616113,
     &   0.00594398, 0.00573448, 0.00553612, 0.00534527, 0.00516098,
     &   0.00498304, 0.00481121, 0.00464530, 0.00448509, 0.00433040,
     &   0.00418103, 0.00403681, 0.00389755, 0.00376309, 0.00363360,
     &   0.00350983, 0.00339025, 0.00327474, 0.00316316, 0.00305536,
     &   0.00295119, 0.00285032, 0.00275286, 0.00265870, 0.00256687,
     &   0.00247758, 0.00239137, 0.00230783, 0.00222650, 0.00214802,
     &   0.00207226, 0.00199879, 0.00192793, 0.00185956, 0.00179367,
     &   0.00173012, 0.00166882, 0.00160967, 0.00155259, 0.00149754,
     &   0.00144445, 0.00139327, 0.00134390, 0.00129627, 0.00125042,
     &   0.00120619, 0.00116353, 0.00112242, 0.00108277, 0.00104453,
     &   0.00100769, 0.00097219, 0.00093795, 0.00090494, 0.00087317,
     &   0.00084252, 0.00081295, 0.00078452, 0.00075708, 0.00073060/
      data (ggi(i,2),i=1,mxwv)/
     &   0.84917140, 0.84926456, 0.84936696, 0.84919518, 0.84908676,
     &   0.84889764, 0.84881067, 0.84856558, 0.84849083, 0.84841669,
     &   0.84816706, 0.84799922, 0.84766787, 0.84742087, 0.84716910,
     &   0.84669441, 0.84648794, 0.84621769, 0.84593588, 0.84557152,
     &   0.84517574, 0.84470445, 0.84429395, 0.84385592, 0.84338874,
     &   0.84299600, 0.84242749, 0.84194636, 0.84143001, 0.84092623,
     &   0.84034145, 0.83968282, 0.83917385, 0.83852208, 0.83786619,
     &   0.83718598, 0.83651030, 0.83579898, 0.83501047, 0.83424908,
     &   0.83344120, 0.83264542, 0.83178341, 0.83089077, 0.82996136,
     &   0.82901311, 0.82803476, 0.82702070, 0.82598084, 0.82490480,
     &   0.82378823, 0.82265246, 0.82148319, 0.82028925, 0.81906003,
     &   0.81781399, 0.81654078, 0.81524265, 0.81392950, 0.81259841,
     &   0.81125748, 0.80990350, 0.80854684, 0.80718958, 0.80583549,
     &   0.80449080, 0.80316085, 0.80185473, 0.80057502, 0.79933429,
     &   0.79813409, 0.79698664, 0.79589570, 0.79487097, 0.79392213,
     &   0.79306108, 0.79230011, 0.79164577, 0.79111117, 0.79070884,
     &   0.79044741, 0.79033697, 0.79037809, 0.79056978, 0.79093295,
     &   0.79147685, 0.79220688, 0.79312509, 0.79426038, 0.79561424,
     &   0.79737663, 0.79962528, 0.80235118, 0.80466044, 0.80642235,
     &   0.80828089, 0.81037319, 0.81278962, 0.81544471, 0.81820953,
     &   0.82113928, 0.82421690, 0.82743859, 0.83076888, 0.83408225,
     &   0.83764076, 0.84156382, 0.84610569, 0.85076064, 0.85425222,
     &   0.85689265, 0.85942078, 0.86188591, 0.86482960, 0.86828721,
     &   0.87192333, 0.87568557, 0.87968713, 0.88364244, 0.88774800,
     &   0.89181852, 0.89612395, 0.90068567, 0.90559161, 0.91111666,
     &   0.91797650, 0.92525935, 0.93318057, 0.94029433, 0.94308221,
     &   0.93246245, 0.90762490, 0.89092463, 0.86711878, 0.85522097,
     &   0.85717922, 0.85129422, 0.82542664, 0.79088098, 0.76473993,
     &   0.75694960, 0.76204813, 0.77066338, 0.78112841, 0.79004753,
     &   0.79800308, 0.80457574, 0.81026953, 0.81534708, 0.81890559,
     &   0.82195395, 0.82469428, 0.82722872, 0.82989496, 0.83183336,
     &   0.83147663, 0.82899904, 0.82329947, 0.81911016, 0.81781822,
     &   0.81654161, 0.81581968, 0.81548071, 0.81511676, 0.81467724,
     &   0.81461549, 0.81455046, 0.81447977, 0.81441712, 0.81426167,
     &   0.81389618, 0.81053841, 0.80645800, 0.80118477, 0.79568779,
     &   0.79041481, 0.78602189, 0.78247619, 0.77872699, 0.77428758,
     &   0.76973355, 0.76524407, 0.76065886, 0.75589448, 0.75142258,
     &   0.74668378, 0.74158508, 0.73617482, 0.73067921, 0.72528714,
     &   0.72025174, 0.71548736, 0.71022648, 0.70471579, 0.69896567,
     &   0.69294691, 0.68654519, 0.67981690, 0.67269862, 0.66507792,
     &   0.65676397, 0.64742184, 0.63667083, 0.62384343, 0.60865778,
     &   0.59332752, 0.57937968, 0.56869352, 0.55878782, 0.55046654,
     &   0.54255033, 0.53547692, 0.52896595, 0.52300751, 0.51824027,
     &   0.51371914, 0.50946736, 0.50580972, 0.50166601, 0.49755046,
     &   0.49356103, 0.48906791, 0.48379770, 0.47720644, 0.46972069,
     &   0.46119544, 0.45218688, 0.44287264, 0.43326879, 0.42376509,
     &   0.41481608, 0.40595838, 0.39649197, 0.38664594, 0.37663409,
     &   0.36613703, 0.35544324, 0.34502843, 0.33545977, 0.32656282,
     &   0.31756175, 0.30795959, 0.29777062, 0.28750828, 0.27742225,
     &   0.26754534, 0.25785923, 0.24839330, 0.23930289, 0.23056434,
     &   0.22216430, 0.21404478, 0.20619054, 0.19860987, 0.19130133,
     &   0.18426859, 0.17750756, 0.17099294, 0.16471666, 0.15867044,
     &   0.15283981, 0.14717169, 0.14171445, 0.13645975, 0.13139947,
     &   0.12644352, 0.12159006, 0.11692380, 0.11243618, 0.10806518,
     &   0.10377247, 0.09971109, 0.09586556, 0.09222864, 0.08876160,
     &   0.08544204, 0.08226056, 0.07925844, 0.07639313, 0.07363933,
     &   0.07096922, 0.06836597, 0.06582751, 0.06334890, 0.06105047,
     &   0.05924718, 0.05746062, 0.05627456, 0.05715230, 0.05877275,
     &   0.05867526, 0.05726555, 0.05582485, 0.05459800, 0.05360858,
     &   0.05270187, 0.05203269, 0.05143032, 0.04981858, 0.04745170,
     &   0.04401491, 0.04130376, 0.03923849, 0.03793414, 0.03745477,
     &   0.03683023, 0.03625653, 0.03578776, 0.03541326, 0.03489154,
     &   0.03417274, 0.03334953, 0.03244369, 0.03154372, 0.03060090,
     &   0.02967687, 0.02873661, 0.02778024, 0.02684781, 0.02585855,
     &   0.02488202, 0.02391864, 0.02299587, 0.02212896, 0.02130678,
     &   0.02051512, 0.01975966, 0.01903583, 0.01833849, 0.01767066,
     &   0.01702731, 0.01641069, 0.01583209, 0.01527392, 0.01473545,
     &   0.01421598, 0.01371483, 0.01323136, 0.01276494, 0.01231497,
     &   0.01188087, 0.01146206, 0.01106562, 0.01068422, 0.01031594,
     &   0.00996032, 0.00961695, 0.00928539, 0.00896524, 0.00865611,
     &   0.00835762, 0.00806940, 0.00779111, 0.00752240, 0.00726364,
     &   0.00701630, 0.00677736, 0.00654652, 0.00632353, 0.00610812,
     &   0.00589994, 0.00569832, 0.00550354, 0.00531536, 0.00513181,
     &   0.00495333, 0.00478102, 0.00461404, 0.00445146, 0.00429457,
     &   0.00414313, 0.00399627, 0.00385461, 0.00371796, 0.00358624,
     &   0.00345920, 0.00333665, 0.00321841, 0.00310431, 0.00299426,
     &   0.00288812, 0.00278580, 0.00268710, 0.00259190, 0.00250024,
     &   0.00241182, 0.00232652, 0.00224433, 0.00216507, 0.00208861,
     &   0.00201496, 0.00194399, 0.00187553, 0.00180953, 0.00174603,
     &   0.00168475, 0.00162563, 0.00156877, 0.00151391, 0.00146096/
      data (ggi(i,3),i=1,mxwv)/
     &   0.85731214, 0.85751832, 0.85769582, 0.85763127, 0.85768604,
     &   0.85763150, 0.85752910, 0.85752320, 0.85745639, 0.85750520,
     &   0.85742235, 0.85738337, 0.85708910, 0.85693592, 0.85684413,
     &   0.85660189, 0.85646498, 0.85621679, 0.85595161, 0.85581541,
     &   0.85546273, 0.85530627, 0.85497016, 0.85468125, 0.85443079,
     &   0.85408998, 0.85377288, 0.85343581, 0.85305929, 0.85269463,
     &   0.85228992, 0.85192668, 0.85154629, 0.85099095, 0.85069889,
     &   0.85023099, 0.84971017, 0.84940726, 0.84888673, 0.84845698,
     &   0.84793580, 0.84745103, 0.84697056, 0.84644598, 0.84585786,
     &   0.84529459, 0.84478503, 0.84409571, 0.84356952, 0.84295863,
     &   0.84233254, 0.84167403, 0.84101832, 0.84032494, 0.83966959,
     &   0.83887935, 0.83813560, 0.83738202, 0.83658934, 0.83576691,
     &   0.83493620, 0.83403969, 0.83314896, 0.83222657, 0.83125597,
     &   0.83027440, 0.82925826, 0.82822347, 0.82715118, 0.82605857,
     &   0.82493895, 0.82379597, 0.82261276, 0.82141232, 0.82018697,
     &   0.81894636, 0.81769210, 0.81642693, 0.81515050, 0.81388038,
     &   0.81261504, 0.81136030, 0.81011015, 0.80886012, 0.80762446,
     &   0.80641752, 0.80524963, 0.80413151, 0.80308330, 0.80212706,
     &   0.80152977, 0.80151254, 0.80227512, 0.80231804, 0.80132455,
     &   0.80033767, 0.79948217, 0.79904205, 0.79899687, 0.79910892,
     &   0.79948801, 0.80015701, 0.80116028, 0.80247766, 0.80401713,
     &   0.80604094, 0.80891079, 0.81330240, 0.81870443, 0.82241613,
     &   0.82478410, 0.82702869, 0.82926118, 0.83249348, 0.83698964,
     &   0.84220809, 0.84800673, 0.85455728, 0.86140716, 0.86874408,
     &   0.87623596, 0.88429636, 0.89292371, 0.90218478, 0.91246170,
     &   0.92479515, 0.93736798, 0.95006543, 0.96054292, 0.96502995,
     &   0.95502400, 0.93037206, 0.91381347, 0.88978171, 0.87796861,
     &   0.88211477, 0.88510287, 0.87315297, 0.83507568, 0.78570020,
     &   0.75208068, 0.74180454, 0.74340969, 0.75362378, 0.76536810,
     &   0.77708125, 0.78805768, 0.79854274, 0.80850369, 0.81622535,
     &   0.82326323, 0.82998604, 0.83645207, 0.84328443, 0.84903896,
     &   0.85117799, 0.85002869, 0.84332681, 0.83905494, 0.83957446,
     &   0.84024423, 0.84191298, 0.84428531, 0.84667689, 0.84900165,
     &   0.85194093, 0.85488957, 0.85783505, 0.86078650, 0.86360306,
     &   0.86610591, 0.86431295, 0.86156297, 0.85709888, 0.85232991,
     &   0.84795195, 0.84499478, 0.84340358, 0.84156454, 0.83875245,
     &   0.83585775, 0.83315665, 0.83039367, 0.82746953, 0.82511652,
     &   0.82243043, 0.81926054, 0.81571627, 0.81220913, 0.80911303,
     &   0.80689496, 0.80533332, 0.80312854, 0.80077583, 0.79832697,
     &   0.79573309, 0.79282141, 0.78989345, 0.78687322, 0.78386104,
     &   0.78094399, 0.77781820, 0.77360320, 0.76741683, 0.75775671,
     &   0.74516881, 0.73197037, 0.72116786, 0.71116471, 0.70283335,
     &   0.69484293, 0.68759382, 0.68071568, 0.67414892, 0.66863853,
     &   0.66312528, 0.65776843, 0.65307921, 0.64809889, 0.64303142,
     &   0.63820076, 0.63349152, 0.62871897, 0.62365615, 0.61831939,
     &   0.61274010, 0.60681701, 0.60059381, 0.59411353, 0.58729154,
     &   0.58024347, 0.57309926, 0.56584638, 0.55838126, 0.55068594,
     &   0.54277521, 0.53458345, 0.52621317, 0.51783425, 0.50951731,
     &   0.50101513, 0.49203783, 0.48249784, 0.47260541, 0.46244609,
     &   0.45203879, 0.44136575, 0.43044001, 0.41943035, 0.40836319,
     &   0.39727059, 0.38614506, 0.37498906, 0.36385530, 0.35279247,
     &   0.34184843, 0.33106044, 0.32043698, 0.31000233, 0.29977682,
     &   0.28976548, 0.27989513, 0.27028051, 0.26092851, 0.25184318,
     &   0.24285835, 0.23398384, 0.22540589, 0.21711864, 0.20900692,
     &   0.20100541, 0.19343153, 0.18625684, 0.17946242, 0.17297190,
     &   0.16674344, 0.16075818, 0.15509589, 0.14967351, 0.14444257,
     &   0.13935012, 0.13436420, 0.12948166, 0.12468329, 0.12016890,
     &   0.11658015, 0.11294117, 0.11030856, 0.11156409, 0.11480656,
     &   0.11513623, 0.11266553, 0.11004021, 0.10780244, 0.10605422,
     &   0.10449169, 0.10344395, 0.10269877, 0.09976295, 0.09506969,
     &   0.08793950, 0.08237317, 0.07817542, 0.07553288, 0.07463518,
     &   0.07344513, 0.07236794, 0.07152173, 0.07089716, 0.06997933,
     &   0.06863035, 0.06704286, 0.06526260, 0.06348125, 0.06159545,
     &   0.05974023, 0.05784176, 0.05590114, 0.05400637, 0.05198567,
     &   0.04999034, 0.04802433, 0.04614521, 0.04438427, 0.04271778,
     &   0.04111495, 0.03958770, 0.03812613, 0.03671932, 0.03537368,
     &   0.03407831, 0.03283788, 0.03167582, 0.03055526, 0.02947469,
     &   0.02843264, 0.02742770, 0.02645853, 0.02552382, 0.02462232,
     &   0.02375283, 0.02291420, 0.02212080, 0.02135766, 0.02062088,
     &   0.01990955, 0.01922277, 0.01855971, 0.01791952, 0.01730142,
     &   0.01670465, 0.01612846, 0.01557215, 0.01503502, 0.01451783,
     &   0.01402353, 0.01354602, 0.01308474, 0.01263913, 0.01220867,
     &   0.01179267, 0.01138973, 0.01100046, 0.01062440, 0.01025755,
     &   0.00990084, 0.00955647, 0.00922273, 0.00889777, 0.00858420,
     &   0.00828153, 0.00798802, 0.00770489, 0.00743179, 0.00716854,
     &   0.00691466, 0.00666975, 0.00643344, 0.00620542, 0.00598548,
     &   0.00577337, 0.00556888, 0.00537163, 0.00518137, 0.00499819,
     &   0.00482147, 0.00465100, 0.00448674, 0.00432834, 0.00417553,
     &   0.00402831, 0.00388648, 0.00374964, 0.00361774, 0.00349081,
     &   0.00336834, 0.00325016, 0.00313652, 0.00302686, 0.00292103/
      data (ggi(i,4),i=1,mxwv)/
     &   0.86354595, 0.86387378, 0.86413205, 0.86419100, 0.86422288,
     &   0.86427242, 0.86433119, 0.86443490, 0.86452848, 0.86455113,
     &   0.86457717, 0.86457628, 0.86446059, 0.86437178, 0.86437982,
     &   0.86421132, 0.86415386, 0.86410737, 0.86397141, 0.86383533,
     &   0.86373389, 0.86348313, 0.86330944, 0.86324066, 0.86297172,
     &   0.86276370, 0.86253339, 0.86231703, 0.86193711, 0.86176825,
     &   0.86156946, 0.86123502, 0.86095035, 0.86073852, 0.86043340,
     &   0.86010987, 0.85968465, 0.85946202, 0.85907412, 0.85879147,
     &   0.85842693, 0.85807860, 0.85774779, 0.85727656, 0.85688710,
     &   0.85652345, 0.85615265, 0.85575706, 0.85528415, 0.85483712,
     &   0.85449290, 0.85404968, 0.85355610, 0.85312176, 0.85266221,
     &   0.85217464, 0.85172027, 0.85118085, 0.85069901, 0.85021353,
     &   0.84965587, 0.84921789, 0.84863478, 0.84808910, 0.84746087,
     &   0.84692204, 0.84639084, 0.84579170, 0.84516639, 0.84450972,
     &   0.84398657, 0.84330082, 0.84262276, 0.84194064, 0.84124148,
     &   0.84050077, 0.83976346, 0.83899713, 0.83821011, 0.83742809,
     &   0.83661664, 0.83580232, 0.83494693, 0.83405042, 0.83311909,
     &   0.83215487, 0.83116984, 0.83018994, 0.82916421, 0.82816952,
     &   0.82757306, 0.82770216, 0.82884902, 0.82878834, 0.82705367,
     &   0.82511985, 0.82308894, 0.82140428, 0.82006902, 0.81868273,
     &   0.81738156, 0.81619018, 0.81516963, 0.81427956, 0.81345153,
     &   0.81287485, 0.81317407, 0.81551582, 0.81953561, 0.82102489,
     &   0.82033187, 0.81902516, 0.81723422, 0.81641126, 0.81729722,
     &   0.81934178, 0.82237464, 0.82671618, 0.83210492, 0.83858246,
     &   0.84589791, 0.85473263, 0.86530066, 0.87780613, 0.89306289,
     &   0.91311407, 0.93446958, 0.95546496, 0.97123039, 0.97769755,
     &   0.96824318, 0.94388378, 0.92750490, 0.90339875, 0.89163810,
     &   0.89680916, 0.90399241, 0.90662426, 0.88911843, 0.84652519,
     &   0.79877818, 0.76724547, 0.74883264, 0.74460483, 0.74767816,
     &   0.75290531, 0.76069516, 0.77065963, 0.78178132, 0.79104638,
     &   0.80020177, 0.80996084, 0.82001293, 0.83169460, 0.84248418,
     &   0.84785253, 0.84818053, 0.83737630, 0.83020800, 0.83127660,
     &   0.83294308, 0.83665371, 0.84194493, 0.84751421, 0.85307115,
     &   0.85979772, 0.86658484, 0.87336135, 0.88012540, 0.88659400,
     &   0.89264452, 0.89250737, 0.89094067, 0.88682079, 0.88206333,
     &   0.87775147, 0.87540203, 0.87526834, 0.87495059, 0.87323362,
     &   0.87141740, 0.86994582, 0.86845738, 0.86677080, 0.86601371,
     &   0.86484480, 0.86303002, 0.86072308, 0.85854781, 0.85708266,
     &   0.85702944, 0.85800797, 0.85812747, 0.85812110, 0.85807353,
     &   0.85791397, 0.85742182, 0.85708213, 0.85679901, 0.85680407,
     &   0.85733366, 0.85806596, 0.85786474, 0.85587513, 0.85014510,
     &   0.84028894, 0.82874978, 0.81882548, 0.80963886, 0.80207026,
     &   0.79482174, 0.78824687, 0.78194654, 0.77581888, 0.77057147,
     &   0.76512247, 0.75965375, 0.75464767, 0.74935979, 0.74371111,
     &   0.73804140, 0.73263568, 0.72746491, 0.72283596, 0.71847183,
     &   0.71474427, 0.71100259, 0.70716035, 0.70332587, 0.69908053,
     &   0.69380438, 0.68804616, 0.68261850, 0.67729807, 0.67192739,
     &   0.66729426, 0.66284418, 0.65851980, 0.65340877, 0.64711559,
     &   0.64022440, 0.63332695, 0.62688315, 0.62094045, 0.61484838,
     &   0.60857582, 0.60221910, 0.59584361, 0.58918631, 0.58218908,
     &   0.57483935, 0.56710190, 0.55899405, 0.55052477, 0.54167122,
     &   0.53245729, 0.52289951, 0.51299149, 0.50276053, 0.49223804,
     &   0.48144868, 0.47034889, 0.45906255, 0.44763228, 0.43610087,
     &   0.42428482, 0.41219944, 0.40013352, 0.38813147, 0.37601596,
     &   0.36367574, 0.35172105, 0.34016061, 0.32910264, 0.31844157,
     &   0.30808532, 0.29801974, 0.28844926, 0.27922574, 0.27026030,
     &   0.26145995, 0.25276443, 0.24417309, 0.23559672, 0.22727811,
     &   0.22055119, 0.21342406, 0.20751449, 0.20776242, 0.21277854,
     &   0.21450683, 0.21097265, 0.20686291, 0.20325723, 0.20056030,
     &   0.19831036, 0.19712707, 0.19755411, 0.19415170, 0.18691573,
     &   0.17322907, 0.16198197, 0.15339760, 0.14774854, 0.14600536,
     &   0.14376703, 0.14182289, 0.14044149, 0.13971111, 0.13859823,
     &   0.13657883, 0.13400170, 0.13093664, 0.12778251, 0.12431413,
     &   0.12083100, 0.11719103, 0.11338978, 0.10961869, 0.10549580,
     &   0.10138208, 0.09729552, 0.09338834, 0.08973753, 0.08629080,
     &   0.08297966, 0.07983184, 0.07682534, 0.07393590, 0.07117860,
     &   0.06852850, 0.06599586, 0.06363205, 0.06135518, 0.05916191,
     &   0.05704901, 0.05501337, 0.05305203, 0.05116215, 0.04934097,
     &   0.04758589, 0.04589439, 0.04429648, 0.04276063, 0.04127862,
     &   0.03984851, 0.03846842, 0.03713657, 0.03585123, 0.03461072,
     &   0.03341345, 0.03225788, 0.03114253, 0.03006596, 0.02902969,
     &   0.02803972, 0.02708360, 0.02616015, 0.02526824, 0.02440680,
     &   0.02357438, 0.02276794, 0.02198900, 0.02123664, 0.02050266,
     &   0.01978899, 0.01910012, 0.01843258, 0.01778257, 0.01715545,
     &   0.01655023, 0.01596339, 0.01539737, 0.01485144, 0.01432528,
     &   0.01381786, 0.01332843, 0.01285618, 0.01240055, 0.01196107,
     &   0.01153727, 0.01112871, 0.01073461, 0.01035449, 0.00998852,
     &   0.00963548, 0.00929491, 0.00896676, 0.00865031, 0.00834502,
     &   0.00805093, 0.00776759, 0.00749421, 0.00723069, 0.00697712,
     &   0.00673243, 0.00649633, 0.00626929, 0.00605019, 0.00583875/
      data (ggi(i,5),i=1,mxwv)/
     &   0.86838096, 0.86875856, 0.86908442, 0.86922479, 0.86933666,
     &   0.86943299, 0.86961442, 0.86972928, 0.86985797, 0.86996901,
     &   0.87007201, 0.87014949, 0.87014484, 0.87012798, 0.87012559,
     &   0.87017876, 0.87011158, 0.87016016, 0.87010252, 0.87005627,
     &   0.87000841, 0.86992210, 0.86978942, 0.86971080, 0.86960274,
     &   0.86950409, 0.86931765, 0.86922896, 0.86904919, 0.86887044,
     &   0.86871040, 0.86852163, 0.86835635, 0.86814195, 0.86795092,
     &   0.86772966, 0.86752564, 0.86732262, 0.86705369, 0.86682278,
     &   0.86657232, 0.86627513, 0.86598843, 0.86579013, 0.86551493,
     &   0.86521584, 0.86494446, 0.86460012, 0.86430830, 0.86403984,
     &   0.86367214, 0.86339486, 0.86301410, 0.86268014, 0.86237687,
     &   0.86204690, 0.86167049, 0.86134934, 0.86091506, 0.86055094,
     &   0.86018711, 0.85983330, 0.85942614, 0.85913426, 0.85864145,
     &   0.85823268, 0.85784298, 0.85743535, 0.85708499, 0.85662627,
     &   0.85628480, 0.85584235, 0.85532409, 0.85493201, 0.85450584,
     &   0.85408115, 0.85353297, 0.85309374, 0.85271180, 0.85222512,
     &   0.85178685, 0.85136116, 0.85097802, 0.85041732, 0.84994894,
     &   0.84942216, 0.84892517, 0.84832358, 0.84786195, 0.84739792,
     &   0.84750330, 0.84861159, 0.85105985, 0.85187608, 0.85047966,
     &   0.84879333, 0.84692645, 0.84547734, 0.84446496, 0.84331936,
     &   0.84221393, 0.84117162, 0.84026653, 0.83942169, 0.83853674,
     &   0.83780444, 0.83813238, 0.84112406, 0.84615624, 0.84754169,
     &   0.84580934, 0.84287065, 0.83874601, 0.83521438, 0.83340871,
     &   0.83265251, 0.83260894, 0.83360553, 0.83550674, 0.83796716,
     &   0.84081024, 0.84482038, 0.85049701, 0.85833484, 0.87017781,
     &   0.89104879, 0.91909635, 0.95126897, 0.97576815, 0.98497438,
     &   0.97604096, 0.95203799, 0.93583339, 0.91169918, 0.89989978,
     &   0.90539831, 0.91353375, 0.92159641, 0.91971308, 0.89743727,
     &   0.85954458, 0.82665408, 0.80163753, 0.78942090, 0.78513712,
     &   0.78199524, 0.78234386, 0.78613597, 0.79192972, 0.79652435,
     &   0.80151880, 0.80843174, 0.81661129, 0.82880801, 0.84197265,
     &   0.84944063, 0.85108382, 0.83348817, 0.81915843, 0.81601423,
     &   0.81419849, 0.81609708, 0.82153934, 0.82835704, 0.83576417,
     &   0.84591317, 0.85681725, 0.86820453, 0.88001001, 0.89146018,
     &   0.90276057, 0.90520024, 0.90522403, 0.90135294, 0.89595729,
     &   0.89067113, 0.88753659, 0.88812786, 0.88899964, 0.88778406,
     &   0.88631666, 0.88539517, 0.88455653, 0.88332641, 0.88366002,
     &   0.88352442, 0.88248128, 0.88063532, 0.87893230, 0.87829667,
     &   0.87993032, 0.88337511, 0.88559616, 0.88768411, 0.88976085,
     &   0.89172405, 0.89324391, 0.89502597, 0.89694661, 0.89934087,
     &   0.90253925, 0.90608829, 0.90853316, 0.90901518, 0.90537524,
     &   0.89698350, 0.88647991, 0.87727243, 0.86880010, 0.86195111,
     &   0.85545212, 0.84965134, 0.84414417, 0.83881807, 0.83432990,
     &   0.82958782, 0.82473695, 0.82013643, 0.81513250, 0.80945992,
     &   0.80329686, 0.79705817, 0.79085726, 0.78564465, 0.78108543,
     &   0.77800393, 0.77534139, 0.77286035, 0.77071422, 0.76825815,
     &   0.76409483, 0.75902587, 0.75453782, 0.75040478, 0.74637717,
     &   0.74396956, 0.74215633, 0.74089068, 0.73829252, 0.73334700,
     &   0.72721839, 0.72127998, 0.71663326, 0.71340317, 0.71021879,
     &   0.70704192, 0.70416385, 0.70184112, 0.69941545, 0.69676477,
     &   0.69385946, 0.69065410, 0.68723279, 0.68362385, 0.67970580,
     &   0.67550164, 0.67100930, 0.66612589, 0.66083783, 0.65513271,
     &   0.64899886, 0.64241886, 0.63537943, 0.62787741, 0.61991292,
     &   0.61142838, 0.60235256, 0.59274513, 0.58263242, 0.57184297,
     &   0.56014174, 0.54799092, 0.53545177, 0.52293658, 0.51045066,
     &   0.49785882, 0.48519766, 0.47292006, 0.46087933, 0.44897196,
     &   0.43708691, 0.42513299, 0.41312411, 0.40079215, 0.38826188,
     &   0.37798098, 0.36652920, 0.35578287, 0.35208037, 0.35617396,
     &   0.35890427, 0.35470846, 0.34929320, 0.34410274, 0.34002972,
     &   0.33670491, 0.33484378, 0.33721039, 0.33651474, 0.33140498,
     &   0.31408641, 0.29631740, 0.28128579, 0.26981828, 0.26586258,
     &   0.26146874, 0.25774226, 0.25517273, 0.25425673, 0.25364563,
     &   0.25183871, 0.24921700, 0.24579599, 0.24210177, 0.23774494,
     &   0.23316461, 0.22822882, 0.22284484, 0.21719675, 0.21052060,
     &   0.20350261, 0.19604886, 0.18869008, 0.18169537, 0.17497513,
     &   0.16843066, 0.16214304, 0.15608481, 0.15021770, 0.14458449,
     &   0.13914658, 0.13393623, 0.12907989, 0.12439491, 0.11987738,
     &   0.11552306, 0.11132742, 0.10728579, 0.10339330, 0.09964504,
     &   0.09603606, 0.09256144, 0.08928916, 0.08614790, 0.08311960,
     &   0.08020022, 0.07738582, 0.07467259, 0.07205678, 0.06953481,
     &   0.06710318, 0.06475854, 0.06249764, 0.06031736, 0.05822078,
     &   0.05622032, 0.05428965, 0.05242627, 0.05062773, 0.04889172,
     &   0.04721497, 0.04559001, 0.04402156, 0.04250762, 0.04103084,
     &   0.03959535, 0.03821066, 0.03686936, 0.03556348, 0.03430438,
     &   0.03308989, 0.03191287, 0.03077812, 0.02968409, 0.02863011,
     &   0.02761402, 0.02663424, 0.02568915, 0.02477753, 0.02389844,
     &   0.02305088, 0.02223398, 0.02144614, 0.02068634, 0.01995495,
     &   0.01924949, 0.01856903, 0.01791344, 0.01728128, 0.01667147,
     &   0.01608405, 0.01551815, 0.01497217, 0.01444589, 0.01393951,
     &   0.01345088, 0.01297937, 0.01252600, 0.01208848, 0.01166624/
      data (ggi(i,6),i=1,mxwv)/
     &   0.87211460, 0.87250054, 0.87290180, 0.87308013, 0.87326646,
     &   0.87341970, 0.87361413, 0.87382364, 0.87396908, 0.87413549,
     &   0.87432647, 0.87444997, 0.87450641, 0.87457401, 0.87463570,
     &   0.87466305, 0.87472379, 0.87478226, 0.87484366, 0.87484998,
     &   0.87487835, 0.87483084, 0.87478560, 0.87476498, 0.87468469,
     &   0.87468803, 0.87460333, 0.87453461, 0.87444782, 0.87436163,
     &   0.87427795, 0.87414104, 0.87405950, 0.87393260, 0.87376136,
     &   0.87368381, 0.87353325, 0.87337106, 0.87320298, 0.87311167,
     &   0.87286401, 0.87272090, 0.87253100, 0.87235028, 0.87214327,
     &   0.87196612, 0.87184113, 0.87153298, 0.87133908, 0.87115264,
     &   0.87092948, 0.87066299, 0.87039733, 0.87019694, 0.86992651,
     &   0.86968297, 0.86932713, 0.86921030, 0.86894697, 0.86864859,
     &   0.86842763, 0.86811930, 0.86784267, 0.86755341, 0.86729580,
     &   0.86698449, 0.86662751, 0.86644572, 0.86605936, 0.86583889,
     &   0.86540848, 0.86521250, 0.86488497, 0.86455214, 0.86427909,
     &   0.86390573, 0.86357230, 0.86330497, 0.86288124, 0.86265224,
     &   0.86242467, 0.86211646, 0.86187357, 0.86157715, 0.86127752,
     &   0.86089891, 0.86059821, 0.86025608, 0.85976595, 0.85972464,
     &   0.86027038, 0.86210948, 0.86564142, 0.86713147, 0.86586392,
     &   0.86425000, 0.86241645, 0.86112815, 0.86041868, 0.85955983,
     &   0.85877168, 0.85808843, 0.85759044, 0.85720241, 0.85677582,
     &   0.85656017, 0.85777766, 0.86247283, 0.86962467, 0.87208539,
     &   0.87064701, 0.86762863, 0.86295080, 0.85875630, 0.85657376,
     &   0.85556304, 0.85523623, 0.85594505, 0.85749227, 0.85907441,
     &   0.86033303, 0.86195737, 0.86424363, 0.86721301, 0.87249482,
     &   0.88623512, 0.90951407, 0.94370192, 0.97657281, 0.98917103,
     &   0.98069805, 0.95706266, 0.94099015, 0.91681218, 0.90492719,
     &   0.91054976, 0.91885984, 0.92823571, 0.93279898, 0.92492247,
     &   0.90004659, 0.87198770, 0.84781528, 0.83580482, 0.83194387,
     &   0.82864749, 0.82871908, 0.83215129, 0.83719033, 0.84045821,
     &   0.84361374, 0.84855461, 0.85447776, 0.86481184, 0.87677634,
     &   0.88387138, 0.88597357, 0.86657989, 0.84795535, 0.83949620,
     &   0.83192277, 0.82816517, 0.82890147, 0.83179164, 0.83569789,
     &   0.84396309, 0.85431594, 0.86652118, 0.88070738, 0.89532304,
     &   0.91126752, 0.91691226, 0.91885489, 0.91596919, 0.91046572,
     &   0.90414584, 0.89879417, 0.89847970, 0.89956403, 0.89800155,
     &   0.89580387, 0.89425510, 0.89295840, 0.89078230, 0.89101493,
     &   0.89092451, 0.88964587, 0.88690472, 0.88396615, 0.88226908,
     &   0.88417518, 0.88977790, 0.89375889, 0.89769417, 0.90174347,
     &   0.90575397, 0.90909624, 0.91279995, 0.91673136, 0.92132956,
     &   0.92706025, 0.93325996, 0.93797249, 0.94030994, 0.93805128,
     &   0.93054593, 0.92065972, 0.91191608, 0.90392560, 0.89760011,
     &   0.89168519, 0.88654560, 0.88179141, 0.87732542, 0.87381166,
     &   0.87015879, 0.86649358, 0.86309236, 0.85923648, 0.85460222,
     &   0.84910703, 0.84282571, 0.83566451, 0.82895350, 0.82263029,
     &   0.81825483, 0.81479764, 0.81197995, 0.81001514, 0.80822921,
     &   0.80411553, 0.79847056, 0.79342633, 0.78900969, 0.78498912,
     &   0.78381354, 0.78382361, 0.78513134, 0.78462899, 0.78040040,
     &   0.77416027, 0.76818442, 0.76442868, 0.76321661, 0.76226634,
     &   0.76150346, 0.76140571, 0.76240152, 0.76339161, 0.76417524,
     &   0.76470971, 0.76493782, 0.76502168, 0.76503181, 0.76478058,
     &   0.76432908, 0.76370573, 0.76275116, 0.76146227, 0.75983441,
     &   0.75786787, 0.75559944, 0.75295484, 0.74992520, 0.74650025,
     &   0.74282122, 0.73880482, 0.73424256, 0.72912103, 0.72342581,
     &   0.71691245, 0.70940202, 0.70085424, 0.69167840, 0.68200642,
     &   0.67168874, 0.66075504, 0.64975744, 0.63865614, 0.62738830,
     &   0.61587459, 0.60400933, 0.59181446, 0.57879215, 0.56474072,
     &   0.55312717, 0.53957844, 0.52570742, 0.51755250, 0.51795936,
     &   0.51969588, 0.51537287, 0.50945234, 0.50308090, 0.49731916,
     &   0.49214229, 0.48782438, 0.48847565, 0.48954117, 0.48949090,
     &   0.47991022, 0.46390310, 0.44673562, 0.43007836, 0.42219958,
     &   0.41447735, 0.40761873, 0.40211445, 0.39889261, 0.39729971,
     &   0.39517438, 0.39266431, 0.38974142, 0.38661045, 0.38304085,
     &   0.37923121, 0.37536398, 0.37126458, 0.36670384, 0.36101365,
     &   0.35461843, 0.34689870, 0.33870435, 0.33049348, 0.32215154,
     &   0.31368065, 0.30518368, 0.29666618, 0.28810906, 0.27955130,
     &   0.27101874, 0.26258501, 0.25448290, 0.24647875, 0.23858696,
     &   0.23082100, 0.22319329, 0.21571513, 0.20839651, 0.20124617,
     &   0.19427152, 0.18747862, 0.18104693, 0.17482202, 0.16877306,
     &   0.16290158, 0.15720814, 0.15169252, 0.14635366, 0.14118978,
     &   0.13619852, 0.13137695, 0.12672170, 0.12222897, 0.11790897,
     &   0.11379270, 0.10982004, 0.10598692, 0.10228913, 0.09872243,
     &   0.09527965, 0.09194119, 0.08872246, 0.08561946, 0.08259300,
     &   0.07965332, 0.07682229, 0.07408309, 0.07141794, 0.06885269,
     &   0.06638243, 0.06399202, 0.06169080, 0.05947514, 0.05734333,
     &   0.05529058, 0.05331330, 0.05140788, 0.04957163, 0.04780236,
     &   0.04609793, 0.04445642, 0.04287431, 0.04134941, 0.03988253,
     &   0.03846831, 0.03710479, 0.03579175, 0.03452610, 0.03330557,
     &   0.03213032, 0.03099849, 0.02990673, 0.02885461, 0.02784255,
     &   0.02686610, 0.02592400, 0.02501832, 0.02414439, 0.02330104/
      data (ggi(i,7),i=1,mxwv)/
     &   0.87489736, 0.87536228, 0.87580419, 0.87601978, 0.87623501,
     &   0.87644792, 0.87665993, 0.87689215, 0.87713051, 0.87734789,
     &   0.87764251, 0.87774098, 0.87784582, 0.87795979, 0.87807024,
     &   0.87815166, 0.87831157, 0.87835789, 0.87845975, 0.87846845,
     &   0.87857407, 0.87858820, 0.87860405, 0.87861580, 0.87863743,
     &   0.87864137, 0.87864751, 0.87862623, 0.87860060, 0.87858325,
     &   0.87852675, 0.87849015, 0.87841409, 0.87838513, 0.87829834,
     &   0.87822765, 0.87812150, 0.87808865, 0.87809438, 0.87776786,
     &   0.87778407, 0.87766737, 0.87752956, 0.87743974, 0.87739658,
     &   0.87720048, 0.87703884, 0.87692267, 0.87676561, 0.87664968,
     &   0.87648410, 0.87628216, 0.87616801, 0.87598616, 0.87584031,
     &   0.87567508, 0.87547106, 0.87526476, 0.87506282, 0.87490451,
     &   0.87471068, 0.87456602, 0.87435168, 0.87417197, 0.87394166,
     &   0.87375855, 0.87356663, 0.87334198, 0.87316120, 0.87294936,
     &   0.87274522, 0.87257886, 0.87229490, 0.87211519, 0.87182957,
     &   0.87166476, 0.87139982, 0.87119293, 0.87101024, 0.87079394,
     &   0.87066066, 0.87049103, 0.87038189, 0.87018633, 0.86990988,
     &   0.86973697, 0.86951125, 0.86929208, 0.86912382, 0.86900789,
     &   0.86991549, 0.87243515, 0.87710911, 0.87919140, 0.87782729,
     &   0.87602842, 0.87392509, 0.87254107, 0.87187999, 0.87102556,
     &   0.87026834, 0.86964446, 0.86927855, 0.86905372, 0.86878699,
     &   0.86880833, 0.87063873, 0.87685049, 0.88602275, 0.88939589,
     &   0.88808012, 0.88485980, 0.87957078, 0.87481499, 0.87254775,
     &   0.87176722, 0.87187141, 0.87332505, 0.87585932, 0.87834668,
     &   0.88025701, 0.88232601, 0.88468713, 0.88688713, 0.89011818,
     &   0.90051132, 0.91849756, 0.94621319, 0.97738570, 0.99166113,
     &   0.98353404, 0.96022242, 0.94422466, 0.91996735, 0.90798366,
     &   0.91369164, 0.92208660, 0.93176490, 0.93848187, 0.93865818,
     &   0.92570221, 0.90386218, 0.88089544, 0.86870730, 0.86527020,
     &   0.86260271, 0.86380816, 0.86881191, 0.87544841, 0.87994230,
     &   0.88412040, 0.88986540, 0.89619434, 0.90621847, 0.91684711,
     &   0.92278701, 0.92461038, 0.90893048, 0.89198440, 0.88335657,
     &   0.87497205, 0.86984807, 0.86897194, 0.86998409, 0.87157649,
     &   0.87733382, 0.88502771, 0.89454907, 0.90629107, 0.91854757,
     &   0.93262810, 0.93836540, 0.94045883, 0.93885386, 0.93500090,
     &   0.93000871, 0.92455918, 0.92306060, 0.92336857, 0.92147356,
     &   0.91870147, 0.91629559, 0.91411424, 0.91060263, 0.90945196,
     &   0.90826494, 0.90590507, 0.90141857, 0.89575839, 0.89032817,
     &   0.88886052, 0.89370137, 0.89714098, 0.90097338, 0.90540570,
     &   0.91028595, 0.91436553, 0.91899598, 0.92416441, 0.93040222,
     &   0.93848342, 0.94747895, 0.95448643, 0.95845783, 0.95725816,
     &   0.95037115, 0.94089997, 0.93247509, 0.92481881, 0.91886896,
     &   0.91338760, 0.90876234, 0.90462297, 0.90089512, 0.89829606,
     &   0.89574659, 0.89340532, 0.89162701, 0.88958377, 0.88709831,
     &   0.88413751, 0.88041413, 0.87537229, 0.86956131, 0.86294156,
     &   0.85715687, 0.85191441, 0.84723741, 0.84357655, 0.84082448,
     &   0.83556044, 0.82788473, 0.81990546, 0.81233013, 0.80530614,
     &   0.80261242, 0.80202776, 0.80450666, 0.80521810, 0.80033422,
     &   0.79176861, 0.78295738, 0.77750510, 0.77663648, 0.77659875,
     &   0.77723813, 0.77931231, 0.78351158, 0.78797728, 0.79230440,
     &   0.79638225, 0.80008072, 0.80365342, 0.80720711, 0.81043726,
     &   0.81344217, 0.81626815, 0.81867486, 0.82067162, 0.82226688,
     &   0.82347870, 0.82439262, 0.82490426, 0.82501912, 0.82474071,
     &   0.82436675, 0.82382596, 0.82276607, 0.82118684, 0.81923968,
     &   0.81676531, 0.81319636, 0.80846018, 0.80287498, 0.79663545,
     &   0.78960156, 0.78176433, 0.77360624, 0.76516479, 0.75640935,
     &   0.74729240, 0.73770267, 0.72765034, 0.71648002, 0.70364016,
     &   0.69305831, 0.68020046, 0.66625619, 0.65638417, 0.65435702,
     &   0.65504020, 0.65146577, 0.64630198, 0.64015591, 0.63388097,
     &   0.62770528, 0.62131339, 0.61806768, 0.61758286, 0.61917257,
     &   0.62052071, 0.61365849, 0.60157758, 0.58548135, 0.57491457,
     &   0.56512362, 0.55591118, 0.54745370, 0.54058194, 0.53571296,
     &   0.53117913, 0.52682227, 0.52265739, 0.51838368, 0.51422042,
     &   0.50994319, 0.50600219, 0.50230789, 0.49837613, 0.49449265,
     &   0.49037835, 0.48558050, 0.48037100, 0.47487730, 0.46907040,
     &   0.46305850, 0.45682245, 0.45037875, 0.44373208, 0.43682143,
     &   0.42971131, 0.42241168, 0.41499060, 0.40742370, 0.39971462,
     &   0.39186892, 0.38389328, 0.37579608, 0.36758679, 0.35927647,
     &   0.35087746, 0.34240332, 0.33411080, 0.32582510, 0.31751582,
     &   0.30919680, 0.30088246, 0.29258755, 0.28432706, 0.27611610,
     &   0.26796952, 0.25990200, 0.25192770, 0.24406014, 0.23634697,
     &   0.22888875, 0.22156101, 0.21437353, 0.20733497, 0.20045301,
     &   0.19373281, 0.18716472, 0.18075943, 0.17452340, 0.16837765,
     &   0.16235748, 0.15652721, 0.15085737, 0.14531362, 0.13996680,
     &   0.13481236, 0.12982033, 0.12501794, 0.12039964, 0.11596362,
     &   0.11170006, 0.10760162, 0.10366062, 0.09987106, 0.09622794,
     &   0.09272630, 0.08936160, 0.08612531, 0.08301214, 0.08002397,
     &   0.07714792, 0.07437937, 0.07171776, 0.06915582, 0.06668823,
     &   0.06431539, 0.06203297, 0.05983338, 0.05771556, 0.05568038,
     &   0.05371813, 0.05182608, 0.05000858, 0.04825566, 0.04656487/
      data (ggi(i,8),i=1,mxwv)/
     &   0.87702900, 0.87750286, 0.87797087, 0.87823170, 0.87848151,
     &   0.87873423, 0.87918609, 0.87923157, 0.87950087, 0.87977129,
     &   0.88003421, 0.88023639, 0.88036680, 0.88052553, 0.88067180,
     &   0.88080782, 0.88095564, 0.88109183, 0.88121873, 0.88132393,
     &   0.88141263, 0.88146991, 0.88155776, 0.88160396, 0.88164806,
     &   0.88169473, 0.88173652, 0.88174814, 0.88177133, 0.88178670,
     &   0.88177919, 0.88179058, 0.88179141, 0.88177693, 0.88174957,
     &   0.88173079, 0.88169396, 0.88167149, 0.88161951, 0.88156986,
     &   0.88150948, 0.88146520, 0.88139462, 0.88134205, 0.88123500,
     &   0.88119894, 0.88110691, 0.88102382, 0.88094038, 0.88086313,
     &   0.88075858, 0.88066620, 0.88053167, 0.88045263, 0.88034385,
     &   0.88023865, 0.88012064, 0.88000244, 0.87989795, 0.87980473,
     &   0.87965238, 0.87953591, 0.87940776, 0.87927681, 0.87918669,
     &   0.87899941, 0.87889147, 0.87872767, 0.87860870, 0.87849033,
     &   0.87835717, 0.87824869, 0.87808877, 0.87795341, 0.87779617,
     &   0.87764204, 0.87747824, 0.87734622, 0.87714136, 0.87713099,
     &   0.87707418, 0.87704492, 0.87699425, 0.87693119, 0.87680399,
     &   0.87670964, 0.87650484, 0.87638730, 0.87632322, 0.87632942,
     &   0.87761837, 0.88097847, 0.88703430, 0.88978845, 0.88823950,
     &   0.88613898, 0.88365626, 0.88202226, 0.88132119, 0.88037926,
     &   0.87954634, 0.87886858, 0.87853670, 0.87834972, 0.87814313,
     &   0.87824601, 0.88066286, 0.88850969, 0.89980119, 0.90393406,
     &   0.90249646, 0.89875138, 0.89242822, 0.88661581, 0.88382643,
     &   0.88288945, 0.88306129, 0.88492942, 0.88818324, 0.89138317,
     &   0.89390159, 0.89665413, 0.89980876, 0.90275431, 0.90675056,
     &   0.91843033, 0.93579048, 0.95839447, 0.98161435, 0.99325639,
     &   0.98530501, 0.96223730, 0.94626170, 0.92189318, 0.90981185,
     &   0.91559756, 0.92407387, 0.93385553, 0.94116622, 0.94491476,
     &   0.94105834, 0.92730904, 0.90775907, 0.89568383, 0.89203686,
     &   0.88915932, 0.89056623, 0.89604878, 0.90325737, 0.90821546,
     &   0.91282392, 0.91883940, 0.92519528, 0.93424124, 0.94278073,
     &   0.94702858, 0.94815660, 0.93735641, 0.92395926, 0.91687906,
     &   0.90968078, 0.90542096, 0.90524334, 0.90681678, 0.90879345,
     &   0.91443926, 0.92136651, 0.92925239, 0.93816435, 0.94654000,
     &   0.95505744, 0.95787472, 0.95849162, 0.95731509, 0.95522135,
     &   0.95260626, 0.94966692, 0.94867682, 0.94865847, 0.94739848,
     &   0.94557375, 0.94386870, 0.94223994, 0.93958068, 0.93831581,
     &   0.93703747, 0.93492013, 0.93099838, 0.92550969, 0.91903788,
     &   0.91473961, 0.91643131, 0.91713780, 0.91831559, 0.92027193,
     &   0.92310011, 0.92520738, 0.92783654, 0.93134832, 0.93639177,
     &   0.94450635, 0.95508730, 0.96386683, 0.96907544, 0.96861637,
     &   0.96215349, 0.95297867, 0.94479007, 0.93737257, 0.93168330,
     &   0.92649525, 0.92221189, 0.91847128, 0.91521281, 0.91320026,
     &   0.91135025, 0.90985376, 0.90917945, 0.90843785, 0.90761822,
     &   0.90696359, 0.90629041, 0.90511256, 0.90318638, 0.90028328,
     &   0.89703238, 0.89341265, 0.88959008, 0.88600516, 0.88303906,
     &   0.87823004, 0.87100232, 0.86207819, 0.85224664, 0.84193170,
     &   0.83532202, 0.83073848, 0.83093196, 0.83064669, 0.82378435,
     &   0.81089908, 0.79541075, 0.78279436, 0.77687871, 0.77268893,
     &   0.77015036, 0.77058202, 0.77534741, 0.78157002, 0.78836071,
     &   0.79537594, 0.80218756, 0.80914634, 0.81638634, 0.82332432,
     &   0.83008820, 0.83671045, 0.84274369, 0.84819371, 0.85307109,
     &   0.85740441, 0.86131197, 0.86467367, 0.86750984, 0.86983991,
     &   0.87210381, 0.87422240, 0.87571764, 0.87661237, 0.87718785,
     &   0.87732846, 0.87623101, 0.87385130, 0.87049812, 0.86643070,
     &   0.86153626, 0.85580093, 0.84966624, 0.84321105, 0.83642268,
     &   0.82927221, 0.82164830, 0.81355280, 0.80425525, 0.79299402,
     &   0.78385818, 0.77237499, 0.75951463, 0.75006205, 0.74803442,
     &   0.74895465, 0.74660075, 0.74277550, 0.73779410, 0.73229510,
     &   0.72656864, 0.71997279, 0.71447879, 0.71187681, 0.71243256,
     &   0.72048289, 0.72161776, 0.71573585, 0.70331126, 0.69275677,
     &   0.68329298, 0.67407542, 0.66495985, 0.65636337, 0.64919442,
     &   0.64264446, 0.63639814, 0.63052559, 0.62444240, 0.61868644,
     &   0.61278033, 0.60712254, 0.60171598, 0.59619337, 0.59167331,
     &   0.58739585, 0.58383155, 0.58033121, 0.57650852, 0.57254881,
     &   0.56853402, 0.56437922, 0.56011420, 0.55577713, 0.55135763,
     &   0.54684478, 0.54219896, 0.53729630, 0.53229904, 0.52720213,
     &   0.52200025, 0.51668799, 0.51125973, 0.50570977, 0.50003225,
     &   0.49422151, 0.48827195, 0.48218122, 0.47597626, 0.46965465,
     &   0.46321201, 0.45664436, 0.44994792, 0.44312108, 0.43616062,
     &   0.42906573, 0.42183623, 0.41447309, 0.40697780, 0.39938506,
     &   0.39179289, 0.38409445, 0.37629509, 0.36840087, 0.36041933,
     &   0.35237446, 0.34434521, 0.33623254, 0.32804963, 0.31970280,
     &   0.31122857, 0.30272046, 0.29415566, 0.28549361, 0.27685335,
     &   0.26825371, 0.25964844, 0.25114310, 0.24275860, 0.23452401,
     &   0.22644874, 0.21854599, 0.21082552, 0.20329954, 0.19598044,
     &   0.18887758, 0.18199916, 0.17534190, 0.16890720, 0.16271180,
     &   0.15673484, 0.15097319, 0.14543256, 0.14010046, 0.13496809,
     &   0.13004008, 0.12530832, 0.12075571, 0.11638103, 0.11218712,
     &   0.10815117, 0.10426702, 0.10054480, 0.09696128, 0.09351071/
      data (ggi(i,9),i=1,mxwv)/
     &   0.87861216, 0.87912714, 0.87961048, 0.87989771, 0.88015419,
     &   0.88045174, 0.88072830, 0.88100988, 0.88131195, 0.88159961,
     &   0.88191193, 0.88211364, 0.88225728, 0.88245320, 0.88263482,
     &   0.88280320, 0.88297671, 0.88315302, 0.88330525, 0.88342798,
     &   0.88357282, 0.88366103, 0.88373339, 0.88384557, 0.88393360,
     &   0.88399094, 0.88407284, 0.88412911, 0.88418782, 0.88423204,
     &   0.88427269, 0.88429838, 0.88433433, 0.88435364, 0.88437861,
     &   0.88439059, 0.88438499, 0.88441080, 0.88438809, 0.88436329,
     &   0.88437539, 0.88435322, 0.88433629, 0.88432211, 0.88421333,
     &   0.88425434, 0.88420820, 0.88416022, 0.88414431, 0.88405228,
     &   0.88404387, 0.88398880, 0.88393426, 0.88387388, 0.88382107,
     &   0.88376039, 0.88370198, 0.88363826, 0.88354838, 0.88348168,
     &   0.88342309, 0.88335329, 0.88325542, 0.88320518, 0.88313800,
     &   0.88304681, 0.88296896, 0.88289082, 0.88281608, 0.88278264,
     &   0.88269329, 0.88265270, 0.88258392, 0.88247150, 0.88237703,
     &   0.88225985, 0.88217372, 0.88211381, 0.88202369, 0.88204759,
     &   0.88208342, 0.88218290, 0.88223863, 0.88224030, 0.88216460,
     &   0.88207263, 0.88198572, 0.88195962, 0.88192153, 0.88211077,
     &   0.88384169, 0.88824773, 0.89600974, 0.89953816, 0.89772618,
     &   0.89521414, 0.89219284, 0.89022022, 0.88942087, 0.88831621,
     &   0.88734370, 0.88657820, 0.88619983, 0.88603407, 0.88583034,
     &   0.88600457, 0.88909823, 0.89893109, 0.91256785, 0.91742581,
     &   0.91582614, 0.91149068, 0.90394008, 0.89676589, 0.89320081,
     &   0.89191365, 0.89198589, 0.89414859, 0.89801866, 0.90173972,
     &   0.90456200, 0.90759873, 0.91106200, 0.91421902, 0.91852558,
     &   0.93224186, 0.95138603, 0.97166264, 0.98712981, 0.99436945,
     &   0.98643923, 0.96352768, 0.94753546, 0.92303532, 0.91086286,
     &   0.91672194, 0.92527878, 0.93513221, 0.94259757, 0.94754589,
     &   0.94885170, 0.94281578, 0.92904377, 0.91857743, 0.91510749,
     &   0.91221404, 0.91349638, 0.91887504, 0.92572075, 0.93032610,
     &   0.93451089, 0.93975532, 0.94502383, 0.95185995, 0.95752108,
     &   0.95991820, 0.96029598, 0.95403159, 0.94481820, 0.93950540,
     &   0.93371493, 0.93027365, 0.93050712, 0.93235248, 0.93456882,
     &   0.93987775, 0.94588572, 0.95207924, 0.95822030, 0.96320325,
     &   0.96720278, 0.96763885, 0.96713793, 0.96598178, 0.96477473,
     &   0.96359318, 0.96261293, 0.96236354, 0.96231294, 0.96169502,
     &   0.96089458, 0.96016288, 0.95942676, 0.95826197, 0.95773053,
     &   0.95711833, 0.95604551, 0.95402884, 0.95104790, 0.94718951,
     &   0.94423604, 0.94497985, 0.94494987, 0.94507951, 0.94558775,
     &   0.94661868, 0.94698578, 0.94739658, 0.94822210, 0.94993538,
     &   0.95441270, 0.96255887, 0.97056401, 0.97573411, 0.97548461,
     &   0.96925932, 0.96029502, 0.95228016, 0.94502527, 0.93949974,
     &   0.93448442, 0.93039072, 0.92685962, 0.92383349, 0.92209685,
     &   0.92055541, 0.91941339, 0.91918170, 0.91895413, 0.91877759,
     &   0.91903824, 0.91970581, 0.92050463, 0.92105937, 0.92119813,
     &   0.92098552, 0.92030317, 0.91923994, 0.91800463, 0.91666961,
     &   0.91434044, 0.91074061, 0.90597856, 0.90001529, 0.89289916,
     &   0.88731909, 0.88245511, 0.88105732, 0.87982208, 0.87359488,
     &   0.86139870, 0.84443849, 0.82708508, 0.81474429, 0.80317056,
     &   0.79265922, 0.78519297, 0.78332424, 0.78440309, 0.78737509,
     &   0.79183400, 0.79694194, 0.80320889, 0.81091559, 0.81908399,
     &   0.82786471, 0.83720589, 0.84616733, 0.85464323, 0.86255389,
     &   0.86985534, 0.87659317, 0.88265854, 0.88805199, 0.89278567,
     &   0.89742756, 0.90187275, 0.90550929, 0.90838367, 0.91091967,
     &   0.91301948, 0.91366345, 0.91285694, 0.91091794, 0.90818310,
     &   0.90457714, 0.90009391, 0.89516687, 0.88990366, 0.88430488,
     &   0.87835324, 0.87193900, 0.86506182, 0.85695267, 0.84675574,
     &   0.83866948, 0.82823300, 0.81638521, 0.80800879, 0.80724674,
     &   0.80938607, 0.80833244, 0.80586404, 0.80228645, 0.79818672,
     &   0.79382873, 0.78858125, 0.78370869, 0.78042197, 0.77895218,
     &   0.78832126, 0.79356980, 0.79157293, 0.78230286, 0.77314705,
     &   0.76518047, 0.75736427, 0.74942994, 0.74151385, 0.73438507,
     &   0.72764087, 0.72093368, 0.71433550, 0.70726514, 0.70031285,
     &   0.69300628, 0.68547165, 0.67774314, 0.66982085, 0.66321880,
     &   0.65712357, 0.65319151, 0.64990377, 0.64639097, 0.64303654,
     &   0.63975614, 0.63645667, 0.63315028, 0.62987131, 0.62674826,
     &   0.62361676, 0.62042904, 0.61702919, 0.61358297, 0.61008787,
     &   0.60654104, 0.60293984, 0.59928191, 0.59556472, 0.59178591,
     &   0.58794308, 0.58403379, 0.57974291, 0.57535118, 0.57091630,
     &   0.56643617, 0.56190848, 0.55733067, 0.55269986, 0.54801291,
     &   0.54326642, 0.53845650, 0.53357917, 0.52862978, 0.52357388,
     &   0.51834142, 0.51305032, 0.50769496, 0.50226957, 0.49676803,
     &   0.49117112, 0.48541486, 0.47958785, 0.47368118, 0.46767673,
     &   0.46155202, 0.45529982, 0.44889754, 0.44231009, 0.43555477,
     &   0.42862305, 0.42148268, 0.41414553, 0.40661177, 0.39889035,
     &   0.39098251, 0.38289407, 0.37463105, 0.36620566, 0.35763535,
     &   0.34893969, 0.34014118, 0.33125597, 0.32230702, 0.31333917,
     &   0.30435616, 0.29538226, 0.28645572, 0.27759159, 0.26880825,
     &   0.26014522, 0.25162235, 0.24323891, 0.23502812, 0.22702031,
     &   0.21919525, 0.21156321, 0.20417121, 0.19698486, 0.19000819/
      data (ggi(i,10),i=1,mxwv)/
     &   0.87979376, 0.88032848, 0.88083392, 0.88113421, 0.88143677,
     &   0.88173145, 0.88203132, 0.88235635, 0.88264561, 0.88296133,
     &   0.88326079, 0.88352317, 0.88370222, 0.88391024, 0.88411182,
     &   0.88430208, 0.88449323, 0.88469124, 0.88486171, 0.88502699,
     &   0.88517451, 0.88529623, 0.88541865, 0.88552934, 0.88563472,
     &   0.88574493, 0.88583863, 0.88592499, 0.88600731, 0.88607621,
     &   0.88614625, 0.88620913, 0.88626486, 0.88631040, 0.88635892,
     &   0.88640118, 0.88644695, 0.88647515, 0.88650399, 0.88654381,
     &   0.88654196, 0.88654375, 0.88656640, 0.88657153, 0.88658404,
     &   0.88657010, 0.88657951, 0.88657016, 0.88657296, 0.88656485,
     &   0.88654953, 0.88654172, 0.88714188, 0.88648868, 0.88646346,
     &   0.88645452, 0.88641918, 0.88639730, 0.88634604, 0.88633174,
     &   0.88629454, 0.88626850, 0.88622606, 0.88619983, 0.88616955,
     &   0.88613683, 0.88609660, 0.88607973, 0.88605475, 0.88605869,
     &   0.88604754, 0.88604963, 0.88603532, 0.88595802, 0.88588345,
     &   0.88584608, 0.88580573, 0.88581634, 0.88577354, 0.88585353,
     &   0.88598418, 0.88617688, 0.88634092, 0.88640004, 0.88638735,
     &   0.88634479, 0.88632965, 0.88634443, 0.88643354, 0.88668120,
     &   0.88898635, 0.89472604, 0.90456676, 0.90896869, 0.90682262,
     &   0.90379131, 0.90007579, 0.89764601, 0.89668661, 0.89533091,
     &   0.89413399, 0.89317906, 0.89274049, 0.89254421, 0.89230603,
     &   0.89253563, 0.89646429, 0.90868044, 0.92476171, 0.93022448,
     &   0.92848057, 0.92359602, 0.91475803, 0.90598249, 0.90142864,
     &   0.89965779, 0.89956105, 0.90205628, 0.90665066, 0.91095221,
     &   0.91406715, 0.91733652, 0.92099941, 0.92413270, 0.92856389,
     &   0.94336253, 0.96317536, 0.98099470, 0.99096775, 0.99514866,
     &   0.98717970, 0.96434897, 0.94831508, 0.92367476, 0.91141993,
     &   0.91734707, 0.92598063, 0.93590057, 0.94342828, 0.94869637,
     &   0.95217532, 0.95145297, 0.94410676, 0.93665594, 0.93394721,
     &   0.93152255, 0.93276757, 0.93747848, 0.94327670, 0.94704199,
     &   0.95035034, 0.95425290, 0.95792890, 0.96218264, 0.96518654,
     &   0.96615005, 0.96599674, 0.96308529, 0.95794392, 0.95465624,
     &   0.95069665, 0.94822532, 0.94856626, 0.95017272, 0.95205963,
     &   0.95616853, 0.96049207, 0.96455652, 0.96811312, 0.97063696,
     &   0.97212183, 0.97148764, 0.97055143, 0.96940392, 0.96853417,
     &   0.96793222, 0.96781260, 0.96790284, 0.96782964, 0.96750915,
     &   0.96720397, 0.96696639, 0.96669918, 0.96635556, 0.96627432,
     &   0.96608490, 0.96566695, 0.96491885, 0.96382827, 0.96238154,
     &   0.96138304, 0.96219093, 0.96250582, 0.96287417, 0.96339291,
     &   0.96411461, 0.96444833, 0.96475303, 0.96514386, 0.96579385,
     &   0.96773350, 0.97197193, 0.97682559, 0.98040169, 0.97976696,
     &   0.97362471, 0.96480590, 0.95691127, 0.94975901, 0.94432712,
     &   0.93940437, 0.93540674, 0.93197727, 0.92905933, 0.92744422,
     &   0.92603165, 0.92502546, 0.92495328, 0.92489690, 0.92491436,
     &   0.92542505, 0.92645055, 0.92779893, 0.92912191, 0.93031877,
     &   0.93132830, 0.93198889, 0.93235546, 0.93252581, 0.93229401,
     &   0.93152797, 0.93037713, 0.92898679, 0.92701304, 0.92425054,
     &   0.92202544, 0.92000103, 0.91952801, 0.91887331, 0.91540736,
     &   0.90829754, 0.89733404, 0.88459027, 0.87427866, 0.86341333,
     &   0.85222799, 0.84256935, 0.83717424, 0.83405441, 0.83238333,
     &   0.83197147, 0.83196044, 0.83305508, 0.83589453, 0.83963567,
     &   0.84474093, 0.85138506, 0.85839379, 0.86563969, 0.87297726,
     &   0.88024527, 0.88723594, 0.89394855, 0.90026438, 0.90609854,
     &   0.91197270, 0.91774392, 0.92269945, 0.92684251, 0.93069410,
     &   0.93413043, 0.93593293, 0.93613827, 0.93506306, 0.93311942,
     &   0.93027020, 0.92651713, 0.92229575, 0.91773015, 0.91282851,
     &   0.90758020, 0.90187413, 0.89571095, 0.88830084, 0.87874442,
     &   0.87132609, 0.86155802, 0.85040843, 0.84297812, 0.84355748,
     &   0.84712535, 0.84725481, 0.84596300, 0.84365994, 0.84101284,
     &   0.83827382, 0.83499861, 0.83305395, 0.83225995, 0.83155298,
     &   0.83807749, 0.84286100, 0.84174609, 0.83395487, 0.82649809,
     &   0.82020181, 0.81414944, 0.80815220, 0.80242020, 0.79749197,
     &   0.79273629, 0.78779131, 0.78261924, 0.77675301, 0.77054012,
     &   0.76363128, 0.75582349, 0.74693936, 0.73723072, 0.72783619,
     &   0.71840322, 0.71183252, 0.70637661, 0.70097148, 0.69618779,
     &   0.69165581, 0.68738163, 0.68333447, 0.67950350, 0.67625403,
     &   0.67317420, 0.67024541, 0.66743612, 0.66467988, 0.66196764,
     &   0.65929055, 0.65663987, 0.65400732, 0.65138477, 0.64876449,
     &   0.64613914, 0.64350194, 0.64027697, 0.63694346, 0.63360316,
     &   0.63025236, 0.62688738, 0.62350488, 0.62010193, 0.61667567,
     &   0.61322343, 0.60974312, 0.60623258, 0.60268980, 0.59901553,
     &   0.59495693, 0.59089035, 0.58681422, 0.58272731, 0.57862824,
     &   0.57442844, 0.56967968, 0.56498164, 0.56033063, 0.55579168,
     &   0.55134094, 0.54692262, 0.54252720, 0.53815275, 0.53379744,
     &   0.52946311, 0.52530861, 0.52110183, 0.51683092, 0.51247317,
     &   0.50802636, 0.50348043, 0.49882972, 0.49405795, 0.48914522,
     &   0.48408338, 0.47886485, 0.47346637, 0.46787786, 0.46210799,
     &   0.45612791, 0.44993117, 0.44352534, 0.43689722, 0.43004137,
     &   0.42297319, 0.41569436, 0.40819597, 0.40049344, 0.39261612,
     &   0.38455194, 0.37631485, 0.36795920, 0.35946748, 0.35085794/
      data (ggi(i,11),i=1,mxwv)/
     &   0.88066339, 0.88120908, 0.88173294, 0.88205522, 0.88236547,
     &   0.88267845, 0.88299239, 0.88331431, 0.88364393, 0.88397080,
     &   0.88429195, 0.88461709, 0.88473970, 0.88498348, 0.88520390,
     &   0.88541657, 0.88562816, 0.88583797, 0.88603282, 0.88621503,
     &   0.88637757, 0.88652039, 0.88665456, 0.88679475, 0.88691747,
     &   0.88704973, 0.88715374, 0.88726604, 0.88736075, 0.88745922,
     &   0.88754660, 0.88762981, 0.88771355, 0.88778538, 0.88785505,
     &   0.88791400, 0.88797486, 0.88802952, 0.88807821, 0.88813043,
     &   0.88817018, 0.88819098, 0.88825345, 0.88827866, 0.88835782,
     &   0.88833857, 0.88836157, 0.88838464, 0.88840622, 0.88841337,
     &   0.88843554, 0.88845074, 0.88845468, 0.88846695, 0.88847953,
     &   0.88849545, 0.88847977, 0.88849252, 0.88848519, 0.88851607,
     &   0.88849652, 0.88849056, 0.88848889, 0.88849419, 0.88850379,
     &   0.88850605, 0.88850141, 0.88850945, 0.88852823, 0.88855696,
     &   0.88860232, 0.88866580, 0.88859779, 0.88861579, 0.88860649,
     &   0.88858974, 0.88859046, 0.88861984, 0.88866389, 0.88882238,
     &   0.88902396, 0.88933867, 0.88959175, 0.88972080, 0.88974708,
     &   0.88975030, 0.88976580, 0.88983673, 0.88995016, 0.89034355,
     &   0.89339954, 0.90084314, 0.91316867, 0.91850615, 0.91597867,
     &   0.91234374, 0.90779412, 0.90476966, 0.90359145, 0.90188682,
     &   0.90037119, 0.89915407, 0.89858395, 0.89832759, 0.89800942,
     &   0.89826846, 0.90323257, 0.91823560, 0.93656152, 0.94236481,
     &   0.94055212, 0.93530101, 0.92530113, 0.91476488, 0.90899682,
     &   0.90659070, 0.90625381, 0.90912789, 0.91458327, 0.91954356,
     &   0.92306191, 0.92646646, 0.93031567, 0.93338442, 0.93776107,
     &   0.95344085, 0.97247976, 0.98636436, 0.99275845, 0.99565822,
     &   0.98766810, 0.96486235, 0.94877279, 0.92399305, 0.91166526,
     &   0.91765422, 0.92635828, 0.93634081, 0.94391459, 0.94926751,
     &   0.95343864, 0.95540911, 0.95319784, 0.94931072, 0.94780892,
     &   0.94630069, 0.94742376, 0.95117557, 0.95546168, 0.95814556,
     &   0.96042442, 0.96290946, 0.96510208, 0.96731728, 0.96860933,
     &   0.96879452, 0.96837735, 0.96729606, 0.96512389, 0.96364540,
     &   0.96160340, 0.96019381, 0.96060711, 0.96184611, 0.96321714,
     &   0.96589863, 0.96850723, 0.97075891, 0.97252941, 0.97369009,
     &   0.97410107, 0.97312063, 0.97207224, 0.97094482, 0.97018766,
     &   0.96977717, 0.96995217, 0.97016162, 0.97008073, 0.96986955,
     &   0.96974742, 0.96969098, 0.96960610, 0.96959198, 0.96967834,
     &   0.96965051, 0.96950549, 0.96931767, 0.96913141, 0.96895415,
     &   0.96905959, 0.96984696, 0.97035998, 0.97090423, 0.97152799,
     &   0.97221911, 0.97278267, 0.97345287, 0.97421879, 0.97519898,
     &   0.97677088, 0.97919279, 0.98174733, 0.98368287, 0.98247862,
     &   0.97635037, 0.96763343, 0.95981675, 0.95272458, 0.94734240,
     &   0.94246662, 0.93851686, 0.93513787, 0.93227422, 0.93072075,
     &   0.92937338, 0.92843693, 0.92844021, 0.92846286, 0.92856294,
     &   0.92916483, 0.93030107, 0.93179739, 0.93332368, 0.93480086,
     &   0.93615466, 0.93721682, 0.93803525, 0.93867207, 0.93882686,
     &   0.93862557, 0.93840992, 0.93842572, 0.93828690, 0.93776536,
     &   0.93743682, 0.93718022, 0.93735957, 0.93711954, 0.93552172,
     &   0.93227226, 0.92685300, 0.91981798, 0.91370970, 0.90669924,
     &   0.89891648, 0.89190871, 0.88820475, 0.88620311, 0.88518304,
     &   0.88494724, 0.88469523, 0.88498515, 0.88634503, 0.88802856,
     &   0.89054918, 0.89409631, 0.89775276, 0.90153104, 0.90543002,
     &   0.90940338, 0.91321909, 0.91712129, 0.92105407, 0.92494911,
     &   0.92912775, 0.93351781, 0.93757397, 0.94120055, 0.94482338,
     &   0.94825530, 0.95023984, 0.95071483, 0.94994199, 0.94832754,
     &   0.94582945, 0.94244283, 0.93860263, 0.93442756, 0.92992455,
     &   0.92508125, 0.91978729, 0.91404158, 0.90704256, 0.89786971,
     &   0.89085722, 0.88148582, 0.87074673, 0.86389637, 0.86532712,
     &   0.86987442, 0.87081742, 0.87031996, 0.86885744, 0.86717349,
     &   0.86552829, 0.86358345, 0.86403656, 0.86649483, 0.86970085,
     &   0.87498027, 0.87718159, 0.87491196, 0.86705559, 0.86077291,
     &   0.85561657, 0.85082155, 0.84630245, 0.84248394, 0.83994228,
     &   0.83771849, 0.83548546, 0.83318025, 0.83038884, 0.82732356,
     &   0.82366312, 0.81932664, 0.81397164, 0.80746692, 0.80017656,
     &   0.79177207, 0.78460199, 0.77777672, 0.77067822, 0.76392931,
     &   0.75708234, 0.75034338, 0.74370748, 0.73714429, 0.73133099,
     &   0.72570044, 0.72037840, 0.71579009, 0.71135604, 0.70708501,
     &   0.70297903, 0.69904447, 0.69528079, 0.69168657, 0.68825865,
     &   0.68499178, 0.68187875, 0.67820239, 0.67452371, 0.67096698,
     &   0.66752595, 0.66419339, 0.66096187, 0.65782326, 0.65476936,
     &   0.65179163, 0.64888167, 0.64603102, 0.64323121, 0.64030689,
     &   0.63680935, 0.63337010, 0.62998265, 0.62664062, 0.62333786,
     &   0.61988097, 0.61529011, 0.61084288, 0.60653073, 0.60243827,
     &   0.59855002, 0.59479159, 0.59113210, 0.58756268, 0.58412963,
     &   0.58084953, 0.57811159, 0.57541138, 0.57273859, 0.57005566,
     &   0.56737715, 0.56470078, 0.56203973, 0.55937952, 0.55669236,
     &   0.55398345, 0.55125880, 0.54847819, 0.54563630, 0.54276937,
     &   0.53981775, 0.53677589, 0.53366989, 0.53046495, 0.52714401,
     &   0.52373302, 0.52021658, 0.51655453, 0.51274061, 0.50876880,
     &   0.50463271, 0.50032556, 0.49582809, 0.49114525, 0.48627853/
      data (ggi(i,12),i=1,mxwv)/
     &   0.88130456, 0.88185936, 0.88239551, 0.88272601, 0.88305002,
     &   0.88337380, 0.88370150, 0.88405013, 0.88436913, 0.88471287,
     &   0.88504660, 0.88533360, 0.88554657, 0.88577622, 0.88600510,
     &   0.88623244, 0.88645703, 0.88668329, 0.88689429, 0.88708818,
     &   0.88725978, 0.88742155, 0.88757086, 0.88771790, 0.88786697,
     &   0.88800770, 0.88813609, 0.88826048, 0.88837469, 0.88848609,
     &   0.88859314, 0.88868964, 0.88878590, 0.88888061, 0.88895899,
     &   0.88904518, 0.88912338, 0.88919657, 0.88924992, 0.88932472,
     &   0.88938785, 0.88944894, 0.88950509, 0.88951111, 0.88960493,
     &   0.88965160, 0.88969654, 0.88973826, 0.88977927, 0.88981545,
     &   0.88985711, 0.88987285, 0.88992161, 0.88995242, 0.88998097,
     &   0.89000374, 0.89003915, 0.89006901, 0.89009422, 0.89011389,
     &   0.89013702, 0.89016229, 0.89019102, 0.89023012, 0.89026618,
     &   0.89029294, 0.89031744, 0.89035165, 0.89040911, 0.89048624,
     &   0.89056647, 0.89068520, 0.89073426, 0.89071530, 0.89068800,
     &   0.89070189, 0.89073473, 0.89081204, 0.89086831, 0.89112365,
     &   0.89142704, 0.89186597, 0.89223427, 0.89243048, 0.89249152,
     &   0.89251173, 0.89255542, 0.89267302, 0.89287096, 0.89337265,
     &   0.89740676, 0.90701556, 0.92215866, 0.92839813, 0.92549962,
     &   0.92123675, 0.91575497, 0.91202831, 0.91055310, 0.90841180,
     &   0.90647882, 0.90490443, 0.90414399, 0.90377647, 0.90332532,
     &   0.90359950, 0.90984130, 0.92792749, 0.94782466, 0.95351326,
     &   0.95178097, 0.94653469, 0.93579859, 0.92353439, 0.91634929,
     &   0.91315466, 0.91247791, 0.91580611, 0.92225921, 0.92793381,
     &   0.93165380, 0.93535984, 0.93935156, 0.94230515, 0.94656360,
     &   0.96264893, 0.97952163, 0.98911625, 0.99343938, 0.99597210,
     &   0.98799092, 0.96517259, 0.94902152, 0.92410946, 0.91171813,
     &   0.91776186, 0.92652911, 0.93656898, 0.94418383, 0.94957358,
     &   0.95392293, 0.95690757, 0.95768845, 0.95679438, 0.95656013,
     &   0.95615399, 0.95723540, 0.95986187, 0.96263027, 0.96433794,
     &   0.96576613, 0.96718967, 0.96839797, 0.96945864, 0.96995634,
     &   0.96983933, 0.96933150, 0.96898603, 0.96836644, 0.96806073,
     &   0.96736467, 0.96695894, 0.96743304, 0.96831805, 0.96927601,
     &   0.97083521, 0.97226429, 0.97344714, 0.97433794, 0.97495061,
     &   0.97502232, 0.97396290, 0.97290039, 0.97178876, 0.97106767,
     &   0.97071308, 0.97097415, 0.97122532, 0.97115219, 0.97098136,
     &   0.97091985, 0.97092581, 0.97090358, 0.97099829, 0.97114587,
     &   0.97117788, 0.97112960, 0.97114319, 0.97130191, 0.97165209,
     &   0.97224277, 0.97298092, 0.97355002, 0.97413927, 0.97477359,
     &   0.97541934, 0.97606641, 0.97688955, 0.97785187, 0.97909343,
     &   0.98076165, 0.98275363, 0.98454392, 0.98573810, 0.98418641,
     &   0.97807467, 0.96942991, 0.96165949, 0.95459759, 0.94923800,
     &   0.94438255, 0.94045442, 0.93709958, 0.93426359, 0.93274486,
     &   0.93143547, 0.93054074, 0.93059105, 0.93066269, 0.93081337,
     &   0.93146878, 0.93266201, 0.93422228, 0.93582129, 0.93739104,
     &   0.93885148, 0.94003636, 0.94099349, 0.94177675, 0.94206232,
     &   0.94204611, 0.94213253, 0.94261688, 0.94314420, 0.94351035,
     &   0.94397092, 0.94446802, 0.94487554, 0.94477618, 0.94406718,
     &   0.94287169, 0.94086522, 0.93790489, 0.93505311, 0.93131614,
     &   0.92668945, 0.92227262, 0.92018056, 0.91937596, 0.91935235,
     &   0.91995215, 0.92055464, 0.92156821, 0.92333567, 0.92524946,
     &   0.92758209, 0.93042916, 0.93310940, 0.93563187, 0.93800712,
     &   0.94022739, 0.94217938, 0.94404846, 0.94584525, 0.94758236,
     &   0.94945365, 0.95148247, 0.95344043, 0.95529455, 0.95730406,
     &   0.95933199, 0.96045351, 0.96044856, 0.95949590, 0.95786464,
     &   0.95543927, 0.95218676, 0.94852966, 0.94456446, 0.94028664,
     &   0.93567878, 0.93062705, 0.92512846, 0.91837311, 0.90942484,
     &   0.90264648, 0.89349306, 0.88296258, 0.87640011, 0.87826210,
     &   0.88330489, 0.88468528, 0.88461578, 0.88358808, 0.88237321,
     &   0.88123709, 0.87985557, 0.88123685, 0.88518256, 0.89089638,
     &   0.89722735, 0.89835244, 0.89520329, 0.88698846, 0.88122356,
     &   0.87660325, 0.87240171, 0.86856341, 0.86561084, 0.86419696,
     &   0.86324978, 0.86247569, 0.86185753, 0.86098820, 0.86014062,
     &   0.85899180, 0.85778570, 0.85634351, 0.85425770, 0.85191923,
     &   0.84881622, 0.84575772, 0.84244359, 0.83858252, 0.83451009,
     &   0.83002120, 0.82521212, 0.82007962, 0.81457996, 0.80921400,
     &   0.80359459, 0.79786384, 0.79262978, 0.78726637, 0.78179133,
     &   0.77622414, 0.77058595, 0.76490498, 0.75918847, 0.75347745,
     &   0.74779141, 0.74215508, 0.73608863, 0.72998315, 0.72394931,
     &   0.71800786, 0.71217847, 0.70647895, 0.70092565, 0.69553268,
     &   0.69031209, 0.68527383, 0.68042523, 0.67577171, 0.67110068,
     &   0.66582286, 0.66074908, 0.65588075, 0.65121734, 0.64675641,
     &   0.64215529, 0.63562381, 0.62946069, 0.62366122, 0.61822706,
     &   0.61320245, 0.60857701, 0.60423726, 0.60010928, 0.59640563,
     &   0.59314781, 0.59100783, 0.58910364, 0.58740634, 0.58584231,
     &   0.58442330, 0.58313537, 0.58199441, 0.58096784, 0.57999909,
     &   0.57909560, 0.57826823, 0.57743424, 0.57658303, 0.57579553,
     &   0.57494873, 0.57403600, 0.57312047, 0.57213670, 0.57105720,
     &   0.56995142, 0.56879979, 0.56752706, 0.56613201, 0.56460977,
     &   0.56296575, 0.56120086, 0.55925572, 0.55719405, 0.55501711/
      data (ggi(i,13),i=1,mxwv)/
     &   0.88176787, 0.88233030, 0.88287544, 0.88321048, 0.88354802,
     &   0.88388175, 0.88421369, 0.88455391, 0.88497645, 0.88525784,
     &   0.88559914, 0.88589621, 0.88610947, 0.88635570, 0.88659734,
     &   0.88683420, 0.88706934, 0.88730079, 0.88752598, 0.88773096,
     &   0.88791746, 0.88808781, 0.88825196, 0.88840854, 0.88856405,
     &   0.88871366, 0.88885254, 0.88898760, 0.88911897, 0.88924497,
     &   0.88935965, 0.88947034, 0.88958246, 0.88968349, 0.88977987,
     &   0.88987684, 0.88996911, 0.89005482, 0.89013535, 0.89021611,
     &   0.89029014, 0.89036846, 0.89043260, 0.89050347, 0.89063716,
     &   0.89062822, 0.89068967, 0.89074582, 0.89080125, 0.89086139,
     &   0.89090741, 0.89096159, 0.89100915, 0.89105791, 0.89111030,
     &   0.89114273, 0.89119983, 0.89125282, 0.89129454, 0.89133686,
     &   0.89137602, 0.89142442, 0.89147878, 0.89154613, 0.89159930,
     &   0.89163822, 0.89169592, 0.89175153, 0.89184439, 0.89196080,
     &   0.89209944, 0.89225805, 0.89233547, 0.89230490, 0.89230239,
     &   0.89232647, 0.89238673, 0.89249080, 0.89259416, 0.89294720,
     &   0.89336276, 0.89396697, 0.89445990, 0.89473408, 0.89481878,
     &   0.89479709, 0.89491165, 0.89506918, 0.89532238, 0.89599049,
     &   0.90131444, 0.91349667, 0.93169469, 0.93863505, 0.93546116,
     &   0.93065953, 0.92426121, 0.91977048, 0.91795242, 0.91528738,
     &   0.91284060, 0.91081309, 0.90980172, 0.90927804, 0.90864760,
     &   0.90890193, 0.91668153, 0.93787712, 0.95802271, 0.96300292,
     &   0.96154940, 0.95685148, 0.94619191, 0.93257153, 0.92389596,
     &   0.91974276, 0.91864043, 0.92247760, 0.93004495, 0.93643171,
     &   0.94040060, 0.94424218, 0.94828641, 0.95106107, 0.95510209,
     &   0.97075701, 0.98420942, 0.99033076, 0.99371439, 0.99616367,
     &   0.98820323, 0.96534932, 0.94913638, 0.92410302, 0.91165680,
     &   0.91774821, 0.92657071, 0.93666196, 0.94431365, 0.94973052,
     &   0.95412838, 0.95740408, 0.95947427, 0.96038705, 0.96109498,
     &   0.96159834, 0.96267641, 0.96440351, 0.96607649, 0.96715319,
     &   0.96807623, 0.96892661, 0.96966469, 0.97025162, 0.97048295,
     &   0.97028315, 0.96975541, 0.96963030, 0.96961218, 0.96985805,
     &   0.96993291, 0.97010976, 0.97062290, 0.97128659, 0.97199202,
     &   0.97292328, 0.97376961, 0.97449893, 0.97507775, 0.97553355,
     &   0.97553086, 0.97446322, 0.97340572, 0.97230405, 0.97159654,
     &   0.97125995, 0.97154623, 0.97181469, 0.97175181, 0.97159773,
     &   0.97155780, 0.97158617, 0.97158653, 0.97171557, 0.97188687,
     &   0.97194237, 0.97192711, 0.97200227, 0.97226757, 0.97279227,
     &   0.97355753, 0.97428107, 0.97487253, 0.97547889, 0.97611600,
     &   0.97674596, 0.97741687, 0.97828656, 0.97930562, 0.98062414,
     &   0.98232734, 0.98423541, 0.98589283, 0.98691958, 0.98525810,
     &   0.97917867, 0.97058159, 0.96283317, 0.95578039, 0.95042574,
     &   0.94557405, 0.94165188, 0.93830580, 0.93548185, 0.93398291,
     &   0.93269598, 0.93182731, 0.93190914, 0.93201363, 0.93219817,
     &   0.93289006, 0.93412220, 0.93572336, 0.93736482, 0.93897969,
     &   0.94048822, 0.94172424, 0.94273597, 0.94357628, 0.94391644,
     &   0.94396669, 0.94414794, 0.94477522, 0.94550705, 0.94616675,
     &   0.94690812, 0.94768608, 0.94818455, 0.94814652, 0.94778836,
     &   0.94746476, 0.94713491, 0.94654983, 0.94586289, 0.94456959,
     &   0.94256669, 0.94042343, 0.93957496, 0.93947011, 0.93985379,
     &   0.94064212, 0.94145262, 0.94255424, 0.94415063, 0.94582981,
     &   0.94774300, 0.94988871, 0.95186234, 0.95366979, 0.95531738,
     &   0.95680910, 0.95814061, 0.95935321, 0.96045280, 0.96144563,
     &   0.96248215, 0.96356601, 0.96454626, 0.96542907, 0.96632856,
     &   0.96716005, 0.96739358, 0.96680117, 0.96556967, 0.96383041,
     &   0.96138871, 0.95818388, 0.95462382, 0.95078021, 0.94663745,
     &   0.94217253, 0.93726838, 0.93191928, 0.92530906, 0.91648728,
     &   0.90983582, 0.90079004, 0.89034688, 0.88391441, 0.88600159,
     &   0.89129645, 0.89290643, 0.89306104, 0.89225572, 0.89126897,
     &   0.89036882, 0.88922769, 0.89090604, 0.89528263, 0.90174878,
     &   0.90884250, 0.91003853, 0.90679759, 0.89854389, 0.89300901,
     &   0.88864732, 0.88472486, 0.88118607, 0.87856781, 0.87755728,
     &   0.87705725, 0.87678564, 0.87674576, 0.87653172, 0.87645632,
     &   0.87620217, 0.87614548, 0.87622708, 0.87600857, 0.87604618,
     &   0.87584239, 0.87550390, 0.87493777, 0.87397999, 0.87277144,
     &   0.87133700, 0.86964512, 0.86770469, 0.86550903, 0.86322612,
     &   0.86071503, 0.85800284, 0.85531050, 0.85243976, 0.84939176,
     &   0.84615868, 0.84273535, 0.83911794, 0.83530354, 0.83129001,
     &   0.82707661, 0.82266444, 0.81777358, 0.81262279, 0.80726683,
     &   0.80171233, 0.79596734, 0.79004180, 0.78394771, 0.77769876,
     &   0.77131057, 0.76480055, 0.75818777, 0.75149298, 0.74456006,
     &   0.73690206, 0.72919738, 0.72147554, 0.71376270, 0.70608771,
     &   0.69803602, 0.68719238, 0.67652953, 0.66610724, 0.65569562,
     &   0.64549351, 0.63576972, 0.62630212, 0.61687696, 0.60816962,
     &   0.60027361, 0.59420335, 0.58877230, 0.58396691, 0.57971168,
     &   0.57602555, 0.57289237, 0.57034045, 0.56831920, 0.56673527,
     &   0.56560093, 0.56493354, 0.56456232, 0.56445575, 0.56477553,
     &   0.56525010, 0.56584996, 0.56669158, 0.56762594, 0.56858295,
     &   0.56969118, 0.57089680, 0.57203317, 0.57309139, 0.57406372,
     &   0.57492793, 0.57567775, 0.57621098, 0.57662112, 0.57690614/
cccc
      if (icall .eq. 0) then
        wmin=log(wlmin)
        wmax=log(wlmax)
        wstep=(wmax-wmin)/(mxwv-1)
        icall=1
      endif

      fw=1+(log(wl)-wmin)/wstep
      fw=min(max(fw,one),float(mxwv)-eps)
      iw=int(fw)
      fw=fw-iw

      fr=1.+((log(abs(re)))/log(2.)-1.)*2
      fr=min(max(fr,one),float(mre)-eps)
      ir=int(fr)
      fr=fr-ir
      if(re .lt. 0.) then 
        qc=qqi(iw  ,ir  )*(1.-fw)*(1.-fr) + qqi(iw+1,ir  )*fw*(1.-fr)+
     &     qqi(iw  ,ir+1)*(1.-fw)*fr      + qqi(iw+1,ir+1)*fw*fr

        wc=wwi(iw  ,ir  )*(1.-fw)*(1.-fr) + wwi(iw+1,ir  )*fw*(1.-fr)+
     &     wwi(iw  ,ir+1)*(1.-fw)*fr      + wwi(iw+1,ir+1)*fw*fr

        gc=ggi(iw  ,ir  )*(1.-fw)*(1.-fr) + ggi(iw+1,ir  )*fw*(1.-fr)+
     &     ggi(iw  ,ir+1)*(1.-fw)*fr      + ggi(iw+1,ir+1)*fw*fr

      else
        qc=qq(iw  ,ir  )*(1.-fw)*(1.-fr) + qq(iw+1,ir  )*fw*(1.-fr)+
     &     qq(iw  ,ir+1)*(1.-fw)*fr      + qq(iw+1,ir+1)*fw*fr

        wc=ww(iw  ,ir  )*(1.-fw)*(1.-fr) + ww(iw+1,ir  )*fw*(1.-fr)+
     &     ww(iw  ,ir+1)*(1.-fw)*fr      + ww(iw+1,ir+1)*fw*fr

        gc=gg(iw  ,ir  )*(1.-fw)*(1.-fr) + gg(iw+1,ir  )*fw*(1.-fr)+
     &     gg(iw  ,ir+1)*(1.-fw)*fr      + gg(iw+1,ir+1)*fw*fr

      endif
      return
      end

c=======================================================================





c file:                  taugas.f
c
c external routines:     kdistr,absint,taugas
c                        gasinit, gasset, depthscl, readk
c
c internal routines:     slf296,slf260,frn296,sint,c4dta,hno3,
c                        hertda,o2cont,o2int,o3hht,o3int,o3uv,
c                        c8dta,cxdta,abcdta,schrun,volmix
c
c modules:               gasblk, trcblk, kdstblk
c
c=======================================================================
      module gasblk
      use params, only: mxly, kr
      integer, parameter :: nta=9

      real(kr) :: bma(11), bmb(11), bmc(11), bms(11), cps(11)

      real(kr) :: cph2o(3515), cpo3(447), cpco2(1219), cpco(173),
     &        cpch4(493), cpn2o(704), cpo2(382), cpnh3(431), cpno(62),
     &        cpno2(142), cpso2(226)

      integer :: iwlh2o(15), iwlo3(6), iwlco2(11), iwlco(4), iwlch4(5), 
     &           iwln2o(12), iwlo2(7), iwlnh3(3), iwlno(2), iwlno2(4),
     &           iwlso2(5), iwhh2o(15), iwho3(6), iwhco2(11), iwhco(4),
     &           iwhch4(5), iwhn2o(12), iwho2(7), iwhnh3(3), iwhno(2),
     &           iwhno2(4), iwhso2(5), ibnd(11)


c --- double exponential fit parameters for h2o  

      data cph2o(1:100)/
     &  -0.59366,  -0.16679,   0.42846,   0.87819,   1.26357, 
     &   1.59247,   1.86372,   2.11483,   2.31810,   2.44040, 
     &   2.55998,   2.69879,   2.79810,   2.89747,   2.98118, 
     &   3.04863,   3.09568,   3.15381,   3.22984,   3.23785, 
     &   3.20991,   3.14246,   3.03461,   2.98864,   3.03520, 
     &   3.08981,   3.10027,   3.11302,   3.10266,   3.05765, 
     &   3.06166,   3.01593,   2.95500,   2.95328,   2.95297, 
     &   2.91497,   2.83753,   2.74642,   2.70474,   2.75606, 
     &   2.84097,   2.89052,   2.89886,   2.86150,   2.78032, 
     &   2.67212,   2.52752,   2.39301,   2.38109,   2.43965, 
     &   2.46195,   2.39329,   2.22943,   2.15815,   2.16157, 
     &   2.29683,   2.40335,   2.35569,   2.29239,   2.12968, 
     &   2.03781,   1.94313,   1.86282,   1.87312,   1.88177, 
     &   1.95321,   1.94145,   1.92602,   1.92812,   1.90587, 
     &   2.04943,   1.95396,   1.78078,   1.60325,   1.55071, 
     &   1.49473,   1.46485,   1.50231,   1.39831,   1.30664, 
     &   1.14704,   0.96109,   0.93139,   1.00613,   1.11827, 
     &   1.13529,   1.07767,   0.96652,   0.90777,   0.91973, 
     &   0.90622,   0.93883,   0.90861,   0.81968,   0.79852, 
     &   0.69385,   0.56997,   0.49693,   0.40867,   0.37846/
      data cph2o(101:200)/
     &   0.44490,   0.53554,   0.59020,   0.59196,   0.50771, 
     &   0.34361,   0.20796,   0.15417,   0.13600,   0.14235, 
     &   0.12700,   0.08853,   0.06715,   0.11430,   0.15016, 
     &   0.15016,   0.13964,   0.04897,  -0.04476,  -0.16953, 
     &  -0.30196,  -0.39901,  -0.42462,  -0.39340,  -0.35671, 
     &  -0.30771,  -0.31570,  -0.35021,  -0.47016,  -0.62308, 
     &  -0.77946,  -0.85086,  -0.82482,  -0.83468,  -0.83991, 
     &  -0.89726,  -0.90918,  -0.84484,  -0.71025,  -0.62777, 
     &  -0.66324,  -0.76848,  -1.03341,  -1.27044,  -1.49576, 
     &  -1.61769,  -1.53549,  -1.47958,  -1.33160,  -1.29625, 
     &  -1.40768,  -1.52411,  -1.72765,  -1.82510,  -1.76468, 
     &  -1.70983,  -1.59977,  -1.50730,  -1.46683,  -1.39464, 
     &  -1.43093,  -1.58947,  -1.78778,  -2.06146,  -2.33634, 
     &  -2.40749,  -2.49065,  -2.44182,  -2.25150,  -2.19801, 
     &  -2.08624,  -2.10309,  -2.27174,  -2.36492,  -2.45781, 
     &  -2.44508,  -2.36196,  -2.38101,  -2.48058,  -2.61957, 
     &  -2.74895,  -2.74245,  -2.63961,  -2.61588,  -2.61569, 
     &  -2.71770,  -2.92220,  -3.01021,  -2.99432,  -2.89456, 
     &  -2.79847,  -2.73359,  -2.69055,  -2.65898,  -2.60837, 
     &  -2.63170,  -2.79096,  -2.97394,  -3.15934,  -3.17057/
      data cph2o(201:300)/
     &  -2.95258,  -2.78308,  -2.69196,  -2.60867,  -2.62239, 
     &  -2.62637,  -2.62950,  -2.71010,  -2.72574,  -2.71317, 
     &  -2.61321,  -2.51967,  -2.42437,  -2.38734,  -2.45056, 
     &  -2.47843,  -2.58702,  -2.56472,  -2.44706,  -2.30814, 
     &  -2.12582,  -2.02697,  -1.99880,  -2.05659,  -2.05701, 
     &  -2.06643,  -2.04721,  -1.90723,  -1.90946,  -1.92812, 
     &  -1.86522,  -1.88820,  -1.77270,  -1.60669,  -1.51740, 
     &  -1.40182,  -1.38758,  -1.38799,  -1.41620,  -1.43182, 
     &  -1.37124,  -1.28249,  -1.09992,  -0.99724,  -0.97950, 
     &  -0.99952,  -1.09066,  -1.09980,  -1.00750,  -0.87259, 
     &  -0.70131,  -0.48309,  -0.30502,  -0.20407,  -0.13886, 
     &  -0.19661,  -0.24505,  -0.28415,  -0.34466,  -0.34496, 
     &  -0.28657,  -0.09485,   0.16770,   0.38311,   0.48553, 
     &   0.49475,   0.49074,   0.52493,   0.57439,   0.60303, 
     &   0.66919,   0.75656,   0.90385,   1.04976,   1.13836, 
     &   1.20132,   1.21963,   1.30344,   1.41212,   1.46770, 
     &   1.47630,   1.45559,   1.43315,   1.49679,   1.62749, 
     &   1.68517,   1.70120,   1.66090,   1.59891,   1.64107, 
     &   1.76792,   1.93419,   2.09362,   2.13280,   2.07959, 
     &   2.01987,   1.96835,   2.03073,   2.17591,   2.32257/
      data cph2o(301:400)/
     &   2.49261,   2.60881,   2.66112,   2.68139,   2.70360, 
     &   2.70568,   2.67997,   2.66478,   2.63655,   2.59716, 
     &   2.57555,   2.58781,   2.58940,   2.50826,   2.28771, 
     &   1.95070,   1.59144,   1.31269,   1.21786,   1.22507, 
     &   1.31945,   1.53875,   1.78543,   2.02655,   2.22881, 
     &   2.32061,   2.34163,   2.39432,   2.43073,   2.53438, 
     &   2.55861,   2.51156,   2.46499,   2.46254,   2.51561, 
     &   2.56373,   2.62430,   2.67999,   2.68386,   2.68780, 
     &   2.68227,   2.59536,   2.42505,   2.29307,   2.17816, 
     &   2.11945,   2.20521,   2.32197,   2.38083,   2.38052, 
     &   2.25417,   2.11473,   2.06142,   2.02788,   2.01508, 
     &   1.97680,   1.91586,   1.87253,   1.83706,   1.80766, 
     &   1.67367,   1.45528,   1.29956,   1.18809,   1.20246, 
     &   1.33650,   1.45778,   1.48886,   1.40546,   1.22716, 
     &   1.01444,   0.91282,   0.87247,   0.83576,   0.80170, 
     &   0.71481,   0.66927,   0.65846,   0.66839,   0.68503, 
     &   0.66215,   0.72413,   0.78703,   0.77831,   0.71136, 
     &   0.51200,   0.35931,   0.30680,   0.33365,   0.36267, 
     &   0.32095,   0.25710,   0.12363,  -0.02266,  -0.18001, 
     &  -0.28048,  -0.27808,  -0.19047,  -0.08151,  -0.09169/
      data cph2o(401:500)/
     &  -0.16662,  -0.24404,  -0.27238,  -0.27345,  -0.32244, 
     &  -0.42037,  -0.54071,  -0.63500,  -0.69930,  -0.77174, 
     &  -0.83521,  -0.86639,  -0.82329,  -0.78820,  -0.82340, 
     &  -0.83838,  -0.91387,  -0.96524,  -0.96364,  -1.05757, 
     &  -1.12747,  -1.19973,  -1.27071,  -1.30173,  -1.34436, 
     &  -1.35556,  -1.35990,  -1.30386,  -1.26726,  -1.28022, 
     &  -1.32843,  -1.43599,  -1.55929,  -1.69416,  -1.79362, 
     &  -1.86416,  -1.90037,  -1.91305,  -1.94866,  -1.95483, 
     &  -1.92284,  -1.87535,  -1.83065,  -1.86043,  -1.93470, 
     &  -2.01410,  -2.07677,  -2.07980,  -2.01822,  -1.96078, 
     &  -1.95185,  -1.96638,  -2.05704,  -2.17667,  -2.24120, 
     &  -2.27833,  -2.33268,  -2.37375,  -2.43075,  -2.54346, 
     &  -2.60789,  -2.68442,  -2.78402,  -2.83736,  -2.89622, 
     &  -2.95598,  -3.03170,  -3.13338,  -3.26736,  -3.41725, 
     &  -3.51456,  -3.61586,  -3.67210,  -3.67841,  -3.72135, 
     &  -3.74941,  -3.78822,  -3.85868,  -3.90419,  -3.91592, 
     &  -3.97897,  -4.00562,  -4.08675,  -4.18795,  -4.15833, 
     &  -4.18094,  -4.18872,  -4.25849,  -4.42026,  -4.57444, 
     &  -4.64021,  -4.58636,  -4.51788,  -4.46274,  -4.44165, 
     &  -4.45450,  -4.42101,  -4.35067,  -4.30493,  -4.23157/
      data cph2o(501:600)/
     &  -4.11952,  -4.01918,  -3.93341,  -3.81424,  -3.70572, 
     &  -3.62484,  -3.48143,  -3.35886,  -3.26514,  -3.15517, 
     &  -3.02814,  -2.95147,  -2.83444,  -2.68908,  -2.62390, 
     &  -2.50458,  -2.39841,  -2.35516,  -2.24360,  -2.18204, 
     &  -2.16652,  -2.08381,  -2.02597,  -1.99880,  -1.90122, 
     &  -1.84045,  -1.82575,  -1.74889,  -1.70489,  -1.66792, 
     &  -1.60475,  -1.59789,  -1.59221,  -1.60854,  -1.66569, 
     &  -1.68527,  -1.72998,  -1.79886,  -1.81356,  -1.82715, 
     &  -1.79425,  -1.61106,  -1.40549,  -1.24369,  -1.15433, 
     &  -1.23589,  -1.44178,  -1.64717,  -1.78560,  -1.84622, 
     &  -1.77824,  -1.69071,  -1.66066,  -1.58765,  -1.54222, 
     &  -1.51960,  -1.45477,  -1.39881,  -1.38659,  -1.37586, 
     &  -1.36025,  -1.39179,  -1.36927,  -1.35455,  -1.38734, 
     &  -1.40292,  -1.45598,  -1.51545,  -1.56173,  -1.62478, 
     &  -1.69200,  -1.75192,  -1.81120,  -1.83354,  -1.87063, 
     &  -1.89006,  -1.88485,  -1.90298,  -1.85403,  -1.82001, 
     &  -1.82495,  -1.82901,  -1.90076,  -1.93649,  -1.83304, 
     &  -1.70268,  -1.52380,  -1.41443,  -1.41301,  -1.39373, 
     &  -1.34561,  -1.20932,  -1.03186,  -0.85296,  -0.71145, 
     &  -0.59825,  -0.51884,  -0.51690,  -0.51723,  -0.52224/
      data cph2o(601:700)/
     &  -0.50043,  -0.40989,  -0.32204,  -0.24881,  -0.18653, 
     &  -0.17548,  -0.22729,  -0.32885,  -0.46183,  -0.47994, 
     &  -0.36042,  -0.23072,  -0.12160,  -0.06422,  -0.14924, 
     &  -0.21674,  -0.17913,  -0.15803,  -0.04515,   0.14450, 
     &   0.28118,   0.39718,   0.49818,   0.51040,   0.44761, 
     &   0.29666,   0.01147,  -0.32421,  -0.66518,  -0.96090, 
     &  -1.13017,  -1.18009,  -1.08032,  -0.80133,  -0.52001, 
     &  -0.33748,  -0.22519,  -0.20871,  -0.26962,  -0.22592, 
     &  -0.15919,  -0.07358,   0.09367,   0.20019,   0.25965, 
     &   0.27816,   0.28577,   0.22305,   0.17722,   0.14469, 
     &   0.06694,   0.07268,   0.10103,   0.14554,   0.20352, 
     &   0.25681,   0.25790,   0.21316,   0.15965,   0.08703, 
     &   0.01638,  -0.03529,  -0.03274,  -0.08812,  -0.12524, 
     &  -0.13536,  -0.23808,  -0.28262,  -0.30082,  -0.29252, 
     &  -0.13320,   0.05226,   0.17657,   0.21670,   0.12268, 
     &   0.00438,  -0.03051,  -0.00359,   0.02967,   0.04460, 
     &  -0.01109,  -0.06041,  -0.07485,  -0.02511,   0.07116, 
     &   0.18506,   0.27668,   0.32130,   0.35452,   0.39867, 
     &   0.36470,   0.34978,   0.36519,   0.38993,   0.47009, 
     &   0.54349,   0.60193,   0.67101,   0.73253,   0.84100/
      data cph2o(701:800)/
     &   0.92974,   1.00406,   1.06301,   1.07261,   1.09629, 
     &   1.10790,   1.10959,   1.11710,   1.15716,   1.24152, 
     &   1.34834,   1.45152,   1.53939,   1.59331,   1.60894, 
     &   1.63833,   1.67031,   1.74144,   1.82069,   1.90463, 
     &   1.98593,   2.02996,   2.10254,   2.16357,   2.16140, 
     &   2.11190,   2.06655,   2.02241,   2.02978,   2.06771, 
     &   2.04985,   2.02048,   1.99566,   2.01593,   2.11269, 
     &   2.22805,   2.27037,   2.23480,   2.16907,   2.09990, 
     &   2.08096,   2.10710,   2.15298,   2.19061,   2.25811, 
     &   2.34221,   2.43200,   2.59765,   2.72007,   2.77243, 
     &   2.71671,   2.56246,   2.33896,   2.14412,   1.97864, 
     &   1.79640,   1.73371,   1.71380,   1.74950,   1.91932, 
     &   2.10063,   2.26262,   2.36884,   2.42988,   2.47605, 
     &   2.51875,   2.53371,   2.51476,   2.47425,   2.40051, 
     &   2.39254,   2.39540,   2.35342,   2.33460,   2.26830, 
     &   2.17169,   2.09605,   2.04747,   2.01127,   1.89721, 
     &   1.74928,   1.55948,   1.38069,   1.34831,   1.35751, 
     &   1.35809,   1.34286,   1.25929,   1.16743,   1.09595, 
     &   1.00365,   0.87965,   0.76257,   0.64206,   0.56343, 
     &   0.49943,   0.40691,   0.29104,   0.18437,   0.12690/
      data cph2o(801:900)/
     &   0.09157,   0.13377,   0.18899,   0.20257,   0.19155, 
     &   0.09384,  -0.01238,  -0.14283,  -0.26122,  -0.31851, 
     &  -0.45610,  -0.58273,  -0.65867,  -0.73100,  -0.66169, 
     &  -0.52264,  -0.46798,  -0.50258,  -0.59104,  -0.72925, 
     &  -0.81067,  -0.80914,  -0.86943,  -0.92975,  -0.92524, 
     &  -0.88289,  -0.79203,  -0.69250,  -0.68167,  -0.75444, 
     &  -0.86193,  -0.97556,  -1.10473,  -1.20018,  -1.24824, 
     &  -1.27702,  -1.22693,  -1.18773,  -1.13552,  -1.14015, 
     &  -1.21589,  -1.26394,  -1.39464,  -1.46192,  -1.52629, 
     &  -1.64635,  -1.71511,  -1.78752,  -1.79358,  -1.77801, 
     &  -1.75599,  -1.77196,  -1.83224,  -1.89985,  -1.98528, 
     &  -2.09408,  -2.24126,  -2.37607,  -2.43218,  -2.43830, 
     &  -2.38400,  -2.33538,  -2.43573,  -2.52275,  -2.67290, 
     &  -2.83451,  -2.93019,  -3.01749,  -3.02463,  -2.99666, 
     &  -2.95414,  -2.91300,  -2.96493,  -3.07471,  -3.25693, 
     &  -3.47657,  -3.67222,  -3.88925,  -3.97727,  -3.94079, 
     &  -3.81920,  -3.66194,  -3.59739,  -3.64351,  -3.74016, 
     &  -3.90037,  -4.04679,  -4.07663,  -4.03256,  -3.91836, 
     &  -3.80990,  -3.76032,  -3.77951,  -3.84240,  -3.90305, 
     &  -3.92223,  -3.82628,  -3.65450,  -3.44339,  -3.25756/
      data cph2o(901:1000)/
     &  -3.09919,  -3.00901,  -2.95747,  -2.88271,  -2.82108, 
     &  -2.72633,  -2.59367,  -2.46775,  -2.36235,  -2.28438, 
     &  -2.27343,  -2.30886,  -2.33620,  -2.27813,  -2.20677, 
     &  -2.16170,  -2.14594,  -2.24245,  -2.36299,  -2.42996, 
     &  -2.50866,  -2.55678,  -2.50968,  -2.47465,  -2.42796, 
     &  -2.37981,  -2.34092,  -2.30518,  -2.26753,  -2.27390, 
     &  -2.44156,  -2.72384,  -3.06108,  -3.38056,  -3.48970, 
     &  -3.41674,  -3.36528,  -3.27790,  -3.15495,  -3.01945, 
     &  -2.81869,  -2.66003,  -2.56096,  -2.49017,  -2.46335, 
     &  -2.51454,  -2.59743,  -2.67025,  -2.78841,  -2.77863, 
     &  -2.63881,  -2.54169,  -2.40240,  -2.37146,  -2.46253, 
     &  -2.54291,  -2.65346,  -2.69467,  -2.69130,  -2.65025, 
     &  -2.59152,  -2.56343,  -2.50785,  -2.44665,  -2.41418, 
     &  -2.34553,  -2.28223,  -2.25278,  -2.20694,  -2.16892, 
     &  -2.14295,  -2.14341,  -2.16443,  -2.24853,  -2.38594, 
     &  -2.49449,  -2.58047,  -2.55462,  -2.41673,  -2.35641, 
     &  -2.32619,  -2.34603,  -2.40102,  -2.30576,  -2.20532, 
     &  -2.09307,  -2.00782,  -2.00039,  -1.91252,  -1.80383, 
     &  -1.65749,  -1.55728,  -1.59262,  -1.70939,  -1.83569, 
     &  -1.84895,  -1.71457,  -1.53813,  -1.41904,  -1.37588/
      data cph2o(1001:1100)/
     &  -1.39458,  -1.39135,  -1.35232,  -1.30470,  -1.24821, 
     &  -1.20394,  -1.19607,  -1.15995,  -1.13948,  -1.11024, 
     &  -1.03785,  -0.99804,  -0.95430,  -0.92707,  -0.93592, 
     &  -0.93528,  -0.86881,  -0.75121,  -0.55836,  -0.35056, 
     &  -0.22085,  -0.13412,  -0.12673,  -0.13867,  -0.11656, 
     &  -0.07357,   0.01888,   0.11050,   0.20428,   0.29291, 
     &   0.35923,   0.43608,   0.47266,   0.49792,   0.54978, 
     &   0.60489,   0.67778,   0.71787,   0.73606,   0.74796, 
     &   0.75193,   0.81728,   0.87972,   0.95990,   1.07451, 
     &   1.13098,   1.17565,   1.19031,   1.20334,   1.27687, 
     &   1.35910,   1.41924,   1.37988,   1.28213,   1.16286, 
     &   1.08658,   1.06554,   1.03702,   1.01290,   0.95519, 
     &   0.94231,   0.94216,   0.95764,   1.03405,   1.11309, 
     &   1.27076,   1.48131,   1.66125,   1.76502,   1.68299, 
     &   1.50126,   1.28195,   1.13724,   1.09863,   1.12031, 
     &   1.23502,   1.34328,   1.39556,   1.40851,   1.40939, 
     &   1.40259,   1.39505,   1.38427,   1.33724,   1.29860, 
     &   1.34354,   1.43194,   1.50874,   1.54493,   1.48740, 
     &   1.37260,   1.26973,   1.21297,   1.11026,   0.97625, 
     &   0.87238,   0.76100,   0.71825,   0.73936,   0.69604/
      data cph2o(1101:1200)/
     &   0.64138,   0.59585,   0.51097,   0.44903,   0.40524, 
     &   0.29892,   0.21583,   0.19145,   0.15378,   0.13759, 
     &   0.09412,  -0.04455,  -0.18870,  -0.28538,  -0.37204, 
     &  -0.46390,  -0.57884,  -0.70647,  -0.78911,  -0.79511, 
     &  -0.76645,  -0.76146,  -0.80163,  -0.83155,  -0.86672, 
     &  -0.92994,  -0.99971,  -1.10990,  -1.25701,  -1.32841, 
     &  -1.33350,  -1.35269,  -1.31799,  -1.35095,  -1.48830, 
     &  -1.57874,  -1.67539,  -1.72874,  -1.68087,  -1.67518, 
     &  -1.73066,  -1.77654,  -1.79238,  -1.81386,  -1.77187, 
     &  -1.73774,  -1.78673,  -1.82129,  -1.86174,  -1.87867, 
     &  -1.92986,  -1.95895,  -1.98042,  -2.10738,  -2.14350, 
     &  -2.22883,  -2.35165,  -2.30593,  -2.31343,  -2.23607, 
     &  -2.17791,  -2.29047,  -2.40740,  -2.60466,  -2.70413, 
     &  -2.67647,  -2.64479,  -2.62274,  -2.66727,  -2.67591, 
     &  -2.66531,  -2.64576,  -2.69566,  -2.79611,  -2.90809, 
     &  -2.99381,  -2.94495,  -2.94833,  -2.97002,  -3.01283, 
     &  -3.07907,  -3.08348,  -3.06412,  -3.08084,  -3.20105, 
     &  -3.32453,  -3.49652,  -3.63219,  -3.65897,  -3.69476, 
     &  -3.63741,  -3.54369,  -3.44992,  -3.41310,  -3.43168, 
     &  -3.48306,  -3.57513,  -3.59385,  -3.59684,  -3.60814/
      data cph2o(1201:1300)/
     &  -3.50612,  -3.41284,  -3.34107,  -3.27248,  -3.26950, 
     &  -3.31027,  -3.32205,  -3.29589,  -3.29768,  -3.28777, 
     &  -3.29950,  -3.39843,  -3.43784,  -3.47042,  -3.54250, 
     &  -3.55457,  -3.69278,  -3.82390,  -3.91709,  -4.02428, 
     &  -3.97802,  -4.04945,  -3.99837,  -3.96096,  -4.01515, 
     &  -4.01286,  -4.27890,  -4.64526,  -4.92520,  -5.20714, 
     &  -5.02961,  -4.88315,  -4.85584,  -4.76921,  -4.54440, 
     &  -4.33075,  -4.16671,  -4.04406,  -4.09564,  -4.11792, 
     &  -4.14522,  -4.19109,  -4.14906,  -4.22221,  -4.35301, 
     &  -4.47867,  -4.50537,  -4.41913,  -4.24856,  -4.05892, 
     &  -3.91396,  -3.73977,  -3.60042,  -3.52610,  -3.50040, 
     &  -3.55218,  -3.66025,  -3.77097,  -3.87835,  -3.96454, 
     &  -3.93046,  -3.92926,  -3.96805,  -3.99038,  -4.10179, 
     &  -4.21981,  -4.24013,  -4.26190,  -4.27753,  -4.25594, 
     &  -4.28500,  -4.29071,  -4.26155,  -4.16114,  -4.04160, 
     &  -3.91756,  -3.82524,  -3.76258,  -3.74207,  -3.77017, 
     &  -3.80666,  -3.92858,  -4.01356,  -4.10145,  -4.16708, 
     &  -4.09123,  -4.00345,  -3.88032,  -3.81171,  -3.80771, 
     &  -3.83212,  -3.88507,  -3.81399,  -3.70048,  -3.58376, 
     &  -3.46350,  -3.42785,  -3.41629,  -3.40329,  -3.36172/
      data cph2o(1301:1400)/
     &  -3.26599,  -3.16908,  -3.10954,  -3.03394,  -2.95828, 
     &  -2.85536,  -2.71469,  -2.60076,  -2.48946,  -2.38513, 
     &  -2.32220,  -2.30051,  -2.34186,  -2.37590,  -2.33267, 
     &  -2.21087,  -2.03216,  -1.91013,  -1.82328,  -1.77996, 
     &  -1.76714,  -1.72488,  -1.71325,  -1.67669,  -1.62963, 
     &  -1.60411,  -1.54027,  -1.47681,  -1.37155,  -1.25978, 
     &  -1.23494,  -1.26986,  -1.33751,  -1.37220,  -1.28322, 
     &  -1.14853,  -1.03021,  -0.89832,  -0.84340,  -0.83317, 
     &  -0.78856,  -0.76905,  -0.69209,  -0.53147,  -0.37401, 
     &  -0.25508,  -0.21755,  -0.22627,  -0.23936,  -0.22223, 
     &  -0.17345,  -0.11880,  -0.10331,  -0.15444,  -0.20353, 
     &  -0.25350,  -0.26628,  -0.13441,   0.02358,   0.13657, 
     &   0.22032,   0.19637,   0.12621,   0.07999,   0.04393, 
     &  -0.01900,  -0.06543,  -0.08129,  -0.14847,  -0.17765, 
     &  -0.23113,  -0.29309,  -0.28723,  -0.27521,  -0.20013, 
     &  -0.11575,  -0.00428,   0.10976,   0.16530,   0.18309, 
     &   0.13200,   0.10610,   0.10394,   0.13621,   0.17117, 
     &   0.17251,   0.18671,   0.16161,   0.16640,   0.18417, 
     &   0.18573,   0.24876,   0.26103,   0.28476,   0.33612, 
     &   0.30642,   0.30150,   0.27173,   0.21976,   0.23130/
      data cph2o(1401:1500)/
     &   0.27376,   0.30887,   0.34334,   0.34765,   0.31180, 
     &   0.30774,   0.31256,   0.35423,   0.42454,   0.44493, 
     &   0.43846,   0.44507,   0.43684,   0.49327,   0.53868, 
     &   0.51933,   0.54592,   0.54951,   0.63201,   0.74737, 
     &   0.80266,   0.88719,   0.87874,   0.84412,   0.84352, 
     &   0.81737,   0.86380,   0.94765,   0.95553,   0.93965, 
     &   0.90241,   0.91481,   1.00917,   1.11552,   1.15202, 
     &   1.06885,   0.96737,   0.85164,   0.80701,   0.82571, 
     &   0.87391,   0.98520,   1.07042,   1.18051,   1.29004, 
     &   1.37755,   1.48278,   1.47663,   1.40851,   1.27508, 
     &   1.11986,   0.98454,   0.88260,   0.82338,   0.79509, 
     &   0.83355,   0.91046,   1.04503,   1.21868,   1.36672, 
     &   1.46155,   1.47085,   1.46520,   1.42619,   1.37940, 
     &   1.41333,   1.43128,   1.45974,   1.54526,   1.53032, 
     &   1.48103,   1.39624,   1.26267,   1.17261,   1.09232, 
     &   1.05888,   1.01929,   0.94626,   0.87615,   0.73334, 
     &   0.61962,   0.52576,   0.40124,   0.32424,   0.20042, 
     &   0.05769,  -0.09325,  -0.27407,  -0.40779,  -0.52559, 
     &  -0.58490,  -0.57916,  -0.54457,  -0.50743,  -0.45937, 
     &  -0.41861,  -0.41520,  -0.39164,  -0.36510,  -0.30857/
      data cph2o(1501:1600)/
     &  -0.23157,  -0.18280,  -0.15878,  -0.21295,  -0.29332, 
     &  -0.39457,  -0.54826,  -0.71006,  -0.87700,  -0.96819, 
     &  -0.98703,  -0.93748,  -0.83916,  -0.78698,  -0.76209, 
     &  -0.80754,  -0.93347,  -1.06076,  -1.15801,  -1.16256, 
     &  -1.09618,  -1.03195,  -1.05522,  -1.13586,  -1.23387, 
     &  -1.33214,  -1.32682,  -1.33648,  -1.38038,  -1.42553, 
     &  -1.49769,  -1.52950,  -1.54445,  -1.56745,  -1.61707, 
     &  -1.69148,  -1.76787,  -1.82556,  -1.84347,  -1.86221, 
     &  -1.87097,  -1.84614,  -1.88659,  -1.98535,  -2.12108, 
     &  -2.27740,  -2.39335,  -2.39886,  -2.33846,  -2.30442, 
     &  -2.27409,  -2.29854,  -2.39124,  -2.56427,  -2.73609, 
     &  -2.88840,  -3.00443,  -3.02685,  -3.09379,  -3.16003, 
     &  -3.13090,  -3.06189,  -3.00807,  -2.95169,  -3.01568, 
     &  -3.11918,  -3.18931,  -3.35446,  -3.46712,  -3.51002, 
     &  -3.48618,  -3.36603,  -3.29278,  -3.32935,  -3.47177, 
     &  -3.61763,  -3.68930,  -3.67420,  -3.62078,  -3.67644, 
     &  -3.76717,  -3.78944,  -3.79818,  -3.75336,  -3.74321, 
     &  -3.86778,  -3.96899,  -4.05004,  -4.15451,  -4.17979, 
     &  -4.22704,  -4.28851,  -4.25560,  -4.21920,  -4.27564, 
     &  -4.42921,  -4.58506,  -4.70967,  -4.80136,  -4.64650/
      data cph2o(1601:1700)/
     &  -4.65341,  -4.51995,  -4.42433,  -4.42137,  -4.44853, 
     &  -4.44819,  -4.49132,  -4.49176,  -4.52929,  -4.58468, 
     &  -4.60533,  -4.62362,  -4.60168,  -4.59803,  -4.45292, 
     &  -4.26920,  -4.09891,  -3.92615,  -3.86016,  -3.69436, 
     &  -3.53699,  -3.38584,  -3.23356,  -3.19281,  -3.14232, 
     &  -3.11326,  -3.04386,  -2.90514,  -2.80270,  -2.68808, 
     &  -2.62726,  -2.61349,  -2.57111,  -2.54465,  -2.47142, 
     &  -2.42795,  -2.40936,  -2.37936,  -2.41255,  -2.40417, 
     &  -2.41017,  -2.39774,  -2.33861,  -2.23985,  -2.08388, 
     &  -2.00350,  -1.93767,  -1.91020,  -1.92815,  -1.89802, 
     &  -1.85648,  -1.84229,  -1.86062,  -1.89799,  -1.95863, 
     &  -2.01858,  -2.05596,  -2.06508,  -2.02824,  -1.93392, 
     &  -1.83965,  -1.74890,  -1.71252,  -1.72275,  -1.71193, 
     &  -1.68781,  -1.66945,  -1.64316,  -1.63675,  -1.69286, 
     &  -1.70297,  -1.72751,  -1.75100,  -1.73714,  -1.79804, 
     &  -1.84371,  -1.86235,  -1.88812,  -1.83704,  -1.77649, 
     &  -1.70661,  -1.60188,  -1.50341,  -1.43505,  -1.46076, 
     &  -1.51651,  -1.57911,  -1.61619,  -1.55812,  -1.49706, 
     &  -1.45230,  -1.42832,  -1.44314,  -1.52138,  -1.60752, 
     &  -1.62106,  -1.64265,  -1.64250,  -1.64573,  -1.74951/
      data cph2o(1701:1800)/
     &  -1.80667,  -1.76036,  -1.68790,  -1.57515,  -1.53228, 
     &  -1.57292,  -1.61350,  -1.65583,  -1.63563,  -1.58694, 
     &  -1.56417,  -1.53128,  -1.54079,  -1.55014,  -1.53022, 
     &  -1.53190,  -1.50230,  -1.50260,  -1.49991,  -1.45992, 
     &  -1.41944,  -1.31703,  -1.21850,  -1.14990,  -1.08809, 
     &  -1.04748,  -1.01748,  -0.95109,  -0.84680,  -0.74538, 
     &  -0.60472,  -0.50362,  -0.46372,  -0.42447,  -0.44838, 
     &  -0.44419,  -0.40683,  -0.38084,  -0.33053,  -0.32846, 
     &  -0.33572,  -0.31158,  -0.29906,  -0.20305,  -0.13083, 
     &  -0.09973,  -0.06963,  -0.12740,  -0.20199,  -0.29978, 
     &  -0.35896,  -0.38843,  -0.41730,  -0.45017,  -0.51507, 
     &  -0.56213,  -0.57297,  -0.50844,  -0.42276,  -0.29372, 
     &  -0.08843,   0.09240,   0.25840,   0.28311,   0.13891, 
     &  -0.06768,  -0.28207,  -0.39760,  -0.40444,  -0.31138, 
     &  -0.14305,  -0.02128,   0.04782,   0.08894,   0.10200, 
     &   0.09648,   0.10814,   0.09787,   0.04275,   0.07559, 
     &   0.12150,   0.14186,   0.19034,   0.13856,   0.07934, 
     &   0.05903,  -0.00117,  -0.04140,  -0.11747,  -0.21938, 
     &  -0.28241,  -0.37335,  -0.49225,  -0.58631,  -0.68229, 
     &  -0.75086,  -0.77623,  -0.84652,  -0.93691,  -1.00829/
      data cph2o(1801:1900)/
     &  -1.07836,  -1.10936,  -1.10990,  -1.10672,  -1.07623, 
     &  -1.03447,  -1.01613,  -1.00369,  -0.99511,  -1.06778, 
     &  -1.12221,  -1.14258,  -1.19379,  -1.17257,  -1.15262, 
     &  -1.17033,  -1.16389,  -1.14503,  -1.13643,  -1.12323, 
     &  -1.19203,  -1.33003,  -1.47540,  -1.65339,  -1.68424, 
     &  -1.66968,  -1.67118,  -1.61782,  -1.65910,  -1.73337, 
     &  -1.81449,  -1.93135,  -2.03554,  -2.03827,  -1.99609, 
     &  -2.00710,  -2.03895,  -2.19678,  -2.30931,  -2.30301, 
     &  -2.23226,  -2.07787,  -2.03277,  -2.03851,  -2.10514, 
     &  -2.23452,  -2.33474,  -2.44465,  -2.43944,  -2.37675, 
     &  -2.35973,  -2.37611,  -2.48915,  -2.59681,  -2.62562, 
     &  -2.61907,  -2.61274,  -2.73225,  -2.84636,  -2.91882, 
     &  -2.95084,  -2.84617,  -2.83687,  -2.84531,  -2.82928, 
     &  -2.88406,  -2.93621,  -3.00526,  -3.09956,  -3.16051, 
     &  -3.18338,  -3.25056,  -3.38003,  -3.56102,  -3.72396, 
     &  -3.80811,  -3.82369,  -3.79760,  -3.90921,  -4.04910, 
     &  -4.14132,  -4.22416,  -4.16634,  -4.21193,  -4.37375, 
     &  -4.54004,  -4.54848,  -4.34009,  -4.10097,  -3.93945, 
     &  -3.99014,  -4.18155,  -4.46321,  -4.84035,  -4.95672, 
     &  -4.88529,  -4.92967,  -5.09480,  -5.27981,  -5.39165/
      data cph2o(1901:2000)/
     &  -5.32774,  -5.16805,  -5.26308,  -5.53619,  -5.93153, 
     &  -6.48485,  -6.38350,  -6.02883,  -5.76237,  -5.65535, 
     &  -5.58220,  -5.58090,  -5.69939,  -5.87562,  -6.23761, 
     &  -6.45380,  -6.50710,  -6.40861,  -6.18069,  -6.15034, 
     &  -6.12957,  -6.08168,  -6.05912,  -6.20029,  -6.35916, 
     &  -6.63834,  -7.22799,  -6.87579,  -6.38557,  -6.05701, 
     &  -5.77145,  -5.71889,  -5.54063,  -5.34887,  -5.20440, 
     &  -5.01687,  -4.88229,  -4.75732,  -4.61829,  -4.47540, 
     &  -4.40382,  -4.22901,  -4.07893,  -3.91067,  -3.71540, 
     &  -3.66982,  -3.60413,  -3.59635,  -3.66139,  -3.67630, 
     &  -3.61574,  -3.49060,  -3.33033,  -3.18950,  -3.19004, 
     &  -3.27293,  -3.43811,  -3.58539,  -3.69658,  -3.64411, 
     &  -3.52966,  -3.51758,  -3.45900,  -3.56858,  -3.67516, 
     &  -3.75396,  -3.80574,  -3.77074,  -3.74231,  -3.63809, 
     &  -3.64323,  -3.59911,  -3.62673,  -3.64385,  -3.54801, 
     &  -3.49160,  -3.38461,  -3.33358,  -3.21719,  -3.04173, 
     &  -2.89493,  -2.77334,  -2.79171,  -2.91085,  -3.04844, 
     &  -3.23627,  -3.31742,  -3.35484,  -3.39756,  -3.34285, 
     &  -3.36017,  -3.34117,  -3.26031,  -3.20256,  -3.07615, 
     &  -2.98533,  -3.01199,  -3.13943,  -3.33780,  -3.54162/
      data cph2o(2001:2100)/
     &  -3.64413,  -3.59251,  -3.59490,  -3.60162,  -3.66139, 
     &  -3.81236,  -3.87304,  -4.04749,  -4.11623,  -4.09447, 
     &  -4.12708,  -3.91916,  -3.77960,  -3.62012,  -3.44890, 
     &  -3.42739,  -3.42156,  -3.36932,  -3.34675,  -3.22941, 
     &  -3.12258,  -3.12447,  -3.07216,  -3.06608,  -3.04637, 
     &  -2.99581,  -3.00597,  -2.94524,  -2.83430,  -2.69244, 
     &  -2.53460,  -2.44553,  -2.36211,  -2.25128,  -2.12504, 
     &  -1.99329,  -1.94694,  -1.96858,  -2.02552,  -2.02890, 
     &  -1.95458,  -1.83064,  -1.68469,  -1.63148,  -1.63055, 
     &  -1.64868,  -1.68433,  -1.65098,  -1.54445,  -1.45543, 
     &  -1.39405,  -1.35500,  -1.38974,  -1.43708,  -1.49729, 
     &  -1.58141,  -1.63709,  -1.71988,  -1.74834,  -1.78729, 
     &  -1.81439,  -1.79445,  -1.80727,  -1.78446,  -1.77116, 
     &  -1.69515,  -1.57106,  -1.41358,  -1.22505,  -1.11749, 
     &  -1.06719,  -1.05722,  -1.05923,  -1.08022,  -1.08249, 
     &  -1.05940,  -1.05527,  -0.97884,  -0.90009,  -0.86984, 
     &  -0.84202,  -0.84891,  -0.86571,  -0.87771,  -0.86436, 
     &  -0.89675,  -0.95811,  -0.95681,  -0.98685,  -0.91920, 
     &  -0.79481,  -0.73405,  -0.63486,  -0.61580,  -0.66083, 
     &  -0.69059,  -0.75323,  -0.74477,  -0.65052,  -0.58475/
      data cph2o(2101:2200)/
     &  -0.56151,  -0.61494,  -0.70313,  -0.70147,  -0.64776, 
     &  -0.57626,  -0.52669,  -0.56405,  -0.57813,  -0.57452, 
     &  -0.57656,  -0.52371,  -0.48121,  -0.47066,  -0.44204, 
     &  -0.42321,  -0.43939,  -0.40019,  -0.34592,  -0.36666, 
     &  -0.36117,  -0.41494,  -0.53334,  -0.63311,  -0.73668, 
     &  -0.83196,  -0.91543,  -0.92801,  -0.91893,  -0.82619, 
     &  -0.64369,  -0.45814,  -0.28838,  -0.20295,  -0.12845, 
     &  -0.12789,  -0.14668,  -0.10804,  -0.12206,  -0.08664, 
     &  -0.05495,  -0.09929,  -0.16477,  -0.24481,  -0.32305, 
     &  -0.39276,  -0.44000,  -0.52873,  -0.60139,  -0.69141, 
     &  -0.79857,  -0.89923,  -1.00968,  -1.08832,  -1.14958, 
     &  -1.21303,  -1.28067,  -1.38492,  -1.47822,  -1.51729, 
     &  -1.55518,  -1.53633,  -1.51062,  -1.50327,  -1.51801, 
     &  -1.57645,  -1.65941,  -1.73134,  -1.75165,  -1.72655, 
     &  -1.71606,  -1.73263,  -1.74728,  -1.79286,  -1.73848, 
     &  -1.66180,  -1.56283,  -1.40366,  -1.32738,  -1.25309, 
     &  -1.25065,  -1.26987,  -1.24009,  -1.22822,  -1.19404, 
     &  -1.20867,  -1.23645,  -1.19332,  -1.13591,  -1.08205, 
     &  -1.04976,  -1.14128,  -1.23489,  -1.27858,  -1.33065, 
     &  -1.28360,  -1.22682,  -1.18706,  -1.15823,  -1.14067/
      data cph2o(2201:2300)/
     &  -1.16633,  -1.17506,  -1.15970,  -1.19126,  -1.19843, 
     &  -1.30385,  -1.42862,  -1.58004,  -1.72327,  -1.78743, 
     &  -1.86895,  -1.85190,  -1.80529,  -1.69422,  -1.49103, 
     &  -1.32529,  -1.20009,  -1.13762,  -1.11678,  -1.13199, 
     &  -1.16550,  -1.16402,  -1.17932,  -1.17405,  -1.15184, 
     &  -1.20924,  -1.29157,  -1.34831,  -1.38571,  -1.42632, 
     &  -1.43812,  -1.50800,  -1.62119,  -1.70590,  -1.86161, 
     &  -2.00714,  -2.11745,  -2.25960,  -2.34777,  -2.44254, 
     &  -2.64264,  -2.83979,  -3.04320,  -3.29364,  -3.41153, 
     &  -3.49359,  -3.60572,  -3.67873,  -3.78090,  -3.85398, 
     &  -3.88200,  -3.83753,  -3.77740,  -3.85401,  -3.79646, 
     &  -3.72746,  -3.70451,  -3.59083,  -3.70223,  -3.88363, 
     &  -4.03077,  -4.20725,  -4.19594,  -4.16725,  -4.13410, 
     &  -4.16791,  -4.12138,  -4.08875,  -4.14355,  -4.10163, 
     &  -4.19018,  -4.25695,  -4.31184,  -4.47906,  -4.51148, 
     &  -4.57929,  -4.59458,  -4.62081,  -4.83031,  -5.02522, 
     &  -5.15710,  -5.27403,  -5.22837,  -5.32058,  -5.55260, 
     &  -5.72630,  -5.86735,  -5.86402,  -5.84419,  -5.89720, 
     &  -6.15533,  -6.51283,  -6.98011,  -7.28495,  -7.08784, 
     &  -6.77605,  -6.49215,  -6.43947,  -6.42083,  -6.59354/
      data cph2o(2301:2400)/
     &  -6.78419,  -6.98883,  -7.11018,  -6.93420,  -6.83581, 
     &  -6.87136,  -6.96133,  -7.28561,  -8.27079,  -8.59451, 
     &  -9.45197,  -8.33631,  -8.21424,  -6.89777,  -6.27923, 
     &  -5.89945,  -5.66364,  -5.69459,  -5.87082,  -5.81185, 
     &  -5.70141,  -5.45890,  -5.24048,  -5.30703,  -5.32430, 
     &  -5.18694,  -5.03410,  -4.82279,  -4.72208,  -4.55097, 
     &  -4.36284,  -4.20326,  -4.04534,  -4.05883,  -4.01183, 
     &  -3.93857,  -3.83212,  -3.66113,  -3.56021,  -3.45969, 
     &  -3.38518,  -3.33373,  -3.32721,  -3.34771,  -3.35412, 
     &  -3.34584,  -3.22701,  -3.14293,  -3.09481,  -3.05706, 
     &  -3.13587,  -3.18156,  -3.26336,  -3.34559,  -3.38498, 
     &  -3.39054,  -3.33695,  -3.34959,  -3.36191,  -3.53258, 
     &  -3.66238,  -3.68946,  -3.69155,  -3.52990,  -3.48606, 
     &  -3.41160,  -3.34144,  -3.31933,  -3.26341,  -3.22609, 
     &  -3.18298,  -3.12800,  -3.02166,  -2.93903,  -2.84135, 
     &  -2.69864,  -2.63582,  -2.60853,  -2.59699,  -2.64799, 
     &  -2.71846,  -2.70856,  -2.67797,  -2.67978,  -2.58432, 
     &  -2.57052,  -2.57883,  -2.48977,  -2.47541,  -2.43446, 
     &  -2.39253,  -2.42823,  -2.44830,  -2.49704,  -2.54904, 
     &  -2.54865,  -2.51223,  -2.39407,  -2.28582,  -2.22341/
      data cph2o(2401:2500)/
     &  -2.18280,  -2.17308,  -2.15234,  -2.10486,  -2.08564, 
     &  -2.08578,  -2.09615,  -2.11136,  -2.10200,  -2.06832, 
     &  -2.04134,  -2.00194,  -1.95185,  -1.92025,  -1.85931, 
     &  -1.85988,  -1.91696,  -2.01129,  -2.15097,  -2.20539, 
     &  -2.21302,  -2.22309,  -2.24412,  -2.30471,  -2.33847, 
     &  -2.25445,  -2.08096,  -1.85098,  -1.61538,  -1.45841, 
     &  -1.42089,  -1.54484,  -1.74234,  -1.96839,  -2.18038, 
     &  -2.28394,  -2.31701,  -2.24125,  -2.05966,  -1.88694, 
     &  -1.78105,  -1.69960,  -1.64107,  -1.62909,  -1.58168, 
     &  -1.56599,  -1.59412,  -1.56739,  -1.56346,  -1.54456, 
     &  -1.55619,  -1.61000,  -1.67072,  -1.75312,  -1.82511, 
     &  -1.87588,  -1.89436,  -1.94377,  -1.96038,  -2.02291, 
     &  -2.14131,  -2.19637,  -2.27114,  -2.33418,  -2.36152, 
     &  -2.44688,  -2.53819,  -2.61011,  -2.69105,  -2.73774, 
     &  -2.76700,  -2.82031,  -2.85910,  -2.88525,  -2.95422, 
     &  -2.99210,  -3.06247,  -3.12280,  -3.12274,  -3.13972, 
     &  -3.09998,  -3.11771,  -3.10670,  -3.00116,  -2.91302, 
     &  -2.75838,  -2.66379,  -2.65726,  -2.62212,  -2.59431, 
     &  -2.55351,  -2.49874,  -2.47486,  -2.52563,  -2.54608, 
     &  -2.54311,  -2.54938,  -2.49038,  -2.49664,  -2.52688/
      data cph2o(2501:2600)/
     &  -2.58688,  -2.67000,  -2.71830,  -2.77113,  -2.80244, 
     &  -2.84845,  -2.87062,  -2.83368,  -2.69338,  -2.52225, 
     &  -2.40864,  -2.34429,  -2.40612,  -2.55941,  -2.73915, 
     &  -2.94409,  -3.12344,  -3.27308,  -3.32104,  -3.27077, 
     &  -3.13129,  -2.92504,  -2.78515,  -2.71007,  -2.66733, 
     &  -2.62518,  -2.62279,  -2.59906,  -2.56745,  -2.59548, 
     &  -2.53657,  -2.50849,  -2.47640,  -2.46021,  -2.53747, 
     &  -2.62220,  -2.76449,  -2.88041,  -2.96357,  -3.02153, 
     &  -3.06178,  -3.14581,  -3.25318,  -3.44687,  -3.69634, 
     &  -3.90497,  -4.09399,  -4.22346,  -4.29749,  -4.51957, 
     &  -4.79572,  -5.03708,  -5.27140,  -5.34657,  -5.44757, 
     &  -5.52207,  -5.57087,  -5.64385,  -5.80294,  -5.90763, 
     &  -5.94797,  -5.85439,  -5.62721,  -5.45121,  -5.40324, 
     &  -5.38540,  -5.39409,  -5.59404,  -5.69955,  -5.76877, 
     &  -5.86764,  -5.78129,  -5.88887,  -6.12206,  -6.37505, 
     &  -6.85575,  -7.13884,  -6.98622,  -6.96112,  -6.84830, 
     &  -6.72456,  -6.67530,  -6.65251,  -6.66033,  -6.88151, 
     &  -7.11199,  -7.33922,  -7.61766,  -7.66585,  -7.87464, 
     &  -8.59519,  -9.04047,  -9.30602,  -9.51273,  -8.93688, 
     &  -9.43151,  -8.84005, -10.00000,  -9.65151,  -8.98420/
      data cph2o(2601:2700)/
     & -10.00000,  -9.43151,  -9.68331, -10.00000,  -9.43067, 
     &  -9.90527, -10.00000,  -9.98810,  -9.65151,  -9.74384, 
     &  -9.20004, -10.00000,  -9.20004, -10.00000, -10.00000, 
     &  -9.85239,  -8.09585,  -7.66916,  -7.89183,  -8.46587, 
     &  -8.79342,  -8.93440,  -8.68356,  -8.83423,  -8.01626, 
     &  -7.91911,  -8.27604,  -9.44864,  -9.69462, -10.00000, 
     &  -9.69462,  -8.31857,  -7.91867,  -7.86404,  -8.32240, 
     &  -8.68705,  -9.61515,  -9.25284,  -8.68705,  -8.28789, 
     &  -7.63730,  -8.25919, -10.00000,  -9.51758, -10.00000, 
     & -10.00000,  -9.51758,  -8.05261,  -7.76848,  -8.24255, 
     &  -9.34171,  -9.19941,  -8.56505,  -7.78955,  -7.23750, 
     &  -6.64136,  -6.41854,  -6.14335,  -5.86704,  -5.70840, 
     &  -5.48179,  -5.23814,  -5.00650,  -4.80407,  -4.69986, 
     &  -4.70404,  -4.80846,  -4.99355,  -5.19947,  -5.33300, 
     &  -5.30550,  -5.17017,  -5.05309,  -4.95685,  -4.79959, 
     &  -4.65496,  -4.54077,  -4.44407,  -4.43768,  -4.47167, 
     &  -4.40429,  -4.30236,  -4.22549,  -4.15283,  -4.06900, 
     &  -3.99244,  -3.87562,  -3.76949,  -3.78198,  -3.77003, 
     &  -3.67364,  -3.52061,  -3.34459,  -3.20426,  -3.15993, 
     &  -3.13851,  -3.09692,  -3.07747,  -3.02936,  -3.02192/
      data cph2o(2701:2800)/
     &  -2.96720,  -2.90584,  -2.79069,  -2.65042,  -2.62072, 
     &  -2.53133,  -2.50554,  -2.48745,  -2.41611,  -2.43532, 
     &  -2.40913,  -2.38723,  -2.33337,  -2.21812,  -2.15072, 
     &  -2.08313,  -2.06151,  -2.10585,  -2.13670,  -2.18757, 
     &  -2.23995,  -2.26992,  -2.34229,  -2.38059,  -2.38087, 
     &  -2.33246,  -2.21618,  -2.14795,  -2.12707,  -2.09130, 
     &  -2.05043,  -1.95550,  -1.77077,  -1.66044,  -1.58224, 
     &  -1.51702,  -1.54004,  -1.54074,  -1.53712,  -1.54938, 
     &  -1.52352,  -1.49404,  -1.51985,  -1.57774,  -1.64393, 
     &  -1.65332,  -1.56238,  -1.45105,  -1.39264,  -1.40635, 
     &  -1.46167,  -1.50365,  -1.47751,  -1.47268,  -1.45992, 
     &  -1.46654,  -1.50223,  -1.47576,  -1.47730,  -1.46977, 
     &  -1.45630,  -1.44490,  -1.43097,  -1.43664,  -1.49313, 
     &  -1.63372,  -1.81848,  -1.97938,  -2.06009,  -2.10124, 
     &  -2.02376,  -1.95095,  -1.86835,  -1.70161,  -1.54835, 
     &  -1.37614,  -1.25130,  -1.17408,  -1.17670,  -1.19790, 
     &  -1.21344,  -1.27252,  -1.28030,  -1.31031,  -1.34321, 
     &  -1.34056,  -1.35264,  -1.38137,  -1.44648,  -1.56735, 
     &  -1.72096,  -1.88960,  -2.06793,  -2.19583,  -2.29229, 
     &  -2.34400,  -2.34120,  -2.36407,  -2.39688,  -2.45450/
      data cph2o(2801:2900)/
     &  -2.53132,  -2.58327,  -2.60585,  -2.60773,  -2.60606, 
     &  -2.64243,  -2.71853,  -2.78182,  -2.84459,  -2.83665, 
     &  -2.78522,  -2.71157,  -2.61628,  -2.53314,  -2.45692, 
     &  -2.41679,  -2.41535,  -2.45500,  -2.51120,  -2.57918, 
     &  -2.62854,  -2.63617,  -2.61204,  -2.53833,  -2.43992, 
     &  -2.37490,  -2.34880,  -2.34476,  -2.36650,  -2.36553, 
     &  -2.34094,  -2.33633,  -2.30483,  -2.26933,  -2.25141, 
     &  -2.22841,  -2.27610,  -2.33673,  -2.37913,  -2.44271, 
     &  -2.48571,  -2.57146,  -2.64200,  -2.61103,  -2.47198, 
     &  -2.27897,  -2.15874,  -2.06557,  -2.05371,  -2.06333, 
     &  -2.04641,  -2.04429,  -2.01040,  -2.00804,  -1.99416, 
     &  -2.05499,  -2.09948,  -2.09706,  -2.10517,  -2.01267, 
     &  -1.99934,  -2.03719,  -2.12114,  -2.29537,  -2.44295, 
     &  -2.55926,  -2.66007,  -2.73808,  -2.75972,  -2.78032, 
     &  -2.67031,  -2.44995,  -2.27133,  -2.11654,  -2.02598, 
     &  -2.01402,  -2.04264,  -2.04511,  -2.02975,  -2.00687, 
     &  -1.94840,  -1.93975,  -1.97104,  -2.01554,  -2.09336, 
     &  -2.15829,  -2.26705,  -2.40356,  -2.55216,  -2.78858, 
     &  -3.00697,  -3.22778,  -3.44413,  -3.55439,  -3.66412, 
     &  -3.73884,  -3.92224,  -4.18922,  -4.41150,  -4.55122/
      data cph2o(2901:3000)/
     &  -4.48637,  -4.29339,  -4.19248,  -4.28419,  -4.41178, 
     &  -4.60369,  -4.81645,  -4.83716,  -4.93818,  -4.87772, 
     &  -4.65255,  -4.40133,  -4.14378,  -4.05431,  -4.02425, 
     &  -4.04257,  -4.11794,  -4.12153,  -4.16374,  -4.17459, 
     &  -4.10274,  -4.04733,  -4.00176,  -4.01760,  -4.13393, 
     &  -4.29085,  -4.38409,  -4.39975,  -4.33916,  -4.31515, 
     &  -4.35426,  -4.44628,  -4.51842,  -4.52596,  -4.53399, 
     &  -4.54590,  -4.63864,  -4.75657,  -4.86504,  -4.95580, 
     &  -5.03365,  -5.14879,  -5.33868,  -5.62972,  -5.92584, 
     &  -6.30299,  -6.62922,  -6.70773,  -6.97810,  -7.35919, 
     &  -7.64909,  -8.62765,  -8.55378,  -7.76305,  -7.47054, 
     &  -7.07789,  -7.11538,  -7.34052,  -7.75694,  -9.17126, 
     & -10.00000,  -9.86547,  -8.71424,  -8.66552,  -8.31602, 
     &  -8.41339,  -7.92192,  -8.66385,  -8.99856,  -9.65437, 
     &  -9.36822,  -9.46517,  -9.43986,  -8.65941, -10.00000, 
     & -10.00000,  -8.82641,  -8.56244,  -7.93689,  -7.68823, 
     &  -7.55818,  -7.05113,  -6.76446,  -6.49313,  -6.24749, 
     &  -6.12617,  -6.05220,  -6.13798,  -6.07909,  -5.86845, 
     &  -5.69141,  -5.50496,  -5.48376,  -5.56108,  -5.42768, 
     &  -5.29615,  -5.10664,  -4.88111,  -4.78669,  -4.62385/
      data cph2o(3001:3100)/
     &  -4.52174,  -4.49073,  -4.45792,  -4.54129,  -4.54480, 
     &  -4.51341,  -4.47258,  -4.27643,  -4.18091,  -4.09557, 
     &  -4.04222,  -4.11247,  -4.14851,  -4.16970,  -4.11065, 
     &  -4.04809,  -4.00745,  -3.99879,  -4.07978,  -4.12451, 
     &  -4.19723,  -4.17393,  -4.09022,  -4.02101,  -3.87998, 
     &  -3.79109,  -3.66411,  -3.50066,  -3.40580,  -3.32713, 
     &  -3.30194,  -3.35131,  -3.35137,  -3.29933,  -3.20658, 
     &  -3.06263,  -2.97995,  -2.98759,  -2.99176,  -3.00756, 
     &  -2.97359,  -2.85849,  -2.81640,  -2.77094,  -2.75469, 
     &  -2.77297,  -2.71165,  -2.69187,  -2.64524,  -2.60542, 
     &  -2.60059,  -2.57842,  -2.59991,  -2.58577,  -2.60792, 
     &  -2.66006,  -2.70803,  -2.79094,  -2.81048,  -2.79532, 
     &  -2.79499,  -2.84578,  -2.90638,  -2.96270,  -2.90633, 
     &  -2.71535,  -2.54313,  -2.37822,  -2.31125,  -2.35246, 
     &  -2.49011,  -2.68215,  -2.83136,  -2.96357,  -2.95873, 
     &  -2.90544,  -2.84387,  -2.70352,  -2.58329,  -2.49207, 
     &  -2.41735,  -2.35522,  -2.30279,  -2.25786,  -2.22067, 
     &  -2.20741,  -2.19735,  -2.20181,  -2.22358,  -2.27247, 
     &  -2.33737,  -2.39631,  -2.45029,  -2.49867,  -2.56939, 
     &  -2.64313,  -2.77129,  -2.92580,  -3.05513,  -3.23728/
      data cph2o(3101:3200)/
     &  -3.31415,  -3.33588,  -3.39544,  -3.43947,  -3.57455, 
     &  -3.69955,  -3.77227,  -3.76260,  -3.70753,  -3.70942, 
     &  -3.73899,  -3.82827,  -3.93052,  -4.10437,  -4.24931, 
     &  -4.35000,  -4.42069,  -4.25644,  -4.21454,  -4.17061, 
     &  -4.11168,  -4.16038,  -4.16686,  -4.19465,  -4.23251, 
     &  -4.27305,  -4.21672,  -4.13963,  -4.07622,  -3.97332, 
     &  -3.96263,  -3.95541,  -3.97392,  -4.03776,  -4.07778, 
     &  -4.01771,  -3.87070,  -3.70710,  -3.59495,  -3.62243, 
     &  -3.69528,  -3.76171,  -3.76582,  -3.65793,  -3.61555, 
     &  -3.59708,  -3.63113,  -3.63691,  -3.57465,  -3.55435, 
     &  -3.47507,  -3.49075,  -3.53253,  -3.57495,  -3.68837, 
     &  -3.68628,  -3.68771,  -3.64979,  -3.60831,  -3.56633, 
     &  -3.48250,  -3.37856,  -3.22908,  -3.14286,  -3.11346, 
     &  -3.13691,  -3.26625,  -3.44333,  -3.64611,  -3.86925, 
     &  -4.08285,  -4.22794,  -4.25115,  -4.14282,  -3.85944, 
     &  -3.59027,  -3.43514,  -3.31856,  -3.24442,  -3.22555, 
     &  -3.18795,  -3.20363,  -3.30589,  -3.41248,  -3.60718, 
     &  -3.70563,  -3.65431,  -3.57332,  -3.47347,  -3.47521, 
     &  -3.53388,  -3.72003,  -3.97569,  -4.31048,  -4.87330, 
     &  -5.39648,  -6.27322,  -8.18185,  -8.07588,  -8.20933/
      data cph2o(3201:3300)/
     &  -8.60643,  -8.83713,  -9.01727,  -9.15690,  -9.41970, 
     &  -9.51520,  -9.63843,  -9.87539,  -9.94314, -10.00000, 
     & -10.00000,  -9.99542,  -9.97748,  -9.94374,  -9.93287, 
     &  -9.90450,  -9.85082,  -9.82140,  -9.73549,  -9.64536, 
     &  -9.59412,  -9.54635,  -9.26735,  -9.23243,  -9.05763, 
     &  -8.93240,  -8.74549,  -8.52992,  -8.25637,  -8.13836, 
     &  -7.66071,  -7.35897,  -7.37375,  -7.09925,  -6.98326, 
     &  -6.89298,  -6.79545,  -6.97172,  -6.67558,  -6.37369, 
     &  -6.21189,  -5.94606,  -5.84975,  -5.83536,  -5.82878, 
     &  -5.78456,  -5.68334,  -5.44809,  -5.28421,  -5.06970, 
     &  -4.89514,  -4.80192,  -4.73588,  -4.78558,  -4.78127, 
     &  -4.73462,  -4.54889,  -4.33093,  -4.18543,  -4.09190, 
     &  -4.11204,  -4.13402,  -4.13401,  -3.97210,  -3.79621, 
     &  -3.65860,  -3.55511,  -3.57549,  -3.57633,  -3.53833, 
     &  -3.46143,  -3.34082,  -3.23729,  -3.17300,  -3.14437, 
     &  -3.10547,  -3.05061,  -2.96941,  -2.86694,  -2.79500, 
     &  -2.75350,  -2.75307,  -2.77146,  -2.79530,  -2.76451, 
     &  -2.68758,  -2.63931,  -2.57797,  -2.58894,  -2.59717, 
     &  -2.52817,  -2.47282,  -2.42360,  -2.45382,  -2.56145, 
     &  -2.61304,  -2.59963,  -2.52689,  -2.46472,  -2.46461/
      data cph2o(3301:3400)/
     &  -2.45407,  -2.39432,  -2.25523,  -2.14408,  -2.05525, 
     &  -2.01888,  -2.07413,  -2.12889,  -2.25990,  -2.39692, 
     &  -2.48925,  -2.54855,  -2.53415,  -2.54460,  -2.50455, 
     &  -2.46921,  -2.42259,  -2.28066,  -2.22625,  -2.17393, 
     &  -2.13289,  -2.19687,  -2.21326,  -2.23949,  -2.27620, 
     &  -2.26819,  -2.29009,  -2.29281,  -2.25201,  -2.17355, 
     &  -2.07947,  -2.03121,  -2.01967,  -2.04954,  -2.08143, 
     &  -2.06833,  -2.05240,  -2.05599,  -2.06967,  -2.12334, 
     &  -2.21510,  -2.29897,  -2.40035,  -2.52428,  -2.62702, 
     &  -2.73003,  -2.87671,  -2.99894,  -3.10548,  -3.25316, 
     &  -3.32982,  -3.39709,  -3.53992,  -3.63406,  -3.74020, 
     &  -3.92706,  -3.96893,  -3.93910,  -3.93559,  -3.82934, 
     &  -3.82006,  -3.87551,  -3.89939,  -3.94509,  -3.95617, 
     &  -3.96332,  -3.96114,  -3.99122,  -4.01273,  -4.01717, 
     &  -4.02888,  -4.04697,  -4.10112,  -4.14864,  -4.27169, 
     &  -4.32135,  -4.33175,  -4.41165,  -4.36331,  -4.39914, 
     &  -4.42505,  -4.40381,  -4.48901,  -4.44885,  -4.38473, 
     &  -4.32458,  -4.19760,  -4.16511,  -4.15683,  -4.14102, 
     &  -4.11365,  -4.10673,  -4.13026,  -4.13652,  -4.19636, 
     &  -4.19684,  -4.14832,  -4.05676,  -3.96205,  -3.90165/
      data cph2o(3401:3500)/
     &  -3.84404,  -3.86524,  -3.83773,  -3.69609,  -3.55481, 
     &  -3.42043,  -3.33841,  -3.37637,  -3.44611,  -3.49193, 
     &  -3.52932,  -3.44601,  -3.36757,  -3.31227,  -3.23777, 
     &  -3.21254,  -3.19842,  -3.22310,  -3.28352,  -3.27914, 
     &  -3.23481,  -3.12437,  -3.04729,  -3.06777,  -3.09818, 
     &  -3.19530,  -3.24569,  -3.24974,  -3.30729,  -3.27728, 
     &  -3.25317,  -3.22055,  -3.15996,  -3.17334,  -3.17694, 
     &  -3.12288,  -3.04593,  -2.99049,  -2.98361,  -3.06492, 
     &  -3.19818,  -3.31628,  -3.42190,  -3.47775,  -3.55095, 
     &  -3.56669,  -3.53409,  -3.38883,  -3.17115,  -3.00955, 
     &  -2.89158,  -2.83770,  -2.86055,  -2.86096,  -2.83436, 
     &  -2.82886,  -2.78602,  -2.80289,  -2.85454,  -2.89629, 
     &  -2.99573,  -3.11206,  -3.27394,  -3.47183,  -3.64849, 
     &  -3.79741,  -3.91130,  -4.08705,  -4.24317,  -4.41275, 
     &  -4.55729,  -4.55082,  -4.66958,  -4.82149,  -4.94204, 
     &  -5.13772,  -5.22105,  -5.20710,  -5.18691,  -5.09729, 
     &  -5.03217,  -4.96344,  -4.97810,  -5.03506,  -5.05380, 
     &  -5.08007,  -5.10835,  -5.13285,  -5.24491,  -5.44530, 
     &  -5.65236,  -5.80563,  -5.75192,  -5.58691,  -5.38023, 
     &  -5.31721,  -5.30923,  -5.34087,  -5.39044,  -5.38089/
      data cph2o(3501:3515)/
     &  -5.43438,  -5.52124,  -5.79590,  -6.25048,  -6.78272, 
     &  -8.29899,  -8.50913,  -8.77871,  -8.91512,  -9.13453, 
     &  -9.37455,  -9.56578,  -9.71290,  -9.89385, -10.00000/

c --- double exponential fit parameters for o3   

      data cpo3(1:100)/
     &  -2.04270,  -1.89660,  -1.62630,  -1.38960,  -1.21700, 
     &  -1.09960,  -1.02140,  -0.96730,  -0.92490,  -0.88960, 
     &  -0.86120,  -0.84170,  -0.83600,  -0.84830,  -0.87850, 
     &  -0.92730,  -0.99320,  -1.07200,  -1.16390,  -1.26620, 
     &  -1.37710,  -1.49760,  -1.62740,  -1.77120,  -1.92890, 
     &  -2.10270,  -2.29480,  -2.49870,  -2.73210,  -2.99920, 
     &  -3.30450,  -3.69940,  -4.10220,  -4.64670,  -5.13280, 
     &  -5.64810,  -6.16340,  -6.67870,  -7.19400,  -7.70930, 
     &  -8.00000,  -7.92740,  -7.64180,  -7.35620,  -7.07060, 
     &  -6.78500,  -6.49940,  -6.21380,  -5.92820,  -5.64260, 
     &  -5.35700,  -5.07140,  -4.78580,  -4.50020,  -4.21460, 
     &  -3.92900,  -3.62130,  -3.34070,  -3.07220,  -2.82260, 
     &  -2.59140,  -2.37780,  -2.18230,  -2.00570,  -1.84560, 
     &  -1.69910,  -1.56590,  -1.44360,  -1.33230,  -1.23190, 
     &  -1.14070,  -1.05500,  -0.97330,  -0.90330,  -0.85840, 
     &  -0.85270,  -0.88380,  -0.92190,  -0.93600,  -0.90250, 
     &  -0.84020,  -0.79130,  -0.77940,  -0.81230,  -0.87500, 
     &  -0.94840,  -1.02060,  -1.08640,  -1.15200,  -1.22020, 
     &  -1.29280,  -1.37450,  -1.46410,  -1.56110,  -1.66690, 
     &  -1.78160,  -1.90510,  -2.03830,  -2.17960,  -2.33120/
      data cpo3(101:200)/
     &  -2.49060,  -2.65690,  -2.83540,  -3.01790,  -3.21210, 
     &  -3.41060,  -3.62080,  -3.83320,  -4.05840,  -4.28540, 
     &  -4.49790,  -4.71750,  -4.91090,  -5.12460,  -5.33440, 
     &  -5.54420,  -5.75400,  -5.96380,  -6.17360,  -6.38340, 
     &  -6.59320,  -6.80300,  -7.01280,  -6.90110,  -6.25900, 
     &  -5.81190,  -5.16030,  -4.33270,  -3.68490,  -3.12530, 
     &  -2.63040,  -2.19030,  -1.80190,  -1.45850,  -1.15330, 
     &  -0.87700,  -0.61660,  -0.36300,  -0.11020,   0.13360, 
     &   0.35250,   0.53260,   0.66780,   0.75100,   0.77520, 
     &   0.78260,   0.78740,   0.80060,   0.82410,   0.76140, 
     &   0.56620,   0.19490,  -0.27700,  -0.61990,  -0.83470, 
     &  -0.95860,  -1.01680,  -1.05010,  -1.08160,  -1.09800, 
     &  -1.08330,  -1.04240,  -0.99720,  -0.97240,  -0.98550, 
     &  -1.03650,  -1.11870,  -1.21500,  -1.31420,  -1.41030, 
     &  -1.49980,  -1.59330,  -1.69380,  -1.80610,  -1.93320, 
     &  -2.07370,  -2.22790,  -2.39660,  -2.57870,  -2.77550, 
     &  -2.98550,  -3.20900,  -3.44650,  -3.69670,  -3.96330, 
     &  -4.24610,  -4.55020,  -4.89120,  -5.28450,  -5.76540, 
     &  -6.41940,  -6.92880,  -7.43820,  -7.94760,  -8.00000, 
     &  -7.54320,  -6.92730,  -6.31150,  -5.54310,  -4.95630/
      data cpo3(201:300)/
     &  -4.46400,  -4.03710,  -3.65330,  -3.30690,  -2.98770, 
     &  -2.70420,  -2.45070,  -2.23550,  -2.06510,  -1.94770, 
     &  -1.87050,  -1.84220,  -1.82350,  -1.77820,  -1.73670, 
     &  -1.70120,  -1.72080,  -1.83530,  -2.03310,  -2.30770, 
     &  -2.59960,  -2.75170,  -2.72630,  -2.66710,  -2.64150, 
     &  -2.64490,  -2.66130,  -2.65890,  -2.60830,  -2.52500, 
     &  -2.45290,  -2.41570,  -2.42980,  -2.49060,  -2.58230, 
     &  -2.68730,  -2.78080,  -2.86120,  -2.93030,  -3.00220, 
     &  -3.08730,  -3.18440,  -3.29290,  -3.41580,  -3.53610, 
     &  -3.67100,  -3.80620,  -3.95200,  -4.11400,  -4.26350, 
     &  -4.43950,  -4.61380,  -4.83720,  -5.08370,  -5.33020, 
     &  -5.36650,  -5.43580,  -5.06510,  -4.84160,  -4.52930, 
     &  -4.25470,  -4.00390,  -3.78180,  -3.58500,  -3.40910, 
     &  -3.25090,  -3.09340,  -2.94850,  -2.80550,  -2.67050, 
     &  -2.54820,  -2.43620,  -2.33800,  -2.24860,  -2.16450, 
     &  -2.08340,  -2.00350,  -1.90810,  -1.76810,  -1.57680, 
     &  -1.36150,  -1.14630,  -0.94820,  -0.78000,  -0.63360, 
     &  -0.50920,  -0.41050,  -0.34950,  -0.32740,  -0.31330, 
     &  -0.30230,  -0.28590,  -0.30550,  -0.43740,  -0.69720, 
     &  -1.10640,  -1.49040,  -1.96870,  -2.44980,  -2.59710/
      data cpo3(301:400)/
     &  -2.52200,  -2.43010,  -2.34670,  -2.29010,  -2.27460, 
     &  -2.30210,  -2.36350,  -2.44200,  -2.50880,  -2.54850, 
     &  -2.56170,  -2.56560,  -2.57710,  -2.61340,  -2.68220, 
     &  -2.78850,  -2.93790,  -3.12000,  -3.32600,  -3.54640, 
     &  -3.77360,  -4.03110,  -4.36510,  -4.77940,  -5.51520, 
     &  -6.12400,  -7.21930,  -8.00000,  -7.97210,  -7.61180, 
     &  -7.25150,  -6.89130,  -6.53100,  -6.17070,  -5.81050, 
     &  -5.45020,  -5.08990,  -4.72970,  -4.36940,  -3.94620, 
     &  -3.60220,  -3.28860,  -3.02340,  -2.78630,  -2.57970, 
     &  -2.40730,  -2.27600,  -2.18940,  -2.13590,  -2.11600, 
     &  -2.08080,  -2.01510,  -1.96660,  -1.94090,  -1.98680, 
     &  -2.14500,  -2.39650,  -2.80420,  -3.55000,  -4.82750, 
     &  -5.63780,  -6.44820,  -7.25850,  -8.00000,  -8.00000, 
     &  -7.62780,  -7.25560,  -6.88340,  -6.51110,  -6.13890, 
     &  -5.76670,  -5.39450,  -5.02230,  -4.65010,  -4.27790, 
     &  -3.90560,  -3.53340,  -3.38280,  -3.24520,  -3.14110, 
     &  -3.04030,  -2.94280,  -2.84360,  -2.75730,  -2.68530, 
     &  -2.60400,  -2.52180,  -2.41210,  -2.35470,  -2.19700, 
     &  -2.06680,  -1.91210,  -1.76170,  -1.61530,  -1.46880, 
     &  -1.40220,  -1.34470,  -1.26690,  -1.19020,  -1.18050/
      data cpo3(401:447)/
     &  -1.17070,  -1.16090,  -1.16090,  -1.18050,  -1.19990, 
     &  -1.42140,  -1.63480,  -1.75190,  -1.97300,  -2.20780, 
     &  -2.46080,  -2.53370,  -2.59230,  -2.66160,  -2.63840, 
     &  -2.62710,  -2.61540,  -2.55700,  -2.49830,  -2.44800, 
     &  -2.38900,  -2.36630,  -2.34310,  -2.33140,  -2.32000, 
     &  -2.32000,  -2.33140,  -2.34310,  -2.35470,  -2.37770, 
     &  -2.40040,  -2.52180,  -2.64990,  -2.76940,  -2.90570, 
     &  -3.02860,  -3.15430,  -3.36960,  -3.60530,  -4.19770, 
     &  -4.78110,  -5.29330,  -5.75540,  -6.45420,  -7.02390, 
     &  -7.59370,  -8.00000/

c --- double exponential fit parameters for co2  

      data cpco2(1:100)/
     &  -9.84950,  -9.64840,  -9.44720,  -9.24610,  -9.04490, 
     &  -8.95440,  -8.61270,  -8.40760,  -8.27100,  -8.03910, 
     &  -7.94850,  -7.96380,  -7.78490,  -7.62780,  -7.14180, 
     &  -6.78230,  -6.38260,  -6.03230,  -5.75010,  -5.52490, 
     &  -5.33040,  -5.01050,  -4.77030,  -4.57140,  -4.39190, 
     &  -4.29740,  -4.13700,  -3.87610,  -3.59360,  -3.28520, 
     &  -3.00160,  -2.73030,  -2.48680,  -2.27410,  -2.09360, 
     &  -1.94240,  -1.80920,  -1.68430,  -1.53720,  -1.38030, 
     &  -1.20430,  -0.99300,  -0.77240,  -0.55090,  -0.34650, 
     &  -0.17850,  -0.04700,   0.04490,   0.11140,   0.13670, 
     &   0.09100,   0.00660,  -0.12690,  -0.29940,  -0.49340, 
     &  -0.71010,  -0.90870,  -1.10040,  -1.26940,  -1.40640, 
     &  -1.56220,  -1.68100,  -1.78410,  -1.89730,  -2.02740, 
     &  -2.20790,  -2.42640,  -2.67630,  -2.93120,  -3.18960, 
     &  -3.42620,  -3.59790,  -3.70510,  -3.73720,  -3.79830, 
     &  -3.91540,  -4.05200,  -4.25670,  -4.46610,  -4.66700, 
     &  -4.92260,  -5.22030,  -5.55970,  -5.64030,  -5.70390, 
     &  -5.76740,  -5.83100,  -5.89480,  -5.95030,  -6.02170, 
     &  -6.03920,  -5.98550,  -5.86200,  -5.68340,  -5.50830, 
     &  -5.34730,  -5.20280,  -5.07990,  -4.96280,  -4.83790/
      data cpco2(101:200)/
     &  -4.70320,  -4.55840,  -4.42130,  -4.31980,  -4.27860, 
     &  -4.28430,  -4.30990,  -4.32100,  -4.27690,  -4.22290, 
     &  -4.21790,  -4.29500,  -4.47890,  -4.75500,  -5.09020, 
     &  -5.43290,  -5.66890,  -5.66080,  -5.45820,  -5.19690, 
     &  -4.94190,  -4.71060,  -4.50840,  -4.34090,  -4.22110, 
     &  -4.15630,  -4.12590,  -4.11080,  -4.08030,  -4.02110, 
     &  -3.98240,  -4.00530,  -4.12210,  -4.35040,  -4.67410, 
     &  -5.08260,  -5.58570,  -6.23010,  -7.08290,  -8.13440, 
     &  -8.86010,  -9.04570,  -9.12310,  -9.07280,  -9.14130, 
     &  -9.12210,  -9.18820,  -9.27520,  -9.22370,  -9.36040, 
     &  -9.30580,  -9.54550,  -9.55670,  -9.37540,  -8.77560, 
     &  -8.09040,  -7.48270,  -6.95850,  -6.50950,  -6.11940, 
     &  -5.78240,  -5.49100,  -5.25320,  -5.08400,  -4.99200, 
     &  -4.95770,  -4.96380,  -4.97410,  -4.95550,  -4.94660, 
     &  -4.97740,  -5.07190,  -5.25580,  -5.52130,  -5.86330, 
     &  -6.28770,  -6.78780,  -7.26020,  -7.29400,  -6.85240, 
     &  -6.33720,  -5.88540,  -5.50650,  -5.20110,  -4.97760, 
     &  -4.84710,  -4.78850,  -4.77830,  -4.78150,  -4.75380, 
     &  -4.72280,  -4.72590,  -4.78600,  -4.92310,  -5.12700, 
     &  -5.38310,  -5.68490,  -6.03510,  -6.44370,  -6.91600/
      data cpco2(201:300)/
     &  -7.48150,  -8.14370,  -8.94490,  -9.85640,  -9.89030, 
     &  -9.43650,  -8.98260,  -8.52880,  -8.11840,  -7.65550, 
     &  -7.16730,  -6.72260,  -6.34230,  -6.04100,  -5.81540, 
     &  -5.65190,  -5.51860,  -5.38590,  -5.22790,  -5.02380, 
     &  -4.78650,  -4.53430,  -4.28460,  -4.05600,  -3.87170, 
     &  -3.76240,  -3.72310,  -3.73350,  -3.83120,  -3.98540, 
     &  -4.19300,  -4.48950,  -4.73940,  -4.88920,  -4.94990, 
     &  -4.93920,  -4.97870,  -5.11290,  -5.33300,  -5.60930, 
     &  -5.88620,  -6.05810,  -6.02740,  -5.83560,  -5.59890, 
     &  -5.37380,  -5.16610,  -4.94720,  -4.70200,  -4.43540, 
     &  -4.14390,  -3.85610,  -3.59440,  -3.36940,  -3.21000, 
     &  -3.10410,  -3.04110,  -3.04710,  -3.10770,  -3.23050, 
     &  -3.42740,  -3.61150,  -3.75420,  -3.86660,  -3.93380, 
     &  -4.00790,  -4.09620,  -4.21420,  -4.14330,  -4.28700, 
     &  -4.47960,  -4.66180,  -4.82040,  -4.94990,  -4.98620, 
     &  -5.01710,  -5.02820,  -5.05800,  -5.03980,  -4.94650, 
     &  -4.78160,  -4.55380,  -4.29750,  -4.02860,  -3.75280, 
     &  -3.47150,  -3.18990,  -2.90410,  -2.61270,  -2.32120, 
     &  -2.04350,  -1.78940,  -1.55310,  -1.33820,  -1.15150, 
     &  -0.99900,  -0.88330,  -0.80060,  -0.72270,  -0.62880/
      data cpco2(301:400)/
     &  -0.49770,  -0.32490,  -0.13490,   0.05760,   0.24870, 
     &   0.43860,   0.62600,   0.80810,   0.96810,   1.08590, 
     &   1.15220,   1.18610,   1.20390,   1.22550,   1.25870, 
     &   1.24730,   1.14570,   0.91390,   0.52500,   0.01730, 
     &  -0.57960,  -1.39440,  -2.38410,  -2.72440,  -2.92640, 
     &  -3.06890,  -3.21200,  -3.33530,  -3.45100,  -3.55660, 
     &  -3.65180,  -3.74600,  -3.85000,  -3.96800,  -4.09810, 
     &  -4.22590,  -4.33690,  -4.43290,  -4.53050,  -4.62640, 
     &  -4.74380,  -4.88420,  -5.02480,  -5.14480,  -5.23710, 
     &  -5.27810,  -5.32990,  -5.37660,  -5.42330,  -5.46990, 
     &  -5.51660,  -5.56330,  -5.66460,  -5.75930,  -5.84610, 
     &  -5.92290,  -5.98180,  -6.00650,  -5.97470,  -5.87410, 
     &  -5.72300,  -5.56200,  -5.43890,  -5.37880,  -5.36790, 
     &  -5.38270,  -5.38370,  -5.34600,  -5.31860,  -5.33940, 
     &  -5.43200,  -5.60950,  -5.84460,  -6.09920,  -6.33990, 
     &  -6.54990,  -6.74340,  -6.93590,  -7.12190,  -7.28180, 
     &  -7.39840,  -7.48810,  -7.54520,  -7.59940,  -7.64450, 
     &  -7.67340,  -7.64220,  -7.50570,  -7.26500,  -6.99750, 
     &  -6.77490,  -6.63980,  -6.58750,  -6.59120,  -6.61920, 
     &  -6.61550,  -6.58660,  -6.58510,  -6.63820,  -6.77360/
      data cpco2(401:500)/
     &  -7.00090,  -7.28960,  -7.63270,  -7.97670,  -8.26330, 
     &  -8.47440,  -8.54550,  -8.58130,  -8.60250,  -8.64590, 
     &  -8.89480,  -9.14360,  -9.39250,  -9.64130,  -9.89020, 
     &  -9.80060,  -9.50490,  -9.19470,  -8.72540,  -8.44100, 
     &  -8.17810,  -8.01820,  -7.93810,  -7.87930,  -7.76360, 
     &  -7.55490,  -7.29620,  -7.02440,  -6.75560,  -6.48880, 
     &  -6.24430,  -6.04220,  -5.90880,  -5.85900,  -5.88900, 
     &  -5.98500,  -6.09490,  -6.11640,  -6.02070,  -5.85920, 
     &  -5.71100,  -5.63280,  -5.63690,  -5.72740,  -5.90690, 
     &  -6.17200,  -6.52030,  -6.95860,  -7.47760,  -8.06070, 
     &  -8.55140,  -8.70110,  -8.42320,  -7.92740,  -7.61590, 
     &  -7.38360,  -7.19690,  -7.05230,  -6.76850,  -6.40220, 
     &  -6.03540,  -5.71250,  -5.46590,  -5.30880,  -5.25460, 
     &  -5.29910,  -5.38190,  -5.46150,  -5.41170,  -5.21070, 
     &  -5.01030,  -4.82320,  -4.70710,  -4.68500,  -4.73850, 
     &  -4.87970,  -5.10240,  -5.40150,  -5.77580,  -6.22250, 
     &  -6.66810,  -6.91270,  -6.89190,  -6.69720,  -6.50120, 
     &  -6.31230,  -6.10910,  -5.86410,  -5.58890,  -5.30570, 
     &  -5.03400,  -4.78260,  -4.54760,  -4.32770,  -4.12240, 
     &  -3.93330,  -3.76750,  -3.63240,  -3.51630,  -3.40430/
      data cpco2(501:600)/
     &  -3.27440,  -3.11800,  -2.95570,  -2.82540,  -2.73590, 
     &  -2.67210,  -2.60840,  -2.51050,  -2.37720,  -2.23170, 
     &  -2.08660,  -1.95210,  -1.82920,  -1.71100,  -1.59920, 
     &  -1.48730,  -1.36460,  -1.22600,  -1.07210,  -0.92810, 
     &  -0.83790,  -0.81230,  -0.82610,  -0.84830,  -0.83050, 
     &  -0.77920,  -0.76260,  -0.82280,  -0.99080,  -1.25030, 
     &  -1.53470,  -1.79340,  -1.98370,  -2.07150,  -2.03750, 
     &  -1.89750,  -1.69060,  -1.44970,  -1.20480,  -0.98310, 
     &  -0.81250,  -0.71570,  -0.67070,  -0.65320,  -0.62970, 
     &  -0.57060,  -0.52630,  -0.54890,  -0.68570,  -0.97930, 
     &  -1.39620,  -1.86730,  -2.36550,  -3.54360,  -4.04240, 
     &  -4.40840,  -4.68480,  -4.86630,  -4.95160,  -4.97900, 
     &  -4.99230,  -5.02070,  -5.05960,  -5.09580,  -5.10180, 
     &  -5.06360,  -5.03540,  -5.05460,  -5.14540,  -5.32740, 
     &  -5.58630,  -5.88890,  -6.17700,  -6.35550,  -6.40960, 
     &  -6.43710,  -6.51120,  -6.66800,  -6.91830,  -7.24180, 
     &  -7.58270,  -7.87040,  -8.05510,  -8.17050,  -8.25000, 
     &  -8.35540,  -8.39610,  -8.43540,  -8.39200,  -8.27850, 
     &  -8.04990,  -7.74370,  -7.41300,  -7.11530,  -6.88610, 
     &  -6.74220,  -6.67860,  -6.67740,  -6.70530,  -6.70900/
      data cpco2(601:700)/
     &  -6.67940,  -6.60550,  -6.48270,  -6.34540,  -6.24010, 
     &  -6.19920,  -6.26760,  -6.48330,  -6.84900,  -7.43100, 
     &  -8.46060,  -9.73640,  -9.87710,  -9.88400,  -9.95590, 
     &  -9.94890,  -9.60030,  -9.09100,  -8.57930,  -8.20590, 
     &  -7.90990,  -7.71570,  -7.61450,  -7.59640,  -7.59420, 
     &  -7.52560,  -7.31900,  -6.99860,  -6.68840,  -6.41020, 
     &  -6.17690,  -5.98820,  -5.84210,  -5.74990,  -5.72010, 
     &  -5.71890,  -5.71080,  -5.66690,  -5.59550,  -5.56860, 
     &  -5.62870,  -5.80000,  -6.08550,  -6.43980,  -6.77930, 
     &  -6.94270,  -6.92050,  -6.83630,  -6.70590,  -6.52720, 
     &  -6.29030,  -6.00850,  -5.72240,  -5.47220,  -5.27720, 
     &  -5.15010,  -5.07680,  -5.02190,  -4.95790,  -4.85550, 
     &  -4.72130,  -4.58680,  -4.45940,  -4.33870,  -4.22190, 
     &  -4.10020,  -3.98120,  -3.88760,  -3.82070,  -3.76730, 
     &  -3.71200,  -3.62230,  -3.49120,  -3.34440,  -3.19830, 
     &  -3.07320,  -3.02620,  -3.00780,  -3.01230,  -3.02130, 
     &  -2.99570,  -2.92610,  -2.87700,  -2.88870,  -2.98530, 
     &  -3.16090,  -3.36430,  -3.54680,  -3.67590,  -3.74880, 
     &  -3.77040,  -3.75350,  -3.71130,  -3.63680,  -3.52770, 
     &  -3.38120,  -3.20200,  -3.00430,  -2.80200,  -2.61220/
      data cpco2(701:800)/
     &  -2.45240,  -2.34050,  -2.28380,  -2.25210,  -2.23190, 
     &  -2.19600,  -2.15620,  -2.17320,  -2.29130,  -2.54760, 
     &  -2.93820,  -3.39660,  -3.85250,  -4.25410,  -4.56820, 
     &  -4.73760,  -4.75240,  -4.67330,  -4.51700,  -4.31230, 
     &  -4.08910,  -3.85650,  -3.62180,  -3.39090,  -3.17850, 
     &  -3.01000,  -2.91050,  -2.85880,  -2.82860,  -2.79120, 
     &  -2.72070,  -2.67290,  -2.68580,  -2.77450,  -2.94140, 
     &  -3.14450,  -3.36170,  -3.59540,  -3.85080,  -4.17390, 
     &  -4.51220,  -4.89850,  -5.34260,  -5.87370,  -6.47340, 
     &  -7.07150,  -7.50420,  -7.60340,  -7.51430,  -7.43580, 
     &  -7.40890,  -7.39690,  -7.38130,  -7.30180,  -7.18580, 
     &  -7.06330,  -6.99620,  -6.99050,  -7.03190,  -7.13310, 
     &  -7.20540,  -7.18560,  -7.05610,  -6.79660,  -6.47710, 
     &  -6.19960,  -5.95930,  -5.75600,  -5.53700,  -5.28360, 
     &  -5.09660,  -4.95830,  -4.91260,  -5.00220,  -5.13700, 
     &  -5.34650,  -5.62790,  -5.93640,  -6.36950,  -6.96020, 
     &  -7.68230,  -8.27010,  -8.64270,  -9.07280,  -9.53660, 
     &  -9.95880,  -9.98710,  -9.67620,  -9.33580,  -8.99540, 
     &  -8.51400,  -8.20660,  -7.97420,  -7.85790,  -7.80730, 
     &  -7.78940,  -7.74660,  -7.70090,  -7.63930,  -7.58890/
      data cpco2(801:900)/
     &  -7.56970,  -7.52000,  -7.39080,  -7.17960,  -6.96100, 
     &  -6.78690,  -6.69720,  -6.67350,  -6.67750,  -6.64950, 
     &  -6.52920,  -6.34350,  -6.13710,  -5.92680,  -5.72540, 
     &  -5.54330,  -5.40230,  -5.32920,  -5.30900,  -5.31710, 
     &  -5.31930,  -5.27050,  -5.20850,  -5.18350,  -5.21860, 
     &  -5.33670,  -5.53050,  -5.77250,  -6.02280,  -6.21500, 
     &  -6.28570,  -6.26340,  -6.22500,  -6.22340,  -6.26160, 
     &  -6.29310,  -6.25080,  -6.09710,  -5.86790,  -5.61950, 
     &  -5.39060,  -5.19440,  -5.02160,  -4.85660,  -4.69190, 
     &  -4.52550,  -4.37850,  -4.28790,  -4.25830,  -4.26360, 
     &  -4.27680,  -4.24840,  -4.18530,  -4.15860,  -4.20790, 
     &  -4.36510,  -4.64070,  -5.01410,  -5.47190,  -6.00150, 
     &  -6.51730,  -6.78290,  -6.68050,  -6.41800,  -6.07930, 
     &  -5.74040,  -5.42040,  -5.12650,  -4.86340,  -4.63780, 
     &  -4.45590,  -4.33600,  -4.27520,  -4.24610,  -4.22570, 
     &  -4.17680,  -4.10680,  -4.07430,  -4.11930,  -4.27320, 
     &  -4.54640,  -4.92560,  -5.40900,  -6.01840,  -6.79850, 
     &  -7.70780,  -8.34570,  -8.51600,  -8.61060,  -8.81750, 
     &  -9.19220,  -9.67750,  -9.74230,  -9.19800,  -8.41200, 
     &  -7.74990,  -7.16850,  -6.68170,  -6.27010,  -5.93010/
      data cpco2(901:1000)/
     &  -5.65670,  -5.45210,  -5.32890,  -5.27760,  -5.26300, 
     &  -5.25470,  -5.20830,  -5.12960,  -5.08230,  -5.09140, 
     &  -5.18060,  -5.35030,  -5.56000,  -5.78770,  -5.99360, 
     &  -6.17200,  -6.38010,  -6.63710,  -6.99640,  -7.50100, 
     &  -8.16280,  -8.99510,  -9.89310, -10.00000, -10.00000, 
     & -10.00000, -10.00000, -10.00000, -10.00000, -10.00000, 
     & -10.00000,  -9.49670,  -8.91980,  -8.50810,  -8.12550, 
     &  -7.82860,  -7.54780,  -7.14870,  -6.78530,  -6.55370, 
     &  -6.39310,  -6.41070,  -6.50870,  -6.66070,  -6.90260, 
     &  -7.21040,  -7.44450,  -7.63030,  -7.63460,  -7.45210, 
     &  -7.22110,  -7.00430,  -6.79030,  -6.56660,  -6.34990, 
     &  -6.15340,  -5.99880,  -5.90330,  -5.87600,  -5.86930, 
     &  -5.82770,  -5.72820,  -5.62620,  -5.58650,  -5.66650, 
     &  -5.92280,  -6.33990,  -7.01800,  -8.42300, -10.00000, 
     & -10.00000, -10.00000,  -9.40900,  -8.82720,  -8.30570, 
     &  -7.88850,  -7.50440,  -7.15600,  -6.82920,  -6.52500, 
     &  -6.24610,  -5.99040,  -5.75330,  -5.52950,  -5.31350, 
     &  -5.10580,  -4.91520,  -4.74630,  -4.60540,  -4.49370, 
     &  -4.39280,  -4.28380,  -4.16260,  -4.03870,  -3.92950, 
     &  -3.86120,  -3.85010,  -3.86470,  -3.86250,  -3.80990/
      data cpco2(1001:1100)/
     &  -3.73510,  -3.71790,  -3.85490,  -4.23120,  -4.76320, 
     &  -5.42700,  -6.42000,  -8.14140,  -9.04510,  -9.53260, 
     &  -9.83010,  -9.94720,  -9.82740,  -8.97970,  -8.42980, 
     &  -7.89060,  -7.44770,  -7.07500,  -6.76980,  -6.53380, 
     &  -6.37390,  -6.29800,  -6.27390,  -6.27260,  -6.25550, 
     &  -6.19890,  -6.15290,  -6.16540,  -6.25840,  -6.46100, 
     &  -6.78050,  -7.22350,  -7.81910,  -8.58500,  -9.60840, 
     & -10.00000, -10.00000,  -9.91990,  -9.10930,  -8.44900, 
     &  -7.91580,  -7.43640,  -7.04000,  -6.69580,  -6.41310, 
     &  -6.18550,  -6.01580,  -5.91230,  -5.87000,  -5.85300, 
     &  -5.83400,  -5.78660,  -5.72240,  -5.70480,  -5.76530, 
     &  -5.92810,  -6.22340,  -6.66460,  -7.29570,  -8.27990, 
     &  -9.94570, -10.00000, -10.00000, -10.00000, -10.00000, 
     & -10.00000, -10.00000, -10.00000,  -9.27660,  -8.62010, 
     &  -8.07640,  -7.63740,  -7.27520,  -6.98020,  -6.75780, 
     &  -6.61630,  -6.55460,  -6.53920,  -6.53970,  -6.51320, 
     &  -6.45310,  -6.41610,  -6.44820,  -6.56830,  -6.80860, 
     &  -7.17620,  -7.67720,  -8.35740,  -9.21880, -10.00000, 
     & -10.00000,  -9.53500,  -8.96860,  -8.53290,  -8.19200, 
     &  -7.92370,  -7.67970,  -7.50390,  -7.36670,  -7.28560/
      data cpco2(1101:1200)/
     &  -7.19690,  -7.07450,  -6.93300,  -6.79260,  -6.68180, 
     &  -6.61440,  -6.56430,  -6.51830,  -6.49100,  -6.44810, 
     &  -6.35670,  -6.21770,  -6.05660,  -5.90960,  -5.79750, 
     &  -5.70930,  -5.61650,  -5.51270,  -5.41240,  -5.34260, 
     &  -5.30610,  -5.26480,  -5.18640,  -5.08760,  -5.02260, 
     &  -5.03970,  -5.19050,  -5.48580,  -5.91010,  -6.48510, 
     &  -6.78620,  -6.53680,  -6.27650,  -6.03980,  -5.82600, 
     &  -5.63970,  -5.47990,  -5.34380,  -5.22740,  -5.14110, 
     &  -5.09170,  -5.04730,  -4.98200,  -4.91140,  -4.86340, 
     &  -4.88440,  -5.03630,  -5.33510,  -5.78020,  -6.53870, 
     &  -8.37350,  -9.99770,  -9.75060,  -9.18870,  -8.68240, 
     &  -8.34880,  -8.05330,  -7.86640,  -7.73460,  -7.69340, 
     &  -7.66740,  -7.62680,  -7.54510,  -7.46770,  -7.45200, 
     &  -7.54710,  -7.79130,  -8.19170,  -8.88350, -10.00000, 
     & -10.00000, -10.00000, -10.00000, -10.00000,  -9.72340, 
     &  -8.99690,  -8.57760,  -8.17370,  -7.86400,  -7.57290, 
     &  -7.31860,  -7.09730,  -6.91310,  -6.77820,  -6.70730, 
     &  -6.67680,  -6.63030,  -6.54060,  -6.45090,  -6.39500, 
     &  -6.43450,  -6.62700,  -6.95070,  -7.50280,  -8.64280, 
     & -10.00000, -10.00000, -10.00000, -10.00000,  -9.53030/
      data cpco2(1201:1219)/
     &  -8.93690,  -8.49520,  -8.14650,  -7.85670,  -7.61770, 
     &  -7.42490,  -7.28760,  -7.22060,  -7.19480,  -7.15520, 
     &  -7.07730,  -6.98840,  -6.94020,  -6.98390,  -7.17730, 
     &  -7.49990,  -8.06430,  -9.14800, -10.00000/

c --- double exponential fit parameters for co   

      data cpco(1:100)/
     &  -4.68680,  -4.41270,  -3.94610,  -3.56620,  -3.29210, 
     &  -3.10810,  -2.98070,  -2.89770,  -2.85800,  -2.84610, 
     &  -2.85870,  -2.90290,  -2.96460,  -3.04800,  -3.15890, 
     &  -3.28360,  -3.42770,  -3.59930,  -3.79630,  -4.01640, 
     &  -4.27990,  -4.57500,  -4.87220,  -5.27410,  -5.68190, 
     &  -6.07990,  -6.48280,  -6.88570,  -7.28860,  -7.69150, 
     &  -8.09440,  -8.49730,  -8.90020,  -9.30310,  -9.70600, 
     & -10.00000, -10.00000,  -9.53120,  -8.89770,  -8.26420, 
     &  -7.57670,  -6.99720,  -6.54080,  -6.12190,  -5.67340, 
     &  -5.26580,  -4.86860,  -4.49180,  -4.14230,  -3.81330, 
     &  -3.49980,  -3.21040,  -2.94430,  -2.71380,  -2.50840, 
     &  -2.31090,  -2.12450,  -1.93870,  -1.76080,  -1.60540, 
     &  -1.47330,  -1.35940,  -1.25400,  -1.14800,  -1.03410, 
     &  -0.92160,  -0.81890,  -0.72350,  -0.63620,  -0.55490, 
     &  -0.48560,  -0.44010,  -0.42680,  -0.46570,  -0.55710, 
     &  -0.65730,  -0.74040,  -0.75230,  -0.66010,  -0.53800, 
     &  -0.42110,  -0.33670,  -0.31670,  -0.33200,  -0.37530, 
     &  -0.44890,  -0.54380,  -0.66530,  -0.80520,  -0.96900, 
     &  -1.15060,  -1.35220,  -1.57910,  -1.82480,  -2.10730, 
     &  -2.42460,  -2.78770,  -3.21520,  -3.70890,  -4.28320/
      data cpco(101:173)/
     &  -4.95180,  -5.72510,  -6.53190,  -7.48790,  -9.08850, 
     & -10.00000, -10.00000,  -9.56110,  -9.08750,  -8.61390, 
     &  -7.97470,  -7.52500,  -7.19310,  -6.85960,  -6.57410, 
     &  -6.29220,  -6.00980,  -5.76690,  -5.53450,  -5.32290, 
     &  -5.14610,  -4.98820,  -4.84930,  -4.72390,  -4.60640, 
     &  -4.50090,  -4.40710,  -4.33220,  -4.26610,  -4.19260, 
     &  -4.09560,  -3.96110,  -3.79840,  -3.63140,  -3.47570, 
     &  -3.34080,  -3.22370,  -3.12190,  -3.03250,  -2.94940, 
     &  -2.87650,  -2.81170,  -2.75310,  -2.70230,  -2.66350, 
     &  -2.64400,  -2.65500,  -2.72250,  -2.81610,  -2.90150, 
     &  -2.92410,  -2.82280,  -2.67260,  -2.53200,  -2.42910, 
     &  -2.37720,  -2.37320,  -2.39950,  -2.45740,  -2.54860, 
     &  -2.66640,  -2.82090,  -3.01290,  -3.25160,  -3.54820, 
     &  -3.91650,  -4.37140,  -4.93260,  -5.63940,  -6.51630, 
     &  -7.60630,  -9.35750, -10.00000/

c --- double exponential fit parameters for ch4  

      data cpch4(1:100)/
     & -10.00000,  -9.45770,  -8.88660,  -8.22460,  -7.79400, 
     &  -7.17340,  -6.79650,  -6.56950,  -6.19290,  -5.91690, 
     &  -5.74520,  -5.47310,  -5.30010,  -5.18720,  -4.96720, 
     &  -4.84740,  -4.69390,  -4.52100,  -4.33770,  -4.13460, 
     &  -3.93220,  -3.73390,  -3.50770,  -3.27190,  -3.02960, 
     &  -2.81240,  -2.61990,  -2.44790,  -2.25020,  -2.05410, 
     &  -1.88000,  -1.70920,  -1.57910,  -1.43790,  -1.29920, 
     &  -1.17350,  -1.05100,  -0.96460,  -0.87790,  -0.80020, 
     &  -0.75740,  -0.73560,  -0.74780,  -0.75120,  -0.69060, 
     &  -0.55940,  -0.44170,  -0.40190,  -0.50270,  -0.76280, 
     &  -0.96250,  -1.04310,  -1.00680,  -0.87810,  -0.75590, 
     &  -0.66280,  -0.61280,  -0.61180,  -0.65750,  -0.76200, 
     &  -0.92170,  -1.12640,  -1.36600,  -1.63520,  -1.92640, 
     &  -2.22660,  -2.51230,  -2.74720,  -2.88200,  -2.91290, 
     &  -2.91450,  -2.88540,  -2.85080,  -2.85120,  -2.82020, 
     &  -2.80230,  -2.80040,  -2.78000,  -2.81750,  -2.84130, 
     &  -2.89430,  -2.98760,  -3.06880,  -3.24240,  -3.40640, 
     &  -3.57590,  -3.76300,  -3.89250,  -4.07740,  -4.32430, 
     &  -4.59640,  -3.86540,  -3.09740,  -2.59670,  -2.24820, 
     &  -2.10160,  -2.14880,  -2.32610,  -2.64480,  -3.04460/
      data cpch4(101:200)/
     &  -3.39580,  -3.65100,  -3.70490,  -3.72400,  -3.59920, 
     &  -3.49370,  -3.36760,  -3.22300,  -3.16300,  -3.06910, 
     &  -3.07760,  -3.08720,  -3.09740,  -3.12230,  -3.12850, 
     &  -3.12120,  -3.13330,  -3.16740,  -3.16680,  -3.24330, 
     &  -3.23980,  -3.31350,  -3.39750,  -3.44270,  -3.64340, 
     &  -3.75280,  -3.94660,  -4.19400,  -4.33620,  -4.55390, 
     &  -4.74100,  -4.91550,  -5.13450,  -5.39080,  -5.55920, 
     &  -5.82700,  -6.02890,  -6.23650,  -6.67300,  -7.05380, 
     &  -7.62160,  -8.56970,  -9.84830, -10.00000,  -9.35770, 
     &  -8.59500,  -7.83230,  -7.06960,  -6.30690,  -5.54420, 
     &  -5.15010,  -4.88530,  -4.69000,  -4.52620,  -4.39570, 
     &  -4.28230,  -4.27360,  -4.20540,  -4.11680,  -3.99860, 
     &  -3.87120,  -3.86920,  -3.87770,  -3.89650,  -3.90920, 
     &  -3.87880,  -3.76610,  -3.69000,  -3.62390,  -3.55970, 
     &  -3.51930,  -3.49060,  -3.44150,  -3.37300,  -3.35790, 
     &  -3.34270,  -3.32080,  -3.30480,  -3.31360,  -3.29040, 
     &  -3.25450,  -3.22410,  -3.14530,  -3.01870,  -2.94270, 
     &  -2.86300,  -2.81460,  -2.86040,  -2.89220,  -2.96500, 
     &  -2.99590,  -2.89200,  -2.79890,  -2.70280,  -2.65060, 
     &  -2.72850,  -2.84200,  -2.93040,  -2.96220,  -2.87260/
      data cpch4(201:300)/
     &  -2.75660,  -2.67450,  -2.63370,  -2.65330,  -2.68000, 
     &  -2.70980,  -2.74790,  -2.68590,  -2.62160,  -2.57010, 
     &  -2.46830,  -2.44260,  -2.44630,  -2.41940,  -2.45780, 
     &  -2.48940,  -2.46390,  -2.48250,  -2.49980,  -2.43810, 
     &  -2.41230,  -2.36540,  -2.26980,  -2.23870,  -2.23640, 
     &  -2.20290,  -2.17800,  -2.14330,  -2.03550,  -1.94580, 
     &  -1.87230,  -1.79360,  -1.76390,  -1.77820,  -1.80220, 
     &  -1.81150,  -1.78180,  -1.69860,  -1.61690,  -1.59750, 
     &  -1.65450,  -1.77420,  -1.89370,  -1.95440,  -1.89420, 
     &  -1.77610,  -1.63920,  -1.52360,  -1.45510,  -1.42210, 
     &  -1.42450,  -1.41740,  -1.41770,  -1.37760,  -1.33490, 
     &  -1.29090,  -1.24700,  -1.21620,  -1.18500,  -1.16770, 
     &  -1.14490,  -1.12290,  -1.10310,  -1.07950,  -1.06870, 
     &  -1.06920,  -1.09040,  -1.11660,  -1.15110,  -1.19510, 
     &  -1.23210,  -1.28310,  -1.27160,  -1.19020,  -0.97150, 
     &  -0.66540,  -0.41030,  -0.30110,  -0.50490,  -0.86590, 
     &  -1.17770,  -1.38470,  -1.43590,  -1.39080,  -1.29920, 
     &  -1.19230,  -1.09510,  -1.02130,  -0.95780,  -0.92990, 
     &  -0.92070,  -0.92920,  -0.97250,  -1.01260,  -1.07500, 
     &  -1.11490,  -1.16360,  -1.20590,  -1.26380,  -1.33270/
      data cpch4(301:400)/
     &  -1.40790,  -1.49830,  -1.57110,  -1.68720,  -1.78700, 
     &  -1.92660,  -2.07740,  -2.21190,  -2.38750,  -2.51550, 
     &  -2.68220,  -2.83720,  -3.00320,  -3.24130,  -3.50580, 
     &  -3.95080,  -4.51330,  -5.35360,  -8.08150,  -8.90810, 
     &  -9.81550, -10.00000,  -7.47570,  -5.16020,  -4.24540, 
     &  -3.76400,  -3.32560,  -3.01030,  -2.77260,  -2.55100, 
     &  -2.38490,  -2.23180,  -2.10800,  -2.00860,  -1.92900, 
     &  -1.89020,  -1.87500,  -1.87000,  -1.84760,  -1.73900, 
     &  -1.57240,  -1.42840,  -1.34250,  -1.37910,  -1.51320, 
     &  -1.65080,  -1.72830,  -1.66840,  -1.54320,  -1.44470, 
     &  -1.37730,  -1.34900,  -1.36420,  -1.40160,  -1.47130, 
     &  -1.58360,  -1.69840,  -1.80850,  -1.84860,  -1.74640, 
     &  -1.63380,  -1.55550,  -1.55520,  -1.69350,  -1.81650, 
     &  -1.84170,  -1.76970,  -1.63460,  -1.55890,  -1.54660, 
     &  -1.56040,  -1.63070,  -1.68670,  -1.75930,  -1.80510, 
     &  -1.81670,  -1.85180,  -1.85590,  -1.85470,  -1.89070, 
     &  -1.88510,  -1.89330,  -1.90810,  -1.90250,  -1.94510, 
     &  -1.99240,  -2.03210,  -2.08160,  -2.10260,  -2.11370, 
     &  -2.13510,  -2.16290,  -2.18760,  -2.23400,  -2.29600, 
     &  -2.37470,  -2.49700,  -2.62440,  -2.76410,  -2.89120/
      data cpch4(401:493)/
     &  -3.03280,  -3.19440,  -3.38770,  -3.45660,  -3.16620, 
     &  -2.72530,  -2.39920,  -2.22140,  -2.20220,  -2.39780, 
     &  -2.74490,  -3.26390,  -3.93110,  -4.14700,  -3.93510, 
     &  -3.74710,  -3.62450,  -3.47910,  -3.47100,  -3.42100, 
     &  -3.41250,  -3.44750,  -3.41400,  -3.49080,  -3.51640, 
     &  -3.59440,  -3.74030,  -3.81920,  -4.01770,  -4.18330, 
     &  -4.35180,  -4.64860,  -4.87780,  -5.25420,  -5.78340, 
     &  -6.34510,  -7.72120, -10.00000,  -9.91340,  -7.91810, 
     &  -6.08150,  -5.43970,  -4.98750,  -4.61540,  -4.48460, 
     &  -4.35410,  -4.30370,  -4.30730,  -4.24710,  -4.25930, 
     &  -4.19840,  -4.18950,  -4.16970,  -4.15780,  -4.19500, 
     &  -4.18780,  -4.22990,  -4.22090,  -4.26460,  -4.31230, 
     &  -4.39110,  -4.45880,  -4.18730,  -3.83530,  -3.52820, 
     &  -3.30550,  -3.33510,  -3.56710,  -3.87500,  -4.26450, 
     &  -4.47860,  -4.42930,  -4.31830,  -4.19960,  -4.08790, 
     &  -4.01690,  -3.97870,  -3.95360,  -3.94540,  -3.92830, 
     &  -3.91660,  -3.91520,  -3.93360,  -3.95610,  -3.99320, 
     &  -4.09340,  -4.23170,  -4.50840,  -4.94600,  -5.49580, 
     &  -6.54920,  -8.56040,  -9.62020/

c --- double exponential fit parameters for n2o  

      data cpn2o(1:100)/
     &  -2.80030,  -2.66280,  -2.43130,  -2.25790,  -2.17000, 
     &  -2.17020,  -2.24900,  -2.40030,  -2.62640,  -2.92190, 
     &  -3.29540,  -3.76840,  -4.26210,  -4.75580,  -5.24950, 
     &  -5.74320,  -6.23690,  -6.73060,  -7.22430,  -7.71800, 
     &  -8.21170,  -8.70540,  -9.19910,  -9.69280, -10.00000, 
     &  -9.71850,  -8.89260,  -8.06670,  -7.23070,  -6.41490, 
     &  -5.48720,  -4.70830,  -4.03190,  -3.47520,  -3.01550, 
     &  -2.60460,  -2.20570,  -1.81370,  -1.47410,  -1.19140, 
     &  -0.96030,  -0.79230,  -0.66290,  -0.58490,  -0.54020, 
     &  -0.49750,  -0.51480,  -0.55920,  -0.65210,  -0.81480, 
     &  -1.01860,  -1.27640,  -1.58730,  -1.96380,  -2.38810, 
     &  -2.80830,  -3.23920,  -3.69340,  -4.06820,  -4.13660, 
     &  -3.94230,  -3.71430,  -3.49750,  -3.26020,  -3.09760, 
     &  -2.98150,  -2.91530,  -2.95960,  -3.02810,  -3.12640, 
     &  -3.26500,  -3.39060,  -3.57170,  -3.83120,  -4.17060, 
     &  -4.60770,  -5.18390,  -5.92240,  -6.98620,  -7.69010, 
     &  -8.39400,  -9.09790,  -9.80180,  -9.91540,  -9.22710, 
     &  -8.53880,  -7.85040,  -7.16210,  -6.24280,  -5.60510, 
     &  -5.09710,  -4.72370,  -4.41040,  -4.20500,  -4.06810, 
     &  -4.02780,  -4.03070,  -4.04920,  -4.03330,  -3.97100/
      data cpn2o(101:200)/
     &  -3.92490,  -3.93600,  -4.03160,  -4.23170,  -4.54140, 
     &  -4.97870,  -5.56230,  -6.33350,  -7.99680,  -9.66010, 
     &  -9.54860,  -8.85170,  -8.15480,  -7.45790,  -6.76100, 
     &  -6.06410,  -5.36720,  -4.67030,  -3.69180,  -3.06560, 
     &  -2.57960,  -2.18760,  -1.86460,  -1.59190,  -1.35870, 
     &  -1.16840,  -1.02860,  -0.94700,  -0.92710,  -0.94420, 
     &  -0.96950,  -0.97530,  -0.95730,  -0.95500,  -1.00000, 
     &  -1.10700,  -1.27910,  -1.49760,  -1.72810,  -1.92770, 
     &  -2.02270,  -1.95770,  -1.76250,  -1.50200,  -1.21860, 
     &  -0.92700,  -0.63260,  -0.34290,  -0.07680,   0.15000, 
     &   0.32150,   0.41040,   0.43850,   0.42880,   0.41850, 
     &   0.45700,   0.49720,   0.49870,   0.42160,   0.23600, 
     &  -0.03190,  -0.37140,  -0.75390,  -1.15340,  -1.58550, 
     &  -2.06100,  -2.60680,  -3.26350,  -4.10380,  -5.27610, 
     &  -6.14370,  -7.00790,  -7.94400,  -8.88010,  -9.81620, 
     & -10.00000,  -9.59510,  -9.13050,  -8.66590,  -8.20130, 
     &  -7.73670,  -7.27210,  -6.80750,  -6.15980,  -5.86950, 
     &  -5.35100,  -4.94910,  -4.63100,  -4.38460,  -4.07840, 
     &  -3.77630,  -3.59010,  -3.46070,  -3.43860,  -3.54810, 
     &  -3.70140,  -3.93100,  -4.22510,  -4.45930,  -4.82100/
      data cpn2o(201:300)/
     &  -5.34940,  -6.12860,  -7.59810, -10.00000, -10.00000, 
     & -10.00000, -10.00000,  -6.37430,  -5.55920,  -5.01290, 
     &  -4.60750,  -4.31710,  -4.09280,  -3.75370,  -3.54060, 
     &  -3.38690,  -3.29130,  -3.36330,  -3.49320,  -3.69240, 
     &  -4.00740,  -4.25040,  -4.53890,  -4.94250,  -5.47410, 
     &  -6.20690,  -7.59810, -10.00000, -10.00000, -10.00000, 
     &  -6.92150,  -6.07980,  -5.19340,  -4.62880,  -4.13160, 
     &  -3.73220,  -3.40890,  -3.15730,  -2.95730,  -2.72980, 
     &  -2.56150,  -2.43820,  -2.35230,  -2.37740,  -2.45080, 
     &  -2.57550,  -2.77570,  -2.99040,  -3.27330,  -3.65240, 
     &  -4.15990,  -4.79520,  -5.70040,  -6.87620,  -6.98220, 
     &  -6.24840,  -5.76130,  -5.25860,  -4.86740,  -4.66330, 
     &  -4.53320,  -4.51580,  -4.65930,  -4.84270,  -5.09170, 
     &  -5.57810,  -6.06450,  -6.55090,  -7.03730,  -7.52370, 
     &  -8.01010,  -8.49650,  -8.98290,  -9.46930,  -9.95570, 
     &  -9.71300,  -8.66090,  -7.60890,  -6.55680,  -5.08800, 
     &  -4.45270,  -3.93020,  -3.44380,  -2.97010,  -2.54230, 
     &  -2.16160,  -1.80760,  -1.47630,  -1.15800,  -0.84450, 
     &  -0.54550,  -0.25060,   0.02340,   0.27750,   0.51130, 
     &   0.71540,   0.89290,   1.03590,   1.13060,   1.16970/
      data cpn2o(301:400)/
     &   1.18070,   1.18030,   1.19740,   1.24660,   1.26290, 
     &   1.20680,   1.04720,   0.76950,   0.40830,  -0.02440, 
     &  -0.54770,  -1.22020,  -2.10670,  -2.95080,  -3.21070, 
     &  -3.15870,  -2.96000,  -2.76410,  -2.63240,  -2.56710, 
     &  -2.56640,  -2.60880,  -2.64250,  -2.66060,  -2.68950, 
     &  -2.75510,  -2.88370,  -3.08840,  -3.37460,  -3.70780, 
     &  -4.09750,  -4.62720,  -5.24840, -10.00000, -10.00000, 
     & -10.00000,  -7.35710,  -5.02870,  -4.30470,  -3.64310, 
     &  -3.10260,  -2.61220,  -2.19410,  -1.84540,  -1.57260, 
     &  -1.38290,  -1.28180,  -1.25050,  -1.25790,  -1.27310, 
     &  -1.25020,  -1.20920,  -1.20440,  -1.25770,  -1.39420, 
     &  -1.62620,  -1.93470,  -2.28300,  -2.53860,  -2.48010, 
     &  -2.16710,  -1.80610,  -1.47260,  -1.17970,  -0.93770, 
     &  -0.75420,  -0.63920,  -0.58990,  -0.57430,  -0.56690, 
     &  -0.53390,  -0.47450,  -0.44710,  -0.47790,  -0.58770, 
     &  -0.79640,  -1.09420,  -1.48120,  -1.95930,  -2.51400, 
     &  -3.13500,  -3.81020,  -4.58250,  -5.59820,  -6.41930, 
     &  -7.24030,  -8.06140,  -8.88250,  -9.70350,  -9.89100, 
     &  -8.98760,  -8.08430,  -7.18090,  -6.15010,  -5.37420, 
     &  -4.73520,  -4.20510,  -3.75250,  -3.35620,  -2.99160/
      data cpn2o(401:500)/
     &  -2.66490,  -2.38720,  -2.14990,  -1.97470,  -1.79820, 
     &  -1.65180,  -1.55820,  -1.48380,  -1.50040,  -1.58210, 
     &  -1.69120,  -1.86730,  -2.07560,  -2.33510,  -2.70200, 
     &  -3.19210,  -3.84090,  -4.70850,  -5.95880,  -6.58290, 
     &  -8.55850,  -9.85840,  -9.97230,  -9.42150,  -8.87070, 
     &  -8.31990,  -7.76910,  -7.21830,  -6.55670,  -6.43450, 
     &  -5.64480,  -5.05290,  -4.46430,  -3.96240,  -3.52310, 
     &  -3.13950,  -2.80670,  -2.52320,  -2.28580,  -2.08200, 
     &  -1.90490,  -1.75540,  -1.64850,  -1.59590,  -1.58380, 
     &  -1.59610,  -1.59970,  -1.57340,  -1.56150,  -1.59740, 
     &  -1.70590,  -1.90340,  -2.16310,  -2.41810,  -2.54270, 
     &  -2.45920,  -2.25130,  -2.01870,  -1.78790,  -1.56120, 
     &  -1.33990,  -1.12650,  -0.92260,  -0.73790,  -0.57900, 
     &  -0.45730,  -0.39520,  -0.36830,  -0.35110,  -0.32160, 
     &  -0.25560,  -0.21260,  -0.25930,  -0.43610,  -0.77020, 
     &  -1.20890,  -1.70600,  -2.29370,  -3.11330,  -4.44190, 
     &  -6.01190,  -6.94570, -10.00000, -10.00000, -10.00000, 
     & -10.00000,  -7.03940,  -5.96370,  -5.23170,  -4.64190, 
     &  -4.16630,  -3.78740,  -3.50000,  -3.30860,  -3.21430, 
     &  -3.19260,  -3.21050,  -3.23080,  -3.19710,  -3.15100/
      data cpn2o(501:600)/
     &  -3.14020,  -3.19690,  -3.34770,  -3.60050,  -3.95340, 
     &  -4.41170,  -4.97290,  -5.60090,  -6.21790,  -5.98450, 
     &  -5.55020,  -4.90100,  -4.34010,  -3.82320,  -3.38020, 
     &  -2.99720,  -2.67470,  -2.41430,  -2.22090,  -2.10800, 
     &  -2.06820,  -2.06870,  -2.07750,  -2.04850,  -1.98470, 
     &  -1.95310,  -1.98700,  -2.11100,  -2.33660,  -2.62930, 
     &  -2.89220,  -2.94740,  -2.76270,  -2.49990,  -2.25540, 
     &  -2.05370,  -1.90620,  -1.82680,  -1.79410,  -1.77660, 
     &  -1.74680,  -1.67670,  -1.61300,  -1.60850,  -1.68490, 
     &  -1.85990,  -2.12580,  -2.45380,  -2.82050,  -3.20280, 
     &  -3.59880,  -4.06910,  -4.71170,  -5.63200,  -6.48060, 
     &  -7.37310,  -8.26020,  -9.14740, -10.00000, -10.00000, 
     &  -9.53400,  -9.02820,  -8.52240,  -8.01660,  -7.51090, 
     &  -7.00510,  -6.41170,  -6.01480,  -5.48780,  -5.17420, 
     &  -4.88590,  -4.48730,  -4.22490,  -4.02850,  -3.86690, 
     &  -3.82470,  -3.76520,  -3.65210,  -3.49060,  -3.26130, 
     &  -3.03070,  -2.81560,  -2.61720,  -2.42640,  -2.24420, 
     &  -2.07750,  -1.94320,  -1.87030,  -1.85230,  -1.85520, 
     &  -1.84430,  -1.78140,  -1.71040,  -1.70430,  -1.79520, 
     &  -2.02050,  -2.39680,  -2.93740,  -3.76890,  -5.31590/
      data cpn2o(601:700)/
     &  -7.41390,  -9.51190,  -9.79650,  -9.15110,  -8.50570, 
     &  -7.86030,  -7.21490,  -6.56950,  -6.24150,  -5.58290, 
     &  -5.02960,  -4.56600,  -4.17220,  -3.83640,  -3.55510, 
     &  -3.33980,  -3.19700,  -3.13630,  -3.12320,  -3.12570, 
     &  -3.09990,  -3.02880,  -2.97460,  -2.98750,  -3.09250, 
     &  -3.31370,  -3.64960,  -4.02760,  -4.19580,  -3.97600, 
     &  -3.61790,  -3.27250,  -2.96530,  -2.69620,  -2.46770, 
     &  -2.28280,  -2.15470,  -2.09490,  -2.07630,  -2.06060, 
     &  -2.01420,  -1.92390,  -1.86180,  -1.88130,  -2.00990, 
     &  -2.28250,  -2.70710,  -3.32770,  -4.33000,  -6.21510, 
     &  -8.35430, -10.00000,  -9.72750,  -9.12570,  -8.52390, 
     &  -7.92210,  -7.32030,  -6.71850,  -6.60890,  -5.88770, 
     &  -5.45270,  -5.08790,  -4.65980,  -4.38060,  -4.18300, 
     &  -4.04260,  -4.01750,  -4.01780,  -3.98110,  -3.92440, 
     &  -3.80560,  -3.69680,  -3.64350,  -3.63260,  -3.63390, 
     &  -3.61570,  -3.54780,  -3.48260,  -3.48070,  -3.56650, 
     &  -3.76500,  -4.07180,  -4.39800,  -4.50750,  -4.33580, 
     &  -4.07650,  -3.86740,  -3.72210,  -3.65880,  -3.64290, 
     &  -3.63710,  -3.60140,  -3.52090,  -3.46160,  -3.47740, 
     &  -3.59570,  -3.84810,  -4.25980,  -4.87840,  -5.82660/
      data cpn2o(701:704)/
     &  -6.74680,  -8.13520,  -9.22080, -10.00000/

c --- double exponential fit parameters for o2   

      data cpo2(1:100)/
     &  -6.13630,  -6.17940,  -6.25380,  -6.37050,  -6.51100, 
     &  -6.61620,  -6.75050,  -6.78960,  -6.83050,  -6.84710, 
     &  -6.82820,  -6.87720,  -6.86800,  -6.93320,  -6.95110, 
     &  -7.00480,  -7.06620,  -7.10430,  -7.20550,  -7.24430, 
     &  -7.35200,  -7.40790,  -7.49980,  -7.59240,  -7.66820, 
     &  -7.79930,  -7.87120,  -8.01610,  -8.11020,  -8.24850, 
     &  -8.37580,  -8.49420,  -8.65320,  -8.75540,  -8.94530, 
     &  -9.06650,  -9.26310,  -9.43870,  -9.63250,  -9.87570, 
     & -10.06280, -10.37610, -10.54780, -10.91470, -11.20520, 
     & -11.51290, -11.82060, -12.12830, -12.43600, -12.74370, 
     & -13.05140, -13.35910, -13.66680, -13.97450, -13.94580, 
     & -13.76920, -13.50480, -13.14220, -13.02420, -12.66840, 
     & -12.35710, -12.24280, -11.84920, -11.64270, -11.51730, 
     & -11.21080, -11.15840, -11.01960, -10.80400, -10.80590, 
     & -10.58280, -10.45800, -10.41700, -10.18230, -10.14350, 
     & -10.00300,  -9.81360,  -9.77720,  -9.56800,  -9.45950, 
     &  -9.35020,  -9.14110,  -9.04760,  -8.86280,  -8.70510, 
     &  -8.58380,  -8.42820,  -8.32710,  -8.19580,  -8.08380, 
     &  -7.96520,  -7.83710,  -7.74760,  -7.64310,  -7.57360, 
     &  -7.51490,  -7.41940,  -7.26880,  -7.07220,  -6.88150/
      data cpo2(101:200)/
     &  -6.76270,  -6.80550,  -6.91140,  -6.99360,  -7.05190, 
     &  -7.05970,  -7.06800,  -7.12420,  -7.20880,  -7.32650, 
     &  -7.46730,  -7.63260,  -7.81100,  -8.00960,  -8.21040, 
     &  -8.40360,  -8.58530,  -8.72520,  -8.85110,  -8.94270, 
     &  -9.03750,  -9.12280,  -9.22460,  -9.32910,  -9.44360, 
     &  -9.57160,  -9.69510,  -9.84080,  -9.97590, -10.14890, 
     & -10.30270, -10.51780, -10.72650, -10.97870, -11.29390, 
     & -11.55520, -11.95950, -12.24360, -12.69420, -13.20110, 
     & -13.81910, -13.92160, -13.72930, -13.53700, -13.34470, 
     & -13.15230, -12.96000, -12.76770, -12.57540, -12.38300, 
     & -12.19070, -11.99480, -11.77590, -11.59260, -11.42140, 
     & -11.24930, -11.10940, -10.94770, -10.83320, -10.73230, 
     & -10.63800, -10.57250, -10.44090, -10.20130,  -9.88390, 
     &  -9.65460,  -9.50530,  -9.46380,  -9.55260,  -9.65580, 
     &  -9.74300,  -9.79580,  -9.78960,  -9.83200,  -9.94470, 
     & -10.12210, -10.37070, -10.66230, -10.97610, -11.22710, 
     & -11.40910, -11.49210, -11.60150, -11.69450, -11.83330, 
     & -11.99850, -12.17880, -12.38220, -12.66050, -13.07960, 
     & -13.35280, -13.64630, -13.93980, -13.70340, -13.31500, 
     & -13.11770, -12.64620, -12.48680, -12.22050, -11.96500/
      data cpo2(201:300)/
     & -11.69410, -11.43770, -11.21360, -10.95670, -10.79800, 
     & -10.55460, -10.39520, -10.24030, -10.04910,  -9.92260, 
     &  -9.78710,  -9.65570,  -9.61060,  -9.51420,  -9.47630, 
     &  -9.41630,  -9.23480,  -9.10880,  -8.79460,  -8.58760, 
     &  -8.31280,  -8.09450,  -7.91270,  -7.72290,  -7.58600, 
     &  -7.42150,  -7.27260,  -7.11790,  -6.95160,  -6.80750, 
     &  -6.64130,  -6.50430,  -6.35190,  -6.21120,  -6.08390, 
     &  -5.93370,  -5.83210,  -5.69690,  -5.59230,  -5.50760, 
     &  -5.40020,  -5.34130,  -5.28260,  -5.24580,  -5.28770, 
     &  -5.37430,  -5.46540,  -5.52620,  -5.44290,  -5.24300, 
     &  -5.02840,  -4.84640,  -4.75340,  -4.78250,  -4.94620, 
     &  -5.22900,  -5.64400,  -6.18890,  -6.84270,  -7.77310, 
     &  -9.16880,  -9.68930, -10.18530, -10.76700, -11.46110, 
     & -12.30810, -13.14760, -13.81920, -13.58710, -13.21890, 
     & -12.97050, -12.48250, -12.13010, -11.94300, -11.66360, 
     & -11.31970, -11.16780, -10.89670, -10.60020, -10.48570, 
     & -10.19860,  -9.97310,  -9.85470,  -9.58170,  -9.43820, 
     &  -9.30420,  -9.07550,  -8.99440,  -8.80600,  -8.65430, 
     &  -8.54410,  -8.35560,  -8.25570,  -8.09590,  -7.97170, 
     &  -7.84530,  -7.70760,  -7.59100,  -7.45670,  -7.34390/
      data cpo2(301:382)/
     &  -7.22480,  -7.12360,  -7.02090,  -6.93450,  -6.84040, 
     &  -6.75600,  -6.67440,  -6.58700,  -6.52780,  -6.48090, 
     &  -6.50420,  -6.57970,  -6.65640,  -6.69390,  -6.59120, 
     &  -6.37760,  -6.14380,  -6.00620,  -6.04690,  -6.30810, 
     &  -6.81990,  -7.43070,  -8.13450,  -9.11900, -10.42030, 
     & -11.46980, -12.59420, -13.53160, -13.86930, -13.93920, 
     & -13.68850, -13.43770, -13.18690, -12.93620, -12.68540, 
     & -12.37200, -12.28520, -11.93310, -11.75750, -11.62970, 
     & -11.32900, -11.12050, -11.00840, -10.72430, -10.55430, 
     & -10.44850, -10.17640, -10.07590,  -9.93040,  -9.71960, 
     &  -9.66300,  -9.47740,  -9.36380,  -9.26750,  -9.11210, 
     &  -9.03680,  -8.90250,  -8.80280,  -8.70120,  -8.59090, 
     &  -8.51210,  -8.41410,  -8.34440,  -8.26870,  -8.20030, 
     &  -8.15710,  -8.11410,  -8.12610,  -8.18480,  -8.23950, 
     &  -8.24780,  -8.08770,  -7.78800,  -7.56110,  -7.44870, 
     &  -7.48800,  -7.76440,  -8.21420,  -8.87650, -10.10910, 
     & -12.44830, -13.72280/

c --- double exponential fit parameters for nh3  

      data cpnh3(1:100)/
     &  -5.71420,  -5.28540,  -4.51630,  -3.97950,  -3.43930, 
     &  -2.87350,  -2.49470,  -2.22900,  -2.06240,  -1.96160, 
     &  -1.87070,  -1.77120,  -1.64730,  -1.53760,  -1.43150, 
     &  -1.33280,  -1.23910,  -1.17680,  -1.13020,  -1.07550, 
     &  -1.02720,  -0.98840,  -0.95010,  -0.92870,  -0.91010, 
     &  -0.89820,  -0.88880,  -0.87090,  -0.86200,  -0.86450, 
     &  -0.86760,  -0.89100,  -0.90840,  -0.93280,  -0.95460, 
     &  -0.97430,  -0.99830,  -1.02020,  -1.05690,  -1.08240, 
     &  -1.10860,  -1.14750,  -1.17900,  -1.20590,  -1.26680, 
     &  -1.32370,  -1.38010,  -1.42710,  -1.49200,  -1.54030, 
     &  -1.58480,  -1.64980,  -1.73820,  -1.82940,  -1.92030, 
     &  -2.06940,  -2.21340,  -2.36220,  -2.55160,  -2.76330, 
     &  -2.93440,  -3.11720,  -3.35430,  -3.56710,  -3.75040, 
     &  -3.98840,  -4.26330,  -4.55050,  -4.78370,  -5.03500, 
     &  -5.37330,  -5.64780,  -5.88560,  -6.10410,  -6.33750, 
     &  -6.57090,  -6.80430,  -7.03770,  -7.26200,  -7.09500, 
     &  -6.92790,  -6.76080,  -6.59380,  -6.42670,  -6.25970, 
     &  -6.09260,  -5.88420,  -5.75600,  -5.58440,  -5.42480, 
     &  -5.25730,  -5.07710,  -4.92440,  -4.79030,  -4.65120, 
     &  -4.51690,  -4.39610,  -4.26070,  -4.17050,  -4.12940/
      data cpnh3(101:200)/
     &  -4.06110,  -3.95380,  -3.88210,  -3.75920,  -3.67540, 
     &  -3.68300,  -3.69770,  -3.69250,  -3.66320,  -3.58990, 
     &  -3.52180,  -3.52650,  -3.65350,  -3.80680,  -3.98180, 
     &  -4.05740,  -3.97890,  -3.88580,  -3.81200,  -3.89270, 
     &  -3.87990,  -3.86230,  -3.39840,  -2.88570,  -2.58140, 
     &  -2.40660,  -2.38500,  -2.54150,  -2.81610,  -3.22650, 
     &  -3.71770,  -3.99320,  -4.06830,  -4.07850,  -3.99120, 
     &  -3.74180,  -3.47420,  -3.26510,  -3.07150,  -2.95000, 
     &  -2.86690,  -2.77230,  -2.66140,  -2.56130,  -2.43720, 
     &  -2.30850,  -2.16960,  -2.03020,  -1.91660,  -1.80710, 
     &  -1.72210,  -1.63700,  -1.54530,  -1.44870,  -1.35390, 
     &  -1.25700,  -1.16180,  -1.11310,  -1.08240,  -1.05590, 
     &  -1.01900,  -0.97210,  -0.92180,  -0.86800,  -0.85560, 
     &  -0.85680,  -0.87130,  -0.89840,  -0.90760,  -0.90240, 
     &  -0.88820,  -0.89680,  -0.94920,  -1.00890,  -1.08460, 
     &  -1.15560,  -1.17920,  -1.19460,  -1.19640,  -1.21730, 
     &  -1.24240,  -1.17440,  -0.97430,  -0.63500,  -0.29750, 
     &  -0.07050,   0.01440,  -0.09780,  -0.35360,  -0.56300, 
     &  -0.54790,  -0.37840,  -0.17970,  -0.11510,  -0.30850, 
     &  -0.61800,  -0.97180,  -1.29260,  -1.27480,  -1.12170/
      data cpnh3(201:300)/
     &  -1.01970,  -0.93000,  -0.88170,  -0.87230,  -0.83090, 
     &  -0.78040,  -0.70750,  -0.64310,  -0.61760,  -0.60120, 
     &  -0.60790,  -0.62720,  -0.63040,  -0.61930,  -0.60260, 
     &  -0.58820,  -0.60290,  -0.63170,  -0.68620,  -0.74470, 
     &  -0.79210,  -0.82750,  -0.85950,  -0.88560,  -0.92360, 
     &  -0.99340,  -1.06930,  -1.14600,  -1.21000,  -1.28630, 
     &  -1.35930,  -1.42920,  -1.50290,  -1.60540,  -1.70670, 
     &  -1.81100,  -1.93500,  -2.03460,  -2.13050,  -2.22940, 
     &  -2.37240,  -2.49170,  -2.62180,  -2.80560,  -2.96930, 
     &  -3.11010,  -3.27900,  -3.53150,  -3.70110,  -3.89520, 
     &  -4.15270,  -4.41210,  -4.52440,  -4.85990,  -5.19400, 
     &  -5.55890,  -5.81700,  -6.14020,  -6.46330,  -6.78650, 
     &  -7.10960,  -7.43280,  -7.75590,  -8.00000,  -7.81990, 
     &  -7.59880,  -7.37780,  -7.15670,  -6.93570,  -6.71460, 
     &  -6.49360,  -6.27250,  -6.05150,  -5.83040,  -5.59630, 
     &  -5.38830,  -5.23190,  -5.05360,  -4.90290,  -4.77890, 
     &  -4.58670,  -4.34140,  -4.13990,  -3.97840,  -3.75530, 
     &  -3.57730,  -3.41230,  -3.22540,  -3.03840,  -2.92430, 
     &  -2.77550,  -2.58090,  -2.47260,  -2.32060,  -2.12090, 
     &  -2.03310,  -1.90160,  -1.74580,  -1.69270,  -1.59580/
      data cpnh3(301:400)/
     &  -1.48630,  -1.44920,  -1.37300,  -1.28590,  -1.25540, 
     &  -1.21290,  -1.16890,  -1.18020,  -1.19480,  -1.18820, 
     &  -1.21850,  -1.24640,  -1.25220,  -1.29460,  -1.35870, 
     &  -1.39710,  -1.44880,  -1.52610,  -1.54950,  -1.54780, 
     &  -1.49260,  -1.31150,  -1.04550,  -0.79870,  -0.59720, 
     &  -0.46640,  -0.42440,  -0.44260,  -0.49520,  -0.57720, 
     &  -0.68450,  -0.80970,  -0.94430,  -1.09040,  -1.22320, 
     &  -1.28530,  -1.29490,  -1.27080,  -1.18960,  -1.14670, 
     &  -1.11870,  -1.07000,  -1.03920,  -1.02270,  -1.01780, 
     &  -1.00890,  -1.00210,  -0.97060,  -0.95690,  -0.99280, 
     &  -1.03100,  -1.07670,  -1.10530,  -1.12410,  -1.17170, 
     &  -1.22030,  -1.27720,  -1.33560,  -1.38550,  -1.47340, 
     &  -1.57010,  -1.65720,  -1.76380,  -1.86520,  -1.99180, 
     &  -2.14490,  -2.23880,  -2.32510,  -2.39360,  -2.45250, 
     &  -2.59980,  -2.71470,  -2.77040,  -2.78520,  -2.75240, 
     &  -2.76460,  -2.85070,  -3.04220,  -3.26420,  -3.52010, 
     &  -3.63280,  -3.76240,  -3.95050,  -4.13990,  -4.30870, 
     &  -4.38590,  -4.42950,  -4.44930,  -4.33170,  -4.18920, 
     &  -4.05450,  -3.93560,  -3.91170,  -4.00010,  -4.06270, 
     &  -4.08330,  -4.09970,  -4.06590,  -4.02640,  -4.08930/
      data cpnh3(401:431)/
     &  -4.18320,  -4.25220,  -4.31820,  -4.39490,  -4.41910, 
     &  -4.45800,  -4.59970,  -4.72820,  -4.83700,  -5.00410, 
     &  -5.16440,  -5.21010,  -5.41450,  -5.51140,  -5.69860, 
     &  -5.80570,  -5.95290,  -6.10000,  -6.24720,  -6.39430, 
     &  -6.54150,  -6.68860,  -6.83580,  -6.98290,  -7.13010, 
     &  -7.27720,  -7.42440,  -7.57150,  -7.71870,  -7.86580, 
     &  -8.00000/

c --- double exponential fit parameters for no   

      data cpno/
     &  -7.92650,  -7.56490,  -7.20330,  -6.84180,  -6.48020, 
     &  -6.06470,  -5.71930,  -5.39550,  -5.14750,  -4.82330, 
     &  -4.51940,  -4.31840,  -3.96640,  -3.70450,  -3.33980, 
     &  -3.03680,  -2.72820,  -2.44480,  -2.17910,  -1.93150, 
     &  -1.70460,  -1.49840,  -1.31330,  -1.14860,  -1.00360, 
     &  -0.87760,  -0.76990,  -0.68110,  -0.61240,  -0.56630, 
     &  -0.54880,  -0.56730,  -0.60760,  -0.67910,  -0.75530, 
     &  -0.78110,  -0.77110,  -0.68400,  -0.57040,  -0.47910, 
     &  -0.41380,  -0.39500,  -0.41890,  -0.47940,  -0.57510, 
     &  -0.70620,  -0.87510,  -1.08520,  -1.34060,  -1.64730, 
     &  -2.00680,  -2.43350,  -2.90680,  -3.45950,  -4.03700, 
     &  -4.67950,  -5.27040,  -5.86130,  -6.45220,  -7.04310, 
     &  -7.63400,  -8.00000/

c --- double exponential fit parameters for no2  

      data cpno2(1:100)/
     &  -6.00000,  -5.84190,  -5.53130,  -5.10480,  -4.95120, 
     &  -4.58300,  -4.26760,  -3.97830,  -3.71500,  -3.47820, 
     &  -3.25410,  -3.05970,  -2.86250,  -2.69890,  -2.53230, 
     &  -2.39040,  -2.25610,  -2.13460,  -2.03200,  -1.92840, 
     &  -1.85840,  -1.77780,  -1.72220,  -1.67760,  -1.60240, 
     &  -1.56580,  -1.49170,  -1.41170,  -1.37060,  -1.30450, 
     &  -1.29140,  -1.32920,  -1.36660,  -1.42680,  -1.45640, 
     &  -1.40760,  -1.32840,  -1.28040,  -1.24970,  -1.25190, 
     &  -1.31230,  -1.37040,  -1.41920,  -1.48780,  -1.53010, 
     &  -1.55750,  -1.59120,  -1.62500,  -1.65440,  -1.68490, 
     &  -1.73400,  -1.77480,  -1.81710,  -1.86790,  -1.92560, 
     &  -1.98090,  -2.03860,  -2.11120,  -2.17690,  -2.24620, 
     &  -2.31990,  -2.41290,  -2.51560,  -2.65750,  -2.88250, 
     &  -3.18310,  -3.62090,  -4.22710,  -5.52900,  -6.00000, 
     &  -6.00000,  -5.54150,  -4.89640,  -4.25130,  -3.60630, 
     &  -2.96120,  -2.17330,  -1.55140,  -1.02600,  -0.58170, 
     &  -0.20300,   0.12310,   0.40980,   0.66530,   0.88850, 
     &   1.07160,   1.20250,   1.26970,   1.29260,   1.30060, 
     &   1.31280,   1.34490,   1.36560,   1.32450,   1.18680, 
     &   0.93100,   0.59070,   0.20560,  -0.23370,  -0.76330/
      data cpno2(101:142)/
     &  -1.45410,  -2.44510,  -3.18220,  -3.91930,  -4.65650, 
     &  -5.39360,  -6.00000,  -6.00000,  -5.76060,  -5.34220, 
     &  -4.92380,  -4.50550,  -4.08710,  -3.66870,  -3.25040, 
     &  -2.83200,  -2.37360,  -1.95650,  -1.57690,  -1.24000, 
     &  -0.93840,  -0.67810,  -0.46300,  -0.29440,  -0.17830, 
     &  -0.12130,  -0.10330,  -0.09340,  -0.07230,  -0.02670, 
     &   0.00160,  -0.03940,  -0.17000,  -0.41410,  -0.78610, 
     &  -1.29510,  -2.03790,  -3.09840,  -3.86920,  -4.63990, 
     &  -5.41070,  -6.00000/

c --- double exponential fit parameters for so2  

      data cpso2(1:100)/
     &  -0.93120,  -0.81010,  -0.57290,  -0.35900,  -0.20160, 
     &  -0.09710,  -0.03330,   0.00480,   0.02280,   0.02140, 
     &  -0.00440,  -0.05670,  -0.13340,  -0.23150,  -0.34510, 
     &  -0.47410,  -0.61980,  -0.78540,  -0.97640,  -1.19220, 
     &  -1.43260,  -1.69510,  -1.96870,  -2.27880,  -2.60340, 
     &  -2.93980,  -3.35510,  -3.77040,  -4.18570,  -4.60100, 
     &  -5.01630,  -5.43160,  -5.84690,  -6.26220,  -6.67750, 
     &  -7.09280,  -7.50810,  -7.92340,  -8.00000,  -7.42090, 
     &  -6.69940,  -5.97780,  -5.25630,  -4.42480,  -3.73690, 
     &  -3.09170,  -2.52000,  -2.03030,  -1.63070,  -1.30560, 
     &  -1.03730,  -0.81890,  -0.63950,  -0.48800,  -0.35740, 
     &  -0.23690,  -0.12370,  -0.02610,   0.02500,   0.01860, 
     &  -0.01940,  -0.06590,  -0.06380,  -0.00650,   0.04680, 
     &   0.06820,   0.03550,  -0.04310,  -0.13340,  -0.21750, 
     &  -0.29540,  -0.37380,  -0.45880,  -0.55710,  -0.67290, 
     &  -0.81310,  -0.98050,  -1.18310,  -1.43340,  -1.73540, 
     &  -2.10650,  -2.57050,  -3.12380,  -3.76910,  -4.57930, 
     &  -5.70120,  -6.56030,  -7.41950,  -8.00000,  -7.93020, 
     &  -7.65630,  -7.38240,  -7.10850,  -6.83460,  -6.56070, 
     &  -6.28680,  -6.01290,  -5.73900,  -5.46510,  -5.19120/
      data cpso2(101:200)/
     &  -4.91730,  -4.64340,  -4.36950,  -4.09560,  -3.82170, 
     &  -3.54780,  -3.27390,  -3.00000,  -2.72610,  -2.45220, 
     &  -2.17830,  -1.93170,  -1.70730,  -1.50040,  -1.31360, 
     &  -1.14440,  -0.99010,  -0.85050,  -0.72380,  -0.60830, 
     &  -0.50250,  -0.40160,  -0.30470,  -0.21120,  -0.12630, 
     &  -0.06560,  -0.04140,  -0.05090,  -0.07310,  -0.08020, 
     &  -0.04830,   0.00320,   0.03390,   0.02490,  -0.02960, 
     &  -0.11700,  -0.21410,  -0.30690,  -0.39680,  -0.48810, 
     &  -0.58810,  -0.70190,  -0.82990,  -0.97290,  -1.13050, 
     &  -1.30360,  -1.49240,  -1.70000,  -1.93060,  -2.19060, 
     &  -2.49590,  -2.86130,  -3.31760,  -3.92360,  -4.68470, 
     &  -5.25610,  -4.70820,  -4.11100,  -3.65820,  -3.19630, 
     &  -2.70630,  -1.96430,  -1.30890,  -0.68560,  -0.04120, 
     &   0.36780,   0.67120,   0.90310,   1.05770,   1.11450, 
     &   1.12720,   1.13000,   1.12370,   1.14590,   1.10470, 
     &   0.96170,   0.71070,   0.32540,  -0.23220,  -1.06120, 
     &  -1.77150,  -2.60890,  -3.02250,  -3.35420,  -3.73390, 
     &  -4.19860,  -4.78520,  -5.63900,  -6.27400,  -6.90910, 
     &  -7.54410,  -8.00000,  -8.00000,  -7.56980,  -6.88150, 
     &  -6.19330,  -5.35300,  -4.86020,  -4.12860,  -2.99220/
      data cpso2(201:226)/
     &  -2.35250,  -1.89050,  -1.51780,  -1.22950,  -1.00820, 
     &  -0.84840,  -0.76340,  -0.73400,  -0.72030,  -0.71670, 
     &  -0.70970,  -0.72970,  -0.83910,  -1.04720,  -1.36070, 
     &  -1.77200,  -2.29570,  -3.05660,  -4.10730,  -4.53370, 
     &  -4.94810,  -5.45420,  -6.24450,  -6.81480,  -7.38500, 
     &  -7.95530/

c     wavenumber-low and wavenumber-high specify a band region
c     for a molecular absorber.  -999 indicates the end of
c     absorption bands for any specific absorber.

c --- wavelength index limits for h2o 

      data iwlh2o/
     &      0,    350,   1005,   1645,   2535,   3425,   4315, 
     &   6155,   8005,   9620,  11545,  13075,  14865,  16340, 
     &   -999/
      data iwhh2o/
     &    345,   1000,   1640,   2530,   3420,   4310,   6150, 
     &   8000,   9615,  11540,  13070,  14860,  16045,  17860, 
     &   -999/

c --- wavelength index limits for o3  

      data iwlo3/
     &      0,    515,   1630,   2670,   2850,   -999/
      data iwho3/
     &    200,   1275,   2295,   2845,   3260,   -999/

c --- wavelength index limits for co2 

      data iwlco2/
     &    425,    840,   1805,   3070,   3760,   4530,   5905, 
     &   7395,   8030,   9340,   -999/
      data iwhco2/
     &    835,   1440,   2855,   3755,   4065,   5380,   7025, 
     &   7785,   8335,   9670,   -999/

c --- wavelength index limits for co  

      data iwlco/
     &      0,   1940,   4040,   -999/
      data iwhco/
     &    175,   2285,   4370,   -999/

c --- wavelength index limits for ch4 

      data iwlch4/
     &   1065,   2345,   4110,   5865,   -999/
      data iwhch4/
     &   1775,   3230,   4690,   6135,   -999/

c --- wavelength index limits for n2o 

      data iwln2o/
     &      0,    490,    865,   1065,   1545,   2090,   2705, 
     &   3245,   4260,   4540,   4910,   -999/
      data iwhn2o/
     &    120,    775,    995,   1385,   2040,   2655,   2865, 
     &   3925,   4470,   4785,   5165,   -999/

c --- wavelength index limits for o2  

      data iwlo2/
     &      0,   7650,   9235,  12850,  14300,  15695,   -999/
      data iwho2/
     &    265,   8080,   9490,  13220,  14600,  15955,   -999/

c --- wavelength index limits for nh3 

      data iwlnh3/
     &      0,    390,   -999/
      data iwhnh3/
     &    385,   2150,   -999/

c --- wavelength index limits for no  

      data iwlno/
     &   1700,   -999/
      data iwhno/
     &   2005,   -999/

c --- wavelength index limits for no2 

      data iwlno2/
     &    580,   1515,   2800,   -999/
      data iwhno2/
     &    925,   1695,   2970,   -999/

c --- wavelength index limits for so2 

      data iwlso2/
     &      0,    400,    950,   2415,   -999/
      data iwhso2/
     &    185,    650,   1460,   2580,   -999/

      end module gasblk
c=======================================================================
      subroutine kdistr(nz,mxly,mxq,uu,dtauk,tauk,wtk,twgp)

c   PURPOSE: compute 3 component k-distribution optical depths
c            and weights.  The parameters in /aabbcc/
c            are set up by routine TAUGAS, which should be called
c            before KDISTR at each new wavelength.
c
c   input parameters:
c   ----------------
c     nz       number of active layers
c     mxly     dimensioned size of first subscript of dtauk,tauk,wtk,twgp
c     mxq      dimensioned size of first subscript of uu
c     uu       absorption integrals from ABSINT 
c
c   output parameters:
c   -----------------
c   dtauk(nz,3)   molecular optical thickness of layer n, term k 
c   tauk(nz,3)    cumulative optical depth upto layer n, term k
c   wtk(nz,3)     layer by layer weighting for k-distribution fit
c     
c
c   internal working array (dimensioned in calling routine)
c   -------------------------------------------------------
c   twgp(nz,3)  - sum of optical depth * probability by molecule
c
c   new varables          11 molecules  by joseph h pierluissi
c   dpwj  probability for each molecule  fit double exponential
c   gkwj  band dependent  scaling of densitys to get k  amount
c   cps is the stored values of Pierluissi band model coefficients
c   cp1s = 10**cps
c   ibnd maps bands to molecules
c   wtk  is the  effective probability by layer
c   dtaum is defined as the sum of the optical depths by molecule

      use params, only: kr
      use gasblk, only: bma,bmb,bmc,bms,cps,ibnd
      implicit none

      integer, parameter :: nk=3

      integer :: mxly, mxq, m, k, mol, iw, n, nz, nm, ib

      real(kr) ::  dtauk(mxly,nk), tauk(mxly,nk), wtk(mxly,nk),
     &     twgp(mxly,nk), fac(nk),gkwj(nk,11),dpwj(nk,11),cp1s(11),
     &     taudb(11), uu(mxq,mxly), dpc, taukk, duu, wpth, smwtk

      data fac /1.0,0.09,0.015/

cccc

      dpc = 1./3.
      
      do 20 m = 1,11
        cp1s(m)= 10.**cps(m)
        taudb(m)=0.
 20   continue

      do  40   k = 1,nk
        do  30  mol = 1,11
          iw = ibnd(mol)
          gkwj(k,mol) = 0.
          dpwj(k,mol) = 0.
          if(iw .gt. 0) then
            gkwj(k,mol) = fac(k) * bmc(mol)
            if(k .eq. 1) dpwj(k,mol)=bma(mol)
            if(k .eq. 2) dpwj(k,mol)=bmb(mol)
            if(k .eq. 3) dpwj(k,mol)=1.-bma(mol)-bmb(mol)
          endif
 30     continue
 40   continue
      
c   evaluate the weighted k distribution quantities for
c   water vapor, ozone and uniformly mixed gases

      taukk=0.
      do 80 k=1, nk
        do 70 n=1,nz
          tauk(n,k)=0.
          dtauk(n,k)= 0.
          nm=nz-n+1

          twgp(n,k)=0.
          do 60 mol=1,11
            ib = ibnd(mol)
            if(ib.lt.0) go to 60
            if(nm.eq.nz) then
              duu=uu(ib,nm)
            else
              duu=uu(ib,nm)-uu(ib,nm+1)
            endif
            wpth = duu*gkwj(k,mol)
            dtauk(n,k)=dtauk(n,k)+wpth*cp1s(mol)
            twgp(n,k)=twgp(n,k)+wpth*cp1s(mol)*dpwj(k,mol)
 60       continue
          
          wtk(n,k) = dpc
          if(dtauk(n,k).ne.0) wtk(n,k)=twgp(n,k)/dtauk(n,k)
          
c     effective probability by layer, wtk is based on 
c     molecular probability weighted by optical depth
          
 70     continue
 80   continue
      
      do 100 n=1,nz
        smwtk = wtk(n,1) + wtk(n,2) + wtk(n,3)
        if(smwtk .eq. .0) write(8, *) 'smwtk == 0'
        do 90 k=1, nk
          wtk(n,k) = wtk(n,k)/smwtk
          if(n.eq.1) then
            tauk(n,k) = dtauk(n,k)
          else
            tauk(n,k) = dtauk(n,k)+tauk(n-1,k)
          endif
 90     continue
 100  continue
      
      return
      end

c=======================================================================

      subroutine absint(uu,nz,z,p,t,wh,wo,idb)
c
c     calculates absorber integrals from top to bottom of atmosphere
c
c    input:  nz         number of atmospheric layers
c            z          layer altitudes
c            p          pressure in mb
c            t          temperature in kelvin 
c            wh         water vapor mass density in g/m3
c            wo         o3 mass density in g/m3
c            idb        if non-zero, write diagnostic print out 
c
c    output: uu         absorption integrals
c
c
c     k      absorption process
c    ----    --------------------------------------
c     1-2    temp dep o2  1395 to 1760
c     3      o2*o2 and o2*n2 continuum (solomon et al 1998) 
c     4      n2 continuum
c     5      h2o self broadened continuum at 296 k
c     6      molecular (rayliegh) scattering
c     7      not used
c     8      ultraviolet and visible ozone
c     9      h2o self continuum 296-260  (tx(9) total trans)
c    10      h2o foriegn continuum
c    11      hno3 (nitric acid)
c    12-16   not used
c    17-30   h2o (line absorption) (double exponential models)
c    31-35   o3  (line absorption) (double exponential models)
c    36-43   co2 (line absorption) (double exponential models)
c    44-45   co  (line absorption) (double exponential models)
c    46      ch4 (line absorption) (double exponential models)
c    47      n2o (line absorption) (double exponential models)
c    50-51   o2  (line absorption) (double exponential models)
c    52-53   nh3 (line absorption) (double exponential models)
c    54      no  (line absorption) (double exponential models)
c    55      no2 (line absorption) (double exponential models)
c    56-57   so2 (line absorption) (double exponential models)
c    58      o2  hertzberg continuum (uv)
c    59-60   temp dep o3
c    61-62   not used
c    63      o2  1395 to 1760

      use params, only: mxly,mxq,one,zero,pzero,tzero,alosch,pmo,kr

      implicit none      

      real(kr), parameter :: 
     &                   xlosch=alosch*1.e5, 
     &                   conjoe=0.1/alosch,  
     &                   con=3.3429e21,      
     &                   rhzero=tzero/296.0  

c conjoe = 1.e5(cm/km) * 1.e-6(ppmv) / alosch
c con    = 0.1/molecular_mass_of_h2o 
c          con/xlosch = conversion from g/m3 => density in loschmidts



      real(kr) ::  z(mxly), p(mxly), t(mxly), wh(mxly), wo(mxly),
     &     uu(mxq,mxly), dd(mxq,mxly), tt, vfo3, vfn2, vfo2, vfco2,
     &     vfch4, vfn2o, vfco, vfno2, vfso2, vfnh3, vfno, vfhno3, pp,
     &     pss, tss, f1, f2, wair, rhoair, rhoh2o, rhofrn, wo2d, conh2o,
     &     cono3, conco2, conco, conch4, conn2o, conno2, connh3, conno,
     &     conso2, cono2, scfac, airwt, sclht, ztop, ptop, ttop, dz,
     &     tbar, dp, drho, den1, uutop, den2, denmin, denave, du, 
     &     du5=0, du8=0, tfac
      
      integer :: i, nz, idb, j, k
      logical :: ozn
cccc

      uu=0.

      if(maxval(p).eq.0.) return

      do 100 i=1,nz
        call volmix(z(i),vfn2,vfo2,vfco2,vfch4,vfn2o,vfco,
     &       vfno2,vfso2,vfnh3,vfno,vfhno3)
        tt=t(i)
        pp=p(i)
        pss=pp/pzero
        tss=tzero/tt
        f1=(pp/pzero)/(tt/tzero)
        f2=(pp/pzero)*sqrt(tzero/tt)
        wair=alosch*f1

        rhoair=f1
        rhoh2o=con*wh(i)/xlosch     
        rhofrn=rhoair-rhoh2o
        wo2d=conjoe*wair*vfo2*pss
        vfo3=wo(i)/(3*pmo*wair)
c
c                             rhoair - air density in loschmidts
c                             rhoh2o - water vapor density in loschmidts
c                             
c
c vfo3 = o3 volume fraction in parts per million
c
        dd(1,i)=wo2d*tt
        dd(2,i)=wo2d*(tt-220.)**2
        dd(3,i)=f1**2
        dd(4,i)=1.e-6*vfn2*f1*f2
        dd(5,i)=xlosch*rhoh2o**2/rhzero
        dd(6,i)=f1
        dd(8,i)=conjoe*wair*vfo3
        dd(10,i)=xlosch*rhoh2o*rhofrn/rhzero
        dd(11,i)=f1*vfhno3*(1.e-6*1.e5)
        dd(63,i)=wo2d
c
c for h2o
c    wh(i) is in g/m3,  conh2o is in g/cm2/km
c
        conh2o=wh(i)*.1
        dd(17,i)=conh2o*pss**0.9810*tss**( 0.3324) !     0-  345 cm-1 
        dd(18,i)=conh2o*pss**1.1406*tss**(-2.6343) !   350- 1000 cm-1 
        dd(19,i)=conh2o*pss**0.9834*tss**(-2.5294) !  1005- 1640 cm-1 
        dd(20,i)=conh2o*pss**1.0443*tss**(-2.4359) !  1645- 2530 cm-1 
        dd(21,i)=conh2o*pss**0.9681*tss**(-1.9537) !  2535- 3420 cm-1 
        dd(22,i)=conh2o*pss**0.9555*tss**(-1.5378) !  3425- 4310 cm-1 
        dd(23,i)=conh2o*pss**0.9362*tss**(-1.6338) !  4315- 6150 cm-1 
        dd(24,i)=conh2o*pss**0.9233*tss**(-0.9398) !  6155- 8000 cm-1 
        dd(25,i)=conh2o*pss**0.8658*tss**(-0.1034) !  8005- 9615 cm-1 
        dd(26,i)=conh2o*pss**0.8874*tss**(-0.2576) !  9620-11540 cm-1 
        dd(27,i)=conh2o*pss**0.7982*tss**( 0.0588) ! 11545-13070 cm-1 
        dd(28,i)=conh2o*pss**0.8088*tss**( 0.2816) ! 13075-14860 cm-1 
        dd(29,i)=conh2o*pss**0.6642*tss**( 0.2764) ! 14865-16045 cm-1 
        dd(30,i)=conh2o*pss**0.6656*tss**( 0.5061) ! 16340-17860 cm-1 
c
c for o3
c    cono3 in units of (1.e5 loschmidts)=(1 atm cm/km)=(2.6868e24 cm-3)
c
        cono3 = conjoe*wair*vfo3
        dd(31,i)=cono3 *pss**0.4200*tss**( 1.3909) !     0-  200 cm-1
        dd(32,i)=cono3 *pss**0.4221*tss**( 0.7678) !   515- 1275 cm-1
        dd(33,i)=cono3 *pss**0.3739*tss**( 0.1225) !  1630- 2295 cm-1
        dd(34,i)=cono3 *pss**0.1770*tss**( 0.9827) !  2670- 2845 cm-1
        dd(35,i)=cono3 *pss**0.3921*tss**( 0.1942) !  2850- 3260 cm-1
c
c for co2
c
        conco2= conjoe*wair*vfco2
        dd(36,i)=conco2*pss**0.6705*tss**(-2.2560)
        dd(37,i)=conco2*pss**0.7038*tss**(-5.0768)
        dd(38,i)=conco2*pss**0.7258*tss**(-1.6740)
        dd(39,i)=conco2*pss**0.6982*tss**(-1.8107)
        dd(40,i)=conco2*pss**0.8867*tss**(-0.5327)
        dd(41,i)=conco2*pss**0.7883*tss**(-1.3244)
        dd(42,i)=conco2*pss**0.6899*tss**(-0.8152)
        dd(43,i)=conco2*pss**0.6035*tss**( 0.6026)
c
c for co
c
        conco = conjoe*wair*vfco
        dd(44,i)=conco *pss**0.7589*tss**( 0.6911)
        dd(45,i)=conco *pss**0.9267*tss**( 0.1716)
c
c for ch4
c
        conch4= conjoe*wair*vfch4
        dd(46,i)=conch4*pss**0.7139*tss**(-0.4185)
c
c for n2o
c
        conn2o= conjoe*wair*vfn2o
        dd(47,i)=conn2o*pss**0.3783*tss**( 0.9399)
        dd(48,i)=conn2o*pss**0.7203*tss**(-0.1836)
        dd(49,i)=conn2o*pss**0.7764*tss**( 1.1931)
c
c for o2
c
        cono2 = conjoe*wair*vfo2
        dd(50,i)=cono2 *pss**1.1879*tss**( 2.9738) !     0-  265 cm-1
        dd(51,i)=cono2 *pss**0.9353*tss**( 0.1936) !  7650-52710 cm-1
c
c for nh3
c
        connh3= conjoe*wair*vfnh3
        dd(52,i)=connh3*pss**0.8023*tss**(-0.9111)
        dd(53,i)=connh3*pss**0.6968*tss**( 0.3377)
c
c for no
c
        conno = conjoe*wair*vfno
        dd(54,i)=conno *pss**0.5265*tss**(-0.4702)
c
c for no2
c
        conno2= conjoe*wair*vfno2
        dd(55,i)=conno2*pss**0.3956*tss**(-0.0545)
c
c for so2
c
        conso2= conjoe*wair*vfso2
        dd(56,i)=conso2*pss**0.2943*tss**( 1.2316)
        dd(57,i)=conso2*pss**0.2135*tss**( 0.0733)
c
c   herzberg continuum pressure dependence calculation, shardanand 1977
c      and   yoshino et al 1988
c
        dd(58,i)=(1.+.83*f1)*cono2
c
        if(idb.ne.0) then
          if(i.eq.1) print 800
          print '(i4,0pf9.2,f9.3,f7.1,1x,11es9.2)',
     &       i,z(i),p(i),t(i), dd(17,i),dd(31,i),
     &       dd(36,i),dd(44,i),dd(46,i),dd(47,i),
     &       dd(50,i),dd(52,i),dd(54,i),dd(55,i),dd(56,i)
        endif

 100  continue
c
c absorption integrals
c
      do 120 j=1,nz
        do 110 k=1,mxq
          uu(k,j)=0.
 110    continue
 120  continue
c
c use scale height to get first step of integration
c
      scfac=exp(-1.)
      airwt=(2.66e-23*(vfo2+1.e6-vfn2)+2.24e-23*vfn2)*1.e-6
      sclht=1.e-5*1.38e-16*t(nz)/(airwt*980.67)
c     
      do 200 i=nz,1,-1
        if(i.eq.nz) then
          ztop=2*z(i)-z(i-1)
          ptop=p(i)**2/p(i-1)
          ttop=t(i)
        else
          ztop=z(i+1)
          ptop=p(i+1)
          ttop=t(i+1)
        endif
        dz=ztop-z(i)
        if(p(i).eq.ptop) then 
          tbar=.5*(ttop+t(i))
        else 
          dp=(p(i)-ptop)/log(p(i)/ptop)
          drho=(p(i)/t(i)-ptop/ttop)/log(p(i)*ttop/(ptop*t(i)))
          tbar=dp/drho
        endif
        do 150 k=1,mxq
          den1=dd(k,i)
          if(i.eq.nz) then
            uutop=0.
            den2=dd(k,i)*scfac
          else
            uutop=uu(k,i+1)
            den2=dd(k,i+1)
          endif
          ozn=.false.
          if(k.eq.8) ozn=.true.
          if(k.ge.31.and.k.le.35) ozn=.true.
          if(k.ge.59.and.k.le.60) ozn=.true.
          denmin=min(den1,den2)
          denave=.5*(den1+den2)
          if(denmin.gt.0..and.denmin.lt.0.999*denave.and..not.ozn) then
            du=dz*(den1-den2)/log(den1/den2)
          else
            du=dz*denave
          endif
          if(k.eq.5) du5=du
          if(k.eq.8) du8=du
          if(k.eq.9) then
            tfac=(296.-tbar)/(296.-260.)
            tfac=max(zero,min(one,tfac))
            uu(9,i)=uutop+du5*tfac
          elseif(k.eq.59) then
            uu(59,i)=uutop+.269*du8*(tbar-273.15)
          elseif(k.eq.60) then
            uu(60,i)=uutop+.269*du8*(tbar-273.15)**2
          else
            uu(k,i)=uutop+du
          endif
 150    continue
        if(idb.ne.0) then
          if(i.eq.nz) print 900
          print '(i4,0pf9.2,f9.3,f7.1,1x,11es9.2)',
     &       i,z(i),p(i),t(i), uu(17,i),uu(31,i),
     &       uu(36,i),uu(44,i),uu(46,i),uu(47,i),
     &       uu(50,i),uu(52,i),uu(54,i),uu(55,i),uu(56,i)
        endif

 200  continue
c
 800  format('1',/,'  atmospheric profiles',//,
     + '  (if a molecule has more than one band, then the data for the',
     + ' first band are shown.)'//
     1 '   i      z       p       t      h2o      o3       co2',
     2 '      co       ch4      n2o      o2       nh3      no ',
     3 '      no2      so2'/
     4 '         (km)    (mb)    (k) g/cm**2/km  (            ',
     5 '          atm cm/km                                   ',
     6 '                  )')
 900  format('1',/,'  absorption integrals',//,
     + '  (if a molecule has more than one band, then the data for the',
     + ' first band are shown.)'//
     1 '   i      z       p       t      h2o      o3       co2',
     2 '      co       ch4      n2o      o2       nh3      no ',
     3 '      no2      so2'/
     4 '         (km)    (mb)    (k) g/cm**2     (            ',
     5 '          atm cm                                      ',
     6 '                  )')

      return
      end
c=======================================================================

      subroutine taugas(wl,uu,amu0,nz,z,dtauc,dtaul,idb)
c
c     calculates optical depth due to atmospheric gases and aerosols
c
c  input:  wl        wavelength in microns
c          uu        path integral uu(mxq,nxlayr) computed by ABSINT
c          amu0      cosine of solar zenith angle (at surface)
c          nz        number of atmospheric layers
c                    nz=1  => integration to bottom level only
c          z         heights of atmospheric layers
c          idb       if non-zero print optical depth profiles
c
c  output: 
c          dtauc(k)  the optical depth of a single layer, k, due to 
c                    continuum processes (follows Beer's law).
c
c          dtaul(k)  the negative log of the mean transmission through
c                    a single layer k, due to line absorption (no
c                    continuum processes).  BE AWARE OF THIS: the
c                    negative log of the transmission through a given
c                    layer depends on the column density of absorbers 
c                    above that layer.
c
c          NOTE: the order of the depth layers are reversed in this routine,
c                uu(k,1) will be used to compute the optical depth dtauc(nz)
c                    
c
c                    continuum and line absorption due to 
c
c                    1. uniformly mixed atmospheric gases (co2,n2,o2...), 
c                    2. h2o
c                    3. ozone
c                    4. trace gases.
c
c          gasabs    gas absorption diagnostics at surface, gasabs(9),
c                    due to both line and continuum processes
c                   1    2     3   4    5   6     7    8      9  
c                  h2o  h2o_c co2  o3  n2o  co  ch4  o2+n2  trace
c     
      use params, only: mxq, mxly, kr, pzero, re, grav
      use gasblk, only: cps, iwlh2o, iwhh2o, iwlco2, iwhco2,
     &     iwlo3, iwho3, iwln2o, iwhn2o, iwlco, iwhco, iwlch4, iwhch4,
     &     iwlo2, iwho2, iwlno, iwhno, iwlso2, iwhso2, iwlno2, iwhno2,
     &     iwlnh3, iwhnh3, cph2o, cpco2, cpo3, cpn2o, cpco, cpch4,
     &     cpo2, cpno, cpso2, cpno2, cpnh3, ibnd, bms
      implicit none

      real(kr), parameter ::
     &     bigexp=87.,
     &     airrho=1.273,        ! air density at stp (kg m-3)
     &     awlmax=20.,          ! maximum exponent
     &     amfac=10.*grav*airrho/pzero ! convert scale height to airmass

      integer :: i, nz, indh2o, indo3, indco2, indco, indch4, indn2o,
     &     indo2, indnh3, indno, indno2, indso2, iv, im, k, ib, idb

      real(kr) ::  uu(mxq,nz),w(mxq),dtauc(*),dtaul(*),
     &     tauc(mxly),taul(mxly),z(*), tau(11), amu0, 
     &     wl, v, sh2ot1, fh2o, t0, t1, alpha2, xh2o, vtemp, xd,
     &     radfn0, radfn1, wfac, ya, yb, fdg, abn2, abno3, abo2,
     &     sigo20, sigo2a, sigo2b, sigo4, doz1, doz2, doz3, c0,
     &     ct1, ct2, abo3a, sh2ot0, rayla, zim, zi, zbar, tcunif,
     &     tch2o, tco3, tctrc, cp, awl, gasabs(9)

      real(kr), external :: raysig
      logical, save :: first=.true.

      data indh2o,indo3,indco2,indco,indch4/5*1./
      data indn2o,indo2,indnh3,indno,indno2,indso2/6*1./
      


cccc
      do 10 i=1,nz
        tauc(i)=0.
        taul(i)=0.
 10   continue

c convert from wavelength in microns to wavenumber in cm-1

      iv=5*(int(10000.0/wl)/5)
      v=float(iv)
      v=10000./wl

c water continuum (v < 10000 cm-1)
c slf296 loads self-broadened water vapor continuum at 296k
c slf260 loads self-broadened water vapor continuum at 296k
c frn296 loads foreign-broadened water vapor continuum at 296k

      call slf296(v,sh2ot0)
      call slf260(v,sh2ot1)
      call frn296(v,fh2o)
      t0=296.
      t1=260.
      if(sh2ot0.gt.0.) then
        alpha2=200.**2
        xh2o=(1.-0.2333*(alpha2/((v-1050.)**2+alpha2)) )
        sh2ot0=sh2ot0*xh2o
        sh2ot1=sh2ot1*xh2o
      endif

c protect against exponential underflow at high frequency

      vtemp=v/0.6952
      if(vtemp/t1.le.bigexp) then
        xd=exp(-v/(t0*0.6952))
        radfn0=v*(1.-xd)/(1.+xd)
        xd=exp(-v/(t1*0.6952))
        radfn1=v*(1.-xd)/(1.+xd)
      else
        radfn0=v
        radfn1=v
      endif

      wfac=1.e-20
      ya=exp(-log(1.025*3.159e-8)+(2.75e-4)*v)
      yb=exp(-log(8.97e-6)+(1.300e-3)*v)
      fdg=1./(ya+yb)

c  n2 continuum absorption coefficient (2080 < v < 2740)

      call c4dta(abn2,v)

c hno3 absorption calculation (850 < v < 1735)

      call hno3(v,abno3)

c hertda computes hertzburg uv o2 absorption ( v > 36000 )  
c o2cont computes o2 continuum ( 1395 < v < 1760 )

      call hertda(abo2,v)
      call o2cont(v,sigo20,sigo2a,sigo2b)

c o2 * o2 collision-radiative process 

      call o4cont(wl,sigo4)

c diffuse ozone

c o3uv computes uv ozone for 40800-54054cm-1 (185-245nm)

c o3hht ozone hartley band for 24370-40800 cm-1 (245-410 nm)
c   temperature dependent coefficient

c c8dta ozone chappius band for 13000-24200 cm-1 (413-769nm)


      doz1=0.
      doz2=0.
      doz3=0.
      if(v.gt.40800) then       ! wl < 0.245um
        call o3uv(v,c0)
        doz1=.269*c0
      elseif(v.gt.24370) then   ! wl < 0.410
        call o3hht(v,c0,ct1,ct2)
        doz1=.269*c0
        doz2=c0*ct1
        doz3=c0*ct2
      elseif(v.ge.13000. .and. v.le.24200) then ! 0.413 < wl < 0.76923
        call c8dta(abo3a,v)
        doz1=abo3a
      endif

c rayleigh scattering (included here for print out only)

      rayla=raysig(v)

c cxdta locates coefficient for double exponential

      call cxdta(cps(1),v,iwlh2o,iwhh2o,cph2o,indh2o)
      call cxdta(cps(2),v,iwlco2,iwhco2,cpco2,indco2)
      call cxdta(cps(3),v,iwlo3, iwho3, cpo3, indo3 )
      call cxdta(cps(4),v,iwln2o,iwhn2o,cpn2o,indn2o)
      call cxdta(cps(5),v,iwlco, iwhco, cpco, indco )
      call cxdta(cps(6),v,iwlch4,iwhch4,cpch4,indch4)
      call cxdta(cps(7),v,iwlo2, iwho2, cpo2, indo2 )
      call cxdta(cps(8),v,iwlno, iwhno, cpno, indno )
      call cxdta(cps(9),v,iwlso2,iwhso2,cpso2,indso2)
      call cxdta(cps(10),v,iwlno2,iwhno2,cpno2,indno2)
      call cxdta(cps(11),v,iwlnh3,iwhnh3,cpnh3,indnh3)

c  abcdta moves double exponential coefficients to new bm? arrays

      call abcdta(iv)

c schrun computes uv o2 schumann-runge band model parameters

      if(v.gt.49600) call schrun(v,cps(7))

      zim=z(nz)
      do 40 i=nz,1,-1
        im=nz-i+1
        zi=z(i)
        zbar=0.5*(zi+zim)
        zim=zi
        do 20 k=1,mxq
          if (i .eq. nz) then
            w(k)=uu(k,i)/amuz(zi)
          else
            w(k)=w(k)+(uu(k,i)-uu(k,i+1))/amuz(zbar)
          endif
 20     continue

        tcunif=+sigo4*w(3)
     &         +abn2*w(4)
     &         +sigo20*(w(63)
     &         +sigo2a*(w(1)-220*w(63))
     &         +sigo2b*w(2))
     &         +abo2*w(58)

        tch2o=sh2ot0*radfn0*(wfac*w(5))
     &        +((sh2ot1*radfn1)-(sh2ot0*radfn0))*(wfac*w(9))
     &        +(fh2o+fdg)*radfn0*(wfac*w(10))

        tco3=doz1*w(8)+doz2*w(59)+doz3*w(60)
        tctrc=abno3*w(11)

c  compute transmitance from double exponential band model for
c       1    2    3    4    5    6    7    8    9   10   11
c      h2o, co2,  o3, n2o,  co, ch4,  o2,  no, so2, no2, nh3

        tauc(im)=tcunif+tch2o+tco3+tctrc

c        write(*,'(a/(10i11))') 'ibnd:',(ibnd(k),k=1,11)
c        write(*,'(a/(10es11.3))') 'cps:',(cps(k),k=1,11)
c        write(*,'(a/(10es11.3))') 'w:',(w(ibnd(k)),k=1,11)

        do k=1,11
          ib=ibnd(k)
          if(ib.gt.0) then 
            cp=cps(k)
            if(cp.gt.-awlmax .and. w(ib).gt.1.e-20) then
              awl=bms(k)*(cp+log10(w(ib)))
              awl=min(awl,awlmax)              
              tau(k)=10.**awl
            else
              tau(k)=0.
            endif
            taul(im)=taul(im)+tau(k)
          endif
        enddo

c  gas absorption diagnostics in gasabs(9)
c    1    2      3   4    5   6     7      8    9   
c   h2o  h2o_c  co2  o3  n2o  co  ch4  o2+n2  trace 

        gasabs(1)=tau(1)
        gasabs(2)=tch2o
        gasabs(3)=tau(2)
        gasabs(4)=tau(3)+tco3
        gasabs(5)=tau(4)            
        gasabs(6)=tau(5)         
        gasabs(7)=tau(6)
        gasabs(8)=tau(7)+tcunif
        gasabs(9)=tau(8)+tau(9)+tau(10)+tau(11)

        if(idb.ge.2) then
          if(i.eq.nz) then
            print '(/,a,f11.4,5x,a,f8.2)','wl=',wl,'wavenumber=',v
            print '(3x,11a11)','z','h2o','h2o_c','co2','o3',
     &           'n2o','co','ch4','o2+n2','trace','total'
          endif            
          print '(i3,11es11.3)', i,z(i),gasabs(1:9),sum(gasabs(1:9))
        elseif(idb.eq.1.and.i.eq.1) then
c         wl h2o co2 o3 n2o co ch4 o2+n2 trace total
          print '(f11.4,9es11.3)',
     &      wl,gasabs(1)+gasabs(2),gasabs(3:9),sum(gasabs(1:9))
        endif
 40   continue
      
      dtauc(1)=tauc(1)
      dtaul(1)=taul(1)
 9876 format(/a,4(/10es11.3))
c      write(*,*) 'taugasx    amu0=',amu0
c      write(*,9876) 'tauc+taul',(tauc(i)+taul(i),i=1,nz)

      do 50 i=2,nz
        dtauc(i)=tauc(i)-tauc(i-1)
        dtaul(i)=taul(i)-taul(i-1)
 50   continue
c      write(*,9876) 'dtauc+dtaul',(dtauc(i)+dtaul(i),i=1,nz)
      return

      contains

c non-spherical earth correction to the cosine of the solar zenith angle
c Draw a line extending from a point on the earths surface out to space
c if amu0 is the solar zenith angle at the surface then amuz(zzz) is the
c angle between the line and the local zenith direction at altitude zzz. 

c statement function:

      real(kr) function amuz(zzz)
      real(kr) zzz
      amuz = sqrt(1.-(1.-amu0**2)*(re/(re+zzz))**2)
      return
      end function

      end
c=======================================================================

c
      subroutine slf296(v,sh2ot0)

c     water vapor self continuum at 296K

      use params, only: kr
      implicit none
      integer, parameter :: npts=2003
      real(kr) :: v, sh2ot0, v1=-20, v2=20000.0, dv=10., s(npts)


      ! units of (cm**3/mol) * 1.e-20

      data s(1:100)/
     &  1.1109e-01 ,1.0573e-01, 1.0162e-01, 1.0573e-01, 1.1109e-01,
     &  1.2574e-01, 1.3499e-01, 1.4327e-01, 1.5065e-01, 1.5164e-01,
     &  1.5022e-01, 1.3677e-01, 1.3115e-01, 1.2253e-01, 1.1271e-01,
     &  1.0070e-01, 8.7495e-02, 8.0118e-02, 6.9940e-02, 6.2034e-02,
     &  5.6051e-02, 4.7663e-02, 4.2450e-02, 3.6690e-02, 3.3441e-02,
     &  3.0711e-02, 2.5205e-02, 2.2113e-02, 1.8880e-02, 1.6653e-02,
     &  1.4626e-02, 1.2065e-02, 1.0709e-02, 9.1783e-03, 7.7274e-03,
     &  6.7302e-03, 5.6164e-03, 4.9089e-03, 4.1497e-03, 3.5823e-03,
     &  3.1124e-03, 2.6414e-03, 2.3167e-03, 2.0156e-03, 1.7829e-03,
     &  1.5666e-03, 1.3928e-03, 1.2338e-03, 1.0932e-03, 9.7939e-04,
     &  8.8241e-04, 7.9173e-04, 7.1296e-04, 6.4179e-04, 5.8031e-04,
     &  5.2647e-04, 4.7762e-04, 4.3349e-04, 3.9355e-04, 3.5887e-04,
     &  3.2723e-04, 2.9919e-04, 2.7363e-04, 2.5013e-04, 2.2876e-04,
     &  2.0924e-04, 1.9193e-04, 1.7618e-04, 1.6188e-04, 1.4891e-04,
     &  1.3717e-04, 1.2647e-04, 1.1671e-04, 1.0786e-04, 9.9785e-05,
     &  9.2350e-05, 8.5539e-05, 7.9377e-05, 7.3781e-05, 6.8677e-05,
     &  6.3993e-05, 5.9705e-05, 5.5788e-05, 5.2196e-05, 4.8899e-05,
     &  4.5865e-05, 4.3079e-05, 4.0526e-05, 3.8182e-05, 3.6025e-05,
     &  3.4038e-05, 3.2203e-05, 3.0511e-05, 2.8949e-05, 2.7505e-05,
     &  2.6170e-05, 2.4933e-05, 2.3786e-05, 2.2722e-05, 2.1736e-05/
      data s(101:200)/
     &  2.0819e-05, 1.9968e-05, 1.9178e-05, 1.8442e-05, 1.7760e-05,
     &  1.7127e-05, 1.6541e-05, 1.5997e-05, 1.5495e-05, 1.5034e-05,
     &  1.4614e-05, 1.4230e-05, 1.3883e-05, 1.3578e-05, 1.3304e-05,
     &  1.3069e-05, 1.2876e-05, 1.2732e-05, 1.2626e-05, 1.2556e-05,
     &  1.2544e-05, 1.2604e-05, 1.2719e-05, 1.2883e-05, 1.3164e-05,
     &  1.3581e-05, 1.4187e-05, 1.4866e-05, 1.5669e-05, 1.6717e-05,
     &  1.8148e-05, 2.0268e-05, 2.2456e-05, 2.5582e-05, 2.9183e-05,
     &  3.3612e-05, 3.9996e-05, 4.6829e-05, 5.5055e-05, 6.5897e-05,
     &  7.5360e-05, 8.7213e-05, 1.0046e-04, 1.1496e-04, 1.2943e-04,
     &  1.5049e-04, 1.6973e-04, 1.8711e-04, 2.0286e-04, 2.2823e-04,
     &  2.6780e-04, 2.8766e-04, 3.1164e-04, 3.3640e-04, 3.6884e-04,
     &  3.9159e-04, 3.8712e-04, 3.7433e-04, 3.4503e-04, 3.1003e-04,
     &  2.8027e-04, 2.5253e-04, 2.3408e-04, 2.2836e-04, 2.4442e-04,
     &  2.7521e-04, 2.9048e-04, 3.0489e-04, 3.2646e-04, 3.3880e-04,
     &  3.3492e-04, 3.0987e-04, 2.9482e-04, 2.8711e-04, 2.6068e-04,
     &  2.2683e-04, 1.9996e-04, 1.7788e-04, 1.6101e-04, 1.3911e-04,
     &  1.2013e-04, 1.0544e-04, 9.4224e-05, 8.1256e-05, 7.3667e-05,
     &  6.2233e-05, 5.5906e-05, 5.1619e-05, 4.5140e-05, 4.0273e-05,
     &  3.3268e-05, 3.0258e-05, 2.6440e-05, 2.3103e-05, 2.0749e-05,
     &  1.8258e-05, 1.6459e-05, 1.4097e-05, 1.2052e-05, 1.0759e-05/
      data s(201:300)/
     &  9.1400e-06, 8.1432e-06, 7.1460e-06, 6.4006e-06, 5.6995e-06,
     &  4.9372e-06, 4.4455e-06, 3.9033e-06, 3.4740e-06, 3.1269e-06,
     &  2.8059e-06, 2.5558e-06, 2.2919e-06, 2.0846e-06, 1.8983e-06,
     &  1.7329e-06, 1.5929e-06, 1.4631e-06, 1.3513e-06, 1.2461e-06,
     &  1.1519e-06, 1.0682e-06, 9.9256e-07, 9.2505e-07, 8.6367e-07,
     &  8.0857e-07, 7.5674e-07, 7.0934e-07, 6.6580e-07, 6.2580e-07,
     &  5.8853e-07, 5.5333e-07, 5.2143e-07, 4.9169e-07, 4.6431e-07,
     &  4.3898e-07, 4.1564e-07, 3.9405e-07, 3.7403e-07, 3.5544e-07,
     &  3.3819e-07, 3.2212e-07, 3.0714e-07, 2.9313e-07, 2.8003e-07,
     &  2.6777e-07, 2.5628e-07, 2.4551e-07, 2.3540e-07, 2.2591e-07,
     &  2.1701e-07, 2.0866e-07, 2.0082e-07, 1.9349e-07, 1.8665e-07,
     &  1.8027e-07, 1.7439e-07, 1.6894e-07, 1.6400e-07, 1.5953e-07,
     &  1.5557e-07, 1.5195e-07, 1.4888e-07, 1.4603e-07, 1.4337e-07,
     &  1.4093e-07, 1.3828e-07, 1.3569e-07, 1.3270e-07, 1.2984e-07,
     &  1.2714e-07, 1.2541e-07, 1.2399e-07, 1.2102e-07, 1.1878e-07,
     &  1.1728e-07, 1.1644e-07, 1.1491e-07, 1.1305e-07, 1.1235e-07,
     &  1.1228e-07, 1.1224e-07, 1.1191e-07, 1.1151e-07, 1.1098e-07,
     &  1.1068e-07, 1.1109e-07, 1.1213e-07, 1.1431e-07, 1.1826e-07,
     &  1.2322e-07, 1.3025e-07, 1.4066e-07, 1.5657e-07, 1.7214e-07,
     &  1.9449e-07, 2.2662e-07, 2.6953e-07, 3.1723e-07, 3.7028e-07/
      data s(301:400)/
     &  4.4482e-07, 5.3852e-07, 6.2639e-07, 7.2175e-07, 7.7626e-07,
     &  8.7248e-07, 9.6759e-07, 1.0102e-06, 1.0620e-06, 1.1201e-06,
     &  1.2107e-06, 1.2998e-06, 1.3130e-06, 1.2856e-06, 1.2350e-06,
     &  1.1489e-06, 1.0819e-06, 1.0120e-06, 9.4795e-07, 9.2858e-07,
     &  9.8060e-07, 1.0999e-06, 1.1967e-06, 1.2672e-06, 1.3418e-06,
     &  1.3864e-06, 1.4330e-06, 1.4592e-06, 1.4598e-06, 1.4774e-06,
     &  1.4726e-06, 1.4820e-06, 1.5050e-06, 1.4984e-06, 1.5181e-06,
     &  1.5888e-06, 1.6850e-06, 1.7690e-06, 1.9277e-06, 2.1107e-06,
     &  2.3068e-06, 2.5347e-06, 2.8069e-06, 3.1345e-06, 3.5822e-06,
     &  3.9051e-06, 4.3422e-06, 4.8704e-06, 5.5351e-06, 6.3454e-06,
     &  7.2690e-06, 8.2974e-06, 9.7609e-06, 1.1237e-05, 1.3187e-05,
     &  1.5548e-05, 1.8784e-05, 2.1694e-05, 2.5487e-05, 3.0092e-05,
     &  3.5385e-05, 4.2764e-05, 4.9313e-05, 5.5800e-05, 6.2968e-05,
     &  7.1060e-05, 7.7699e-05, 8.7216e-05, 8.9335e-05, 9.2151e-05,
     &  9.2779e-05, 9.4643e-05, 9.7978e-05, 1.0008e-04, 1.0702e-04,
     &  1.1026e-04, 1.0828e-04, 1.0550e-04, 1.0432e-04, 1.0428e-04,
     &  9.8980e-05, 9.4992e-05, 9.5159e-05, 1.0058e-04, 1.0738e-04,
     &  1.1550e-04, 1.1229e-04, 1.0596e-04, 1.0062e-04, 9.1742e-05,
     &  8.4492e-05, 6.8099e-05, 5.6295e-05, 4.6502e-05, 3.8071e-05,
     &  3.0721e-05, 2.3297e-05, 1.8688e-05, 1.4830e-05, 1.2049e-05/
      data s(401:500)/
     &  9.6754e-06, 7.9192e-06, 6.6673e-06, 5.6468e-06, 4.8904e-06,
     &  4.2289e-06, 3.6880e-06, 3.2396e-06, 2.8525e-06, 2.5363e-06,
     &  2.2431e-06, 1.9949e-06, 1.7931e-06, 1.6164e-06, 1.4431e-06,
     &  1.2997e-06, 1.1559e-06, 1.0404e-06, 9.4300e-07, 8.4597e-07,
     &  7.6133e-07, 6.8623e-07, 6.2137e-07, 5.6345e-07, 5.1076e-07,
     &  4.6246e-07, 4.1906e-07, 3.8063e-07, 3.4610e-07, 3.1554e-07,
     &  2.8795e-07, 2.6252e-07, 2.3967e-07, 2.1901e-07, 2.0052e-07,
     &  1.8384e-07, 1.6847e-07, 1.5459e-07, 1.4204e-07, 1.3068e-07,
     &  1.2036e-07, 1.1095e-07, 1.0237e-07, 9.4592e-08, 8.7530e-08,
     &  8.1121e-08, 7.5282e-08, 6.9985e-08, 6.5189e-08, 6.0874e-08,
     &  5.6989e-08, 5.3530e-08, 5.0418e-08, 4.7745e-08, 4.5367e-08,
     &  4.3253e-08, 4.1309e-08, 3.9695e-08, 3.8094e-08, 3.6482e-08,
     &  3.4897e-08, 3.3500e-08, 3.2302e-08, 3.0854e-08, 2.9698e-08,
     &  2.8567e-08, 2.7600e-08, 2.6746e-08, 2.5982e-08, 2.5510e-08,
     &  2.5121e-08, 2.4922e-08, 2.4909e-08, 2.5013e-08, 2.5216e-08,
     &  2.5589e-08, 2.6049e-08, 2.6451e-08, 2.6978e-08, 2.7687e-08,
     &  2.8600e-08, 2.9643e-08, 3.0701e-08, 3.2058e-08, 3.3695e-08,
     &  3.5558e-08, 3.7634e-08, 3.9875e-08, 4.2458e-08, 4.5480e-08,
     &  4.8858e-08, 5.2599e-08, 5.7030e-08, 6.2067e-08, 6.7911e-08,
     &  7.4579e-08, 8.1902e-08, 8.9978e-08, 9.9870e-08, 1.1102e-07/
      data s(501:600)/
     &  1.2343e-07, 1.3732e-07, 1.5394e-07, 1.7318e-07, 1.9383e-07,
     &  2.1819e-07, 2.4666e-07, 2.8109e-07, 3.2236e-07, 3.7760e-07,
     &  4.4417e-07, 5.2422e-07, 6.1941e-07, 7.4897e-07, 9.2041e-07,
     &  1.1574e-06, 1.4126e-06, 1.7197e-06, 2.1399e-06, 2.6266e-06,
     &  3.3424e-06, 3.8418e-06, 4.5140e-06, 5.0653e-06, 5.8485e-06,
     &  6.5856e-06, 6.8937e-06, 6.9121e-06, 6.9005e-06, 6.9861e-06,
     &  6.8200e-06, 6.6089e-06, 6.5809e-06, 7.3496e-06, 8.0311e-06,
     &  8.3186e-06, 8.4260e-06, 9.0644e-06, 9.4965e-06, 9.4909e-06,
     &  9.0160e-06, 9.1494e-06, 9.3629e-06, 9.5944e-06, 9.5459e-06,
     &  8.9919e-06, 8.6040e-06, 7.8613e-06, 7.1567e-06, 6.2677e-06,
     &  5.1899e-06, 4.4188e-06, 3.7167e-06, 3.0636e-06, 2.5573e-06,
     &  2.0317e-06, 1.6371e-06, 1.3257e-06, 1.0928e-06, 8.9986e-07,
     &  7.4653e-07, 6.1111e-07, 5.1395e-07, 4.3500e-07, 3.7584e-07,
     &  3.2633e-07, 2.8413e-07, 2.4723e-07, 2.1709e-07, 1.9294e-07,
     &  1.7258e-07, 1.5492e-07, 1.3820e-07, 1.2389e-07, 1.1189e-07,
     &  1.0046e-07, 9.0832e-08, 8.2764e-08, 7.4191e-08, 6.7085e-08,
     &  6.0708e-08, 5.4963e-08, 4.9851e-08, 4.5044e-08, 4.0916e-08,
     &  3.7220e-08, 3.3678e-08, 3.0663e-08, 2.7979e-08, 2.5495e-08,
     &  2.3286e-08, 2.1233e-08, 1.9409e-08, 1.7770e-08, 1.6260e-08,
     &  1.4885e-08, 1.3674e-08, 1.2543e-08, 1.1551e-08, 1.0655e-08/
      data s(601:700)/
     &  9.8585e-09, 9.1398e-09, 8.4806e-09, 7.8899e-09, 7.3547e-09,
     &  6.8670e-09, 6.4131e-09, 5.9930e-09, 5.6096e-09, 5.2592e-09,
     &  4.9352e-09, 4.6354e-09, 4.3722e-09, 4.1250e-09, 3.9081e-09,
     &  3.7118e-09, 3.5372e-09, 3.3862e-09, 3.2499e-09, 3.1324e-09,
     &  3.0313e-09, 2.9438e-09, 2.8686e-09, 2.8050e-09, 2.7545e-09,
     &  2.7149e-09, 2.6907e-09, 2.6724e-09, 2.6649e-09, 2.6642e-09,
     &  2.6725e-09, 2.6871e-09, 2.7056e-09, 2.7357e-09, 2.7781e-09,
     &  2.8358e-09, 2.9067e-09, 2.9952e-09, 3.1020e-09, 3.2253e-09,
     &  3.3647e-09, 3.5232e-09, 3.7037e-09, 3.9076e-09, 4.1385e-09,
     &  4.3927e-09, 4.6861e-09, 5.0238e-09, 5.4027e-09, 5.8303e-09,
     &  6.3208e-09, 6.8878e-09, 7.5419e-09, 8.3130e-09, 9.1952e-09,
     &  1.0228e-08, 1.1386e-08, 1.2792e-08, 1.4521e-08, 1.6437e-08,
     &  1.8674e-08, 2.1160e-08, 2.4506e-08, 2.8113e-08, 3.2636e-08,
     &  3.7355e-08, 4.2234e-08, 4.9282e-08, 5.7358e-08, 6.6743e-08,
     &  7.8821e-08, 9.4264e-08, 1.1542e-07, 1.3684e-07, 1.6337e-07,
     &  2.0056e-07, 2.3252e-07, 2.6127e-07, 2.9211e-07, 3.3804e-07,
     &  3.7397e-07, 3.8205e-07, 3.8810e-07, 3.9499e-07, 3.9508e-07,
     &  3.7652e-07, 3.5859e-07, 3.6198e-07, 3.7871e-07, 4.0925e-07,
     &  4.2717e-07, 4.8241e-07, 5.2008e-07, 5.6530e-07, 5.9531e-07,
     &  6.1994e-07, 6.5080e-07, 6.6355e-07, 6.9193e-07, 6.9930e-07/
      data s(701:800)/
     &  7.3058e-07, 7.4678e-07, 7.9193e-07, 8.3627e-07, 9.1267e-07,
     &  1.0021e-06, 1.1218e-06, 1.2899e-06, 1.4447e-06, 1.7268e-06,
     &  2.0025e-06, 2.3139e-06, 2.5599e-06, 2.8920e-06, 3.3059e-06,
     &  3.5425e-06, 3.9522e-06, 4.0551e-06, 4.2818e-06, 4.2892e-06,
     &  4.4210e-06, 4.5614e-06, 4.6739e-06, 4.9482e-06, 5.1118e-06,
     &  5.0986e-06, 4.9417e-06, 4.9022e-06, 4.8449e-06, 4.8694e-06,
     &  4.8111e-06, 4.9378e-06, 5.3231e-06, 5.7362e-06, 6.2350e-06,
     &  6.0951e-06, 5.7281e-06, 5.4585e-06, 4.9032e-06, 4.3009e-06,
     &  3.4776e-06, 2.8108e-06, 2.2993e-06, 1.7999e-06, 1.3870e-06,
     &  1.0750e-06, 8.5191e-07, 6.7951e-07, 5.5336e-07, 4.6439e-07,
     &  4.0243e-07, 3.5368e-07, 3.1427e-07, 2.7775e-07, 2.4486e-07,
     &  2.1788e-07, 1.9249e-07, 1.7162e-07, 1.5115e-07, 1.3478e-07,
     &  1.2236e-07, 1.1139e-07, 1.0092e-07, 9.0795e-08, 8.2214e-08,
     &  7.4691e-08, 6.7486e-08, 6.0414e-08, 5.4584e-08, 4.8754e-08,
     &  4.3501e-08, 3.8767e-08, 3.4363e-08, 3.0703e-08, 2.7562e-08,
     &  2.4831e-08, 2.2241e-08, 1.9939e-08, 1.8049e-08, 1.6368e-08,
     &  1.4863e-08, 1.3460e-08, 1.2212e-08, 1.1155e-08, 1.0185e-08,
     &  9.3417e-09, 8.5671e-09, 7.8292e-09, 7.1749e-09, 6.5856e-09,
     &  6.0588e-09, 5.5835e-09, 5.1350e-09, 4.7395e-09, 4.3771e-09,
     &  4.0476e-09, 3.7560e-09, 3.4861e-09, 3.2427e-09, 3.0240e-09/
      data s(801:900)/
     &  2.8278e-09, 2.6531e-09, 2.4937e-09, 2.3511e-09, 2.2245e-09,
     &  2.1133e-09, 2.0159e-09, 1.9330e-09, 1.8669e-09, 1.8152e-09,
     &  1.7852e-09, 1.7752e-09, 1.7823e-09, 1.8194e-09, 1.8866e-09,
     &  1.9759e-09, 2.0736e-09, 2.2083e-09, 2.3587e-09, 2.4984e-09,
     &  2.6333e-09, 2.8160e-09, 3.0759e-09, 3.3720e-09, 3.6457e-09,
     &  4.0668e-09, 4.4541e-09, 4.7976e-09, 5.0908e-09, 5.4811e-09,
     &  6.1394e-09, 6.3669e-09, 6.5714e-09, 6.8384e-09, 7.1918e-09,
     &  7.3741e-09, 7.2079e-09, 7.2172e-09, 7.2572e-09, 7.3912e-09,
     &  7.6188e-09, 8.3291e-09, 8.7885e-09, 9.2412e-09, 1.0021e-08,
     &  1.0752e-08, 1.1546e-08, 1.1607e-08, 1.1949e-08, 1.2346e-08,
     &  1.2516e-08, 1.2826e-08, 1.3053e-08, 1.3556e-08, 1.4221e-08,
     &  1.5201e-08, 1.6661e-08, 1.8385e-08, 2.0585e-08, 2.3674e-08,
     &  2.7928e-08, 3.3901e-08, 4.1017e-08, 4.9595e-08, 6.0432e-08,
     &  7.6304e-08, 9.0764e-08, 1.0798e-07, 1.2442e-07, 1.4404e-07,
     &  1.6331e-07, 1.8339e-07, 2.0445e-07, 2.2288e-07, 2.3083e-07,
     &  2.3196e-07, 2.3919e-07, 2.3339e-07, 2.3502e-07, 2.3444e-07,
     &  2.6395e-07, 2.9928e-07, 3.0025e-07, 3.0496e-07, 3.1777e-07,
     &  3.4198e-07, 3.4739e-07, 3.2696e-07, 3.4100e-07, 3.5405e-07,
     &  3.7774e-07, 3.8285e-07, 3.6797e-07, 3.5800e-07, 3.2283e-07,
     &  2.9361e-07, 2.4881e-07, 2.0599e-07, 1.7121e-07, 1.3641e-07/
      data s(901:1000)/
     &  1.1111e-07, 8.9413e-08, 7.3455e-08, 6.2078e-08, 5.2538e-08,
     &  4.5325e-08, 3.9005e-08, 3.4772e-08, 3.1203e-08, 2.8132e-08,
     &  2.5250e-08, 2.2371e-08, 2.0131e-08, 1.7992e-08, 1.6076e-08,
     &  1.4222e-08, 1.2490e-08, 1.1401e-08, 1.0249e-08, 9.2279e-09,
     &  8.5654e-09, 7.6227e-09, 6.9648e-09, 6.2466e-09, 5.7252e-09,
     &  5.3800e-09, 4.6960e-09, 4.2194e-09, 3.7746e-09, 3.3813e-09,
     &  3.0656e-09, 2.6885e-09, 2.4311e-09, 2.1572e-09, 1.8892e-09,
     &  1.7038e-09, 1.4914e-09, 1.3277e-09, 1.1694e-09, 1.0391e-09,
     &  9.2779e-10, 8.3123e-10, 7.4968e-10, 6.8385e-10, 6.2915e-10,
     &  5.7784e-10, 5.2838e-10, 4.8382e-10, 4.4543e-10, 4.1155e-10,
     &  3.7158e-10, 3.3731e-10, 3.0969e-10, 2.8535e-10, 2.6416e-10,
     &  2.4583e-10, 2.2878e-10, 2.1379e-10, 2.0073e-10, 1.8907e-10,
     &  1.7866e-10, 1.6936e-10, 1.6119e-10, 1.5424e-10, 1.4847e-10,
     &  1.4401e-10, 1.4068e-10, 1.3937e-10, 1.3943e-10, 1.4281e-10,
     &  1.4766e-10, 1.5701e-10, 1.7079e-10, 1.8691e-10, 2.0081e-10,
     &  2.1740e-10, 2.4847e-10, 2.6463e-10, 2.7087e-10, 2.7313e-10,
     &  2.8352e-10, 2.9511e-10, 2.8058e-10, 2.7227e-10, 2.7356e-10,
     &  2.8012e-10, 2.8034e-10, 2.9031e-10, 3.1030e-10, 3.3745e-10,
     &  3.8152e-10, 4.0622e-10, 4.2673e-10, 4.3879e-10, 4.5488e-10,
     &  4.7179e-10, 4.6140e-10, 4.6339e-10, 4.6716e-10, 4.7024e-10/
      data s(1001:1100)/
     &  4.7931e-10, 4.8503e-10, 4.9589e-10, 4.9499e-10, 5.0363e-10,
     &  5.3184e-10, 5.6451e-10, 6.0932e-10, 6.6469e-10, 7.4076e-10,
     &  8.3605e-10, 9.4898e-10, 1.0935e-09, 1.2593e-09, 1.4913e-09,
     &  1.8099e-09, 2.1842e-09, 2.7284e-09, 3.2159e-09, 3.7426e-09,
     &  4.5226e-09, 5.3512e-09, 6.1787e-09, 6.8237e-09, 7.9421e-09,
     &  9.0002e-09, 9.6841e-09, 9.9558e-09, 1.0232e-08, 1.0591e-08,
     &  1.0657e-08, 1.0441e-08, 1.0719e-08, 1.1526e-08, 1.2962e-08,
     &  1.4336e-08, 1.6150e-08, 1.8417e-08, 2.0725e-08, 2.3426e-08,
     &  2.5619e-08, 2.7828e-08, 3.0563e-08, 3.3438e-08, 3.6317e-08,
     &  4.0400e-08, 4.4556e-08, 5.0397e-08, 5.3315e-08, 5.9185e-08,
     &  6.5311e-08, 6.9188e-08, 7.7728e-08, 7.9789e-08, 8.6598e-08,
     &  8.7768e-08, 9.1773e-08, 9.7533e-08, 1.0007e-07, 1.0650e-07,
     &  1.0992e-07, 1.0864e-07, 1.0494e-07, 1.0303e-07, 1.0031e-07,
     &  1.0436e-07, 1.0537e-07, 1.1184e-07, 1.2364e-07, 1.3651e-07,
     &  1.4881e-07, 1.4723e-07, 1.4118e-07, 1.3371e-07, 1.1902e-07,
     &  1.0007e-07, 7.9628e-08, 6.4362e-08, 5.0243e-08, 3.8133e-08,
     &  2.9400e-08, 2.3443e-08, 1.9319e-08, 1.6196e-08, 1.4221e-08,
     &  1.2817e-08, 1.1863e-08, 1.1383e-08, 1.1221e-08, 1.1574e-08,
     &  1.1661e-08, 1.2157e-08, 1.2883e-08, 1.3295e-08, 1.4243e-08,
     &  1.4240e-08, 1.4614e-08, 1.4529e-08, 1.4685e-08, 1.4974e-08/
      data s(1101:1200)/
     &  1.4790e-08, 1.4890e-08, 1.4704e-08, 1.4142e-08, 1.3374e-08,
     &  1.2746e-08, 1.2172e-08, 1.2336e-08, 1.2546e-08, 1.3065e-08,
     &  1.4090e-08, 1.5215e-08, 1.6540e-08, 1.6144e-08, 1.5282e-08,
     &  1.4358e-08, 1.2849e-08, 1.0998e-08, 8.6956e-09, 7.0881e-09,
     &  5.5767e-09, 4.2792e-09, 3.2233e-09, 2.5020e-09, 1.9985e-09,
     &  1.5834e-09, 1.3015e-09, 1.0948e-09, 9.4141e-10, 8.1465e-10,
     &  7.1517e-10, 6.2906e-10, 5.5756e-10, 4.9805e-10, 4.3961e-10,
     &  3.9181e-10, 3.5227e-10, 3.1670e-10, 2.8667e-10, 2.5745e-10,
     &  2.3212e-10, 2.0948e-10, 1.8970e-10, 1.7239e-10, 1.5659e-10,
     &  1.4301e-10, 1.3104e-10, 1.2031e-10, 1.1095e-10, 1.0262e-10,
     &  9.5130e-11, 8.8595e-11, 8.2842e-11, 7.7727e-11, 7.3199e-11,
     &  6.9286e-11, 6.5994e-11, 6.3316e-11, 6.1244e-11, 5.9669e-11,
     &  5.8843e-11, 5.8832e-11, 5.9547e-11, 6.1635e-11, 6.4926e-11,
     &  7.0745e-11, 7.8802e-11, 8.6724e-11, 1.0052e-10, 1.1575e-10,
     &  1.3626e-10, 1.5126e-10, 1.6751e-10, 1.9239e-10, 2.1748e-10,
     &  2.2654e-10, 2.2902e-10, 2.3240e-10, 2.4081e-10, 2.3930e-10,
     &  2.2378e-10, 2.2476e-10, 2.2791e-10, 2.4047e-10, 2.5305e-10,
     &  2.8073e-10, 3.1741e-10, 3.6592e-10, 4.1495e-10, 4.6565e-10,
     &  5.0990e-10, 5.5607e-10, 6.1928e-10, 6.6779e-10, 7.3350e-10,
     &  8.1434e-10, 8.9635e-10, 9.9678e-10, 1.1256e-09, 1.2999e-09/
      data s(1201:1300)/
     &  1.4888e-09, 1.7642e-09, 1.9606e-09, 2.2066e-09, 2.4601e-09,
     &  2.7218e-09, 3.0375e-09, 3.1591e-09, 3.2852e-09, 3.2464e-09,
     &  3.3046e-09, 3.2710e-09, 3.2601e-09, 3.3398e-09, 3.7446e-09,
     &  4.0795e-09, 4.0284e-09, 4.0584e-09, 4.1677e-09, 4.5358e-09,
     &  4.4097e-09, 4.2744e-09, 4.5449e-09, 4.8147e-09, 5.2656e-09,
     &  5.2476e-09, 5.0275e-09, 4.7968e-09, 4.3654e-09, 3.9530e-09,
     &  3.2447e-09, 2.6489e-09, 2.1795e-09, 1.7880e-09, 1.4309e-09,
     &  1.1256e-09, 9.1903e-10, 7.6533e-10, 6.3989e-10, 5.5496e-10,
     &  4.9581e-10, 4.5722e-10, 4.3898e-10, 4.3505e-10, 4.3671e-10,
     &  4.5329e-10, 4.6827e-10, 4.9394e-10, 5.1122e-10, 5.1649e-10,
     &  5.0965e-10, 4.9551e-10, 4.8928e-10, 4.7947e-10, 4.7989e-10,
     &  4.9071e-10, 4.8867e-10, 4.7260e-10, 4.5756e-10, 4.5400e-10,
     &  4.5993e-10, 4.4042e-10, 4.3309e-10, 4.4182e-10, 4.6735e-10,
     &  5.0378e-10, 5.2204e-10, 5.0166e-10, 4.6799e-10, 4.3119e-10,
     &  3.8803e-10, 3.3291e-10, 2.6289e-10, 2.1029e-10, 1.7011e-10,
     &  1.3345e-10, 1.0224e-10, 7.8207e-11, 6.2451e-11, 5.0481e-11,
     &  4.1507e-11, 3.5419e-11, 3.0582e-11, 2.6900e-11, 2.3778e-11,
     &  2.1343e-11, 1.9182e-11, 1.7162e-11, 1.5391e-11, 1.3877e-11,
     &  1.2619e-11, 1.1450e-11, 1.0461e-11, 9.6578e-12, 8.9579e-12,
     &  8.3463e-12, 7.8127e-12, 7.3322e-12, 6.9414e-12, 6.6037e-12/
      data s(1301:1400)/
     &  6.3285e-12, 6.1095e-12, 5.9387e-12, 5.8118e-12, 5.7260e-12,
     &  5.6794e-12, 5.6711e-12, 5.7003e-12, 5.7670e-12, 5.8717e-12,
     &  6.0151e-12, 6.1984e-12, 6.4232e-12, 6.6918e-12, 7.0065e-12,
     &  7.3705e-12, 7.7873e-12, 8.2612e-12, 8.7972e-12, 9.4009e-12,
     &  1.0079e-11, 1.0840e-11, 1.1692e-11, 1.2648e-11, 1.3723e-11,
     &  1.4935e-11, 1.6313e-11, 1.7905e-11, 1.9740e-11, 2.1898e-11,
     &  2.4419e-11, 2.7426e-11, 3.0869e-11, 3.4235e-11, 3.7841e-11,
     &  4.1929e-11, 4.6776e-11, 5.2123e-11, 5.8497e-11, 6.5294e-11,
     &  7.4038e-11, 8.4793e-11, 9.6453e-11, 1.1223e-10, 1.2786e-10,
     &  1.4882e-10, 1.7799e-10, 2.0766e-10, 2.4523e-10, 2.8591e-10,
     &  3.3386e-10, 4.0531e-10, 4.7663e-10, 5.4858e-10, 6.3377e-10,
     &  7.1688e-10, 8.4184e-10, 9.5144e-10, 1.0481e-09, 1.1356e-09,
     &  1.2339e-09, 1.3396e-09, 1.4375e-09, 1.5831e-09, 1.7323e-09,
     &  1.9671e-09, 2.2976e-09, 2.6679e-09, 3.0777e-09, 3.4321e-09,
     &  3.8192e-09, 4.2711e-09, 4.4903e-09, 4.8931e-09, 5.2253e-09,
     &  5.4040e-09, 5.6387e-09, 5.6704e-09, 6.0345e-09, 6.1079e-09,
     &  6.2576e-09, 6.4039e-09, 6.3776e-09, 6.1878e-09, 5.8616e-09,
     &  5.7036e-09, 5.5840e-09, 5.6905e-09, 5.8931e-09, 6.2478e-09,
     &  6.8291e-09, 7.4528e-09, 7.6078e-09, 7.3898e-09, 6.7573e-09,
     &  5.9827e-09, 5.0927e-09, 4.0099e-09, 3.1933e-09, 2.4296e-09/
      data s(1401:1500)/
     &  1.8485e-09, 1.4595e-09, 1.2017e-09, 1.0164e-09, 8.7433e-10,
     &  7.7108e-10, 7.0049e-10, 6.5291e-10, 6.1477e-10, 5.9254e-10,
     &  5.8150e-10, 5.7591e-10, 5.8490e-10, 5.8587e-10, 5.9636e-10,
     &  6.2408e-10, 6.5479e-10, 7.0480e-10, 7.2313e-10, 7.5524e-10,
     &  8.0863e-10, 8.3386e-10, 9.2342e-10, 9.6754e-10, 1.0293e-09,
     &  1.0895e-09, 1.1330e-09, 1.2210e-09, 1.2413e-09, 1.2613e-09,
     &  1.2671e-09, 1.2225e-09, 1.1609e-09, 1.0991e-09, 1.0600e-09,
     &  1.0570e-09, 1.0818e-09, 1.1421e-09, 1.2270e-09, 1.3370e-09,
     &  1.4742e-09, 1.4946e-09, 1.4322e-09, 1.3210e-09, 1.1749e-09,
     &  1.0051e-09, 7.8387e-10, 6.1844e-10, 4.6288e-10, 3.4164e-10,
     &  2.5412e-10, 1.9857e-10, 1.5876e-10, 1.2966e-10, 1.0920e-10,
     &  9.4811e-11, 8.3733e-11, 7.3906e-11, 6.7259e-11, 6.1146e-11,
     &  5.7119e-11, 5.3546e-11, 4.8625e-11, 4.4749e-11, 4.1089e-11,
     &  3.7825e-11, 3.4465e-11, 3.1018e-11, 2.8109e-11, 2.5610e-11,
     &  2.2859e-11, 2.0490e-11, 1.8133e-11, 1.5835e-11, 1.3949e-11,
     &  1.2295e-11, 1.0799e-11, 9.6544e-12, 8.7597e-12, 7.9990e-12,
     &  7.3973e-12, 6.9035e-12, 6.4935e-12, 6.1195e-12, 5.8235e-12,
     &  5.5928e-12, 5.4191e-12, 5.2993e-12, 5.2338e-12, 5.2272e-12,
     &  5.2923e-12, 5.4252e-12, 5.6523e-12, 5.9433e-12, 6.3197e-12,
     &  6.9016e-12, 7.5016e-12, 8.2885e-12, 9.4050e-12, 1.0605e-11/
      data s(1501:1600)/
     &  1.2257e-11, 1.3622e-11, 1.5353e-11, 1.7543e-11, 1.9809e-11,
     &  2.2197e-11, 2.4065e-11, 2.6777e-11, 2.9751e-11, 3.2543e-11,
     &  3.5536e-11, 3.9942e-11, 4.6283e-11, 5.4556e-11, 6.5490e-11,
     &  7.6803e-11, 9.0053e-11, 1.0852e-10, 1.2946e-10, 1.4916e-10,
     &  1.7748e-10, 2.0073e-10, 2.2485e-10, 2.5114e-10, 2.7715e-10,
     &  3.1319e-10, 3.3305e-10, 3.5059e-10, 3.5746e-10, 3.6311e-10,
     &  3.7344e-10, 3.6574e-10, 3.7539e-10, 3.9434e-10, 4.3510e-10,
     &  4.3340e-10, 4.2588e-10, 4.3977e-10, 4.6062e-10, 4.7687e-10,
     &  4.6457e-10, 4.8578e-10, 5.2344e-10, 5.6752e-10, 5.8702e-10,
     &  5.6603e-10, 5.3784e-10, 4.9181e-10, 4.3272e-10, 3.5681e-10,
     &  2.8814e-10, 2.3320e-10, 1.8631e-10, 1.4587e-10, 1.1782e-10,
     &  9.8132e-11, 8.2528e-11, 6.9174e-11, 6.1056e-11, 5.3459e-11,
     &  4.7116e-11, 4.1878e-11, 3.8125e-11, 3.6347e-11, 3.5071e-11,
     &  3.3897e-11, 3.3541e-11, 3.3563e-11, 3.5469e-11, 3.8111e-11,
     &  3.8675e-11, 4.1333e-11, 4.3475e-11, 4.6476e-11, 4.9761e-11,
     &  5.1380e-11, 5.4135e-11, 5.3802e-11, 5.5158e-11, 5.6864e-11,
     &  5.9311e-11, 6.3827e-11, 6.7893e-11, 6.8230e-11, 6.6694e-11,
     &  6.6018e-11, 6.4863e-11, 6.5893e-11, 6.3813e-11, 6.4741e-11,
     &  6.8630e-11, 7.0255e-11, 7.0667e-11, 6.8810e-11, 6.4104e-11,
     &  5.8136e-11, 4.7242e-11, 3.7625e-11, 3.1742e-11, 2.5581e-11/
      data s(1601:1700)/
     &  1.8824e-11, 1.3303e-11, 9.6919e-12, 7.5353e-12, 6.0986e-12,
     &  5.0742e-12, 4.3094e-12, 3.7190e-12, 3.2520e-12, 2.8756e-12,
     &  2.5680e-12, 2.3139e-12, 2.1025e-12, 1.9257e-12, 1.7777e-12,
     &  1.6539e-12, 1.5508e-12, 1.4657e-12, 1.3966e-12, 1.3417e-12,
     &  1.2998e-12, 1.2700e-12, 1.2514e-12, 1.2437e-12, 1.2463e-12,
     &  1.2592e-12, 1.2823e-12, 1.3157e-12, 1.3596e-12, 1.4144e-12,
     &  1.4806e-12, 1.5588e-12, 1.6497e-12, 1.7544e-12, 1.8738e-12,
     &  2.0094e-12, 2.1626e-12, 2.3354e-12, 2.5297e-12, 2.7483e-12,
     &  2.9941e-12, 3.2708e-12, 3.5833e-12, 3.9374e-12, 4.3415e-12,
     &  4.8079e-12, 5.3602e-12, 5.9816e-12, 6.7436e-12, 7.6368e-12,
     &  8.6812e-12, 9.8747e-12, 1.1350e-11, 1.3181e-11, 1.5406e-11,
     &  1.7868e-11, 2.0651e-11, 2.4504e-11, 2.9184e-11, 3.4159e-11,
     &  3.9979e-11, 4.8704e-11, 5.7856e-11, 6.7576e-11, 7.9103e-11,
     &  9.4370e-11, 1.1224e-10, 1.3112e-10, 1.5674e-10, 1.8206e-10,
     &  2.0576e-10, 2.3187e-10, 2.7005e-10, 3.0055e-10, 3.3423e-10,
     &  3.6956e-10, 3.8737e-10, 4.2630e-10, 4.5154e-10, 4.8383e-10,
     &  5.3582e-10, 5.8109e-10, 6.3741e-10, 6.3874e-10, 6.3870e-10,
     &  6.5818e-10, 6.5056e-10, 6.5291e-10, 6.3159e-10, 6.3984e-10,
     &  6.4549e-10, 6.5444e-10, 6.7035e-10, 6.7665e-10, 6.9124e-10,
     &  6.8451e-10, 6.9255e-10, 6.9923e-10, 7.0396e-10, 6.7715e-10/
      data s(1701:1800)/
     &  6.0371e-10, 5.3774e-10, 4.6043e-10, 3.7635e-10, 2.9484e-10,
     &  2.2968e-10, 1.8185e-10, 1.4191e-10, 1.1471e-10, 9.4790e-11,
     &  7.9613e-11, 6.7989e-11, 5.9391e-11, 5.2810e-11, 4.7136e-11,
     &  4.2618e-11, 3.8313e-11, 3.4686e-11, 3.1669e-11, 2.9110e-11,
     &  2.6871e-11, 2.5074e-11, 2.4368e-11, 2.3925e-11, 2.4067e-11,
     &  2.4336e-11, 2.4704e-11, 2.5823e-11, 2.7177e-11, 2.9227e-11,
     &  3.1593e-11, 3.5730e-11, 4.0221e-11, 4.3994e-11, 4.8448e-11,
     &  5.3191e-11, 5.8552e-11, 6.3458e-11, 6.6335e-11, 7.2457e-11,
     &  7.9091e-11, 8.2234e-11, 8.7668e-11, 8.7951e-11, 9.2952e-11,
     &  9.6157e-11, 9.5926e-11, 1.0120e-10, 1.0115e-10, 9.9577e-11,
     &  9.6633e-11, 9.2891e-11, 9.3315e-11, 9.5584e-11, 1.0064e-10,
     &  1.0509e-10, 1.1455e-10, 1.2443e-10, 1.2963e-10, 1.2632e-10,
     &  1.1308e-10, 1.0186e-10, 8.5880e-11, 6.7863e-11, 5.1521e-11,
     &  3.7780e-11, 2.8842e-11, 2.2052e-11, 1.7402e-11, 1.4406e-11,
     &  1.1934e-11, 1.0223e-11, 8.9544e-12, 7.9088e-12, 7.0675e-12,
     &  6.2222e-12, 5.6051e-12, 5.0502e-12, 4.5578e-12, 4.2636e-12,
     &  3.9461e-12, 3.7599e-12, 3.5215e-12, 3.2467e-12, 3.0018e-12,
     &  2.6558e-12, 2.3928e-12, 2.0707e-12, 1.7575e-12, 1.5114e-12,
     &  1.2941e-12, 1.1004e-12, 9.5175e-13, 8.2894e-13, 7.3253e-13,
     &  6.5551e-13, 5.9098e-13, 5.3548e-13, 4.8697e-13, 4.4413e-13/
      data s(1801:1900)/
     &  4.0600e-13, 3.7188e-13, 3.4121e-13, 3.1356e-13, 2.8856e-13,
     &  2.6590e-13, 2.4533e-13, 2.2663e-13, 2.0960e-13, 1.9407e-13,
     &  1.7990e-13, 1.6695e-13, 1.5512e-13, 1.4429e-13, 1.3437e-13,
     &  1.2527e-13, 1.1693e-13, 1.0927e-13, 1.0224e-13, 9.5767e-14,
     &  8.9816e-14, 8.4335e-14, 7.9285e-14, 7.4626e-14, 7.0325e-14,
     &  6.6352e-14, 6.2676e-14, 5.9274e-14, 5.6121e-14, 5.3195e-14,
     &  5.0479e-14, 4.7953e-14, 4.5602e-14, 4.3411e-14, 4.1367e-14,
     &  3.9456e-14, 3.7670e-14, 3.5996e-14, 3.4427e-14, 3.2952e-14,
     &  3.1566e-14, 3.0261e-14, 2.9030e-14, 2.7868e-14, 2.6770e-14,
     &  2.5730e-14, 2.4745e-14, 2.3809e-14, 2.2921e-14, 2.2076e-14,
     &  2.1271e-14, 2.0504e-14, 1.9772e-14, 1.9073e-14, 1.8404e-14,
     &  1.7764e-14, 1.7151e-14, 1.6564e-14, 1.6000e-14, 1.5459e-14,
     &  1.4939e-14, 1.4439e-14, 1.3958e-14, 1.3495e-14, 1.3049e-14,
     &  1.2620e-14, 1.2206e-14, 1.1807e-14, 1.1422e-14, 1.1050e-14,
     &  1.0691e-14, 1.0345e-14, 1.0010e-14, 9.6870e-15, 9.3747e-15,
     &  9.0727e-15, 8.7808e-15, 8.4986e-15, 8.2257e-15, 7.9617e-15,
     &  7.7064e-15, 7.4594e-15, 7.2204e-15, 6.9891e-15, 6.7653e-15,
     &  6.5488e-15, 6.3392e-15, 6.1363e-15, 5.9399e-15, 5.7499e-15,
     &  5.5659e-15, 5.3878e-15, 5.2153e-15, 5.0484e-15, 4.8868e-15,
     &  4.7303e-15, 4.5788e-15, 4.4322e-15, 4.2902e-15, 4.1527e-15/
      data s(1901:2003)/
     &  4.0196e-15, 3.8907e-15, 3.7659e-15, 3.6451e-15, 3.5281e-15,
     &  3.4149e-15, 3.3052e-15, 3.1991e-15, 3.0963e-15, 2.9967e-15,
     &  2.9004e-15, 2.8071e-15, 2.7167e-15, 2.6293e-15, 2.5446e-15,
     &  2.4626e-15, 2.3833e-15, 2.3064e-15, 2.2320e-15, 2.1600e-15,
     &  2.0903e-15, 2.0228e-15, 1.9574e-15, 1.8942e-15, 1.8329e-15,
     &  1.7736e-15, 1.7163e-15, 1.6607e-15, 1.6069e-15, 1.5548e-15,
     &  1.5044e-15, 1.4557e-15, 1.4084e-15, 1.3627e-15, 1.3185e-15,
     &  1.2757e-15, 1.2342e-15, 1.1941e-15, 1.1552e-15, 1.1177e-15,
     &  1.0813e-15, 1.0461e-15, 1.0120e-15, 9.7900e-16, 9.4707e-16,
     &  9.1618e-16, 8.8628e-16, 8.5734e-16, 8.2933e-16, 8.0223e-16,
     &  7.7600e-16, 7.5062e-16, 7.2606e-16, 7.0229e-16, 6.7929e-16,
     &  6.5703e-16, 6.3550e-16, 6.1466e-16, 5.9449e-16, 5.7498e-16,
     &  5.5610e-16, 5.3783e-16, 5.2015e-16, 5.0305e-16, 4.8650e-16,
     &  4.7049e-16, 4.5500e-16, 4.4002e-16, 4.2552e-16, 4.1149e-16,
     &  3.9792e-16, 3.8479e-16, 3.7209e-16, 3.5981e-16, 3.4792e-16,
     &  3.3642e-16, 3.2530e-16, 3.1454e-16, 3.0413e-16, 2.9406e-16,
     &  2.8432e-16, 2.7490e-16, 2.6579e-16, 2.5697e-16, 2.4845e-16,
     &  2.4020e-16, 2.3223e-16, 2.2451e-16, 2.1705e-16, 2.0984e-16,
     &  2.0286e-16, 1.9611e-16, 1.8958e-16, 1.8327e-16, 1.7716e-16,
     &  1.7126e-16, 1.6555e-16, 1.6003e-16, 1.5469e-16, 1.4952e-16,
     &  1.4453e-16, 1.3970e-16, 1.3503e-16/
cccc

      call sint(v1,v,dv,npts,s,sh2ot0)
      return
      end
c=======================================================================
      subroutine slf260(v,sh2ot1)

c     water vapor self continuum at 260K

      use params, only: kr
      implicit none
      integer, parameter :: npts=2003
      real(kr) :: v, sh2ot1, v1=-20, v2=20000.0, dv=10., s(npts)

      ! units of (cm**3/mol) * 1.e-20

      data s(1:100)/
     &  1.7750e-01, 1.7045e-01, 1.6457e-01, 1.7045e-01, 1.7750e-01,
     &  2.0036e-01, 2.1347e-01, 2.2454e-01, 2.3428e-01, 2.3399e-01,
     &  2.3022e-01, 2.0724e-01, 1.9712e-01, 1.8317e-01, 1.6724e-01,
     &  1.4780e-01, 1.2757e-01, 1.1626e-01, 1.0098e-01, 8.9033e-02,
     &  7.9770e-02, 6.7416e-02, 5.9588e-02, 5.1117e-02, 4.6218e-02,
     &  4.2179e-02, 3.4372e-02, 2.9863e-02, 2.5252e-02, 2.2075e-02,
     &  1.9209e-02, 1.5816e-02, 1.3932e-02, 1.1943e-02, 1.0079e-02,
     &  8.7667e-03, 7.4094e-03, 6.4967e-03, 5.5711e-03, 4.8444e-03,
     &  4.2552e-03, 3.6953e-03, 3.2824e-03, 2.9124e-03, 2.6102e-03,
     &  2.3370e-03, 2.1100e-03, 1.9008e-03, 1.7145e-03, 1.5573e-03,
     &  1.4206e-03, 1.2931e-03, 1.1803e-03, 1.0774e-03, 9.8616e-04,
     &  9.0496e-04, 8.3071e-04, 7.6319e-04, 7.0149e-04, 6.4637e-04,
     &  5.9566e-04, 5.4987e-04, 5.0768e-04, 4.6880e-04, 4.3317e-04,
     &  4.0037e-04, 3.7064e-04, 3.4325e-04, 3.1809e-04, 2.9501e-04,
     &  2.7382e-04, 2.5430e-04, 2.3630e-04, 2.1977e-04, 2.0452e-04,
     &  1.9042e-04, 1.7740e-04, 1.6544e-04, 1.5442e-04, 1.4425e-04,
     &  1.3486e-04, 1.2618e-04, 1.1817e-04, 1.1076e-04, 1.0391e-04,
     &  9.7563e-05, 9.1696e-05, 8.6272e-05, 8.1253e-05, 7.6607e-05,
     &  7.2302e-05, 6.8311e-05, 6.4613e-05, 6.1183e-05, 5.8001e-05,
     &  5.5048e-05, 5.2307e-05, 4.9761e-05, 4.7395e-05, 4.5197e-05/
      data s(101:200)/
     &  4.3155e-05, 4.1256e-05, 3.9491e-05, 3.7849e-05, 3.6324e-05,
     &  3.4908e-05, 3.3594e-05, 3.2374e-05, 3.1244e-05, 3.0201e-05,
     &  2.9240e-05, 2.8356e-05, 2.7547e-05, 2.6814e-05, 2.6147e-05,
     &  2.5551e-05, 2.5029e-05, 2.4582e-05, 2.4203e-05, 2.3891e-05,
     &  2.3663e-05, 2.3531e-05, 2.3483e-05, 2.3516e-05, 2.3694e-05,
     &  2.4032e-05, 2.4579e-05, 2.5234e-05, 2.6032e-05, 2.7119e-05,
     &  2.8631e-05, 3.0848e-05, 3.3262e-05, 3.6635e-05, 4.0732e-05,
     &  4.5923e-05, 5.3373e-05, 6.1875e-05, 7.2031e-05, 8.5980e-05,
     &  9.8642e-05, 1.1469e-04, 1.3327e-04, 1.5390e-04, 1.7513e-04,
     &  2.0665e-04, 2.3609e-04, 2.6220e-04, 2.8677e-04, 3.2590e-04,
     &  3.8624e-04, 4.1570e-04, 4.5207e-04, 4.9336e-04, 5.4500e-04,
     &  5.8258e-04, 5.8086e-04, 5.6977e-04, 5.3085e-04, 4.8020e-04,
     &  4.3915e-04, 4.0343e-04, 3.7853e-04, 3.7025e-04, 3.9637e-04,
     &  4.4675e-04, 4.7072e-04, 4.9022e-04, 5.2076e-04, 5.3676e-04,
     &  5.2755e-04, 4.8244e-04, 4.5473e-04, 4.3952e-04, 3.9614e-04,
     &  3.4086e-04, 2.9733e-04, 2.6367e-04, 2.3767e-04, 2.0427e-04,
     &  1.7595e-04, 1.5493e-04, 1.3851e-04, 1.1874e-04, 1.0735e-04,
     &  9.0490e-05, 8.1149e-05, 7.4788e-05, 6.5438e-05, 5.8248e-05,
     &  4.8076e-05, 4.3488e-05, 3.7856e-05, 3.3034e-05, 2.9592e-05,
     &  2.6088e-05, 2.3497e-05, 2.0279e-05, 1.7526e-05, 1.5714e-05/
      data s(201:300)/
     &  1.3553e-05, 1.2145e-05, 1.0802e-05, 9.7681e-06, 8.8196e-06,
     &  7.8291e-06, 7.1335e-06, 6.4234e-06, 5.8391e-06, 5.3532e-06,
     &  4.9079e-06, 4.5378e-06, 4.1716e-06, 3.8649e-06, 3.5893e-06,
     &  3.3406e-06, 3.1199e-06, 2.9172e-06, 2.7348e-06, 2.5644e-06,
     &  2.4086e-06, 2.2664e-06, 2.1359e-06, 2.0159e-06, 1.9051e-06,
     &  1.8031e-06, 1.7074e-06, 1.6185e-06, 1.5356e-06, 1.4584e-06,
     &  1.3861e-06, 1.3179e-06, 1.2545e-06, 1.1951e-06, 1.1395e-06,
     &  1.0873e-06, 1.0384e-06, 9.9250e-07, 9.4935e-07, 9.0873e-07,
     &  8.7050e-07, 8.3446e-07, 8.0046e-07, 7.6834e-07, 7.3800e-07,
     &  7.0931e-07, 6.8217e-07, 6.5648e-07, 6.3214e-07, 6.0909e-07,
     &  5.8725e-07, 5.6655e-07, 5.4693e-07, 5.2835e-07, 5.1077e-07,
     &  4.9416e-07, 4.7853e-07, 4.6381e-07, 4.5007e-07, 4.3728e-07,
     &  4.2550e-07, 4.1450e-07, 4.0459e-07, 3.9532e-07, 3.8662e-07,
     &  3.7855e-07, 3.7041e-07, 3.6254e-07, 3.5420e-07, 3.4617e-07,
     &  3.3838e-07, 3.3212e-07, 3.2655e-07, 3.1865e-07, 3.1203e-07,
     &  3.0670e-07, 3.0252e-07, 2.9749e-07, 2.9184e-07, 2.8795e-07,
     &  2.8501e-07, 2.8202e-07, 2.7856e-07, 2.7509e-07, 2.7152e-07,
     &  2.6844e-07, 2.6642e-07, 2.6548e-07, 2.6617e-07, 2.6916e-07,
     &  2.7372e-07, 2.8094e-07, 2.9236e-07, 3.1035e-07, 3.2854e-07,
     &  3.5481e-07, 3.9377e-07, 4.4692e-07, 5.0761e-07, 5.7715e-07/
      data s(301:400)/
     &  6.7725e-07, 8.0668e-07, 9.3716e-07, 1.0797e-06, 1.1689e-06,
     &  1.3217e-06, 1.4814e-06, 1.5627e-06, 1.6519e-06, 1.7601e-06,
     &  1.9060e-06, 2.0474e-06, 2.0716e-06, 2.0433e-06, 1.9752e-06,
     &  1.8466e-06, 1.7526e-06, 1.6657e-06, 1.5870e-06, 1.5633e-06,
     &  1.6520e-06, 1.8471e-06, 1.9953e-06, 2.0975e-06, 2.2016e-06,
     &  2.2542e-06, 2.3081e-06, 2.3209e-06, 2.2998e-06, 2.3056e-06,
     &  2.2757e-06, 2.2685e-06, 2.2779e-06, 2.2348e-06, 2.2445e-06,
     &  2.3174e-06, 2.4284e-06, 2.5290e-06, 2.7340e-06, 2.9720e-06,
     &  3.2332e-06, 3.5392e-06, 3.9013e-06, 4.3334e-06, 4.9088e-06,
     &  5.3428e-06, 5.9142e-06, 6.6106e-06, 7.4709e-06, 8.5019e-06,
     &  9.6835e-06, 1.0984e-05, 1.2831e-05, 1.4664e-05, 1.7080e-05,
     &  2.0103e-05, 2.4148e-05, 2.7948e-05, 3.2855e-05, 3.9046e-05,
     &  4.6429e-05, 5.6633e-05, 6.6305e-05, 7.6048e-05, 8.7398e-05,
     &  1.0034e-04, 1.1169e-04, 1.2813e-04, 1.3354e-04, 1.3952e-04,
     &  1.4204e-04, 1.4615e-04, 1.5144e-04, 1.5475e-04, 1.6561e-04,
     &  1.7135e-04, 1.6831e-04, 1.6429e-04, 1.6353e-04, 1.6543e-04,
     &  1.5944e-04, 1.5404e-04, 1.5458e-04, 1.6287e-04, 1.7277e-04,
     &  1.8387e-04, 1.7622e-04, 1.6360e-04, 1.5273e-04, 1.3667e-04,
     &  1.2364e-04, 9.7576e-05, 7.9140e-05, 6.4241e-05, 5.1826e-05,
     &  4.1415e-05, 3.1347e-05, 2.5125e-05, 2.0027e-05, 1.6362e-05/
      data s(401:500)/
     &  1.3364e-05, 1.1117e-05, 9.4992e-06, 8.1581e-06, 7.1512e-06,
     &  6.2692e-06, 5.5285e-06, 4.9000e-06, 4.3447e-06, 3.8906e-06,
     &  3.4679e-06, 3.1089e-06, 2.8115e-06, 2.5496e-06, 2.2982e-06,
     &  2.0861e-06, 1.8763e-06, 1.7035e-06, 1.5548e-06, 1.4107e-06,
     &  1.2839e-06, 1.1706e-06, 1.0709e-06, 9.8099e-07, 8.9901e-07,
     &  8.2394e-07, 7.5567e-07, 6.9434e-07, 6.3867e-07, 5.8845e-07,
     &  5.4263e-07, 5.0033e-07, 4.6181e-07, 4.2652e-07, 3.9437e-07,
     &  3.6497e-07, 3.3781e-07, 3.1292e-07, 2.9011e-07, 2.6915e-07,
     &  2.4989e-07, 2.3215e-07, 2.1582e-07, 2.0081e-07, 1.8700e-07,
     &  1.7432e-07, 1.6264e-07, 1.5191e-07, 1.4207e-07, 1.3306e-07,
     &  1.2484e-07, 1.1737e-07, 1.1056e-07, 1.0451e-07, 9.9060e-08,
     &  9.4135e-08, 8.9608e-08, 8.5697e-08, 8.1945e-08, 7.8308e-08,
     &  7.4808e-08, 7.1686e-08, 6.8923e-08, 6.5869e-08, 6.3308e-08,
     &  6.0840e-08, 5.8676e-08, 5.6744e-08, 5.5016e-08, 5.3813e-08,
     &  5.2792e-08, 5.2097e-08, 5.1737e-08, 5.1603e-08, 5.1656e-08,
     &  5.1989e-08, 5.2467e-08, 5.2918e-08, 5.3589e-08, 5.4560e-08,
     &  5.5869e-08, 5.7403e-08, 5.8968e-08, 6.0973e-08, 6.3432e-08,
     &  6.6245e-08, 6.9353e-08, 7.2686e-08, 7.6541e-08, 8.0991e-08,
     &  8.5950e-08, 9.1429e-08, 9.7851e-08, 1.0516e-07, 1.1349e-07,
     &  1.2295e-07, 1.3335e-07, 1.4488e-07, 1.5864e-07, 1.7412e-07/
      data s(501:600)/
     &  1.9140e-07, 2.1078e-07, 2.3369e-07, 2.5996e-07, 2.8848e-07,
     &  3.2169e-07, 3.5991e-07, 4.0566e-07, 4.5969e-07, 5.3094e-07,
     &  6.1458e-07, 7.1155e-07, 8.3045e-07, 9.9021e-07, 1.2042e-06,
     &  1.4914e-06, 1.8145e-06, 2.2210e-06, 2.7831e-06, 3.4533e-06,
     &  4.4446e-06, 5.1989e-06, 6.2289e-06, 7.1167e-06, 8.3949e-06,
     &  9.6417e-06, 1.0313e-05, 1.0485e-05, 1.0641e-05, 1.0898e-05,
     &  1.0763e-05, 1.0506e-05, 1.0497e-05, 1.1696e-05, 1.2654e-05,
     &  1.3029e-05, 1.3175e-05, 1.4264e-05, 1.4985e-05, 1.4999e-05,
     &  1.4317e-05, 1.4616e-05, 1.4963e-05, 1.5208e-05, 1.4942e-05,
     &  1.3879e-05, 1.3087e-05, 1.1727e-05, 1.0515e-05, 9.0073e-06,
     &  7.3133e-06, 6.1181e-06, 5.0623e-06, 4.1105e-06, 3.3915e-06,
     &  2.6711e-06, 2.1464e-06, 1.7335e-06, 1.4302e-06, 1.1847e-06,
     &  9.9434e-07, 8.2689e-07, 7.0589e-07, 6.0750e-07, 5.3176e-07,
     &  4.6936e-07, 4.1541e-07, 3.6625e-07, 3.2509e-07, 2.9156e-07,
     &  2.6308e-07, 2.3819e-07, 2.1421e-07, 1.9366e-07, 1.7626e-07,
     &  1.5982e-07, 1.4567e-07, 1.3354e-07, 1.2097e-07, 1.1029e-07,
     &  1.0063e-07, 9.2003e-08, 8.4245e-08, 7.7004e-08, 7.0636e-08,
     &  6.4923e-08, 5.9503e-08, 5.4742e-08, 5.0450e-08, 4.6470e-08,
     &  4.2881e-08, 3.9550e-08, 3.6541e-08, 3.3803e-08, 3.1279e-08,
     &  2.8955e-08, 2.6858e-08, 2.4905e-08, 2.3146e-08, 2.1539e-08/
      data s(601:700)/
     &  2.0079e-08, 1.8746e-08, 1.7517e-08, 1.6396e-08, 1.5369e-08,
     &  1.4426e-08, 1.3543e-08, 1.2724e-08, 1.1965e-08, 1.1267e-08,
     &  1.0617e-08, 1.0010e-08, 9.4662e-09, 8.9553e-09, 8.4988e-09,
     &  8.0807e-09, 7.7043e-09, 7.3721e-09, 7.0707e-09, 6.8047e-09,
     &  6.5702e-09, 6.3634e-09, 6.1817e-09, 6.0239e-09, 5.8922e-09,
     &  5.7824e-09, 5.7019e-09, 5.6368e-09, 5.5940e-09, 5.5669e-09,
     &  5.5583e-09, 5.5653e-09, 5.5837e-09, 5.6243e-09, 5.6883e-09,
     &  5.7800e-09, 5.8964e-09, 6.0429e-09, 6.2211e-09, 6.4282e-09,
     &  6.6634e-09, 6.9306e-09, 7.2336e-09, 7.5739e-09, 7.9562e-09,
     &  8.3779e-09, 8.8575e-09, 9.3992e-09, 1.0004e-08, 1.0684e-08,
     &  1.1450e-08, 1.2320e-08, 1.3311e-08, 1.4455e-08, 1.5758e-08,
     &  1.7254e-08, 1.8927e-08, 2.0930e-08, 2.3348e-08, 2.6074e-08,
     &  2.9221e-08, 3.2770e-08, 3.7485e-08, 4.2569e-08, 4.8981e-08,
     &  5.5606e-08, 6.2393e-08, 7.1901e-08, 8.2921e-08, 9.5513e-08,
     &  1.1111e-07, 1.3143e-07, 1.5971e-07, 1.8927e-07, 2.2643e-07,
     &  2.7860e-07, 3.2591e-07, 3.7024e-07, 4.2059e-07, 4.9432e-07,
     &  5.5543e-07, 5.7498e-07, 5.9210e-07, 6.1005e-07, 6.1577e-07,
     &  5.9193e-07, 5.6602e-07, 5.7403e-07, 6.0050e-07, 6.4723e-07,
     &  6.7073e-07, 7.5415e-07, 8.0982e-07, 8.7658e-07, 9.1430e-07,
     &  9.4459e-07, 9.8347e-07, 9.8768e-07, 1.0153e-06, 1.0066e-06/
      data s(701:800)/
     &  1.0353e-06, 1.0353e-06, 1.0722e-06, 1.1138e-06, 1.1923e-06,
     &  1.2947e-06, 1.4431e-06, 1.6537e-06, 1.8662e-06, 2.2473e-06,
     &  2.6464e-06, 3.1041e-06, 3.4858e-06, 4.0167e-06, 4.6675e-06,
     &  5.0983e-06, 5.7997e-06, 6.0503e-06, 6.4687e-06, 6.5396e-06,
     &  6.7986e-06, 7.0244e-06, 7.2305e-06, 7.6732e-06, 7.9783e-06,
     &  7.9846e-06, 7.7617e-06, 7.7657e-06, 7.7411e-06, 7.8816e-06,
     &  7.8136e-06, 8.0051e-06, 8.5799e-06, 9.1659e-06, 9.8646e-06,
     &  9.4920e-06, 8.7670e-06, 8.2034e-06, 7.2297e-06, 6.2324e-06,
     &  4.9315e-06, 3.9128e-06, 3.1517e-06, 2.4469e-06, 1.8815e-06,
     &  1.4627e-06, 1.1698e-06, 9.4686e-07, 7.8486e-07, 6.6970e-07,
     &  5.8811e-07, 5.2198e-07, 4.6809e-07, 4.1671e-07, 3.7006e-07,
     &  3.3066e-07, 2.9387e-07, 2.6415e-07, 2.3409e-07, 2.0991e-07,
     &  1.9132e-07, 1.7519e-07, 1.5939e-07, 1.4368e-07, 1.3050e-07,
     &  1.1883e-07, 1.0772e-07, 9.6884e-08, 8.7888e-08, 7.8956e-08,
     &  7.1024e-08, 6.3824e-08, 5.7256e-08, 5.1769e-08, 4.7037e-08,
     &  4.2901e-08, 3.8970e-08, 3.5467e-08, 3.2502e-08, 2.9827e-08,
     &  2.7389e-08, 2.5111e-08, 2.3056e-08, 2.1267e-08, 1.9610e-08,
     &  1.8133e-08, 1.6775e-08, 1.5491e-08, 1.4329e-08, 1.3265e-08,
     &  1.2300e-08, 1.1420e-08, 1.0593e-08, 9.8475e-09, 9.1585e-09,
     &  8.5256e-09, 7.9525e-09, 7.4226e-09, 6.9379e-09, 6.4950e-09/
      data s(801:900)/
     &  6.0911e-09, 5.7242e-09, 5.3877e-09, 5.0821e-09, 4.8051e-09,
     &  4.5554e-09, 4.3315e-09, 4.1336e-09, 3.9632e-09, 3.8185e-09,
     &  3.7080e-09, 3.6296e-09, 3.5804e-09, 3.5776e-09, 3.6253e-09,
     &  3.7115e-09, 3.8151e-09, 3.9804e-09, 4.1742e-09, 4.3581e-09,
     &  4.5306e-09, 4.7736e-09, 5.1297e-09, 5.5291e-09, 5.9125e-09,
     &  6.4956e-09, 7.0362e-09, 7.5318e-09, 7.9947e-09, 8.6438e-09,
     &  9.7227e-09, 1.0130e-08, 1.0549e-08, 1.1064e-08, 1.1702e-08,
     &  1.2043e-08, 1.1781e-08, 1.1838e-08, 1.1917e-08, 1.2131e-08,
     &  1.2476e-08, 1.3611e-08, 1.4360e-08, 1.5057e-08, 1.6247e-08,
     &  1.7284e-08, 1.8420e-08, 1.8352e-08, 1.8722e-08, 1.9112e-08,
     &  1.9092e-08, 1.9311e-08, 1.9411e-08, 1.9884e-08, 2.0508e-08,
     &  2.1510e-08, 2.3143e-08, 2.5050e-08, 2.7596e-08, 3.1231e-08,
     &  3.6260e-08, 4.3410e-08, 5.2240e-08, 6.3236e-08, 7.7522e-08,
     &  9.8688e-08, 1.1859e-07, 1.4341e-07, 1.6798e-07, 1.9825e-07,
     &  2.2898e-07, 2.6257e-07, 2.9884e-07, 3.3247e-07, 3.4936e-07,
     &  3.5583e-07, 3.7150e-07, 3.6580e-07, 3.7124e-07, 3.7030e-07,
     &  4.1536e-07, 4.6656e-07, 4.6677e-07, 4.7507e-07, 4.9653e-07,
     &  5.3795e-07, 5.4957e-07, 5.2238e-07, 5.4690e-07, 5.6569e-07,
     &  5.9844e-07, 5.9835e-07, 5.6522e-07, 5.4123e-07, 4.7904e-07,
     &  4.2851e-07, 3.5603e-07, 2.8932e-07, 2.3655e-07, 1.8592e-07/
      data s(901:1000)/
     &  1.4943e-07, 1.1971e-07, 9.8482e-08, 8.3675e-08, 7.1270e-08,
     &  6.2496e-08, 5.4999e-08, 4.9821e-08, 4.5387e-08, 4.1340e-08,
     &  3.7453e-08, 3.3298e-08, 3.0120e-08, 2.7032e-08, 2.4236e-08,
     &  2.1500e-08, 1.8988e-08, 1.7414e-08, 1.5706e-08, 1.4192e-08,
     &  1.3204e-08, 1.1759e-08, 1.0737e-08, 9.6309e-09, 8.8179e-09,
     &  8.2619e-09, 7.2264e-09, 6.4856e-09, 5.8037e-09, 5.2093e-09,
     &  4.7205e-09, 4.1749e-09, 3.7852e-09, 3.3915e-09, 3.0089e-09,
     &  2.7335e-09, 2.4398e-09, 2.2031e-09, 1.9786e-09, 1.7890e-09,
     &  1.6266e-09, 1.4830e-09, 1.3576e-09, 1.2518e-09, 1.1587e-09,
     &  1.0726e-09, 9.9106e-10, 9.1673e-10, 8.5084e-10, 7.9147e-10,
     &  7.2882e-10, 6.7342e-10, 6.2593e-10, 5.8294e-10, 5.4435e-10,
     &  5.0997e-10, 4.7806e-10, 4.4931e-10, 4.2357e-10, 4.0023e-10,
     &  3.7909e-10, 3.5999e-10, 3.4285e-10, 3.2776e-10, 3.1468e-10,
     &  3.0377e-10, 2.9479e-10, 2.8877e-10, 2.8512e-10, 2.8617e-10,
     &  2.8976e-10, 3.0001e-10, 3.1718e-10, 3.3898e-10, 3.5857e-10,
     &  3.8358e-10, 4.3131e-10, 4.5741e-10, 4.6948e-10, 4.7594e-10,
     &  4.9529e-10, 5.1563e-10, 4.9475e-10, 4.8369e-10, 4.8829e-10,
     &  5.0047e-10, 5.0203e-10, 5.1954e-10, 5.5352e-10, 5.9928e-10,
     &  6.7148e-10, 7.1121e-10, 7.4317e-10, 7.6039e-10, 7.8313e-10,
     &  8.0684e-10, 7.8553e-10, 7.8312e-10, 7.8537e-10, 7.8872e-10/
      data s(1001:1100)/
     &  8.0185e-10, 8.1004e-10, 8.2608e-10, 8.2525e-10, 8.3857e-10,
     &  8.7920e-10, 9.2451e-10, 9.8661e-10, 1.0629e-09, 1.1659e-09,
     &  1.2922e-09, 1.4387e-09, 1.6254e-09, 1.8425e-09, 2.1428e-09,
     &  2.5477e-09, 3.0379e-09, 3.7570e-09, 4.4354e-09, 5.1802e-09,
     &  6.2769e-09, 7.4894e-09, 8.7474e-09, 9.8037e-09, 1.1582e-08,
     &  1.3293e-08, 1.4471e-08, 1.5025e-08, 1.5580e-08, 1.6228e-08,
     &  1.6413e-08, 1.6020e-08, 1.6393e-08, 1.7545e-08, 1.9590e-08,
     &  2.1449e-08, 2.3856e-08, 2.7050e-08, 3.0214e-08, 3.3733e-08,
     &  3.6487e-08, 3.9353e-08, 4.2660e-08, 4.6385e-08, 4.9955e-08,
     &  5.5313e-08, 6.0923e-08, 6.8948e-08, 7.3649e-08, 8.2602e-08,
     &  9.2212e-08, 9.9080e-08, 1.1319e-07, 1.1790e-07, 1.2941e-07,
     &  1.3199e-07, 1.3914e-07, 1.4843e-07, 1.5300e-07, 1.6419e-07,
     &  1.7095e-07, 1.6988e-07, 1.6494e-07, 1.6327e-07, 1.6067e-07,
     &  1.6909e-07, 1.7118e-07, 1.8106e-07, 1.9857e-07, 2.1696e-07,
     &  2.3385e-07, 2.2776e-07, 2.1402e-07, 1.9882e-07, 1.7362e-07,
     &  1.4308e-07, 1.1158e-07, 8.8781e-08, 6.8689e-08, 5.2062e-08,
     &  4.0427e-08, 3.2669e-08, 2.7354e-08, 2.3200e-08, 2.0580e-08,
     &  1.8676e-08, 1.7329e-08, 1.6621e-08, 1.6433e-08, 1.6953e-08,
     &  1.7134e-08, 1.7948e-08, 1.9107e-08, 1.9875e-08, 2.1416e-08,
     &  2.1556e-08, 2.2265e-08, 2.2171e-08, 2.2534e-08, 2.3029e-08/
      data s(1101:1200)/
     &  2.2828e-08, 2.3143e-08, 2.2965e-08, 2.2223e-08, 2.1108e-08,
     &  2.0265e-08, 1.9516e-08, 1.9941e-08, 2.0312e-08, 2.1080e-08,
     &  2.2611e-08, 2.4210e-08, 2.6069e-08, 2.5097e-08, 2.3318e-08,
     &  2.1543e-08, 1.8942e-08, 1.5960e-08, 1.2386e-08, 9.9340e-09,
     &  7.7502e-09, 5.9462e-09, 4.5113e-09, 3.5523e-09, 2.8844e-09,
     &  2.3394e-09, 1.9584e-09, 1.6749e-09, 1.4624e-09, 1.2809e-09,
     &  1.1359e-09, 1.0087e-09, 9.0166e-10, 8.1079e-10, 7.2219e-10,
     &  6.4922e-10, 5.8803e-10, 5.3290e-10, 4.8590e-10, 4.4111e-10,
     &  4.0184e-10, 3.6644e-10, 3.3529e-10, 3.0789e-10, 2.8286e-10,
     &  2.6089e-10, 2.4125e-10, 2.2355e-10, 2.0783e-10, 1.9370e-10,
     &  1.8088e-10, 1.6948e-10, 1.5929e-10, 1.5013e-10, 1.4193e-10,
     &  1.3470e-10, 1.2841e-10, 1.2307e-10, 1.1865e-10, 1.1502e-10,
     &  1.1243e-10, 1.1099e-10, 1.1066e-10, 1.1216e-10, 1.1529e-10,
     &  1.2171e-10, 1.3128e-10, 1.4153e-10, 1.5962e-10, 1.8048e-10,
     &  2.0936e-10, 2.3165e-10, 2.5746e-10, 2.9600e-10, 3.3707e-10,
     &  3.5267e-10, 3.5953e-10, 3.6822e-10, 3.8363e-10, 3.8286e-10,
     &  3.5883e-10, 3.6154e-10, 3.6653e-10, 3.8507e-10, 4.0250e-10,
     &  4.4435e-10, 4.9889e-10, 5.6932e-10, 6.3599e-10, 7.0281e-10,
     &  7.5777e-10, 8.1279e-10, 8.8910e-10, 9.3400e-10, 1.0076e-09,
     &  1.0945e-09, 1.1898e-09, 1.3108e-09, 1.4725e-09, 1.7028e-09/
      data s(1201:1300)/
     &  1.9619e-09, 2.3527e-09, 2.6488e-09, 3.0327e-09, 3.4396e-09,
     &  3.8797e-09, 4.4115e-09, 4.6853e-09, 4.9553e-09, 4.9551e-09,
     &  5.1062e-09, 5.0996e-09, 5.1119e-09, 5.2283e-09, 5.8297e-09,
     &  6.3439e-09, 6.2675e-09, 6.3296e-09, 6.5173e-09, 7.1685e-09,
     &  7.0528e-09, 6.8856e-09, 7.3182e-09, 7.6990e-09, 8.3461e-09,
     &  8.1946e-09, 7.7153e-09, 7.2411e-09, 6.4511e-09, 5.7336e-09,
     &  4.6105e-09, 3.6962e-09, 2.9944e-09, 2.4317e-09, 1.9399e-09,
     &  1.5331e-09, 1.2633e-09, 1.0613e-09, 9.0136e-10, 7.9313e-10,
     &  7.1543e-10, 6.6485e-10, 6.4225e-10, 6.3980e-10, 6.4598e-10,
     &  6.7428e-10, 7.0270e-10, 7.4694e-10, 7.7946e-10, 7.9395e-10,
     &  7.8716e-10, 7.6933e-10, 7.6220e-10, 7.4825e-10, 7.4805e-10,
     &  7.6511e-10, 7.6492e-10, 7.4103e-10, 7.1979e-10, 7.1686e-10,
     &  7.3403e-10, 7.1142e-10, 7.0212e-10, 7.1548e-10, 7.5253e-10,
     &  8.0444e-10, 8.2378e-10, 7.8004e-10, 7.1712e-10, 6.4978e-10,
     &  5.7573e-10, 4.8675e-10, 3.7945e-10, 3.0118e-10, 2.4241e-10,
     &  1.9100e-10, 1.4816e-10, 1.1567e-10, 9.4183e-11, 7.7660e-11,
     &  6.5270e-11, 5.6616e-11, 4.9576e-11, 4.4137e-11, 3.9459e-11,
     &  3.5759e-11, 3.2478e-11, 2.9419e-11, 2.6703e-11, 2.4365e-11,
     &  2.2412e-11, 2.0606e-11, 1.9067e-11, 1.7800e-11, 1.6695e-11,
     &  1.5729e-11, 1.4887e-11, 1.4135e-11, 1.3519e-11, 1.2992e-11/
      data s(1301:1400)/
     &  1.2563e-11, 1.2223e-11, 1.1962e-11, 1.1775e-11, 1.1657e-11,
     &  1.1605e-11, 1.1619e-11, 1.1697e-11, 1.1839e-11, 1.2046e-11,
     &  1.2319e-11, 1.2659e-11, 1.3070e-11, 1.3553e-11, 1.4113e-11,
     &  1.4754e-11, 1.5480e-11, 1.6298e-11, 1.7214e-11, 1.8236e-11,
     &  1.9372e-11, 2.0635e-11, 2.2036e-11, 2.3590e-11, 2.5317e-11,
     &  2.7242e-11, 2.9400e-11, 3.1849e-11, 3.4654e-11, 3.7923e-11,
     &  4.1695e-11, 4.6055e-11, 5.0940e-11, 5.5624e-11, 6.0667e-11,
     &  6.6261e-11, 7.2692e-11, 7.9711e-11, 8.7976e-11, 9.6884e-11,
     &  1.0775e-10, 1.2093e-10, 1.3531e-10, 1.5404e-10, 1.7315e-10,
     &  1.9862e-10, 2.3341e-10, 2.7014e-10, 3.1716e-10, 3.6957e-10,
     &  4.3233e-10, 5.2566e-10, 6.2251e-10, 7.2149e-10, 8.3958e-10,
     &  9.5931e-10, 1.1388e-09, 1.2973e-09, 1.4442e-09, 1.5638e-09,
     &  1.6974e-09, 1.8489e-09, 1.9830e-09, 2.1720e-09, 2.3662e-09,
     &  2.6987e-09, 3.1697e-09, 3.6907e-09, 4.2625e-09, 4.7946e-09,
     &  5.3848e-09, 6.0897e-09, 6.4730e-09, 7.1483e-09, 7.7432e-09,
     &  8.0851e-09, 8.5013e-09, 8.5909e-09, 9.1890e-09, 9.3124e-09,
     &  9.5936e-09, 9.8787e-09, 9.9036e-09, 9.6712e-09, 9.2036e-09,
     &  9.0466e-09, 8.9380e-09, 9.1815e-09, 9.5092e-09, 1.0027e-08,
     &  1.0876e-08, 1.1744e-08, 1.1853e-08, 1.1296e-08, 1.0134e-08,
     &  8.8245e-09, 7.3930e-09, 5.7150e-09, 4.4884e-09, 3.4027e-09/
      data s(1401:1500)/
     &  2.6054e-09, 2.0790e-09, 1.7267e-09, 1.4724e-09, 1.2722e-09,
     &  1.1234e-09, 1.0186e-09, 9.4680e-10, 8.8854e-10, 8.5127e-10,
     &  8.3157e-10, 8.2226e-10, 8.3395e-10, 8.3294e-10, 8.4725e-10,
     &  8.8814e-10, 9.3697e-10, 1.0112e-09, 1.0412e-09, 1.0948e-09,
     &  1.1810e-09, 1.2267e-09, 1.3690e-09, 1.4512e-09, 1.5568e-09,
     &  1.6552e-09, 1.7321e-09, 1.8797e-09, 1.9210e-09, 1.9686e-09,
     &  1.9917e-09, 1.9357e-09, 1.8486e-09, 1.7575e-09, 1.7113e-09,
     &  1.7163e-09, 1.7623e-09, 1.8536e-09, 1.9765e-09, 2.1334e-09,
     &  2.3237e-09, 2.3259e-09, 2.1833e-09, 1.9785e-09, 1.7308e-09,
     &  1.4596e-09, 1.1198e-09, 8.7375e-10, 6.5381e-10, 4.8677e-10,
     &  3.6756e-10, 2.9155e-10, 2.3735e-10, 1.9590e-10, 1.6638e-10,
     &  1.4549e-10, 1.2947e-10, 1.1511e-10, 1.0548e-10, 9.6511e-11,
     &  9.0469e-11, 8.5170e-11, 7.7804e-11, 7.1971e-11, 6.6213e-11,
     &  6.1063e-11, 5.5881e-11, 5.0508e-11, 4.5932e-11, 4.1997e-11,
     &  3.7672e-11, 3.3972e-11, 3.0318e-11, 2.6769e-11, 2.3874e-11,
     &  2.1336e-11, 1.9073e-11, 1.7313e-11, 1.5904e-11, 1.4684e-11,
     &  1.3698e-11, 1.2873e-11, 1.2175e-11, 1.1542e-11, 1.1024e-11,
     &  1.0602e-11, 1.0267e-11, 1.0012e-11, 9.8379e-12, 9.7482e-12,
     &  9.7564e-12, 9.8613e-12, 1.0092e-11, 1.0418e-11, 1.0868e-11,
     &  1.1585e-11, 1.2351e-11, 1.3372e-11, 1.4841e-11, 1.6457e-11/
      data s(1501:1600)/
     &  1.8681e-11, 2.0550e-11, 2.2912e-11, 2.5958e-11, 2.9137e-11,
     &  3.2368e-11, 3.4848e-11, 3.8462e-11, 4.2190e-11, 4.5629e-11,
     &  4.9022e-11, 5.4232e-11, 6.1900e-11, 7.1953e-11, 8.5368e-11,
     &  9.9699e-11, 1.1734e-10, 1.4185e-10, 1.7017e-10, 1.9813e-10,
     &  2.3859e-10, 2.7304e-10, 3.0971e-10, 3.5129e-10, 3.9405e-10,
     &  4.5194e-10, 4.8932e-10, 5.2436e-10, 5.4098e-10, 5.5542e-10,
     &  5.7794e-10, 5.6992e-10, 5.8790e-10, 6.1526e-10, 6.8034e-10,
     &  6.7956e-10, 6.6864e-10, 6.9329e-10, 7.2971e-10, 7.6546e-10,
     &  7.5078e-10, 7.8406e-10, 8.3896e-10, 9.0111e-10, 9.1994e-10,
     &  8.7189e-10, 8.1426e-10, 7.3097e-10, 6.3357e-10, 5.1371e-10,
     &  4.0936e-10, 3.2918e-10, 2.6255e-10, 2.0724e-10, 1.6879e-10,
     &  1.4165e-10, 1.1989e-10, 1.0125e-10, 8.9629e-11, 7.8458e-11,
     &  6.8826e-11, 6.0935e-11, 5.5208e-11, 5.2262e-11, 5.0260e-11,
     &  4.8457e-11, 4.7888e-11, 4.8032e-11, 5.0838e-11, 5.4668e-11,
     &  5.5790e-11, 6.0056e-11, 6.3811e-11, 6.8848e-11, 7.4590e-11,
     &  7.8249e-11, 8.3371e-11, 8.3641e-11, 8.6591e-11, 8.9599e-11,
     &  9.3487e-11, 1.0066e-10, 1.0765e-10, 1.0851e-10, 1.0619e-10,
     &  1.0557e-10, 1.0460e-10, 1.0796e-10, 1.0523e-10, 1.0674e-10,
     &  1.1261e-10, 1.1431e-10, 1.1408e-10, 1.0901e-10, 9.9105e-11,
     &  8.8077e-11, 6.9928e-11, 5.4595e-11, 4.5401e-11, 3.6313e-11/
      data s(1601:1700)/
     &  2.6986e-11, 1.9463e-11, 1.4577e-11, 1.1583e-11, 9.5492e-12,
     &  8.0770e-12, 6.9642e-12, 6.0966e-12, 5.4046e-12, 4.8431e-12,
     &  4.3815e-12, 3.9987e-12, 3.6790e-12, 3.4113e-12, 3.1868e-12,
     &  2.9992e-12, 2.8434e-12, 2.7153e-12, 2.6120e-12, 2.5311e-12,
     &  2.4705e-12, 2.4290e-12, 2.4053e-12, 2.3988e-12, 2.4087e-12,
     &  2.4349e-12, 2.4771e-12, 2.5355e-12, 2.6103e-12, 2.7019e-12,
     &  2.8110e-12, 2.9383e-12, 3.0848e-12, 3.2518e-12, 3.4405e-12,
     &  3.6527e-12, 3.8902e-12, 4.1555e-12, 4.4510e-12, 4.7801e-12,
     &  5.1462e-12, 5.5539e-12, 6.0086e-12, 6.5171e-12, 7.0884e-12,
     &  7.7357e-12, 8.4831e-12, 9.3096e-12, 1.0282e-11, 1.1407e-11,
     &  1.2690e-11, 1.4148e-11, 1.5888e-11, 1.7992e-11, 2.0523e-11,
     &  2.3342e-11, 2.6578e-11, 3.0909e-11, 3.6228e-11, 4.2053e-11,
     &  4.9059e-11, 5.9273e-11, 7.0166e-11, 8.2298e-11, 9.7071e-11,
     &  1.1673e-10, 1.4010e-10, 1.6621e-10, 2.0127e-10, 2.3586e-10,
     &  2.7050e-10, 3.0950e-10, 3.6584e-10, 4.1278e-10, 4.6591e-10,
     &  5.2220e-10, 5.5246e-10, 6.1500e-10, 6.5878e-10, 7.1167e-10,
     &  7.9372e-10, 8.6975e-10, 9.6459e-10, 9.7368e-10, 9.8142e-10,
     &  1.0202e-09, 1.0200e-09, 1.0356e-09, 1.0092e-09, 1.0269e-09,
     &  1.0366e-09, 1.0490e-09, 1.0717e-09, 1.0792e-09, 1.1016e-09,
     &  1.0849e-09, 1.0929e-09, 1.0971e-09, 1.0969e-09, 1.0460e-09/
      data s(1701:1800)/
     &  9.2026e-10, 8.1113e-10, 6.8635e-10, 5.5369e-10, 4.2908e-10,
     &  3.3384e-10, 2.6480e-10, 2.0810e-10, 1.6915e-10, 1.4051e-10,
     &  1.1867e-10, 1.0158e-10, 8.8990e-11, 7.9175e-11, 7.0440e-11,
     &  6.3453e-11, 5.7009e-11, 5.1662e-11, 4.7219e-11, 4.3454e-11,
     &  4.0229e-11, 3.7689e-11, 3.6567e-11, 3.5865e-11, 3.5955e-11,
     &  3.5928e-11, 3.6298e-11, 3.7629e-11, 3.9300e-11, 4.1829e-11,
     &  4.4806e-11, 5.0534e-11, 5.6672e-11, 6.2138e-11, 6.8678e-11,
     &  7.6111e-11, 8.4591e-11, 9.2634e-11, 9.8085e-11, 1.0830e-10,
     &  1.1949e-10, 1.2511e-10, 1.3394e-10, 1.3505e-10, 1.4342e-10,
     &  1.4874e-10, 1.4920e-10, 1.5872e-10, 1.5972e-10, 1.5821e-10,
     &  1.5425e-10, 1.4937e-10, 1.5089e-10, 1.5521e-10, 1.6325e-10,
     &  1.6924e-10, 1.8265e-10, 1.9612e-10, 2.0176e-10, 1.9359e-10,
     &  1.7085e-10, 1.5197e-10, 1.2646e-10, 9.8552e-11, 7.4530e-11,
     &  5.5052e-11, 4.2315e-11, 3.2736e-11, 2.6171e-11, 2.1909e-11,
     &  1.8286e-11, 1.5752e-11, 1.3859e-11, 1.2288e-11, 1.1002e-11,
     &  9.7534e-12, 8.8412e-12, 8.0169e-12, 7.2855e-12, 6.8734e-12,
     &  6.4121e-12, 6.1471e-12, 5.7780e-12, 5.3478e-12, 4.9652e-12,
     &  4.4043e-12, 3.9862e-12, 3.4684e-12, 2.9681e-12, 2.5791e-12,
     &  2.2339e-12, 1.9247e-12, 1.6849e-12, 1.4863e-12, 1.3291e-12,
     &  1.2021e-12, 1.0947e-12, 1.0015e-12, 9.1935e-13, 8.4612e-13/
      data s(1801:1900)/
     &  7.8036e-13, 7.2100e-13, 6.6718e-13, 6.1821e-13, 5.7353e-13,
     &  5.3269e-13, 4.9526e-13, 4.6093e-13, 4.2937e-13, 4.0034e-13,
     &  3.7361e-13, 3.4895e-13, 3.2621e-13, 3.0520e-13, 2.8578e-13,
     &  2.6782e-13, 2.5120e-13, 2.3581e-13, 2.2154e-13, 2.0832e-13,
     &  1.9605e-13, 1.8466e-13, 1.7408e-13, 1.6425e-13, 1.5511e-13,
     &  1.4661e-13, 1.3869e-13, 1.3131e-13, 1.2444e-13, 1.1803e-13,
     &  1.1205e-13, 1.0646e-13, 1.0124e-13, 9.6358e-14, 9.1789e-14,
     &  8.7509e-14, 8.3498e-14, 7.9735e-14, 7.6202e-14, 7.2882e-14,
     &  6.9760e-14, 6.6822e-14, 6.4053e-14, 6.1442e-14, 5.8978e-14,
     &  5.6650e-14, 5.4448e-14, 5.2364e-14, 5.0389e-14, 4.8516e-14,
     &  4.6738e-14, 4.5048e-14, 4.3441e-14, 4.1911e-14, 4.0453e-14,
     &  3.9063e-14, 3.7735e-14, 3.6467e-14, 3.5254e-14, 3.4093e-14,
     &  3.2980e-14, 3.1914e-14, 3.0891e-14, 2.9909e-14, 2.8965e-14,
     &  2.8058e-14, 2.7185e-14, 2.6344e-14, 2.5535e-14, 2.4755e-14,
     &  2.4002e-14, 2.3276e-14, 2.2576e-14, 2.1899e-14, 2.1245e-14,
     &  2.0613e-14, 2.0002e-14, 1.9411e-14, 1.8839e-14, 1.8285e-14,
     &  1.7749e-14, 1.7230e-14, 1.6727e-14, 1.6240e-14, 1.5768e-14,
     &  1.5310e-14, 1.4867e-14, 1.4436e-14, 1.4019e-14, 1.3614e-14,
     &  1.3221e-14, 1.2840e-14, 1.2471e-14, 1.2112e-14, 1.1764e-14,
     &  1.1425e-14, 1.1097e-14, 1.0779e-14, 1.0469e-14, 1.0169e-14/
      data s(1901:2000)/
     &  9.8775e-15, 9.5943e-15, 9.3193e-15, 9.0522e-15, 8.7928e-15,
     &  8.5409e-15, 8.2962e-15, 8.0586e-15, 7.8278e-15, 7.6036e-15,
     &  7.3858e-15, 7.1742e-15, 6.9687e-15, 6.7691e-15, 6.5752e-15,
     &  6.3868e-15, 6.2038e-15, 6.0260e-15, 5.8533e-15, 5.6856e-15,
     &  5.5226e-15, 5.3642e-15, 5.2104e-15, 5.0610e-15, 4.9158e-15,
     &  4.7748e-15, 4.6378e-15, 4.5047e-15, 4.3753e-15, 4.2497e-15,
     &  4.1277e-15, 4.0091e-15, 3.8939e-15, 3.7820e-15, 3.6733e-15,
     &  3.5677e-15, 3.4651e-15, 3.3655e-15, 3.2686e-15, 3.1746e-15,
     &  3.0832e-15, 2.9944e-15, 2.9082e-15, 2.8244e-15, 2.7431e-15,
     &  2.6640e-15, 2.5872e-15, 2.5126e-15, 2.4401e-15, 2.3697e-15,
     &  2.3014e-15, 2.2349e-15, 2.1704e-15, 2.1077e-15, 2.0468e-15,
     &  1.9877e-15, 1.9302e-15, 1.8744e-15, 1.8202e-15, 1.7675e-15,
     &  1.7164e-15, 1.6667e-15, 1.6184e-15, 1.5716e-15, 1.5260e-15,
     &  1.4818e-15, 1.4389e-15, 1.3971e-15, 1.3566e-15, 1.3172e-15,
     &  1.2790e-15, 1.2419e-15, 1.2058e-15, 1.1708e-15, 1.1368e-15,
     &  1.1037e-15, 1.0716e-15, 1.0405e-15, 1.0102e-15, 9.8079e-16,
     &  9.5224e-16, 9.2451e-16, 8.9758e-16, 8.7142e-16, 8.4602e-16,
     &  8.2136e-16, 7.9740e-16, 7.7414e-16, 7.5154e-16, 7.2961e-16,
     &  7.0830e-16, 6.8761e-16, 6.6752e-16, 6.4801e-16, 6.2906e-16,
     &  6.1066e-16, 5.9280e-16, 5.7545e-16, 5.5860e-16, 5.4224e-16/
      data s(2001:2003)/
     &  5.2636e-16, 5.1094e-16, 4.9596e-16/

cccc
      call sint(v1,v,dv,npts,s,sh2ot1)
      return
      end
c=======================================================================
      subroutine frn296(v,fh2o)

c     water vapor foreign continuum at 296K

      use params, only: kr
      implicit none
      integer, parameter :: npts=2003
      real(kr) :: v, fh2o, v1=-20, v2=20000.0, dv=10., f(npts)

      ! units of (cm**3/mol) * 1.e-20

      data f(1:100)/
     & 1.2859e-02, 1.1715e-02, 1.1038e-02, 1.1715e-02, 1.2859e-02, 
     & 1.5326e-02, 1.6999e-02, 1.8321e-02, 1.9402e-02, 1.9570e-02, 
     & 1.9432e-02, 1.7572e-02, 1.6760e-02, 1.5480e-02, 1.3984e-02, 
     & 1.2266e-02, 1.0467e-02, 9.4526e-03, 8.0485e-03, 6.9484e-03, 
     & 6.1416e-03, 5.0941e-03, 4.4836e-03, 3.8133e-03, 3.4608e-03, 
     & 3.1487e-03, 2.4555e-03, 2.0977e-03, 1.7266e-03, 1.4920e-03, 
     & 1.2709e-03, 9.8081e-04, 8.5063e-04, 6.8822e-04, 5.3809e-04, 
     & 4.4679e-04, 3.3774e-04, 2.7979e-04, 2.1047e-04, 1.6511e-04, 
     & 1.2993e-04, 9.3033e-05, 7.4360e-05, 5.6428e-05, 4.5442e-05, 
     & 3.4575e-05, 2.7903e-05, 2.1374e-05, 1.6075e-05, 1.3022e-05, 
     & 1.0962e-05, 8.5959e-06, 6.9125e-06, 5.3808e-06, 4.3586e-06, 
     & 3.6394e-06, 2.9552e-06, 2.3547e-06, 1.8463e-06, 1.6036e-06, 
     & 1.3483e-06, 1.1968e-06, 1.0333e-06, 8.4484e-07, 6.7195e-07, 
     & 5.0947e-07, 4.2343e-07, 3.4453e-07, 2.7830e-07, 2.3063e-07, 
     & 1.9951e-07, 1.7087e-07, 1.4393e-07, 1.2575e-07, 1.0750e-07, 
     & 8.2325e-08, 5.7524e-08, 4.4482e-08, 3.8106e-08, 3.4315e-08, 
     & 2.9422e-08, 2.5069e-08, 2.2402e-08, 1.9349e-08, 1.6152e-08, 
     & 1.2208e-08, 8.9660e-09, 7.1322e-09, 6.1028e-09, 5.2938e-09, 
     & 4.5350e-09, 3.4977e-09, 2.9511e-09, 2.4734e-09, 2.0508e-09, 
     & 1.8507e-09, 1.6373e-09, 1.5171e-09, 1.3071e-09, 1.2462e-09/
      data f(101:200)/
     & 1.2148e-09, 1.2590e-09, 1.3153e-09, 1.3301e-09, 1.4483e-09, 
     & 1.6944e-09, 2.0559e-09, 2.2954e-09, 2.6221e-09, 3.2606e-09, 
     & 4.2392e-09, 5.2171e-09, 6.2553e-09, 8.2548e-09, 9.5842e-09, 
     & 1.1280e-08, 1.3628e-08, 1.7635e-08, 2.1576e-08, 2.4835e-08, 
     & 3.0014e-08, 3.8485e-08, 4.7440e-08, 5.5202e-08, 7.0897e-08, 
     & 9.6578e-08, 1.3976e-07, 1.8391e-07, 2.3207e-07, 2.9960e-07, 
     & 4.0408e-07, 5.9260e-07, 7.8487e-07, 1.0947e-06, 1.4676e-06, 
     & 1.9325e-06, 2.6587e-06, 3.4534e-06, 4.4376e-06, 5.8061e-06, 
     & 7.0141e-06, 8.4937e-06, 1.0186e-05, 1.2034e-05, 1.3837e-05, 
     & 1.6595e-05, 1.9259e-05, 2.1620e-05, 2.3681e-05, 2.7064e-05, 
     & 3.2510e-05, 3.5460e-05, 3.9109e-05, 4.2891e-05, 4.7757e-05, 
     & 5.0981e-05, 5.0527e-05, 4.8618e-05, 4.4001e-05, 3.7982e-05, 
     & 3.2667e-05, 2.7794e-05, 2.4910e-05, 2.4375e-05, 2.7316e-05, 
     & 3.2579e-05, 3.5499e-05, 3.8010e-05, 4.1353e-05, 4.3323e-05, 
     & 4.3004e-05, 3.9790e-05, 3.7718e-05, 3.6360e-05, 3.2386e-05, 
     & 2.7409e-05, 2.3626e-05, 2.0631e-05, 1.8371e-05, 1.5445e-05, 
     & 1.2989e-05, 1.1098e-05, 9.6552e-06, 8.0649e-06, 7.2365e-06, 
     & 5.9137e-06, 5.2759e-06, 4.8860e-06, 4.1321e-06, 3.5918e-06, 
     & 2.7640e-06, 2.4892e-06, 2.1018e-06, 1.7848e-06, 1.5855e-06, 
     & 1.3569e-06, 1.1986e-06, 9.4693e-07, 7.4097e-07, 6.3443e-07/
      data f(201:300)/
     & 4.8131e-07, 4.0942e-07, 3.3316e-07, 2.8488e-07, 2.3461e-07, 
     & 1.7397e-07, 1.4684e-07, 1.0953e-07, 8.5396e-08, 6.9261e-08, 
     & 5.4001e-08, 4.5430e-08, 3.2791e-08, 2.5995e-08, 2.0225e-08, 
     & 1.5710e-08, 1.3027e-08, 1.0229e-08, 8.5277e-09, 6.5249e-09, 
     & 5.0117e-09, 3.9906e-09, 3.2332e-09, 2.7847e-09, 2.4570e-09, 
     & 2.3359e-09, 2.0599e-09, 1.8436e-09, 1.6559e-09, 1.4910e-09, 
     & 1.2794e-09, 9.8229e-10, 8.0054e-10, 6.0769e-10, 4.5646e-10, 
     & 3.3111e-10, 2.4428e-10, 1.8007e-10, 1.3291e-10, 9.7974e-11, 
     & 7.8271e-11, 6.3833e-11, 5.4425e-11, 4.6471e-11, 4.0209e-11, 
     & 3.5227e-11, 3.1212e-11, 2.8840e-11, 2.7762e-11, 2.7935e-11, 
     & 3.2012e-11, 3.9525e-11, 5.0303e-11, 6.8027e-11, 9.3954e-11, 
     & 1.2986e-10, 1.8478e-10, 2.5331e-10, 3.4827e-10, 4.6968e-10, 
     & 6.2380e-10, 7.9106e-10, 1.0026e-09, 1.2102e-09, 1.4146e-09, 
     & 1.6154e-09, 1.7510e-09, 1.8575e-09, 1.8742e-09, 1.8700e-09, 
     & 1.8582e-09, 1.9657e-09, 2.1204e-09, 2.0381e-09, 2.0122e-09, 
     & 2.0436e-09, 2.1213e-09, 2.0742e-09, 1.9870e-09, 2.0465e-09, 
     & 2.1556e-09, 2.2222e-09, 2.1977e-09, 2.1047e-09, 1.9334e-09, 
     & 1.7357e-09, 1.5754e-09, 1.4398e-09, 1.4018e-09, 1.5459e-09, 
     & 1.7576e-09, 2.1645e-09, 2.9480e-09, 4.4439e-09, 5.8341e-09, 
     & 8.0757e-09, 1.1658e-08, 1.6793e-08, 2.2694e-08, 2.9468e-08/
      data f(301:400)/
     & 3.9278e-08, 5.2145e-08, 6.4378e-08, 7.7947e-08, 8.5321e-08, 
     & 9.7848e-08, 1.0999e-07, 1.1489e-07, 1.2082e-07, 1.2822e-07, 
     & 1.4053e-07, 1.5238e-07, 1.5454e-07, 1.5018e-07, 1.4048e-07, 
     & 1.2359e-07, 1.0858e-07, 9.3486e-08, 8.1638e-08, 7.7690e-08, 
     & 8.4625e-08, 1.0114e-07, 1.1430e-07, 1.2263e-07, 1.3084e-07, 
     & 1.3380e-07, 1.3573e-07, 1.3441e-07, 1.2962e-07, 1.2638e-07, 
     & 1.1934e-07, 1.1371e-07, 1.0871e-07, 9.8843e-08, 9.1877e-08, 
     & 9.1050e-08, 9.3213e-08, 9.2929e-08, 1.0155e-07, 1.1263e-07, 
     & 1.2370e-07, 1.3636e-07, 1.5400e-07, 1.7656e-07, 2.1329e-07, 
     & 2.3045e-07, 2.5811e-07, 2.9261e-07, 3.4259e-07, 4.0770e-07, 
     & 4.8771e-07, 5.8081e-07, 7.2895e-07, 8.7482e-07, 1.0795e-06, 
     & 1.3384e-06, 1.7208e-06, 2.0677e-06, 2.5294e-06, 3.1123e-06, 
     & 3.7900e-06, 4.7752e-06, 5.6891e-06, 6.6261e-06, 7.6246e-06, 
     & 8.7730e-06, 9.6672e-06, 1.0980e-05, 1.1287e-05, 1.1670e-05, 
     & 1.1635e-05, 1.1768e-05, 1.2039e-05, 1.2253e-05, 1.3294e-05, 
     & 1.4005e-05, 1.3854e-05, 1.3420e-05, 1.3003e-05, 1.2645e-05, 
     & 1.1715e-05, 1.1258e-05, 1.1516e-05, 1.2494e-05, 1.3655e-05, 
     & 1.4931e-05, 1.4649e-05, 1.3857e-05, 1.3120e-05, 1.1791e-05, 
     & 1.0637e-05, 8.2760e-06, 6.5821e-06, 5.1959e-06, 4.0158e-06, 
     & 3.0131e-06, 2.0462e-06, 1.4853e-06, 1.0365e-06, 7.3938e-07/
      data f(401:500)/
     & 4.9752e-07, 3.4148e-07, 2.4992e-07, 1.8363e-07, 1.4591e-07, 
     & 1.1380e-07, 9.0588e-08, 7.3697e-08, 6.0252e-08, 5.1868e-08, 
     & 4.2660e-08, 3.6163e-08, 3.2512e-08, 2.9258e-08, 2.4238e-08, 
     & 2.1209e-08, 1.6362e-08, 1.3871e-08, 1.2355e-08, 9.6940e-09, 
     & 7.7735e-09, 6.2278e-09, 5.2282e-09, 4.3799e-09, 3.5545e-09, 
     & 2.7527e-09, 2.0950e-09, 1.6344e-09, 1.2689e-09, 1.0403e-09, 
     & 8.4880e-10, 6.3461e-10, 4.7657e-10, 3.5220e-10, 2.7879e-10, 
     & 2.3021e-10, 1.6167e-10, 1.1732e-10, 8.9206e-11, 7.0596e-11, 
     & 5.8310e-11, 4.4084e-11, 3.1534e-11, 2.5068e-11, 2.2088e-11, 
     & 2.2579e-11, 2.2637e-11, 2.5705e-11, 3.2415e-11, 4.6116e-11, 
     & 6.5346e-11, 9.4842e-11, 1.2809e-10, 1.8211e-10, 2.4052e-10, 
     & 3.0270e-10, 3.5531e-10, 4.2402e-10, 4.6730e-10, 4.7942e-10, 
     & 4.6813e-10, 4.5997e-10, 4.5788e-10, 4.0311e-10, 3.7367e-10, 
     & 3.3149e-10, 2.9281e-10, 2.5231e-10, 2.1152e-10, 1.9799e-10, 
     & 1.8636e-10, 1.9085e-10, 2.0786e-10, 2.2464e-10, 2.3785e-10, 
     & 2.5684e-10, 2.7499e-10, 2.6962e-10, 2.6378e-10, 2.6297e-10, 
     & 2.6903e-10, 2.7035e-10, 2.5394e-10, 2.5655e-10, 2.7184e-10, 
     & 2.9013e-10, 3.0585e-10, 3.0791e-10, 3.1667e-10, 3.4343e-10, 
     & 3.7365e-10, 4.0269e-10, 4.7260e-10, 5.6584e-10, 6.9791e-10, 
     & 8.6569e-10, 1.0393e-09, 1.2067e-09, 1.5047e-09, 1.8583e-09/
      data f(501:600)/
     & 2.2357e-09, 2.6498e-09, 3.2483e-09, 3.9927e-09, 4.6618e-09, 
     & 5.5555e-09, 6.6609e-09, 8.2139e-09, 1.0285e-08, 1.3919e-08, 
     & 1.8786e-08, 2.5150e-08, 3.3130e-08, 4.5442e-08, 6.3370e-08, 
     & 9.0628e-08, 1.2118e-07, 1.5927e-07, 2.1358e-07, 2.7825e-07, 
     & 3.7671e-07, 4.4894e-07, 5.4442e-07, 6.2240e-07, 7.3004e-07, 
     & 8.3384e-07, 8.7933e-07, 8.8080e-07, 8.6939e-07, 8.6541e-07, 
     & 8.2055e-07, 7.7278e-07, 7.5989e-07, 8.6909e-07, 9.7945e-07, 
     & 1.0394e-06, 1.0646e-06, 1.1509e-06, 1.2017e-06, 1.1915e-06, 
     & 1.1259e-06, 1.1549e-06, 1.1938e-06, 1.2356e-06, 1.2404e-06, 
     & 1.1716e-06, 1.1149e-06, 1.0073e-06, 8.9845e-07, 7.6639e-07, 
     & 6.1517e-07, 5.0887e-07, 4.1269e-07, 3.2474e-07, 2.5698e-07, 
     & 1.8893e-07, 1.4009e-07, 1.0340e-07, 7.7724e-08, 5.7302e-08, 
     & 4.2178e-08, 2.9603e-08, 2.1945e-08, 1.6301e-08, 1.2806e-08, 
     & 1.0048e-08, 7.8970e-09, 6.1133e-09, 4.9054e-09, 4.1985e-09, 
     & 3.6944e-09, 3.2586e-09, 2.7362e-09, 2.3647e-09, 2.1249e-09, 
     & 1.8172e-09, 1.6224e-09, 1.5158e-09, 1.2361e-09, 1.0682e-09, 
     & 9.2312e-10, 7.9220e-10, 6.8174e-10, 5.6147e-10, 4.8268e-10, 
     & 4.1534e-10, 3.3106e-10, 2.8275e-10, 2.4584e-10, 2.0742e-10, 
     & 1.7840e-10, 1.4664e-10, 1.2390e-10, 1.0497e-10, 8.5038e-11, 
     & 6.7008e-11, 5.6355e-11, 4.3323e-11, 3.6914e-11, 3.2262e-11/
      data f(601:700)/
     & 3.0749e-11, 3.0318e-11, 2.9447e-11, 2.9918e-11, 3.0668e-11, 
     & 3.1315e-11, 3.0329e-11, 2.8259e-11, 2.6065e-11, 2.3578e-11, 
     & 2.0469e-11, 1.6908e-11, 1.4912e-11, 1.1867e-11, 9.9730e-12, 
     & 8.1014e-12, 6.7528e-12, 6.3133e-12, 5.8599e-12, 6.0145e-12, 
     & 6.5105e-12, 7.0537e-12, 7.4973e-12, 7.8519e-12, 8.5039e-12, 
     & 9.1995e-12, 1.0694e-11, 1.1659e-11, 1.2685e-11, 1.3087e-11, 
     & 1.3222e-11, 1.2634e-11, 1.1077e-11, 9.6259e-12, 8.3202e-12, 
     & 7.4857e-12, 6.8069e-12, 6.7496e-12, 7.3116e-12, 8.0171e-12, 
     & 8.6394e-12, 9.2659e-12, 1.0048e-11, 1.0941e-11, 1.2226e-11, 
     & 1.3058e-11, 1.5193e-11, 1.8923e-11, 2.3334e-11, 2.8787e-11, 
     & 3.6693e-11, 4.8295e-11, 6.4260e-11, 8.8269e-11, 1.1865e-10, 
     & 1.5961e-10, 2.0605e-10, 2.7349e-10, 3.7193e-10, 4.8216e-10, 
     & 6.1966e-10, 7.7150e-10, 1.0195e-09, 1.2859e-09, 1.6535e-09, 
     & 2.0316e-09, 2.3913e-09, 3.0114e-09, 3.7495e-09, 4.6504e-09, 
     & 5.9145e-09, 7.6840e-09, 1.0304e-08, 1.3010e-08, 1.6441e-08, 
     & 2.1475e-08, 2.5892e-08, 2.9788e-08, 3.3820e-08, 4.0007e-08, 
     & 4.4888e-08, 4.5765e-08, 4.6131e-08, 4.6239e-08, 4.4849e-08, 
     & 4.0729e-08, 3.6856e-08, 3.6164e-08, 3.7606e-08, 4.1457e-08, 
     & 4.3750e-08, 5.1150e-08, 5.6054e-08, 6.1586e-08, 6.4521e-08, 
     & 6.6494e-08, 6.9024e-08, 6.8893e-08, 7.0901e-08, 6.9760e-08/
      data f(701:800)/
     & 7.1485e-08, 7.0740e-08, 7.3764e-08, 7.6618e-08, 8.4182e-08, 
     & 9.3838e-08, 1.0761e-07, 1.2851e-07, 1.4748e-07, 1.8407e-07, 
     & 2.2109e-07, 2.6392e-07, 2.9887e-07, 3.4493e-07, 4.0336e-07, 
     & 4.3551e-07, 4.9231e-07, 5.0728e-07, 5.3781e-07, 5.3285e-07, 
     & 5.4496e-07, 5.5707e-07, 5.6944e-07, 6.1123e-07, 6.4317e-07, 
     & 6.4581e-07, 6.1999e-07, 6.0191e-07, 5.7762e-07, 5.7241e-07, 
     & 5.7013e-07, 6.0160e-07, 6.6905e-07, 7.4095e-07, 8.2121e-07, 
     & 8.0947e-07, 7.6145e-07, 7.2193e-07, 6.3722e-07, 5.4316e-07, 
     & 4.2186e-07, 3.2528e-07, 2.5207e-07, 1.8213e-07, 1.2658e-07, 
     & 8.6746e-08, 6.0216e-08, 4.1122e-08, 2.8899e-08, 2.1740e-08, 
     & 1.7990e-08, 1.5593e-08, 1.3970e-08, 1.2238e-08, 1.0539e-08, 
     & 9.2386e-09, 7.8481e-09, 6.8704e-09, 5.7615e-09, 5.0434e-09, 
     & 4.6886e-09, 4.3770e-09, 3.9768e-09, 3.5202e-09, 3.1854e-09, 
     & 2.9009e-09, 2.5763e-09, 2.2135e-09, 1.9455e-09, 1.6248e-09, 
     & 1.3368e-09, 1.0842e-09, 8.4254e-10, 6.7414e-10, 5.4667e-10, 
     & 4.5005e-10, 3.4932e-10, 2.6745e-10, 2.2053e-10, 1.8162e-10, 
     & 1.4935e-10, 1.1618e-10, 9.1888e-11, 8.0672e-11, 6.8746e-11, 
     & 6.2668e-11, 5.5715e-11, 4.5074e-11, 3.7669e-11, 3.2082e-11, 
     & 2.8085e-11, 2.4838e-11, 1.9791e-11, 1.6964e-11, 1.3887e-11, 
     & 1.1179e-11, 9.7499e-12, 7.8255e-12, 6.3698e-12, 5.3265e-12/
      data f(801:900)/
     & 4.6588e-12, 4.4498e-12, 3.9984e-12, 3.7513e-12, 3.7176e-12, 
     & 3.9148e-12, 4.2702e-12, 5.0090e-12, 6.5801e-12, 8.7787e-12, 
     & 1.2718e-11, 1.8375e-11, 2.5304e-11, 3.5403e-11, 4.8842e-11, 
     & 6.4840e-11, 8.0911e-11, 1.0136e-10, 1.2311e-10, 1.4203e-10, 
     & 1.5869e-10, 1.8093e-10, 2.1370e-10, 2.5228e-10, 2.8816e-10, 
     & 3.4556e-10, 3.9860e-10, 4.4350e-10, 4.7760e-10, 5.2357e-10, 
     & 6.0827e-10, 6.3635e-10, 6.5886e-10, 6.8753e-10, 7.2349e-10, 
     & 7.2789e-10, 6.8232e-10, 6.6081e-10, 6.4232e-10, 6.3485e-10, 
     & 6.4311e-10, 7.2235e-10, 7.7263e-10, 8.1668e-10, 9.0324e-10, 
     & 9.7643e-10, 1.0535e-09, 1.0195e-09, 1.0194e-09, 1.0156e-09, 
     & 9.6792e-10, 9.2725e-10, 8.7347e-10, 8.4484e-10, 8.2647e-10, 
     & 8.4363e-10, 9.1261e-10, 1.0051e-09, 1.1511e-09, 1.4037e-09, 
     & 1.8066e-09, 2.4483e-09, 3.2739e-09, 4.3194e-09, 5.6902e-09, 
     & 7.7924e-09, 9.7376e-09, 1.2055e-08, 1.4303e-08, 1.6956e-08, 
     & 1.9542e-08, 2.2233e-08, 2.5186e-08, 2.7777e-08, 2.8943e-08, 
     & 2.8873e-08, 2.9417e-08, 2.7954e-08, 2.7524e-08, 2.7040e-08, 
     & 3.1254e-08, 3.6843e-08, 3.7797e-08, 3.8713e-08, 4.0135e-08, 
     & 4.2824e-08, 4.3004e-08, 4.0279e-08, 4.2781e-08, 4.5220e-08, 
     & 4.8948e-08, 5.0172e-08, 4.8499e-08, 4.7182e-08, 4.2204e-08, 
     & 3.7701e-08, 3.0972e-08, 2.4654e-08, 1.9543e-08, 1.4609e-08/
      data f(901:1000)/
     & 1.1171e-08, 8.3367e-09, 6.3791e-09, 5.0790e-09, 4.0655e-09, 
     & 3.3658e-09, 2.7882e-09, 2.4749e-09, 2.2287e-09, 2.0217e-09, 
     & 1.8191e-09, 1.5897e-09, 1.4191e-09, 1.2448e-09, 1.0884e-09, 
     & 9.3585e-10, 7.9429e-10, 7.3214e-10, 6.5008e-10, 5.7549e-10, 
     & 5.4300e-10, 4.7251e-10, 4.3451e-10, 3.8446e-10, 3.5589e-10, 
     & 3.4432e-10, 2.8209e-10, 2.4620e-10, 2.1278e-10, 1.8406e-10, 
     & 1.6314e-10, 1.3261e-10, 1.1696e-10, 9.6865e-11, 7.6814e-11, 
     & 6.6411e-11, 5.0903e-11, 4.0827e-11, 3.0476e-11, 2.3230e-11, 
     & 1.7707e-11, 1.3548e-11, 1.0719e-11, 9.3026e-12, 8.7967e-12, 
     & 8.3136e-12, 7.3918e-12, 6.5293e-12, 5.9243e-12, 5.3595e-12, 
     & 3.5266e-12, 2.2571e-12, 1.6150e-12, 1.1413e-12, 8.4998e-13, 
     & 7.0803e-13, 5.1747e-13, 4.0694e-13, 3.6528e-13, 3.3670e-13, 
     & 3.1341e-13, 2.9390e-13, 2.8680e-13, 3.1283e-13, 3.7294e-13, 
     & 5.0194e-13, 6.7919e-13, 1.0455e-12, 1.5230e-12, 2.3932e-12, 
     & 3.4231e-12, 5.0515e-12, 7.3193e-12, 9.9406e-12, 1.2193e-11, 
     & 1.4742e-11, 1.9269e-11, 2.1816e-11, 2.2750e-11, 2.2902e-11, 
     & 2.3888e-11, 2.4902e-11, 2.2160e-11, 2.0381e-11, 1.9903e-11, 
     & 2.0086e-11, 1.9304e-11, 2.0023e-11, 2.2244e-11, 2.5450e-11, 
     & 3.1228e-11, 3.4560e-11, 3.6923e-11, 3.7486e-11, 3.8124e-11, 
     & 3.8317e-11, 3.4737e-11, 3.3037e-11, 3.1724e-11, 2.9840e-11/
      data f(1001:1100)/
     & 2.8301e-11, 2.5857e-11, 2.3708e-11, 1.9452e-11, 1.6232e-11, 
     & 1.5174e-11, 1.4206e-11, 1.4408e-11, 1.5483e-11, 1.8642e-11, 
     & 2.3664e-11, 3.0181e-11, 4.0160e-11, 5.2287e-11, 7.2754e-11, 
     & 1.0511e-10, 1.4531e-10, 2.0998e-10, 2.6883e-10, 3.3082e-10, 
     & 4.2638e-10, 5.3132e-10, 6.3617e-10, 7.1413e-10, 8.5953e-10, 
     & 9.9715e-10, 1.0796e-09, 1.0978e-09, 1.1052e-09, 1.1095e-09, 
     & 1.0641e-09, 9.7881e-10, 9.6590e-10, 1.0332e-09, 1.1974e-09, 
     & 1.3612e-09, 1.5829e-09, 1.8655e-09, 2.1465e-09, 2.4779e-09, 
     & 2.7370e-09, 2.9915e-09, 3.3037e-09, 3.6347e-09, 3.9587e-09, 
     & 4.4701e-09, 5.0122e-09, 5.8044e-09, 6.1916e-09, 6.9613e-09, 
     & 7.7863e-09, 8.2820e-09, 9.4359e-09, 9.7387e-09, 1.0656e-08, 
     & 1.0746e-08, 1.1210e-08, 1.1905e-08, 1.2194e-08, 1.3145e-08, 
     & 1.3738e-08, 1.3634e-08, 1.3011e-08, 1.2511e-08, 1.1805e-08, 
     & 1.2159e-08, 1.2390e-08, 1.3625e-08, 1.5678e-08, 1.7886e-08, 
     & 1.9933e-08, 1.9865e-08, 1.9000e-08, 1.7812e-08, 1.5521e-08, 
     & 1.2593e-08, 9.5635e-09, 7.2987e-09, 5.2489e-09, 3.5673e-09, 
     & 2.4206e-09, 1.6977e-09, 1.2456e-09, 9.3744e-10, 7.8379e-10, 
     & 6.9960e-10, 6.6451e-10, 6.8521e-10, 7.4234e-10, 8.6658e-10, 
     & 9.4972e-10, 1.0791e-09, 1.2359e-09, 1.3363e-09, 1.5025e-09, 
     & 1.5368e-09, 1.6152e-09, 1.6184e-09, 1.6557e-09, 1.7035e-09/
      data f(1101:1200)/
     & 1.6916e-09, 1.7237e-09, 1.7175e-09, 1.6475e-09, 1.5335e-09, 
     & 1.4272e-09, 1.3282e-09, 1.3459e-09, 1.4028e-09, 1.5192e-09, 
     & 1.7068e-09, 1.9085e-09, 2.1318e-09, 2.1020e-09, 1.9942e-09, 
     & 1.8654e-09, 1.6391e-09, 1.3552e-09, 1.0186e-09, 7.8540e-10, 
     & 5.7022e-10, 3.9247e-10, 2.5441e-10, 1.6699e-10, 1.1132e-10, 
     & 6.8989e-11, 4.5255e-11, 3.1106e-11, 2.3161e-11, 1.7618e-11, 
     & 1.4380e-11, 1.1601e-11, 9.7148e-12, 8.4519e-12, 6.5392e-12, 
     & 5.4113e-12, 4.7624e-12, 4.0617e-12, 3.6173e-12, 2.8608e-12, 
     & 2.2724e-12, 1.7436e-12, 1.3424e-12, 1.0358e-12, 7.3064e-13, 
     & 5.4500e-13, 4.0551e-13, 2.8642e-13, 2.1831e-13, 1.6860e-13, 
     & 1.2086e-13, 1.0150e-13, 9.3550e-14, 8.4105e-14, 7.3051e-14, 
     & 6.9796e-14, 7.9949e-14, 1.0742e-13, 1.5639e-13, 2.1308e-13, 
     & 3.1226e-13, 4.6853e-13, 6.6917e-13, 1.0088e-12, 1.4824e-12, 
     & 2.2763e-12, 3.3917e-12, 4.4585e-12, 6.3187e-12, 8.4189e-12, 
     & 1.1302e-11, 1.3431e-11, 1.5679e-11, 1.9044e-11, 2.2463e-11, 
     & 2.3605e-11, 2.3619e-11, 2.3505e-11, 2.3805e-11, 2.2549e-11, 
     & 1.9304e-11, 1.8382e-11, 1.7795e-11, 1.8439e-11, 1.9146e-11, 
     & 2.1966e-11, 2.6109e-11, 3.1883e-11, 3.7872e-11, 4.3966e-11, 
     & 4.8789e-11, 5.3264e-11, 5.9705e-11, 6.3744e-11, 7.0163e-11, 
     & 7.9114e-11, 8.8287e-11, 9.9726e-11, 1.1498e-10, 1.3700e-10/
      data f(1201:1300)/
     & 1.6145e-10, 1.9913e-10, 2.2778e-10, 2.6216e-10, 2.9770e-10, 
     & 3.3405e-10, 3.7821e-10, 3.9552e-10, 4.1322e-10, 4.0293e-10, 
     & 4.0259e-10, 3.8853e-10, 3.7842e-10, 3.8551e-10, 4.4618e-10, 
     & 5.0527e-10, 5.0695e-10, 5.1216e-10, 5.1930e-10, 5.5794e-10, 
     & 5.3320e-10, 5.2008e-10, 5.6888e-10, 6.1883e-10, 6.9006e-10, 
     & 6.9505e-10, 6.6768e-10, 6.3290e-10, 5.6753e-10, 5.0327e-10, 
     & 3.9830e-10, 3.1147e-10, 2.4416e-10, 1.8860e-10, 1.3908e-10, 
     & 9.9156e-11, 7.3779e-11, 5.6048e-11, 4.2457e-11, 3.4505e-11, 
     & 2.9881e-11, 2.7865e-11, 2.8471e-11, 3.1065e-11, 3.4204e-11, 
     & 3.9140e-11, 4.3606e-11, 4.9075e-11, 5.3069e-11, 5.5236e-11, 
     & 5.5309e-11, 5.3832e-11, 5.3183e-11, 5.1783e-11, 5.2042e-11, 
     & 5.4422e-11, 5.5656e-11, 5.4409e-11, 5.2659e-11, 5.1696e-11, 
     & 5.1726e-11, 4.9003e-11, 4.9050e-11, 5.1700e-11, 5.6818e-11, 
     & 6.3129e-11, 6.6542e-11, 6.4367e-11, 5.9908e-11, 5.4470e-11, 
     & 4.7903e-11, 3.9669e-11, 2.9651e-11, 2.2286e-11, 1.6742e-11, 
     & 1.1827e-11, 7.7739e-12, 4.8805e-12, 3.1747e-12, 2.0057e-12, 
     & 1.2550e-12, 8.7434e-13, 6.2755e-13, 4.9752e-13, 4.0047e-13, 
     & 3.5602e-13, 3.0930e-13, 2.4903e-13, 1.9316e-13, 1.4995e-13, 
     & 1.2059e-13, 8.7242e-14, 6.4511e-14, 5.3300e-14, 4.3741e-14, 
     & 3.4916e-14, 2.6560e-14, 1.6923e-14, 1.1816e-14, 6.7071e-15/
      data f(1301:1400)/
     & 3.6474e-15, 2.0686e-15, 1.1925e-15, 6.8948e-16, 3.9661e-16, 
     & 2.2576e-16, 1.2669e-16, 6.9908e-17, 3.7896e-17, 2.0280e-17, 
     & 1.1016e-17, 6.7816e-18, 6.0958e-18, 8.9913e-18, 1.7201e-17, 
     & 3.4964e-17, 7.0722e-17, 1.4020e-16, 2.7167e-16, 5.1478e-16, 
     & 9.5500e-16, 1.7376e-15, 3.1074e-15, 5.4789e-15, 9.5640e-15, 
     & 1.6635e-14, 2.9145e-14, 5.2179e-14, 8.8554e-14, 1.4764e-13, 
     & 2.3331e-13, 3.5996e-13, 5.2132e-13, 6.3519e-13, 7.3174e-13, 
     & 8.3752e-13, 9.8916e-13, 1.1515e-12, 1.4034e-12, 1.6594e-12, 
     & 2.1021e-12, 2.7416e-12, 3.4135e-12, 4.5517e-12, 5.5832e-12, 
     & 7.2303e-12, 9.9484e-12, 1.2724e-11, 1.6478e-11, 2.0588e-11, 
     & 2.5543e-11, 3.3625e-11, 4.1788e-11, 5.0081e-11, 6.0144e-11, 
     & 6.9599e-11, 8.4408e-11, 9.7143e-11, 1.0805e-10, 1.1713e-10, 
     & 1.2711e-10, 1.3727e-10, 1.4539e-10, 1.6049e-10, 1.7680e-10, 
     & 2.0557e-10, 2.4967e-10, 3.0096e-10, 3.5816e-10, 4.0851e-10, 
     & 4.6111e-10, 5.2197e-10, 5.5043e-10, 6.0324e-10, 6.4983e-10, 
     & 6.7498e-10, 7.0545e-10, 7.0680e-10, 7.5218e-10, 7.5723e-10, 
     & 7.7840e-10, 8.0081e-10, 8.0223e-10, 7.7271e-10, 7.1676e-10, 
     & 6.7819e-10, 6.4753e-10, 6.5844e-10, 7.0163e-10, 7.7503e-10, 
     & 8.8152e-10, 9.9022e-10, 1.0229e-09, 9.9296e-10, 8.9911e-10, 
     & 7.7813e-10, 6.3785e-10, 4.7491e-10, 3.5280e-10, 2.4349e-10/
      data f(1401:1500)/
     & 1.6502e-10, 1.1622e-10, 8.6715e-11, 6.7360e-11, 5.3910e-11, 
     & 4.5554e-11, 4.1300e-11, 3.9728e-11, 3.9000e-11, 3.9803e-11, 
     & 4.1514e-11, 4.3374e-11, 4.6831e-11, 4.8921e-11, 5.1995e-11, 
     & 5.7242e-11, 6.2759e-11, 7.0801e-11, 7.4555e-11, 7.9754e-11, 
     & 8.7616e-11, 9.1171e-11, 1.0349e-10, 1.1047e-10, 1.2024e-10, 
     & 1.2990e-10, 1.3725e-10, 1.5005e-10, 1.5268e-10, 1.5535e-10, 
     & 1.5623e-10, 1.5009e-10, 1.4034e-10, 1.3002e-10, 1.2225e-10, 
     & 1.1989e-10, 1.2411e-10, 1.3612e-10, 1.5225e-10, 1.7202e-10, 
     & 