# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hyperstate', 'hyperstate.schema']

package_data = \
{'': ['*']}

install_requires = \
['click>=8.0.3,<9.0.0',
 'dill>=0.3.4,<0.4.0',
 'docstring-parser>=0.13,<0.14',
 'msgpack-numpy>=0.4.7,<0.5.0',
 'msgpack>=1.0.3,<2.0.0',
 'python-ron>=0.2.1,<0.3.0']

setup_kwargs = {
    'name': 'hyperstate',
    'version': '0.3.9',
    'description': 'Library for managing hyperparameters and mutable state of machine learning training systems.',
    'long_description': '# HyperState\n\n[![PyPI](https://img.shields.io/pypi/v/hyperstate.svg?style=flat-square)](https://pypi.org/project/hyperstate/)\n\nOpinionated library for managing hyperparameter configs and mutable program state of machine learning training systems.\n\n**Key Features**:\n- (De)serialize nested Python dataclasses as [Rusty Object Notation](https://github.com/ron-rs/ron)\n- Override any config value from the command line\n- Automatic checkpointing and restoration of full program state\n- Checkpoints are (partially) human readable and can be modified in a text editor\n- Powerful tools for versioning and schema evolution that can detect breaking changes and make it easy to restructure your program while remaining backwards compatible with old checkpoints\n- Large binary objects in checkpoints can be loaded lazily only when accessed\n- DSL for hyperparameter schedules \n- (planned) Edit hyperparameters of running experiments on the fly without restarts\n- (planned) Usable without fermented vegetables\n\n## Quick start guide\n\nAll you need to use HyperState is a (nested) dataclass for your hyperparameters:\n\n```python\nfrom dataclasses import dataclass\n\n\n@dataclass\nclass OptimizerConfig:\n    lr: float = 0.003\n    batch_size: int = 512\n\n\n@dataclass\nclass NetConfig:\n    hidden_size: int = 128\n    num_layers: int = 2\n\n\n@dataclass\nclass Config:\n    optimizer: OptimizerConfig\n    net: NetConfig\n    steps: int = 100\n```\n\nThe `hyperstate.load` function can load values from a config file and/or apply specific overrides from the command line.\n\n```python\nimport argparse\nimport hyperstate\n\nif __name__ == "__main__":\n    parser = argparse.ArgumentParser()\n    parser.add_argument("--config", type=str, default=None, help="Path to config file")\n    parser.add_argument("--hps", nargs="+", help="Override hyperparameter value")\n    args = parser.parse_args()\n    config = hyperstate.load(Config, path=args.config, overrides=args.hps)\n    print(config)\n```\n\n```shell\n$ python main.py --hps net.num_layers=96 steps=50\nConfig(optimizer=OptimizerConfig(lr=0.003, batch_size=512), net=NetConfig(hidden_size=128, num_layers=96), steps=50)\n```\n\n```shell\n$ cat config.ron\nConfig(\n    optimizer: (\n        lr: 0.05,\n        batch_size: 4096,\n    ),\n)\n$ python main.py --config=config.ron\nConfig(optimizer=OptimizerConfig(lr=0.05, batch_size=4096), net=NetConfig(hidden_size=128, num_layers=2), steps=100)\n```\n\nThe full code for this example can be found in [examples/basic-config](examples/basic-config).\n\nLearn more about:\n- [Configs](#configs)\n- [Versioning and schema evolution](#versioning)\n- [Serializing complex objects](#unstable-feature-serializable)\n- [Checkpointing and schedules](#unstable-feature-hyperstate)\n- [Example application](examples/mnist)\n\n## Configs\n\nHyperState supports a strictly typed subset of Python objects:\n- dataclasses\n- containers: `Dict`, `List`, `Tuple`, `Optional`\n- primitives: `int`, `float`, `str`, `Enum`\n- objects with custom serialization logic: [`hyperstate.Serializable`](#serializable)\n\nUse `hyperstate.dump` to serialize configs.\nThe second argument to `dump` is a path to a file, and can be omitted to return the serialized config as a string instead of saving it to a file:\n\n```python\n>>> print(hyperstate.dump(Config(lr=0.1, batch_size=256))\nConfig(\n    lr: 0.1,\n    batch_size: 256,\n)\n```\n\nUse `hyperstate.load` to deserialize configs.\nThe `load` method takes the type of the config as the first argugment, and allows you to optionally specify the path to a config file and/or a `List[str]` of overrides:\n\n```python\n@dataclass\nclass OptimizerConfig:\n    lr: float\n    batch_size: int\n\n@dataclass\nclass Config:\n    optimzer: OptimizerConfig\n    steps: int\n\n\nconfig = hyperstate.load(Config, path="config.ron", overrides=["optimizer.lr=0.1", "steps=100"])\n```\n\n## Versioning\n\nVersioning allows you to modify your `Config` class while still remaining compatible with checkpoints recorded at previous version.\nTo benefit from versionining, your config must inherit `hyperstate.Versioned` and implement its `version` function:\n\n```python\n@dataclass\nclass Config(hyperstate.Versioned):\n    lr: float\n    batch_size: int\n    \n    @classmethod\n    def version(clz) -> int:\n        return 0\n```\n\nWhen serializing the config, hyperstate will now record an additional `version` field with the value of the current version.\nAny snapshots that contain configs without a version field are assumed to have a version of `0`.\n\n### `RewriteRule`\n\nNow suppose you modify your `Config` class, e.g. by renaming the `lr` field to `learning_rate`.\nTo still be able to load old configs that are using `lr` instead of `learning_rate`, you increase the `version` to `1` and add an entry to the dictionary returned by `upgrade_rules` that tells HyperState to change `lr` to `learning_rate` when upgrading configs from version `0`.\n\n```python\nfrom dataclasses import dataclass\nfrom typing import Dict, List\nfrom hyperstate import Versioned\nfrom hyperstate.schema.rewrite_rule import RenameField, RewriteRule\n\n@dataclass\nclass Config(Versioned):\n    learning_rate: float\n    batch_size: int\n    \n    @classmethod\n    def version(clz) -> int:\n        return 1\n\n    @classmethod\n    def upgrade_rules(clz) -> Dict[int, List[RewriteRule]]:\n        """\n        Returns a list of rewrite rules that can be applied to the given version\n        to make it compatible with the next version.\n        """\n        return {\n            0: [RenameField(old_field=("lr",), new_field=("learning_rate",))],\n        }\n```\n\nIn the majority of cases, you don\'t actually have to manually write out `RewriteRule`s.\nInstead, they are generated for you automatically by the [Schema Evolution CLI](#schema-evolution-cli).\n\n### Schema evolution CLI\n\nHyperState comes with a command line tool for managing changes to your config schema.\nTo access the CLI, simply add the following code to the Python file defining your config:\n\n```python\n# config.py\nfrom hyperstate import schema_evolution_cli\n\nif __name__ == "__main__":\n    schema_evolution_cli(Config)\n```\n\nRun `python config.py` to see a list of available commands, described in more detail below.\n\n#### `dump-schema`\n\nThe `dump-schema` command creates a file describing the schema of your config.\nThis file should commited to version control, and is used to detect changes to the config schema and perform automatic upgrades.\n\n#### `check-schema`\n\nThe `check-schema` command compares your config class to a schema file and detects any backwards incompatible changes.\nIt also emits a suggested list of [`RewriteRule`](#rewrite-rule)s that can upgrade old configs to the new schema.\nHyperState does not always guess the correct `RewriteRule`s so you still need to check that they are correct.\n\n```\n$ python config.py check-schema\nWARN  field renamed to learning_rate: lr\nWARN  schema changed but version identical\nSchema incompatible\n\nProposed mitigations\n- add upgrade rules:\n    0: [\n        RenameField(old_field=(\'lr\',), new_field=(\'learning_rate\',)),\n    ],\n- bump version to 1\n```\n\n#### `upgrade-schema`\n\nThe `upgrade-schema` command functions much the same as `check-schema`, but also updates your schema config files once all backwards-incompatability issues have been address.\n\n#### `upgrade-config`\n\nThe `upgrade-config` command takes a list of paths to config files, and upgrades them to the latest version.\n\n### Automated Tests\n\nTo prevent accidental backwards-incompatible modifications of your `Config` class, you can use the following code as an automated test that checks your config `Class` against a schema file created with [`dump-schema`](#dump-schema): \n\n```python\nfrom hyperstate.schema.schema_change import Severity\nfrom hyperstate.schema.schema_checker import SchemaChecker\nfrom hyperstate.schema.types import load_schema\nfrom config import Config\n\ndef test_schema():\n    old = load_schema("config-schema.ron")\n    checker = SchemaChecker(old, Config)\n    if checker.severity() >= Severity.WARN:\n        checker.print_report()\n    assert checker.severity() == Severity.INFO\n```\n\n## _[unstable feature]_ `Serializable`\n\nYou can define custom serialization logic for a class by inheriting from `hyperstate.Serializable` and implementing the `serialize` and `deserialize` methods.\n\n```python\nfrom dataclasses import dataclass\n\nimport torch\nimport torch.nn as nn\nimport torch.nn.functional as F\nimport hyperstate\n\n@dataclass\nclass Config:\n   inputs: int\n\nclass LinearRegression(nn.Module, hyperstate.Serializable):\n    def __init__(self, inputs):\n        super(Net, self).__init__()\n        self.fc1 = nn.Linear(inputs, 1)\n        \n    def forward(self, x):\n        return self.fc1(x)\n    \n    # `serialize` should return a representation of the object consisting only of\n    # primitives, containers, numpy arrays and torch tensors.\n    def serialize(self) -> Any:\n        return self.state_dict()\n\n    # `deserialize` should take a serialized representation of the object and\n    # return an instance of the class. The `ctx` argument allows you to pass\n    # additional information to the deserialization function.\n    @classmethod\n    def deserialize(clz, state_dict, ctx):\n        net = clz(ctx["config"].inputs)\n        return net.load_state_dict(state_dict)\n\n@dataclass\nclass State:\n    net: LinearRegression\n\nconfig = hyperstate.load("config.ron")\nstate = hyperstate.load("state.ron", ctx={"config": config})\n```\n\nObjects that implement `Serializable` are stored in separate files using a binary encoding.\nIn the above example, calling `hyperstate.dump(state, "checkpoint/state.ron")` will result in the following file structure:\n\n```\ncheckpoint\n├── state.net.blob\n└── state.ron\n```\n\n### _[unstable feature]_ `Lazy`\n\nIf you inherit from `hyperstate.Lazy`, any fields with `Serializable` types will only be loaded/deserialized when accessed. If the `.blob` file for a field is missing, HyperState will not raise an error unless the corresponding field is accessed.\n\n### _[unstable feature]_`blob`\n\nTo include objects in your state that do not directly implement `hyperstate.Serializable`, you can seperately implement `hyperstate.Serializable` and use the `blob` function to mix in the `Serializable` implementation:\n\n```python\nimport torch.optim as optim\nimport torch.nn as nn\nimport hyperstate\n\nclass SerializableOptimizer(hyperstate.Serializable):\n    def serialize(self):\n        return self.state_dict()\n\n    @classmethod\n    def deserialize(clz, state_dict: Any, config: Config, state: "State") -> optim.Optimizer:\n        optimizer = blob(optim.SerializableAdam, mixin=SerializableOptimizer)(state.net.parameters())\n        optimizer.load_state_dict(state_dict)\n        return optimizer\n\n@dataclass\nclass State(hyperstate.Lazy):\n    net: nn.Module\n    optimizer: blob(Adam, mixin=SerializableOptimizer)\n```\n\n\n\n## _[unstable feature]_ `HyperState`\n\nTo unlock the full power of HyperState, you must inherit from the `HyperState` class.\nThis class combines an immutable config and mutable state, and provides automatic checkpointing, hyperparameter schedules, and the on-the-fly changes to the config and state (not implemented yet).\n\n```python\nfrom dataclasses import dataclass\n\nimport torch\nimport torch.nn as nn\nimport torch.nn.functional as F\nimport hyperstate\n\n@dataclass\nclass Config:\n   inputs: int\n   steps: int\n\nclass LinearRegression(nn.Module, hyperstate.Serializable):\n    def __init__(self, inputs):\n        super(Net, self).__init__()\n        self.fc1 = nn.Linear(inputs, 1)\n    def forward(self, x):\n        return self.fc1(x)\n    def serialize(self) -> Any:\n        return self.state_dict()\n    @classmethod\n    def deserialize(clz, state_dict, ctx):\n        net = clz(ctx["config"].inputs)\n        return net.load_state_dict(state_dict)\n\n@dataclass\nclass State:\n    net: LinearRegression\n    step: int\n\n\nclass Trainer(HyperState[Config, State]):\n    def __init__(\n        self,\n        # Path to the config file\n        initial_config: str,\n        # Optional path to the checkpoint directory, which enables automatic checkpointing.\n        # If any checkpoint files are present, they will be used to initialize the state.\n        checkpoint_dir: Optional[str] = None,\n        # List of manually specified config overrides.\n        config_overrides: Optional[List[str]] = None,\n    ):\n        super().__init__(Config, State, initial_config, checkpoint_dir, overrides=config_overrides)\n\n    def initial_state(self) -> State:\n        """\n        This function is called to initialize the state if no checkpoint files are found.\n        """\n        return State(net=LinearRegression(self.config.inputs))\n\n    def train(self) -> None:\n        for step in range(self.state.step, self.config.steps):\n            # training code...\n\n            self.state.step = step\n            # At the end of each iteration, call `self.step()` to checkpoint the state and apply hyperparameter schedules.\n            self.step()\n```\n\n### _[unstable feature]_ Checkpointing\n\nWhen using the `HyperState` object, the config and state are automatically checkpointed to the configured directory when calling the `step` method.\n\n### _[unstable feature]_ Schedules\n\nAny `int`/`float` fields in the config can also be set to a schedule that will be updated at each step.\nFor example, the following config defines a schedule that linearly decays the learning rate from 1.0 to 0.1 over 1000 steps:\n\n```rust\nConfig(\n    lr: Schedule(\n      key: "state.step",\n      schedule: [\n        (0, 1.0),\n        "lin",\n        (1000, 0.1),\n      ],\n    ),\n    batch_size: 256,\n)\n```\n\nWhen you call `step()`, all config values that are schedules will be updated.\n\n\n## License\n\nHyperState is dual-licensed under the MIT license and Apache License (Version 2.0).\n\nSee LICENSE-MIT and LICENSE-APACHE for more information.\n',
    'author': 'Clemens Winter',
    'author_email': 'clemenswinter1@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
