/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { LambdaContext } from '../lib/aws-lambda';
import { CfnRequestEvent, SimpleCustomResource } from '../lib/custom-resource';
import { VersionProvider } from '../lib/version-provider';
/**
 * The input to this Custom Resource
 */
export interface IVersionProviderResourceProperties {
    /**
     * The version of Deadline to look up.
     */
    readonly versionString?: string;
    /**
     * A random string that can be used to force the Lambda to run again which is intended to allow for Deadline updates
     * to be discovered.
     */
    readonly forceRun?: string;
}
/**
 * Output of this Custom Resource, it is all made up of a flat structure of strings to avoid issues with how the
 * results are tokenized by CDK to be used in constructs.
 */
export interface FlatVersionedUriOutput {
    /**
     * The S3 bucket holding the installers. This makes the assumption they are all in the same bucket.
     */
    readonly S3Bucket: string;
    /**
     * The major version of the Linux installer. For example, "a" in "a.b.c.d".
     */
    readonly MajorVersion: string;
    /**
     * The minor version of the Linux installer. For example, "b" in "a.b.c.d".
     */
    readonly MinorVersion: string;
    /**
     * The release version of the Linux installer. For example, "c" in "a.b.c.d".
     */
    readonly ReleaseVersion: string;
    /**
     * The patch version of the Linux installer. For example, "d" in "a.b.c.d".
     */
    readonly LinuxPatchVersion: string;
    /**
     * The object key of the Deadline repository installer for Linux.
     */
    readonly LinuxRepositoryInstaller: string;
    /**
     * The object key of the Deadline client installer for Linux.
     */
    readonly LinuxClientInstaller: string;
}
/**
 * This custom resource will parse and return the S3 URI's of the Deadline installers and Docker recipes for use in
 * any constructs that need to install Deadline.
 */
export declare class VersionProviderResource extends SimpleCustomResource {
    readonly versionProvider: VersionProvider;
    constructor();
    /**
     * @inheritdoc
     */
    validateInput(data: object): boolean;
    /**
     * @inheritdoc
     */
    doCreate(physicalId: string, resourceProperties: IVersionProviderResourceProperties): Promise<FlatVersionedUriOutput>;
    /**
     * @inheritdoc
     */
    doDelete(physicalId: string, resourceProperties: IVersionProviderResourceProperties): Promise<void>;
    private implementsIVersionProviderResourceProperties;
    /**
     * Parses the S3 bucket name from an S3 URI.
     */
    private parseS3BucketName;
    /**
     * Parses the S3 object key from an S3 URI.
     */
    private parseS3ObjectKey;
    private findRegex;
}
/**
 * The handler used to provide the installer links for the requested version
 */
export declare function handler(event: CfnRequestEvent, context: LambdaContext): Promise<string>;
