"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.writePaddingFile = exports.determineNextSequentialFilename = exports.diskUsage = exports.listNumberedFiles = exports.nextSequentialFile = exports.ensureIsDirectory = void 0;
/* eslint-disable no-console */
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const util_1 = require("util");
/**
 * Verifies that the path given in the argument exists and is a directory.
 * @param location
 */
async function ensureIsDirectory(location) {
    try {
        const stat = await fs_1.promises.stat(location);
        if (!stat.isDirectory()) {
            throw Error(`Must be a directory: ${location}`);
        }
    }
    catch (err) {
        throw Error(`No such file or directory: ${location}`);
    }
}
exports.ensureIsDirectory = ensureIsDirectory;
/**
 * Given a filename that is assumed to be numeric, return the next numeric
 * filename in increasing order padded out to 5 digits.
 * @param filename
 * @returns
 */
function nextSequentialFile(filename) {
    const value = parseInt(filename);
    return (value + 1).toString().padStart(5, '0');
}
exports.nextSequentialFile = nextSequentialFile;
/**
 * List all of the names in the given directory that are numeric.
 * @param location Path of the directory to list. Assumed to be a directory.
 * @returns Array of the names of numeric contents in the directory, sorted into increasing order.
 */
async function listNumberedFiles(location) {
    const dirContents = await fs_1.promises.readdir(location);
    const digitsRegex = /^\d+$/;
    const numericFiles = dirContents.filter(name => digitsRegex.test(name)).sort();
    return numericFiles;
}
exports.listNumberedFiles = listNumberedFiles;
/**
 * Invoke "du -sh -BM" on the given location, to determine the total size in MB stored
 * in the filesystem location.
 * @param location Directory location.
 * @returns Filesystem size under the location, in MiB.
 */
async function diskUsage(location) {
    await ensureIsDirectory(location);
    const execPromise = util_1.promisify(child_process_1.exec);
    const { stdout, stderr } = await execPromise(`/usr/bin/du -sh -BM ${location}`);
    if (stderr) {
        throw Error(stderr);
    }
    // stdout will be: <number>M\t<location>\n
    const size = parseInt(stdout);
    if (isNaN(size)) {
        throw Error(`Unexpected error. Could not parse size of directory ${location}: ${stdout}`);
    }
    return size;
}
exports.diskUsage = diskUsage;
/**
 * Inspect the filenames in the given directory location, and return the next highest numeric
 * filename that does not already exist.
 * e.g.
 *  i) Empty dir -> 00000
 *  ii) '00000', '00002' -> '00003'
 * @param location Directory to inspect.
 * @returns
 */
async function determineNextSequentialFilename(location) {
    const numericFiles = await listNumberedFiles(location);
    if (numericFiles.length == 0) {
        return '00000';
    }
    return nextSequentialFile(numericFiles[numericFiles.length - 1]);
}
exports.determineNextSequentialFilename = determineNextSequentialFilename;
/**
 * Writes a file of zeroes to the given location.
 * @param filename Name of the file to create.
 * @param filesize Size of the file in MiB. Must be a multiple of 10.
 */
async function writePaddingFile(filename, filesize) {
    const execPromise = util_1.promisify(child_process_1.exec);
    const numberOfBlocks = filesize / 32;
    const command = `dd if=/dev/zero of=${filename} bs=32M count=${numberOfBlocks}`;
    console.log(`Writing ${filesize}MiB to ${filename}: ${command}`);
    const { stderr } = await execPromise(command);
    console.log(stderr);
}
exports.writePaddingFile = writePaddingFile;
//# sourceMappingURL=data:application/json;base64,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