"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const crypto_1 = require("crypto");
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const dynalite = require("dynalite");
const sinon_1 = require("sinon");
const composite_table_1 = require("../composite-table");
// Enable/disable debugging statements.
const DEBUG = false;
if (!DEBUG) {
    console.debug = () => { };
}
// Test class to expose constructor and tableName
class TestTable extends composite_table_1.CompositeStringIndexTable {
    constructor(client, name, primaryKey, sortKey) {
        super(client, name, primaryKey, sortKey);
    }
}
/**
 * This class is being refactored to use mocks over an actualy connection to DynamoDB. All the DynamoDB tests were
 * converted to use dynalite, a package that sets up a server to replicate DynamoDB locally. These tests can be found
 * in the first describe() block. Newer tests have been written using aws-sdk-mock and found in the describe() block
 * that follows. Any updates to these tests should translate dynalite tests to aws-sdk-mocks and new tests should use
 * aws-sdk-mock.
 */
describe('Tests using dynalite', () => {
    const SUFFIX = crypto_1.randomBytes(8).toString('hex');
    const GOOD_TABLE_NAME = `RFDKTestGood-${SUFFIX}`;
    const BAD_TABLE1_NAME = `RFDKTestBad1-${SUFFIX}`; // Missing the sort key
    const BAD_TABLE2_NAME = `RFDKTestBad2-${SUFFIX}`; // Incorrect type on primary key
    const BAD_TABLE3_NAME = `RFDKTestBad3-${SUFFIX}`; // Incorrect type on sort key
    const dynaliteServer = dynalite({
        createTableMs: 5,
        deleteTableMs: 5,
        updateTableMs: 5,
    });
    let dynamoClient;
    beforeAll(async () => {
        const dynaPort = 43266;
        await dynaliteServer.listen(dynaPort, (err) => {
            if (err) {
                throw err;
            }
        });
        dynamoClient = new AWS.DynamoDB({
            credentials: new AWS.Credentials({
                accessKeyId: '',
                secretAccessKey: '',
            }),
            endpoint: `http://localhost:${dynaPort}`,
            region: 'us-west-2',
        });
        function createTableRequest(tableName, primaryKeyType, sortKey) {
            const request = {
                TableName: tableName,
                AttributeDefinitions: [
                    {
                        AttributeName: 'PrimKey',
                        AttributeType: primaryKeyType,
                    },
                ],
                KeySchema: [
                    {
                        AttributeName: 'PrimKey',
                        KeyType: 'HASH',
                    },
                ],
                BillingMode: 'PAY_PER_REQUEST',
                Tags: [
                    {
                        Key: 'RFDKTesting',
                        Value: 'RFDKTesting',
                    },
                ],
            };
            if (sortKey) {
                request.AttributeDefinitions.push({
                    AttributeName: 'SortKey',
                    AttributeType: sortKey.KeyType,
                });
                request.KeySchema.push({
                    AttributeName: 'SortKey',
                    KeyType: 'RANGE',
                });
            }
            return request;
        }
        let request = createTableRequest(GOOD_TABLE_NAME, 'S', { KeyType: 'S' });
        let response = await dynamoClient.createTable(request).promise();
        let table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${GOOD_TABLE_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        request = createTableRequest(BAD_TABLE1_NAME, 'S');
        response = await dynamoClient.createTable(request).promise();
        table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${BAD_TABLE1_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        request = createTableRequest(BAD_TABLE2_NAME, 'N', { KeyType: 'S' });
        response = await dynamoClient.createTable(request).promise();
        table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${BAD_TABLE2_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        request = createTableRequest(BAD_TABLE3_NAME, 'S', { KeyType: 'N' });
        response = await dynamoClient.createTable(request).promise();
        table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${BAD_TABLE3_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        let waiting = true;
        do {
            const promises = [];
            for (const name of [GOOD_TABLE_NAME, BAD_TABLE1_NAME, BAD_TABLE2_NAME, BAD_TABLE3_NAME]) {
                promises.push(dynamoClient.describeTable({
                    TableName: name,
                }).promise());
            }
            const responses = await Promise.all(promises);
            waiting = !responses.every(item => { var _a; return ((_a = item.Table) === null || _a === void 0 ? void 0 : _a.TableStatus) === 'ACTIVE'; });
            await new Promise(resolve => setTimeout(resolve, 2000));
        } while (waiting);
    }, 60000);
    afterAll(async () => {
        await dynaliteServer.close();
    });
    test('delete table ResourceNotFoundException', async () => {
        const tableName = 'NonExistantTable';
        const pk = 'PrimKey';
        const sk = 'SortKey';
        const table = new TestTable(dynamoClient, tableName, pk, sk);
        expect(table.tableName).toBeDefined();
        await expect(table.deleteTable()).resolves.not.toThrowError();
        expect(table.tableName).toBeUndefined();
    });
    describe('fromExisting fails on bad table name', () => {
        test.each([
            [BAD_TABLE1_NAME],
            [BAD_TABLE2_NAME],
            [BAD_TABLE3_NAME],
        ])('tableName = %p', async (tableName) => {
            // WHEN
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, tableName))
                // THEN
                .rejects
                .toThrow();
        });
    });
    test('fromExising succeeds on good table name', async () => {
        // WHEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // THEN
        expect(table.primaryKey).toBe('PrimKey');
        expect(table.sortKey).toBe('SortKey');
    });
    test('putItem/getItem success', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const attributes = {
            TestAttribute: 'test value',
        };
        const putRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
            attributes,
        };
        const getRequest = {
            primaryKeyValue: putRequest.primaryKeyValue,
            sortKeyValue: putRequest.sortKeyValue,
        };
        // THEN
        await expect(table.putItem(putRequest)).resolves.toBe(true);
        await expect(table.getItem(getRequest)).resolves.toMatchObject(attributes);
    });
    test.each([
        [true], [false],
    ])('putItem overwrite %p', async (allowOverwrite) => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const putRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
            attributes: {
                TestAttribute: 'test value',
            },
            allow_overwrite: true,
        };
        await table.putItem(putRequest);
        // THEN
        putRequest.allow_overwrite = allowOverwrite;
        await expect(table.putItem(putRequest)).resolves.toBe(allowOverwrite);
    });
    test('getItem no match', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const getRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
        };
        // THEN
        await expect(table.getItem(getRequest)).resolves.toBeUndefined();
    });
    test('deleteItem success', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const attributes = {
            TestAttribute: 'test value',
        };
        const putRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
            attributes,
        };
        const deleteRequest = {
            primaryKeyValue: putRequest.primaryKeyValue,
            sortKeyValue: putRequest.sortKeyValue,
        };
        await table.putItem(putRequest);
        // THEN
        await expect(table.deleteItem(deleteRequest)).resolves.toBe(true);
    });
    test('deleteItem no match', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const deleteRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
        };
        // THEN
        await expect(table.deleteItem(deleteRequest)).resolves.toBe(false);
    });
    test('query success', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const primaryKeyValue = crypto_1.randomBytes(24).toString('hex');
        const expected = {};
        for (let i = 0; i < 20; i++) {
            const sortKeyValue = crypto_1.randomBytes(24).toString('hex');
            expected[sortKeyValue] = {
                TestAttribute: `value${i}`,
            };
        }
        for (const [key, value] of Object.entries(expected)) {
            const putRequest = {
                primaryKeyValue,
                sortKeyValue: key,
                attributes: value,
            };
            await table.putItem(putRequest);
        }
        // THEN
        await expect(table.query(primaryKeyValue, 5)).resolves.toMatchObject(expected);
    });
});
describe('Tests using aws-sdk-mock', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('DynamoDB');
    });
    describe('fromExisting tests', () => {
        test('Table not found', async () => {
            const callback = sinon_1.fake.resolves({ Table: undefined });
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', callback);
            const client = new AWS.DynamoDB();
            const tableName = 'Nonexistant';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not describeTable for Table '${tableName}'`);
        });
        test('KeySchema not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({ Table: { KeySchema: undefined } }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not get KeySchema for Table '${tableName}'`);
        });
        test('AttributeDefinitions not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [],
                    AttributeDefinitions: undefined,
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not get Attributes for Table '${tableName}'`);
        });
        test('PrimaryKey not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'SortKey',
                            KeyType: 'RANGE',
                        },
                    ],
                    AttributeDefinitions: {},
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not find PrimaryKey of Table '${tableName}'`);
        });
        test('SortKey not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'PrimaryKey',
                            KeyType: 'HASH',
                        },
                    ],
                    AttributeDefinitions: {},
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not find SortKey of Table '${tableName}'`);
        });
        test('PrimaryKey AttributeDefinition not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'PrimaryKey',
                            KeyType: 'HASH',
                        },
                        {
                            AttributeName: 'SortKey',
                            KeyType: 'RANGE',
                        },
                    ],
                    AttributeDefinitions: [
                        {
                            AttributeName: 'SortKey',
                            AttributeType: 'S',
                        },
                    ],
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError("Primary key 'PrimaryKey' must be string type");
        });
        test('SortKey AttributeDefinition not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'PrimaryKey',
                            KeyType: 'HASH',
                        },
                        {
                            AttributeName: 'SortKey',
                            KeyType: 'RANGE',
                        },
                    ],
                    AttributeDefinitions: [
                        {
                            AttributeName: 'PrimaryKey',
                            AttributeType: 'S',
                        },
                    ],
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError("Sort key 'SortKey' must be string type");
        });
    });
    describe('createNew tests', () => {
        test('DynamoDB.createTable() failure throws Error', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'createTable', sinon_1.fake.rejects({}));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            await expect(composite_table_1.CompositeStringIndexTable.createNew({
                client,
                name,
                primaryKeyName,
                sortKeyName,
            }))
                .rejects
                .toThrow();
        });
        test('success', async () => {
            const tableName = 'TestTable';
            const pk = 'PrimKey';
            const sk = 'SortKey';
            const tags = [
                {
                    Key: 'RFDKTesting',
                    Value: 'RFDKTesting',
                },
            ];
            aws_sdk_mock_1.mock('DynamoDB', 'createTable', sinon_1.fake.resolves({}));
            const client = new AWS.DynamoDB();
            const table = await composite_table_1.CompositeStringIndexTable.createNew({
                client,
                name: tableName,
                primaryKeyName: pk,
                sortKeyName: sk,
                tags,
            });
            expect(table.primaryKey).toBe(pk);
            expect(table.sortKey).toBe(sk);
        });
    });
    describe('deleteTable tests', () => {
        test('success', async () => {
            const tableName = 'TestTable';
            const pk = 'PrimKey';
            const sk = 'SortKey';
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', sinon_1.fake.resolves({}));
            const client = new AWS.DynamoDB();
            const table = new TestTable(client, tableName, pk, sk);
            await expect(table.deleteTable()).resolves.not.toThrow();
        });
        test('Table already deleted', async () => {
            const deleteFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.deleteTable()).resolves.toBe(undefined);
            expect(deleteFake.callCount).toEqual(1);
        });
        test('DynamoDB.deleteTable() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.deleteTable()).rejects.toThrowError();
        });
    });
    describe('putItem tests', () => {
        test('Table already deleted', async () => {
            const deleteFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteFake);
            const putFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'putItem', putFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.putItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' }))
                .rejects
                .toThrowError('Attempt to PutItem in deleted table');
            expect(deleteFake.callCount).toEqual(1);
            expect(putFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.putItem() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'putItem', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.putItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' })).rejects.toThrowError();
        });
    });
    describe('getItem tests', () => {
        test('Table already deleted', async () => {
            const deleteFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteFake);
            const getFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'getItem', getFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.getItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' }))
                .rejects
                .toThrowError('Attempt to GetItem from deleted table');
            expect(deleteFake.callCount).toEqual(1);
            expect(getFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.getItem() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'getItem', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.getItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' })).rejects.toThrowError();
        });
    });
    describe('deleteItem tests', () => {
        test('Table already deleted', async () => {
            const deleteTableFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteTableFake);
            const deleteItemFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteItem', deleteItemFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.deleteItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' }))
                .rejects
                .toThrowError('Attempt to DeleteItem from deleted table');
            expect(deleteTableFake.callCount).toEqual(1);
            expect(deleteItemFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.deleteItem() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'deleteItem', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.deleteItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' })).rejects.toThrowError();
        });
    });
    describe('query tests', () => {
        test('Returns empty', async () => {
            const queryFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'query', queryFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.query('TestPrimVal'))
                .resolves
                .toEqual({});
            expect(queryFake.callCount).toEqual(1);
        });
        test('Table already deleted', async () => {
            const deleteTableFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteTableFake);
            const queryFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'query', queryFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.query('TestPrimVal'))
                .rejects
                .toThrowError('Attempt to Query a deleted table');
            expect(deleteTableFake.callCount).toEqual(1);
            expect(queryFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.query() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'query', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.query('TestPrimVal')).rejects.toThrowError();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcG9zaXRlLXRhYmxlLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjb21wb3NpdGUtdGFibGUudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOztBQUVILCtCQUErQjtBQUUvQiw2REFBNkQ7QUFDN0QsbUNBQXFDO0FBQ3JDLCtCQUErQjtBQUMvQiwrQ0FBNkQ7QUFDN0QscUNBQXFDO0FBQ3JDLGlDQUE2QjtBQUU3Qix3REFBK0Q7QUFFL0QsdUNBQXVDO0FBQ3ZDLE1BQU0sS0FBSyxHQUFHLEtBQUssQ0FBQztBQUNwQixJQUFJLENBQUMsS0FBSyxFQUFFO0lBQ1YsT0FBTyxDQUFDLEtBQUssR0FBRyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7Q0FDM0I7QUFFRCxpREFBaUQ7QUFDakQsTUFBTSxTQUFVLFNBQVEsMkNBQXlCO0lBRy9DLFlBQ0UsTUFBb0IsRUFDcEIsSUFBWSxFQUNaLFVBQWtCLEVBQ2xCLE9BQWU7UUFFZixLQUFLLENBQ0gsTUFBTSxFQUNOLElBQUksRUFDSixVQUFVLEVBQ1YsT0FBTyxDQUNSLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFFRDs7Ozs7O0dBTUc7QUFDSCxRQUFRLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO0lBQ3BDLE1BQU0sTUFBTSxHQUFHLG9CQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzlDLE1BQU0sZUFBZSxHQUFHLGdCQUFnQixNQUFNLEVBQUUsQ0FBQztJQUNqRCxNQUFNLGVBQWUsR0FBRyxnQkFBZ0IsTUFBTSxFQUFFLENBQUMsQ0FBQyx1QkFBdUI7SUFDekUsTUFBTSxlQUFlLEdBQUcsZ0JBQWdCLE1BQU0sRUFBRSxDQUFDLENBQUMsZ0NBQWdDO0lBQ2xGLE1BQU0sZUFBZSxHQUFHLGdCQUFnQixNQUFNLEVBQUUsQ0FBQyxDQUFDLDZCQUE2QjtJQUMvRSxNQUFNLGNBQWMsR0FBRyxRQUFRLENBQUM7UUFDOUIsYUFBYSxFQUFFLENBQUM7UUFDaEIsYUFBYSxFQUFFLENBQUM7UUFDaEIsYUFBYSxFQUFFLENBQUM7S0FDakIsQ0FBQyxDQUFDO0lBQ0gsSUFBSSxZQUEwQixDQUFDO0lBRS9CLFNBQVMsQ0FBQyxLQUFLLElBQUksRUFBRTtRQUNuQixNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUM7UUFDdkIsTUFBTSxjQUFjLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxDQUFDLEdBQVEsRUFBRSxFQUFFO1lBQ2pELElBQUksR0FBRyxFQUFFO2dCQUFFLE1BQU0sR0FBRyxDQUFDO2FBQUU7UUFDekIsQ0FBQyxDQUFDLENBQUM7UUFFSCxZQUFZLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDO1lBQzlCLFdBQVcsRUFBRSxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUM7Z0JBQy9CLFdBQVcsRUFBRSxFQUFFO2dCQUNmLGVBQWUsRUFBRSxFQUFFO2FBQ3BCLENBQUM7WUFDRixRQUFRLEVBQUUsb0JBQW9CLFFBQVEsRUFBRTtZQUN4QyxNQUFNLEVBQUUsV0FBVztTQUNwQixDQUFDLENBQUM7UUFFSCxTQUFTLGtCQUFrQixDQUFDLFNBQWlCLEVBQUUsY0FBc0IsRUFBRSxPQUE2QjtZQUNsRyxNQUFNLE9BQU8sR0FBRztnQkFDZCxTQUFTLEVBQUUsU0FBUztnQkFDcEIsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLGFBQWEsRUFBRSxTQUFTO3dCQUN4QixhQUFhLEVBQUUsY0FBYztxQkFDOUI7aUJBQ0Y7Z0JBQ0QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLGFBQWEsRUFBRSxTQUFTO3dCQUN4QixPQUFPLEVBQUUsTUFBTTtxQkFDaEI7aUJBQ0Y7Z0JBQ0QsV0FBVyxFQUFFLGlCQUFpQjtnQkFDOUIsSUFBSSxFQUFFO29CQUNKO3dCQUNFLEdBQUcsRUFBRSxhQUFhO3dCQUNsQixLQUFLLEVBQUUsYUFBYTtxQkFDckI7aUJBQ0Y7YUFDRixDQUFDO1lBQ0YsSUFBSSxPQUFPLEVBQUU7Z0JBQ1gsT0FBTyxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQztvQkFDaEMsYUFBYSxFQUFFLFNBQVM7b0JBQ3hCLGFBQWEsRUFBRSxPQUFPLENBQUMsT0FBTztpQkFDL0IsQ0FBQyxDQUFDO2dCQUNILE9BQU8sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDO29CQUNyQixhQUFhLEVBQUUsU0FBUztvQkFDeEIsT0FBTyxFQUFFLE9BQU87aUJBQ2pCLENBQUMsQ0FBQzthQUNKO1lBQ0QsT0FBTyxPQUFPLENBQUM7UUFDakIsQ0FBQztRQUVELElBQUksT0FBTyxHQUFHLGtCQUFrQixDQUFDLGVBQWUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUN6RSxJQUFJLFFBQVEsR0FBbUMsTUFBTSxZQUFZLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ2pHLElBQUksS0FBSyxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztRQUN0QyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQUUsTUFBTSxLQUFLLENBQUMsb0JBQW9CLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FBRTtRQUNuRSxPQUFPLENBQUMsS0FBSyxDQUFDLDRCQUE0QixLQUFLLENBQUMsU0FBUyxHQUFHLENBQUMsQ0FBQztRQUU5RCxPQUFPLEdBQUcsa0JBQWtCLENBQUMsZUFBZSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ25ELFFBQVEsR0FBRyxNQUFNLFlBQVksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDN0QsS0FBSyxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztRQUNsQyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQUUsTUFBTSxLQUFLLENBQUMsb0JBQW9CLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FBRTtRQUNuRSxPQUFPLENBQUMsS0FBSyxDQUFDLDRCQUE0QixLQUFLLENBQUMsU0FBUyxHQUFHLENBQUMsQ0FBQztRQUU5RCxPQUFPLEdBQUcsa0JBQWtCLENBQUMsZUFBZSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ3JFLFFBQVEsR0FBRyxNQUFNLFlBQVksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDN0QsS0FBSyxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztRQUNsQyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQUUsTUFBTSxLQUFLLENBQUMsb0JBQW9CLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FBRTtRQUNuRSxPQUFPLENBQUMsS0FBSyxDQUFDLDRCQUE0QixLQUFLLENBQUMsU0FBUyxHQUFHLENBQUMsQ0FBQztRQUU5RCxPQUFPLEdBQUcsa0JBQWtCLENBQUMsZUFBZSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ3JFLFFBQVEsR0FBRyxNQUFNLFlBQVksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDN0QsS0FBSyxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztRQUNsQyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQUUsTUFBTSxLQUFLLENBQUMsb0JBQW9CLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FBRTtRQUNuRSxPQUFPLENBQUMsS0FBSyxDQUFDLDRCQUE0QixLQUFLLENBQUMsU0FBUyxHQUFHLENBQUMsQ0FBQztRQUU5RCxJQUFJLE9BQU8sR0FBWSxJQUFJLENBQUM7UUFDNUIsR0FBRztZQUNELE1BQU0sUUFBUSxHQUFHLEVBQUUsQ0FBQztZQUNwQixLQUFLLE1BQU0sSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFLGVBQWUsRUFBRSxlQUFlLEVBQUUsZUFBZSxDQUFDLEVBQUU7Z0JBQ3ZGLFFBQVEsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQztvQkFDdkMsU0FBUyxFQUFFLElBQUk7aUJBQ2hCLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO2FBQ2Y7WUFDRCxNQUFNLFNBQVMsR0FBRyxNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDOUMsT0FBTyxHQUFHLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRSxXQUFDLE9BQUEsT0FBQSxJQUFJLENBQUMsS0FBSywwQ0FBRSxXQUFXLE1BQUssUUFBUSxDQUFBLEVBQUEsQ0FBQyxDQUFDO1lBQ3pFLE1BQU0sSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDekQsUUFBUSxPQUFPLEVBQUU7SUFDcEIsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRVYsUUFBUSxDQUFDLEtBQUssSUFBSSxFQUFFO1FBQ2xCLE1BQU0sY0FBYyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQy9CLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEtBQUssSUFBSSxFQUFFO1FBQ3hELE1BQU0sU0FBUyxHQUFXLGtCQUFrQixDQUFDO1FBQzdDLE1BQU0sRUFBRSxHQUFXLFNBQVMsQ0FBQztRQUM3QixNQUFNLEVBQUUsR0FBVyxTQUFTLENBQUM7UUFFN0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxTQUFTLENBQ3pCLFlBQVksRUFDWixTQUFTLEVBQ1QsRUFBRSxFQUNGLEVBQUUsQ0FDSCxDQUFDO1FBQ0YsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUN0QyxNQUFNLE1BQU0sQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLFlBQVksRUFBRSxDQUFDO1FBQzlELE1BQU0sQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDMUMsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1FBQ3BELElBQUksQ0FBQyxJQUFJLENBQUM7WUFDUixDQUFDLGVBQWUsQ0FBQztZQUNqQixDQUFDLGVBQWUsQ0FBQztZQUNqQixDQUFDLGVBQWUsQ0FBQztTQUNsQixDQUFDLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxFQUFFLFNBQWlCLEVBQUUsRUFBRTtZQUMvQyxPQUFPO1lBQ1AsTUFBTSxNQUFNLENBQUMsMkNBQXlCLENBQUMsWUFBWSxDQUFDLFlBQVksRUFBRSxTQUFTLENBQUMsQ0FBQztnQkFDM0UsT0FBTztpQkFDTixPQUFPO2lCQUNQLE9BQU8sRUFBRSxDQUFDO1FBQ2YsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx5Q0FBeUMsRUFBRSxLQUFLLElBQUksRUFBRTtRQUN6RCxPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsTUFBTSwyQ0FBeUIsQ0FBQyxZQUFZLENBQUMsWUFBWSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBRTFGLE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN6QyxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUN4QyxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx5QkFBeUIsRUFBRSxLQUFLLElBQUksRUFBRTtRQUN6QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsTUFBTSwyQ0FBeUIsQ0FBQyxZQUFZLENBQUMsWUFBWSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBRTFGLE9BQU87UUFDUCxNQUFNLFVBQVUsR0FBRztZQUNqQixhQUFhLEVBQUUsWUFBWTtTQUM1QixDQUFDO1FBQ0YsTUFBTSxVQUFVLEdBQUc7WUFDakIsZUFBZSxFQUFFLG9CQUFXLENBQUMsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQztZQUNoRCxZQUFZLEVBQUUsb0JBQVcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDO1lBQzdDLFVBQVU7U0FDWCxDQUFDO1FBQ0YsTUFBTSxVQUFVLEdBQUc7WUFDakIsZUFBZSxFQUFFLFVBQVUsQ0FBQyxlQUFlO1lBQzNDLFlBQVksRUFBRSxVQUFVLENBQUMsWUFBWTtTQUN0QyxDQUFDO1FBRUYsT0FBTztRQUNQLE1BQU0sTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVELE1BQU0sTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzdFLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztRQUNSLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUM7S0FDaEIsQ0FBQyxDQUFDLHNCQUFzQixFQUFFLEtBQUssRUFBRSxjQUF1QixFQUFFLEVBQUU7UUFDM0QsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLE1BQU0sMkNBQXlCLENBQUMsWUFBWSxDQUFDLFlBQVksRUFBRSxlQUFlLENBQUMsQ0FBQztRQUUxRixPQUFPO1FBQ1AsTUFBTSxVQUFVLEdBQUc7WUFDakIsZUFBZSxFQUFFLG9CQUFXLENBQUMsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQztZQUNoRCxZQUFZLEVBQUUsb0JBQVcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDO1lBQzdDLFVBQVUsRUFBRTtnQkFDVixhQUFhLEVBQUUsWUFBWTthQUM1QjtZQUNELGVBQWUsRUFBRSxJQUFJO1NBQ3RCLENBQUM7UUFDRixNQUFNLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFaEMsT0FBTztRQUNQLFVBQVUsQ0FBQyxlQUFlLEdBQUcsY0FBYyxDQUFDO1FBQzVDLE1BQU0sTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ3hFLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEtBQUssSUFBSSxFQUFFO1FBQ2xDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxNQUFNLDJDQUF5QixDQUFDLFlBQVksQ0FBQyxZQUFZLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFFMUYsT0FBTztRQUNQLE1BQU0sVUFBVSxHQUFHO1lBQ2pCLGVBQWUsRUFBRSxvQkFBVyxDQUFDLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUM7WUFDaEQsWUFBWSxFQUFFLG9CQUFXLENBQUMsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQztTQUM5QyxDQUFDO1FBRUYsT0FBTztRQUNQLE1BQU0sTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDbkUsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsb0JBQW9CLEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDcEMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLE1BQU0sMkNBQXlCLENBQUMsWUFBWSxDQUFDLFlBQVksRUFBRSxlQUFlLENBQUMsQ0FBQztRQUUxRixPQUFPO1FBQ1AsTUFBTSxVQUFVLEdBQUc7WUFDakIsYUFBYSxFQUFFLFlBQVk7U0FDNUIsQ0FBQztRQUNGLE1BQU0sVUFBVSxHQUFHO1lBQ2pCLGVBQWUsRUFBRSxvQkFBVyxDQUFDLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUM7WUFDaEQsWUFBWSxFQUFFLG9CQUFXLENBQUMsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQztZQUM3QyxVQUFVO1NBQ1gsQ0FBQztRQUNGLE1BQU0sYUFBYSxHQUFHO1lBQ3BCLGVBQWUsRUFBRSxVQUFVLENBQUMsZUFBZTtZQUMzQyxZQUFZLEVBQUUsVUFBVSxDQUFDLFlBQVk7U0FDdEMsQ0FBQztRQUNGLE1BQU0sS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUVoQyxPQUFPO1FBQ1AsTUFBTSxNQUFNLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDcEUsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMscUJBQXFCLEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDckMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLE1BQU0sMkNBQXlCLENBQUMsWUFBWSxDQUFDLFlBQVksRUFBRSxlQUFlLENBQUMsQ0FBQztRQUUxRixPQUFPO1FBQ1AsTUFBTSxhQUFhLEdBQUc7WUFDcEIsZUFBZSxFQUFFLG9CQUFXLENBQUMsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQztZQUNoRCxZQUFZLEVBQUUsb0JBQVcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDO1NBQzlDLENBQUM7UUFFRixPQUFPO1FBQ1AsTUFBTSxNQUFNLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDckUsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsZUFBZSxFQUFFLEtBQUssSUFBSSxFQUFFO1FBQy9CLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxNQUFNLDJDQUF5QixDQUFDLFlBQVksQ0FBQyxZQUFZLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFFMUYsT0FBTztRQUNQLE1BQU0sZUFBZSxHQUFXLG9CQUFXLENBQUMsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hFLE1BQU0sUUFBUSxHQUE4QyxFQUFFLENBQUM7UUFDL0QsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEVBQUUsRUFBRTtZQUMzQixNQUFNLFlBQVksR0FBVyxvQkFBVyxDQUFDLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUM3RCxRQUFRLENBQUMsWUFBWSxDQUFDLEdBQUc7Z0JBQ3ZCLGFBQWEsRUFBRSxRQUFRLENBQUMsRUFBRTthQUMzQixDQUFDO1NBQ0g7UUFDRCxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUNuRCxNQUFNLFVBQVUsR0FBRztnQkFDakIsZUFBZTtnQkFDZixZQUFZLEVBQUUsR0FBRztnQkFDakIsVUFBVSxFQUFFLEtBQUs7YUFDbEIsQ0FBQztZQUNGLE1BQU0sS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUNqQztRQUVELE9BQU87UUFDUCxNQUFNLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDakYsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQywwQkFBMEIsRUFBRSxHQUFHLEVBQUU7SUFDeEMsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLDZCQUFjLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQyxDQUFDLENBQUM7SUFFSCxTQUFTLENBQUMsR0FBRyxFQUFFO1FBQ2Isc0JBQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUN0QixDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxvQkFBb0IsRUFBRSxHQUFHLEVBQUU7UUFDbEMsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEtBQUssSUFBSSxFQUFFO1lBQ2pDLE1BQU0sUUFBUSxHQUFHLFlBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQztZQUNyRCxtQkFBSSxDQUFDLFVBQVUsRUFBRSxlQUFlLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDNUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbEMsTUFBTSxTQUFTLEdBQUcsYUFBYSxDQUFDO1lBRWhDLE1BQU0sTUFBTSxDQUFDLDJDQUF5QixDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsU0FBUyxDQUFDLENBQUM7aUJBQ3BFLE9BQU87aUJBQ1AsWUFBWSxDQUFDLHNDQUFzQyxTQUFTLEdBQUcsQ0FBQyxDQUFDO1FBQ3RFLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHFCQUFxQixFQUFFLEtBQUssSUFBSSxFQUFFO1lBQ3JDLG1CQUFJLENBQUMsVUFBVSxFQUFFLGVBQWUsRUFBRSxZQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3RGLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2xDLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQztZQUU5QixNQUFNLE1BQU0sQ0FBQywyQ0FBeUIsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO2lCQUNwRSxPQUFPO2lCQUNQLFlBQVksQ0FBQyxzQ0FBc0MsU0FBUyxHQUFHLENBQUMsQ0FBQztRQUN0RSxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxLQUFLLElBQUksRUFBRTtZQUNoRCxtQkFBSSxDQUFDLFVBQVUsRUFBRSxlQUFlLEVBQUUsWUFBSSxDQUFDLFFBQVEsQ0FBQztnQkFDOUMsS0FBSyxFQUFFO29CQUNMLFNBQVMsRUFBRSxFQUFFO29CQUNiLG9CQUFvQixFQUFFLFNBQVM7aUJBQ2hDO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFDSixNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQyxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUM7WUFFOUIsTUFBTSxNQUFNLENBQUMsMkNBQXlCLENBQUMsWUFBWSxDQUFDLE1BQU0sRUFBRSxTQUFTLENBQUMsQ0FBQztpQkFDcEUsT0FBTztpQkFDUCxZQUFZLENBQUMsdUNBQXVDLFNBQVMsR0FBRyxDQUFDLENBQUM7UUFDdkUsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsc0JBQXNCLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDdEMsbUJBQUksQ0FBQyxVQUFVLEVBQUUsZUFBZSxFQUFFLFlBQUksQ0FBQyxRQUFRLENBQUM7Z0JBQzlDLEtBQUssRUFBRTtvQkFDTCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsYUFBYSxFQUFFLFNBQVM7NEJBQ3hCLE9BQU8sRUFBRSxPQUFPO3lCQUNqQjtxQkFDRjtvQkFDRCxvQkFBb0IsRUFBRSxFQUFFO2lCQUN6QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ0osTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbEMsTUFBTSxTQUFTLEdBQUcsV0FBVyxDQUFDO1lBRTlCLE1BQU0sTUFBTSxDQUFDLDJDQUF5QixDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsU0FBUyxDQUFDLENBQUM7aUJBQ3BFLE9BQU87aUJBQ1AsWUFBWSxDQUFDLHVDQUF1QyxTQUFTLEdBQUcsQ0FBQyxDQUFDO1FBQ3ZFLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG1CQUFtQixFQUFFLEtBQUssSUFBSSxFQUFFO1lBQ25DLG1CQUFJLENBQUMsVUFBVSxFQUFFLGVBQWUsRUFBRSxZQUFJLENBQUMsUUFBUSxDQUFDO2dCQUM5QyxLQUFLLEVBQUU7b0JBQ0wsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLGFBQWEsRUFBRSxZQUFZOzRCQUMzQixPQUFPLEVBQUUsTUFBTTt5QkFDaEI7cUJBQ0Y7b0JBQ0Qsb0JBQW9CLEVBQUUsRUFBRTtpQkFDekI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2xDLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQztZQUU5QixNQUFNLE1BQU0sQ0FBQywyQ0FBeUIsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO2lCQUNwRSxPQUFPO2lCQUNQLFlBQVksQ0FBQyxvQ0FBb0MsU0FBUyxHQUFHLENBQUMsQ0FBQztRQUNwRSxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywwQ0FBMEMsRUFBRSxLQUFLLElBQUksRUFBRTtZQUMxRCxtQkFBSSxDQUFDLFVBQVUsRUFBRSxlQUFlLEVBQUUsWUFBSSxDQUFDLFFBQVEsQ0FBQztnQkFDOUMsS0FBSyxFQUFFO29CQUNMLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxhQUFhLEVBQUUsWUFBWTs0QkFDM0IsT0FBTyxFQUFFLE1BQU07eUJBQ2hCO3dCQUNEOzRCQUNFLGFBQWEsRUFBRSxTQUFTOzRCQUN4QixPQUFPLEVBQUUsT0FBTzt5QkFDakI7cUJBQ0Y7b0JBQ0Qsb0JBQW9CLEVBQUU7d0JBQ3BCOzRCQUNFLGFBQWEsRUFBRSxTQUFTOzRCQUN4QixhQUFhLEVBQUUsR0FBRzt5QkFDbkI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2xDLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQztZQUU5QixNQUFNLE1BQU0sQ0FBQywyQ0FBeUIsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO2lCQUNwRSxPQUFPO2lCQUNQLFlBQVksQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1FBQ2xFLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEtBQUssSUFBSSxFQUFFO1lBQ3ZELG1CQUFJLENBQUMsVUFBVSxFQUFFLGVBQWUsRUFBRSxZQUFJLENBQUMsUUFBUSxDQUFDO2dCQUM5QyxLQUFLLEVBQUU7b0JBQ0wsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLGFBQWEsRUFBRSxZQUFZOzRCQUMzQixPQUFPLEVBQUUsTUFBTTt5QkFDaEI7d0JBQ0Q7NEJBQ0UsYUFBYSxFQUFFLFNBQVM7NEJBQ3hCLE9BQU8sRUFBRSxPQUFPO3lCQUNqQjtxQkFDRjtvQkFDRCxvQkFBb0IsRUFBRTt3QkFDcEI7NEJBQ0UsYUFBYSxFQUFFLFlBQVk7NEJBQzNCLGFBQWEsRUFBRSxHQUFHO3lCQUNuQjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ0osTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbEMsTUFBTSxTQUFTLEdBQUcsV0FBVyxDQUFDO1lBRTlCLE1BQU0sTUFBTSxDQUFDLDJDQUF5QixDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsU0FBUyxDQUFDLENBQUM7aUJBQ3BFLE9BQU87aUJBQ1AsWUFBWSxDQUFDLHdDQUF3QyxDQUFDLENBQUM7UUFDNUQsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxHQUFHLEVBQUU7UUFDL0IsSUFBSSxDQUFDLDZDQUE2QyxFQUFFLEtBQUssSUFBSSxFQUFFO1lBQzdELG1CQUFJLENBQUMsVUFBVSxFQUFFLGFBQWEsRUFBRSxZQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDbEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQUcsV0FBVyxDQUFDO1lBQ3pCLE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQztZQUNwQyxNQUFNLFdBQVcsR0FBRyxTQUFTLENBQUM7WUFFOUIsTUFBTSxNQUFNLENBQUMsMkNBQXlCLENBQUMsU0FBUyxDQUFDO2dCQUMvQyxNQUFNO2dCQUNOLElBQUk7Z0JBQ0osY0FBYztnQkFDZCxXQUFXO2FBQ1osQ0FBQyxDQUFDO2lCQUNBLE9BQU87aUJBQ1AsT0FBTyxFQUFFLENBQUM7UUFDZixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDekIsTUFBTSxTQUFTLEdBQVcsV0FBVyxDQUFDO1lBQ3RDLE1BQU0sRUFBRSxHQUFXLFNBQVMsQ0FBQztZQUM3QixNQUFNLEVBQUUsR0FBVyxTQUFTLENBQUM7WUFDN0IsTUFBTSxJQUFJLEdBQUc7Z0JBQ1g7b0JBQ0UsR0FBRyxFQUFFLGFBQWE7b0JBQ2xCLEtBQUssRUFBRSxhQUFhO2lCQUNyQjthQUNGLENBQUM7WUFFRixtQkFBSSxDQUFDLFVBQVUsRUFBRSxhQUFhLEVBQUUsWUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ25ELE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2xDLE1BQU0sS0FBSyxHQUFHLE1BQU0sMkNBQXlCLENBQUMsU0FBUyxDQUFDO2dCQUN0RCxNQUFNO2dCQUNOLElBQUksRUFBRSxTQUFTO2dCQUNmLGNBQWMsRUFBRSxFQUFFO2dCQUNsQixXQUFXLEVBQUUsRUFBRTtnQkFDZixJQUFJO2FBQ0wsQ0FBQyxDQUFDO1lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDbEMsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDakMsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7UUFDakMsSUFBSSxDQUFDLFNBQVMsRUFBRSxLQUFLLElBQUksRUFBRTtZQUN6QixNQUFNLFNBQVMsR0FBVyxXQUFXLENBQUM7WUFDdEMsTUFBTSxFQUFFLEdBQVcsU0FBUyxDQUFDO1lBQzdCLE1BQU0sRUFBRSxHQUFXLFNBQVMsQ0FBQztZQUM3QixtQkFBSSxDQUFDLFVBQVUsRUFBRSxhQUFhLEVBQUUsWUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBRW5ELE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2xDLE1BQU0sS0FBSyxHQUFHLElBQUksU0FBUyxDQUN6QixNQUFNLEVBQ04sU0FBUyxFQUNULEVBQUUsRUFDRixFQUFFLENBQ0gsQ0FBQztZQUVGLE1BQU0sTUFBTSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDM0QsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDdkMsTUFBTSxVQUFVLEdBQUcsWUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNyQyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxhQUFhLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDNUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQUcsV0FBVyxDQUFDO1lBQ3pCLE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQztZQUNwQyxNQUFNLFdBQVcsR0FBRyxTQUFTLENBQUM7WUFDOUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxTQUFTLENBQzNCLE1BQU0sRUFDTixJQUFJLEVBQ0osY0FBYyxFQUNkLFdBQVcsQ0FDWixDQUFDO1lBQ0YsTUFBTSxPQUFPLENBQUMsV0FBVyxFQUFFLENBQUM7WUFFNUIsTUFBTSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUM3RCxNQUFNLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxQyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxLQUFLLElBQUksRUFBRTtZQUNoRCxtQkFBSSxDQUFDLFVBQVUsRUFBRSxhQUFhLEVBQUUsWUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztZQUMzRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQyxNQUFNLElBQUksR0FBRyxXQUFXLENBQUM7WUFDekIsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDO1lBQ3BDLE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQztZQUM5QixNQUFNLE9BQU8sR0FBRyxJQUFJLFNBQVMsQ0FDM0IsTUFBTSxFQUNOLElBQUksRUFDSixjQUFjLEVBQ2QsV0FBVyxDQUNaLENBQUM7WUFDRixNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDN0QsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxlQUFlLEVBQUUsR0FBRyxFQUFFO1FBQzdCLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxLQUFLLElBQUksRUFBRTtZQUN2QyxNQUFNLFVBQVUsR0FBRyxZQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ3JDLG1CQUFJLENBQUMsVUFBVSxFQUFFLGFBQWEsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM1QyxNQUFNLE9BQU8sR0FBRyxZQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ2xDLG1CQUFJLENBQUMsVUFBVSxFQUFFLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUVyQyxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQyxNQUFNLElBQUksR0FBRyxXQUFXLENBQUM7WUFDekIsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDO1lBQ3BDLE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQztZQUM5QixNQUFNLE9BQU8sR0FBRyxJQUFJLFNBQVMsQ0FDM0IsTUFBTSxFQUNOLElBQUksRUFDSixjQUFjLEVBQ2QsV0FBVyxDQUNaLENBQUM7WUFDRixNQUFNLE9BQU8sQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUU1QixNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsZUFBZSxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQztpQkFDM0YsT0FBTztpQkFDUCxZQUFZLENBQUMscUNBQXFDLENBQUMsQ0FBQztZQUN2RCxNQUFNLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN4QyxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3pDLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLDRCQUE0QixFQUFFLEtBQUssSUFBSSxFQUFFO1lBQzVDLG1CQUFJLENBQUMsVUFBVSxFQUFFLFNBQVMsRUFBRSxZQUFJLENBQUMsT0FBTyxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3ZELE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLFdBQVcsQ0FBQztZQUN6QixNQUFNLGNBQWMsR0FBRyxZQUFZLENBQUM7WUFDcEMsTUFBTSxXQUFXLEdBQUcsU0FBUyxDQUFDO1lBQzlCLE1BQU0sT0FBTyxHQUFHLElBQUksU0FBUyxDQUMzQixNQUFNLEVBQ04sSUFBSSxFQUNKLGNBQWMsRUFDZCxXQUFXLENBQ1osQ0FBQztZQUNGLE1BQU0sTUFBTSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxlQUFlLEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxDQUFDO1FBQ3hILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsZUFBZSxFQUFFLEdBQUcsRUFBRTtRQUM3QixJQUFJLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDdkMsTUFBTSxVQUFVLEdBQUcsWUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNyQyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxhQUFhLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDNUMsTUFBTSxPQUFPLEdBQUcsWUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNsQyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxTQUFTLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFFckMsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQUcsV0FBVyxDQUFDO1lBQ3pCLE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQztZQUNwQyxNQUFNLFdBQVcsR0FBRyxTQUFTLENBQUM7WUFDOUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxTQUFTLENBQzNCLE1BQU0sRUFDTixJQUFJLEVBQ0osY0FBYyxFQUNkLFdBQVcsQ0FDWixDQUFDO1lBQ0YsTUFBTSxPQUFPLENBQUMsV0FBVyxFQUFFLENBQUM7WUFFNUIsTUFBTSxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLGVBQWUsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7aUJBQzNGLE9BQU87aUJBQ1AsWUFBWSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7WUFDekQsTUFBTSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUN6QyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyw0QkFBNEIsRUFBRSxLQUFLLElBQUksRUFBRTtZQUM1QyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxTQUFTLEVBQUUsWUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztZQUN2RCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQyxNQUFNLElBQUksR0FBRyxXQUFXLENBQUM7WUFDekIsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDO1lBQ3BDLE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQztZQUM5QixNQUFNLE9BQU8sR0FBRyxJQUFJLFNBQVMsQ0FDM0IsTUFBTSxFQUNOLElBQUksRUFDSixjQUFjLEVBQ2QsV0FBVyxDQUNaLENBQUM7WUFDRixNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsZUFBZSxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUN4SCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtRQUNoQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDdkMsTUFBTSxlQUFlLEdBQUcsWUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUMxQyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxhQUFhLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDakQsTUFBTSxjQUFjLEdBQUcsWUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUN6QyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxZQUFZLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFFL0MsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQUcsV0FBVyxDQUFDO1lBQ3pCLE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQztZQUNwQyxNQUFNLFdBQVcsR0FBRyxTQUFTLENBQUM7WUFDOUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxTQUFTLENBQzNCLE1BQU0sRUFDTixJQUFJLEVBQ0osY0FBYyxFQUNkLFdBQVcsQ0FDWixDQUFDO1lBQ0YsTUFBTSxPQUFPLENBQUMsV0FBVyxFQUFFLENBQUM7WUFFNUIsTUFBTSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFLGVBQWUsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7aUJBQzlGLE9BQU87aUJBQ1AsWUFBWSxDQUFDLDBDQUEwQyxDQUFDLENBQUM7WUFDNUQsTUFBTSxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDN0MsTUFBTSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUNoRCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywrQkFBK0IsRUFBRSxLQUFLLElBQUksRUFBRTtZQUMvQyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxZQUFZLEVBQUUsWUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztZQUMxRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQyxNQUFNLElBQUksR0FBRyxXQUFXLENBQUM7WUFDekIsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDO1lBQ3BDLE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQztZQUM5QixNQUFNLE9BQU8sR0FBRyxJQUFJLFNBQVMsQ0FDM0IsTUFBTSxFQUNOLElBQUksRUFDSixjQUFjLEVBQ2QsV0FBVyxDQUNaLENBQUM7WUFDRixNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEVBQUUsZUFBZSxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUMzSCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGFBQWEsRUFBRSxHQUFHLEVBQUU7UUFDM0IsSUFBSSxDQUFDLGVBQWUsRUFBRSxLQUFLLElBQUksRUFBRTtZQUMvQixNQUFNLFNBQVMsR0FBRyxZQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ3BDLG1CQUFJLENBQUMsVUFBVSxFQUFFLE9BQU8sRUFBRSxTQUFTLENBQUMsQ0FBQztZQUVyQyxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQyxNQUFNLElBQUksR0FBRyxXQUFXLENBQUM7WUFDekIsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDO1lBQ3BDLE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQztZQUM5QixNQUFNLE9BQU8sR0FBRyxJQUFJLFNBQVMsQ0FDM0IsTUFBTSxFQUNOLElBQUksRUFDSixjQUFjLEVBQ2QsV0FBVyxDQUNaLENBQUM7WUFFRixNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO2lCQUN2QyxRQUFRO2lCQUNSLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNmLE1BQU0sQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3pDLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHVCQUF1QixFQUFFLEtBQUssSUFBSSxFQUFFO1lBQ3ZDLE1BQU0sZUFBZSxHQUFHLFlBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDMUMsbUJBQUksQ0FBQyxVQUFVLEVBQUUsYUFBYSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ2pELE1BQU0sU0FBUyxHQUFHLFlBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDcEMsbUJBQUksQ0FBQyxVQUFVLEVBQUUsT0FBTyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBRXJDLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLFdBQVcsQ0FBQztZQUN6QixNQUFNLGNBQWMsR0FBRyxZQUFZLENBQUM7WUFDcEMsTUFBTSxXQUFXLEdBQUcsU0FBUyxDQUFDO1lBQzlCLE1BQU0sT0FBTyxHQUFHLElBQUksU0FBUyxDQUMzQixNQUFNLEVBQ04sSUFBSSxFQUNKLGNBQWMsRUFDZCxXQUFXLENBQ1osQ0FBQztZQUNGLE1BQU0sT0FBTyxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBRTVCLE1BQU0sTUFBTSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7aUJBQ3ZDLE9BQU87aUJBQ1AsWUFBWSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7WUFDcEQsTUFBTSxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDN0MsTUFBTSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUMzQyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywwQkFBMEIsRUFBRSxLQUFLLElBQUksRUFBRTtZQUMxQyxtQkFBSSxDQUFDLFVBQVUsRUFBRSxPQUFPLEVBQUUsWUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNyRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQyxNQUFNLElBQUksR0FBRyxXQUFXLENBQUM7WUFDekIsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDO1lBQ3BDLE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQztZQUM5QixNQUFNLE9BQU8sR0FBRyxJQUFJLFNBQVMsQ0FDM0IsTUFBTSxFQUNOLElBQUksRUFDSixjQUFjLEVBQ2QsV0FBVyxDQUNaLENBQUM7WUFDRixNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxDQUFDO1FBQ3BFLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbi8qIGVzbGludC1kaXNhYmxlIG5vLWNvbnNvbGUgKi9cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGltcG9ydC9uby1leHRyYW5lb3VzLWRlcGVuZGVuY2llc1xuaW1wb3J0IHsgcmFuZG9tQnl0ZXMgfSBmcm9tICdjcnlwdG8nO1xuaW1wb3J0ICogYXMgQVdTIGZyb20gJ2F3cy1zZGsnO1xuaW1wb3J0IHsgbW9jaywgcmVzdG9yZSwgc2V0U0RLSW5zdGFuY2UgfSBmcm9tICdhd3Mtc2RrLW1vY2snO1xuaW1wb3J0ICogYXMgZHluYWxpdGUgZnJvbSAnZHluYWxpdGUnO1xuaW1wb3J0IHsgZmFrZSB9IGZyb20gJ3Npbm9uJztcblxuaW1wb3J0IHsgQ29tcG9zaXRlU3RyaW5nSW5kZXhUYWJsZSB9IGZyb20gJy4uL2NvbXBvc2l0ZS10YWJsZSc7XG5cbi8vIEVuYWJsZS9kaXNhYmxlIGRlYnVnZ2luZyBzdGF0ZW1lbnRzLlxuY29uc3QgREVCVUcgPSBmYWxzZTtcbmlmICghREVCVUcpIHtcbiAgY29uc29sZS5kZWJ1ZyA9ICgpID0+IHsgfTtcbn1cblxuLy8gVGVzdCBjbGFzcyB0byBleHBvc2UgY29uc3RydWN0b3IgYW5kIHRhYmxlTmFtZVxuY2xhc3MgVGVzdFRhYmxlIGV4dGVuZHMgQ29tcG9zaXRlU3RyaW5nSW5kZXhUYWJsZSB7XG4gIHB1YmxpYyB0YWJsZU5hbWU6IHN0cmluZyB8IHVuZGVmaW5lZDtcblxuICBwdWJsaWMgY29uc3RydWN0b3IoXG4gICAgY2xpZW50OiBBV1MuRHluYW1vREIsXG4gICAgbmFtZTogc3RyaW5nLFxuICAgIHByaW1hcnlLZXk6IHN0cmluZyxcbiAgICBzb3J0S2V5OiBzdHJpbmcsXG4gICkge1xuICAgIHN1cGVyKFxuICAgICAgY2xpZW50LFxuICAgICAgbmFtZSxcbiAgICAgIHByaW1hcnlLZXksXG4gICAgICBzb3J0S2V5LFxuICAgICk7XG4gIH1cbn1cblxuLyoqXG4gKiBUaGlzIGNsYXNzIGlzIGJlaW5nIHJlZmFjdG9yZWQgdG8gdXNlIG1vY2tzIG92ZXIgYW4gYWN0dWFseSBjb25uZWN0aW9uIHRvIER5bmFtb0RCLiBBbGwgdGhlIER5bmFtb0RCIHRlc3RzIHdlcmVcbiAqIGNvbnZlcnRlZCB0byB1c2UgZHluYWxpdGUsIGEgcGFja2FnZSB0aGF0IHNldHMgdXAgYSBzZXJ2ZXIgdG8gcmVwbGljYXRlIER5bmFtb0RCIGxvY2FsbHkuIFRoZXNlIHRlc3RzIGNhbiBiZSBmb3VuZFxuICogaW4gdGhlIGZpcnN0IGRlc2NyaWJlKCkgYmxvY2suIE5ld2VyIHRlc3RzIGhhdmUgYmVlbiB3cml0dGVuIHVzaW5nIGF3cy1zZGstbW9jayBhbmQgZm91bmQgaW4gdGhlIGRlc2NyaWJlKCkgYmxvY2tcbiAqIHRoYXQgZm9sbG93cy4gQW55IHVwZGF0ZXMgdG8gdGhlc2UgdGVzdHMgc2hvdWxkIHRyYW5zbGF0ZSBkeW5hbGl0ZSB0ZXN0cyB0byBhd3Mtc2RrLW1vY2tzIGFuZCBuZXcgdGVzdHMgc2hvdWxkIHVzZVxuICogYXdzLXNkay1tb2NrLlxuICovXG5kZXNjcmliZSgnVGVzdHMgdXNpbmcgZHluYWxpdGUnLCAoKSA9PiB7XG4gIGNvbnN0IFNVRkZJWCA9IHJhbmRvbUJ5dGVzKDgpLnRvU3RyaW5nKCdoZXgnKTtcbiAgY29uc3QgR09PRF9UQUJMRV9OQU1FID0gYFJGREtUZXN0R29vZC0ke1NVRkZJWH1gO1xuICBjb25zdCBCQURfVEFCTEUxX05BTUUgPSBgUkZES1Rlc3RCYWQxLSR7U1VGRklYfWA7IC8vIE1pc3NpbmcgdGhlIHNvcnQga2V5XG4gIGNvbnN0IEJBRF9UQUJMRTJfTkFNRSA9IGBSRkRLVGVzdEJhZDItJHtTVUZGSVh9YDsgLy8gSW5jb3JyZWN0IHR5cGUgb24gcHJpbWFyeSBrZXlcbiAgY29uc3QgQkFEX1RBQkxFM19OQU1FID0gYFJGREtUZXN0QmFkMy0ke1NVRkZJWH1gOyAvLyBJbmNvcnJlY3QgdHlwZSBvbiBzb3J0IGtleVxuICBjb25zdCBkeW5hbGl0ZVNlcnZlciA9IGR5bmFsaXRlKHtcbiAgICBjcmVhdGVUYWJsZU1zOiA1LFxuICAgIGRlbGV0ZVRhYmxlTXM6IDUsXG4gICAgdXBkYXRlVGFibGVNczogNSxcbiAgfSk7XG4gIGxldCBkeW5hbW9DbGllbnQ6IEFXUy5EeW5hbW9EQjtcblxuICBiZWZvcmVBbGwoYXN5bmMgKCkgPT4ge1xuICAgIGNvbnN0IGR5bmFQb3J0ID0gNDMyNjY7XG4gICAgYXdhaXQgZHluYWxpdGVTZXJ2ZXIubGlzdGVuKGR5bmFQb3J0LCAoZXJyOiBhbnkpID0+IHtcbiAgICAgIGlmIChlcnIpIHsgdGhyb3cgZXJyOyB9XG4gICAgfSk7XG5cbiAgICBkeW5hbW9DbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKHtcbiAgICAgIGNyZWRlbnRpYWxzOiBuZXcgQVdTLkNyZWRlbnRpYWxzKHtcbiAgICAgICAgYWNjZXNzS2V5SWQ6ICcnLFxuICAgICAgICBzZWNyZXRBY2Nlc3NLZXk6ICcnLFxuICAgICAgfSksXG4gICAgICBlbmRwb2ludDogYGh0dHA6Ly9sb2NhbGhvc3Q6JHtkeW5hUG9ydH1gLFxuICAgICAgcmVnaW9uOiAndXMtd2VzdC0yJyxcbiAgICB9KTtcblxuICAgIGZ1bmN0aW9uIGNyZWF0ZVRhYmxlUmVxdWVzdCh0YWJsZU5hbWU6IHN0cmluZywgcHJpbWFyeUtleVR5cGU6IHN0cmluZywgc29ydEtleT86IHsgS2V5VHlwZTogc3RyaW5nIH0pOiBBV1MuRHluYW1vREIuQ3JlYXRlVGFibGVJbnB1dCB7XG4gICAgICBjb25zdCByZXF1ZXN0ID0ge1xuICAgICAgICBUYWJsZU5hbWU6IHRhYmxlTmFtZSxcbiAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnUHJpbUtleScsXG4gICAgICAgICAgICBBdHRyaWJ1dGVUeXBlOiBwcmltYXJ5S2V5VHlwZSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnUHJpbUtleScsXG4gICAgICAgICAgICBLZXlUeXBlOiAnSEFTSCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgQmlsbGluZ01vZGU6ICdQQVlfUEVSX1JFUVVFU1QnLFxuICAgICAgICBUYWdzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgS2V5OiAnUkZES1Rlc3RpbmcnLFxuICAgICAgICAgICAgVmFsdWU6ICdSRkRLVGVzdGluZycsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH07XG4gICAgICBpZiAoc29ydEtleSkge1xuICAgICAgICByZXF1ZXN0LkF0dHJpYnV0ZURlZmluaXRpb25zLnB1c2goe1xuICAgICAgICAgIEF0dHJpYnV0ZU5hbWU6ICdTb3J0S2V5JyxcbiAgICAgICAgICBBdHRyaWJ1dGVUeXBlOiBzb3J0S2V5LktleVR5cGUsXG4gICAgICAgIH0pO1xuICAgICAgICByZXF1ZXN0LktleVNjaGVtYS5wdXNoKHtcbiAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnU29ydEtleScsXG4gICAgICAgICAgS2V5VHlwZTogJ1JBTkdFJyxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgICByZXR1cm4gcmVxdWVzdDtcbiAgICB9XG5cbiAgICBsZXQgcmVxdWVzdCA9IGNyZWF0ZVRhYmxlUmVxdWVzdChHT09EX1RBQkxFX05BTUUsICdTJywgeyBLZXlUeXBlOiAnUycgfSk7XG4gICAgbGV0IHJlc3BvbnNlOiBBV1MuRHluYW1vREIuQ3JlYXRlVGFibGVPdXRwdXQgPSBhd2FpdCBkeW5hbW9DbGllbnQuY3JlYXRlVGFibGUocmVxdWVzdCkucHJvbWlzZSgpO1xuICAgIGxldCB0YWJsZSA9IHJlc3BvbnNlLlRhYmxlRGVzY3JpcHRpb247XG4gICAgaWYgKCF0YWJsZSkgeyB0aHJvdyBFcnJvcihgQ291bGQgbm90IGNyZWF0ZSAke0dPT0RfVEFCTEVfTkFNRX1gKTsgfVxuICAgIGNvbnNvbGUuZGVidWcoYENyZWF0ZWQgRHluYW1vREIgdGFibGU6ICcke3RhYmxlLlRhYmxlTmFtZX0nYCk7XG5cbiAgICByZXF1ZXN0ID0gY3JlYXRlVGFibGVSZXF1ZXN0KEJBRF9UQUJMRTFfTkFNRSwgJ1MnKTtcbiAgICByZXNwb25zZSA9IGF3YWl0IGR5bmFtb0NsaWVudC5jcmVhdGVUYWJsZShyZXF1ZXN0KS5wcm9taXNlKCk7XG4gICAgdGFibGUgPSByZXNwb25zZS5UYWJsZURlc2NyaXB0aW9uO1xuICAgIGlmICghdGFibGUpIHsgdGhyb3cgRXJyb3IoYENvdWxkIG5vdCBjcmVhdGUgJHtCQURfVEFCTEUxX05BTUV9YCk7IH1cbiAgICBjb25zb2xlLmRlYnVnKGBDcmVhdGVkIER5bmFtb0RCIHRhYmxlOiAnJHt0YWJsZS5UYWJsZU5hbWV9J2ApO1xuXG4gICAgcmVxdWVzdCA9IGNyZWF0ZVRhYmxlUmVxdWVzdChCQURfVEFCTEUyX05BTUUsICdOJywgeyBLZXlUeXBlOiAnUycgfSk7XG4gICAgcmVzcG9uc2UgPSBhd2FpdCBkeW5hbW9DbGllbnQuY3JlYXRlVGFibGUocmVxdWVzdCkucHJvbWlzZSgpO1xuICAgIHRhYmxlID0gcmVzcG9uc2UuVGFibGVEZXNjcmlwdGlvbjtcbiAgICBpZiAoIXRhYmxlKSB7IHRocm93IEVycm9yKGBDb3VsZCBub3QgY3JlYXRlICR7QkFEX1RBQkxFMl9OQU1FfWApOyB9XG4gICAgY29uc29sZS5kZWJ1ZyhgQ3JlYXRlZCBEeW5hbW9EQiB0YWJsZTogJyR7dGFibGUuVGFibGVOYW1lfSdgKTtcblxuICAgIHJlcXVlc3QgPSBjcmVhdGVUYWJsZVJlcXVlc3QoQkFEX1RBQkxFM19OQU1FLCAnUycsIHsgS2V5VHlwZTogJ04nIH0pO1xuICAgIHJlc3BvbnNlID0gYXdhaXQgZHluYW1vQ2xpZW50LmNyZWF0ZVRhYmxlKHJlcXVlc3QpLnByb21pc2UoKTtcbiAgICB0YWJsZSA9IHJlc3BvbnNlLlRhYmxlRGVzY3JpcHRpb247XG4gICAgaWYgKCF0YWJsZSkgeyB0aHJvdyBFcnJvcihgQ291bGQgbm90IGNyZWF0ZSAke0JBRF9UQUJMRTNfTkFNRX1gKTsgfVxuICAgIGNvbnNvbGUuZGVidWcoYENyZWF0ZWQgRHluYW1vREIgdGFibGU6ICcke3RhYmxlLlRhYmxlTmFtZX0nYCk7XG5cbiAgICBsZXQgd2FpdGluZzogYm9vbGVhbiA9IHRydWU7XG4gICAgZG8ge1xuICAgICAgY29uc3QgcHJvbWlzZXMgPSBbXTtcbiAgICAgIGZvciAoY29uc3QgbmFtZSBvZiBbR09PRF9UQUJMRV9OQU1FLCBCQURfVEFCTEUxX05BTUUsIEJBRF9UQUJMRTJfTkFNRSwgQkFEX1RBQkxFM19OQU1FXSkge1xuICAgICAgICBwcm9taXNlcy5wdXNoKGR5bmFtb0NsaWVudC5kZXNjcmliZVRhYmxlKHtcbiAgICAgICAgICBUYWJsZU5hbWU6IG5hbWUsXG4gICAgICAgIH0pLnByb21pc2UoKSk7XG4gICAgICB9XG4gICAgICBjb25zdCByZXNwb25zZXMgPSBhd2FpdCBQcm9taXNlLmFsbChwcm9taXNlcyk7XG4gICAgICB3YWl0aW5nID0gIXJlc3BvbnNlcy5ldmVyeShpdGVtID0+IGl0ZW0uVGFibGU/LlRhYmxlU3RhdHVzID09PSAnQUNUSVZFJyk7XG4gICAgICBhd2FpdCBuZXcgUHJvbWlzZShyZXNvbHZlID0+IHNldFRpbWVvdXQocmVzb2x2ZSwgMjAwMCkpO1xuICAgIH0gd2hpbGUgKHdhaXRpbmcpO1xuICB9LCA2MDAwMCk7XG5cbiAgYWZ0ZXJBbGwoYXN5bmMgKCkgPT4ge1xuICAgIGF3YWl0IGR5bmFsaXRlU2VydmVyLmNsb3NlKCk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlbGV0ZSB0YWJsZSBSZXNvdXJjZU5vdEZvdW5kRXhjZXB0aW9uJywgYXN5bmMgKCkgPT4ge1xuICAgIGNvbnN0IHRhYmxlTmFtZTogc3RyaW5nID0gJ05vbkV4aXN0YW50VGFibGUnO1xuICAgIGNvbnN0IHBrOiBzdHJpbmcgPSAnUHJpbUtleSc7XG4gICAgY29uc3Qgc2s6IHN0cmluZyA9ICdTb3J0S2V5JztcblxuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRlc3RUYWJsZShcbiAgICAgIGR5bmFtb0NsaWVudCxcbiAgICAgIHRhYmxlTmFtZSxcbiAgICAgIHBrLFxuICAgICAgc2ssXG4gICAgKTtcbiAgICBleHBlY3QodGFibGUudGFibGVOYW1lKS50b0JlRGVmaW5lZCgpO1xuICAgIGF3YWl0IGV4cGVjdCh0YWJsZS5kZWxldGVUYWJsZSgpKS5yZXNvbHZlcy5ub3QudG9UaHJvd0Vycm9yKCk7XG4gICAgZXhwZWN0KHRhYmxlLnRhYmxlTmFtZSkudG9CZVVuZGVmaW5lZCgpO1xuICB9KTtcblxuICBkZXNjcmliZSgnZnJvbUV4aXN0aW5nIGZhaWxzIG9uIGJhZCB0YWJsZSBuYW1lJywgKCkgPT4ge1xuICAgIHRlc3QuZWFjaChbXG4gICAgICBbQkFEX1RBQkxFMV9OQU1FXSxcbiAgICAgIFtCQURfVEFCTEUyX05BTUVdLFxuICAgICAgW0JBRF9UQUJMRTNfTkFNRV0sXG4gICAgXSkoJ3RhYmxlTmFtZSA9ICVwJywgYXN5bmMgKHRhYmxlTmFtZTogc3RyaW5nKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBhd2FpdCBleHBlY3QoQ29tcG9zaXRlU3RyaW5nSW5kZXhUYWJsZS5mcm9tRXhpc3RpbmcoZHluYW1vQ2xpZW50LCB0YWJsZU5hbWUpKVxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93KCk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2Zyb21FeGlzaW5nIHN1Y2NlZWRzIG9uIGdvb2QgdGFibGUgbmFtZScsIGFzeW5jICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFibGUgPSBhd2FpdCBDb21wb3NpdGVTdHJpbmdJbmRleFRhYmxlLmZyb21FeGlzdGluZyhkeW5hbW9DbGllbnQsIEdPT0RfVEFCTEVfTkFNRSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHRhYmxlLnByaW1hcnlLZXkpLnRvQmUoJ1ByaW1LZXknKTtcbiAgICBleHBlY3QodGFibGUuc29ydEtleSkudG9CZSgnU29ydEtleScpO1xuICB9KTtcblxuICB0ZXN0KCdwdXRJdGVtL2dldEl0ZW0gc3VjY2VzcycsIGFzeW5jICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHRhYmxlID0gYXdhaXQgQ29tcG9zaXRlU3RyaW5nSW5kZXhUYWJsZS5mcm9tRXhpc3RpbmcoZHluYW1vQ2xpZW50LCBHT09EX1RBQkxFX05BTUUpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGF0dHJpYnV0ZXMgPSB7XG4gICAgICBUZXN0QXR0cmlidXRlOiAndGVzdCB2YWx1ZScsXG4gICAgfTtcbiAgICBjb25zdCBwdXRSZXF1ZXN0ID0ge1xuICAgICAgcHJpbWFyeUtleVZhbHVlOiByYW5kb21CeXRlcygyNCkudG9TdHJpbmcoJ2hleCcpLFxuICAgICAgc29ydEtleVZhbHVlOiByYW5kb21CeXRlcygyNCkudG9TdHJpbmcoJ2hleCcpLFxuICAgICAgYXR0cmlidXRlcyxcbiAgICB9O1xuICAgIGNvbnN0IGdldFJlcXVlc3QgPSB7XG4gICAgICBwcmltYXJ5S2V5VmFsdWU6IHB1dFJlcXVlc3QucHJpbWFyeUtleVZhbHVlLFxuICAgICAgc29ydEtleVZhbHVlOiBwdXRSZXF1ZXN0LnNvcnRLZXlWYWx1ZSxcbiAgICB9O1xuXG4gICAgLy8gVEhFTlxuICAgIGF3YWl0IGV4cGVjdCh0YWJsZS5wdXRJdGVtKHB1dFJlcXVlc3QpKS5yZXNvbHZlcy50b0JlKHRydWUpO1xuICAgIGF3YWl0IGV4cGVjdCh0YWJsZS5nZXRJdGVtKGdldFJlcXVlc3QpKS5yZXNvbHZlcy50b01hdGNoT2JqZWN0KGF0dHJpYnV0ZXMpO1xuICB9KTtcblxuICB0ZXN0LmVhY2goW1xuICAgIFt0cnVlXSwgW2ZhbHNlXSxcbiAgXSkoJ3B1dEl0ZW0gb3ZlcndyaXRlICVwJywgYXN5bmMgKGFsbG93T3ZlcndyaXRlOiBib29sZWFuKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB0YWJsZSA9IGF3YWl0IENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuZnJvbUV4aXN0aW5nKGR5bmFtb0NsaWVudCwgR09PRF9UQUJMRV9OQU1FKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBwdXRSZXF1ZXN0ID0ge1xuICAgICAgcHJpbWFyeUtleVZhbHVlOiByYW5kb21CeXRlcygyNCkudG9TdHJpbmcoJ2hleCcpLFxuICAgICAgc29ydEtleVZhbHVlOiByYW5kb21CeXRlcygyNCkudG9TdHJpbmcoJ2hleCcpLFxuICAgICAgYXR0cmlidXRlczoge1xuICAgICAgICBUZXN0QXR0cmlidXRlOiAndGVzdCB2YWx1ZScsXG4gICAgICB9LFxuICAgICAgYWxsb3dfb3ZlcndyaXRlOiB0cnVlLFxuICAgIH07XG4gICAgYXdhaXQgdGFibGUucHV0SXRlbShwdXRSZXF1ZXN0KTtcblxuICAgIC8vIFRIRU5cbiAgICBwdXRSZXF1ZXN0LmFsbG93X292ZXJ3cml0ZSA9IGFsbG93T3ZlcndyaXRlO1xuICAgIGF3YWl0IGV4cGVjdCh0YWJsZS5wdXRJdGVtKHB1dFJlcXVlc3QpKS5yZXNvbHZlcy50b0JlKGFsbG93T3ZlcndyaXRlKTtcbiAgfSk7XG5cbiAgdGVzdCgnZ2V0SXRlbSBubyBtYXRjaCcsIGFzeW5jICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHRhYmxlID0gYXdhaXQgQ29tcG9zaXRlU3RyaW5nSW5kZXhUYWJsZS5mcm9tRXhpc3RpbmcoZHluYW1vQ2xpZW50LCBHT09EX1RBQkxFX05BTUUpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGdldFJlcXVlc3QgPSB7XG4gICAgICBwcmltYXJ5S2V5VmFsdWU6IHJhbmRvbUJ5dGVzKDI0KS50b1N0cmluZygnaGV4JyksXG4gICAgICBzb3J0S2V5VmFsdWU6IHJhbmRvbUJ5dGVzKDI0KS50b1N0cmluZygnaGV4JyksXG4gICAgfTtcblxuICAgIC8vIFRIRU5cbiAgICBhd2FpdCBleHBlY3QodGFibGUuZ2V0SXRlbShnZXRSZXF1ZXN0KSkucmVzb2x2ZXMudG9CZVVuZGVmaW5lZCgpO1xuICB9KTtcblxuICB0ZXN0KCdkZWxldGVJdGVtIHN1Y2Nlc3MnLCBhc3luYyAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB0YWJsZSA9IGF3YWl0IENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuZnJvbUV4aXN0aW5nKGR5bmFtb0NsaWVudCwgR09PRF9UQUJMRV9OQU1FKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhdHRyaWJ1dGVzID0ge1xuICAgICAgVGVzdEF0dHJpYnV0ZTogJ3Rlc3QgdmFsdWUnLFxuICAgIH07XG4gICAgY29uc3QgcHV0UmVxdWVzdCA9IHtcbiAgICAgIHByaW1hcnlLZXlWYWx1ZTogcmFuZG9tQnl0ZXMoMjQpLnRvU3RyaW5nKCdoZXgnKSxcbiAgICAgIHNvcnRLZXlWYWx1ZTogcmFuZG9tQnl0ZXMoMjQpLnRvU3RyaW5nKCdoZXgnKSxcbiAgICAgIGF0dHJpYnV0ZXMsXG4gICAgfTtcbiAgICBjb25zdCBkZWxldGVSZXF1ZXN0ID0ge1xuICAgICAgcHJpbWFyeUtleVZhbHVlOiBwdXRSZXF1ZXN0LnByaW1hcnlLZXlWYWx1ZSxcbiAgICAgIHNvcnRLZXlWYWx1ZTogcHV0UmVxdWVzdC5zb3J0S2V5VmFsdWUsXG4gICAgfTtcbiAgICBhd2FpdCB0YWJsZS5wdXRJdGVtKHB1dFJlcXVlc3QpO1xuXG4gICAgLy8gVEhFTlxuICAgIGF3YWl0IGV4cGVjdCh0YWJsZS5kZWxldGVJdGVtKGRlbGV0ZVJlcXVlc3QpKS5yZXNvbHZlcy50b0JlKHRydWUpO1xuICB9KTtcblxuICB0ZXN0KCdkZWxldGVJdGVtIG5vIG1hdGNoJywgYXN5bmMgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgdGFibGUgPSBhd2FpdCBDb21wb3NpdGVTdHJpbmdJbmRleFRhYmxlLmZyb21FeGlzdGluZyhkeW5hbW9DbGllbnQsIEdPT0RfVEFCTEVfTkFNRSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgZGVsZXRlUmVxdWVzdCA9IHtcbiAgICAgIHByaW1hcnlLZXlWYWx1ZTogcmFuZG9tQnl0ZXMoMjQpLnRvU3RyaW5nKCdoZXgnKSxcbiAgICAgIHNvcnRLZXlWYWx1ZTogcmFuZG9tQnl0ZXMoMjQpLnRvU3RyaW5nKCdoZXgnKSxcbiAgICB9O1xuXG4gICAgLy8gVEhFTlxuICAgIGF3YWl0IGV4cGVjdCh0YWJsZS5kZWxldGVJdGVtKGRlbGV0ZVJlcXVlc3QpKS5yZXNvbHZlcy50b0JlKGZhbHNlKTtcbiAgfSk7XG5cbiAgdGVzdCgncXVlcnkgc3VjY2VzcycsIGFzeW5jICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHRhYmxlID0gYXdhaXQgQ29tcG9zaXRlU3RyaW5nSW5kZXhUYWJsZS5mcm9tRXhpc3RpbmcoZHluYW1vQ2xpZW50LCBHT09EX1RBQkxFX05BTUUpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHByaW1hcnlLZXlWYWx1ZTogc3RyaW5nID0gcmFuZG9tQnl0ZXMoMjQpLnRvU3RyaW5nKCdoZXgnKTtcbiAgICBjb25zdCBleHBlY3RlZDogeyBba2V5OiBzdHJpbmddOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IH0gPSB7fTtcbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IDIwOyBpKyspIHtcbiAgICAgIGNvbnN0IHNvcnRLZXlWYWx1ZTogc3RyaW5nID0gcmFuZG9tQnl0ZXMoMjQpLnRvU3RyaW5nKCdoZXgnKTtcbiAgICAgIGV4cGVjdGVkW3NvcnRLZXlWYWx1ZV0gPSB7XG4gICAgICAgIFRlc3RBdHRyaWJ1dGU6IGB2YWx1ZSR7aX1gLFxuICAgICAgfTtcbiAgICB9XG4gICAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMoZXhwZWN0ZWQpKSB7XG4gICAgICBjb25zdCBwdXRSZXF1ZXN0ID0ge1xuICAgICAgICBwcmltYXJ5S2V5VmFsdWUsXG4gICAgICAgIHNvcnRLZXlWYWx1ZToga2V5LFxuICAgICAgICBhdHRyaWJ1dGVzOiB2YWx1ZSxcbiAgICAgIH07XG4gICAgICBhd2FpdCB0YWJsZS5wdXRJdGVtKHB1dFJlcXVlc3QpO1xuICAgIH1cblxuICAgIC8vIFRIRU5cbiAgICBhd2FpdCBleHBlY3QodGFibGUucXVlcnkocHJpbWFyeUtleVZhbHVlLCA1KSkucmVzb2x2ZXMudG9NYXRjaE9iamVjdChleHBlY3RlZCk7XG4gIH0pO1xufSk7XG5cbmRlc2NyaWJlKCdUZXN0cyB1c2luZyBhd3Mtc2RrLW1vY2snLCAoKSA9PiB7XG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIHNldFNES0luc3RhbmNlKEFXUyk7XG4gIH0pO1xuXG4gIGFmdGVyRWFjaCgoKSA9PiB7XG4gICAgcmVzdG9yZSgnRHluYW1vREInKTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2Zyb21FeGlzdGluZyB0ZXN0cycsICgpID0+IHtcbiAgICB0ZXN0KCdUYWJsZSBub3QgZm91bmQnLCBhc3luYyAoKSA9PiB7XG4gICAgICBjb25zdCBjYWxsYmFjayA9IGZha2UucmVzb2x2ZXMoeyBUYWJsZTogdW5kZWZpbmVkIH0pO1xuICAgICAgbW9jaygnRHluYW1vREInLCAnZGVzY3JpYmVUYWJsZScsIGNhbGxiYWNrKTtcbiAgICAgIGNvbnN0IGNsaWVudCA9IG5ldyBBV1MuRHluYW1vREIoKTtcbiAgICAgIGNvbnN0IHRhYmxlTmFtZSA9ICdOb25leGlzdGFudCc7XG5cbiAgICAgIGF3YWl0IGV4cGVjdChDb21wb3NpdGVTdHJpbmdJbmRleFRhYmxlLmZyb21FeGlzdGluZyhjbGllbnQsIHRhYmxlTmFtZSkpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoYENvdWxkIG5vdCBkZXNjcmliZVRhYmxlIGZvciBUYWJsZSAnJHt0YWJsZU5hbWV9J2ApO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnS2V5U2NoZW1hIG5vdCBmb3VuZCcsIGFzeW5jICgpID0+IHtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ2Rlc2NyaWJlVGFibGUnLCBmYWtlLnJlc29sdmVzKHsgVGFibGU6IHsgS2V5U2NoZW1hOiB1bmRlZmluZWQgfSB9KSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCB0YWJsZU5hbWUgPSAnVGVzdFRhYmxlJztcblxuICAgICAgYXdhaXQgZXhwZWN0KENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuZnJvbUV4aXN0aW5nKGNsaWVudCwgdGFibGVOYW1lKSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3dFcnJvcihgQ291bGQgbm90IGdldCBLZXlTY2hlbWEgZm9yIFRhYmxlICcke3RhYmxlTmFtZX0nYCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdBdHRyaWJ1dGVEZWZpbml0aW9ucyBub3QgZm91bmQnLCBhc3luYyAoKSA9PiB7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdkZXNjcmliZVRhYmxlJywgZmFrZS5yZXNvbHZlcyh7XG4gICAgICAgIFRhYmxlOiB7XG4gICAgICAgICAgS2V5U2NoZW1hOiBbXSxcbiAgICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogdW5kZWZpbmVkLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuICAgICAgY29uc3QgY2xpZW50ID0gbmV3IEFXUy5EeW5hbW9EQigpO1xuICAgICAgY29uc3QgdGFibGVOYW1lID0gJ1Rlc3RUYWJsZSc7XG5cbiAgICAgIGF3YWl0IGV4cGVjdChDb21wb3NpdGVTdHJpbmdJbmRleFRhYmxlLmZyb21FeGlzdGluZyhjbGllbnQsIHRhYmxlTmFtZSkpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoYENvdWxkIG5vdCBnZXQgQXR0cmlidXRlcyBmb3IgVGFibGUgJyR7dGFibGVOYW1lfSdgKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ1ByaW1hcnlLZXkgbm90IGZvdW5kJywgYXN5bmMgKCkgPT4ge1xuICAgICAgbW9jaygnRHluYW1vREInLCAnZGVzY3JpYmVUYWJsZScsIGZha2UucmVzb2x2ZXMoe1xuICAgICAgICBUYWJsZToge1xuICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnU29ydEtleScsXG4gICAgICAgICAgICAgIEtleVR5cGU6ICdSQU5HRScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IHt9LFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuICAgICAgY29uc3QgY2xpZW50ID0gbmV3IEFXUy5EeW5hbW9EQigpO1xuICAgICAgY29uc3QgdGFibGVOYW1lID0gJ1Rlc3RUYWJsZSc7XG5cbiAgICAgIGF3YWl0IGV4cGVjdChDb21wb3NpdGVTdHJpbmdJbmRleFRhYmxlLmZyb21FeGlzdGluZyhjbGllbnQsIHRhYmxlTmFtZSkpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoYENvdWxkIG5vdCBmaW5kIFByaW1hcnlLZXkgb2YgVGFibGUgJyR7dGFibGVOYW1lfSdgKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ1NvcnRLZXkgbm90IGZvdW5kJywgYXN5bmMgKCkgPT4ge1xuICAgICAgbW9jaygnRHluYW1vREInLCAnZGVzY3JpYmVUYWJsZScsIGZha2UucmVzb2x2ZXMoe1xuICAgICAgICBUYWJsZToge1xuICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnUHJpbWFyeUtleScsXG4gICAgICAgICAgICAgIEtleVR5cGU6ICdIQVNIJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczoge30sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCB0YWJsZU5hbWUgPSAnVGVzdFRhYmxlJztcblxuICAgICAgYXdhaXQgZXhwZWN0KENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuZnJvbUV4aXN0aW5nKGNsaWVudCwgdGFibGVOYW1lKSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3dFcnJvcihgQ291bGQgbm90IGZpbmQgU29ydEtleSBvZiBUYWJsZSAnJHt0YWJsZU5hbWV9J2ApO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnUHJpbWFyeUtleSBBdHRyaWJ1dGVEZWZpbml0aW9uIG5vdCBmb3VuZCcsIGFzeW5jICgpID0+IHtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ2Rlc2NyaWJlVGFibGUnLCBmYWtlLnJlc29sdmVzKHtcbiAgICAgICAgVGFibGU6IHtcbiAgICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQXR0cmlidXRlTmFtZTogJ1ByaW1hcnlLZXknLFxuICAgICAgICAgICAgICBLZXlUeXBlOiAnSEFTSCcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnU29ydEtleScsXG4gICAgICAgICAgICAgIEtleVR5cGU6ICdSQU5HRScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQXR0cmlidXRlTmFtZTogJ1NvcnRLZXknLFxuICAgICAgICAgICAgICBBdHRyaWJ1dGVUeXBlOiAnUycsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCB0YWJsZU5hbWUgPSAnVGVzdFRhYmxlJztcblxuICAgICAgYXdhaXQgZXhwZWN0KENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuZnJvbUV4aXN0aW5nKGNsaWVudCwgdGFibGVOYW1lKSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3dFcnJvcihcIlByaW1hcnkga2V5ICdQcmltYXJ5S2V5JyBtdXN0IGJlIHN0cmluZyB0eXBlXCIpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnU29ydEtleSBBdHRyaWJ1dGVEZWZpbml0aW9uIG5vdCBmb3VuZCcsIGFzeW5jICgpID0+IHtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ2Rlc2NyaWJlVGFibGUnLCBmYWtlLnJlc29sdmVzKHtcbiAgICAgICAgVGFibGU6IHtcbiAgICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQXR0cmlidXRlTmFtZTogJ1ByaW1hcnlLZXknLFxuICAgICAgICAgICAgICBLZXlUeXBlOiAnSEFTSCcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnU29ydEtleScsXG4gICAgICAgICAgICAgIEtleVR5cGU6ICdSQU5HRScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQXR0cmlidXRlTmFtZTogJ1ByaW1hcnlLZXknLFxuICAgICAgICAgICAgICBBdHRyaWJ1dGVUeXBlOiAnUycsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCB0YWJsZU5hbWUgPSAnVGVzdFRhYmxlJztcblxuICAgICAgYXdhaXQgZXhwZWN0KENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuZnJvbUV4aXN0aW5nKGNsaWVudCwgdGFibGVOYW1lKSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3dFcnJvcihcIlNvcnQga2V5ICdTb3J0S2V5JyBtdXN0IGJlIHN0cmluZyB0eXBlXCIpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnY3JlYXRlTmV3IHRlc3RzJywgKCkgPT4ge1xuICAgIHRlc3QoJ0R5bmFtb0RCLmNyZWF0ZVRhYmxlKCkgZmFpbHVyZSB0aHJvd3MgRXJyb3InLCBhc3luYyAoKSA9PiB7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdjcmVhdGVUYWJsZScsIGZha2UucmVqZWN0cyh7fSkpO1xuICAgICAgY29uc3QgY2xpZW50ID0gbmV3IEFXUy5EeW5hbW9EQigpO1xuICAgICAgY29uc3QgbmFtZSA9ICdUZXN0VGFibGUnO1xuICAgICAgY29uc3QgcHJpbWFyeUtleU5hbWUgPSAnUHJpbWFyeUtleSc7XG4gICAgICBjb25zdCBzb3J0S2V5TmFtZSA9ICdTb3J0S2V5JztcblxuICAgICAgYXdhaXQgZXhwZWN0KENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuY3JlYXRlTmV3KHtcbiAgICAgICAgY2xpZW50LFxuICAgICAgICBuYW1lLFxuICAgICAgICBwcmltYXJ5S2V5TmFtZSxcbiAgICAgICAgc29ydEtleU5hbWUsXG4gICAgICB9KSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3coKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3N1Y2Nlc3MnLCBhc3luYyAoKSA9PiB7XG4gICAgICBjb25zdCB0YWJsZU5hbWU6IHN0cmluZyA9ICdUZXN0VGFibGUnO1xuICAgICAgY29uc3QgcGs6IHN0cmluZyA9ICdQcmltS2V5JztcbiAgICAgIGNvbnN0IHNrOiBzdHJpbmcgPSAnU29ydEtleSc7XG4gICAgICBjb25zdCB0YWdzID0gW1xuICAgICAgICB7XG4gICAgICAgICAgS2V5OiAnUkZES1Rlc3RpbmcnLFxuICAgICAgICAgIFZhbHVlOiAnUkZES1Rlc3RpbmcnLFxuICAgICAgICB9LFxuICAgICAgXTtcblxuICAgICAgbW9jaygnRHluYW1vREInLCAnY3JlYXRlVGFibGUnLCBmYWtlLnJlc29sdmVzKHt9KSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCB0YWJsZSA9IGF3YWl0IENvbXBvc2l0ZVN0cmluZ0luZGV4VGFibGUuY3JlYXRlTmV3KHtcbiAgICAgICAgY2xpZW50LFxuICAgICAgICBuYW1lOiB0YWJsZU5hbWUsXG4gICAgICAgIHByaW1hcnlLZXlOYW1lOiBwayxcbiAgICAgICAgc29ydEtleU5hbWU6IHNrLFxuICAgICAgICB0YWdzLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdCh0YWJsZS5wcmltYXJ5S2V5KS50b0JlKHBrKTtcbiAgICAgIGV4cGVjdCh0YWJsZS5zb3J0S2V5KS50b0JlKHNrKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2RlbGV0ZVRhYmxlIHRlc3RzJywgKCkgPT4ge1xuICAgIHRlc3QoJ3N1Y2Nlc3MnLCBhc3luYyAoKSA9PiB7XG4gICAgICBjb25zdCB0YWJsZU5hbWU6IHN0cmluZyA9ICdUZXN0VGFibGUnO1xuICAgICAgY29uc3QgcGs6IHN0cmluZyA9ICdQcmltS2V5JztcbiAgICAgIGNvbnN0IHNrOiBzdHJpbmcgPSAnU29ydEtleSc7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdkZWxldGVUYWJsZScsIGZha2UucmVzb2x2ZXMoe30pKTtcblxuICAgICAgY29uc3QgY2xpZW50ID0gbmV3IEFXUy5EeW5hbW9EQigpO1xuICAgICAgY29uc3QgdGFibGUgPSBuZXcgVGVzdFRhYmxlKFxuICAgICAgICBjbGllbnQsXG4gICAgICAgIHRhYmxlTmFtZSxcbiAgICAgICAgcGssXG4gICAgICAgIHNrLFxuICAgICAgKTtcblxuICAgICAgYXdhaXQgZXhwZWN0KHRhYmxlLmRlbGV0ZVRhYmxlKCkpLnJlc29sdmVzLm5vdC50b1Rocm93KCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdUYWJsZSBhbHJlYWR5IGRlbGV0ZWQnLCBhc3luYyAoKSA9PiB7XG4gICAgICBjb25zdCBkZWxldGVGYWtlID0gZmFrZS5yZXNvbHZlcyh7fSk7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdkZWxldGVUYWJsZScsIGRlbGV0ZUZha2UpO1xuICAgICAgY29uc3QgY2xpZW50ID0gbmV3IEFXUy5EeW5hbW9EQigpO1xuICAgICAgY29uc3QgbmFtZSA9ICdUZXN0VGFibGUnO1xuICAgICAgY29uc3QgcHJpbWFyeUtleU5hbWUgPSAnUHJpbWFyeUtleSc7XG4gICAgICBjb25zdCBzb3J0S2V5TmFtZSA9ICdTb3J0S2V5JztcbiAgICAgIGNvbnN0IHN1YmplY3QgPSBuZXcgVGVzdFRhYmxlKFxuICAgICAgICBjbGllbnQsXG4gICAgICAgIG5hbWUsXG4gICAgICAgIHByaW1hcnlLZXlOYW1lLFxuICAgICAgICBzb3J0S2V5TmFtZSxcbiAgICAgICk7XG4gICAgICBhd2FpdCBzdWJqZWN0LmRlbGV0ZVRhYmxlKCk7XG5cbiAgICAgIGF3YWl0IGV4cGVjdChzdWJqZWN0LmRlbGV0ZVRhYmxlKCkpLnJlc29sdmVzLnRvQmUodW5kZWZpbmVkKTtcbiAgICAgIGV4cGVjdChkZWxldGVGYWtlLmNhbGxDb3VudCkudG9FcXVhbCgxKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ0R5bmFtb0RCLmRlbGV0ZVRhYmxlKCkgZmFpbHVyZScsIGFzeW5jICgpID0+IHtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ2RlbGV0ZVRhYmxlJywgZmFrZS5yZWplY3RzKG5ldyBFcnJvcigpKSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCBuYW1lID0gJ1Rlc3RUYWJsZSc7XG4gICAgICBjb25zdCBwcmltYXJ5S2V5TmFtZSA9ICdQcmltYXJ5S2V5JztcbiAgICAgIGNvbnN0IHNvcnRLZXlOYW1lID0gJ1NvcnRLZXknO1xuICAgICAgY29uc3Qgc3ViamVjdCA9IG5ldyBUZXN0VGFibGUoXG4gICAgICAgIGNsaWVudCxcbiAgICAgICAgbmFtZSxcbiAgICAgICAgcHJpbWFyeUtleU5hbWUsXG4gICAgICAgIHNvcnRLZXlOYW1lLFxuICAgICAgKTtcbiAgICAgIGF3YWl0IGV4cGVjdChzdWJqZWN0LmRlbGV0ZVRhYmxlKCkpLnJlamVjdHMudG9UaHJvd0Vycm9yKCk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdwdXRJdGVtIHRlc3RzJywgKCkgPT4ge1xuICAgIHRlc3QoJ1RhYmxlIGFscmVhZHkgZGVsZXRlZCcsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IGRlbGV0ZUZha2UgPSBmYWtlLnJlc29sdmVzKHt9KTtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ2RlbGV0ZVRhYmxlJywgZGVsZXRlRmFrZSk7XG4gICAgICBjb25zdCBwdXRGYWtlID0gZmFrZS5yZXNvbHZlcyh7fSk7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdwdXRJdGVtJywgcHV0RmFrZSk7XG5cbiAgICAgIGNvbnN0IGNsaWVudCA9IG5ldyBBV1MuRHluYW1vREIoKTtcbiAgICAgIGNvbnN0IG5hbWUgPSAnVGVzdFRhYmxlJztcbiAgICAgIGNvbnN0IHByaW1hcnlLZXlOYW1lID0gJ1ByaW1hcnlLZXknO1xuICAgICAgY29uc3Qgc29ydEtleU5hbWUgPSAnU29ydEtleSc7XG4gICAgICBjb25zdCBzdWJqZWN0ID0gbmV3IFRlc3RUYWJsZShcbiAgICAgICAgY2xpZW50LFxuICAgICAgICBuYW1lLFxuICAgICAgICBwcmltYXJ5S2V5TmFtZSxcbiAgICAgICAgc29ydEtleU5hbWUsXG4gICAgICApO1xuICAgICAgYXdhaXQgc3ViamVjdC5kZWxldGVUYWJsZSgpO1xuXG4gICAgICBhd2FpdCBleHBlY3Qoc3ViamVjdC5wdXRJdGVtKHsgcHJpbWFyeUtleVZhbHVlOiAnVGVzdFByaW1WYWwnLCBzb3J0S2V5VmFsdWU6ICdUZXN0U29ydFZhbCcgfSkpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoJ0F0dGVtcHQgdG8gUHV0SXRlbSBpbiBkZWxldGVkIHRhYmxlJyk7XG4gICAgICBleHBlY3QoZGVsZXRlRmFrZS5jYWxsQ291bnQpLnRvRXF1YWwoMSk7XG4gICAgICBleHBlY3QocHV0RmFrZS5ub3RDYWxsZWQpLnRvQmVUcnV0aHkoKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ0R5bmFtb0RCLnB1dEl0ZW0oKSBmYWlsdXJlJywgYXN5bmMgKCkgPT4ge1xuICAgICAgbW9jaygnRHluYW1vREInLCAncHV0SXRlbScsIGZha2UucmVqZWN0cyhuZXcgRXJyb3IoKSkpO1xuICAgICAgY29uc3QgY2xpZW50ID0gbmV3IEFXUy5EeW5hbW9EQigpO1xuICAgICAgY29uc3QgbmFtZSA9ICdUZXN0VGFibGUnO1xuICAgICAgY29uc3QgcHJpbWFyeUtleU5hbWUgPSAnUHJpbWFyeUtleSc7XG4gICAgICBjb25zdCBzb3J0S2V5TmFtZSA9ICdTb3J0S2V5JztcbiAgICAgIGNvbnN0IHN1YmplY3QgPSBuZXcgVGVzdFRhYmxlKFxuICAgICAgICBjbGllbnQsXG4gICAgICAgIG5hbWUsXG4gICAgICAgIHByaW1hcnlLZXlOYW1lLFxuICAgICAgICBzb3J0S2V5TmFtZSxcbiAgICAgICk7XG4gICAgICBhd2FpdCBleHBlY3Qoc3ViamVjdC5wdXRJdGVtKHsgcHJpbWFyeUtleVZhbHVlOiAnVGVzdFByaW1WYWwnLCBzb3J0S2V5VmFsdWU6ICdUZXN0U29ydFZhbCcgfSkpLnJlamVjdHMudG9UaHJvd0Vycm9yKCk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdnZXRJdGVtIHRlc3RzJywgKCkgPT4ge1xuICAgIHRlc3QoJ1RhYmxlIGFscmVhZHkgZGVsZXRlZCcsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IGRlbGV0ZUZha2UgPSBmYWtlLnJlc29sdmVzKHt9KTtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ2RlbGV0ZVRhYmxlJywgZGVsZXRlRmFrZSk7XG4gICAgICBjb25zdCBnZXRGYWtlID0gZmFrZS5yZXNvbHZlcyh7fSk7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdnZXRJdGVtJywgZ2V0RmFrZSk7XG5cbiAgICAgIGNvbnN0IGNsaWVudCA9IG5ldyBBV1MuRHluYW1vREIoKTtcbiAgICAgIGNvbnN0IG5hbWUgPSAnVGVzdFRhYmxlJztcbiAgICAgIGNvbnN0IHByaW1hcnlLZXlOYW1lID0gJ1ByaW1hcnlLZXknO1xuICAgICAgY29uc3Qgc29ydEtleU5hbWUgPSAnU29ydEtleSc7XG4gICAgICBjb25zdCBzdWJqZWN0ID0gbmV3IFRlc3RUYWJsZShcbiAgICAgICAgY2xpZW50LFxuICAgICAgICBuYW1lLFxuICAgICAgICBwcmltYXJ5S2V5TmFtZSxcbiAgICAgICAgc29ydEtleU5hbWUsXG4gICAgICApO1xuICAgICAgYXdhaXQgc3ViamVjdC5kZWxldGVUYWJsZSgpO1xuXG4gICAgICBhd2FpdCBleHBlY3Qoc3ViamVjdC5nZXRJdGVtKHsgcHJpbWFyeUtleVZhbHVlOiAnVGVzdFByaW1WYWwnLCBzb3J0S2V5VmFsdWU6ICdUZXN0U29ydFZhbCcgfSkpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoJ0F0dGVtcHQgdG8gR2V0SXRlbSBmcm9tIGRlbGV0ZWQgdGFibGUnKTtcbiAgICAgIGV4cGVjdChkZWxldGVGYWtlLmNhbGxDb3VudCkudG9FcXVhbCgxKTtcbiAgICAgIGV4cGVjdChnZXRGYWtlLm5vdENhbGxlZCkudG9CZVRydXRoeSgpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnRHluYW1vREIuZ2V0SXRlbSgpIGZhaWx1cmUnLCBhc3luYyAoKSA9PiB7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdnZXRJdGVtJywgZmFrZS5yZWplY3RzKG5ldyBFcnJvcigpKSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCBuYW1lID0gJ1Rlc3RUYWJsZSc7XG4gICAgICBjb25zdCBwcmltYXJ5S2V5TmFtZSA9ICdQcmltYXJ5S2V5JztcbiAgICAgIGNvbnN0IHNvcnRLZXlOYW1lID0gJ1NvcnRLZXknO1xuICAgICAgY29uc3Qgc3ViamVjdCA9IG5ldyBUZXN0VGFibGUoXG4gICAgICAgIGNsaWVudCxcbiAgICAgICAgbmFtZSxcbiAgICAgICAgcHJpbWFyeUtleU5hbWUsXG4gICAgICAgIHNvcnRLZXlOYW1lLFxuICAgICAgKTtcbiAgICAgIGF3YWl0IGV4cGVjdChzdWJqZWN0LmdldEl0ZW0oeyBwcmltYXJ5S2V5VmFsdWU6ICdUZXN0UHJpbVZhbCcsIHNvcnRLZXlWYWx1ZTogJ1Rlc3RTb3J0VmFsJyB9KSkucmVqZWN0cy50b1Rocm93RXJyb3IoKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2RlbGV0ZUl0ZW0gdGVzdHMnLCAoKSA9PiB7XG4gICAgdGVzdCgnVGFibGUgYWxyZWFkeSBkZWxldGVkJywgYXN5bmMgKCkgPT4ge1xuICAgICAgY29uc3QgZGVsZXRlVGFibGVGYWtlID0gZmFrZS5yZXNvbHZlcyh7fSk7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdkZWxldGVUYWJsZScsIGRlbGV0ZVRhYmxlRmFrZSk7XG4gICAgICBjb25zdCBkZWxldGVJdGVtRmFrZSA9IGZha2UucmVzb2x2ZXMoe30pO1xuICAgICAgbW9jaygnRHluYW1vREInLCAnZGVsZXRlSXRlbScsIGRlbGV0ZUl0ZW1GYWtlKTtcblxuICAgICAgY29uc3QgY2xpZW50ID0gbmV3IEFXUy5EeW5hbW9EQigpO1xuICAgICAgY29uc3QgbmFtZSA9ICdUZXN0VGFibGUnO1xuICAgICAgY29uc3QgcHJpbWFyeUtleU5hbWUgPSAnUHJpbWFyeUtleSc7XG4gICAgICBjb25zdCBzb3J0S2V5TmFtZSA9ICdTb3J0S2V5JztcbiAgICAgIGNvbnN0IHN1YmplY3QgPSBuZXcgVGVzdFRhYmxlKFxuICAgICAgICBjbGllbnQsXG4gICAgICAgIG5hbWUsXG4gICAgICAgIHByaW1hcnlLZXlOYW1lLFxuICAgICAgICBzb3J0S2V5TmFtZSxcbiAgICAgICk7XG4gICAgICBhd2FpdCBzdWJqZWN0LmRlbGV0ZVRhYmxlKCk7XG5cbiAgICAgIGF3YWl0IGV4cGVjdChzdWJqZWN0LmRlbGV0ZUl0ZW0oeyBwcmltYXJ5S2V5VmFsdWU6ICdUZXN0UHJpbVZhbCcsIHNvcnRLZXlWYWx1ZTogJ1Rlc3RTb3J0VmFsJyB9KSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3dFcnJvcignQXR0ZW1wdCB0byBEZWxldGVJdGVtIGZyb20gZGVsZXRlZCB0YWJsZScpO1xuICAgICAgZXhwZWN0KGRlbGV0ZVRhYmxlRmFrZS5jYWxsQ291bnQpLnRvRXF1YWwoMSk7XG4gICAgICBleHBlY3QoZGVsZXRlSXRlbUZha2Uubm90Q2FsbGVkKS50b0JlVHJ1dGh5KCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdEeW5hbW9EQi5kZWxldGVJdGVtKCkgZmFpbHVyZScsIGFzeW5jICgpID0+IHtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ2RlbGV0ZUl0ZW0nLCBmYWtlLnJlamVjdHMobmV3IEVycm9yKCkpKTtcbiAgICAgIGNvbnN0IGNsaWVudCA9IG5ldyBBV1MuRHluYW1vREIoKTtcbiAgICAgIGNvbnN0IG5hbWUgPSAnVGVzdFRhYmxlJztcbiAgICAgIGNvbnN0IHByaW1hcnlLZXlOYW1lID0gJ1ByaW1hcnlLZXknO1xuICAgICAgY29uc3Qgc29ydEtleU5hbWUgPSAnU29ydEtleSc7XG4gICAgICBjb25zdCBzdWJqZWN0ID0gbmV3IFRlc3RUYWJsZShcbiAgICAgICAgY2xpZW50LFxuICAgICAgICBuYW1lLFxuICAgICAgICBwcmltYXJ5S2V5TmFtZSxcbiAgICAgICAgc29ydEtleU5hbWUsXG4gICAgICApO1xuICAgICAgYXdhaXQgZXhwZWN0KHN1YmplY3QuZGVsZXRlSXRlbSh7IHByaW1hcnlLZXlWYWx1ZTogJ1Rlc3RQcmltVmFsJywgc29ydEtleVZhbHVlOiAnVGVzdFNvcnRWYWwnIH0pKS5yZWplY3RzLnRvVGhyb3dFcnJvcigpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgncXVlcnkgdGVzdHMnLCAoKSA9PiB7XG4gICAgdGVzdCgnUmV0dXJucyBlbXB0eScsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IHF1ZXJ5RmFrZSA9IGZha2UucmVzb2x2ZXMoe30pO1xuICAgICAgbW9jaygnRHluYW1vREInLCAncXVlcnknLCBxdWVyeUZha2UpO1xuXG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCBuYW1lID0gJ1Rlc3RUYWJsZSc7XG4gICAgICBjb25zdCBwcmltYXJ5S2V5TmFtZSA9ICdQcmltYXJ5S2V5JztcbiAgICAgIGNvbnN0IHNvcnRLZXlOYW1lID0gJ1NvcnRLZXknO1xuICAgICAgY29uc3Qgc3ViamVjdCA9IG5ldyBUZXN0VGFibGUoXG4gICAgICAgIGNsaWVudCxcbiAgICAgICAgbmFtZSxcbiAgICAgICAgcHJpbWFyeUtleU5hbWUsXG4gICAgICAgIHNvcnRLZXlOYW1lLFxuICAgICAgKTtcblxuICAgICAgYXdhaXQgZXhwZWN0KHN1YmplY3QucXVlcnkoJ1Rlc3RQcmltVmFsJykpXG4gICAgICAgIC5yZXNvbHZlc1xuICAgICAgICAudG9FcXVhbCh7fSk7XG4gICAgICBleHBlY3QocXVlcnlGYWtlLmNhbGxDb3VudCkudG9FcXVhbCgxKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ1RhYmxlIGFscmVhZHkgZGVsZXRlZCcsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IGRlbGV0ZVRhYmxlRmFrZSA9IGZha2UucmVzb2x2ZXMoe30pO1xuICAgICAgbW9jaygnRHluYW1vREInLCAnZGVsZXRlVGFibGUnLCBkZWxldGVUYWJsZUZha2UpO1xuICAgICAgY29uc3QgcXVlcnlGYWtlID0gZmFrZS5yZXNvbHZlcyh7fSk7XG4gICAgICBtb2NrKCdEeW5hbW9EQicsICdxdWVyeScsIHF1ZXJ5RmFrZSk7XG5cbiAgICAgIGNvbnN0IGNsaWVudCA9IG5ldyBBV1MuRHluYW1vREIoKTtcbiAgICAgIGNvbnN0IG5hbWUgPSAnVGVzdFRhYmxlJztcbiAgICAgIGNvbnN0IHByaW1hcnlLZXlOYW1lID0gJ1ByaW1hcnlLZXknO1xuICAgICAgY29uc3Qgc29ydEtleU5hbWUgPSAnU29ydEtleSc7XG4gICAgICBjb25zdCBzdWJqZWN0ID0gbmV3IFRlc3RUYWJsZShcbiAgICAgICAgY2xpZW50LFxuICAgICAgICBuYW1lLFxuICAgICAgICBwcmltYXJ5S2V5TmFtZSxcbiAgICAgICAgc29ydEtleU5hbWUsXG4gICAgICApO1xuICAgICAgYXdhaXQgc3ViamVjdC5kZWxldGVUYWJsZSgpO1xuXG4gICAgICBhd2FpdCBleHBlY3Qoc3ViamVjdC5xdWVyeSgnVGVzdFByaW1WYWwnKSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3dFcnJvcignQXR0ZW1wdCB0byBRdWVyeSBhIGRlbGV0ZWQgdGFibGUnKTtcbiAgICAgIGV4cGVjdChkZWxldGVUYWJsZUZha2UuY2FsbENvdW50KS50b0VxdWFsKDEpO1xuICAgICAgZXhwZWN0KHF1ZXJ5RmFrZS5ub3RDYWxsZWQpLnRvQmVUcnV0aHkoKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ0R5bmFtb0RCLnF1ZXJ5KCkgZmFpbHVyZScsIGFzeW5jICgpID0+IHtcbiAgICAgIG1vY2soJ0R5bmFtb0RCJywgJ3F1ZXJ5JywgZmFrZS5yZWplY3RzKG5ldyBFcnJvcigpKSk7XG4gICAgICBjb25zdCBjbGllbnQgPSBuZXcgQVdTLkR5bmFtb0RCKCk7XG4gICAgICBjb25zdCBuYW1lID0gJ1Rlc3RUYWJsZSc7XG4gICAgICBjb25zdCBwcmltYXJ5S2V5TmFtZSA9ICdQcmltYXJ5S2V5JztcbiAgICAgIGNvbnN0IHNvcnRLZXlOYW1lID0gJ1NvcnRLZXknO1xuICAgICAgY29uc3Qgc3ViamVjdCA9IG5ldyBUZXN0VGFibGUoXG4gICAgICAgIGNsaWVudCxcbiAgICAgICAgbmFtZSxcbiAgICAgICAgcHJpbWFyeUtleU5hbWUsXG4gICAgICAgIHNvcnRLZXlOYW1lLFxuICAgICAgKTtcbiAgICAgIGF3YWl0IGV4cGVjdChzdWJqZWN0LnF1ZXJ5KCdUZXN0UHJpbVZhbCcpKS5yZWplY3RzLnRvVGhyb3dFcnJvcigpO1xuICAgIH0pO1xuICB9KTtcbn0pO1xuIl19