"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const http_1 = require("http");
const net_1 = require("net");
const deadline_client_1 = require("../../deadline-client");
const spot_event_plugin_client_1 = require("../spot-event-plugin-client");
describe('SpotEventPluginClient', () => {
    const poolsColection = {
        Pools: ['pool_name'],
        ObsoletePools: [],
    };
    const groupsColection = {
        Pools: ['group_name'],
        ObsoletePools: [],
    };
    const successfulPoolResponse = {
        data: { ...poolsColection },
        fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
    };
    const successfulGroupResponse = {
        data: { ...groupsColection },
        fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
    };
    let spotEventPluginClient;
    let describeDataResponse;
    let successfulResponse;
    let consoleLogMock;
    let consoleErrorMock;
    beforeEach(() => {
        consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
        consoleErrorMock = jest.spyOn(console, 'error').mockReturnValue(undefined);
        describeDataResponse = {
            data: {
                ServerData: [{
                        ID: 'event.plugin.spot',
                        ConcurrencyToken: 'token',
                    }],
            },
            fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
        };
        successfulResponse = {
            data: {},
            fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
        };
        spotEventPluginClient = new spot_event_plugin_client_1.SpotEventPluginClient(new deadline_client_1.DeadlineClient({
            host: 'test',
            port: 0,
            protocol: 'HTTP',
        }));
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn();
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn();
    });
    afterEach(() => {
        jest.clearAllMocks();
    });
    test('successful saveServerData', async () => {
        // GIVEN
        const configuration = 'configuration';
        const mockSuccessfulPostRequest = jest.fn((_a) => Promise.resolve(successfulResponse));
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['concurrencyToken'] = jest.fn().mockResolvedValue('token');
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = mockSuccessfulPostRequest;
        const result = await spotEventPluginClient.saveServerData(configuration);
        // THEN
        expect(result).toBeTruthy();
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledTimes(1);
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/Saving server data configuration:/);
        expect(consoleLogMock.mock.calls[1][0]).toMatch(configuration);
    });
    test('failed saveServerData on post request', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['concurrencyToken'] = jest.fn().mockResolvedValue('token');
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockRejectedValue(statusMessage);
        const result = await spotEventPluginClient.saveServerData('configuration');
        // THEN
        expect(result).toBeFalsy();
        expect(consoleErrorMock.mock.calls.length).toBe(1);
        expect(consoleErrorMock.mock.calls[0][0]).toMatch(`Failed to save server data. Reason: ${statusMessage}`);
    });
    test('failed saveServerData on concurrency token', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['concurrencyToken'] = jest.fn().mockRejectedValue(statusMessage);
        const result = await spotEventPluginClient.saveServerData('configuration');
        // THEN
        expect(result).toBeFalsy();
        expect(consoleErrorMock.mock.calls.length).toBe(1);
        expect(consoleErrorMock.mock.calls[0][0]).toMatch(`Failed to save server data. Reason: ${statusMessage}`);
    });
    test('successful configureSpotEventPlugin', async () => {
        // GIVEN
        const configs = [
            {
                Key: 'testkey',
                Value: 'testValue',
            },
            {
                Key: 'testkey2',
                Value: 'testValue2',
            },
        ];
        const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(successfulResponse));
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = mockConfigureSpotEventPlugin;
        const result = await spotEventPluginClient.configureSpotEventPlugin(configs);
        // THEN
        expect(result).toBeTruthy();
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledTimes(1);
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/Saving plugin configuration:/);
        expect(consoleLogMock.mock.calls[1][0]).toEqual(configs);
    });
    test('failed configureSpotEventPlugin', async () => {
        // GIVEN
        const configs = [
            {
                Key: 'testkey',
                Value: 'testValue',
            },
            {
                Key: 'testkey2',
                Value: 'testValue2',
            },
        ];
        const statusMessage = 'error message';
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockRejectedValue(statusMessage);
        const result = await spotEventPluginClient.configureSpotEventPlugin(configs);
        // THEN
        expect(result).toBeFalsy();
        expect(consoleErrorMock.mock.calls.length).toBe(1);
        expect(consoleErrorMock.mock.calls[0][0]).toMatch(`Failed to save plugin configuration. Reason: ${statusMessage}`);
    });
    test('valid concurrency token', async () => {
        // GIVEN
        const concurrencyToken = 'TOKEN';
        const validResponse = {
            data: {
                ServerData: [{
                        ID: 'event.plugin.spot',
                        ConcurrencyToken: concurrencyToken,
                    }],
            },
        };
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['describeServerData'] = jest.fn().mockResolvedValue(validResponse);
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['concurrencyToken']();
        // THEN
        expect(result).toBe(concurrencyToken);
    });
    test('returns empty token if no event plugin id entry', async () => {
        // GIVEN
        const noSpotEventOluginResponse = {
            data: {
                ServerData: [{
                        ID: 'NOT.event.plugin.spot',
                        ConcurrencyToken: 'token',
                    }],
            },
        };
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['describeServerData'] = jest.fn().mockResolvedValue(noSpotEventOluginResponse);
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['concurrencyToken']();
        // THEN
        expect(result).toBe('');
    });
    test('throws if invalid server data', async () => {
        // GIVEN
        const invalidDescribeDataResponse = {
            data: {
                NotServerData: {},
            },
        };
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['describeServerData'] = jest.fn().mockResolvedValue(invalidDescribeDataResponse);
        // eslint-disable-next-line dot-notation
        const promise = spotEventPluginClient['concurrencyToken']();
        // THEN
        await expect(promise).rejects.toThrowError(`Failed to receive a ConcurrencyToken. Invalid response: ${invalidDescribeDataResponse.data}.`);
    });
    test('successful describeServerData', async () => {
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockResolvedValue(describeDataResponse);
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['describeServerData']();
        // THEN
        expect(result).toEqual(describeDataResponse);
    });
    test('failed describeServerData', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockRejectedValue(statusMessage);
        // eslint-disable-next-line dot-notation
        const promise = spotEventPluginClient['describeServerData']();
        // THEN
        await expect(promise).rejects.toEqual(statusMessage);
    });
    test.each([
        [spot_event_plugin_client_1.CollectionType.Group, successfulGroupResponse, groupsColection],
        [spot_event_plugin_client_1.CollectionType.Pool, successfulPoolResponse, poolsColection],
    ])('Successful getCollection for %s', async (type, response, expectedResult) => {
        // GIVEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn().mockResolvedValue(response);
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['getCollection'](type);
        // THEN
        expect(result).toEqual(expectedResult);
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].GetRequest).toBeCalledTimes(1);
        expect(consoleLogMock).toBeCalledTimes(1);
        expect(consoleLogMock).toBeCalledWith(expect.stringMatching(`Getting ${type} collection:`));
    });
    test('failed getCollection', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn().mockRejectedValue(new Error(statusMessage));
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['getCollection'](spot_event_plugin_client_1.CollectionType.Group);
        // THEN
        expect(result).toBeUndefined();
        expect(consoleErrorMock).toBeCalledTimes(1);
        expect(consoleErrorMock).toBeCalledWith(expect.stringMatching(`Failed to get group collection. Reason: ${statusMessage}`));
    });
    test('failed getCollection with invalid response', async () => {
        // GIVEN
        const invalidGroupResponse = {
            data: {
                Pools: {},
            },
        };
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn().mockResolvedValue(invalidGroupResponse);
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['getCollection'](spot_event_plugin_client_1.CollectionType.Group);
        // THEN
        expect(result).toBeUndefined();
        expect(consoleErrorMock).toBeCalledTimes(1);
        expect(consoleErrorMock).toBeCalledWith(expect.stringMatching(`Failed to receive a group collection. Invalid response: ${JSON.stringify(invalidGroupResponse.data)}.`));
    });
    test.each([
        [spot_event_plugin_client_1.CollectionType.Group, groupsColection],
        [spot_event_plugin_client_1.CollectionType.Pool, poolsColection],
    ])('successful saveCollection for %s', async (type, expectedResult) => {
        // GIVEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockResolvedValue({});
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['saveCollection'](expectedResult, type);
        // THEN
        expect(result).toBeTruthy();
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledTimes(1);
        expect(consoleLogMock).toBeCalledTimes(2);
        expect(consoleLogMock).toBeCalledWith(expect.stringMatching(`Saving ${type} collection:`));
        expect(consoleLogMock).toBeCalledWith(expectedResult);
    });
    test('failed saveCollection', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockRejectedValue(new Error(statusMessage));
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['saveCollection'](groupsColection, spot_event_plugin_client_1.CollectionType.Group);
        // THEN
        expect(result).toBeFalsy();
        expect(consoleErrorMock).toBeCalledTimes(1);
        expect(consoleErrorMock).toBeCalledWith(expect.stringMatching(`Failed to save group collection. Reason: ${statusMessage}`));
    });
    test.each([
        [[], ['gr1', 'gr2']],
        [['gr1', 'gr2'], ['gr1', 'gr2']],
        [['gr1', 'gr2'], ['gr1', 'gr3']],
        [['gr1', 'gr2'], []],
    ])('successful call addGroup with existing groups %s and added groups %s', async (currentGroupsCollection, addedGroupsCollection) => {
        // GIVEN
        const obsoletePools = ['obsolete_pool'];
        const groupResponse = {
            data: {
                Pools: currentGroupsCollection,
                ObsoletePools: obsoletePools,
            },
            fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
        };
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn().mockResolvedValue(groupResponse);
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockReturnValue(true);
        // WHEN
        await spotEventPluginClient.addGroups(addedGroupsCollection);
        // THEN
        const requestsCount = addedGroupsCollection.length > 0 ? 1 : 0;
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].GetRequest).toBeCalledTimes(requestsCount);
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledTimes(requestsCount);
        if (requestsCount > 0) {
            // eslint-disable-next-line dot-notation, jest/no-conditional-expect
            expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledWith('/db/settings/collections/groups/save', {
                Pools: Array.from(new Set(currentGroupsCollection.concat(addedGroupsCollection))),
                ObsoletePools: obsoletePools,
            }, {
                headers: { 'Content-Type': 'application/json; charset=utf-8' },
            });
        }
    });
    test.each([
        [[], ['pool1', 'pool2']],
        [['pool1', 'pool2'], ['pool1', 'pool2']],
        [['pool1', 'pool2'], ['pool1', 'pool3']],
        [['pool1', 'pool2'], []],
    ])('successful call addPool with existing pools %s and added pools %s', async (currentPoolsCollection, addedPoolsCollection) => {
        // GIVEN
        const obsoletePools = ['obsolete_pool'];
        const poolResponse = {
            data: {
                Pools: currentPoolsCollection,
                ObsoletePools: obsoletePools,
            },
            fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
        };
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn().mockResolvedValue(poolResponse);
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockReturnValue(true);
        // WHEN
        await spotEventPluginClient.addPools(addedPoolsCollection);
        // THEN
        const requestsCount = addedPoolsCollection.length > 0 ? 1 : 0;
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].GetRequest).toBeCalledTimes(requestsCount);
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledTimes(requestsCount);
        if (requestsCount > 0) {
            // eslint-disable-next-line dot-notation, jest/no-conditional-expect
            expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledWith('/db/settings/collections/pools/save', {
                Pools: Array.from(new Set(currentPoolsCollection.concat(addedPoolsCollection))),
                ObsoletePools: obsoletePools,
            }, {
                headers: { 'Content-Type': 'application/json; charset=utf-8' },
            });
        }
    });
    test('does not add groups when groups were not retrieved successfully', async () => {
        // GIVEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn().mockResolvedValue(undefined);
        // WHEN
        const added = await spotEventPluginClient.addGroups(['group']);
        // THEN
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].GetRequest).toBeCalledTimes(1);
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).not.toBeCalled();
        expect(added).toBe(false);
    });
    test('does not add pools when pools were not retrieved successfully', async () => {
        // GIVEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn().mockResolvedValue(undefined);
        // WHEN
        const added = await spotEventPluginClient.addPools(['pool']);
        // THEN
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].GetRequest).toBeCalledTimes(1);
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).not.toBeCalled();
        expect(added).toBe(false);
    });
});
//# sourceMappingURL=data:application/json;base64,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