"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const wait_for_stable_service_1 = require("../lib/wait-for-stable-service");
describe('WaitForStableService', () => {
    let app;
    let stack;
    let isolatedStack;
    let cluster;
    let taskDefinition;
    let service;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        cluster = new aws_ecs_1.Cluster(stack, 'Cluster');
        cluster.addCapacity('ASG', {
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: 1,
            maxCapacity: 1,
        });
        taskDefinition = new aws_ecs_1.Ec2TaskDefinition(stack, 'RCSTask');
        taskDefinition.addContainer('Test', {
            image: aws_ecs_1.ContainerImage.fromAsset(__dirname),
            memoryLimitMiB: 7500,
        });
        service = new aws_ecs_1.Ec2Service(stack, 'Service', {
            cluster,
            taskDefinition,
        });
    });
    test('creates a custom resource', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_WaitForStableService', {
            cluster: isolatedStack.resolve(cluster.clusterArn),
            services: [isolatedStack.resolve(service.serviceArn)],
        }));
    });
    test('creates lambda correctly', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        assert_1.expect(isolatedStack).to(assert_1.countResourcesLike('AWS::Lambda::Function', 1, {
            Handler: 'wait-for-stable-service.wait',
            Environment: {
                Variables: {
                    DEBUG: 'false',
                },
            },
            Runtime: 'nodejs12.x',
            Timeout: 900,
        }));
    });
    test('adds policies to the lambda role', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
            ManagedPolicyArns: assert_1.arrayWith(isolatedStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole').managedPolicyArn)),
            Policies: [{
                    PolicyDocument: assert_1.objectLike({
                        Statement: [{
                                Action: 'ecs:DescribeServices',
                                Effect: 'Allow',
                                Resource: assert_1.arrayWith(isolatedStack.resolve(cluster.clusterArn), isolatedStack.resolve(service.serviceArn)),
                            }],
                    }),
                }],
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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