"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_1 = require("../lib");
let app;
let stack;
let spotFleetStack;
let vpc;
let renderQueue;
let rcsImage;
const groupName = 'Group_Name';
const deadlineGroups = [
    groupName,
];
const workerMachineImage = new aws_ec2_1.GenericLinuxImage({
    'us-east-1': 'ami-any',
});
const instanceTypes = [
    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
];
const maxCapacity = 1;
describe('SpotEventPluginFleet', () => {
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'infraStack', {
            env: {
                region: 'us-east-1',
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'VersionQuery');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        spotFleetStack = new core_1.Stack(app, 'SpotFleetStack', {
            env: {
                region: 'us-east-1',
            },
        });
    });
    describe('created with default values', () => {
        test('creates a security group', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.securityGroups).toBeDefined();
            expect(fleet.securityGroups.length).toBe(1);
            assert_1.expect(spotFleetStack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
        });
        test('allows connection to the render queue', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
                SourceSecurityGroupId: spotFleetStack.resolve(fleet.connections.securityGroups[0].securityGroupId),
            }));
        });
        test('creates a spot fleet instance role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('creates an instance profile', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.instanceProfile).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::InstanceProfile', {
                Roles: assert_1.arrayWith({
                    Ref: spotFleetStack.getLogicalId(fleet.fleetInstanceRole.node.defaultChild),
                }),
            }));
        });
        test('creates a spot fleet role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('adds group names to user data', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBeDefined();
            expect(renderedUserData).toMatch(groupName.toLocaleLowerCase());
        });
        test('adds RFDK tags', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            expect(fleet.tags).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: rfdkTag.name,
                    Value: rfdkTag.value,
                })),
            }));
        });
        test('uses default LogGroup prefix %s', () => {
            // GIVEN
            const id = 'SpotFleet';
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: '/renderfarm/' + id,
            }));
        });
        test('sets default allocation strategy', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(lib_1.SpotFleetAllocationStrategy.LOWEST_PRICE);
        });
        test('does not set valid until property', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.validUntil).toBeUndefined();
        });
        test('does not set valid block devices', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.blockDevices).toBeUndefined();
        });
        test('does not set ssh key', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.keyName).toBeUndefined();
        });
        test('creates launch template configs for each instance type', () => {
            // WHEN
            const moreInstanceTypes = [
                new aws_ec2_1.InstanceType('t2.small'),
                new aws_ec2_1.InstanceType('c5.large'),
            ];
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                instanceTypes: moreInstanceTypes,
            });
            // THEN
            expect(fleet._launchTemplateConfigs.length).toBeGreaterThanOrEqual(moreInstanceTypes.length);
            moreInstanceTypes.forEach(instanceType => {
                expect(fleet._launchTemplateConfigs.some(ltc => {
                    return ltc.Overrides.some(override => override.InstanceType === instanceType.toString());
                })).toBeTruthy();
            });
        });
        test('creates launch template configs for each subnet id', () => {
            // WHEN
            const subnets = vpc.selectSubnets({ subnetType: aws_ec2_1.SubnetType.PRIVATE });
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: subnets,
            });
            // THEN
            expect(fleet._launchTemplateConfigs.length).toBeGreaterThanOrEqual(subnets.subnets.length);
            subnets.subnetIds.forEach(subnetId => {
                expect(fleet._launchTemplateConfigs.some(ltc => {
                    return ltc.Overrides.some(override => override.SubnetId === subnetId);
                })).toBeTruthy();
            });
        });
        test('add tag indicating resource tracker is enabled', () => {
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    TagSpecifications: assert_1.arrayWith({
                        ResourceType: 'instance',
                        Tags: assert_1.arrayWith({
                            Key: 'DeadlineTrackedAWSResource',
                            Value: 'SpotEventPlugin',
                        }),
                    }),
                }),
            }));
        });
        test('adds multiple fleet security groups to launch template', () => {
            // GIVEN
            const securityGroups = [
                new aws_ec2_1.SecurityGroup(stack, 'NewFleetSecurityGroup1', { vpc }),
                new aws_ec2_1.SecurityGroup(stack, 'NewFleetSecurityGroup2', { vpc }),
            ];
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet2', {
                vpc,
                renderQueue,
                deadlineGroups: ['group2'],
                instanceTypes: [new aws_ec2_1.InstanceType('t2.micro')],
                workerMachineImage,
                maxCapacity: 1,
                securityGroups,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    SecurityGroupIds: securityGroups.map(sg => spotFleetStack.resolve(sg.securityGroupId)),
                }),
            }));
        });
        test('adds fleet tags to launch template', () => {
            // GIVEN
            const tag = {
                key: 'mykey',
                value: 'myvalue',
            };
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
            });
            // WHEN
            core_1.Tags.of(fleet).add(tag.key, tag.value);
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    TagSpecifications: assert_1.arrayWith({
                        ResourceType: lib_1.SpotFleetResourceType.INSTANCE.toString(),
                        Tags: assert_1.arrayWith({
                            Key: tag.key,
                            Value: tag.value,
                        }),
                    }),
                }),
            }));
        });
    });
    describe('created with custom values', () => {
        test('uses provided required properties', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const imageConfig = workerMachineImage.getImage(fleet);
            // THEN
            expect(fleet.deadlineGroups).toStrictEqual(deadlineGroups.map(group => group.toLocaleLowerCase()));
            expect(fleet.instanceTypes).toBe(instanceTypes);
            expect(fleet.osType).toBe(imageConfig.osType);
            expect(fleet.maxCapacity).toBe(maxCapacity);
        });
        test('uses provided security group', () => {
            // GIVEN
            const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(1);
            expect(fleet.securityGroups).toContainEqual(sg);
        });
        test('uses multiple provided security groups', () => {
            // GIVEN
            const sg1 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'sg-123456789', {
                allowAllOutbound: false,
            });
            const sg2 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG2', 'sg-987654321', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg1,
                    sg2,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(2);
            expect(fleet.securityGroups).toContainEqual(sg1);
            expect(fleet.securityGroups).toContainEqual(sg2);
        });
        test('adds to provided user data', () => {
            // GIVEN
            const originalCommands = 'original commands';
            const originalUserData = workerMachineImage.getImage(spotFleetStack).userData;
            originalUserData.addCommands(originalCommands);
            const renderedOriginalUser = originalUserData.render();
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userData: originalUserData,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBe(originalUserData);
            expect(renderedUserData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(originalCommands)));
            expect(renderedUserData).not.toEqual(renderedOriginalUser);
        });
        test('uses provided spot fleet instance role from the same stack', () => {
            // GIVEN
            const spotFleetInstanceRole = new aws_iam_1.Role(spotFleetStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                fleetInstanceRole: spotFleetInstanceRole,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBe(spotFleetInstanceRole);
            assert_1.expect(spotFleetStack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('throws if provided spot fleet instance role is not from the same stack', () => {
            // GIVEN
            const otherStack = new core_1.Stack(app, 'OtherStack', {
                env: { region: 'us-east-1' },
            });
            const spotFleetInstanceRole = new aws_iam_1.Role(otherStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            function createSpotEventPluginFleet() {
                new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    fleetInstanceRole: spotFleetInstanceRole,
                });
            }
            // THEN
            expect(createSpotEventPluginFleet).toThrowError('Fleet instance role should be created on the same stack as SpotEventPluginFleet to avoid circular dependencies.');
        });
        test('uses provided spot fleet role', () => {
            // GIVEN
            const fleetRole = new aws_iam_1.Role(stack, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                fleetRole: fleetRole,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBe(fleetRole);
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('tags resources deployed by CDK', () => {
            // GIVEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const tagName = 'name';
            const tagValue = 'tagValue';
            // WHEN
            core_1.Tags.of(fleet).add(tagName, tagValue);
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: tagName,
                    Value: tagValue,
                })),
            }));
        });
        test('uses provided subnets', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            const expectedSubnetId = stack.resolve(vpc.privateSubnets[0].subnetId);
            // THEN
            expect(stack.resolve(fleet.subnets.subnetIds)).toContainEqual(expectedSubnetId);
        });
        test('.defaultSubnets is false when subnets provided', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            // THEN
            expect(fleet.defaultSubnets).toBeFalsy();
        });
        test('uses provided allocation strategy', () => {
            // GIVEN
            const allocationStartegy = lib_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                allocationStrategy: allocationStartegy,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(allocationStartegy);
        });
        test('adds deadline region to user data', () => {
            // GIVEN
            const deadlineRegion = 'someregion';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlineRegion: deadlineRegion,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(deadlineRegion);
        });
        test('adds deadline pools to user data', () => {
            // GIVEN
            const pool1 = 'Pool1';
            const pool2 = 'pool2';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlinePools: [
                    pool1,
                    pool2,
                ],
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(pool1.toLocaleLowerCase());
            expect(renderedUserData).toMatch(pool2.toLocaleLowerCase());
        });
        test('uses provided ssh key name', () => {
            // GIVEN
            const keyName = 'test-key-name';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                keyName: keyName,
            });
            // THEN
            expect(fleet.keyName).toEqual(keyName);
        });
        test('UserData is added by UserDataProvider', () => {
            // WHEN
            class UserDataProvider extends lib_1.InstanceUserDataProvider {
                preCloudWatchAgent(host) {
                    host.userData.addCommands('echo preCloudWatchAgent');
                }
                preRenderQueueConfiguration(host) {
                    host.userData.addCommands('echo preRenderQueueConfiguration');
                }
                preWorkerConfiguration(host) {
                    host.userData.addCommands('echo preWorkerConfiguration');
                }
                postWorkerLaunch(host) {
                    host.userData.addCommands('echo postWorkerLaunch');
                }
            }
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userDataProvider: new UserDataProvider(spotFleetStack, 'UserDataProvider'),
            });
            const userData = fleet.userData.render();
            // THEN
            expect(userData).toContain('echo preCloudWatchAgent');
            expect(userData).toContain('echo preRenderQueueConfiguration');
            expect(userData).toContain('echo preWorkerConfiguration');
            expect(userData).toContain('echo postWorkerLaunch');
        });
        test.each([
            'test-prefix/',
            '',
        ])('uses custom LogGroup prefix %s', (testPrefix) => {
            // GIVEN
            const id = 'SpotFleet';
            const logGroupProps = {
                logGroupPrefix: testPrefix,
            };
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                logGroupProps: logGroupProps,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: testPrefix + id,
            }));
        });
        test('adds tag indicating resource tracker is not enabled', () => {
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                trackInstancesWithResourceTracker: false,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    TagSpecifications: assert_1.arrayWith({
                        ResourceType: 'instance',
                        Tags: assert_1.arrayWith({
                            Key: 'DeadlineResourceTracker',
                            Value: 'SpotEventPlugin',
                        }),
                    }),
                }),
            }));
        });
    });
    describe('allowing remote control', () => {
        test('from CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('to CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('from SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('from other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
    });
    describe('validation with', () => {
        describe('instance types', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyInstanceTypes = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: emptyInstanceTypes,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one instance type is required for a Spot Fleet Request Configuration/);
            });
            test('passes with at least one', () => {
                // GIVEN
                const oneInstanceType = [aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL)];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: oneInstanceType,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('subnets', () => {
            test('error if no subnets provided', () => {
                // GIVEN
                const invalidSubnets = {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    availabilityZones: ['dummy zone'],
                };
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    vpcSubnets: invalidSubnets,
                });
                // THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.ERROR);
                expect(fleet.node.metadataEntry[0].data).toMatch(/Did not find any subnets matching/);
            });
        });
        describe('groups', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyDeadlineGroups = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: emptyDeadlineGroups,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one Deadline Group is required for a Spot Fleet Request Configuration/);
            });
            test.each([
                'none',
                'with space',
                'group_*',
            ])('throws with %s', (invalidGroupName) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [invalidGroupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineGroups'/);
            });
            test('passes with valid group name', () => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [groupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('region', () => {
            test.each([
                'none',
                'all',
                'unrecognized',
                'none@123',
                'None',
            ])('throws with %s', (deadlineRegion) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineRegion'/);
            });
            test('passes with reserved name as substring', () => {
                // GIVEN
                const deadlineRegion = 'none123';
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('Block Device Tests', () => {
            test('Warning if no BlockDevices provided', () => {
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                });
                // THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
            });
            test('No Warnings if Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: '/dev/xvda',
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry).toHaveLength(0);
            });
            test('Warnings if non-Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('Warnings for BlockDevices without encryption specified', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('No warnings for Ephemeral blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry).toHaveLength(0);
            });
            test('No warnings for Suppressed blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry).toHaveLength(0);
            });
            test('throws if block devices without iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.IO1;
                // WHEN
                function createSpotEventPluginFleet() {
                    return new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups: [
                            groupName,
                        ],
                        instanceTypes: [
                            aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                        ],
                        workerMachineImage,
                        maxCapacity: 1,
                        blockDevices: [{
                                deviceName,
                                volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                    volumeType,
                                }),
                            }],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/iops property is required with volumeType: EbsDeviceVolumeType.IO1/);
            });
            test('warning if block devices with iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const iops = 100;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups: [
                        groupName,
                    ],
                    instanceTypes: [
                        aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                    ],
                    workerMachineImage,
                    maxCapacity: 1,
                    blockDevices: [{
                            deviceName,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                iops,
                                volumeType,
                            }),
                        }],
                });
                // THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3BvdC1ldmVudC1wbHVnaW4tZmxlZXQudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNwb3QtZXZlbnQtcGx1Z2luLWZsZWV0LnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7QUFFSCwrQkFBK0I7QUFFL0IsaUNBQWlDO0FBRWpDLDRDQVF5QjtBQUN6Qiw4REFHa0M7QUFDbEMsOENBVzBCO0FBQzFCLDhDQUcwQjtBQUMxQiw4Q0FJMEI7QUFDMUIsMEVBQTJFO0FBQzNFLHdDQUt1QjtBQUN2Qiw4REFBd0Q7QUFDeEQsNkVBRTZDO0FBRTdDLGdDQVVnQjtBQUVoQixJQUFJLEdBQVEsQ0FBQztBQUNiLElBQUksS0FBWSxDQUFDO0FBQ2pCLElBQUksY0FBcUIsQ0FBQztBQUMxQixJQUFJLEdBQVMsQ0FBQztBQUNkLElBQUksV0FBeUIsQ0FBQztBQUM5QixJQUFJLFFBQW9CLENBQUM7QUFFekIsTUFBTSxTQUFTLEdBQUcsWUFBWSxDQUFDO0FBQy9CLE1BQU0sY0FBYyxHQUFHO0lBQ3JCLFNBQVM7Q0FDVixDQUFDO0FBQ0YsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLDJCQUFpQixDQUFDO0lBQy9DLFdBQVcsRUFBRSxTQUFTO0NBQ3ZCLENBQUMsQ0FBQztBQUNILE1BQU0sYUFBYSxHQUFHO0lBQ3BCLHNCQUFZLENBQUMsRUFBRSxDQUFDLHVCQUFhLENBQUMsRUFBRSxFQUFFLHNCQUFZLENBQUMsS0FBSyxDQUFDO0NBQ3RELENBQUM7QUFDRixNQUFNLFdBQVcsR0FBRyxDQUFDLENBQUM7QUFFdEIsUUFBUSxDQUFDLHNCQUFzQixFQUFFLEdBQUcsRUFBRTtJQUNwQyxVQUFVLENBQUMsR0FBRyxFQUFFO1FBQ2QsR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDaEIsS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxZQUFZLEVBQUU7WUFDbkMsR0FBRyxFQUFFO2dCQUNILE1BQU0sRUFBRSxXQUFXO2FBQ3BCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsR0FBRyxHQUFHLElBQUksYUFBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUM1QixRQUFRLEdBQUcsd0JBQWMsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDL0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsQ0FBQztRQUN4RCxXQUFXLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7WUFDekMsR0FBRztZQUNILE1BQU0sRUFBRSxFQUFFLHNCQUFzQixFQUFFLFFBQVEsRUFBRTtZQUM1QyxVQUFVLEVBQUUsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7Z0JBQzlDLEdBQUc7Z0JBQ0gsT0FBTztnQkFDUCx5QkFBeUIsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUU7YUFDOUMsQ0FBQztZQUNGLGlCQUFpQixFQUFFLEVBQUUsV0FBVyxFQUFFLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxFQUFFO1lBQ3RELE9BQU87U0FDUixDQUFDLENBQUM7UUFDSCxjQUFjLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGdCQUFnQixFQUFFO1lBQ2hELEdBQUcsRUFBRTtnQkFDSCxNQUFNLEVBQUUsV0FBVzthQUNwQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtRQUMzQyxJQUFJLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO1lBQ3BDLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ2xFLEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixXQUFXO2FBQ1osQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7WUFDM0MsTUFBTSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzVDLGVBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyx5QkFBeUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzdFLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtZQUNqRCxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNsRSxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVzthQUNaLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO2dCQUM5RSxVQUFVLEVBQUUsS0FBSztnQkFDakIsTUFBTSxFQUFFLFFBQVEsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxFQUFFLEVBQUUsQ0FBQztnQkFDekQscUJBQXFCLEVBQUUsY0FBYyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUM7YUFDbkcsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQ0FBb0MsRUFBRSxHQUFHLEVBQUU7WUFDOUMsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBQzlDLGVBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQzlELHdCQUF3QixFQUFFLG1CQUFVLENBQUM7b0JBQ25DLFNBQVMsRUFBRSxDQUFDLG1CQUFVLENBQUM7NEJBQ3JCLE1BQU0sRUFBRSxnQkFBZ0I7NEJBQ3hCLE1BQU0sRUFBRSxPQUFPOzRCQUNmLFNBQVMsRUFBRTtnQ0FDVCxPQUFPLEVBQUUsbUJBQW1COzZCQUM3Qjt5QkFDRixDQUFDLENBQUM7aUJBQ0osQ0FBQztnQkFDRixpQkFBaUIsRUFBRSxrQkFBUyxDQUMxQixjQUFjLENBQUMsT0FBTyxDQUFDLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsZ0RBQWdELENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUNsSTthQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsNkJBQTZCLEVBQUUsR0FBRyxFQUFFO1lBQ3ZDLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ2xFLEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixXQUFXO2FBQ1osQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7WUFDNUMsZUFBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywyQkFBMkIsRUFBRTtnQkFDekUsS0FBSyxFQUFFLGtCQUFTLENBQUM7b0JBQ2YsR0FBRyxFQUFFLGNBQWMsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxZQUEwQixDQUFDO2lCQUMxRixDQUFDO2FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7WUFDckMsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUN0QyxlQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdCQUFnQixFQUFFO2dCQUM5RCx3QkFBd0IsRUFBRSxtQkFBVSxDQUFDO29CQUNuQyxTQUFTLEVBQUUsQ0FBQyxtQkFBVSxDQUFDOzRCQUNyQixNQUFNLEVBQUUsZ0JBQWdCOzRCQUN4QixNQUFNLEVBQUUsT0FBTzs0QkFDZixTQUFTLEVBQUU7Z0NBQ1QsT0FBTyxFQUFFLHlCQUF5Qjs2QkFDbkM7eUJBQ0YsQ0FBQyxDQUFDO2lCQUNKLENBQUM7Z0JBQ0YsaUJBQWlCLEVBQUUsa0JBQVMsQ0FDMUIsY0FBYyxDQUFDLE9BQU8sQ0FBQyx1QkFBYSxDQUFDLHdCQUF3QixDQUFDLDRDQUE0QyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FDOUg7YUFDRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtZQUN6QyxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNsRSxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVzthQUNaLENBQUMsQ0FBQztZQUNILE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUVqRCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUNyQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FBQztRQUNsRSxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLEVBQUU7WUFDMUIsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFDSCxNQUFNLE9BQU8sR0FBRyx3QkFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRWpDLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBQ2pDLGVBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMseUJBQXlCLEVBQUU7Z0JBQ3ZFLElBQUksRUFBRSxrQkFBUyxDQUNiLG1CQUFVLENBQUM7b0JBQ1QsR0FBRyxFQUFFLE9BQU8sQ0FBQyxJQUFJO29CQUNqQixLQUFLLEVBQUUsT0FBTyxDQUFDLEtBQUs7aUJBQ3JCLENBQUMsQ0FDSDthQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO1lBQzNDLFFBQVE7WUFDUixNQUFNLEVBQUUsR0FBRyxXQUFXLENBQUM7WUFFdkIsT0FBTztZQUNQLElBQUksMEJBQW9CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtnQkFDbEMsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7Z0JBQ3ZELGVBQWUsRUFBRSxDQUFDO2dCQUNsQixZQUFZLEVBQUUsY0FBYyxHQUFHLEVBQUU7YUFDbEMsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxrQ0FBa0MsRUFBRSxHQUFHLEVBQUU7WUFDNUMsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxpQ0FBMkIsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUNyRixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxtQ0FBbUMsRUFBRSxHQUFHLEVBQUU7WUFDN0MsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUMzQyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxrQ0FBa0MsRUFBRSxHQUFHLEVBQUU7WUFDNUMsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUM3QyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxzQkFBc0IsRUFBRSxHQUFHLEVBQUU7WUFDaEMsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUN4QyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx3REFBd0QsRUFBRSxHQUFHLEVBQUU7WUFDbEUsT0FBTztZQUNQLE1BQU0saUJBQWlCLEdBQW1CO2dCQUN4QyxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO2dCQUM1QixJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO2FBQzdCLENBQUM7WUFDRixNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ2xFLEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGtCQUFrQjtnQkFDbEIsV0FBVztnQkFDWCxhQUFhLEVBQUUsaUJBQWlCO2FBQ2pDLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLE1BQU0sQ0FBQyxDQUFDLHNCQUFzQixDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzdGLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsRUFBRTtnQkFDdkMsTUFBTSxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUU7b0JBQzdDLE9BQVEsR0FBNEIsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLFlBQVksS0FBSyxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztnQkFDckgsQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNuQixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG9EQUFvRCxFQUFFLEdBQUcsRUFBRTtZQUM5RCxPQUFPO1lBQ1AsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDdEUsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNsRSxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsYUFBYTtnQkFDYixjQUFjO2dCQUNkLGtCQUFrQjtnQkFDbEIsV0FBVztnQkFDWCxVQUFVLEVBQUUsT0FBTzthQUNwQixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzNGLE9BQU8sQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dCQUNuQyxNQUFNLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRTtvQkFDN0MsT0FBUSxHQUE0QixDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsUUFBUSxLQUFLLFFBQVEsQ0FBQyxDQUFDO2dCQUNsRyxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ25CLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO1lBQzFELE9BQU87WUFDUCxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ3BELEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxhQUFhO2dCQUNiLGNBQWM7Z0JBQ2Qsa0JBQWtCO2dCQUNsQixXQUFXO2FBQ1osQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7Z0JBQ3hFLGtCQUFrQixFQUFFLG1CQUFVLENBQUM7b0JBQzdCLGlCQUFpQixFQUFFLGtCQUFTLENBQUM7d0JBQzNCLFlBQVksRUFBRSxVQUFVO3dCQUN4QixJQUFJLEVBQUUsa0JBQVMsQ0FBQzs0QkFDZCxHQUFHLEVBQUUsNEJBQTRCOzRCQUNqQyxLQUFLLEVBQUUsaUJBQWlCO3lCQUN6QixDQUFDO3FCQUNILENBQUM7aUJBQ0gsQ0FBQzthQUNILENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO1lBQ2xFLFFBQVE7WUFDUixNQUFNLGNBQWMsR0FBRztnQkFDckIsSUFBSSx1QkFBYSxDQUFDLEtBQUssRUFBRSx3QkFBd0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDO2dCQUMzRCxJQUFJLHVCQUFhLENBQUMsS0FBSyxFQUFFLHdCQUF3QixFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUM7YUFDNUQsQ0FBQztZQUVGLE9BQU87WUFDUCxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxZQUFZLEVBQUU7Z0JBQ3JELEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjLEVBQUUsQ0FBQyxRQUFRLENBQUM7Z0JBQzFCLGFBQWEsRUFBRSxDQUFDLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDN0Msa0JBQWtCO2dCQUNsQixXQUFXLEVBQUUsQ0FBQztnQkFDZCxjQUFjO2FBQ2YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7Z0JBQ3hFLGtCQUFrQixFQUFFLG1CQUFVLENBQUM7b0JBQzdCLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztpQkFDdkYsQ0FBQzthQUNILENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO1lBQzlDLFFBQVE7WUFDUixNQUFNLEdBQUcsR0FBRztnQkFDVixHQUFHLEVBQUUsT0FBTztnQkFDWixLQUFLLEVBQUUsU0FBUzthQUNqQixDQUFDO1lBQ0YsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNsRSxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsYUFBYTtnQkFDYixjQUFjO2dCQUNkLGtCQUFrQjtnQkFDbEIsV0FBVzthQUNaLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxXQUFJLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUV2QyxPQUFPO1lBQ1AsZUFBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtnQkFDeEUsa0JBQWtCLEVBQUUsbUJBQVUsQ0FBQztvQkFDN0IsaUJBQWlCLEVBQUUsa0JBQVMsQ0FBQzt3QkFDM0IsWUFBWSxFQUFFLDJCQUFxQixDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUU7d0JBQ3ZELElBQUksRUFBRSxrQkFBUyxDQUFDOzRCQUNkLEdBQUcsRUFBRSxHQUFHLENBQUMsR0FBRzs0QkFDWixLQUFLLEVBQUUsR0FBRyxDQUFDLEtBQUs7eUJBQ2pCLENBQUM7cUJBQ0gsQ0FBQztpQkFDSCxDQUFDO2FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtRQUMxQyxJQUFJLENBQUMsbUNBQW1DLEVBQUUsR0FBRyxFQUFFO1lBQzdDLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ2xFLEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixXQUFXO2FBQ1osQ0FBQyxDQUFDO1lBQ0gsTUFBTSxXQUFXLEdBQUcsa0JBQWtCLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRXZELE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ25HLE1BQU0sQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ2hELE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUM5QyxNQUFNLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUM5QyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7WUFDeEMsUUFBUTtZQUNSLE1BQU0sRUFBRSxHQUFHLHVCQUFhLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUU7Z0JBQ3hFLGdCQUFnQixFQUFFLEtBQUs7YUFDeEIsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsY0FBYyxFQUFFO29CQUNkLEVBQUU7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxxQkFBWSxDQUFDLHlCQUF5QixDQUFDLENBQUMsQ0FBQztZQUN6RSxNQUFNLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDNUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDbEQsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsd0NBQXdDLEVBQUUsR0FBRyxFQUFFO1lBQ2xELFFBQVE7WUFDUixNQUFNLEdBQUcsR0FBRyx1QkFBYSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUMxRSxnQkFBZ0IsRUFBRSxLQUFLO2FBQ3hCLENBQUMsQ0FBQztZQUNILE1BQU0sR0FBRyxHQUFHLHVCQUFhLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQzFFLGdCQUFnQixFQUFFLEtBQUs7YUFDeEIsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsY0FBYyxFQUFFO29CQUNkLEdBQUc7b0JBQ0gsR0FBRztpQkFDSjthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsS0FBSyxDQUFDLHFCQUFZLENBQUMseUJBQXlCLENBQUMsQ0FBQyxDQUFDO1lBQ3pFLE1BQU0sQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1QyxNQUFNLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUNqRCxNQUFNLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNuRCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyw0QkFBNEIsRUFBRSxHQUFHLEVBQUU7WUFDdEMsUUFBUTtZQUNSLE1BQU0sZ0JBQWdCLEdBQUcsbUJBQW1CLENBQUM7WUFDN0MsTUFBTSxnQkFBZ0IsR0FBRyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLENBQUMsUUFBUSxDQUFDO1lBQzlFLGdCQUFnQixDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1lBQy9DLE1BQU0sb0JBQW9CLEdBQUcsZ0JBQWdCLENBQUMsTUFBTSxFQUFFLENBQUM7WUFFdkQsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsUUFBUSxFQUFFLGdCQUFnQjthQUMzQixDQUFDLENBQUM7WUFDSCxNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7WUFFakQsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDOUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLHNDQUFnQixDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pGLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUM3RCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyw0REFBNEQsRUFBRSxHQUFHLEVBQUU7WUFDdEUsUUFBUTtZQUNSLE1BQU0scUJBQXFCLEdBQUcsSUFBSSxjQUFJLENBQUMsY0FBYyxFQUFFLHVCQUF1QixFQUFFO2dCQUM5RSxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztnQkFDcEQsZUFBZSxFQUFFO29CQUNmLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsZ0RBQWdELENBQUM7aUJBQ3pGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsaUJBQWlCLEVBQUUscUJBQXFCO2FBQ3pDLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLENBQUM7WUFDNUQsZUFBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLEVBQUUsQ0FBQywyQkFBa0IsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUU7Z0JBQ25FLHdCQUF3QixFQUFFLG1CQUFVLENBQUM7b0JBQ25DLFNBQVMsRUFBRSxDQUFDLG1CQUFVLENBQUM7NEJBQ3JCLE1BQU0sRUFBRSxnQkFBZ0I7NEJBQ3hCLE1BQU0sRUFBRSxPQUFPOzRCQUNmLFNBQVMsRUFBRTtnQ0FDVCxPQUFPLEVBQUUsbUJBQW1COzZCQUM3Qjt5QkFDRixDQUFDLENBQUM7aUJBQ0osQ0FBQztnQkFDRixpQkFBaUIsRUFBRSxrQkFBUyxDQUMxQixjQUFjLENBQUMsT0FBTyxDQUFDLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsZ0RBQWdELENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUNsSTthQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsd0VBQXdFLEVBQUUsR0FBRyxFQUFFO1lBQ2xGLFFBQVE7WUFDUixNQUFNLFVBQVUsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsWUFBWSxFQUFFO2dCQUM5QyxHQUFHLEVBQUUsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO2FBQzdCLENBQUMsQ0FBQztZQUNILE1BQU0scUJBQXFCLEdBQUcsSUFBSSxjQUFJLENBQUMsVUFBVSxFQUFFLHVCQUF1QixFQUFFO2dCQUMxRSxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztnQkFDcEQsZUFBZSxFQUFFO29CQUNmLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsZ0RBQWdELENBQUM7aUJBQ3pGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFNBQVMsMEJBQTBCO2dCQUNqQyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7b0JBQ3BELEdBQUc7b0JBQ0gsV0FBVztvQkFDWCxjQUFjO29CQUNkLGFBQWE7b0JBQ2Isa0JBQWtCO29CQUNsQixXQUFXO29CQUNYLGlCQUFpQixFQUFFLHFCQUFxQjtpQkFDekMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQztZQUVELE9BQU87WUFDUCxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxZQUFZLENBQUMsaUhBQWlILENBQUMsQ0FBQztRQUNySyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7WUFDekMsUUFBUTtZQUNSLE1BQU0sU0FBUyxHQUFHLElBQUksY0FBSSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7Z0JBQzdDLFNBQVMsRUFBRSxJQUFJLDBCQUFnQixDQUFDLHlCQUF5QixDQUFDO2dCQUMxRCxlQUFlLEVBQUU7b0JBQ2YsdUJBQWEsQ0FBQyx3QkFBd0IsQ0FBQyw0Q0FBNEMsQ0FBQztpQkFDckY7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNsRSxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsU0FBUyxFQUFFLFNBQVM7Z0JBQ3BCLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDeEMsZUFBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBZ0IsQ0FBQyxnQkFBZ0IsRUFBRTtnQkFDakUsd0JBQXdCLEVBQUUsbUJBQVUsQ0FBQztvQkFDbkMsU0FBUyxFQUFFLENBQUMsbUJBQVUsQ0FBQzs0QkFDckIsTUFBTSxFQUFFLGdCQUFnQjs0QkFDeEIsTUFBTSxFQUFFLE9BQU87NEJBQ2YsU0FBUyxFQUFFO2dDQUNULE9BQU8sRUFBRSx5QkFBeUI7NkJBQ25DO3lCQUNGLENBQUMsQ0FBQztpQkFDSixDQUFDO2dCQUNGLGlCQUFpQixFQUFFLGtCQUFTLENBQzFCLEtBQUssQ0FBQyxPQUFPLENBQUMsdUJBQWEsQ0FBQyx3QkFBd0IsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLENBQ3JIO2FBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLEVBQUU7WUFDMUMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFDSCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUM7WUFDdkIsTUFBTSxRQUFRLEdBQUcsVUFBVSxDQUFDO1lBRTVCLE9BQU87WUFDUCxXQUFJLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFFdEMsT0FBTztZQUNQLGVBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMseUJBQXlCLEVBQUU7Z0JBQ3ZFLElBQUksRUFBRSxrQkFBUyxDQUNiLG1CQUFVLENBQUM7b0JBQ1QsR0FBRyxFQUFFLE9BQU87b0JBQ1osS0FBSyxFQUFFLFFBQVE7aUJBQ2hCLENBQUMsQ0FDSDthQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO1lBQ2pDLFFBQVE7WUFDUixNQUFNLGNBQWMsR0FBb0I7Z0JBQ3RDLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE9BQU87YUFDL0IsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ2xFLEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixXQUFXO2dCQUNYLFVBQVUsRUFBRSxjQUFjO2FBQzNCLENBQUMsQ0FBQztZQUNILE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBRXZFLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDbEYsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO1lBQzFELFFBQVE7WUFDUixNQUFNLGNBQWMsR0FBb0I7Z0JBQ3RDLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE9BQU87YUFDL0IsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ2xFLEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixXQUFXO2dCQUNYLFVBQVUsRUFBRSxjQUFjO2FBQzNCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQzNDLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtZQUM3QyxRQUFRO1lBQ1IsTUFBTSxrQkFBa0IsR0FBRyxpQ0FBMkIsQ0FBQyxrQkFBa0IsQ0FBQztZQUUxRSxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNsRSxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVztnQkFDWCxrQkFBa0IsRUFBRSxrQkFBa0I7YUFDdkMsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUMvRCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxtQ0FBbUMsRUFBRSxHQUFHLEVBQUU7WUFDN0MsUUFBUTtZQUNSLE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQztZQUVwQyxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNsRSxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVztnQkFDWCxjQUFjLEVBQUUsY0FBYzthQUMvQixDQUFDLENBQUM7WUFDSCxNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7WUFFakQsT0FBTztZQUNQLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUNuRCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxrQ0FBa0MsRUFBRSxHQUFHLEVBQUU7WUFDNUMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQztZQUN0QixNQUFNLEtBQUssR0FBRyxPQUFPLENBQUM7WUFFdEIsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsYUFBYSxFQUFFO29CQUNiLEtBQUs7b0JBQ0wsS0FBSztpQkFDTjthQUNGLENBQUMsQ0FBQztZQUNILE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUVqRCxPQUFPO1lBQ1AsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUM7WUFDNUQsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDOUQsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsNEJBQTRCLEVBQUUsR0FBRyxFQUFFO1lBQ3RDLFFBQVE7WUFDUixNQUFNLE9BQU8sR0FBRyxlQUFlLENBQUM7WUFFaEMsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsT0FBTyxFQUFFLE9BQU87YUFDakIsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtZQUNqRCxPQUFPO1lBQ1AsTUFBTSxnQkFBaUIsU0FBUSw4QkFBd0I7Z0JBQ3JELGtCQUFrQixDQUFDLElBQVc7b0JBQzVCLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLHlCQUF5QixDQUFDLENBQUM7Z0JBQ3ZELENBQUM7Z0JBQ0QsMkJBQTJCLENBQUMsSUFBVztvQkFDckMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsa0NBQWtDLENBQUMsQ0FBQztnQkFDaEUsQ0FBQztnQkFDRCxzQkFBc0IsQ0FBQyxJQUFXO29CQUNoQyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO2dCQUMzRCxDQUFDO2dCQUNELGdCQUFnQixDQUFDLElBQVc7b0JBQzFCLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLHVCQUF1QixDQUFDLENBQUM7Z0JBQ3JELENBQUM7YUFDRjtZQUVELE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtnQkFDbEUsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7Z0JBQ1gsZ0JBQWdCLEVBQUUsSUFBSSxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUUsa0JBQWtCLENBQUM7YUFDM0UsQ0FBQyxDQUFDO1lBRUgsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUV6QyxPQUFPO1lBQ1AsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1lBQ3RELE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxTQUFTLENBQUMsa0NBQWtDLENBQUMsQ0FBQztZQUMvRCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsU0FBUyxDQUFDLDZCQUE2QixDQUFDLENBQUM7WUFDMUQsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ3RELENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztZQUNSLGNBQWM7WUFDZCxFQUFFO1NBQ0gsQ0FBQyxDQUFDLGdDQUFnQyxFQUFFLENBQUMsVUFBa0IsRUFBRSxFQUFFO1lBQzFELFFBQVE7WUFDUixNQUFNLEVBQUUsR0FBRyxXQUFXLENBQUM7WUFDdkIsTUFBTSxhQUFhLEdBQUc7Z0JBQ3BCLGNBQWMsRUFBRSxVQUFVO2FBQzNCLENBQUM7WUFFRixPQUFPO1lBQ1AsSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO2dCQUNsQyxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVztnQkFDWCxhQUFhLEVBQUUsYUFBYTthQUM3QixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO2dCQUN2RCxlQUFlLEVBQUUsQ0FBQztnQkFDbEIsWUFBWSxFQUFFLFVBQVUsR0FBRyxFQUFFO2FBQzlCLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMscURBQXFELEVBQUUsR0FBRyxFQUFFO1lBQy9ELE9BQU87WUFDUCxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ3BELEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxhQUFhO2dCQUNiLGNBQWM7Z0JBQ2Qsa0JBQWtCO2dCQUNsQixXQUFXO2dCQUNYLGlDQUFpQyxFQUFFLEtBQUs7YUFDekMsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7Z0JBQ3hFLGtCQUFrQixFQUFFLG1CQUFVLENBQUM7b0JBQzdCLGlCQUFpQixFQUFFLGtCQUFTLENBQUM7d0JBQzNCLFlBQVksRUFBRSxVQUFVO3dCQUN4QixJQUFJLEVBQUUsa0JBQVMsQ0FBQzs0QkFDZCxHQUFHLEVBQUUseUJBQXlCOzRCQUM5QixLQUFLLEVBQUUsaUJBQWlCO3lCQUN6QixDQUFDO3FCQUNILENBQUM7aUJBQ0gsQ0FBQzthQUNILENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7UUFDdkMsSUFBSSxDQUFDLFdBQVcsRUFBRSxHQUFHLEVBQUU7WUFDckIsUUFBUTtZQUNSLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQztZQUN2QixNQUFNLGlCQUFpQixHQUFHLENBQUMsQ0FBQztZQUU1QixPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO2dCQUN6RCxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVzthQUNaLENBQUMsQ0FBQztZQUVILEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxjQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7WUFFeEQsT0FBTztZQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMseUJBQXlCLEVBQUU7Z0JBQzlELG1CQUFtQixFQUFFLENBQUMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFLENBQUM7Z0JBQzlDLG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxNQUFNLEVBQUUsY0FBYzt3QkFDdEIsV0FBVyxFQUFFLHNDQUFzQzt3QkFDbkQsUUFBUSxFQUFFLFFBQVE7d0JBQ2xCLFVBQVUsRUFBRSxLQUFLO3dCQUNqQixNQUFNLEVBQUUsUUFBUSxHQUFHLGlCQUFpQjtxQkFDckM7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUU7WUFDbkIsUUFBUTtZQUNSLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQztZQUN2QixNQUFNLGlCQUFpQixHQUFHLENBQUMsQ0FBQztZQUU1QixPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO2dCQUN6RCxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVzthQUNaLENBQUMsQ0FBQztZQUVILEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxjQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7WUFFdEQsT0FBTztZQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMseUJBQXlCLEVBQUU7Z0JBQzlELG1CQUFtQixFQUFFLENBQUMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFLENBQUM7Z0JBQzlDLG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxNQUFNLEVBQUUsY0FBYzt3QkFDdEIsV0FBVyxFQUFFLHNDQUFzQzt3QkFDbkQsUUFBUSxFQUFFLFFBQVE7d0JBQ2xCLFVBQVUsRUFBRSxLQUFLO3dCQUNqQixNQUFNLEVBQUUsUUFBUSxHQUFHLGlCQUFpQjtxQkFDckM7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG9CQUFvQixFQUFFLEdBQUcsRUFBRTtZQUM5QixRQUFRO1lBQ1IsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDO1lBQ3ZCLE1BQU0saUJBQWlCLEdBQUcsQ0FBQyxDQUFDO1lBRTVCLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7Z0JBQ3pELEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixXQUFXO2FBQ1osQ0FBQyxDQUFDO1lBQ0gsTUFBTSxhQUFhLEdBQUcsdUJBQWEsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1lBRXJGLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUU1QyxPQUFPO1lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDckUsUUFBUSxFQUFFLFFBQVE7Z0JBQ2xCLFVBQVUsRUFBRSxLQUFLO2dCQUNqQixxQkFBcUIsRUFBRSxjQUFjO2dCQUNyQyxNQUFNLEVBQUUsUUFBUSxHQUFHLGlCQUFpQjthQUNyQyxDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtZQUM1QixRQUFRO1lBQ1IsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDO1lBQ3ZCLE1BQU0saUJBQWlCLEdBQUcsQ0FBQyxDQUFDO1lBRTVCLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7Z0JBQ3pELEdBQUc7Z0JBQ0gsV0FBVztnQkFDWCxjQUFjO2dCQUNkLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixXQUFXO2FBQ1osQ0FBQyxDQUFDO1lBQ0gsTUFBTSxhQUFhLEdBQUcsdUJBQWEsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1lBRXJGLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUUxQyxPQUFPO1lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDckUsUUFBUSxFQUFFLFFBQVE7Z0JBQ2xCLFVBQVUsRUFBRSxLQUFLO2dCQUNqQixxQkFBcUIsRUFBRSxjQUFjO2dCQUNyQyxNQUFNLEVBQUUsUUFBUSxHQUFHLGlCQUFpQjthQUNyQyxDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtZQUM1QixRQUFRO1lBQ1IsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDO1lBQ3ZCLE1BQU0saUJBQWlCLEdBQUcsQ0FBQyxDQUFDO1lBQzVCLE1BQU0sVUFBVSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxZQUFZLEVBQUU7Z0JBQzlDLEdBQUcsRUFBRSxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7YUFDN0IsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtnQkFDekQsR0FBRztnQkFDSCxXQUFXO2dCQUNYLGNBQWM7Z0JBQ2QsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFdBQVc7YUFDWixDQUFDLENBQUM7WUFDSCxNQUFNLGFBQWEsR0FBRyx1QkFBYSxDQUFDLG1CQUFtQixDQUFDLFVBQVUsRUFBRSxJQUFJLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFFMUYsS0FBSyxDQUFDLHNCQUFzQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBRTVDLE9BQU87WUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO2dCQUNyRSxRQUFRLEVBQUUsUUFBUTtnQkFDbEIsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLHFCQUFxQixFQUFFLGNBQWM7Z0JBQ3JDLE1BQU0sRUFBRSxRQUFRLEdBQUcsaUJBQWlCO2FBQ3JDLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFO1lBQzFCLFFBQVE7WUFDUixNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUM7WUFDdkIsTUFBTSxpQkFBaUIsR0FBRyxDQUFDLENBQUM7WUFDNUIsTUFBTSxVQUFVLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFlBQVksRUFBRTtnQkFDOUMsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTthQUM3QixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO2dCQUN6RCxHQUFHO2dCQUNILFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsV0FBVzthQUNaLENBQUMsQ0FBQztZQUNILE1BQU0sYUFBYSxHQUFHLHVCQUFhLENBQUMsbUJBQW1CLENBQUMsVUFBVSxFQUFFLElBQUksRUFBRSxjQUFjLENBQUMsQ0FBQztZQUUxRixLQUFLLENBQUMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLENBQUM7WUFFMUMsT0FBTztZQUNQLGVBQVMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7Z0JBQzFFLFFBQVEsRUFBRSxRQUFRO2dCQUNsQixVQUFVLEVBQUUsS0FBSztnQkFDakIscUJBQXFCLEVBQUUsY0FBYztnQkFDckMsTUFBTSxFQUFFLFFBQVEsR0FBRyxpQkFBaUI7YUFDckMsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGlCQUFpQixFQUFFLEdBQUcsRUFBRTtRQUMvQixRQUFRLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFO1lBQzlCLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7Z0JBQzdCLFFBQVE7Z0JBQ1IsTUFBTSxrQkFBa0IsR0FBbUIsRUFBRSxDQUFDO2dCQUU5QyxPQUFPO2dCQUNQLFNBQVMsMEJBQTBCO29CQUNqQyxJQUFJLDBCQUFvQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7d0JBQzNDLEdBQUc7d0JBQ0gsV0FBVzt3QkFDWCxjQUFjO3dCQUNkLGFBQWEsRUFBRSxrQkFBa0I7d0JBQ2pDLGtCQUFrQjt3QkFDbEIsV0FBVztxQkFDWixDQUFDLENBQUM7Z0JBQ0wsQ0FBQztnQkFFRCxPQUFPO2dCQUNQLE1BQU0sQ0FBQywwQkFBMEIsQ0FBQyxDQUFDLFlBQVksQ0FBQywrRUFBK0UsQ0FBQyxDQUFDO1lBQ25JLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLDBCQUEwQixFQUFFLEdBQUcsRUFBRTtnQkFDcEMsUUFBUTtnQkFDUixNQUFNLGVBQWUsR0FBRyxDQUFFLHNCQUFZLENBQUMsRUFBRSxDQUFDLHVCQUFhLENBQUMsRUFBRSxFQUFFLHNCQUFZLENBQUMsS0FBSyxDQUFDLENBQUUsQ0FBQztnQkFFbEYsT0FBTztnQkFDUCxTQUFTLDBCQUEwQjtvQkFDakMsSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO3dCQUMzQyxHQUFHO3dCQUNILFdBQVc7d0JBQ1gsY0FBYzt3QkFDZCxhQUFhLEVBQUUsZUFBZTt3QkFDOUIsa0JBQWtCO3dCQUNsQixXQUFXO3FCQUNaLENBQUMsQ0FBQztnQkFDTCxDQUFDO2dCQUVELE9BQU87Z0JBQ1AsTUFBTSxDQUFDLDBCQUEwQixDQUFDLENBQUMsR0FBRyxDQUFDLFlBQVksRUFBRSxDQUFDO1lBQ3hELENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRTtZQUN2QixJQUFJLENBQUMsOEJBQThCLEVBQUUsR0FBRyxFQUFFO2dCQUN4QyxRQUFRO2dCQUNSLE1BQU0sY0FBYyxHQUFHO29CQUNyQixVQUFVLEVBQUUsb0JBQVUsQ0FBQyxPQUFPO29CQUM5QixpQkFBaUIsRUFBRSxDQUFDLFlBQVksQ0FBQztpQkFDbEMsQ0FBQztnQkFFRixPQUFPO2dCQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLFdBQVcsRUFBRTtvQkFDbEUsR0FBRztvQkFDSCxXQUFXO29CQUNYLGNBQWM7b0JBQ2QsYUFBYTtvQkFDYixrQkFBa0I7b0JBQ2xCLFdBQVc7b0JBQ1gsVUFBVSxFQUFFLGNBQWM7aUJBQzNCLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsaURBQXlCLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ2xGLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsbUNBQW1DLENBQUMsQ0FBQztZQUN4RixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLFFBQVEsRUFBRSxHQUFHLEVBQUU7WUFDdEIsSUFBSSxDQUFDLG1CQUFtQixFQUFFLEdBQUcsRUFBRTtnQkFDN0IsUUFBUTtnQkFDUixNQUFNLG1CQUFtQixHQUFhLEVBQUUsQ0FBQztnQkFFekMsT0FBTztnQkFDUCxTQUFTLDBCQUEwQjtvQkFDakMsSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO3dCQUMzQyxHQUFHO3dCQUNILFdBQVc7d0JBQ1gsYUFBYTt3QkFDYixrQkFBa0I7d0JBQ2xCLFdBQVc7d0JBQ1gsY0FBYyxFQUFFLG1CQUFtQjtxQkFDcEMsQ0FBQyxDQUFDO2dCQUNMLENBQUM7Z0JBRUQsT0FBTztnQkFDUCxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxZQUFZLENBQUMsZ0ZBQWdGLENBQUMsQ0FBQztZQUNwSSxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLENBQUM7Z0JBQ1IsTUFBTTtnQkFDTixZQUFZO2dCQUNaLFNBQVM7YUFDVixDQUFDLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQyxnQkFBd0IsRUFBRSxFQUFFO2dCQUNoRCxPQUFPO2dCQUNQLFNBQVMsMEJBQTBCO29CQUNqQyxJQUFJLDBCQUFvQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7d0JBQzNDLEdBQUc7d0JBQ0gsV0FBVzt3QkFDWCxhQUFhO3dCQUNiLGtCQUFrQjt3QkFDbEIsV0FBVzt3QkFDWCxjQUFjLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQztxQkFDbkMsQ0FBQyxDQUFDO2dCQUNMLENBQUM7Z0JBRUQsT0FBTztnQkFDUCxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxZQUFZLENBQUMsaURBQWlELENBQUMsQ0FBQztZQUNyRyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7Z0JBQ3hDLE9BQU87Z0JBQ1AsU0FBUywwQkFBMEI7b0JBQ2pDLElBQUksMEJBQW9CLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTt3QkFDM0MsR0FBRzt3QkFDSCxXQUFXO3dCQUNYLGFBQWE7d0JBQ2Isa0JBQWtCO3dCQUNsQixXQUFXO3dCQUNYLGNBQWMsRUFBRSxDQUFDLFNBQVMsQ0FBQztxQkFDNUIsQ0FBQyxDQUFDO2dCQUNMLENBQUM7Z0JBRUQsT0FBTztnQkFDUCxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxHQUFHLENBQUMsWUFBWSxFQUFFLENBQUM7WUFDeEQsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyxRQUFRLEVBQUUsR0FBRyxFQUFFO1lBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUM7Z0JBQ1IsTUFBTTtnQkFDTixLQUFLO2dCQUNMLGNBQWM7Z0JBQ2QsVUFBVTtnQkFDVixNQUFNO2FBQ1AsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLENBQUMsY0FBc0IsRUFBRSxFQUFFO2dCQUM5QyxPQUFPO2dCQUNQLFNBQVMsMEJBQTBCO29CQUNqQyxJQUFJLDBCQUFvQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7d0JBQzNDLEdBQUc7d0JBQ0gsV0FBVzt3QkFDWCxjQUFjO3dCQUNkLGFBQWE7d0JBQ2Isa0JBQWtCO3dCQUNsQixXQUFXO3dCQUNYLGNBQWMsRUFBRSxjQUFjO3FCQUMvQixDQUFDLENBQUM7Z0JBQ0wsQ0FBQztnQkFFRCxPQUFPO2dCQUNQLE1BQU0sQ0FBQywwQkFBMEIsQ0FBQyxDQUFDLFlBQVksQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO1lBQ3JHLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtnQkFDbEQsUUFBUTtnQkFDUixNQUFNLGNBQWMsR0FBRyxTQUFTLENBQUM7Z0JBRWpDLE9BQU87Z0JBQ1AsU0FBUywwQkFBMEI7b0JBQ2pDLElBQUksMEJBQW9CLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTt3QkFDM0MsR0FBRzt3QkFDSCxXQUFXO3dCQUNYLGNBQWM7d0JBQ2QsYUFBYTt3QkFDYixrQkFBa0I7d0JBQ2xCLFdBQVc7d0JBQ1gsY0FBYyxFQUFFLGNBQWM7cUJBQy9CLENBQUMsQ0FBQztnQkFDTCxDQUFDO2dCQUVELE9BQU87Z0JBQ1AsTUFBTSxDQUFDLDBCQUEwQixDQUFDLENBQUMsR0FBRyxDQUFDLFlBQVksRUFBRSxDQUFDO1lBQ3hELENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsb0JBQW9CLEVBQUUsR0FBRyxFQUFFO1lBQ2xDLElBQUksQ0FBQyxxQ0FBcUMsRUFBRSxHQUFHLEVBQUU7Z0JBQy9DLE9BQU87Z0JBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO29CQUNsRSxHQUFHO29CQUNILFdBQVc7b0JBQ1gsY0FBYztvQkFDZCxhQUFhO29CQUNiLGtCQUFrQjtvQkFDbEIsV0FBVztpQkFDWixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLGlEQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNqRixNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLG9SQUFvUixDQUFDLENBQUM7WUFDelUsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO2dCQUMxRCxNQUFNLFdBQVcsR0FBRyxFQUFFLENBQUM7Z0JBRXZCLE9BQU87Z0JBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO29CQUNsRSxHQUFHO29CQUNILFdBQVc7b0JBQ1gsY0FBYztvQkFDZCxhQUFhO29CQUNiLGtCQUFrQjtvQkFDbEIsV0FBVztvQkFDWCxZQUFZLEVBQUUsQ0FBRTs0QkFDZCxVQUFVLEVBQUUsV0FBVzs0QkFDdkIsTUFBTSxFQUFFLG1DQUFpQixDQUFDLEdBQUcsQ0FBRSxXQUFXLEVBQUUsRUFBQyxTQUFTLEVBQUUsSUFBSSxFQUFDLENBQUM7eUJBQy9ELENBQUM7aUJBQ0gsQ0FBQyxDQUFDO2dCQUVILE1BQU07Z0JBQ04sTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ25ELENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGlEQUFpRCxFQUFFLEdBQUcsRUFBRTtnQkFDM0QsTUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO2dCQUN2QixNQUFNLFdBQVcsR0FBRyxXQUFXLENBQUM7Z0JBQ2hDLE1BQU0sRUFBRSxHQUFHLFdBQVcsQ0FBQztnQkFFdkIsT0FBTztnQkFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxFQUFFLEVBQUU7b0JBQ3pELEdBQUc7b0JBQ0gsV0FBVztvQkFDWCxjQUFjO29CQUNkLGFBQWE7b0JBQ2Isa0JBQWtCO29CQUNsQixXQUFXO29CQUNYLFlBQVksRUFBRSxDQUFFOzRCQUNkLFVBQVUsRUFBRSxXQUFXOzRCQUN2QixNQUFNLEVBQUUsbUNBQWlCLENBQUMsR0FBRyxDQUFFLFdBQVcsRUFBRSxFQUFDLFNBQVMsRUFBRSxLQUFLLEVBQUMsQ0FBQzt5QkFDaEUsQ0FBQztpQkFDSCxDQUFDLENBQUM7Z0JBRUgsTUFBTTtnQkFDTixNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLGlEQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNqRixNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLHFCQUFxQixXQUFXLHdCQUF3QixFQUFFLCtIQUErSCxDQUFDLENBQUM7WUFDOU8sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO2dCQUNsRSxNQUFNLFdBQVcsR0FBRyxFQUFFLENBQUM7Z0JBQ3ZCLE1BQU0sV0FBVyxHQUFHLFdBQVcsQ0FBQztnQkFDaEMsTUFBTSxFQUFFLEdBQUcsV0FBVyxDQUFDO2dCQUV2QixPQUFPO2dCQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsY0FBYyxFQUFFLEVBQUUsRUFBRTtvQkFDekQsR0FBRztvQkFDSCxXQUFXO29CQUNYLGNBQWM7b0JBQ2QsYUFBYTtvQkFDYixrQkFBa0I7b0JBQ2xCLFdBQVc7b0JBQ1gsWUFBWSxFQUFFLENBQUU7NEJBQ2QsVUFBVSxFQUFFLFdBQVc7NEJBQ3ZCLE1BQU0sRUFBRSxtQ0FBaUIsQ0FBQyxHQUFHLENBQUUsV0FBVyxDQUFFO3lCQUM3QyxDQUFDO2lCQUNILENBQUMsQ0FBQztnQkFFSCxNQUFNO2dCQUNOLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsaURBQXlCLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ2pGLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMscUJBQXFCLFdBQVcsd0JBQXdCLEVBQUUsK0hBQStILENBQUMsQ0FBQztZQUM5TyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyw4Q0FBOEMsRUFBRSxHQUFHLEVBQUU7Z0JBQ3hELE1BQU0sV0FBVyxHQUFHLFdBQVcsQ0FBQztnQkFFaEMsT0FBTztnQkFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLDBCQUFvQixDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUU7b0JBQ2xFLEdBQUc7b0JBQ0gsV0FBVztvQkFDWCxjQUFjO29CQUNkLGFBQWE7b0JBQ2Isa0JBQWtCO29CQUNsQixXQUFXO29CQUNYLFlBQVksRUFBRSxDQUFFOzRCQUNkLFVBQVUsRUFBRSxXQUFXOzRCQUN2QixNQUFNLEVBQUUsbUNBQWlCLENBQUMsU0FBUyxDQUFFLENBQUMsQ0FBRTt5QkFDekMsQ0FBQztpQkFDSCxDQUFDLENBQUM7Z0JBRUgsTUFBTTtnQkFDTixNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbkQsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsK0NBQStDLEVBQUUsR0FBRyxFQUFFO2dCQUN6RCxNQUFNLFdBQVcsR0FBRyxXQUFXLENBQUM7Z0JBRWhDLE9BQU87Z0JBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxjQUFjLEVBQUUsV0FBVyxFQUFFO29CQUNsRSxHQUFHO29CQUNILFdBQVc7b0JBQ1gsY0FBYztvQkFDZCxhQUFhO29CQUNiLGtCQUFrQjtvQkFDbEIsV0FBVztvQkFDWCxZQUFZLEVBQUUsQ0FBRTs0QkFDZCxVQUFVLEVBQUUsV0FBVzs0QkFDdkIsTUFBTSxFQUFFLG1DQUFpQixDQUFDLFFBQVEsRUFBRTt5QkFDckMsQ0FBQztpQkFDSCxDQUFDLENBQUM7Z0JBRUgsTUFBTTtnQkFDTixNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbkQsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO2dCQUN0RSxRQUFRO2dCQUNSLE1BQU0sVUFBVSxHQUFHLFdBQVcsQ0FBQztnQkFDL0IsTUFBTSxVQUFVLEdBQUcsRUFBRSxDQUFDO2dCQUN0QixNQUFNLFVBQVUsR0FBRyxxQ0FBbUIsQ0FBQyxHQUFHLENBQUM7Z0JBRTNDLE9BQU87Z0JBQ1AsU0FBUywwQkFBMEI7b0JBQ2pDLE9BQU8sSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUU7d0JBQzdELEdBQUc7d0JBQ0gsV0FBVzt3QkFDWCxjQUFjLEVBQUU7NEJBQ2QsU0FBUzt5QkFDVjt3QkFDRCxhQUFhLEVBQUU7NEJBQ2Isc0JBQVksQ0FBQyxFQUFFLENBQUMsdUJBQWEsQ0FBQyxFQUFFLEVBQUUsc0JBQVksQ0FBQyxLQUFLLENBQUM7eUJBQ3REO3dCQUNELGtCQUFrQjt3QkFDbEIsV0FBVyxFQUFFLENBQUM7d0JBQ2QsWUFBWSxFQUFFLENBQUM7Z0NBQ2IsVUFBVTtnQ0FDVixNQUFNLEVBQUUsbUNBQWlCLENBQUMsR0FBRyxDQUFDLFVBQVUsRUFBRTtvQ0FDeEMsVUFBVTtpQ0FDWCxDQUFDOzZCQUNILENBQUM7cUJBQ0gsQ0FBQyxDQUFDO2dCQUNMLENBQUM7Z0JBRUQsT0FBTztnQkFDUCxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxZQUFZLENBQUMsb0VBQW9FLENBQUMsQ0FBQztZQUN4SCxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7Z0JBQ3BFLFFBQVE7Z0JBQ1IsTUFBTSxVQUFVLEdBQUcsV0FBVyxDQUFDO2dCQUMvQixNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUM7Z0JBQ3RCLE1BQU0sSUFBSSxHQUFHLEdBQUcsQ0FBQztnQkFDakIsTUFBTSxVQUFVLEdBQUcscUNBQW1CLENBQUMsUUFBUSxDQUFDO2dCQUVoRCxPQUFPO2dCQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksMEJBQW9CLENBQUMsS0FBSyxFQUFFLHNCQUFzQixFQUFFO29CQUNwRSxHQUFHO29CQUNILFdBQVc7b0JBQ1gsY0FBYyxFQUFFO3dCQUNkLFNBQVM7cUJBQ1Y7b0JBQ0QsYUFBYSxFQUFFO3dCQUNiLHNCQUFZLENBQUMsRUFBRSxDQUFDLHVCQUFhLENBQUMsRUFBRSxFQUFFLHNCQUFZLENBQUMsS0FBSyxDQUFDO3FCQUN0RDtvQkFDRCxrQkFBa0I7b0JBQ2xCLFdBQVcsRUFBRSxDQUFDO29CQUNkLFlBQVksRUFBRSxDQUFDOzRCQUNiLFVBQVU7NEJBQ1YsTUFBTSxFQUFFLG1DQUFpQixDQUFDLEdBQUcsQ0FBQyxVQUFVLEVBQUU7Z0NBQ3hDLElBQUk7Z0NBQ0osVUFBVTs2QkFDWCxDQUFDO3lCQUNILENBQUM7aUJBQ0gsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxpREFBeUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDakYsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO1lBQ3ZILENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuLyogZXNsaW50LWRpc2FibGUgbm8tY29uc29sZSAqL1xuXG4vKiBlc2xpbnQtZGlzYWJsZSBkb3Qtbm90YXRpb24gKi9cblxuaW1wb3J0IHtcbiAgYXJyYXlXaXRoLFxuICBjb3VudFJlc291cmNlcyxcbiAgY291bnRSZXNvdXJjZXNMaWtlLFxuICBleHBlY3QgYXMgZXhwZWN0Q0RLLFxuICBoYXZlUmVzb3VyY2UsXG4gIGhhdmVSZXNvdXJjZUxpa2UsXG4gIG9iamVjdExpa2UsXG59IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQge1xuICBCbG9ja0RldmljZVZvbHVtZSxcbiAgRWJzRGV2aWNlVm9sdW1lVHlwZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWF1dG9zY2FsaW5nJztcbmltcG9ydCB7XG4gIEdlbmVyaWNMaW51eEltYWdlLFxuICBJbnN0YW5jZUNsYXNzLFxuICBJbnN0YW5jZVNpemUsXG4gIEluc3RhbmNlVHlwZSxcbiAgSVZwYyxcbiAgUGVlcixcbiAgU2VjdXJpdHlHcm91cCxcbiAgU3VibmV0U2VsZWN0aW9uLFxuICBTdWJuZXRUeXBlLFxuICBWcGMsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHtcbiAgQXNzZXRJbWFnZSxcbiAgQ29udGFpbmVySW1hZ2UsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnO1xuaW1wb3J0IHtcbiAgTWFuYWdlZFBvbGljeSxcbiAgUm9sZSxcbiAgU2VydmljZVByaW5jaXBhbCxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBBcnRpZmFjdE1ldGFkYXRhRW50cnlUeXBlIH0gZnJvbSAnQGF3cy1jZGsvY2xvdWQtYXNzZW1ibHktc2NoZW1hJztcbmltcG9ydCB7XG4gIEFwcCxcbiAgQ2ZuRWxlbWVudCxcbiAgU3RhY2ssXG4gIFRhZ3MsXG59IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgdGFnRmllbGRzIH0gZnJvbSAnLi4vLi4vY29yZS9saWIvcnVudGltZS1pbmZvJztcbmltcG9ydCB7XG4gIGVzY2FwZVRva2VuUmVnZXgsXG59IGZyb20gJy4uLy4uL2NvcmUvdGVzdC90b2tlbi1yZWdleC1oZWxwZXJzJztcbmltcG9ydCB7IExhdW5jaFRlbXBsYXRlQ29uZmlnIH0gZnJvbSAnLi4vLi4vbGFtYmRhcy9ub2RlanMvY29uZmlndXJlLXNwb3QtZXZlbnQtcGx1Z2luJztcbmltcG9ydCB7XG4gIElIb3N0LFxuICBJbnN0YW5jZVVzZXJEYXRhUHJvdmlkZXIsXG4gIElSZW5kZXJRdWV1ZSxcbiAgUmVuZGVyUXVldWUsXG4gIFJlcG9zaXRvcnksXG4gIFZlcnNpb25RdWVyeSxcbiAgU3BvdEV2ZW50UGx1Z2luRmxlZXQsXG4gIFNwb3RGbGVldEFsbG9jYXRpb25TdHJhdGVneSxcbiAgU3BvdEZsZWV0UmVzb3VyY2VUeXBlLFxufSBmcm9tICcuLi9saWInO1xuXG5sZXQgYXBwOiBBcHA7XG5sZXQgc3RhY2s6IFN0YWNrO1xubGV0IHNwb3RGbGVldFN0YWNrOiBTdGFjaztcbmxldCB2cGM6IElWcGM7XG5sZXQgcmVuZGVyUXVldWU6IElSZW5kZXJRdWV1ZTtcbmxldCByY3NJbWFnZTogQXNzZXRJbWFnZTtcblxuY29uc3QgZ3JvdXBOYW1lID0gJ0dyb3VwX05hbWUnO1xuY29uc3QgZGVhZGxpbmVHcm91cHMgPSBbXG4gIGdyb3VwTmFtZSxcbl07XG5jb25zdCB3b3JrZXJNYWNoaW5lSW1hZ2UgPSBuZXcgR2VuZXJpY0xpbnV4SW1hZ2Uoe1xuICAndXMtZWFzdC0xJzogJ2FtaS1hbnknLFxufSk7XG5jb25zdCBpbnN0YW5jZVR5cGVzID0gW1xuICBJbnN0YW5jZVR5cGUub2YoSW5zdGFuY2VDbGFzcy5UMiwgSW5zdGFuY2VTaXplLlNNQUxMKSxcbl07XG5jb25zdCBtYXhDYXBhY2l0eSA9IDE7XG5cbmRlc2NyaWJlKCdTcG90RXZlbnRQbHVnaW5GbGVldCcsICgpID0+IHtcbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgYXBwID0gbmV3IEFwcCgpO1xuICAgIHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ2luZnJhU3RhY2snLCB7XG4gICAgICBlbnY6IHtcbiAgICAgICAgcmVnaW9uOiAndXMtZWFzdC0xJyxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycpO1xuICAgIHJjc0ltYWdlID0gQ29udGFpbmVySW1hZ2UuZnJvbUFzc2V0KF9fZGlybmFtZSk7XG4gICAgY29uc3QgdmVyc2lvbiA9IG5ldyBWZXJzaW9uUXVlcnkoc3RhY2ssICdWZXJzaW9uUXVlcnknKTtcbiAgICByZW5kZXJRdWV1ZSA9IG5ldyBSZW5kZXJRdWV1ZShzdGFjaywgJ1JRJywge1xuICAgICAgdnBjLFxuICAgICAgaW1hZ2VzOiB7IHJlbW90ZUNvbm5lY3Rpb25TZXJ2ZXI6IHJjc0ltYWdlIH0sXG4gICAgICByZXBvc2l0b3J5OiBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG9zaXRvcnknLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgdmVyc2lvbixcbiAgICAgICAgc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5nczogeyBlbmFibGVkOiBmYWxzZSB9LFxuICAgICAgfSksXG4gICAgICB0cmFmZmljRW5jcnlwdGlvbjogeyBleHRlcm5hbFRMUzogeyBlbmFibGVkOiBmYWxzZSB9IH0sXG4gICAgICB2ZXJzaW9uLFxuICAgIH0pO1xuICAgIHNwb3RGbGVldFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1Nwb3RGbGVldFN0YWNrJywge1xuICAgICAgZW52OiB7XG4gICAgICAgIHJlZ2lvbjogJ3VzLWVhc3QtMScsXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnY3JlYXRlZCB3aXRoIGRlZmF1bHQgdmFsdWVzJywgKCkgPT4ge1xuICAgIHRlc3QoJ2NyZWF0ZXMgYSBzZWN1cml0eSBncm91cCcsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoZmxlZXQuc2VjdXJpdHlHcm91cHMpLnRvQmVEZWZpbmVkKCk7XG4gICAgICBleHBlY3QoZmxlZXQuc2VjdXJpdHlHcm91cHMubGVuZ3RoKS50b0JlKDEpO1xuICAgICAgZXhwZWN0Q0RLKHNwb3RGbGVldFN0YWNrKS50byhjb3VudFJlc291cmNlcygnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnLCAxKSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhbGxvd3MgY29ubmVjdGlvbiB0byB0aGUgcmVuZGVyIHF1ZXVlJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzcG90RmxlZXRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICAgICAgVG9Qb3J0OiBwYXJzZUludChyZW5kZXJRdWV1ZS5lbmRwb2ludC5wb3J0QXNTdHJpbmcoKSwgMTApLFxuICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHNwb3RGbGVldFN0YWNrLnJlc29sdmUoZmxlZXQuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHNbMF0uc2VjdXJpdHlHcm91cElkKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2NyZWF0ZXMgYSBzcG90IGZsZWV0IGluc3RhbmNlIHJvbGUnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGZsZWV0LmZsZWV0SW5zdGFuY2VSb2xlKS50b0JlRGVmaW5lZCgpO1xuICAgICAgZXhwZWN0Q0RLKHNwb3RGbGVldFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICAgICAgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OiBvYmplY3RMaWtlKHtcbiAgICAgICAgICBTdGF0ZW1lbnQ6IFtvYmplY3RMaWtlKHtcbiAgICAgICAgICAgIEFjdGlvbjogJ3N0czpBc3N1bWVSb2xlJyxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgICBTZXJ2aWNlOiAnZWMyLmFtYXpvbmF3cy5jb20nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KV0sXG4gICAgICAgIH0pLFxuICAgICAgICBNYW5hZ2VkUG9saWN5QXJuczogYXJyYXlXaXRoKFxuICAgICAgICAgIHNwb3RGbGVldFN0YWNrLnJlc29sdmUoTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FXU1RoaW5rYm94RGVhZGxpbmVTcG90RXZlbnRQbHVnaW5Xb3JrZXJQb2xpY3knKS5tYW5hZ2VkUG9saWN5QXJuKSxcbiAgICAgICAgKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2NyZWF0ZXMgYW4gaW5zdGFuY2UgcHJvZmlsZScsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoZmxlZXQuaW5zdGFuY2VQcm9maWxlKS50b0JlRGVmaW5lZCgpO1xuICAgICAgZXhwZWN0Q0RLKHNwb3RGbGVldFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6SW5zdGFuY2VQcm9maWxlJywge1xuICAgICAgICBSb2xlczogYXJyYXlXaXRoKHtcbiAgICAgICAgICBSZWY6IHNwb3RGbGVldFN0YWNrLmdldExvZ2ljYWxJZChmbGVldC5mbGVldEluc3RhbmNlUm9sZS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KSxcbiAgICAgICAgfSksXG4gICAgICB9KSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdjcmVhdGVzIGEgc3BvdCBmbGVldCByb2xlJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChmbGVldC5mbGVldFJvbGUpLnRvQmVEZWZpbmVkKCk7XG4gICAgICBleHBlY3RDREsoc3BvdEZsZWV0U3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpSb2xlJywge1xuICAgICAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IG9iamVjdExpa2Uoe1xuICAgICAgICAgIFN0YXRlbWVudDogW29iamVjdExpa2Uoe1xuICAgICAgICAgICAgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICAgIFNlcnZpY2U6ICdzcG90ZmxlZXQuYW1hem9uYXdzLmNvbScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pXSxcbiAgICAgICAgfSksXG4gICAgICAgIE1hbmFnZWRQb2xpY3lBcm5zOiBhcnJheVdpdGgoXG4gICAgICAgICAgc3BvdEZsZWV0U3RhY2sucmVzb2x2ZShNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVDMlNwb3RGbGVldFRhZ2dpbmdSb2xlJykubWFuYWdlZFBvbGljeUFybiksXG4gICAgICAgICksXG4gICAgICB9KSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhZGRzIGdyb3VwIG5hbWVzIHRvIHVzZXIgZGF0YScsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuICAgICAgY29uc3QgcmVuZGVyZWRVc2VyRGF0YSA9IGZsZWV0LnVzZXJEYXRhLnJlbmRlcigpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoZmxlZXQudXNlckRhdGEpLnRvQmVEZWZpbmVkKCk7XG4gICAgICBleHBlY3QocmVuZGVyZWRVc2VyRGF0YSkudG9NYXRjaChncm91cE5hbWUudG9Mb2NhbGVMb3dlckNhc2UoKSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhZGRzIFJGREsgdGFncycsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuICAgICAgY29uc3QgcmZka1RhZyA9IHRhZ0ZpZWxkcyhmbGVldCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChmbGVldC50YWdzKS50b0JlRGVmaW5lZCgpO1xuICAgICAgZXhwZWN0Q0RLKHNwb3RGbGVldFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgVGFnczogYXJyYXlXaXRoKFxuICAgICAgICAgIG9iamVjdExpa2Uoe1xuICAgICAgICAgICAgS2V5OiByZmRrVGFnLm5hbWUsXG4gICAgICAgICAgICBWYWx1ZTogcmZka1RhZy52YWx1ZSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3VzZXMgZGVmYXVsdCBMb2dHcm91cCBwcmVmaXggJXMnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgaWQgPSAnU3BvdEZsZWV0JztcblxuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHN0YWNrLCBpZCwge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpMb2dSZXRlbnRpb24nLCB7XG4gICAgICAgIFJldGVudGlvbkluRGF5czogMyxcbiAgICAgICAgTG9nR3JvdXBOYW1lOiAnL3JlbmRlcmZhcm0vJyArIGlkLFxuICAgICAgfSkpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnc2V0cyBkZWZhdWx0IGFsbG9jYXRpb24gc3RyYXRlZ3knLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGZsZWV0LmFsbG9jYXRpb25TdHJhdGVneSkudG9FcXVhbChTcG90RmxlZXRBbGxvY2F0aW9uU3RyYXRlZ3kuTE9XRVNUX1BSSUNFKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2RvZXMgbm90IHNldCB2YWxpZCB1bnRpbCBwcm9wZXJ0eScsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoZmxlZXQudmFsaWRVbnRpbCkudG9CZVVuZGVmaW5lZCgpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnZG9lcyBub3Qgc2V0IHZhbGlkIGJsb2NrIGRldmljZXMnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGZsZWV0LmJsb2NrRGV2aWNlcykudG9CZVVuZGVmaW5lZCgpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnZG9lcyBub3Qgc2V0IHNzaCBrZXknLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGZsZWV0LmtleU5hbWUpLnRvQmVVbmRlZmluZWQoKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2NyZWF0ZXMgbGF1bmNoIHRlbXBsYXRlIGNvbmZpZ3MgZm9yIGVhY2ggaW5zdGFuY2UgdHlwZScsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IG1vcmVJbnN0YW5jZVR5cGVzOiBJbnN0YW5jZVR5cGVbXSA9IFtcbiAgICAgICAgbmV3IEluc3RhbmNlVHlwZSgndDIuc21hbGwnKSxcbiAgICAgICAgbmV3IEluc3RhbmNlVHlwZSgnYzUubGFyZ2UnKSxcbiAgICAgIF07XG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgIGluc3RhbmNlVHlwZXM6IG1vcmVJbnN0YW5jZVR5cGVzLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChmbGVldC5fbGF1bmNoVGVtcGxhdGVDb25maWdzLmxlbmd0aCkudG9CZUdyZWF0ZXJUaGFuT3JFcXVhbChtb3JlSW5zdGFuY2VUeXBlcy5sZW5ndGgpO1xuICAgICAgbW9yZUluc3RhbmNlVHlwZXMuZm9yRWFjaChpbnN0YW5jZVR5cGUgPT4ge1xuICAgICAgICBleHBlY3QoZmxlZXQuX2xhdW5jaFRlbXBsYXRlQ29uZmlncy5zb21lKGx0YyA9PiB7XG4gICAgICAgICAgcmV0dXJuIChsdGMgYXMgTGF1bmNoVGVtcGxhdGVDb25maWcpLk92ZXJyaWRlcy5zb21lKG92ZXJyaWRlID0+IG92ZXJyaWRlLkluc3RhbmNlVHlwZSA9PT0gaW5zdGFuY2VUeXBlLnRvU3RyaW5nKCkpO1xuICAgICAgICB9KSkudG9CZVRydXRoeSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdjcmVhdGVzIGxhdW5jaCB0ZW1wbGF0ZSBjb25maWdzIGZvciBlYWNoIHN1Ym5ldCBpZCcsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHN1Ym5ldHMgPSB2cGMuc2VsZWN0U3VibmV0cyh7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSB9KTtcbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgdnBjU3VibmV0czogc3VibmV0cyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoZmxlZXQuX2xhdW5jaFRlbXBsYXRlQ29uZmlncy5sZW5ndGgpLnRvQmVHcmVhdGVyVGhhbk9yRXF1YWwoc3VibmV0cy5zdWJuZXRzLmxlbmd0aCk7XG4gICAgICBzdWJuZXRzLnN1Ym5ldElkcy5mb3JFYWNoKHN1Ym5ldElkID0+IHtcbiAgICAgICAgZXhwZWN0KGZsZWV0Ll9sYXVuY2hUZW1wbGF0ZUNvbmZpZ3Muc29tZShsdGMgPT4ge1xuICAgICAgICAgIHJldHVybiAobHRjIGFzIExhdW5jaFRlbXBsYXRlQ29uZmlnKS5PdmVycmlkZXMuc29tZShvdmVycmlkZSA9PiBvdmVycmlkZS5TdWJuZXRJZCA9PT0gc3VibmV0SWQpO1xuICAgICAgICB9KSkudG9CZVRydXRoeSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhZGQgdGFnIGluZGljYXRpbmcgcmVzb3VyY2UgdHJhY2tlciBpcyBlbmFibGVkJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoc3BvdEZsZWV0U3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpMYXVuY2hUZW1wbGF0ZScsIHtcbiAgICAgICAgTGF1bmNoVGVtcGxhdGVEYXRhOiBvYmplY3RMaWtlKHtcbiAgICAgICAgICBUYWdTcGVjaWZpY2F0aW9uczogYXJyYXlXaXRoKHtcbiAgICAgICAgICAgIFJlc291cmNlVHlwZTogJ2luc3RhbmNlJyxcbiAgICAgICAgICAgIFRhZ3M6IGFycmF5V2l0aCh7XG4gICAgICAgICAgICAgIEtleTogJ0RlYWRsaW5lVHJhY2tlZEFXU1Jlc291cmNlJyxcbiAgICAgICAgICAgICAgVmFsdWU6ICdTcG90RXZlbnRQbHVnaW4nLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgfSksXG4gICAgICAgIH0pLFxuICAgICAgfSkpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnYWRkcyBtdWx0aXBsZSBmbGVldCBzZWN1cml0eSBncm91cHMgdG8gbGF1bmNoIHRlbXBsYXRlJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHNlY3VyaXR5R3JvdXBzID0gW1xuICAgICAgICBuZXcgU2VjdXJpdHlHcm91cChzdGFjaywgJ05ld0ZsZWV0U2VjdXJpdHlHcm91cDEnLCB7IHZwYyB9KSxcbiAgICAgICAgbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdOZXdGbGVldFNlY3VyaXR5R3JvdXAyJywgeyB2cGMgfSksXG4gICAgICBdO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQyJywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwczogWydncm91cDInXSxcbiAgICAgICAgaW5zdGFuY2VUeXBlczogW25ldyBJbnN0YW5jZVR5cGUoJ3QyLm1pY3JvJyldLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5OiAxLFxuICAgICAgICBzZWN1cml0eUdyb3VwcyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoc3BvdEZsZWV0U3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpMYXVuY2hUZW1wbGF0ZScsIHtcbiAgICAgICAgTGF1bmNoVGVtcGxhdGVEYXRhOiBvYmplY3RMaWtlKHtcbiAgICAgICAgICBTZWN1cml0eUdyb3VwSWRzOiBzZWN1cml0eUdyb3Vwcy5tYXAoc2cgPT4gc3BvdEZsZWV0U3RhY2sucmVzb2x2ZShzZy5zZWN1cml0eUdyb3VwSWQpKSxcbiAgICAgICAgfSksXG4gICAgICB9KSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhZGRzIGZsZWV0IHRhZ3MgdG8gbGF1bmNoIHRlbXBsYXRlJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHRhZyA9IHtcbiAgICAgICAga2V5OiAnbXlrZXknLFxuICAgICAgICB2YWx1ZTogJ215dmFsdWUnLFxuICAgICAgfTtcbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBUYWdzLm9mKGZsZWV0KS5hZGQodGFnLmtleSwgdGFnLnZhbHVlKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKHNwb3RGbGVldFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6TGF1bmNoVGVtcGxhdGUnLCB7XG4gICAgICAgIExhdW5jaFRlbXBsYXRlRGF0YTogb2JqZWN0TGlrZSh7XG4gICAgICAgICAgVGFnU3BlY2lmaWNhdGlvbnM6IGFycmF5V2l0aCh7XG4gICAgICAgICAgICBSZXNvdXJjZVR5cGU6IFNwb3RGbGVldFJlc291cmNlVHlwZS5JTlNUQU5DRS50b1N0cmluZygpLFxuICAgICAgICAgICAgVGFnczogYXJyYXlXaXRoKHtcbiAgICAgICAgICAgICAgS2V5OiB0YWcua2V5LFxuICAgICAgICAgICAgICBWYWx1ZTogdGFnLnZhbHVlLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgfSksXG4gICAgICAgIH0pLFxuICAgICAgfSkpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnY3JlYXRlZCB3aXRoIGN1c3RvbSB2YWx1ZXMnLCAoKSA9PiB7XG4gICAgdGVzdCgndXNlcyBwcm92aWRlZCByZXF1aXJlZCBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgfSk7XG4gICAgICBjb25zdCBpbWFnZUNvbmZpZyA9IHdvcmtlck1hY2hpbmVJbWFnZS5nZXRJbWFnZShmbGVldCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChmbGVldC5kZWFkbGluZUdyb3VwcykudG9TdHJpY3RFcXVhbChkZWFkbGluZUdyb3Vwcy5tYXAoZ3JvdXAgPT4gZ3JvdXAudG9Mb2NhbGVMb3dlckNhc2UoKSkpO1xuICAgICAgZXhwZWN0KGZsZWV0Lmluc3RhbmNlVHlwZXMpLnRvQmUoaW5zdGFuY2VUeXBlcyk7XG4gICAgICBleHBlY3QoZmxlZXQub3NUeXBlKS50b0JlKGltYWdlQ29uZmlnLm9zVHlwZSk7XG4gICAgICBleHBlY3QoZmxlZXQubWF4Q2FwYWNpdHkpLnRvQmUobWF4Q2FwYWNpdHkpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndXNlcyBwcm92aWRlZCBzZWN1cml0eSBncm91cCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzZyA9IFNlY3VyaXR5R3JvdXAuZnJvbVNlY3VyaXR5R3JvdXBJZChzdGFjaywgJ1NHJywgJ3NnLTEyMzQ1Njc4OScsIHtcbiAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogZmFsc2UsXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICBzZWN1cml0eUdyb3VwczogW1xuICAgICAgICAgIHNnLFxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzcG90RmxlZXRTdGFjaykubm90VG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cCcpKTtcbiAgICAgIGV4cGVjdChmbGVldC5zZWN1cml0eUdyb3Vwcy5sZW5ndGgpLnRvQmUoMSk7XG4gICAgICBleHBlY3QoZmxlZXQuc2VjdXJpdHlHcm91cHMpLnRvQ29udGFpbkVxdWFsKHNnKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3VzZXMgbXVsdGlwbGUgcHJvdmlkZWQgc2VjdXJpdHkgZ3JvdXBzJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHNnMSA9IFNlY3VyaXR5R3JvdXAuZnJvbVNlY3VyaXR5R3JvdXBJZChzdGFjaywgJ1NHMScsICdzZy0xMjM0NTY3ODknLCB7XG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlLFxuICAgICAgfSk7XG4gICAgICBjb25zdCBzZzIgPSBTZWN1cml0eUdyb3VwLmZyb21TZWN1cml0eUdyb3VwSWQoc3RhY2ssICdTRzInLCAnc2ctOTg3NjU0MzIxJywge1xuICAgICAgICBhbGxvd0FsbE91dGJvdW5kOiBmYWxzZSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgIHNlY3VyaXR5R3JvdXBzOiBbXG4gICAgICAgICAgc2cxLFxuICAgICAgICAgIHNnMixcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoc3BvdEZsZWV0U3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnKSk7XG4gICAgICBleHBlY3QoZmxlZXQuc2VjdXJpdHlHcm91cHMubGVuZ3RoKS50b0JlKDIpO1xuICAgICAgZXhwZWN0KGZsZWV0LnNlY3VyaXR5R3JvdXBzKS50b0NvbnRhaW5FcXVhbChzZzEpO1xuICAgICAgZXhwZWN0KGZsZWV0LnNlY3VyaXR5R3JvdXBzKS50b0NvbnRhaW5FcXVhbChzZzIpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnYWRkcyB0byBwcm92aWRlZCB1c2VyIGRhdGEnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgb3JpZ2luYWxDb21tYW5kcyA9ICdvcmlnaW5hbCBjb21tYW5kcyc7XG4gICAgICBjb25zdCBvcmlnaW5hbFVzZXJEYXRhID0gd29ya2VyTWFjaGluZUltYWdlLmdldEltYWdlKHNwb3RGbGVldFN0YWNrKS51c2VyRGF0YTtcbiAgICAgIG9yaWdpbmFsVXNlckRhdGEuYWRkQ29tbWFuZHMob3JpZ2luYWxDb21tYW5kcyk7XG4gICAgICBjb25zdCByZW5kZXJlZE9yaWdpbmFsVXNlciA9IG9yaWdpbmFsVXNlckRhdGEucmVuZGVyKCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgdXNlckRhdGE6IG9yaWdpbmFsVXNlckRhdGEsXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHJlbmRlcmVkVXNlckRhdGEgPSBmbGVldC51c2VyRGF0YS5yZW5kZXIoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGZsZWV0LnVzZXJEYXRhKS50b0JlKG9yaWdpbmFsVXNlckRhdGEpO1xuICAgICAgZXhwZWN0KHJlbmRlcmVkVXNlckRhdGEpLnRvTWF0Y2gobmV3IFJlZ0V4cChlc2NhcGVUb2tlblJlZ2V4KG9yaWdpbmFsQ29tbWFuZHMpKSk7XG4gICAgICBleHBlY3QocmVuZGVyZWRVc2VyRGF0YSkubm90LnRvRXF1YWwocmVuZGVyZWRPcmlnaW5hbFVzZXIpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndXNlcyBwcm92aWRlZCBzcG90IGZsZWV0IGluc3RhbmNlIHJvbGUgZnJvbSB0aGUgc2FtZSBzdGFjaycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzcG90RmxlZXRJbnN0YW5jZVJvbGUgPSBuZXcgUm9sZShzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldEluc3RhbmNlUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FXU1RoaW5rYm94RGVhZGxpbmVTcG90RXZlbnRQbHVnaW5Xb3JrZXJQb2xpY3knKSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgIGZsZWV0SW5zdGFuY2VSb2xlOiBzcG90RmxlZXRJbnN0YW5jZVJvbGUsXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGZsZWV0LmZsZWV0SW5zdGFuY2VSb2xlKS50b0JlKHNwb3RGbGVldEluc3RhbmNlUm9sZSk7XG4gICAgICBleHBlY3RDREsoc3BvdEZsZWV0U3RhY2spLnRvKGNvdW50UmVzb3VyY2VzTGlrZSgnQVdTOjpJQU06OlJvbGUnLCAxLCB7XG4gICAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDogb2JqZWN0TGlrZSh7XG4gICAgICAgICAgU3RhdGVtZW50OiBbb2JqZWN0TGlrZSh7XG4gICAgICAgICAgICBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgU2VydmljZTogJ2VjMi5hbWF6b25hd3MuY29tJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSldLFxuICAgICAgICB9KSxcbiAgICAgICAgTWFuYWdlZFBvbGljeUFybnM6IGFycmF5V2l0aChcbiAgICAgICAgICBzcG90RmxlZXRTdGFjay5yZXNvbHZlKE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBV1NUaGlua2JveERlYWRsaW5lU3BvdEV2ZW50UGx1Z2luV29ya2VyUG9saWN5JykubWFuYWdlZFBvbGljeUFybiksXG4gICAgICAgICksXG4gICAgICB9KSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCd0aHJvd3MgaWYgcHJvdmlkZWQgc3BvdCBmbGVldCBpbnN0YW5jZSByb2xlIGlzIG5vdCBmcm9tIHRoZSBzYW1lIHN0YWNrJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG90aGVyU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnT3RoZXJTdGFjaycsIHtcbiAgICAgICAgZW52OiB7IHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgICAgIH0pO1xuICAgICAgY29uc3Qgc3BvdEZsZWV0SW5zdGFuY2VSb2xlID0gbmV3IFJvbGUob3RoZXJTdGFjaywgJ1Nwb3RGbGVldEluc3RhbmNlUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FXU1RoaW5rYm94RGVhZGxpbmVTcG90RXZlbnRQbHVnaW5Xb3JrZXJQb2xpY3knKSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBmdW5jdGlvbiBjcmVhdGVTcG90RXZlbnRQbHVnaW5GbGVldCgpIHtcbiAgICAgICAgbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgICBmbGVldEluc3RhbmNlUm9sZTogc3BvdEZsZWV0SW5zdGFuY2VSb2xlLFxuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KS50b1Rocm93RXJyb3IoJ0ZsZWV0IGluc3RhbmNlIHJvbGUgc2hvdWxkIGJlIGNyZWF0ZWQgb24gdGhlIHNhbWUgc3RhY2sgYXMgU3BvdEV2ZW50UGx1Z2luRmxlZXQgdG8gYXZvaWQgY2lyY3VsYXIgZGVwZW5kZW5jaWVzLicpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndXNlcyBwcm92aWRlZCBzcG90IGZsZWV0IHJvbGUnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgZmxlZXRSb2xlID0gbmV3IFJvbGUoc3RhY2ssICdGbGVldFJvbGUnLCB7XG4gICAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ3Nwb3RmbGVldC5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgIE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQW1hem9uRUMyU3BvdEZsZWV0VGFnZ2luZ1JvbGUnKSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZmxlZXRSb2xlOiBmbGVldFJvbGUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChmbGVldC5mbGVldFJvbGUpLnRvQmUoZmxlZXRSb2xlKTtcbiAgICAgIGV4cGVjdENESyhzcG90RmxlZXRTdGFjaykubm90VG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDogb2JqZWN0TGlrZSh7XG4gICAgICAgICAgU3RhdGVtZW50OiBbb2JqZWN0TGlrZSh7XG4gICAgICAgICAgICBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgU2VydmljZTogJ3Nwb3RmbGVldC5hbWF6b25hd3MuY29tJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSldLFxuICAgICAgICB9KSxcbiAgICAgICAgTWFuYWdlZFBvbGljeUFybnM6IGFycmF5V2l0aChcbiAgICAgICAgICBzdGFjay5yZXNvbHZlKE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQW1hem9uRUMyU3BvdEZsZWV0VGFnZ2luZ1JvbGUnKS5tYW5hZ2VkUG9saWN5QXJuKSxcbiAgICAgICAgKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3RhZ3MgcmVzb3VyY2VzIGRlcGxveWVkIGJ5IENESycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHRhZ05hbWUgPSAnbmFtZSc7XG4gICAgICBjb25zdCB0YWdWYWx1ZSA9ICd0YWdWYWx1ZSc7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIFRhZ3Mub2YoZmxlZXQpLmFkZCh0YWdOYW1lLCB0YWdWYWx1ZSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzcG90RmxlZXRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIFRhZ3M6IGFycmF5V2l0aChcbiAgICAgICAgICBvYmplY3RMaWtlKHtcbiAgICAgICAgICAgIEtleTogdGFnTmFtZSxcbiAgICAgICAgICAgIFZhbHVlOiB0YWdWYWx1ZSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3VzZXMgcHJvdmlkZWQgc3VibmV0cycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBwcml2YXRlU3VibmV0czogU3VibmV0U2VsZWN0aW9uID0ge1xuICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgIHZwY1N1Ym5ldHM6IHByaXZhdGVTdWJuZXRzLFxuICAgICAgfSk7XG4gICAgICBjb25zdCBleHBlY3RlZFN1Ym5ldElkID0gc3RhY2sucmVzb2x2ZSh2cGMucHJpdmF0ZVN1Ym5ldHNbMF0uc3VibmV0SWQpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2sucmVzb2x2ZShmbGVldC5zdWJuZXRzLnN1Ym5ldElkcykpLnRvQ29udGFpbkVxdWFsKGV4cGVjdGVkU3VibmV0SWQpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnLmRlZmF1bHRTdWJuZXRzIGlzIGZhbHNlIHdoZW4gc3VibmV0cyBwcm92aWRlZCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBwcml2YXRlU3VibmV0czogU3VibmV0U2VsZWN0aW9uID0ge1xuICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgIHZwY1N1Ym5ldHM6IHByaXZhdGVTdWJuZXRzLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChmbGVldC5kZWZhdWx0U3VibmV0cykudG9CZUZhbHN5KCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCd1c2VzIHByb3ZpZGVkIGFsbG9jYXRpb24gc3RyYXRlZ3knLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgYWxsb2NhdGlvblN0YXJ0ZWd5ID0gU3BvdEZsZWV0QWxsb2NhdGlvblN0cmF0ZWd5LkNBUEFDSVRZX09QVElNSVpFRDtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICBhbGxvY2F0aW9uU3RyYXRlZ3k6IGFsbG9jYXRpb25TdGFydGVneSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoZmxlZXQuYWxsb2NhdGlvblN0cmF0ZWd5KS50b0VxdWFsKGFsbG9jYXRpb25TdGFydGVneSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhZGRzIGRlYWRsaW5lIHJlZ2lvbiB0byB1c2VyIGRhdGEnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgZGVhZGxpbmVSZWdpb24gPSAnc29tZXJlZ2lvbic7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgZGVhZGxpbmVSZWdpb246IGRlYWRsaW5lUmVnaW9uLFxuICAgICAgfSk7XG4gICAgICBjb25zdCByZW5kZXJlZFVzZXJEYXRhID0gZmxlZXQudXNlckRhdGEucmVuZGVyKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChyZW5kZXJlZFVzZXJEYXRhKS50b01hdGNoKGRlYWRsaW5lUmVnaW9uKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2FkZHMgZGVhZGxpbmUgcG9vbHMgdG8gdXNlciBkYXRhJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHBvb2wxID0gJ1Bvb2wxJztcbiAgICAgIGNvbnN0IHBvb2wyID0gJ3Bvb2wyJztcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICBkZWFkbGluZVBvb2xzOiBbXG4gICAgICAgICAgcG9vbDEsXG4gICAgICAgICAgcG9vbDIsXG4gICAgICAgIF0sXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHJlbmRlcmVkVXNlckRhdGEgPSBmbGVldC51c2VyRGF0YS5yZW5kZXIoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHJlbmRlcmVkVXNlckRhdGEpLnRvTWF0Y2gocG9vbDEudG9Mb2NhbGVMb3dlckNhc2UoKSk7XG4gICAgICBleHBlY3QocmVuZGVyZWRVc2VyRGF0YSkudG9NYXRjaChwb29sMi50b0xvY2FsZUxvd2VyQ2FzZSgpKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3VzZXMgcHJvdmlkZWQgc3NoIGtleSBuYW1lJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGtleU5hbWUgPSAndGVzdC1rZXktbmFtZSc7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAga2V5TmFtZToga2V5TmFtZSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoZmxlZXQua2V5TmFtZSkudG9FcXVhbChrZXlOYW1lKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ1VzZXJEYXRhIGlzIGFkZGVkIGJ5IFVzZXJEYXRhUHJvdmlkZXInLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjbGFzcyBVc2VyRGF0YVByb3ZpZGVyIGV4dGVuZHMgSW5zdGFuY2VVc2VyRGF0YVByb3ZpZGVyIHtcbiAgICAgICAgcHJlQ2xvdWRXYXRjaEFnZW50KGhvc3Q6IElIb3N0KTogdm9pZCB7XG4gICAgICAgICAgaG9zdC51c2VyRGF0YS5hZGRDb21tYW5kcygnZWNobyBwcmVDbG91ZFdhdGNoQWdlbnQnKTtcbiAgICAgICAgfVxuICAgICAgICBwcmVSZW5kZXJRdWV1ZUNvbmZpZ3VyYXRpb24oaG9zdDogSUhvc3QpOiB2b2lkIHtcbiAgICAgICAgICBob3N0LnVzZXJEYXRhLmFkZENvbW1hbmRzKCdlY2hvIHByZVJlbmRlclF1ZXVlQ29uZmlndXJhdGlvbicpO1xuICAgICAgICB9XG4gICAgICAgIHByZVdvcmtlckNvbmZpZ3VyYXRpb24oaG9zdDogSUhvc3QpOiB2b2lkIHtcbiAgICAgICAgICBob3N0LnVzZXJEYXRhLmFkZENvbW1hbmRzKCdlY2hvIHByZVdvcmtlckNvbmZpZ3VyYXRpb24nKTtcbiAgICAgICAgfVxuICAgICAgICBwb3N0V29ya2VyTGF1bmNoKGhvc3Q6IElIb3N0KTogdm9pZCB7XG4gICAgICAgICAgaG9zdC51c2VyRGF0YS5hZGRDb21tYW5kcygnZWNobyBwb3N0V29ya2VyTGF1bmNoJyk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICB1c2VyRGF0YVByb3ZpZGVyOiBuZXcgVXNlckRhdGFQcm92aWRlcihzcG90RmxlZXRTdGFjaywgJ1VzZXJEYXRhUHJvdmlkZXInKSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCB1c2VyRGF0YSA9IGZsZWV0LnVzZXJEYXRhLnJlbmRlcigpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignZWNobyBwcmVDbG91ZFdhdGNoQWdlbnQnKTtcbiAgICAgIGV4cGVjdCh1c2VyRGF0YSkudG9Db250YWluKCdlY2hvIHByZVJlbmRlclF1ZXVlQ29uZmlndXJhdGlvbicpO1xuICAgICAgZXhwZWN0KHVzZXJEYXRhKS50b0NvbnRhaW4oJ2VjaG8gcHJlV29ya2VyQ29uZmlndXJhdGlvbicpO1xuICAgICAgZXhwZWN0KHVzZXJEYXRhKS50b0NvbnRhaW4oJ2VjaG8gcG9zdFdvcmtlckxhdW5jaCcpO1xuICAgIH0pO1xuXG4gICAgdGVzdC5lYWNoKFtcbiAgICAgICd0ZXN0LXByZWZpeC8nLFxuICAgICAgJycsXG4gICAgXSkoJ3VzZXMgY3VzdG9tIExvZ0dyb3VwIHByZWZpeCAlcycsICh0ZXN0UHJlZml4OiBzdHJpbmcpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBpZCA9ICdTcG90RmxlZXQnO1xuICAgICAgY29uc3QgbG9nR3JvdXBQcm9wcyA9IHtcbiAgICAgICAgbG9nR3JvdXBQcmVmaXg6IHRlc3RQcmVmaXgsXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3RhY2ssIGlkLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICBsb2dHcm91cFByb3BzOiBsb2dHcm91cFByb3BzLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdDdXN0b206OkxvZ1JldGVudGlvbicsIHtcbiAgICAgICAgUmV0ZW50aW9uSW5EYXlzOiAzLFxuICAgICAgICBMb2dHcm91cE5hbWU6IHRlc3RQcmVmaXggKyBpZCxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2FkZHMgdGFnIGluZGljYXRpbmcgcmVzb3VyY2UgdHJhY2tlciBpcyBub3QgZW5hYmxlZCcsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgIHRyYWNrSW5zdGFuY2VzV2l0aFJlc291cmNlVHJhY2tlcjogZmFsc2UsXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKHNwb3RGbGVldFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6TGF1bmNoVGVtcGxhdGUnLCB7XG4gICAgICAgIExhdW5jaFRlbXBsYXRlRGF0YTogb2JqZWN0TGlrZSh7XG4gICAgICAgICAgVGFnU3BlY2lmaWNhdGlvbnM6IGFycmF5V2l0aCh7XG4gICAgICAgICAgICBSZXNvdXJjZVR5cGU6ICdpbnN0YW5jZScsXG4gICAgICAgICAgICBUYWdzOiBhcnJheVdpdGgoe1xuICAgICAgICAgICAgICBLZXk6ICdEZWFkbGluZVJlc291cmNlVHJhY2tlcicsXG4gICAgICAgICAgICAgIFZhbHVlOiAnU3BvdEV2ZW50UGx1Z2luJyxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIH0pLFxuICAgICAgICB9KSxcbiAgICAgIH0pKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2FsbG93aW5nIHJlbW90ZSBjb250cm9sJywgKCkgPT4ge1xuICAgIHRlc3QoJ2Zyb20gQ0lEUicsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBmcm9tUG9ydCA9IDU2MDMyO1xuICAgICAgY29uc3QgbWF4V29ya2Vyc1Blckhvc3QgPSA4O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICB9KTtcblxuICAgICAgZmxlZXQuYWxsb3dSZW1vdGVDb250cm9sRnJvbShQZWVyLmlwdjQoJzEyNy4wLjAuMS8yNCcpKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgU2VjdXJpdHlHcm91cEVncmVzczogW3sgQ2lkcklwOiAnMC4wLjAuMC8wJyB9XSxcbiAgICAgICAgU2VjdXJpdHlHcm91cEluZ3Jlc3M6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBDaWRySXA6ICcxMjcuMC4wLjEvMjQnLFxuICAgICAgICAgICAgRGVzY3JpcHRpb246ICdXb3JrZXIgcmVtb3RlIGNvbW1hbmQgbGlzdGVuaW5nIHBvcnQnLFxuICAgICAgICAgICAgRnJvbVBvcnQ6IGZyb21Qb3J0LFxuICAgICAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgICAgICBUb1BvcnQ6IGZyb21Qb3J0ICsgbWF4V29ya2Vyc1Blckhvc3QsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3RvIENJRFInLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgZnJvbVBvcnQgPSA1NjAzMjtcbiAgICAgIGNvbnN0IG1heFdvcmtlcnNQZXJIb3N0ID0gODtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgfSk7XG5cbiAgICAgIGZsZWV0LmFsbG93UmVtb3RlQ29udHJvbFRvKFBlZXIuaXB2NCgnMTI3LjAuMC4xLzI0JykpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJywge1xuICAgICAgICBTZWN1cml0eUdyb3VwRWdyZXNzOiBbeyBDaWRySXA6ICcwLjAuMC4wLzAnIH1dLFxuICAgICAgICBTZWN1cml0eUdyb3VwSW5ncmVzczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIENpZHJJcDogJzEyNy4wLjAuMS8yNCcsXG4gICAgICAgICAgICBEZXNjcmlwdGlvbjogJ1dvcmtlciByZW1vdGUgY29tbWFuZCBsaXN0ZW5pbmcgcG9ydCcsXG4gICAgICAgICAgICBGcm9tUG9ydDogZnJvbVBvcnQsXG4gICAgICAgICAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICAgICAgICAgIFRvUG9ydDogZnJvbVBvcnQgKyBtYXhXb3JrZXJzUGVySG9zdCxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSkpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnZnJvbSBTZWN1cml0eUdyb3VwJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGZyb21Qb3J0ID0gNTYwMzI7XG4gICAgICBjb25zdCBtYXhXb3JrZXJzUGVySG9zdCA9IDg7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIH0pO1xuICAgICAgY29uc3Qgc2VjdXJpdHlHcm91cCA9IFNlY3VyaXR5R3JvdXAuZnJvbVNlY3VyaXR5R3JvdXBJZChzdGFjaywgJ1NHJywgJ3NnLTEyMzQ1Njc4OScpO1xuXG4gICAgICBmbGVldC5hbGxvd1JlbW90ZUNvbnRyb2xGcm9tKHNlY3VyaXR5R3JvdXApO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICAgICAgRnJvbVBvcnQ6IGZyb21Qb3J0LFxuICAgICAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICAgICAgU291cmNlU2VjdXJpdHlHcm91cElkOiAnc2ctMTIzNDU2Nzg5JyxcbiAgICAgICAgVG9Qb3J0OiBmcm9tUG9ydCArIG1heFdvcmtlcnNQZXJIb3N0LFxuICAgICAgfSkpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndG8gU2VjdXJpdHlHcm91cCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBmcm9tUG9ydCA9IDU2MDMyO1xuICAgICAgY29uc3QgbWF4V29ya2Vyc1Blckhvc3QgPSA4O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHNlY3VyaXR5R3JvdXAgPSBTZWN1cml0eUdyb3VwLmZyb21TZWN1cml0eUdyb3VwSWQoc3RhY2ssICdTRycsICdzZy0xMjM0NTY3ODknKTtcblxuICAgICAgZmxlZXQuYWxsb3dSZW1vdGVDb250cm9sVG8oc2VjdXJpdHlHcm91cCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgICBGcm9tUG9ydDogZnJvbVBvcnQsXG4gICAgICAgIElwUHJvdG9jb2w6ICd0Y3AnLFxuICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6ICdzZy0xMjM0NTY3ODknLFxuICAgICAgICBUb1BvcnQ6IGZyb21Qb3J0ICsgbWF4V29ya2Vyc1Blckhvc3QsXG4gICAgICB9KSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdmcm9tIG90aGVyIHN0YWNrJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGZyb21Qb3J0ID0gNTYwMzI7XG4gICAgICBjb25zdCBtYXhXb3JrZXJzUGVySG9zdCA9IDg7XG4gICAgICBjb25zdCBvdGhlclN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ290aGVyU3RhY2snLCB7XG4gICAgICAgIGVudjogeyByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgfSk7XG4gICAgICBjb25zdCBzZWN1cml0eUdyb3VwID0gU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKG90aGVyU3RhY2ssICdTRycsICdzZy0xMjM0NTY3ODknKTtcblxuICAgICAgZmxlZXQuYWxsb3dSZW1vdGVDb250cm9sRnJvbShzZWN1cml0eUdyb3VwKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgICAgIEZyb21Qb3J0OiBmcm9tUG9ydCxcbiAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgIFNvdXJjZVNlY3VyaXR5R3JvdXBJZDogJ3NnLTEyMzQ1Njc4OScsXG4gICAgICAgIFRvUG9ydDogZnJvbVBvcnQgKyBtYXhXb3JrZXJzUGVySG9zdCxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3RvIG90aGVyIHN0YWNrJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGZyb21Qb3J0ID0gNTYwMzI7XG4gICAgICBjb25zdCBtYXhXb3JrZXJzUGVySG9zdCA9IDg7XG4gICAgICBjb25zdCBvdGhlclN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ290aGVyU3RhY2snLCB7XG4gICAgICAgIGVudjogeyByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgfSk7XG4gICAgICBjb25zdCBzZWN1cml0eUdyb3VwID0gU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKG90aGVyU3RhY2ssICdTRycsICdzZy0xMjM0NTY3ODknKTtcblxuICAgICAgZmxlZXQuYWxsb3dSZW1vdGVDb250cm9sVG8oc2VjdXJpdHlHcm91cCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhvdGhlclN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgICAgIEZyb21Qb3J0OiBmcm9tUG9ydCxcbiAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgIFNvdXJjZVNlY3VyaXR5R3JvdXBJZDogJ3NnLTEyMzQ1Njc4OScsXG4gICAgICAgIFRvUG9ydDogZnJvbVBvcnQgKyBtYXhXb3JrZXJzUGVySG9zdCxcbiAgICAgIH0pKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ3ZhbGlkYXRpb24gd2l0aCcsICgpID0+IHtcbiAgICBkZXNjcmliZSgnaW5zdGFuY2UgdHlwZXMnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCd0aHJvd3Mgd2l0aCBlbXB0eScsICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgZW1wdHlJbnN0YW5jZVR5cGVzOiBJbnN0YW5jZVR5cGVbXSA9IFtdO1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgZnVuY3Rpb24gY3JlYXRlU3BvdEV2ZW50UGx1Z2luRmxlZXQoKSB7XG4gICAgICAgICAgbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICAgICAgdnBjLFxuICAgICAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgICAgIGluc3RhbmNlVHlwZXM6IGVtcHR5SW5zdGFuY2VUeXBlcyxcbiAgICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QoY3JlYXRlU3BvdEV2ZW50UGx1Z2luRmxlZXQpLnRvVGhyb3dFcnJvcigvQXQgbGVhc3Qgb25lIGluc3RhbmNlIHR5cGUgaXMgcmVxdWlyZWQgZm9yIGEgU3BvdCBGbGVldCBSZXF1ZXN0IENvbmZpZ3VyYXRpb24vKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdwYXNzZXMgd2l0aCBhdCBsZWFzdCBvbmUnLCAoKSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IG9uZUluc3RhbmNlVHlwZSA9IFsgSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDIsIEluc3RhbmNlU2l6ZS5TTUFMTCkgXTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGZ1bmN0aW9uIGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KCkge1xuICAgICAgICAgIG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgICAgICBpbnN0YW5jZVR5cGVzOiBvbmVJbnN0YW5jZVR5cGUsXG4gICAgICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KS5ub3QudG9UaHJvd0Vycm9yKCk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCdzdWJuZXRzJywgKCkgPT4ge1xuICAgICAgdGVzdCgnZXJyb3IgaWYgbm8gc3VibmV0cyBwcm92aWRlZCcsICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgaW52YWxpZFN1Ym5ldHMgPSB7XG4gICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFLFxuICAgICAgICAgIGF2YWlsYWJpbGl0eVpvbmVzOiBbJ2R1bW15IHpvbmUnXSxcbiAgICAgICAgfTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgICB2cGNTdWJuZXRzOiBpbnZhbGlkU3VibmV0cyxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QoZmxlZXQubm9kZS5tZXRhZGF0YUVudHJ5WzBdLnR5cGUpLnRvTWF0Y2goQXJ0aWZhY3RNZXRhZGF0YUVudHJ5VHlwZS5FUlJPUik7XG4gICAgICAgIGV4cGVjdChmbGVldC5ub2RlLm1ldGFkYXRhRW50cnlbMF0uZGF0YSkudG9NYXRjaCgvRGlkIG5vdCBmaW5kIGFueSBzdWJuZXRzIG1hdGNoaW5nLyk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCdncm91cHMnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCd0aHJvd3Mgd2l0aCBlbXB0eScsICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgZW1wdHlEZWFkbGluZUdyb3Vwczogc3RyaW5nW10gPSBbXTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGZ1bmN0aW9uIGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KCkge1xuICAgICAgICAgIG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICAgICAgZGVhZGxpbmVHcm91cHM6IGVtcHR5RGVhZGxpbmVHcm91cHMsXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChjcmVhdGVTcG90RXZlbnRQbHVnaW5GbGVldCkudG9UaHJvd0Vycm9yKC9BdCBsZWFzdCBvbmUgRGVhZGxpbmUgR3JvdXAgaXMgcmVxdWlyZWQgZm9yIGEgU3BvdCBGbGVldCBSZXF1ZXN0IENvbmZpZ3VyYXRpb24vKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmVhY2goW1xuICAgICAgICAnbm9uZScsXG4gICAgICAgICd3aXRoIHNwYWNlJyxcbiAgICAgICAgJ2dyb3VwXyonLCAvLyB3aXRoIHdpbGRjYXJkXG4gICAgICBdKSgndGhyb3dzIHdpdGggJXMnLCAoaW52YWxpZEdyb3VwTmFtZTogc3RyaW5nKSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgZnVuY3Rpb24gY3JlYXRlU3BvdEV2ZW50UGx1Z2luRmxlZXQoKSB7XG4gICAgICAgICAgbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICAgICAgdnBjLFxuICAgICAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgICAgICBkZWFkbGluZUdyb3VwczogW2ludmFsaWRHcm91cE5hbWVdLFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QoY3JlYXRlU3BvdEV2ZW50UGx1Z2luRmxlZXQpLnRvVGhyb3dFcnJvcigvSW52YWxpZCB2YWx1ZTogLisgZm9yIHByb3BlcnR5ICdkZWFkbGluZUdyb3VwcycvKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdwYXNzZXMgd2l0aCB2YWxpZCBncm91cCBuYW1lJywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGZ1bmN0aW9uIGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KCkge1xuICAgICAgICAgIG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICAgICAgZGVhZGxpbmVHcm91cHM6IFtncm91cE5hbWVdLFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QoY3JlYXRlU3BvdEV2ZW50UGx1Z2luRmxlZXQpLm5vdC50b1Rocm93RXJyb3IoKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3JlZ2lvbicsICgpID0+IHtcbiAgICAgIHRlc3QuZWFjaChbXG4gICAgICAgICdub25lJywgLy8gcmVnaW9uIGFzICdub25lJ1xuICAgICAgICAnYWxsJywgLy8gcmVnaW9uIGFzICdhbGwnXG4gICAgICAgICd1bnJlY29nbml6ZWQnLCAvLyByZWdpb24gYXMgJ3VucmVjb2duaXplZCdcbiAgICAgICAgJ25vbmVAMTIzJywgLy8gcmVnaW9uIHdpdGggaW52YWxpZCBjaGFyYWN0ZXJzXG4gICAgICAgICdOb25lJywgLy8gcmVnaW9uIHdpdGggY2FzZS1pbnNlbnNpdGl2ZSBuYW1lXG4gICAgICBdKSgndGhyb3dzIHdpdGggJXMnLCAoZGVhZGxpbmVSZWdpb246IHN0cmluZykgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGZ1bmN0aW9uIGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KCkge1xuICAgICAgICAgIG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgICAgICBkZWFkbGluZVJlZ2lvbjogZGVhZGxpbmVSZWdpb24sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChjcmVhdGVTcG90RXZlbnRQbHVnaW5GbGVldCkudG9UaHJvd0Vycm9yKC9JbnZhbGlkIHZhbHVlOiAuKyBmb3IgcHJvcGVydHkgJ2RlYWRsaW5lUmVnaW9uJy8pO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ3Bhc3NlcyB3aXRoIHJlc2VydmVkIG5hbWUgYXMgc3Vic3RyaW5nJywgKCkgPT4ge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBjb25zdCBkZWFkbGluZVJlZ2lvbiA9ICdub25lMTIzJztcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGZ1bmN0aW9uIGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KCkge1xuICAgICAgICAgIG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgICAgICBkZWFkbGluZVJlZ2lvbjogZGVhZGxpbmVSZWdpb24sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChjcmVhdGVTcG90RXZlbnRQbHVnaW5GbGVldCkubm90LnRvVGhyb3dFcnJvcigpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnQmxvY2sgRGV2aWNlIFRlc3RzJywgKCkgPT4ge1xuICAgICAgdGVzdCgnV2FybmluZyBpZiBubyBCbG9ja0RldmljZXMgcHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3QgZmxlZXQgPSBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3BvdEZsZWV0U3RhY2ssICdTcG90RmxlZXQnLCB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgICAgIGluc3RhbmNlVHlwZXMsXG4gICAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICAgIG1heENhcGFjaXR5LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChmbGVldC5ub2RlLm1ldGFkYXRhRW50cnlbMF0udHlwZSkudG9NYXRjaChBcnRpZmFjdE1ldGFkYXRhRW50cnlUeXBlLldBUk4pO1xuICAgICAgICBleHBlY3QoZmxlZXQubm9kZS5tZXRhZGF0YUVudHJ5WzBdLmRhdGEpLnRvTWF0Y2goJ2JlaW5nIGNyZWF0ZWQgd2l0aG91dCBiZWluZyBwcm92aWRlZCBhbnkgYmxvY2sgZGV2aWNlcyBzbyB0aGUgU291cmNlIEFNSVxcJ3MgZGV2aWNlcyB3aWxsIGJlIHVzZWQuIFdvcmtlcnMgY2FuIGhhdmUgYWNjZXNzIHRvIHNlbnNpdGl2ZSBkYXRhIHNvIGl0IGlzIHJlY29tbWVuZGVkIHRvIGVpdGhlciBleHBsaWNpdGx5IGVuY3J5cHQgdGhlIGRldmljZXMgb24gdGhlIHdvcmtlciBmbGVldCBvciB0byBlbnN1cmUgdGhlIHNvdXJjZSBBTUlcXCdzIERyaXZlcyBhcmUgZW5jcnlwdGVkLicpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ05vIFdhcm5pbmdzIGlmIEVuY3J5cHRlZCBCbG9ja0RldmljZXMgUHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgICAgIGNvbnN0IFZPTFVNRV9TSVpFID0gNTA7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgJ1Nwb3RGbGVldCcsIHtcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgICAgYmxvY2tEZXZpY2VzOiBbIHtcbiAgICAgICAgICAgIGRldmljZU5hbWU6ICcvZGV2L3h2ZGEnLFxuICAgICAgICAgICAgdm9sdW1lOiBCbG9ja0RldmljZVZvbHVtZS5lYnMoIFZPTFVNRV9TSVpFLCB7ZW5jcnlwdGVkOiB0cnVlfSksXG4gICAgICAgICAgfV0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vVEhFTlxuICAgICAgICBleHBlY3QoZmxlZXQubm9kZS5tZXRhZGF0YUVudHJ5KS50b0hhdmVMZW5ndGgoMCk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnV2FybmluZ3MgaWYgbm9uLUVuY3J5cHRlZCBCbG9ja0RldmljZXMgUHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgICAgIGNvbnN0IFZPTFVNRV9TSVpFID0gNTA7XG4gICAgICAgIGNvbnN0IERFVklDRV9OQU1FID0gJy9kZXYveHZkYSc7XG4gICAgICAgIGNvbnN0IGlkID0gJ1Nwb3RGbGVldCc7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBmbGVldCA9IG5ldyBTcG90RXZlbnRQbHVnaW5GbGVldChzcG90RmxlZXRTdGFjaywgaWQsIHtcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgICAgZGVhZGxpbmVHcm91cHMsXG4gICAgICAgICAgaW5zdGFuY2VUeXBlcyxcbiAgICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICAgICAgbWF4Q2FwYWNpdHksXG4gICAgICAgICAgYmxvY2tEZXZpY2VzOiBbIHtcbiAgICAgICAgICAgIGRldmljZU5hbWU6IERFVklDRV9OQU1FLFxuICAgICAgICAgICAgdm9sdW1lOiBCbG9ja0RldmljZVZvbHVtZS5lYnMoIFZPTFVNRV9TSVpFLCB7ZW5jcnlwdGVkOiBmYWxzZX0pLFxuICAgICAgICAgIH1dLFxuICAgICAgICB9KTtcblxuICAgICAgICAvL1RIRU5cbiAgICAgICAgZXhwZWN0KGZsZWV0Lm5vZGUubWV0YWRhdGFFbnRyeVswXS50eXBlKS50b01hdGNoKEFydGlmYWN0TWV0YWRhdGFFbnRyeVR5cGUuV0FSTik7XG4gICAgICAgIGV4cGVjdChmbGVldC5ub2RlLm1ldGFkYXRhRW50cnlbMF0uZGF0YSkudG9NYXRjaChgVGhlIEJsb2NrRGV2aWNlIFxcXCIke0RFVklDRV9OQU1FfVxcXCIgb24gdGhlIHNwb3QtZmxlZXQgJHtpZH0gaXMgbm90IGVuY3J5cHRlZC4gV29ya2VycyBjYW4gaGF2ZSBhY2Nlc3MgdG8gc2Vuc2l0aXZlIGRhdGEgc28gaXQgaXMgcmVjb21tZW5kZWQgdG8gZW5jcnlwdCB0aGUgZGV2aWNlcyBvbiB0aGUgd29ya2VyIGZsZWV0LmApO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ1dhcm5pbmdzIGZvciBCbG9ja0RldmljZXMgd2l0aG91dCBlbmNyeXB0aW9uIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICAgICAgY29uc3QgVk9MVU1FX1NJWkUgPSA1MDtcbiAgICAgICAgY29uc3QgREVWSUNFX05BTUUgPSAnL2Rldi94dmRhJztcbiAgICAgICAgY29uc3QgaWQgPSAnU3BvdEZsZWV0JztcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCBpZCwge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgICBibG9ja0RldmljZXM6IFsge1xuICAgICAgICAgICAgZGV2aWNlTmFtZTogREVWSUNFX05BTUUsXG4gICAgICAgICAgICB2b2x1bWU6IEJsb2NrRGV2aWNlVm9sdW1lLmVicyggVk9MVU1FX1NJWkUgKSxcbiAgICAgICAgICB9XSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy9USEVOXG4gICAgICAgIGV4cGVjdChmbGVldC5ub2RlLm1ldGFkYXRhRW50cnlbMF0udHlwZSkudG9NYXRjaChBcnRpZmFjdE1ldGFkYXRhRW50cnlUeXBlLldBUk4pO1xuICAgICAgICBleHBlY3QoZmxlZXQubm9kZS5tZXRhZGF0YUVudHJ5WzBdLmRhdGEpLnRvTWF0Y2goYFRoZSBCbG9ja0RldmljZSBcXFwiJHtERVZJQ0VfTkFNRX1cXFwiIG9uIHRoZSBzcG90LWZsZWV0ICR7aWR9IGlzIG5vdCBlbmNyeXB0ZWQuIFdvcmtlcnMgY2FuIGhhdmUgYWNjZXNzIHRvIHNlbnNpdGl2ZSBkYXRhIHNvIGl0IGlzIHJlY29tbWVuZGVkIHRvIGVuY3J5cHQgdGhlIGRldmljZXMgb24gdGhlIHdvcmtlciBmbGVldC5gKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdObyB3YXJuaW5ncyBmb3IgRXBoZW1lcmFsIGJsb2NrRGV2aWNlVm9sdW1lcycsICgpID0+IHtcbiAgICAgICAgY29uc3QgREVWSUNFX05BTUUgPSAnL2Rldi94dmRhJztcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgICBibG9ja0RldmljZXM6IFsge1xuICAgICAgICAgICAgZGV2aWNlTmFtZTogREVWSUNFX05BTUUsXG4gICAgICAgICAgICB2b2x1bWU6IEJsb2NrRGV2aWNlVm9sdW1lLmVwaGVtZXJhbCggMCApLFxuICAgICAgICAgIH1dLFxuICAgICAgICB9KTtcblxuICAgICAgICAvL1RIRU5cbiAgICAgICAgZXhwZWN0KGZsZWV0Lm5vZGUubWV0YWRhdGFFbnRyeSkudG9IYXZlTGVuZ3RoKDApO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ05vIHdhcm5pbmdzIGZvciBTdXBwcmVzc2VkIGJsb2NrRGV2aWNlVm9sdW1lcycsICgpID0+IHtcbiAgICAgICAgY29uc3QgREVWSUNFX05BTUUgPSAnL2Rldi94dmRhJztcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHNwb3RGbGVldFN0YWNrLCAnU3BvdEZsZWV0Jywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICByZW5kZXJRdWV1ZSxcbiAgICAgICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzLFxuICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICBtYXhDYXBhY2l0eSxcbiAgICAgICAgICBibG9ja0RldmljZXM6IFsge1xuICAgICAgICAgICAgZGV2aWNlTmFtZTogREVWSUNFX05BTUUsXG4gICAgICAgICAgICB2b2x1bWU6IEJsb2NrRGV2aWNlVm9sdW1lLm5vRGV2aWNlKCksXG4gICAgICAgICAgfV0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vVEhFTlxuICAgICAgICBleHBlY3QoZmxlZXQubm9kZS5tZXRhZGF0YUVudHJ5KS50b0hhdmVMZW5ndGgoMCk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgndGhyb3dzIGlmIGJsb2NrIGRldmljZXMgd2l0aG91dCBpb3BzIGFuZCB3cm9uZyB2b2x1bWUgdHlwZScsICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgZGV2aWNlTmFtZSA9ICcvZGV2L3h2ZGEnO1xuICAgICAgICBjb25zdCB2b2x1bWVTaXplID0gNTA7XG4gICAgICAgIGNvbnN0IHZvbHVtZVR5cGUgPSBFYnNEZXZpY2VWb2x1bWVUeXBlLklPMTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGZ1bmN0aW9uIGNyZWF0ZVNwb3RFdmVudFBsdWdpbkZsZWV0KCkge1xuICAgICAgICAgIHJldHVybiBuZXcgU3BvdEV2ZW50UGx1Z2luRmxlZXQoc3RhY2ssICdTcG90RXZlbnRQbHVnaW5GbGVldCcsIHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgICAgZGVhZGxpbmVHcm91cHM6IFtcbiAgICAgICAgICAgICAgZ3JvdXBOYW1lLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGluc3RhbmNlVHlwZXM6IFtcbiAgICAgICAgICAgICAgSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDMsIEluc3RhbmNlU2l6ZS5MQVJHRSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgd29ya2VyTWFjaGluZUltYWdlLFxuICAgICAgICAgICAgbWF4Q2FwYWNpdHk6IDEsXG4gICAgICAgICAgICBibG9ja0RldmljZXM6IFt7XG4gICAgICAgICAgICAgIGRldmljZU5hbWUsXG4gICAgICAgICAgICAgIHZvbHVtZTogQmxvY2tEZXZpY2VWb2x1bWUuZWJzKHZvbHVtZVNpemUsIHtcbiAgICAgICAgICAgICAgICB2b2x1bWVUeXBlLFxuICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIH1dLFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QoY3JlYXRlU3BvdEV2ZW50UGx1Z2luRmxlZXQpLnRvVGhyb3dFcnJvcigvaW9wcyBwcm9wZXJ0eSBpcyByZXF1aXJlZCB3aXRoIHZvbHVtZVR5cGU6IEVic0RldmljZVZvbHVtZVR5cGUuSU8xLyk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnd2FybmluZyBpZiBibG9jayBkZXZpY2VzIHdpdGggaW9wcyBhbmQgd3Jvbmcgdm9sdW1lIHR5cGUnLCAoKSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IGRldmljZU5hbWUgPSAnL2Rldi94dmRhJztcbiAgICAgICAgY29uc3Qgdm9sdW1lU2l6ZSA9IDUwO1xuICAgICAgICBjb25zdCBpb3BzID0gMTAwO1xuICAgICAgICBjb25zdCB2b2x1bWVUeXBlID0gRWJzRGV2aWNlVm9sdW1lVHlwZS5TVEFOREFSRDtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHN0YWNrLCAnU3BvdEV2ZW50UGx1Z2luRmxlZXQnLCB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgICAgIGRlYWRsaW5lR3JvdXBzOiBbXG4gICAgICAgICAgICBncm91cE5hbWUsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzOiBbXG4gICAgICAgICAgICBJbnN0YW5jZVR5cGUub2YoSW5zdGFuY2VDbGFzcy5UMywgSW5zdGFuY2VTaXplLkxBUkdFKSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHdvcmtlck1hY2hpbmVJbWFnZSxcbiAgICAgICAgICBtYXhDYXBhY2l0eTogMSxcbiAgICAgICAgICBibG9ja0RldmljZXM6IFt7XG4gICAgICAgICAgICBkZXZpY2VOYW1lLFxuICAgICAgICAgICAgdm9sdW1lOiBCbG9ja0RldmljZVZvbHVtZS5lYnModm9sdW1lU2l6ZSwge1xuICAgICAgICAgICAgICBpb3BzLFxuICAgICAgICAgICAgICB2b2x1bWVUeXBlLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgfV0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KGZsZWV0Lm5vZGUubWV0YWRhdGFFbnRyeVswXS50eXBlKS50b01hdGNoKEFydGlmYWN0TWV0YWRhdGFFbnRyeVR5cGUuV0FSTik7XG4gICAgICAgIGV4cGVjdChmbGVldC5ub2RlLm1ldGFkYXRhRW50cnlbMF0uZGF0YSkudG9NYXRjaCgnaW9wcyB3aWxsIGJlIGlnbm9yZWQgd2l0aG91dCB2b2x1bWVUeXBlOiBFYnNEZXZpY2VWb2x1bWVUeXBlLklPMScpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xufSk7XG4iXX0=