"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const sinon = require("sinon");
const lib_1 = require("../../core/lib");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
describe('DocumentDB', () => {
    let stack;
    let vpc;
    let database;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        if (!database.secret) {
            throw new Error('secret cannot be null');
        }
    });
    test('Grants access to Document DB Secret', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: { Ref: 'DbClusterSecretAttachment4201A1ED' },
                    }],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=(');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            ToPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'DbClusterInstance155835CE5',
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency to attributes', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency throws when cluster implementation changed', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        docdb.node.defaultChild = database; // Trick addChildDependency() into thinking this is a real construct.
        // THEN
        expect(() => {
            connection.addChildDependency(instance);
        }).toThrowError(/The internal implementation of the AWS CDK's DocumentDB cluster construct may have changed./);
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds warning annotation when a security group cannot be added due to unsupported IDatabaseCluster implementation', () => {
        // GIVEN
        class FakeDatabaseCluster extends core_1.Resource {
            constructor(scope, id) {
                super(scope, id);
                this.clusterIdentifier = '';
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new aws_docdb_1.Endpoint('address', 123);
                this.clusterReadEndpoint = new aws_docdb_1.Endpoint('readAddress', 123);
                this.instanceEndpoints = [];
                this.securityGroupId = '';
                this.connections = new aws_ec2_1.Connections();
                this.stack = core_1.Stack.of(scope);
                this.env = { account: this.stack.account, region: this.stack.region };
            }
            asSecretAttachmentTarget() {
                throw new Error('Method not implemented.');
            }
        }
        const fakeDatabase = new FakeDatabaseCluster(stack, 'FakeDatabase');
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: fakeDatabase, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(fakeDatabase.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(new RegExp(`Failed to add the following security groups to ${fakeDatabase.node.id}: .*\\. ` +
                    'The \\"database\\" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.')),
            }),
        ]));
    });
    // This test can be removed once the following CDK PR is merged:
    // https://github.com/aws/aws-cdk/pull/13290
    test('adds warning annotation when a security group cannot be added due to implementation changes in DatabaseCluster', () => {
        // GIVEN
        if (!database.node.tryRemoveChild('Resource')) {
            throw new Error('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed. The addSecurityGroup method needs to be updated.');
        }
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(database.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(new RegExp(`Failed to add the following security groups to ${database.node.id}: .*\\. ` +
                    'The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.')),
            }),
        ]));
    });
    test('Document DB connection is pointed to correct construct', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(connection.databaseConstruct).toEqual(database);
    });
});
describe('DocumentDB Version Checks', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    });
    test('Compatible version', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadataEntry.length).toBe(0);
    });
    test('When from attributes', () => {
        // GIVEN
        const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
        const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
            clusterEndpointAddress: '1.2.3.4',
            clusterIdentifier: 'foo',
            instanceEndpointAddresses: ['1.2.3.5'],
            instanceIdentifiers: ['i0'],
            port: 27001,
            readerEndpointAddress: '1.2.3.6',
            securityGroup: sg,
        });
        // WHEN
        const databaseConnection = lib_2.DatabaseConnection.forDocDB({ database, login: secret });
        // THEN
        expect(database.node.metadataEntry.length).toBe(0);
        expect(databaseConnection.databaseConstruct).toBeUndefined();
    });
    test('No engineVersion given', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:error',
                data: 'engineVersion must be 3.6.0 to be compatible with Deadline',
            }),
        ]));
    });
    test('engineVersion not 3.6.0', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '4.0.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:error',
                data: 'engineVersion must be 3.6.0 to be compatible with Deadline',
            }),
        ]));
    });
});
describe('MongoDB', () => {
    let stack;
    let vpc;
    let database;
    let clientCert;
    let dnsZone;
    let serverCert;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const hostname = 'mongo';
        const zoneName = 'deadline.internal';
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        clientCert = new lib_1.X509CertificatePem(stack, 'ClientCert', {
            subject: {
                cn: 'dbuser',
                ou: 'TestClient',
            },
            signingCertificate: caCert,
        });
        database = new lib_1.MongoDbInstance(stack, 'MongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname,
                dnsZone,
                serverCertificate: serverCert,
            },
        });
    });
    test('Grants access to certificate Secrets', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'ClientCert',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: 'ClientCertPassphrase8A71E1E1',
                        },
                    },
                ],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true');
        expect(userData).toContain('["--dbhost"]="mongo.deadline.internal"');
        expect(userData).toContain('["--dbport"]=27017');
        expect(userData).toContain('["--dbclientcert"]=');
        expect(userData).toContain('["--dbcertpass"]=$CERT_PASSWORD');
    });
    test('addConnectionDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addConnectionDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_deadline_database(){\n');
        expect(userData).toContain('\nexport -f configure_deadline_database');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nexport DB_CERT_FILE=');
        expect(userData).toContain('\nexport DB_CERT_PASSWORD=');
    });
    test('defines required container environment variables', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_URI');
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_PASSWORD_URI');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 27017,
            ToPort: 27017,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'MongoDbServerAsgInstanceSecurityGroupCE623335',
                    'GroupId',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
                'MongoDbServerAsgASG47B3D94E',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds security group', () => {
        // GIVEN
        const dbSpy = sinon.spy(database, 'addSecurityGroup');
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(dbSpy.calledOnce).toBeTruthy();
    });
    test('Mongo DB connection is pointed to correct construct', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.databaseConstruct).toEqual(database.mongoDataVolume);
    });
    test('Mongo DB imported from attributes', () => {
        // GIVEN
        const volume = aws_ec2_1.Volume.fromVolumeAttributes(stack, 'Volume', {
            availabilityZone: 'dummy zone',
            volumeId: 'vol-05abe246af',
        });
        const mongoDB = new lib_1.MongoDbInstance(stack, 'ImportedMongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname: 'mongo',
                dnsZone,
                serverCertificate: serverCert,
                mongoDataVolume: { volume },
            },
        });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database: mongoDB, clientCertificate: clientCert });
        // THEN
        expect(connection.databaseConstruct).toBeUndefined();
    });
});
//# sourceMappingURL=data:application/json;base64,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