"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const lib_1 = require("../lib");
const spot_event_plugin_fleet_1 = require("../lib/spot-event-plugin-fleet");
describe('ConfigureSpotEventPlugin', () => {
    let stack;
    let vpc;
    let region;
    let renderQueue;
    let version;
    let app;
    let fleet;
    let groupName;
    const workerMachineImage = new aws_ec2_1.GenericWindowsImage({
        'us-east-1': 'ami-any',
    });
    beforeEach(() => {
        region = 'us-east-1';
        app = new core_1.App();
        stack = new core_1.Stack(app, 'stack', {
            env: {
                region,
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        groupName = 'group_name1';
        fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
            vpc,
            renderQueue: renderQueue,
            deadlineGroups: [
                groupName,
            ],
            instanceTypes: [
                aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
            ],
            workerMachineImage,
            maxCapacity: 1,
        });
    });
    describe('creates a custom resource', () => {
        test('with default spot event plugin properties', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'Disabled',
                    DeleteInterruptedSlaves: false,
                    DeleteTerminatedSlaves: false,
                    IdleShutdown: 10,
                    Logging: 'Standard',
                    PreJobTaskMode: 'Conservative',
                    Region: core_1.Stack.of(renderQueue).region,
                    ResourceTracker: true,
                    StaggerInstances: 50,
                    State: 'Global Enabled',
                    StrictHardCap: false,
                }),
            })));
        });
        test('with custom spot event plugin properties', () => {
            // GIVEN
            const configuration = {
                awsInstanceStatus: lib_1.SpotEventPluginDisplayInstanceStatus.EXTRA_INFO_0,
                deleteEC2SpotInterruptedWorkers: true,
                deleteSEPTerminatedWorkers: true,
                idleShutdown: core_1.Duration.minutes(20),
                loggingLevel: lib_1.SpotEventPluginLoggingLevel.VERBOSE,
                preJobTaskMode: lib_1.SpotEventPluginPreJobTaskMode.NORMAL,
                region: 'us-west-2',
                enableResourceTracker: false,
                maximumInstancesStartedPerCycle: 10,
                state: lib_1.SpotEventPluginState.DISABLED,
                strictHardCap: true,
            };
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'ExtraInfo0',
                    DeleteInterruptedSlaves: true,
                    DeleteTerminatedSlaves: true,
                    IdleShutdown: 20,
                    Logging: 'Verbose',
                    PreJobTaskMode: 'Normal',
                    Region: 'us-west-2',
                    ResourceTracker: false,
                    StaggerInstances: 10,
                    State: 'Disabled',
                    StrictHardCap: true,
                }),
            })));
        });
        test('without spot fleets', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: assert_1.ABSENT,
            }));
        });
        test('provides RQ connection parameters to custom resource', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueue.endpoint.hostname),
                    port: stack.resolve(renderQueue.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueue.endpoint.applicationProtocol.toString()),
                }),
            })));
        });
        test('with default spot fleet request configuration', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: {
                    [groupName]: {
                        AllocationStrategy: 'lowestPrice',
                        IamFleetRole: stack.resolve(fleet.fleetRole.roleArn),
                        LaunchTemplateConfigs: [
                            {
                                LaunchTemplateSpecification: {
                                    Version: '$Latest',
                                    LaunchTemplateId: stack.resolve(fleet.launchTemplate.launchTemplateId),
                                },
                            },
                        ],
                        TagSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            ResourceType: 'spot-fleet-request',
                            Tags: assert_1.arrayWith({
                                Key: rfdkTag.name,
                                Value: rfdkTag.value,
                            }),
                        })),
                    },
                },
            }));
        });
        test('adds policies to the render queue', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: [
                                'arn:aws:ec2:*:*:spot-fleet-request/*',
                                'arn:aws:ec2:*:*:volume/*',
                            ],
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('adds resource tracker policy even if rt disabled', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration: {
                    enableResourceTracker: false,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
        });
        test.each([
            undefined,
            [],
        ])('without spot fleet', (noFleets) => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: noFleets,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.ABSENT,
            })));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: 'arn:aws:ec2:*:*:spot-fleet-request/*',
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('fleet with validUntil', () => {
            // GIVEN
            const validUntil = core_1.Expiration.atDate(new Date(2022, 11, 17));
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                validUntil,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        ValidUntil: validUntil.date.toISOString(),
                    }),
                }),
            })));
        });
    });
    test('only one object allowed per render queue', () => {
        // GIVEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/Only one ConfigureSpotEventPlugin construct is allowed per render queue./);
    });
    test('can create multiple objects with different render queues', () => {
        // GIVEN
        const renderQueue2 = new lib_1.RenderQueue(stack, 'RQ2', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository2', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
            vpc,
            renderQueue: renderQueue2,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('Custom::RFDK_ConfigureSpotEventPlugin', 2));
    });
    test('throws with not supported render queue', () => {
        // GIVEN
        const invalidRenderQueue = {};
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: invalidRenderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/The provided render queue is not an instance of RenderQueue class. Some functionality is not supported./);
    });
    test('tagSpecifications returns undefined if fleet does not have tags', () => {
        // GIVEN
        const mockFleet = {
            tags: {
                hasTags: jest.fn().mockReturnValue(false),
            },
        };
        const mockedFleet = mockFleet;
        const config = new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = stack.resolve(config['tagSpecifications'](mockedFleet, lib_1.SpotFleetResourceType.INSTANCE));
        // THEN
        expect(result).toBeUndefined();
    });
    describe('with TLS', () => {
        let renderQueueWithTls;
        let caCert;
        beforeEach(() => {
            const host = 'renderqueue';
            const zoneName = 'deadline-test.internal';
            caCert = new core_2.X509CertificatePem(stack, 'RootCA', {
                subject: {
                    cn: 'SampleRootCA',
                },
            });
            renderQueueWithTls = new lib_1.RenderQueue(stack, 'RQ with TLS', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository2', {
                    vpc,
                    version,
                }),
                version,
                hostname: {
                    zone: new aws_route53_1.PrivateHostedZone(stack, 'DnsZone', {
                        vpc,
                        zoneName: zoneName,
                    }),
                    hostname: host,
                },
                trafficEncryption: {
                    externalTLS: {
                        rfdkCertificate: new core_2.X509CertificatePem(stack, 'RQCert', {
                            subject: {
                                cn: `${host}.${zoneName}`,
                            },
                            signingCertificate: caCert,
                        }),
                    },
                },
            });
        });
        test('Lambda role can get the ca secret', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(renderQueueWithTls.certChain.secretArn),
                        },
                    ],
                },
                Roles: [
                    {
                        Ref: 'ConfigureSpotEventPluginConfiguratorServiceRole341B4735',
                    },
                ],
            }));
        });
        test('creates a custom resource with connection', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueueWithTls.endpoint.hostname),
                    port: stack.resolve(renderQueueWithTls.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueueWithTls.endpoint.applicationProtocol.toString()),
                    caCertificateArn: stack.resolve(renderQueueWithTls.certChain.secretArn),
                }),
            })));
        });
    });
    test('throws with the same group name', () => {
        // WHEN
        function createConfigureSpotEventPlugin() {
            const duplicateFleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'DuplicateSpotFleet', {
                vpc,
                renderQueue,
                workerMachineImage: fleet.machineImage,
                instanceTypes: fleet.instanceTypes,
                maxCapacity: fleet.maxCapacity,
                deadlineGroups: fleet.deadlineGroups,
            });
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                    duplicateFleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(`Bad Group Name: ${groupName}. Group names in Spot Fleet Request Configurations should be unique.`);
    });
    test('uses selected subnets', () => {
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'configure-spot-event-plugin.configureSEP',
            VpcConfig: {
                SubnetIds: [
                    stack.resolve(vpc.privateSubnets[0].subnetId),
                ],
            },
        }));
    });
    describe('throws with wrong deadline version', () => {
        test.each([
            ['10.1.9'],
            ['10.1.10'],
        ])('%s', (versionString) => {
            // GIVEN
            const newStack = new core_1.Stack(app, 'NewStack');
            version = new lib_1.VersionQuery(newStack, 'OldVersion', {
                version: versionString,
            });
            renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(newStack, 'Repository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                }),
                trafficEncryption: { externalTLS: { enabled: false } },
                version,
            });
            // WHEN
            function createConfigureSpotEventPlugin() {
                new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                    vpc,
                    renderQueue: renderQueue,
                    spotFleets: [
                        fleet,
                    ],
                });
            }
            // THEN
            expect(createConfigureSpotEventPlugin).toThrowError(`Minimum supported Deadline version for ConfigureSpotEventPlugin is 10.1.12.0. Received: ${versionString}.`);
        });
    });
    test('does not throw with min deadline version', () => {
        // GIVEN
        const versionString = '10.1.12';
        const newStack = new core_1.Stack(app, 'NewStack');
        version = new lib_1.VersionQuery(newStack, 'OldVersion', {
            version: versionString,
        });
        renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(newStack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(newStack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).not.toThrow();
    });
    describe('secrets management enabled', () => {
        beforeEach(() => {
            region = 'us-east-1';
            app = new core_1.App();
            stack = new core_1.Stack(app, 'stack', {
                env: {
                    region,
                },
            });
            vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
            version = new lib_1.VersionQuery(stack, 'Version');
            renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository', {
                    vpc,
                    version,
                }),
                version,
            });
            groupName = 'group_name1';
        });
        test('a fleet without vpcSubnets specified => warns about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            expect(fleet.node.metadataEntry).toContainEqual(expect.objectContaining({
                type: 'aws:cdk:warning',
                data: 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components',
            }));
        });
        test('a fleet with vpcSubnets specified => does not warn about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleetWithSubnets', {
                vpc,
                vpcSubnets: {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                },
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            expect(fleet.node.metadataEntry).not.toContainEqual(expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(/dedicated subnet/i),
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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