"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 *
 * @stability stable
 */
class VersionQuery extends VersionQueryBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.expression = props === null || props === void 0 ? void 0 : props.version;
        const match = ((_b = props === null || props === void 0 ? void 0 : props.version) !== null && _b !== void 0 ? _b : '').match(VersionQuery.EXPRESSION_REGEX);
        if (match === null) {
            throw new Error(`Invalid version expression "${props.version}`);
        }
        this.pinnedVersionComponents = (match
            // First capture group is the entire matched string, so slice it off
            .slice(1)
            // Capture groups that are not matched return as undefined, so we filter them out
            .filter(component => component)
            // Parse strings to numbers
            .map(component => Number(component)));
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'version-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props === null || props === void 0 ? void 0 : props.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props === null || props === void 0 ? void 0 : props.version),
        };
        this.deadlineResource = new core_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = this.versionComponent({
            expressionIndex: 0,
            customResourceAttribute: 'MajorVersion',
        });
        this.minorVersion = this.versionComponent({
            expressionIndex: 1,
            customResourceAttribute: 'MinorVersion',
        });
        this.releaseVersion = this.versionComponent({
            expressionIndex: 2,
            customResourceAttribute: 'ReleaseVersion',
        });
        const installerBucket = aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', this.deadlineResource.getAttString('S3Bucket'));
        this.linuxInstallers = {
            patchVersion: core_1.Token.asNumber(this.deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: this.deadlineResource.getAttString('LinuxRepositoryInstaller'),
                s3Bucket: installerBucket,
            },
            client: {
                objectKey: this.deadlineResource.getAttString('LinuxClientInstaller'),
                s3Bucket: installerBucket,
            },
        };
    }
    versionComponent(args) {
        const { expressionIndex, customResourceAttribute } = args;
        return this.pinnedVersionComponents.length > expressionIndex
            ? this.pinnedVersionComponents[expressionIndex]
            : core_1.Token.asNumber(this.deadlineResource.getAtt(customResourceAttribute));
    }
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     *
     * @stability stable
     */
    linuxFullVersionString() {
        const major = core_1.Token.isUnresolved(this.majorVersion) ? core_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = core_1.Token.isUnresolved(this.minorVersion) ? core_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = core_1.Token.isUnresolved(this.releaseVersion) ? core_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = core_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? core_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    /**
     * Returns whether this version is less than another version.
     *
     * @stability stable
     */
    isLessThan(other) {
        if (other.patchVersion !== 0) {
            throw new Error('Cannot compare a VersionQuery to a fully-qualified version with a non-zero patch number');
        }
        // We compare each component from highest order to lowest
        const componentGetters = [
            v => v.majorVersion,
            v => v.minorVersion,
            v => v.releaseVersion,
        ];
        for (const componentGetter of componentGetters) {
            const thisComponent = componentGetter(this);
            const otherComponent = componentGetter(other);
            if (core_1.Token.isUnresolved(thisComponent)) {
                // Unresolved components are unpinned. These will resolve to the latest and are not less than any provided
                // version
                return false;
            }
            else {
                const componentDiff = thisComponent - otherComponent;
                if (componentDiff !== 0) {
                    // If the components are different, return whether this component is smaller than the other component
                    return componentDiff < 0;
                }
            }
        }
        // If we've exited the loop naturally, it means all version components are pinned and equal. This means the version
        // is not less than the other, they are the same
        return false;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version === null || version === void 0 ? void 0 : version.split('.').map(x => parseInt(x));
        if (!components || (components === null || components === void 0 ? void 0 : components.length) != 4) {
            return false;
        }
        return true;
    }
    /**
     * A string representation of the version using the best available information at synthesis-time.
     *
     * This value is not guaranteed to be resolved, and is intended for output to CDK users.
     *
     * @stability stable
     */
    get versionString() {
        var _b;
        return (_b = this.expression) !== null && _b !== void 0 ? _b : '(latest)';
    }
}
exports.VersionQuery = VersionQuery;
_a = JSII_RTTI_SYMBOL_1;
VersionQuery[_a] = { fqn: "aws-rfdk.deadline.VersionQuery", version: "0.40.0" };
/**
 * Regular expression for valid version query expressions
 */
VersionQuery.EXPRESSION_REGEX = /^(?:(\d+)(?:\.(\d+)(?:\.(\d+)(?:\.(\d+))?)?)?)?$/;
//# sourceMappingURL=data:application/json;base64,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