"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const worker_configuration_1 = require("./worker-configuration");
/**
 * This construct reperesents a fleet from the Spot Fleet Request created by the Spot Event Plugin.
 *
 * This fleet is intended to be used as input for the {@link @aws-rfdk/deadline#ConfigureSpotEventPlugin} construct.
 *
 * The construct itself doesn't create the Spot Fleet Request, but deploys all the resources
 * required for the Spot Fleet Request and generates the Spot Fleet Configuration setting:
 * a one to one mapping between a Deadline Group and Spot Fleet Request Configurations.
 *
 * ![architecture diagram](/diagrams/deadline/SpotEventPluginFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Instance Role, corresponding IAM Policy and an Instance Profile.
 * - A Fleet Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 * - A security Group if security groups are not provided.
 * - An EC2 Launch Template for the Spot Fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
class SpotEventPluginFleet extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.defaultSubnets = !props.vpcSubnets;
        this.deadlineGroups = props.deadlineGroups.map(group => group.toLocaleLowerCase());
        this.deadlinePools = (_b = props.deadlinePools) === null || _b === void 0 ? void 0 : _b.map(pool => pool.toLocaleLowerCase());
        this.validateProps(props);
        this.securityGroups = (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [new aws_ec2_1.SecurityGroup(this, 'SpotFleetSecurityGroup', { vpc: props.vpc })];
        this.connections = new aws_ec2_1.Connections({ securityGroups: this.securityGroups });
        this.connections.allowToDefaultPort(props.renderQueue.endpoint);
        this.fleetInstanceRole = (_d = props.fleetInstanceRole) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(this, 'SpotFleetInstanceRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
            ],
            description: `Spot Fleet instance role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.fleetInstanceRole.roleName],
        });
        this.grantPrincipal = this.fleetInstanceRole;
        this.fleetRole = (_e = props.fleetRole) !== null && _e !== void 0 ? _e : new aws_iam_1.Role(this, 'SpotFleetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
            description: `Spot Fleet role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.blockDevices = props.blockDevices;
        this.subnets = props.vpc.selectSubnets((_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE });
        this.instanceTypes = props.instanceTypes;
        this.allocationStrategy = (_g = props.allocationStrategy) !== null && _g !== void 0 ? _g : spot_event_plugin_fleet_ref_1.SpotFleetAllocationStrategy.LOWEST_PRICE;
        this.maxCapacity = props.maxCapacity;
        this.validUntil = props.validUntil;
        this.keyName = props.keyName;
        const imageConfig = props.workerMachineImage.getImage(this);
        this.osType = imageConfig.osType;
        this.userData = (_h = props.userData) !== null && _h !== void 0 ? _h : imageConfig.userData;
        this.machineImage = props.workerMachineImage;
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this,
            cloudWatchLogSettings: {
                logGroupPrefix: SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: {
                groups: this.deadlineGroups,
                pools: this.deadlinePools,
                region: props.deadlineRegion,
            },
            userDataProvider: props.userDataProvider,
        });
        this.remoteControlPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + SpotEventPluginFleet.MAX_WORKERS_PER_HOST);
        this.tags = new core_1.TagManager(core_1.TagType.KEY_VALUE, 'RFDK::SpotEventPluginFleet');
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
        this.launchTemplate = this.createLaunchTemplate((_j = props.trackInstancesWithResourceTracker) !== null && _j !== void 0 ? _j : true);
        this._launchTemplateConfigs = this.createLaunchTemplateConfigs();
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowRemoteControlTo().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlFrom(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlFrom(other) {
        this.connections.allowFrom(other.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowRemoteControlFrom().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlTo(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlTo(other) {
        other.connections.allowTo(this.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    createLaunchTemplate(resourceTrackerEnabled) {
        const launchTemplate = new aws_ec2_1.LaunchTemplate(this, 'LaunchTemplate', {
            blockDevices: this.blockDevices,
            role: this.fleetInstanceRole,
            machineImage: this.machineImage,
            keyName: this.keyName,
            securityGroup: this.securityGroups[0],
            userData: this.userData,
        });
        if (this.securityGroups.length > 1) {
            launchTemplate.connections.addSecurityGroup(...this.securityGroups.slice(1));
        }
        core_1.Tags.of(launchTemplate).add(resourceTrackerEnabled ? 'DeadlineTrackedAWSResource' : 'DeadlineResourceTracker', 'SpotEventPlugin');
        return launchTemplate;
    }
    createLaunchTemplateConfigs() {
        const launchTemplateConfigs = [];
        // Create a launch template config for each instance type + subnet pair
        this.instanceTypes.forEach(instanceType => {
            this.subnets.subnetIds.forEach(subnetId => {
                launchTemplateConfigs.push({
                    LaunchTemplateSpecification: {
                        Version: aws_ec2_1.LaunchTemplateSpecialVersions.LATEST_VERSION,
                        LaunchTemplateId: this.launchTemplate.launchTemplateId,
                        LaunchTemplateName: this.launchTemplate.launchTemplateName,
                    },
                    Overrides: [{
                            InstanceType: instanceType.toString(),
                            SubnetId: subnetId,
                        }],
                });
            });
        });
        return launchTemplateConfigs;
    }
    validateProps(props) {
        this.validateFleetInstanceRole(props.fleetInstanceRole);
        this.validateInstanceTypes(props.instanceTypes);
        this.validateSubnets(props.vpc, props.vpcSubnets);
        this.validateGroups('deadlineGroups', this.deadlineGroups);
        this.validateRegion('deadlineRegion', props.deadlineRegion);
        this.validateBlockDevices(props.blockDevices);
    }
    validateFleetInstanceRole(fleetInstanceRole) {
        if (fleetInstanceRole) {
            if (core_1.Stack.of(fleetInstanceRole) !== core_1.Stack.of(this)) {
                throw new Error(`Fleet instance role should be created on the same stack as ${this.constructor.name} to avoid circular dependencies.`);
            }
        }
    }
    validateInstanceTypes(array) {
        if (array.length === 0) {
            throw new Error('At least one instance type is required for a Spot Fleet Request Configuration');
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const { subnets } = vpc.selectSubnets(vpcSubnets);
        if (subnets.length === 0) {
            core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(vpcSubnets)}', please use a different selection.`);
        }
    }
    validateGroups(property, array) {
        const regex = /^(?!none$)[a-z0-9-_]+$/g;
        if (array.length === 0) {
            throw new Error('At least one Deadline Group is required for a Spot Fleet Request Configuration');
        }
        array.forEach(value => {
            if (!regex.test(value)) {
                throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
            }
        });
    }
    validateRegion(property, region) {
        const regex = /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i;
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The spot-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                const { iops, volumeType } = device.volume.ebsDevice;
                if (!iops) {
                    if (volumeType === aws_ec2_1.EbsDeviceVolumeType.IO1) {
                        throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                    }
                }
                else if (volumeType !== aws_ec2_1.EbsDeviceVolumeType.IO1) {
                    core_1.Annotations.of(this).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the spot-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
}
exports.SpotEventPluginFleet = SpotEventPluginFleet;
_a = JSII_RTTI_SYMBOL_1;
SpotEventPluginFleet[_a] = { fqn: "aws-rfdk.deadline.SpotEventPluginFleet", version: "0.40.0" };
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
SpotEventPluginFleet.MAX_WORKERS_PER_HOST = 8;
//# sourceMappingURL=data:application/json;base64,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